# flcheck --
#
# This file implements package flcheck, a fusion logic modelchecker 
# using the CUDD and BuDDy BDD packages
#
# Copyright (C) 2009-2025  Antonio Cau, Ben Moszkowski and Helge Janicke

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

#
#

::bdd_tcl::load_file policy_library.tcl

#########################################################################
# Role-Based Access Control policy enforcement example
#########################################################################
#
# the set of users is { AC, HJ } 

proc users {} {
   return [list  AC HJ ]
}

# the set of roles { ADMIN, USER }

proc roles {} {
     return [list ADMIN USER]
}

# the set of all subjects is the union of users and roles

proc subjects {} {
    return [concat [users] [roles] ]
}

# the set of objects is { RM, SERVER } 
proc objects {} {
    return [list RM SERVER]
}

# the set of actions 
# { rm-activate-admin, rm-activate-user, server-create }

proc actions { object } {
    if { $object == "RM" } { 
        return [actions_activate] 
    } else {
      if {$object == "SERVER"} {
        return "CREATE"
      } else {
         return {}
      }
    }
}

proc actions_activate {} {
    set actions {}
    foreach a [roles] {
      lappend actions "ACTIVATE$a"
    }
     return $actions 
}


proc action_activate { role } {
   return "ACTIVATE$role"
}


# the user AC is assigned the role ADMIN 

proc rule_activate_1 {} {
    global consequences
    set c [autho_plus AC RM [action_activate ADMIN]]
    array set consequences [list $c 0]
    #puts "$c : $consequences($c)"
    return [r_sfb_0 true $c]
}

# the user HJ is assigned the role USER 

proc rule_activate_2 {} {
    global consequences
    set c [autho_plus HJ RM [action_activate USER]]
    array set consequences [list $c 0]
    return [r_sfb_0 true $c]
}

# the user HJ is temporarily promoted to act as ADMIN if user AC is ill

proc rule_activate_3 {} {
    global consequences
    set c [autho_plus HJ RM [action_activate ADMIN]]
    array set consequences [list $c 0]
    return [r_sfb_0 [ill AC] $c]
}

# any user that has called in sick cannot activate any role

proc rule_activate_4 {} {
   global consequences
   set l_rules {}
    foreach user [users] {
      foreach action [actions "RM"] {
          set x [ill $user]
          set c [autho_minus $user "RM" $action]
          array set consequences [list $c 0]
          #puts "$c : $consequences($c)"
          set rule [r_sfb_0 $x $c]
          lappend l_rules $rule
      }
    }
    set z [l_and $l_rules]
    return $z
}

# assign permission < SERVER, CREATE > to the role ADMIN 

proc rule_create_1 {} {
    global consequences
    set c [autho_plus ADMIN SERVER CREATE]
    array set consequences [list $c 0]
    return "[r_sfb_true $c]"
}


#
# Conflict Resolution, Decision Rule:
# As we define our RBAC example as a hybrid policy, e.g. both positive
# and negative authorisations are present in the same policy, we can
# create conflicts.  It is not generally necessary to remove conflicts
# between positive and negative rules, however there must be an
# unambiguous definition of which decision is being taken in case a
# conflict arises.  We capture this in a standard decision rule:

proc rule_conflict_resolution {} {
  set l_rules {}
    foreach subject [subjects] {
     foreach object [objects] {
      foreach action [actions $object] {
          set x1 "[not [autho_minus $subject $object $action]]"
          set x [and [autho_plus $subject $object $action] $x1]
            set y [autho_derived $subject $object $action]
            set rule [r_sfb_0 $y $x]
            lappend l_rules $rule
      }
     }
    }
    set z [l_and $l_rules]
    return $z
}

#\end{small}
#%\subsubsection{Conflict Analysis and Property Verification}
#For the analysis of the policy, we expand the policy into its normal
#form by expanding the sets and then complete the policy specification
#with a set of default rules of the form $\itlFalse \itlWFollows
#c(s,o,a)$ where $c\in\{\autho^d,\autho^-,\autho^+\},s\in S, o \in O, a
#\in A_o$ and $a(s,o,a)$ does not occur as any consequence. It has been
#shown in \cite{Siewe2005} that the resulting policy is a refinement of
#the original specification. We refer to this specification as the
#model $M$ of our policy.

proc default_rules {} {
   global consequences
    set l_rules {}
    foreach c [array names consequences] {
    #puts "$c : $consequences($c)"
        if { $consequences($c) == 1 } {
            set x [false]
            set rule [r_sfb_0 $x $c]
            lappend l_rules $rule
        }
    }
    if { [llength $l_rules] == 0 } {
      return "true"
    } else {
      set z [l_and $l_rules]
      return $z
    }
}

proc collect_set {} {
    global consequences
    array unset consequences
    foreach subject [subjects] {
     foreach object [objects] {
      foreach action [actions $object] {
          array set consequences [list [autho_plus $subject $object $action] 1]
          array set consequences [list [autho_minus $subject $object $action] 1]
      }
     }
    }
}


proc model_1_enf {} {
    collect_set
    set x [list \
            [rule_activate_1] \
            [rule_activate_2] \
            [rule_activate_3] \
            [rule_activate_4] \
            [rule_create_1] \
            [rule_conflict_resolution] \
          ]
    set y [default_rules]
    set y [and [equiv [ill HJ] [ill HJ]] $y]
    return [and [l_and $x] $y]
}

proc autho_plus { s o a } {
   set x "_"
   return "AUTHOP$x$s$x$o$x$a"
} 

proc autho_minus { s o a } {
   set x "_"
   return "AUTHON$x$s$x$o$x$a"
} 

proc autho_derived { s o a } {
   set x "_"
   return "AUTHO$x$s$x$o$x$a"
} 

proc ill { user } {
  set y "ILL"
  set x "_"
  return "$y$x$user"
}

proc not_ill { user } {
  set y "ILL"
  set x "_"
    return "(not $y$x$user)"
}

proc both_not_ill {} {
  set x "(and [not_ill HJ] [not_ill AC])"
  return "$x"
}

####################################################################

# input 0:  (not ill(ac), not ill(hj))
proc input_0_enf {} {
    return [list "(and [not_ill AC] [not_ill HJ])"]
}

# incremental enforcement
proc test_7_enf {} {
    global INPUT
    set INPUT [input_0_enf]
    return "[always_r [model_1_enf] ]"
}

# input 1: (not ill(ac), not ill(hj)) 
#          (ill(ac), not ill(hj))
proc input_1_enf {} {
    return [list "(and [not_ill AC] [not_ill HJ] )" \
                "(and [ill AC] [not_ill HJ] )"]
}

# incremental
proc test_8_enf {} {
    global INPUT
    set INPUT [input_1_enf]
    return "[always_r [model_1_enf]]"
}

# input 2: (not ill(ac), not ill(hj)) 
#          (ill(ac), not ill(hj)) 
#          (ill(ac), ill(hj))
#          (not ill(ac), ill(hj)) 
proc input_2_enf {} {
    return [list "(and [not_ill AC] [not_ill HJ] )" \
                "(and [ill AC] [not_ill HJ] )" \
                "(and [ill AC] [ill HJ] )" \
                "(and [not_ill AC] [ill HJ] )"]
}

# incremental
proc test_9_enf {} {
    global INPUT
    set INPUT [input_2_enf]
    return "[always_r [model_1_enf]]"
}

# interactive
# input should be of the form
# [list p] or end
# where p is
# true
# [not_ill AC]
# [not_ill AC] [not_ill HJ]
# [ill AC] [not_ill HJ]
# etc
#
proc test_10_i {} {
    return "[always_r [model_1_enf] ]"
}




