(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                Dimitar Guelev
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)



theory QPITL_Deep_GNF imports
 QPITL_Deep

begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="cvc4 z3 e spass vampire " ]
*)
text \<open>
This theory proves that all QPITL formulae can be written in Guarded Normal Form (GNF)
\cite{GuelevMoszkowski2022, bowman03:_decis_proced_compl_axiom_finit}.  
\<close>

section \<open>Guarded Normal Form\<close>

subsection \<open>Definitions\<close>

definition mutex
where " mutex F =
   (\<forall> k1 k2. 0 \<le> k1  \<and> k1 < k2  \<and> k2 < length  F \<longrightarrow> (\<turnstile> inot ((F! k1) iand (F! k2))))" 

definition mutex_single
 where " mutex_single f F =
         (\<forall>  k2. 0 \<le> k2  \<and> k2 < length  F \<longrightarrow> (\<turnstile> inot (f iand (F! k2)))) " 


definition add_to_now_single
 where 
   " add_to_now_single f L   = ( (map (\<lambda>x. ( f iand x) ) L)@ 
                                 (map (\<lambda>x. ( inot f iand x)) L))"


definition add_to_next_single
 where 
   " add_to_next_single f L   = ( (map (\<lambda>x. ( f ior x)) L)@ L)"


definition add_to_single
 where 
  " add_to_single f1 f2 L1 L2 = 
     filter (\<lambda> (x,y).  (\<exists> \<sigma>. \<sigma> \<Turnstile> x)) (zip (add_to_now_single f1 L1) (add_to_next_single f2 L2)) "


definition add_to_next_single_a
 where 
   " add_to_next_single_a f L   = ( (map (\<lambda>x. ( (if f = ifalse then x else f ior x))) L)@ L)"


definition add_to_now
where " add_to_now L1 = foldr (add_to_now_single) L1  [itrue] "
 

definition add_to_next
where " add_to_next L1 = foldr (add_to_next_single) L1 [ifalse]" 

definition add_to_next_a
where " add_to_next_a L1 = foldr (add_to_next_single_a) L1 [ifalse]" 


definition add_to
 where "add_to L1 L2 = filter (\<lambda> (x,y). (\<exists>\<sigma> . \<sigma> \<Turnstile> x)  )  (zip  (add_to_now L1) (add_to_next L1)) " 


fun GNF_empty :: " qpitl \<Rightarrow>  qpitl "
where
  " GNF_empty (ifalse)      = ifalse  "
| " GNF_empty ($n)          = ($n)  "
| " GNF_empty (f iimp g)    = ((inot (GNF_empty f)) ior (GNF_empty g)) "
| " GNF_empty (next f)      = (ifalse) "
| " GNF_empty (f schop g)   = ((GNF_empty f) iand (GNF_empty g))"
| " GNF_empty (schopstar f) = itrue " 
| " GNF_empty (omega f)     = ifalse "
| " GNF_empty (Ex n. f)     = (Ex n. (GNF_empty f)) " 


fun GNF_state :: "qpitl \<Rightarrow> qpitl list"
where
  " GNF_state (ifalse)      = [itrue]  "
| " GNF_state ($n)          = [($n)]  "
| " GNF_state (f iimp g)    = ( ((add_to_now (GNF_state f)   )) @ (GNF_state g))   "
| " GNF_state (next f)      = [itrue] "
| " GNF_state (f schop g)   = ( (GNF_state f) @ (map (\<lambda>x. x iand (GNF_empty f)) (GNF_state g)))"
| " GNF_state (schopstar f) = (GNF_state f) " 
| " GNF_state (omega f)     = (GNF_state f) "
| " GNF_state (Ex n. f)     = (map (\<lambda>x. (Ex n. x)) (GNF_state f))" 

fun GNF_next :: "qpitl \<Rightarrow>  qpitl list"
where
  " GNF_next (ifalse)      = [ifalse]  "
| " GNF_next ($n)      = [itrue]  "
| " GNF_next (f iimp g)    = ( (map (\<lambda>x. (inot x)) (add_to_next (GNF_next f) )) @ (GNF_next g))   "
| " GNF_next (next f)      = [f] "
| " GNF_next (f schop g)   = ( (map (\<lambda>x. x schop g) (GNF_next f)) @  (GNF_next g)) "
| " GNF_next (schopstar f) = (map (\<lambda>x. x schop (schopstar f)) (GNF_next f))  " 
| " GNF_next (omega f)     = (map (\<lambda>x. x schop (omega f)) (GNF_next f)) "
| " GNF_next (Ex n. f)     = (map (\<lambda>x. (Ex n. x)) (GNF_next f)) " 

fun GNF_next_a :: "qpitl \<Rightarrow> qpitl list"
where
  " GNF_next_a (ifalse)      = [ifalse]  "
| " GNF_next_a ($n)      = [itrue]  "
| " GNF_next_a (f iimp g)    = ( (map (\<lambda>x. (inot x)) (add_to_next_a (GNF_next_a f) )) @ (GNF_next_a g))   "
| " GNF_next_a (next f)      = [f] "
| " GNF_next_a (f schop g)   = ( (map (\<lambda>x. x schop g) (GNF_next_a f)) @  (GNF_next_a g)) "
| " GNF_next_a (schopstar f) = (map (\<lambda>x. x schop (schopstar f)) (GNF_next_a f))  " 
| " GNF_next_a (omega f)     = (map (\<lambda>x. x schop (omega f)) (GNF_next_a f)) "
| " GNF_next_a (Ex n. f)     = (map (\<lambda>x. (Ex n. f)) (GNF_next_a f)) " 


definition full_system 
 where "full_system F = ((\<turnstile> big_ior F) \<and> mutex F)" 


definition GNF_cons 
 where "GNF_cons ae a A =
         ((ae iand empty) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  a A  )))"

definition GNF
 where "GNF f =
        (GNF_cons (GNF_empty f) (GNF_state f) (GNF_next f)) " 


definition GNF_prop
 where "GNF_prop X =
        (\<exists> ae a A. 
          (state_qpitl ae) \<and> (state_qpitl_list a) \<and>
          full_system a \<and> length a = length A \<and>
          (\<turnstile> X ieqv GNF_cons ae a A)
        )" 


definition GNF_det 
 where " GNF_det f = 
         (GNF_cons (GNF_empty f) 
                   (foldr (add_to_now_single ) (GNF_state f) [itrue ])
                   (foldr (add_to_next_single) (GNF_next  f) [ifalse])
         ) " 


subsection \<open>Mutex lemmas\<close>

lemma mutex_single_empty: 
shows "mutex_single f [] "
unfolding mutex_single_def by auto

lemma mutex_single_Cons: 
 shows " mutex_single f (f1 # F) \<longleftrightarrow>
         (\<turnstile> inot (f iand f1)) \<and> mutex_single f F " 
unfolding mutex_single_def using less_Suc_eq_0_disj by auto


lemma mutex_single_append: 
 shows " mutex_single f (F1@ F2) \<longleftrightarrow> mutex_single f F1 \<and> mutex_single f F2 " 
proof (induct F1 arbitrary: F2)
case Nil
then show ?case unfolding mutex_single_def by simp
next
case (Cons a F1)
then show ?case  
   proof -
    have 1: "mutex_single f ((a # F1) @ F2)  \<longleftrightarrow>
             (\<turnstile> inot (f iand a)) \<and> mutex_single f (F1 @F2)" 
       by (simp add: mutex_single_Cons)
    have 2: "mutex_single f (F1 @F2) \<longleftrightarrow>  mutex_single f F1 \<and> mutex_single f F2" 
      using Cons by blast
    have 3: " (\<turnstile> inot (f iand a)) \<and> (mutex_single f F1 \<and> mutex_single f F2) \<longleftrightarrow>
              (mutex_single f (a # F1) \<and> mutex_single f F2)" 
      using mutex_single_Cons by blast
    show ?thesis 
    using "1" "2" "3" by presburger
 qed
qed



lemma mutex_empty: 
shows "mutex [] "
unfolding mutex_def by auto

lemma mutex_Cons:
shows "mutex (f # F1) \<longleftrightarrow>  mutex_single f F1 \<and> mutex F1  "
unfolding mutex_def mutex_single_def
using   less_Suc_eq_0_disj  
apply auto
apply (metis diff_Suc_1' nth_Cons')
by (metis diff_Suc_1' nth_Cons_Suc)



lemma mutex_append:
 shows " mutex (F1 @ F2) \<longleftrightarrow>  
         mutex F1 \<and> foldr (\<lambda>x y. (mutex_single x F2) \<and> y) F1 True  \<and> mutex F2 " 
proof (induct F1 arbitrary: F2)
case Nil
then show ?case 
by (simp add: mutex_empty)
next
case (Cons a F1)
then show ?case 
   proof -
    have 1: "mutex ((a # F1) @ F2) \<longleftrightarrow>
             mutex_single a F1 \<and> mutex_single a F2  \<and> mutex (F1@F2)"
        by (simp add: mutex_Cons mutex_single_append) 
    have 2: "foldr (\<lambda>x. (\<and>) (mutex_single x F2)) (a # F1) True \<longleftrightarrow>
             mutex_single a F2 \<and> foldr (\<lambda>x. (\<and>) (mutex_single x F2)) ( F1) True" 
        by simp
    have 3: "mutex (F1 @ F2) \<longleftrightarrow> (mutex F1 \<and> foldr (\<lambda>x. (\<and>) (mutex_single x F2)) F1 True \<and> mutex F2)" 
      using Cons by blast
    have 4: "mutex_single a F1 \<and> mutex_single a F2  \<and> mutex (F1@F2) \<longleftrightarrow>
             mutex_single a F1 \<and> mutex_single a F2  \<and> 
             (mutex F1 \<and> foldr (\<lambda>x. (\<and>) (mutex_single x F2)) F1 True \<and> mutex F2)" 
     using 1 3 by auto
    have 5: "mutex_single a F1 \<and> mutex_single a F2  \<and> 
             (mutex F1 \<and> foldr (\<lambda>x. (\<and>) (mutex_single x F2)) F1 True \<and> mutex F2) \<longleftrightarrow>
             (mutex (a # F1) \<and> foldr (\<lambda>x. (\<and>) (mutex_single x F2)) (a # F1) True \<and> mutex F2)" 
         using "2" mutex_Cons by blast
    show ?thesis 
    using "1" "4" "5" by presburger
  qed
qed

lemma unfold_mutex_single: 
 "foldr (\<lambda>x y. (mutex_single x F2) \<and> y) F1 True \<longleftrightarrow>
  (\<forall> i. 0 \<le> i \<and> i < length F1 \<longrightarrow> (mutex_single (F1 ! i) F2)) " 
proof (induct F1 arbitrary: F2)
case Nil
then show ?case by simp
next
case (Cons a F1)
then show ?case 
  using less_Suc_eq_0_disj by auto
qed


subsection \<open>Add to lemmas\<close>

lemma length_add_to_now_single: 
shows "length (add_to_now_single f L) = (if L=[] then 0 else 2*(length L)) " 
proof (induct L)
case Nil
then show ?case unfolding add_to_now_single_def by simp
next
case (Cons a L)
then show ?case unfolding add_to_now_single_def by simp
qed



lemma length_add_to_next_single: 
shows "length (add_to_next_single f L) = (if L=[] then 0 else 2*(length L)) " 
proof (induct L)
case Nil
then show ?case unfolding add_to_next_single_def by simp
next
case (Cons a L)
then show ?case unfolding add_to_next_single_def by simp
qed

lemma length_add_to_next_single_a: 
shows "length (add_to_next_single_a f L) = (if L=[] then 0 else 2*(length L)) " 
proof (induct L)
case Nil
then show ?case unfolding add_to_next_single_a_def by simp
next
case (Cons a L)
then show ?case unfolding add_to_next_single_a_def by simp
qed

lemma length_add_to_now_next_gen: 
assumes "length xs = length ys" 
        " length L1 = length L2" 
shows "length (foldr add_to_now_single  xs L1) =
       length (foldr add_to_next_single ys L2) " 
using assms
proof (induct xs ys rule: list_induct2)
case Nil
then show ?case by simp
next
case (Cons x xs y ys)
then show ?case 
   using Cons unfolding add_to_now_single_def add_to_next_single_def
   by simp
qed

lemma length_add_to_now_next_a_gen: 
assumes "length xs = length ys" 
        " length L1 = length L2" 
shows "length (foldr add_to_now_single  xs L1) =
       length (foldr add_to_next_single_a ys L2) " 
using assms
proof (induct xs ys rule: list_induct2)
case Nil
then show ?case by simp
next
case (Cons x xs y ys)
then show ?case 
   using Cons unfolding add_to_now_single_def add_to_next_single_a_def
   by simp
qed

lemma length_add_to_now_next: 
assumes "length xs = length ys" 
shows "length (foldr add_to_now_single  xs [itrue ,ifalse]) =
       length (foldr add_to_next_single ys [ifalse,ifalse]) " 
using assms
by (simp add: length_add_to_now_next_gen)

lemma length_add_to_now_next_a: 
assumes "length xs = length ys" 
shows "length (foldr add_to_now_single  xs [itrue ,ifalse]) =
       length (foldr add_to_next_single_a ys [ifalse,ifalse]) " 
using assms
by (simp add: length_add_to_now_next_a_gen)



lemma add_to_now_single_to_big_ior_inv:
 assumes "\<turnstile> big_ior L"
 shows   "\<turnstile> big_ior (add_to_now_single f L)  " 
 unfolding add_to_now_single_def using assms 
big_ior_append[of "(map (\<lambda>x. ( f iand x)) L)" "(map (\<lambda>x. ( inot f iand x)  ) L)"]
by (auto simp add: big_ior_defs) 
 (metis iand_defs nth_map)

lemma big_ior_foldr_add_to_now_single_inv_a: 
shows "\<turnstile> big_ior (foldr add_to_now_single xs [itrue])"
proof (induct xs)
case Nil
then show ?case  
using big_ior_defs by fastforce
next
case (Cons a xs)
then show ?case
     using Cons add_to_now_single_to_big_ior_inv by auto
qed 

lemma big_ior_foldr_add_to_now_single_inv: 
shows "\<turnstile> big_ior (foldr add_to_now_single xs [itrue, ifalse])" 
proof (induct xs)
case Nil
then show ?case  
using big_ior_defs by fastforce
next
case (Cons a xs)
then show ?case
     using Cons add_to_now_single_to_big_ior_inv by auto
qed


lemma add_to_now_single_to_mutex_inv: 
 assumes "mutex L"
 shows   "mutex ( add_to_now_single f L)"
proof -
 have 1: "mutex (map (\<lambda>x. f iand x) L)"
   using assms unfolding mutex_def by auto
 have 2: " mutex (map (\<lambda>x. inot f iand x) L)" 
    using assms unfolding mutex_def by auto
 have 3: "  foldr (\<lambda>x. (\<and>) (mutex_single x (map (\<lambda>x. (inot f) iand  x ) L))) (map (\<lambda>x. f iand x) L) True 
    \<longleftrightarrow> (\<forall>i. 0 \<le> i \<and> i < length (map (\<lambda>x. f iand x) L) \<longrightarrow> mutex_single (map (\<lambda>x. f iand x) L ! i) (map (\<lambda>x. (inot f) iand x) L))"
   using  unfold_mutex_single[of "(map (\<lambda>x. (inot f) iand x) L)" "(map (\<lambda>x. f iand x) L)"] 
     by blast
 have 4: "(\<forall>i. 0 \<le> i \<and> i < length (map (\<lambda>x. f iand x) L) \<longrightarrow> mutex_single (map (\<lambda>x. f iand x) L ! i) (map (\<lambda>x. (inot f) iand x) L))" 
   using assms unfolding mutex_single_def by auto  
 show ?thesis  unfolding add_to_now_single_def using assms 
   mutex_append[of "(map (\<lambda>x. f iand x ) L)" "(map (\<lambda>x. (inot f) iand x  ) L)"  ]
  using "1" "2" "3" "4" by blast
qed
 
lemma mutex_foldr_add_to_now_single_inv_a: 
shows " mutex (foldr add_to_now_single xs [itrue])" 
proof (induct xs)
case Nil
then show ?case unfolding mutex_def by simp
next
case (Cons a xs)
then show ?case by (simp add: Cons.hyps add_to_now_single_to_mutex_inv)
qed

lemma mutex_foldr_add_to_now_single_inv: 
shows " mutex (foldr add_to_now_single xs [itrue, ifalse])" 
proof (induct xs)
case Nil
then show ?case unfolding mutex_def by simp
next
case (Cons a xs)
then show ?case by (simp add: Cons.hyps add_to_now_single_to_mutex_inv)
qed


lemma add_to_now_single_to_state_qpitl_list:
 assumes "state_qpitl f"
         "state_qpitl_list L"
 shows " state_qpitl_list ( add_to_now_single f L ) " 
using assms unfolding add_to_now_single_def
by (simp add: inot_d_def state_qpitl_list_append state_qpitl_list_map)
 
lemma state_qpitl_foldr_add_to_now_single_inv_a: 
assumes "state_qpitl_list xs" 
shows "  state_qpitl_list (foldr add_to_now_single xs [itrue])" 
using assms 
proof (induct xs)
case Nil
then show ?case using Nil 
by (simp add: inot_d_def itrue_d_def state_qpitl_list_def)
next
case (Cons a xs)
then show ?case 
   by simp 
    (metis add_to_now_single_to_state_qpitl_list append_Cons append_Nil length_Cons nth_Cons_0 state_qpitl_list_append state_qpitl_list_defs zero_less_Suc)
qed

lemma state_qpitl_foldr_add_to_now_single_inv: 
assumes "state_qpitl_list xs" 
shows "  state_qpitl_list (foldr add_to_now_single xs [itrue, ifalse])" 
using assms 
proof (induct xs)
case Nil
then show ?case using Nil 
by (simp add: inot_d_def itrue_d_def state_qpitl_list_def)
next
case (Cons a xs)
then show ?case 
   by simp 
    (metis add_to_now_single_to_state_qpitl_list append_Cons append_Nil length_Cons nth_Cons_0 state_qpitl_list_append state_qpitl_list_defs zero_less_Suc)
qed

lemma det_big_ior_ieqv_nondet_big_ior_a: 
assumes "length L1 = length L2"      
shows "\<turnstile> big_ior ( map2 (\<lambda> x y. x iand (next  y)) 
                   (foldr (add_to_now_single) L1 [itrue])
                   (foldr (add_to_next_single) L2 [ifalse])) 
         ieqv 
         big_ior (map2 (\<lambda>x y. x iand (next y)) L1 L2) "
using assms
proof (induct L1 L2 rule: list_induct2)
case Nil
then show ?case by (simp add: big_ior_def) 
next
case (Cons x xs y ys)
then show ?case 
    proof (cases "xs = [] \<and> ys = [] ")
    case True
    then show ?thesis using Cons 
     unfolding add_to_now_single_def add_to_next_single_def big_ior_def big_iand_def
      by simp   
    next
    case False
    then show ?thesis 
      proof -
       have 1: "(foldr add_to_now_single (x # xs) [itrue]) = 
                ( (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue])) @ 
                  (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue])))"  
           by (simp add: add_to_now_single_def)
       have 2: "(foldr add_to_next_single (y # ys) [ifalse]) = 
                ( (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse]))@
                                   (foldr add_to_next_single ys [ifalse]) )"
          by (simp add: add_to_next_single_def)
       have 3: "length xs = length ys"
           by (simp add: Cons.hyps(1))
       have 4: "\<turnstile> big_ior (map2 (\<lambda>x y. x iand next  y) 
                          (foldr add_to_now_single xs [itrue])
                          (foldr add_to_next_single ys [ifalse])) ieqv 
                  big_ior (map2 (\<lambda>x y. x iand next y) xs ys) " 
          using "3" Cons.hyps(2) False by fastforce
       have 5: "big_ior (map2 (\<lambda>x y. x iand next  y) 
                        (foldr add_to_now_single (x # xs) [itrue])
                        (foldr add_to_next_single (y # ys) [ifalse])) =
                big_ior (map2 (\<lambda>x y. x iand next  y) 
                        ( (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue])) @ 
                          (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue])))
                        ( (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse]))@
                                           (foldr add_to_next_single ys [ifalse]) )
                         )"
          using "1" "2" by presburger 
       have 6: "length (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue])) =
                length (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse]))"
          by (simp add: "3" length_add_to_now_next_gen)
       have 7: " length (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue])) =
                 length (foldr add_to_next_single ys [ifalse])"
           using "6" by auto
       have 75: "length (foldr add_to_now_single xs [itrue]) = 
                 length (foldr add_to_next_single ys [ifalse])" 
         using "7" by force
       have 8: "\<turnstile> big_ior (map2 (\<lambda>x y. x iand next y) 
                        ( (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue])) @ 
                          (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue])))
                        ( (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse]))@
                                           (foldr add_to_next_single ys [ifalse]) )
                         ) ieqv
                   big_ior ( (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue]))
                             (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse])) )
                           @ 
                             (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue]))
                             (foldr add_to_next_single ys [ifalse]) )
                          )"
             using "6" by force
       have 9: "\<turnstile> big_ior ( (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue]))
                             (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse])) )
                           @ 
                             (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue]))
                             (foldr add_to_next_single ys [ifalse]) )
                          ) ieqv
                    (big_ior  (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue]))
                             (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse])) )) ior
                    (big_ior (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue]))
                                                     (foldr add_to_next_single ys [ifalse]) ))"
          using big_ior_append by blast    
       have 10: "\<turnstile> (big_ior  (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue]))
                             (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse])) )) ieqv
                   (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand next (y ior y1)) 
                             ( (foldr add_to_now_single xs [itrue]))
                             ( (foldr add_to_next_single ys [ifalse])))) " 
              using map2_map_a[of "(foldr add_to_now_single xs [itrue])" 
            "(foldr add_to_next_single ys [ifalse])" x y ] using "7" by auto  
       have 11: "\<turnstile> (big_ior (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue]))
                                                          (foldr add_to_next_single ys [ifalse]) )) ieqv
                   (big_ior (map2 (\<lambda>x1 y1. ((inot x) iand x1) iand next y1)
                             ( (foldr add_to_now_single xs [itrue]))
                               (foldr add_to_next_single ys [ifalse]) )) " 
           using map2_map_b[of "(foldr add_to_now_single xs [itrue])"
             "(foldr add_to_next_single ys [ifalse])" x] 75
          by auto
       have 12: "\<turnstile> big_ior (foldr add_to_now_single xs [itrue])" 
         using big_ior_foldr_add_to_now_single_inv_a by metis
       have 13: "0 < length (foldr add_to_now_single xs [itrue])"
         using "12" big_ior_empty by auto
       have 14: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand next (y ior y1)) 
                             ( (foldr add_to_now_single xs [itrue]))
                             ( (foldr add_to_next_single ys [ifalse])))) ior
                    (big_ior (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue]))
                                                     (foldr add_to_next_single ys [ifalse]) ))
                    ieqv
                    (x iand next y) ior 
                    (big_ior  (map2 (\<lambda>x1 y1. x1 iand next  y1) 
                             ( (foldr add_to_now_single xs [itrue]))
                             ( (foldr add_to_next_single ys [ifalse]))))" 
            using big_ior_split_off[of "( (foldr add_to_now_single xs [itrue]))"
              "( (foldr add_to_next_single ys [ifalse]))" x y] 12 13
           by (simp add: "75" map2_map_b)
       have 15: "\<turnstile> big_ior (map2 (\<lambda>x y. x iand next y) (x # xs) (y # ys))  ieqv
                   (x iand next y) ior 
                    big_ior (map2 (\<lambda>x y. x iand next y) xs  ys) " 
            using big_ior_Cons_alt by auto
      have 16: "\<turnstile> (x iand next y) ior 
                    (big_ior  (map2 (\<lambda>x1 y1. x1 iand next  y1) 
                             ( (foldr add_to_now_single xs [itrue]))
                             ( (foldr add_to_next_single ys [ifalse])))) ieqv
                  big_ior (map2 (\<lambda>x y. x iand next y) (x # xs) (y # ys))" 
          using 4 15 by auto
       show ?thesis 
       using "10" "14" "16" "5" "6" "9" by force
    qed
    qed
qed

lemma det_big_ior_ieqv_nondet_big_ior_a_a: 
assumes "length L1 = length L2"      
shows "\<turnstile> big_ior ( map2 (\<lambda> x y. x iand (next  y)) 
                   (foldr (add_to_now_single) L1 [itrue])
                   (foldr (add_to_next_single_a) L2 [ifalse])) 
         ieqv 
         big_ior (map2 (\<lambda>x y. x iand (next y)) L1 L2) "
using assms
proof (induct L1 L2 rule: list_induct2)
case Nil
then show ?case by (simp add: big_ior_def) 
next
case (Cons x xs y ys)
then show ?case 
    proof (cases "xs = [] \<and> ys = [] ")
    case True
    then show ?thesis using Cons 
     unfolding add_to_now_single_def add_to_next_single_a_def big_ior_def big_iand_def
      by simp   
    next
    case False
    then show ?thesis 
      proof -
       have 1: "(foldr add_to_now_single (x # xs) [itrue]) = 
                ( (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue])) @ 
                  (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue])))"  
           by (simp add: add_to_now_single_def)
       have 2: "(foldr add_to_next_single_a (y # ys) [ifalse]) = 
                ( (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse]))@
                                   (foldr add_to_next_single_a ys [ifalse]) )"
          by (simp add: add_to_next_single_a_def)
       have 3: "length xs = length ys"
           by (simp add: Cons.hyps(1))
       have 4: "\<turnstile> big_ior (map2 (\<lambda>x y. x iand next  y) 
                          (foldr add_to_now_single xs [itrue])
                          (foldr add_to_next_single_a ys [ifalse])) ieqv 
                  big_ior (map2 (\<lambda>x y. x iand next y) xs ys) " 
          using "3" Cons.hyps(2) False by fastforce
       have 5: "big_ior (map2 (\<lambda>x y. x iand next  y) 
                        (foldr add_to_now_single (x # xs) [itrue])
                        (foldr add_to_next_single_a (y # ys) [ifalse])) =
                big_ior (map2 (\<lambda>x y. x iand next  y) 
                        ( (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue])) @ 
                          (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue])))
                        ( (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse]))@
                                           (foldr add_to_next_single_a ys [ifalse]) )
                         )"
          using "1" "2" by presburger 
       have 6: "length (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue])) =
                length (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse]))"
          by (simp add: "3" length_add_to_now_next_a_gen)
       have 7: " length (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue])) =
                 length (foldr add_to_next_single_a ys [ifalse])"
           using "6" by auto
       have 75: "length (foldr add_to_now_single xs [itrue]) = 
                 length (foldr add_to_next_single_a ys [ifalse])" 
         using "7" by force
       have 8: "\<turnstile> big_ior (map2 (\<lambda>x y. x iand next y) 
                        ( (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue])) @ 
                          (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue])))
                        ( (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse]))@
                                           (foldr add_to_next_single_a ys [ifalse]) )
                         ) ieqv
                   big_ior ( (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue]))
                             (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse])) )
                           @ 
                             (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue]))
                             (foldr add_to_next_single_a ys [ifalse]) )
                          )"
             using "6" by force
       have 9: "\<turnstile> big_ior ( (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue]))
                             (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse])) )
                           @ 
                             (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue]))
                             (foldr add_to_next_single_a ys [ifalse]) )
                          ) ieqv
                    (big_ior  (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue]))
                             (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse])) )) ior
                    (big_ior (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue]))
                                                     (foldr add_to_next_single_a ys [ifalse]) ))"
          using big_ior_append by blast    
       have 10: "\<turnstile> (big_ior  (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue]))
                             (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse])) )) ieqv
                   (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand next ((if y = ifalse then y1 else y ior y1))) 
                             ( (foldr add_to_now_single xs [itrue]))
                             ( (foldr add_to_next_single_a ys [ifalse])))) " 
              using map2_map_a_a[of "(foldr add_to_now_single xs [itrue])" 
            "(foldr add_to_next_single_a ys [ifalse])" x y ] using "7" by auto  
       have 11: "\<turnstile> (big_ior (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue]))
                                                          (foldr add_to_next_single_a ys [ifalse]) )) ieqv
                   (big_ior (map2 (\<lambda>x1 y1. ((inot x) iand x1) iand next y1)
                             ( (foldr add_to_now_single xs [itrue]))
                               (foldr add_to_next_single_a ys [ifalse]) )) " 
           using map2_map_b[of "(foldr add_to_now_single xs [itrue])"
             "(foldr add_to_next_single_a ys [ifalse])" x] 75
          by auto
       have 12: "\<turnstile> big_ior (foldr add_to_now_single xs [itrue])" 
         using big_ior_foldr_add_to_now_single_inv_a by metis
       have 13: "0 < length (foldr add_to_now_single xs [itrue])"
         using "12" big_ior_empty by auto
       have 14: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand next ((if y = ifalse then y1 else y ior y1))) 
                             ( (foldr add_to_now_single xs [itrue]))
                             ( (foldr add_to_next_single_a ys [ifalse])))) ior
                    (big_ior (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue]))
                                                     (foldr add_to_next_single_a ys [ifalse]) ))
                    ieqv
                    (x iand next y) ior 
                    (big_ior  (map2 (\<lambda>x1 y1. x1 iand next  y1) 
                             ( (foldr add_to_now_single xs [itrue]))
                             ( (foldr add_to_next_single_a ys [ifalse]))))" 
            using big_ior_split_off_a[of "( (foldr add_to_now_single xs [itrue]))"
              "( (foldr add_to_next_single_a ys [ifalse]))" x y] 12 13 
           by (simp add: "75" map2_map_b)
       have 15: "\<turnstile> big_ior (map2 (\<lambda>x y. x iand next y) (x # xs) (y # ys))  ieqv
                   (x iand next y) ior 
                    big_ior (map2 (\<lambda>x y. x iand next y) xs  ys) " 
            using big_ior_Cons_alt by auto
      have 16: "\<turnstile> (x iand next y) ior 
                    (big_ior  (map2 (\<lambda>x1 y1. x1 iand next  y1) 
                             ( (foldr add_to_now_single xs [itrue]))
                             ( (foldr add_to_next_single_a ys [ifalse])))) ieqv
                  big_ior (map2 (\<lambda>x y. x iand next y) (x # xs) (y # ys))" 
          using 4 15 by auto
       show ?thesis 
       using "10" "14" "16" "5" "6" "9" by force 
    qed
    qed
qed

lemma det_big_ior_ieqv_nondet_big_ior: 
assumes "length L1 = length L2"      
shows "\<turnstile> big_ior ( map2 (\<lambda> x y. x iand (next  y)) 
                   (foldr (add_to_now_single) L1 [itrue,ifalse])
                   (foldr (add_to_next_single) L2 [ifalse,ifalse])) 
         ieqv 
         big_ior (map2 (\<lambda>x y. x iand (next y)) L1 L2) "
using assms
proof (induct L1 L2 rule: list_induct2)
case Nil
then show ?case by (simp add: big_ior_def) 
next
case (Cons x xs y ys)
then show ?case 
    proof (cases "xs = [] \<and> ys = [] ")
    case True
    then show ?thesis using Cons 
     unfolding add_to_now_single_def add_to_next_single_def big_ior_def big_iand_def
      by simp   
    next
    case False
    then show ?thesis 
      proof -
       have 1: "(foldr add_to_now_single (x # xs) [itrue,ifalse]) = 
                ( (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse])) @ 
                  (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse])))"  
           by (simp add: add_to_now_single_def)
       have 2: "(foldr add_to_next_single (y # ys) [ifalse,ifalse]) = 
                ( (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse,ifalse]))@
                                   (foldr add_to_next_single ys [ifalse,ifalse]) )"
          by (simp add: add_to_next_single_def)
       have 3: "length xs = length ys"
           by (simp add: Cons.hyps(1))
       have 4: "\<turnstile> big_ior (map2 (\<lambda>x y. x iand next  y) 
                          (foldr add_to_now_single xs [itrue,ifalse])
                          (foldr add_to_next_single ys [ifalse,ifalse])) ieqv 
                  big_ior (map2 (\<lambda>x y. x iand next y) xs ys) " 
          using "3" Cons.hyps(2) False by fastforce
       have 5: "big_ior (map2 (\<lambda>x y. x iand next  y) 
                        (foldr add_to_now_single (x # xs) [itrue,ifalse])
                        (foldr add_to_next_single (y # ys) [ifalse,ifalse])) =
                big_ior (map2 (\<lambda>x y. x iand next  y) 
                        ( (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse])) @ 
                          (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse])))
                        ( (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse,ifalse]))@
                                           (foldr add_to_next_single ys [ifalse,ifalse]) )
                         )"
          using "1" "2" by presburger 
       have 6: "length (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse])) =
                length (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse,ifalse]))"
          by (simp add: "3" length_add_to_now_next)
       have 7: " length (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse])) =
                 length (foldr add_to_next_single ys [ifalse,ifalse])"
           using "6" by auto
       have 75: "length (foldr add_to_now_single xs [itrue, ifalse]) = 
                 length (foldr add_to_next_single ys [ifalse, ifalse])" 
         using "7" by force
       have 8: "\<turnstile> big_ior (map2 (\<lambda>x y. x iand next y) 
                        ( (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse])) @ 
                          (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse])))
                        ( (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse,ifalse]))@
                                           (foldr add_to_next_single ys [ifalse,ifalse]) )
                         ) ieqv
                   big_ior ( (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                             (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse,ifalse])) )
                           @ 
                             (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                             (foldr add_to_next_single ys [ifalse,ifalse]) )
                          )"
             using "6" by force
       have 9: "\<turnstile> big_ior ( (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                             (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse,ifalse])) )
                           @ 
                             (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                             (foldr add_to_next_single ys [ifalse,ifalse]) )
                          ) ieqv
                    (big_ior  (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                             (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse,ifalse])) )) ior
                    (big_ior (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                                                     (foldr add_to_next_single ys [ifalse,ifalse]) ))"
          using big_ior_append by blast    
       have 10: "\<turnstile> (big_ior  (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                             (map (\<lambda>x1. y ior x1) (foldr add_to_next_single ys [ifalse,ifalse])) )) ieqv
                   (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand next (y ior y1)) 
                             ( (foldr add_to_now_single xs [itrue,ifalse]))
                             ( (foldr add_to_next_single ys [ifalse,ifalse])))) " 
              using map2_map_a[of "(foldr add_to_now_single xs [itrue,ifalse])" 
            "(foldr add_to_next_single ys [ifalse,ifalse])" x y ] using "7" by auto  
       have 11: "\<turnstile> (big_ior (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                                                          (foldr add_to_next_single ys [ifalse,ifalse]) )) ieqv
                   (big_ior (map2 (\<lambda>x1 y1. ((inot x) iand x1) iand next y1)
                             ( (foldr add_to_now_single xs [itrue,ifalse]))
                               (foldr add_to_next_single ys [ifalse,ifalse]) )) " 
           using map2_map_b[of "(foldr add_to_now_single xs [itrue,ifalse])"
             "(foldr add_to_next_single ys [ifalse,ifalse])" x] 75
          by auto
       have 12: "\<turnstile> big_ior (foldr add_to_now_single xs [itrue, ifalse])" 
         using big_ior_foldr_add_to_now_single_inv by metis
       have 13: "0 < length (foldr add_to_now_single xs [itrue, ifalse])"
         using "12" big_ior_empty by auto
       have 14: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand next (y ior y1)) 
                             ( (foldr add_to_now_single xs [itrue,ifalse]))
                             ( (foldr add_to_next_single ys [ifalse,ifalse])))) ior
                    (big_ior (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                                                     (foldr add_to_next_single ys [ifalse,ifalse]) ))
                    ieqv
                    (x iand next y) ior 
                    (big_ior  (map2 (\<lambda>x1 y1. x1 iand next  y1) 
                             ( (foldr add_to_now_single xs [itrue,ifalse]))
                             ( (foldr add_to_next_single ys [ifalse,ifalse]))))" 
            using big_ior_split_off[of "( (foldr add_to_now_single xs [itrue,ifalse]))"
              "( (foldr add_to_next_single ys [ifalse,ifalse]))" x y] 12 13
           by (simp add: "75" map2_map_b)
       have 15: "\<turnstile> big_ior (map2 (\<lambda>x y. x iand next y) (x # xs) (y # ys))  ieqv
                   (x iand next y) ior 
                    big_ior (map2 (\<lambda>x y. x iand next y) xs  ys) " 
            using big_ior_Cons_alt by auto
      have 16: "\<turnstile> (x iand next y) ior 
                    (big_ior  (map2 (\<lambda>x1 y1. x1 iand next  y1) 
                             ( (foldr add_to_now_single xs [itrue,ifalse]))
                             ( (foldr add_to_next_single ys [ifalse,ifalse])))) ieqv
                  big_ior (map2 (\<lambda>x y. x iand next y) (x # xs) (y # ys))" 
          using 4 15 by auto
       show ?thesis 
       using "10" "14" "16" "5" "6" "9" by force
    qed
    qed
qed

lemma det_big_ior_ieqv_nondet_big_ior_b: 
assumes "length L1 = length L2"      
shows "\<turnstile> big_ior ( map2 (\<lambda> x y. x iand (next  y)) 
                   (foldr (add_to_now_single) L1 [itrue,ifalse])
                   (foldr (add_to_next_single_a) L2 [ifalse,ifalse])) 
         ieqv 
         big_ior (map2 (\<lambda>x y. x iand (next y)) L1 L2) "
using assms
proof (induct L1 L2 rule: list_induct2)
case Nil
then show ?case by (simp add: big_ior_def) 
next
case (Cons x xs y ys)
then show ?case 
    proof (cases "xs = [] \<and> ys = [] ")
    case True
    then show ?thesis using Cons 
     unfolding add_to_now_single_def add_to_next_single_a_def big_ior_def big_iand_def
      by simp   
    next
    case False
    then show ?thesis 
      proof -
       have 1: "(foldr add_to_now_single (x # xs) [itrue,ifalse]) = 
                ( (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse])) @ 
                  (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse])))"  
           by (simp add: add_to_now_single_def)
       have 2: "(foldr add_to_next_single_a (y # ys) [ifalse,ifalse]) = 
                ( (map (\<lambda>x1. (if y =ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse,ifalse]))@
                                   (foldr add_to_next_single_a ys [ifalse,ifalse]) )"
          by (simp add: add_to_next_single_a_def)
       have 3: "length xs = length ys"
           by (simp add: Cons.hyps(1))
       have 4: "\<turnstile> big_ior (map2 (\<lambda>x y. x iand next  y) 
                          (foldr add_to_now_single xs [itrue,ifalse])
                          (foldr add_to_next_single_a ys [ifalse,ifalse])) ieqv 
                  big_ior (map2 (\<lambda>x y. x iand next y) xs ys) " 
          using "3" Cons.hyps(2) False by fastforce
       have 5: "big_ior (map2 (\<lambda>x y. x iand next  y) 
                        (foldr add_to_now_single (x # xs) [itrue,ifalse])
                        (foldr add_to_next_single_a (y # ys) [ifalse,ifalse])) =
                big_ior (map2 (\<lambda>x y. x iand next  y) 
                        ( (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse])) @ 
                          (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse])))
                        ( (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse,ifalse]))@
                                           (foldr add_to_next_single_a ys [ifalse,ifalse]) )
                         )"
          using "1" "2" by presburger 
       have 6: "length (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse])) =
                length (map (\<lambda>x1. y ior x1) (foldr add_to_next_single_a ys [ifalse,ifalse]))"
          by (simp add: "3" length_add_to_now_next_a)
       have 7: " length (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse])) =
                 length (foldr add_to_next_single_a ys [ifalse,ifalse])"
           using "6" by auto
       have 75: "length (foldr add_to_now_single xs [itrue, ifalse]) = 
                 length (foldr add_to_next_single_a ys [ifalse, ifalse])" 
         using "7" by force
       have 8: "\<turnstile> big_ior (map2 (\<lambda>x y. x iand next y) 
                        ( (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse])) @ 
                          (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse])))
                        ( (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse,ifalse]))@
                                           (foldr add_to_next_single_a ys [ifalse,ifalse]) )
                         ) ieqv
                   big_ior ( (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                             (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse,ifalse])) )
                           @ 
                             (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                             (foldr add_to_next_single_a ys [ifalse,ifalse]) )
                          )"
             using "6" by force
       have 9: "\<turnstile> big_ior ( (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                             (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse,ifalse])) )
                           @ 
                             (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                             (foldr add_to_next_single_a ys [ifalse,ifalse]) )
                          ) ieqv
                    (big_ior  (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                             (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse,ifalse])) )) ior
                    (big_ior (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                                                     (foldr add_to_next_single_a ys [ifalse,ifalse]) ))"
          using big_ior_append by blast    
       have 10: "\<turnstile> (big_ior  (map2 (\<lambda>x y. x iand next y) 
                             (map (\<lambda>x1. x iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                             (map (\<lambda>x1. (if y = ifalse then x1 else y ior x1)) (foldr add_to_next_single_a ys [ifalse,ifalse])) )) ieqv
                   (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand next ((if y = ifalse then y1 else y ior y1))) 
                             ( (foldr add_to_now_single xs [itrue,ifalse]))
                             ( (foldr add_to_next_single_a ys [ifalse,ifalse])))) " 
              using map2_map_a_a[of "(foldr add_to_now_single xs [itrue,ifalse])" 
            "(foldr add_to_next_single_a ys [ifalse,ifalse])" x y ] using "7" by auto  
       have 11: "\<turnstile> (big_ior (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                                                          (foldr add_to_next_single_a ys [ifalse,ifalse]) )) ieqv
                   (big_ior (map2 (\<lambda>x1 y1. ((inot x) iand x1) iand next y1)
                             ( (foldr add_to_now_single xs [itrue,ifalse]))
                               (foldr add_to_next_single_a ys [ifalse,ifalse]) )) " 
           using map2_map_b[of "(foldr add_to_now_single xs [itrue,ifalse])"
             "(foldr add_to_next_single_a ys [ifalse,ifalse])" x] 75
          by auto
       have 12: "\<turnstile> big_ior (foldr add_to_now_single xs [itrue, ifalse])" 
         using big_ior_foldr_add_to_now_single_inv by metis
       have 13: "0 < length (foldr add_to_now_single xs [itrue, ifalse])"
         using "12" big_ior_empty by auto
       have 14: "\<turnstile> (big_ior  (map2 (\<lambda>x1 y1. (x iand x1) iand next (if y= ifalse then y1 else y ior y1)) 
                             ( (foldr add_to_now_single xs [itrue,ifalse]))
                             ( (foldr add_to_next_single_a ys [ifalse,ifalse])))) ior
                    (big_ior (map2 (\<lambda>x y. x iand next y)
                             (map (\<lambda>x1. (inot x) iand x1) (foldr add_to_now_single xs [itrue,ifalse]))
                                                     (foldr add_to_next_single_a ys [ifalse,ifalse]) ))
                    ieqv
                    (x iand next y) ior 
                    (big_ior  (map2 (\<lambda>x1 y1. x1 iand next  y1) 
                             ( (foldr add_to_now_single xs [itrue,ifalse]))
                             ( (foldr add_to_next_single_a ys [ifalse,ifalse]))))" 
            using big_ior_split_off_a[of "( (foldr add_to_now_single xs [itrue,ifalse]))"
              "( (foldr add_to_next_single_a ys [ifalse,ifalse]))" x y] 12 13
           by (simp add: "75" map2_map_b)
       have 15: "\<turnstile> big_ior (map2 (\<lambda>x y. x iand next y) (x # xs) (y # ys))  ieqv
                   (x iand next y) ior 
                    big_ior (map2 (\<lambda>x y. x iand next y) xs  ys) " 
            using big_ior_Cons_alt by auto
      have 16: "\<turnstile> (x iand next y) ior 
                    (big_ior  (map2 (\<lambda>x1 y1. x1 iand next  y1) 
                             ( (foldr add_to_now_single xs [itrue,ifalse]))
                             ( (foldr add_to_next_single_a ys [ifalse,ifalse])))) ieqv
                  big_ior (map2 (\<lambda>x y. x iand next y) (x # xs) (y # ys))" 
          using 4 15 by auto
       show ?thesis 
       using "10" "14" "16" "5" "6" "9" by force
    qed
    qed
qed


lemma mutex_single_filter_sat: 
 assumes "mutex_single f L1"
 shows "mutex_single f ( filter (\<lambda>x. (\<exists> \<sigma>. \<sigma> \<Turnstile> x)) L1)"
using assms
proof (induct L1)
case Nil
then show ?case by simp
next
case (Cons a L1)
then show ?case 
by (simp add: mutex_single_Cons)
qed

lemma mutex_filter_sat: 
 assumes "mutex L1"
 shows "mutex ( filter (\<lambda>x. (\<exists> \<sigma>. \<sigma> \<Turnstile> x)) L1)"
using assms
proof (induct L1)
case Nil
then show ?case unfolding mutex_def by simp
next
case (Cons a L1)
then show ?case 
by (simp add: mutex_Cons mutex_single_filter_sat)
qed

  
lemma state_qpitl_list_filter_sat: 
 assumes "state_qpitl_list L"
 shows "state_qpitl_list ( filter (\<lambda>x. (\<exists> \<sigma>. \<sigma> \<Turnstile> x)) L)"
using assms
proof (induct L)
case Nil
then show ?case by simp
next
case (Cons a L)
then show ?case 
by (simp add: state_qpitl_list_def)
qed
 


lemma big_ior_itrue: 
 "\<turnstile> big_ior [itrue]"
by (simp add: big_ior_defs)

lemma mutex_itrue: 
 "mutex [itrue] "
by (simp add: mutex_Cons mutex_empty mutex_single_empty)

lemma big_ior_mutex_start:
 " (\<turnstile> big_ior [ifalse,itrue]) \<and> mutex [ifalse,itrue]" 
unfolding big_ior_def mutex_def
using less_Suc_eq by force


subsection \<open>GNF lemmas\<close>



lemma GNF_defs: 
assumes " \<turnstile> F ieqv (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) "  
        " state_qpitl F_e" 
        " state_qpitl_list F_now "
        " length F_now = length F_next" 
shows "(\<sigma> \<Turnstile> F) \<longleftrightarrow>
       ((nlength \<sigma> = 0 \<and> ((NNil (nfirst \<sigma>)) \<Turnstile> F_e)) \<or>
        (nlength \<sigma> > 0 \<and> 
     (\<exists>i<length F_now. ((NNil (nfirst \<sigma>)) \<Turnstile> F_now !i ) \<and> ((ndropn 1 \<sigma>) \<Turnstile> F_next !i)) ))" 
proof -
 have 1: " (\<sigma> \<Turnstile> F) \<longleftrightarrow>
           (\<sigma> \<Turnstile> (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next)))) 
           " 
 using assms itl_ieq by blast 
 have 2: "(\<sigma> \<Turnstile> ( F_e iand empty )) \<longleftrightarrow>
          (nlength \<sigma> = 0 \<and> ((NNil (nfirst \<sigma>)) \<Turnstile> F_e))"
          using assms state_qpitl_defs empty_defs iand_defs by blast
 have 3: "length (map2 (\<lambda>x y. x iand next y) F_now F_next) = length F_now" 
      by (simp add: assms(4))
 have 4: "
          (\<sigma> \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) \<longleftrightarrow>
          (\<exists>i<length F_now. (\<sigma> \<Turnstile>  (\<lambda> x y. x iand (next y)) (F_now!i) (F_next!i))) " 
  by (simp add: assms(4) big_ior_map2_defs)
 have 5: "(\<exists>i<length F_now. (\<sigma> \<Turnstile>  (\<lambda> x y. x iand (next y)) (F_now!i) (F_next!i))) \<longleftrightarrow>
           (nlength \<sigma> > 0 \<and> 
     (\<exists>i<length F_now. ((NNil (nfirst \<sigma>)) \<Turnstile> F_now !i ) \<and> ((ndropn 1 \<sigma>) \<Turnstile> F_next !i)) )"  
     by simp
      (metis assms(3) co.enat.exhaust_sel enat_le_plus_same(2) not_one_le_zero plus_1_eSuc(2)
       state_qpitl_defs state_qpitl_list_defs)
 show ?thesis 
 using "1" "2" "4" "5" by auto
qed  



lemma big_ior_iand_next_big_iand_iimp: 
 assumes "(\<turnstile> big_ior F)"
         "mutex F"
         "length F = length G" 
 shows " \<turnstile> big_ior (map2 (\<lambda>x y. x iand (next y)) F G) ieqv 
           big_iand (map2 (\<lambda>x y. x iimp (next y)) F G) " 
using assms unfolding valid_def mutex_def
apply (auto simp add: big_ior_defs big_iand_defs map2_iand_next_defs)
apply (metis not_less_iff_gr_or_eq)
by meson

lemma inot_big_iand_next_iimp_big_iand_iimp_inot: 
 assumes "(\<turnstile> big_ior F)"
         "mutex F"
         "length F = length G" 
 shows " \<turnstile> more iimp (inot big_iand (map2 (\<lambda>x y. x iimp (next y))  F G) ieqv
                      big_iand  (map2 (\<lambda> x y. x iimp (next (inot y)))  F G))" 
using assms unfolding mutex_def
apply (auto simp add: big_iand_defs big_ior_defs more_d_def)
apply (metis linorder_less_linear)
by meson


lemma inot_big_ior_iand_big_ior_iand_inot: 
 assumes "(\<turnstile> big_ior F)"
         "mutex F"
         "length F = length G" 
 shows " \<turnstile> (inot (big_ior (map2 (\<lambda>x y. x iand ( y)) F G)) ieqv 
           big_ior (map2 (\<lambda> x y. x iand ( (inot y)))  F  G))  " 
using assms unfolding mutex_def
apply (auto simp add: big_ior_defs big_iand_defs map2_iand_inot_defs)
apply meson
by (metis not_less_iff_gr_or_eq)

lemma inot_big_ior_iand_big_ior_iand_next_inot: 
 assumes "(\<turnstile> big_ior F)"
         "mutex F"
         "length F = length G" 
 shows " \<turnstile> more iimp (inot (big_ior (map2 (\<lambda>x y. x iand (next y)) F G)) ieqv 
           big_ior (map2 (\<lambda> x y. x iand (next (inot y)))  F  G))  " 
using assms unfolding mutex_def
apply (auto simp add: big_ior_defs big_iand_defs map2_iand_inot_next_defs more_d_def)
apply (metis One_nat_def gr_zeroI map2_iand_inot_next_defs not_one_le_zero semantics_qpitl.simps(4))
by (metis linorder_less_linear)



lemma more_iimp_inot_big_ior_big_ior_map_inot: 
 assumes "(\<turnstile> big_ior F)"
         "mutex F"
         "length F = length G" 
shows " \<turnstile> more iimp 
         ((inot (big_ior (map2 (\<lambda>x y. x iand (next y)) F G))) ieqv
          big_ior (map2 (\<lambda>x y. x iand (next y)) F (map (\<lambda>x. (inot x)) G)))"
proof -
 have 1: " (map2 (\<lambda>x y. x iand (next y)) F (map (\<lambda>x. (inot x)) G)) =
           (map2 (\<lambda>x y. x iand (next (inot y))) F G) "
        using map2_map[of F G "(\<lambda>x y. x iand (next y))" id "(\<lambda>x. (inot x))"]
          assms by auto
 show ?thesis 
 using "1" assms inot_big_ior_iand_big_ior_iand_next_inot by auto   
qed




lemma GNF_iimp_step: 
 assumes " \<turnstile> F ieqv (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) "  
        " state_qpitl F_e" 
        " state_qpitl_list F_now"
        " length F_now = length F_next" 
        " \<turnstile> G ieqv (( G_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  G_now G_next))) "  
        " state_qpitl G_e" 
        " state_qpitl_list G_now "
        " length G_now = length G_next"
 shows " (\<sigma> \<Turnstile> F iimp G) \<longleftrightarrow>
         (\<sigma> \<Turnstile> ( ( (inot F_e) ior G_e) iand empty) ior
               (big_ior  
                (map2 (\<lambda> x y. x iand (next y)) 
                      ((add_to_now F_now ) @G_now) 
                      ((map (\<lambda>x. (inot x)) (add_to_next F_next )) @ G_next) )))   "
proof (cases "nlength \<sigma> = 0")
case True
then show ?thesis 
   proof -
    have 1: "(\<sigma> \<Turnstile> F iimp G) \<longleftrightarrow> (\<not>(\<sigma> \<Turnstile> F) \<or> (\<sigma> \<Turnstile>G)) " 
       by auto
    have 2: "(\<not>(\<sigma> \<Turnstile> F) \<or> (\<sigma> \<Turnstile>G)) \<longleftrightarrow>
             (\<not>((NNil (nfirst \<sigma>)) \<Turnstile> F_e) \<or> ((NNil (nfirst \<sigma>)) \<Turnstile> G_e))"
     using 
      assms True 
      by (metis (mono_tags, lifting) GNF_defs less_numeral_extra(3))
    have 3: "state_qpitl (inot F_e ior G_e)" 
      by (simp add: assms(2) assms(6) inot_d_def ior_d_def)
    have 35: "state_qpitl_list [ifalse, itrue]" 
        unfolding state_qpitl_list_def itrue_d_def inot_d_def by auto
    have 4: "state_qpitl_list ((add_to_now F_now ) @ G_now)" 
        by (simp add: add_to_now_def assms(3) assms(7) state_qpitl_foldr_add_to_now_single_inv_a 
          state_qpitl_list_append)
    have 5: "length (add_to_now F_now  @ G_now) = 
             length (map inot_d (add_to_next F_next ) @ G_next)" 
       by (simp add: add_to_next_def add_to_now_def assms(4) assms(8) length_add_to_now_next_gen)
    have 6: "(\<not>((NNil (nfirst \<sigma>)) \<Turnstile> F_e) \<or> ((NNil (nfirst \<sigma>)) \<Turnstile> G_e)) \<longleftrightarrow>
             (\<sigma> \<Turnstile> ( ( (inot F_e) ior G_e) iand empty) ior
               (big_ior  
                (map2 (\<lambda> x y. x iand (next y)) 
                   ((add_to_now F_now )@G_now) 
                   ((map (\<lambda>x. (inot x)) (add_to_next F_next )) @ G_next) ) ))"
        using GNF_defs[of 
               "( ( (inot F_e) ior G_e) iand empty) ior
               (big_ior  
                (map2 (\<lambda> x y. x iand (next y)) 
                ((add_to_now F_now )@G_now) 
                ((map (\<lambda>x. (inot x)) (add_to_next F_next )) @ G_next) ))"
                 "( (inot F_e) ior G_e)" 
                            "((add_to_now F_now )@G_now)"
                            "((map (\<lambda>x. (inot x)) (add_to_next F_next )) @ G_next)" \<sigma> ]
        using assms  
        using "3" "4" "5" True by force
     show ?thesis 
     using "1" "2" "6" by blast
   qed
next
case False
then show ?thesis 
  proof -
   have 8: "(\<sigma> \<Turnstile> F iimp G) \<longleftrightarrow> (\<sigma> \<Turnstile> (inot F) ior G)"
       by auto
   have 9: "(\<sigma> \<Turnstile> (inot F)) \<longleftrightarrow>
            (\<sigma> \<Turnstile> (inot (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))))"
      using False assms(1) by auto
   have 10: "\<turnstile> (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next)) ieqv
               big_ior ( map2 (\<lambda> x y. x iand (next  y)) 
                   (foldr (add_to_now_single) F_now [itrue])
                   (foldr (add_to_next_single) F_next [ifalse]))" 
     using assms(4) det_big_ior_ieqv_nondet_big_ior_a by auto
   have 101: "(\<sigma> \<Turnstile> (inot (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next)))) \<longleftrightarrow>
              (\<sigma> \<Turnstile> (inot (big_ior ( map2 (\<lambda> x y. x iand (next  y)) 
                   (foldr (add_to_now_single) F_now [itrue])
                   (foldr (add_to_next_single) F_next [ifalse])))))"
      using "10" by auto
   have 11: "\<turnstile> big_ior (foldr (add_to_now_single) F_now [itrue])" 
     using big_ior_foldr_add_to_now_single_inv_a by blast
   have 12: "mutex (foldr (add_to_now_single) F_now [itrue])"
     using mutex_foldr_add_to_now_single_inv_a by blast 
   have 13: "length (foldr (add_to_now_single) F_now [itrue]) =
             length (foldr (add_to_next_single) F_next [ifalse])"
     by (simp add: assms(4) length_add_to_now_next_gen) 
   have 14: "\<turnstile> more iimp
              ( (inot (big_ior ( map2 (\<lambda> x y. x iand (next  y)) 
                   (foldr (add_to_now_single) F_now [itrue])
                   (foldr (add_to_next_single) F_next [ifalse])))) ieqv
                (big_ior  
                (map2 (\<lambda> x y. x iand (next y)) 
                      (foldr (add_to_now_single) F_now [itrue]) 
                      ((map (\<lambda>x. (inot x)) (foldr (add_to_next_single) F_next [ifalse]))))))"
      using "11" "12" "13" more_iimp_inot_big_ior_big_ior_map_inot by blast
   have 15: "(\<sigma> \<Turnstile> (inot (big_ior ( map2 (\<lambda> x y. x iand (next  y)) 
                   (foldr (add_to_now_single) F_now [itrue])
                   (foldr (add_to_next_single) F_next [ifalse]))))) \<longleftrightarrow>
            (\<sigma> \<Turnstile> (big_ior  
                (map2 (\<lambda> x y. x iand (next y)) 
                      (foldr (add_to_now_single) F_now [itrue]) 
                      ((map (\<lambda>x. (inot x)) (foldr (add_to_next_single) F_next [ifalse]))))))"
       using 14 False by auto
       (metis (no_types, lifting) empty_d_def empty_defs inot_defs, 
        metis (no_types, lifting) empty_d_def empty_defs inot_defs)
   have 16: "(\<sigma> \<Turnstile> G) \<longleftrightarrow>
              (\<sigma> \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand (next y))  G_now G_next)))"
    using False assms(5) by auto
   have 17: " (\<sigma> \<Turnstile> (inot F) ior G) \<longleftrightarrow>
              (\<sigma> \<Turnstile> (big_ior  (map2 (\<lambda> x y. x iand (next y)) 
                      (foldr (add_to_now_single) F_now [itrue]) 
                      ((map (\<lambda>x. (inot x)) (foldr (add_to_next_single) F_next [ifalse]))))) ior
                    (big_ior (map2 (\<lambda> x y. x iand (next y))  G_now G_next)))"
                    using "101" "15" "16" "9" by auto 
   have 18: "(\<sigma> \<Turnstile> (big_ior  (map2 (\<lambda> x y. x iand (next y)) 
                      (foldr (add_to_now_single) F_now [itrue]) 
                      ((map (\<lambda>x. (inot x)) (foldr (add_to_next_single) F_next [ifalse]))))) ior
                    (big_ior (map2 (\<lambda> x y. x iand (next y))  G_now G_next))) \<longleftrightarrow>
            (\<sigma> \<Turnstile> (big_ior  (map2 (\<lambda> x y. x iand (next y)) 
                      ((foldr (add_to_now_single) F_now [itrue]) @ G_now)
                      ((map (\<lambda>x. (inot x)) (foldr (add_to_next_single) F_next [ifalse])) @ G_next) ) ))"
   using "13" big_ior_append by auto
   show ?thesis unfolding add_to_now_def add_to_next_def
   using "17" "18" False by auto  
 qed            
qed 
 



lemma GNF_inot_step: 
 assumes " \<turnstile> F ieqv (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) "  
        " state_qpitl F_e" 
        " state_qpitl_list F_now"
        " length F_now = length F_next" 
shows " (\<sigma> \<Turnstile> (inot F) ) \<longleftrightarrow>
         (\<sigma> \<Turnstile> (  (inot F_e) iand empty) ior
               (big_ior  
                (map2 (\<lambda> x y. x iand (next y)) 
                      ((add_to_now F_now ) @[ifalse]) 
                      ((map (\<lambda>x. (inot x)) (add_to_next F_next )) @ [ifalse]) )))   "
proof -
 have 1: "(\<sigma> \<Turnstile> (inot F) ) \<longleftrightarrow> (\<sigma> \<Turnstile> F iimp ifalse)"
   by auto
 have 2: " \<turnstile> ifalse ieqv ifalse iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) [ifalse] [ifalse])" 
    by (auto simp add: big_ior_defs)
 have 3: "(\<sigma> \<Turnstile> F iimp ifalse) \<longleftrightarrow>
          (\<sigma> \<Turnstile> ( ( (inot F_e) ior ifalse) iand empty) ior
               (big_ior  
                (map2 (\<lambda> x y. x iand (next y)) 
                      ((add_to_now F_now ) @ [ifalse]) 
                      ((map (\<lambda>x. (inot x)) (add_to_next F_next )) @ [ifalse]) )))"
    using GNF_iimp_step[of F F_e F_now F_next ifalse ifalse "[ifalse]" "[ifalse]" \<sigma>]
     assms 
    by (metis "2" length_Cons less_Suc0 list.size(3) nth_Cons_0 state_qpitl.simps(1) 
        state_qpitl_list_defs)
 show ?thesis 
 using "3" by auto
qed

lemma GNF_inot_step_valid: 
 assumes " \<turnstile> F ieqv (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) "  
        " state_qpitl F_e" 
        " state_qpitl_list F_now"
        " length F_now = length F_next" 
shows " \<turnstile> (inot F)  ieqv
         ( (  (inot F_e) iand empty) ior
               (big_ior  
                (map2 (\<lambda> x y. x iand (next y)) 
                      ((add_to_now F_now ) @[ifalse]) 
                      ((map (\<lambda>x. (inot x)) (add_to_next F_next )) @ [ifalse]) )))   "
using assms GNF_inot_step[of F F_e F_now F_next ]  
using itl_ieq by blast



lemma GNF_inot_inot_step_valid: 
 assumes " \<turnstile> F ieqv (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) "  
        " state_qpitl F_e" 
        " state_qpitl_list F_now"
        " length F_now = length F_next" 
shows " \<turnstile> (inot (inot F))  ieqv
         ( (  F_e iand empty) ior
               (big_ior  (map2 (\<lambda> x y. x iand (next y)) F_now F_next )))   "
using assms by simp



lemma GNF_ior_step: 
 assumes " \<turnstile> F ieqv (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) "  
        " state_qpitl F_e" 
        " state_qpitl_list F_now"
        " length F_now = length F_next" 
        " \<turnstile> G ieqv (( G_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  G_now G_next))) "  
        " state_qpitl G_e" 
        " state_qpitl_list G_now "
        " length G_now = length G_next"
 shows " (\<sigma> \<Turnstile> F ior G) \<longleftrightarrow>
         (\<sigma> \<Turnstile> ( ( F_e ior G_e) iand empty) ior
               (big_ior  (map2 (\<lambda> x y. x iand (next y)) (F_now  @ G_now) (F_next  @ G_next) )))   "
proof -
 have 1: "(\<sigma> \<Turnstile> F ior G) \<longleftrightarrow> 
          (\<sigma> \<Turnstile> (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) ior
                (( G_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  G_now G_next))))"
   using assms by auto
 have 2: "(\<sigma> \<Turnstile> (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) ior
                (( G_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  G_now G_next)))) \<longleftrightarrow>
          (\<sigma> \<Turnstile> ( ( F_e ior G_e) iand empty) ior
               (big_ior  (map2 (\<lambda> x y. x iand (next y)) (F_now  @ G_now) (F_next  @ G_next) )))"   
   using  assms big_ior_map2_append by fastforce
 show ?thesis 
 using "1" "2" by blast
qed


lemma GNF_schop_step: 
 assumes " \<turnstile> F ieqv (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) "  
        " state_qpitl F_e" 
        " state_qpitl_list F_now"
        " length F_now = length F_next" 
        " \<turnstile> G ieqv (( G_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  G_now G_next))) "  
        " state_qpitl G_e" 
        " state_qpitl_list G_now "
        " length G_now = length G_next"
 shows " (\<sigma> \<Turnstile> F schop G) \<longleftrightarrow>
          (\<sigma> \<Turnstile> (F_e iand G_e) iand empty ior
                (big_ior (map2 (\<lambda> x y. x iand next y) 
                         (F_now @ (map (\<lambda>x. x iand F_e) G_now)) 
                         ((map (\<lambda>x. x schop G) F_next) @  G_next))) ) " 
proof (cases "nlength \<sigma> = 0")
case True
then show ?thesis 
  proof -
   have 1: "(\<sigma> \<Turnstile> F schop G) \<longleftrightarrow> (\<sigma> \<Turnstile> F) \<and> (\<sigma> \<Turnstile> G)"
       using True by (auto simp add: ntaken_all zero_enat_def) 
   have 2: "(\<sigma> \<Turnstile> F) \<and> (\<sigma> \<Turnstile> G) \<longleftrightarrow> ((NNil (nfirst \<sigma>)) \<Turnstile> F_e) \<and> ((NNil (nfirst \<sigma>)) \<Turnstile> G_e)" 
     using GNF_defs True assms 
     by (metis less_numeral_extra(3))
   have 3: "state_qpitl (F_e iand G_e)"
     by (simp add: assms(2) assms(6) iand_d_def inot_d_def ior_d_def)
   have 4: "state_qpitl_list (F_now @ map (\<lambda>x. x iand F_e) G_now)" 
     using assms state_qpitl_list_append state_qpitl_list_map_alt by blast
   have 5: "length (F_now @ map (\<lambda>x. x iand F_e) G_now) = 
            length (map (\<lambda>x.  x schop G ) F_next @ G_next)" 
      by (simp add: assms(4) assms(8))
   have 6: "((NNil (nfirst \<sigma>)) \<Turnstile> F_e) \<and> ((NNil (nfirst \<sigma>)) \<Turnstile> G_e) \<longleftrightarrow>
            (\<sigma> \<Turnstile> (F_e iand G_e) iand empty ior
                (big_ior (map2 (\<lambda> x y. x iand next y) 
                         (F_now @ (map (\<lambda>x. x iand F_e) G_now)) 
                         ((map (\<lambda>x. x schop G) F_next) @  G_next))) )"
       using GNF_defs[of "(F_e iand G_e) iand empty ior
                (big_ior (map2 (\<lambda> x y. x iand next y) 
                         (F_now @ (map (\<lambda>x. x iand F_e) G_now)) 
                         ((map (\<lambda>x. x schop G) F_next) @  G_next))) "
                 "(F_e iand G_e)" "(F_now @ (map (\<lambda>x. x iand F_e) G_now))"
                 "((map (\<lambda>x. x schop G) F_next) @  G_next)" \<sigma>] 
     using assms 
     by (simp add: "3" "4" True)
   show ?thesis 
   using "1" "2" "6" by presburger
  qed
next
case False
  assume a: "nlength \<sigma> \<noteq> 0" 
  show ?thesis (is "?lhs \<longleftrightarrow> ?rhs") 
   proof 
     assume l:?lhs
     show ?rhs
     proof -
      have 7: "(\<exists>i. enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> F) \<and> (ndropn i \<sigma> \<Turnstile> G))" 
       using l by simp
      obtain i where 8: "enat i \<le> nlength \<sigma> \<and> (ntaken i \<sigma> \<Turnstile> F) \<and> (ndropn i \<sigma> \<Turnstile> G)"
        using 7 by blast
      have 9: "i=0 \<Longrightarrow> ?rhs" 
        proof -
          assume b:"i=0"
         have 10: "(ntaken 0 \<sigma> \<Turnstile> F) \<and> (\<sigma> \<Turnstile> G)"
           using "8" b by auto  
         have 11: " ( \<sigma> \<Turnstile> F_e)"
            using assms 
            by (metis (mono_tags, lifting) "8" GNF_defs  b less_numeral_extra(3) min_def ntaken_nfirst 
                ntaken_nlength state_qpitl_defs zero_enat_def)
         have 12: "\<sigma> \<Turnstile> F_e iand big_ior (map2 (\<lambda> x y. x iand next y) G_now G_next)" 
             using "10" 11 a assms(5) by force
         have 13: "\<sigma> \<Turnstile> big_ior (map2 (\<lambda> x y. x iand next y) 
                        (map (\<lambda>x. x iand F_e) G_now) G_next)"
           using iand_big_ior_dist "12" assms(8) itl_ieq by blast
         show ?thesis
         using "13" assms(4) assms(8) big_ior_map2_append by force 
        qed
     have 14: "0<i \<and> enat i \<le> nlength \<sigma> \<Longrightarrow> ?rhs"
       proof -
        assume c: "0<i \<and> enat i \<le> nlength \<sigma>" 
       have 15: " (ntaken i \<sigma> \<Turnstile> F) \<and> (ndropn i \<sigma> \<Turnstile> G)" 
         by (simp add: "8")
       have 16: " (ntaken i \<sigma> \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand next y) F_now F_next))) " 
          using "8" assms(1) c enat_0_iff(1) by auto
       have 17: "((ntaken i \<sigma>) \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand next y) F_now F_next))) \<longleftrightarrow>
                 (\<exists>ia<length F_now. ntaken i \<sigma> \<Turnstile> F_now ! ia iand next F_next ! ia) " 
          using "16" big_ior_map2_defs[of F_now F_next "(ntaken i \<sigma>)" "(\<lambda> x y. x iand next y)"  ]
         assms by blast
       have 18: "length F_now = length (map (\<lambda>x. x schop G) F_next)"
         using assms(4) by fastforce
       have 19: "(\<sigma> \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand next y)
                        F_now  (map (\<lambda>x. x schop G) F_next) ))) \<longleftrightarrow>
                 (\<exists>i<length F_now. \<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop G ) F_next ! i)"
          using 18  big_ior_map2_defs[of F_now "(map (\<lambda>x. x schop G) F_next)" \<sigma> "(\<lambda> x y. x iand next y)"]
       by blast
       obtain j where  20: "j<length F_now \<and> (ntaken i \<sigma> \<Turnstile> F_now ! j iand next F_next ! j)"
          using "16" "17" by blast 
       have 21: "enat (i-1) \<le> nlength \<sigma> - enat (Suc 0)" 
          by (metis One_nat_def c enat_minus_mono1 idiff_enat_enat)
       have 22: "( (ntaken (i-1) (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! j))"
         by (metis "20" One_nat_def Suc_leI c diff_add iand_defs ntaken_ndropn_swap plus_enat_simps(1) 
             semantics_qpitl.simps(4)) 
       have 23: "(  (ndropn (i-1) (ndropn (Suc 0) \<sigma>) \<Turnstile> G))" 
        using 8 by (simp add:  c ndropn_ndropn)
       have 24: "( \<sigma> \<Turnstile> F_now ! j iand next ((F_next ! j) schop G ) )"
          using 20 15 by simp 
           (metis "21" "22" "23" assms(3) ntaken_nfirst state_qpitl_defs state_qpitl_list_defs)
       have 25: "(\<exists>i<length F_now. \<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop G ) F_next ! i)"   
             using "20" "24" assms(4) by fastforce
       show ?thesis
       using "18" "19" "25" assms(8) big_ior_map2_append by auto 
     qed
    show ?thesis 
    using "14" "8" "9" by blast
   qed
   next
    assume r: "?rhs" 
    show "?lhs" 
     proof -
      have 26: "\<sigma> \<Turnstile> big_ior (map2 (\<lambda>x y. x iand next y) (F_now @ map (\<lambda>x. x iand F_e) G_now) (map (\<lambda>x.  x schop G ) F_next @ G_next))"  
        using a r by auto   
      have 27: "length F_now = length (map (\<lambda>x.  x schop G ) F_next)"
         by (simp add: assms(4)) 
      have 28: "length (map (\<lambda>x. x iand F_e) G_now) = length (G_next)" 
         using assms(8) by fastforce
      have 29: "(\<sigma> \<Turnstile> (big_ior (map2 (\<lambda>x y. x iand next y) (F_now) (map (\<lambda>x.  x schop G ) F_next)))) 
                      \<or>
                  (\<sigma> \<Turnstile>   ((big_ior (map2 (\<lambda>x y. x iand next y) (map (\<lambda>x. x iand F_e) G_now) (G_next)   )))) "
        using "26" "27" "28" big_ior_map2_append itl_ieq ior_defs by blast
      have 30: "(\<sigma> \<Turnstile> (big_ior (map2 (\<lambda>x y. x iand next y) (F_now) (map (\<lambda>x.  x schop G ) F_next)))) \<longleftrightarrow>
                 (\<exists>i<length F_now. \<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop G ) F_next ! i)" 
         using big_ior_map2_defs[of F_now "(map (\<lambda>x.  x schop G ) F_next)" \<sigma> "(\<lambda>x y. x iand next y)"]
          using "27" by fastforce
      have 31: "(\<exists>i<length F_now. \<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop G ) F_next ! i) \<Longrightarrow> ?lhs"
        proof -
         assume d: "(\<exists>i<length F_now. \<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop G ) F_next ! i)"  
         obtain i where 32: "i<length F_now \<and> (\<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop G ) F_next ! i)"
          using d by blast
         have 33: "(\<sigma> \<Turnstile> F_now ! i)"
           using "32" iand_defs by blast 
         have 34: "map (\<lambda>x.  x schop G ) F_next ! i = (F_next !i) schop G"
             using "32" assms(4) by auto 
         have 35: "1 \<le> nlength \<sigma> \<and> (\<exists>ia. enat ia \<le> nlength \<sigma> - enat (Suc 0) \<and> (ntaken ia (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! i) \<and> (ndropn ia (ndropn (Suc 0) \<sigma>) \<Turnstile> G))"
            using 32 34 by simp  
         obtain ia where 36: "enat ia \<le> nlength \<sigma> - enat (Suc 0) \<and> (ntaken ia (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! i) \<and> (ndropn ia (ndropn (Suc 0) \<sigma>) \<Turnstile> G)"
          using 35 by blast
         have 37: "(ntaken (ia+1) ( \<sigma>) \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) \<longleftrightarrow>
                    (\<exists>i<length F_now. ntaken (ia+1) ( \<sigma>) \<Turnstile> F_now ! i iand next F_next ! i)" 
           using assms  big_ior_map2_defs[of F_now F_next "(ntaken (ia+1) ( \<sigma>))" "(\<lambda> x y. x iand (next y))"] by blast
         have 38: "(ntaken (ia+1)  \<sigma> \<Turnstile> F_now ! i iand next F_next ! i)"
           by simp 
            (metis "32" "33" "35" "36" One_nat_def add.commute assms(3) enat_min_eq enat_ord_simps(1) 
             le_add1 ntaken_ndropn_swap ntaken_nfirst one_enat_def plus_1_eq_Suc state_qpitl_defs
              state_qpitl_list_defs)
         have 39: " (ntaken (ia+1)  \<sigma> \<Turnstile> F)"
           using "32" "37" "38" assms(1) by auto  
         have 40: "(ndropn (ia+1)  \<sigma>) \<Turnstile> G"
           by (metis "36" One_nat_def add.commute ndropn_ndropn) 
         have 41: "(ia+1) \<le> nlength \<sigma>"  
            by (metis "35" "36" One_nat_def enat_min_eq one_enat_def plus_enat_simps(1))
         show ?thesis 
         using "39" "40" "41" by auto
       qed
      have 42: "(\<sigma> \<Turnstile>   ((big_ior (map2 (\<lambda>x y. x iand next y) (map (\<lambda>x. x iand F_e) G_now) (G_next)   )))) \<longleftrightarrow>
                 (\<exists>i<length (map (\<lambda>x. x iand F_e) G_now). \<sigma> \<Turnstile> map (\<lambda>x. x iand F_e) G_now ! i iand next G_next ! i)" 
        using big_ior_map2_defs[of "(map (\<lambda>x. x iand F_e) G_now)" G_next \<sigma> "(\<lambda>x y. x iand next y)"] 
           using "28" by fastforce
      have 43: "(\<exists>i<length (map (\<lambda>x. x iand F_e) G_now). \<sigma> \<Turnstile> map (\<lambda>x. x iand F_e) G_now ! i iand next G_next ! i) \<Longrightarrow> ?lhs"
        proof -
         assume e: "(\<exists>i<length (map (\<lambda>x. x iand F_e) G_now). \<sigma> \<Turnstile> map (\<lambda>x. x iand F_e) G_now ! i iand next G_next ! i)"  
         obtain i where 44: "i<length (map (\<lambda>x. x iand F_e) G_now) \<and> (\<sigma> \<Turnstile> map (\<lambda>x. x iand F_e) G_now ! i iand next G_next ! i)"
            using e by blast
         have 45: "(ntaken 0 \<sigma>) \<Turnstile> ( G_now ! i) iand F_e"
           using assms 
           by (metis "28" "44"  iand_defs ntaken_0 nth_map state_qpitl_defs state_qpitl_list_defs) 
         have 46: "(ntaken 0 \<sigma>) \<Turnstile> F"  
            using "45" assms(1) by fastforce
         have 47: "(ndropn 0 \<sigma>) \<Turnstile> G_now ! i iand next G_next ! i"
           using "44" by fastforce
         have 48: "((ndropn 0 \<sigma>) \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand (next y))  G_now G_next))) \<longleftrightarrow> 
                    (\<exists>i<length G_now. ndropn 0 \<sigma> \<Turnstile> G_now ! i iand next G_next ! i)" 
           using big_ior_map2_defs[of G_now G_next "(ndropn 0 \<sigma>)"  "(\<lambda> x y. x iand (next y))"]
            using assms(8) by fastforce
         have 49: "(ndropn 0 \<sigma>) \<Turnstile> G"
           using "44" "47" "48" assms(5) by auto 
         show ?thesis 
         by (metis "46" "49" semantics_qpitl.simps(5) zero_enat_def zero_le)
       qed
     show ?thesis 
     using "29" "30" "31" "42" "43" by fastforce  
    qed
  qed  
qed


lemma GNF_schopstar_step: 
 assumes " \<turnstile> F ieqv (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) "  
        " state_qpitl F_e" 
        " state_qpitl_list F_now"
        " length F_now = length F_next" 
 shows  " (\<sigma> \<Turnstile> schopstar F) \<longleftrightarrow>
          (\<sigma> \<Turnstile> (itrue iand empty) ior 
                (big_ior (map2 (\<lambda> x y. x iand next y) F_now (map (\<lambda>x. x schop (schopstar F)) F_next))))" 
        (is "?lhs \<longleftrightarrow> ?rhs")
proof -
 have 1: "(\<sigma> \<Turnstile> schopstar F) \<longleftrightarrow> 
          (\<exists>l. nnth l 0 = 0 \<and> nfinite l \<and> nidx l \<and> enat (nlast l) = nlength \<sigma> \<and> nfinite \<sigma> \<and> 
           (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> F)))"
   by simp  
 have 2: " ?lhs \<Longrightarrow> ?rhs" 
   proof -
    assume l: "?lhs" 
    obtain l where 3: "nnth l 0 = 0 \<and> nfinite l \<and> nidx l \<and> enat (nlast l) = nlength \<sigma> \<and> nfinite \<sigma> \<and> 
           (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> F))" 
      using l 1 by blast
    have 4: " nlength l = 0 \<Longrightarrow> ?rhs" 
      proof -
       assume n: "nlength l = 0" 
       have 5: " nlength \<sigma> = 0 "
         by (metis "3" n nnth_nlast the_enat_0 zero_enat_def)
       show ?thesis 
       by (simp add: "5")
      qed
    have 6: "0< nlength l \<Longrightarrow> ?rhs"
     proof -
      assume g: " 0 < nlength l " 
      have 7: " 0 < nlength \<sigma> " 
         by (metis "3" g gr_zeroI less_numeral_extra(3) nfinite_conv_nlength_enat nidx_less_last_1 
             nnth_nlast the_enat.simps the_enat_0)
      have 8: "(nsubn \<sigma> (nnth l 0) (nnth l (Suc 0)) \<Turnstile> F)"
        using 3 g by (metis zero_enat_def)
      have 9: " (nsubn \<sigma> (nnth l 0) (nnth l (Suc 0))) = (ntaken (nnth l (Suc 0)) \<sigma>)"
          using 3 by (simp add: nsubn_def1)
      have 10: " (ntaken (nnth l (Suc 0)) \<sigma>) \<Turnstile> F "
          using 9 8 by auto
      have 11: "(ntaken (nnth l (Suc 0)) \<sigma>) \<Turnstile> (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next)))"
         using "10" assms(1) itl_ieq by blast 
      have 12: "(ntaken (nnth l (Suc 0)) \<sigma>) \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))"
        by (metis "11" "3" "7" "9" One_nat_def eSuc_enat empty_defs enat_defs(1) g iand_defs ileI1 
            ior_defs less_numeral_extra(1) linorder_le_cases nidx_gr_first nsubn_nlength_gr_one 
            ntaken_all zero_less_iff_neq_zero) 
      have 13: "0 < nlength (ntaken (nnth l (Suc 0)) \<sigma>)"
           by (metis "3" "7" "9" One_nat_def eSuc_enat enat_defs(1) g ileI1 less_numeral_extra(1) 
               linorder_le_cases nidx_gr_first nsubn_nlength_gr_one ntaken_all) 
      have 14: "(nlength (ntaken (nnth l (Suc 0)) \<sigma>) > 0 \<and> 
                (\<exists>i<length F_now. ((NNil (nfirst (ntaken (nnth l (Suc 0)) \<sigma>))) \<Turnstile> F_now !i ) \<and> 
                                  ((ndropn 1 (ntaken (nnth l (Suc 0)) \<sigma>)) \<Turnstile> F_next !i)) )"
          using GNF_defs[of F F_e F_now F_next "(ntaken (nnth l (Suc 0)) \<sigma>)" ] 12 13 
          by (metis "10" assms(1) assms(2) assms(3) assms(4) order_less_irrefl)
      obtain i where 15: "i<length F_now \<and> ((NNil (nfirst (ntaken (nnth l (Suc 0)) \<sigma>))) \<Turnstile> F_now !i ) \<and> 
                          ((ndropn 1 (ntaken (nnth l (Suc 0)) \<sigma>)) \<Turnstile> F_next !i) "
             using 14 by blast 
      have 19: " ((ndropn 1 \<sigma>) \<Turnstile> F_next !i schop (schopstar F)) \<longleftrightarrow>
                (\<exists>ia. enat ia \<le> nlength \<sigma> - enat (Suc 0) \<and>
         (ntaken ia (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! i) \<and>
         (\<exists>l. nnth l 0 = 0 \<and>
              nfinite l \<and>
              nidx l \<and>
              enat (nlast l) = nlength \<sigma> - enat (Suc 0) - enat ia \<and> nfinite \<sigma> \<and> 
             (\<forall>i. enat i < nlength l \<longrightarrow> 
               (nsubn (ndropn ia (ndropn (Suc 0) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> F)))) " 
        by simp
      have 191: "enat ((nnth l (Suc 0))-1 ) \<le>  nlength \<sigma> - enat (Suc 0)"
          by (metis "3" One_nat_def eSuc_enat enat_minus_mono1 enat_ord_simps(1) g idiff_enat_enat 
              ileI1 nidx_all_le_nlast zero_enat_def) 
      have 192: "ntaken ((nnth l (Suc 0))-1 ) (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! i"
        by (metis "15" "191" "3" "7" One_nat_def Suc_leI diff_add eSuc_enat enat_min_eq g ileI1 
            less_numeral_extra(1) nidx_gr_first ntaken_ndropn_swap zero_enat_def)
      have 193: "nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) 0 = 0 " 
        using enat_defs(1) by auto
      have 194: "nfinite (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l))"
        by (simp add: "3") 
      have 195: "nidx (ndropn 1 l) " 
         by (metis "3" g ileI1 nidx_ndropn one_eSuc one_enat_def)
      have 196: "nnth (ndropn 1 l) 0 = (nnth l (Suc 0))"
        by simp
      have 197: "nidx (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l))" 
         using  nidx_shiftm[of "(ndropn 1 l)" "(nnth l (Suc 0))"   ] 
         by (metis "195" "196" One_nat_def)
      have 198: "nlast (ndropn (Suc 0) l) = nlast l " 
         by (metis Suc_ile_eq g nappend_NNil nappend_ntaken_ndropn nlast_NCons ntaken_0 zero_enat_def)
      have 199: " nlast (nmap (\<lambda>x. x - nnth l (Suc 0)) (ndropn (Suc 0) l)) = 
                nlength \<sigma> - nnth l (Suc 0)" 
          using nlast_nmap[of "(ndropn (Suc 0) l)" "(\<lambda>x. x - (nnth l (Suc 0)))"]  
          using "194" nfinite_nmap 
          by (metis "198" "3" idiff_enat_enat)
      have 200: "nlength \<sigma> - nnth l (Suc 0) = nlength \<sigma> - enat (Suc 0) - ((nnth l (Suc 0))-1 )"
          by auto 
             (metis "3" One_nat_def Suc_pred g ileI1 less_numeral_extra(1) ndropn_ndropn 
              ndropn_nlength nidx_gr_first one_eSuc one_enat_def plus_1_eq_Suc) 
      have 201: " enat (nlast (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l))) =
                  nlength \<sigma> - enat (Suc 0) - ((nnth l (Suc 0))-1 )"
          using "199" "200" by presburger  
      have 202: "nlength (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) = nlength l - (Suc 0)"
         by force 
      have 203: "(ndropn ((nnth l (Suc 0))-1 ) (ndropn (Suc 0) \<sigma>)) =  (ndropn (nnth l (Suc 0)) \<sigma>)"
         by (metis "3" One_nat_def Suc_pred g ileI1 ndropn_ndropn nidx_gr_first one_eSuc 
             one_enat_def plus_1_eq_Suc zero_less_one) 
      have 204: "\<And>j. j< nlength l - (Suc 0) \<Longrightarrow> 
              nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) j =
              nnth l (j+ (Suc 0)) -  (nnth l (Suc 0))"
          by simp
      have 2041: "\<And>j. j< nlength l - (Suc 0) \<Longrightarrow> 
              nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) (Suc j) =
              nnth l ((Suc j) + (Suc 0)) -  (nnth l (Suc 0))" 
          by (metis add.right_neutral add_Suc_right eSuc_enat ileI1 ndropn_nlength ndropn_nnth 
              nnth_nmap plus_1_eq_Suc)
      have 205: "\<And>j. j< nlength l - (Suc 0) \<Longrightarrow>
                   nnth l (j+ (Suc 0)) -  (nnth l (Suc 0)) < nnth l ((Suc j)+ (Suc 0)) -  (nnth l (Suc 0))" 
          using 204 197 unfolding nidx_expand by auto
           (metis "204" One_nat_def add.commute eSuc_enat ileI1 plus_1_eq_Suc)
      have 206: "\<And>j. j< nlength l - (Suc 0) \<Longrightarrow>
                     (nsubn (ndropn ((nnth l (Suc 0))-1 ) (ndropn (Suc 0) \<sigma>)) 
                (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) j) 
                (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) (Suc j))) =
                     (nsubn (ndropn (nnth l (Suc 0)) \<sigma>)
                           (nnth l (j+ (Suc 0)) -  (nnth l (Suc 0)))
                           (nnth l ((Suc j)+ (Suc 0)) -  (nnth l (Suc 0))))  "
           using 203 204  2041 by simp
      have 207: "\<And>j. j< nlength l - (Suc 0) \<Longrightarrow> 
                    (nsubn (ndropn (nnth l (Suc 0)) \<sigma>)
                           (nnth l (j+ (Suc 0)) -  (nnth l (Suc 0)))
                           (nnth l ((Suc j)+ (Suc 0)) -  (nnth l (Suc 0)))) = 
                 (nsubn \<sigma> (nnth l (Suc j)) (nnth l (Suc (Suc j))))  "
            using nsubn_ndropn[of _ _ \<sigma>  "(nnth l (Suc 0))"] by simp 
          (metis (no_types, lifting) "197" "202" "204" "205" add.commute add.right_neutral 
           bot_nat_0.not_eq_extremum diff_add less_nat_zero_code nidx_gr_first order_less_imp_le
             plus_1_eq_Suc zero_less_diff)
      have 208: "\<And>j. j< nlength l - (Suc 0) \<Longrightarrow>
                     (nsubn \<sigma> (nnth l (Suc j)) (nnth l (Suc (Suc j)))) \<Turnstile> F"
         using 3 
         by (metis eSuc_enat enat_min g ileI1 one_eSuc plus_1_eSuc(2) zero_enat_def)   
      have 209: "(\<forall>i. enat i < nlength (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) \<longrightarrow> 
               (nsubn (ndropn ((nnth l (Suc 0))-1 ) (ndropn (Suc 0) \<sigma>)) 
                (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) i) 
                (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) (Suc i)) \<Turnstile> F))" 
           using "202" "206" "207" "208" by presburger
      have 210: "( enat ((nnth l (Suc 0))-1 ) \<le> nlength \<sigma> - enat (Suc 0) \<and>
         (ntaken ((nnth l (Suc 0))-1 ) (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! i) \<and>
         ( nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) 0 = 0 \<and>
              nfinite (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) \<and>
              nidx (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) \<and>
              enat (nlast (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l))) = 
              nlength \<sigma> - enat (Suc 0) - enat ((nnth l (Suc 0))-1 ) \<and> nfinite \<sigma> \<and> 
             (\<forall>i. enat i < nlength (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) \<longrightarrow> 
               (nsubn (ndropn ((nnth l (Suc 0))-1 ) (ndropn (Suc 0) \<sigma>)) 
                      (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) i) 
                      (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) (Suc i)) \<Turnstile> F))))"
          using 191 192 193 194 197 201 209 3   by blast 
      have 20: "((ndropn 1 \<sigma>) \<Turnstile> F_next !i schop (schopstar F))" 
         using 210 "19" by blast
      have 21: "\<sigma> \<Turnstile> F_now ! i"
         by (metis "15" assms(3) ntaken_nfirst state_qpitl_defs state_qpitl_list_defs) 
      have 22: "length F_now = length (map (\<lambda>x.  x schop schopstar F  ) F_next)"
         by (simp add: assms(4))
      have 23: " (\<sigma> \<Turnstile> big_ior (map2 (\<lambda>x y. x iand next y) F_now (map (\<lambda>x.  x schop schopstar F  ) F_next))) \<longleftrightarrow>
                 (\<exists>i<length F_now. \<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop schopstar F  ) F_next ! i)" 
        using  big_ior_map2_defs[of F_now  "(map (\<lambda>x. x schop (schopstar F)) F_next)" \<sigma> "(\<lambda> x y. x iand next y)"]       
          22 by auto
      have 24: "( \<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop schopstar F  ) F_next ! i)"
          by (metis "15" "20" "21" "7" assms(4) iand_defs ileI1 nth_map one_eSuc semantics_qpitl.simps(4))        
      show ?thesis using "15" "23" "24" ior_defs by blast
    qed
   show ?thesis 
   using "4" "6" zero_less_iff_neq_zero by blast
 qed
 have 1000: "?rhs \<Longrightarrow> ?lhs"
  proof -
   assume r: "?rhs"
    have 1001: " nlength \<sigma> = 0 \<or> (\<sigma> \<Turnstile> big_ior (map2 (\<lambda>x y. x iand next y) F_now (map (\<lambda>x.  x schop schopstar F  ) F_next)))" 
        using r by simp
    have 1002: "nlength \<sigma> = 0 \<Longrightarrow> ?lhs"
       proof -
        assume n: "nlength \<sigma> = 0"
        have 1003: "nnth (NNil 0) 0 = 0"
          by (simp add: nnth_NNil)
        have 1004: " nfinite (NNil 0)"
           by simp
        have 1005: "  nidx (NNil 0) "
            using Suc_ile_eq nidx_expand by auto
        have 1006: " enat (nlast (NNil 0)) = nlength \<sigma>"
           by (simp add: n zero_enat_def)
        have 1007: "  nfinite \<sigma> "
            using n 
            by (simp add: nfinite_conv_nlength_enat zero_enat_def)
        have 1008: " (\<forall>i. enat i < nlength (NNil 0) \<longrightarrow> (nsubn \<sigma> (nnth (NNil 0) i) (nnth (NNil 0) (Suc i)) \<Turnstile> F))" 
         by auto
        show ?thesis apply simp         
           using "1003" "1005" "1006" "1007" "1008" by blast
      qed
   have 1009: "(\<sigma> \<Turnstile> big_ior (map2 (\<lambda>x y. x iand next y) F_now (map (\<lambda>x.  x schop schopstar F  ) F_next)))
               \<Longrightarrow> ?lhs"
     proof -
      assume b: "(\<sigma> \<Turnstile> big_ior (map2 (\<lambda>x y. x iand next y) F_now (map (\<lambda>x.  x schop schopstar F  ) F_next)))"
      have 1010: "length F_now = length (map (\<lambda>x.  x schop schopstar F  ) F_next)"
         by (simp add: assms(4))
      have 1011: "(\<exists>i<length F_now. \<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop schopstar F  ) F_next ! i)"   
          using 1010 b big_ior_map2_defs[of F_now  "(map (\<lambda>x. x schop (schopstar F)) F_next)" \<sigma> "(\<lambda> x y. x iand next y)"]
        by blast
      obtain i where 1012: "i<length F_now \<and> (\<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop schopstar F  ) F_next ! i)"
         using 1011 by blast
      have 1013: "(\<sigma> \<Turnstile> F_now ! i)"
         using 1012 by auto
      have 1014: "((ndropn 1 \<sigma>) \<Turnstile> F_next !i schop (schopstar F))" 
            using 1012 assms(4) by fastforce  
      have 1015: "((ndropn 1 \<sigma>) \<Turnstile> F_next !i schop (schopstar F)) \<longleftrightarrow> 
              (\<exists>ia. enat ia \<le> nlength \<sigma> - enat (Suc 0) \<and>
          (ntaken ia (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! i) \<and>
          (\<exists>l. nnth l 0 = 0 \<and>
               nfinite l \<and>
               nidx l \<and>
               enat (nlast l) = nlength \<sigma> - enat (Suc 0) - enat ia \<and> nfinite \<sigma> \<and>
                (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn (ndropn ia (ndropn (Suc 0) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> F))))"
        by simp 
      obtain ia where 1016: "enat ia \<le> nlength \<sigma> - enat (Suc 0) \<and>
          (ntaken ia (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! i) \<and>
          (\<exists>l. nnth l 0 = 0 \<and>
               nfinite l \<and>
               nidx l \<and>
               enat (nlast l) = nlength \<sigma> - enat (Suc 0) - enat ia \<and> nfinite \<sigma> \<and>
                (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn (ndropn ia (ndropn (Suc 0) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> F)))"
         using 1014 1015 by blast
     obtain l where 1017: "nnth l 0 = 0 \<and>
               nfinite l \<and>
               nidx l \<and>
               enat (nlast l) = nlength \<sigma> - enat (Suc 0) - enat ia \<and> nfinite \<sigma> \<and>
                (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn (ndropn ia (ndropn (Suc 0) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> F))"
      using 1016 by blast
     have 1018: "0<nlength \<sigma>" 
        using "1012" by force
     have 10183: "nlength \<sigma> - enat (Suc 0) - enat ia = nlength \<sigma> - Suc ia" 
           by (metis One_nat_def ndropn_ndropn ndropn_nlength plus_1_eq_Suc)
     have 10184: "Suc ia \<le> nlength \<sigma> " 
        by (metis "1016" "1018" One_nat_def eSuc_enat enat_min_eq ileI1 one_eSuc one_enat_def plus_1_eSuc(2))
     have 10185: "nlength \<sigma> - Suc ia + Suc ia = nlength \<sigma>" 
         by (metis "10184" add.commute enat.simps(3) enat_add_sub_same le_iff_add)
     have 1019: " (nlength \<sigma> - enat (Suc 0) - enat ia) + Suc ia = nlength \<sigma>" 
        by (simp add: "10183" "10185")
     have 1020: "nlast (nmap (\<lambda>x. x+ Suc ia) l) = nlength \<sigma>"      
       by (metis "1017" "1019" nlast_nmap plus_enat_simps(1))
     have 1021: "nlast (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) = nlength \<sigma>" 
         using "1020" by force
     have 1022: "nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) 0 = 0" 
        by simp
     have 1023: " 0 < nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) 1 " 
         using zero_enat_def by force
     have 1024: " nfinite (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l))" 
        by (simp add: "1017")
     have 1025: "nidx (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l))"
        using "1017" nidx_Ncons_shift by blast 
     have 1028: "(nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) 0) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc 0))) =
                 (nsubn \<sigma> 0 (Suc ia)) " 
          using "1017" zero_enat_def by fastforce
     have 1029: "((nsubn \<sigma> 0 (Suc ia)) \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) \<longleftrightarrow>
                   (\<exists>i<length F_now. nsubn \<sigma> 0 (Suc ia) \<Turnstile> F_now ! i iand next F_next ! i)"
         using  big_ior_map2_defs[of F_now F_next "(nsubn \<sigma> 0 (Suc ia))" "(\<lambda> x y. x iand (next y))"]
           assms by blast 
     have 1030: "nsubn \<sigma> 0 (Suc ia) \<Turnstile> F_now ! i "
         using 1016 
         by (metis "1012" "1013" assms(3) ndropn_0 nsubn_def1 ntaken_nfirst state_qpitl_defs state_qpitl_list_defs)
     have 1031: "nsubn \<sigma> 0 (Suc ia) \<Turnstile>  next F_next ! i"
          using 1016 ndropn_nsubn[of "Suc ia" \<sigma> "Suc 0" 0]  
          by (metis "1019" One_nat_def add_diff_cancel_left' enat_le_plus_same(2) ileI1 le_add1 
          nsubn_def1 nsubn_nlength_gr_one one_eSuc plus_1_eq_Suc semantics_qpitl.simps(4) zero_less_Suc)
     have 1034: "(nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) 0) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc 0)) \<Turnstile> 
                   (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))))"  
       using "1012" "1028" "1029" "1030" "1031" by auto
     have 1035: "\<And>j. j<nlength l \<Longrightarrow>
                      (nnth ( (nmap (\<lambda>x. x+ Suc ia) l)) j) = nnth l j + Suc ia "
          by simp      
     have 1036: "\<And>j. j<nlength l \<Longrightarrow>
                      (nnth ( (nmap (\<lambda>x. x+ Suc ia) l)) (Suc j)) = nnth l (Suc j) + Suc ia" 
           using Suc_ile_eq by force
     have 1037: "\<And>j. j<nlength l \<Longrightarrow>
                     nnth l j + Suc ia < nnth l (Suc j) + Suc ia"
          by (metis "1017" add_strict_right_mono eSuc_enat ileI1 nidx_expand) 
     have 1038: "\<And>j. j<nlength l \<Longrightarrow>
                     enat ( nnth l (Suc j) + Suc ia) \<le> nlength \<sigma> " 
            using 1017 nidx_expand by simp 
          (metis "1020" "1025" "1036" add_Suc_right eSuc_enat enat_ord_simps(1) ileI1 nfinite_nmap
           nidx_LCons_1 nidx_all_le_nlast nlength_nmap)  
     have 1039: "\<And>j. j<nlength l \<Longrightarrow>
                    Suc 0 + (nnth l j + Suc ia) \<le>  nnth l (Suc j) + Suc ia"
             using "1037" by fastforce 
     have 1040: "\<And>j. j<nlength l \<Longrightarrow> 
                      ndropn (Suc 0) (nsubn \<sigma> (nnth l j + Suc ia) (nnth l (Suc j) + Suc ia)) = 
                      nsubn \<sigma> (Suc 0 + (nnth l j + Suc ia)) (nnth l (Suc j) + Suc ia)" 
         using ndropn_nsubn[of _ \<sigma> "Suc 0" ]  using "1038" "1039" by presburger
     have 1041: "\<And>j.  j<nlength l \<Longrightarrow> 
                   (nsubn (ndropn ia (ndropn (Suc 0) \<sigma>)) (nnth l j) (nnth l (Suc j))) =
                   nsubn \<sigma> ((nnth l j) + Suc ia) ((nnth l (Suc j)) + Suc ia)"
           using nsubn_ndropn[of _ _ \<sigma> "Suc ia" ]
             by (metis "1037" One_nat_def add_less_cancel_right ndropn_ndropn plus_1_eq_Suc)
     have 1280: "(nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) 0) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc 0)) \<Turnstile> F)" 
        using "1034" assms(1) by simp 
     have 1281: "(\<forall>i. enat i < nlength ( (nmap (\<lambda>x. x+ Suc ia) l)) \<longrightarrow> 
                 (nsubn \<sigma> (nnth ( (nmap (\<lambda>x. x+ Suc ia) l)) i) (nnth ( (nmap (\<lambda>x. x+ Suc ia) l)) (Suc i)) \<Turnstile> F))"
        using "1017" "1036" "1041" by auto 
     have 1290: "(\<forall>i. enat i < nlength (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) \<longrightarrow> 
          (nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) i) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc i)) \<Turnstile> F)) \<longleftrightarrow>
                 (nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) 0) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc 0)) \<Turnstile> F) \<and>
               (\<forall>i. enat i < nlength ( (nmap (\<lambda>x. x+ Suc ia) l)) \<longrightarrow> 
                 (nsubn \<sigma> (nnth ( (nmap (\<lambda>x. x+ Suc ia) l)) i) (nnth ( (nmap (\<lambda>x. x+ Suc ia) l)) (Suc i)) \<Turnstile> F))" 
          using forall_ncons_split[of "(nmap (\<lambda>x. x+ Suc ia) l)" 
            "\<lambda> x y. (nsubn \<sigma> x y \<Turnstile> F)"  ] by (simp add: "1017")         
     have 1300: "(\<forall>i. enat i < nlength (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) \<longrightarrow> 
          (nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) i) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc i)) \<Turnstile> F))"     
        using "1280" "1281" "1290" by blast
      have 1400: "\<exists>l. nnth l 0 = 0 \<and> nfinite l \<and> nidx l \<and> enat (nlast l) = nlength \<sigma> \<and> nfinite \<sigma> \<and> 
                  (\<forall>i. enat i < nlength l \<longrightarrow> (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> F))" 
        by (meson "1017" "1021" "1022" "1025" "1300" nfinite_NConsI nfinite_nmap)
      show ?thesis using 1400 by simp 
     qed
    show ?thesis 
    using "1001" "1002" "1009" by blast
   qed
  show ?thesis 
  using "1000" "2" by blast
qed



lemma GNF_omega_step: 
 assumes " \<turnstile> F ieqv (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) "  
        " state_qpitl F_e" 
        " state_qpitl_list F_now"
        " length F_now = length F_next" 
 shows  " (\<sigma> \<Turnstile> omega F) \<longleftrightarrow>
          (\<sigma> \<Turnstile> (ifalse iand empty) ior 
                (big_ior (map2 (\<lambda> x y. x iand next y) F_now (map (\<lambda>x. x schop (omega F)) F_next))))" 
        (is "?lhs \<longleftrightarrow> ?rhs")
proof -
 have 1: "(\<sigma> \<Turnstile> omega F) \<longleftrightarrow> 
          (\<exists>l. \<not>nfinite l \<and> nnth l 0 = 0 \<and>  nidx l \<and> 
             (\<forall>i. (nnth l i) \<le> nlength \<sigma>) \<and>
           (\<forall>i.  (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> F)))"
   by simp  
 have 2: " ?lhs \<Longrightarrow> ?rhs" 
   proof -
    assume l: "?lhs" 
    obtain l where 3: "\<not>nfinite l \<and> nnth l 0 = 0 \<and>  nidx l \<and> 
             (\<forall>i. (nnth l i) \<le> nlength \<sigma>) \<and>
           (\<forall>i.  (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> F))" 
      using l 1 by blast
    have 6: "\<not>nfinite \<sigma>" 
      using "3" infinite_nidx_imp_infinite_interval by blast
    have 7: " 0 < nlength \<sigma> " 
      using "6" enat_defs(1) nfinite_conv_nlength_enat by fastforce
    have 8: "(nsubn \<sigma> (nnth l 0) (nnth l (Suc 0)) \<Turnstile> F)"
      using "3" by blast
    have 9: " (nsubn \<sigma> (nnth l 0) (nnth l (Suc 0))) = (ntaken (nnth l (Suc 0)) \<sigma>)"
          using 3 by (simp add: nsubn_def1)
    have 10: " (ntaken (nnth l (Suc 0)) \<sigma>) \<Turnstile> F "
          using 9 8 by auto
    have 11: "(ntaken (nnth l (Suc 0)) \<sigma>) \<Turnstile> (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next)))"
         using "10" assms(1) itl_ieq by blast 
    have 12: "(ntaken (nnth l (Suc 0)) \<sigma>) \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))"
        by (metis "11" "3" "9" One_nat_def empty_defs enat_defs(2) iand_defs iless_eSuc0 ior_defs 
            less_numeral_extra(1) less_numeral_extra(3) nidx_gr_first nlength_eq_enat_nfiniteD 
             not_less nsubn_nlength_gr_one one_eSuc zero_enat_def)      
    have 13: "0 < nlength (ntaken (nnth l (Suc 0)) \<sigma>)"
     by (metis "3" "9" One_nat_def iless_eSuc0 less_numeral_extra(1) linorder_not_less 
         nidx_gr_first nlength_eq_enat_nfiniteD nsubn_nlength_gr_one one_eSuc one_enat_def zero_enat_def)
    have 14: "(nlength (ntaken (nnth l (Suc 0)) \<sigma>) > 0 \<and> 
                (\<exists>i<length F_now. ((NNil (nfirst (ntaken (nnth l (Suc 0)) \<sigma>))) \<Turnstile> F_now !i ) \<and> 
                  ((ndropn 1 (ntaken (nnth l (Suc 0)) \<sigma>)) \<Turnstile> F_next !i)) )"
          using GNF_defs[of F F_e F_now F_next "(ntaken (nnth l (Suc 0)) \<sigma>)" ] 12 13 
          by (metis "10" assms(1) assms(2) assms(3) assms(4) order_less_irrefl)
    obtain i where 15: "i<length F_now \<and> ((NNil (nfirst (ntaken (nnth l (Suc 0)) \<sigma>))) \<Turnstile> F_now !i ) \<and> 
                          ((ndropn 1 (ntaken (nnth l (Suc 0)) \<sigma>)) \<Turnstile> F_next !i) "
       using 14 by blast 
    have 19: " ((ndropn 1 \<sigma>) \<Turnstile> F_next !i schop (omega F)) \<longleftrightarrow>
               (\<exists>ia. enat ia \<le> nlength \<sigma> - enat (Suc 0) \<and>
          (ntaken ia (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! i) \<and>
          (\<exists>l. \<not> nfinite l \<and>
               nnth l 0 = 0 \<and> nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma> - enat (Suc 0) - enat ia) \<and> 
               (\<forall>i. nsubn (ndropn ia (ndropn (Suc 0) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> F))) " 
       by simp
    have 191: "enat ((nnth l (Suc 0))-1 ) \<le>  nlength \<sigma> - enat (Suc 0)"
       by (metis "3" One_nat_def enat_minus_mono1 idiff_enat_enat)    
    have 192: "ntaken ((nnth l (Suc 0))-1 ) (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! i"
     using "13" "15" "191" "3" "7" 
       by (simp add: enat_0_iff(1) ntaken_ndropn_swap)
    have 193: "nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) 0 = 0 " 
        using enat_defs(1) by auto
    have 194: "\<not>nfinite (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l))"
        by (simp add: "3") 
    have 195: "nidx (ndropn 1 l) " 
      by (metis "3" One_nat_def Suc_ile_eq dual_order.order_iff_strict enat_defs(1) 
          nfinite_conv_nlength_enat nidx_ndropn zero_le)
    have 196: "nnth (ndropn 1 l) 0 = (nnth l (Suc 0))"
        by simp
    have 197: "nidx (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l))" 
         using  nidx_shiftm[of "(ndropn 1 l)" "(nnth l (Suc 0))"   ] 
         by (metis "195" "196" One_nat_def)
    have 198: "(\<forall>i. enat (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) i) \<le> nlength \<sigma> - enat (Suc 0) - enat ((nnth l (Suc 0))-1 ))" 
      by (metis "6" ndropn_nlength nfinite_ndropn nfinite_ntaken nle_le ntaken_all)
    have 202: "nlength (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) = nlength l - (Suc 0)"
         by force 
    have 203: "(ndropn ((nnth l (Suc 0))-1 ) (ndropn (Suc 0) \<sigma>)) =  (ndropn (nnth l (Suc 0)) \<sigma>)"
      by (metis "3" One_nat_def Suc_pred ileI1 less_numeral_extra(1) ndropn_ndropn nidx_gr_first
           nlength_eq_enat_nfiniteD one_eSuc one_enat_def plus_1_eq_Suc zero_enat_def zero_less_iff_neq_zero)   
    have 204: "\<And>j. j< nlength l - (Suc 0) \<Longrightarrow> 
              nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) j =
              nnth l (j+ (Suc 0)) -  (nnth l (Suc 0))"
          by simp
    have 2041: "\<And>j. j< nlength l - (Suc 0) \<Longrightarrow> 
              nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) (Suc j) =
              nnth l ((Suc j) + (Suc 0)) -  (nnth l (Suc 0))" 
          by (metis add.right_neutral add_Suc_right eSuc_enat ileI1 ndropn_nlength ndropn_nnth nnth_nmap plus_1_eq_Suc)
    have 205: "\<And>j. j< nlength l - (Suc 0) \<Longrightarrow>
                   nnth l (j+ (Suc 0)) -  (nnth l (Suc 0)) < nnth l ((Suc j)+ (Suc 0)) -  (nnth l (Suc 0))" 
          using 204 197 unfolding nidx_expand apply auto
          by (metis "204" One_nat_def add.commute eSuc_enat ileI1 plus_1_eq_Suc)
    have 206: "\<And>j. j< nlength l - (Suc 0) \<Longrightarrow>
                     (nsubn (ndropn ((nnth l (Suc 0))-1 ) (ndropn (Suc 0) \<sigma>)) 
                (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) j) 
                (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) (Suc j))) =
                     (nsubn (ndropn (nnth l (Suc 0)) \<sigma>)
                           (nnth l (j+ (Suc 0)) -  (nnth l (Suc 0)))
                           (nnth l ((Suc j)+ (Suc 0)) -  (nnth l (Suc 0))))  "
           using 203 204  2041 by simp
    have 207: "\<And>j. j< nlength l - (Suc 0) \<Longrightarrow> 
                    (nsubn (ndropn (nnth l (Suc 0)) \<sigma>)
                           (nnth l (j+ (Suc 0)) -  (nnth l (Suc 0)))
                           (nnth l ((Suc j)+ (Suc 0)) -  (nnth l (Suc 0)))) = 
                 (nsubn \<sigma> (nnth l (Suc j)) (nnth l (Suc (Suc j))))  "
            using nsubn_ndropn[of _ _ \<sigma>  "(nnth l (Suc 0))"] by simp 
            (metis (no_types, lifting) "197" "202" "204" "205" add.commute add.right_neutral
              bot_nat_0.not_eq_extremum diff_add less_nat_zero_code nidx_gr_first 
              order_less_imp_le plus_1_eq_Suc zero_less_diff)
    have 208: "\<And>j. j< nlength l - (Suc 0) \<Longrightarrow>
                     (nsubn \<sigma> (nnth l (Suc j)) (nnth l (Suc (Suc j)))) \<Turnstile> F"
         using 3 by blast
    have 209: "(\<forall>i. enat i < nlength (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) \<longrightarrow> 
               (nsubn (ndropn ((nnth l (Suc 0))-1 ) (ndropn (Suc 0) \<sigma>)) 
                (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) i) 
                (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) (Suc i)) \<Turnstile> F))" 
           using "202" "206" "207" "208" by presburger
    have 2090: " nlength (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) = \<infinity>" 
        by (metis "194" enat2_cases nlength_eq_enat_nfiniteD)
    have 2091: "(\<forall>i. 
               (nsubn (ndropn ((nnth l (Suc 0))-1 ) (ndropn (Suc 0) \<sigma>)) 
                (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) i) 
                (nnth (nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l)) (Suc i)) \<Turnstile> F))" 
       using 2090 209 
       using enat_ord_code(4) by presburger
    let ?ia = "((nnth l (Suc 0))-1 )"  
    let ?l = "(nmap (\<lambda>x. x - (nnth l (Suc 0))) (ndropn (Suc 0) l))" 
    have 210: "( enat ?ia \<le> nlength \<sigma> - enat (Suc 0) \<and>
               (ntaken ?ia (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! i) \<and>
                (\<not>nfinite ?l \<and>
                 nnth ?l 0 = 0 \<and>
                 nidx ?l \<and>
                 (\<forall>i. enat (nnth ?l i) \<le> nlength \<sigma> - enat (Suc 0) - enat ?ia) \<and>  
                 (\<forall>i. 
                   (nsubn (ndropn ?ia (ndropn (Suc 0) \<sigma>)) 
                      (nnth ?l i) 
                      (nnth ?l (Suc i)) \<Turnstile> F))
               )
              )"
          using "191" "192" "193" "194" "197" "198" "2091" by blast
    have 20: "((ndropn 1 \<sigma>) \<Turnstile> F_next !i schop (omega F))" 
         using 210 "19" by blast
    have 21: "\<sigma> \<Turnstile> F_now ! i"
         by (metis "15" assms(3) ntaken_nfirst state_qpitl_defs state_qpitl_list_defs) 
    have 22: "length F_now = length (map (\<lambda>x.  x schop omega F  ) F_next)"
         by (simp add: assms(4))
    have 23: " (\<sigma> \<Turnstile> big_ior (map2 (\<lambda>x y. x iand next y) F_now (map (\<lambda>x.  x schop omega F  ) F_next))) \<longleftrightarrow>
                 (\<exists>i<length F_now. \<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop omega F  ) F_next ! i)" 
        using  big_ior_map2_defs[of F_now  "(map (\<lambda>x. x schop (omega F)) F_next)" \<sigma> "(\<lambda> x y. x iand next y)"]       
          22 by auto
      have 24: "( \<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop omega F  ) F_next ! i)"
          by (metis "15" "20" "21" "7" assms(4) iand_defs ileI1 nth_map one_eSuc semantics_qpitl.simps(4))        
      show ?thesis using "15" "23" "24" ior_defs by blast
    qed
 have 1000: "?rhs \<Longrightarrow> ?lhs"
  proof -
   assume r: "?rhs"
    have 1001: " (\<sigma> \<Turnstile> big_ior (map2 (\<lambda>x y. x iand next y) F_now (map (\<lambda>x.  x schop omega F  ) F_next)))" 
        using r by simp    
    have 1010: "length F_now = length (map (\<lambda>x.  x schop omega F  ) F_next)"
         by (simp add: assms(4))
    have 1011: "(\<exists>i<length F_now. \<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop omega F  ) F_next ! i)"   
          using 1001 1010  big_ior_map2_defs[of F_now  "(map (\<lambda>x. x schop (omega F)) F_next)" \<sigma> "(\<lambda> x y. x iand next y)"]
        by blast
    obtain i where 1012: "i<length F_now \<and> (\<sigma> \<Turnstile> F_now ! i iand next map (\<lambda>x.  x schop omega F  ) F_next ! i)"
         using 1011 by blast
    have 1013: "(\<sigma> \<Turnstile> F_now ! i)"
         using 1012 by auto
    have 1014: "((ndropn 1 \<sigma>) \<Turnstile> F_next !i schop (omega F))" 
         using 1012 assms(4) by fastforce  
    have 1015: "((ndropn 1 \<sigma>) \<Turnstile> F_next !i schop (omega F)) \<longleftrightarrow> 
              (\<exists>ia. enat ia \<le> nlength \<sigma> - enat (Suc 0) \<and>
          (ntaken ia (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! i) \<and>
          (\<exists>l. \<not> nfinite l \<and>
               nnth l 0 = 0 \<and> nidx l \<and> 
               (\<forall>i. enat (nnth l i) \<le> nlength \<sigma> - enat (Suc 0) - enat ia) \<and> 
               (\<forall>i. nsubn (ndropn ia (ndropn (Suc 0) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> F)))"
        by simp 
    obtain ia where 1016: "enat ia \<le> nlength \<sigma> - enat (Suc 0) \<and>
          (ntaken ia (ndropn (Suc 0) \<sigma>) \<Turnstile> F_next ! i) \<and>
          (\<exists>l. \<not> nfinite l \<and>
               nnth l 0 = 0 \<and> nidx l \<and> 
               (\<forall>i. enat (nnth l i) \<le> nlength \<sigma> - enat (Suc 0) - enat ia) \<and> 
               (\<forall>i. nsubn (ndropn ia (ndropn (Suc 0) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> F))"
         using 1014 1015 by blast
    obtain l where 1017: "\<not> nfinite l \<and>
               nnth l 0 = 0 \<and> nidx l \<and> 
               (\<forall>i. enat (nnth l i) \<le> nlength \<sigma> - enat (Suc 0) - enat ia) \<and> 
               (\<forall>i. nsubn (ndropn ia (ndropn (Suc 0) \<sigma>)) (nnth l i) (nnth l (Suc i)) \<Turnstile> F)"
      using 1016 by blast
   have 1018: "\<not> nfinite \<sigma>" 
        by (metis "1016" infinite_nidx_imp_infinite_interval ndropn_nlength nfinite_ndropn_a)
   have 1022: "nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) 0 = 0" 
        by simp
   have 1023: " 0 < nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) 1 " 
         using zero_enat_def by force
   have 1024: " \<not>nfinite (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l))" 
        by (simp add: "1017")
   have 1025: "nidx (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l))"
        using "1017" nidx_Ncons_shift_alt by blast 
   have 1026: "(\<forall>i. (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) i) \<le> nlength \<sigma>) " 
     by (meson "1018" enat_ile nle_le nlength_eq_enat_nfiniteD)
   have 1028: "(nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) 0) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc 0))) =
                 (nsubn \<sigma> 0 (Suc ia)) " 
          using "1017" zero_enat_def by fastforce
   have 1029: "((nsubn \<sigma> 0 (Suc ia)) \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) \<longleftrightarrow>
                   (\<exists>i<length F_now. nsubn \<sigma> 0 (Suc ia) \<Turnstile> F_now ! i iand next F_next ! i)"
         using  big_ior_map2_defs[of F_now F_next "(nsubn \<sigma> 0 (Suc ia))" "(\<lambda> x y. x iand (next y))"]
           assms by blast 
   have 1030: "nsubn \<sigma> 0 (Suc ia) \<Turnstile> F_now ! i "
         using 1016 
         by (metis "1012" "1013" assms(3) ndropn_0 nsubn_def1 ntaken_nfirst state_qpitl_defs state_qpitl_list_defs)
   have 1031: "nsubn \<sigma> 0 (Suc ia) \<Turnstile>  next F_next ! i"
       using 1016 ndropn_nsubn[of "Suc ia" \<sigma> "Suc 0" 0]  
       by (metis "1022" "1023" "1026" "1028" One_nat_def add.commute diff_zero enat_min_eq enat_ord_simps(1)
           ileI1 leD ndropn_0 nsubn_def1 nsubn_nlength_gr_one ntaken_ndropn_swap one_eSuc one_enat_def 
           plus_1_eq_Suc semantics_qpitl.simps(4) zero_less_iff_neq_zero)
   have 1034: "(nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) 0) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc 0)) \<Turnstile> 
                   (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))))"  
       using "1012" "1028" "1029" "1030" "1031" by auto
   have 1035: "\<And>j. j<nlength l \<Longrightarrow>
                      (nnth ( (nmap (\<lambda>x. x+ Suc ia) l)) j) = nnth l j + Suc ia "
          by simp      
   have 1036: "\<And>j. j<nlength l \<Longrightarrow>
                      (nnth ( (nmap (\<lambda>x. x+ Suc ia) l)) (Suc j)) = nnth l (Suc j) + Suc ia" 
           using Suc_ile_eq by force
   have 1037: "\<And>j. j<nlength l \<Longrightarrow>
                     nnth l j + Suc ia < nnth l (Suc j) + Suc ia"
          by (metis "1017" add_strict_right_mono eSuc_enat ileI1 nidx_expand) 
   have 1038: "\<And>j. j<nlength l \<Longrightarrow>
                     enat ( nnth l (Suc j) + Suc ia) \<le> nlength \<sigma> " 
        using 1017 nidx_expand by simp 
         (meson "1018" enat_ile nle_le nlength_eq_enat_nfiniteD)
   have 1039: "\<And>j. j<nlength l \<Longrightarrow>
                    Suc 0 + (nnth l j + Suc ia) \<le>  nnth l (Suc j) + Suc ia"
             using "1037" by fastforce 
   have 1040: "\<And>j. j<nlength l \<Longrightarrow> 
                      ndropn (Suc 0) (nsubn \<sigma> (nnth l j + Suc ia) (nnth l (Suc j) + Suc ia)) = 
                      nsubn \<sigma> (Suc 0 + (nnth l j + Suc ia)) (nnth l (Suc j) + Suc ia)" 
         using ndropn_nsubn[of _ \<sigma> "Suc 0" ]  using "1038" "1039" by presburger
   have 1041: "\<And>j.  j<nlength l \<Longrightarrow> 
                   (nsubn (ndropn ia (ndropn (Suc 0) \<sigma>)) (nnth l j) (nnth l (Suc j))) =
                   nsubn \<sigma> ((nnth l j) + Suc ia) ((nnth l (Suc j)) + Suc ia)"
           using nsubn_ndropn[of _ _ \<sigma> "Suc ia" ]
             by (metis "1037" One_nat_def add_less_cancel_right ndropn_ndropn plus_1_eq_Suc)
   have 1280: "(nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) 0) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc 0)) \<Turnstile> F)" 
        using "1034" assms(1) by simp 
   have 1281: "(\<forall>i. enat i < nlength ( (nmap (\<lambda>x. x+ Suc ia) l)) \<longrightarrow> 
                 (nsubn \<sigma> (nnth ( (nmap (\<lambda>x. x+ Suc ia) l)) i) (nnth ( (nmap (\<lambda>x. x+ Suc ia) l)) (Suc i)) \<Turnstile> F))"
        using "1017" "1036" "1041"
        by (metis "1035" nlength_nmap) 
   have 1290: "(\<forall>i. enat i < nlength (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) \<longrightarrow> 
          (nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) i) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc i)) \<Turnstile> F)) \<longleftrightarrow>
                 (nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) 0) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc 0)) \<Turnstile> F) \<and>
               (\<forall>i. enat i < nlength ( (nmap (\<lambda>x. x+ Suc ia) l)) \<longrightarrow> 
                 (nsubn \<sigma> (nnth ( (nmap (\<lambda>x. x+ Suc ia) l)) i) (nnth ( (nmap (\<lambda>x. x+ Suc ia) l)) (Suc i)) \<Turnstile> F))" 
          using forall_ncons_split_alt[of "(nmap (\<lambda>x. x+ Suc ia) l)" 
            "\<lambda> x y. (nsubn \<sigma> x y \<Turnstile> F)"  ]  
          by (simp add: "1017" )         
   have 1300: "(\<forall>i. enat i < nlength (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) \<longrightarrow> 
          (nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) i) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc i)) \<Turnstile> F))"     
        using "1280" "1281" "1290" by blast
   have 1340: "nlength (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) = \<infinity>" 
     by (metis "1024" enat2_cases nlength_eq_enat_nfiniteD)
   have 1350: "(\<forall>i.
          (nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) i) (nnth (NCons 0 (nmap (\<lambda>x. x+ Suc ia) l)) (Suc i)) \<Turnstile> F))" 
       using "1300" "1340" enat_ord_code(4) by presburger   
   have 1400: "(\<exists>(l:: nat nellist). 
              \<not>nfinite l \<and> (nnth l 0) = 0 \<and> nidx l \<and> 
              (\<forall>i. (nnth l i) \<le> nlength \<sigma>) \<and>
              (\<forall>i. ( (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> F) ) 
           ) "
    by (meson "1017" "1022" "1025" "1026" "1350" nfinite_NCons nfinite_nmap)
      show ?thesis using 1400 by simp 
     qed
  show ?thesis 
  using "1000" "2" by blast
qed


lemma big_ior_dist_exists: 
 " \<turnstile> (Ex n. big_ior L) ieqv big_ior (map (\<lambda> x. (Ex n. x)) L) " 
apply (auto simp add: big_ior_defs)
apply fastforce
by blast

lemma more_defs [simp]: 
  "(\<sigma> \<Turnstile> more) = ( (nlength \<sigma>) > 0) "
by (auto simp add: empty_d_def more_d_def ileI1 one_eSuc)


lemma state_qpitl_exists_dist_iand: 
 assumes "state_qpitl w"
 shows " \<turnstile>  ((Ex n. w iand next f) ieqv (Ex n. w) iand next (Ex n. f)) "
proof -
 have 1: "\<And> \<sigma>. 
          (\<sigma> \<Turnstile> (Ex n. w iand next f)) \<Longrightarrow> (\<sigma> \<Turnstile> (Ex n. w) iand next (Ex n. f))" 
 using assms by (auto simp add: nlength_upd)
  (metis One_nat_def ndropn_nlength ndropn_upd one_enat_def)
 have 2: "\<And> \<sigma>.  (\<sigma> \<Turnstile> (Ex n. w) iand next (Ex n. f)) \<Longrightarrow> (\<sigma> \<Turnstile> (Ex n. w iand next f))"
  proof -
   fix \<sigma>
   assume a: "(\<sigma> \<Turnstile> (Ex n. w) iand next (Ex n. f))" 
   show "(\<sigma> \<Turnstile> (Ex n. w iand next f))"
   proof -
    have 3: "(\<sigma> \<Turnstile> (Ex n. w)) "
      using a by auto
    obtain l where 4: "nlength l = nlength \<sigma> \<and> (upd \<sigma> n l \<Turnstile> w) " 
      using 3 by auto    
    have 5: " nlength \<sigma> > 0 \<and>(ndropn 1 \<sigma> \<Turnstile> (Ex n. f))"
     using a by auto
    obtain l1 where 6: "nlength l1 = nlength(ndropn 1 \<sigma>) \<and> (upd (ndropn 1 \<sigma>) n l1 \<Turnstile> f) " 
        using 5 by auto
    have 7: "(NNil (nfirst (upd \<sigma> n l)) \<Turnstile> w)"
      using assms "4" state_qpitl_defs by blast
    have 8: "(NNil (nfirst (upd \<sigma> n (nappend (NNil (nfirst l)) l1)))) = 
             (NNil (nfirst (upd \<sigma> n l)))"
      by (metis "4" "5" "6" One_nat_def nappend_NNil ndropn_0 ndropn_Suc_conv_ndropn nfirst_eq_nnth_zero nfirst_upd nlength_NCons nnth_0 zero_enat_def)
    have 9: "(NNil (nfirst (upd \<sigma> n (nappend (NNil (nfirst l)) l1)))) \<Turnstile> w" 
       using "7" "8" by fastforce  
    have 10: "(ndropn 1 (upd \<sigma> n (nappend (NNil (nfirst l)) l1)) \<Turnstile> f)" 
        by (metis "5" "6" One_nat_def ileI1 nappend_NNil ndropn_0 ndropn_Suc_NCons ndropn_Suc_conv_ndropn ndropn_upd nlength_NCons one_eSuc one_enat_def zero_enat_def)
    have 11: " nlength (nappend (NNil (nfirst l)) l1) = nlength \<sigma>" 
       by (metis 5 "6" One_nat_def  nappend_NNil ndropn_0 ndropn_Suc_conv_ndropn nlength_NCons zero_enat_def)
    have 12: "(upd \<sigma> n (nappend (NNil (nfirst l)) l1) \<Turnstile> w iand next f)"
      by (metis "10" "11" "9" a assms iand_defs nlength_upd semantics_qpitl.simps(4) state_qpitl_defs)  
    have 13: "(\<sigma> \<Turnstile> (Ex n. w iand next f)) \<longleftrightarrow>
              (\<exists> l. nlength l = nlength \<sigma> \<and> (upd \<sigma> n l \<Turnstile> w iand next f))"
       by (metis semantics_qpitl.simps(8)) 
    show ?thesis 
    using "11" "12" "13" by blast
   qed
  qed
 show ?thesis unfolding valid_def 
 by (metis "1" "2" ieqv_defs) 
qed


lemma big_ior_state_qpitl_exists_dist_iand: 
 assumes "state_qpitl_list F_now"
        " length F_now = length F_next " 
 shows " \<turnstile> ((Ex n. (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next)))
                 ieqv
                 (big_ior (map2 (\<lambda> x y. x iand (next y)) 
                   (map (\<lambda>x. (Ex n. x)) F_now) 
                   (map (\<lambda>x. (Ex n. x)) F_next)))) "
proof -
 have 1: "\<turnstile> (Ex n. (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) ieqv
            (big_ior (map (\<lambda> x. (Ex n. x)) (map2 (\<lambda> x y. x iand (next y))  F_now F_next)))" 
   using big_ior_dist_exists by auto
 have 2: "\<And>\<sigma>. (\<sigma> \<Turnstile>  ((big_ior (map (\<lambda> x. (Ex n. x)) (map2 (\<lambda> x y. x iand (next y))  F_now F_next))))) =
               (\<sigma> \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand (next y)) 
                   (map (\<lambda>x. (Ex n. x)) F_now) 
                   (map (\<lambda>x. (Ex n. x)) F_next)))) " 
   proof -
    fix \<sigma>
    show "(\<sigma> \<Turnstile> big_ior (map (exists_d n) (map2 (\<lambda>x y. x iand next y) F_now F_next))) = 
          (\<sigma> \<Turnstile> big_ior (map2 (\<lambda>x y. x iand next y) (map (exists_d n) F_now) (map (exists_d n) F_next))) " 
    proof -
     have 3: "(\<sigma> \<Turnstile> big_ior (map (exists_d n) (map2 (\<lambda>x y. x iand next y) F_now F_next))) = 
              (\<exists>i<length (map (exists_d n) (map2 (\<lambda>x y. x iand next y) F_now F_next)). 
                 \<sigma> \<Turnstile> map (exists_d n) (map2 (\<lambda>x y. x iand next y) F_now F_next) ! i)"
      unfolding big_ior_defs  by simp
     have 4: "\<And> i. i< length F_now \<Longrightarrow> 
                    (\<sigma> \<Turnstile> map (exists_d n) (map2 (\<lambda>x y. x iand next y) F_now F_next) ! i) =
                    (\<sigma> \<Turnstile> (Ex n. F_now ! i iand next F_next ! i))"
       by (simp add: assms(2)) 
     have 5: "\<And> i. i< length F_now \<Longrightarrow> 
               (\<sigma> \<Turnstile> (Ex n. F_now ! i iand next F_next ! i)) =
               (\<sigma> \<Turnstile> (Ex n. F_now ! i) iand next (Ex n. F_next ! i))"
       using assms(1) itl_ieq state_qpitl_exists_dist_iand state_qpitl_list_defs by blast 
     have 6: "(\<sigma> \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand (next y)) 
                   (map (\<lambda>x. (Ex n. x)) F_now) 
                   (map (\<lambda>x. (Ex n. x)) F_next)))) = 
              (\<exists>i<length (map2 (\<lambda>x y. x iand next y) (map (exists_d n) F_now) (map (exists_d n) F_next)).
                \<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next y) (map (exists_d n) F_now) (map (exists_d n) F_next) ! i) "
       unfolding big_ior_defs   by simp
     have 7: "\<And> i . i< length F_now \<Longrightarrow> 
               (\<sigma> \<Turnstile> map2 (\<lambda>x y. x iand next y) (map (exists_d n) F_now) (map (exists_d n) F_next) ! i) =
                (\<sigma> \<Turnstile> ((map (exists_d n) F_now) ! i) iand next ((map (exists_d n) F_next) ! i))   " 
       by (simp add: assms(2))
     have 8: "\<And> i . i< length F_now \<Longrightarrow> 
                (\<sigma> \<Turnstile> ((map (exists_d n) F_now) ! i) iand next ((map (exists_d n) F_next) ! i)) =
                (\<sigma> \<Turnstile> (Ex n. F_now ! i) iand next (Ex n. F_next ! i))" 
       using assms(2) by auto
     show ?thesis 
     using "3" "5" "6" assms(2) by auto
    qed
   qed
 show ?thesis 
 using "1" "2" by auto 
qed   
    


lemma GNF_exists_step: 
 assumes " \<turnstile> F ieqv (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))) "  
        " state_qpitl F_e" 
        " state_qpitl_list F_now"
        " length F_now = length F_next" 
 shows  " (\<sigma> \<Turnstile> (Ex n. F) ) \<longleftrightarrow>
          (\<sigma> \<Turnstile> ((Ex n. F_e) iand empty) ior 
                (big_ior (map2 (\<lambda> x y. x iand next y) 
                          (map (\<lambda>x. (Ex n. x)) F_now) 
                          (map (\<lambda>x. (Ex n. x)) F_next))))" 
proof -
  have 1: "(\<sigma> \<Turnstile> (Ex n. F) ) =
           (\<sigma> \<Turnstile> (Ex n. (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next)))))" 
   using assms by simp
  have 2: "(\<sigma> \<Turnstile> (Ex n. (( F_e iand empty ) ior (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next))))) =
           ((\<sigma> \<Turnstile> (Ex n. (F_e iand empty))) \<or> (\<sigma> \<Turnstile> (Ex n. (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next)))))"
    by auto
  have 3: "(\<sigma> \<Turnstile> (Ex n. (F_e iand empty))) = (\<sigma> \<Turnstile> (Ex n. F_e) iand empty)"
    using nlength_upd by auto  
  have 4: "(\<sigma> \<Turnstile> (Ex n. (big_ior (map2 (\<lambda> x y. x iand (next y))  F_now F_next)))) =
           (\<sigma> \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand next y) 
                          (map (\<lambda>x. (Ex n. x)) F_now) 
                          (map (\<lambda>x. (Ex n. x)) F_next))))"
    using assms big_ior_state_qpitl_exists_dist_iand itl_ieq by blast 
  have 5: "(\<sigma> \<Turnstile> ((Ex n. F_e) iand empty) ior 
                (big_ior (map2 (\<lambda> x y. x iand next y) 
                          (map (\<lambda>x. (Ex n. x)) F_now) 
                          (map (\<lambda>x. (Ex n. x)) F_next)))) =
          ((\<sigma> \<Turnstile> (Ex n. F_e) iand empty) \<or>
           (\<sigma> \<Turnstile> (big_ior (map2 (\<lambda> x y. x iand next y) 
                          (map (\<lambda>x. (Ex n. x)) F_now) 
                          (map (\<lambda>x. (Ex n. x)) F_next)))) ) " 
   by auto
  show ?thesis 
  using "1" "2" "3" "4" "5" by blast
qed


lemma GNF_atom_base:
 " \<turnstile> ($n) ieqv (GNF ($n))  "  
unfolding GNF_def  GNF_cons_def big_ior_def
by (auto)
   (metis add_0 ileI1 linorder_not_less not_less_zero order_neq_le_trans plus_1_eSuc(2))


lemma GNF_ifalse_base: 
 " \<turnstile> ifalse ieqv (GNF ifalse) " 
unfolding GNF_def  GNF_cons_def big_ior_def
by auto 

lemma GNF_next_step: 
 " \<turnstile> (next f) ieqv (GNF (next f)) " 
unfolding GNF_def  GNF_cons_def big_ior_def
by (auto )


lemma exists_GNF_nondet: 
 "\<exists> ae a A. (state_qpitl ae) \<and> (state_qpitl_list a) \<and> length a = length A \<and> 
    (\<turnstile> f ieqv (ae iand empty) ior (big_ior (map2 (\<lambda> x y. x iand next y) a A)))" 
proof (induction f)
case false_d
then show ?case 
   proof -
    have f1: "\<turnstile> ifalse ieqv (GNF ifalse)" 
       using GNF_ifalse_base by auto
    have f2: "state_qpitl_list (GNF_state ifalse)"
       unfolding  state_qpitl_list_def by (simp add: inot_d_def itrue_d_def)
    have f3: "length (GNF_state ifalse) = length (GNF_next ifalse)"
       by simp
    show ?thesis 
    by (metis "f1" "f2" "f3" GNF_cons_def GNF_def GNF_empty.simps(1) state_qpitl.simps(1)) 
   qed
next
case (atom_d x)
then show ?case 
   proof -
    have a1: "\<turnstile>($x) ieqv (GNF ($ x))"
       using GNF_atom_base by blast  
    have a2: "state_qpitl_list (GNF_state ($x))"
       unfolding  state_qpitl_list_def by simp
    have a3: "length (GNF_state ($x)) = length (GNF_next ($x))"
     by simp
    show ?thesis 
    by (metis GNF_cons_def GNF_def GNF_empty.simps(2) a1 a2 a3 state_qpitl.simps(2))
   qed
next
case (iimp_d f1 f2)
then show ?case 
  proof -
   obtain ae1 a1 A1 where i1: "state_qpitl ae1 \<and> state_qpitl_list a1 \<and> length a1 = length A1 \<and> 
       (\<turnstile> f1 ieqv ae1 iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) a1 A1))" 
       using iimp_d.IH(1) by blast
   obtain ae2 a2 A2 where i2: "state_qpitl ae2 \<and> state_qpitl_list a2 \<and> length a2 = length A2 \<and>
       (\<turnstile> f2 ieqv ae2 iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) a2 A2))" 
       using iimp_d.IH(2) by blast
   have i3: "\<turnstile> (f1 iimp f2) ieqv ( ( (inot ae1) ior ae2) iand empty) ior
               (big_ior  
                (map2 (\<lambda> x y. x iand (next y)) 
                      ((add_to_now a1 ) @ a2) 
                      ((map (\<lambda>x. (inot x)) (add_to_next A1 )) @ A2) ))"
        using GNF_iimp_step i1 i2 itl_ieq by force
   have i4: "length ((add_to_now a1 ) @ a2) = length ((map (\<lambda>x. (inot x)) (add_to_next A1 )) @ A2)"
     by (simp add: add_to_next_def add_to_now_def i1 i2 length_add_to_now_next_gen) 
   have i5: "state_qpitl_list ((add_to_now a1 ) @ a2)"
     by (simp add: add_to_now_def i1 i2 state_qpitl_foldr_add_to_now_single_inv_a state_qpitl_list_append) 
    show ?thesis 
    by (metis i1 i2 i3 i4 i5 inot_d_def ior_d_def state_qpitl.simps(1) state_qpitl.simps(3))
  qed
next
case (next_d f)
then show ?case 
   proof -
    obtain ae a A where n1: " state_qpitl ae \<and> state_qpitl_list a \<and> length a = length A \<and> 
       (\<turnstile> f ieqv ae iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) a A) )" 
       using next_d.IH by blast
    have n2: "\<turnstile> (next f) ieqv (GNF (next f))"
     using GNF_next_step by blast 
    have n3: "state_qpitl_list (GNF_state (next f))" 
      by (simp add: inot_d_def itrue_d_def state_qpitl_list_defs)
    have n4: "length (GNF_state (next f)) = length (GNF_next (next f)) " 
      by simp
    show ?thesis 
    by (metis GNF_cons_def GNF_def GNF_empty.simps(4) n2 n3 n4 state_qpitl.simps(1))
  qed
next
case (chop_d f1 f2)
then show ?case 
   proof -
    obtain ae1 a1 A1 where c1: "state_qpitl ae1 \<and> state_qpitl_list a1 \<and> length a1 = length A1 \<and> 
      (\<turnstile> f1 ieqv ae1 iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) a1 A1) )"
     using chop_d.IH(1) by blast
    obtain ae2 a2 A2 where c2: "state_qpitl ae2 \<and> state_qpitl_list a2 \<and> length a2 = length A2 \<and> 
      (\<turnstile> f2 ieqv ae2 iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) a2 A2) )"
     using chop_d.IH(2) by blast
    have c3: "\<turnstile> f1 schop f2 ieqv (ae1 iand ae2) iand empty ior
                (big_ior (map2 (\<lambda> x y. x iand next y) 
                         (a1 @ (map (\<lambda>x. x iand ae1) a2)) 
                         ((map (\<lambda>x. x schop f2) A1) @  A2)))"
      using GNF_schop_step[of f1 ae1 a1 A1 f2 ae2 a2 A2] c1 c2 itl_ieq by blast
    have c4: "length (a1 @ (map (\<lambda>x. x iand ae1) a2)) = length ((map (\<lambda>x. x schop f2) A1) @  A2)"
       by (simp add: c1 c2) 
    have c5: "state_qpitl_list (a1 @ (map (\<lambda>x. x iand ae1) a2))"
     by (simp add: c1 c2 state_qpitl_list_append state_qpitl_list_map_alt) 
    have c6: " state_qpitl (ae1 iand ae2)"
         by (simp add: c1 c2 iand_d_def inot_d_def ior_d_def)
    show ?thesis using c1 c2 c3 c4 c5 c6 by blast
  qed
next
case (schopstar_d f)
then show ?case 
   proof -
    obtain ae a A where s1: " state_qpitl ae \<and> state_qpitl_list a \<and> length a = length A \<and> 
      (\<turnstile> f ieqv ae iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) a A) )" 
      using schopstar_d.IH by blast
    have s2: "\<turnstile> schopstar f ieqv (itrue iand empty) ior 
                (big_ior (map2 (\<lambda> x y. x iand next y) a (map (\<lambda>x. x schop (schopstar f)) A)))" 
      using GNF_schopstar_step[of f ae a A] s1 using itl_ieq by blast
    have s3: "length a = length (map (\<lambda>x. x schop (schopstar f)) A)" 
      by (simp add: s1)
    have s4: "state_qpitl itrue"
       by (simp add: inot_d_def itrue_d_def)
    show ?thesis
    by (metis  s1 s2 s3 s4)
   qed
next
case (omega_d f)
then show ?case 
   proof -
    obtain ae a A where o1: " state_qpitl ae \<and> state_qpitl_list a \<and> length a = length A \<and> 
      (\<turnstile> f ieqv ae iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) a A) )" 
      using omega_d.IH by blast
    have o2: "\<turnstile> omega f ieqv (ifalse iand empty) ior 
                (big_ior (map2 (\<lambda> x y. x iand next y) a (map (\<lambda>x. x schop (omega f)) A)))" 
      using GNF_omega_step[of f ae a A] o1 using itl_ieq by blast  
    have o3: "length a = length (map (\<lambda>x. x schop (omega f)) A)" 
      by (simp add: o1)
    show ?thesis 
    using o1 o2 o3 state_qpitl.simps(1) by blast
   qed
next 
case (exists_d n f)
then show ?case
  proof -
   obtain ae a A where e1: " state_qpitl ae \<and> state_qpitl_list a \<and> length a = length A \<and> 
      (\<turnstile> f ieqv ae iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) a A) )" 
    using exists_d.IH by blast
   have e2: "\<turnstile> (Ex n. f) ieqv 
               ((Ex n. ae) iand empty) ior (big_ior (map2 (\<lambda> x y. x iand next y) 
                  (map (\<lambda> x. (Ex n. x)) a) (map (\<lambda> x. (Ex n. x)) A)))"
     using GNF_exists_step[of f ae a A _ n] e1 using itl_ieq  by blast
   have e3: "length (map (\<lambda> x. (Ex n. x)) a) = length (map (\<lambda> x. (Ex n. x)) A) "
      using e1 by simp
   have e4: "state_qpitl (Ex n. ae)"
     by (simp add: e1)
   have e5: "state_qpitl_list (map (\<lambda> x. (Ex n. x)) a)"
     using e1 state_qpitl_list_defs by auto 
   show ?thesis 
    using e1 e2 e3 e4 e5 by blast  
  qed
qed

lemma state_qpitl_GNF_empty: 
 "state_qpitl (GNF_empty f)"
proof (induction f)
case false_d
then show ?case by simp
next
case (atom_d x)
then show ?case by simp
next
case (iimp_d f1 f2)
then show ?case  apply simp  unfolding inot_d_def ior_d_def by simp
next
case (next_d f)
then show ?case by simp
next
case (chop_d f1 f2)
then show ?case apply simp unfolding iand_d_def inot_d_def ior_d_def by simp
next
case (schopstar_d f)
then show ?case apply simp unfolding itrue_d_def inot_d_def by simp
next
case (omega_d f)
then show ?case by simp 
next
case (exists_d x1 f)
then show ?case by simp
qed
 

lemma state_qpitl_map_exists: 
 assumes " state_qpitl_list xs"
 shows   " state_qpitl_list (map (\<lambda> x. Ex n. x) xs) " 
using assms unfolding state_qpitl_list_defs by simp


lemma state_qpitl_list_GNF_state: 
 "state_qpitl_list (GNF_state f) " 
proof (induction f)
case false_d
then show ?case unfolding state_qpitl_list_def apply simp unfolding itrue_d_def inot_d_def by simp
next
case (atom_d x)
then show ?case unfolding state_qpitl_list_def by simp
next
case (iimp_d f1 f2)
then show ?case unfolding state_qpitl_list_def apply simp
using add_to_now_def state_qpitl_foldr_add_to_now_single_inv_a state_qpitl_list_def by auto
next
case (next_d f)
then show ?case unfolding state_qpitl_list_def apply simp unfolding itrue_d_def inot_d_def by simp
next
case (chop_d f1 f2)
then show ?case unfolding state_qpitl_list_def apply simp 
using state_qpitl_GNF_empty state_qpitl_list_def state_qpitl_list_map_alt by force
next
case (schopstar_d f)
then show ?case unfolding state_qpitl_list_def by simp 
next
case (omega_d f)
then show ?case unfolding state_qpitl_list_def by simp 
next
case (exists_d x1 f)
then show ?case unfolding state_qpitl_list_def  apply simp 
using state_qpitl_list_def state_qpitl_map_exists by blast
qed

lemma length_GNF_state_eq_length_GNF_next: 
 "length (GNF_state f) = length (GNF_next f) " 
proof (induction f)
case false_d
then show ?case by simp
next
case (atom_d x)
then show ?case by simp
next
case (iimp_d f1 f2)
then show ?case
by (simp add: add_to_next_def add_to_now_def length_add_to_now_next_gen)
next
case (next_d f)
then show ?case by simp
next
case (chop_d f1 f2)
then show ?case by simp 
next
case (schopstar_d f)
then show ?case by simp
next
case (omega_d f)
then show ?case by simp
next
case (exists_d x1 f)
then show ?case by simp
qed



lemma ieqv_GNF: 
 " \<turnstile> f ieqv GNF f" 
proof (induction f)
case false_d
then show ?case
using GNF_ifalse_base by auto
next
case (atom_d x)
then show ?case using GNF_atom_base by blast
next
case (iimp_d f1 f2)
then show ?case 
  proof -
   have i1: "\<turnstile> f1 ieqv GNF_empty f1 iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) (GNF_state f1) (GNF_next f1)) "
     using iimp_d by (simp add: GNF_cons_def GNF_def)
   have i2: "\<turnstile> f2 ieqv GNF_empty f2 iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) (GNF_state f2) (GNF_next f2))"
     using iimp_d by (simp add: GNF_cons_def GNF_def) 
   have i3: "state_qpitl (GNF_empty f1)"
     by (simp add: state_qpitl_GNF_empty) 
   have i4: "state_qpitl_list (GNF_state f1)" 
     by (simp add: state_qpitl_list_GNF_state)
   have i5: "state_qpitl (GNF_empty f2)"
     by (simp add: state_qpitl_GNF_empty) 
   have i6: "state_qpitl_list (GNF_state f2)" 
     by (simp add: state_qpitl_list_GNF_state)
   have i7: "length (GNF_state f1) = length (GNF_next f1)" 
     by (simp add: length_GNF_state_eq_length_GNF_next)
   have i8: "length (GNF_state f2) = length (GNF_next f2)" 
     by (simp add: length_GNF_state_eq_length_GNF_next)
   show ?thesis using
    GNF_iimp_step[of f1 "GNF_empty f1" "(GNF_state f1)" "(GNF_next f1)" 
                     f2 "GNF_empty f2" "(GNF_state f2)" "(GNF_next f2)"] 
    using GNF_cons_def GNF_def i1 i2 i3 i4 i5 i6 i7 i8 by fastforce  
  qed
next
case (next_d f)
then show ?case using GNF_next_step by blast
next
case (chop_d f1 f2)
then show ?case 
  proof -
   have c1: "\<turnstile> f1 ieqv GNF_empty f1 iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) (GNF_state f1) (GNF_next f1)) "
     using chop_d by (simp add: GNF_cons_def GNF_def)
   have c2: "\<turnstile> f2 ieqv GNF_empty f2 iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) (GNF_state f2) (GNF_next f2))"
     using chop_d by (simp add: GNF_cons_def GNF_def) 
   have c3: "state_qpitl (GNF_empty f1)"
     by (simp add: state_qpitl_GNF_empty) 
   have c4: "state_qpitl_list (GNF_state f1)" 
     by (simp add: state_qpitl_list_GNF_state)
   have c5: "state_qpitl (GNF_empty f2)"
     by (simp add: state_qpitl_GNF_empty) 
   have c6: "state_qpitl_list (GNF_state f2)" 
     by (simp add: state_qpitl_list_GNF_state)
   have c7: "length (GNF_state f1) = length (GNF_next f1)" 
     by (simp add: length_GNF_state_eq_length_GNF_next)
   have c8: "length (GNF_state f2) = length (GNF_next f2)" 
     by (simp add: length_GNF_state_eq_length_GNF_next)
   show ?thesis using
    GNF_schop_step[of f1 "GNF_empty f1" "(GNF_state f1)" "(GNF_next f1)" 
                     f2 "GNF_empty f2" "(GNF_state f2)" "(GNF_next f2)"] 
    unfolding GNF_def GNF_cons_def using c1 c2 c3 c4 c5 c6 c7 c8 by fastforce
   qed
next
case (schopstar_d f)
then show ?case 
   proof -
    have s1: "\<turnstile> f ieqv GNF_empty f iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) (GNF_state f) (GNF_next f)) "
     using schopstar_d by (simp add: GNF_cons_def GNF_def)
    have s2: "state_qpitl (GNF_empty f)"
     by (simp add: state_qpitl_GNF_empty) 
    have s3: "state_qpitl_list (GNF_state f)" 
     by (simp add: state_qpitl_list_GNF_state)
    have s4: "length (GNF_state f) = length (GNF_next f)" 
     by (simp add: length_GNF_state_eq_length_GNF_next)
    show ?thesis using
     GNF_schopstar_step[of f "GNF_empty f" "(GNF_state f)" "(GNF_next f)"] 
      unfolding GNF_def GNF_cons_def using s1 s2 s3 s4 by fastforce 
   qed
next
case (omega_d f)
then show ?case 
  proof -
    have o1: "\<turnstile> f ieqv GNF_empty f iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) (GNF_state f) (GNF_next f)) "
     using omega_d by (simp add: GNF_cons_def GNF_def)
    have o2: "state_qpitl (GNF_empty f)"
     by (simp add: state_qpitl_GNF_empty) 
    have o3: "state_qpitl_list (GNF_state f)" 
     by (simp add: state_qpitl_list_GNF_state)
    have o4: "length (GNF_state f) = length (GNF_next f)" 
     by (simp add: length_GNF_state_eq_length_GNF_next)
    show ?thesis using
     GNF_omega_step[of f "GNF_empty f" "(GNF_state f)" "(GNF_next f)"] 
      unfolding GNF_def GNF_cons_def using o1 o2 o3 o4 by fastforce 
   qed
next
case (exists_d x1 f)
then show ?case 
  proof -
    have e1: "\<turnstile> f ieqv GNF_empty f iand empty ior big_ior (map2 (\<lambda>x y. x iand next y) (GNF_state f) (GNF_next f)) "
     using exists_d by (simp add: GNF_cons_def GNF_def)
    have e2: "state_qpitl (GNF_empty f)"
     by (simp add: state_qpitl_GNF_empty) 
    have e3: "state_qpitl_list (GNF_state f)" 
     by (simp add: state_qpitl_list_GNF_state)
    have e4: "length (GNF_state f) = length (GNF_next f)" 
     by (simp add: length_GNF_state_eq_length_GNF_next)
    show ?thesis using
     GNF_exists_step[of f "GNF_empty f" "(GNF_state f)" "(GNF_next f)"] 
      unfolding GNF_def GNF_cons_def using e1 e2 e3 e4 by fastforce 
   qed
qed


lemma exist_GNF_det: 
 " GNF_prop f"
proof -
 have 1: "\<exists> ae a A. (state_qpitl ae) \<and> (state_qpitl_list a) \<and> length a = length A \<and> 
    (\<turnstile> f ieqv (ae iand empty) ior (big_ior (map2 (\<lambda> x y. x iand next y) a A)))"
   using exists_GNF_nondet by blast  
 obtain ae a A where 2: "(state_qpitl ae) \<and> (state_qpitl_list a) \<and> length a = length A \<and> 
    (\<turnstile> f ieqv (ae iand empty) ior (big_ior (map2 (\<lambda> x y. x iand next y) a A)))"
  using 1 by blast
 have 3: "\<turnstile> (big_ior (map2 (\<lambda> x y. x iand next y) a A)) ieqv
             big_ior (map2 (\<lambda> x y. x iand (next  y)) 
                   (foldr (add_to_now_single) a [itrue])
                   (foldr (add_to_next_single) A [ifalse]))"  
     using "2" det_big_ior_ieqv_nondet_big_ior_a by auto
 have 4: "\<turnstile> f ieqv (ae iand empty) ior
             big_ior ( map2 (\<lambda> x y. x iand (next  y)) 
                   (foldr (add_to_now_single) a [itrue])
                   (foldr (add_to_next_single) A [ifalse]))"
    using "2" "3" by fastforce 
 have 5: "state_qpitl_list  (foldr (add_to_now_single) a [itrue])"
   by (simp add: "2" state_qpitl_foldr_add_to_now_single_inv_a) 
 have 6: "length (foldr (add_to_now_single) a [itrue]) =
          length (foldr (add_to_next_single) A [ifalse])"
     by (simp add: "2" length_add_to_now_next_gen) 
 have 7: "full_system (foldr (add_to_now_single) a [itrue])" 
     using big_ior_foldr_add_to_now_single_inv_a full_system_def mutex_foldr_add_to_now_single_inv_a by blast
 show ?thesis unfolding GNF_prop_def GNF_cons_def
 using "2" "4" "5" "6" "7" by blast
qed

lemma state_qpitl_list_GNF_det_state: 
 "state_qpitl_list  (foldr (add_to_now_single) (GNF_state f) [itrue])" 
by (simp add: state_qpitl_foldr_add_to_now_single_inv_a state_qpitl_list_GNF_state)


lemma full_system_GNF_det_state: 
 "full_system (foldr (add_to_now_single) (GNF_state f) [itrue]) " 
using big_ior_foldr_add_to_now_single_inv_a full_system_def 
mutex_foldr_add_to_now_single_inv_a by auto

lemma ieqv_GNF_det: 
 "\<turnstile> f ieqv GNF_det f" 
using ieqv_GNF[of f] 
unfolding GNF_def GNF_det_def GNF_cons_def 
using det_big_ior_ieqv_nondet_big_ior_a length_GNF_state_eq_length_GNF_next by auto

end
