(*  Title     : A Shallow Encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

section \<open>Projection operator\<close>   

theory Projection
imports Omega
begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="
  cvc4 z3 e spass vampire
"]
*)
text \<open>
This theory introduces the projection operator fproj \cite{moszkowski95:_compos}. 
The projection operator is defined and we prove the soundness of the rules and axiom system.
We also provide the infinite projection operator oproj which was defined in \cite{moszkowski17}.

 
\<close>
(*
declare [[show_types]]


declare [[show_consts]]
*)

subsection \<open>Definitions\<close>

primcorec pfilt ::  "'a intervals \<Rightarrow> nat nellist \<Rightarrow> 'a intervals"
 where 
  "pfilt \<sigma> ls = (case ls of (NNil n) \<Rightarrow> (NNil (nnth \<sigma> n)) |
                       (NCons n ls1) \<Rightarrow> (NCons (nnth \<sigma> n) (pfilt \<sigma> ls1)))"

simps_of_case pfilt_code [code, simp, nitpick_simp]: pfilt.code

primcorec lsum :: "'a intervals intervals \<Rightarrow> nat \<Rightarrow> nat nellist"
 where
  "lsum nells a = (case nells of (NNil nell) \<Rightarrow> (NNil (a+(the_enat (nlength nell)))) |
                     (NCons nell nells1) \<Rightarrow> (NCons (a+(the_enat (nlength nell))) 
                                               (lsum nells1 (a+(the_enat (nlength nell)))))) "

simps_of_case lsum_code [code, simp, nitpick_simp]: lsum.code
 
definition addzero :: " nat nellist \<Rightarrow>  nat nellist"
 where "addzero ls = (if nlength ls = 0 then 
        (if nfirst ls = 0 then ls else (NCons 0  ls)) else (NCons 0  ls))"

definition powerinterval :: "('a::world) formula \<Rightarrow> 'a intervals \<Rightarrow> nat nellist \<Rightarrow> bool"
 where "powerinterval F \<sigma> l = 
        (\<forall> (i::nat). (enat i)<nlength l \<longrightarrow> ((nsubn \<sigma> (nnth l i) (nnth l (Suc i)) ) \<Turnstile> F))"

definition cppl :: "('a::world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a intervals \<Rightarrow> nat nellist"
 where "cppl f g \<sigma> = (\<some> ls.  nidx ls \<and> (nnth ls 0) = 0 \<and> powerinterval f \<sigma> ls \<and> 
                          ((nfinite ls \<and> nfinite \<sigma> \<and>  (nlast ls) = the_enat(nlength \<sigma>) ) \<or>
                           (\<not> nfinite ls \<and> \<not>nfinite \<sigma> ))  \<and>
                          ((pfilt \<sigma> ls) \<Turnstile> g))"

primcorec lcppl :: " ('a ::world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a intervals \<Rightarrow> nat nellist \<Rightarrow> nat intervals intervals"
 where  "lcppl  f g \<sigma> ls = 
         (case ls of (NNil x) \<Rightarrow> (NNil (nmap (\<lambda>l. l+x) (cppl f g (nsubn \<sigma> x x)))) |
                (NCons x ls1) \<Rightarrow>
              (case ls1 of (NNil y) \<Rightarrow> (NNil (nmap (\<lambda>l. l+x) (cppl f g (nsubn \<sigma> x y)))) |
                      (NCons y ls2) \<Rightarrow> (NCons (nmap (\<lambda>l. l+x) (cppl f g (nsubn \<sigma> x y))) (lcppl f g \<sigma> ls1)))) "

simps_of_case lcppl_code [code, simp, nitpick_simp]: lcppl.code

definition fprojection_d :: "('a:: world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
where "fprojection_d F G \<equiv> \<lambda>\<sigma>. (\<exists> ls.  nidx ls  \<and> (nnth ls 0) = 0 \<and> nfinite ls \<and> nfinite \<sigma>  \<and>
                                   (nlast ls) = the_enat(nlength \<sigma>) \<and> 
                                  powerinterval F \<sigma> ls \<and> ((pfilt \<sigma> ls) \<Turnstile> G)
                                )"


definition oprojection_d :: "('a:: world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
where "oprojection_d F G \<equiv> \<lambda>\<sigma>. (\<exists> ls.  nidx ls  \<and> (nnth ls 0) =0 \<and> \<not>nfinite ls \<and> \<not> nfinite \<sigma>  \<and>
                                  powerinterval F \<sigma> ls \<and> ((pfilt \<sigma> ls) \<Turnstile> G)
                                )"

syntax
 "_fprojection_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ fproj _)" [84,84] 83)
 "_oprojection_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ oproj _)" [84,84] 83)

syntax (ASCII)
 "_fprojection_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ fproj _)" [84,84] 83)
 "_oprojection_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ oproj _)" [84,84] 83)

translations
 "_fprojection_d"    \<rightleftharpoons> "CONST fprojection_d"
 "_oprojection_d"    \<rightleftharpoons> "CONST oprojection_d"


definition ufprojection_d :: "('a:: world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
where "ufprojection_d F G \<equiv> LIFT(\<not>(F fproj (\<not> G)))"


definition uoprojection_d :: "('a:: world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
where "uoprojection_d F G \<equiv> LIFT(\<not>(F oproj (\<not> G)))"

syntax
 "_ufprojection_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ ufproj _)" [84,84] 83)
syntax
 "_uoprojection_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ uoproj _)" [84,84] 83)


syntax (ASCII)
 "_ufprojection_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ ufproj _)" [84,84] 83)
 "_uoprojection_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ uoproj _)" [84,84] 83)

translations
 "_ufprojection_d"    \<rightleftharpoons> "CONST ufprojection_d"
 "_uoprojection_d"    \<rightleftharpoons> "CONST uoprojection_d"

definition fdp_d :: "('a:: world) formula \<Rightarrow> 'a formula"
where "fdp_d F \<equiv> LIFT(#True  fproj  F)"

definition fbp_d :: "('a:: world) formula \<Rightarrow> 'a formula"
where "fbp_d F \<equiv> LIFT(#True ufproj F)"

definition odp_d :: "('a:: world) formula \<Rightarrow> 'a formula"
where "odp_d F \<equiv> LIFT(#True oproj F)"

definition obp_d :: "('a:: world) formula \<Rightarrow> 'a formula"
where "obp_d F \<equiv> LIFT(#True uoproj F)"

syntax
 "_fdp_d"     :: "lift \<Rightarrow> lift"        ("(fdp _)" [88] 87)
 "_fbp_d"     :: "lift \<Rightarrow> lift"        ("(fbp _)" [88] 87)
 "_odp_d"     :: "lift \<Rightarrow> lift"        ("(odp _)" [88] 87)
 "_obp_d"     :: "lift \<Rightarrow> lift"        ("(obp _)" [88] 87)

syntax (ASCII)
 "_fdp_d"     :: "lift \<Rightarrow> lift"        ("(fdp _)" [88] 87)
 "_fbp_d"     :: "lift \<Rightarrow> lift"        ("(fbp _)" [88] 87)
 "_odp_d"     :: "lift \<Rightarrow> lift"        ("(odp _)" [88] 87)
 "_obp_d"     :: "lift \<Rightarrow> lift"        ("(obp _)" [88] 87)

translations
 "_fdp_d"    \<rightleftharpoons> "CONST fdp_d"
 "_fbp_d"    \<rightleftharpoons> "CONST fbp_d"
 "_odp_d"    \<rightleftharpoons> "CONST odp_d"
 "_obp_d"    \<rightleftharpoons> "CONST obp_d"


abbreviation "all_nfinite nells \<equiv> (\<forall>nell \<in> nset nells. nfinite nell) " 

abbreviation "all_gr_zero nells \<equiv> (\<forall>nell \<in> nset nells. 0<nlength nell) " 



subsection \<open>Lemmas\<close>

subsubsection \<open>Misc\<close>

lemma all_nfinite_nnth_a: 
assumes "\<forall>i \<le>nlength nells. nfinite (nnth nells i)"
shows "all_nfinite nells " 
using assms 
by (metis in_nset_conv_nnth)

lemma all_nfinite_nnth_b: 
assumes "all_nfinite nells "  
shows "\<forall>i \<le>nlength nells. nfinite (nnth nells i)"
using assms 
by (meson in_nset_conv_nnth)

lemma all_gr_zero_nnth_a: 
assumes "\<forall>i \<le>nlength nells. 0 < nlength (nnth nells i)"
shows "all_gr_zero nells " 
using assms 
by (metis in_nset_conv_nnth)

lemma all_gr_zero_nnth_b: 
assumes "all_gr_zero nells "  
shows "\<forall>i \<le>nlength nells. 0 < nlength (nnth nells i)"
using assms 
by (meson in_nset_conv_nnth)

lemma all_nfinite_nnth_ntaken:
 assumes "all_nfinite nells " 
         "n \<le> nlength nells "
 shows   "all_nfinite (ntaken n nells) "
using assms 
by (meson nset_ntaken subsetD)

lemma all_nfinite_nnth_ndropn:
 assumes "all_nfinite nells " 
         "n \<le> nlength nells "
 shows   "all_nfinite (ndropn n nells) "
using assms 
by (metis nset_ndropn subsetD)

lemma all_nfinite_nnth_nsubn:
 assumes "all_nfinite nells " 
         "n \<le> nlength nells " 
         "k \<le> n "
 shows   "all_nfinite (nsubn nells k n) "
using assms 
by (metis in_mono nset_ndropn nset_ntaken nsubn_def1)

lemma all_gr_zero_nnth_ntaken:
 assumes "all_gr_zero nells " 
         "n \<le> nlength nells "
 shows   "all_gr_zero (ntaken n nells) "
using assms 
by (meson nset_ntaken subsetD)

lemma all_gr_zero_nnth_ndropn:
 assumes "all_gr_zero nells " 
         "n \<le> nlength nells "
 shows   "all_gr_zero (ndropn n nells) "
using assms 
by (metis nset_ndropn subsetD)

lemma all_gr_zero_nnth_nsubn:
 assumes "all_gr_zero nells " 
         "n \<le> nlength nells " 
         "k \<le> n "
 shows   "all_gr_zero (nsubn nells k n) "
using assms 
by (metis in_mono nset_ndropn nset_ntaken nsubn_def1)



subsubsection \<open>pfilt Lemmas\<close>

lemma pfilt_nmap:
 " pfilt nell ls = nmap (\<lambda>x. (nnth nell x)) ls " 
proof (coinduction arbitrary: nell ls)
case (Eq_nellist nell1) 
then show ?case by simp  (metis nellist.case_eq_if) 
qed


lemma pfilt_nlength:
 "nlength(pfilt nell ls) = nlength ls"
by (simp add: pfilt_nmap)


lemma pfilt_nnth:
 assumes " i \<le> nlength (pfilt nell ls) "
 shows   " (nnth (pfilt nell ls) i) = (nnth nell (nnth ls i)) "
using assms
by (simp add: pfilt_nmap)

lemma pfilt_expand: 
 " (nell1 = (pfilt nell ls)) =
   ( nlength nell1 = nlength ls \<and>
     (\<forall> (i::nat). i \<le> nlength nell1\<longrightarrow> (nnth nell1 i) = (nnth nell (nnth ls i))))"
by (metis nellist_eq_nnth_eq pfilt_nlength pfilt_nnth)


lemma pfilt_nfuse:
   " pfilt nell (nfuse ls1 ls2) = (nfuse (pfilt nell ls1) (pfilt nell ls2))"
using pfilt_nmap[of nell "(nfuse ls1 ls2)"] pfilt_nmap[of nell ls1] pfilt_nmap[of nell ls2] 
by (simp add: nmap_nfuse) 

lemma nfuse_pfilt_nlength:
shows  "nlength (pfilt (nfuse (ntaken (nnth ls n) nell) (ndropn (nnth ls n) nell)) ls) =
        nlength (nfuse (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
               (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )))"
by (metis add.right_neutral linorder_le_cases ndropn_all nfuse_nlength nfuse_ntaken_ndropn_nlength 
    nlength_NNil nlength_nmap ntaken_all pfilt_nlength)
 
lemma nfuse_pfilt_nnth_a:
 assumes "nidx  ls"
         "(nnth ls 0) = 0"
         " (nlast ls) = the_enat(nlength nell)"
         "nfinite nell" 
         "nfinite ls" 
         "i \<le> nlength ls"
         "n \<le> nlength ls"
 shows   " nnth (pfilt (nfuse (ntaken (nnth ls n) nell) (ndropn (nnth ls n) nell)) ls) i =
           nnth nell (nnth ls i)"
using assms 
by (metis linorder_le_cases ndropn_all nfinite_conv_nlength_enat nfuse_ntaken_ndropn ntaken_all 
    pfilt_nlength pfilt_nnth)


lemma nfuse_pfilt_nnth_a_alt:
 assumes "nidx  ls"
         "(nnth ls 0) = 0"
         "\<not>nfinite nell" 
         "\<not>nfinite ls" 
         "i \<le> nlength ls"
         "n \<le> nlength ls"
 shows   " nnth (pfilt (nfuse (ntaken (nnth ls n) nell) (ndropn (nnth ls n) nell)) ls) i =
           nnth nell (nnth ls i)"
using assms
by (metis  linorder_le_cases nfinite_ntaken nfuse_ntaken_ndropn ntaken_all pfilt_nlength pfilt_nnth)


lemma nfuse_pfilt_nnth_b:
 assumes "nidx ls"
         "nnth ls 0 = 0"
         "(nlast ls) = the_enat(nlength nell)"
         "nfinite nell" 
         "nfinite ls" 
         "i \<le> nlength ls"
         "n \<le> nlength ls"
 shows   "nnth (nfuse (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
                      (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x - (nnth ls n)) (ndropn n ls) ))) i =
          nnth nell (nnth ls i)"
proof -
 have 1: "i \<le> nlength(nfuse (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
                  (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x - (nnth ls n)) (ndropn n ls) ))) "
    using assms 
    by (metis nfuse_pfilt_nlength pfilt_nlength) 
 have 2: " nlast(pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) = nnth nell (nnth ls n) "
  using assms pfilt_nnth[of _ "(ntaken (nnth ls n) nell)" "(ntaken n ls)" ]
      by (simp add: nlength_eq_enat_nfiniteD nnth_nlast ntaken_nnth pfilt_nlength) 
 have 3: "nfirst(pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) =
          nnth (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) 0"
      by (metis ndropn_0 ndropn_nfirst)
 have 4: "nnth (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) 0 =
          nnth (ndropn (nnth ls n) nell) (nnth (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ) 0)"
         using pfilt_nnth 
         by (metis i0_lb zero_enat_def)
 have 5: "nnth (ndropn (nnth ls n) nell) (nnth (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ) 0) =
          nnth  (ndropn (nnth ls n) nell) (nnth ls (n+0) - (nnth ls n))"
         using zero_enat_def by force  
 have 6: "nnth  (ndropn (nnth ls n) nell) (nnth ls (n+0) - (nnth ls n)) = 
          nnth  (ndropn (nnth ls n) nell) 0 "
    by simp 
 have 7: "nnth  (ndropn (nnth ls n) nell) 0 = nnth nell (nnth ls n) "
    using assms by simp
 have 8: "nlast(pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) =
          nfirst(pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) "
   using "2" "4" "5" "7" "3" "6" by presburger
 have 10: "nnth (nfuse (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
                  (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ))) i =
          (if i \<le> nlength (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
           then nnth (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) i
           else nnth (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) 
                    (i - (the_enat (nlength (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)))))) "
    using "1" "8" by (meson nfuse_nnth not_le_imp_less)
 have 11: "nlength (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) = n "
   using assms by (simp add: pfilt_nlength)
 have 12: "i\<le>n \<longrightarrow> nnth (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) i = 
                   nnth (ntaken (nnth ls n) nell) (nnth (ntaken n ls ) i)"
    by (simp add: "11" pfilt_nnth)
 have 13: "i\<le>n \<longrightarrow> nnth (ntaken (nnth ls n) nell) (nnth (ntaken n ls ) i) =
                   nnth (ntaken (nnth ls n) nell) (nnth ls i) "
       by (simp add: ntaken_nnth)
 have 15: " i\<le>n \<longrightarrow> nnth (ntaken (nnth ls n) nell) (nnth ls i) = nnth nell (nnth ls i)"
    using assms by (simp add:  nidx_less_eq ntaken_nnth)
 have 16: "nnth (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) 
                    (i - (the_enat (nlength (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)))))  =
           nnth (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) 
                    (i - n)"
    by (simp add: "11") 
 have 17: "nnth (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) (i - n) =
           nnth (ndropn (nnth ls n) nell) (nnth (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ) (i -n)) "
    using assms pfilt_nnth[of "i-n" "(ndropn (nnth ls n) nell)" 
                               "(nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )" ]
      by (metis enat_minus_mono1 idiff_enat_enat ndropn_nlength nlength_nmap pfilt_nlength) 
 have 18: "i> n \<longrightarrow> nnth (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ) (i -n) =
           nnth ls (n+(i-n)) - (nnth ls n)"
    using assms 
    by (metis linorder_le_cases ndropn_nnth nfinite_ntaken nlast_nmap nlength_nmap nnth_nmap ntaken_all
        ntaken_nlast)
 have 19: "i>n \<longrightarrow> nnth (ndropn (nnth ls n) nell) (nnth (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ) (i -n)) 
                   = nnth (ndropn (nnth ls n) nell) ((nnth ls i) - (nnth ls n))" 
    by (simp add: "18")    
 have 20 : "i>n \<longrightarrow> nnth (ndropn (nnth ls n) nell) ((nnth ls i) - (nnth ls n)) = 
                    nnth nell ((nnth ls n) + ((nnth ls i) - (nnth ls n)))"
    using assms ndropn_nnth by blast
 have 22: "i>n \<longrightarrow> (nnth ls n) + ((nnth ls i) - (nnth ls n)) =  (nnth ls i)"
     using assms by (simp add: nidx_less_eq)
 have 23: "i>n \<longrightarrow> nnth nell ((nnth ls n) + ((nnth ls i) - (nnth ls n))) = nnth nell (nnth ls i) "
    by (simp add: "22")
 from 10 show ?thesis 
 by (metis "11" "12" "13" "15" "16" "17" "19" "20" "23" enat_ord_simps(1) not_le_imp_less)
qed


lemma nfuse_pfilt_nnth_b_alt:
 assumes "nidx ls"
         "nnth ls 0 = 0"
         "\<not>nfinite nell" 
         "\<not>nfinite ls " 
         "i \<le> nlength ls"
         "n \<le> nlength ls"
 shows   "nnth (nfuse (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
                      (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x - (nnth ls n)) (ndropn n ls) ))) i =
          nnth nell (nnth ls i)"
proof -
 have 1: "i \<le> nlength(nfuse (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
                  (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x - (nnth ls n)) (ndropn n ls) ))) "
    using assms 
    by (metis nfuse_pfilt_nlength pfilt_nlength) 
 have 2: " nlast(pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) = nnth nell (nnth ls n) "
  using assms pfilt_nnth[of _ "(ntaken (nnth ls n) nell)" "(ntaken n ls)" ]
      by (simp add: nlength_eq_enat_nfiniteD nnth_nlast ntaken_nnth pfilt_nlength) 
 have 3: "nfirst(pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) =
          nnth (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) 0"
      by (metis ndropn_0 ndropn_nfirst)
 have 4: "nnth (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) 0 =
          nnth (ndropn (nnth ls n) nell) (nnth (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ) 0)"
         using pfilt_nnth 
         by (metis i0_lb zero_enat_def)
 have 5: "nnth (ndropn (nnth ls n) nell) (nnth (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ) 0) =
          nnth  (ndropn (nnth ls n) nell) (nnth ls (n+0) - (nnth ls n))"
         using zero_enat_def by force  
 have 6: "nnth  (ndropn (nnth ls n) nell) (nnth ls (n+0) - (nnth ls n)) = 
          nnth  (ndropn (nnth ls n) nell) 0 "
    by simp 
 have 7: "nnth  (ndropn (nnth ls n) nell) 0 = nnth nell (nnth ls n) "
    using assms by simp
 have 8: "nlast(pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) =
          nfirst(pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) "
   using "2" "4" "5" "7" "3" "6" by presburger
 have 10: "nnth (nfuse (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
                  (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ))) i =
          (if i \<le> nlength (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
           then nnth (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) i
           else nnth (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) 
                    (i - (the_enat (nlength (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)))))) "
    using "1" "8" by (meson nfuse_nnth not_le_imp_less)
 have 11: "nlength (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) = n "
   using assms by (simp add: pfilt_nlength)
 have 12: "i\<le>n \<longrightarrow> nnth (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) i = 
                   nnth (ntaken (nnth ls n) nell) (nnth (ntaken n ls ) i)"
    by (simp add: "11" pfilt_nnth)
 have 13: "i\<le>n \<longrightarrow> nnth (ntaken (nnth ls n) nell) (nnth (ntaken n ls ) i) =
                   nnth (ntaken (nnth ls n) nell) (nnth ls i) "
       by (simp add: ntaken_nnth)
 have 15: " i\<le>n \<longrightarrow> nnth (ntaken (nnth ls n) nell) (nnth ls i) = nnth nell (nnth ls i)"
     using assms by (simp add:  nidx_less_eq ntaken_nnth)
 have 16: "nnth (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) 
                    (i - (the_enat (nlength (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)))))  =
           nnth (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) 
                    (i - n)"
    by (simp add: "11") 
 have 17: "nnth (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )) (i - n) =
           nnth (ndropn (nnth ls n) nell) (nnth (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ) (i -n)) "
    using assms pfilt_nnth[of "i-n" "(ndropn (nnth ls n) nell)" 
                              "(nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) )" ]
      by (metis enat_minus_mono1 idiff_enat_enat ndropn_nlength nlength_nmap pfilt_nlength) 
 have 18: "i> n \<longrightarrow> nnth (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ) (i -n) =
           nnth ls (n+(i-n)) - (nnth ls n)"
    using assms 
    by (metis linorder_le_cases ndropn_nnth nfinite_ntaken nlast_nmap nlength_nmap nnth_nmap ntaken_all 
        ntaken_nlast)
 have 19: "i>n \<longrightarrow> nnth (ndropn (nnth ls n) nell) (nnth (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ) (i -n)) 
                   = nnth (ndropn (nnth ls n) nell) ((nnth ls i) - (nnth ls n))" 
    by (simp add: "18")    
 have 20 : "i>n \<longrightarrow> nnth (ndropn (nnth ls n) nell) ((nnth ls i) - (nnth ls n)) = 
                    nnth nell ((nnth ls n) + ((nnth ls i) - (nnth ls n)))"
    using assms ndropn_nnth by blast
 have 22: "i>n \<longrightarrow> (nnth ls n) + ((nnth ls i) - (nnth ls n)) =  (nnth ls i)"
     using assms by (simp add: nidx_less_eq)
 have 23: "i>n \<longrightarrow> nnth nell ((nnth ls n) + ((nnth ls i) - (nnth ls n))) = nnth nell (nnth ls i) "
    by (simp add: "22")
 from 10 show ?thesis 
 by (metis "11" "12" "13" "15" "16" "17" "19" "20" "23" enat_ord_simps(1) not_le_imp_less)
qed


lemma nfuse_pfilt_nnth:
assumes "nidx  ls"
        "nnth ls 0 = 0"
        "(nlast ls)  = the_enat(nlength nell)"
        "nfinite nell" 
        "nfinite ls "
        "i \<le> nlength ls"
        "n \<le> nlength ls"
shows  "nnth (pfilt (nfuse (ntaken (nnth ls n) nell) (ndropn (nnth ls n) nell)) ls) i =
        nnth (nfuse (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
                  (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ))) i"
using assms nfuse_pfilt_nnth_a[of "ls" "nell" "i" "n"]   
            nfuse_pfilt_nnth_b[of "ls" "nell" "i" "n"] 
by simp


lemma nfuse_pfilt_nnth_alt:
assumes "nidx  ls"
        "nnth ls 0 = 0"
        "\<not>nfinite nell" 
        "\<not>nfinite ls "
        "i \<le> nlength ls"
        "n \<le> nlength ls"
shows  "nnth (pfilt (nfuse (ntaken (nnth ls n) nell) (ndropn (nnth ls n) nell)) ls) i =
        nnth (nfuse (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
                  (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ))) i"
using assms nfuse_pfilt_nnth_a_alt[of "ls" "nell" "i" "n"]   
            nfuse_pfilt_nnth_b_alt[of "ls" "nell" "i" "n"]
by simp 

lemma nfuse_pfilt:
assumes "nidx  ls"
        "nnth ls 0 = 0"
        "(nlast ls) = the_enat(nlength nell)"
        "nfinite nell" 
        "nfinite ls" 
        "n \<le> nlength ls"
shows  "pfilt (nfuse (ntaken (nnth ls n) nell) (ndropn (nnth ls n) nell)) ls =
        nfuse (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
             (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ))"
using assms nfuse_pfilt_nlength nfuse_pfilt_nnth 
nellist_eq_nnth_eq by (metis pfilt_nlength)

lemma nfuse_pfilt_alt:
assumes "nidx  ls"
        "nnth ls 0 = 0"
        "\<not>nfinite nell"
        "\<not>nfinite ls "  
        "n \<le> nlength ls"
shows  "pfilt (nfuse (ntaken (nnth ls n) nell) (ndropn (nnth ls n) nell)) ls =
        nfuse (pfilt (ntaken (nnth ls n) nell) (ntaken n ls)) 
             (pfilt (ndropn (nnth ls n) nell) (nmap (\<lambda>x. x- (nnth ls n)) (ndropn n ls) ))"
using assms nfuse_pfilt_nlength nfuse_pfilt_nnth_alt 
nellist_eq_nnth_eq by (metis pfilt_nlength)

lemma nfuse_pfilt_a:
 assumes "nidx  ls1"
         "nfinite ls1" 
         "nnth ls1 0 = 0" 
         "nidx ls2"
         "nfinite ls2" 
         "nnth ls2 0 = nlast ls1"  
         "(nlast ls2) = the_enat(nlength nell)"
         "nfinite nell" 
 shows   " pfilt (nfuse (ntaken (nlast ls1) nell) (ndropn (nfirst ls2) nell)) (nfuse ls1 ls2) =
           nfuse (pfilt (ntaken (nlast ls1) nell) ls1) 
                (pfilt (ndropn (nfirst ls2) nell) (nmap (\<lambda>x. x- (nfirst ls2)) ls2)) "
proof -
 have 0: "\<exists>ll. (enat ll) = nlength ls1" 
    using assms(2) nfinite_nlength_enat by fastforce
 obtain ll where 01: " (enat ll) = nlength ls1" 
      using 0 by auto
 have 1: "nlast ls1 = nfirst ls2" 
  using assms by (metis nlast_NNil ntaken_0 ntaken_nlast)
 have 2: "nidx  (nfuse ls1 ls2)"
   using assms nidx_nfuse by blast
 have 20: "nnth (nfuse ls1 ls2) 0 = 0 " 
   by (metis "1" assms(3) nfuse_nnth zero_enat_def zero_le)
 have 3: "nlast(nfuse ls1 ls2) = the_enat(nlength nell)"
   using assms 
   by (simp add: "1" nlast_nfuse)
 have 4: "ll \<le> nlength (nfuse ls1 ls2) " 
     by (simp add: "01" nfuse_nlength)
 have 5: "pfilt (nfuse (ntaken (nnth (nfuse ls1 ls2) ll) nell)
                     (ndropn (nnth (nfuse ls1 ls2) ll) nell))
               (nfuse ls1 ls2)
          =
         nfuse (pfilt (ntaken (nnth (nfuse ls1 ls2) ll) nell) (ntaken ll (nfuse ls1 ls2))) 
              (pfilt (ndropn (nnth (nfuse ls1 ls2) ll) nell) 
                    (nmap (\<lambda> x. x- (nnth (nfuse ls1 ls2) ll)) (ndropn ll (nfuse ls1 ls2))))"
   using "2" "3" "4" nfuse_pfilt[of "(nfuse ls1 ls2)" nell ll] 
   using "20" assms nfuse_nfinite by blast
 have 6: "pfilt (nfuse (ntaken (nlast ls1) nell) (ndropn (nfirst ls2) nell)) (nfuse ls1 ls2) =
          pfilt (nfuse (ntaken (nnth (nfuse ls1 ls2) ll) nell) 
                     (ndropn (nnth (nfuse ls1 ls2) ll) nell)) 
               (nfuse ls1 ls2)" 
    using "1" "4" 
    by (metis "01" assms(2) ntaken_nfuse ntaken_nlast the_enat.simps)
 have 7: "(pfilt (ntaken (nlast ls1) nell) ls1) =
          (pfilt (ntaken (nnth (nfuse ls1 ls2) ll) nell) (ntaken ll (nfuse ls1 ls2)))" 
    using "1" "4" 
    by (metis "01" assms(2) ntaken_nfuse ntaken_nlast the_enat.simps)
 have 8: "(pfilt (ndropn (nfirst ls2) nell) (nmap (\<lambda>x. x- (nfirst ls2)) ls2)) =
          (pfilt (ndropn (nnth (nfuse ls1 ls2) ll) nell) 
                (nmap (\<lambda>x. x- (nnth (nfuse ls1 ls2) ll)) (ndropn ll (nfuse ls1 ls2)) ))"
    using "1" "4" 
    by (metis "01" ndropn_nfirst ndropn_nfuse nfinite_conv_nlength_enat the_enat.simps)
 show ?thesis using "5" "6" "7" "8" by auto
qed  


lemma nfuse_pfilt_a_alt:
 assumes "nidx  ls1"
         "nfinite ls1" 
         "nnth ls1 0 = 0" 
         "nidx ls2"
         "\<not>nfinite ls2" 
         "nnth ls2 0 = nlast ls1"  
         "\<not>nfinite nell" 
 shows   " pfilt (nfuse (ntaken (nlast ls1) nell) (ndropn (nfirst ls2) nell)) (nfuse ls1 ls2) =
           nfuse (pfilt (ntaken (nlast ls1) nell) ls1) 
                (pfilt (ndropn (nfirst ls2) nell) (nmap (\<lambda>x. x- (nfirst ls2)) ls2)) "
proof -
 have 0: "\<exists>ll. (enat ll) = nlength ls1" 
    using assms(2) nfinite_nlength_enat by fastforce
 obtain ll where 01: " (enat ll) = nlength ls1" 
      using 0 by auto
 have 1: "nlast ls1 = nfirst ls2" 
  using assms by (metis nlast_NNil ntaken_0 ntaken_nlast)
 have 2: "nidx  (nfuse ls1 ls2)"
   using assms nidx_nfuse by blast
 have 20: "nnth (nfuse ls1 ls2) 0 = 0 " 
   by (metis "1" assms(3) nfuse_nnth zero_enat_def zero_le)
 have 4: "ll \<le> nlength (nfuse ls1 ls2) " 
     by (simp add: "01" nfuse_nlength)
 have 5: "pfilt (nfuse (ntaken (nnth (nfuse ls1 ls2) ll) nell)
                     (ndropn (nnth (nfuse ls1 ls2) ll) nell))
               (nfuse ls1 ls2)
          =
         nfuse (pfilt (ntaken (nnth (nfuse ls1 ls2) ll) nell) (ntaken ll (nfuse ls1 ls2))) 
              (pfilt (ndropn (nnth (nfuse ls1 ls2) ll) nell) 
                    (nmap (\<lambda> x. x- (nnth (nfuse ls1 ls2) ll)) (ndropn ll (nfuse ls1 ls2))))"
   using "2"  "4" nfuse_pfilt_alt[of "(nfuse ls1 ls2)" nell ll] "20" assms nfuse_nfinite by blast
 have 6: "pfilt (nfuse (ntaken (nlast ls1) nell) (ndropn (nfirst ls2) nell)) (nfuse ls1 ls2) =
          pfilt (nfuse (ntaken (nnth (nfuse ls1 ls2) ll) nell) 
                     (ndropn (nnth (nfuse ls1 ls2) ll) nell)) 
               (nfuse ls1 ls2)" 
    using "1" "4" 
    by (metis "01" assms(2) ntaken_nfuse ntaken_nlast the_enat.simps)
 have 7: "(pfilt (ntaken (nlast ls1) nell) ls1) =
          (pfilt (ntaken (nnth (nfuse ls1 ls2) ll) nell) (ntaken ll (nfuse ls1 ls2)))" 
    using "1" "4" 
    by (metis "01" assms(2) ntaken_nfuse ntaken_nlast the_enat.simps)
 have 8: "(pfilt (ndropn (nfirst ls2) nell) (nmap (\<lambda>x. x- (nfirst ls2)) ls2)) =
          (pfilt (ndropn (nnth (nfuse ls1 ls2) ll) nell) 
                (nmap (\<lambda>x. x- (nnth (nfuse ls1 ls2) ll)) (ndropn ll (nfuse ls1 ls2)) ))"
    using "1" "4" 
    by (metis "01" ndropn_nfirst ndropn_nfuse nfinite_conv_nlength_enat the_enat.simps)
 show ?thesis using "5" "6" "7" "8" by auto
qed  

lemma pfilt_ntaken:
assumes "n \<le> nlength ls"
shows   "  ntaken n (pfilt \<sigma> ls) = pfilt \<sigma> (ntaken n ls) "
proof -
 have 1: "nlength (ntaken n (pfilt \<sigma> ls)) = nlength ( pfilt \<sigma> (ntaken n ls))"
   by (simp add: assms pfilt_nlength)
 have 2: "\<forall> (i::nat). i\<le>nlength(ntaken n (pfilt \<sigma> ls))\<longrightarrow> 
           (nnth (ntaken n (pfilt \<sigma> ls)) i) = (nnth ( pfilt \<sigma> (ntaken n ls)) i)"
     by (simp add: pfilt_nmap)
 show ?thesis using "1" "2" nellist_eq_nnth_eq by blast
qed


lemma pfilt_ntaken_nidx:
assumes "n \<le> nlength nell"
        "nidx  ls"
        "nnth ls 0 = 0" 
        "nfinite ls" 
        "nlast ls  = n "
shows  " pfilt nell ls = pfilt (ntaken n nell) ls"
proof -
 have 1: " nlength(pfilt nell ls) = nlength (pfilt (ntaken n nell) ls) " 
   by (simp add: pfilt_nlength)
 have 2: "\<forall>(i::nat). i\<le>nlength(pfilt nell ls) \<longrightarrow> 
           (nnth (pfilt nell ls) i) = (nnth (pfilt (ntaken n nell) ls) i)"
   using assms 
   by (auto simp add: nidx_all_le_nlast ntaken_nnth pfilt_nlength pfilt_nnth) 
 show ?thesis 
 by (simp add: "1" "2" nellist_eq_nnth_eq)
qed


lemma pfilt_ndropn:
assumes "n \<le> nlength ls"
shows  "  ndropn n (pfilt nell ls) = pfilt nell (ndropn n ls) "
proof -
 have 1: "nlength (ndropn n (pfilt nell ls)) = nlength (pfilt nell (ndropn n ls))"
   by (simp add: assms pfilt_nlength)
 have 2: "\<forall> (i::nat). i\<le>nlength (ndropn n (pfilt nell ls)) \<longrightarrow> 
          (nnth (ndropn n (pfilt nell ls)) i) = (nnth (pfilt nell (ndropn n ls)) i) "
     by (simp add: ndropn_nmap pfilt_nmap)
 show ?thesis using "1" "2" nellist_eq_nnth_eq by blast
qed


lemma pfilt_ndropn_nidx_nlength:
assumes "(enat n) \<le> nlength nell"
        "nidx  ls"
        "nnth ls 0 = 0 "
shows " nlength (pfilt nell (nmap (\<lambda>x. x+ n) ls)) = nlength (pfilt (ndropn n nell) ls)"
using assms by (simp add: pfilt_nlength)

lemma pfilt_ndropn_nidx_nnth:
assumes "(enat n) \<le> nlength nell"
        "nidx ls"
        "nnth ls 0 = 0" 
        "i\<le>nlength ls"
shows " nnth (pfilt nell (nmap (\<lambda>x. x+  n) ls)) i = nnth (pfilt (ndropn n nell) ls) i"
proof -
 have 1: " nnth (pfilt nell (nmap (\<lambda>x. x+ n) ls)) i = nnth nell (nnth (nmap (\<lambda>x. x+ n) ls) i)  "
 by (simp add: assms pfilt_nnth pfilt_nlength) 
 have 2: "nnth nell (nnth (nmap (\<lambda>x. x+ n) ls) i) = (nnth nell ((nnth ls i) + n))"
   by (simp add: assms)
 have 3: " nnth (pfilt (ndropn n nell) ls) i = nnth (ndropn n nell) (nnth ls i)  "  
   by (simp add: assms pfilt_nnth pfilt_nlength)
 have 4: "nnth (ndropn n nell) (nnth ls i) = (nnth nell ((nnth ls i) + n)) "
    by (metis ntaken_ndropn_nlast ntaken_nlast) 
 show ?thesis 
 using "1" "2" "3" "4" by presburger
qed


lemma pfilt_ndropn_nidx:
assumes "(enat n) \<le> nlength nell"
        "nidx ls"
        " nnth ls 0 = 0 " 
shows " pfilt nell (nmap (\<lambda>x. x+ n) ls) = pfilt (ndropn n nell) ls"
using assms  pfilt_ndropn_nidx_nlength pfilt_ndropn_nidx_nnth nellist_eq_nnth_eq
by (simp add: pfilt_ndropn_nidx_nnth nellist_eq_nnth_eq pfilt_nlength)

lemma pfilt_pfilt:
 " (nnth (pfilt (pfilt nell ls1) ls2) k) =  (nnth nell  (nnth ls1 (nnth ls2 k)))"
by (metis enat_le_plus_same(1) gen_nlength_def ndropn_nmap nlength_code nnth_nmap nnth_zero_ndropn 
    pfilt_nmap)

lemma pfilt_pfilt_nmap:
 "(pfilt (pfilt nell ls1) ls2) = (pfilt nell (nmap (\<lambda> x. (nnth ls1 x)) ls2))"
by (simp add: pfilt_expand pfilt_nlength pfilt_pfilt)


lemma pfilt_nmap_pfilt:
 "(pfilt (pfilt nell ls1) ls2) = pfilt nell (pfilt ls1 ls2)"
by (metis pfilt_nmap pfilt_pfilt_nmap)

lemma pfilt_nsubn:
 assumes "k \<le> n"
         "n \<le> nlength ls"
 shows   " (nsubn (pfilt nell ls)) k n  = (pfilt nell (nsubn ls k n))"
using assms 
by (simp add: ndropn_nmap nsubn_def1 pfilt_nmap)

lemma pfilt_nfusecat_nmap:
 "  (pfilt nell (nfusecat lls)) = (nfusecat (nmap (\<lambda> ls . (pfilt nell ls)) lls))    "
proof -
 have 1: "pfilt nell (nfusecat lls) = nmap (nnth nell) (nfusecat lls)" 
   using pfilt_nmap[of nell "(nfusecat lls)"] by blast
 have 2: "(nfusecat (nmap (\<lambda> ls . (pfilt  nell ls)) lls)) =
          (nfusecat (nmap ( nmap (nnth nell) ) lls))" 
   using pfilt_nmap by metis
 have 3: " nmap (nnth nell) (nfusecat lls) = (nfusecat (nmap ( nmap (nnth nell) ) lls))"
   by (simp add: nmap_nfusecat)
 show ?thesis 
 by (simp add: "1" "2" "3")
qed

subsubsection \<open>powerinterval lemmas\<close>


lemma powerinterval_splita0:
 assumes " nidx  ls"
         "nnth ls 0 = 0 " 
         " n \<le> nlength ls "
         "nfinite ls " 
         "nfinite \<sigma> " 
         " nlast ls  = the_enat(nlength \<sigma>)"
         " i < nlength(ntaken n ls)"
 shows   "  (nsubn (ntaken (nnth ls n) \<sigma>)  (nnth (ntaken n ls) i) (nnth (ntaken n ls) (Suc i)) ) =
            (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) " 
proof -
 have 01: "(nnth ls n) \<le> nlength \<sigma>"
   using assms
     by (metis enat_ord_simps(1) nfinite_nlength_enat nidx_less_eq nnth_nlast the_enat.simps 
         verit_comp_simplify1(2)) 
 have 02: "(nnth (ntaken n ls) i) \<le> (nnth (ntaken n ls) (Suc i))"
    using assms 
   by (metis  dual_order.trans eSuc_enat enat_ord_simps(1) ileI1 le_add2 min_def nidx_less_eq 
       ntaken_nlength ntaken_nnth plus_1_eq_Suc)
 have 03: "(nnth (ntaken n ls) (Suc i)) \<le> (nnth ls n)"
    using assms 
    by (metis eSuc_enat enat_ord_simps(1) ileI1 min_def nidx_less_eq ntaken_nlength ntaken_nnth) 
  show ?thesis unfolding nsubn_def1 using 01 02 03 assms  
   by (simp add: ntaken_nnth ntaken_ndropn_swap order_subst2)
qed

lemma powerinterval_splita0_alt:
 assumes " nidx  ls"
         "nnth ls 0 = 0 " 
         " n \<le> nlength ls "
         "\<not>nfinite ls " 
         "\<not>nfinite \<sigma> " 
         " i < nlength(ntaken n ls)"
 shows   "  (nsubn (ntaken (nnth ls n) \<sigma>)  (nnth (ntaken n ls) i) (nnth (ntaken n ls) (Suc i)) ) =
            (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) " 
proof -
 have 01: "(nnth ls n) \<le> nlength \<sigma>"
   by (meson assms(5) enat_less_imp_le less_enatE nfinite_conv_nlength_enat)
 have 02: "(nnth (ntaken n ls) i) \<le> (nnth (ntaken n ls) (Suc i))"
  using assms 
  by (metis  dual_order.trans eSuc_enat enat_ord_simps(1) ileI1 le_add2 min_def nidx_less_eq 
      ntaken_nlength ntaken_nnth plus_1_eq_Suc)
 have 03: "(nnth (ntaken n ls) (Suc i)) \<le> (nnth ls n)"
   using assms 
  by (metis  eSuc_enat enat_ord_simps(1) ileI1 min_def nidx_less_eq ntaken_nlength ntaken_nnth)  
  show ?thesis unfolding nsubn_def1 using 01 02 03 assms  
   by (simp add: ntaken_nnth ntaken_ndropn_swap order_subst2)
qed

lemma powerinterval_splita:
 assumes " nidx ls"
         "nnth ls 0 = 0 " 
         "nfinite ls " 
         " n \<le> nlength ls"
         "nfinite \<sigma> " 
         " nlast ls  = the_enat(nlength \<sigma>)"
         " powerinterval f \<sigma> ls "
 shows   " powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls)"
proof -
 have 0: "(\<forall>i.  i<nlength ls \<longrightarrow> ( (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)) "
   using assms by (simp add: powerinterval_def)
 have 1: " powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls) =
           (\<forall>i. i < nlength(ntaken n ls) \<longrightarrow>
                ((nsubn (ntaken (nnth ls n) \<sigma>) (nnth (ntaken n ls) i) (nnth (ntaken n ls) (Suc i)) ) \<Turnstile> f)) "
       using powerinterval_def by blast
 have 2: "(\<forall>i. i < nlength(ntaken n ls) \<longrightarrow>
                ((nsubn (ntaken (nnth ls n) \<sigma>)  (nnth (ntaken n ls) i) (nnth (ntaken n ls) (Suc i)) ) \<Turnstile> f)) "
  proof 
    fix i
    show "i < nlength(ntaken n ls) \<longrightarrow>
                ((nsubn (ntaken (nnth ls n) \<sigma>) (nnth (ntaken n ls) i) (nnth (ntaken n ls) (Suc i)) ) \<Turnstile> f) " 
    proof -
     have 21: " i<n \<longrightarrow> ((nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)  " 
       using "0" assms less_le_trans by (metis enat_ord_simps(2))
     have 22: "i<nlength(ntaken n ls) \<longrightarrow> 
               ((nsubn \<sigma> (nnth (ntaken n ls) i) (nnth (ntaken n ls) (Suc i)) ) \<Turnstile> f) "
        by (simp add: "21" assms(4) ntaken_nnth)
     have 23: "i<nlength(ntaken n ls) \<longrightarrow>  
               ((nsubn (ntaken (nnth ls n) \<sigma>) (nnth (ntaken n ls) i) (nnth (ntaken n ls) (Suc i)) ) \<Turnstile> f)"
      using "22" assms powerinterval_splita0
      using "21" by fastforce
     show ?thesis using "23" by blast
    qed
   qed
  show ?thesis using "1" "2" by blast
qed

lemma powerinterval_splita_alt:
 assumes " nidx ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls " 
         " n \<le> nlength ls"
         "\<not>nfinite \<sigma> " 
         " powerinterval f \<sigma> ls "
 shows   " powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls)"
proof -
 have 0: "(\<forall>i.  i<nlength ls \<longrightarrow> ( (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)) "
   using assms by (simp add: powerinterval_def)
 have 1: " powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls) =
           (\<forall>i. i < nlength(ntaken n ls) \<longrightarrow>
                ((nsubn (ntaken (nnth ls n) \<sigma>) (nnth (ntaken n ls) i) (nnth (ntaken n ls) (Suc i)) ) \<Turnstile> f)) "
       using powerinterval_def by blast
 have 2: "(\<forall>i. i < nlength(ntaken n ls) \<longrightarrow>
                ((nsubn (ntaken (nnth ls n) \<sigma>)  (nnth (ntaken n ls) i) (nnth (ntaken n ls) (Suc i)) ) \<Turnstile> f)) "
  proof 
    fix i
    show "i < nlength(ntaken n ls) \<longrightarrow>
                ((nsubn (ntaken (nnth ls n) \<sigma>) (nnth (ntaken n ls) i) (nnth (ntaken n ls) (Suc i)) ) \<Turnstile> f) " 
    proof -
     have 21: " i<n \<longrightarrow> ((nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)  " 
       using "0" assms less_le_trans by (metis enat_ord_simps(2))
     have 22: "i<nlength(ntaken n ls) \<longrightarrow> 
               ((nsubn \<sigma> (nnth (ntaken n ls) i) (nnth (ntaken n ls) (Suc i)) ) \<Turnstile> f) "
        by (simp add: "21" assms(4) ntaken_nnth)
     have 23: "i<nlength(ntaken n ls) \<longrightarrow>  
               ((nsubn (ntaken (nnth ls n) \<sigma>) (nnth (ntaken n ls) i) (nnth (ntaken n ls) (Suc i)) ) \<Turnstile> f)"
      using "22" assms powerinterval_splita0_alt
      using "21" by fastforce
     show ?thesis using "23" by blast
    qed
   qed
  show ?thesis using "1" "2" by blast
qed

lemma powerinterval_splitb0:
 assumes " nidx  ls "
         "nnth ls 0 = 0 " 
         "nfinite ls " 
         " n \<le> nlength ls"
         "nfinite \<sigma> " 
         " nlast ls =  the_enat(nlength \<sigma>)"
         "  i < (nlength ls -n) "
 shows   "(nsubn (ndropn (nnth ls n) \<sigma>)
                 (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) i) 
                 (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) (Suc i)) 
                   ) =
          (nsubn \<sigma> (nnth ls (i+n)) (nnth ls ((Suc i)+n)) ) "
proof -
 have 1: "nlength (((nmap (\<lambda>x. x-  (nnth ls n))  (ndropn n ls)))) = (nlength ls) - n "
      by (simp add: assms) 
 have 2: " i<(nlength ls -n) \<longrightarrow> 
               (nnth (((nmap (\<lambda>x .x- (nnth ls n))  (ndropn n ls)))) i) =
                (nnth ls (n + i)) - (nnth ls n) "
           by simp
 have 3: "i<(nlength ls -n) \<longrightarrow> 
              (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) (Suc i)) =
              (nnth ls (n + (Suc i))) - (nnth ls n) "
       using assms by (metis eSuc_enat ileI1 ndropn_nlength ndropn_nnth nnth_nmap)
 have 4: "i<(nlength ls -n) \<longrightarrow>
              (nsubn (ndropn (nnth ls n) \<sigma>)
                     (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) i) 
                     (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) (Suc i)) 
                   ) =
                (nsubn (ndropn (nnth ls n) \<sigma>)
                     ( (nnth ls (n + i)) - (nnth ls n) )
                     ( (nnth ls (n + (Suc i))) - (nnth ls n) )
                   ) " 
          by (simp add: "2" "3")
 have 5: " i<(nlength ls -n) \<longrightarrow> (nnth ls (n + i))  < (nnth ls (n + (Suc i)))"
  using assms 
  by (metis Suc_ile_eq add.commute add_Suc_right enat_min nidx_expand plus_enat_simps(1))
 have 6: "i<(nlength ls -n) \<longrightarrow> (nnth ls n) \<le> (nnth ls (n + i))" 
    using assms 
    by (metis add.commute enat_min le_add1 nidx_less_eq order_less_imp_le plus_enat_simps(1))
 have 7: "i<(nlength ls -n) \<longrightarrow> (nnth ls n) \<le> (nnth ls (n + (Suc i)))" 
    using "5" "6" by linarith
 have 8: " i<(nlength ls -n) \<longrightarrow> (nnth ls (n + i)) - (nnth ls n) < (nnth ls (n + (Suc i))) - (nnth ls n)" 
   using "5" "6" diff_less_mono by blast
 have 10: "i<(nlength ls -n) \<longrightarrow> 
              (nsubn (ndropn (nnth ls n) \<sigma>)
                     ( (nnth ls (n + i)) - (nnth ls n) )
                     ( (nnth ls (n + (Suc i))) - (nnth ls n) )
                   ) = 
               (nsubn \<sigma> (nnth ls (i+n)) (nnth ls ((Suc i)+n)) )  "
   by (metis "6" "7" "8" add.commute diff_add nsubn_ndropn)
 show ?thesis 
 using "2" "3" "10" assms by auto   
qed

lemma powerinterval_splitb0_alt:
 assumes " nidx  ls "
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls " 
         " n \<le> nlength ls"
         "\<not>nfinite \<sigma> " 
         "  i < (nlength ls -n) "
 shows   "(nsubn (ndropn (nnth ls n) \<sigma>)
                 (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) i) 
                 (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) (Suc i)) 
                   ) =
          (nsubn \<sigma> (nnth ls (i+n)) (nnth ls ((Suc i)+n)) ) "
proof -
 have 1: "nlength (((nmap (\<lambda>x. x-  (nnth ls n))  (ndropn n ls)))) = (nlength ls) - n "
      by (simp add: assms) 
 have 2: " i<(nlength ls -n) \<longrightarrow> 
               (nnth (((nmap (\<lambda>x .x- (nnth ls n))  (ndropn n ls)))) i) =
                (nnth ls (n + i)) - (nnth ls n) "
           by simp
 have 3: "i<(nlength ls -n) \<longrightarrow> 
              (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) (Suc i)) =
              (nnth ls (n + (Suc i))) - (nnth ls n) "
       using assms by (metis eSuc_enat ileI1 ndropn_nlength ndropn_nnth nnth_nmap)
 have 4: "i<(nlength ls -n) \<longrightarrow>
              (nsubn (ndropn (nnth ls n) \<sigma>)
                     (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) i) 
                     (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) (Suc i)) 
                   ) =
                (nsubn (ndropn (nnth ls n) \<sigma>)
                     ( (nnth ls (n + i)) - (nnth ls n) )
                     ( (nnth ls (n + (Suc i))) - (nnth ls n) )
                   ) " 
          by (simp add: "2" "3")
 have 5: " i<(nlength ls -n) \<longrightarrow> (nnth ls (n + i))  < (nnth ls (n + (Suc i)))"
  using assms 
  by (metis Suc_ile_eq add.commute add_Suc_right enat_min nidx_expand plus_enat_simps(1))
 have 6: "i<(nlength ls -n) \<longrightarrow> (nnth ls n) \<le> (nnth ls (n + i))" 
    using assms 
    by (metis add.commute enat_min le_add1 nidx_less_eq order_less_imp_le plus_enat_simps(1))
 have 7: "i<(nlength ls -n) \<longrightarrow> (nnth ls n) \<le> (nnth ls (n + (Suc i)))" 
    using "5" "6" by linarith
 have 8: " i<(nlength ls -n) \<longrightarrow> (nnth ls (n + i)) - (nnth ls n) < (nnth ls (n + (Suc i))) - (nnth ls n)" 
   using "5" "6" diff_less_mono by blast
 have 9: "i<(nlength ls -n) \<longrightarrow> (nnth ls (n + (Suc i))) - (nnth ls n) \<le> nlength \<sigma> - (nnth ls n)"
    using assms 
    by (simp add: nfinite_conv_nlength_enat)
  have 10: "i<(nlength ls -n) \<longrightarrow> 
              (nsubn (ndropn (nnth ls n) \<sigma>)
                     ( (nnth ls (n + i)) - (nnth ls n) )
                     ( (nnth ls (n + (Suc i))) - (nnth ls n) )
                   ) = 
               (nsubn \<sigma> (nnth ls (i+n)) (nnth ls ((Suc i)+n)) )  "
   by (metis "6" "7" "8" add.commute diff_add nsubn_ndropn)
 show ?thesis 
 using "2" "3" "10" assms by auto   
qed

lemma powerinterval_splitb:
 assumes " nidx  ls"
         "nnth ls 0 = 0 " 
         "nfinite ls " 
         "  n \<le> nlength ls"
         "nfinite \<sigma> " 
         " nlast ls = the_enat(nlength \<sigma>) "
         " powerinterval f \<sigma> ls "
 shows   " powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))"
proof -
 have 0: "(\<forall> i. i<nlength ls \<longrightarrow> ( (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)) "
   using assms by (simp add: powerinterval_def)
 have 1: "powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))) =
         (\<forall> i. i<nlength (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) \<longrightarrow> 
            ( (nsubn (ndropn (nnth ls n) \<sigma>)
                   (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) i) 
                   (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) (Suc i)) 
                   ) \<Turnstile> f))   "
     by (simp add: powerinterval_def)
 have 2: "(\<forall> i. i<nlength(((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) \<longrightarrow> 
            ( (nsubn (ndropn (nnth ls n) \<sigma>)
                   (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) i) 
                   (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) (Suc i)) 
                   ) \<Turnstile> f)) "
  proof 
   fix i
   show "i<nlength(((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) \<longrightarrow> 
            ( (nsubn (ndropn (nnth ls n) \<sigma>)
                   (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) i) 
                   (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) (Suc i)) 
                   ) \<Turnstile> f) "
   proof -
    have 21: " i< (nlength ls) -n \<longrightarrow> 
               ((nsubn \<sigma> (nnth ls (i+n)) (nnth ls ((Suc i)+n)) ) \<Turnstile> f)"
        using "0" assms(4) enat_min by auto
    show ?thesis 
    using "21" assms powerinterval_splitb0 by fastforce
   qed
  qed
 show ?thesis using "1" "2" by blast
qed

lemma powerinterval_splitb_alt:
 assumes " nidx  ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls " 
         "  n \<le> nlength ls"
         "\<not>nfinite \<sigma> " 
         " powerinterval f \<sigma> ls "
 shows   " powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))"
proof -
 have 0: "(\<forall> i. i<nlength ls \<longrightarrow> ( (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)) "
   using assms by (simp add: powerinterval_def)
 have 1: "powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))) =
         (\<forall> i. i<nlength (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) \<longrightarrow> 
            ( (nsubn (ndropn (nnth ls n) \<sigma>)
                   (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) i) 
                   (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) (Suc i)) 
                   ) \<Turnstile> f))   "
     by (simp add: powerinterval_def)
 have 2: "(\<forall> i. i<nlength(((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) \<longrightarrow> 
            ( (nsubn (ndropn (nnth ls n) \<sigma>)
                   (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) i) 
                   (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) (Suc i)) 
                   ) \<Turnstile> f)) "
  proof 
   fix i
   show "i<nlength(((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) \<longrightarrow> 
            ( (nsubn (ndropn (nnth ls n) \<sigma>)
                   (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) i) 
                   (nnth (((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) (Suc i)) 
                   ) \<Turnstile> f) "
   proof -
    have 21: " i< (nlength ls) -n \<longrightarrow> 
               ((nsubn \<sigma> (nnth ls (i+n)) (nnth ls ((Suc i)+n)) ) \<Turnstile> f)"
        using "0" assms(4) enat_min by auto
    show ?thesis 
    using "21" assms powerinterval_splitb0_alt by fastforce
   qed
  qed
 show ?thesis using "1" "2" by blast
qed
       
lemma powerinterval_split:
 assumes " nidx  ls"
         "nnth ls 0 = 0 " 
         "nfinite ls " 
         "  n \<le> nlength ls"
         "nfinite \<sigma> " 
         " nlast ls =  the_enat(nlength \<sigma>)"
 shows   " powerinterval f \<sigma> ls =
           (powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls) \<and>
            powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))))"
proof -
 have 1: "powerinterval f \<sigma> ls \<Longrightarrow> 
          powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls) "
   by (simp add: assms powerinterval_splita)
 have 2: "powerinterval f \<sigma> ls \<Longrightarrow> 
          powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))) "
   by (simp add: assms powerinterval_splitb)
 have 3: "powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls) =
          (\<forall>i. i<n \<longrightarrow> ( (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)) "
    using assms by (simp add:  powerinterval_def powerinterval_splita0)
 have 4: "powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))) =
         (\<forall> i. i<(nlength ls) -n \<longrightarrow> ((nsubn \<sigma> (nnth ls (i+n)) (nnth ls ((Suc i)+n)) ) \<Turnstile> f)) "
    unfolding powerinterval_def using assms powerinterval_splitb0[of ls n \<sigma>] by (simp)
 have 5: "(\<forall> i. i<(nlength ls) -n \<longrightarrow> ((nsubn \<sigma> (nnth ls (i+n)) (nnth ls ((Suc i)+n)) ) \<Turnstile> f)) =
          (\<forall> i. n \<le> i \<and> i<(nlength ls) \<longrightarrow> ((nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)) "
     using assms  enat_min by auto
         (metis diff_add diff_less_mono enat_ord_simps(2) idiff_enat_enat nfinite_nlength_enat)
 have 6: "(powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls) \<and>
            powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) \<Longrightarrow>
           powerinterval f \<sigma> ls "
  using "3" "4" "5" assms powerinterval_def 
  by (metis not_less_eq not_less_less_Suc_eq order.order_iff_strict)
 show ?thesis 
 using "1" "2" 6 by blast  
qed

lemma powerinterval_split_alt:
 assumes " nidx  ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls " 
         "  n \<le> nlength ls"
         "\<not>nfinite \<sigma> " 
 shows   " powerinterval f \<sigma> ls =
           (powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls) \<and>
            powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))))"
proof -
 have 1: "powerinterval f \<sigma> ls \<Longrightarrow> 
          powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls) "
   by (simp add: assms powerinterval_splita_alt)
 have 2: "powerinterval f \<sigma> ls \<Longrightarrow> 
          powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))) "
   by (simp add: assms powerinterval_splitb_alt)
 have 3: "powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls) =
          (\<forall>i. i<n \<longrightarrow> ( (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)) "
  using assms by (simp add:  powerinterval_def powerinterval_splita0_alt)
 have 4: "powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))) =
         (\<forall> i. i<(nlength ls) -n \<longrightarrow> ((nsubn \<sigma> (nnth ls (i+n)) (nnth ls ((Suc i)+n)) ) \<Turnstile> f)) "
    unfolding powerinterval_def using powerinterval_splitb0_alt[of ls n \<sigma>] assms
    by simp
 have 5: "(\<forall> i. i<(nlength ls) -n \<longrightarrow> ((nsubn \<sigma> (nnth ls (i+n)) (nnth ls ((Suc i)+n)) ) \<Turnstile> f)) =
          (\<forall> i. n \<le> i \<and> i<(nlength ls) \<longrightarrow> ((nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)) "
     using assms  enat_min
    by auto
      (metis diff_add ndropn_nlength nfinite_ndropn_b nfinite_ntaken not_le_imp_less ntaken_all)
 have 6: "(powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls) \<and>
            powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) \<Longrightarrow>
           powerinterval f \<sigma> ls "
  using "3" "4" "5" assms powerinterval_def 
  by (metis not_less_eq not_less_less_Suc_eq order.order_iff_strict)
 show ?thesis 
 using "1" "2" 6 by blast  
qed

lemma powerinterval_nfuse:
 assumes " nidx  ls1"
         " nnth ls1 0 = 0 " 
         " nidx  ls2"
         " nnth ls2 0 = 0 "
         " nfinite ls1 " 
         " nlast ls1 = cp"
         "  cp \<le> nlength \<sigma> "
         " nfinite ls2" 
         " nfinite \<sigma> " 
         " nlast ls2  = the_enat(nlength \<sigma>) - cp"
 shows   " powerinterval f \<sigma> (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) =
           (powerinterval f (ntaken cp \<sigma>) ls1 \<and> 
            powerinterval f (ndropn cp \<sigma>) ls2 )"
proof -
 have 1: "nidx  (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) "
  using assms nidx_nfuse[of "ls1" "(nmap (\<lambda>x. x+ cp) ls2)"]   
    using Suc_ile_eq nidx_expand zero_enat_def by force  
 have 2: "cp = (nnth ls1 (the_enat (nlength ls1)))"
  using assms using nnth_nlast by blast
 have 3: "nlength ls1 \<le> nlength (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))"
   by (simp add: nfuse_nlength)
 have 30: "nfinite (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))  "
   using assms 
 by (metis  nfinite_conv_nlength_enat nfuse_nlength nlength_nmap plus_enat_simps(1))
 have 31: "nfirst (nmap (\<lambda>x. x+ cp) ls2) = nnth (nmap (\<lambda>x. x+ cp) ls2) 0"
     by (metis ndropn_0 ndropn_nfirst) 
 have 32: "nnth (nmap (\<lambda>x. x+ cp) ls2) 0 = cp"
   using assms nnth_nmap[of 0 ls2 "(\<lambda>x. x+ cp)"] 
   using zero_enat_def by auto   
 have 33: "nlast ls1 = nfirst (nmap (\<lambda>x. x+ cp) ls2)" 
     by (simp add: "31" "32" assms(6))
 have 34: "nlast (nmap (\<lambda>x. x + cp) ls2) = the_enat(nlength \<sigma>)" 
  using assms 
   by (metis diff_add enat.simps(3) enat_ord_simps(1) enat_the_enat nfinite_conv_nlength_enat nlast_nmap)
 have 4: "nlast  (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) = the_enat(nlength \<sigma>)"
     using assms using 30 assms nlast_nfuse[of ls1 "(nmap (\<lambda>x. x+ cp) ls2)"]  "33" "34" by presburger
  have 5: "cp = nnth (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) (the_enat (nlength ls1))"
   using assms 2
 by (metis  enat.simps(3) enat_the_enat nfinite_conv_nlength_enat nfuse_def1 nnth_nappend
      not_le_imp_less order_less_imp_le)
 have 50: "nlength (nmap (\<lambda>x. x- cp) (nmap (\<lambda>x. x+ cp) ls2)) = nlength ls2" 
    by simp
 have 51: "\<And>j. j \<le> nlength ls2 \<longrightarrow> 
                nnth (nmap (\<lambda>x. x- cp) (nmap (\<lambda>x. x+ cp) ls2)) j = nnth ls2 j " 
     by simp
 have 6: "(nmap (\<lambda>x. x- cp) (nmap (\<lambda>x. x+ cp) ls2)) = ls2 "
  using 50 51 by (simp add: nellist_eq_nnth_eq)  
  have 70: "ntaken (the_enat (nlength ls1)) (nfuse ls1 (nmap (\<lambda>x::nat. x + cp) ls2)) = ls1" 
   by (simp add: 33 assms(5) ntaken_nfuse)
 have 71: "ndropn (the_enat (nlength ls1)) (nfuse ls1 (nmap (\<lambda>x::nat. x + cp) ls2)) = 
           nmap (\<lambda>x::nat. x + cp) ls2" 
   by (simp add: 33 assms(5) ndropn_nfuse)
 have 72: "nidx (nfuse (ls1::nat nellist) (nmap (\<lambda>x::nat. x + (cp::nat)) (ls2::nat nellist)))"
   by (simp add: "1")
 have 73: "nnth (nfuse ls1 (nmap (\<lambda>x::nat. x + cp) ls2)) (0::nat) = (0::nat)" 
       by (metis 33 assms(2) nfuse_nnth zero_enat_def zero_le)
 have 74: "enat (the_enat (nlength ls1)) \<le> nlength (nfuse ls1 (nmap (\<lambda>x::nat. x + cp) ls2))"
   by (metis "70" assms(5) enat.simps(3) enat_the_enat min_def nfinite_nlength_enat ntaken_nlength
        order_eq_refl)
 have 75: "enat (nlast (nfuse ls1 (nmap (\<lambda>x::nat. x + cp) ls2))) = nlength \<sigma>"
    using "4" assms enat_the_enat nfinite_conv_nlength_enat by auto
 have 76: "nfinite \<sigma>" 
   by (simp add: assms(9))
 have 80: "powerinterval f \<sigma> (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) =
    (powerinterval f (ntaken (nnth (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) (the_enat (nlength ls1))) \<sigma>)
      (ntaken (the_enat (nlength ls1)) (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2))) \<and>
     powerinterval f (ndropn (nnth (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) (the_enat (nlength ls1))) \<sigma>)
      (nmap (\<lambda>x. x - nnth (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) (the_enat (nlength ls1)))
        (ndropn (the_enat (nlength ls1)) (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)))))"
     using 72 73 30 74 76 75 4 using powerinterval_split[of "(nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))" 
      "(the_enat (nlength ls1))" \<sigma> f]  
     by fastforce
  have 81: "powerinterval f (ntaken (nnth (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) (the_enat (nlength ls1))) \<sigma>)
      (ntaken (the_enat (nlength ls1)) (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2))) =
            powerinterval f (ntaken cp \<sigma>) ls1" 
    using "5" "70" by auto
  have 82: "powerinterval f (ndropn (nnth (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) (the_enat (nlength ls1))) \<sigma>)
      (nmap (\<lambda>x. x - nnth (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) (the_enat (nlength ls1)))
        (ndropn (the_enat (nlength ls1)) (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)))) =
         powerinterval f (ndropn cp \<sigma>) ls2  " 
        using "5" "6" "71" by presburger
  show ?thesis 
  using "80" "81" "82" by blast
qed

lemma powerinterval_nfuse_alt:
 assumes " nidx  ls1"
         " nnth ls1 0 = 0 " 
         " nidx  ls2"
         " nnth ls2 0 = 0 "
         " nfinite ls1 " 
         " nlast ls1 = cp"
         "  cp \<le> nlength \<sigma> "
         " \<not>nfinite ls2" 
         " \<not>nfinite \<sigma> " 
 shows   " powerinterval f \<sigma> (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) =
           (powerinterval f (ntaken cp \<sigma>) ls1 \<and> 
            powerinterval f (ndropn cp \<sigma>) ls2 )"
proof -
 have 1: "nidx  (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) "
  using assms nidx_nfuse[of "ls1" "(nmap (\<lambda>x. x+ cp) ls2)"]   
    using Suc_ile_eq nidx_expand zero_enat_def by force  
 have 2: "cp = (nnth ls1 (the_enat (nlength ls1)))"
  using assms using nnth_nlast by blast
 have 3: "nlength ls1 \<le> nlength (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))"
   by (simp add: nfuse_nlength)
 have 30: "\<not>nfinite (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))  "
  by (metis assms(8) enat_ile enat_le_plus_same(2) nfinite_conv_nlength_enat nfuse_nlength nlength_nmap)     
 have 5: "cp = nnth (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) (the_enat (nlength ls1))"
 by (metis "2" assms(5) enat.simps(3) enat_the_enat nfinite_conv_nlength_enat nfuse_def1 nnth_nappend 
     not_le_imp_less order_less_imp_le)
 have 50: "nlength (nmap (\<lambda>x. x- cp) (nmap (\<lambda>x. x+ cp) ls2)) = nlength ls2" 
    by simp
 have 51: "\<And>j. j \<le> nlength ls2 \<longrightarrow> 
                nnth (nmap (\<lambda>x. x- cp) (nmap (\<lambda>x. x+ cp) ls2)) j = nnth ls2 j " 
     by simp
 have 6: "(nmap (\<lambda>x. x- cp) (nmap (\<lambda>x. x+ cp) ls2)) = ls2 "
  using 50 51 by (simp add: nellist_eq_nnth_eq)  
 have 7: "nlast ls1 = nfirst (nmap (\<lambda>x. x+ cp) ls2)"
      by (metis add_0 assms(4) assms(6) nfinite_conv_nlength_enat nfinite_ntaken nlast_NNil 
          nlength_NNil nnth_nmap ntaken_0 ntaken_nlast the_enat.simps the_enat_0 zero_le)
 have 70: "ntaken (the_enat (nlength ls1)) (nfuse ls1 (nmap (\<lambda>x::nat. x + cp) ls2)) = ls1" 
   by (simp add: "7" assms(5) ntaken_nfuse)
 have 71: "ndropn (the_enat (nlength ls1)) (nfuse ls1 (nmap (\<lambda>x::nat. x + cp) ls2)) = 
           nmap (\<lambda>x::nat. x + cp) ls2" 
   by (simp add: "7" assms(5) ndropn_nfuse)
 have 72: "nidx (nfuse (ls1::nat nellist) (nmap (\<lambda>x::nat. x + (cp::nat)) (ls2::nat nellist)))"
   by (simp add: "1")
 have 73: "nnth (nfuse ls1 (nmap (\<lambda>x::nat. x + cp) ls2)) (0::nat) = (0::nat)" 
       by (metis "7" assms(2) nfuse_nnth zero_enat_def zero_le)
 have 74: "enat (the_enat (nlength ls1)) \<le> nlength (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2))"
   by (metis "70" assms(5) enat.simps(3) enat_the_enat min_def nfinite_nlength_enat ntaken_nlength 
       order_eq_refl)
 have 76: "\<not>nfinite \<sigma>" 
   by (simp add: assms(9))
 have 80: "powerinterval f \<sigma> (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) =
    (powerinterval f (ntaken (nnth (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) (the_enat (nlength ls1))) \<sigma>)
      (ntaken (the_enat (nlength ls1)) (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2))) \<and>
     powerinterval f (ndropn (nnth (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) (the_enat (nlength ls1))) \<sigma>)
      (nmap (\<lambda>x::nat. x - nnth (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) (the_enat (nlength ls1)))
        (ndropn (the_enat (nlength ls1)) (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)))))"
     using 72 73 30 74 76  
     using powerinterval_split_alt[of "(nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))" 
     "(the_enat (nlength ls1))" \<sigma> f]  
      by blast
  have 81: "powerinterval f (ntaken (nnth (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) (the_enat (nlength ls1))) \<sigma>)
      (ntaken (the_enat (nlength ls1)) (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2))) =
            powerinterval f (ntaken cp \<sigma>) ls1" 
    using "5" "70" by auto
  have 82: "powerinterval f (ndropn (nnth (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) (the_enat (nlength ls1))) \<sigma>)
      (nmap (\<lambda>x. x - nnth (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)) (the_enat (nlength ls1)))
        (ndropn (the_enat (nlength ls1)) (nfuse ls1 (nmap (\<lambda>x. x + cp) ls2)))) =
         powerinterval f (ndropn cp \<sigma>) ls2  " 
        using "5" "6" "71" by presburger
  show ?thesis 
  using "80" "81" "82" by blast
qed

subsubsection \<open>cppl lemmas\<close>

lemma cppl_expand:
assumes "(\<exists> ls. nidx  ls \<and> nnth ls 0 = 0 \<and> powerinterval f \<sigma> ls \<and>
              ((nfinite ls \<and> nfinite \<sigma> \<and> nlast ls = the_enat(nlength \<sigma>)) \<or>
               (\<not>nfinite ls \<and> \<not> nfinite \<sigma>)) \<and>
             ((pfilt \<sigma> ls) \<Turnstile> g  )) "
shows  " nidx  (cppl f g \<sigma>) \<and> nnth (cppl f g \<sigma>) 0 = 0 \<and> powerinterval f \<sigma> (cppl f g \<sigma>) \<and>
          ((nfinite (cppl f g \<sigma>) \<and> nfinite \<sigma> \<and> nlast  (cppl f g \<sigma>)  = the_enat(nlength \<sigma>)) \<or>
           (\<not> nfinite (cppl f g \<sigma>) \<and> \<not>nfinite \<sigma>)) \<and> ((pfilt \<sigma> (cppl f g \<sigma>)) \<Turnstile> g)"
proof -
 have 0: "cppl f g \<sigma> = (\<some> ls. nidx  ls \<and> nnth ls 0 = 0 \<and> powerinterval f \<sigma> ls \<and>
                             ((nfinite ls \<and> nfinite \<sigma> \<and> nlast ls  = the_enat(nlength \<sigma>)) \<or>
                              (\<not>nfinite ls \<and> \<not>nfinite \<sigma>)) \<and> ((pfilt \<sigma> ls) \<Turnstile> g)) "
   using cppl_def by blast
 have 1: "(\<exists> ls. nidx  ls \<and> nnth ls 0 = 0 \<and> powerinterval f \<sigma> ls \<and> 
               ((nfinite ls \<and> nfinite \<sigma> \<and> nlast ls  = the_enat(nlength \<sigma>)) \<or>
                (\<not>nfinite ls \<and> \<not>nfinite \<sigma>)) \<and> ((pfilt \<sigma> ls) \<Turnstile> g))"
      using assms by auto
 have 2: "nidx  (cppl f g \<sigma>) \<and> nnth (cppl f g \<sigma>) 0 = 0 \<and> powerinterval f \<sigma> (cppl f g \<sigma>) \<and>
           ((nfinite (cppl f g \<sigma>) \<and> nfinite \<sigma> \<and> nlast (cppl f g \<sigma>) = the_enat(nlength \<sigma>)) \<or>
            (\<not>nfinite (cppl f g \<sigma>) \<and> \<not>nfinite \<sigma>)) \<and> ((pfilt \<sigma> (cppl f g \<sigma>)) \<Turnstile> g)"
  using "0" "1" 
   someI_ex[of "\<lambda>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> powerinterval f \<sigma> ls \<and>
                    ((nfinite ls \<and> nfinite \<sigma> \<and>  nlast ls = the_enat(nlength \<sigma>)) \<or>
                     (\<not>nfinite ls \<and> \<not> nfinite \<sigma>)) \<and>
            ((pfilt \<sigma> ls) \<Turnstile> g)"]  by simp
 show ?thesis 
 using "2" by blast
qed  

lemma cppl_fprojection:
 "(\<sigma> \<Turnstile> f  fproj  g) =  
  ( nidx  (cppl f g \<sigma>) \<and> nnth (cppl f g \<sigma>) 0 = 0  \<and> powerinterval f \<sigma> (cppl f g \<sigma>) \<and>
    nfinite (cppl f g \<sigma>) \<and> nfinite \<sigma> \<and> 
    nlast (cppl f g \<sigma>) = the_enat(nlength \<sigma>) \<and> g (pfilt \<sigma> (cppl f g \<sigma>)) )"
using cppl_expand[of f \<sigma> g ] 
 by  (simp add: fprojection_d_def)
     (metis nfinite_conv_nlength_enat the_enat.simps)


lemma cppl_oprojection:
 "(\<sigma> \<Turnstile> f oproj g) =  
  ( nidx  (cppl f g \<sigma>) \<and> nnth (cppl f g \<sigma>) 0 = 0  \<and> powerinterval f \<sigma> (cppl f g \<sigma>) \<and>
    \<not>nfinite (cppl f g \<sigma>) \<and> \<not>nfinite \<sigma>  \<and> g (pfilt \<sigma> (cppl f g \<sigma>)) )"
using cppl_expand by (simp add: oprojection_d_def, blast)


lemma cppl_empty:
 assumes "nlength \<sigma> = 0"
         "(\<sigma> \<Turnstile> f  fproj  g)"
 shows   "(cppl f g \<sigma>) = (NNil 0)"
 using assms cppl_fprojection[of f g \<sigma>]  
 by (metis gr_zeroI less_numeral_extra(3) ndropn_0 ndropn_nlast nfinite_conv_nlength_enat 
     nidx_less_last_1 nnth_nlast the_enat.simps the_enat_0)

lemma cppl_empty_a:
 assumes "nlength \<sigma> = 0"
         "(cppl f g \<sigma>) = NNil 0"
         "g(pfilt \<sigma> (NNil 0))"
 shows   "(\<sigma> \<Turnstile> f  fproj  g)"
proof -
 have 1: "nidx  (NNil 0) "
   by (simp add: nidx_def)
 have 10: "nnth (NNil 0) 0 = 0 " 
     by (simp add: nnth_NNil)
 have 11: "nfinite (NNil 0)" 
    by simp
 have 12: "nfinite \<sigma>" 
   by (simp add: assms(1) nlength_eq_enat_nfiniteD zero_enat_def)
 have 2: "powerinterval f \<sigma> (NNil 0) "
   by (simp add: powerinterval_def)
 have 3: "nlast (NNil 0) =  nlength \<sigma> "
    by (simp add: assms)
 have 4: " g(pfilt \<sigma> (NNil 0))"
    using assms by blast
 from 1 10 11 12 2 3 4 show ?thesis using assms 
 by (simp add: cppl_fprojection zero_enat_def)
qed  


lemma cppl_more:
 assumes "nlength \<sigma> >0"
         "(\<sigma> \<Turnstile> f  fproj  g)"
 shows   "nlength(cppl f g \<sigma>)>0"
using assms 
by (metis cppl_fprojection enat_add_sub_same gen_nlength_def gr_zeroI i0_ne_infinity ndropn_nlast 
    ndropn_nlength nlength_NNil nlength_code nnth_nlast the_enat_0 zero_enat_def)

lemma cppl_more_infinite:
 assumes "nlength \<sigma> >0"
         "(\<sigma> \<Turnstile> f oproj g)"
 shows   "nlength(cppl f g \<sigma>)>0"
using assms
by (simp add: cppl_oprojection nfinite_conv_nlength_enat)


lemma cppl_more_than_first:
 assumes "nlength \<sigma> >0"
         "(\<sigma> \<Turnstile> f  fproj  g)"
 shows   "(nnth (cppl f g \<sigma>) 0) = 0"
using assms 
using cppl_fprojection by blast

lemma cppl_more_than_first_alt:
 assumes "nlength \<sigma> >0"
         "(\<sigma> \<Turnstile> f oproj g)"
 shows   "(nnth (cppl f g \<sigma>) 0) = 0"
using assms 
using cppl_oprojection by blast

lemma cppl_more_than_last:
 assumes "nlength \<sigma> >0"
         "(\<sigma> \<Turnstile> f  fproj  g)"
 shows   "nlast (cppl f g \<sigma>) = the_enat(nlength \<sigma>)"
using assms cppl_fprojection by blast

lemma cppl_sub_more:
 assumes "n < k"
         "k \<le> nlength \<sigma>"
         "((nsubn \<sigma> n k ) \<Turnstile> f  fproj  g)"
 shows   " nlength(cppl f g (nsubn \<sigma> n k )) > 0"
using assms cppl_fprojection[of f g "(nsubn \<sigma> n k )" ] 
using cppl_more nsubn_nlength_gr_one by blast

lemma cppl_bounds:
 assumes "n < k"
         "k \<le> nlength \<sigma>"
         "((nsubn \<sigma> n k ) \<Turnstile> f  fproj  g)"
         "i < nlength (cppl f g (nsubn \<sigma> n k ))" 
 shows   "0 \<le> (nnth (cppl f g (nsubn \<sigma> n k)) i) \<and> (nnth (cppl f g (nsubn \<sigma> n k )) i) \<le> k -n"
using assms 
using cppl_fprojection[of f g "(nsubn \<sigma> n k )"] using nsubn_nlength[of \<sigma> n k ] 
by simp
   (metis enat_minus_mono1 idiff_enat_enat min_def nfinite_conv_nlength_enat nidx_less_last_1 
    nnth_nlast order_less_imp_le the_enat.simps)  

lemma cppl_nmap_bounds:
 assumes "n < k"
         "k \<le> nlength \<sigma>"
         "((nsubn \<sigma> n k ) \<Turnstile> f  fproj  g)"
         "i < nlength (nmap (\<lambda>x. x+  n) (cppl f g (nsubn \<sigma> n k))) " 
 shows   "n \<le> (nnth (nmap (\<lambda>x. x+ n) (cppl f g (nsubn \<sigma> n k))) i) \<and> 
              (nnth (nmap (\<lambda>x.  x+n) (cppl f g (nsubn \<sigma> n k ))) i) \<le> k "
using assms 
using cppl_bounds by fastforce

lemma cppl_nfirst:
 assumes "(nsubn \<sigma> x1a (nfirst ls) ) \<Turnstile> f  fproj  g"
 shows   "nfirst((nmap (\<lambda>x. x+x1a) (cppl f g (nsubn \<sigma> x1a (nfirst ls) ))))  = x1a"
proof -
 have 1: "( nidx  (cppl f g (nsubn \<sigma> x1a (nfirst ls))) \<and> 
            nnth (cppl f g (nsubn \<sigma> x1a (nfirst ls))) 0 = 0 \<and> 
            powerinterval f (nsubn \<sigma> x1a (nfirst ls)) (cppl f g (nsubn \<sigma> x1a (nfirst ls))) \<and>
            nfinite (cppl f g (nsubn \<sigma> x1a (nfirst ls))) \<and> nfinite (nsubn \<sigma> x1a (nfirst ls)) \<and>
            nlast (cppl f g (nsubn \<sigma> x1a (nfirst ls)))  = 
            the_enat(nlength (nsubn \<sigma> x1a (nfirst ls))) \<and> 
           g (pfilt (nsubn \<sigma> x1a (nfirst ls)) (cppl f g (nsubn \<sigma> x1a (nfirst ls) ))) ) " 
          using cppl_fprojection assms by auto
 have 3: " (nnth (cppl f g (nsubn \<sigma> x1a (nfirst ls) )) 0) =0"
  by (simp add: "1")
 show ?thesis
 by (metis "3" add_cancel_right_left nfirst_eq_nnth_zero nnth_nmap zero_enat_def zero_le) 
qed

lemma cppl_nfirst_same:
 assumes "(nsubn \<sigma> x1a x1a) \<Turnstile> f  fproj  g"
 shows   "nfirst((nmap (\<lambda>x. x+ x1a) (cppl f g (nsubn \<sigma> x1a x1a))))  = x1a"
proof -
 have 1: " nfirst (NNil x1a ) = x1a"
   by (metis NNil_eq_ntake_iff nellist.inject(1))  
 from 1 cppl_nfirst show ?thesis 
   by (metis assms)
qed

lemma cppl_nlast:
 assumes " ((nsubn \<sigma> x (nfirst ls) ) \<Turnstile> f  fproj  g)"
         "  x < nfirst ls"
         "  nfirst ls \<le> nlength \<sigma> "
 shows   "nlast((nmap (\<lambda>y. y+ x) (cppl f g (nsubn \<sigma> x (nfirst ls)))))  = nfirst ls"
 proof -
  have 01: "( nidx  (cppl f g (nsubn \<sigma> x (nfirst ls))) \<and> 
            nnth (cppl f g (nsubn \<sigma> x (nfirst ls))) 0 = 0 \<and>
            powerinterval f (nsubn \<sigma> x (nfirst ls)) (cppl f g (nsubn \<sigma> x (nfirst ls) )) \<and>
            nfinite (cppl f g (nsubn \<sigma> x (nfirst ls))) \<and> nfinite (nsubn \<sigma> x (nfirst ls)) \<and>
            nlast (cppl f g (nsubn \<sigma> x (nfirst ls))) = the_enat(nlength (nsubn \<sigma> x (nfirst ls))) \<and> 
            g (pfilt (nsubn \<sigma> x (nfirst ls)) (cppl f g (nsubn \<sigma> x (nfirst ls)))) ) "
     using cppl_fprojection assms by (simp add: cppl_fprojection)
  have 02: "nlast (cppl f g (nsubn \<sigma> x (nfirst ls))) = 
             the_enat(nlength (nsubn \<sigma> x (nfirst ls) )) "
        using 01 by auto
  have 04: "x < nfirst ls "
    using assms by blast
  have 05: "nlength (nsubn \<sigma> x (nfirst ls)) = (nfirst ls) -x "
   using assms by (metis enat_minus_mono1 idiff_enat_enat min.orderE nsubn_nlength)
  have 06: "nlast((nmap (\<lambda>y. y+ x) (cppl f g (nsubn \<sigma> x (nfirst ls))))) =
             ((nfirst ls) -x)+x "
      using "01" "05" by auto
  show ?thesis using "04" "06" by auto
qed

lemma cppl_nlast_i:
 assumes "((nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> f  fproj  g)"
         "(nnth ls i) < (nnth ls (Suc i))"
         "(nnth ls (Suc i)) \<le> nlength \<sigma> "
 shows   "nlast((nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))))  = 
          (nnth ls (Suc i))"
 proof -
  have 01: "( nidx (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) \<and> 
             nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) 0 = 0 \<and>
            powerinterval f (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) 
                            (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) \<and>
             nfinite (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) \<and>
             nfinite (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<and>
             nlast (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) = 
               the_enat(nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) \<and> 
           g (pfilt (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) 
                    (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))) )"
     using assms by (simp add: cppl_fprojection)
  have 02: "nlast (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) = 
               the_enat(nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) "
        using 01 by auto
  have 04: "(nnth ls i) < (nnth ls (Suc i)) "
        by (simp add: assms)
  have 05: "nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) = (nnth ls (Suc i)) -(nnth ls i) "
      by (metis assms(3) enat_minus_mono1 idiff_enat_enat min.orderE nsubn_nlength)
  have 06: "nlast((nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))))) =
             ((nnth ls (Suc i)) -(nnth ls i))+(nnth ls i) "
     using "01" "05" by auto
  show ?thesis using "04" "06" by auto
qed


subsubsection \<open>lcppl lemmas\<close>


lemma lcppl_nnth:
 assumes "nidx ls"
         " i < nlength ls"
 shows   " (nnth (lcppl f g \<sigma> ls) i) = 
           (nmap (\<lambda>x .x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )))"
using assms
proof (induct i arbitrary: ls)
case 0
then show ?case
  proof (cases ls)
  case (NNil x1)
  then show ?thesis
  using "0.prems"(2) by auto
  next
  case (NCons x21 x22)
  then show ?thesis using NCons
    proof (cases "is_NNil x22")
    case True
    then show ?thesis using NCons by simp 
     (metis is_NNil_def lcppl_code(2) nnth_NNil)
    next
    case False
    then show ?thesis using NCons by simp
     (metis lcppl_code(3) nellist.collapse(2) nnth_0)
    qed
  qed
next
case (Suc i)
then show ?case 
  proof (cases ls)
  case (NNil x1)
  then show ?thesis 
  using Suc.prems(2) by auto
  next
  case (NCons x21 x22)
  then show ?thesis 
    proof (cases "is_NNil x22")
    case True
    then show ?thesis 
    by (metis NCons Suc.prems(2) enat_0_iff(1) ile0_eq iless_Suc_eq nat.simps(3) nellist.collapse(1) 
        nlength_NCons nlength_NNil)
    next
    case False
    then show ?thesis using NCons by simp 
        (metis (no_types, lifting) Suc(1) Suc.prems(1) Suc.prems(2) Suc_ile_eq iless_Suc_eq 
           lcppl_code(3) nellist.collapse(2)  nellist.sel(5) nidx_expand nlength_NCons nnth_ntl)
    qed
  qed
qed


lemma lcppl_nlength:
 assumes "nfinite ls" 
         "nidx  ls"
         " nlength ls > 0 " 
 shows   " nlength(lcppl f g \<sigma> ls) = (epred (nlength ls))"
using assms 
proof (induct ls rule: nfinite_induct)
case (NNil y)
then show ?case by simp
next
case (NCons x ls)
then show ?case 
   proof (cases "is_NNil ls")
   case True
   then show ?thesis using NCons 
   by (metis co.enat.sel(2) lcppl_code(2) nellist.collapse(1) nlength_NCons nlength_NNil)
   next
   case False
   then show ?thesis using NCons unfolding nidx_expand by simp 
    (metis NCons.prems(1) co.enat.exhaust_sel lcppl_code(3) ndropn_0 ndropn_nlast nellist.collapse(2) 
     nellist.disc(1) nidx_LCons_1 nidx_expand nlength_NCons the_enat_0)
   qed
qed

lemma lcppl_nfinite:
 assumes "nidx  ls"
 shows   " nfinite ls \<longleftrightarrow> nfinite(lcppl f g \<sigma> ls) " (is "?lhs \<longleftrightarrow> ?rhs")
proof
 assume a: ?lhs 
 show   ?rhs
   using a assms 
    proof (induct ls rule: nfinite_induct)
    case (NNil y)
    then show ?case by simp
    next
    case (NCons x ls)
    then show ?case 
       proof (cases "is_NNil ls")
       case True
       then show ?thesis using NCons 
       by simp
       next
       case False
       then show ?thesis using NCons by simp 
        (metis lcppl_code(3) nellist.collapse(2) nfinite_NConsI)
       qed
    qed
next
 assume b: ?rhs
 show ?lhs
 using b assms 
  proof (induct zs\<equiv>"(lcppl f g \<sigma> ls)" arbitrary: ls rule: nfinite_induct)
  case (NNil y)
  then show ?case 
  by (metis is_NNil_imp_nfinite lcppl.disc(2) nellist.disc(1) nfinite_ntl)
  next
  case (NCons x ls1)
  then show ?case 
    proof (cases "is_NNil ls1")
    case True
    then show ?thesis using NCons 
      by (metis is_NNil_imp_nfinite lcppl_code(3) nellist.collapse(2) nellist.disc(2) nellist.sel(5)
          nfinite_ntl)
    next
    case False
    then show ?thesis using NCons unfolding nidx_expand 
      by (metis NCons.hyps(2) NCons.prems is_NNil_imp_nfinite lcppl_code(3) nellist.collapse(2) 
          nellist.inject(2) nfinite_ntl nidx_LCons_1)
    qed
  qed      
qed

lemma lcppl_nlength_alt:
 assumes "\<not>nfinite ls" 
         "nidx  ls"
 shows   " nlength(lcppl f g \<sigma> ls) = (epred (nlength ls))"     
proof -
 have 1: "\<not> nfinite (lcppl f g \<sigma> ls)"
    using assms(1) assms(2) lcppl_nfinite by blast
 have 2: "epred (nlength ls ) = \<infinity> " 
    using assms
    by (metis enat2_cases epred_Infty nlength_eq_enat_nfiniteD)
 have 3: "nlength (lcppl f g \<sigma> ls) = \<infinity> "
 by (meson "1" enat2_cases nlength_eq_enat_nfiniteD) 
 show ?thesis
 by (simp add: "2" "3")
qed 



lemma lcppl_nlength_zero:
 assumes "nidx ls"
         "nlength ls = 0" 
 shows   "nlength(lcppl f g \<sigma> ls) = 0"
using assms
by (metis (no_types, lifting) is_NNil_def lcppl.ctr(1) le_numeral_extra(3) nlength_NNil 
    ntaken_0 ntaken_all zero_enat_def)

lemma lcppl_nlast:
 assumes "nidx  ls"
         "nfinite ls"
         "nfinite \<sigma>"  
         "nlast ls = the_enat(nlength \<sigma>)"
         " nlength ls >0 "
 shows   " nlast (lcppl f g \<sigma> ls) = 
             (nmap (\<lambda>x. x+ (nnth ls (the_enat (epred(nlength ls))))) 
                 (cppl f g (nsubn \<sigma> (nnth ls (the_enat (epred (nlength ls))))
                                    (nnth ls (the_enat (nlength ls))) )))"
proof -
 have 1: "nlast (lcppl f g \<sigma> ls) = (nnth (lcppl f g \<sigma> ls) ((the_enat (nlength (lcppl f g \<sigma> ls))))) "
    using assms lcppl_nfinite nnth_nlast by blast
 have 2: "nlast  (lcppl f g \<sigma> ls)  =
          (nmap (\<lambda>x. x+ (nnth ls ((the_enat (nlength (lcppl f g \<sigma> ls)))))) 
               (cppl f g (nsubn \<sigma> (nnth ls ((the_enat (nlength (lcppl f g \<sigma> ls))))) 
                                  (nnth ls (Suc (the_enat (nlength (lcppl f g \<sigma> ls))))) 
                             ))) "
      using assms lcppl_nnth[of ls "((the_enat (nlength (lcppl f g \<sigma> ls))))" f g \<sigma>] 
       lcppl_nlength[of ls f g \<sigma>]  
       by (metis "1" co.enat.exhaust_sel eSuc_enat enat_ord_simps(2) lcppl_nfinite less_add_same_cancel2 
           nfinite_nlength_enat plus_1_eq_Suc the_enat.simps zero_less_Suc zero_less_iff_neq_zero)
  have 3: "(nmap (\<lambda>x. x+ (nnth ls (the_enat (nlength (lcppl f g \<sigma> ls))))) 
               (cppl f g (nsubn \<sigma> (nnth ls ((the_enat (nlength (lcppl f g \<sigma> ls))))) 
                                  (nnth ls (Suc (the_enat (nlength (lcppl f g \<sigma> ls))))) 
                             ))) =
          (nmap (\<lambda>x. x+ (nnth ls ((the_enat (epred(nlength ls)))))) 
               (cppl f g (nsubn \<sigma> (nnth ls ((the_enat (epred (nlength ls))))) 
                                  (nnth ls (Suc (the_enat (epred(nlength ls))))) ))) "
        using assms lcppl_nlength 
        by (metis (no_types, lifting) nellist.map_cong0)
 have 4: "(Suc (the_enat (epred(nlength ls)))) = (the_enat (nlength ls))"
   by (metis assms(2) assms(5) co.enat.exhaust_sel enat.distinct(2) epred_Infty infinity_ne_i0 
       nfinite_conv_nlength_enat not_gr_zero the_enat_eSuc) 
 show ?thesis 
 using "1" "2" "3" "4" by presburger
qed

lemma lcppl_nlast_nlast:
 assumes "nidx  ls"
         "nfinite ls" 
         "nfinite \<sigma>" 
         "nlast ls = the_enat(nlength \<sigma>)"
         "((nsubn \<sigma> (nnth ls (the_enat (epred(nlength ls))) ) (nnth ls (the_enat(nlength ls)))) \<Turnstile> f  fproj  g)"
         " nlength ls > 0 "
 shows   " nlast (nlast (lcppl f g \<sigma> ls)) = the_enat(nlength \<sigma>) "
proof -
 have 2: "(nlast (lcppl f g \<sigma> ls)) =  
             (nmap (\<lambda>x. x+ (nnth ls (the_enat (epred(nlength ls))))) 
                 (cppl f g (nsubn \<sigma> (nnth ls (the_enat (epred (nlength ls)))) 
                                    (nnth ls (the_enat (nlength ls))) )))" 
    using assms lcppl_nlast[of ls \<sigma> f g]  by blast
 have 3: "nlast (nmap (\<lambda>x. x+ (nnth ls (the_enat (epred(nlength ls))))) 
                 (cppl f g (nsubn \<sigma> (nnth ls (the_enat (epred (nlength ls)))) 
                                    (nnth ls (the_enat (nlength ls))) ))) =
           (\<lambda>x. x+ (nnth ls (the_enat (epred(nlength ls))))) 
           (nlast  (cppl f g (nsubn \<sigma> (nnth ls (the_enat (epred (nlength ls)))) 
                                      (nnth ls (the_enat (nlength ls))) )))"
       using nnth_nmap assms by (simp add: cppl_fprojection)   
 have 4: "nnth ls (the_enat (epred (nlength ls))) < nnth ls (Suc (the_enat (epred (nlength ls))))" 
    using assms unfolding nidx_expand 
    by (metis Suc_ile_eq co.enat.exhaust_sel eSuc_enat enat.simps(3) enat_ord_simps(2) enat_the_enat 
        i0_less i0_ne_infinity lessI nfinite_conv_nlength_enat)
 have 5: "enat (nnth ls (Suc (the_enat (epred (nlength ls))))) \<le> nlength \<sigma>" 
    using assms 
    by (metis Orderings.order_eq_iff co.enat.exhaust_sel enat.simps(3) enat_the_enat epred_simps(5)
         i0_less i0_ne_infinity nfinite_conv_nlength_enat nnth_nlast the_enat_eSuc)
 have 6: "nlast (nmap (\<lambda>x. x+ (nnth ls (the_enat (epred(nlength ls))))) 
                 (cppl f g (nsubn \<sigma> (nnth ls (the_enat (epred (nlength ls)))) 
                                    (nnth ls (the_enat (nlength ls))) ))) =
           nnth ls (Suc (the_enat (epred (nlength ls))))"
    using cppl_nlast_i[of f g \<sigma> ls "(the_enat (epred (nlength ls)))"] assms 4 5
    by (metis  co.enat.exhaust_sel enat.simps(3) epred_simps(5) i0_less i0_ne_infinity 
        nfinite_conv_nlength_enat the_enat_eSuc)
 have 7: "nnth ls (Suc (the_enat (epred (nlength ls)))) = the_enat(nlength \<sigma>)"
    using assms
    by (metis  co.enat.collapse enat_eSuc_iff nfinite_nlength_enat nnth_nlast the_enat.simps 
      zero_less_iff_neq_zero) 
 show ?thesis 
 by (simp add: "2" "6" "7")
qed

lemma lcppl_zero_zero:
  assumes "nidx  ls"
          "nfinite ls"
          "nfinite \<sigma>" 
          " nlast ls  = the_enat(nlength \<sigma>)"
          " nlength ls = 0"
 shows   "(nnth (lcppl f g \<sigma> ls) 0) = 
          (nmap (\<lambda>x. x+ (nnth ls 0)) (cppl f g (nsubn \<sigma> (nnth ls 0) (nnth ls 0))))"
proof (cases ls)
case (NNil x1)
then show ?thesis 
by (metis lcppl_code(1)  nnth_NNil)
next
case (NCons x21 ls)
then show ?thesis 
using assms(5) by auto
qed
 
lemma lcppl_nfirst:
 assumes "nidx ls"
         "nfinite ls" 
         "nfinite \<sigma> " 
         "nlast ls = the_enat(nlength \<sigma>) "
         "(\<forall>i.  i<nlength ls \<longrightarrow> ((nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> f  fproj  g))  "
         " nlength ls > 0"
 shows   " nfirst(nfirst((lcppl f g \<sigma> ls))) = nfirst ls"
proof -
 have 01: "(nfirst((lcppl f g \<sigma> ls))) = 
           (nmap (\<lambda>x. x+ (nnth ls 0)) (cppl f g (nsubn \<sigma> (nnth ls 0) (nnth ls (Suc 0)))))" 
      using assms 
      by (metis lcppl_nnth ndropn_0 ndropn_nfirst zero_enat_def)
 have 02: "nlength ls > 0 \<longrightarrow> 
           nfirst((nmap (\<lambda>x. x+ (nnth ls 0)) (cppl f g (nsubn \<sigma> (nnth ls 0) (nnth ls (Suc 0)))))) =
            (nnth ls 0)" 
            using assms 01 cppl_fprojection[of f g  ]
            by (metis cppl_nfirst enat_0_iff(1) ndropn_nfirst) 
 show ?thesis 
 using "01" "02" assms 
 by (metis ndropn_0 ndropn_nfirst)
qed

lemma lcppl_nfirst_alt:
 assumes "nidx ls"
         "\<not>nfinite ls" 
         "\<not>nfinite \<sigma> " 
         "(\<forall>i.  i<nlength ls \<longrightarrow> ((nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> f  fproj  g))  "
 shows   " nfirst(nfirst((lcppl f g \<sigma> ls))) = nfirst ls"
proof -
 have 01: "(nfirst((lcppl f g \<sigma> ls))) = 
           (nmap (\<lambda>x. x+ (nnth ls 0)) (cppl f g (nsubn \<sigma> (nnth ls 0) (nnth ls (Suc 0)))))" 
      using assms 
      by (metis i0_less lcppl_nnth  nfinite_ntaken nlength_eq_enat_nfiniteD nnth_NNil nnth_nlast 
          ntaken_0 ntaken_nlast zero_enat_def)   
 have 02: "nfirst((nmap (\<lambda>x. x+ (nnth ls 0)) (cppl f g (nsubn \<sigma> (nnth ls 0) (nnth ls (Suc 0)))))) =
            (nnth ls 0)" 
      using assms 01 cppl_fprojection[of f g  ] 
      by (metis (no_types, lifting) add_0 gr_zeroI ndropn_0 ndropn_nfirst nlength_eq_enat_nfiniteD 
          nnth_nmap zero_enat_def zero_le)
 show ?thesis 
 using "01" "02" assms 
 by (metis ndropn_0 ndropn_nfirst)
qed

 
lemma lcppl_nfusecat_nlastnfirst:
 assumes "nfinite ls" 
          "nidx  ls"
         "nfinite \<sigma>"
         "nlast ls  = the_enat(nlength \<sigma>)"
         "((nlength ls = 0 \<and> ((nsubn \<sigma> (nnth ls 0) (nnth ls 0) ) \<Turnstile> f  fproj  g)) \<or>
           (nlength ls > 0 \<and> (\<forall>i.  i<nlength ls \<longrightarrow> ((nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g))))"
 shows   "nlastnfirst (lcppl f g \<sigma> ls)"
proof (cases "is_NNil ls")
case True
then show ?thesis 
by (metis is_NNil_def lcppl_code(1) nlastnfirst_NNil)
next
case False
then show ?thesis 
  proof (auto simp add: nlastnfirst_def1)  
  fix i
  assume a: "enat (Suc i) \<le> nlength (lcppl f g \<sigma> ls) " 
  assume b: "\<not> is_NNil ls " 
  show "  nlast (nnth (lcppl f g \<sigma> ls) i) = nfirst (nnth (lcppl f g \<sigma> ls) (Suc i))" 
   proof -
    have 1: " nlength ls > 0 "
      by (metis a assms(2) assms(5) enat.inject ile0_eq lcppl_nlength_zero old.nat.distinct(1) 
          zero_enat_def) 
    have 2: "(\<forall>i.  i<nlength ls\<longrightarrow> ((nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g))"
      using assms(5) by auto  
    have 3: "nidx ls " 
      by (simp add: assms(2))
    have 4: " i< nlength ls"
      by (metis "1" "3" Suc_ile_eq a assms(1) co.enat.exhaust_sel dual_order.order_iff_strict 
          iless_Suc_eq lcppl_nlength less_numeral_extra(3))
    have 5: "(nnth (lcppl f g \<sigma> ls) i) = 
              nmap (\<lambda>x::nat. x + nnth ls i) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))"
       using lcppl_nnth[of ls i f g \<sigma>] 
       using "3" "4" by blast
    have 6: " (Suc i) < nlength ls" 
        by (metis "1" "3" a assms(1) co.enat.exhaust_sel iless_Suc_eq lcppl_nlength less_numeral_extra(3))
    have 7: "(nnth (lcppl f g \<sigma> ls) (Suc i)) = 
              nmap (\<lambda>x::nat. x + nnth ls (Suc i)) (cppl f g (nsubn \<sigma> (nnth ls (Suc i)) (nnth ls (Suc (Suc i)))))" 
        using lcppl_nnth[of ls "Suc i"  f g \<sigma>]
        using "3" "6" by blast 
    have 71: "nnth ls i < nnth ls (Suc i)" 
        using assms nidx_expand[of ls] "6" order_less_imp_le by blast 
    have 72: "enat (nnth ls (Suc i)) \<le> nlength \<sigma>" 
      using assms 
      by (metis "3" "6"  dual_order.order_iff_strict enat_ord_simps(1) nfinite_conv_nlength_enat 
          nidx_less_last_1 nnth_nlast the_enat.simps)
    have 8: "nlast (nmap (\<lambda>x::nat. x + nnth ls i) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))) =
               (nnth ls (Suc i)) "
        using assms 2 3 4 5 6 cppl_nlast_i[of f g \<sigma> ls i ]  
        using "71" "72" by blast
    have 9: "nfirst (nmap (\<lambda>x::nat. x + nnth ls (Suc i)) 
                          (cppl f g (nsubn \<sigma> (nnth ls (Suc i)) (nnth ls (Suc (Suc i)))))) =
               (nnth ls (Suc i))"
       using 2 6 cppl_fprojection[of f g ] 
       by (metis add_0 nlast_NNil nnth_nmap ntaken_0 ntaken_nlast zero_enat_def zero_order(1))
    show ?thesis using "8" "9" "5" "7" by presburger
   qed
  qed
qed

lemma lcppl_nfusecat_nlastnfirst_alt:
 assumes "\<not>nfinite ls" 
          "nidx  ls"
         "\<not>nfinite \<sigma>"
         " (\<forall>i.  i<nlength ls\<longrightarrow> ((nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g))"
 shows   "nlastnfirst (lcppl f g \<sigma> ls)"
proof (auto simp add: nlastnfirst_def1)  
fix i
assume a: "enat (Suc i) \<le> nlength (lcppl f g \<sigma> ls) " 
show "  nlast (nnth (lcppl f g \<sigma> ls) i) = nfirst (nnth (lcppl f g \<sigma> ls) (Suc i))" 
proof -
    have 2: "(\<forall>i.  i<nlength ls\<longrightarrow> ((nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g))"
      using assms by auto  
    have 3: "nidx ls " 
      by (simp add: assms)
    have 4: " i< nlength ls"
      by (meson assms(1) enat_iless linorder_less_linear nfinite_conv_nlength_enat)
    have 5: "(nnth (lcppl f g \<sigma> ls) i) = 
              nmap (\<lambda>x::nat. x + nnth ls i) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))"
       using lcppl_nnth[of ls i f g \<sigma>] 
       using "3" "4" by blast
    have 6: " (Suc i) < nlength ls" 
      by (metis "4" assms(1) eSuc_enat ileI1 nlength_eq_enat_nfiniteD order_neq_le_trans)
    have 7: "(nnth (lcppl f g \<sigma> ls) (Suc i)) = 
              nmap (\<lambda>x::nat. x + nnth ls (Suc i)) 
                   (cppl f g (nsubn \<sigma> (nnth ls (Suc i)) (nnth ls (Suc (Suc i)))))" 
        using lcppl_nnth[of ls "Suc i"  f g \<sigma>]
           using "3" "6" by blast 
    have 8: "nlast (nmap (\<lambda>x::nat. x + nnth ls i) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))) =
               (nnth ls (Suc i)) "
        using assms 2 3 4 5 6 
        by (simp add: cppl_nlast_i nfinite_conv_nlength_enat nidx_expand)
    have 9: "nfirst (nmap (\<lambda>x::nat. x + nnth ls (Suc i)) 
                          (cppl f g (nsubn \<sigma> (nnth ls (Suc i)) (nnth ls (Suc (Suc i)))))) =
               (nnth ls (Suc i))"
       using 2 6 cppl_fprojection[of f g ] 
       by (metis add_0 nlast_NNil nnth_nmap ntaken_0 ntaken_nlast zero_enat_def zero_order(1))
    show ?thesis using "8" "9" "5" "7" by presburger
   qed
qed

lemma lcppl_nfusecat_nlength:
 assumes "nidx  ls"
         "nfinite ls"
         "nfinite \<sigma>" 
         "nlast ls  = the_enat(nlength \<sigma>) " 
         "((nlength ls =0 \<and> ((nsubn \<sigma> (nnth ls 0 ) (nnth ls 0 )  \<Turnstile> f  fproj  g))) \<or>
           (nlength ls >0 \<and> (\<forall> i. i<nlength ls \<longrightarrow> ((nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> f  fproj  g))))"
 shows   "(nlength ls =0 \<longrightarrow> nlength(nfusecat (lcppl f g \<sigma> ls)) = 0) \<and>
          (nlength ls >0 \<longrightarrow> 
            nlength(nfusecat (lcppl f g \<sigma> ls)) = 
            ( (\<Sum>k=0..(the_enat (epred(nlength ls))). ( (nlength (nnth (lcppl f g \<sigma> ls) k))))))"
proof -
 have 1: "nlastnfirst (lcppl f g \<sigma> ls)" 
  using assms lcppl_nfusecat_nlastnfirst by blast
 have 20: "nlength ls = 0 \<longrightarrow> 
            (lcppl f g \<sigma> ls) =
              (NNil ((nmap (\<lambda>x. x+ (nnth ls 0)) (cppl f g (nsubn \<sigma> (nnth ls 0) (nnth ls 0) ))))) "
      by (metis ile0_eq lcppl_code(1)  nellist_eq_nnth_eq nlength_NNil nnth_NNil  the_enat.simps 
          the_enat_0)
 have 2: "nlength ls = 0 \<longrightarrow> 
           nfusecat (lcppl f g \<sigma> ls) =  
           ((nmap (\<lambda>x. x+ (nnth ls 0)) (cppl f g (nsubn \<sigma> (nnth ls 0) (nnth ls 0) ))))" 
       using "20" nfusecat_NNil by auto   
 have 3: "nlength ls = 0 \<longrightarrow> 
           nlength ((nmap (\<lambda>x. x+ (nnth ls 0)) (cppl f g (nsubn \<sigma> (nnth ls 0) (nnth ls 0) )))) = 0"
    using assms 
    by (metis cppl_empty diff_self_eq_0 idiff_enat_enat less_numeral_extra(3) min.orderE ndropn_nlast 
        ndropn_nlength nlength_NNil nlength_nmap nnth_nlast not_le_imp_less nsubn_nlength the_enat_0 
        zero_enat_def)
 have 4: "nlength ls = 0 \<longrightarrow> nlength(nfusecat (lcppl f g \<sigma> ls)) = 0"
    using "2" "3" by auto 
 have 5: "nfinite (lcppl f g \<sigma> ls) " 
    using assms(1) assms(2) lcppl_nfinite by blast
 have 6: " all_nfinite (lcppl f g \<sigma> ls)"
 by (metis (no_types, lifting) "20" all_nfinite_nnth_a assms(1) assms(2) assms(5) co.enat.exhaust_sel
     cppl_fprojection iless_Suc_eq lcppl_nlength lcppl_nnth nfinite_nmap nnth_NNil 
     not_less_iff_gr_or_eq)  
 have 7: "nlength (lcppl f g \<sigma> ls) = (epred(nlength ls))"
    by (metis assms(1) assms(2) assms(5) epred_0 lcppl_nlength lcppl_nlength_zero) 
 have 8: "nlength ls > 0 \<longrightarrow>
            nlength(nfusecat (lcppl f g \<sigma> ls)) = 
            (\<Sum>k=0..(the_enat (epred(nlength ls))). ((nlength (nnth (lcppl f g \<sigma> ls) k)))) "
   using nfusecat_nlength_nfinite[of "(lcppl f g \<sigma> ls)"] 5 6 7 1 by presburger
  show ?thesis 
  using "4" "8" by blast
qed

lemma lcppl_nfusecat_nidx:
 assumes "nidx  ls"
         "nnth ls 0 = 0 "
         "nfinite ls"  
         "nfinite \<sigma>" 
         " nlast ls  = the_enat(nlength \<sigma>)" 
         "(\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         "nlength ls > 0 "
 shows   "nidx  (nfusecat ( (lcppl f g \<sigma> ls)))"
proof -
 have 0: "nlength \<sigma> >0 \<longrightarrow> nlength ls >0 "
    using assms by auto
 have 2: "nlength \<sigma> >0 \<longrightarrow> nlastnfirst (lcppl f g \<sigma> ls)"
    using assms lcppl_nfusecat_nlastnfirst by blast
 have 3: " (\<forall>i<nlength ls. nidx (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))) "
     using assms cppl_fprojection by auto
 have 4: " (\<forall>i<nlength ls. 
              nidx (nmap (\<lambda>x. x+(nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))))"
    using "3" by (simp add: Suc_ile_eq nidx_expand)
 have 5: "nlength \<sigma> >0 \<longrightarrow>
           (\<forall>i<nlength ls. nidx (nnth (lcppl f g \<sigma> ls) i) ) "
    using assms by (simp add: "4" lcppl_nnth)
 have 6: "nlength \<sigma> >0 \<longrightarrow> nlength (lcppl f g \<sigma> ls) = epred(nlength ls)"
    using assms lcppl_nlength by blast
 have 7: " nlength \<sigma> >0 \<longrightarrow>
            (\<forall>i\<le>nlength ls. 
              nidx  (nnth (lcppl f g \<sigma> ls) i) ) "
     using  "5" assms 
     by (metis "6" Extended_Nat.eSuc_mono antisym_conv2 co.enat.exhaust_sel enat_the_enat epred_simps(5)
          i0_ne_infinity iless_Suc_eq lcppl_nfinite nnth_beyond nnth_nlast)
 have 68: "\<And>i. i<nlength ls \<Longrightarrow> 0 < nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Longrightarrow> 
               (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> (f  fproj  g)  \<Longrightarrow> 
                0 < nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) " 
      using cppl_more by blast
 have 69: "\<And>i. i<nlength ls \<Longrightarrow> 0 < nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))"
    using assms unfolding nidx_expand by simp 
     (metis assms(1) dual_order.order_iff_strict eSuc_enat enat_ord_simps(2) ileI1 less_numeral_extra(3) 
      nfinite_conv_nlength_enat nidx_less_last_1 nnth_nlast nsubn_nlength_gr_one the_enat.simps)
 have 70: "nlength \<sigma> >0 \<longrightarrow> (\<forall>i<nlength ls.  (0::enat) < nlength  (nnth (lcppl f g \<sigma> ls) i))" 
     using lcppl_nnth[of ls _ f g \<sigma>] "68" "69" assms(1) assms(6) by auto
 have 700: "nlength \<sigma> >0 \<longrightarrow> (\<forall>lx\<in>nset (lcppl f g \<sigma> ls). (0::enat) < nlength lx)"
     using 70 assms 
     by (metis  co.enat.exhaust_sel iless_Suc_eq in_nset_conv_nnth lcppl_nlength less_numeral_extra(3))
 have 71: "nlength \<sigma> >0 \<longrightarrow> (\<forall>lx \<in> nset (lcppl f g \<sigma> ls). nfinite lx)" 
     using assms unfolding cppl_fprojection 
     by (metis (no_types, lifting) co.enat.exhaust_sel i0_less iless_Suc_eq in_nset_conv_nnth 
     lcppl_nlength lcppl_nnth nfinite_nmap)
 have 8: "nlength \<sigma> >0 \<longrightarrow>
           nidx  (nfusecat ( (lcppl f g \<sigma> ls))) "
     using assms nidx_nfusecat[of "( (lcppl f g \<sigma> ls))" ] 700 71 
     by (metis "2" 5 co.enat.exhaust_sel iless_Suc_eq lcppl_nlength zero_less_iff_neq_zero)
  from 8  show ?thesis using assms 
  by (metis gr_zeroI less_numeral_extra(3) nfinite_nlength_enat nidx_less_last_1 nnth_nlast 
      the_enat.simps zero_enat_def)
qed

lemma lcppl_nfusecat_nidx_alt:
 assumes "nidx  ls"
         "nnth ls 0 = 0 "
         "\<not>nfinite ls"  
         "\<not>nfinite \<sigma>" 
         "(\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
 shows   "nidx  (nfusecat ( (lcppl f g \<sigma> ls)))"
proof -
 have 2: "nlastnfirst (lcppl f g \<sigma> ls)"
    using assms lcppl_nfusecat_nlastnfirst_alt by blast
 have 3: " (\<forall>i<nlength ls. nidx (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))) "
     using assms cppl_fprojection by auto
 have 4: " (\<forall>i<nlength ls. 
              nidx (nmap (\<lambda>x. x+(nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))))"
    using "3" by (simp add: Suc_ile_eq nidx_expand)
 have 5: "(\<forall>i<nlength ls. nidx (nnth (lcppl f g \<sigma> ls) i) ) "
    using assms by (simp add: "4" lcppl_nnth)
 have 7: " (\<forall>i<nlength ls. 
              nidx  (nnth (lcppl f g \<sigma> ls) i) ) "
     using  "5" by simp
 have 60: "\<And>j. (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j))) \<Turnstile> (f  fproj  g) " 
    by (metis assms(3) assms(5) leI nfinite_ntaken ntaken_all)
 have 61: "\<And>j. nnth ls j < nnth ls (Suc j) " 
   by (metis Suc_ile_eq assms(1) assms(3) nfinite_ntaken nidx_expand not_le_imp_less ntaken_all)
 have 62: "\<And>j. enat (nnth ls (Suc j)) \<le> nlength \<sigma>" 
    by (meson assms(4) enat_iless enat_less_imp_le nfinite_conv_nlength_enat)
 have 63: "\<And>j. nlast (nmap (\<lambda>x::nat. x + nnth ls j) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j))))) = 
               nnth ls (Suc j) " 
    using "60" "61" "62" cppl_nlast_i by blast
 have 68: "\<And>i. i<nlength ls \<Longrightarrow> 0 < nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Longrightarrow> 
               (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> (f  fproj  g)  \<Longrightarrow> 
                0 < nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) " 
      using cppl_more by blast
 have 69: "\<And>i. i<nlength ls \<Longrightarrow> 0 < nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))"
    using assms unfolding nidx_expand using "61" "62" nsubn_nlength_gr_one by blast
 have 700: "nlength \<sigma> >0 \<longrightarrow> (\<forall>i<nlength ls.  (0::enat) < nlength  (nnth (lcppl f g \<sigma> ls) i))" 
     using lcppl_nnth[of ls _ f g \<sigma>] "68" "69" assms by (simp )
 have 701: "nlength \<sigma> >0 \<longrightarrow> (\<forall>lx\<in>nset (lcppl f g \<sigma> ls). (0::enat) < nlength lx)"
     using 700 assms 
     by (metis co.enat.exhaust_sel iless_Suc_eq in_nset_conv_nnth lcppl_nlength_alt
          nlength_eq_enat_nfiniteD zero_enat_def)
 have 70: " (\<forall>lx\<in>nset (lcppl f g \<sigma> ls). (0::enat) < nlength lx)"
    using "701" assms nfinite_conv_nlength_enat zero_enat_def by fastforce
 have 71: " (\<forall>lx \<in> nset (lcppl f g \<sigma> ls). nfinite lx)" 
     using assms 
     by (metis (no_types, lifting) co.enat.exhaust_sel cppl_fprojection iless_Suc_eq in_nset_conv_nnth 
         lcppl_nlength_alt lcppl_nnth nfinite_nmap nlength_eq_enat_nfiniteD zero_enat_def)
 have 8: "nidx  (nfusecat ( (lcppl f g \<sigma> ls))) "
     using assms nidx_nfusecat[of "( (lcppl f g \<sigma> ls))" ] 70 71 
     by (metis "2" "7" co.enat.exhaust_sel iless_Suc_eq lcppl_nlength_alt 
     nlength_eq_enat_nfiniteD zero_enat_def)
  from 8  show ?thesis using assms by blast
qed

lemma lcppl_nlength_all_nfinite:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "nfinite ls" 
         "nfinite \<sigma>" 
         "nlast ls = the_enat(nlength \<sigma>)"
         "(\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> f  fproj  g)"
         " nlength \<sigma> >0"
 shows  " (\<forall> j \<le> nlength (lcppl f g \<sigma> ls). nfinite(nnth (lcppl f g \<sigma> ls) j) ) "
proof 
   fix j
   show "j \<le> nlength (lcppl f g \<sigma> ls) \<longrightarrow> nfinite ( nnth (lcppl f g \<sigma> ls) j) "
   proof -
   have 1: "nlength \<sigma>>0 \<longrightarrow> nlastnfirst (lcppl f g \<sigma> ls)"
       using assms 
       by (metis enat.distinct(2) enat_the_enat i0_less lcppl_nfusecat_nlastnfirst 
           nfinite_conv_nlength_enat nnth_nlast the_enat_0)
   have 2: "nlength \<sigma>>0 \<longrightarrow> nlength ls >0"
    using assms 
    by (metis enat.distinct(2) enat_the_enat gr_zeroI nfinite_conv_nlength_enat nnth_nlast the_enat_0)
   have 3: "nlength \<sigma>>0 \<longrightarrow> j \<le> nlength (lcppl f g \<sigma> ls) \<longrightarrow> 
           (nnth (lcppl f g \<sigma> ls) j) = 
             (nmap (\<lambda>x. x+ (nnth ls j)) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j)))))" 
     using assms lcppl_nnth[of "ls" "j" "f" "g" "\<sigma>" ]
      by (metis "2" co.enat.exhaust_sel iless_Suc_eq lcppl_nlength not_gr_zero) 
   have 4: "nlength \<sigma>>0 \<longrightarrow> j \<le> nlength (lcppl f g \<sigma> ls) \<longrightarrow>
            nfinite (nmap (\<lambda>x. x+ (nnth ls j)) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j)))))  "
     using assms 2 
     by (metis co.enat.exhaust_sel cppl_fprojection iless_Suc_eq lcppl_nlength nfinite_nmap not_gr_zero)
   show ?thesis 
   using  "3" "4" assms(7) by presburger
   qed
qed 

lemma lcppl_nlength_all_gr_zero:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "nfinite ls" 
         "nfinite \<sigma>" 
         "nlast ls = the_enat(nlength \<sigma>)"
         "(\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> f  fproj  g)"
         " nlength \<sigma> >0"
 shows  " (\<forall> j \<le> nlength (lcppl f g \<sigma> ls). nlength(nnth (lcppl f g \<sigma> ls) j) > 0) "
proof 
   fix j
   show "j \<le> nlength (lcppl f g \<sigma> ls) \<longrightarrow> 0 < nlength ( nnth (lcppl f g \<sigma> ls) j) "
   proof -
   have 1: "nlength \<sigma>>0 \<longrightarrow> nlastnfirst (lcppl f g \<sigma> ls)"
       using assms 
       by (metis enat.distinct(2) enat_the_enat i0_less lcppl_nfusecat_nlastnfirst 
           nfinite_conv_nlength_enat nnth_nlast the_enat_0)
   have 2: "nlength \<sigma>>0 \<longrightarrow> nlength ls >0"
    using assms 
    by (metis enat.distinct(2) enat_the_enat gr_zeroI nfinite_conv_nlength_enat nnth_nlast the_enat_0)
   have 3: "nlength \<sigma>>0 \<longrightarrow> j \<le> nlength (lcppl f g \<sigma> ls) \<longrightarrow> 
           (nnth (lcppl f g \<sigma> ls) j) = 
             (nmap (\<lambda>x. x+ (nnth ls j)) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j)))))" 
     using assms lcppl_nnth[of "ls" "j" "f" "g" "\<sigma>" ]
      by (metis "2" co.enat.exhaust_sel iless_Suc_eq lcppl_nlength not_gr_zero) 
   have 4: "nlength \<sigma>>0 \<longrightarrow> j \<le> nlength (lcppl f g \<sigma> ls) \<longrightarrow>
            nlength (nmap (\<lambda>x. x+ (nnth ls j)) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j))))) > 0 "
     using assms 2 cppl_more[of "(nsubn \<sigma> (nnth ls j) (nnth ls (Suc j)))" f g ] 
       by simp 
        (metis co.enat.exhaust_sel eSuc_enat enat_le_plus_same(2) enat_ord_simps(1) gen_nlength_def 
         i0_less ileI1 iless_Suc_eq lcppl_nlength nfinite_nlength_enat nidx_expand nidx_less_eq 
         nlength_code nnth_nlast nsubn_nlength_gr_one the_enat.simps)
   show ?thesis 
   using  "3" "4" assms(7) by presburger
   qed
qed 

lemma lcppl_nlength_all_nfinite_alt:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls" 
         "\<not>nfinite \<sigma>" 
         "(\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> f  fproj  g)"
 shows  " (\<forall> j \<le> nlength (lcppl f g \<sigma> ls). nfinite(nnth (lcppl f g \<sigma> ls) j) ) "
proof 
   fix j
   show "j \<le> nlength (lcppl f g \<sigma> ls) \<longrightarrow>  nfinite ( nnth (lcppl f g \<sigma> ls) j) "
   proof -
   have 1: " nlastnfirst (lcppl f g \<sigma> ls)"
       using assms 
       using lcppl_nfusecat_nlastnfirst_alt by blast
   have 3: " j \<le> nlength (lcppl f g \<sigma> ls) \<longrightarrow> 
           (nnth (lcppl f g \<sigma> ls) j) = 
             (nmap (\<lambda>x. x+ (nnth ls j)) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j)))))" 
     using assms lcppl_nnth[of "ls" "j" "f" "g" "\<sigma>" ] by (metis leI nfinite_ntaken ntaken_all)
   have 4: " j \<le> nlength (lcppl f g \<sigma> ls) \<longrightarrow>
            nfinite (nmap (\<lambda>x. x+ (nnth ls j)) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j)))))  "
     using assms 
     by (metis Suc_ile_eq cppl_fprojection linorder_le_cases nfinite_nmap nfinite_ntaken ntaken_all)
   show ?thesis 
   using  "3" "4" assms by presburger
   qed
qed 

lemma lcppl_nlength_all_gr_zero_alt:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls" 
         "\<not>nfinite \<sigma>" 
         "(\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> f  fproj  g)"
 shows  " (\<forall> j \<le> nlength (lcppl f g \<sigma> ls). nlength(nnth (lcppl f g \<sigma> ls) j) > 0) "
proof 
   fix j
   show "j \<le> nlength (lcppl f g \<sigma> ls) \<longrightarrow> 0 < nlength ( nnth (lcppl f g \<sigma> ls) j) "
   proof -
   have 1: " nlastnfirst (lcppl f g \<sigma> ls)"
       using assms 
       using lcppl_nfusecat_nlastnfirst_alt by blast
   have 3: " j \<le> nlength (lcppl f g \<sigma> ls) \<longrightarrow> 
           (nnth (lcppl f g \<sigma> ls) j) = 
             (nmap (\<lambda>x. x+ (nnth ls j)) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j)))))" 
     using assms lcppl_nnth[of "ls" "j" "f" "g" "\<sigma>" ] by (metis leI nfinite_ntaken ntaken_all)
   have 4: " j \<le> nlength (lcppl f g \<sigma> ls) \<longrightarrow>
            nlength (nmap (\<lambda>x. x+ (nnth ls j)) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j))))) > 0 "
     using assms 
     by (metis Suc_ile_eq cppl_more enat_ile linorder_le_cases nfinite_conv_nlength_enat 
         nidx_expand nlength_nmap nsubn_nlength_gr_one)
   show ?thesis 
   using  "3" "4" assms by presburger
   qed
qed   

lemma lcppl_nlast_nnth:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "nfinite ls " 
         "nfinite \<sigma>" 
         "nlast ls  = the_enat(nlength \<sigma>)"
         "(\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " nlength \<sigma> > 0 " 
         " j \<le> nlength (lcppl f g \<sigma> ls) "
 shows  "nlast(nnth (lcppl f g \<sigma> ls) j) = (nnth ls (Suc j))"
proof -
 have 0: "nlength \<sigma> >0 \<longrightarrow> nlength ls >0"
    using assms 
    by (metis enat.distinct(2) enat_the_enat nfinite_nlength_enat nnth_nlast the_enat_0 
        zero_less_iff_neq_zero)
 have 1: "nlength \<sigma>>0 \<longrightarrow> 
          j \<le> nlength (lcppl f g \<sigma> ls)   \<longrightarrow>
           nnth (lcppl f g \<sigma> ls) j = 
            (nmap (\<lambda>x. x+ (nnth ls j)) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j)) )))" 
    using assms lcppl_nnth[of "ls" "j" "f" "g" "\<sigma>"] 
        by (metis "0" co.enat.exhaust_sel iless_Suc_eq lcppl_nlength not_gr_zero)
 have 2: "nlength \<sigma>>0 \<longrightarrow> 
          j \<le> nlength (lcppl f g \<sigma> ls)   \<longrightarrow> (nnth ls (Suc j)) \<le> nlength \<sigma>"  
     using assms 
     by (metis dual_order.eq_iff enat.simps(3) enat_ord_simps(1) enat_the_enat 
         nfinite_conv_nlength_enat nidx_all_le_nlast nnth_beyond not_le_imp_less)
 have 3: "nlength \<sigma>>0 \<longrightarrow> 
          j \<le> nlength (lcppl f g \<sigma> ls)   \<longrightarrow>
          nlast (nmap (\<lambda>x. x+ (nnth ls j)) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j))))) = 
           (nnth ls (Suc j))"
    using assms cppl_nlast_i[of f g \<sigma> ls j]  
      by (metis "0" "2" Suc_ile_eq co.enat.exhaust_sel i0_less iless_Suc_eq lcppl_nlength nidx_expand)
 show ?thesis using "1" "3" assms 
 by presburger
qed  

lemma lcppl_nlast_nnth_alt:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls " 
         "\<not>nfinite \<sigma>" 
         "(\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " j \<le> nlength (lcppl f g \<sigma> ls) "
 shows  "nlast(nnth (lcppl f g \<sigma> ls) j) = (nnth ls (Suc j))"
proof -
 have 1: " 
          j \<le> nlength (lcppl f g \<sigma> ls)   \<longrightarrow>
           nnth (lcppl f g \<sigma> ls) j = 
            (nmap (\<lambda>x. x+ (nnth ls j)) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j)) )))
          " 
    using assms lcppl_nnth[of "ls" "j" "f" "g" "\<sigma>"] by (metis leI nfinite_ntaken ntaken_all)
 have 2: "
          j \<le> nlength (lcppl f g \<sigma> ls)   \<longrightarrow> (nnth ls (Suc j)) \<le> nlength \<sigma>"  
     using assms 
     by (simp add: nfinite_conv_nlength_enat)
 have 3: "j \<le> nlength (lcppl f g \<sigma> ls)   \<longrightarrow>
          nlast (nmap (\<lambda>x. x+ (nnth ls j)) (cppl f g (nsubn \<sigma> (nnth ls j) (nnth ls (Suc j))))) = 
           (nnth ls (Suc j))"
    using assms cppl_nlast_i[of f g \<sigma> ls j]  
    by (metis "2" co.enat.exhaust_sel eSuc_enat ileI1 iless_Suc_eq lcppl_nlength_alt
         nidx_expand nlength_eq_enat_nfiniteD zero_enat_def)
 show ?thesis using "1" "3" assms 
 by presburger
qed  

lemma lcppl_nfusecat_pfilt_fpower_help:
 assumes "nidx  ls"
         "nnth ls 0 = 0 "
         "nfinite ls " 
         "nfinite \<sigma>" 
         "nlast ls  = the_enat(nlength \<sigma>)" 
         "(\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " nlength \<sigma> > 0 " 
 shows   " (\<forall> i<nlength ls. g (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) )"
proof -
 have 1: "(\<forall> i<nlength ls. g (pfilt (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) 
                                 (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) )) "
    using assms cppl_fprojection by blast 
 have 2: "nlength \<sigma> >0 \<longrightarrow> 
           (\<forall> i<nlength ls. 
            (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) =
            (pfilt \<sigma> (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))))))
          "
    using assms by (simp add: lcppl_nnth)
 have 3: "nlength \<sigma> >0 \<longrightarrow> 
           (\<forall> i<nlength ls. 
              nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) =
              nlength (pfilt (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) 
                                 (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) ) 
           ) 
          " 
        by (simp add: "2" pfilt_nlength)
 have 4: "nlength \<sigma> >0 \<longrightarrow> 
           (\<forall> i<nlength ls.
            (\<forall> j\<le> nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)).
               (nnth (pfilt \<sigma> (nmap (\<lambda>x. x+ (nnth ls i)) 
                              (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )))) j) 
                =
                (nnth \<sigma> ((nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) j) +(nnth ls i) ))
            )
          )"  
        using nnth_nmap  pfilt_nmap by (metis "2" nlength_nmap)
 have 5: "nlength \<sigma> >0 \<longrightarrow> 
           (\<forall> i<nlength ls.
            (\<forall> j\<le> nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)).
               (nnth (pfilt (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) 
                                 (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) ) j) =
               (nnth (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) 
                    (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) j)) 
            ))"
          by (simp add: "3" pfilt_nnth)
 have 6: "nlength \<sigma> >0 \<longrightarrow> 
           (\<forall> i<nlength ls.
               (nnth ls (Suc i)) \<le> nlength \<sigma> 
           )"
     using assms 
     by (metis eSuc_enat enat_le_plus_same(2) enat_ord_simps(1) gen_nlength_def ileI1 
         nfinite_conv_nlength_enat nidx_less_eq nlength_code nnth_nlast the_enat.simps)
 have 7: "nlength \<sigma> >0 \<longrightarrow> 
           (\<forall> i<nlength ls.
                (nnth ls i) \<le> (nnth ls (Suc i))
           ) "
       using assms 
       by (metis eSuc_enat ileI1 le_add2 nidx_less_eq plus_1_eq_Suc)
 have 70: "\<And>i j . i<nlength ls \<Longrightarrow> (nnth ls i) < (nnth ls (Suc i)) \<Longrightarrow>
    enat (nnth ls (Suc i)) \<le> nlength \<sigma> \<Longrightarrow>
    (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> (f  fproj  g)  \<Longrightarrow>
    enat j < nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) \<Longrightarrow>
    0 \<le> nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) j \<and> 
    nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) j \<le> (nnth ls (Suc i)) - (nnth ls i) "
          using cppl_bounds by blast
 have 71: "\<And>i.  i<nlength ls \<Longrightarrow> (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> (f  fproj  g)  \<Longrightarrow>
               nnth ls i < nnth ls (Suc i) \<Longrightarrow>
              enat (nnth ls (Suc i)) \<le> nlength \<sigma> \<Longrightarrow>
              nlast (nmap (\<lambda>x. x + nnth ls i) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))) = 
              nnth ls (Suc i)" 
        using cppl_nlast_i by blast        
 have 8: "nlength \<sigma> >0 \<longrightarrow> 
           (\<forall> i<nlength ls.
            (\<forall> j\<le> nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)). 
                (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) j) \<le> 
                (nnth ls (Suc i)) - (nnth ls i)
            ))"
    using 6 assms 70 71 unfolding cppl_fprojection nidx_expand by simp 
    (metis add_diff_cancel_right' dual_order.eq_iff eSuc_enat ileI1 not_le_imp_less ntaken_all 
     ntaken_nlast)        
 have 9: "nlength \<sigma> >0 \<longrightarrow> 
           (\<forall> i<nlength ls.
            (\<forall> j\<le> nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)).
              (nnth (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) 
                    (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) j)) =
              (nnth \<sigma> ((nnth  (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) j) + (nnth ls i)))
            ))
         "   using 6 7 8 
         by (simp add: add.commute nsubn_def1 ntaken_nnth)
  have 10: "nlength \<sigma> >0 \<longrightarrow> 
           (\<forall> i<nlength ls.
            (\<forall> j\<le> nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)).
               (pfilt (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) 
                                 (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) ) =
              (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) 
             ))" 
            using 2 3 4 9 by (simp add: pfilt_expand pfilt_nlength pfilt_nnth) 
  show ?thesis using "1" "10" assms  by fastforce
qed

lemma lcppl_nfusecat_pfilt_fpower_help_alt:
 assumes "nidx  ls"
         "nnth ls 0 = 0 "
         "\<not>nfinite ls " 
         "\<not>nfinite \<sigma>" 
         "(\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
 shows   " (\<forall> i<nlength ls. g (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) )"
proof -
 have 1: "(\<forall> i<nlength ls. g (pfilt (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) 
                                 (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) )) "
    using assms cppl_fprojection by blast 
 have 2: "(\<forall> i<nlength ls. 
            (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) =
            (pfilt \<sigma> (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))))))"
    using assms by (simp add: lcppl_nnth)
 have 3: "(\<forall> i<nlength ls. 
              nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) =
              nlength (pfilt (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) 
                                 (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) ) 
           )" 
        by (simp add: "2" pfilt_nlength)
 have 4: "(\<forall> i<nlength ls.
            (\<forall> j\<le> nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)).
               (nnth (pfilt \<sigma> (nmap (\<lambda>x. x+ (nnth ls i)) 
                                    (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )))) j) 
                =
                (nnth \<sigma> ((nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) j) +(nnth ls i) ))
            )
          )"  
        using nnth_nmap  pfilt_nmap  by (metis "2" nlength_nmap)
 have 5: " (\<forall> i<nlength ls.
            (\<forall> j\<le> nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)).
               (nnth (pfilt (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) 
                                 (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) ) j) =
               (nnth (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) 
                    (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) j)) 
            ))"
          by (simp add: "3" pfilt_nnth)
 have 6: " (\<forall> i<nlength ls.
               (nnth ls (Suc i)) \<le> nlength \<sigma> 
           )"
     using assms 
     by (simp add: nfinite_conv_nlength_enat)
 have 7: " (\<forall> i<nlength ls.
                (nnth ls i) \<le> (nnth ls (Suc i))
           ) "
       using assms 
       by (metis eSuc_enat ileI1 le_add2 nidx_less_eq plus_1_eq_Suc)
 have 70: "\<And>i j . i<nlength ls \<Longrightarrow> (nnth ls i) < (nnth ls (Suc i)) \<Longrightarrow>
    enat (nnth ls (Suc i)) \<le> nlength \<sigma> \<Longrightarrow>
    (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> (f  fproj  g)  \<Longrightarrow>
    enat j < nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) \<Longrightarrow>
    0 \<le> nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) j \<and> 
    nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) j \<le> (nnth ls (Suc i)) - (nnth ls i) "
          using cppl_bounds by blast
 have 71: "\<And>i.  i<nlength ls \<Longrightarrow> (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> (f  fproj  g)  \<Longrightarrow>
               nnth ls i < nnth ls (Suc i) \<Longrightarrow>
              enat (nnth ls (Suc i)) \<le> nlength \<sigma> \<Longrightarrow>
              nlast (nmap (\<lambda>x. x + nnth ls i) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))) = 
              nnth ls (Suc i)" 
        using cppl_nlast_i by blast 
 have 8: "(\<forall> i<nlength ls.
            (\<forall> j\<le> nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)). 
                (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) j) \<le> 
                (nnth ls (Suc i)) - (nnth ls i)
            ))"
        using assms using 6 assms 70 71 unfolding cppl_fprojection nidx_expand by simp
         (metis add_diff_cancel_right' dual_order.eq_iff eSuc_enat ileI1 not_le_imp_less ntaken_all 
          ntaken_nlast)    
 have 9: "(\<forall> i<nlength ls.
            (\<forall> j\<le> nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)).
              (nnth (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) 
                    (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) j)) =
              (nnth \<sigma> ((nnth  (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) j) + (nnth ls i)))
            ))"   
      using 6 7 8 by (simp add: add.commute nsubn_def1 ntaken_nnth)
  have 10: " (\<forall> i<nlength ls.
            (\<forall> j\<le> nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)).
               (pfilt (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) 
                                 (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) ) =
              (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) 
             ))" 
           using 2 3 4 9  by (simp add: pfilt_expand pfilt_nlength pfilt_nnth) 
  show ?thesis using "1" "10" assms  by fastforce
qed


subsubsection \<open>lsum lemmas\<close>


lemma lsum_NNil:
 "lsum (NNil nell)  a = NNil (a+(the_enat (nlength nell)))"
by simp


lemma lsum_addzero_NNil:
assumes "nfinite nell" 
shows " addzero (lsum (NNil nell) 0) =  
       (if nlength nell = 0 then (NNil 0) else  (NCons 0  (NNil (the_enat (nlength nell))) ))"
using assms unfolding addzero_def 
by simp 
 (metis less_nat_zero_code ndropn_0 ndropn_nfirst ndropn_nlast nlength_NNil nnth_NNil)

lemma lsum_eq_NNil_conv: 
 " (lsum nells a) = (NNil b) \<longleftrightarrow>   is_NNil nells \<and> a+ (the_enat (nlength(nfirst nells))) =  b" 
by (metis NNil_eq_ntake_iff lsum.disc_iff(1) lsum_code(1) nellist.collapse(1) nellist.disc(1) 
    nellist.inject(1))


lemma lsum_eq_NCons_conv: 
 "lsum nells a = (NCons b nells1) \<longleftrightarrow> 
   (\<exists> nell nells'. nells = (NCons nell nells') \<and> b = a+(the_enat (nlength nell)) \<and>
               nells1 = lsum nells' (a+(the_enat (nlength nell)))) " 
by (cases nells) (simp_all, blast) 


lemma lsum_addzero_NCons:
 " addzero (lsum (NCons nell nells) 0) = (NCons 0  (lsum (NCons nell nells) 0))"
by (simp add: addzero_def)


lemma lsum_nfirst:
 "nfirst (lsum nells a) = a+(the_enat (nlength(nfirst nells)))"
proof (cases nells)
case (NNil x1)
then show ?thesis by simp (metis NNil_eq_ntake_iff nellist.inject(1))
next
case (NCons x21 nells1)
then show ?thesis by simp (metis NNil_eq_ntake_iff nnth_0 nnth_NNil ntaken_0 ntaken_nlast)
qed


lemma nfinite_lsum_conv_a: 
assumes "nfinite nells"
shows   "nfinite (lsum nells a)" 
using assms
proof (induction nells arbitrary:  a rule:nfinite_induct)
case (NNil y)
then show ?case by simp
next
case (NCons x nells)
then show ?case by simp
qed


lemma nfinite_lsum_conv_b: 
assumes "nfinite (lsum nells a)"
shows   "nfinite nells" 
using assms
proof (induct zs\<equiv>"lsum nells a" arbitrary: a nells rule:nfinite_induct)
case (NNil y)
then show ?case 
by (metis is_NNil_imp_nfinite lsum_eq_NNil_conv)
next
case (NCons x nell)
then show ?case 
by (metis is_NNil_imp_nfinite lsum_code(2) nellist.collapse(2) nellist.sel(5) nfinite_NConsI)
qed


lemma nfinite_lsum_conv:
 "nfinite (lsum nells a) \<longleftrightarrow> nfinite nells" 
using nfinite_lsum_conv_a nfinite_lsum_conv_b by blast

 
lemma lsum_nlength:
 "nlength (lsum nells a) = nlength nells"
proof (cases "nfinite nells")
case True
then show ?thesis 
   proof (induction nells arbitrary: a rule: nfinite_induct)
   case (NNil y)
   then show ?case by simp
   next
   case (NCons x nell)
   then show ?case by simp
   qed
next
case False
then show ?thesis 
   proof (coinduction arbitrary: nells a rule: enat_coinduct)
   case (Eq_enat nells1)
   then show ?case 
      proof -
       have 1: "(nlength (lsum nells1 a) = (0::enat)) = (nlength nells1 = (0::enat))" 
         by (metis Eq_enat nfinite_lsum_conv_b nlength_eq_enat_nfiniteD zero_enat_def)
    show ?thesis 
    by (metis "1" Eq_enat nbutlast_not_nfinite nfinite_lsum_conv_b nlength_nbutlast)
   qed
 qed
qed

  
lemma lsum_addzero_nfirst:
 "nfirst (addzero (lsum nells 0)) = 0"
by (metis addzero_def ndropn_nfirst ndropn_nfuse nellist.disc(1) nellist.disc(2) nfuse_leftneutral 
    nfuse_nappend nlast_NNil nlength_NNil nnth_0 the_enat_0)


lemma lsum_addzero_nlength:
assumes "nfinite(nfirst nells)" 
shows "(nlength nells = 0 \<and> nlength(nfirst nells) = 0 \<longrightarrow> 
   nlength (addzero (lsum nells 0)) =  0)
  \<and>
  (nlength nells = 0 \<and> nlength(nfirst nells) > 0 \<longrightarrow> 
   nlength (addzero (lsum nells 0)) =  1)
  \<and>
  (nlength nells > 0  \<longrightarrow>
   nlength (addzero (lsum nells 0)) =  (nlength nells) +1)
 "
using assms
by (simp add: addzero_def eSuc_plus_1 lsum_nfirst lsum_nlength)
   (metis enat.distinct(2) enat_the_enat nfinite_nlength_enat zero_enat_def)

lemma sum_nnth_help:
assumes  "i>0"
         " i \<le> nlength nells +1"
shows " (\<Sum>k = 0..(i-1). nlength ( nnth (nells) k)) =
        (\<Sum>k = 1..i. nlength ( nnth (nells) (k-1))) "
using assms 
proof 
  (induct i)
  case 0
  then show ?case by blast
  next
  case (Suc i)
  then show ?case 
  proof (cases i)
  case 0
  then show ?thesis by simp
  next
  case (Suc nat)
  then show ?thesis 
      proof -
       have 1: "(0::nat) < (i::nat)" 
        by (simp add: Suc)
       have 2: "enat i \<le> nlength (nells::'a nellist nellist) + (1::enat)"
        using Suc.prems(2) Suc_ile_eq by auto 
       have 3: "(\<Sum>k::nat = 0::nat..i - (1::nat). nlength (nnth nells k)) = 
                (\<Sum>k::nat = 1::nat..i. nlength (nnth nells (k - (1::nat))))" 
        using "1" "2" Suc.hyps by blast
       have 4: "(\<Sum>k::nat = 0::nat..Suc i - (1::nat). nlength (nnth nells k)) = 
                (\<Sum>k::nat = 0::nat..i. nlength (nnth nells k))"
         by simp 
       have 5: "(\<Sum>k::nat = 0::nat..i. nlength (nnth nells k)) = 
                 nlength (nnth nells i) + (\<Sum>k::nat = 0::nat..(i-1). nlength (nnth nells k))" 
        by (simp add: Suc)
       have 6: " nlength (nnth nells i) + (\<Sum>k::nat = 1::nat..i. nlength (nnth nells (k - (1::nat)))) =
                 (\<Sum>k::nat = 1::nat..Suc i. nlength (nnth nells (k - (1::nat))))" 
               by simp
       show ?thesis 
       using "3" "4" "5" "6" by presburger
     qed
  qed
qed    


lemma lsum_nnth:
 assumes "i\<le> nlength nells" 
         " all_nfinite nells" 
 shows "nnth (lsum nells a) i = a+(\<Sum>k::nat= 0..i. (the_enat (nlength(nnth nells k))))"
using assms
proof 
 (induct i arbitrary: a nells)
 case 0
 then show ?case 
   proof (cases nells)
   case (NNil x1)
   then show ?thesis by (simp add: nnth_NNil)
   next
   case (NCons x21 x22)
   then show ?thesis by simp      
   qed
 next
 case (Suc i)
 then show ?case 
    proof (cases nells)
    case (NNil x1)
    then show ?thesis using Suc.prems enat_0_iff(1) by simp 
    next
    case (NCons x21 x22)
    then show ?thesis 
      proof -
       have 1: "nnth (lsum nells a) (Suc i) = nnth (lsum x22 (a + (the_enat (nlength x21)))) i" 
         by (simp add: NCons)
       have 2: " enat (i::nat) \<le> nlength x22"
           using NCons Suc.prems Suc_ile_eq by auto 
       have 20: "all_nfinite x22"
          by (simp add: NCons Suc.prems(2))
       have 3: " nnth (lsum x22 (a + (the_enat (nlength x21)))) i = 
                (a + (the_enat (nlength x21))) + 
                  (\<Sum>k::nat = 0::nat..i. (the_enat (nlength (nnth x22 k))))" 
          using Suc.hyps[of x22 "(a + (the_enat (nlength x21)))"] 2  20 by blast
       have 4: "a + (\<Sum>k::nat = 0::nat..Suc i. (the_enat (nlength (nnth nells k)))) = 
                a + (the_enat ((nlength (nnth nells 0)))) +  
                    (\<Sum>k::nat = 1::nat..Suc i. (the_enat (nlength (nnth nells k))))" 
           by (simp add: sum.atLeast_Suc_atMost)
       have 5: "(nlength (nnth nells 0)) = nlength x21"
         by (simp add: NCons) 
       have 6: "(\<Sum>k::nat = 1::nat..Suc i. (the_enat (nlength (nnth nells k)))) =
                (\<Sum>k::nat = 0::nat.. i. (the_enat (nlength (nnth nells (k+1)))))"
             using sum.shift_bounds_cl_nat_ivl[of "\<lambda>k. (the_enat (nlength (nnth nells k)))" 0 1 i  ] 
            by simp 
       have 7: "(\<Sum>k::nat = 0::nat.. i. (the_enat (nlength (nnth nells (k+1))))) = 
                (\<Sum>k::nat = 0::nat.. i. (the_enat (nlength (nnth x22 (k)))))"
          using NCons by auto   
       show ?thesis 
       using "1" "3" "4" "5" "6" "7" by presburger
     qed      
   qed
qed

lemma lsum_addzero_nnth:
 assumes "i\<le> nlength (addzero (lsum nells 0))" 
         "nfinite (nfirst nells)" 
 shows "(nlength nells = 0 \<and> nlength(nfirst nells) = 0 \<longrightarrow>
          nnth (addzero (lsum nells 0)) i = (nnth (lsum nells 0) i) )
        \<and>
        (nlength nells = 0 \<and> nlength(nfirst nells) > 0 \<longrightarrow>
          nnth (addzero (lsum nells 0)) i = (nnth (NCons 0 (lsum nells 0))) i) 
        \<and>
        (nlength nells > 0 \<longrightarrow>
          nnth (addzero (lsum nells 0)) i = (nnth (NCons 0 (lsum nells 0)) i) )
       "
using assms using lsum_addzero_nlength[of nells ]  lsum_nlength[of nells 0] 
lsum_nfirst[of nells 0] using addzero_def by auto 


lemma lsum_nlast:
assumes "nfinite nells" 
        "all_nfinite nells" 
shows " nlast (lsum nells a) = a+(\<Sum>k::nat= 0..(the_enat (nlength nells)). (the_enat (nlength(nnth nells k))))"
using assms 
by (metis (no_types, lifting) enat.simps(3) enat_le_plus_same(2) enat_the_enat gen_nlength_def 
    lsum_nlength lsum_nnth nfinite_lsum_conv_a nfinite_nlength_enat nlength_code nnth_nlast sum.cong)


lemma lsum_addzero_nlast:
assumes "nfinite nells" 
shows "nlast (addzero (lsum nells 0)) = nlast(lsum nells 0)"
by (simp add: addzero_def)

lemma lsum_nnth_nfinite:
 assumes "i \<le> nlength nells"
         "all_gr_zero nells" 
         "all_nfinite nells" 
 shows "(\<Sum>k::nat= 0..i. (the_enat (nlength(nnth nells k)))) < \<infinity>" 
using assms 
using enat_ord_code(4) by blast


lemma sum_finite: 
 assumes "all_nfinite nells" 
         " i\<le> nlength nells"
 shows   "(\<Sum>k= 0..i. (nlength(nnth nells k))) < \<infinity> " 
using assms
proof (induction i arbitrary: nells)
case 0
then show ?case 
  proof (cases nells)
  case (NNil x1)
  then show ?thesis using 0 by (simp add: nfinite_nlength_enat nnth_NNil)
  next
  case (NCons x21 x22)
  then show ?thesis using 0 using nfinite_nlength_enat by simp blast
  qed
next
case (Suc i)
then show ?case 
   proof (cases nells)
   case (NNil x1)
   then show ?thesis using Suc by simp (metis enat.inject old.nat.distinct(2) zero_enat_def)
   next
   case (NCons x21 x22)
   then show ?thesis 
     proof -
      have 1: "(\<Sum>k = 0..Suc i. nlength (nnth nells k)) =
               nlength (nnth nells 0) + (\<Sum>k = 1..Suc i. nlength (nnth nells k)) " 
        by (metis One_nat_def sum.atLeast0_atMost_Suc_shift sum.atLeast_Suc_atMost_Suc_shift)
      have 2: "(\<Sum>k = 1..Suc i. nlength (nnth nells k)) =
               (\<Sum>k = 0.. i. nlength (nnth nells (k+1)))"
          using sum.shift_bounds_cl_nat_ivl[of "\<lambda>k . (nlength (nnth nells k))" 0 1 "i"]   
            by (metis  One_nat_def Suc_eq_plus1 ) 
      have 3: " (\<Sum>k = 0.. i. nlength (nnth nells (k+1))) = (\<Sum>k = 0.. i. nlength (nnth x22 k)) " 
          using NCons by auto
      have 4: "all_nfinite x22"
        by (simp add: NCons Suc.prems(1))
      have 5: "i\<le> nlength x22" 
        using NCons Suc.prems(2) Suc_ile_eq by auto
      have 6: "(\<Sum>k = 0.. i. nlength (nnth x22 k)) < \<infinity>" 
        using Suc.IH[of x22]  4 5 by blast
      have 7: "nlength (nnth nells 0)< \<infinity>"  
         by (metis Suc.prems(1) all_nfinite_nnth_b enat.distinct(2) enat_ord_simps(4)
              nfinite_nlength_enat zero_enat_def zero_le)
      have 8: "nlength (nnth nells 0) + (\<Sum>k = 0.. i. nlength (nnth x22 k)) < \<infinity>" 
        using "6" "7" by force
      show ?thesis using "1" "2" "3" "8" by presburger
     qed
   qed
qed

lemma sum_the_enat: 
 assumes "all_nfinite nells" 
         " i\<le> nlength nells"
 shows   " (\<Sum>k= 0..i.  the_enat(nlength(nnth nells k))) = the_enat(\<Sum>k= 0..i. (nlength(nnth nells k)))" 
using assms
proof (induction i arbitrary: nells)
case 0
then show ?case 
   proof (cases nells)
   case (NNil x1)
   then show ?thesis using 0 by simp
   next
   case (NCons x21 x22)
   then show ?thesis using 0 by simp 
   qed
next
case (Suc i)
then show ?case 
   proof (cases nells)
   case (NNil x1)
   then show ?thesis using Suc by  simp (metis enat_0_iff(2) old.nat.distinct(2))
   next
   case (NCons x21 x22)
   then show ?thesis 
      proof -
       have 1: "(\<Sum>k = 0..Suc i. the_enat (nlength (nnth nells k))) =
                the_enat(nlength (nnth nells 0)) + (\<Sum>k = 1..Suc i. the_enat (nlength (nnth nells k)))" 
          by (simp add: sum.atLeast_Suc_atMost)       
       have 2: "the_enat(nlength (nnth nells 0)) = the_enat(nlength x21)"
         using NCons by simp
       have 3: "(\<Sum>k = 1..Suc i. the_enat (nlength (nnth nells k))) =
                (\<Sum>k = 0..i. the_enat (nlength (nnth nells (k+1))))"
           using sum.shift_bounds_cl_nat_ivl[of "\<lambda>k .the_enat (nlength (nnth nells k))" 0 1 "i"] 
            by (metis One_nat_def Suc_eq_plus1)
       have 4: "(\<Sum>k = 0..i. the_enat (nlength (nnth nells (k+1)))) =
                (\<Sum>k = 0..i. the_enat (nlength (nnth x22 k)))" 
          using NCons by auto
       have 5: "all_nfinite x22"
         by (simp add: NCons Suc.prems(1)) 
       have 6: "i\<le> nlength x22" 
         using NCons Suc.prems(2) Suc_ile_eq by auto
       have 7: "(\<Sum>k = 0..i. the_enat (nlength (nnth x22 k))) = 
                the_enat(\<Sum>k = 0..i. (nlength (nnth x22 k)))" 
         using Suc.IH[of x22]  5 6 by blast
       have 8: "the_enat(\<Sum>k = 0..i. (nlength (nnth x22 k))) =
                the_enat(\<Sum>k = 0..i. (nlength (nnth nells (k+1))))"
              using NCons by auto
       have 9: "the_enat(\<Sum>k = 0..i. (nlength (nnth nells (k+1)))) =
                the_enat(\<Sum>k = 1..Suc i. (nlength (nnth nells k)))"
           using sum.shift_bounds_cl_nat_ivl[of "\<lambda>k . (nlength (nnth nells k))" 0 1 "i"]   
            by (metis One_nat_def Suc_eq_plus1)
       have 10: "(\<Sum>k = 1..Suc i. (nlength (nnth nells k))) < \<infinity>" 
          proof -
           have 100: "(\<Sum>k = 1..Suc i. (nlength (nnth nells k))) =
                      (\<Sum>k = 0.. i. (nlength (nnth nells (k+1))))"
                  using sum.shift_bounds_cl_nat_ivl[of "\<lambda>k . (nlength (nnth nells k))" 0 1 "i"]   
                  by (metis One_nat_def Suc_eq_plus1)
           have 101: "(\<Sum>k = 0.. i. (nlength (nnth nells (k+1)))) =
                      (\<Sum>k = 0.. i. (nlength (nnth x22 k)))" 
              using NCons by auto
           have 102: "(\<Sum>k = 0.. i. (nlength (nnth x22 k))) < \<infinity>"   
              using sum_finite[of x22 i] "5" "6" by blast  
           show ?thesis 
           using "100" "101" "102" by presburger
          qed
       have 11: "\<exists>m . (enat m) = (\<Sum>k = 1..Suc i. (nlength (nnth nells k)))" 
           by (metis "10" less_infinityE)
        obtain m where 12: "(enat m) = (\<Sum>k = 1..Suc i. (nlength (nnth nells k)))"
           using  11 by blast
       have 13: "\<exists>n. (enat n) = (nlength (nnth nells 0)) " 
            by (metis Suc.prems(1) all_nfinite_nnth_b nfinite_nlength_enat zero_enat_def zero_le)
       obtain n where 14: "(enat n) = (nlength (nnth nells 0))"
          using 13 by blast 
       have 15: "the_enat(nlength (nnth nells 0)) + the_enat(\<Sum>k = 1..Suc i. (nlength (nnth nells k))) =
                 the_enat(nlength (nnth nells 0) + (\<Sum>k = 1..Suc i. (nlength (nnth nells k))))" 
           by (metis "12" "14" plus_enat_simps(1) the_enat.simps)
       have 16: "the_enat(nlength (nnth nells 0)) + the_enat(\<Sum>k = 1..Suc i. (nlength (nnth nells k))) =
                 the_enat(\<Sum>k = 0..Suc i. (nlength (nnth nells k)))"  
           using sum.atLeast_Suc_atMost[of 0 "Suc i" "\<lambda>k. nlength (nnth nells k)" ]  
           by (metis "15" One_nat_def zero_le)     
       show ?thesis 
       using "1" "16" "3" "4" "7" "8" "9" by presburger
     qed     
   qed
qed



lemma lsum_nnth_leq_Suc:
 assumes "i < nlength nells"
         " all_gr_zero nells "
         " all_nfinite nells " 
         "a < \<infinity>" 
 shows " nnth (lsum nells a) i < nnth (lsum nells a) (Suc i)"
proof -
 have 1: "nnth (lsum nells a) i = a+(\<Sum>k::nat= 0..i. (the_enat (nlength(nnth nells k)))) "
     using assms less_imp_le_nat lsum_nnth by (metis order_less_imp_le )
 have 2: "nnth (lsum nells a) (Suc i) = a+(\<Sum>k::nat= 0..(Suc i). (the_enat (nlength(nnth nells k))))" 
   using assms Suc_ile_eq lsum_nnth by blast
 have 3: "(\<Sum>k::nat= 0..(Suc i). (the_enat (nlength(nnth nells k)))) =
          (\<Sum>k::nat= 0..i. (the_enat (nlength(nnth nells k)))) + (the_enat (nlength(nnth nells (Suc i))))  " 
      using sum.atLeast0_atMost_Suc by blast
 have 4: " nlength(nnth nells (Suc i)) > 0"
    using assms by (metis eSuc_enat ileI1 in_nset_conv_nnth)
 have 5: "(\<Sum>k::nat= 0..i. (the_enat (nlength(nnth nells k)))) < \<infinity>" 
   using lsum_nnth_nfinite[of i nells] assms order.order_iff_strict by blast
 have 6: "nlength(nnth nells (Suc i)) < \<infinity>" 
      using assms 
      by (metis all_nfinite_nnth_b  eSuc_enat enat.distinct(2) enat_ord_simps(4) ileI1 nfinite_nlength_enat)
 have 7: "a+ (\<Sum>k::nat= 0..i. (the_enat (nlength(nnth nells k)))) < 
          a+ (\<Sum>k::nat= 0..(Suc i). (the_enat (nlength(nnth nells k))))"
    using "4" "5" assms "6" enat_the_enat zero_enat_def by fastforce
 show ?thesis 
 using "1" "2" "7" by presburger
qed 


lemma lsum_addzero_nnth_leq_Suc:
 assumes "i < nlength(addzero (lsum nells 0))"
         "all_gr_zero nells "
         "all_nfinite nells " 
 shows " nnth (addzero (lsum nells 0)) i < nnth (addzero (lsum nells 0)) (Suc i)"
using assms 
proof (cases i)
case 0
then show ?thesis 
   proof (cases nells)
   case (NNil x1)
   then show ?thesis using 0 assms unfolding addzero_def  by simp 
   (metis enat_0_iff(2) ndropn_nfirst ndropn_nlast nfinite_code(1) nfinite_nlength_enat nlast_NNil 
    nlength_NNil nnth_nlast the_enat.simps)
   next
   case (NCons x21 x22)
   then show ?thesis using 0 assms unfolding addzero_def by simp 
   (metis eSuc_infinity enat_the_enat gr_zeroI ndropn_eq_NNil ndropn_nlast not_eSuc_ilei0 
    zero_enat_def)
   qed 
   next
case (Suc nat)
then show ?thesis
  proof (cases nells)
  case (NNil x1)
  then show ?thesis
  using Suc addzero_def assms(1) enat_0_iff(1) by fastforce
  next
  case (NCons x21 x22)
  then show ?thesis 
    proof -
     have 1: "nat = 0 \<Longrightarrow> ?thesis " 
       using Suc assms unfolding addzero_def by auto 
          (simp add: Suc_ile_eq lsum_nlength lsum_nnth_leq_Suc,
          metis Extended_Nat.eSuc_mono NCons One_nat_def assms(1) assms(2) enat_ord_code(4) 
          lsum_addzero_NCons lsum_nlength lsum_nnth_leq_Suc nlength_NCons one_eSuc one_enat_def 
          zero_enat_def)
     have 2: "nat>0 \<Longrightarrow> ?thesis"
        using Suc assms unfolding addzero_def by auto
         (simp add: Suc_ile_eq lsum_nlength lsum_nnth_leq_Suc,
          metis NCons Suc_ile_eq assms(1) assms(2) enat_ord_code(4) iless_Suc_eq lsum_addzero_NCons 
          lsum_nlength lsum_nnth_leq_Suc nlength_NCons)
     show ?thesis using 1 2 by blast
    qed
   qed
qed


lemma lsum_nidx:
assumes "all_gr_zero nells "
        "all_nfinite nells "  
shows "nidx  (lsum nells a)"
using assms unfolding nidx_expand
by (simp add: Suc_ile_eq lsum_nlength lsum_nnth_leq_Suc)


lemma lsum_addzero_nidx:
assumes "all_gr_zero nells " 
        "all_nfinite nells  " 
shows "nidx  (addzero (lsum nells 0))"
using assms unfolding nidx_expand 
using Suc_ile_eq lsum_addzero_nnth_leq_Suc by blast

lemma pfilt_nfuse_lsum_a:
 assumes "nlast nell  = nfirst nell1"
         "nlength nell > 0 "
         "nlength nell1 > 0 "
         "nfinite nell"
         "nfinite nell1" 
 shows   " pfilt (nfuse nell nell1) (lsum  (NNil nell1) (the_enat (nlength nell))) =
           pfilt nell1 (lsum (NNil nell1) 0) " 
proof -
 have 1: "(pfilt (nfuse nell nell1) (lsum  (NNil nell1) (the_enat (nlength nell)))) =
          nmap (nnth (nfuse nell nell1)) (lsum  (NNil nell1) (the_enat (nlength nell)))" 
    unfolding pfilt_nmap  by simp 
 have 2: "(pfilt (nell1) ( (lsum (NNil nell1) 0))) = nmap (nnth nell1) (lsum (NNil nell1) (0::nat))" 
   unfolding pfilt_nmap by simp 
 have 3: "nlength (nmap (nnth (nfuse nell nell1)) (lsum (NNil nell1) (the_enat (nlength nell)))) =
          nlength (nmap (nnth nell1) (lsum (NNil nell1) (0::nat)))"
     by (simp add: lsum_nlength) 
 have 4: "\<And>j. j \<le> nlength (nmap (nnth nell1) (lsum (NNil nell1) (0::nat))) \<longrightarrow>
            nnth ((nmap (nnth (nfuse nell nell1)) (lsum  (NNil nell1) (the_enat (nlength nell))))) j =
            nnth ((nmap (nnth nell1) (lsum  (NNil nell1) (0::nat)))) j "
      by (metis assms(1) assms(4) lsum_NNil ndropn_nfuse ndropn_nnth nellist.simps(14) plus_nat.add_0)
 have 5: "nmap (nnth (nfuse nell nell1)) (lsum  (NNil nell1) (the_enat (nlength nell))) =
          (nmap (nnth nell1) (lsum (NNil nell1) (0::nat)))" 
     using 3 4 
     nellist_eq_nnth_eq[of "nmap (nnth (nfuse nell nell1)) (lsum (NNil nell1) (the_enat (nlength nell)))" 
          "(nmap (nnth nell1) (lsum (NNil nell1) (0::nat)))"]
        by presburger
 show ?thesis using 5 1 2  
 by force
qed

 
lemma pfilt_nfusecat_lsum_a:
assumes "nlastnfirst (NCons nell nells)"
        "all_gr_zero nells "
        "all_nfinite nells " 
        "nlength nell > 0"
        "nfinite nell" 
shows "(pfilt (nfuse nell (nfusecat nells)) (lsum nells (the_enat (nlength nell)))) =
          (pfilt (nfusecat nells) ( (lsum nells 0)))"
proof -
 have 1: "(pfilt (nfuse nell (nfusecat nells)) (lsum nells (the_enat (nlength nell)))) =
          nmap (nnth (nfuse nell (nfusecat nells))) (lsum nells (the_enat (nlength nell)))" 
   unfolding pfilt_nmap  by simp 
 have 2: "(pfilt (nfusecat nells) ( (lsum nells 0))) = nmap (nnth (nfusecat nells)) (lsum nells (0::nat))" 
   unfolding pfilt_nmap by simp 
 have 3: "nlength (nmap (nnth (nfuse nell (nfusecat nells))) (lsum nells (the_enat (nlength nell)))) =
          nlength (nmap (nnth (nfusecat nells)) (lsum nells (0::nat)))"
    by (simp add: lsum_nlength) 
 have 4: "\<And>j. (enat j) \<le> nlength (nmap (nnth (nfusecat nells)) (lsum nells (0::nat))) \<Longrightarrow>
            nnth ((nmap (nnth (nfuse nell (nfusecat nells))) (lsum nells (the_enat (nlength nell))))) j =
            nnth ((nmap (nnth (nfusecat nells)) (lsum nells (0::nat)))) j "
   proof -
    fix j
    assume a: "(enat j) \<le> nlength (nmap (nnth (nfusecat nells)) (lsum nells (0::nat)))"       
    show   "nnth ((nmap (nnth (nfuse nell (nfusecat nells))) (lsum nells (the_enat (nlength nell))))) j =
            nnth ((nmap (nnth (nfusecat nells)) (lsum nells (0::nat)))) j" 
    proof -
     have 5: "nnth ((nmap (nnth (nfuse nell (nfusecat nells))) (lsum nells (the_enat (nlength nell))))) j = 
              nnth (nfuse nell (nfusecat nells)) (nnth (lsum nells (the_enat (nlength nell))) j)"
        using nnth_nmap[of j  "(lsum nells (the_enat (nlength nell)))" "(nnth (nfuse nell (nfusecat nells)))" ]   
        a "3" by auto
     have 6: "(nnth (lsum nells (the_enat (nlength nell))) j) = 
              the_enat (nlength nell) + (\<Sum>k::nat = 0::nat..j. the_enat (nlength (nnth nells k)))" 
         using lsum_nnth[of j nells "(the_enat (nlength nell))"] 
         by (metis a assms(3) lsum_nlength nlength_nmap)
     have 7: "nnth ((nmap (nnth (nfusecat nells)) (lsum nells (0::nat)))) j =
              nnth (nfusecat nells) (nnth (lsum nells (0::nat)) j)" 
           using nnth_nmap[of j "(lsum nells (0::nat))" "(nnth (nfusecat nells))"] using a by auto
     have 8: "(nnth (lsum nells (0::nat)) j) = (\<Sum>k::nat = 0::nat..j. the_enat (nlength (nnth nells k)))"
           by (metis (no_types, lifting) "6" a add_0 add_diff_cancel_left' assms(3) lsum_nlength 
               lsum_nnth nlength_nmap) 
     have 9: "nlast nell = nfirst (nfusecat nells)" 
          by (metis assms(1) nfirst_nfusecat_nfirst nlastnfirst_LCons)
     have 10: "(the_enat (nlength nell) + (\<Sum>k::nat = 0::nat..j. the_enat (nlength (nnth nells k)))) \<le>
              nlength (nfuse nell (nfusecat nells))"  
         proof (cases "nfinite nells")
         case True
         then show ?thesis
           proof -
             have 101: "j \<le> nlength nells" 
                 by (metis a lsum_nlength nlength_nmap)
             have 11: "nlength (nfusecat nells) = 
                       (\<Sum>i::nat = 0::nat..the_enat (nlength nells). nlength (nnth nells i))" 
              using nfusecat_nlength_nfinite[of nells ] 
                by (metis True assms(1) assms(3) nlastnfirst_LCons)
             have 12: "nlength (nfuse nell (nfusecat nells)) = 
                       nlength nell + (\<Sum>i::nat = 0::nat..the_enat (nlength nells). nlength (nnth nells i))" 
               using nfuse_nlength[of nell "nfusecat nells" ] "11" by presburger
             have 13: "(\<Sum>k::nat = 0::nat..j. the_enat (nlength (nnth nells k))) = 
                       (the_enat (\<Sum>k::nat = 0::nat..j.  (nlength (nnth nells k))))" 
               by (metis "101" assms(3)  sum_the_enat)
             have 14: "j< the_enat (nlength nells) \<Longrightarrow>
                       (\<Sum>k::nat = 0::nat..j.  (nlength (nnth nells k))) \<le> 
                       (\<Sum>i::nat = 0::nat..the_enat (nlength nells). nlength (nnth nells i))" 
                by (metis bot_nat_0.extremum canonically_ordered_monoid_add_class.lessE 
                    enat_le_plus_same(1) sum.ub_add_nat)                   
             have 141: "j= the_enat (nlength nells) \<Longrightarrow>
                        (\<Sum>k::nat = 0::nat..j.  (nlength (nnth nells k))) \<le> 
                        (\<Sum>i::nat = 0::nat..the_enat (nlength nells). nlength (nnth nells i))" 
                by blast
             have 142: "(\<Sum>k::nat = 0::nat..j.  (nlength (nnth nells k))) \<le> 
                        (\<Sum>i::nat = 0::nat..the_enat (nlength nells). nlength (nnth nells i))"
               using 14 141 "101" True nfinite_nlength_enat by fastforce  
             have 143: "enat (the_enat (nlength nell)) = nlength nell" 
                 by (simp add: assms(5) enat_the_enat nfinite_nlength_enat)
             have 144: "enat (the_enat (\<Sum>k = 0..j. nlength (nnth nells k))) =
                        (\<Sum>k = 0..j. nlength (nnth nells k))" 
                 by (metis "11" "142" True assms(1) assms(3) dual_order.refl enat_ord_code(4) 
                     enat_the_enat leD nfinite_nlength_enat nfusecat_nlength_nfinite
                      nlastnfirst_LCons sum_finite)
             have 143: "enat (the_enat (nlength nell) + (\<Sum>k = 0..j. the_enat (nlength (nnth nells k)))) =
                        nlength nell + (\<Sum>k = 0..j. nlength (nnth nells k))" 
                 using 13 
                 by (metis "143" "144" plus_enat_simps(1))
             have 15: "(the_enat (nlength nell) + (\<Sum>k::nat = 0::nat..j. the_enat (nlength (nnth nells k)))) \<le>
                        nlength nell + (\<Sum>i::nat = 0::nat..the_enat (nlength nells). nlength (nnth nells i))" 
                using 142 6 8 13 by (metis "143" add_left_mono)
             show ?thesis 
             using "12" "15" by presburger
           qed
         next
         case False
         then show ?thesis 
             proof -
              have 200: "\<not>nfinite (nfuse nell (nfusecat nells)) "
                using assms by (metis False nfuse_nfinite nfusecat_nfinite nlastnfirst_LCons) 
              show ?thesis 
              by (metis "200" linorder_le_cases nfinite_ntaken ntaken_all)
             qed
         qed
     have 30: "nnth (nfuse nell (nfusecat nells)) (the_enat (nlength nell) + 
               (\<Sum>k::nat = 0::nat..j. the_enat (nlength (nnth nells k)))) =
              nnth (nfusecat nells) (\<Sum>k::nat = 0::nat..j. the_enat (nlength (nnth nells k)))" 
         by (metis "9" assms(5) ndropn_nfuse ndropn_nnth)
     show ?thesis 
     using "30" "5" "6" "7" "8" by presburger
   qed
  qed
  show ?thesis
  by (metis "3" "4" nellist_eq_nnth_eq pfilt_nmap) 
qed

lemma pfilt_nfusecat_lsum:
assumes "nlastnfirst (NCons nell nells)"
        "all_gr_zero nells "
        "all_nfinite nells"
        " nlength nell > 0"
        "nfinite nell" 
shows " (pfilt (nfusecat (NCons nell nells)) (addzero (lsum (NCons nell nells) 0))) = 
        (NCons (nfirst nell) (NCons  (nlast nell)   (pfilt (nfusecat nells) (lsum nells 0))))"
proof -
 have 1: " nfusecat (NCons nell nells) = nfuse nell (nfusecat nells) "
   by simp
 have 2: "(pfilt (nfusecat (NCons nell nells)) (addzero (lsum (NCons nell nells) 0))) =
          (pfilt (nfuse nell (nfusecat nells)) (addzero (lsum (NCons nell nells) 0)))  "
  using "1" by simp
 have 3: "addzero (lsum (NCons nell nells) 0) =   
         (NCons 0 (NCons (the_enat (nlength nell))  (lsum nells (the_enat (nlength nell))))) "
   using lsum_addzero_NCons by auto
 have 4: "(pfilt (nfuse nell (nfusecat nells)) (addzero (lsum (NCons nell nells) 0))) =
          (pfilt (nfuse nell (nfusecat nells)) (NCons 0 (NCons (the_enat (nlength nell))  
                 (lsum nells (the_enat (nlength nell)))))) "
   using "3" by auto 
 have 5: " (pfilt (nfuse nell (nfusecat nells)) (NCons 0 (NCons (the_enat (nlength nell))  
              (lsum nells (the_enat (nlength nell)))))) =
            (NCons (nnth (nfuse nell (nfusecat nells)) 0)  
            (NCons (nnth (nfuse nell (nfusecat nells)) (the_enat (nlength nell))) 
            (pfilt (nfuse nell (nfusecat nells)) (lsum nells (the_enat (nlength nell))))))  "
   by simp
 have 6: "(nnth (nfuse nell (nfusecat nells)) 0) = (nnth nell 0)"
   using assms 
   by (metis nfirst_nfusecat_nfirst nfuse_nnth nlastnfirst_LCons zero_enat_def zero_le)
 have 7: "(nnth (nfuse nell (nfusecat nells)) (the_enat (nlength nell))) = 
          (nnth nell (the_enat (nlength nell))) "
   using assms 
   by (metis ndropn_nfirst ndropn_nfuse nfirst_nfusecat_nfirst nlastnfirst_LCons nnth_nlast) 
 have 8: "nidx (addzero (lsum  nells 0))"
   using assms lsum_addzero_nidx by blast  
 have 9: "(pfilt (nfuse nell (nfusecat nells)) (lsum nells (the_enat (nlength nell)))) =
          (pfilt (nfusecat nells) (lsum nells 0))"
     using assms pfilt_nfusecat_lsum_a by blast
   show ?thesis 
   using "3" "6" "7" "9" 
   by (metis "2" "5" assms(5) nlast_NNil nnth_nlast ntaken_0 ntaken_nlast)
qed   

lemma pfilt_nfusecat_lsum_1:
assumes "nlastnfirst (NCons nell  nells)" 
        "all_gr_zero nells  "
        "all_nfinite nells  "
        " nlength nell > 0"
        "nfinite nell" 
shows " (pfilt (nfusecat (NCons nell nells)) ((lsum (NCons nell nells) 0))) = 
         (NCons (nlast nell)  (pfilt (nfusecat nells) (lsum nells 0)))"
using assms 
pfilt_nfusecat_lsum[of nell nells]
by (metis lsum_addzero_NCons nellist.inject(2) pfilt_code(2))

lemma pfilt_nfusecat_lsum_2:
assumes "nlastnfirst (nells)"
        "all_gr_zero nells "
        "all_nfinite nells "
        " j \<le> nlength nells "
shows   " (nnth (pfilt (nfusecat (nells)) ((lsum (nells) 0))) j) = nlast(nnth nells j)"
proof -
 have 1: "(pfilt (nfusecat (nells)) ((lsum (nells) 0))) = 
          nmap (nnth (nfusecat nells)) (lsum nells (0::nat)) " 
   using pfilt_nmap[of "(nfusecat (nells))" "(lsum (nells) 0)"]  by simp
 have 2: "(nnth (pfilt (nfusecat (nells)) ((lsum (nells) 0))) j) = 
           nnth (nfusecat nells) (nnth (lsum nells (0::nat)) j)"
    using 1 nnth_nmap[of j "((lsum (nells) 0))" "(nnth (nfusecat nells))" ] 
    by (simp add: assms(4) lsum_nlength)   
 have 3: "(nnth (lsum nells (0::nat)) j) = (\<Sum>k::nat = 0::nat..j. the_enat (nlength (nnth nells k)))" 
    by (metis add_0 assms(3) assms(4) lsum_nnth)
 have 4: "(\<Sum>k::nat = 0::nat..j. the_enat (nlength (nnth nells k))) = 
          the_enat ((\<Sum>k::nat = 0::nat..j. (nlength (nnth nells k)))) " 
     by (metis assms(3) assms(4) sum_the_enat)
 have 40: "nfinite (ntaken j nells)"
      by simp
 have 41: "nlastnfirst (ntaken j nells)"
     by (simp add: assms(1) assms(4) nlastnfirst_ntaken)
 have 42: "all_nfinite (ntaken j nells )"
    by (metis assms(3)  nset_ntaken subset_iff) 
 have 5: "nnth (nfusecat nells) (the_enat ((\<Sum>k::nat = 0::nat..j. (nlength (nnth nells k))))) =
           nlast(nnth nells j)" 
   using nlastfirst_nfusecat_nlast[of "ntaken j nells"  ] nfusecat_ntake[of j nells]  assms
      nlastnfirst_ntaken[of j nells] ntake_eq_ntaken ntaken_nlast[of j nells] 
     by (metis "3" "4" "40" "42"  enat_ord_simps(4) enat_the_enat ntaken_nlast sum_finite)
 show ?thesis 
 using "2" "3" "4" "5" by presburger
qed



lemma pfilt_nfusecat_lsum_3:
assumes "nlastnfirst (nells)"
        "all_gr_zero nells "
        "all_nfinite nells "
        " j \<le> nlength (addzero (lsum nells 0)) " 
shows   " ( j=0 \<longrightarrow> (nnth (pfilt (nfusecat (nells)) (addzero(lsum (nells) 0))) j) = nfirst(nfirst nells))
          \<and>
          (j>0 \<longrightarrow> (nnth (pfilt (nfusecat (nells)) (addzero(lsum (nells) 0))) j) = nlast(nnth nells (j-1)))
        "
using assms 
proof -
 have 1: "j \<le> nlength (addzero (lsum nells 0)) \<and> j=0 \<longrightarrow>
           (nnth (pfilt (nfusecat (nells)) (addzero(lsum (nells) 0))) j) = nfirst(nfirst nells)  " 
     using assms 
     by (metis lsum_addzero_nfirst nfinite_ntaken nfirst_nfusecat_nfirst nnth_NNil nnth_nlast 
         ntaken_0 ntaken_nlast pfilt_nlength pfilt_nnth)
 have 2: "j \<le> nlength (addzero (lsum nells 0)) \<and> j>0 \<longrightarrow>
        (nnth (pfilt (nfusecat nells) (addzero(lsum (nells) 0))) j) = 
         (nnth (nfusecat nells) (nnth  (addzero(lsum (nells) 0)) j))" 
    by (simp add: pfilt_nlength pfilt_nnth)
 have 3: "j \<le> nlength (addzero (lsum nells 0)) \<and> j>0 \<longrightarrow> 
           nnth  (addzero(lsum (nells) 0)) j = nnth (lsum nells 0) (j-1)"    
    by (metis Suc_diff_1 addzero_def enat_0_iff(1) le_zero_eq nat_less_le nnth_Suc_NCons)
 have 20: "j \<le> nlength (addzero (lsum nells 0)) \<and> j>0 \<longrightarrow> enat ((j::nat) - (1::nat)) \<le> nlength nells" 
    by (metis Suc_diff_1 Suc_ile_eq addzero_def dual_order.strict_trans1 enat_0_iff(1) iless_Suc_eq 
        lsum_nlength nlength_NCons not_gr_zero)
 have 4: "j \<le> nlength (addzero (lsum nells 0)) \<and> j>0 \<longrightarrow> 
          (nnth (nfusecat nells) (nnth (lsum nells 0) (j-1))) = nlast(nnth nells (j -1)) "
    using assms 20 pfilt_nfusecat_lsum_2[of nells "j-1" ] 
    by (metis lsum_nlength pfilt_expand)
 show ?thesis 
 using "1" "2" "3" "4" 
 by (simp add: assms(4))
qed

  
lemma pfilt_nfusecat_lsum_4:
 assumes "nlastnfirst nells"
          "all_gr_zero nells "
          "all_nfinite nells "
          "nfinite nells" 
 shows   " (nnth (addzero (lsum nells 0)) (the_enat (nlength (addzero (lsum nells 0))))) \<le> nlength (nfusecat nells)"
proof - 
 have 2: "nlength (nfusecat nells) = (\<Sum>i = 0..the_enat (nlength nells). nlength (nnth nells i))" 
     using assms nfusecat_nlength_nfinite by blast
 have 3: "nlast (lsum nells 0) = 
          (\<Sum>k = 0..the_enat (nlength nells). the_enat (nlength (nnth nells k)))  "
    by (simp add: assms(3) assms(4) lsum_nlast) 
 have 4: "(\<Sum>k = 0..the_enat (nlength nells). the_enat (nlength (nnth nells k))) =
          the_enat(\<Sum>k = 0..the_enat (nlength nells).  (nlength (nnth nells k))) " 
     by (simp add: assms(3) assms(4) enat_the_enat nfinite_nlength_enat sum_the_enat)
 have 5: "enat (\<Sum>k = 0..the_enat (nlength nells). the_enat (nlength (nnth nells k))) \<le> 
          (\<Sum>i = 0..the_enat (nlength nells). nlength (nnth nells i))"
    using assms by (metis "4" dual_order.refl enat_ord_code(3) enat_the_enat) 
 have 6: "nlast (addzero (lsum nells 0)) \<le> nlength (nfusecat nells)"
     unfolding addzero_def using  3 2 5 
     by simp 
 have 7: "nlast (addzero (lsum nells 0)) = 
          (nnth (addzero (lsum nells 0)) (the_enat (nlength (addzero (lsum nells 0)))))" 
   by (simp add: addzero_def assms(4) nfinite_lsum_conv_a nnth_nlast)
 show ?thesis using assms 6 7 by auto    
qed

lemma nfusecat_nlength_b:
 assumes "nlastnfirst nells"
         "all_nfinite nells " 
         " 1\<le> i"
         " i\<le> nlength nells" 
         " j \<le> nlength(nnth nells i) " 
         "nfinite nells" 
         " all_gr_zero nells" 
  shows  "(nnth ((lsum nells 0) ) (i-1) + j \<le> nlength (nfusecat nells)) "
proof -
 have 0: " i-1 \<le> nlength nells "
      by (metis Suc_ile_eq assms(3) assms(4) le_add_diff_inverse order_less_imp_le plus_1_eq_Suc)
 have 1: "( nnth ((lsum nells 0) ) (i-1) =  (\<Sum>k::nat= 0..(i-1). the_enat(nlength(nnth nells k)))) "
       using 0 lsum_nnth[of "i-1"  nells 0 ] assms by presburger
 have 2: "nlength (nfusecat nells) = (\<Sum>k::nat= 0..(the_enat(nlength nells)). nlength(nnth nells k))" 
     using assms nfusecat_nlength_nfinite by metis
 have 20: "(\<Sum>k::nat= 0..(i). the_enat(nlength(nnth nells k))) = 
           (\<Sum>k = 0..i - 1. the_enat (nlength (nnth nells k))) + 
           the_enat (nlength (nnth nells (Suc (i - 1))))"
    using assms  sum.atLeast0_atMost_Suc[of "\<lambda>k. the_enat(nlength(nnth nells k))" "i-1" ]  by simp
 have 21: "(\<Sum>k = 0..i - 1. the_enat (nlength (nnth nells k))) < \<infinity>" 
    using enat_ord_code(4) by blast
 have 22: "the_enat (nlength (nnth nells (Suc (i - 1)))) < \<infinity>" 
     by auto
 have 23: "(\<Sum>k::nat= 0..(i). the_enat(nlength(nnth nells k))) < \<infinity>" 
    using enat_ord_simps(4) by blast
 have 3: "(\<Sum>k::nat= 0..(i-1). the_enat(nlength(nnth nells k))) + j \<le> 
               (\<Sum>k::nat= 0..(i). the_enat(nlength(nnth nells k))) " 
   using 20 assms by simp 
    (metis (no_types, lifting)  all_nfinite_nnth_b enat_ord_simps(1) enat_the_enat infinity_ileE 
        ndropn_eq_NNil ndropn_nlast)
 have 4: "(\<Sum>k::nat= 0..(i). the_enat(nlength(nnth nells k))) \<le> 
             (\<Sum>k::nat= 0..(the_enat(nlength nells)). the_enat(nlength(nnth nells k)))" 
        using sum.ub_add_nat[of 0 i  "\<lambda> k. the_enat(nlength(nnth nells k))" "(the_enat(nlength nells))-i" ] 
         using assms(4) assms(6) nfinite_nlength_enat by fastforce 
 have 5: "(\<Sum>k::nat= 0..(the_enat(nlength nells)). the_enat(nlength(nnth nells k))) \<le> 
          (\<Sum>k::nat= 0..(the_enat(nlength nells)). nlength(nnth nells k))"   
     using sum_the_enat[of nells "(the_enat(nlength nells))"] assms
     by (metis  leI less_enatE order_less_imp_not_eq the_enat.simps)   
 show ?thesis 
 using "1" "2" "3" "4" 5  by simp 
    (meson dual_order.trans enat_ord_simps(1))
qed

lemma nfusecat_nlength_b_alt:
 assumes "nlastnfirst nells"
         "all_nfinite nells " 
         " 1\<le> i"
         " i\<le> nlength nells" 
         " j \<le> nlength(nnth nells i) " 
         "\<not>nfinite nells" 
         "all_gr_zero nells " 
  shows  "nnth ((lsum nells 0) ) (i-1) + j \<le> nlength (nfusecat nells) "
proof -
 have 1: "\<not>nfinite (nfusecat nells)"
  using assms by (metis  nfusecat_nfinite) 
 show ?thesis 
 by (meson "1" enat_ile linorder_le_cases nfinite_conv_nlength_enat)
qed

lemma pfilt_nfusecat_lsum_5:
 assumes "nlastnfirst nells"
         "all_gr_zero nells "
         "all_nfinite nells "
         " (enat i) \<le> nlength (addzero (lsum nells 0)) "
         "nfinite nells" 
 shows   " (nnth (addzero (lsum nells 0)) i) \<le> nlength (nfusecat nells)"
using assms 
proof -
 have 2: "nlength (nfusecat nells) = (\<Sum>i = 0..the_enat (nlength nells). nlength (nnth nells i))" 
    using  assms nfusecat_nlength_nfinite by blast
 have 3: " nlength nells > 0 \<Longrightarrow> nlength (lsum nells 0) >0 " 
     by (simp add: lsum_nlength)
 have 4: "nlength nells > 0 \<Longrightarrow> (nnth (addzero (lsum nells 0)) i) = nnth (NCons 0 (lsum nells 0)) i "
   using assms unfolding addzero_def using 3 by simp 
 have 5: "nlength nells > 0 \<and> i=0 \<Longrightarrow> (nnth (addzero (lsum nells 0)) i) = 0" 
     using "4" by auto
 have 6: "nlength nells > 0 \<and> i>0 \<Longrightarrow> (nnth (addzero (lsum nells 0)) i) = nnth (lsum nells 0) (i -1) " 
     by (metis "4" Suc_diff_1 nnth_Suc_NCons)
 have 7: "nlength nells > 0 \<and> i>0 \<Longrightarrow> nnth (lsum nells 0) (i -1) =
            (\<Sum>k = 0..(i-1). the_enat (nlength (nnth nells k)))"
      using assms lsum_nnth[of "i-1" nells 0] 
      by (metis Suc_diff_1 Suc_ile_eq add_0 addzero_def i0_less iless_Suc_eq lsum_nlength 
          nlength_NCons) 
 have 8: "nlength nells > 0 \<and> i>0 \<Longrightarrow> 
         (\<Sum>k = 0..(i-1). the_enat (nlength (nnth nells k))) =
          the_enat(\<Sum>k = 0..(i-1).  (nlength (nnth nells k))) " 
      using assms sum_the_enat[of nells "i-1"  ] 
      by (metis Suc_diff_1 Suc_ile_eq addzero_def i0_less iless_Suc_eq lsum_nlength nlength_NCons)   
 have 9: "nlength nells > 0 \<and> i>0 \<Longrightarrow> nnth (lsum nells 0) (i -1) \<le>  nlength (nfusecat nells)"
      using assms nfusecat_nlength_b[of nells i 0] pfilt_nfusecat_lsum_4[of nells] 
       3 4 6 lsum_nlength[of nells] unfolding addzero_def by simp 
       (metis iless_Suc_eq order.order_iff_strict the_enat.simps zero_enat_def zero_le)
 have 10: " nlength nells = 0 \<and> i = 0 \<Longrightarrow>  nnth (NCons 0 (lsum nells 0)) i \<le> nlength (nfusecat nells) " 
     using assms zero_enat_def by auto
 have 11: " nlength nells = 0 \<and> i = 1 \<Longrightarrow>  nnth (NCons 0 (lsum nells 0)) i \<le> nlength (nfusecat nells)" 
   using assms 
   by (metis addzero_def lsum_nlength nlength_NCons not_one_le_zero one_eSuc one_enat_def 
       pfilt_nfusecat_lsum_4 the_enat.simps)
 have 12: "nfinite (nfirst nells) "
      by (metis assms(3) assms(5) nconcat_expand nfinite_nappend nfinite_nconcat)
 have 13: "nlength (nfirst nells) > 0 "
   using assms 
   by (metis in_nset_conv_nnth nlast_NNil ntaken_0 ntaken_nlast zero_enat_def zero_le) 
 have 14: "nlength nells = 0  \<Longrightarrow> i \<le> 1"
     using assms  lsum_addzero_nlength[of nells] 
     by (metis "12" "13" enat_ord_simps(1) one_enat_def)      
 have 15: "nnth (NCons 0 (lsum nells 0)) i \<le> nlength (nfusecat nells)" 
    by (metis "10" "11" "14" "4" "5" "6" "9" One_nat_def Suc_leI dual_order.antisym not_gr_zero 
        zero_enat_def zero_le)  
 show ?thesis 
 by (metis "12" "13" "15" assms(4) gr_zeroI lsum_addzero_nnth)
qed

lemma pfilt_nfusecat_lsum_5_alt:
 assumes "nlastnfirst (nells)"
         "all_gr_zero nells "
         "all_nfinite nells "
         " (enat i) \<le> nlength (addzero (lsum nells 0)) "
         "\<not>nfinite nells" 
 shows   " (nnth (addzero (lsum nells 0)) i) \<le> nlength (nfusecat nells)"
using assms 
by (metis enat_ile linorder_le_cases nfinite_conv_nlength_enat nfusecat_nfinite)


lemma lsum_shift:
 assumes "nlastnfirst nells"
         "all_gr_zero nells "
         "all_nfinite nells "
         " i\<le>nlength nells "  
 shows   "  nnth (lsum nells a) i = a+ nnth (lsum nells 0) i"
using assms by (simp add: lsum_nnth)


lemma lsum_nfusecat_nnth_lsum_nnth:
assumes "nlastnfirst nells"
        "all_gr_zero nells "
        "all_nfinite nells"
        "i\<le>nlength nells" 
        " j\<le> nlength(nnth nells i)" 
shows    "  (nnth (nfusecat nells) ((nnth (addzero (lsum nells 0)) i) +j) ) = (nnth (nnth nells i) j)  "
using assms
proof (induction i arbitrary: nells j)
case 0
then show ?case 
   proof (cases nells)
   case (NNil x1)
   then show ?thesis using 0 by simp 
    (metis add_cancel_right_left addzero_def ndropn_nfirst ndropn_nlast nfinite_NNil nlast_NNil 
     nnth_0 nnth_NNil)
   next
   case (NCons x21 x22)
   then show ?thesis using 0 by simp
    (metis "0.prems"(1) "0.prems"(2) "0.prems"(3) add_cancel_right_left addzero_def eSuc_ne_0 
       nfirst_nfusecat_nfirst nfuse_nnth nfusecat_NCons nfusecat_nlength_a nlength_NCons nnth_0) 
   qed
next
case (Suc i)
then show ?case 
   proof (cases nells)
   case (NNil x1)
   then show ?thesis using Suc by (simp add: zero_enat_def)
   next
   case (NCons x21 x22)
   then show ?thesis 
     proof -
      have 0: "(0::enat) < nlength nells" 
         by (simp add: NCons)
      have 1: "nlastnfirst x22"
       using NCons Suc.prems(1) by auto 
      have 2: "all_gr_zero x22"
        using NCons Suc.prems(2) by auto
      have 3: "all_nfinite x22 "
        using NCons Suc.prems(3) by auto
      have 4: "  enat (i::nat) \<le> nlength x22"
        using NCons Suc.prems(4) Suc_ile_eq by auto 
      have 5: "nnth (nfusecat nells) (nnth (addzero (lsum nells (0::nat))) (Suc i) + j) =
               nnth (nfuse x21 (nfusecat x22)) (nnth (addzero (lsum nells (0::nat))) (Suc i) + j) " 
        by (simp add: NCons)
      have 6: " nlength (addzero (lsum nells (0::nat))) = nlength nells + (1::enat) "
         using lsum_addzero_nlength[of nells ] 
         by (metis NCons lsum_addzero_NCons lsum_nlength nlength_NCons plus_1_eSuc(2))
      have 7: "enat (Suc (i::nat) ) \<le> nlength (addzero (lsum (nells::'a nellist nellist) (0::nat)))" 
          by (simp add: "6" Suc.prems(4) order_less_imp_le plus_1_eSuc(2))
      have 8: "(nnth (addzero (lsum nells (0::nat))) (Suc i)) = 
                nnth (NCons (0::nat) (lsum nells (0::nat))) (Suc i)" 
        using lsum_addzero_nnth[of "(Suc i)" nells ] by (metis NCons lsum_addzero_NCons)
      have 9: "nnth (NCons (0::nat) (lsum nells (0::nat))) (Suc i) = nnth (lsum nells 0) i"
        by simp 
      have 10: "nnth (lsum nells 0) i  = (\<Sum>k::nat = 0::nat..i. the_enat (nlength (nnth nells k)))"
        using lsum_nnth[of i nells 0] 
        by (metis Suc.prems(3) Suc.prems(4) Suc_ile_eq add_0 order_less_imp_le)  
      have 11: "nlength x21 \<le> (\<Sum>k::nat = 0::nat..i. the_enat (nlength (nnth nells k))) + j " 
        proof (cases i)
        case 0
        then show ?thesis using NCons by simp 
        (metis Suc.prems(3) enat_ord_simps(1) enat_the_enat infinity_ileE le_add1 ndropn_eq_NNil 
         ndropn_nlast nellist.set_intros(2))
        next
        case (Suc nat)
        then show ?thesis 
            proof -
             have 12: "(\<Sum>k::nat = 0::nat..i. the_enat (nlength (nnth nells k))) =
                       the_enat (nlength (nnth nells 0)) + 
                       (\<Sum>k::nat = 1::nat..i. the_enat (nlength (nnth nells k)))" 
               by (simp add: sum.atLeast_Suc_atMost)
             have 13: " (nlength (nnth nells 0))  = nlength x21"
               by (simp add: NCons) 
             have 14: "nlength x21 \<le> 
                       the_enat (nlength x21) + 
                       (\<Sum>k::nat = 1::nat..i. the_enat (nlength (nnth nells k))) + j" 
                using NCons Suc.prems(3) nfinite_nlength_enat by fastforce
             show ?thesis 
             using "12" "13" "14" by presburger
            qed
       qed
      have 15: "nlast x21 = nfirst (nfusecat x22) "
        by (metis NCons Suc.prems(1) nfirst_nfusecat_nfirst nlastnfirst_LCons) 
      have 16: "enat (nnth (addzero (lsum nells 0)) (Suc i) + j) \<le> nlength (nfuse x21 (nfusecat x22))" 
          by (metis "8" "9" NCons Suc.prems(1) Suc.prems(2) Suc.prems(3) Suc.prems(4) Suc.prems(5) 
              add_diff_cancel_left' le_add1 nfusecat_NCons nfusecat_nlength_b nfusecat_nlength_b_alt 
              plus_1_eq_Suc)      
      have 17: "nlength x21 \<le> enat (nnth (addzero (lsum nells (0::nat))) (Suc i) + j)" 
           using "10" "11" "8" "9" by presburger
      have 18: "nnth (nfuse x21 (nfusecat x22)) (nnth (addzero (lsum nells (0::nat))) (Suc i) + j) = 
                nnth (nfusecat x22) (nnth (addzero (lsum nells (0::nat))) (Suc i) + j - (the_enat(nlength x21)))"
       using nfuse_nnth_var[of "(nnth (addzero (lsum nells (0::nat))) (Suc i) + j)" x21 "nfusecat x22"] 
        using "15" "16" "17" by blast
      have 19: "i=0 \<Longrightarrow> nnth (addzero (lsum nells (0::nat))) (Suc i) =  (the_enat(nlength x21))"
           using "8" NCons by auto
      have 20: "i=0 \<Longrightarrow> nnth (nfusecat x22) 0 = nnth (nnth x22 i) 0"
        using Suc.IH[of x22 0] 
        by (metis "1" "2" "3" "4" add.right_neutral all_gr_zero_nnth_b lsum_addzero_NCons 
            lsum_addzero_nfirst nnth_0 ntaken_0 ntaken_nlast order_less_imp_le zero_enat_def)
      have 21: "i=0 \<Longrightarrow> ?thesis"
         by (metis "1" "18" "19" "2" "3" "4" "5" NCons Suc.IH Suc.prems(5) add.commute add.right_neutral
              add_diff_cancel_left' lsum_addzero_NCons lsum_addzero_nfirst nnth_0 nnth_Suc_NCons ntaken_0 
            ntaken_nlast)
      have 22: "0<i \<Longrightarrow> (\<Sum>k::nat = 0::nat..i. the_enat (nlength (nnth nells k))) = 
                (the_enat(nlength x21)) + (\<Sum>k::nat = 0::nat..(i-1). the_enat (nlength (nnth x22 k)))" 
         proof -
          assume a1: "0<i" 
          have 23: "(\<Sum>k::nat = 0::nat..i. the_enat (nlength (nnth nells k))) = 
                    the_enat(nlength x21) + (\<Sum>k::nat = 1::nat..i. the_enat (nlength (nnth nells k)))" 
             by (simp add: NCons sum.atLeast_Suc_atMost)
          have 24: "(\<Sum>k::nat = 1::nat..i. the_enat (nlength (nnth nells k))) = 
                    (\<Sum>k::nat = 0::nat..(i-1). the_enat (nlength (nnth nells (k+1))))"
            using sum.shift_bounds_cl_nat_ivl[of "\<lambda>k .the_enat (nlength (nnth nells k))" 0 1 "i-1"]
             by (simp add: a1)
          have 25: "(\<Sum>k::nat = 0::nat..(i-1). the_enat (nlength (nnth nells (k+1)))) = 
                    (\<Sum>k::nat = 0::nat..(i-1). the_enat (nlength (nnth x22 (k))))"
           using NCons by auto
          show ?thesis 
          using "23" "24" "25" by presburger 
       qed
     have 26: "0<i \<Longrightarrow> nnth (addzero (lsum nells (0::nat))) (Suc i) - (the_enat(nlength x21)) = 
                       (\<Sum>k::nat = 0::nat..(i-1). the_enat (nlength (nnth x22 (k))))" 
       by (simp add: "10" "22" "8")  
     have 260: "0 < nfirst (lsum x22 0)" 
         proof (cases x22)
         case (NNil x1)
         then show ?thesis by simp 
          (metis "2" "3" NNil_eq_ntake_iff enat_the_enat gr0I infinity_ileE less_numeral_extra(3) 
           ndropn_eq_NNil ndropn_nlast nellist.set_intros(1) nlast_NNil zero_enat_def)
         next
         case (NCons x21 x22)
         then show ?thesis by simp 
          (metis "2" "3"  gr0I  less_numeral_extra(3) ndropn_nfirst ndropn_nfuse ndropn_nlast
           nellist.set_intros(2) nfinite_NNil nfuse_leftneutral nlast_NNil nlength_NNil nnth_0 the_enat_0)
         qed
     have 27: "0<i \<Longrightarrow> (\<Sum>k::nat = 0::nat..(i-1). the_enat (nlength (nnth x22 (k)))) =
                        nnth (addzero (lsum x22 0)) (i)"
           unfolding addzero_def using NCons  26 3 4 8 260 apply simp 
            using lsum_nnth 
            by (metis Suc_ile_eq Suc_pred add_0 nnth_Suc_NCons order_less_imp_le)
     have 28: "0<i \<Longrightarrow> 
               nnth (nfusecat x22) (nnth (addzero (lsum nells (0::nat))) (Suc i) + j - (the_enat(nlength x21))) =
               nnth (nnth x22 i) j"
        using Suc.IH[of x22 j] 
        by (metis "1" "10" "2" "22" "26" "27" "3" "4" "8" NCons Nat.add_diff_assoc2 Suc.prems(5) 
            le_add1 nnth_Suc_NCons)
     have 29: "0<i \<Longrightarrow> ?thesis " 
       using "18" "28" NCons by fastforce
     show ?thesis 
     using "21" "29" by blast
    qed
   qed
qed 


lemma lcppl_lsum_less_th_equal:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "nfinite ls" 
         "nfinite \<sigma>" 
         " nlast ls  = (the_enat (nlength \<sigma>))"
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " nlength \<sigma> > 0 "
         " i < nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) " 
 shows   " (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)) \<le>
           nlength( (nfusecat (lcppl f g \<sigma> ls)))"
proof -
 have 1: "nlastnfirst (lcppl f g \<sigma> ls)" 
    using assms 
    by (metis enat.distinct(2) enat_the_enat i0_less lcppl_nfusecat_nlastnfirst 
        nfinite_conv_nlength_enat nnth_nlast the_enat_0)
 have 2: "all_gr_zero  (lcppl f g \<sigma> ls)  " 
   using assms all_gr_zero_nnth_a lcppl_nlength_all_gr_zero by blast
 have 3: "all_nfinite (lcppl f g \<sigma> ls) " 
  using assms all_nfinite_nnth_a lcppl_nlength_all_nfinite by blast
 have 4: " enat (Suc (i::nat)) \<le> nlength (addzero (lsum (lcppl f g \<sigma> ls) (0::nat))) " 
   using assms using Suc_ile_eq by blast
 have 5: " nfinite (lcppl f g \<sigma> ls) " 
  using assms using lcppl_nfinite by blast
 show ?thesis using 
 pfilt_nfusecat_lsum_5[of "(lcppl f g \<sigma> ls)" "Suc i" ] 
 using "1" "2" "3" "4" "5" by blast
qed

lemma lcppl_lsum_less_th_equal_alt:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls" 
         "\<not>nfinite \<sigma>" 
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " i < nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) " 
 shows   " (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)) \<le>
           nlength( (nfusecat (lcppl f g \<sigma> ls)))"
proof -
 have 1: "nlastnfirst (lcppl f g \<sigma> ls)" 
    using assms 
    using lcppl_nfusecat_nlastnfirst_alt by blast
 have 2: "all_gr_zero  (lcppl f g \<sigma> ls) " 
   using assms all_gr_zero_nnth_a lcppl_nlength_all_gr_zero_alt by blast
 have 3: " all_nfinite  (lcppl f g \<sigma> ls)  " 
  using assms all_nfinite_nnth_a lcppl_nlength_all_nfinite_alt by blast
 have 4: " enat (Suc (i::nat)) \<le> nlength (addzero (lsum (lcppl f g \<sigma> ls) (0::nat))) " 
   using assms using Suc_ile_eq by blast
 show ?thesis 
 using "1" "2" "3" "4" pfilt_nfusecat_lsum_5_alt using assms(1) assms(3) lcppl_nfinite by blast
qed


lemma lcppl_lsum_nlength:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "nfinite ls " 
         "nfinite \<sigma>" 
         " nlast ls = (the_enat (nlength \<sigma>))"
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " nlength \<sigma> >0 " 
 shows   "nlength (addzero (lsum ((lcppl f g \<sigma> ls)) 0)) = nlength ls"
proof -
 have 1: "nlength \<sigma> >0 \<longrightarrow> nlength (lcppl f g \<sigma> ls) = nlength ls -1"
   using assms 
   by (metis epred_0 epred_conv_minus i0_less lcppl_nlength lcppl_nlength_zero)
 have 2: "nlength \<sigma>>0 \<longrightarrow> nlength ls >0"
    using assms 
    by (metis enat.distinct(2) enat_the_enat gr_zeroI nfinite_conv_nlength_enat nnth_nlast the_enat_0)
 have 3: "nlength \<sigma> > 0 \<longrightarrow> nlength (nfirst (lcppl f g \<sigma> ls)) > 0"  
   using assms 
   by (metis lcppl_nlength_all_gr_zero ndropn_0 ndropn_nfirst nfinite_nlength_enat the_enat.simps 
       zero_enat_def zero_le)
 have 30: "nlastnfirst (lcppl f g \<sigma> ls)"
     using "2" assms lcppl_nfusecat_nlastnfirst by blast 
 have 31: "nfinite (nfirst (lcppl f g \<sigma> ls))" 
     using assms 
     by (metis all_nfinite_nnth_a lcppl_nfinite lcppl_nlength_all_nfinite nconcat_expand 
         nfinite_nappend nfinite_nconcat)
 have 4: "nlength \<sigma> >0 \<longrightarrow> nlength (lcppl f g \<sigma> ls) = 0 \<longrightarrow> 
           nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) = nlength ls "
      using 1 2 3 assms lsum_addzero_nlength[of "(lcppl f g \<sigma> ls)" ] 30 31 
      by (metis co.enat.exhaust_sel i0_less lcppl_nlength one_eSuc)
 have 5: "nlength \<sigma> >0 \<longrightarrow> nlength (lcppl f g \<sigma> ls) > 0 \<longrightarrow> 
        nlength (addzero (lsum ((lcppl f g \<sigma> ls)) 0)) = nlength ls" 
    using assms 
    by (metis "2" "4" addzero_def co.enat.exhaust_sel lcppl_nlength lcppl_nlength_zero lsum_nlength 
        nlength_NCons)
  show ?thesis using "4" "5" 
  using assms(7) gr_zeroI by blast
qed

lemma lcppl_lsum_nlength_alt:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls " 
         "\<not>nfinite \<sigma>" 
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
 shows   "nlength (addzero (lsum ((lcppl f g \<sigma> ls)) 0)) = nlength ls"
proof -
 have 1: " nlength (lcppl f g \<sigma> ls) = nlength ls -1"
   using assms 
   by (simp add: epred_conv_minus lcppl_nlength_alt)
 have 2: " nlength ls >0"
    using assms 
    by (simp add: nfinite_conv_nlength_enat)
 have 3: " nlength (nfirst (lcppl f g \<sigma> ls)) > 0"  
   using assms lcppl_nlength_all_gr_zero_alt[of ls \<sigma> f g]
   by (metis ndropn_0 ndropn_nfirst zero_enat_def zero_le) 
 have 4: " nlength (lcppl f g \<sigma> ls) = 0 \<longrightarrow> 
           nlength (addzero (lsum ((lcppl f g \<sigma> ls)) 0)) = nlength ls "
     using "1" "2" "3" assms 
     by (metis lcppl_nfinite nlength_eq_enat_nfiniteD zero_enat_def)
 have 5: " nlength (lcppl f g \<sigma> ls) > 0 \<longrightarrow> 
        nlength (addzero (lsum ((lcppl f g \<sigma> ls)) 0)) = nlength ls" 
    using assms 
    by (metis "4" addzero_def co.enat.exhaust_sel lcppl_nlength_alt lcppl_nlength_zero lsum_nlength 
        nlength_NCons)
  show ?thesis using "4" "5" 
  using assms gr_zeroI by blast
qed

lemma lcppl_lsum_nnth:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "nfinite ls"
         "nfinite \<sigma>" 
         " nlast ls = the_enat(nlength \<sigma>) "
         "(\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> f  fproj  g)"
         " nlength \<sigma> > 0 " 
         " j \<le> nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) "
 shows  " ( j=0 \<longrightarrow> 
          (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) j) =
           nfirst(nfirst (lcppl f g \<sigma> ls)) )
          \<and>
          (j>0 \<longrightarrow> (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) j) =
            nlast(nnth (lcppl f g \<sigma> ls) (j-1)))
         "
proof -
 have 0: "nlength \<sigma> >0 \<longrightarrow> nlength ls >0" 
   using assms 
   by (metis enat.distinct(2) enat_the_enat gr_zeroI nfinite_conv_nlength_enat nnth_nlast the_enat_0)
 have 1: "nlength \<sigma> >0 \<longrightarrow> nlastnfirst (lcppl f g \<sigma> ls)" 
    using "0" assms 
    by (simp add: enat_the_enat lcppl_nfusecat_nlastnfirst nfinite_conv_nlength_enat)
 have 2: "nlength \<sigma>>0 \<longrightarrow> 
         (\<forall> j \<le> nlength (lcppl f g \<sigma> ls). nlength(nnth (lcppl f g \<sigma> ls) j) > 0) "
   using assms 
   by (metis enat.distinct(2) enat_the_enat lcppl_nlength_all_gr_zero nfinite_conv_nlength_enat)
 have 3: " nlength \<sigma>>0 \<longrightarrow> 
           nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) =
           nlength (lcppl f g \<sigma> ls) +1" 
       using assms 
       by (metis "0" co.enat.exhaust_sel i0_less lcppl_lsum_nlength lcppl_nlength plus_1_eSuc(2))
 have 4: " nlength \<sigma> > 0 \<longrightarrow>
           (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) 0) =
             nfirst(nfirst (lcppl f g \<sigma> ls)) "
    by (metis lsum_addzero_nfirst nfirst_nfusecat_nfirst nlast_NNil ntaken_0 ntaken_nlast pfilt_nnth 
        zero_enat_def zero_le)
 have 5: "nlength \<sigma> > 0 \<longrightarrow> 
          j \<le> nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) \<and> j>0 \<longrightarrow>
          (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) (j))
           = nlast(nnth (lcppl f g \<sigma> ls) (j-1))"
     using assms pfilt_nfusecat_lsum_3[of "(lcppl f g \<sigma> ls)" j ] 
     by (metis "1" "2" all_gr_zero_nnth_a all_nfinite_nnth_a lcppl_nlength_all_nfinite)
 from 4 5 show ?thesis using assms 
 by blast
qed   

lemma lcppl_lsum_nnth_alt:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls"
         "\<not>nfinite \<sigma>" 
         "(\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> f  fproj  g)"
         " j \<le> nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) "
 shows  " ( j=0 \<longrightarrow> 
          (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) j) =
           nfirst(nfirst (lcppl f g \<sigma> ls)) )
          \<and>
          (j>0 \<longrightarrow> (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) j) =
            nlast(nnth (lcppl f g \<sigma> ls) (j-1)))"
proof -
 have 1: " nlastnfirst (lcppl f g \<sigma> ls)" 
    using  assms 
    using lcppl_nfusecat_nlastnfirst_alt by blast
 have 2: "(\<forall> j \<le> nlength (lcppl f g \<sigma> ls). nlength(nnth (lcppl f g \<sigma> ls) j) > 0) "
   using assms lcppl_nlength_all_gr_zero_alt by blast
 have 3: " nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) = nlength (lcppl f g \<sigma> ls) +1" 
       using assms 
       by (simp add: lcppl_lsum_nlength_alt lcppl_nlength_alt nfinite_conv_nlength_enat)
 have 4: "(nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) 0) =
             nfirst(nfirst (lcppl f g \<sigma> ls)) "
    by (metis lsum_addzero_nfirst nfirst_nfusecat_nfirst nlast_NNil ntaken_0 ntaken_nlast pfilt_nnth 
        zero_enat_def zero_le)
 have 5: " j \<le> nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) \<and> j>0 \<longrightarrow>
          (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) (j))
           = nlast(nnth (lcppl f g \<sigma> ls) (j-1))"
     using assms pfilt_nfusecat_lsum_3[of "(lcppl f g \<sigma> ls)" j ] 
     using "1" "2" all_gr_zero_nnth_a all_nfinite_nnth_a lcppl_nlength_all_nfinite_alt by blast
 from 4 5 show ?thesis using assms 
 by blast
qed   

lemma lcppl_lsum_nnth_a:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "nfinite ls"
         "nfinite \<sigma>" 
         " nlast ls = the_enat( nlength \<sigma>) " 
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " nlength \<sigma>>0 "
         " j \<le> nlength ls " 
 shows  " (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) j) =
          (nnth ls j)"
proof -
 have 1: "nlength \<sigma>>0 \<longrightarrow> nlength ls>0 "
  using assms 
  by (metis enat.distinct(2) enat_the_enat nfinite_nlength_enat nnth_nlast the_enat_0 
      zero_less_iff_neq_zero)
 have 2: "nlength \<sigma>>0 \<longrightarrow> nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) = nlength ls"
    by (simp add: assms lcppl_lsum_nlength) 
 have 3: "nlength \<sigma>>0 \<longrightarrow> 
          j \<le> nlength ls \<longrightarrow>
            ( j=0 \<longrightarrow> 
          (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) j) =
           nfirst(nfirst (lcppl f g \<sigma> ls)) )
          \<and>
          (j>0 \<longrightarrow> (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) j) = 
           nlast(nnth (lcppl f g \<sigma> ls) (j-1)))" 
     by (metis "2" assms lcppl_lsum_nnth)
 have 4: "nlength \<sigma>>0 \<longrightarrow> 
          j \<le> nlength ls \<and> j=0 \<longrightarrow> nfirst(nfirst (lcppl f g \<sigma> ls)) = (nnth ls j) " 
     using assms lcppl_nfirst[of "ls" "\<sigma>" "f" "g"] 
     by (metis "1"  nlast_NNil ntaken_0 ntaken_nlast)  
 have 5: "nlength \<sigma>>0 \<longrightarrow> 
          j \<le> nlength ls \<and> j>0 \<longrightarrow> j-1 \<le> nlength (lcppl f g \<sigma> ls)"
      using assms 
      by (metis "1" Suc_diff_1 Suc_ile_eq co.enat.exhaust_sel i0_less iless_Suc_eq lcppl_nlength)
 have 6: "nlength \<sigma>>0 \<longrightarrow> 
          j \<le> nlength ls \<and> j>0 \<longrightarrow> nlast(nnth (lcppl f g \<sigma> ls) (j-1)) = (nnth ls j)"
    using lcppl_nlast_nnth assms
    by (metis "5" Suc_diff_1  enat.distinct(2) enat_the_enat nfinite_conv_nlength_enat)
 show ?thesis 
 using "3" "4" "6" using assms 
 by (metis not_gr_zero)
qed

lemma lcppl_lsum_nnth_a_alt:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls"
         "\<not>nfinite \<sigma>"       
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " j \<le> nlength ls " 
 shows  " (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) j) =
          (nnth ls j)"
proof -
 have 2: " nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) = nlength ls"
    by (simp add: assms lcppl_lsum_nlength_alt)
 have 3: "j \<le> nlength ls \<longrightarrow>
            ( j=0 \<longrightarrow> 
          (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) j) =
           nfirst(nfirst (lcppl f g \<sigma> ls)) )
          \<and>
          (j>0 \<longrightarrow> (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) j) = 
           nlast(nnth (lcppl f g \<sigma> ls) (j-1)))" 
     by (metis "2" assms lcppl_lsum_nnth_alt)
 have 4: "j \<le> nlength ls \<and> j=0 \<longrightarrow> nfirst(nfirst (lcppl f g \<sigma> ls)) = (nnth ls j) " 
     using assms lcppl_nfirst_alt[of "ls" "\<sigma>" "f" "g"] by (metis ndropn_0 ndropn_nfirst)
 have 5: "j \<le> nlength ls \<and> j>0 \<longrightarrow> j-1 \<le> nlength (lcppl f g \<sigma> ls)"
      using assms 
      by (meson enat_ile lcppl_nfinite linorder_le_cases nfinite_conv_nlength_enat)
 have 6: "j \<le> nlength ls \<and> j>0 \<longrightarrow> nlast(nnth (lcppl f g \<sigma> ls) (j-1)) = (nnth ls j)"
    using lcppl_nlast_nnth_alt assms 
    by (metis "5" Suc_pred')
 show ?thesis 
 using "3" "4" "6" using assms 
 by (metis not_gr_zero)
qed

lemma lcppl_pfilt_nfusecat_lsum:
 assumes "nidx ls"
         "nnth ls 0 = 0 " 
         "nfinite ls"
         "nfinite \<sigma> " 
         " nlast ls = the_enat( nlength \<sigma>)"
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " nlength \<sigma>>0 " 
 shows  " (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) = ls"
using assms 
proof -
 have 0: "nlength \<sigma>>0 \<longrightarrow>   nlength ls > 0" 
   using assms 
   by (metis enat.distinct(2) enat_the_enat gr_zeroI nfinite_conv_nlength_enat nnth_nlast the_enat_0) 
 have 1: " nlength \<sigma>>0 \<longrightarrow>  
          nlength (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) = nlength ls"
   using assms 
   by (simp add: lcppl_lsum_nlength pfilt_nlength)
 have 2: "nlength \<sigma>>0 \<longrightarrow>  
           (\<forall> j. j\<le>nlength ls \<longrightarrow>
          (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) j) =
          (nnth ls j))  "
     by (simp add: assms lcppl_lsum_nnth_a)
 from 1 2 show ?thesis using assms nellist_eq_nnth_eq 
 by metis
qed

lemma lcppl_pfilt_nfusecat_lsum_alt:
 assumes "nidx ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls"
         "\<not>nfinite \<sigma> " 
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
 shows  " (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) = ls"
using assms 
proof -
 have 1: "nlength (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) = nlength ls"
   using assms by (simp add: lcppl_lsum_nlength_alt pfilt_nlength)
 have 2: " (\<forall> j. j\<le>nlength ls \<longrightarrow>
          (nnth (pfilt (nfusecat ((lcppl f g \<sigma> ls))) (addzero(lsum ((lcppl f g \<sigma> ls)) 0))) j) =
          (nnth ls j))  "
     by (simp add: assms lcppl_lsum_nnth_a_alt)
 from 1 2 show ?thesis using assms nellist_eq_nnth_eq 
 by metis
qed

lemma lcppl_nfusecat_pfilt_powerinterval:
 assumes "nidx ls"
         "nnth ls 0 = 0 " 
         "nfinite ls"
         "nfinite \<sigma>" 
         " nlast ls  = the_enat (nlength \<sigma>) "  
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         "  nlength \<sigma> >0 " 
 shows   "  powerinterval g (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) (addzero (lsum (lcppl f g \<sigma> ls) 0))"
proof -
 have 0: "nlength \<sigma> > 0 \<longrightarrow> nlength ls >0 "
 using assms 
 by (metis enat.distinct(2) enat_the_enat gr_zeroI nfinite_conv_nlength_enat nnth_nlast the_enat_0)
 have 01: "(\<forall> i<nlength ls. 
            g (pfilt (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))))"
   using assms cppl_fprojection by auto
 have 02: "nlength \<sigma> > 0 \<longrightarrow> (\<forall> i<nlength ls. g (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) )"
   using "0" assms lcppl_nfusecat_pfilt_fpower_help 
   by (metis enat.distinct(2) enat_the_enat nfinite_conv_nlength_enat)
 have 03 : "powerinterval g (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) (addzero (lsum (lcppl f g \<sigma> ls) 0)) =
           ( \<forall>i<nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)).
             g (nsubn (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) 
                      (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) 
                      (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                    ))" 
        by (simp add: powerinterval_def)
  have 04: "nlength \<sigma> >0 \<longrightarrow>  nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) = nlength ls"
    using assms lcppl_lsum_nlength by blast
  have 05: "nlength \<sigma> >0 \<longrightarrow>
           (\<forall> i<nlength ls. 
              (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i))  =
              (nsubn  (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))
                   (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) 
                   (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                   ) )
      " 
   proof -
    have 06: "nlength \<sigma> >0 \<longrightarrow> 
              (\<forall> i<nlength ls. nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) =
                    nlength (nnth (lcppl f g \<sigma> ls) i))"
      using pfilt_nlength by blast
    have 07: "nlength \<sigma> >0 \<longrightarrow> 
              (\<forall> i<nlength ls. nlength (nnth (lcppl f g \<sigma> ls) i) =
                nlength (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )))) 
              " 
        using assms by (simp add: lcppl_nnth)
    have 08: "nlength \<sigma> >0 \<longrightarrow> 
              (\<forall> i<nlength ls. 
                nlength (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )))  =
                nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )))
              " 
         using nlength_nmap by blast
    have 09: "nlength \<sigma> >0 \<longrightarrow> 
              (\<forall>i<nlength ls. 
               nlength (nsubn  (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))
                               (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i)
                               (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                           ) =
               (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)) -  
               (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) )
             " using assms
             by (metis "04" enat_minus_mono1 idiff_enat_enat lcppl_lsum_less_th_equal min.orderE 
         nsubn_nlength pfilt_nlength) 
    have 10: "nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls. (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)) =
                 (nnth (NCons 0 (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)))  
              " 
        using "04" addzero_def by auto
    have 11: "nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls. (nnth (NCons 0 (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)) =
                 (\<Sum>k::nat= 0..(i). nlength(nnth (lcppl f g \<sigma> ls) k)) )
              "   using "04" 
        proof simp_all
          assume "nlength \<sigma> \<noteq> (0::enat) \<longrightarrow> nlength (addzero (lsum (lcppl f g \<sigma> ls) (0::nat))) = nlength ls"
          show   " nlength \<sigma> \<noteq> (0::enat) \<longrightarrow>
                  (\<forall>i::nat. enat i < nlength ls \<longrightarrow> enat (nnth (lsum (lcppl f g \<sigma> ls) (0::nat)) i) = 
                  (\<Sum>k::nat = 0::nat..i. nlength (nnth (lcppl f g \<sigma> ls) k)))"
           proof 
            assume " nlength \<sigma> \<noteq> (0::enat)"
            show " (\<forall>i::nat. enat i < nlength ls \<longrightarrow> 
                  enat (nnth (lsum (lcppl f g \<sigma> ls) (0::nat)) i) = 
                  (\<Sum>k::nat = 0::nat..i. nlength (nnth (lcppl f g \<sigma> ls) k))) " 
            proof 
             fix i
             show "i < nlength ls \<longrightarrow>
                   enat (nnth (lsum (lcppl f g \<sigma> ls) (0::nat)) i) = 
                  (\<Sum>k::nat = 0::nat..i. nlength (nnth (lcppl f g \<sigma> ls) k)) " 
                   proof 
                     assume a: "i < nlength ls" 
                     show "enat (nnth (lsum (lcppl f g \<sigma> ls) 0) i) = 
                           (\<Sum>k = 0..i. nlength (nnth (lcppl f g \<sigma> ls) k)) "
                     proof -
                     have 110: " all_nfinite (lcppl f g \<sigma> ls) "
                         using all_nfinite_nnth_a assms lcppl_nlength_all_nfinite by blast   
                     have 111: "nnth (lsum (lcppl f g \<sigma> ls) 0) i =  
                                (\<Sum>k = 0..i. the_enat (nlength (nnth (lcppl f g \<sigma> ls) k)))" 
                     using lsum_nnth[of i "(lcppl f g \<sigma> ls)" 0] a assms 
                     by (metis "0" "110" add_cancel_right_left co.enat.exhaust_sel iless_Suc_eq 
                         lcppl_nlength not_gr_zero)
                     have 112: "(\<Sum>k = 0..i. the_enat (nlength (nnth (lcppl f g \<sigma> ls) k))) =
                                the_enat(\<Sum>k = 0..i.  (nlength (nnth (lcppl f g \<sigma> ls) k)))" 
                      using sum_the_enat[of "(lcppl f g \<sigma> ls)" i]   assms 
                      by (metis "0" "110" a co.enat.exhaust_sel gr_implies_not_zero iless_Suc_eq 
                          lcppl_nlength)
                     have 113: "(\<Sum>k = 0..i.  (nlength (nnth (lcppl f g \<sigma> ls) k))) < \<infinity> " 
                       using sum_finite[of "(lcppl f g \<sigma> ls)" i] 0 110 assms a
                       by (metis co.enat.exhaust_sel gr_implies_not_zero iless_Suc_eq lcppl_nlength)  
                     have 114: "enat (the_enat(\<Sum>k = 0..i.  (nlength (nnth (lcppl f g \<sigma> ls) k)))) =
                                 (\<Sum>k = 0..i.  (nlength (nnth (lcppl f g \<sigma> ls) k))) " 
                        using "113" enat_ord_simps(4) enat_the_enat by blast
                     show ?thesis  using "111" "112" "114" by presburger
                   qed
                qed 
            qed
           qed
        qed              
    have 12: "nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls. (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) =
                 (nnth (NCons 0 (lsum (lcppl f g \<sigma> ls) 0)) i) )"      
       using "04" addzero_def by auto
    have 121: "nlength \<sigma> >0 \<longrightarrow> (nnth (NCons 0 (lsum (lcppl f g \<sigma> ls) 0)) 0) = 0 "
          by simp  
    have 13: "nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls. (nnth (NCons 0 (lsum (lcppl f g \<sigma> ls) 0)) i) =
               (if i = 0 then  0
                  else (\<Sum>k::nat= 0..(i-1). nlength(nnth (lcppl f g \<sigma> ls) k)) ))    
              " 
         using 11 
         by (metis "121" Suc_diff_1 Suc_ile_eq not_gr_zero order_less_imp_le zero_enat_def)
    have 14: "nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls. 
              (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)) -  
              (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) =
              (\<Sum>k::nat= 0..(i). nlength(nnth (lcppl f g \<sigma> ls) k)) -
              (if i = 0 then 0 else
                  (\<Sum>k::nat= 0..(i-1). nlength(nnth (lcppl f g \<sigma> ls) k)) ) )" 
      using "10" "11" "12" "13" by simp (metis One_nat_def idiff_enat_enat not_gr_zero)
    have 15: "nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls.  
                (\<Sum>k::nat= 0..(i). nlength(nnth (lcppl f g \<sigma> ls) k)) -
              (if i = 0 then 0 else
                  (\<Sum>k::nat= 0..(i-1). nlength(nnth (lcppl f g \<sigma> ls) k)) ) =
                (if i = 0 then nlength(nnth (lcppl f g \<sigma> ls) 0) else
                   (\<Sum>k::nat= 0..(i). nlength(nnth (lcppl f g \<sigma> ls) k)) -
                             (\<Sum>k::nat= 0..(i-1). nlength(nnth (lcppl f g \<sigma> ls) k)) ))" 
        by (simp add: Nitpick.case_nat_unfold)
     have 16: "nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls.  
                 (if i = 0 then nlength (nnth (lcppl f g \<sigma> ls) 0) else
                   (\<Sum>k::nat= 0..(i). nlength(nnth (lcppl f g \<sigma> ls) k)) -
                             (\<Sum>k::nat= 0..(i-1). nlength(nnth (lcppl f g \<sigma> ls) k)) ) = 
                 (if i = 0 then nlength(nnth (lcppl f g \<sigma> ls) 0) else
                   nlength(nnth (lcppl f g \<sigma> ls) i)  ) )"
         using 13  sum.cl_ivl_Suc[of "\<lambda>k. nlength(nnth (lcppl f g \<sigma> ls) k)" 0]
         by (metis Suc_diff_1 enat.distinct(2) enat_add_sub_same less_nat_zero_code not_gr_zero)  
    have 17: "nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls. 
                  (if i = 0 then nlength(nnth (lcppl f g \<sigma> ls) 0) else
                    nlength(nnth (lcppl f g \<sigma> ls) i)  ) =
                  nlength(nnth (lcppl f g \<sigma> ls) i))" 
      by (simp add: Nitpick.case_nat_unfold)
    have 18: "nlength \<sigma> >0 \<longrightarrow> 
              (\<forall> i<nlength ls. nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) =
                nlength (nsubn  (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))
                                (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) 
                                (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                            ) )
             "   
       by (simp add: "09" "14" "15" "16" pfilt_nlength)
    have 19: "nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls. 
                (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i).
                 (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)) \<le> 
                  nlength (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))))"
         by (simp add: "04" assms lcppl_lsum_less_th_equal pfilt_nlength)
    have 22: " nlength \<sigma> >0 \<longrightarrow>  nlastnfirst (lcppl f g \<sigma> ls)"
      using "0" assms 
      by (simp add: enat_the_enat lcppl_nfusecat_nlastnfirst nfinite_conv_nlength_enat)
   have 23: " nlength \<sigma> >0 \<longrightarrow>  (\<forall> j \<le> nlength (lcppl f g \<sigma> ls). nlength(nnth (lcppl f g \<sigma> ls) j) > 0)"
     using assms 
     by (metis enat.distinct(2) enat_the_enat lcppl_nlength_all_gr_zero nfinite_conv_nlength_enat)
   have 190: "nlength \<sigma> >0 \<longrightarrow> 
                (\<forall> i<nlength ls. 
                   (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i).
                      (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) \<le> 
                      (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                   ))" 
       using 0 04 06 09 18 23 lsum_nlength[of "(lcppl f g \<sigma> ls)" 0] lcppl_nlength[of ls f g \<sigma>] assms
        by simp
         (metis (no_types, lifting) co.enat.exhaust_sel diff_is_0_eq' iless_Suc_eq le_cases zero_enat_def)
   have 20: "nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls. 
                 (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i). 
                   (nnth (nsubn  (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) 
                                 (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) 
                                 (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                             ) j) =
                   (nnth (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) 
                        ((nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) +j) ) ) ) 
              " using nsubn_nnth[of "(pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))" ] by  simp 
      (metis "06" "09" "18" assms(7) enat_ord_simps(1) min.orderE)
  have 21: "nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls. 
                 (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i).
                  (nnth (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) 
                       ((nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) +j) ) =
                  (nnth \<sigma> (nnth (nfusecat (lcppl f g \<sigma> ls)) 
                              ((nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) +j) )) ) )"
       using pfilt_nlength[of \<sigma> "(nfusecat (lcppl f g \<sigma> ls))" ] 
         pfilt_nnth[of _ \<sigma> "(nfusecat (lcppl f g \<sigma> ls))" ] 
         by (metis nnth_0 pfilt_code(2) pfilt_pfilt)
   have 24: "nlength \<sigma> >0 \<longrightarrow>
             (\<forall> i\<le>nlength (lcppl f g \<sigma> ls). 
                 (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i). 
                  ( (nnth (nfusecat (lcppl f g \<sigma> ls)) ((nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) +j) )) =
                  ( (nnth (nnth (lcppl f g \<sigma> ls) i) j))  ))
            " 
    using assms lsum_nfusecat_nnth_lsum_nnth[of "(lcppl f g \<sigma> ls)"] lcppl_nlength[of ls f g \<sigma>]  
       0 06 09 18 22 23 
       by (metis all_gr_zero_nnth_a all_nfinite_nnth_a lcppl_nlength_all_nfinite) 
   have 241: "nlength \<sigma> >0 \<longrightarrow> nlength (lcppl f g \<sigma> ls) = nlength ls -1"
     using "0" assms 
     by (simp add: epred_conv_minus lcppl_nlength) 
   have 25: " nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls. 
                 (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i). 
                  (nnth \<sigma> (nnth (nfusecat (lcppl f g \<sigma> ls)) 
                              ((nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) +j) )) =
                  (nnth (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) j) ))"
       using 0 04 24 lsum_nlength[of "(lcppl f g \<sigma> ls)" 0]  pfilt_nlength[of \<sigma> ]  pfilt_nnth 
       by (metis addzero_def i0_less iless_Suc_eq nlength_NCons)    
    have 26: "nlength \<sigma> >0 \<longrightarrow> 
               (\<forall> i<nlength ls. 
                 (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i). 
                  (nnth (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) j) =
                  (nnth (nsubn  (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))
                                (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i)
                                (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                            ) j) ) )
             "
     by (simp add: "20" "21" "25")
   from 18 26 show ?thesis using nellist_eq_nnth_eq 
   by (metis "06")
   qed
 show ?thesis 
 using "02" "03" "04" "05" assms(7) by force
qed

lemma lcppl_nfusecat_pfilt_powerinterval_alt:
 assumes "nidx ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls"
         "\<not>nfinite \<sigma>" 
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
 shows   "  powerinterval g (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) (addzero (lsum (lcppl f g \<sigma> ls) 0))"
proof -
 have 01: "(\<forall> i<nlength ls. 
            g (pfilt (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))))"
   using assms cppl_fprojection by auto
 have 02: " (\<forall> i<nlength ls. g (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) )"
   using  assms lcppl_nfusecat_pfilt_fpower_help_alt  
   by (metis  enat_the_enat nfinite_conv_nlength_enat)
 have 03 : "powerinterval g (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) (addzero (lsum (lcppl f g \<sigma> ls) 0)) =
           ( \<forall>i<nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)).
             g (nsubn (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) 
                      (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) 
                      (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                    ))" 
        by (simp add: powerinterval_def)
  have 04: "  nlength (addzero (lsum (lcppl f g \<sigma> ls) 0)) = nlength ls"
    using assms lcppl_lsum_nlength_alt by blast
  have 05: "(\<forall> i<nlength ls. 
              (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i))  =
              (nsubn  (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))
                   (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) 
                   (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                   ) )" 
   proof -
    have 06: "
              (\<forall> i<nlength ls. nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) =
                    nlength (nnth (lcppl f g \<sigma> ls) i))"
      using pfilt_nlength by blast
    have 07: " (\<forall> i<nlength ls. nlength (nnth (lcppl f g \<sigma> ls) i) =
                nlength (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )))) " 
        using assms by (simp add: lcppl_nnth)
    have 08: " (\<forall> i<nlength ls. 
                nlength (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )))  =
                nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )))" 
         using nlength_nmap by blast
    have 09: " (\<forall>i<nlength ls. 
               nlength (nsubn  (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))
                               (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i)
                               (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                           ) =
               (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)) -  
               (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) )" 
        using assms 
       by (metis "04" enat_minus_mono1 idiff_enat_enat lcppl_lsum_less_th_equal_alt min_def 
           nsubn_nlength pfilt_nlength)
    have 10: " (\<forall> i<nlength ls. (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)) =
                 (nnth (NCons 0 (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)))  " 
        using "04" addzero_def by auto
    have 11: " (\<forall> i<nlength ls. (nnth (NCons 0 (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)) =
                 (\<Sum>k::nat= 0..(i). nlength(nnth (lcppl f g \<sigma> ls) k)) )"  
        using "04" 
        proof simp_all
          assume " nlength (addzero (lsum (lcppl f g \<sigma> ls) (0::nat))) = nlength ls"
          show   "(\<forall>i::nat. enat i < nlength ls \<longrightarrow> enat (nnth (lsum (lcppl f g \<sigma> ls) (0::nat)) i) = 
                  (\<Sum>k::nat = 0::nat..i. nlength (nnth (lcppl f g \<sigma> ls) k)))"
            proof 
             fix i
             show "i < nlength ls \<longrightarrow>
                   enat (nnth (lsum (lcppl f g \<sigma> ls) (0::nat)) i) = 
                  (\<Sum>k::nat = 0::nat..i. nlength (nnth (lcppl f g \<sigma> ls) k)) " 
                  proof 
                     assume a: "i < nlength ls" 
                     show "enat (nnth (lsum (lcppl f g \<sigma> ls) 0) i) = 
                           (\<Sum>k = 0..i. nlength (nnth (lcppl f g \<sigma> ls) k)) "
                     proof -
                     have 110: " all_nfinite (lcppl f g \<sigma> ls) "
                         using all_nfinite_nnth_a assms lcppl_nlength_all_nfinite_alt by blast 
                     have 111: "nnth (lsum (lcppl f g \<sigma> ls) 0) i =  (\<Sum>k = 0..i. the_enat (nlength (nnth (lcppl f g \<sigma> ls) k)))" 
                     using lsum_nnth[of i "(lcppl f g \<sigma> ls)" 0] a assms 
                     by (metis "04" "110" add_cancel_right_left addzero_def iless_Suc_eq lsum_nlength 
                         nlength_NCons nlength_eq_enat_nfiniteD zero_enat_def)
                     have 112: "(\<Sum>k = 0..i. the_enat (nlength (nnth (lcppl f g \<sigma> ls) k))) =
                                the_enat(\<Sum>k = 0..i.  (nlength (nnth (lcppl f g \<sigma> ls) k)))" 
                      using sum_the_enat[of "(lcppl f g \<sigma> ls)" i]   assms 
                      by (metis "110" lcppl_nfinite linorder_le_cases nfinite_ntaken ntaken_all)
                     have 113: "(\<Sum>k = 0..i.  (nlength (nnth (lcppl f g \<sigma> ls) k))) < \<infinity> " 
                       using sum_finite[of "(lcppl f g \<sigma> ls)" i]  110 04 assms a 
                       by (metis addzero_def iless_Suc_eq lsum_nlength nlength_NCons 
                           nlength_eq_enat_nfiniteD zero_enat_def)
                     have 114: "enat (the_enat(\<Sum>k = 0..i.  (nlength (nnth (lcppl f g \<sigma> ls) k)))) =
                                 (\<Sum>k = 0..i.  (nlength (nnth (lcppl f g \<sigma> ls) k))) " 
                        using "113" enat_ord_simps(4) enat_the_enat by blast
                     show ?thesis  using "111" "112" "114" by presburger
                   qed
                  qed
              qed
       qed            
    have 12: " (\<forall> i<nlength ls. (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) =
                 (nnth (NCons 0 (lsum (lcppl f g \<sigma> ls) 0)) i) )"      
       using "04" addzero_def by auto
    have 121: " (nnth (NCons 0 (lsum (lcppl f g \<sigma> ls) 0)) 0) = 0 "
          by simp  
    have 13: " (\<forall> i<nlength ls. (nnth (NCons 0 (lsum (lcppl f g \<sigma> ls) 0)) i) =
               (if i = 0 then  0
                  else (\<Sum>k::nat= 0..(i-1). nlength(nnth (lcppl f g \<sigma> ls) k)) )) " 
         using 11 
         by (metis "121" Suc_diff_1 Suc_ile_eq not_gr_zero order_less_imp_le zero_enat_def)
    have 14: " (\<forall> i<nlength ls. 
              (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)) -  
              (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) =
              (\<Sum>k::nat= 0..(i). nlength(nnth (lcppl f g \<sigma> ls) k)) -
              (if i = 0 then 0 else
                  (\<Sum>k::nat= 0..(i-1). nlength(nnth (lcppl f g \<sigma> ls) k)) ) )
             " 
      using "10" "11" "12" "13" by simp (metis One_nat_def idiff_enat_enat not_gr_zero)
    have 15: " (\<forall> i<nlength ls.  
                (\<Sum>k::nat= 0..(i). nlength(nnth (lcppl f g \<sigma> ls) k)) -
              (if i = 0 then 0 else
                  (\<Sum>k::nat= 0..(i-1). nlength(nnth (lcppl f g \<sigma> ls) k)) ) =
                (if i = 0 then nlength(nnth (lcppl f g \<sigma> ls) 0) else
                   (\<Sum>k::nat= 0..(i). nlength(nnth (lcppl f g \<sigma> ls) k)) -
                             (\<Sum>k::nat= 0..(i-1). nlength(nnth (lcppl f g \<sigma> ls) k)) ))
             " 
        by (simp add: Nitpick.case_nat_unfold)
     have 16: " (\<forall> i<nlength ls.  
                 (if i = 0 then nlength (nnth (lcppl f g \<sigma> ls) 0) else
                   (\<Sum>k::nat= 0..(i). nlength(nnth (lcppl f g \<sigma> ls) k)) -
                             (\<Sum>k::nat= 0..(i-1). nlength(nnth (lcppl f g \<sigma> ls) k)) ) = 
                 (if i = 0 then nlength(nnth (lcppl f g \<sigma> ls) 0) else
                   nlength(nnth (lcppl f g \<sigma> ls) i)  ) )"
         using 13  sum.cl_ivl_Suc[of "\<lambda>k. nlength(nnth (lcppl f g \<sigma> ls) k)" 0]
          by (metis Suc_diff_1 enat.distinct(2) enat_add_sub_same less_nat_zero_code not_gr_zero) 
    have 17: " (\<forall> i<nlength ls. 
                  (if i = 0 then nlength(nnth (lcppl f g \<sigma> ls) 0) else
                    nlength(nnth (lcppl f g \<sigma> ls) i)  ) =
                  nlength(nnth (lcppl f g \<sigma> ls) i))" 
      by (simp add: Nitpick.case_nat_unfold)
    have 18: " (\<forall> i<nlength ls. nlength (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) =
                nlength (nsubn  (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))
                                (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) 
                                (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                            ) )"   
       by (simp add: "09" "14" "15" "16" pfilt_nlength)
    have 19: " (\<forall> i<nlength ls. 
                (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i).
                 (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i)) \<le> 
                  nlength (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))))" 
             by (simp add: "04" assms lcppl_lsum_less_th_equal_alt pfilt_nlength)
    have 22: "   nlastnfirst (lcppl f g \<sigma> ls)"
      using  assms lcppl_nfusecat_nlastnfirst_alt by blast
   have 23: "   (\<forall> j \<le> nlength (lcppl f g \<sigma> ls). nlength(nnth (lcppl f g \<sigma> ls) j) > 0)"
     using assms lcppl_nlength_all_gr_zero_alt by blast
   have 190: " (\<forall> i<nlength ls. 
                   (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i).
                      (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) \<le> 
                      (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                   ))" 
     using  04 06 09 18 23 lsum_nlength[of "(lcppl f g \<sigma> ls)" 0] lcppl_nlength_alt[of ls f g \<sigma>] assms
     by simp
     (metis (no_types, opaque_lifting) co.enat.exhaust_sel diff_is_0_eq' iless_Suc_eq nfinite_ntaken
      not_le_imp_less ntaken_all order_less_imp_le zero_enat_def)
   have 20: "(\<forall> i<nlength ls. 
                 (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i). 
                   (nnth (nsubn  (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) 
                                 (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) 
                                 (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                             ) j) =
                   (nnth (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) 
                        ((nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) +j) ) ) ) " 
        using nsubn_nnth[of  "(pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))" ] by simp
        (metis "06" "09" "18" enat_ord_simps(1) min.orderE)
  have 21: " (\<forall> i<nlength ls. 
                 (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i).
                  (nnth (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) 
                       ((nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) +j) ) =
                  (nnth \<sigma> (nnth (nfusecat (lcppl f g \<sigma> ls)) 
                              ((nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) +j) )) ) )" 
     using pfilt_nlength[of \<sigma> "(nfusecat (lcppl f g \<sigma> ls))" ] 
           pfilt_nnth[of _ \<sigma> "(nfusecat (lcppl f g \<sigma> ls))" ]
     by (metis enat_ord_code(4) enat_the_enat not_le_imp_less ntaken_all ntaken_nlast order_less_imp_le) 
   have 24: "(\<forall> i\<le>nlength (lcppl f g \<sigma> ls). 
                 (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i). 
                  ( (nnth (nfusecat (lcppl f g \<sigma> ls)) ((nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) +j) )) =
                  ( (nnth (nnth (lcppl f g \<sigma> ls) i) j))  ))" 
    using assms lsum_nfusecat_nnth_lsum_nnth[of "(lcppl f g \<sigma> ls)"]  04 22 23 
    all_gr_zero_nnth_a all_nfinite_nnth_a lcppl_nlength_all_nfinite_alt by blast
   have 241: " nlength (lcppl f g \<sigma> ls) = nlength ls -1"
     using  assms by (simp add: epred_conv_minus lcppl_nlength_alt)
   have 25: " (\<forall> i<nlength ls. 
                 (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i). 
                  (nnth \<sigma> (nnth (nfusecat (lcppl f g \<sigma> ls)) 
                              ((nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i) +j) )) =
                  (nnth (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) j) ))"
       using  04 24 lsum_nlength[of "(lcppl f g \<sigma> ls)" 0]  pfilt_nlength[of \<sigma> ]  pfilt_nnth 
      assms(3)
      by (metis "241" enat2_cases enat_ord_code(3) epred_Infty epred_conv_minus 
          nfinite_conv_nlength_enat) 
    have 26: "(\<forall> i<nlength ls. 
                 (\<forall> j\<le> nlength(nnth (lcppl f g \<sigma> ls) i). 
                  (nnth (pfilt \<sigma> (nnth (lcppl f g \<sigma> ls) i)) j) =
                  (nnth (nsubn  (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))
                                (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) i)
                                (nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) (Suc i))
                            ) j) ) )"
     by (simp add: "20" "21" "25")
   from 18 26 show ?thesis using nellist_eq_nnth_eq 
   by (metis "06")
  qed
 show ?thesis 
 using "02" "03" "04" "05" assms by force
qed

lemma lcppl_nfusecat_pfilt_nlength:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "nfinite ls "
         "nfinite \<sigma> " 
         " nlast ls = the_enat (nlength \<sigma>)" 
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " nlength \<sigma> >0 " 
 shows   " nlast (addzero (lsum (lcppl f g \<sigma> ls) 0))  =
          the_enat(nlength (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))) "
proof -
 have 0: " nlength ls > 0"
    using assms 
    by (metis enat.distinct(2) enat_the_enat gr_zeroI nfinite_conv_nlength_enat nnth_nlast the_enat_0)
 have 1: "nlength (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) = nlength ( (nfusecat (lcppl f g \<sigma> ls)))  " 
   using pfilt_nlength by blast
 have 10: "nfinite (lcppl f g \<sigma> ls)"
     using assms(1) assms(3) lcppl_nfinite by blast
 have 11: "nlastnfirst (lcppl f g \<sigma> ls)" 
    using assms 
    by (metis "0"   lcppl_nfusecat_nlastnfirst nfinite_conv_nlength_enat)
 have 12: " \<forall>lx \<in> nset (lcppl f g \<sigma> ls). 0 < nlength lx"
     using assms 
     by (metis  enat.distinct(2) enat_the_enat in_nset_conv_nnth lcppl_nlength_all_gr_zero 
         nfinite_conv_nlength_enat) 
 have 13: "\<forall>lx \<in> nset (lcppl f g \<sigma> ls). nfinite lx" 
   by (metis (no_types, lifting) "0" assms(1) assms(3) assms(6) co.enat.exhaust_sel 
       cppl_fprojection i0_less iless_Suc_eq in_nset_conv_nnth lcppl_nlength lcppl_nnth nfinite_nmap)
 have 141: " \<forall>i \<le> nlength (lcppl f g \<sigma> ls). 0 < nlength (nnth (lcppl f g \<sigma> ls) i)"
     using 12 assms
     by (metis  enat.distinct(2) enat_the_enat lcppl_nlength_all_gr_zero nfinite_conv_nlength_enat) 
 have 142: "\<forall>i \<le> nlength (lcppl f g \<sigma> ls). nfinite (nnth (lcppl f g \<sigma> ls) i)" 
   using 13 assms 
   by (meson in_nset_conv_nnth)
 have 15: "nfinite  (nfusecat (lcppl f g \<sigma> ls))"
     using nfusecat_nfinite[of "(lcppl f g \<sigma> ls)"]  "10" "11" "12" "13" by blast
 have 2: " nlength ( (nfusecat (lcppl f g \<sigma> ls))) = 
            (the_enat (\<Sum>k::nat= 0..(the_enat(nlength (lcppl f g \<sigma> ls))). nlength(nnth (lcppl f g \<sigma> ls) k)))"
   using "0" assms 
   by (metis "10" "11" "13" "15" nfinite_conv_nlength_enat nfusecat_nlength_nfinite  the_enat.simps)
 have 3: " 
      nlast( addzero (lsum (lcppl f g \<sigma> ls) 0)) = nlast( (lsum (lcppl f g \<sigma> ls) 0))  "
    using lsum_addzero_nlast 
    using assms(1) assms(3) lcppl_nfinite by blast
 have 4: " nlast( (lsum (lcppl f g \<sigma> ls) 0)) = 
          (\<Sum>k::nat = 0::nat..the_enat (nlength (lcppl f g \<sigma> ls)). the_enat (nlength (nnth (lcppl f g \<sigma> ls) k)))"
   using assms lsum_nlast[of "(lcppl f g \<sigma> ls)" 0] "10" "13" plus_nat.add_0 by presburger
 have 5: " (\<Sum>k::nat = 0..the_enat (nlength (lcppl f g \<sigma> ls)). the_enat (nlength (nnth (lcppl f g \<sigma> ls) k))) =
          (the_enat (\<Sum>k::nat = 0..the_enat(nlength (lcppl f g \<sigma> ls)). nlength(nnth (lcppl f g \<sigma> ls) k)))"
      using assms lsum_nnth[of "the_enat (nlength (lcppl f g \<sigma> ls))" "(lcppl f g \<sigma> ls)" 0 ]
      sum_the_enat[of "(lcppl f g \<sigma> ls)" "the_enat(nlength (lcppl f g \<sigma> ls))"] 
      by (metis "13" dual_order.refl enat_ord_code(3) enat_the_enat)     
 show ?thesis using "1" "2" "3" "4" 5 assms 
 by (simp add: lcppl_lsum_nlength nfinite_conv_nlength_enat)
qed


subsection \<open>Soundness of Projection Axioms\<close>

lemma PJ00sem:
 "\<sigma> \<Turnstile> \<not>(f  fproj  inf) " 
by (auto simp add: fprojection_d_def infinite_defs  nfinite_conv_nlength_enat pfilt_nlength)

lemma OPJ00sem:
 "\<sigma> \<Turnstile> \<not>(f oproj finite) "
by (auto simp add: oprojection_d_def finite_defs  nfinite_conv_nlength_enat pfilt_nlength) 

lemma PJ01sem:
 "\<sigma> \<Turnstile> (f  fproj  g) \<longrightarrow> finite"
by (auto simp add: fprojection_d_def finite_defs  nfinite_conv_nlength_enat pfilt_nlength)
 
lemma OPJ01sem:
 "\<sigma> \<Turnstile> (f oproj g) \<longrightarrow> inf"
by (auto simp add: oprojection_d_def infinite_defs  nfinite_conv_nlength_enat pfilt_nlength) 

lemma PJ02sem:
 "\<sigma> \<Turnstile> (f  fproj  g) = ((f \<and> finite)  fproj  g)"
by (auto simp add: fprojection_d_def finite_defs powerinterval_def  nfinite_conv_nlength_enat 
    pfilt_nlength)
 (metis enat_ord_simps(2) nfinite_nlength_enat nsubn_nfinite)
 

lemma OPJ02sem:
 "\<sigma> \<Turnstile> (f oproj g) = ((f \<and> finite) oproj g)"
by (auto simp add: oprojection_d_def finite_defs powerinterval_def  nfinite_conv_nlength_enat 
    pfilt_nlength)
 (metis nfinite_conv_nlength_enat nfinite_ntaken nsubn_def1)

lemma PJ03sem:
 "\<sigma> \<Turnstile> (f  fproj  g) = (f  fproj  (g \<and> finite))"
by (auto simp add: fprojection_d_def finite_defs powerinterval_def nfinite_conv_nlength_enat 
    pfilt_nlength)

lemma OPJ03sem:
 "\<sigma> \<Turnstile> (f oproj g) = (f oproj (g \<and> inf))"
by (auto simp add: oprojection_d_def infinite_defs powerinterval_def  nfinite_conv_nlength_enat 
    pfilt_nlength)

subsubsection \<open>PJ1\<close>

lemma PJ1sem:
 "(\<sigma> \<Turnstile> f  fproj  ( g \<or> h) = (f  fproj  g \<or> f  fproj  h))"
by (simp add: fprojection_d_def) blast

lemma OPJ1sem:
 "(\<sigma> \<Turnstile> f oproj ( g \<or> h) = (f oproj g \<or> f oproj h))"
by (simp add: oprojection_d_def) blast

subsubsection \<open>PJ2\<close>

lemma PJ2sem:
 "(\<sigma> \<Turnstile> f  fproj  empty = empty)"
proof auto
  show "(\<sigma> \<Turnstile> f  fproj  empty) \<Longrightarrow> \<sigma> \<Turnstile> empty " 
  unfolding fprojection_d_def empty_defs 
  by (metis enat.distinct(2) enat_the_enat nfinite_nlength_enat nnth_nlast pfilt_nlength the_enat_0)
  show "\<sigma> \<Turnstile> empty \<Longrightarrow> (\<sigma> \<Turnstile> f  fproj  empty)" 
  unfolding fprojection_d_def empty_defs  powerinterval_def nidx_expand 
  by (metis enat_ord_simps(2) leD nfinite_conv_nlength_enat nlast_NNil nlength_NNil nnth_NNil 
      not_iless0 pfilt_nlength the_enat_0 zero_enat_def zero_less_Suc)
qed

lemma OPJ2sem:
 "(\<sigma> \<Turnstile> \<not>(f oproj empty))"
unfolding oprojection_d_def empty_defs pfilt_nlength 
using nlength_eq_enat_nfiniteD by (simp add: zero_enat_def) blast


subsubsection \<open>PJ3\<close>

lemma PJ3help:
assumes "nfinite \<sigma>" 
shows  "nsubn \<sigma> 0 (the_enat (nlength \<sigma>))  = \<sigma>"
using assms 
by (metis diff_zero dual_order.refl ndropn_0 nfinite_conv_nlength_enat nsubn_def1 ntaken_all 
    the_enat.simps)

lemma PJ3help1:
assumes "f \<sigma> \<and> 0 < nlength \<sigma> \<and> nfinite \<sigma>"
shows   " (\<exists>l. nidx  l \<and> nnth l 0 = 0 \<and> nfinite l \<and> nfinite \<sigma> \<and>
        nlast l = (the_enat (nlength \<sigma>)) \<and>
        (\<forall>i<nlength l. f (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) )) \<and>
        (\<exists>\<sigma>1. nlength \<sigma>1 = nlength l \<and>
      (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (nnth l i)) \<and> nlength \<sigma>1 = Suc 0))"
proof -
 have 1: "nidx  (NCons 0 (NNil (the_enat(nlength \<sigma>)))) " 
   using assms unfolding nidx_expand by simp 
   (metis Suc_ile_eq assms enat_0_iff(1) enat_ord_simps(2) iless_eSuc0 nfinite_nlength_enat nnth_0 
    nnth_NNil the_enat.simps)
 have 2: "nnth (NCons 0 (NNil (the_enat(nlength \<sigma>)))) ((the_enat(nlength (NCons 0 (NNil (nlength \<sigma>)))))) =
          (the_enat (nlength \<sigma>)) " 
   by (metis nfinite_NCons nfinite_NNil nlast_NCons nlast_NNil nlength_NCons nlength_NNil nnth_nlast)
 have 3: "(\<forall>i<nlength (NCons 0 (NNil (the_enat(nlength \<sigma>)))).
             f (nsubn \<sigma> (nnth (NCons 0 (NNil (the_enat(nlength \<sigma>)))) i) 
                        (nnth (NCons 0 (NNil (the_enat(nlength \<sigma>)))) (Suc i)) )) "     
   using assms PJ3help 
   by (metis enat_0_iff(1) iless_eSuc0 nlength_NCons nlength_NNil nnth_0 nnth_NNil nnth_Suc_NCons)
 have 4: "nlength (NCons (nnth \<sigma> (0)) (NNil (nnth \<sigma> (the_enat(nlength \<sigma>))))) =
          nlength (NCons 0 (NNil (the_enat(nlength \<sigma>)))) " 
    by simp
 have 5: "(\<forall>i\<le>nlength  (NCons (nnth \<sigma> (0)) (NNil (nnth \<sigma> (the_enat(nlength \<sigma>))))). 
      nnth  (NCons (nnth \<sigma> (0)) (NNil (nnth \<sigma> (the_enat(nlength \<sigma>))))) i = 
      nnth \<sigma> (nnth (NCons 0 (NNil (the_enat(nlength \<sigma>)))) i)) "
   by (metis iless_Suc_eq le_zero_eq ndropn_nlast nfinite_NCons nfinite_NNil nlast_NCons 
       nlength_NCons nlength_NNil nnth_0 nnth_NNil nnth_zero_ndropn order_neq_le_trans 
       the_enat.simps the_enat_0)
 have 6: "nlength (NCons (nnth \<sigma> (0)) (NNil (nnth \<sigma> (the_enat(nlength \<sigma>))))) = Suc 0 " 
   using one_eSuc one_enat_def by auto
 have 7: "nfinite (NCons 0 (NNil (the_enat(nlength \<sigma>))))"
   by simp 
 have 8: "nlast (NCons 0 (NNil (the_enat(nlength \<sigma>)))) = (the_enat (nlength \<sigma>))"
    by simp 
 have 9: "nnth (NCons 0 (NNil (the_enat(nlength \<sigma>)))) 0 = 0 " 
     using nnth_0 by simp
 show ?thesis 
 using "1" "2" "3" "4" "5" "6" 7 8 9 assms by blast
qed


lemma PJ3sem:
"(\<sigma> \<Turnstile> f  fproj  skip = (f \<and> more \<and> finite))"
proof -
 have 1: "(\<sigma> \<Turnstile> f  fproj  skip) \<Longrightarrow> (\<sigma> \<Turnstile> f \<and> more \<and> finite)"
   unfolding cppl_fprojection pfilt_expand nidx_expand skip_defs more_defs finite_defs powerinterval_def 
   by (metis (no_types, lifting) One_nat_def PJ3help eSuc_enat i0_less ileI1 intensional_rews(3) 
       less_numeral_extra(1) less_numeral_extra(3) nnth_nlast pfilt_nlength the_enat.simps zero_enat_def)
 have 2: "(\<sigma> \<Turnstile> f \<and> more \<and> finite) \<Longrightarrow> (\<sigma> \<Turnstile> f  fproj  skip)"
   by (simp add:  fprojection_d_def finite_defs skip_defs more_defs powerinterval_def pfilt_nlength ) 
     (metis PJ3help1 i0_less nfinite_conv_nlength_enat the_enat.simps)
 show ?thesis 
 using "1" "2" unl_lift2 by blast
qed

lemma OPJ3sem:
"(\<sigma> \<Turnstile> \<not>(f oproj skip) )"
unfolding oprojection_d_def skip_defs pfilt_nlength 
using nlength_eq_enat_nfiniteD by auto

subsubsection \<open>PJ4\<close>

lemma PJ4semchaina:
assumes "(\<sigma> \<Turnstile> f  fproj  (g;h))"
 shows  "(\<sigma> \<Turnstile> (f  fproj  g) ; (f  fproj  h))"
proof -
 have 1: "(\<sigma> \<Turnstile> f  fproj  (g;h))" 
  using assms by auto
 have 2: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and>
         nlast  ls  = (the_enat (nlength \<sigma>)) \<and>
         powerinterval f \<sigma> ls \<and>
         (\<exists>n\<le>nlength (pfilt \<sigma> ls). g (ntaken n (pfilt \<sigma> ls)) \<and> h (ndropn n (pfilt \<sigma> ls)))) "
  using assms unfolding chop_defs fprojection_d_def
  by (metis nfinite_conv_nlength_enat pfilt_nlength the_enat.simps)
  obtain ls where 3: "nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and>
         nlast ls  = (the_enat (nlength \<sigma>)) \<and>
         powerinterval f \<sigma> ls \<and>
         (\<exists>n\<le>nlength (pfilt \<sigma> ls). g (ntaken n (pfilt \<sigma> ls)) \<and> h (ndropn n (pfilt \<sigma> ls)))" 
    using "2" by auto
 have 4: "nidx  ls \<and> nnth ls 0 = 0"  
   using "3" by auto
 have 5: "powerinterval f \<sigma> ls" 
   using "3" by auto
 have 6: "nlast  ls  = the_enat (nlength \<sigma>)"
   using "3" by auto
 have 7: "(\<exists>n\<le>nlength (pfilt \<sigma> ls). g (ntaken n (pfilt \<sigma> ls)) \<and> h (ndropn n (pfilt \<sigma> ls)))" 
   using "3" by auto
  obtain n where 8: "n\<le>nlength (pfilt \<sigma> ls) \<and> g (ntaken n (pfilt \<sigma> ls)) \<and> h (ndropn n (pfilt \<sigma> ls)) " 
  using "7" by auto
 have 9: "n\<le>nlength (pfilt \<sigma> ls) " 
  using "8" by auto
 have 10: " g (ntaken n (pfilt \<sigma> ls))" 
  using "8" by auto
 have 11: "h (ndropn n (pfilt \<sigma> ls))" 
  using "8" by auto
 have 12: "nidx  (ntaken n ls) \<and> nnth (ntaken n ls) 0 = 0" 
   using 4 8 unfolding nidx_expand  pfilt_nlength by simp
   (metis Suc_leD bot_nat_0.extremum dual_order.trans enat_ord_simps(1) min.orderE ntaken_nnth) 
 have 13: "nidx  (ndropn n ls) \<and> nnth (ndropn n ls) 0 = (nnth ls n)"
  using 4 9 unfolding pfilt_nlength nidx_expand  
  using nidx_expand nidx_ndropn by auto
 have 131: "((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))) = ndropn n (nmap (\<lambda>x. x - nnth ls n) ls) " 
    using ndropn_nmap[of n "(\<lambda>x. x- (nnth ls n))" ls]  by presburger
 have 132: " nnth (ndropn n (nmap (\<lambda>x. x - nnth ls n) ls)) 0 = 0 "
        by (metis "13" "131" diff_self_eq_0 nnth_nmap zero_enat_def zero_le) 
 have 133: "\<And>j. enat j \<le> nlength ls \<Longrightarrow> nnth (nmap (\<lambda>x. x - nnth ls n) ls) j = nnth ls j - nnth ls n " 
     using nnth_nmap by blast
 have 134: "nlength (ndropn n (nmap (\<lambda>x. x - nnth ls n) ls)) = nlength ls - enat n " 
        by simp 
 have 135: " \<And>j. j \<le> nlength ls - enat n \<Longrightarrow>  
              nnth (ndropn n (nmap (\<lambda>x. x - nnth ls n) ls)) j = nnth ls (n+ j) - nnth ls n  " 
      by (metis "131" ndropn_nlength ndropn_nnth nnth_nmap) 
 have 136: "\<And>j. (Suc j) \<le> nlength ls - enat n \<Longrightarrow>  
              nnth (ndropn n (nmap (\<lambda>x. x - nnth ls n) ls)) (Suc j) = nnth ls (n+ (Suc j)) - nnth ls n" 
    using "135" by blast
 have 137: "\<And>j. (Suc j) \<le> nlength ls - enat n \<Longrightarrow> 
                 nnth ls (n+ j) - nnth ls n <  nnth ls (n+ (Suc j)) - nnth ls n " 
     by (metis "13" Suc_ile_eq diff_less_mono le_add1 le_add_same_cancel1 ndropn_nlength ndropn_nnth 
         nidx_expand nidx_less_eq order_less_imp_le)     
 have 14: "nidx  ( ndropn n (nmap (\<lambda>x. x - nnth ls n) ls)) " 
    by (metis "134" "135" "137" Suc_ile_eq dual_order.strict_iff_order nidx_expand)
 have 15: "g  (pfilt \<sigma> (ntaken n ls))"
   by (metis "10" "9" pfilt_nlength pfilt_ntaken)
 have 16: "h (pfilt \<sigma> (ndropn n ls))"
   by (metis "8" pfilt_ndropn pfilt_nlength)
 have 17: "g  (pfilt (ntaken (nnth ls n) \<sigma>) (ntaken n ls))"
    by (metis "12" "15" nfinite_ntaken nle_le ntaken_all ntaken_nlast pfilt_ntaken_nidx)
 have 170: "nmap (\<lambda>x. nnth \<sigma> (nnth ls n + x)) (nmap (\<lambda>x. x - nnth ls n) (ndropn n ls)) =
            nmap ((\<lambda>x. nnth \<sigma> (nnth ls n + x)) \<circ> (\<lambda>x. x - nnth ls n)) (ndropn n ls) " 
        using nellist.map_comp by blast
 have 171: "nmap ((\<lambda>x. nnth \<sigma> (nnth ls n + x)) \<circ> (\<lambda>x. x - nnth ls n)) (ndropn n ls) =
            nmap (nnth \<sigma>) (ndropn n ls)"
      proof -
       have 172: "nlength (nmap ((\<lambda>x. nnth \<sigma> (nnth ls n + x)) \<circ> (\<lambda>x. x - nnth ls n)) (ndropn n ls)) =
                  nlength (nmap (nnth \<sigma>) (ndropn n ls))"
               by auto 
       have 173: "\<And>j. j\<le> nlength (nmap (nnth \<sigma>) (ndropn n ls)) \<Longrightarrow>
                      nnth (nmap ((\<lambda>x. nnth \<sigma> (nnth ls n + x)) \<circ> (\<lambda>x. x - nnth ls n)) (ndropn n ls)) j =
                      nnth (nmap (nnth \<sigma>) (ndropn n ls)) j"
         by auto 
         (metis "13" "131" "134" le_add1 le_add_diff_inverse le_add_same_cancel1 ndropn_nnth 
          nidx_less_eq nlength_nmap) 
       show ?thesis 
       by (metis "172" "173" nellist_eq_nnth_eq)
      qed
 have 171: "(pfilt (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) =
            (pfilt \<sigma> (ndropn n ls))" 
       using pfilt_nmap[of "(ndropn (nnth ls n) \<sigma>)" "((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))"]
             pfilt_nmap[of \<sigma> "(ndropn n ls)"]  "170" "171" by force
        
 have 18: "h (pfilt (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))))" 
     using "16" "171" by auto
 have 19: "powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls)"
    by (metis "3" "9"  nfinite_conv_nlength_enat pfilt_nlength powerinterval_splita)
 have 20: "powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))" 
   using powerinterval_split[of ls n \<sigma> f] 
   by (metis "3" "9" pfilt_nlength)
 have 21: "(nnth ls n) \<le> nlength \<sigma> "
   by (metis "3" "9" enat_ord_simps(2) nfinite_conv_nlength_enat nidx_less_last_1 nnth_nlast 
       order.order_iff_strict pfilt_nlength the_enat.simps)
 have 22: "nnth (ntaken n ls) (the_enat(nlength (ntaken n ls))) = (nnth ls n)"
    by (metis "9" min_def ntaken_nlength ntaken_nnth  pfilt_nlength the_enat.simps)
 have 222: "nlast (ntaken n ls) = (nnth ls n)"
     by (simp add: ntaken_nlast) 
 have 23: " nnth ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))
                (the_enat(nlength ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))))) =
            (the_enat(nlength \<sigma>)) -(nnth ls n)" 
    by (metis "171" "222" "3" "9" enat_le_plus_same(2) enat_ord_simps(3) enat_the_enat gen_nlength_def
         ndropn_nfirst nfinite_ndropn_a nfinite_ntaken nfuse_ntaken_ndropn nlast_nfuse nlength_code 
        nnth_nlast nnth_nmap pfilt_nlength)
 have 223: "nlast ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))) =  (the_enat(nlength \<sigma>)) -(nnth ls n)"
   by (metis "23" "3" nfinite_ndropn_a nfinite_nmap nnth_nlast) 
 have 224: "(nmap (\<lambda>x. x+ (nnth ls n) ) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) = ndropn n ls"
     proof -
      have 2241: "nlength (nmap (\<lambda>x. x+ (nnth ls n) ) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) = 
                  nlength (ndropn n ls)"
           by auto
      have 2242: "\<And>j. j\<le>nlength (ndropn n ls) \<longrightarrow> 
                    nnth (nmap (\<lambda>x. x+ (nnth ls n) ) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) j =
                    nnth (ndropn n ls) j"
           by (metis "13" eq_imp_le le_add_diff_inverse2 nidx_gr_first nlength_nmap nnth_nmap 
               not_gr_zero order_less_imp_le)
      show ?thesis using 2241 2242 nellist_eq_nnth_eq 
      by metis            
   qed
 have 24: "ls = nfuse (ntaken n ls) 
                    (nmap (\<lambda>x. x+ (nnth ls n) ) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))))"
   using nfuse_ntaken_ndropn 
   by (metis "224" "9" pfilt_nlength)
 have 241: "nfinite (ntaken n ls)"
    by simp 
 have 242: " nfinite (ntaken (nnth ls n) \<sigma>)"
    by simp 
 have 243: "nlast (ntaken n ls) = (the_enat(nlength (ntaken (nnth ls n) \<sigma>)))" 
    by (simp add: "21" "222")
 have 25: "(\<exists>ls1. nidx  ls1 \<and> nnth ls1 0 = 0 \<and> nfinite ls1 \<and> nfinite (ntaken (nnth ls n) \<sigma>) \<and>
              nlast ls1 = (the_enat(nlength (ntaken (nnth ls n) \<sigma>))) \<and>
             powerinterval f (ntaken (nnth ls n) \<sigma>) ls1 \<and> g (pfilt (ntaken (nnth ls n) \<sigma>) ls1))"
    using "12" "17" "19" "241" "242" "243" by blast
 have 251: "nfinite (ndropn (nnth ls n) \<sigma>)"
    by (simp add: "3") 
 have 252: "nlast (nmap (\<lambda>x. x - nnth ls n) (ndropn n ls)) = the_enat(nlength (ndropn (nnth ls n) \<sigma>))" 
     by (metis "223" "3" idiff_enat_enat ndropn_nlength nfinite_nlength_enat the_enat.simps)
 have 26: "(\<exists>ls2. nidx  ls2 \<and> nnth ls2 0 = 0 \<and> nfinite ls2 \<and> nfinite (ndropn (nnth ls n) \<sigma>) \<and>
              nlast ls2 = the_enat(nlength (ndropn (nnth ls n) \<sigma>)) \<and>
             powerinterval f (ndropn (nnth ls n) \<sigma>) ls2 \<and> h (pfilt (ndropn (nnth ls n) \<sigma>) ls2))"
   by (metis "131" "132" "14" "18" "20" "252" "3" nfinite_ndropn_a nfinite_nmap)
 have 27: "(ntaken (nnth ls n) \<sigma>) \<Turnstile> f fproj g"
     by (metis "25" fprojection_d_def)
 have 28: " (ndropn (nnth ls n) \<sigma>) \<Turnstile> f fproj h"  
    by (metis "26" fprojection_d_def nfinite_nlength_enat the_enat.simps)
  show ?thesis 
  using "21" "27" "28" chop_defs by auto
qed

lemma OPJ4semchaina:
assumes "(\<sigma> \<Turnstile> f oproj ((g \<and> finite);h))"
 shows  "(\<sigma> \<Turnstile> (f  fproj  g) ; (f oproj h))"
proof -
 have 1: "(\<sigma> \<Turnstile> f oproj ((g \<and> finite);h))" 
  using assms by auto
 have 2: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and>
         powerinterval f \<sigma> ls \<and>
         (\<exists>n\<le>nlength (pfilt \<sigma> ls). g (ntaken n (pfilt \<sigma> ls)) \<and> h (ndropn n (pfilt \<sigma> ls)))) "
  using assms unfolding chop_defs oprojection_d_def finite_defs 
  by auto
  obtain ls where 3: "nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and>
         powerinterval f \<sigma> ls \<and>
         (\<exists>n\<le>nlength (pfilt \<sigma> ls). g (ntaken n (pfilt \<sigma> ls)) \<and> h (ndropn n (pfilt \<sigma> ls)))" 
    using "2" by auto
 have 4: "nidx  ls \<and> nnth ls 0 = 0"  
   using "3" by auto
 have 5: "powerinterval f \<sigma> ls" 
   using "3" by auto
 have 7: "(\<exists>n\<le>nlength (pfilt \<sigma> ls). g (ntaken n (pfilt \<sigma> ls)) \<and> h (ndropn n (pfilt \<sigma> ls)))" 
   using "3" by auto
  obtain n where 8: "n\<le>nlength (pfilt \<sigma> ls) \<and> g (ntaken n (pfilt \<sigma> ls)) \<and> h (ndropn n (pfilt \<sigma> ls)) " 
  using "7" by auto
 have 9: "n\<le>nlength (pfilt \<sigma> ls) " 
  using "8" by auto
 have 10: " g (ntaken n (pfilt \<sigma> ls))" 
  using "8" by auto
 have 11: "h (ndropn n (pfilt \<sigma> ls))" 
  using "8" by auto
 have 12: "nidx  (ntaken n ls) \<and> nnth (ntaken n ls) 0 = 0" 
   using 4 8 unfolding nidx_expand  pfilt_nlength by simp
   (metis "3" Suc_leD bot_nat_0.extremum min.orderE nfinite_ntaken nle_le ntaken_all ntaken_nnth) 
 have 13: "nidx  (ndropn n ls) \<and> nnth (ndropn n ls) 0 = (nnth ls n)"
  using 4 9 unfolding pfilt_nlength nidx_expand  
  using nidx_expand nidx_ndropn by auto
 have 131: "((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))) = ndropn n (nmap (\<lambda>x. x - nnth ls n) ls) " 
    using ndropn_nmap[of n "(\<lambda>x. x- (nnth ls n))" ls]  by presburger
 have 132: " nnth (ndropn n (nmap (\<lambda>x. x - nnth ls n) ls)) 0 = 0 "
        by (metis "13" "131" diff_self_eq_0 nnth_nmap zero_enat_def zero_le) 
 have 133: "\<And>j. enat j \<le> nlength ls \<Longrightarrow> nnth (nmap (\<lambda>x. x - nnth ls n) ls) j = nnth ls j - nnth ls n " 
     using nnth_nmap by blast
 have 134: "nlength (ndropn n (nmap (\<lambda>x. x - nnth ls n) ls)) = nlength ls - enat n " 
        by simp 
 have 135: " \<And>j. j \<le> nlength ls - enat n \<Longrightarrow>  
              nnth (ndropn n (nmap (\<lambda>x. x - nnth ls n) ls)) j = nnth ls (n+ j) - nnth ls n  " 
      by (metis "131" ndropn_nlength ndropn_nnth nnth_nmap) 
 have 136: "\<And>j. (Suc j) \<le> nlength ls - enat n \<Longrightarrow>  
              nnth (ndropn n (nmap (\<lambda>x. x - nnth ls n) ls)) (Suc j) = nnth ls (n+ (Suc j)) - nnth ls n" 
    using "135" by blast
 have 137: "\<And>j. (Suc j) \<le> nlength ls - enat n \<Longrightarrow> 
                 nnth ls (n+ j) - nnth ls n <  nnth ls (n+ (Suc j)) - nnth ls n " 
     by (metis "13" Suc_ile_eq diff_less_mono le_add1 le_add_same_cancel1 ndropn_nlength ndropn_nnth 
         nidx_expand nidx_less_eq order_less_imp_le)   
 have 14: "nidx  ( ndropn n (nmap (\<lambda>x. x - nnth ls n) ls))" 
   by (metis "134" "135" "137" Suc_ile_eq dual_order.order_iff_strict nidx_expand)
 have 15: "g  (pfilt \<sigma> (ntaken n ls))"
   by (metis "10" "9" pfilt_nlength pfilt_ntaken)
 have 16: "h (pfilt \<sigma> (ndropn n ls))"
   by (metis "8" pfilt_ndropn pfilt_nlength)
 have 17: "g  (pfilt (ntaken (nnth ls n) \<sigma>) (ntaken n ls))"
    by (metis "12" "15" nfinite_ntaken nle_le ntaken_all ntaken_nlast pfilt_ntaken_nidx)
 have 170: "nmap (\<lambda>x. nnth \<sigma> (nnth ls n + x)) (nmap (\<lambda>x. x - nnth ls n) (ndropn n ls)) =
            nmap ((\<lambda>x. nnth \<sigma> (nnth ls n + x)) \<circ> (\<lambda>x. x - nnth ls n)) (ndropn n ls) " 
        using nellist.map_comp by blast
 have 171: "nmap ((\<lambda>x. nnth \<sigma> (nnth ls n + x)) \<circ> (\<lambda>x. x - nnth ls n)) (ndropn n ls) =
            nmap (nnth \<sigma>) (ndropn n ls)"
      proof -
       have 172: "nlength (nmap ((\<lambda>x. nnth \<sigma> (nnth ls n + x)) \<circ> (\<lambda>x. x - nnth ls n)) (ndropn n ls)) =
                  nlength (nmap (nnth \<sigma>) (ndropn n ls))"
               by auto 
       have 173: "\<And>j. j\<le> nlength (nmap (nnth \<sigma>) (ndropn n ls)) \<Longrightarrow>
                      nnth (nmap ((\<lambda>x. nnth \<sigma> (nnth ls n + x)) \<circ> (\<lambda>x. x - nnth ls n)) (ndropn n ls)) j =
                      nnth (nmap (nnth \<sigma>) (ndropn n ls)) j"
         by auto 
         (metis "13" "131" "134" le_add1 le_add_diff_inverse le_add_same_cancel1 ndropn_nnth 
          nidx_less_eq nlength_nmap) 
       show ?thesis 
       by (metis "172" "173" nellist_eq_nnth_eq)
      qed
 have 171: "(pfilt (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) =
            (pfilt \<sigma> (ndropn n ls))" 
       using pfilt_nmap[of "(ndropn (nnth ls n) \<sigma>)" "((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))"]
             pfilt_nmap[of \<sigma> "(ndropn n ls)"]  "170" "171" by force
 have 18: "h (pfilt (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))))" 
       using "16" "171" by auto
 have 19: "powerinterval f (ntaken (nnth ls n) \<sigma>) (ntaken n ls)"
     by (metis "3" "8" pfilt_nlength powerinterval_splita_alt)
 have 20: "powerinterval f (ndropn (nnth ls n) \<sigma>) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))" 
   using powerinterval_split[of ls n \<sigma> f] 
   by (metis "3" "8" pfilt_nlength powerinterval_splitb_alt)
 have 21: "(nnth ls n) \<le> nlength \<sigma> "
   by (metis "3" linorder_le_cases nfinite_ntaken ntaken_all)
 have 22: "nnth (ntaken n ls) (the_enat(nlength (ntaken n ls))) = (nnth ls n)"
    by (metis "9" min_def ntaken_nlength ntaken_nnth  pfilt_nlength the_enat.simps)
 have 222: "nlast (ntaken n ls) = (nnth ls n)"
     by (simp add: ntaken_nlast) 
 have 224: "(nmap (\<lambda>x. x+ (nnth ls n) ) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) = ndropn n ls"
     proof -
      have 2241: "nlength (nmap (\<lambda>x. x+ (nnth ls n) ) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) = 
                  nlength (ndropn n ls)"
           by auto
      have 2242: "\<And>j. j\<le>nlength (ndropn n ls) \<longrightarrow> 
                    nnth (nmap (\<lambda>x. x+ (nnth ls n) ) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls)))) j =
                    nnth (ndropn n ls) j"
      by (metis "13" "132" "9" diff_is_0_eq le_add_diff_inverse2 nidx_gr_first nlength_nmap nnth_nmap
           nnth_zero_ndropn not_gr_zero order_less_imp_le pfilt_nlength)
      show ?thesis using 2241 2242 nellist_eq_nnth_eq 
      by metis            
   qed
 have 24: "ls = nfuse (ntaken n ls) 
                    (nmap (\<lambda>x. x+ (nnth ls n) ) ((nmap (\<lambda>x. x- (nnth ls n))  (ndropn n ls))))"
   using nfuse_ntaken_ndropn 
   by (metis "224" "9" pfilt_nlength)
 have 25: "(\<exists>ls1. nidx  ls1 \<and> nnth ls1 0 = 0 \<and> nfinite ls1 \<and> nfinite (ntaken (nnth ls n) \<sigma>) \<and>
              nlast ls1 = (the_enat(nlength (ntaken (nnth ls n) \<sigma>))) \<and>
             powerinterval f (ntaken (nnth ls n) \<sigma>) ls1 \<and> g (pfilt (ntaken (nnth ls n) \<sigma>) ls1))"
       using "12" "17" "19" "21" "222" "3"  
       by (metis min.orderE nfinite_ntaken ntaken_nlength the_enat.simps)
 have 26: "(\<exists>ls2. nidx  ls2 \<and> nnth ls2 0 = 0 \<and> \<not>nfinite ls2 \<and> \<not>nfinite (ndropn (nnth ls n) \<sigma>) \<and>
             powerinterval f (ndropn (nnth ls n) \<sigma>) ls2 \<and> h (pfilt (ndropn (nnth ls n) \<sigma>) ls2))"
    using assms "14" "18" "20"  "3" 132 131 by (metis nfinite_ndropn nfinite_nmap)    
 have 27: "(ntaken (nnth ls n) \<sigma>) \<Turnstile> f fproj g"
     by (metis "25" fprojection_d_def)
 have 28: " (ndropn (nnth ls n) \<sigma>) \<Turnstile> f oproj h"  
    by (metis "26" oprojection_d_def)
  show ?thesis 
  using "21" "27" "28" chop_defs by auto
qed

lemma PJ4semchainb:
assumes "(\<sigma> \<Turnstile> (f  fproj  g) ; (f  fproj  h))"
 shows  "(\<sigma> \<Turnstile> f  fproj  (g;h))"
proof - 
 have 1: "(\<exists>n\<le>nlength \<sigma>.
        (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite (ntaken n \<sigma>) \<and>
              nlast ls  = (the_enat(nlength (ntaken n \<sigma>))) \<and>
             powerinterval f (ntaken n \<sigma>) ls \<and> g (pfilt (ntaken n \<sigma>) ls)) \<and>
        (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite (ndropn n \<sigma>) \<and> 
              nlast ls = (the_enat(nlength (ndropn n \<sigma>))) \<and>
             powerinterval f (ndropn n \<sigma>) ls \<and> h (pfilt (ndropn n \<sigma>) ls))) "
    using assms unfolding chop_defs  fprojection_d_def by simp  blast
 obtain cp where 2: "cp\<le>nlength \<sigma> \<and> 
               (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite (ntaken cp \<sigma>) \<and>
              nlast ls  = (the_enat(nlength (ntaken cp \<sigma>))) \<and>
             powerinterval f (ntaken cp \<sigma>) ls \<and> g (pfilt (ntaken cp \<sigma>) ls)) \<and>
        (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite (ndropn cp \<sigma>) \<and> 
              nlast ls = (the_enat(nlength (ndropn cp \<sigma>))) \<and>
             powerinterval f (ndropn cp \<sigma>) ls \<and> h (pfilt (ndropn cp \<sigma>) ls))"
   using "1" by auto
 have 3: "cp\<le>nlength \<sigma>"
  using "2" by auto
 have 4: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite (ntaken cp \<sigma>) \<and>
              nlast ls  = (the_enat(nlength (ntaken cp \<sigma>))) \<and>
             powerinterval f (ntaken cp \<sigma>) ls \<and> g (pfilt (ntaken cp \<sigma>) ls))" 
  using "2" by auto
 obtain ls1 where 5: "nidx  ls1 \<and> nnth ls1 0 = 0 \<and> nfinite ls1 \<and> nfinite (ntaken cp \<sigma>) \<and>
              nlast ls1  = (the_enat(nlength (ntaken cp \<sigma>))) \<and>
             powerinterval f (ntaken cp \<sigma>) ls1 \<and> g (pfilt (ntaken cp \<sigma>) ls1)"
   using "4" by auto
 have 6: "nidx  ls1 \<and> nnth ls1 0 = 0"  
   using "5" by auto
 have 61: "nfinite ls1 \<and> nfinite (ntaken cp \<sigma>)"
    using 5 by auto 
 have 7: " nlast ls1  = the_enat(nlength (ntaken cp \<sigma>))"  
   using "5" by auto
 have 8: "powerinterval f (ntaken cp \<sigma>) ls1" 
    using "5" by auto
 have 9: "g (pfilt (ntaken cp \<sigma>) ls1)"
   using "5" by auto
 have 10: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite (ndropn cp \<sigma>) \<and> 
              nlast ls = (the_enat(nlength (ndropn cp \<sigma>))) \<and>
             powerinterval f (ndropn cp \<sigma>) ls \<and> h (pfilt (ndropn cp \<sigma>) ls))" 
   using "2" by auto
 obtain ls2 where 11: "nidx  ls2 \<and> nnth ls2 0 = 0 \<and> nfinite ls2 \<and> nfinite (ndropn cp \<sigma>) \<and> 
              nlast ls2 = (the_enat(nlength (ndropn cp \<sigma>))) \<and>
             powerinterval f (ndropn cp \<sigma>) ls2 \<and> h (pfilt (ndropn cp \<sigma>) ls2)"  
   using "10" by auto
 have 12: "nidx  ls2 \<and> nnth ls2 0 = 0" 
   using "11" by auto
 have 13: " nlast ls2  = the_enat(nlength (ndropn cp \<sigma>))"
     using "11" by auto
 have 14: "powerinterval f (ndropn cp \<sigma>) ls2" 
    using "11" by auto
 have 15: "h (pfilt (ndropn cp \<sigma>) ls2)" 
    using "11" by auto
 have 150: "nlast ls1 = cp"
    by (simp add: "2" "7")
 have 151: "nfirst (nmap (\<lambda>x. x+ cp) ls2) = cp"
    by (metis "11" NNil_eq_nmap_conv nlast_NNil ntake_eq_NNil_iff ntake_nmap ntaken_0 ntaken_nlast 
        plus_nat.add_0) 
 have 152: "nnth (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) 0 = 0"
   by (metis "150" "151" "5" nfuse_nnth zero_enat_def zero_le)
 have 153: "nidx ls1  \<and> nnth ls1 0 = 0 \<and> nfinite ls1 \<and> nlast ls1 = cp "
   by (simp add: "150" "5")
 have 154: "nidx ls2 \<and> nnth ls2 0 = 0 \<and> nfinite ls2 \<and> nfinite \<sigma> \<and> nlast ls2 = the_enat (nlength \<sigma>) - cp" 
   by (metis "11" idiff_enat_enat ndropn_nlength nfinite_conv_nlength_enat nfinite_ndropn the_enat.simps)   
 have 16: "nidx  (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) \<and> nnth (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) 0 = 0"
    using nidx_nfuse_nidx[of ls1 ls2 cp \<sigma> ]  153 154 3 by fastforce
 have 17: "nlast ls1 = nfirst (nmap (\<lambda>x. x+ cp) ls2)"
    by (simp add: "150" "151")
 have 18: "nnth (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) (the_enat(nlength ls1)) = cp"
   by (metis "151" "17" "5" ntaken_nfuse ntaken_nlast)
 have 19: "nlast (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))  = (the_enat (nlength \<sigma>)) " 
   by (metis "154" "17" "3" "5" diff_add enat.distinct(2) enat_ord_simps(1) enat_the_enat 
       nfinite_conv_nlength_enat nlast_nfuse nlast_nmap)
 have 20: "powerinterval f \<sigma> (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))" 
   using powerinterval_nfuse[of "ls1" "(ls2)" "cp" "\<sigma>" "f"] "11" "150" "154" "3" "5" by fastforce
 have 21: "\<sigma> = nfuse (ntaken cp \<sigma>) (ndropn cp \<sigma>)"
      by (simp add: "2" nfuse_ntaken_ndropn)
 have 22: "nnth ((nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))) (the_enat(nlength ls1)) = cp"
     using "18" by blast
 have 23: " (ntaken (the_enat(nlength ls1)) (pfilt \<sigma> (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) )) =
            (pfilt (ntaken cp \<sigma>) ls1)" 
   by (metis "151" "17" "2" "5" enat.distinct(2) enat_le_plus_same(1) enat_the_enat 
       nfinite_conv_nlength_enat nfuse_nlength ntaken_nfuse pfilt_ntaken pfilt_ntaken_nidx)
 have 24: "g (ntaken (the_enat(nlength ls1)) (pfilt \<sigma> (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) ))"
   by (simp add: "23" "9")    
 have 25: "(ndropn (the_enat(nlength ls1)) (pfilt \<sigma> (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) )) =
           (pfilt (ndropn cp \<sigma>) ls2)"
   using pfilt_ndropn[of "(the_enat(nlength ls1))" "(nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))" \<sigma>]  
         pfilt_ndropn_nidx[of cp \<sigma> ls2 ] 
   by (metis "12" "17" "23" "3" "5" enat_ord_code(4) enat_the_enat min_def 
       ndropn_nfuse nle_le ntaken_nlength order_less_imp_le pfilt_nlength)
 have 26: "nlength ls1 \<le> nlength (pfilt \<sigma> (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)))"
       by (simp add: nfuse_nlength pfilt_nlength)
 have 27: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and>
          nlast ls  = the_enat(nlength \<sigma>) \<and>
         powerinterval f \<sigma> ls \<and>
         (\<exists>n\<le>nlength (pfilt \<sigma> ls). g (ntaken n (pfilt \<sigma> ls)) \<and> h (ndropn n (pfilt \<sigma> ls))))"
      using 11 16 19 20 24 25 26
      by (metis "5" nfinite_ndropn nfinite_nlength_enat nfinite_nmap pfilt_nmap the_enat.simps)  
 show ?thesis unfolding fprojection_d_def using chop_nfuse nfuse_ntaken_ndropn 27 
 by (metis  ndropn_nfirst nfinite_conv_nlength_enat nfinite_ntaken ntaken_nlast)
qed 


lemma OPJ4semchainb:
assumes "(\<sigma> \<Turnstile> (f  fproj  g) ; (f oproj h))"
 shows  "(\<sigma> \<Turnstile> f oproj ((g \<and> finite);h))"
proof - 
 have 1: "(\<exists>n\<le>nlength \<sigma>.
        (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite (ntaken n \<sigma>) \<and>
              nlast ls  = (the_enat(nlength (ntaken n \<sigma>))) \<and>
             powerinterval f (ntaken n \<sigma>) ls \<and> g (pfilt (ntaken n \<sigma>) ls)) \<and>
        (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite (ndropn n \<sigma>) \<and> 
             powerinterval f (ndropn n \<sigma>) ls \<and> h (pfilt (ndropn n \<sigma>) ls))) "
    using assms unfolding chop_defs  fprojection_d_def oprojection_d_def
   by simp blast  
 obtain cp where 2: "cp\<le>nlength \<sigma> \<and> 
               (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite (ntaken cp \<sigma>) \<and>
              nlast ls  = (the_enat(nlength (ntaken cp \<sigma>))) \<and>
             powerinterval f (ntaken cp \<sigma>) ls \<and> g (pfilt (ntaken cp \<sigma>) ls)) \<and>
        (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite (ndropn cp \<sigma>) \<and> 
             powerinterval f (ndropn cp \<sigma>) ls \<and> h (pfilt (ndropn cp \<sigma>) ls))"
   using "1" by auto
 have 3: "cp\<le>nlength \<sigma>"
  using "2" by auto
 have 4: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite (ntaken cp \<sigma>) \<and>
              nlast ls  = (the_enat(nlength (ntaken cp \<sigma>))) \<and>
             powerinterval f (ntaken cp \<sigma>) ls \<and> g (pfilt (ntaken cp \<sigma>) ls))" 
  using "2" by auto
 obtain ls1 where 5: "nidx  ls1 \<and> nnth ls1 0 = 0 \<and> nfinite ls1 \<and> nfinite (ntaken cp \<sigma>) \<and>
              nlast ls1  = (the_enat(nlength (ntaken cp \<sigma>))) \<and>
             powerinterval f (ntaken cp \<sigma>) ls1 \<and> g (pfilt (ntaken cp \<sigma>) ls1)"
   using "4" by auto
 have 6: "nidx  ls1 \<and> nnth ls1 0 = 0"  
   using "5" by auto
 have 61: "nfinite ls1 \<and> nfinite (ntaken cp \<sigma>)"
    using 5 by auto 
 have 7: " nlast ls1  = the_enat(nlength (ntaken cp \<sigma>))"  
   using "5" by auto
 have 8: "powerinterval f (ntaken cp \<sigma>) ls1" 
    using "5" by auto
 have 9: "g (pfilt (ntaken cp \<sigma>) ls1)"
   using "5" by auto
 have 10: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite (ndropn cp \<sigma>) \<and> 
             powerinterval f (ndropn cp \<sigma>) ls \<and> h (pfilt (ndropn cp \<sigma>) ls))" 
   using "2" by auto
 obtain ls2 where 11: "nidx  ls2 \<and> nnth ls2 0 = 0 \<and> \<not>nfinite ls2 \<and> \<not>nfinite (ndropn cp \<sigma>) \<and> 
             powerinterval f (ndropn cp \<sigma>) ls2 \<and> h (pfilt (ndropn cp \<sigma>) ls2)"  
   using "10" by auto
 have 12: "nidx  ls2 \<and> nnth ls2 0 = 0" 
   using "11" by auto
 have 14: "powerinterval f (ndropn cp \<sigma>) ls2" 
    using "11" by auto
 have 15: "h (pfilt (ndropn cp \<sigma>) ls2)" 
    using "11" by auto
  have 150: "nlast ls1 = cp"
    by (simp add: "2" "7")
 have 151: "nfirst (nmap (\<lambda>x. x+ cp) ls2) = cp"
    by (metis "11" NNil_eq_nmap_conv nlast_NNil ntake_eq_NNil_iff ntake_nmap ntaken_0 ntaken_nlast 
        plus_nat.add_0) 
 have 152: "nnth (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) 0 = 0"
   by (metis "150" "151" "5" nfuse_nnth zero_enat_def zero_le)
 have 153: "nidx ls1  \<and> nnth ls1 0 = 0 \<and> nfinite ls1 \<and> nlast ls1 = cp "
   by (simp add: "150" "5")
 have 154: "nidx ls2 \<and> nnth ls2 0 = 0 \<and> \<not>nfinite ls2 \<and> \<not>nfinite \<sigma> "
    using "11" nfinite_ndropn_a by blast  
 have 16: "nidx  (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))  \<and> nnth (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) 0 = 0 "
    using nidx_nfuse_nidx_infinite[of ls1 ls2 cp \<sigma>  ]  "150" "154" "5" by fastforce
 have 17: "nlast ls1 = nfirst (nmap (\<lambda>x. x+ cp) ls2)"
   by (simp add: "150" "151")
 have 18: "nnth (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) (the_enat(nlength ls1)) = cp"
      by (metis "150" "17" "5" ntaken_nfuse ntaken_nlast)  
 have 20: "powerinterval f \<sigma> (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))" 
   using powerinterval_nfuse_alt[of "ls1" "(ls2)" "cp" "\<sigma>" "f"]  
     using "14" "154" "3" "5" by fastforce
 have 21: "\<sigma> = nfuse (ntaken cp \<sigma>) (ndropn cp \<sigma>)"
      by (simp add: "2" nfuse_ntaken_ndropn)
 have 22: "nnth ((nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))) (the_enat(nlength ls1)) = cp"
     using "18" by blast
 have 23: " (ntaken (the_enat(nlength ls1)) (pfilt \<sigma> (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) )) =
            (pfilt (ntaken cp \<sigma>) ls1)"
    by (metis "150" "17" "2" "5" enat.distinct(2) enat_le_plus_same(1) enat_the_enat 
        nfinite_conv_nlength_enat nfuse_nlength ntaken_nfuse pfilt_ntaken pfilt_ntaken_nidx)
 have 24: "g (ntaken (the_enat(nlength ls1)) (pfilt \<sigma> (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) ))"
   by (simp add: "23" "9")    
 have 25: "(ndropn (the_enat(nlength ls1)) (pfilt \<sigma> (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)) )) =
           (pfilt (ndropn cp \<sigma>) ls2)"
   using pfilt_ndropn[of "(the_enat(nlength ls1))" "(nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2))" \<sigma>]  
         pfilt_ndropn_nidx[of cp \<sigma> ls2 ] 
         by (metis "12" "17" "2" "23" "5" enat_ord_code(4) enat_the_enat min_def ndropn_nfuse nle_le 
         ntaken_nlength order_less_imp_le pfilt_nlength)  
 have 26: "nlength ls1 \<le> nlength (pfilt \<sigma> (nfuse ls1 (nmap (\<lambda>x. x+ cp) ls2)))"
       by (simp add: nfuse_nlength pfilt_nlength)
 have 27: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and>
         powerinterval f \<sigma> ls \<and>
         (\<exists>n\<le>nlength (pfilt \<sigma> ls). g (ntaken n (pfilt \<sigma> ls)) \<and> h (ndropn n (pfilt \<sigma> ls))))"
      using 11 16  20 24 25 26
      by (metis "154" "5" nfinite_nlength_enat nfinite_nmap nfuse_nfinite the_enat.simps)
 show ?thesis unfolding fprojection_d_def oprojection_d_def  chop_nfuse   nfuse_ntaken_ndropn 
      finite_defs by simp 
       (metis "27" ndropn_nfirst nfinite_ntaken nfuse_ntaken_ndropn ntaken_nlast)
qed 

lemma PJ4sem:
"(\<sigma> \<Turnstile> f  fproj  (g;h) = (f  fproj  g) ; (f  fproj  h))"
using PJ4semchaina PJ4semchainb unl_lift2 by blast

lemma OPJ4sem:
"(\<sigma> \<Turnstile> f oproj ((g \<and> finite);h) = (f  fproj  g) ; (f oproj h))"
using OPJ4semchaina OPJ4semchainb unl_lift2 by blast


subsubsection \<open>PJ5\<close>

lemma PJ5sem:
 "(\<sigma> \<Turnstile> f  fproj  init(g) \<longrightarrow> init(g))"
by (simp add: fprojection_d_def init_defs)
 (metis ndropn_0 ndropn_nfirst ntaken_0 pfilt_code(1) pfilt_ntaken zero_enat_def zero_le)

lemma OPJ5sem:
 "(\<sigma> \<Turnstile> f oproj init(g) \<longrightarrow> init(g))"
by (simp add: oprojection_d_def init_defs)
 (metis nlast_NNil ntaken_0 ntaken_nlast pfilt_nlength pfilt_nnth zero_enat_def zero_le)


subsubsection \<open>PJ6\<close>

lemma PJ6help1:
 assumes "nidx  ls"
         "nnth ls 0 = 0 "
         "nfinite ls"
         "nfinite \<sigma> " 
         " nlast ls = the_enat(nlength \<sigma>)"
 shows   "(\<forall> i. 0\<le> i \<and> i<nlength ls \<longrightarrow> nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )
          = (nnth ls (Suc i)) - (nnth ls i)) "
proof 
  fix i
  show "0 \<le> i \<and> i < nlength ls \<longrightarrow>
         nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) = nnth ls (Suc i) - nnth ls i"
      using assms nsubn_nlength[of \<sigma> ] unfolding nidx_expand 
      by (metis assms(1) dual_order.order_iff_strict eSuc_enat enat_minus_mono1 enat_ord_simps(1) 
          idiff_enat_enat ileI1 min.orderE nfinite_conv_nlength_enat nidx_less_last_1 nnth_nlast 
          the_enat.simps)
qed

lemma OPJ6help1:
 assumes "nidx  ls"
         "nnth ls 0 = 0 "
         "\<not>nfinite ls"
         "\<not>nfinite \<sigma> " 
 shows   "(\<forall> i. 0\<le> i \<and> i<nlength ls \<longrightarrow> nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )
          = (nnth ls (Suc i)) - (nnth ls i)) "
proof 
  fix i
  show "0 \<le> i \<and> i < nlength ls \<longrightarrow>
         nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) = nnth ls (Suc i) - nnth ls i"
  using assms unfolding nidx_expand nsubn_def1 by (simp add: nfinite_conv_nlength_enat)
qed


lemma PJ6help2:
assumes " nidx  ls"
        " nnth ls 0 = 0 " 
        "nfinite ls " 
        "nfinite \<sigma>" 
        " nlast ls  = the_enat(nlength \<sigma>) \<and>
         (\<forall>i<nlength ls. nnth ls (Suc i) - nnth ls i = Suc 0) "
shows   " (\<forall>i\<le>nlength ls. nnth ls i = i)"
proof 
 fix i
 show " i\<le>nlength ls \<longrightarrow> nnth ls i = i"
 proof
  (induct i)
  case 0
  then show ?case using assms by blast
  next
  case (Suc i)
  then show ?case
  by (metis One_nat_def Suc_ile_eq assms(1) assms(5) diff_add nidx_expand order_less_imp_le 
      plus_1_eq_Suc)
  qed
qed


lemma OPJ6help2:
assumes " nidx  ls"
        " nnth ls 0 = 0 " 
        "\<not>nfinite ls " 
        "\<not>nfinite \<sigma>" 
        "(\<forall>i<nlength ls. nnth ls (Suc i) - nnth ls i = Suc 0) "
shows   " (\<forall>i\<le>nlength ls. nnth ls i = i)"
proof 
 fix i
 show " i\<le>nlength ls \<longrightarrow> nnth ls i = i"
 proof
  (induct i)
  case 0
  then show ?case using assms by blast
  next
  case (Suc i)
  then show ?case
  by (metis One_nat_def Suc_ile_eq assms(1) assms(5) diff_add nidx_expand order_less_imp_le 
      plus_1_eq_Suc)
  qed
qed

lemma PJ6help3:
assumes " nidx  ls"
        " nnth ls 0 = 0 " 
        "nfinite ls " 
        "nfinite \<sigma>" 
        " nlast  ls  = the_enat(nlength \<sigma>)"
        " (\<forall>i\<le>nlength ls. nnth ls i = i) "
shows   " (\<forall>i<nlength ls. nnth ls (Suc i) - nnth ls i = Suc 0)"
proof
 fix i
 show "i<nlength ls \<longrightarrow> nnth ls (Suc i) - nnth ls i = Suc 0"
    by (metis One_nat_def add_diff_cancel_right' assms(6) eSuc_enat ileI1 order_less_imp_le 
        plus_1_eq_Suc)
qed

lemma OPJ6help3:
assumes " nidx  ls"
        " nnth ls 0 = 0 " 
        "\<not>nfinite ls " 
        "\<not>nfinite \<sigma>" 
        " (\<forall>i\<le>nlength ls. nnth ls i = i) "
shows   " (\<forall>i<nlength ls. nnth ls (Suc i) - nnth ls i = Suc 0)"
proof
 fix i
 show "i<nlength ls \<longrightarrow> nnth ls (Suc i) - nnth ls i = Suc 0"
   by (metis One_nat_def add_diff_cancel_right' assms(5) eSuc_enat ileI1 order_less_imp_le 
       plus_1_eq_Suc)
qed



lemma PJ6help4:
assumes "nfinite \<sigma>" 
shows " (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and>
         ls = ntaken (the_enat (nlength \<sigma>)) (niterates Suc 0)  \<and>
          nlast ls  = the_enat(nlength \<sigma>) \<and>
         (\<forall>i\<le>nlength ls. nnth ls i = i) \<and> 
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (i)) \<and> \<sigma>1 = \<sigma>))
 "
proof -
 have 1: " nidx (ntaken (the_enat(nlength \<sigma>)) (niterates Suc 0))" 
     using nidx_ntaken_niterates_Suc by presburger
 have 2: " nnth (ntaken (the_enat (nlength \<sigma>)) (niterates Suc 0)) 0 = 0 " 
    by (simp add: ntaken_nnth)
 have 3: " nlast (ntaken (the_enat (nlength \<sigma>)) (niterates Suc 0)) = the_enat(nlength \<sigma>)"
   by (simp add: ntaken_nlast) 
 have 4: " (\<forall>i\<le>nlength (ntaken (the_enat (nlength \<sigma>)) (niterates Suc 0)). 
             nnth (ntaken (the_enat (nlength \<sigma>)) (niterates Suc 0)) i = i)" 
   by (simp add: ntaken_nnth)
 have 5: " (\<exists>\<sigma>1. nlength \<sigma>1 = nlength (ntaken (the_enat (nlength \<sigma>)) (niterates Suc 0)) \<and> 
           (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> i) \<and> \<sigma>1 = \<sigma>)" 
   using assms   by (simp add: enat_the_enat nfinite_nlength_enat)
 show ?thesis 
 using "1" "2" "3" "4" "5" assms nfinite_ntaken by blast
qed


lemma OPJ6help4:
assumes "\<not>nfinite \<sigma>" 
shows " (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and>
         ls = (niterates Suc 0)  \<and>
         (\<forall>i\<le>nlength ls. nnth ls i = i) \<and> 
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (i)) \<and> \<sigma>1 = \<sigma>))
 "
proof -
 have 1: " nidx ((niterates Suc 0))" 
     using nidx_expand nnth_niterates_Suc by presburger
 have 2: " nnth ( (niterates Suc 0)) 0 = 0 " 
    by (simp)
 have 4: " (\<forall>i\<le>nlength ( (niterates Suc 0)). nnth ( (niterates Suc 0)) i = i)" 
   by (simp)
 have 5: " (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ( (niterates Suc 0)) \<and> 
           (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> i) \<and> \<sigma>1 = \<sigma>)" 
    using assms  by (simp add: nfinite_conv_nlength_enat)
 show ?thesis 
 using "1" "2"  "4" "5" assms by blast
qed


lemma PJ6help5:
assumes "nfinite \<sigma> " 
shows "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and> 
          nlast  ls  = the_enat(nlength \<sigma>) \<and>
         (\<forall>i<nlength ls. nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) = Suc 0) \<and>
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> ( nnth ls i)) \<and> g \<sigma>1))
   = g \<sigma> "
proof -
 have "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and>
          nlast  ls  = the_enat (nlength \<sigma>) \<and>
         (\<forall>i<nlength ls. nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) = Suc 0) \<and>
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (nnth ls i)) \<and> g \<sigma>1))
       = 
       (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and> 
          nlast ls  = the_enat(nlength \<sigma>) \<and>
         (\<forall>i<nlength ls. nnth ls (Suc i) - nnth ls i = Suc 0) \<and>
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> ( nnth ls i)) \<and> g \<sigma>1))"
     using PJ6help1[of _ \<sigma>] assms by auto
  also have " ... = 
            (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and>
          nlast  ls  = the_enat(nlength \<sigma>) \<and>
         (\<forall>i\<le>nlength ls. nnth ls i = i) \<and>
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (nnth ls i)) \<and> g \<sigma>1))
            "
       using PJ6help2 PJ6help3 by blast
 also have " ... =
             (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and>
          nlast ls  = the_enat(nlength \<sigma>) \<and>
         (\<forall>i\<le>nlength ls. nnth ls i = i) \<and>
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (i)) \<and> g \<sigma>1))
           "
      using assms by auto (metis, metis)   
 also have " ... =
           (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and>
          nlast ls  = the_enat(nlength \<sigma>) \<and>
         (\<forall>i\<le>nlength ls. nnth ls i = i) \<and> 
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (i)) \<and> \<sigma>1 = \<sigma> \<and> g \<sigma>))"
        by auto
        (metis dual_order.refl enat.distinct(2) enat_the_enat nellist_eq_nnth_eq nfinite_nlength_enat
          nnth_nlast)
 also have " ... =
             g \<sigma>
           " 
   using PJ6help4 assms by auto
 finally show "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and>
          nlast ls  = the_enat(nlength \<sigma>) \<and>
         (\<forall>i<nlength ls. nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) = Suc 0) \<and>
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> ( nnth ls i)) \<and> g \<sigma>1))
   = g \<sigma>
              " .
qed

lemma OPJ6help5:
assumes "\<not>nfinite \<sigma> " 
shows "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and> 
         (\<forall>i<nlength ls. nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) = Suc 0) \<and>
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> ( nnth ls i)) \<and> g \<sigma>1))
   = g \<sigma> "
proof -
 have "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and>
         (\<forall>i<nlength ls. nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) = Suc 0) \<and>
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (nnth ls i)) \<and> g \<sigma>1))
       = 
       (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and> 
         (\<forall>i<nlength ls. nnth ls (Suc i) - nnth ls i = Suc 0) \<and>
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> ( nnth ls i)) \<and> g \<sigma>1))"
     using OPJ6help1[of _ \<sigma>] assms by auto
  also have " ... = 
            (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and>
         (\<forall>i\<le>nlength ls. nnth ls i = i) \<and>
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (nnth ls i)) \<and> g \<sigma>1))
            "
       using OPJ6help2[of _ \<sigma>] OPJ6help3[of _ \<sigma>] by blast
 also have " ... =
             (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and>
         (\<forall>i\<le>nlength ls. nnth ls i = i) \<and>
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (i)) \<and> g \<sigma>1))
           "
      using assms by auto (metis, metis) 
 also have " ... =
           (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and>
         (\<forall>i\<le>nlength ls. nnth ls i = i) \<and> 
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (i)) \<and> \<sigma>1 = \<sigma> \<and> g \<sigma>))
           "
      by auto (metis OPJ6help4 nellist_eq_nnth_eq) 
 also have " ... =
             g \<sigma>
           " 
   using OPJ6help4 assms by auto
 finally show "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and>
         (\<forall>i<nlength ls. nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) = Suc 0) \<and>
         (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> ( nnth ls i)) \<and> g \<sigma>1))
   = g \<sigma>
              " .
qed

lemma PJ6sem:
"(\<sigma> \<Turnstile> skip  fproj  g = (g \<and> finite))"
proof -
 have 1: "(\<sigma> \<Turnstile> skip  fproj  g = (g \<and> finite)) =
          ((\<exists>l. nidx  l \<and> nnth l 0 = 0 \<and> nfinite l \<and> nfinite \<sigma> \<and> nlast l  = the_enat(nlength \<sigma>) \<and>
         (\<forall>i<nlength l.  (nsubn \<sigma> (nnth l i) (nnth l (Suc i)) ) \<Turnstile> skip) \<and> g (pfilt \<sigma> l)) =
         (g \<sigma> \<and> nfinite \<sigma>) )" 
       by (simp add: fprojection_d_def powerinterval_def finite_defs) 
 have 2: "(\<exists>ls. nidx  ls \<and>  nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and> nlast ls  = the_enat(nlength \<sigma>) \<and>
         (\<forall>i<nlength ls.  (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> skip) \<and> g (pfilt \<sigma> ls)) =
         (\<exists>ls \<sigma>1 . nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and> nlast  ls  = the_enat(nlength \<sigma>) \<and>
         (\<forall>i<nlength ls.  (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> skip) \<and> 
          nlength \<sigma>1 = nlength ls \<and>
          (\<forall> i\<le> nlength \<sigma>1. (nnth \<sigma>1 i) = (nnth \<sigma> (nnth ls i))) \<and>
         g \<sigma>1)" 
     using pfilt_expand[of _ \<sigma> ] 
       by auto (blast,metis)
  have 3: "(\<exists>ls \<sigma>1 . nidx  ls \<and>   nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and> 
                    nlast  ls = the_enat(nlength \<sigma>) \<and>
         (\<forall>i<nlength ls.  (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> skip) \<and> 
          nlength \<sigma>1 = nlength ls \<and>
          (\<forall> i\<le> nlength \<sigma>1. (nnth \<sigma>1 i) = (nnth \<sigma> (nnth ls i))) \<and>
         g \<sigma>1) =
         (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and> nlast ls  = the_enat(nlength \<sigma>) \<and>
        (\<forall>i<nlength ls. nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) = Suc 0) \<and>
        (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (nnth ls i)) \<and> g \<sigma>1)) "
   by (simp add: skip_defs)
  have 4: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and>  nlast ls  = the_enat(nlength \<sigma>) \<and>
        (\<forall>i<nlength ls. nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) = Suc 0) \<and>
        (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (nnth ls i)) \<and> g \<sigma>1)) =
        ((g \<sigma>) \<and> nfinite \<sigma>) "
      using PJ6help5[of \<sigma> g]  by auto 
  from 1 2 3 4 show ?thesis 
  by simp
qed

lemma OPJ6sem:
"(\<sigma> \<Turnstile> skip oproj g = (g \<and> inf))"
proof -
 have 1: "(\<sigma> \<Turnstile> skip oproj g = (g \<and> inf)) =
          ((\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and> 
         (\<forall>i<nlength ls.  (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> skip) \<and> g (pfilt \<sigma> ls)) =
         (g \<sigma> \<and> \<not>nfinite \<sigma>) )" 
       by (simp add: oprojection_d_def powerinterval_def infinite_defs) 
 have 2: "(\<exists>ls. nidx  ls \<and>  nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and> 
         (\<forall>i<nlength ls.  (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> skip) \<and> g (pfilt \<sigma> ls)) =
         (\<exists>ls \<sigma>1 . nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and> 
         (\<forall>i<nlength ls.  (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> skip) \<and> 
          nlength \<sigma>1 = nlength ls \<and>
          (\<forall> i\<le> nlength \<sigma>1. (nnth \<sigma>1 i) = (nnth \<sigma> (nnth ls i))) \<and>
         g \<sigma>1)" 
     using pfilt_expand[of _ \<sigma> ] by auto (blast,metis)
  have 3: "(\<exists>ls \<sigma>1 . nidx  ls \<and>   nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and> 
         (\<forall>i<nlength ls.  (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> skip) \<and> 
          nlength \<sigma>1 = nlength ls \<and>
          (\<forall> i\<le> nlength \<sigma>1. (nnth \<sigma>1 i) = (nnth \<sigma> (nnth ls i))) \<and>
         g \<sigma>1) =
         (\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and> 
        (\<forall>i<nlength ls. nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) = Suc 0) \<and>
        (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (nnth ls i)) \<and> g \<sigma>1)) "
   by (simp add: skip_defs)
  have 4: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and>  
        (\<forall>i<nlength ls. nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) = Suc 0) \<and>
        (\<exists>\<sigma>1. nlength \<sigma>1 = nlength ls \<and> (\<forall>i\<le>nlength \<sigma>1. nnth \<sigma>1 i = nnth \<sigma> (nnth ls i)) \<and> g \<sigma>1)) =
        ((g \<sigma>) \<and> \<not>nfinite \<sigma>) "
      using OPJ6help5[of \<sigma> g]  by auto 
  from 1 2 3 4 show ?thesis 
  by simp
qed

subsubsection \<open>PJ7\<close>


lemma PJemptyImp:
assumes " nlength \<sigma> = 0"
 shows  "(\<sigma> \<Turnstile> (f  fproj  g) = g)"  
using assms 
by (auto simp add: fprojection_d_def  powerinterval_def nsubn_def1)
   (metis i0_less less_numeral_extra(3) ndropn_0 ndropn_nlast nfinite_conv_nlength_enat
    nidx_less_last_1 nnth_nlast pfilt_code(1) the_enat.simps the_enat_0,
    metis PJ6help4 ndropn_0 ndropn_nlast nfinite_ntaken not_less_zero ntaken_all ntaken_nlast
    pfilt_code(1) zero_le)

lemma PJ7empty:
 assumes " nlength \<sigma> = 0"
 shows   "(\<sigma> \<Turnstile> f  fproj  (g  fproj  h) = (f  fproj  g)  fproj  h)"
proof -
 have 1: "(\<sigma> \<Turnstile> f  fproj  (g  fproj  h) = (g  fproj  h)) "
   using PJemptyImp assms by blast
 have 2: "(\<sigma> \<Turnstile> (g  fproj  h) = h)"
   using PJemptyImp assms by blast
 have 3: "(\<sigma> \<Turnstile> (f  fproj  g)  fproj  h = h) "
   using PJemptyImp assms by blast
 from 1 2 3 show ?thesis by simp
qed

lemma PJ7helpchain1a_help_1:
 assumes "nidx  ls"
         "nnth ls 0 = 0  "
         "nfinite ls"
         "nfinite \<sigma>" 
         "nlast ls  = the_enat(nlength \<sigma>) "  
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " nlength \<sigma> > 0 " 
 shows   " nidx   (nfusecat ( (lcppl f g \<sigma> ls))) \<and> nnth (nfusecat ( (lcppl f g \<sigma> ls))) 0 = 0"
proof -
 have 0:  "nlength \<sigma> > 0 \<longrightarrow> nlength ls >0"
    using assms 
    by (metis enat.distinct(2) enat_the_enat gr_zeroI nfinite_conv_nlength_enat nnth_nlast the_enat_0)
 have 01: "nfirst ls = 0"
   using assms 
   by (metis ndropn_0 ndropn_nfirst)
 have 02: "nlastnfirst (lcppl f g \<sigma> ls)"
    using assms lcppl_nfusecat_nlastnfirst[of ls \<sigma> f g] 
    by (metis "0")   
 have 1: " nfirst (nfusecat  (lcppl f g \<sigma> ls)) = 0 "
   using assms 0  01  02 lcppl_nfirst[of "ls" "\<sigma>" "f" "g" ] by (metis  nfirst_nfusecat_nfirst )  
 from 1 0 show ?thesis using assms lcppl_nfusecat_nidx[of "ls" "\<sigma>" "f" "g"] 
 by (metis "01"   ntaken_0 ntaken_nlast) 
qed

lemma PJ7helpchain1a_help_1_alt:
 assumes "nidx  ls"
         "nnth ls 0 = 0  "
         "\<not>nfinite ls"
         "\<not>nfinite \<sigma>" 
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " nlength \<sigma> > 0 " 
 shows   " nidx   (nfusecat ( (lcppl f g \<sigma> ls))) \<and> nnth (nfusecat ( (lcppl f g \<sigma> ls))) 0 = 0"
proof -
 have 0:  "nlength \<sigma> > 0 \<longrightarrow> nlength ls >0"
    using assms 
    by (metis  enat_the_enat  nfinite_conv_nlength_enat )
 have 01: "nfirst ls = 0"
   using assms 
   by (metis ndropn_0 ndropn_nfirst)
 have 02: "nlastnfirst (lcppl f g \<sigma> ls)"
    using assms lcppl_nfusecat_nlastnfirst_alt[of ls \<sigma> f g] 
    by (metis )   
 have 1: " nfirst (nfusecat  (lcppl f g \<sigma> ls)) = 0 "
   using assms 0  01  02 
     lcppl_nfirst_alt[of "ls" "\<sigma>" "f" "g" ] 
     by (metis  nfirst_nfusecat_nfirst )  
 from 1 0 show ?thesis using assms lcppl_nfusecat_nidx_alt[of "ls" "\<sigma>" "f" "g"] 
 by (metis "01"   ntaken_0 ntaken_nlast) 
qed

lemma PJ7helpchain1a_help_2:
 assumes "nidx  ls"
         "nnth ls 0 = 0 "
         "nfinite ls"
         "nfinite \<sigma>" 
         " nlast ls = the_enat(nlength \<sigma>) "
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
         " nlength \<sigma> > 0 " 
 shows   " powerinterval f \<sigma> (nfusecat ( (lcppl f g \<sigma> ls)))"
proof -
 have 1: "(\<forall> i<nlength ls. 
             powerinterval f (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) 
                             (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )))"
   using assms cppl_fprojection by blast 
 have 2: "\<forall>i<nlength ls.
           \<forall>ia<nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )).
            f (nsubn  (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )
                      (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) ia)
                      (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) (Suc ia))
                   )" 
   using "1" by (simp add: powerinterval_def)
 have 3: "\<forall>i<nlength ls.
           \<forall>ia<nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )).
              (nsubn (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )
                     (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) ia)
                     (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) (Suc ia))
                   ) =
            (nsubn \<sigma>
                  (nnth (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))) ia)
                  (nnth (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))) (Suc ia))
                  )" 
   proof auto 
     fix i
     fix ia
     assume a0: "enat i < nlength ls"
     assume a1: "enat ia < nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))"
     show "nsubn (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) 
                 (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia)
                 (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia)) =
           nsubn \<sigma> (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia + nnth ls i)
                   (nnth (nmap (\<lambda>x. x + nnth ls i) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))) (Suc ia)) "
     proof -  
      have 300: "0 < nlength \<sigma>"
        using assms(7) by auto 
      have 301: "nnth ls i < nnth ls (Suc i)"
        by (metis a0 assms(1) eSuc_enat ileI1 nidx_expand)  
      have 302: "(nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> (f  fproj  g)"
         by (simp add: a0 assms(6))
      have 303: "the_enat (nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) = (nnth ls (Suc i)) - (nnth ls i) "
         by (simp add: PJ6help1 a0 assms(1) assms(2) assms(3) assms(4) assms(5))
      have 304: "nlast (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) = (nnth ls (Suc i)) - (nnth ls i)"
          using  a0 a1 cppl_fprojection[of f g "(nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))"] 
          using "302" "303" by presburger
      have 305: "(Suc ia) \<le> nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))"
         by (simp add: Suc_ile_eq a1) 
      have 306: "nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia \<le> 
                 nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia)" 
         by (metis "302" a1 cppl_fprojection eSuc_enat ileI1 le_add2 nidx_less_eq plus_1_eq_Suc)
      have 307: "nlast (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))= 
                 nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) 
                      (the_enat(nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))))"
          using "302" cppl_fprojection nnth_nlast by auto 
      have 308: "nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia) \<le>
                 nlast (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))"
            by (metis "302" "305" cppl_fprojection nidx_all_le_nlast)     
      have 310: "enat (nnth ls (Suc i)) \<le> nlength \<sigma>"
         using a0 assms 
          by (metis  dual_order.refl eSuc_enat enat_ord_simps(1) enat_ord_simps(3) enat_the_enat 
              ileI1 nfinite_conv_nlength_enat nidx_less_eq nnth_nlast the_enat.simps) 
      have 30: "nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia) \<le> nnth ls (Suc i) - nnth ls i" 
           using assms cppl_bounds[of  "(nnth ls i)" "(nnth ls (Suc i))" \<sigma> f g "Suc ia"]  
          PJ6help1[of ls \<sigma> ]  a0 a1 cppl_fprojection[of f g "(nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))"]
           by (metis "303" "308")
      have 311: "nsubn (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) 
                 (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia)
                 (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia)) = 
                nsubn \<sigma> (nnth ls i + nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia)
                        (nnth ls i + nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia)) " 
         using nsubn_nsubn_1[of "(nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia)" 
            "(nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia))" "(nnth ls i)" "(nnth ls (Suc i))" \<sigma> ]
              using "30" "301" "306" "310" order_less_imp_le by blast
      have 312: "(nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia + nnth ls i) =
                 (nnth ls i + nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia)"
              by simp 
      have 313: "(nnth ls i + nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia)) =
                 (nnth (nmap (\<lambda>x. x + nnth ls i) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))) (Suc ia))"
           using "305" by force 
      show ?thesis 
      using "311" "312" "313" by presburger
    qed
   qed
  have 4: "\<forall>i<nlength ls.
           \<forall>ia<nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )).
            f (nsubn \<sigma> (nnth (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))) ia)
                  (nnth (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))) (Suc ia))
                  )" 
       using "2" "3" by auto
  have 5: "nlength(lcppl f g \<sigma> ls) = nlength ls -1"
      using assms  lcppl_nlength lcppl_nlength_zero 
      by (metis epred_0 epred_conv_minus i0_less)
  have 6: "nlength \<sigma> >0 \<longrightarrow> nlength ls >0" 
      using assms 
      by (metis enat.distinct(2) enat_the_enat nfinite_nlength_enat nnth_nlast the_enat_0 zero_less_iff_neq_zero)
  have 7: "\<forall>i< nlength ls.
            \<forall>ia<nlength ((nnth (lcppl f g \<sigma> ls) i)).
             f (nsubn \<sigma> (nnth (nnth (lcppl f g \<sigma> ls) i) ia)
                    (nnth (nnth (lcppl f g \<sigma> ls) i) (Suc ia))
                    )" 
      using assms 4 lcppl_nnth by (metis nlength_nmap)
  have 8: "nlength \<sigma> >0 \<longrightarrow>
            (\<forall>i\<le> nlength(lcppl f g \<sigma> ls).
              \<forall>ia<nlength ((nnth (lcppl f g \<sigma> ls) i)).
               f (nsubn \<sigma> (nnth (nnth (lcppl f g \<sigma> ls) i) ia)
                      (nnth (nnth (lcppl f g \<sigma> ls) i) (Suc ia))
                      ))" 
   using 5 6 7 assms by (metis  co.enat.exhaust_sel i0_less iless_Suc_eq lcppl_nlength)
  have 9: "nlength \<sigma> >0 \<longrightarrow> 
            powerinterval f \<sigma> (nfusecat ( (lcppl f g \<sigma> ls))) =
            (\<forall>i<nlength (nfusecat (lcppl f g \<sigma> ls)).
              f (nsubn \<sigma> (nnth (nfusecat (lcppl f g \<sigma> ls)) i) (nnth (nfusecat (lcppl f g \<sigma> ls)) (Suc i)) ))"
   by (simp add: powerinterval_def)
  have 10: "nlength \<sigma> >0 \<longrightarrow> nlastnfirst (lcppl f g \<sigma> ls)"
    using "6" assms lcppl_nfusecat_nlastnfirst 
    by (metis  nfinite_conv_nlength_enat)
  have 11: "nlength \<sigma> >0 \<longrightarrow>
             (\<forall> j \<le> nlength (lcppl f g \<sigma> ls). nlength(nnth (lcppl f g \<sigma> ls) j) > 0)
             " 
       using assms 
       by (metis enat.distinct(2) enat_the_enat lcppl_nlength_all_gr_zero nfinite_conv_nlength_enat)
  have 110: " all_gr_zero (lcppl f g \<sigma> ls) "
     using "11" all_gr_zero_nnth_a assms(7) by blast
  have 111: " all_nfinite (lcppl f g \<sigma> ls)  "
     using all_nfinite_nnth_a assms lcppl_nlength_all_nfinite by blast
  have 12: "nlength \<sigma> >0 \<longrightarrow>
             (\<forall>i\<le> nlength(lcppl f g \<sigma> ls).
               (\<forall>ia<nlength ((nnth (lcppl f g \<sigma> ls) i)).
                 f (nsubn \<sigma> (nnth (nnth (lcppl f g \<sigma> ls) i) ia)
                        (nnth (nnth (lcppl f g \<sigma> ls) i) (Suc ia))
                        ))) =
                  (\<forall>j<nlength (nfusecat (lcppl f g \<sigma> ls)).
                 f (nsubn \<sigma> (nnth (nfusecat (lcppl f g \<sigma> ls)) j) 
                        (nnth (nfusecat (lcppl f g \<sigma> ls)) (Suc j)) 
                        ))
              " 
     using  nfusecat_split_nsubn[of   "(lcppl f g \<sigma> ls)" f \<sigma>] 6 "10" "11" 110 111 assms 
     unfolding nridx_expand  
     by (simp add: Suc_ile_eq)
   show ?thesis using "12" "8" "9" using assms 
   by meson
qed

lemma PJ7helpchain1a_help_2_alt:
 assumes "nidx  ls"
         "nnth ls 0 = 0 "
         "\<not>nfinite ls"
         "\<not>nfinite \<sigma>" 
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g)"
 shows   " powerinterval f \<sigma> (nfusecat ( (lcppl f g \<sigma> ls)))"
proof -
 have 1: "(\<forall> i<nlength ls. 
             powerinterval f (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) 
                             (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )))"
   using assms cppl_fprojection by blast 
 have 2: "\<forall>i<nlength ls.
           \<forall>ia<nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )).
            f (nsubn  (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )
                      (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) ia)
                      (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) (Suc ia))
                   )" 
   using "1" by (simp add: powerinterval_def)
 have 3: "\<forall>i<nlength ls.
           \<forall>ia<nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )).
              (nsubn (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )
                     (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) ia)
                     (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )) (Suc ia))
                   ) =
            (nsubn \<sigma>
                  (nnth (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))) ia)
                  (nnth (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))) (Suc ia))
                  )"
   proof auto 
     fix i
     fix ia
     assume a0: "enat i < nlength ls"
     assume a1: "enat ia < nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))"
     show "nsubn (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) 
                 (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia)
                 (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia)) =
           nsubn \<sigma> (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia + nnth ls i)
                   (nnth (nmap (\<lambda>x. x + nnth ls i) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))) (Suc ia)) "
     proof -  
      have 300: "0 < nlength \<sigma>"
         by (metis assms(4) gr_zeroI nlength_eq_enat_nfiniteD zero_enat_def)
      have 301: "nnth ls i < nnth ls (Suc i)"
        by (metis a0 assms(1) eSuc_enat ileI1 nidx_expand)  
      have 302: "(nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) \<Turnstile> (f  fproj  g)"
           by (simp add: a0 assms(5))
      have 303: "the_enat (nlength (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) = (nnth ls (Suc i)) - (nnth ls i) "
            by (simp add: OPJ6help1 a0 assms(1) assms(2) assms(3) assms(4))
      have 304: "nlast (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) = (nnth ls (Suc i)) - (nnth ls i)"
          using  a0 a1 cppl_fprojection[of f g "(nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))"] 
          using "302" "303" by presburger
      have 305: "(Suc ia) \<le> nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))"
         by (simp add: Suc_ile_eq a1) 
      have 306: "nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia \<le> 
                 nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia)" 
         by (metis "302" a1 cppl_fprojection eSuc_enat ileI1 le_add2 nidx_less_eq plus_1_eq_Suc)
      have 307: "nlast (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))= 
                 nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) 
                      (the_enat(nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))))"
          using "302" cppl_fprojection nnth_nlast by auto 
      have 308: "nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia) \<le>
                 nlast (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))"
            by (metis "302" "305" cppl_fprojection nidx_all_le_nlast)     
      have 310: "enat (nnth ls (Suc i)) \<le> nlength \<sigma>"
         using a0 assms 
          by (metis  enat_ord_simps(3) enat_the_enat nfinite_conv_nlength_enat  ) 
      have 30: "nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia) \<le> nnth ls (Suc i) - nnth ls i" 
           using assms cppl_bounds[of  "(nnth ls i)" "(nnth ls (Suc i))" \<sigma> f g "Suc ia"]  
          PJ6help1[of ls \<sigma> ]  a0 a1 cppl_fprojection[of f g "(nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))"]
           by (metis "303" "308")
      have 311: "nsubn (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))) 
                 (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia)
                 (nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia)) = 
                nsubn \<sigma> (nnth ls i + nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia)
                        (nnth ls i + nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia)) " 
         using nsubn_nsubn_1[of "(nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia)" 
            "(nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia))" "(nnth ls i)" "(nnth ls (Suc i))" \<sigma> ]
              using "30" "301" "306" "310" order_less_imp_le by blast
      have 312: "(nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia + nnth ls i) =
                 (nnth ls i + nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) ia)"
              by simp 
      have 313: "(nnth ls i + nnth (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)))) (Suc ia)) =
                 (nnth (nmap (\<lambda>x. x + nnth ls i) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i))))) (Suc ia))"
           using "305" by force 
      show ?thesis 
      using "311" "312" "313" by presburger
    qed
   qed
  have 4: "\<forall>i<nlength ls.
           \<forall>ia<nlength (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) )).
            f (nsubn \<sigma> (nnth (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))) ia)
                  (nnth (nmap (\<lambda>x. x+ (nnth ls i)) (cppl f g (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ))) (Suc ia))
                  )" 
       using "2" "3" by auto
  have 5: "nlength(lcppl f g \<sigma> ls) = nlength ls -1"
      using assms  lcppl_nlength lcppl_nlength_zero  
      by (simp add: epred_conv_minus lcppl_nlength_alt)
  have 6: "nlength \<sigma> >0 \<longrightarrow> nlength ls >0" 
      using assms 
      by (simp add: nfinite_conv_nlength_enat)
  have 7: "\<forall>i< nlength ls.
            \<forall>ia<nlength ((nnth (lcppl f g \<sigma> ls) i)).
             f (nsubn \<sigma> (nnth (nnth (lcppl f g \<sigma> ls) i) ia)
                    (nnth (nnth (lcppl f g \<sigma> ls) i) (Suc ia))
                    )" 
      using assms 4 lcppl_nnth 
      by (metis nlength_nmap)
  have 8: "(\<forall>i\<le> nlength(lcppl f g \<sigma> ls).
              \<forall>ia<nlength ((nnth (lcppl f g \<sigma> ls) i)).
               f (nsubn \<sigma> (nnth (nnth (lcppl f g \<sigma> ls) i) ia)
                      (nnth (nnth (lcppl f g \<sigma> ls) i) (Suc ia))
                      ))" 
   using 5 6 7 assms enat_ile nfinite_conv_nlength_enat not_le_imp_less by blast
  have 9: "powerinterval f \<sigma> (nfusecat ( (lcppl f g \<sigma> ls))) =
            (\<forall>i<nlength (nfusecat (lcppl f g \<sigma> ls)).
              f (nsubn \<sigma> (nnth (nfusecat (lcppl f g \<sigma> ls)) i) (nnth (nfusecat (lcppl f g \<sigma> ls)) (Suc i)) ))"
   by (simp add: powerinterval_def)
  have 10: " nlastnfirst (lcppl f g \<sigma> ls)"
    using "6" assms lcppl_nfusecat_nlastnfirst_alt 
    by (metis)
  have 11: "(\<forall> j \<le> nlength (lcppl f g \<sigma> ls). nlength(nnth (lcppl f g \<sigma> ls) j) > 0)" 
       using assms lcppl_nlength_all_gr_zero_alt by blast
  have 110: " all_gr_zero (lcppl f g \<sigma> ls) "
     using "11" all_gr_zero_nnth_a assms by blast
  have 111: " all_nfinite (lcppl f g \<sigma> ls)  "
      using all_nfinite_nnth_a assms lcppl_nlength_all_nfinite_alt by blast
  have 12: "(\<forall>i\<le> nlength(lcppl f g \<sigma> ls).
               (\<forall>ia<nlength ((nnth (lcppl f g \<sigma> ls) i)).
                 f (nsubn \<sigma> (nnth (nnth (lcppl f g \<sigma> ls) i) ia)
                        (nnth (nnth (lcppl f g \<sigma> ls) i) (Suc ia))
                        ))) =
                  (\<forall>j<nlength (nfusecat (lcppl f g \<sigma> ls)).
                 f (nsubn \<sigma> (nnth (nfusecat (lcppl f g \<sigma> ls)) j) 
                        (nnth (nfusecat (lcppl f g \<sigma> ls)) (Suc j)) 
                        ))
              " 
     using  nfusecat_split_nsubn[of   "(lcppl f g \<sigma> ls)" f \<sigma>] "10" "11" 110 111 assms 
     unfolding nridx_expand
     by (simp add: Suc_ile_eq)  
   show ?thesis using "12" "8" "9" using assms 
   by meson
qed        

lemma PJ7helpchain1a_help_3:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "nfinite ls" 
         "nfinite \<sigma> " 
         " nlast  ls  = the_enat(nlength \<sigma>)" 
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g) " 
         " h (pfilt \<sigma> ls) "
         " nlength \<sigma> >0 " 
 shows   " nlast (nfusecat (lcppl f g \<sigma> ls)) = nlength \<sigma>"
proof -
 have 0: "nlength \<sigma> > 0 \<longrightarrow> nlength ls >0"
    using assms 
    by (metis PJ6help4 nnth_nlast the_enat_0 zero_less_iff_neq_zero)
 have 00: "nfinite (lcppl f g \<sigma> ls)"
       using assms(1) assms(3) lcppl_nfinite by blast 
 have 01: "nlastnfirst (lcppl f g \<sigma> ls)"
    using "0" assms lcppl_nfusecat_nlastnfirst by blast 
 have 02: "all_nfinite (lcppl f g \<sigma> ls)"
    using all_nfinite_nnth_a assms lcppl_nlength_all_nfinite by blast 
 have 1: "nlength \<sigma> >0 \<longrightarrow>
            nlast (nfusecat (lcppl f g \<sigma> ls)) = nlast(nlast ( (lcppl f g \<sigma> ls)))" 
      using assms 0 nlastfirst_nfusecat_nlast[of "(lcppl f g \<sigma> ls)" ] 00 01 02 by blast
 have 2: "nlength \<sigma> >0 \<longrightarrow>
         nlast ( (lcppl f g \<sigma> ls))
          = (nnth (lcppl f g \<sigma> ls) (the_enat(nlength ls) -1)) " 
       using assms 
       by (metis "0" epred_enat lcppl_nfinite lcppl_nlength nfinite_nlength_enat nnth_nlast 
           the_enat.simps)
 have 3: "nlength \<sigma> >0 \<longrightarrow>
          (nnth (lcppl f g \<sigma> ls) (the_enat(nlength ls) -1)) =
           (nmap (\<lambda>x. x+ (nnth ls (the_enat(nlength ls) -1))) 
                (cppl f g (nsubn \<sigma> (nnth ls (the_enat(nlength ls) -1)) 
                                   (nnth ls (Suc (the_enat(nlength ls) -1))) )))
         " 
       using assms 0 lcppl_nnth[of ls "(the_enat(nlength ls) -1)" f g \<sigma>]  
         by (metis Suc_n_not_le_n add.commute diff_add enat.distinct(2) enat_ord_simps(1)
             enat_ord_simps(4) enat_the_enat ileI1 not_le_imp_less one_eSuc one_enat_def plus_1_eq_Suc)
 have 4: "nlength \<sigma> >0 \<longrightarrow>
          nlast( ( (cppl f g (nsubn \<sigma> (nnth ls (the_enat(nlength ls) -1)) 
                                      (nnth ls (Suc (the_enat(nlength ls) -1))) )))) =
           the_enat(nlength ((nsubn \<sigma> (nnth ls (the_enat(nlength ls) -1)) 
                                      (nnth ls (Suc (the_enat(nlength ls) -1))) )))" 
    using "0" assms cppl_fprojection[of f g 
              "(nsubn \<sigma> (nnth ls (the_enat(nlength ls) -1)) (nnth ls (Suc (the_enat(nlength ls) -1))) )"]
    by (metis Suc_n_not_le_n add.commute diff_add enat.distinct(2) enat_ord_simps(1) 
        enat_the_enat ileI1 nfinite_conv_nlength_enat not_le_imp_less one_eSuc one_enat_def plus_1_eq_Suc)         
 have 5: "nlength \<sigma> >0 \<longrightarrow>
         the_enat(nlength ((nsubn \<sigma> (nnth ls (the_enat(nlength ls) -1)) 
                                    (nnth ls (Suc (the_enat(nlength ls) -1))) ))) =
            (nnth ls (Suc (the_enat(nlength ls) -1))) - (nnth ls (the_enat(nlength ls) -1))     
        "  
     using "0" PJ6help1 assms 
     by (metis Suc_n_not_le_n add.commute diff_add enat_ord_simps(1) ileI1 le_add_same_cancel1 
         nfinite_conv_nlength_enat not_le_imp_less one_eSuc one_enat_def plus_1_eq_Suc the_enat.simps)
 have 60:"nfinite (cppl f g (nsubn \<sigma> (nnth ls (the_enat (nlength ls) - 1)) 
                                     (nnth ls (Suc (the_enat (nlength ls) - 1)))))"
      by (metis "0" Suc_n_not_le_n add.commute assms(3) assms(6) assms(8) cppl_fprojection diff_add 
          enat.distinct(2) enat_ord_simps(1) enat_the_enat ileI1 nfinite_conv_nlength_enat 
          not_le_imp_less one_eSuc one_enat_def plus_1_eq_Suc)
 have 6: "nlength \<sigma> >0 \<longrightarrow>
          nlast ( (nmap (\<lambda>x. x+ (nnth ls (the_enat(nlength ls) -1))) 
                         (cppl f g (nsubn \<sigma> (nnth ls (the_enat(nlength ls) -1)) 
                                            (nnth ls (Suc (the_enat(nlength ls) -1))) )))) =
             (\<lambda>x. x+ (nnth ls (the_enat(nlength ls) -1))) 
              (nlast (cppl f g (nsubn \<sigma> (nnth ls (the_enat(nlength ls) -1)) 
                                        (nnth ls (Suc (the_enat(nlength ls) -1))) )))"    
    using nlast_nmap[of  "(cppl f g (nsubn \<sigma> (nnth ls (the_enat(nlength ls) -1)) 
                                             (nnth ls (Suc (the_enat(nlength ls) -1))) ))" 
     "(\<lambda>x. x+ (nnth ls (the_enat(nlength ls) -1)))" ]   60 by blast 
 have 7: "nlength \<sigma> >0 \<longrightarrow> 
          (\<lambda>x. x+ (nnth ls (the_enat(nlength ls) -1))) 
           (nlast (cppl f g (nsubn \<sigma> (nnth ls (the_enat(nlength ls) -1)) 
                                     (nnth ls (Suc (the_enat(nlength ls) -1))) ))) =
          (\<lambda>x. x+ (nnth ls (the_enat(nlength ls) -1))) 
          ((nnth ls (Suc (the_enat(nlength ls) -1))) - (nnth ls (the_enat(nlength ls) -1)))" 
      using "4" "5" by auto
 have 8: "nlength \<sigma> >0 \<longrightarrow> 
          (\<lambda>x. x+ (nnth ls (the_enat(nlength ls) -1))) 
          ((nnth ls (Suc (the_enat(nlength ls) -1))) - (nnth ls (the_enat(nlength ls) -1))) =
           ((nnth ls (Suc (the_enat(nlength ls) -1))) - 
            (nnth ls (the_enat(nlength ls) -1))) + (nnth ls (the_enat(nlength ls) -1))" 
    by (simp add:  shift_def)
 have 9: "nlength \<sigma> >0 \<longrightarrow>  
          ((nnth ls (Suc (the_enat(nlength ls) -1))) - 
           (nnth ls (the_enat(nlength ls) -1))) + (nnth ls (the_enat(nlength ls) -1)) 
          = (nnth ls (Suc (the_enat(nlength ls) -1)))" 
    using assms 0 
    by (metis Suc_diff_1 cancel_comm_monoid_add_class.diff_cancel diff_add diff_is_0_eq enat_ord_simps(1) 
        enat_ord_simps(2) nfinite_nlength_enat nidx_expand order_less_imp_le the_enat.simps zero_enat_def)
 have 10: "nlength \<sigma> >0 \<longrightarrow> (nnth ls (Suc (the_enat(nlength ls) -1))) = nlength \<sigma> " 
    using assms 0 
    by (metis One_nat_def add.commute diff_add eSuc_enat enat.distinct(2) enat_ord_simps(1) enat_the_enat
         ileI1 nfinite_conv_nlength_enat nnth_nlast plus_1_eq_Suc zero_enat_def)
  show ?thesis 
  using "1" "10" "2" "3" "6" "7" "8" "9" assms by presburger
qed


lemma PJ7helpchain1a_help_4:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "nfinite ls"
         "nfinite \<sigma>" 
         " nlast ls  = the_enat(nlength \<sigma>)" 
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g) " 
         "  h (pfilt \<sigma> ls) "
         " nlength \<sigma> >0 "
 shows   " ((pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) \<Turnstile> g  fproj  h) "
proof -
 have 0: "nlength \<sigma> > 0 \<longrightarrow> nlength ls >0"
    using assms 
    by (metis PJ6help4 nnth_nlast the_enat_0 zero_less_iff_neq_zero)
 have 00: "all_gr_zero (lcppl f g \<sigma> ls)"
    using all_gr_zero_nnth_a assms lcppl_nlength_all_gr_zero by blast 
 have 01: "all_nfinite (lcppl f g \<sigma> ls)"
   using all_nfinite_nnth_a assms lcppl_nlength_all_nfinite by blast 
 have 02: " nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) 0 = 0"
    using 0 assms 
    by (metis  addzero_def lcppl_lsum_nlength less_numeral_extra(3) nnth_0) 
 have 1: "nlength \<sigma> >0 \<longrightarrow> nidx  (addzero (lsum (lcppl f g \<sigma> ls) 0)) \<and> nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) 0 = 0"
    using assms lsum_addzero_nidx[of "(lcppl f g \<sigma> ls)" ]    lcppl_nlength_all_gr_zero[of ls \<sigma> f g]  
     using "00" "01" "02" by blast
 have 2: "nlength \<sigma> > 0 \<longrightarrow> 
          nlast  (addzero (lsum (lcppl f g \<sigma> ls) 0))  =
          the_enat(nlength (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))))"
    using assms lcppl_lsum_nlength[of ls \<sigma> f g]  lcppl_nfusecat_pfilt_nlength[of ls \<sigma> f g] 
      by fastforce
 have 3: "nlength \<sigma> >0 \<longrightarrow>  
          powerinterval g (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) (addzero (lsum (lcppl f g \<sigma> ls) 0))" 
    by (simp add: assms lcppl_nfusecat_pfilt_powerinterval)
 have 4: "nlength \<sigma> >0 \<longrightarrow>
           h (pfilt (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) (addzero (lsum (lcppl f g \<sigma> ls) 0))) "
   by (simp add: assms pfilt_nmap_pfilt lcppl_pfilt_nfusecat_lsum)
show ?thesis using "1" "2" "3" "4" assms fprojection_d_def 
by (metis "0" "00" "01" lcppl_lsum_nlength lcppl_nfinite lcppl_nfusecat_nlastnfirst 
    nfinite_conv_nlength_enat nfinite_nmap nfusecat_nfinite pfilt_nmap)
qed

lemma PJ7helpchain1a_help_4_alt:
 assumes "nidx  ls"
         "nnth ls 0 = 0 " 
         "\<not>nfinite ls"
         "\<not>nfinite \<sigma>" 
         " (\<forall> i<nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g) " 
         "  h (pfilt \<sigma> ls) "
 shows   " ((pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) \<Turnstile> g oproj h) "
proof -
 have 00: "all_gr_zero (lcppl f g \<sigma> ls)"
   using all_gr_zero_nnth_a assms lcppl_nlength_all_gr_zero_alt by blast
 have 01: "all_nfinite (lcppl f g \<sigma> ls)"
   using all_nfinite_nnth_a assms lcppl_nlength_all_nfinite_alt by blast 
 have 02: " nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) 0 = 0"
    using assms 
    by (metis addzero_def lcppl_nfinite nfinite_lsum_conv_b nlength_eq_enat_nfiniteD nnth_0 zero_enat_def) 
 have 1: " nidx  (addzero (lsum (lcppl f g \<sigma> ls) 0)) \<and> nnth (addzero (lsum (lcppl f g \<sigma> ls) 0)) 0 = 0"
    using assms lsum_addzero_nidx "00" "01" "02" by blast
 have 3: " powerinterval g (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) (addzero (lsum (lcppl f g \<sigma> ls) 0))"
   by (simp add: assms lcppl_nfusecat_pfilt_powerinterval_alt)   
 have 4: "h (pfilt (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) (addzero (lsum (lcppl f g \<sigma> ls) 0))) "
    by (simp add: assms lcppl_pfilt_nfusecat_lsum_alt pfilt_nmap_pfilt)
 have 5: " \<not> nfinite (addzero (lsum (lcppl f g \<sigma> ls) 0))" 
   using assms 
   by (simp add: lcppl_lsum_nlength_alt nfinite_conv_nlength_enat)
 have 6: "\<not>nfinite (nfusecat (lcppl f g \<sigma> ls))"
     using assms "00" "01" lcppl_nfinite lcppl_nfusecat_nlastnfirst_alt nfusecat_nfinite by blast 
 have 7: "\<not> nfinite (pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls)))" 
    by (simp add: "6" pfilt_nmap)
show ?thesis using "1"  "3" "4" assms unfolding oprojection_d_def 
using "5" "7" by blast
qed

lemma PJ7helpchain1a:
assumes "nlength \<sigma> > 0"
        "(\<sigma> \<Turnstile> (f  fproj  g)  fproj  h)" 
shows " (\<sigma> \<Turnstile> f  fproj  (g  fproj  h))"
proof -
 have 1: "nlength \<sigma> > 0"
  using assms by auto
 have 2: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and> nlast ls  = the_enat(nlength \<sigma>) \<and>
       powerinterval (LIFT(f  fproj  g)) \<sigma> ls \<and>
         h (pfilt \<sigma> ls)) "  
 using assms using cppl_fprojection[of "LIFT(f  fproj  g)" h \<sigma>]  by blast
 obtain ls where 3: "nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and>  
       nlast ls = the_enat(nlength \<sigma>) \<and>
       powerinterval (LIFT(f  fproj  g)) \<sigma> ls \<and>
         h (pfilt \<sigma> ls)"
  using "2" by blast
 have 4: "nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and> nlast ls  = the_enat(nlength \<sigma>) \<and>
            (\<forall> i <nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g) \<and>
         h (pfilt \<sigma> ls)" 
 using "3"  by (simp add: powerinterval_def) 
 have 6: "nlength ls>0"
  using "1" "4" 
  by (metis PJ6help4 nnth_nlast the_enat_0 zero_less_iff_neq_zero)
 have 7: "nidx   (nfusecat (lcppl f g \<sigma> ls)) \<and> nnth (nfusecat (lcppl f g \<sigma> ls)) 0 = 0" 
 by (metis (no_types, lifting) "1" "4" PJ7helpchain1a_help_1)
 have 8: "powerinterval f \<sigma> (nfusecat (lcppl f g \<sigma> ls))"   
  using "4" "6" PJ7helpchain1a_help_2 assms  by auto
 have 9: "nlast (nfusecat (lcppl f g \<sigma> ls))  = the_enat(nlength \<sigma>) " 
     by (metis "1" "4" PJ7helpchain1a_help_3 the_enat.simps)
 have 10: "((pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) \<Turnstile> g  fproj  h) " 
     using "1" "4" PJ7helpchain1a_help_4 by blast
show ?thesis 
using "10" "7" "8" "9" by (metis "3" fprojection_d_def nfinite_nmap pfilt_nmap)
qed


lemma OPJ7helpchain1a:
assumes "(\<sigma> \<Turnstile> (f  fproj  g) oproj h)" 
shows " (\<sigma> \<Turnstile> f oproj (g oproj h))"
proof -
 have 2: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and>
       powerinterval (LIFT(f  fproj  g)) \<sigma> ls \<and>
         h (pfilt \<sigma> ls)) "  
 using assms using cppl_oprojection[of "LIFT(f  fproj  g)" h \<sigma>]  by blast
 obtain ls where 3: "nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and> 
       powerinterval (LIFT(f  fproj  g)) \<sigma> ls \<and>
         h (pfilt \<sigma> ls)"
  using "2" by blast
 have 4: "nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and> 
            (\<forall> i <nlength ls. (nsubn \<sigma> (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f  fproj  g) \<and>
         h (pfilt \<sigma> ls)" 
 using "3"  by (simp add: powerinterval_def) 
 have 6: "nlength ls>0"
     by (metis "4" gr_zeroI nlength_eq_enat_nfiniteD zero_enat_def)
 have 7: "nidx   (nfusecat (lcppl f g \<sigma> ls)) \<and> nnth (nfusecat (lcppl f g \<sigma> ls)) 0 = 0" 
    using  "4" PJ7helpchain1a_help_1_alt 
    by (metis "6" OPJ6help4)
 have 8: "powerinterval f \<sigma> (nfusecat (lcppl f g \<sigma> ls))"  
    using  "4" PJ7helpchain1a_help_2_alt by blast
 have 10: "((pfilt \<sigma> (nfusecat (lcppl f g \<sigma> ls))) \<Turnstile> g oproj h) " 
    using  "4" PJ7helpchain1a_help_4_alt by blast
show ?thesis 
using "10" "7" "8" by (metis "3" nfinite_conv_nlength_enat oprojection_d_def pfilt_nlength)
qed
    


lemma PJ7helpchain1b:
assumes "nlength \<sigma> > 0"
        "(\<sigma> \<Turnstile> f fproj (g fproj h))" 
shows   "(\<sigma> \<Turnstile> (f fproj g) fproj h)"
proof -
 have 1: "nlength \<sigma> > 0" 
  using assms by auto
 have 2: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and> nlast ls  = the_enat(nlength \<sigma>) \<and>
         powerinterval f \<sigma> ls \<and>
         ((pfilt \<sigma> ls) \<Turnstile> g  fproj  h) ) " 
  using assms cppl_fprojection by blast
 obtain ls where 3: "nidx  ls \<and> nnth ls 0 = 0 \<and> nfinite ls \<and> nfinite \<sigma> \<and>  nlast ls  = the_enat(nlength \<sigma>) \<and>
         powerinterval f \<sigma> ls \<and>
         ((pfilt \<sigma> ls) \<Turnstile> g  fproj  h)" 
  using "2" by blast
 have 4: "(\<exists> lsa. nidx  lsa \<and>  nnth lsa 0 = 0 \<and> nfinite lsa \<and> nfinite (pfilt \<sigma> ls) \<and> 
           nlast  lsa  = the_enat(nlength(pfilt \<sigma> ls)) \<and>
           powerinterval g (pfilt \<sigma> ls) lsa \<and>
           ((pfilt (pfilt \<sigma> ls) lsa) \<Turnstile> h))" 
  using "3" using cppl_fprojection by blast
 obtain lsa where 5: "nidx  lsa \<and> nnth lsa 0 = 0 \<and> nfinite lsa \<and> nfinite (pfilt \<sigma> ls) \<and> 
           nlast lsa = the_enat(nlength(pfilt \<sigma> ls)) \<and>
           powerinterval g (pfilt \<sigma> ls) lsa \<and>
           ((pfilt (pfilt \<sigma> ls) lsa) \<Turnstile> h) " 
  using "4" by blast
 have 6: "nlength ls >0 "
   using "1" "3"  
   by (metis enat.distinct(2) enat_the_enat gr_zeroI nfinite_conv_nlength_enat nnth_nlast the_enat_0)
 have 7: "nlength(pfilt \<sigma> ls) = nlength ls "
    by (simp add: pfilt_nlength)
 have 8: "(pfilt (pfilt \<sigma> ls) lsa) = (pfilt \<sigma> (pfilt ls lsa)) " 
   using pfilt_nmap_pfilt by blast
 have 9: "nlast (pfilt ls lsa)  = the_enat(nlength \<sigma>) "
   by (metis "3" "5" "7" nlast_nmap nnth_nlast pfilt_nmap)
 have 10: "nlength lsa >0"
   using "5" "6" "7" 
   by (metis enat.distinct(2) enat_the_enat gr_zeroI nfinite_conv_nlength_enat nnth_nlast the_enat_0)
 have 11: "nlength (pfilt ls lsa) > 0  "
   by (metis "10" pfilt_nlength)
 have 111: "nnth (pfilt ls lsa) 0 = 0"
   by (metis "3" "5" pfilt_nnth zero_enat_def zero_le)
 have 112: "nlength(pfilt ls lsa) = nlength lsa"
   using pfilt_expand by blast
 have 113: "(\<forall> i< nlength lsa. (nnth (pfilt ls lsa) i) = (nnth ls (nnth lsa i))) "
   by (simp add: pfilt_nmap )
 have 114: "(\<forall> i< nlength lsa. (nnth (pfilt ls lsa) (Suc i)) = (nnth ls (nnth lsa (Suc i)))) "
   by (metis "112" eSuc_enat ileI1 pfilt_nnth)
 have 1141: "\<And>j. j \<le> nlength lsa\<longrightarrow> nnth lsa j \<le> nlength ls" 
   by (metis "5" "7" enat_ord_simps(1) ndropn_eq_NNil ndropn_nlast nfinite_conv_nlength_enat
          nidx_less_eq the_enat.simps)
 have 115: "(\<forall> i< nlength lsa. (nnth ls (nnth lsa i)) < (nnth ls (nnth lsa (Suc i))) ) "
   by (metis "1141" "3" "5" eSuc_enat ileI1 less_imp_Suc_add nidx_expand nidx_less)
 have 12: "nidx  (pfilt ls lsa) \<and> nnth (pfilt ls lsa) 0 = 0 "
   by (simp add: "111" "112" "115" Suc_ile_eq nidx_expand pfilt_nnth)
 have 2021: "(\<forall> i< nlength lsa. nfinite (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))) "
   by (simp add: nsubn_def1)
 have 2022: "\<And>i. i< nlength lsa \<Longrightarrow> (nnth ls (nnth lsa i)) \<le> nlength \<sigma> " 
   by (metis "3" "5" "7" enat.distinct(2) enat_ord_simps(1) enat_the_enat
       nfinite_conv_nlength_enat nidx_all_le_nlast order_less_imp_le)
 have 2023: "\<And>j. j \<le> nlength lsa\<longrightarrow> nnth lsa j \<le> nlength ls" 
   by (metis "5" "7" enat_ord_simps(1) ndropn_eq_NNil ndropn_nlast nfinite_conv_nlength_enat
       nidx_less_eq the_enat.simps)
 have 203: "(\<forall> i< nlength lsa.  nlength (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) =
                    (nnth ls (nnth lsa (Suc i))) - (nnth ls (nnth lsa i)))  " 
   by (metis "112" "113" "114" "12" "3" "5" "9" PJ6help1 le_add1 le_add_same_cancel1 
       nfinite_conv_nlength_enat)
 have 2041: "(\<forall> i< nlength lsa.  (nnth lsa (Suc i)) \<le> nlength ls )"
   by (metis "2023" eSuc_enat ileI1)
 have 204: "(\<forall> i< nlength lsa.  
             nlength (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) =
                     (nnth lsa (Suc i)) - (nnth lsa i))" 
   by (simp add: "3" "5" PJ6help1 pfilt_nlength)
 have 205: "(\<forall> i< nlength lsa.  
                  (\<forall> j \<le> (nnth lsa (Suc i)) - (nnth lsa i). 
                     (nnth (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) ) j) =
                      (nnth ls ((nnth lsa i) +j))
                  )
                )" 
   using "2041" "5" by (simp add: nsubn_def1 ntaken_nnth)
 have 2060: "(\<forall> i< nlength lsa. (nnth lsa i) \<le> (nnth lsa (Suc i)) )" 
   using "5" by (metis eSuc_enat ileI1 le_add2 nidx_less_eq plus_1_eq_Suc)
 have 206: "(\<forall> i< nlength lsa.  
                   (\<forall> j \<le> (nnth lsa (Suc i)) - (nnth lsa i).
                      (nnth ls (nnth lsa i)) \<le> (nnth ls ((nnth lsa i) +j))
                   )
                 )"  
   using "2041" "3" "2060" 
   by (metis le_add1 nidx_all_le_nlast nidx_less_eq nnth_beyond not_le_imp_less order_less_imp_le)
 have 207: " (\<forall> i< nlength lsa.  
                   (\<forall> j \<le> (nnth lsa (Suc i)) - (nnth lsa i).
                    (nnth (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) 
                                (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) j ) =
                      (nnth ls ((nnth lsa i) +j)) - (nnth ls (nnth lsa i))
                    ))" 
   using "204" "205" by auto
 have 208: "(\<forall> i< nlength lsa.  
                   (nnth (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) 
                               (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) 0) = 0 ) "
   by (simp add: "207")
 have 209: "(\<forall> i< nlength lsa.  
                   nlast (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) 
                               (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) =
                   (nnth ls (nnth lsa (Suc i))) - (nnth ls (nnth lsa i))
                ) 
               "   
   using "204" "207" "5" "2060" by (simp add: nsubn_def1 ntaken_ndropn_nlast) 
 have 210: "(\<forall> i< nlength lsa. 
              nidx  (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) \<and>
              nnth (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) 0 = 0  
                ) " 
   using 3 5 7  nidx_expand nidx_shiftm nidx_nsubn by (simp add: "2041" Suc_ile_eq order_less_imp_le)
 have 20: "powerinterval (LIFT(f fproj g)) \<sigma> (pfilt ls lsa)"
    proof -
     have 201: "powerinterval (LIFT(f fproj g)) \<sigma> (pfilt ls lsa) =
               (\<forall> i< nlength lsa. (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) \<Turnstile> f  fproj  g)  "
             unfolding powerinterval_def by (auto simp add: Suc_ile_eq pfilt_nlength pfilt_nnth)     
     have 202: "(\<forall> i< nlength lsa. (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) \<Turnstile> f  fproj  g) =
                (\<forall> i< nlength lsa. 
                  (\<exists> ls1. nidx  ls1 \<and> nnth ls1 0 = 0 \<and> nfinite ls1 \<and> 
                   nfinite (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) \<and>
                   nlast ls1 = the_enat(nlength (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) )) \<and>
                    powerinterval f (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) ls1 \<and>
                    ((pfilt (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) ls1) \<Turnstile> g)   
                     ))
                "
       by (simp add: fprojection_d_def)
     have fg: "(\<forall> i< nlength lsa. 
                  (\<exists> ls1. nidx  ls1 \<and> nnth ls1 0 = 0 \<and> nfinite ls1 \<and> 
                   nfinite (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) \<and>
                   nlast ls1 = the_enat(nlength (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) )) \<and>
                    powerinterval f (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) ls1 \<and>
                    ((pfilt (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) ls1) \<Turnstile> g)   
                     )) " 
     proof auto 
      fix i::nat
      assume a: "enat i < nlength lsa"
      show "\<exists>ls1. nidx ls1 \<and>
              nnth ls1 0 = 0 \<and>
              nfinite ls1 \<and>
              nfinite (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) \<and>
              nlast ls1 = the_enat (nlength (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))) \<and>
              powerinterval f (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) ls1 \<and> 
              g (pfilt (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) ls1)"
       proof -
       have fg1: "nnth (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) 0 = 0"
           using "210" a by blast
       have fg2: "nidx (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) ))"
           using 210 a by blast
       have fg3: "nfinite (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))"
          using "2021" a by blast 
       have fg4: "nlast (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) =
                    the_enat (nlength (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))))"
          using "203" "209" a the_enat.simps by presburger 
       have fg5: "powerinterval f (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                      (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) ))"
       proof (auto simp add: powerinterval_def)
        fix ia::nat
        assume aa: "enat ia < nlength (nsubn ls (nnth lsa i) (nnth lsa (Suc i)))" 
        show "f (nsubn (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                       (nnth (nsubn ls (nnth lsa i) (nnth lsa (Suc i))) ia - nnth ls (nnth lsa i))
                       (nnth (nmap (\<lambda>x. x - nnth ls (nnth lsa i)) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)))) (Suc ia))) " 
        proof -
         have f1: "ia < (nnth lsa (Suc i)) - (nnth lsa i)" 
            by (metis "5" "7" PJ6help1 a aa enat_ord_simps(2) le_add1 le_add_same_cancel1 nsubn_nlength)
         have f2: "nsubn (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                      (nnth ls (nnth lsa i + ia) - nnth ls (nnth lsa i))
                      (nnth ls (nnth lsa i + Suc ia) - nnth ls (nnth lsa i)) =
                nsubn \<sigma> (nnth ls (nnth lsa i + ia)) (nnth ls (nnth lsa i + Suc ia))"
         proof -
          have f3: "(nnth ls (nnth lsa i + ia) - nnth ls (nnth lsa i) \<le> 
                         nnth ls (nnth lsa i + Suc ia) - nnth ls (nnth lsa i))"
            by (metis "2041" "205" "2060" "3" Suc_leI a aa diff_le_mono eSuc_enat f1 ileI1 
                   le_add2 nidx_less_eq nidx_nsubn order_less_imp_le plus_1_eq_Suc)
          have f4: "nnth ls (nnth lsa i) \<le> nnth ls (nnth lsa (Suc i))"
            using "115" a dual_order.order_iff_strict by blast
          have f5: "enat (nnth ls (nnth lsa (Suc i))) \<le> nlength \<sigma>"
            by (metis "2041" "3" a enat_ord_code(4) enat_ord_simps(1) enat_the_enat nidx_all_le_nlast 
                order_less_imp_le)
          have f6: "nnth ls (nnth lsa i + Suc ia) - nnth ls (nnth lsa i) \<le>
                        nnth ls (nnth lsa (Suc i)) - nnth ls (nnth lsa i)" 
            by (metis "2041" "2060" "3" Suc_leI a add_diff_cancel_left' diff_le_mono f1 le_add1 
                le_diff_iff nidx_less_eq)
          show ?thesis using nsubn_nsubn_1[of "(nnth ls (nnth lsa i + ia) - nnth ls (nnth lsa i))" 
                                               "(nnth ls (nnth lsa i + Suc ia) - nnth ls (nnth lsa i))"
                  "(nnth ls (nnth lsa i))" " (nnth ls (nnth lsa (Suc i)))" \<sigma> ] 
            by (metis "206" Suc_leI a f1 f3 f4 f5 f6 le_add_diff_inverse order_less_imp_le)
         qed 
         have f7: "(nnth (nmap (\<lambda>x. x - nnth ls (nnth lsa i)) 
                                  (nsubn ls (nnth lsa i) (nnth lsa (Suc i)))) (Suc ia)) =
                      (nnth (nsubn ls (nnth lsa i) (nnth lsa (Suc i))) (Suc ia) - nnth ls (nnth lsa i))" 
           by (metis aa eSuc_enat ileI1 nnth_nmap)
         have f8: "f (nsubn \<sigma> (nnth ls ((nnth lsa i) +ia)) ((nnth ls ((nnth lsa i)+(Suc ia)))) ) "
          using 2041 3 unfolding powerinterval_def 
            by (metis a add.commute add_Suc_right enat_ord_simps(2) f1 less_diff_conv 
                order_less_le_trans )
         show ?thesis using "205" a f1 f2 f7 f8 by fastforce
        qed
       qed
       have fg6: "g (pfilt (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))
                      (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )))"
         proof -
          have g1: "(pfilt (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))
                      (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) ))) =
                        nmap (nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))))
                             (nmap (\<lambda>x. x - nnth ls (nnth lsa i)) (nsubn ls (nnth lsa i) (nnth lsa (Suc i))))"
            using pfilt_nmap[of "(nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))" 
               "(nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) ))"]  
            by blast
          have g2: "... = 
                       nmap (((nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))))) \<circ>
                             (\<lambda>x. x- (nnth ls (nnth lsa i)))) 
                            (nsubn ls (nnth lsa i) (nnth lsa (Suc i)))"
            using nellist.map_comp by blast  
          have g3: "... =
                       (nsubn (nmap (((nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))))) \<circ>
                                (\<lambda>x. x- (nnth ls (nnth lsa i)))) ls)
                               (nnth lsa i) (nnth lsa (Suc i))) "
            unfolding nsubn_def1 by (simp add: ndropn_nmap)       
          have g4: "\<And>y. (((nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))))) \<circ>
                                (\<lambda>x. x- (nnth ls (nnth lsa i)))) y = 
                            nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                                 (y - nnth ls (nnth lsa i))"
             by simp
          have g5: "(nsubn (nmap (((nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))))) \<circ>
                                (\<lambda>x. x- (nnth ls (nnth lsa i)))) ls)
                               (nnth lsa i) (nnth lsa (Suc i))) = 
                       (nsubn (nmap (\<lambda>y. nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                                         (y - nnth ls (nnth lsa i))) ls) 
                              (nnth lsa i) (nnth lsa (Suc i)))" 
            using g4 by presburger
          have g6: "\<And>j. (nnth lsa i) \<le>j \<and> j \<le> (nnth lsa (Suc i)) \<longrightarrow>
                          nnth (nmap (\<lambda>y. nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                                         (y - nnth ls (nnth lsa i))) ls) j =
                          nnth (nmap (nnth \<sigma>) ls) j"
            proof 
             fix j::nat
             assume aaa: "nnth lsa i \<le> j \<and> j \<le> nnth lsa (Suc i)"
             show "nnth (nmap (\<lambda>y. nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                              (y - nnth ls (nnth lsa i))) ls) j = nnth (nmap (nnth \<sigma>) ls) j"  
             proof -
              have g8: "nnth (nmap (nnth \<sigma>) ls) j = nnth \<sigma> (nnth ls j)"
                by (metis nfinite_ntaken nlast_nmap ntaken_nlast ntaken_nmap)  
              have g9: "nnth (nmap (\<lambda>y. nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                                         (y - nnth ls (nnth lsa i))) ls) j = 
                             (\<lambda>y. nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                                         (y - nnth ls (nnth lsa i))) (nnth ls j)" 
                by (metis (no_types, lifting) nfinite_ntaken nlast_nmap ntaken_nlast ntaken_nmap)
              have g10: "... = 
                           nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                              (nnth ls j - nnth ls (nnth lsa i)) " 
                by auto
              have g11: "... = nnth \<sigma> ((nnth ls (nnth lsa i)) + (nnth ls j - nnth ls (nnth lsa i)))"
                using nsubn_nnth[of \<sigma> "(nnth ls (nnth lsa i))" "(nnth ls (nnth lsa (Suc i)))" 
                  "(nnth ls j - nnth ls (nnth lsa i))"]   
                by (simp add: "2041" "3" a aaa diff_le_mono nidx_less_eq)
              have g12: "... = nnth \<sigma> (nnth ls j)"
                using aaa
                by (metis "3" le_add_diff_inverse nidx_all_le_nlast nidx_less_eq nnth_beyond 
                    not_le_imp_less order_less_imp_le) 
              show ?thesis 
              using g11 g12 g8 g9 by presburger 
             qed
            qed
          have g13: "(pfilt (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))
                      (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) ))) = 
                       (nsubn (pfilt \<sigma> ls) (nnth lsa i) (nnth lsa (Suc i)) )" 
                       by (simp add: "2041" "2060" a g2 g3 g5 g6 nsubn_eq pfilt_nmap)
          show ?thesis using "5" a by (simp add: g13 powerinterval_def)
         qed
       show ?thesis 
       using "203" "2041" "2060" "209" "210" a fg3 fg5 fg6 nsubn_nfinite by fastforce
     qed
   qed
  show ?thesis 
  using "201" "202" fg by blast
 qed
 show ?thesis 
 by (metis "12" "20" "3" "5" "8" "9" fprojection_d_def nfinite_nmap pfilt_nmap)
qed   


lemma OPJ7helpchain1b:
assumes "(\<sigma> \<Turnstile> f oproj (g oproj h))" 
shows   "(\<sigma> \<Turnstile> (f fproj g) oproj h)"
proof -
 have 2: "(\<exists>ls. nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and> 
         powerinterval f \<sigma> ls \<and>
         ((pfilt \<sigma> ls) \<Turnstile> g  oproj  h) ) " 
  using assms cppl_oprojection by blast
 obtain ls where 3: "nidx  ls \<and> nnth ls 0 = 0 \<and> \<not>nfinite ls \<and> \<not>nfinite \<sigma> \<and>  
         powerinterval f \<sigma> ls \<and>
         ((pfilt \<sigma> ls) \<Turnstile> g  oproj  h)" 
  using "2" by blast
 have 4: "(\<exists> lsa. nidx  lsa \<and>  nnth lsa 0 = 0 \<and> \<not>nfinite lsa \<and> \<not>nfinite (pfilt \<sigma> ls) \<and> 
           powerinterval g (pfilt \<sigma> ls) lsa \<and>
           ((pfilt (pfilt \<sigma> ls) lsa) \<Turnstile> h))" 
  using "3" using cppl_oprojection by blast
 obtain lsa where 5: "nidx  lsa \<and> nnth lsa 0 = 0 \<and> \<not>nfinite lsa \<and> \<not>nfinite (pfilt \<sigma> ls) \<and> 
           powerinterval g (pfilt \<sigma> ls) lsa \<and>
           ((pfilt (pfilt \<sigma> ls) lsa) \<Turnstile> h) " 
  using "4" by blast
 have 7: "nlength(pfilt \<sigma> ls) = nlength ls "
    by (simp add: pfilt_nlength)
 have 8: "(pfilt (pfilt \<sigma> ls) lsa) = (pfilt \<sigma> (pfilt ls lsa)) " 
   using pfilt_nmap_pfilt by blast
 have 11: "nlength (pfilt ls lsa) > 0  "
   by (metis "5" gr_zeroI nlength_eq_enat_nfiniteD pfilt_nlength zero_enat_def)
 have 111: "nnth (pfilt ls lsa) 0 = 0"
   by (metis "3" "5" pfilt_nnth zero_enat_def zero_le)
 have 112: "nlength(pfilt ls lsa) = nlength lsa"
   using pfilt_expand by blast
 have 113: "(\<forall> i< nlength lsa. (nnth (pfilt ls lsa) i) = (nnth ls (nnth lsa i))) "
   by (simp add: pfilt_nmap )
 have 114: "(\<forall> i< nlength lsa. (nnth (pfilt ls lsa) (Suc i)) = (nnth ls (nnth lsa (Suc i)))) "
   by (metis "112" eSuc_enat ileI1 pfilt_nnth)
 have 1141: "\<And>j. j \<le> nlength lsa\<longrightarrow> nnth lsa j \<le> nlength ls" 
   by (meson "3" enat_ile linorder_le_cases nfinite_conv_nlength_enat)     
 have 115: "(\<forall> i< nlength lsa. (nnth ls (nnth lsa i)) < (nnth ls (nnth lsa (Suc i))) ) "
   by (metis "1141" "3" "5" eSuc_enat ileI1 less_imp_Suc_add nidx_expand nidx_less)
 have 12: "nidx  (pfilt ls lsa) \<and> nnth (pfilt ls lsa) 0 = 0 "
   by (simp add: "111" "112" "115" Suc_ile_eq nidx_expand pfilt_nnth)
 have 2021: "(\<forall> i< nlength lsa. nfinite (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))) "
   by (simp add: nsubn_def1)
 have 2022: "\<And>i. i< nlength lsa \<Longrightarrow> (nnth ls (nnth lsa i)) \<le> nlength \<sigma> " 
   by (meson "3" enat_ile linorder_le_cases nfinite_conv_nlength_enat)
 have 203: "(\<forall> i< nlength lsa.  nlength (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) =
                    (nnth ls (nnth lsa (Suc i))) - (nnth ls (nnth lsa i)))  " 
   by (metis "112" "113" "114" "12" "3" "5" OPJ6help1 le_add1 le_add_same_cancel1 nfinite_nmap 
       pfilt_nmap)
 have 2041: "(\<forall> i< nlength lsa.  (nnth lsa (Suc i)) \<le> nlength ls )"
   by (metis "1141" eSuc_enat ileI1)
 have 204: "(\<forall> i< nlength lsa.  
             nlength (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) =
                     (nnth lsa (Suc i)) - (nnth lsa i))" 
   by (simp add: "3" "5" OPJ6help1)
 have 205: "(\<forall> i< nlength lsa.  
                  (\<forall> j \<le> (nnth lsa (Suc i)) - (nnth lsa i). 
                     (nnth (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) ) j) =
                      (nnth ls ((nnth lsa i) +j))
                  )
                )" 
   using "2041" "5" by (simp add: nsubn_def1 ntaken_nnth)
 have 2060: "(\<forall> i< nlength lsa. (nnth lsa i) \<le> (nnth lsa (Suc i)) )" 
   using "5" by (metis eSuc_enat ileI1 le_add2 nidx_less_eq plus_1_eq_Suc)
 have 206: "(\<forall> i< nlength lsa.  
                   (\<forall> j \<le> (nnth lsa (Suc i)) - (nnth lsa i).
                      (nnth ls (nnth lsa i)) \<le> (nnth ls ((nnth lsa i) +j))
                   )
                 )"  
   using "2041" "3" "2060" by (simp add: nfinite_conv_nlength_enat nidx_less_eq)
 have 207: " (\<forall> i< nlength lsa.  
                   (\<forall> j \<le> (nnth lsa (Suc i)) - (nnth lsa i).
                    (nnth (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) 
                                (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) j ) =
                      (nnth ls ((nnth lsa i) +j)) - (nnth ls (nnth lsa i))
                    ))" 
   using "204" "205" by auto
 have 208: "(\<forall> i< nlength lsa.  
                   (nnth (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) 
                               (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) 0) = 0 ) "
   by (simp add: "207")
 have 209: "(\<forall> i< nlength lsa.  
                   nlast (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) 
                               (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) =
                   (nnth ls (nnth lsa (Suc i))) - (nnth ls (nnth lsa i))
                ) "   
   using "204" "207" "5" "2060" by (simp add: nsubn_def1 ntaken_ndropn_nlast) 
 have 210: "(\<forall> i< nlength lsa. 
              nidx  (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) \<and>
              nnth (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) 0 = 0  
                ) " 
   using 3 5 7  nidx_expand nidx_shiftm nidx_nsubn by (simp add: "2041" Suc_ile_eq order_less_imp_le)
 have 20: "powerinterval (LIFT(f fproj g)) \<sigma> (pfilt ls lsa)"
    proof -
     have 201: "powerinterval (LIFT(f fproj g)) \<sigma> (pfilt ls lsa) =
               (\<forall> i< nlength lsa. (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) \<Turnstile> f  fproj  g)  "
             unfolding powerinterval_def by (auto simp add: Suc_ile_eq pfilt_nlength pfilt_nnth)     
     have 202: "(\<forall> i< nlength lsa. (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) \<Turnstile> f  fproj  g) =
                (\<forall> i< nlength lsa. 
                  (\<exists> ls1. nidx  ls1 \<and> nnth ls1 0 = 0 \<and> nfinite ls1 \<and> 
                   nfinite (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) \<and>
                   nlast ls1 = the_enat(nlength (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) )) \<and>
                    powerinterval f (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) ls1 \<and>
                    ((pfilt (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) ls1) \<Turnstile> g)   
                     ))"
       by (simp add: fprojection_d_def)
     have fg: "(\<forall> i< nlength lsa. 
                  (\<exists> ls1. nidx  ls1 \<and> nnth ls1 0 = 0 \<and> nfinite ls1 \<and> 
                   nfinite (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) \<and>
                   nlast ls1 = the_enat(nlength (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) )) \<and>
                    powerinterval f (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) ls1 \<and>
                    ((pfilt (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))) ) ls1) \<Turnstile> g)   
                     )) " 
     proof auto 
      fix i::nat
      assume a: "enat i < nlength lsa"
      show "\<exists>ls1. nidx ls1 \<and>
              nnth ls1 0 = 0 \<and>
              nfinite ls1 \<and>
              nfinite (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) \<and>
              nlast ls1 = the_enat (nlength (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))) \<and>
              powerinterval f (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) ls1 \<and> 
              g (pfilt (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) ls1)"
       proof -
       have fg1: "nnth (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) 0 = 0"
           using "210" a by blast
       have fg2: "nidx (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) ))"
           using 210 a by blast
       have fg3: "nfinite (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))"
          using "2021" a by blast 
       have fg4: "nlast (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )) =
                    the_enat (nlength (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))))"
          using "203" "209" a the_enat.simps by presburger 
       have fg5: "powerinterval f (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                      (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) ))"
       proof (auto simp add: powerinterval_def)
        fix ia::nat
        assume aa: "enat ia < nlength (nsubn ls (nnth lsa i) (nnth lsa (Suc i)))" 
        show "f (nsubn (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                       (nnth (nsubn ls (nnth lsa i) (nnth lsa (Suc i))) ia - nnth ls (nnth lsa i))
                       (nnth (nmap (\<lambda>x. x - nnth ls (nnth lsa i)) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)))) (Suc ia))) " 
        proof -
         have f1: "ia < (nnth lsa (Suc i)) - (nnth lsa i)" 
           by (metis "3" "5" OPJ6help1 a aa enat_ord_simps(2) le_add1 le_add_same_cancel1)
         have f2: "nsubn (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                      (nnth ls (nnth lsa i + ia) - nnth ls (nnth lsa i))
                      (nnth ls (nnth lsa i + Suc ia) - nnth ls (nnth lsa i)) =
                nsubn \<sigma> (nnth ls (nnth lsa i + ia)) (nnth ls (nnth lsa i + Suc ia))"
         proof -
          have f3: "(nnth ls (nnth lsa i + ia) - nnth ls (nnth lsa i) \<le> 
                         nnth ls (nnth lsa i + Suc ia) - nnth ls (nnth lsa i))"
            by (metis "2041" "205" "2060" "3" Suc_leI a aa diff_le_mono eSuc_enat f1 ileI1 
                   le_add2 nidx_less_eq nidx_nsubn order_less_imp_le plus_1_eq_Suc)
          have f4: "nnth ls (nnth lsa i) \<le> nnth ls (nnth lsa (Suc i))"
            using "115" a dual_order.order_iff_strict by blast
          have f5: "enat (nnth ls (nnth lsa (Suc i))) \<le> nlength \<sigma>"
            by (metis "2022" "5" a eSuc_enat ileI1 nfinite_conv_nlength_enat order_le_imp_less_or_eq)
          have f6: "nnth ls (nnth lsa i + Suc ia) - nnth ls (nnth lsa i) \<le>
                        nnth ls (nnth lsa (Suc i)) - nnth ls (nnth lsa i)" 
            by (metis "2041" "2060" "3" Suc_leI a add_diff_cancel_left' diff_le_mono f1 le_add1 
                le_diff_iff nidx_less_eq)
          show ?thesis using nsubn_nsubn_1[of "(nnth ls (nnth lsa i + ia) - nnth ls (nnth lsa i))" 
                                               "(nnth ls (nnth lsa i + Suc ia) - nnth ls (nnth lsa i))"
                  "(nnth ls (nnth lsa i))" " (nnth ls (nnth lsa (Suc i)))" \<sigma> ] 
            by (metis "206" Suc_leI a f1 f3 f4 f5 f6 le_add_diff_inverse order_less_imp_le)
         qed 
         have f7: "(nnth (nmap (\<lambda>x. x - nnth ls (nnth lsa i)) 
                                  (nsubn ls (nnth lsa i) (nnth lsa (Suc i)))) (Suc ia)) =
                      (nnth (nsubn ls (nnth lsa i) (nnth lsa (Suc i))) (Suc ia) - nnth ls (nnth lsa i))" 
           by (metis aa eSuc_enat ileI1 nnth_nmap)
         have f8: "f (nsubn \<sigma> (nnth ls ((nnth lsa i) +ia)) ((nnth ls ((nnth lsa i)+(Suc ia)))) ) "
          using 2041 3 unfolding powerinterval_def 
            by (metis a add.commute add_Suc_right enat_ord_simps(2) f1 less_diff_conv 
                order_less_le_trans )
         show ?thesis using "205" a f1 f2 f7 f8 by fastforce
        qed
       qed
       have fg6: "g (pfilt (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))
                      (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) )))"
         proof -
          have g1: "(pfilt (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))
                      (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) ))) =
                        nmap (nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))))
                             (nmap (\<lambda>x. x - nnth ls (nnth lsa i)) (nsubn ls (nnth lsa i) (nnth lsa (Suc i))))"
            using pfilt_nmap[of "(nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))" 
               "(nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) ))"]  
            by blast
          have g2: "... = 
                       nmap (((nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))))) \<circ>
                             (\<lambda>x. x- (nnth ls (nnth lsa i)))) 
                            (nsubn ls (nnth lsa i) (nnth lsa (Suc i)))"
            using nellist.map_comp by blast  
          have g3: "... =
                       (nsubn (nmap (((nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))))) \<circ>
                                (\<lambda>x. x- (nnth ls (nnth lsa i)))) ls)
                               (nnth lsa i) (nnth lsa (Suc i))) "
            unfolding nsubn_def1 by (simp add: ndropn_nmap)       
          have g4: "\<And>y. (((nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))))) \<circ>
                                (\<lambda>x. x- (nnth ls (nnth lsa i)))) y = 
                            nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                                 (y - nnth ls (nnth lsa i))"
             by simp
          have g5: "(nsubn (nmap (((nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))))) \<circ>
                                (\<lambda>x. x- (nnth ls (nnth lsa i)))) ls)
                               (nnth lsa i) (nnth lsa (Suc i))) = 
                       (nsubn (nmap (\<lambda>y. nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                                         (y - nnth ls (nnth lsa i))) ls) 
                              (nnth lsa i) (nnth lsa (Suc i)))" 
            using g4 by presburger
          have g6: "\<And>j. (nnth lsa i) \<le>j \<and> j \<le> (nnth lsa (Suc i)) \<longrightarrow>
                          nnth (nmap (\<lambda>y. nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                                         (y - nnth ls (nnth lsa i))) ls) j =
                          nnth (nmap (nnth \<sigma>) ls) j"
            proof 
             fix j::nat
             assume aaa: "nnth lsa i \<le> j \<and> j \<le> nnth lsa (Suc i)"
             show "nnth (nmap (\<lambda>y. nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                              (y - nnth ls (nnth lsa i))) ls) j = nnth (nmap (nnth \<sigma>) ls) j"  
             proof -
              have g8: "nnth (nmap (nnth \<sigma>) ls) j = nnth \<sigma> (nnth ls j)"
                by (metis nfinite_ntaken nlast_nmap ntaken_nlast ntaken_nmap)  
              have g9: "nnth (nmap (\<lambda>y. nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                                         (y - nnth ls (nnth lsa i))) ls) j = 
                             (\<lambda>y. nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                                         (y - nnth ls (nnth lsa i))) (nnth ls j)" 
                by (metis (no_types, lifting) nfinite_ntaken nlast_nmap ntaken_nlast ntaken_nmap)
              have g10: "... = 
                           nnth (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i)))) 
                              (nnth ls j - nnth ls (nnth lsa i)) " 
                by auto
              have g11: "... = nnth \<sigma> ((nnth ls (nnth lsa i)) + (nnth ls j - nnth ls (nnth lsa i)))"
                using nsubn_nnth[of \<sigma> "(nnth ls (nnth lsa i))" "(nnth ls (nnth lsa (Suc i)))"
                  "(nnth ls j - nnth ls (nnth lsa i))" ]
                by (simp add: "2041" "3" a aaa diff_le_mono nidx_less_eq)
              have g12: "... = nnth \<sigma> (nnth ls j)"
                using aaa 
                by (metis "206" a add_le_imp_le_diff diff_add le_add_diff_inverse)
              show ?thesis 
              using g11 g12 g8 g9 by presburger 
             qed
            qed
          have g13: "(pfilt (nsubn \<sigma> (nnth ls (nnth lsa i)) (nnth ls (nnth lsa (Suc i))))
                      (nmap (\<lambda>x. x- (nnth ls (nnth lsa i))) (nsubn ls (nnth lsa i) (nnth lsa (Suc i)) ))) = 
                       (nsubn (pfilt \<sigma> ls) (nnth lsa i) (nnth lsa (Suc i)) )" 
                       by (simp add: "2041" "2060" a g2 g3 g5 g6 nsubn_eq pfilt_nmap)
          show ?thesis using "5" a by (simp add: g13 powerinterval_def)
         qed
       show ?thesis 
       using "203" "2041" "2060" "209" "210" a fg3 fg5 fg6 nsubn_nfinite by fastforce
     qed
   qed
  show ?thesis 
  using "201" "202" fg by blast
 qed
 show ?thesis 
 by (metis "12" "20" "3" "5" "8" nfinite_nmap oprojection_d_def pfilt_nmap)
qed

lemma PJ7sem:
"(\<sigma> \<Turnstile> f  fproj  (g  fproj  h) = (f  fproj  g)  fproj  h)  "
proof -
 have 1: " nlength \<sigma> >0 \<longrightarrow> (\<sigma> \<Turnstile> f  fproj  (g  fproj  h) = (f  fproj  g)  fproj  h)"
  using PJ7helpchain1a PJ7helpchain1b unl_lift2 by blast
 have 2: " nlength \<sigma> = 0 \<longrightarrow> (\<sigma> \<Turnstile> f  fproj  (g  fproj  h) = (f  fproj  g)  fproj  h)"
   using PJ7empty by blast
 from 1 2 show ?thesis by auto
qed

lemma OPJ7sem:
"(\<sigma> \<Turnstile> f oproj (g oproj h) = (f  fproj  g) oproj h)  "
using OPJ7helpchain1a OPJ7helpchain1b unl_lift2 by blast


subsubsection \<open>PJ8\<close>

lemma PJ8semhelp:
 assumes "nidx  ls" 
         "nnth ls 0 = 0 "
         "nfinite ls"
         "nfinite \<sigma>" 
         "nlast ls  = the_enat(nlength \<sigma>) "
         " (\<forall>n na. na + n \<le> nlength \<sigma> \<longrightarrow> f (nsubn \<sigma> n (n+ na) ) \<longrightarrow> g (nsubn \<sigma> n (n + na) )) "
 shows "
         (\<forall>i<nlength ls. f (nsubn \<sigma> ( nnth ls i) ( nnth ls (Suc i)) )
                       \<longrightarrow> g (nsubn \<sigma>  ( nnth ls i) ( nnth ls (Suc i)) )
         )"
using assms unfolding nidx_expand by auto
 (metis assms(1) diff_add eSuc_enat enat_ord_simps(1) ileI1 le_add_diff_inverse less_imp_le_nat 
  nfinite_nlength_enat nidx_all_le_nlast the_enat.simps)

lemma PJ8semhelp_alt:
 assumes "nidx  ls" 
         "nnth ls 0 = 0 "
         "\<not>nfinite ls"
         "\<not>nfinite \<sigma>" 
         " (\<forall>n na. na + n \<le> nlength \<sigma> \<longrightarrow> f (nsubn \<sigma> n (n+ na) ) \<longrightarrow> g (nsubn \<sigma> n (n + na) )) "
 shows "
         (\<forall>i<nlength ls. f (nsubn \<sigma> ( nnth ls i) ( nnth ls (Suc i)) )
                       \<longrightarrow> g (nsubn \<sigma>  ( nnth ls i) ( nnth ls (Suc i)) )
         )"
using assms unfolding nidx_expand
by auto
 (metis enat_ord_simps(3) enat_the_enat le_add_diff_inverse nlength_eq_enat_nfiniteD order_less_imp_le)

lemma PJ8sem:
"(\<sigma> \<Turnstile> ba(f \<longrightarrow> g) \<longrightarrow> (f  fproj  h)  \<longrightarrow> (g  fproj  h))"
using PJ8semhelp
by (simp add: fprojection_d_def ba_defs powerinterval_def) 
 (metis eSuc_enat enat_ord_simps(1) ileI1 le_add_diff_inverse linorder_le_cases nfinite_nlength_enat
  nidx_expand nidx_less_eq nnth_nlast order_less_imp_le the_enat.simps)

lemma OPJ8sem:
"(\<sigma> \<Turnstile> ba(f \<longrightarrow> g) \<longrightarrow> (f oproj h)  \<longrightarrow> (g oproj h))"
using PJ8semhelp_alt
by (simp add: oprojection_d_def ba_defs powerinterval_def) 
 (metis eSuc_enat enat_ord_code(4) enat_the_enat ileI1 le_add_diff_inverse nidx_expand 
  nlength_eq_enat_nfiniteD order_less_imp_le order_less_imp_le)

subsubsection \<open>PJ9\<close>

lemma PJ9sem:
"(\<sigma> \<Turnstile> f ufproj ( g \<longrightarrow> h) \<longrightarrow> f  fproj  g  \<longrightarrow> f  fproj  h)"
by (simp add: ufprojection_d_def fprojection_d_def)
   (metis less_numeral_extra(3))

lemma OPJ9sem:
"(\<sigma> \<Turnstile> f uoproj ( g \<longrightarrow> h) \<longrightarrow> f oproj g  \<longrightarrow> f oproj h)"
by (simp add: uoprojection_d_def oprojection_d_def)
   (metis less_numeral_extra(3))

subsection \<open>Axioms\<close>



lemma FBpGen:
 assumes "\<turnstile> f"
 shows   "\<turnstile> fbp f"
using assms 
by (simp add: fbp_d_def ufprojection_d_def fprojection_d_def Valid_def)

lemma OBpGen:
 assumes "\<turnstile> f"
 shows   "\<turnstile> obp f"
using assms 
by (simp add: obp_d_def uoprojection_d_def oprojection_d_def Valid_def)

lemma PJ00:
 "\<turnstile> \<not>(f  fproj  inf) " 
using PJ00sem Valid_def by blast

lemma OPJ00:
 "\<turnstile> \<not>(f oproj finite) "
using OPJ00sem Valid_def by blast

lemma PJ01:
 "\<turnstile> (f  fproj  g) \<longrightarrow> finite"
using PJ01sem Valid_def by blast

lemma OPJ01:
 "\<turnstile> (f oproj g) \<longrightarrow> inf"
using OPJ01sem Valid_def by blast

lemma PJ02:
 "\<turnstile> (f  fproj  g) = ((f \<and> finite)  fproj  g)"
using PJ02sem Valid_def by blast

lemma OPJ02:
 "\<turnstile> (f oproj g) = ((f \<and> finite) oproj g)"
using OPJ02sem Valid_def by blast

lemma PJ03:
 "\<turnstile> (f  fproj  g) = (f  fproj  (g \<and> finite))"
using PJ03sem Valid_def by blast

lemma OPJ03:
 "\<turnstile> (f oproj g) = (f oproj (g \<and> inf))"
using OPJ03sem Valid_def by blast


lemma PJ1:
 "\<turnstile> f  fproj  ( g \<or> h) = (f  fproj  g \<or> f  fproj  h)"
using PJ1sem Valid_def by blast

lemma OPJ1:
 "\<turnstile> f oproj ( g \<or> h) = (f oproj g \<or> f oproj h)"
using OPJ1sem Valid_def by blast

lemma PJ2:
 "\<turnstile> f  fproj  empty = empty"
 using PJ2sem Valid_def by blast

lemma OPJ2:
 "\<turnstile> \<not>(f oproj empty) "
 using OPJ2sem Valid_def by blast

lemma PJ3:
 "\<turnstile> f  fproj  skip = (f \<and> more \<and> finite)" 
using PJ3sem Valid_def by blast

lemma OPJ3:
 "\<turnstile> \<not>(f oproj skip)" 
using OPJ3sem Valid_def by blast

lemma PJ4:
 "\<turnstile> f  fproj  (g;h) = (f  fproj  g) ; ( f  fproj  h)"
using PJ4sem Valid_def by blast

lemma OPJ4:
 "\<turnstile> f oproj ((g \<and> finite);h) = (f fproj g) ; ( f oproj h)"
using OPJ4sem Valid_def by blast

lemma PJ5:
 "\<turnstile> f  fproj  init(g) \<longrightarrow> init(g)"
using PJ5sem Valid_def by blast

lemma OPJ5:
 "\<turnstile> f oproj init(g) \<longrightarrow> init(g)"
using OPJ5sem Valid_def by blast

lemma PJ6:
 "\<turnstile> skip  fproj  g = (g \<and> finite)"
using PJ6sem Valid_def by blast

lemma OPJ6:
 "\<turnstile> skip oproj g = (g \<and> inf)"
using OPJ6sem Valid_def by blast

lemma PJ7:
 "\<turnstile> f  fproj  (g  fproj  h) = (f  fproj  g)  fproj  h"
using PJ7sem Valid_def by blast

lemma OPJ7:
 "\<turnstile> f oproj (g oproj h) = (f  fproj  g) oproj h"
using OPJ7sem Valid_def by blast

lemma PJ8:
 "\<turnstile> ba(f \<longrightarrow> g) \<longrightarrow> (f  fproj  h)  \<longrightarrow> (g  fproj  h)"
using PJ8sem Valid_def by blast

lemma OPJ8:
 "\<turnstile> ba(f \<longrightarrow> g) \<longrightarrow> (f oproj h)  \<longrightarrow> (g oproj h)"
using OPJ8sem Valid_def by blast

lemma PJ9:
 "\<turnstile> f ufproj (g \<longrightarrow> h) \<longrightarrow> f  fproj  g  \<longrightarrow> f  fproj  h"
using PJ9sem Valid_def by blast

lemma OPJ9:
 "\<turnstile> f uoproj (g \<longrightarrow> h) \<longrightarrow> f oproj g  \<longrightarrow> f oproj h"
using OPJ9sem Valid_def by blast




subsection \<open>Theorems\<close>

subsubsection \<open>Projection\<close>

lemma FPowerFProjLen:
 "\<turnstile> f  fproj  len n = fpower (f \<and> more) n"
proof 
  (induct n)
  case 0
  then show ?case 
  by (metis PJ2 fpower_d_def len_d_def wpow_0)
  next
  case (Suc n)
  then show ?case 
  by (metis AndMoreAndFiniteEqvAndFmore FiniteChopSkipEqvFiniteAndMore FiniteChopSkipEqvSkipChopFinite 
      FmoreEqvSkipChopFinite PJ3 PJ4sem fpower_d_def intI inteq_reflection len_d_def lift_and_com 
      wpow_Suc)
  qed  

lemma FProjLenExist:
 "\<turnstile> f  fproj  (\<exists>n. len n) = (\<exists> n. f  fproj  len n)"
  by (simp add: Valid_def fprojection_d_def, blast)

lemma FPowerFProjLenExist:
 "\<turnstile> (\<exists> n. f  fproj  len n) = (\<exists> n. fpower (f \<and> more) n)"
using FPowerFProjLen by (simp add: Valid_def FPowerFProjLen, blast)


lemma RightFProjImpFProj:
 assumes "\<turnstile> g1 \<longrightarrow> g2"
 shows   "\<turnstile> f  fproj  g1 \<longrightarrow> f  fproj  g2"
using assms 
by (simp add: Valid_def fprojection_d_def, blast)

lemma RightOProjImpOProj:
 assumes "\<turnstile> g1 \<longrightarrow> g2"
 shows   "\<turnstile> f oproj g1 \<longrightarrow> f oproj g2"
using assms 
by (simp add: Valid_def oprojection_d_def, blast)

lemma LeftFProjImpFProj:
 assumes "\<turnstile> f1 \<longrightarrow> f2"
 shows   "\<turnstile> f1  fproj  g \<longrightarrow> f2  fproj  g"
using assms 
by (simp add: Valid_def fprojection_d_def powerinterval_def, blast)

lemma LeftOProjImpOProj:
 assumes "\<turnstile> f1 \<longrightarrow> f2"
 shows   "\<turnstile> f1 oproj g \<longrightarrow> f2 oproj g"
using assms 
by (simp add: Valid_def oprojection_d_def powerinterval_def, blast)

lemma RightFProjEqvFProj:
 assumes "\<turnstile> g1 = g2"
 shows   "\<turnstile> f  fproj  g1 = f  fproj  g2"
using assms 
by (metis (mono_tags, lifting) Valid_def inteq_reflection unl_lift2)

lemma RightOProjEqvOProj:
 assumes "\<turnstile> g1 = g2"
 shows   "\<turnstile> f oproj g1 = f oproj g2"
using assms 
by (metis (mono_tags, lifting) Valid_def inteq_reflection unl_lift2)

lemma LeftFProjEqvFProj:
 assumes "\<turnstile> f1 = f2"
 shows   "\<turnstile> f1  fproj  g = f2  fproj  g"
using assms 
by (metis (mono_tags, lifting) Valid_def inteq_reflection unl_lift2)

lemma LeftOProjEqvOProj:
 assumes "\<turnstile> f1 = f2"
 shows   "\<turnstile> f1 oproj g = f2 oproj g"
using assms 
by (metis (mono_tags, lifting) Valid_def inteq_reflection unl_lift2)

lemma FProjTrueEqvSChopstar:
 "\<turnstile> f  fproj  #True = (schopstar f)"
proof -
 have 1: "\<turnstile>  finite = ( (\<exists>n. len n))"
   by (simp add: Finite_exist_len)
 have 2: "\<turnstile> f  fproj  #True = f  fproj  (\<exists>n. len n) " 
   using "1"  
   by (metis PJ03 Prop03 Prop10 int_simps(29) inteq_reflection lift_and_com)
 have 3: "\<turnstile> f  fproj  (\<exists>n. len n) = (\<exists> n. fpower (f \<and> more) n)" 
  using FPowerFProjLenExist FProjLenExist by fastforce
 have 4: "\<turnstile> (\<exists> n. fpower (f \<and> more) n) = (schopstar f)" 
    by (metis FPowerstardef int_eq schopstar_d_def)
 show ?thesis using "2" "3" "4" by fastforce
qed

lemma OProjTrueEqvAOmega:
 "\<turnstile> f oproj #True = (aomega f)"
using infinite_nidx_imp_infinite_interval
by (auto simp add: Valid_def oprojection_d_def aomega_d_def powerinterval_def) 
(meson enat_ile linorder_le_cases nfinite_conv_nlength_enat not_le_imp_less, blast)

lemma OProjTrueEqvOmega:
 "\<turnstile> f oproj #True = (omega f)"
by (metis OProjTrueEqvAOmega OmegaEqvAOmega int_eq)

lemma FProjSChopstarEqvSChopstarFProj:
 "\<turnstile> f  fproj  (schopstar g) = (schopstar (f  fproj  g))"
proof -
 have 1: "\<turnstile> f  fproj  (schopstar g) = f  fproj  (g  fproj  #True)"
  by (metis FProjTrueEqvSChopstar RightFProjEqvFProj inteq_reflection) 
 have 2: "\<turnstile> f  fproj  (g  fproj  #True) = (f  fproj  g)  fproj  #True"
  by (simp add: PJ7) 
 have 3: "\<turnstile> (f  fproj  g)  fproj  #True = (schopstar (f  fproj  g))" 
    by (simp add: FProjTrueEqvSChopstar)
 show ?thesis using "1" "2" "3" by fastforce
qed  

lemma OProjOmegaEqvOmegaFProj:
 "\<turnstile> f oproj (omega g) = (omega (f  fproj  g))"
proof -
 have 1: "\<turnstile> f oproj (omega g) = f oproj (g oproj #True)"
   by (metis OProjTrueEqvOmega RightOProjEqvOProj inteq_reflection)
 have 2: "\<turnstile> f oproj (g oproj #True) = (f  fproj  g) oproj #True"
  by (simp add: OPJ7) 
 have 3: "\<turnstile> (f  fproj  g) oproj #True = (omega (f  fproj  g))" 
    by (simp add: OProjTrueEqvOmega)
 show ?thesis using "1" "2" "3" by fastforce
qed


lemma OProjAndImp:
 " \<turnstile> f oproj (g1 \<and> g2) \<longrightarrow> f oproj g1 \<and> f oproj g2  "
by (meson Prop12 RightOProjImpOProj int_iffD1 lift_and_com)

lemma FProjAndImp:
 " \<turnstile> f  fproj  (g1 \<and> g2) \<longrightarrow> f  fproj  g1 \<and> f  fproj  g2  "
by (meson Prop12 RightFProjImpFProj int_iffD1 lift_and_com)


lemma FProjOrDist:
 "\<turnstile> #True  fproj  (f \<or>  g) = ( #True  fproj  f  \<or>  #True  fproj  g) "
using PJ1 by blast

lemma OProjOrDist:
 "\<turnstile> #True oproj (f \<or>  g) = ( #True oproj f  \<or>  #True oproj g) "
using OPJ1 by blast

lemma StateImportFProj:
 "\<turnstile> ((init w) \<and> f  fproj  g) = f  fproj  ((init w) \<and> g)"
by (auto simp add: Valid_def init_defs fprojection_d_def pfilt_nnth nidx_expand)
   (metis nlast_NNil ntaken_0 ntaken_nlast pfilt_nlength pfilt_nnth zero_enat_def zero_le,
    metis nlast_NNil ntaken_0 ntaken_nlast pfilt_nlength pfilt_nnth zero_enat_def zero_le)

lemma StateImportOProj:
 "\<turnstile> ((init w) \<and> f oproj g) = f oproj ((init w) \<and> g)"
by (auto simp add: Valid_def init_defs oprojection_d_def pfilt_nnth nidx_expand)
    (metis ndropn_0 ndropn_nfirst pfilt_nlength pfilt_nnth zero_enat_def zero_le,
    metis ndropn_0 ndropn_nfirst pfilt_nlength pfilt_nnth zero_enat_def zero_le)

lemma FProjStateAndNextEqvStateAndMoreChopFProj:
" \<turnstile> f  fproj  ( (init w) \<and> \<circle> g) = ((init w) \<and> (f \<and> more \<and> finite);(f fproj  g))"
proof -
 have 2: "\<turnstile> (f \<and> more \<and> finite);(f fproj  g) = f  fproj  \<circle> g"  
    using PJ3 PJ4 unfolding next_d_def by (metis inteq_reflection)  
 have 3: "\<turnstile> f  fproj  ( (init w) ) \<longrightarrow> init w "
   by (simp add: PJ5)
 have 4: "\<turnstile> (init w \<and>   f  fproj  \<circle> g) = f  fproj  ( (init w) \<and> \<circle> g) "
  by (simp add: StateImportFProj)
 have 5: "\<turnstile> f  fproj  ( (init w) \<and> \<circle> g) \<longrightarrow> ((init w) \<and> (f \<and> more \<and> finite);(f fproj  g))"
    using "2" "3" FProjAndImp by fastforce
  from 5 4 show ?thesis using "2" by fastforce
qed

lemma OProjStateAndNextEqvStateAndMoreChopFProj:
" \<turnstile> f oproj ( (init w) \<and> \<circle> g) = ((init w) \<and> (f \<and> more \<and> finite);(f oproj g))"
proof -
 have 1: "\<turnstile> (skip \<and> finite) = skip " 
       using itl_defs(1) itl_defs(5) nlength_eq_enat_nfiniteD by fastforce
 have 2: "\<turnstile> (f \<and> more \<and> finite);(f oproj g) = f oproj \<circle> g"  
    using PJ3 OPJ4 unfolding next_d_def by (metis "1" inteq_reflection)
 have 3: "\<turnstile> f oproj ( (init w) ) \<longrightarrow> init w "
   by (simp add: OPJ5)
 have 4: "\<turnstile> (init w \<and>   f oproj \<circle> g) = f oproj ( (init w) \<and> \<circle> g) "
  by (simp add: StateImportOProj)
 have 5: "\<turnstile> f oproj ( (init w) \<and> \<circle> g) \<longrightarrow> ((init w) \<and> (f \<and> more \<and> finite);(f oproj g))"
    using "2" "3" OProjAndImp by fastforce
  from 5 4 show ?thesis using "2" by fastforce
qed


lemma FProjNext:
 "\<turnstile> f  fproj  \<circle> g = (f \<and> more \<and> finite);(f  fproj  g)"
by (metis PJ3 PJ4 inteq_reflection next_d_def)

lemma OProjNext:
 "\<turnstile> f oproj \<circle> g = (f \<and> more \<and> finite);(f oproj g)"
by (metis DiamondEmptyEqvFinite DiamondEqvEmptyOrNextDiamond FiniteChopEqvDiamond 
    FiniteChopSkipEqvSkipChopFinite NowImpDiamond OPJ4 PJ3 Prop05 Prop10 SkipChopEqvNext 
    inteq_reflection)

lemma FProjWnext:
 "\<turnstile> f  fproj  (wnext g) = (empty \<or> (f \<and> more \<and> finite);(f fproj  g))"
proof -
 have 1: "\<turnstile> f  fproj  (wnext g) = f  fproj  (empty \<or> \<circle> g)"
  by (simp add: RightFProjEqvFProj WnextEqvEmptyOrNext)
 have 2: "\<turnstile> f  fproj  (empty \<or> \<circle> g) = (empty \<or> f  fproj  (\<circle> g)) "
   using PJ1 PJ2 by fastforce
 have 3: "\<turnstile> f  fproj  (\<circle> g) = (f \<and> more \<and> finite);(f fproj  g)"
   by (metis PJ3 PJ4 inteq_reflection next_d_def)
 show ?thesis 
 using "1" "2" "3" by fastforce
qed

lemma OProjWnext:
 "\<turnstile> f oproj (wnext g) = ((f \<and> more \<and> finite);(f oproj g))"
proof -
 have 1: "\<turnstile> f oproj (wnext g) = f oproj (empty \<or> \<circle> g)"
  by (simp add: RightOProjEqvOProj WnextEqvEmptyOrNext)
 have 2: "\<turnstile> f oproj (empty \<or> \<circle> g) = ( f oproj (\<circle> g)) "
   using OPJ2[of f] OPJ1[of f "LIFT empty" "LIFT \<circle> g"] by fastforce 
 have 3: "\<turnstile> f oproj (\<circle> g) = (f \<and> more \<and> finite);(f oproj g)"
   by (simp add: OProjNext)
 show ?thesis 
 using "1" "2" "3" by fastforce
qed

lemma FProjIntro:
 assumes "\<turnstile>   f   \<longrightarrow> ((g \<and>  more) \<and> finite); f "
 shows   "\<turnstile> f \<and> finite\<longrightarrow> g  fproj  #True "
using assms SCSIntro[of f g] FProjTrueEqvSChopstar[of g] unfolding schop_d_def 
by fastforce

lemma OProjIntro:
 assumes "\<turnstile>   f   \<longrightarrow> ((g \<and>  more) \<and> finite); f "
 shows   "\<turnstile> f  \<longrightarrow> g oproj #True "
using assms OProjTrueEqvOmega[of g]
by (metis AndMoreAndFiniteEqvAndFmore OmegaIntro inteq_reflection)

lemma RightBoxStateImportFProj:
 "\<turnstile> \<box>(init w) \<and> f  fproj  g \<longrightarrow> f  fproj  (\<box>(init w) \<and> g)"
by (simp add: Valid_def always_defs init_defs fprojection_d_def)
 (metis ndropn_all ndropn_nfirst nfinite_conv_nlength_enat nle_le pfilt_code(1) pfilt_nmap_pfilt) 

lemma RightBoxStateImportOProj:
 "\<turnstile> \<box>(init w) \<and> f oproj g \<longrightarrow> f oproj (\<box>(init w) \<and> g)"
by (simp add: Valid_def always_defs init_defs oprojection_d_def)
 (metis min_def ndropn_nfirst nfinite_conv_nlength_enat nfinite_ntaken ntaken_nlength pfilt_nnth)


lemma LeftBoxStateImportFProjhelp:
 "(\<forall>n. enat n \<le> nlength wa \<longrightarrow> w (NNil (nfirst (ndropn n wa)))) \<and>
         (\<exists>ls. (\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i)) \<and>
              nnth ls 0 = 0 \<and>
              nfinite ls \<and>
              nfinite wa \<and>
              nlast ls = the_enat (nlength wa) \<and> 
              (\<forall>i. enat i < nlength ls \<longrightarrow> f (nsubn wa (nnth ls i) (nnth ls (Suc i)))) \<and> g (pfilt wa ls)) \<longrightarrow>
         (\<exists>ls. (\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i)) \<and>
              nnth ls 0 = 0 \<and>
              nfinite ls \<and>
              nfinite wa \<and>
              nlast ls = the_enat (nlength wa) \<and>
              (\<forall>i. enat i < nlength ls \<longrightarrow>
                   f (nsubn wa (nnth ls i) (nnth ls (Suc i))) \<and>
                   (\<forall>n. enat n \<le> nlength (nsubn wa (nnth ls i) (nnth ls (Suc i))) \<longrightarrow>
                        w (NNil (nfirst (ndropn n (nsubn wa (nnth ls i) (nnth ls (Suc i)))))))) \<and>
              g (pfilt wa ls))"
proof 
 assume 0: "(\<forall>n. enat n \<le> nlength wa \<longrightarrow> w (NNil (nfirst (ndropn n wa)))) \<and>
         (\<exists>ls. (\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i)) \<and>
              nnth ls 0 = 0 \<and>
              nfinite ls \<and>
              nfinite wa \<and>
              nlast ls = the_enat (nlength wa) \<and> 
              (\<forall>i. enat i < nlength ls \<longrightarrow> f (nsubn wa (nnth ls i) (nnth ls (Suc i)))) \<and> g (pfilt wa ls))" 
 show   "(\<exists>ls. (\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i)) \<and>
              nnth ls 0 = 0 \<and>
              nfinite ls \<and>
              nfinite wa \<and>
              nlast ls = the_enat (nlength wa) \<and>
              (\<forall>i. enat i < nlength ls \<longrightarrow>
                   f (nsubn wa (nnth ls i) (nnth ls (Suc i))) \<and>
                   (\<forall>n. enat n \<le> nlength (nsubn wa (nnth ls i) (nnth ls (Suc i))) \<longrightarrow>
                        w (NNil (nfirst (ndropn n (nsubn wa (nnth ls i) (nnth ls (Suc i)))))))) \<and>
              g (pfilt wa ls))"
  proof -
   have 1: "(\<forall>n. enat n \<le> nlength wa \<longrightarrow> w (NNil (nfirst (ndropn n wa)))) "
    using 0 by auto
   have 2: "(\<exists>ls. (\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i)) \<and>
              nnth ls 0 = 0 \<and>
              nfinite ls \<and>
              nfinite wa \<and>
              nlast ls = the_enat (nlength wa) \<and> 
              (\<forall>i. enat i < nlength ls \<longrightarrow> 
                    f (nsubn wa (nnth ls i) (nnth ls (Suc i)))) \<and> g (pfilt wa ls))"
     using 0 by auto
   obtain ls where 3: " (\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i)) \<and>
              nnth ls 0 = 0 \<and>
              nfinite ls \<and>
              nfinite wa \<and>
              nlast ls = the_enat (nlength wa) \<and> 
              (\<forall>i. enat i < nlength ls \<longrightarrow> 
                    f (nsubn wa (nnth ls i) (nnth ls (Suc i)))) \<and> g (pfilt wa ls)"
      using "2" by auto
   have 4: " nnth ls 0 = 0" 
     using "3" by auto
   have 5: "(\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i))"  
     using "3" by auto
   have 6: " nlast  ls  = the_enat(nlength wa)" 
      using "3" by auto
   have 7: "(\<forall>i<nlength ls. f (nsubn wa ( nnth ls i) ( nnth ls (Suc i)) ))" 
      using "3" by auto
   have 8: "g (pfilt wa ls)"
      using "3" by auto
   have 9: "(\<forall>i<nlength ls.
              f (nsubn wa ( nnth ls i) ( nnth ls (Suc i))) \<and>
            (\<forall>n\<le>(nnth ls (Suc i)) -(nnth ls i).
                w (NNil (nnth wa ((nnth ls i)+n)))))"
       using 1 7 
       by (metis "0" ndropn_eq_NNil ndropn_nfirst ndropn_nlast nfinite_conv_nlength_enat 
           nnth_beyond not_le_imp_less the_enat.simps)
   have 10: "(\<forall>i<nlength ls. 
               nlength (nsubn wa ( nnth ls i) ( nnth ls (Suc i)) ) =
                (nnth ls (Suc i)) -(nnth ls i) ) " 
         by (simp add: "3" PJ6help1 Suc_ile_eq nidx_expand)
   have 11: "(\<forall>i<nlength ls.
              (\<forall>n\<le>(nnth ls (Suc i)) -(nnth ls i).
                 nnth (nsubn wa ( nnth ls i) ( nnth ls (Suc i)) ) n =
                 nnth wa ((nnth ls i)+n) )) "
       using "3"  by (simp add: nsubn_def1 ntaken_nnth)
  have 12: "(\<forall>i<nlength ls.
            f (nsubn wa ( nnth ls i) ( nnth ls (Suc i)) ) \<and>
            (\<forall>n\<le>nlength (nsubn wa ( nnth ls i) ( nnth ls (Suc i)) ).
                w (NNil (nnth (nsubn wa ( nnth ls i) ( nnth ls (Suc i))) n))))"
    using "9" "10" "11" by simp 
  have 13: "(\<forall>i. enat i < nlength ls \<longrightarrow>
                   f (nsubn wa (nnth ls i) (nnth ls (Suc i))) \<and>
                   (\<forall>n. enat n \<le> nlength (nsubn wa (nnth ls i) (nnth ls (Suc i))) \<longrightarrow>
                        w (NNil (nfirst (ndropn n (nsubn wa (nnth ls i) (nnth ls (Suc i))))))))" 
       by (simp add: "12" ndropn_nfirst)
  show ?thesis 
  using "13" "3" by blast
  qed
qed

lemma LeftBoxStateImportOProjhelp: 
 "(\<forall>n. enat n \<le> nlength wa \<longrightarrow> w (NNil (nfirst (ndropn n wa)))) \<and>
         (\<exists>ls. (\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i)) \<and>
              nnth ls 0 = 0 \<and>
              \<not> nfinite ls \<and> \<not> nfinite wa \<and> 
           (\<forall>i. enat i < nlength ls \<longrightarrow> f (nsubn wa (nnth ls i) (nnth ls (Suc i)))) \<and> g (pfilt wa ls)) \<longrightarrow>
         (\<exists>ls. (\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i)) \<and>
              nnth ls 0 = 0 \<and>
              \<not> nfinite ls \<and>
              \<not> nfinite wa \<and>
              (\<forall>i. enat i < nlength ls \<longrightarrow>
                   f (nsubn wa (nnth ls i) (nnth ls (Suc i))) \<and>
                   (\<forall>n. enat n \<le> nlength (nsubn wa (nnth ls i) (nnth ls (Suc i))) \<longrightarrow>
                        w (NNil (nfirst (ndropn n (nsubn wa (nnth ls i) (nnth ls (Suc i)))))))) \<and>
              g (pfilt wa ls))"
proof 
 assume 0: "(\<forall>n. enat n \<le> nlength wa \<longrightarrow> w (NNil (nfirst (ndropn n wa)))) \<and>
         (\<exists>ls. (\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i)) \<and>
              nnth ls 0 = 0 \<and>
              \<not> nfinite ls \<and> \<not> nfinite wa \<and> 
           (\<forall>i. enat i < nlength ls \<longrightarrow> f (nsubn wa (nnth ls i) (nnth ls (Suc i)))) \<and> g (pfilt wa ls))" 
 show   "(\<exists>ls. (\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i)) \<and>
              nnth ls 0 = 0 \<and>
              \<not> nfinite ls \<and>
              \<not> nfinite wa \<and>
              (\<forall>i. enat i < nlength ls \<longrightarrow>
                   f (nsubn wa (nnth ls i) (nnth ls (Suc i))) \<and>
                   (\<forall>n. enat n \<le> nlength (nsubn wa (nnth ls i) (nnth ls (Suc i))) \<longrightarrow>
                        w (NNil (nfirst (ndropn n (nsubn wa (nnth ls i) (nnth ls (Suc i)))))))) \<and>
              g (pfilt wa ls))"
  proof -
   have 1: "(\<forall>n. enat n \<le> nlength wa \<longrightarrow> w (NNil (nfirst (ndropn n wa)))) "
    using 0 by auto
   have 2: "(\<exists>ls. (\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i)) \<and>
              nnth ls 0 = 0 \<and>
              \<not> nfinite ls \<and> \<not> nfinite wa \<and> 
           (\<forall>i. enat i < nlength ls \<longrightarrow> f (nsubn wa (nnth ls i) (nnth ls (Suc i)))) \<and> g (pfilt wa ls))"
     using 0 by auto
   obtain ls where 3: " (\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i)) \<and>
              nnth ls 0 = 0 \<and>
              \<not> nfinite ls \<and> \<not> nfinite wa \<and> 
           (\<forall>i. enat i < nlength ls \<longrightarrow> f (nsubn wa (nnth ls i) (nnth ls (Suc i)))) \<and> g (pfilt wa ls)"
      using "2" by auto
   have 4: " nnth ls 0 = 0" 
     using "3" by auto
   have 5: "(\<forall>i. enat (Suc i) \<le> nlength ls \<longrightarrow> nnth ls i < nnth ls (Suc i))"  
     using "3" by auto
   have 7: "(\<forall>i<nlength ls. f (nsubn wa ( nnth ls i) ( nnth ls (Suc i)) ))" 
      using "3" by auto
   have 8: "g (pfilt wa ls)"
      using "3" by auto
   have 9: "(\<forall>i<nlength ls.
              f (nsubn wa ( nnth ls i) ( nnth ls (Suc i))) \<and>
            (\<forall>n\<le>(nnth ls (Suc i)) -(nnth ls i).
                w (NNil (nnth wa ((nnth ls i)+n)))))"
       using 1 7 
       by (metis "0" linorder_le_cases ndropn_eq_NNil ndropn_nfirst nfinite_NNil nfinite_ndropn_b)
   have 10: "(\<forall>i<nlength ls. 
               nlength (nsubn wa ( nnth ls i) ( nnth ls (Suc i)) ) =
                (nnth ls (Suc i)) -(nnth ls i) ) " 
        by (simp add: "3" OPJ6help1 nidx_expand)
   have 11: "(\<forall>i<nlength ls.
              (\<forall>n\<le>(nnth ls (Suc i)) -(nnth ls i).
                 nnth (nsubn wa ( nnth ls i) ( nnth ls (Suc i)) ) n =
                 nnth wa ((nnth ls i)+n) )) "
       using "3"  by (simp add: nsubn_def1 ntaken_nnth)
  have 12: "(\<forall>i<nlength ls.
            f (nsubn wa ( nnth ls i) ( nnth ls (Suc i)) ) \<and>
            (\<forall>n\<le>nlength (nsubn wa ( nnth ls i) ( nnth ls (Suc i)) ).
                w (NNil (nnth (nsubn wa ( nnth ls i) ( nnth ls (Suc i))) n))))"
    using "9" "10" "11" by simp 
  have 13: "(\<forall>i. enat i < nlength ls \<longrightarrow>
                   f (nsubn wa (nnth ls i) (nnth ls (Suc i))) \<and>
                   (\<forall>n. enat n \<le> nlength (nsubn wa (nnth ls i) (nnth ls (Suc i))) \<longrightarrow>
                        w (NNil (nfirst (ndropn n (nsubn wa (nnth ls i) (nnth ls (Suc i))))))))" 
       by (simp add: "12" ndropn_nfirst)
  show ?thesis 
  using "13" "3" by blast
  qed
qed
 
lemma LeftBoxStateImportFProj:
 "\<turnstile> \<box>(init w) \<and> f  fproj  g \<longrightarrow> (f \<and> \<box> (init w))  fproj  g"
using LeftBoxStateImportFProjhelp[of _ w f g]
by (simp add:  Valid_def always_defs init_defs fprojection_d_def powerinterval_def nidx_expand)

lemma LeftBoxStateImportOProj:
 "\<turnstile> \<box>(init w) \<and> f oproj g \<longrightarrow> (f \<and> \<box> (init w)) oproj g"
using LeftBoxStateImportOProjhelp[of _ w f g]
by (simp add:  Valid_def always_defs init_defs oprojection_d_def powerinterval_def nidx_expand)


subsubsection \<open>fdp, fbp odp and obp\<close>

lemma NotFDpEqvFBpNot:
 "\<turnstile> (\<not>(fdp f)) =  fbp (\<not> f)"
by (simp add: fbp_d_def fdp_d_def ufprojection_d_def)

lemma NotODpEqvOBpNot:
 "\<turnstile> (\<not>(odp f)) =  obp (\<not> f)"
by (simp add: obp_d_def odp_d_def uoprojection_d_def)

lemma NotFDBpEqvFDpNot:
 "\<turnstile> (\<not>(fbp f)) = fdp(\<not> f)"
by (simp add: fbp_d_def fdp_d_def ufprojection_d_def)

lemma NotODBpEqvODpNot:
 "\<turnstile> (\<not>(obp f)) = odp(\<not> f)"
by (simp add: obp_d_def odp_d_def uoprojection_d_def)

lemma NowImpFDp:
 "\<turnstile> f \<and> finite\<longrightarrow> fdp f"
proof -
 have 1: "\<turnstile>  (skip \<longrightarrow> #True)"
  by simp
 have 2: "\<turnstile> ba(skip \<longrightarrow> #True)"
  using "1" by (simp add: BaGen)
 have 3: "\<turnstile> ba(skip \<longrightarrow> #True) \<longrightarrow> (skip  fproj  f \<longrightarrow> #True  fproj  f)"
  using PJ8 by blast 
 have 4: "\<turnstile> (skip  fproj  f \<longrightarrow> #True  fproj  f)" 
  using "2" "3" MP by blast
 show ?thesis using 4 PJ6 
 by (metis "4" PJ6 fdp_d_def inteq_reflection)
qed 

lemma NowImpODp:
 "\<turnstile> f \<and> inf \<longrightarrow> odp f"
proof -
 have 1: "\<turnstile>  (skip \<longrightarrow> #True)"
  by simp
 have 2: "\<turnstile> ba(skip \<longrightarrow> #True)"
  using "1" by (simp add: BaGen)
 have 3: "\<turnstile> ba(skip \<longrightarrow> #True) \<longrightarrow> (skip oproj f \<longrightarrow> #True oproj f)"
  using OPJ8 by blast 
 have 4: "\<turnstile> (skip oproj f \<longrightarrow> #True oproj f)" 
  using "2" "3" MP by blast
 show ?thesis using 4 OPJ6 by (metis int_eq odp_d_def)
qed 


lemma FBpElim:
 "\<turnstile> fbp f \<and> finite\<longrightarrow> f"
proof -
 have 1: "\<turnstile> \<not> f \<and> finite\<longrightarrow> fdp (\<not> f)"
   by (simp add: NowImpFDp) 
 hence 2: "\<turnstile> \<not>(fdp (\<not> f)) \<longrightarrow> f \<or> inf"
   unfolding finite_d_def by auto  
 from 2 show ?thesis 
 by (simp add: Prop13 fbp_d_def fdp_d_def finite_d_def ufprojection_d_def) 
qed

lemma OBpElim:
 "\<turnstile> obp f \<and> inf \<longrightarrow> f"
proof -
 have 1: "\<turnstile> \<not> f \<and> inf \<longrightarrow> odp (\<not> f)" 
   by (simp add: NowImpODp) 
 hence 2: "\<turnstile> \<not>(odp (\<not> f)) \<longrightarrow> f \<or> finite"
   unfolding finite_d_def by auto  
 from 2 show ?thesis 
 by (metis InfEqvNotFinite Prop13 inteq_reflection obp_d_def odp_d_def uoprojection_d_def)
qed

lemma FBpImpFDpImpFDp:
  "\<turnstile> fbp (f \<longrightarrow> g) \<longrightarrow>  fdp  f \<longrightarrow>  fdp  g "
proof -
 have 1: "\<turnstile> fbp (f\<longrightarrow> g)  \<longrightarrow> (#True  fproj  f) \<longrightarrow> (#True  fproj  g) "  
   by (simp add: PJ9 fbp_d_def)
 from 1 show ?thesis   by (simp add: fdp_d_def)
qed

lemma OBpImpODpImpODp:
  "\<turnstile> obp (f \<longrightarrow> g) \<longrightarrow>  odp  f \<longrightarrow>  odp  g "
proof -
 have 1: "\<turnstile> obp (f\<longrightarrow> g)  \<longrightarrow> (#True oproj f) \<longrightarrow> (#True oproj g) "  
   by (simp add: OPJ9 obp_d_def)
 from 1 show ?thesis   by (simp add: odp_d_def)
qed

lemma FBpContraPosImpDist:
  "\<turnstile> fbp (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> (fbp  f) \<longrightarrow> (fbp  g) "
proof -
  have  1: "\<turnstile> fbp (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> ( fdp (\<not>  g)) \<longrightarrow> ( fdp (\<not>  f)) " 
     by (rule FBpImpFDpImpFDp)
  hence 2: "\<turnstile> fbp (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> (\<not> ( fdp (\<not>  f))) \<longrightarrow> (\<not> ( fdp (\<not>  g))) " by auto
  from 2 show ?thesis 
  by (simp add: fbp_d_def fdp_d_def ufprojection_d_def)
qed

lemma OBpContraPosImpDist:
  "\<turnstile> obp (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> (obp  f) \<longrightarrow> (obp  g) "
proof -
  have  1: "\<turnstile> obp (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> ( odp (\<not>  g)) \<longrightarrow> ( odp (\<not>  f)) " 
     by (rule OBpImpODpImpODp)
  hence 2: "\<turnstile> obp (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> (\<not> ( odp (\<not>  f))) \<longrightarrow> (\<not> ( odp (\<not>  g))) " by auto
  from 2 show ?thesis 
  by (simp add: obp_d_def odp_d_def uoprojection_d_def)
qed

lemma FBpImpDist:
 "\<turnstile> fbp (f\<longrightarrow> g) \<longrightarrow> (fbp  f) \<longrightarrow> (fbp  g) "
proof -
 have  1: "\<turnstile> (f \<longrightarrow> g) \<longrightarrow> (\<not>  g \<longrightarrow> \<not>  f) " by auto
 hence 2: "\<turnstile> \<not> (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> \<not> (f \<longrightarrow> g) " by auto
 hence 3: "\<turnstile> fbp (\<not> (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> \<not> (f \<longrightarrow> g)) " by (rule FBpGen)
 have  4: "\<turnstile> fbp (\<not> (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> \<not> (f \<longrightarrow> g))
             \<longrightarrow>  
             fbp (f \<longrightarrow> g) \<longrightarrow> fbp (\<not>  g \<longrightarrow> \<not>  f) " by (rule FBpContraPosImpDist)
 have  5: "\<turnstile> fbp (f \<longrightarrow> g) \<longrightarrow> fbp (\<not>  g \<longrightarrow> \<not>  f) " using 3 4 MP by blast
 have  6: "\<turnstile> fbp (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> (fbp  f) \<longrightarrow> (fbp  g) " by (rule FBpContraPosImpDist)
 from 5 6 show ?thesis  using lift_imp_trans by blast 
qed

lemma OBpImpDist:
 "\<turnstile> obp (f\<longrightarrow> g) \<longrightarrow> (obp  f) \<longrightarrow> (obp  g) "
proof -
 have  1: "\<turnstile> (f \<longrightarrow> g) \<longrightarrow> (\<not>  g \<longrightarrow> \<not>  f) " by auto
 hence 2: "\<turnstile> \<not> (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> \<not> (f \<longrightarrow> g) " by auto
 hence 3: "\<turnstile> obp (\<not> (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> \<not> (f \<longrightarrow> g)) " by (rule OBpGen)
 have  4: "\<turnstile> obp (\<not> (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> \<not> (f \<longrightarrow> g))
             \<longrightarrow>  
             obp (f \<longrightarrow> g) \<longrightarrow> obp (\<not>  g \<longrightarrow> \<not>  f) " by (rule OBpContraPosImpDist)
 have  5: "\<turnstile> obp (f \<longrightarrow> g) \<longrightarrow> obp (\<not>  g \<longrightarrow> \<not>  f) " using 3 4 MP by blast
 have  6: "\<turnstile> obp (\<not>  g \<longrightarrow> \<not>  f) \<longrightarrow> (obp  f) \<longrightarrow> (obp  g) " by (rule OBpContraPosImpDist)
 from 5 6 show ?thesis  using lift_imp_trans by blast 
qed

lemma FDpImpDpRule:
 assumes "\<turnstile> f\<longrightarrow> g "
 shows   "\<turnstile>  fdp  f \<longrightarrow>  fdp  g "
proof -
 have  1: "\<turnstile> f\<longrightarrow> g " using assms by auto
 hence 2: "\<turnstile> #True  fproj  f \<longrightarrow> #True  fproj  g " 
   by (metis FBpGen MP PJ9 fbp_d_def)
 from 2 show ?thesis by (simp add: fdp_d_def)
qed

lemma ODpImpODpRule:
 assumes "\<turnstile> f\<longrightarrow> g "
 shows   "\<turnstile>  odp  f \<longrightarrow>  odp  g "
proof -
 have  1: "\<turnstile> f\<longrightarrow> g " using assms by auto
 hence 2: "\<turnstile> #True oproj f \<longrightarrow> #True oproj g " 
   by (metis OBpGen MP OPJ9 obp_d_def)
 from 2 show ?thesis by (simp add: odp_d_def)
qed

lemma FBpImpFBpRule:
 assumes "\<turnstile> f\<longrightarrow> g"
 shows   "\<turnstile> fbp  f \<longrightarrow> fbp  g "
proof -
 have  1: "\<turnstile> f\<longrightarrow> g" using assms by auto
 hence 2: "\<turnstile> \<not>  g \<longrightarrow> \<not>  f " by auto
 hence 3: "\<turnstile>  fdp (\<not>  g) \<longrightarrow>  fdp (\<not>  f) " by (rule FDpImpDpRule)
 hence 4: "\<turnstile> \<not> ( fdp (\<not>  f)) \<longrightarrow> \<not> ( fdp (\<not>  g)) " by auto
 from 4 show ?thesis 
   by (meson FBpGen FBpImpDist MP assms)
qed

lemma OBpImpOBpRule:
 assumes "\<turnstile> f\<longrightarrow> g"
 shows   "\<turnstile> obp  f \<longrightarrow> obp  g "
proof -
 have  1: "\<turnstile> f\<longrightarrow> g" using assms by auto
 hence 2: "\<turnstile> \<not>  g \<longrightarrow> \<not>  f " by auto
 hence 3: "\<turnstile>  odp (\<not>  g) \<longrightarrow>  odp (\<not>  f) " by (rule ODpImpODpRule)
 hence 4: "\<turnstile> \<not> ( odp (\<not>  f)) \<longrightarrow> \<not> ( odp (\<not>  g)) " by auto
 from 4 show ?thesis 
   by (meson OBpGen OBpImpDist MP assms)
qed

lemma FDpEqvFDpRule:
 assumes "\<turnstile> f = g "
 shows   "\<turnstile>  fdp f =  fdp g "
proof -
 have  1: "\<turnstile> f = g " using assms by auto
 hence 2: "\<turnstile> #True  fproj  f =  #True  fproj  g " 
    using RightFProjEqvFProj by blast
 from 2 show ?thesis by (simp add: fdp_d_def)
qed

lemma ODpEqvODpRule:
 assumes "\<turnstile> f = g "
 shows   "\<turnstile>  odp f =  odp g "
proof -
 have  1: "\<turnstile> f = g " using assms by auto
 hence 2: "\<turnstile> #True oproj f =  #True oproj g " 
    using RightOProjEqvOProj by blast
 from 2 show ?thesis by (simp add: odp_d_def)
qed

lemma FBpEqvFBpRule:
 assumes "\<turnstile> f = g "
 shows   "\<turnstile>  fbp f =  fbp  g "
proof -
 have  1: "\<turnstile> f = g " using assms by auto
 hence 2: "\<turnstile> (\<not>  f) = (\<not>  g) " by auto
 hence 3: "\<turnstile>  fdp (\<not>  f) =  fdp (\<not>  g) " by (rule FDpEqvFDpRule)
 hence 4: "\<turnstile> (\<not>  (fdp (\<not>  f))) = (\<not> ( fdp (\<not>  g))) " by auto
 from 4 show ?thesis 
 by (metis FBpImpFBpRule assms int_iffD1 int_iffI inteq_reflection)
qed

lemma OBpEqvOBpRule:
 assumes "\<turnstile> f = g "
 shows   "\<turnstile>  obp f =  obp  g "
proof -
 have  1: "\<turnstile> f = g " using assms by auto
 hence 2: "\<turnstile> (\<not>  f) = (\<not>  g) " by auto
 hence 3: "\<turnstile>  odp (\<not>  f) =  odp (\<not>  g) " by (rule ODpEqvODpRule)
 hence 4: "\<turnstile> (\<not>  (odp (\<not>  f))) = (\<not> ( odp (\<not>  g))) " by auto
 from 4 show ?thesis 
 by (metis OBpImpOBpRule assms int_iffD1 int_iffI inteq_reflection)
qed

lemma FDpState:
 "\<turnstile> fdp (init w) = ((init w) \<and> finite) "
proof -
 have 1: " \<turnstile> init w \<and> finite \<longrightarrow> fdp (init w)" 
  using NowImpFDp[of "LIFT (init w)" ]  by blast
 have 2: "\<turnstile>fdp (init w) \<longrightarrow> init w " 
   unfolding fdp_d_def by (simp add: PJ5)
 have 3: "\<turnstile>fdp (init w) \<longrightarrow> finite "
   unfolding fdp_d_def by (simp add: PJ01) 
 show ?thesis 
 by (simp add: "1" "2" "3" Prop12 int_iffI)
qed

lemma ODpState:
 "\<turnstile> odp (init w) = ((init w) \<and> inf) "
proof -
 have 1: " \<turnstile> init w \<and> inf \<longrightarrow> odp (init w)" 
  using NowImpODp[of "LIFT (init w)" ]  by blast
 have 2: "\<turnstile>odp (init w) \<longrightarrow> init w " 
   unfolding odp_d_def by (simp add: OPJ5)
 have 3: "\<turnstile>odp (init w) \<longrightarrow> inf "
   unfolding odp_d_def by (simp add: OPJ01) 
 show ?thesis 
 by (simp add: "1" "2" "3" Prop12 int_iffI)
qed

lemma StateEqvFBp:
 "\<turnstile>  finite \<longrightarrow> (init w) = fbp  (init w) "
proof -
  have 1: "\<turnstile> (init w) \<longrightarrow> fbp  (init w) "
   by (metis (no_types, lifting) DiEqvNotBiNot DiState Initprop(2) NotDiEqvBiNot PJ5 fbp_d_def
        inteq_reflection lift_imp_neg ufprojection_d_def)
  have 2: "\<turnstile> fbp  (init w) \<and> finite \<longrightarrow> (init w) " using FBpElim by blast
  from 1 2 show ?thesis by fastforce
qed

lemma StateEqvOBp:
 "\<turnstile>  inf \<longrightarrow> (init w) = obp  (init w) "
proof -
  have 1: "\<turnstile> (init w) \<longrightarrow> obp  (init w) "
    by (metis (no_types, lifting) DiEqvNotBiNot DiState Initprop(2) NotDiEqvBiNot NotODpEqvOBpNot 
        ODpState Prop11 Prop12 inteq_reflection lift_imp_neg)
  have 2: "\<turnstile> obp  (init w) \<and> inf \<longrightarrow> (init w) " using OBpElim by blast
  from 1 2 show ?thesis by fastforce
qed

lemma FDpFDpEqvFDp:
 "\<turnstile> fdp (fdp f) = fdp f"
proof -
 have 2: "\<turnstile> #True  fproj  ( #True  fproj  f) = (#True  fproj  #True)  fproj  f" 
   by (simp add: PJ7)
 have 3: "\<turnstile> (#True  fproj  #True) = finite"
   by (metis ChopEmpty EmptyImpFinite NowImpFDp PJ01 Prop10 TrueChopAndFiniteEqvAndFiniteChopFinite 
      fdp_d_def int_eq int_iffI)
 have 4: "\<turnstile> finite  fproj  f = #True  fproj  f " 
       by (metis PJ02 Prop03 Prop10 finite_d_def int_simps(28) inteq_reflection lift_and_com)
 show ?thesis 
 by (metis "2" "3" "4" fdp_d_def int_eq)
qed

lemma ODpODpEqvODp:
 "\<turnstile> odp (odp f) = odp f"
proof -
 have 2: "\<turnstile> #True oproj ( #True oproj f) = (#True fproj #True) oproj f" 
   by (simp add: OPJ7)
 have 3: "\<turnstile> (#True  fproj  #True) = finite"
   by (metis ChopEmpty EmptyImpFinite NowImpFDp PJ01 Prop10 TrueChopAndFiniteEqvAndFiniteChopFinite 
      fdp_d_def int_eq int_iffI)
 have 4: "\<turnstile> finite oproj f = #True oproj f " 
       by (metis OPJ02 Prop03 Prop10 finite_d_def int_simps(28) inteq_reflection lift_and_com)
 show ?thesis 
 by (metis "2" "3" "4" odp_d_def int_eq)
qed


lemma FBpFBpEqvFBp:
 "\<turnstile> fbp (fbp f) = fbp f"
proof -
 have 1: "\<turnstile> fdp (fdp (\<not> f)) = fdp (\<not> f)"
   using FDpFDpEqvFDp by blast 
 have 2: "\<turnstile> (\<not> (fdp (fdp (\<not> f)))) = (\<not> (fdp (\<not> f)))"
   using "1" by auto
 have 3: "\<turnstile> (\<not> (fdp (\<not> f))) = fbp f"
    by (simp add: fbp_d_def fdp_d_def ufprojection_d_def)
 have 4: "\<turnstile> (\<not> (fdp (fdp (\<not> f)))) = fbp (fbp f)"
    by (simp add: fbp_d_def fdp_d_def ufprojection_d_def) 
 from 2 3 4 show ?thesis 
 by fastforce
qed

lemma OBpOBpEqvOBp:
 "\<turnstile> obp (obp f) = obp f"
proof -
 have 1: "\<turnstile> odp (odp (\<not> f)) = odp (\<not> f)"
   using ODpODpEqvODp by blast 
 have 2: "\<turnstile> (\<not> (odp (odp (\<not> f)))) = (\<not> (odp (\<not> f)))"
   using "1" by auto
 have 3: "\<turnstile> (\<not> (odp (\<not> f))) = obp f"
    by (simp add: obp_d_def odp_d_def uoprojection_d_def)
 have 4: "\<turnstile> (\<not> (odp (odp (\<not> f)))) = obp (obp f)"
    by (simp add: obp_d_def odp_d_def uoprojection_d_def) 
 from 2 3 4 show ?thesis 
 by fastforce
qed

lemma FDpOrEqv:
 "\<turnstile>    fdp (f \<or>  g) =  (fdp  f \<or>   fdp  g) " 
proof -
 have 1: "\<turnstile> #True  fproj  (f\<or>  g) = ( #True  fproj  f  \<or>  #True  fproj  g) " 
  using FProjOrDist by auto
 from 1 show ?thesis by (simp add: fdp_d_def)
qed

lemma ODpOrEqv:
 "\<turnstile>    odp (f \<or>  g) =  (odp  f \<or>   odp  g) " 
proof -
 have 1: "\<turnstile> #True oproj (f \<or> g) = ( #True oproj f  \<or>  #True oproj g) " 
  using OProjOrDist by auto
 from 1 show ?thesis by (simp add: odp_d_def)
qed

lemma FBpAndEqv:
 "\<turnstile> fbp(f \<and> g) = (fbp f \<and> fbp g)"
proof -
 have 1: "\<turnstile> fdp ((\<not> f) \<or>  (\<not>g)) =  (fdp  (\<not>f) \<or>   fdp  (\<not>g))"
   using FDpOrEqv by auto
 hence 2: "\<turnstile> (\<not> (fdp ((\<not> f) \<or>  (\<not>g)))) =  (\<not>(fdp  (\<not>f) \<or>   fdp  (\<not>g)))  "
   by auto
 have 3: "\<turnstile> (\<not> (fdp ((\<not> f) \<or>  (\<not>g)))) = fbp ( \<not>((\<not> f) \<or>  (\<not>g)))"
       using NotFDpEqvFBpNot by blast
 have 4: "\<turnstile> (\<not>((\<not> f) \<or>  (\<not>g))) =(f \<and> g)"
   by auto
 hence 5: "\<turnstile> fbp(\<not>((\<not> f) \<or>  (\<not>g))) = fbp(f \<and> g)"
    by (simp add: FBpEqvFBpRule)
 have 6: "\<turnstile> (\<not>(fdp  (\<not>f) \<or>  fdp  (\<not>g))) = ((\<not>(fdp (\<not> f))) \<and> (\<not>(fdp (\<not> g))))"
   by auto
 have 7: "\<turnstile> ((\<not>(fdp (\<not> f))) \<and> (\<not>(fdp (\<not> g)))) = (fbp f \<and> fbp g)"
    by (simp add: fbp_d_def fdp_d_def ufprojection_d_def)
  show ?thesis 
  by (metis "2" "3" "4" "6" "7" inteq_reflection)
qed

lemma OBpAndEqv:
 "\<turnstile> obp(f \<and> g) = (obp f \<and> obp g)"
proof -
 have 1: "\<turnstile> odp ((\<not> f) \<or>  (\<not>g)) =  (odp  (\<not>f) \<or>   odp  (\<not>g))"
   using ODpOrEqv by auto
 hence 2: "\<turnstile> (\<not> (odp ((\<not> f) \<or>  (\<not>g)))) =  (\<not>(odp  (\<not>f) \<or>   odp  (\<not>g)))  "
   by auto
 have 3: "\<turnstile> (\<not> (odp ((\<not> f) \<or>  (\<not>g)))) = obp ( \<not>((\<not> f) \<or>  (\<not>g)))"
       using NotODpEqvOBpNot by blast
 have 4: "\<turnstile> (\<not>((\<not> f) \<or>  (\<not>g))) =(f \<and> g)"
   by auto
 hence 5: "\<turnstile> obp(\<not>((\<not> f) \<or>  (\<not>g))) = obp(f \<and> g)"
    by (simp add: OBpEqvOBpRule)
 have 6: "\<turnstile> (\<not>(odp  (\<not>f) \<or>  odp  (\<not>g))) = ((\<not>(odp (\<not> f))) \<and> (\<not>(odp (\<not> g))))"
   by auto
 have 7: "\<turnstile> ((\<not>(odp (\<not> f))) \<and> (\<not>(odp (\<not> g)))) = (obp f \<and> obp g)"
    by (simp add: obp_d_def odp_d_def uoprojection_d_def)
  show ?thesis 
  by (metis "2" "3" "4" "6" "7" inteq_reflection)
qed

lemma FDpAndA:
 "\<turnstile>    fdp (f \<and> g) \<longrightarrow>  fdp  f "
proof -
 have 1: "\<turnstile> #True  fproj  (f \<and> g) \<longrightarrow> #True  fproj  f " 
   by (meson Prop12 RightFProjImpFProj int_iffD1 lift_and_com)
 from 1 show ?thesis by (simp add: fdp_d_def)
qed

lemma ODpAndA:
 "\<turnstile>    odp (f \<and> g) \<longrightarrow>  odp  f "
proof -
 have 1: "\<turnstile> #True oproj (f \<and> g) \<longrightarrow> #True oproj f " 
   by (meson Prop12 RightOProjImpOProj int_iffD1 lift_and_com)
 from 1 show ?thesis by (simp add: odp_d_def)
qed

lemma FBpOrA:
 "\<turnstile> fbp f \<longrightarrow>   fbp(f \<or> g)"
by (simp add: FBpImpFBpRule intI)

lemma OBpOrA:
 "\<turnstile> obp f \<longrightarrow>   obp(f \<or> g)"
by (simp add: OBpImpOBpRule intI)

lemma FBpOrB:
 "\<turnstile> fbp g \<longrightarrow>   fbp(f \<or> g)"
by (simp add: FBpImpFBpRule intI)

lemma OBpOrB:
 "\<turnstile> obp g \<longrightarrow>   obp(f \<or> g)"
by (simp add: OBpImpOBpRule intI)

lemma FBpOrImpOr:
 "\<turnstile> fbp f \<or> fbp g \<longrightarrow> fbp(f \<or> g)"
using FBpOrA FBpOrB by fastforce

lemma OBpOrImpOr:
 "\<turnstile> obp f \<or> obp g \<longrightarrow> obp(f \<or> g)"
using OBpOrA OBpOrB by fastforce

lemma FDpAndB:
 "\<turnstile>    fdp (f \<and> g) \<longrightarrow>  fdp  g "
proof -
 have 1: "\<turnstile> #True  fproj  (f \<and> g) \<longrightarrow> #True  fproj  g " 
    by (meson Prop12 RightFProjImpFProj int_iffD2 lift_and_com)
 from 1 show ?thesis by (simp add: fdp_d_def)
qed

lemma ODpAndB:
 "\<turnstile>    odp (f \<and> g) \<longrightarrow>  odp  g "
proof -
 have 1: "\<turnstile> #True oproj (f \<and> g) \<longrightarrow> #True oproj g " 
    by (meson Prop12 RightOProjImpOProj int_iffD2 lift_and_com)
 from 1 show ?thesis by (simp add: odp_d_def)
qed

lemma FDpAndImpAnd:
 "\<turnstile>    fdp (f \<and> g) \<longrightarrow>  fdp  f \<and>  fdp  g "  
proof -
 have 1: "\<turnstile>  fdp (f \<and> g) \<longrightarrow>  fdp  f " by (rule FDpAndA)
 have 2: "\<turnstile>  fdp (f \<and> g) \<longrightarrow>  fdp  g " by (rule FDpAndB)
 from 1 2 show ?thesis by fastforce
qed

lemma ODpAndImpAnd:
 "\<turnstile>    odp (f \<and> g) \<longrightarrow>  odp  f \<and>  odp  g "  
proof -
 have 1: "\<turnstile>  odp (f \<and> g) \<longrightarrow>  odp  f " by (rule ODpAndA)
 have 2: "\<turnstile>  odp (f \<and> g) \<longrightarrow>  odp  g " by (rule ODpAndB)
 from 1 2 show ?thesis by fastforce
qed

lemma FDpSkipEqvMore:
 "\<turnstile>    fdp  skip  =  (more \<and> finite) "
proof -
 have 1: "\<turnstile> fdp skip = #True  fproj  skip"
  by (simp add: fdp_d_def)
 have 2: "\<turnstile> #True  fproj  skip = (#True \<and> more \<and> finite)"
 using PJ3 by blast
 have 3: "\<turnstile> (#True \<and> more ) = more"
   by auto
 from 1 2 3 show ?thesis by fastforce
qed 


lemma FDpMoreEqvMore:
 "\<turnstile>    fdp  more  =  (more \<and> finite) "
using FDpFDpEqvFDp FDpSkipEqvMore 
by (metis PJ03 fdp_d_def inteq_reflection)

lemma ODpMoreEqvInf:
 "\<turnstile>    odp  more  =  ( inf) "
by (metis MoreAndInfEqvInf NowImpODp OPJ01 int_iffI inteq_reflection odp_d_def)

lemma FBpEmptyEqvEmpty:
 "\<turnstile> fbp empty = (empty \<or> inf)"
proof -
 have 1: "\<turnstile> fbp empty = (\<not> (fdp  more))"
     by (metis NotFDpEqvFBpNot Prop11 empty_d_def) 
 have 2: "\<turnstile> (\<not> (fdp  more)) = (\<not> (more \<and> finite))"
    using FDpMoreEqvMore by auto 
 have 3: "\<turnstile> (\<not> (more \<and> finite)) = (empty \<or> inf)" 
     unfolding finite_d_def empty_d_def by fastforce
 show ?thesis 
 by (metis "1" "2" "3" int_eq)
qed

lemma OBpEmptyEqvFinite:
 "\<turnstile> obp empty = (finite)"
proof -
 have 1: "\<turnstile> obp empty = (\<not> (odp  more))"
     by (metis NotODpEqvOBpNot Prop11 empty_d_def) 
 have 2: "\<turnstile> (\<not> (odp  more)) = (\<not> (inf))"
    using ODpMoreEqvInf by auto 
 have 3: "\<turnstile> (\<not> (inf)) = (finite)" 
     unfolding finite_d_def by fastforce
 show ?thesis 
 by (metis "1" "2" "3" int_eq)
qed

lemma FDpEmptyEqvEmpty:
 "\<turnstile>    fdp  empty = empty"
proof -
 have 1: "\<turnstile> fdp empty = #True  fproj  empty"
  by (simp add: fdp_d_def)
 have 2: "\<turnstile> #True  fproj  empty = empty"
   by (simp add: PJ2)
 from 1 2 show ?thesis by fastforce
qed

lemma NotODpEmpty:
 "\<turnstile>    \<not>(odp  empty) "
proof -
 have 1: "\<turnstile> odp empty = #True oproj empty"
  by (simp add: odp_d_def)
 show ?thesis 
 by (metis "1" OPJ2 int_eq)
qed

lemma FBpMoreEqvMore:
 "\<turnstile> fbp more = more"
by (metis NotFDBpEqvFDpNot PJ2 empty_d_def fdp_d_def int_eq int_simps(4))


lemma OBpMore:
 "\<turnstile> obp more "
by (metis OPJ2 empty_d_def obp_d_def uoprojection_d_def)


lemma NextFDpImpFDpNext:
 "\<turnstile>  \<circle> (fdp f) \<longrightarrow> fdp (\<circle> f)"
proof -
 have 1: "\<turnstile> fdp(\<circle> f) = #True  fproj  (skip;f) "
  by (simp add: fdp_d_def next_d_def) 
 have 2: "\<turnstile> #True  fproj  (skip;f) = (#True  fproj  skip);(#True  fproj  f)"
  by (simp add: PJ4)
 have 3: "\<turnstile> (#True  fproj  skip) = (#True \<and> more \<and> finite)"
  using PJ3 by blast
 have 4: "\<turnstile> (#True \<and> more) = more"
   by auto
 have 40: "\<turnstile> skip \<longrightarrow> more  "
     by (metis DiIntro DiSkipEqvMore int_eq)
 have 41: "\<turnstile> skip \<longrightarrow> finite" 
    by (metis AndChopB EmptySChop FiniteChopSkipImpFinite Prop11 lift_imp_trans schop_d_def)
 have 5: "\<turnstile> skip;(#True  fproj  f) \<longrightarrow> (more \<and> finite);(#True  fproj  f) "
     by (simp add: "40" "41" LeftChopImpChop Prop12)
 show ?thesis  by (metis "2" "5" FDpSkipEqvMore fdp_d_def inteq_reflection next_d_def)
qed

lemma NextODpImpODpNext:
 "\<turnstile>  \<circle> (odp f) \<longrightarrow> odp (\<circle> f)"
proof -
 have 1: "\<turnstile> odp(\<circle> f) = #True oproj (skip;f) "
  by (simp add: odp_d_def next_d_def) 
 have 10: "\<turnstile> (skip \<and> finite) = skip " 
   by (metis FiniteChopEqvDiamond FiniteChopSkipEqvSkipChopFinite NowImpDiamond Prop10 Prop11
        SkipChopFiniteImpFinite lift_imp_trans)
 have 2: "\<turnstile> #True oproj (skip;f) = (#True  fproj  skip);(#True oproj f)"
    using OPJ4 by (metis "10" inteq_reflection)
 have 3: "\<turnstile> (#True  fproj  skip) = (#True \<and> more \<and> finite)"
  using PJ3 by blast
 have 4: "\<turnstile> (#True \<and> more) = more"
   by auto
 have 40: "\<turnstile> skip \<longrightarrow> more  "
     by (metis DiIntro DiSkipEqvMore int_eq)
 have 41: "\<turnstile> skip \<longrightarrow> finite" 
    by (metis AndChopB EmptySChop FiniteChopSkipImpFinite Prop11 lift_imp_trans schop_d_def)
 have 5: "\<turnstile> skip;(#True oproj f) \<longrightarrow> (more \<and> finite);(#True oproj f) "
     by (simp add: "40" "41" LeftChopImpChop Prop12)
 show ?thesis by (metis "2" "5" FDpSkipEqvMore fdp_d_def inteq_reflection next_d_def odp_d_def)
qed

lemma BoxStateImportFBp:
 "\<turnstile> \<box>(init w)  \<longrightarrow> fbp (\<box>(init w))"
proof -
 have 1: "\<turnstile> fbp (\<box>(init w)) = (\<not>(fdp (\<diamond> (\<not>(init w))))) "
   by (metis NotFDpEqvFBpNot always_d_def int_eq) 
 have 2: "\<turnstile>  (\<diamond> (\<not>(init w))) = (finite;(\<not>(init w)))"
   by (simp add: sometimes_d_def) 
 have 3: "\<turnstile> fdp (finite;(\<not>(init w))) = (fdp finite); (fdp (\<not>(init w)))"
    by (simp add: PJ4 fdp_d_def) 
 have 4: "\<turnstile> (fdp finite) = finite"
    by (metis EmptyOrMoreSplit FDpEmptyEqvEmpty FiniteAndEmptyEqvEmpty FiniteChopSkipEqvFiniteAndMore 
        FiniteChopSkipImpFinite NowImpFDp PJ01 Prop02 Prop10 fdp_d_def int_iffI inteq_reflection) 
 have 5: "\<turnstile> (fdp (\<not>(init w))) = ((\<not>(init w)) \<and> finite) " 
     by (metis FDpState Initprop(2) inteq_reflection)
 have 6: "\<turnstile> finite;((\<not>(init w)) \<and> finite) \<longrightarrow> \<diamond>(\<not>(init w)) " 
     by (metis "2" ChopAndA inteq_reflection)
 show ?thesis
 by (metis "1" "2" "3" "4" "5" "6" always_d_def inteq_reflection lift_imp_neg)
qed

lemma BoxStateImportOBp:
 "\<turnstile> \<box>(init w)  \<longrightarrow> obp (\<box>(init w))"
proof -
 have 1: "\<turnstile> obp (\<box>(init w)) = (\<not>(odp (\<diamond> (\<not>(init w))))) "
   by (metis NotODpEqvOBpNot always_d_def int_eq) 
 have 2: "\<turnstile>  (\<diamond> (\<not>(init w))) = (finite;(\<not>(init w)))"
   by (simp add: sometimes_d_def) 
 have 3: "\<turnstile> odp (finite;(\<not>(init w))) = (fdp finite); (odp (\<not>(init w)))"
   unfolding odp_d_def fdp_d_def 
     using OPJ4[of "LIFT #True" "LIFT finite" "LIFT (\<not>(init w))" ]    
    by (simp add: OPJ4 odp_d_def fdp_d_def ) 
 have 4: "\<turnstile> (fdp finite) = finite"
    by (metis EmptyOrMoreSplit FDpEmptyEqvEmpty FiniteAndEmptyEqvEmpty FiniteChopSkipEqvFiniteAndMore 
        FiniteChopSkipImpFinite NowImpFDp PJ01 Prop02 Prop10 fdp_d_def int_iffI inteq_reflection) 
 have 5: "\<turnstile> (odp (\<not>(init w))) = ((\<not>(init w)) \<and> inf) " 
     by (metis ODpState Initprop(2) inteq_reflection)
 have 6: "\<turnstile> finite;((\<not>(init w)) \<and> inf) \<longrightarrow> \<diamond>(\<not>(init w)) " 
     by (metis "2" ChopAndA inteq_reflection)
 show ?thesis  by (metis "1" "2" "3" "4" "5" "6" always_d_def int_eq lift_imp_neg)
qed

lemma BoxStateEqvFBpBoxState:
 "\<turnstile> finite\<longrightarrow> \<box> (init w) = fbp(\<box> (init w))"
proof -
 have 1: "\<turnstile>  finite \<longrightarrow> fbp(\<box> (init w)) \<longrightarrow> \<box> (init w)" 
     by (metis FBpElim Prop09 inteq_reflection lift_and_com)
 have 2: "\<turnstile>  fbp(\<box> (init w)) = (\<not>(#True  fproj  (\<not> \<box> (init w))))"
   by (simp add: fbp_d_def ufprojection_d_def)
 have 2: "\<turnstile> \<box>(init w) \<and> finite\<longrightarrow> fdp (\<box>(init w))"
    by (metis NowImpFDp)
 have 2: "\<turnstile> \<box> (init w) \<longrightarrow> fbp(\<box> (init w))"
 using BoxStateImportFBp by auto
 from 1 2 show ?thesis by fastforce
qed

lemma BoxStateEqvOBpBoxState:
 "\<turnstile> inf \<longrightarrow> \<box> (init w) = obp(\<box> (init w))"
proof -
 have 1: "\<turnstile>  inf \<longrightarrow> obp(\<box> (init w)) \<longrightarrow> \<box> (init w)" 
     by (metis OBpElim Prop09 inteq_reflection lift_and_com)
 have 2: "\<turnstile>  obp(\<box> (init w)) = (\<not>(#True oproj (\<not> \<box> (init w))))"
   by (simp add: obp_d_def uoprojection_d_def)
 have 2: "\<turnstile> \<box>(init w) \<and> inf \<longrightarrow> odp (\<box>(init w))"
    by (metis NowImpODp)
 have 2: "\<turnstile> \<box> (init w) \<longrightarrow> obp(\<box> (init w))"
 using BoxStateImportOBp by auto
 from 1 2 show ?thesis by fastforce
qed

end
