(*  Title     : A Shallow Encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

section \<open>Pi operator\<close>   

theory Pi
imports  Until Omega 
begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="
  cvc4 z3 e spass vampire verit
"]
*)

text \<open>
This theory introduces the Pi operator 
\cite{moszkowski15:_applic_tempor_projec_inter_concur, 
      moszkowski17}. The Pi operator is defined in terms of the filter operator. 
We prove the soundness of the rules and axiom system. The until operator from 
Until.thy is used as there is a striking similarity of the expressiveness of the until and 
the Pi operator \cite{moszkowski17}. 
\<close>

subsection \<open>Definitions\<close>

definition sfxfilt :: "'a nellist \<Rightarrow> ('a:: world) formula \<Rightarrow> 'a nellist nellist"
 where "sfxfilt s f = (nfilter (\<lambda> \<sigma>. \<sigma> \<Turnstile>f) (ndropns s))"

definition pifilt :: " 'a nellist \<Rightarrow> ('a:: world) formula \<Rightarrow> 'a nellist"
 where "pifilt s f = (nmap (\<lambda>s. nnth s 0) (sfxfilt s f))"

definition pi_d :: "('a:: world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
where "pi_d F G \<equiv> \<lambda>s. ( (\<exists> i \<le> nlength s. (ndropn i s) \<Turnstile> F) \<and> ( (pifilt s F) \<Turnstile> G ) )"

syntax
 "_pi_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ \<Pi> _)" [84,84] 83)

syntax (ASCII)
 "_pi_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ PI _)" [84,84] 83)

translations
 "_pi_d"    \<rightleftharpoons> "CONST pi_d"


definition upi_d :: "('a:: world) formula \<Rightarrow> 'a formula \<Rightarrow> 'a formula"
where "upi_d F G \<equiv> LIFT(\<not>(F \<Pi> (\<not> G)))"


syntax
 "_upi_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ \<Pi>\<^sup>u _)" [84,84] 83)

syntax (ASCII)
 "_upi_d"     :: "[lift,lift] \<Rightarrow> lift"        ("(_ UPI _)" [84,84] 83)

translations
 "_upi_d"    \<rightleftharpoons> "CONST upi_d"



subsection \<open>Semantic Lemmas\<close>

lemma sfxfilter_help:
 "(\<exists> ys \<in> nset (ndropns xs) . f ys) = (\<exists> i \<le> nlength xs. f (ndropn i xs))"
using in_nset_ndropns by auto

lemma pifiltinit_help:
 "(\<exists> y \<in> nset (xs). w (NNil y) ) = (\<exists> i \<le> nlength xs. w (NNil (nnth xs i)))"
by (metis in_nset_conv_nnth)

lemma sfxfilt_nnth:
assumes "(\<exists> i \<le> nlength \<sigma>. (ndropn i \<sigma>) \<Turnstile> f)"
        " i\<le>nlength (sfxfilt \<sigma> f)"
shows   " (nnth (sfxfilt \<sigma> f) i) \<Turnstile> f"
using assms  by (metis in_nset_ndropns nkfilter_nnth_aa sfxfilt_def)

lemma pifilt_exists:
 assumes "(\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>) )"
 shows   "(\<exists>i \<le> nlength(sfxfilt \<sigma> f). (nnth (sfxfilt \<sigma> f) i) \<Turnstile> f)"
using assms by (metis sfxfilt_nnth zero_enat_def zero_le) 

lemma sfxfilt_pifilt_nlength:
shows   "nlength (pifilt \<sigma> f) = nlength (nmap (\<lambda>s. nnth s 0) (sfxfilt \<sigma> f))  "
by (simp add: pifilt_def)

lemma sfxfilt_pifilt_nnth:
assumes "j\<le> nlength (pifilt \<sigma> f)"
shows   " nnth (pifilt \<sigma> f) j = nnth (nmap (\<lambda>s. nnth s 0) (sfxfilt \<sigma> f)) j  "
using assms by (simp add: pifilt_def)

lemma sfxfilt_pifilt:
shows   "(nmap (\<lambda>s. nnth s 0) (sfxfilt \<sigma> f)) = pifilt \<sigma> f"
 by (simp add: pifilt_def)

lemma sfxfilt_nlength_bound:
assumes "(\<exists>i\<le>nlength xs. f (ndropn i xs))"
shows   "nlength (sfxfilt xs f) \<le> nlength xs "
using assms  
by (metis length_nfilter_le ndropns_nlength sfxfilt_def)

lemma pifilt_nlength_bound:
assumes "(\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>))"
shows   "nlength (pifilt \<sigma> f) \<le>  nlength \<sigma> "
using assms by (simp add: pifilt_def sfxfilt_nlength_bound)

lemma sfxfilt_nlength_nnth_bound:
assumes "(\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>))"
        " j\<le> nlength (sfxfilt \<sigma> f)"
shows   "nlength (nnth (sfxfilt \<sigma> f) j) \<le> nlength \<sigma> "
using assms by (simp add: nfilter_ndropns_nnth_bound sfxfilt_def sfxfilter_help) 

lemma sfxfilt_pifilt_nnth_ndropn:
assumes "(\<exists> i \<le> nlength \<sigma>. (ndropn i \<sigma>) \<Turnstile> f)"
        "j\<le> nlength (pifilt \<sigma> f)"
shows   "  nnth (sfxfilt \<sigma> f) j = ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) j) \<sigma>   "
proof -
 have 0: "\<exists>x\<in>nset (ndropns \<sigma>). f x"
   by (simp add: assms(1) sfxfilter_help)
 have 1: "nnth (sfxfilt \<sigma> f) j = nnth (nfilter f (ndropns \<sigma>)) j" 
   by (simp add: sfxfilt_def)
 have 2: "nnth (nfilter f (ndropns \<sigma>)) j = nnth (ndropns \<sigma>) (nnth (nkfilter f 0 (ndropns \<sigma>)) j)"
   using nkfilter_nfilter[of "ndropns \<sigma>" "\<lambda> s. s \<Turnstile> f" j 0] assms unfolding pifilt_def sfxfilt_def
   by (simp add: "0" nkfilter_nlength)
 have 30: "enat (nnth (nkfilter f 0 (ndropns \<sigma>)) j) \<le> nlength \<sigma>" 
   using 0 nkfilter_upperbound[of "(ndropns \<sigma>)" f j 0] assms unfolding pifilt_def sfxfilt_def 
   by (metis gen_nlength_def ndropns_nlength nkfilter_nlength nlength_code nlength_nmap)  
 have 3: "nnth (ndropns \<sigma>) (nnth (nkfilter f 0 (ndropns \<sigma>)) j) =
           ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) j) \<sigma>" 
   using ndropns_nnth[of "(nnth (nkfilter f 0 (ndropns \<sigma>)) j)"  ] 
   using "30" by blast
 show ?thesis by (simp add: "1" "2" "3")
qed

lemma pifilt_nnth:
 assumes "(\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>))"
         "i \<le> nlength (pifilt \<sigma> f)"
shows    "(\<exists> k \<le> nlength \<sigma>. nnth (pifilt \<sigma> f) i = nnth \<sigma> k)"
using assms sfxfilt_pifilt_nnth_ndropn[of "\<sigma>" "f" "i"] 
by (simp add: pifilt_def)
   (metis enat_the_enat infinity_ileE linorder_le_cases ndropn_all ndropn_nfirst)


lemma sfxfilt_nlength_imp:
 assumes "(\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>) \<and> g (ndropn i \<sigma>))"
 shows   " nlength (sfxfilt \<sigma> (LIFT(f \<and> g))) \<le> nlength (sfxfilt \<sigma> f)"
proof -
 have 1: "nlength (sfxfilt \<sigma> (LIFT(f \<and> g))) = 
          nlength (nfilter (\<lambda>ys. f ys \<and> g ys) (ndropns \<sigma>))" 
   by (simp add: sfxfilt_def)
 have 2: "nlength (nfilter f (ndropns \<sigma>)) = nlength (sfxfilt \<sigma> f) "
   by (simp add:  sfxfilt_def)
 have 3: "\<exists> x \<in> nset (ndropns \<sigma>). f x \<and> g x" 
   using assms in_nset_ndropns by blast
 have 4: "nlength (nfilter (\<lambda>x. f x \<and> g x) (ndropns \<sigma>)) \<le> nlength (nfilter f (ndropns \<sigma>))" 
   using "3" nfilter_nlength_imp[of "ndropns \<sigma>" "f" "g"] by auto
 show ?thesis using "1" "2" "4" by auto
qed

lemma pifilt_nlength_imp:
 assumes "(\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>) \<and> g (ndropn i \<sigma>))"
 shows   " nlength (pifilt \<sigma> (LIFT(f \<and> g))) \<le> nlength (pifilt \<sigma> f)"
using assms
by (simp add: sfxfilt_nlength_imp pifilt_def)

lemma nellist_nset_sfxfilt [simp]:
assumes "(\<exists> i \<le> nlength xs. f (ndropn i xs))"
shows "(nset (sfxfilt xs f)) = {ys. ys \<in> nset (ndropns xs) \<and> f (ys)}"
using assms 
proof -
 have 1: "nset (sfxfilt xs f) = nset (nfilter f (ndropns xs))"
   by (simp add: sfxfilt_def) 
 have 2: "\<exists> x \<in> nset (ndropns xs). f x" 
   using assms using in_nset_ndropns by blast
 have 3: "nset (nfilter f (ndropns xs)) = {ys. ys \<in> nset (ndropns xs) \<and> f ys}" 
   using "2" nset_nfilter[of "ndropns xs" "f"] by auto
 show ?thesis by (simp add: "1" "3")
qed 

lemma subset_nfilter:
 assumes "\<exists>x \<in> nset xs. P x"
 shows   " nset(nfilter P xs) \<le> nset(nfilter (\<lambda>x. P x \<or> Q x) xs)"
proof -
 have 1: "\<exists>x \<in> nset xs. P x \<or> Q x "
   using assms by blast 
 have 2: "nset(nfilter (\<lambda>x. P x \<or> Q x) xs) = {x \<in> nset xs. P x \<or> Q x} "
   using assms nset_nfilter[of xs "(\<lambda>x. P x \<or> Q x)"] by blast
 have 3: "nset(nfilter P xs) = {x\<in> nset xs. P x}"
   using assms nset_nfilter[of xs P] by (simp add: Collect_conj_eq) 
 have 4: "{x\<in> nset xs. P x} \<le> {x \<in> nset xs. P x \<or> Q x}"
   by auto
 show ?thesis  by (simp add: "2" "3" "4")  
qed

lemma nellist_subset_sfxfilt [simp]:
assumes "(\<exists> i \<le> nlength xs. f (ndropn i xs))"
shows "(nset (sfxfilt xs f)) \<le> (nset (sfxfilt xs (LIFT(f \<or> g))))"
proof -
 have 1: "\<exists> x \<in> nset (ndropns xs). f x"
   using assms using in_nset_ndropns by blast 
 have 2: "nset (sfxfilt xs f) = nset (nfilter f (ndropns xs))"
   by (simp add: sfxfilt_def)
 have 3: " nset (sfxfilt xs (LIFT(f \<or> g))) = nset (nfilter (\<lambda>x. f x \<or> g x) (ndropns xs))"
   by (simp add: sfxfilt_def)
 have 4: "nset (nfilter f (ndropns xs)) \<le> nset (nfilter (\<lambda>x. f x \<or> g x) (ndropns xs))"
   using "1"  subset_nfilter[of "ndropns xs" "f" "g"] by auto
 show ?thesis using "2" "3" "4" by blast
qed

lemma nellist_nset_nmap:
 "(nset (nmap (\<lambda>s. nnth s 0) xs)) = {(nnth ys 0) | ys. ys \<in> nset xs}" 
by (auto simp add: in_nset_conv_nnth nset_conv_nnth )
   (metis nnth_nmap)

lemma nellist_nset_pifilt [simp]:
assumes "(\<exists> i \<le> nlength xs. f (ndropn i xs))"
shows "(nset (pifilt xs f)) = {(nnth ys 0)| ys. ys \<in> nset(ndropns xs) \<and> f (ys)}"
proof -
 have 1: "(nset (pifilt xs f)) = (nset (nmap (\<lambda>s. nnth s 0) (sfxfilt xs f)))"
   by (simp add: pifilt_def)
 have 2: "(nset (nmap (\<lambda>s. nnth s 0) (sfxfilt xs f))) =
          {(nnth ys 0) | ys. ys \<in> nset (sfxfilt xs f)}"
   using nellist_nset_nmap[of "sfxfilt xs f" ] by auto
 have 3: "{(nnth ys 0) | ys. ys \<in> nset (sfxfilt xs f)} =
          {(nnth ys 0)| ys. ys \<in> nset(ndropns xs) \<and> f (ys)}" 
   using assms by auto
 show ?thesis using "1" "2" "3" by auto
qed

lemma nellist_nnth_sfxfilt_in_nset:
"x \<in> nset (sfxfilt \<sigma> (LIFT(f \<or> g))) =
 (\<exists> i \<le> nlength (sfxfilt \<sigma> (LIFT(f \<or> g))). x =(nnth (sfxfilt \<sigma> (LIFT(f \<or> g))) i))"
by (metis in_nset_conv_nnth)

lemma nfilter_nnth_or:
 assumes " \<exists> x \<in> nset xs. P x"
 shows   " \<exists> x \<in> nset(nfilter (\<lambda>x. P x \<or> Q x) xs). P x"
proof -
 have 0: "\<exists> x \<in> nset xs. P x \<or> Q x" 
   using assms by auto
 have 1: "\<exists>i \<le> nlength(nfilter (\<lambda>x. P x \<or> Q x) xs). P (nnth (nfilter P xs) i) "
   using assms nkfilter_nnth_aa[of xs "(\<lambda>x. P x)"] nkfilter_nnth_aa
   by (metis (mono_tags, lifting) le_cases le_zero_eq zero_enat_def)
 obtain i where 2: "i \<le> nlength(nfilter (\<lambda>x. P x \<or> Q x) xs) \<and>  P (nnth (nfilter P xs) i) \<and>
    (nnth (nfilter P xs) i) \<in> nset (nfilter P xs) "
   by (metis "1" in_nset_conv_nnth le_cases nfinite_ntaken nset_nlast ntaken_all ntaken_nlast)
 have 3: "nset (nfilter P xs) \<le> nset(nfilter (\<lambda>x. P x \<or> Q x) xs)"
   using assms subset_nfilter[of xs P Q] by auto
 have 4: "(nnth (nfilter P xs) i) \<in> nset (nfilter P xs)"
   using 2 by auto
 have 5: "(nnth (nfilter P xs) i) \<in> nset(nfilter (\<lambda>x. P x \<or> Q x) xs)"
   using 3 4 by blast
 show ?thesis  using 2 5 by blast
qed 

lemma sfxfilt_nnth_or:
assumes "(\<exists> i \<le> nlength \<sigma>. (ndropn i \<sigma>) \<Turnstile> f)"
shows   "(\<exists> i \<le> nlength (sfxfilt \<sigma> (LIFT(f \<or> g))). (nnth (sfxfilt \<sigma> (LIFT(f \<or> g)) ) i) \<Turnstile> f)"
proof -
 have 1: "\<exists> x \<in> nset (ndropns \<sigma>). f x" 
   using assms by (simp add: sfxfilter_help)
 have 2: "sfxfilt \<sigma> (LIFT(f \<or> g)) = nfilter (\<lambda> x. f x \<or> g x) (ndropns \<sigma>) "
   by (simp add: sfxfilt_def)
 have 3: "\<exists> x \<in> nset(nfilter (\<lambda> x. f x \<or> g x) (ndropns \<sigma>)). f x "
   using "1" nfilter_nnth_or[of "ndropns \<sigma>" "f" "g"] by auto
 from "2" "3" show ?thesis 
   by (metis (full_types) in_nset_conv_nnth)
qed

lemma NotPiFalse:
 "\<sigma> \<Turnstile> \<not> ((#False) \<Pi> f)"
by (simp add: pi_d_def)

lemma pifilt_true:
 " pifilt \<sigma> (LIFT(#True)) = \<sigma> "
by (simp add: pifilt_def sfxfilt_def nmap_first_ndropns) 

lemma pifilt_state_help:
 "(\<exists> x \<in> nset (ndropns xs) . (LIFT(init w)) x) = (\<exists> x \<in> nset xs. w ((NNil x)))"
proof (auto simp add: init_defs)
 show "\<And>x.  x \<in> nset (ndropns xs) \<Longrightarrow> w (NNil (nfirst x)) \<Longrightarrow> \<exists>x\<in>nset xs. w (NNil x)" 
 using in_nset_ndropns by (metis in_nset_conv_nnth ndropn_nfirst)
 show "\<And>x. x \<in> nset xs \<Longrightarrow> w (NNil x) \<Longrightarrow> \<exists>x\<in>nset (ndropns xs). w (NNil (nfirst x))"
 by (metis in_nset_ndropns_nhd ndropn_0 ndropn_nfirst)
qed

lemma pifilt_init:
 shows   "(pifilt xs (\<lambda>s. s \<Turnstile> init w)) = nfilter (\<lambda>y. w ((NNil y))) xs"
proof (simp add: init_defs pifilt_def sfxfilt_def)
 show "nmap (\<lambda>s. nnth s 0) (nfilter (\<lambda>s. w (NNil (nfirst s))) (ndropns xs)) = 
       nfilter (\<lambda>y. w (NNil y)) xs" 
 proof - 
 have 1: "\<And>x. ((\<lambda>y. w ((NNil y))) \<circ> (\<lambda>s. nnth s 0)) x = (\<lambda>s. w (NNil (nfirst s))) x"
   by simp  (metis ndropn_0 ndropn_nfirst) 
 have 2: "((\<lambda>y. w ((NNil y))) \<circ> (\<lambda>s. nnth s 0)) = (\<lambda>s. w (NNil (nfirst s)))"
   using 1 by blast
 have 4: "nmap (\<lambda>s. nnth s 0) (nfilter (\<lambda>s. w (NNil (nfirst s))) (ndropns xs)) = 
          nfilter (\<lambda>y. w ((NNil y))) (nmap (\<lambda>s. nnth s 0) (ndropns xs)) "
   by (metis (mono_tags, lifting) "1"  nfilter_cong nfilter_nmap)
 have 5: "(nmap (\<lambda>s. nnth s 0) (ndropns xs)) = xs"
   by (simp add: nmap_first_ndropns) 
 show ?thesis 
   by (simp add: "4" "5")
 qed 
qed

lemma pifilt_init_a:
 shows   "(pifilt xs (\<lambda>s. w (NNil (nnth s 0)) )) = nfilter (\<lambda>y. w (NNil y)) xs"
proof -
 have 2: "(pifilt xs (\<lambda>s. s \<Turnstile> init w)) = (pifilt xs (\<lambda>s. w (NNil (nnth s 0)) ))"
   by (metis (no_types, lifting) in_nset_ndropns init_defs ndropn_nfirst nfilter_cong nnth_zero_ndropn
       ntaken_0 pifilt_def sfxfilt_def)
 show ?thesis using pifilt_init 2 by (metis)
qed

lemma pifilt_pifilt :
assumes "(\<exists>i\<le>nlength xs. f (ndropn i xs))"
        "(\<exists>i\<le>nlength (pifilt xs f). w (NNil (nnth (ndropn i (pifilt xs f)) 0)))"
shows "(pifilt (pifilt xs f) (LIFT(init w)))  = pifilt xs (LIFT(f \<and> init w))"
proof -
 have 1: "\<exists>i\<le>nlength (pifilt xs f). (LIFT(init w)) (ndropn i (pifilt xs f))"
   using assms by (simp add: init_defs ndropn_nfirst)
 have 2: "(pifilt (pifilt xs f) (LIFT(init w))) = nfilter (\<lambda>y. w ((NNil y))) (pifilt xs f)" 
   using "1" pifilt_init[of "(pifilt xs f)" "w" ] by auto 
 have 3: "(pifilt xs f) = nmap (\<lambda>s. nnth s 0) (sfxfilt xs f)" 
   by (simp add: assms sfxfilt_pifilt)
 have 4: "(sfxfilt xs f) = nfilter (\<lambda> ys. f ys) (ndropns xs)"
   using sfxfilt_def by blast
 have 5: " (pifilt xs f) = nmap (\<lambda>s. nnth s 0) (nfilter (\<lambda> ys. f ys) (ndropns xs))" 
   by (simp add: "3" "4")
 have 6: "nfilter (\<lambda>y. w (NNil y)) (pifilt xs f)  =
          nfilter (\<lambda>y. w (NNil y)) (nmap (\<lambda>s. nnth s 0) (nfilter (\<lambda> ys. f ys) (ndropns xs)))" 
   using "5" by simp
 have 7: "nfilter (\<lambda>y. w (NNil y)) (nmap (\<lambda>s. nnth s 0) (nfilter (\<lambda> ys. f ys) (ndropns xs))) =
      nmap (\<lambda>s. nnth s 0) (nfilter ((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0)) (nfilter (\<lambda> ys. f ys) (ndropns xs))) " 
   using assms 3 4 by (simp add: nfilter_nmap pifiltinit_help)
 have 8: "\<exists>x\<in>nset (nfilter (\<lambda> ys. f ys) (ndropns xs)). ((\<lambda>y. w ((NNil y)))\<circ>(\<lambda>s. nnth s 0)) x" 
   using assms "3" "4"  in_nset_conv_nnth[of _ "(nfilter f (ndropns xs))"] 
   by simp  
      (metis in_nset_conv_nnth ndropn_0 ndropn_nfirst nlength_nmap nnth_nmap)
 have 9: "\<exists>x\<in>nset (ndropns xs). (\<lambda> ys. f ys) x" 
   using assms by (simp add: sfxfilter_help)
 have 10: "\<exists>x\<in>nset (ndropns xs). ((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0)) x \<and> (\<lambda> ys. f ys) x" 
   using "8" "9"
   using exist_conj[of f "(ndropns xs)" "((\<lambda>y. w (NNil y)) \<circ> (\<lambda>s. nnth s 0))"] by fastforce 
 have 11: "nfilter ((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0)) (nfilter (\<lambda> ys. f ys) (ndropns xs)) =
          nfilter (\<lambda> zs. ((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0)) zs \<and> (\<lambda> ys. f ys) zs) (ndropns xs)" 
   using nfilter_nfilter[of "(\<lambda> ys. f ys)" "(ndropns xs)" "((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0))" ]  
         "8" "10" by blast
 have 12: "\<exists>i\<le>nlength xs. (\<lambda> zs. ((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0)) zs \<and> (\<lambda> ys. f ys) zs) (ndropn i xs) "
   by (metis "10" in_nset_ndropns) 
 have 13: " (nfilter (\<lambda> zs. ((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0)) zs \<and> (\<lambda> ys. f ys) zs) (ndropns xs))
            = ( sfxfilt xs (\<lambda> zs. ((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0)) zs \<and> (\<lambda> ys. f ys) zs)) " 
   by (simp add: sfxfilt_def)
 have 14: "\<exists>i\<le>nlength xs. (\<lambda> zs. ((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0)) zs \<and> (\<lambda> ys. f ys) zs) (ndropn i xs)"
   using "12" by blast
 have 15: "nmap (\<lambda>s. nnth s 0) 
               (( sfxfilt xs (\<lambda> zs. ((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0)) zs \<and> (\<lambda> ys. f ys) zs)) ) = 
           pifilt xs (\<lambda> zs. ((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0)) zs \<and> (\<lambda> ys. f ys) zs)"  
   using "14" by (simp add: sfxfilt_pifilt)
 have 16: "\<And>xs . (\<lambda> zs. ((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0)) zs \<and> (\<lambda> ys. f ys) zs) xs =
            (LIFT(f \<and> init w)) xs" 
   by (simp add: init_defs)  (metis ndropn_0 ndropn_nfirst)
 have 17: "pifilt xs (\<lambda> zs. ((\<lambda>y. w (NNil y))\<circ>(\<lambda>s. nnth s 0)) zs \<and> (\<lambda> ys. f ys) zs) =
           pifilt xs (LIFT(f \<and> init w))" 
   using "16" by presburger
 show ?thesis 
   using "11" "13" "15" "17" "2" "3" "4" "7" by auto    
qed


lemma PiStatesem: 
 " (\<sigma> \<Turnstile> (init w) \<Pi> f) =
   ((\<exists>i\<le>nlength \<sigma>. w (NNil (nnth \<sigma> i))) \<and> f (nfilter (\<lambda>y. w ((NNil y))) \<sigma>))"
by (simp add: pi_d_def init_defs ndropn_nfirst pifilt_init) 


subsection \<open>Soundness of Axioms\<close>

subsubsection \<open>PiK\<close>

lemma PiKsem:
 "\<sigma> \<Turnstile> f1 \<Pi>\<^sup>u ( f \<longrightarrow> g) \<longrightarrow> (  f1 \<Pi>\<^sup>u f \<longrightarrow> f1 \<Pi>\<^sup>u g ) "
by (simp add: upi_d_def init_defs pi_d_def) auto

lemma PiK:
 "\<turnstile> f1 \<Pi>\<^sup>u (f \<longrightarrow> g) \<longrightarrow> (  f1 \<Pi>\<^sup>u f \<longrightarrow> f1 \<Pi>\<^sup>u g )"
using PiKsem Valid_def by blast 

subsubsection \<open>PiDc\<close>

lemma PiDcsem:
 " \<sigma> \<Turnstile> f \<Pi> g \<longrightarrow> f \<Pi>\<^sup>u g"
by (simp add: upi_d_def init_defs pi_d_def)

lemma PiDc:
 "\<turnstile> f \<Pi> g \<longrightarrow> f \<Pi>\<^sup>u g"
using PiDcsem Valid_def by blast

subsubsection \<open>PiN\<close>

lemma PiN:
 assumes "\<turnstile> g"
 shows   "\<turnstile> f \<Pi>\<^sup>u g"
using assms by (simp add: Valid_def pi_d_def upi_d_def)

subsubsection \<open>PiTrueEqvDiamond\<close>

lemma PiTrueEqvDiamond:
 "\<turnstile> f \<Pi> #True = \<diamond> f"
by (simp add: Valid_def pi_d_def itl_defs)

subsubsection \<open>PiOr\<close>

lemma PiOr:
 "\<turnstile> f \<Pi> (g1 \<or> g2) = (f \<Pi> g1 \<or> f \<Pi> g2)"
by (simp add: Valid_def pi_d_def) blast

subsubsection \<open>UPiFalseEqvBoxNot:\<close>

lemma UPiFalseEqvBoxNot:
 "\<turnstile> f \<Pi>\<^sup>u #False = \<box> (\<not>f)"
by (simp add: Valid_def upi_d_def pi_d_def itl_defs)

subsubsection \<open>BoxEqvImpPiEqv\<close>

lemma BoxEqvImpPiEqvsem:
assumes "(\<sigma> \<Turnstile> \<box> (f1 = f2))"
shows   "(\<sigma> \<Turnstile>(f1 \<Pi> g = f2 \<Pi> g))"
proof -
 show   "(\<sigma> \<Turnstile>(f1 \<Pi> g = f2 \<Pi> g))" 
 proof -
  have 1: "\<forall>n\<le>nlength \<sigma>. f1 (ndropn n \<sigma>) = f2 (ndropn n \<sigma>)" 
    using assms by (simp add: itl_defs)
  have 2: "(\<sigma> \<Turnstile>(f1 \<Pi> g)) = ((\<exists>i\<le>nlength \<sigma>. f1 (ndropn i \<sigma>)) \<and> g (pifilt \<sigma> f1))"
    by (simp add: pi_d_def)
  have 3: "(\<exists>i\<le>nlength \<sigma>. f1 (ndropn i \<sigma>)) = (\<exists>i\<le>nlength \<sigma>. f2 (ndropn i \<sigma>))"
    using "1" by blast   
  have 6: "(\<exists>i\<le>nlength \<sigma>. f1 (ndropn i \<sigma>)) \<Longrightarrow>  
           { (ndropn n \<sigma>) | n. n \<le> nlength \<sigma> \<and> f1 (ndropn n \<sigma>)} = 
           { (ndropn n \<sigma>) | n. n \<le> nlength \<sigma> \<and> f2 (ndropn n \<sigma>)} "
    using "1" by blast
  have 7: "(\<exists>i\<le>nlength \<sigma>. f1 (ndropn i \<sigma>)) \<Longrightarrow> (sfxfilt \<sigma> f1) =  (sfxfilt \<sigma> f2)"
    by (metis "1" in_nset_ndropns nfilter_cong sfxfilt_def)
  have 8: "((\<exists>i\<le>nlength \<sigma>. f1 (ndropn i \<sigma>)) \<and> g (pifilt \<sigma> f1)) = 
           ((\<exists>i\<le>nlength \<sigma>. f2 (ndropn i \<sigma>)) \<and> g (pifilt \<sigma> f2))"
             by (metis "3" "7" pifilt_def) 
  show ?thesis 
    by (simp add: "8" pi_d_def) 
 qed
qed
 
lemma BoxEqvImpPiEqv:
 "\<turnstile> \<box> (f1 = f2) \<longrightarrow> (f1 \<Pi> g = f2 \<Pi> g)"
using BoxEqvImpPiEqvsem by (simp add: Valid_def,auto)      


subsubsection \<open>PiDiamondImpDiamond\<close>

lemma PiDiamondImpDiamondsem:
 "\<sigma> \<Turnstile> f \<Pi> (\<diamond> (init w)) \<longrightarrow> \<diamond> (init w)"
by (simp add: Valid_def pi_d_def itl_defs ndropn_nfirst) 
   (metis  pifilt_nnth)

lemma PiDiamondImp:
 "\<turnstile> f \<Pi> (\<diamond> (init w)) \<longrightarrow> \<diamond> (init w)"
 using PiDiamondImpDiamondsem Valid_def by blast


subsubsection \<open>PiAssoc\<close>

                        
lemma PiAssocsem1:
assumes " i \<le> nlength \<sigma>"
        "f (ndropn i \<sigma>)"
        "ia \<le> nlength (pifilt \<sigma> f)"
        "w (NNil (nnth (pifilt \<sigma> f) ia))" 
shows   "\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>) \<and> w (NNil (nnth (ndropn i \<sigma>) 0))"
proof -
 have 1: "(nnth (pifilt \<sigma> f) ia) = (nnth (nmap (\<lambda>s. nnth s 0) (sfxfilt \<sigma> f)) ia)"
   using assms(1) assms(2) assms(3) sfxfilt_pifilt_nnth by blast 
 have 2: "(nnth (nmap (\<lambda>s. nnth s 0) (sfxfilt \<sigma> f)) ia) =
          (\<lambda>s. nnth s 0) (nnth (sfxfilt \<sigma> f) ia)" 
   by (metis assms(3) nlength_nmap nnth_nmap pifilt_def)
 have 3: "f  (nnth (sfxfilt \<sigma> f) ia)"
   using sfxfilt_nnth 
   by (metis assms(1) assms(2) assms(3) nlength_nmap pifilt_def)
 have 4: "nnth (sfxfilt \<sigma> f) ia = ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) ia) \<sigma>"
   using sfxfilt_pifilt_nnth_ndropn  assms(1) assms(2) assms(3) by blast
 have 5: "w (NNil (nnth (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) ia) \<sigma>)  0))  " 
   using "1" "2" "4" assms(4) by auto
 show ?thesis by (metis "3" "4" "5" enat_ile le_cases ndropn_all)
qed
  
lemma PiAssocsem2:
 assumes "i \<le> nlength \<sigma>"
         "f (ndropn i \<sigma>)"
         " w (NNil (nnth \<sigma> i))"
shows "\<exists>j\<le>nlength (pifilt \<sigma> f). w (NNil (nnth (pifilt \<sigma> f) j))"
proof -
 have 1: "\<exists>j\<le>nlength (sfxfilt \<sigma> f). f (nnth (sfxfilt \<sigma> f) j)"
   using assms pifilt_exists by blast
 have 2: " (LIFT (init w)) (ndropn i \<sigma>)"
   by (simp add: assms init_defs ndropn_nfirst)
 have 3: "\<exists>j\<le>nlength (sfxfilt \<sigma> (LIFT(init w))). (LIFT(init w)) (nnth (sfxfilt \<sigma> (LIFT(init w))) j)"
   using pifilt_exists "2" assms by blast
 have 4: "(LIFT (f \<and> init w)) (ndropn i \<sigma>)"
   by (simp add: "2" assms)
 have 5: "\<exists>j\<le>nlength (sfxfilt \<sigma> (LIFT(f \<and> init w))). 
           (LIFT(f \<and> init w)) (nnth (sfxfilt \<sigma> (LIFT(f \<and> init w))) j) "
   using pifilt_exists "4" assms by blast
 have 6: "\<exists>i\<le>nlength \<sigma>. ndropn i \<sigma> \<Turnstile> f \<and> init w"
   using "4" assms by blast
 have 7: "\<exists>j\<le>nlength (sfxfilt \<sigma> (LIFT((f \<and> init w) \<or> (f \<and> \<not> (init w)))  )). 
           (LIFT(f \<and> init w)) (nnth (sfxfilt \<sigma> (LIFT((f \<and> init w) \<or> (f \<and> \<not>(init w))) )) j)" 
   using "6" sfxfilt_nnth_or[of "\<sigma>" "LIFT(f \<and> init w)" "LIFT(f \<and> \<not>(init w))"] 
      by auto
 have 8: "\<And>\<sigma> . (\<sigma> \<Turnstile> ((f \<and> init w) \<or> (f \<and> \<not> (init w)))) = (\<sigma> \<Turnstile> f)"
   by auto  
 have 9: "(sfxfilt \<sigma> (LIFT((f \<and> init w) \<or> (f \<and> \<not> (init w)))  )) = 
          (sfxfilt \<sigma> f)"
   using "8" by (simp add: sfxfilt_def)
 have 10: "\<exists>j\<le>nlength (sfxfilt \<sigma> f). 
           (LIFT(f \<and> init w)) (nnth (sfxfilt \<sigma> f) j)"
   using "7" "9" by auto 
 have 11: "nlength (sfxfilt \<sigma> f) = nlength (pifilt \<sigma> f)" 
   by (simp add: pifilt_def)
 have 12: "\<exists>j\<le>nlength (pifilt \<sigma> f). 
           (LIFT(init w)) (nnth (sfxfilt \<sigma> f) j)" 
   using "10" "11" by auto
 from 12 11 show ?thesis unfolding  pifilt_def init_defs 
   by (metis nlast_NNil nnth_nmap ntaken_0 ntaken_nlast)
qed    
    
lemma PiAssocsema:
 "((\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>)) \<and>
     (\<exists>i\<le>nlength (pifilt \<sigma> f). w (NNil (nnth (ndropn i (pifilt \<sigma> f)) 0)) )) =
  (\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>) \<and> w (NNil (nnth (ndropn i \<sigma>) 0))) "
using PiAssocsem1[of _ \<sigma> f _ w] PiAssocsem2[of _ \<sigma> f w] by fastforce

lemma PiAssocsemb:
 "((\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>)) \<and>
     (\<exists>i\<le>nlength (pifilt \<sigma> f). (LIFT(init w)) (ndropn i (pifilt \<sigma> f)) )) =
  (\<exists>i\<le>nlength \<sigma>. (LIFT(f \<and> init w)) (ndropn i \<sigma>)) "
using PiAssocsem1[of _ \<sigma> f _ w] PiAssocsem2[of _ \<sigma> f w] 
by (simp add: init_defs ndropn_nfirst) blast


lemma PiAssocsem:
 "\<sigma> \<Turnstile> f \<Pi> ((init w) \<Pi>  g) =   (f \<and> (init w)) \<Pi> g"
proof (auto simp add: pi_d_def init_defs)
 fix i 
 fix ia
 assume a0: "g (pifilt (pifilt \<sigma> f) (LIFT(init w)))"
 assume a1: "(enat i) \<le> nlength \<sigma>"  
 assume a2: "f (ndropn i \<sigma>)"
 assume a3: "(enat ia) \<le> nlength (pifilt \<sigma> f)"
 assume a4: " w (NNil (nfirst (ndropn ia (pifilt \<sigma> f))))"   
 show "\<exists>i. enat i \<le> nlength \<sigma> \<and> f (ndropn i \<sigma>) \<and> w (NNil (nfirst (ndropn i \<sigma>)))" 
  using a0 a1 a2 a3 a4 PiAssocsem1 
  by (metis ndropn_nfirst nnth_zero_ndropn)
next
 fix i
 fix ia
 assume a0: "g (pifilt (pifilt \<sigma> f) (LIFT(init w)))"
 assume a1: "(enat i) \<le> nlength \<sigma>"
 assume a2: "f (ndropn i \<sigma>)"
 assume a3: "(enat ia) \<le> nlength (pifilt \<sigma> f)"
 assume a4: "w (NNil (nfirst (ndropn ia (pifilt \<sigma> f))))"      
 show "g (pifilt \<sigma> (LIFT(f \<and> init w)))" 
  using a0 a1 a2 a3 a4 by (metis ndropn_nfirst nnth_zero_ndropn pifilt_pifilt)
next
 fix i
 assume a0: " g (pifilt \<sigma> (LIFT(f \<and> init w)))"
 assume a1: "(enat i) \<le> nlength \<sigma>"
 assume a2: "f (ndropn i \<sigma>)"
 assume a3: " w (NNil (nfirst (ndropn i \<sigma>)))"    
 show "\<exists>i. enat i \<le> nlength (pifilt \<sigma> f) \<and> w (NNil (nfirst (ndropn i (pifilt \<sigma> f))))" 
  using a0 a1 a2 a3 by (metis PiAssocsem2 ndropn_nfirst)
next
 fix i
 assume a0: "g (pifilt \<sigma> (LIFT(f \<and> init w)))"
 assume a1: " (enat i) \<le> nlength \<sigma>"  
 assume a2: "f (ndropn i \<sigma>)"
 assume a3: " w (NNil (nfirst (ndropn i \<sigma>)))"  
 show " g (pifilt (pifilt \<sigma> f) (LIFT(init w))) " 
  using a0 a1 a2 a3 by (metis PiAssocsem2 ndropn_nfirst nnth_zero_ndropn pifilt_pifilt)
qed

lemma  PiAssoc:
 "\<turnstile> f \<Pi> ((init w) \<Pi>  g) =   (f \<and> (init w)) \<Pi> g"
 using PiAssocsem Valid_def by blast


subsubsection \<open>PiNotEqvDiamondAndNotPi\<close>

lemma PiNotEqvDiamondAndNotPisem:
 "\<sigma> \<Turnstile> f \<Pi> (\<not> g) = (\<diamond>f \<and> \<not>(f \<Pi> g))"
by (simp add: pi_d_def itl_defs ) blast

lemma PiNotEqvDiamondAndNotPi:
 "\<turnstile> f \<Pi> (\<not> g) = (\<diamond>f \<and> \<not>(f \<Pi> g))"
using PiNotEqvDiamondAndNotPisem Valid_def by blast


subsubsection \<open>PiSChopDist\<close>

lemma PiSChopDistsema:
 assumes "(\<sigma> \<Turnstile> (init w) \<Pi> (g\<frown>h))"
 shows  " (\<sigma> \<Turnstile> ((init w) \<Pi> g)\<frown>((init w) \<and> ((init w) \<Pi> h)))"
proof -
 have 1: "(\<sigma> \<Turnstile> (init w) \<Pi> (g\<frown>h))" 
   using assms by auto
 have 2: "((\<exists>i\<le>nlength \<sigma>. (LIFT(init w)) (ndropn i \<sigma>)) \<and>
           ((pifilt \<sigma> (LIFT(init w))) \<Turnstile> g\<frown>h)
          )"
   using "1" by (simp add: pi_d_def)
 have 3: "(\<exists>i\<le>nlength \<sigma>. (LIFT(init w)) (ndropn i \<sigma>))"  
   using "2" by auto
 have 4: "((pifilt \<sigma> (LIFT(init w))) \<Turnstile> g\<frown>h)"
   using "2" by auto 
 have 5: "nfilter (\<lambda>y. w ((NNil y))) \<sigma> \<Turnstile> g\<frown>h"
   using pifilt_init by (metis "4")
 have 6: "(\<exists>n. (enat n)\<le>nlength (nfilter (\<lambda>y. w (NNil y)) \<sigma>) \<and>
            g (ntaken n (nfilter (\<lambda>y. w (NNil y)) \<sigma>)) \<and> 
            h (ndropn n (nfilter (\<lambda>y. w (NNil y)) \<sigma>)))   "
   using "5" by (simp add: itl_defs) 
 obtain n where 7: "(enat n)\<le>nlength (nfilter (\<lambda>y. w (NNil y)) \<sigma>) \<and>
                          g (ntaken n (nfilter (\<lambda>y. w (NNil y)) \<sigma>)) \<and> 
                          h (ndropn n (nfilter (\<lambda>y. w (NNil y)) \<sigma>))"
   using 6 by auto
 have 8: "\<exists>i\<le>nlength \<sigma>. w (NNil (nnth \<sigma> i))"
   using "3" using init_defs PiStatesem assms by blast
 have 9: "\<exists> x \<in> nset \<sigma>. w (NNil x)"
   using "8" by (simp add: pifiltinit_help)
 have 10: "(ntaken n (nfilter (\<lambda>y. w (NNil y)) \<sigma>)) = 
           (nfilter (\<lambda>y. w (NNil y)) (ntaken ((nnth (nkfilter (\<lambda>y. w (NNil y)) 0 \<sigma>) n) ) \<sigma>) )"
   by (simp add: "7" "9" nfilter_nkfilter_ntaken_1 nkfilter_nlength)
 have 11: "(ndropn n (nfilter (\<lambda>y. w (NNil y)) \<sigma>)) =
            (nfilter (\<lambda>y. w (NNil y)) (ndropn ((nnth (nkfilter (\<lambda>y. w (NNil y)) 0 \<sigma>) n) ) \<sigma>) )"
   by (simp add: "7" "9" nfilter_nkfilter_ndropn_1 nkfilter_nlength)
 have 12: "g (nfilter (\<lambda>y. w (NNil y)) (ntaken ((nnth (nkfilter (\<lambda>y. w (NNil y)) 0 \<sigma>) n) ) \<sigma>) )"
   using "10" "7" by auto
 have 13: "h (nfilter (\<lambda>y. w (NNil y)) (ndropn ((nnth (nkfilter (\<lambda>y. w (NNil y)) 0 \<sigma> ) n) ) \<sigma>) )" 
   using "11" "7" by auto
 have 14: "((nnth (nkfilter (\<lambda>y. w (NNil y)) 0 \<sigma>) n) ) \<le> nlength \<sigma>" 
   using 7 9 nkfilter_upperbound[of \<sigma> "(\<lambda>y. w (NNil y))" _ 0]  
   by (metis gen_nlength_def nkfilter_nlength nlength_code)
 have 15: "w (NNil (nnth (ndropn ((nnth (nkfilter (\<lambda>y. w (NNil y)) 0 \<sigma>) n) ) \<sigma>) 0))"
   using 7 9 nkfilter_nmap_nfilter[of \<sigma> "\<lambda>x. w (NNil x)"]  nkfilter_nnth_aa[of \<sigma> "\<lambda>x. w (NNil x)"]
   by simp 
     (metis (mono_tags, lifting) "7"  nkfilter_nlength nnth_nmap )
 have 16: "(\<exists>i. (enat i)\<le>nlength (ntaken  ((nnth (nkfilter (\<lambda>y. w (NNil y)) 0 \<sigma>) n) ) \<sigma>) \<and> 
                w (NNil (nnth (ndropn i (ntaken  ((nnth (nkfilter (\<lambda>y. w (NNil y)) 0 \<sigma>) n) ) \<sigma>)) 0)))" 
   using "14" "15"
   by (metis le_cases min.orderE ndropn_0 ndropn_nfirst ntaken_nlength ntaken_nnth) 
 have 17: "(\<exists>i. (enat i)\<le>nlength (ndropn ((nnth (nkfilter (\<lambda>y. w (NNil y)) 0 \<sigma>) n) ) \<sigma>) \<and>
             w (NNil (nnth (ndropn (i + ((nnth (nkfilter (\<lambda>y. w (NNil y)) 0 \<sigma>) n) )) \<sigma>) 0)))"
   using "15" by (metis add.left_neutral zero_enat_def zero_le)
 have 18: "(\<exists>n. (enat n)\<le>nlength \<sigma> \<and>
          (\<exists>i. (enat i) \<le>nlength (ntaken n \<sigma>) \<and> w (NNil (nnth (ndropn i (ntaken n \<sigma>)) 0)) \<and>
          g (nfilter (\<lambda>y. w ((NNil y))) (ntaken n \<sigma>)) \<and>
          w (NNil (nnth (ndropn n \<sigma>) 0)) \<and>
          (\<exists>i. (enat i) \<le>nlength (ndropn n \<sigma>) \<and> w (NNil (nnth (ndropn (i + n) \<sigma>) 0))) \<and> 
          h (nfilter (\<lambda>y. w ((NNil y))) (ndropn n \<sigma>)) ))" 
   using "12" "13" "14" "15" "16" "17" by blast
 have 181: "(\<exists>n. (enat n)\<le>nlength \<sigma> \<and> (\<exists> i. (enat i)\<le> nlength (ntaken n \<sigma>) \<and> w (NNil (nnth (ntaken n \<sigma>) i)) )  )"
   using "18" by auto
 have 182: "(\<exists>n. (enat n)\<le>nlength \<sigma> \<and> (\<exists>i. (enat i)\<le> nlength (ndropn n \<sigma>) \<and>  w (NNil (nnth \<sigma> (i + n))) )  )" 
   using "18" by auto
 have 19: "(\<exists>n. (enat n)\<le>nlength \<sigma> \<and>
         (\<exists>i. (enat i)\<le>nlength (ntaken n \<sigma>) \<and> w (NNil (nnth (ndropn i (ntaken n \<sigma>)) 0)) \<and>
        g (pifilt (ntaken n \<sigma>) (LIFT(init w)) ) \<and>
         w (NNil (nnth (ndropn n \<sigma>) 0)) \<and>
         (\<exists>i. (enat i)\<le>nlength (ndropn n \<sigma>) \<and> w (NNil (nnth (ndropn (i + n) \<sigma>) 0)) \<and> 
         h (pifilt (ndropn n \<sigma>) (LIFT(init w))))))"    
   using 18 pifilt_init[of _ "w"] by auto  
 have 20: "((\<exists>n. (enat n)\<le>nlength \<sigma> \<and>
            (\<exists>i. (enat i)\<le>nlength (ntaken n \<sigma>) \<and> (LIFT(init w)) (ndropn i (ntaken n \<sigma>))) \<and>
              g (pifilt (ntaken n \<sigma>) (LIFT(init w))) \<and>
              (LIFT(init w)) (ndropn n \<sigma>) \<and> 
             (\<exists>i. (enat i)\<le>nlength (ndropn n \<sigma>) \<and> (LIFT(init w)) (ndropn (i + n) \<sigma>)) 
           \<and> h (pifilt (ndropn n \<sigma>) (LIFT(init w)))))" 
   using 19 
   by (metis  init_defs ndropn_nfirst nnth_zero_ndropn ntaken_0)
 have 21: "(\<sigma> \<Turnstile> ((init w) \<Pi> g)\<frown>((init w) \<and> ((init w) \<Pi> h)))"
   using "20" by (simp add: add.commute itl_defs ndropn_ndropn pi_d_def)
 show ?thesis by (simp add: "21")
qed
          
lemma PiSChopDistsemb:
 assumes " (\<sigma> \<Turnstile> ((init w) \<Pi> g)\<frown>((init w) \<and> ((init w) \<Pi> h)))"
 shows  "(\<sigma> \<Turnstile> (init w) \<Pi> (g\<frown>h))"
proof -  
 have 1: "(\<sigma> \<Turnstile> ((init w) \<Pi> g)\<frown>((init w) \<and> ((init w) \<Pi> h)))" 
   using assms by auto
 have 2: "(\<exists>n. (enat n) \<le> nlength \<sigma> \<and> 
            ((ntaken n \<sigma>) \<Turnstile> ((init w) \<Pi> g)) \<and>
            ((ndropn n \<sigma>) \<Turnstile> ((init w) \<and> ((init w) \<Pi> h)))) " 
   using assms schop_defs by blast
 obtain n where 3: "(enat n) \<le> nlength \<sigma> \<and> ((ntaken n \<sigma>) \<Turnstile> ((init w) \<Pi> g)) \<and>
                    ((ndropn n \<sigma>) \<Turnstile> ((init w) \<and> ((init w) \<Pi> h)))" 
   using 2 by auto
 have 4: "((\<exists>i. (enat i)\<le>nlength (ntaken n \<sigma>) \<and> (LIFT(init w)) (ndropn i (ntaken n \<sigma>))) \<and>
           ((pifilt (ntaken n \<sigma>) (LIFT(init w))) \<Turnstile> g)
          )" 
   by (meson "3" pi_d_def)
 have 5: "(\<exists>i. (enat i)\<le>nlength (ntaken n \<sigma>) \<and> (LIFT(init w)) (ndropn i (ntaken n \<sigma>)))" 
   using "4" by auto
 have 6: "g (pifilt (ntaken n \<sigma>) (LIFT(init w)))" 
   using "4" by auto
 have 7: "g (nfilter (\<lambda>y. w ((NNil y))) (ntaken n \<sigma>)) "
   using "5" "6" pifilt_init by (metis)
 have 8: "((\<exists>i. (enat i)\<le>nlength (ndropn n \<sigma>) \<and> (LIFT(init w)) (ndropn i (ndropn n \<sigma>))) \<and>
           ((pifilt (ndropn n \<sigma>) (LIFT(init w))) \<Turnstile> h)
          )" 
   by (metis "3" intensional_rews(3) pi_d_def)
 have 9: "(\<exists>i. (enat i)\<le>nlength (ndropn n \<sigma>) \<and> (LIFT(init w)) (ndropn i (ndropn n \<sigma>)))"
   using "8" by auto
 have 10: "h (pifilt (ndropn n \<sigma>) (LIFT(init w)))"  
   using "8" by auto
 have 11: "h (nfilter (\<lambda>y. w ((NNil y))) (ndropn n \<sigma>))  "
   using "10" "9" pifilt_init by metis 
 have 12: "(\<lambda>y. w ((NNil y))) (nlast (ntaken n \<sigma>))"
   by (metis "3" init_defs intensional_rews(3) ndropn_nfirst ntaken_0 ntaken_nlast)
 have 13: "\<exists> x \<in> nset \<sigma>. (\<lambda>y. w ((NNil y))) x"
   using "12" "3" by (metis in_nset_conv_nnth ntaken_nlast)
 have 14: "\<exists> x \<in> nset (ntaken n \<sigma>) . (\<lambda>y. w ((NNil y))) x" 
   using "12" using nfinite_ntaken nset_nlast by blast
 have 15: "\<exists> x \<in> nset (ndropn n \<sigma>) . (\<lambda>y. w ((NNil y))) x"
   by (metis "12" "3" dual_order.order_iff_strict ndropn_Suc_conv_ndropn ndropn_nlast 
       nellist.set_intros(1) nellist.set_intros(2) nfinite_ntaken ntaken_all ntaken_nlast the_enat.simps) 
 have 16: "(nfilter (\<lambda>y. w ((NNil y))) (ntaken n \<sigma>)) =
           ntaken (the_enat (nlength (nfilter (\<lambda>y. w ((NNil y))) (ntaken n \<sigma>)))) 
                  (nfilter (\<lambda>y. w ((NNil y))) \<sigma>)" 
   using "12" "13" "14" "15" "3" nfilter_chop1_ntaken[of "n" "\<sigma>" "(\<lambda>y. w (NNil y))"] by auto
 have 17: "(nfilter (\<lambda>y. w ((NNil y))) (ndropn n \<sigma>)) =
           ndropn (the_enat (nlength (nfilter (\<lambda>y. w ((NNil y))) (ntaken n \<sigma>)))) 
                  (nfilter (\<lambda>y. w ((NNil y))) \<sigma>)" 
   using "12" "13" "14" "15" "3" nfilter_chop1_ndropn[of "n" "\<sigma>" "(\<lambda>y. w (NNil y))"] by auto
 have 18: "\<exists>n. (enat n)\<le>nlength (nfilter (\<lambda>y. w (NNil y)) \<sigma>) \<and>
            g (ntaken n (nfilter (\<lambda>y. w (NNil y)) \<sigma>)) \<and> 
            h (ndropn n (nfilter (\<lambda>y. w (NNil y)) \<sigma>))"
   by (metis "11" "16" "17" "7" enat_the_enat infinity_ileE le_cases ntaken_all)
 have 19: "nfilter (\<lambda>y. w ((NNil y))) \<sigma> \<Turnstile> g\<frown>h"  
   by (simp add: "18" schop_defs)
 have 20: "((pifilt \<sigma> (LIFT(init w))) \<Turnstile> g\<frown>h)" 
   by (metis "19"  pifilt_init)
 have 21: "(\<exists>i. (enat i)\<le>nlength \<sigma> \<and> (LIFT(init w)) (ndropn i \<sigma>))" 
   using "3" by auto
 show ?thesis by (simp add: "20" "21" pi_d_def)
qed
 
lemma PiSChopDistsem:
 "\<sigma> \<Turnstile> (init w) \<Pi> (g\<frown>h) = ((init w) \<Pi> g)\<frown>((init w) \<and> ((init w) \<Pi> h))"
using PiSChopDistsema PiSChopDistsemb unl_lift2 by blast

lemma PiSChopDist:
 "\<turnstile> (init w) \<Pi> (g\<frown>h) = ((init w) \<Pi> g)\<frown>((init w) \<and> ((init w) \<Pi> h))"
using PiSChopDistsem Valid_def by blast

subsubsection \<open>PiProp\<close>

lemma Pistate:
 "(\<sigma> \<Turnstile> (init w) \<Pi> f) = 
  ((\<exists> x \<in> nset \<sigma>. w (NNil x)) \<and>  ( (nfilter (\<lambda>y. w (NNil y)) \<sigma>) \<Turnstile> f) )"
proof -
 have 1: "(\<sigma> \<Turnstile> (init w) \<Pi> f) =
          ( (\<exists> i\<le> nlength \<sigma>.  w (NNil (nnth \<sigma> i))) \<and> ((pifilt \<sigma> (LIFT(init w))) \<Turnstile> f) )"
   by (auto simp add: pi_d_def init_defs ndropn_nfirst)
 have 2: "(\<exists>i\<le>nlength \<sigma>. (LIFT(init w)) (ndropn i \<sigma>)) =
          (\<exists> i\<le> nlength \<sigma>.  w (NNil (nnth \<sigma> i)))"
   by (auto simp add: init_defs ndropn_nfirst) 
 have 3: "(\<exists> i\<le> nlength \<sigma>.  w (NNil (nnth \<sigma> i))) \<longrightarrow>
           (pifilt \<sigma> (LIFT(init w))) = (nfilter (\<lambda>y. w (NNil y)) \<sigma>)" 
   using pifilt_init using "2" by blast
 have 4: "(\<exists> i. (enat i)\<le> nlength \<sigma> \<and>  w (NNil (nnth \<sigma> i))) =(\<exists> x \<in> nset \<sigma>. w (NNil x)) "
   using in_nset_conv_nnth by force
 show ?thesis 
 using "1" "3" "4" by auto
qed

lemma PiPropsem1a:
 "(\<sigma> \<Turnstile> f \<Pi> $p) = 
   ((\<exists>i. (enat i)\<le>nlength \<sigma> \<and> f (ndropn i \<sigma>)) \<and> p (nnth \<sigma> (nnth (nkfilter f 0 (ndropns \<sigma>)) 0)) )"
using sfxfilt_pifilt_nnth_ndropn[of \<sigma> f] 
by (simp add: pi_d_def current_val_d_def pifilt_def )
   (metis ndropn_0 ndropn_nfirst nnth_nmap zero_enat_def zero_order(1))

lemma PiPropsem2a:
 " (\<sigma> \<Turnstile> (\<not> f) \<U> (f \<and> $p)) =
   (\<exists>k\<le>nlength \<sigma>. f (ndropn k \<sigma>) \<and> p (nnth \<sigma> k) \<and> (\<forall>j<k. \<not> f (ndropn j \<sigma>)))"
by (simp add: until_d_def current_val_d_def ndropn_nfirst)

lemma PiPropsem3a:
 assumes "(\<sigma> \<Turnstile> f \<Pi> $p)"
 shows   " (\<sigma> \<Turnstile> (\<not> f) \<U> (f \<and> $p))"
proof -
 have 1: "((\<exists>i. (enat i)\<le>nlength \<sigma> \<and> f (ndropn i \<sigma>)) \<and> 
           p (nnth \<sigma> (nnth (nkfilter f 0 (ndropns \<sigma>)) 0) ) )"
   using assms  PiPropsem1a by auto
 have 2: "\<exists> x \<in> nset(ndropns \<sigma>). f x" 
   using "1" by (simp add: sfxfilter_help)
 have 3: "\<forall> x \<in> nset(nkfilter f 0 (ndropns \<sigma>)). f  (nnth (ndropns \<sigma>) x)"
   using "2" nkfilter_holds by fastforce
 have 31: "(nnth (nkfilter f 0 (ndropns \<sigma>)) 0) \<le> nlength \<sigma>"
   by (metis "1" "2" dual_order.strict_trans2 enat_ord_simps(2) leI ndropns_nnth nkfilter_not_before) 
 have 4: "f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) 0)  \<sigma>) "
   by (metis "3" "31" in_nset_ndropns in_nset_ndropns_nhd ndropn_0 ndropns_nnth zero_enat_def zero_le) 
 have 5: "(\<forall>j< (nnth (nkfilter f 0 (ndropns \<sigma>)) 0). \<not> f (ndropn j \<sigma>)) " 
   using nkfilter_not_before[of "ndropns \<sigma>" "f" ] 2
   by (simp add: "31" less_imp_le ndropns_nnth order_less_le_subst2)
 have 6: "(\<exists>k\<le>nlength \<sigma>. f (ndropn k \<sigma>) \<and> p (nnth \<sigma> k) \<and> (\<forall>j<k. \<not> f (ndropn j \<sigma>)))"
   using "1" "31" "4" "5" by blast
 show ?thesis using 6 PiPropsem2a by metis
qed

lemma PiPropsem3b:
 assumes "(\<sigma> \<Turnstile> (\<not> f) \<U> (f \<and> $p))"
 shows   " (\<sigma> \<Turnstile> f \<Pi> $p)"
proof -
 have 1: "(\<exists>k\<le>nlength \<sigma>. f (ndropn k \<sigma>) \<and> p (nnth \<sigma> k) \<and> (\<forall>j<k. \<not> f (ndropn j \<sigma>)))"
   using assms  PiPropsem2a by auto
 obtain k where 2: "k\<le>nlength \<sigma> \<and> f (ndropn k \<sigma>) \<and> p (nnth \<sigma> k) \<and> (\<forall>j<k. \<not> f (ndropn j \<sigma>))"
   using 1 by auto
 have 3: "(\<exists>i. (enat i)\<le>nlength \<sigma> \<and> f (ndropn i \<sigma>))"
   using "2" by blast  
 have 31: "\<exists>x \<in> nset(ndropns \<sigma>). f x"
   using "2" in_nset_ndropns by auto
 have 331: "(nnth (nkfilter f 0 (ndropns \<sigma>)) 0) \<le> nlength \<sigma>"
   by (metis "31" gen_nlength_def ndropns_nlength nkfilter_upperbound nlength_code zero_enat_def zero_le)
 have 32: "f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) 0) \<sigma>)"
   using nkfilter_holds[of "ndropns \<sigma>" "f" "0"]  nkfilter_not_before[of "ndropns \<sigma>" "f"]
   by (metis "2" "31" ndropns_nfilter_nnth sfxfilt_def sfxfilt_pifilt_nnth_ndropn zero_enat_def zero_le)
 have 4: "p (nnth \<sigma> (nnth (nkfilter f 0 (ndropns \<sigma>)) 0) )"
   by (metis "2" "31" "32" linorder_neqE_nat ndropns_nnth nkfilter_not_before)
 show ?thesis using 4 3 by (simp add: PiPropsem1a)
qed

lemma PiPropsema:
 "\<sigma> \<Turnstile> f \<Pi> $p = (\<not> f) \<U> (f \<and> $p)"
using PiPropsem3a PiPropsem3b unl_lift2 by blast

lemma PiProp:
 "\<turnstile> f \<Pi> $p = (\<not> f) \<U> (f \<and> $p)"
using PiPropsema Valid_def by blast

subsubsection \<open>PiNext\<close>

lemma PiNextsem1:
 "(\<sigma> \<Turnstile> f \<Pi> (\<circle> g)) =
  ((\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>)) \<and>
    0 < nlength (nfilter f (ndropns \<sigma>)) \<and>
    g (ndropn (Suc 0) (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))))"
using zero_enat_def by (simp add: pi_d_def itl_defs pifilt_def sfxfilt_def )

lemma PiNextsem2:
 "(\<sigma> \<Turnstile> (\<not> f) \<U> (f \<and> \<circle>(f \<Pi> g))) =
  (\<exists>k\<le>nlength \<sigma>.
       f (ndropn k \<sigma>) \<and>
       k < nlength \<sigma> \<and>
       (\<exists>i\<le>nlength \<sigma> - Suc k. f (ndropn (Suc (i + k)) \<sigma>)) \<and>
       g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn (Suc k) \<sigma>)))) \<and> (\<forall>j<k. \<not> f (ndropn j \<sigma>)))"
by (simp add: until_d_def itl_defs pi_d_def pifilt_def sfxfilt_def ndropn_ndropn add.commute)
   (metis add.right_neutral antisym_conv2 enat.simps(3) enat_add_sub_same less_eqE zero_enat_def)
  
lemma PiNextsem3:
 assumes "(\<sigma> \<Turnstile> f \<Pi> (\<circle> g))" 
 shows    "(\<sigma> \<Turnstile> (\<not> f) \<U> (f \<and> \<circle>(f \<Pi> g)))" 
proof -
 have 1: "((\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>)) \<and> 0 < nlength (nfilter f (ndropns \<sigma>)) \<and>
              g (ndropn (Suc 0) (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))))" 
   using assms by (simp add: PiNextsem1)
 have 2: "\<exists> x \<in> nset(ndropns \<sigma>). f x" 
   using "1" by (simp add: sfxfilter_help)
 have 3: "\<forall> x \<in> nset(nkfilter f 0 (ndropns \<sigma>)). f  (nnth (ndropns \<sigma>) x)"
   using "2" nkfilter_holds by fastforce
 have 31: "(nnth (nkfilter f 0 (ndropns \<sigma>)) 0) \<le> nlength \<sigma>"
   by (metis "2" gen_nlength_def i0_lb ndropns_nlength nkfilter_upperbound nlength_code zero_enat_def)
 have 4: "f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) 0)  \<sigma>) "
   by (metis "3" "31" i0_lb in_nset_conv_nnth ndropns_nnth zero_enat_def)
 have 41: "(nnth (nkfilter f 0 (ndropns \<sigma>)) 1) \<in> nset(nkfilter f 0 (ndropns \<sigma>))"
   by (metis "1" "2" One_nat_def eSuc_enat ileI1 in_nset_conv_nnth nkfilter_nlength zero_enat_def)
 have 42: "  (nnth (nkfilter f 0 (ndropns \<sigma>)) 1) \<le> nlength (ndropns \<sigma>) "
   by (metis "2" "41" gen_nlength_def in_nset_conv_nnth nkfilter_upperbound nlength_code)
 have 5: "f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) 1)  \<sigma>)" 
   using "3" "41" "42" by (metis ndropns_nlength ndropns_nnth)
 have 6: "(nnth (nkfilter f 0 (ndropns \<sigma>)) 0) < (nnth (nkfilter f 0 (ndropns \<sigma>)) 1)" 
   by (metis "1" "2" One_nat_def ileI1 nidx_nkfilter_expand nkfilter_nlength one_eSuc one_enat_def)
 have 7: " Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) 0) \<le> nlength \<sigma>" 
   by (metis "42" "6" Suc_leI dual_order.trans enat_ord_simps(1) ndropns_nlength)
 have 8: "0 < nlength (nkfilter f 0 (ndropns \<sigma>))" 
   using "1" "2" nkfilter_nlength by fastforce
 have 9: "(nnth (nkfilter f 0 (ndropns \<sigma>)) 1) \<le> nlength \<sigma>" 
   by (metis "42" ndropns_nlength)    
 have 10: "(Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) 0)) \<le> (nnth (nkfilter f 0 (ndropns \<sigma>)) 1)"
   using "6" Suc_leI by blast 
 have 11: "(nnth (nkfilter f 0 (ndropns \<sigma>)) 1) = 
            (nnth (nkfilter f 0 (ndropns \<sigma>)) 1) - (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) 0)) +
              (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) 0)) " 
   using "10" by auto
 have 12: "(nnth (nkfilter f 0 (ndropns \<sigma>)) 1) - (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) 0)) \<le>
            nlength \<sigma> - Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) 0)"
   using "9" diff_le_mono by (metis enat_minus_mono1 idiff_enat_enat)
 have 13: "\<exists> i \<le> nlength \<sigma> - Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) 0). 
             (nnth (nkfilter f 0 (ndropns \<sigma>)) 1) = (Suc (i + (nnth (nkfilter f 0 (ndropns \<sigma>)) 0)))" 
   using 11 12 by auto
 have 14: "(\<exists>i\<le>nlength \<sigma> - Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) 0). 
            f (ndropn (Suc (i + (nnth (nkfilter f 0 (ndropns \<sigma>)) 0))) \<sigma>))" 
   using "13" "5" by auto 
 have 15: "(ndropn (Suc 0) (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))) =
           (nmap (\<lambda>s. nnth s 0) 
                (nfilter f (ndropns (ndropn (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) 0)) \<sigma>))))" 
   using 2 8 
   by (metis One_nat_def ileI1 nfilter_ndropns_nmap nkfilter_nlength one_eSuc one_enat_def)
 have 16: "g (nmap (\<lambda>s . nnth s 0) 
                  (nfilter f (ndropns (ndropn (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) 0)) \<sigma>))))  "
   using "1" "15" by auto    
 have 17: "\<forall> j< (nnth (nkfilter f 0 (ndropns \<sigma>)) 0). \<not> f (ndropn j \<sigma>)"
   using 2 31 nkfilter_not_before[of "(ndropns \<sigma>)" f]
   by (metis enat_ord_simps(1) less_imp_le ndropns_nnth order.trans)
 have 18: "(\<exists>k\<le>nlength \<sigma>.
              f (ndropn k \<sigma>) \<and>
              k < nlength \<sigma> \<and>
             (\<exists>i\<le>nlength \<sigma> - Suc k. f (ndropn (Suc (i + k)) \<sigma>)) \<and>
                  g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn (Suc k) \<sigma>)))) \<and>
                  (\<forall>j<k. \<not> f (ndropn j \<sigma>)))"
   using "14" "16" "17" "4" "7" Suc_ile_eq less_imp_le by blast
 show ?thesis using 18 by (simp add: PiNextsem2)
qed

lemma PiNextsem4:
 assumes "(\<sigma> \<Turnstile> (\<not> f) \<U> (f \<and> \<circle>(f \<Pi> g)))" 
 shows    "(\<sigma> \<Turnstile> f \<Pi> (\<circle> g))"
proof -
 have 1: "(\<exists>k\<le>nlength \<sigma>.
             f (ndropn k \<sigma>) \<and>
             k < nlength \<sigma> \<and>
             (\<exists>i\<le>nlength \<sigma> - Suc k. f (ndropn (Suc (i + k)) \<sigma>)) \<and>
                g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn (Suc k) \<sigma>)))) \<and>
               (\<forall>j<k. \<not> f (ndropn j \<sigma>)))"
   using assms   by (simp add: PiNextsem2)
 obtain k where 2: "k\<le>nlength \<sigma> \<and> f (ndropn k \<sigma>) \<and> k < nlength \<sigma> \<and> (\<exists>i\<le>nlength \<sigma> - Suc k. 
                    f (ndropn (Suc (i + k)) \<sigma>)) \<and>
       g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn (Suc k) \<sigma>)))) \<and> (\<forall>j<k. \<not> f (ndropn j \<sigma>))"
   using 1 by auto 
 have 3: "\<exists> x \<in> nset (ndropns \<sigma>). f x"
   using "1" using in_nset_ndropns by auto
 have 4: "\<forall> x \<in> nset(nkfilter f 0 (ndropns \<sigma>)). f  (nnth (ndropns \<sigma>) x)"   
   using "3" nkfilter_holds by fastforce
 have 41: "(nnth (nkfilter f 0 (ndropns \<sigma>)) 0) \<le> nlength \<sigma>"
   by (metis "3" gen_nlength_def le_cases le_zero_eq ndropns_nlength nkfilter_upperbound nlength_code
     zero_enat_def) 
 have 5: "f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) 0)  \<sigma>) "
   by (metis "4" "41" i0_lb in_nset_conv_nnth ndropns_nnth zero_enat_def)
 have 6: "0 < nlength (nfilter f (ndropns \<sigma>))"
   using 2 3 nfilter_nlength_zero_conv_a[of "(ndropns \<sigma>)" f]
   by simp
        (metis (no_types, lifting) add.commute eSuc_enat enat.simps(3) enat_add_sub_same 
         enat_less_enat_plusI2 ileI1 iless_Suc_eq less_add_Suc2 less_eqE ndropn_Suc_conv_ndropn 
         ndropn_nlength ndropns_nlength ndropns_nnth nlength_NCons)
 have 61: "(nnth (nkfilter f 0 (ndropns \<sigma>)) 1) \<in> nset(nkfilter f 0 (ndropns \<sigma>))"
   by (metis "3" "6" ileI1 in_nset_conv_nnth nkfilter_nlength one_eSuc one_enat_def) 
 have 62: "(nnth (nkfilter f 0 (ndropns \<sigma>)) 1) \<le> nlength (ndropns \<sigma>)"
   by (metis "3" "61" gen_nlength_def in_nset_conv_nnth nkfilter_upperbound nlength_code)
 have 7: "f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) 1)  \<sigma>)"
   using "4" "61" "62" by (metis ndropns_nlength ndropns_nnth)
 have 8: "(\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>))" 
   using "1" by blast
 have 9: "g (ndropn (Suc 0) (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>))))"
   using 2 3 5 6 nkfilter_not_before[of "(ndropns \<sigma>)" f] nfilter_ndropns_nmap[of \<sigma> f 0]
   by (metis One_nat_def ileI1 ndropns_nnth not_less_iff_gr_or_eq one_eSuc one_enat_def)  
 have 10: "((\<exists>i. (enat i) \<le>nlength \<sigma> \<and> f (ndropn i \<sigma>)) \<and>
            0 < nlength (nfilter f (ndropns \<sigma>)) \<and>
            g (ndropn (Suc 0) (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))))" 
   using "6" "8" "9" by blast
 show ?thesis using 10 
   by (simp add: PiNextsem1)
qed

lemma PiNextsem:
 " (\<sigma> \<Turnstile> f \<Pi> (\<circle> g) = (\<not> f) \<U> (f \<and> \<circle>(f \<Pi> g)))"
using PiNextsem3 PiNextsem4 unl_lift2 by blast

lemma PiNext:
 "\<turnstile> f \<Pi> (\<circle> g) = (\<not> f) \<U> (f \<and> \<circle>(f \<Pi> g))"
using PiNextsem Valid_def by blast


subsubsection \<open>PiUntil\<close>

lemma PiUntilDistsem1:
  "(\<sigma> \<Turnstile> f \<Pi> (g \<U> h)) =
   ((\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>)) \<and>
    (\<exists>k\<le>nlength (nfilter f (ndropns \<sigma>)).
        h (ndropn k (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))) \<and>
        (\<forall>j<k. g (ndropn j (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))))))"
by (simp add: pi_d_def pifilt_def sfxfilt_def until_d_def)

lemma PiUntilDistsem2:
 " (\<sigma> \<Turnstile> ( f \<Pi> g ) \<U> ( f \<Pi> h ) ) =
   (\<exists>k\<le>nlength \<sigma>.
       (\<exists>i\<le>nlength \<sigma> - k. f (ndropn (i + k) \<sigma>)) \<and>
       h (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn k \<sigma>)))) \<and>
       (\<forall>j<k. (\<exists>i\<le>nlength \<sigma> - j. f (ndropn (i + j) \<sigma>)) \<and> 
              g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn j \<sigma>))))))"
by (simp add: until_d_def pi_d_def pifilt_def sfxfilt_def ndropn_ndropn add.commute)

lemma cover:
 assumes "(\<exists>i<k. ff(i) <(j::nat) \<and> j\<le> ff(Suc i))"
         "(\<forall>i<k. ff(i)<ff(Suc i))"
 shows   "ff(0)<j \<and> j \<le>ff(k)  "
using assms 
proof (induct k arbitrary:j)
case 0
then show ?case by simp
next
case (Suc k)
then show ?case 
  proof -
   have 1: "(\<exists>i<k. ff(i) <(j::nat) \<and> j\<le> ff(Suc i)) \<or> (ff (k) <j \<and> j\<le>ff(Suc k))"
     using Suc.prems(1) less_SucE by blast
   have 2: "(\<forall>i<k. ff(i)<ff(Suc i))" 
     using Suc.prems(2) by auto
   have 3: "ff k < ff (Suc k)"
     by (simp add: Suc.prems(2)) 
   have 4: "(ff(0)<j \<and> j \<le>ff(k)) \<or>  (ff (k) <j \<and> j\<le>ff(Suc k))"
     using "1" "2" Suc.hyps by blast 
   have 41: "ff(0)<j"
     using "2" "4" Suc.hyps order_refl by force 
   have 42: "j \<le>ff(Suc k)"
     using "3" "4" by linarith 
   have 5: "ff(0)<j \<and> j \<le>ff(Suc k) " 
     by (simp add: "41" "42")
   show ?thesis 
     by (simp add: "5")
  qed    
qed

lemma cover_a:
 assumes "(\<forall> j.  
          (j\<le> ff 0) \<or> (\<exists>i<k. ff(i) <(j::nat) \<and> j\<le> ff(Suc i)) 
            \<longrightarrow> gg j)
          "
         "(\<forall>i<k. ff(i)<ff(Suc i))"
 shows   "(\<forall> j< ff k. gg j)  "
proof -
 have 1: "(\<forall> j< ff 0. gg j)" 
   by (simp add: assms(1))
 have 2: "(\<forall> j. ff 0 < j \<and> j\<le> ff k \<longrightarrow> gg j)" 
   proof 
     fix j
     show "ff 0 < j \<and> j \<le> ff k \<longrightarrow> gg j"  
      using assms 
       proof (induct k arbitrary: j)
       case 0
       then show ?case by simp
       next
       case (Suc k)
       then show ?case 
         proof -
          have 21: "(\<forall> j. (j\<le> ff 0) \<or> (\<exists>i<k. ff(i) <(j::nat) \<and> j\<le> ff(Suc i))
                     \<or> (ff k < j \<and> j \<le> ff (Suc k)    ) \<longrightarrow> gg j)" 
             using Suc.prems(1) less_SucI by blast
          have 22: "(\<forall>i<k. ff(i)<ff(Suc i))" 
            using Suc.prems(2) by auto
          have 23: "ff k < ff (Suc k)"
            by (simp add: Suc.prems(2))
          have 24: "(\<forall> j.  
                    (j\<le> ff 0) \<or> (ff 0 <j \<and> j \<le> ff k) 
                     \<or> (ff k < j \<and> j \<le> ff (Suc k)    ) 
                   \<longrightarrow> gg j)"  
            using "21" "22" Suc.hyps by blast
          have 25: "ff 0 < j \<and> j \<le> ff (Suc k) \<longrightarrow> gg j"
            using "24" not_less by blast 
          show ?thesis using "25" by blast
         qed
       qed
    qed
  show ?thesis 
  by (metis "2" assms(1) less_or_eq_imp_le linorder_neqE_nat)
qed

lemma PiUntilDistsem3:
 assumes "(\<sigma> \<Turnstile> f \<Pi> (g \<U> h))"
 shows  " (\<sigma> \<Turnstile> ( f \<Pi> g ) \<U> ( f \<Pi> h ) )"
proof -
  have 1: "((\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>)) \<and>
             (\<exists>k\<le>nlength (nfilter f (ndropns \<sigma>)).
               h (ndropn k (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))) \<and>
               (\<forall>j<k. g (ndropn j (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))))))"
    using assms PiUntilDistsem1 by blast
  have 2: "\<exists> x \<in> nset(ndropns \<sigma>). f x"
    using "1" by (simp add: sfxfilter_help)
  have 3: "\<forall> x \<in> nset(nkfilter f 0 (ndropns \<sigma>)). f  (nnth (ndropns \<sigma>) x)"
    using "2" nkfilter_holds by fastforce
  have 4: "(\<exists>k\<le>nlength (nfilter f (ndropns \<sigma>)).
              h (ndropn k (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))) \<and>
             (\<forall>j<k. g (ndropn j (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>))))))"
    using 1 by auto
  obtain k where 5: "k\<le>nlength (nfilter f (ndropns \<sigma>)) \<and> 
                     h (ndropn k (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))) \<and>
                     (\<forall>j<k. g (ndropn j (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))))"
    using 4 by auto
  have 51: "(nnth (nkfilter f 0 (ndropns \<sigma>)) k) \<le> nlength \<sigma>"
    by (metis (mono_tags, lifting) "2" "5" gen_nlength_def ndropns_nlength nkfilter_nlength 
        nkfilter_upperbound nlength_code) 
  have 6: "f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) k)  \<sigma>) "   
    using 2 3 5
    by (metis "1" ndropns_nfilter_nnth nlength_nmap pifilt_def sfxfilt_def sfxfilt_pifilt_nnth_ndropn) 
  have 7: "k=0 \<longrightarrow> 
          (\<exists>i. (enat i)\<le>nlength \<sigma> - k \<and> f (ndropn (i + k) \<sigma>)) \<and>
             h (nmap (\<lambda>s. nnth s 0) (nfilter f  (ndropns (ndropn k \<sigma>)))) \<and>
             (\<forall>j<k. (\<exists>i\<le>nlength \<sigma> - j. f (ndropn (i + j) \<sigma>)) \<and> 
               g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn j \<sigma>)))))"
    using "1" "5" by auto
  have 71:" k>0 \<longrightarrow> (nnth (nkfilter f 0 (ndropns \<sigma>)) (k -1)) \<in> nset(nkfilter f 0 (ndropns \<sigma>))"
    by (metis "2" "5" One_nat_def Suc_ile_eq Suc_pred in_nset_conv_nnth less_imp_le nkfilter_nlength)
  have 72: "k>0 \<longrightarrow> enat (k - 1) \<le> nlength (nkfilter f 0 (ndropns \<sigma>))" 
     by (metis "2" "5" One_nat_def Suc_ile_eq Suc_pred nkfilter_nlength order_less_imp_le)
  have 8: "k>0 \<longrightarrow> f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) (k -1))  \<sigma>)"
    using 3 2 71 72 nkfilter_upperbound[of "(ndropns \<sigma>)" f "k-1" 0]  
    by (metis add_0 ndropns_nlength ndropns_nnth zero_enat_def)
  have 9: "k>0 \<longrightarrow> (nnth (nkfilter f 0 (ndropns \<sigma>)) (k -1)) < (nnth (nkfilter f 0 (ndropns \<sigma>)) k)"    
    by (metis "2" "5" One_nat_def Suc_pred nkfilter_mono nkfilter_nlength)
  have 10: "k>0 \<longrightarrow>  (nnth (nkfilter f 0 (ndropns \<sigma>)) (k -1)) \<le> nlength \<sigma>"
    by (metis "2" "71" gen_nlength_def in_nset_conv_nnth ndropns_nlength nkfilter_upperbound 
        nlength_code) 
  have 11: "k>0 \<longrightarrow> (nnth (nkfilter f 0 (ndropns \<sigma>)) k) \<le> nlength \<sigma>" 
    using nkfilter_upperbound[of "ndropns \<sigma>" "f" "k" "0"] 
        using "51" by blast
  have 12: "k>0 \<longrightarrow> 
             h (nmap (\<lambda>s. nnth s 0) 
                    (nfilter f (ndropns (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) k) \<sigma>))))" 
    using nfilter_nkfilter_ndropn[of "f" "(ndropns \<sigma>)" "k"] 
       ndropns_nfilter_ndropn_a[of k f \<sigma>]
    by (metis "2" "5" ndropn_nmap )  
  have 121: "k>0 \<longrightarrow> 
             h (nmap (\<lambda>s. nnth s 0) 
                    (nfilter f (ndropns (ndropn (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) (k-1))) \<sigma>))))" 
    by (metis "2" "5" One_nat_def Suc_pred nfilter_ndropns_nmap)
  have 13: "k>0 \<longrightarrow> (\<exists>i\<le>nlength \<sigma> - (nnth (nkfilter f 0 (ndropns \<sigma>)) k). 
                      f (ndropn (i + (nnth (nkfilter f 0 (ndropns \<sigma>)) k)) \<sigma>))"
    using "6" by (metis add.left_neutral zero_enat_def zero_le)  
  have 130: " k>0 \<longrightarrow> (\<exists> i. i+ (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) (k-1))) \<le>nlength \<sigma>   \<and>
                (ndropn (i+(Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) (k-1)))) \<sigma>) = 
                            (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) k) \<sigma>))   "
    using "9" by (metis "11" Suc_leI diff_add)
  have 131: "k>0 \<longrightarrow> (\<exists>i. i+ (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) (k-1)))\<le>nlength \<sigma>  \<and>
                       f (ndropn (i + (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) (k-1)))) \<sigma>))"
    using "10" "6" "9" 11 130 by auto
  have 132: "k>0 \<longrightarrow> (\<exists>i\<le>nlength \<sigma> - (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) (k-1))).  
                       f (ndropn (i + (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) (k-1)))) \<sigma>))" 
    using 131 
    by (metis add_diff_cancel_right' enat_minus_mono1 idiff_enat_enat)
  have 14: "k>0 \<longrightarrow> (\<forall> j< (nnth (nkfilter f 0 (ndropns \<sigma>)) k). 
                      (\<exists>i. i+j\<le> nlength \<sigma> \<and> f (ndropn (i + j) \<sigma>)))" 
    using 131 by (metis "11" "6" diff_add less_imp_le plus_enat_simps(1))
  have 141: "k>0 \<longrightarrow> (\<forall> j< (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) (k-1))). 
                        (\<exists>i. i+j\<le>nlength \<sigma> \<and> f (ndropn (i + j) \<sigma>)))" 
    using "14" "9" by auto
  have 142: "k>0 \<longrightarrow> (\<forall> j< (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) (k-1))). 
                        (\<exists>i\<le>nlength \<sigma> -j.   f (ndropn (i + j) \<sigma>)))"
    using 141 by (metis add.commute enat.simps(3) enat_add_sub_same enat_minus_mono1)
  have 15: "(\<forall>j<k. g (ndropn j (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))))" 
    using "5" by blast
  have 151: "(\<forall>j<k. g (nmap (\<lambda>s. nnth s 0) (ndropn j  (nfilter f (ndropns \<sigma>)))))" 
    by (metis "15" ndropn_nmap)
  have 152: "(\<forall>j<k. (ndropn j  (nfilter f (ndropns \<sigma>))) = 
                    (nfilter f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) j) (ndropns \<sigma>))))"
    using nfilter_nkfilter_ndropn_1[of "ndropns \<sigma>" f ] 
    by (simp add: "2" "5" less_imp_le nkfilter_nlength order_less_le_subst2) 
  have 16: "(\<forall>j<k. g  (nmap (\<lambda>s. nnth s 0) 
                           (nfilter f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) j) (ndropns \<sigma>))))) "
    using 151 152 nfilter_nkfilter_ndropn by simp
  have 1610: " (nnth (nkfilter f 0 (ndropns \<sigma>)) k) \<le> nlength(ndropns \<sigma>)"
    by (simp add: "51" ndropns_nlength) 
  have 1611: "(\<forall>j<k.  (nnth (nkfilter f 0 (ndropns \<sigma>)) j) <   (nnth (nkfilter f 0 (ndropns \<sigma>)) k)) "
    by (simp add: "2" "5" nidx_nkfilter_gr nkfilter_nlength)
  have 1612: "(\<forall>j<k.  (nnth (nkfilter f 0 (ndropns \<sigma>)) j) \<le> nlength(ndropns \<sigma>))"
    using "1610" "1611" by (meson enat_ord_simps(2) less_imp_le order_less_le_subst2)
  have 161: "(\<forall>j<k. ( (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) j) (ndropns \<sigma>))) =
                    ( (ndropns (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) j) \<sigma>))))"
    using  "1612" by (simp add: ndropn_ndropns)
  have 17: "(\<forall>j<k. g  (nmap (\<lambda>s. nnth s 0) 
                      (nfilter f (ndropns (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) j) \<sigma>)))))" 
    using "16" "161" by auto
  have 18: "k>0 \<longrightarrow>  
            g  (nmap (\<lambda>s. nnth s 0) 
               (nfilter f (ndropns (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) (k-1)) \<sigma>))))" 
    using 17 by simp
  have 19: "k>0 \<longrightarrow> 
            g  (nmap (\<lambda>s. nnth s 0) 
                    (nfilter f (ndropns (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) 0) \<sigma>))))"
    using "17" by blast 
  have 20: "k>0 \<longrightarrow> (nfilter f (ndropns (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) 0) \<sigma>))) =
                    (nfilter f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) 0) (ndropns \<sigma>)))" 
    using "161" by auto 
 have 200: "nnth (nkfilter f 0 (ndropns \<sigma>)) 0 \<le> nlength (ndropns \<sigma>)"
    using "1610" "1612" by blast
  have 21: "k>0 \<longrightarrow> (\<forall> j\<le> (nnth (nkfilter f 0 (ndropns \<sigma>)) 0). 
                        ( (ndropns (ndropn j \<sigma>))) =
                        ( (ndropn j (ndropns \<sigma>)))) "
    using 200 ndropn_ndropns by (simp add: ndropn_ndropns order_subst2)
  have 22: "k>0 \<longrightarrow> (\<forall> j\<le> (nnth (nkfilter f 0 (ndropns \<sigma>) ) 0). 
                        (nfilter f (ndropns (ndropn j \<sigma>))) =
                        (nfilter f (ndropn j (ndropns \<sigma>))))" 
    using "21" by auto
  have 23: " k>0 \<longrightarrow> 
              (\<forall> j\<le> (nnth (nkfilter f 0 (ndropns \<sigma>) ) 0).  
                (nmap (\<lambda>s. nnth s 0) 
                     (nfilter f (ndropns (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) 0) \<sigma>)))) =
                (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn j \<sigma>))))
               ) "
    by (simp add: "2" "22" nfilter_ndropns_nmap_help_0)
  have 24: "k>0 \<longrightarrow> 
              (\<forall> j\<le> (nnth (nkfilter f 0 (ndropns \<sigma>) ) 0).  
                 g (nmap (\<lambda>s. nnth s 0) 
                        (nfilter f (ndropns (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) 0) \<sigma>)))) =
                 g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn j \<sigma>))))
              )" 
    using "23" by auto
  have 241: "k>0 \<longrightarrow> (\<forall> j\<le> (nnth (nkfilter f 0 (ndropns \<sigma>)  ) 0).  
                      g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn j \<sigma>)))) ) "
    using "19" "24" by blast 
  have 25: "k>0 \<longrightarrow> (\<forall> i < k -1. 
                      (\<forall> l. l\<le>  (nnth (nkfilter f 0 (ndropns \<sigma>)) (Suc i) ) \<and>
                            (nnth (nkfilter f 0 (ndropns \<sigma>)) i ) < l \<longrightarrow>  
                       (nfilter f (ndropn (nnth(nkfilter f 0 (ndropns \<sigma>)) (Suc i)) (ndropns \<sigma>)) =
                       (nfilter f (ndropn l (ndropns  \<sigma>)))     ) )  ) "
    proof auto
      fix i
      fix l
      assume a0: "0 < k "
      assume a1: "i < k - Suc 0 "
      assume a2: "l \<le> nnth (nkfilter f 0 (ndropns \<sigma>)) (Suc i) "
      assume a3: "nnth (nkfilter f 0 (ndropns \<sigma>)) i < l "    
      show "nfilter f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) (Suc i)) (ndropns \<sigma>)) = 
            nfilter f (ndropn l (ndropns \<sigma>))"
      proof -
        have 251: "k=1 \<Longrightarrow>
                   nfilter f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) (Suc i)) (ndropns \<sigma>)) =
                   nfilter f (ndropn l (ndropns \<sigma>))"
          using a1 by force
        have 2511: "1 < k \<Longrightarrow> enat (Suc i) \<le> nlength (nfilter f (ndropns \<sigma>))"
            by (metis (mono_tags, opaque_lifting) "5" One_nat_def Suc_diff_1 Suc_mono a0 a1 
                enat_ord_simps(1) less_imp_le order_subst2)
        then have 252: "1<k  \<Longrightarrow>
                   nfilter f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) (Suc i)) (ndropns \<sigma>)) =
                   nfilter f (ndropn l (ndropns \<sigma>)) "
          using 2 5 a1 nfilter_ndropns_nmap_help_j[of "ndropns \<sigma>" f l i] a2 a3 by fastforce 
        show ?thesis
        using "252" a1 by fastforce
      qed
    qed      
  have 261: "k>0 \<longrightarrow> (\<forall> i < k -1. 
                      (\<forall> l. l\<le>  (nnth (nkfilter f 0 (ndropns \<sigma>)) (Suc i) ) \<and>
                            (nnth (nkfilter f 0 (ndropns \<sigma>)) i ) < l \<longrightarrow> 
                            l \<le> nlength (ndropns \<sigma>) ))" 
    using 1612 by (meson diff_less enat_ord_simps(1) less_one less_trans_Suc order_subst2)
  have 262: "k>0 \<longrightarrow> (\<forall> i < k -1. 
                      (\<forall> l. l\<le>  (nnth (nkfilter f 0 (ndropns \<sigma>)) (Suc i) ) \<and>
                            (nnth (nkfilter f 0 (ndropns \<sigma>)) i ) < l \<longrightarrow> 
                            (ndropn l (ndropns  \<sigma>)) = (ndropns (ndropn l \<sigma>)) ))"
    using "261" by (simp add: ndropn_ndropns) 
  have 26: "k>0 \<longrightarrow> (\<forall> i < k -1. 
                      (\<forall> l. l\<le>  (nnth (nkfilter f 0 (ndropns \<sigma>)) (Suc i) ) \<and>
                            (nnth (nkfilter f 0 (ndropns \<sigma>)) i ) < l \<longrightarrow>  
                       (nmap (\<lambda>s. nnth s 0) 
                         (nfilter f (ndropn (nnth(nkfilter f 0 (ndropns \<sigma>)) (Suc i)) (ndropns \<sigma>))))=
                       (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns  (ndropn l \<sigma>)))     ) )  ) "
    using "25" "262" by auto
  have 27: "k>0 \<longrightarrow> (\<forall> i < k -1. 
                      (\<forall> l. l\<le>  (nnth (nkfilter f 0 (ndropns \<sigma>)) (Suc i) ) \<and>
                            (nnth (nkfilter f 0 (ndropns \<sigma>)) i ) < l \<longrightarrow>  
                      g (nmap (\<lambda>s. nnth s 0) 
                         (nfilter f 
                            (ndropn (nnth(nkfilter f 0 (ndropns \<sigma>)) (Suc i)) (ndropns \<sigma>))))))"
    by (simp add: "16")
  have 28:  "k>0 \<longrightarrow> (\<forall> i < k -1. 
                      (\<forall> l. l\<le>  (nnth (nkfilter f 0 (ndropns \<sigma>)) (Suc i) ) \<and>
                            (nnth (nkfilter f 0 (ndropns \<sigma>)) i ) < l \<longrightarrow>  
                      g (nmap (\<lambda>s. nnth s 0) (nfilter f  (ndropns  (ndropn l \<sigma>)))     )))  "
    using "25" "262" "27" by auto
  have 281: "k>0 \<longrightarrow> 
             (\<forall>j. 
                (j \<le> (nnth (nkfilter f 0 (ndropns \<sigma>)) 0 ) \<or>
                (\<exists> i. i < k -1 \<and> 
                     j\<le>  (nnth (nkfilter f 0 (ndropns \<sigma>)) (Suc i) ) \<and>
                     (nnth (nkfilter f 0 (ndropns \<sigma>)) i ) < j)  ) 
                     \<longrightarrow>  g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns  (ndropn j \<sigma>)))     )) "
    using 241 28 by blast
  have 282: "k>0 \<longrightarrow>  (\<forall>i<k - 1. 
                        nnth (nkfilter f 0 (ndropns \<sigma>)) i < nnth (nkfilter f 0 (ndropns \<sigma>)) (Suc i))"
    using nidx_nkfilter_expand[of "ndropns \<sigma>" f _ 0]
    by (metis (full_types) "2" "72" Suc_ile_eq enat_ord_simps(2) order_less_le_subst2) 
  have 29: "k>0 \<longrightarrow> (\<forall>j< (Suc (nnth (nkfilter f 0 (ndropns \<sigma>)) (k-1))).  
                        g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn j \<sigma>)))))"     
    using 281 282 cover_a[of "\<lambda> x. nnth (nkfilter f 0 (ndropns \<sigma>)) x " "k-1" 
                              "(\<lambda> j. g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn j \<sigma>)))))"]
    using "18" less_antisym by blast
  have 30: "k>0 \<longrightarrow> (\<exists>k\<le>nlength \<sigma>.
       (\<exists>i\<le>nlength \<sigma> - k. f (ndropn (i + k) \<sigma>)) \<and>
       h (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn k \<sigma>)))) \<and>
       (\<forall>j<k. (\<exists>i\<le>nlength \<sigma> - j. f (ndropn (i + j) \<sigma>)) \<and> 
              g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn j \<sigma>))))))"
    using "10" "11" "121" "132" "142" "29" "9" by simp
          (metis add_Suc_right antisym_conv2 eSuc_enat enat_ord_simps(1) ileI1 leD)
  have 31: "(\<exists>k\<le>nlength \<sigma>.
               (\<exists>i\<le>nlength \<sigma> - k. f (ndropn (i + k) \<sigma>)) \<and>
                 h (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn k \<sigma>)))) \<and>
                 (\<forall>j<k. (\<exists>i\<le>nlength \<sigma> - j. f (ndropn (i + j) \<sigma>)) \<and> 
                    g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn j \<sigma>))))))" 
    using 30 7 by (metis not_gr_zero zero_enat_def zero_le)
 show ?thesis using 31 by (simp add: PiUntilDistsem2)
qed 

lemma PiUntilDistsem4:
 assumes "(\<sigma> \<Turnstile> ( f \<Pi> g ) \<U> ( f \<Pi> h ) )"
 shows  "(\<sigma> \<Turnstile> f \<Pi> (g \<U> h)) "
proof -
 have 1: "(\<exists>k\<le>nlength \<sigma>.
            (\<exists>i\<le>nlength \<sigma> - k. f (ndropn (i + k) \<sigma>)) \<and>
            h (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn k \<sigma>)))) \<and>
            (\<forall>j<k. 
              (\<exists>i\<le>nlength \<sigma> - j. f (ndropn (i + j) \<sigma>)) \<and> 
              g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn j \<sigma>))))))" 
   using assms by (simp add: PiUntilDistsem2)
 obtain k where 2: "k\<le>nlength \<sigma> \<and> 
                    (\<exists>i\<le>nlength \<sigma> - k. f (ndropn (i + k) \<sigma>)) \<and>
                    h (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn k \<sigma>)))) \<and>
                    (\<forall>j<k. 
                      (\<exists>i\<le>nlength \<sigma> - j. f (ndropn (i + j) \<sigma>)) \<and> 
                      g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn j \<sigma>)))))" 
   using 1 by auto
 have 3: "(\<exists>i\<le>nlength \<sigma> - k. f (ndropn (i + k) \<sigma>))"
   using 2 by auto
 have 31: "(\<exists>i\<le> nlength (ndropns (ndropn k \<sigma>)). f (nnth (ndropns (ndropn k \<sigma>)) i))"
   by (metis "3" add.commute ndropn_ndropn ndropn_nlength ndropns_nlength ndropns_nnth)
 have 4: "k \<le> nlength \<sigma>"
   using 2 by auto
 obtain i where 5: "(enat i)\<le>nlength (ndropns (ndropn k \<sigma>))\<and> f (nnth (ndropns (ndropn k \<sigma>)) i) \<and> 
                   i = (LEAST na. enat na \<le> nlength (ndropns (ndropn k \<sigma>)) \<and> 
                   f (nnth (ndropns (ndropn k \<sigma>)) na))"
   by (metis (no_types, lifting) "31" LeastI_ex)
 have 51: "i = (LEAST na. enat na \<le> nlength (ndropns (ndropn k \<sigma>)) \<and> f (nnth (ndropns (ndropn k \<sigma>)) na))" 
   using 5 by auto
 have 60: "(enat i)\<le> nlength \<sigma> - k"
   by (metis "5" ndropn_nlength ndropns_nlength) 
 have 601: " k+i \<le> nlength \<sigma>"
   by (metis "4" "60" dual_order.eq_iff enat.simps(3) enat_add_sub_same enat_less_enat_plusI2 
      less_eqE less_imp_le order.not_eq_order_implies_strict plus_enat_simps(1)) 
 have 6: " i+k \<le> nlength \<sigma>"
   by (metis "601" add.commute) 
 have 61: "\<exists>x \<in> nset (ndropns (ndropn k \<sigma>)). f x"
   using "31" exists_Pred_nnth_nset by blast
 have 62: "\<exists>x \<in> nset (ndropns (ndropn (k+i) \<sigma>)). f x"
   by (metis "5" in_nset_conv_nnth ndropn_ndropn ndropn_ndropns nnth_zero_ndropn zero_enat_def zero_le)
 have 7: "(\<exists>i\<le>nlength \<sigma>. f (ndropn i \<sigma>))"
   by (metis "5" "6" add.commute ndropn_ndropn ndropns_nlength ndropns_nnth)
 have 71: "\<exists>x \<in> nset (ndropns \<sigma>). f x"
   using "5" "6" using in_nset_ndropns using "7" by blast
 have 72: "\<exists> x \<in> nset(nkfilter f 0 (ndropns \<sigma>)). f (nnth (ndropns \<sigma>) x) " 
   using 71  nkfilter_holds_b[of "(ndropns \<sigma>)" f]  sfxfilter_help[of \<sigma>] 
   by (metis "71" Nat.add_0_right ndropns_nlength ndropns_nnth)
 have 722: "(nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0) \<le> nlength (ndropns (ndropn k \<sigma>))" 
   by (metis "61" gen_nlength_def nkfilter_upperbound nlength_code zero_enat_def zero_le)
 have 723: "(nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0) \<le> nlength ( (ndropn k \<sigma>))" 
   by (metis "722" ndropns_nlength)
 have 73: "f (ndropn (nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)  (ndropn k \<sigma>)) "
   using nkfilter_holds[of "ndropns (ndropn k \<sigma>)" "f" "0"] 
   by (metis (no_types, lifting) "61" "722" diff_zero ndropns_nfilter_nnth ndropns_nlength 
       ndropns_nnth nkfilter_nfilter zero_enat_def zero_le)
 have 74: "f (ndropn ((nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)+k)  \<sigma>) "
   using "73" by (simp add: add.commute ndropn_ndropn)
 have 75: "f (ndropn k (ndropn (nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0) \<sigma>))  "
   by (simp add: "74" ndropn_ndropn)
 have 76: "(nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0) = nleast f (ndropns (ndropn k \<sigma>))" 
   by (simp add: "61" nkfilter_nleast)
 have 77: "nleast f (ndropns (ndropn k \<sigma>)) = (LEAST na. enat na \<le> nlength (ndropns (ndropn k \<sigma>)) \<and> 
           f (nnth (ndropns (ndropn k \<sigma>)) na))"
   using  61  nleast_conv[of "ndropns (ndropn k \<sigma>)" f] by auto
 have 78: "nleast f (ndropns (ndropn k \<sigma>)) = i   "
   using "5" "77" by blast
 have 8: "h (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn k \<sigma>))))"  
   using 2 by auto
 have 10: "nfirst (nkfilter f k (ndropn k (ndropns \<sigma>))) = 
           ((nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)+k)"
   by (metis "2" "61" diff_add ndropn_0 ndropn_ndropns ndropn_nfirst ndropns_nlength 
       nkfilter_lowerbound nkfilter_nnth_n_zero zero_enat_def zero_le)
 have 101: "nfirst (nkfilter f k (ndropn k (ndropns \<sigma>))) = k+ i"
   by (simp add: "10" "76" "78") 
 have 90: "f (nlast (ntaken ((nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)+k) (ndropns \<sigma>)))"
   by (metis "6" "74" "76" "78" ndropns_nnth ntaken_nlast)
 have 91: "((nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)+k) \<le> nlength (ndropns \<sigma>)"
   using nkfilter_upperbound[of "(ndropns (ndropn k \<sigma>))" f 0 0 ] 
         ndropn_nlength[of k \<sigma>] ndropns_nlength[of "ndropn k \<sigma>"]
   by (simp add: "6" "76" "78" ndropns_nlength)  
 have 92: "((nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)+k) = ((nnth (nkfilter f k (ndropns (ndropn k \<sigma>))) 0))"
   by (simp add: "61" nkfilter_nleast)
 let ?kf = "(the_enat ((nlength(nfilter f (ntaken ((nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)+k) (ndropns \<sigma>))))))" 
 let ?nkf = "(the_enat ((nlength(nkfilter f  0 (ntaken ((nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)+k) (ndropns \<sigma>))))))" 
 let ?pkf = "(the_enat ((nlength (nkfilter f 0 (ntaken k (ndropns \<sigma>))))))" 
 have 93: "?kf = ?nkf"
   by (metis "90" nfinite_ntaken nkfilter_nlength nset_nlast)
 have 94: "((nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)+k) \<le> nnth (nkfilter f 0 (ndropns \<sigma>)) ?nkf  " 
   by (metis "74" "90" "91" add.left_neutral eq_iff ndropn_nfirst ndropns_nlength ndropns_nnth 
       nkfilter_chop1_ndropn nkfilter_nfirst)
 have 95: "(nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0) = 0 \<Longrightarrow> k = nnth (nkfilter f 0 (ndropns \<sigma>)) ?nkf"
   by (metis "10" "90" "91" add_cancel_left_left ndropn_nfirst nkfilter_chop1_ndropn) 
 have 98: "0< ?nkf \<and> 0< (nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)  \<Longrightarrow> 
            nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf -1) < k" 
    proof -
       assume b: "0< ?nkf \<and> 0< (nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)" 
       have 980: "(nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf -1)) \<in> nset (nkfilter f 0 (ndropns \<sigma>)) " 
         by (metis in_nset_conv_nnth le_cases nfinite_ntaken nset_nlast ntaken_all ntaken_nlast)
       have 981: "f ( nnth (ndropns \<sigma>) (nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf -1))  ) "
         using nkfilter_holds[of "(ndropns \<sigma>)" f "(nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf -1))" 0]
         using "71" "980" by auto
       have 982: "\<not> f ( ndropn k \<sigma>)"
         by (metis "10" "4" add_cancel_left_left b gr_implies_not0 ndropn_nfirst ndropns_nnth 
             nkfilter_nfirst) 
       have 984: "f (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf -1)) \<sigma>)"
         using  "71" "980" "981" ndropns_nnth[of "(nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf -1))" \<sigma>]
         by (metis gen_nlength_def in_nset_conv_nnth ndropns_nlength nkfilter_upperbound nlength_code) 
       have 985: "\<And>j. k \<le>j \<Longrightarrow> j< k+i \<Longrightarrow> \<not> f (ndropn j \<sigma>)"
          proof -
           fix j 
           assume a0: "k\<le>j"
           assume a1: "j<k+i"
           show "\<not> f (ndropn j \<sigma>)" 
            proof -
             have 9851: "\<exists>x\<in>nset (ndropn k (ndropns \<sigma>)). f x "
               by (simp add: "4" "61" ndropn_ndropns ndropns_nlength)
             have 9852: "enat k \<le> nlength (ndropns \<sigma>)"
               by (simp add: "4" ndropns_nlength) 
             have 9853: "k \<le> j"
               by (simp add: a0) 
             have 9854: "j < nnth (nkfilter f k (ndropn k (ndropns \<sigma>))) 0"
               using "10" "101" "92" "9852" a1 ndropn_ndropns by fastforce
             have 9855: "\<not> f (nnth (ndropns \<sigma>) j)"
               using  9851 9852 9853 9854 nkfilter_n_not_before[of k "ndropns \<sigma>" f j] by auto
             show ?thesis 
               by (metis "10" "101" "6" "76" "78" "9855" a1 enat_ord_simps(2) less_imp_le
                   ndropns_nnth order_less_le_subst2)
            qed
          qed
       have 986: " ntaken ?nkf (nkfilter f 0 (ndropns \<sigma>)) =
                   nkfilter f 0 (ntaken (nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0 + k) (ndropns \<sigma>))" 
         using nkfilter_chop1_ntaken[of "(nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0 + k)" "(ndropns \<sigma>)" f 0]
         using "90" "91" by blast
       have 987: "\<not> enat (?nkf) \<le> nlength (nkfilter f 0 (ndropns \<sigma>)) \<or> 
                 nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf - 1) < nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf)" 
         using 71  by (meson b diff_less less_one nidx_nkfilter_gr)
       have 988: "nlast (nkfilter f 0 (ntaken (nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0 + k) (ndropns \<sigma>))) = 
                  0 + (nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0 + k)"
         by (simp add: "90" "91" nkfilter_nlast) 
       have 989: "nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf) = nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0 + k"
         using 986 
         by (metis "988" add.left_neutral ntaken_nlast)
       have 990: "nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf) = nfirst (nkfilter f k (ndropn k (ndropns \<sigma>)))"
         using "10" "989" by fastforce
       have 991: "nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf) = k + i"
         using "10" "101" "989" by presburger
       show ?thesis
         by (metis "984" "985" "986" "987" "991" enat_the_enat infinity_ileE le_cases not_le_imp_less ntaken_all)   
   qed
 have 100: " h (nmap (\<lambda>s. nnth s 0) (ndropn ?kf (nfilter f (ndropns \<sigma>)))) "
   using ndropns_nfilter_ndropn_a[of _ f "ndropn k \<sigma>" ] 
         nfilter_chop1_ndropn[of "((nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)+k)" "(ndropns \<sigma>)" f  ]
       "101" "61" "76" "78" "8" "90" "91"
   by simp
      (metis "10" ndropn_0 ndropn_ndropn ndropn_ndropns zero_enat_def zero_le)        
 have 11: "h (ndropn ?kf (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>))))" 
   by (simp add: "100" ndropn_nmap)
 have 111: "?kf \<le> nlength (nfilter f (ndropns \<sigma>))"
   by (metis "90" enat_ile le_cases nfilter_chop1_ntaken ntaken_all the_enat.simps)
 have 112: "nfinite (nfilter f (ntaken ((nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0)+k) (ndropns \<sigma>)))"
   by (metis "90" "91" nfilter_chop1_ntaken nfinite_ntaken)
 have 13: "(\<forall>j<k. 
              (\<exists>x \<in> nset(ndropns (ndropn j \<sigma>)). f x) \<and> 
              g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns (ndropn j \<sigma>)))))"
   by (metis "2" add.commute in_nset_ndropns ndropn_ndropn ndropn_nlength)
 have 151: "(\<forall>jj<?kf. 
               (ndropn jj (nfilter f (ndropns \<sigma>)))  =
               nfilter f (ndropns (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) jj) \<sigma>))
             )"   using nfilter_nkfilter_ndropn  
   by (simp add: "111" "71" less_imp_le ndropns_nfilter_ndropn_a order_less_le_subst2)
 have 152: "(\<And> jj. (enat jj)<?kf \<Longrightarrow> (nnth (nkfilter f 0 (ndropns \<sigma>)) jj) < k ) "     
   proof -
    fix jj
    assume a: "(enat jj) < ?kf" 
    show "nnth (nkfilter f 0 (ndropns \<sigma>)) jj < k "  
     proof -
      have 1522: "(enat jj) \<le> ?nkf -1" 
        using "93" a by auto         
      have 1523: "nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf -1) < nnth (nkfilter f 0 (ndropns \<sigma>)) ?nkf " 
        by (metis "111" "71" "93" a diff_less gr_implies_not_zero less_numeral_extra(1) 
            nidx_nkfilter_gr nkfilter_nlength not_gr_zero zero_enat_def)
      have 15230: "enat (?nkf - 1) \<le> nlength (nkfilter f 0 (ndropns \<sigma>))"  
        by (metis "111" "71" "93" One_nat_def Suc_ile_eq Suc_pred a less_imp_le nkfilter_nlength 
             not_gr_zero not_less_zero zero_enat_def)
      have 1524: "nnth (nkfilter f 0 (ndropns \<sigma>)) jj \<le> nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf-1) " 
        using "1522" "1523" "71" "93" 15230
          nidx_less_eq[of "(nkfilter f 0 (ndropns \<sigma>))" "(jj)" "?nkf -1"] 
          nidx_nkfilter[of "(ndropns \<sigma>)" f 0] 
        using enat_ord_simps(1) by blast
      have 1525: " k \<le>  nnth (nkfilter f 0 (ndropns \<sigma>)) ?nkf " 
        using "94" add_leE by blast
      have 1526: "(nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0) = 0 \<Longrightarrow>
                   nnth (nkfilter f 0 (ndropns \<sigma>)) jj < k"
        using "1523" "1524" "95" by linarith 
      have 1527: "0<(nnth (nkfilter f 0 (ndropns (ndropn k \<sigma>))) 0) \<Longrightarrow> 
                  nnth (nkfilter f 0 (ndropns \<sigma>)) (?nkf -1) < k"
        using "93" "98" a by auto
      show ?thesis using "1524" "1526" "1527" dual_order.strict_trans2 by blast            
    qed
   qed
 have 153: "(\<forall>jj<?kf.
            g (nmap (\<lambda>s. nnth s 0) 
                (nfilter f (ndropns (ndropn (nnth (nkfilter f 0 (ndropns \<sigma>)) jj) \<sigma>)) )
            ))"  
   using "112" "13" "152" nfinite_nlength_enat by force
 have 1530: "(\<forall>jj<?kf.
              g (nmap (\<lambda>s. nnth s 0) (ndropn jj  (nfilter f (ndropns \<sigma>)))))"
   by (simp add: "151" "153")
 have 1531: "(\<forall>jj<?kf.
              g  (ndropn jj  (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))))"
   by (simp add: "1530" ndropn_nmap)
 have 154: "( (\<exists>i. (enat i)\<le>nlength \<sigma> \<and> f (ndropn i \<sigma>)) \<and>
              (\<exists>kk. (enat kk)\<le>nlength (nfilter f (ndropns \<sigma>)) \<and>
                 h (ndropn kk (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))) \<and>
                 (\<forall>jj<kk. g (ndropn jj (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns \<sigma>)))))))" 
   using "11" "111" "1531" "7" by blast
 show ?thesis by (simp add: "154" PiUntilDistsem1)
qed

lemma  PiUntilDistsem:
 " \<sigma> \<Turnstile>  f \<Pi> (g \<U> h) = ( f \<Pi> g ) \<U> ( f \<Pi> h ) "
using PiUntilDistsem3  PiUntilDistsem4 using unl_lift2 by blast

lemma  PiUntilDist:
 " \<turnstile>  f \<Pi> (g \<U> h) = ( f \<Pi> g ) \<U> ( f \<Pi> h ) "
using PiUntilDistsem Valid_def by blast


subsubsection \<open>PiChopstar\<close>

lemma wnextboxnotstatesem:
assumes " k \<le> nlength \<sigma>"
 shows  " (\<forall> j \<le> nlength \<sigma>. k<j \<longrightarrow> \<not> (\<lambda>y. w (NNil y)) (nnth \<sigma> j)) =
             (LIFT(wnext (\<box> (init (\<not> w))))) (ndropn k \<sigma>)  
             "
using assms  
 proof  (auto simp add: itl_defs ndropn_nfirst)
  show "\<And>n. enat k \<le> nlength \<sigma> \<Longrightarrow>
         \<forall>j. enat j \<le> nlength \<sigma> \<longrightarrow> k < j \<longrightarrow> \<not> w (NNil (nnth \<sigma> j)) \<Longrightarrow>
         nlength \<sigma> - enat k \<noteq> enat 0 \<Longrightarrow>
         enat n \<le> nlength \<sigma> - enat k - enat (Suc 0) \<Longrightarrow> w (NNil (nnth \<sigma> (Suc (k + n)))) \<Longrightarrow> False"
          by (metis   add_Suc_right  assms diff_self_eq_0 dual_order.order_iff_strict 
               enat_ord_simps(2) idiff_enat_enat less_add_same_cancel1 linorder_le_cases 
             nfinite_nlength_enat nfinite_ntaken not_le_imp_less ntaken_all ntaken_nlast 
             zero_enat_def zero_less_Suc)
  show "\<And>j. enat k \<le> nlength \<sigma> \<Longrightarrow> enat j \<le> nlength \<sigma> \<Longrightarrow> k < j \<Longrightarrow> w (NNil (nnth \<sigma> j)) \<Longrightarrow>
             nlength \<sigma> - enat k = enat 0 \<Longrightarrow> False"
    by (metis add.right_neutral enat.inject enat_add_sub_same enat_ord_code(4) leD less_eqE 
        min.strict_order_iff min_enat_simps(1) zero_enat_def)
  show " \<And>j. enat k \<le> nlength \<sigma> \<Longrightarrow>
         enat j \<le> nlength \<sigma> \<Longrightarrow>
         k < j \<Longrightarrow> w (NNil (nnth \<sigma> j)) \<Longrightarrow>
         \<forall>n. enat n \<le> nlength \<sigma> - enat k - enat (Suc 0) \<longrightarrow> \<not> w (NNil (nnth \<sigma> (Suc (k + n)))) \<Longrightarrow> False " 
   proof -
     fix j
     assume a0: "enat k \<le> nlength \<sigma>"
     assume a1: "enat j \<le> nlength \<sigma>"
     assume a2: "k < j "   
     assume a3: "w (NNil (nnth \<sigma> j))"  
     assume a4: "\<forall>n. enat n \<le> nlength \<sigma> - enat k - enat (Suc 0) \<longrightarrow> \<not> w (NNil (nnth \<sigma> (Suc (k + n))))"
     show "False"
     proof -
      have 1: " j = (Suc (k + (j - (Suc k))))"
        using a2 by auto   
      have 2: "\<forall>n. enat n + 1 + k\<le> nlength \<sigma>  \<longrightarrow> \<not> w (NNil (nnth \<sigma> (Suc (k + n))))" 
        by auto
           (metis One_nat_def a4 add.commute enat.simps(3) enat_add_sub_same enat_minus_mono1 
            one_enat_def)  
      have 3: "(j - (Suc k)) + 1 + k \<le> nlength \<sigma> " 
        using "1" a1 by simp
      have 4: "\<not> w (NNil (nnth \<sigma> (Suc (k + (j - (Suc k)))))) " 
        using 2 3  eSuc_enat plus_1_eSuc(2) by auto         
      from 1 4 a3 show ?thesis by auto 
     qed
  qed
qed
 
lemma NotStateUntilStateAndsem:
 "(\<sigma> \<Turnstile> (init (\<not> w)) \<U> ((init w) \<and> f) ) =
   (\<exists>k. (enat k)\<le>nlength \<sigma> \<and> w (NNil (nnth \<sigma> k)) \<and> f (ndropn k \<sigma>) \<and> (\<forall>j<k. \<not> w (NNil (nnth \<sigma> j))))
  "
by (auto simp add: until_d_def init_defs ndropn_nfirst) 

lemma StateUntilEqvWPrevChopsem:
  "\<sigma> \<Turnstile> (init w) \<U> f = (wprev (\<box> (init w)))\<frown>f"
proof (auto simp add: until_d_def itl_defs ntaken_nnth ndropn_nfirst  min_absorb1 )
 show "\<And>k. enat k \<le> nlength \<sigma> \<Longrightarrow>
         f (ndropn k \<sigma>) \<Longrightarrow>
         \<forall>j<k. w (NNil (nnth \<sigma> j)) \<Longrightarrow>
         \<exists>n. enat n \<le> nlength \<sigma> \<and>
             (min (enat n) (nlength \<sigma>) = enat 0 \<or>
              (\<forall>na. na \<le> the_enat (min (enat (n - Suc 0)) (epred (nlength \<sigma>))) \<and> na \<le> n \<and> enat na \<le> nlength \<sigma> \<longrightarrow>
                    w (NNil (nnth \<sigma> na)))) \<and>
             f (ndropn n \<sigma>)" 
  by (metis One_nat_def Suc_pred epred_enat epred_min le_imp_less_Suc min.orderE not_gr_zero 
      the_enat.simps)
next
   fix n
   assume a0: "enat n \<le> nlength \<sigma>"
   assume a1: "f (ndropn n \<sigma>)"
   assume a2: " \<forall>na. na \<le> the_enat (min (enat (n - Suc 0)) (epred (nlength \<sigma>))) \<and> na \<le> n \<and> 
                     enat na \<le> nlength \<sigma> \<longrightarrow> w (NNil (nnth \<sigma> na))"
   show " \<exists>k. enat k \<le> nlength \<sigma> \<and> f (ndropn k \<sigma>) \<and> (\<forall>j<k. w (NNil (nnth \<sigma> j)))" 
   proof -
   have 1: "enat n \<le> nlength \<sigma> \<and> f (ndropn n \<sigma>)"
    using a0 a1 by auto
   have 2: "(\<forall>na. na \<le> the_enat (min (enat (n - Suc 0)) (epred (nlength \<sigma>))) \<and> na \<le> n \<and> 
                  enat na \<le> nlength \<sigma> \<longrightarrow> w (NNil (nnth \<sigma> na))) \<longrightarrow>
            (\<forall>j<n. w (NNil (nnth \<sigma> j)))"
     by (auto simp add: min_def)
        (simp add: a0 less_imp_le order_less_le_subst2,
         metis One_nat_def a0 epred_enat epred_min min.absorb_iff2)  
   have 3: "(\<forall>j<n. w (NNil (nnth \<sigma> j)))"
     using "2" a2 by blast 
   show ?thesis
   using "1" "3" by blast    
  qed
qed 

lemma StateUntilEqvWPrevChop:
 "\<turnstile> (init w) \<U> f = (wprev (\<box> (init w)))\<frown>f"
using StateUntilEqvWPrevChopsem Valid_def by blast

lemma UntilChopDist:
 " \<turnstile> (init w) \<U> (g\<frown>h) = ( (init w) \<U> g) \<frown>h"
using StateUntilEqvWPrevChop[of w ]
by (metis SChopAssoc inteq_reflection)

lemma PiEmptysem:
 " \<sigma> \<Turnstile> (init w) \<Pi> empty = (init (\<not> w)) \<U> ((init w) \<and> wnext (\<box> (init (\<not> w))))"
proof -
have 1: "(\<sigma> \<Turnstile> (init w) \<Pi> empty) = 
         ((\<exists>x\<in>nset \<sigma>. w (NNil x)) \<and> nlength (nfilter (\<lambda>y. w (NNil y)) \<sigma>) = 0)"
  by (simp add: Pistate itl_defs zero_enat_def)
have 2: "((\<exists>x\<in>nset \<sigma>. w (NNil x)) \<and> nlength (nfilter (\<lambda>y. w (NNil y)) \<sigma>) = 0) =
        (\<exists> k \<le> nlength \<sigma> . (\<lambda>y. w (NNil y)) (nnth \<sigma> k) \<and> 
             (\<forall> j. j<k \<longrightarrow> \<not> (\<lambda>y. w (NNil y)) (nnth \<sigma> j)) \<and>
             (\<forall> j \<le> nlength \<sigma>. k<j \<longrightarrow> \<not> (\<lambda>y. w (NNil y)) (nnth \<sigma> j)))" 
  by (simp add: nfilter_nlength_zero_conv_2)
have 3: "(\<exists> k \<le> nlength \<sigma> . (\<lambda>y. w (NNil y)) (nnth \<sigma> k) \<and> 
             (\<forall> j. j<k \<longrightarrow> \<not> (\<lambda>y. w (NNil y)) (nnth \<sigma> j)) \<and>
             (\<forall> j \<le> nlength \<sigma>. k<j \<longrightarrow> \<not> (\<lambda>y. w (NNil y)) (nnth \<sigma> j))) =
           (\<exists>k\<le>nlength \<sigma>.
       w (NNil (nnth \<sigma> k)) \<and>
       (LIFT(wnext (\<box> (init (\<not> w))))) (ndropn k \<sigma>) \<and>
       (\<forall>j<k. \<not> w (NNil (nnth \<sigma> j)))) " (is "?lhs =?rhs")
  proof 
   assume a: ?lhs
   show "?rhs" using a wnextboxnotstatesem by auto
   next
   assume b: ?rhs
   show "?lhs" 
    proof -
     obtain k where 1: 
       "enat k \<le> nlength \<sigma> \<and> w (NNil (nnth \<sigma> k)) \<and> 
         (LIFT(wnext (\<box> (init (\<not> w))))) (ndropn k \<sigma>) \<and> (\<forall>j<k. \<not> w (NNil (nnth \<sigma> j)))" 
        using b by auto
      have 2: "enat k \<le> nlength \<sigma> \<and> w (NNil (nnth \<sigma> k)) \<and> (\<forall>j<k. \<not> w (NNil (nnth \<sigma> j))) \<and> 
               (\<forall>j. enat j \<le> nlength \<sigma> \<longrightarrow> k < j \<longrightarrow> \<not> w (NNil (nnth \<sigma> j)))"
        using 1  wnextboxnotstatesem[of k \<sigma> w ] 
        proof -
         have "\<forall>x0. (x0 < k \<longrightarrow> \<not> w (NNil (nnth \<sigma> x0))) = (\<not> x0 < k \<or> \<not> w (NNil (nnth \<sigma> x0)))"
           by meson 
         then have f1: "enat k \<le> nlength \<sigma> \<and> w (NNil (nnth \<sigma> k)) \<and> 
             (LIFT(wnext (\<box> (init (\<not> w))))) (ndropn k \<sigma>) \<and> (\<forall>n. \<not> n < k \<or> \<not> w (NNil (nnth \<sigma> n)))"
           by (metis \<open>enat k \<le> nlength \<sigma> \<and> w (NNil (nnth \<sigma> k)) \<and> 
                (LIFT(wnext (\<box> (init (\<not> w))))) (ndropn k \<sigma>) \<and> (\<forall>j<k. \<not> w (NNil (nnth \<sigma> j)))\<close>)
         have "(enat k \<le> nlength \<sigma> \<longrightarrow> (\<forall>n. enat n \<le> nlength \<sigma> \<and> k < n \<longrightarrow> \<not> w (NNil (nnth \<sigma> n))) = 
                (LIFT(wnext (\<box> (init (\<not> w))))) (ndropn k \<sigma>)) = 
               (\<not> enat k \<le> nlength \<sigma> \<or> (\<forall>n. (\<not> enat n \<le> nlength \<sigma> \<or> \<not> k < n) \<or> \<not> w (NNil (nnth \<sigma> n))) =
                 (LIFT(wnext (\<box> (init (\<not> w))))) (ndropn k \<sigma>))"
           by blast 
         then have "\<not> enat k \<le> nlength \<sigma> \<or> (\<forall>n. (\<not> enat n \<le> nlength \<sigma> \<or> \<not> k < n) \<or> \<not> w (NNil (nnth \<sigma> n))) =
                    (LIFT(wnext (\<box> (init (\<not> w))))) (ndropn k \<sigma>)"
           by (metis \<open>enat k \<le> nlength \<sigma> \<Longrightarrow> (\<forall>j. enat j \<le> nlength \<sigma> \<longrightarrow> k < j \<longrightarrow> \<not> w (NNil (nnth \<sigma> j))) =
                (LIFT(wnext (\<box> (init (\<not> w))))) (ndropn k \<sigma>)\<close>) 
         then show ?thesis using f1 by presburger
        qed 
       show ?thesis using "2" by blast
    qed
  qed
have 4: " (\<exists>k\<le>nlength \<sigma>.
           w (NNil (nnth \<sigma> k)) \<and>
          (LIFT(wnext (\<box> (init (\<not> w))))) (ndropn k \<sigma>) \<and>
        (\<forall>j<k. \<not> w (NNil (nnth \<sigma> j)))) =
          (\<sigma> \<Turnstile> (init (\<not> w)) \<U> ((init w) \<and> wnext (\<box> (init (\<not> w)))))"
  by (simp add: NotStateUntilStateAndsem) 
from 1 2 3 4 show ?thesis by auto
qed
  
lemma PiEmpty:
 "\<turnstile> (init w) \<Pi> empty = (init (\<not> w)) \<U> ((init w) \<and> wnext (\<box> (init (\<not> w))))"
using PiEmptysem Valid_def by blast

lemma StatePiBoxStatesem:
 "\<sigma> \<Turnstile> (init w) \<Pi> f = (init w) \<Pi> (f \<and> \<box> (init w))"
proof -
 have 1: "(\<sigma> \<Turnstile> (init w) \<Pi> f) =
          ((\<exists> x \<in> nset \<sigma>. w (NNil x)) \<and>  ( (nfilter (\<lambda>y. w (NNil y)) \<sigma>) \<Turnstile>  f))" 
   by (metis Pistate)
 have 2: "(\<sigma> \<Turnstile> (init w) \<Pi> (f \<and> \<box> (init w))) = 
          ((\<exists> x \<in> nset \<sigma>. w (NNil x)) \<and>  ( (nfilter (\<lambda>y. w (NNil y)) \<sigma>) \<Turnstile> f \<and> \<box> (init w)))" 
   by (metis Pistate)
 have 3: "( (nfilter (\<lambda>y. w (NNil y)) \<sigma>) \<Turnstile> f \<and> \<box> (init w))
         = (f (nfilter (\<lambda>y. w (NNil y)) \<sigma>) \<and>
        (\<forall>n\<le>nlength (nfilter (\<lambda>y. w (NNil y)) \<sigma>). w (NNil (nnth (nfilter (\<lambda>y. w (NNil y)) \<sigma>) n))))" 
   by (simp add: itl_defs ndropn_nfirst)  
 have 4: "((\<exists> x \<in> nset \<sigma>. w (NNil x)) \<and> (f (nfilter (\<lambda>y. w (NNil y)) \<sigma>) \<and>
           (\<forall>n\<le>nlength (nfilter (\<lambda>y. w (NNil y)) \<sigma>). w (NNil (nnth (nfilter (\<lambda>y. w (NNil y)) \<sigma>) n))))) =
           ((\<exists> x \<in> nset \<sigma>. w (NNil x)) \<and> f (nfilter (\<lambda>y. w (NNil y)) \<sigma>))"
   by (meson nkfilter_nnth_aa)
 show ?thesis using "1" "2" "3" "4" by auto
qed

lemma StatePiBoxState:
 "\<turnstile> (init w) \<Pi> f = (init w) \<Pi> (f \<and> \<box> (init w))"
using StatePiBoxStatesem Valid_def by blast

lemma StatePiUntil1:
 "\<turnstile> ( (init (\<not> w)) \<U> ( (init w) \<and> (init w) \<Pi> f )) =
           (wprev (\<box> (init (\<not> w))))\<frown>( (init w) \<and> (init w) \<Pi> f )  "
using StateUntilEqvWPrevChop by blast
  
lemma StatePiUntilsem2:
 "(\<sigma> \<Turnstile> (wprev (\<box> (init (\<not> w))))\<frown>( (init w) \<and> (init w) \<Pi> f )) =
          (\<sigma> \<Turnstile> ((wprev (\<box> (init (\<not> w))))\<frown> ((init w) \<and> empty)) \<frown> ((init w) \<and> (init w) \<Pi> f ) )"
by (auto simp add: schop_defs init_defs empty_defs  zero_enat_def ndropn_nfirst)
 (metis (no_types, lifting) add.right_neutral dual_order.refl enat.simps(3) enat_add_sub_same 
    min.orderE min.orderE nfinite_ntaken nnth_nlast ntaken_nlast ntaken_nlength the_enat.simps 
  zero_enat_def)
 
  
lemma StatePiUntil2:
 "\<turnstile> (wprev (\<box> (init (\<not> w))))\<frown>( (init w) \<and> (init w) \<Pi> f ) =
          ( ((wprev (\<box> (init (\<not> w))))\<frown> ((init w) \<and> empty)) \<frown> ((init w) \<and> (init w) \<Pi> f ) )"
by (simp add: StatePiUntilsem2 Valid_def)

lemma StatePiUntil3:
 "\<turnstile> ((wprev (\<box> (init (\<not> w))))\<frown> ((init w) \<and> empty)) =
          ( ((init (\<not> w)) \<U> ((init w) \<and> wnext (\<box> (init (\<not> w))))) \<frown> ((init w) \<and> empty)) " 
proof -
 have 1: " \<turnstile> (wprev (\<box> (init (\<not> w))))\<frown> ((init w) \<and> empty) =
             (init (\<not> w)) \<U> ((init w) \<and> empty) "
   by (meson Prop11 StateUntilEqvWPrevChop)
 have 2: "\<turnstile> ((init w) \<and> empty) = ((init w) \<and> wnext (\<box> (init (\<not> w))))\<frown>((init w) \<and> empty)"
   by (auto simp add:  Valid_def itl_defs zero_enat_def ndropn_nfirst) 
      (metis ndropn_0 ndropn_nfirst ,
       metis add.right_neutral add_diff_inverse_nat enat.simps(3) enat_add_sub_same enat_ord_simps(2) 
       iless_Suc_eq less_eqE min.absorb2 min_def ntaken_all one_eSuc one_enat_def order_refl
        plus_1_eq_Suc zero_enat_def zero_le)
 show ?thesis by (metis "1" "2" UntilChopDist inteq_reflection)
qed

lemma StatePiUntilsem4:
"(\<sigma> \<Turnstile> ((init (\<not> w)) \<U> ((init w) \<and> wnext (\<box> (init (\<not> w))))) \<frown> ((init w) \<and> empty)) =
          (\<sigma> \<Turnstile> ((init w) \<Pi> empty)\<frown> ((init w) \<and> empty)) " 
 by (metis PiEmpty inteq_reflection)

lemma StatePiUntil4:
 "\<turnstile> ((init (\<not> w)) \<U> ((init w) \<and> wnext (\<box> (init (\<not> w))))) \<frown> ((init w) \<and> empty) =
          ( ((init w) \<Pi> empty)\<frown> ((init w) \<and> empty))"
by (simp add: StatePiUntilsem4 Valid_def)

lemma StatePiUntilsem:
 "\<sigma> \<Turnstile> (init w) \<Pi> f = (init (\<not> w)) \<U> ( (init w) \<and> (init w) \<Pi> f )"
proof -
 have 2: "(\<sigma> \<Turnstile> (init (\<not> w)) \<U> ( (init w) \<and> (init w) \<Pi> f )) =
          (\<sigma> \<Turnstile> (wprev (\<box> (init (\<not> w))))\<frown>( (init w) \<and> (init w) \<Pi> f ))" 
   using StateUntilEqvWPrevChopsem[of "LIFT(\<not> w)" "LIFT((init w) \<and> (init w) \<Pi> f)" "\<sigma>"]
   by simp
 have 7: "(\<sigma> \<Turnstile> (wprev (\<box> (init (\<not> w))))\<frown>( (init w) \<and> (init w) \<Pi> f )) = 
          (\<sigma> \<Turnstile> (((init w) \<Pi> empty)\<frown> ((init w) \<and> empty))\<frown>( (init w) \<and> (init w) \<Pi> f )) "
   by (metis PiEmpty StatePiUntil2 StatePiUntil3 inteq_reflection) 
 have 8: "(\<sigma> \<Turnstile> (((init w) \<Pi> empty)\<frown> ((init w) \<and> empty))\<frown>( (init w) \<and> (init w) \<Pi> f )) =
          (\<sigma> \<Turnstile> (((init w) \<Pi> empty))\<frown>( (init w) \<and> (init w) \<Pi> f ))" 
  by (auto simp add: schop_defs init_defs empty_defs  zero_enat_def ndropn_nfirst)
   (metis (no_types, opaque_lifting) enat_0_iff(2) min.absorb_iff1 ndropn_all ndropn_nlength 
    nle_le nlength_NNil ntaken_nlast ntaken_nlength ntaken_ntaken) 
 have 9: "(\<sigma> \<Turnstile> (((init w) \<Pi> empty))\<frown>( (init w) \<and> (init w) \<Pi> f )) =
          (\<sigma> \<Turnstile> (init w) \<Pi> (empty\<frown>f))"    
   using PiSChopDistsema PiSChopDistsemb by blast
 have 10: "(\<sigma> \<Turnstile> (init w) \<Pi> (empty\<frown>f)) = (\<sigma> \<Turnstile> (init w) \<Pi> f)"
   by (metis EmptySChop inteq_reflection)
 show ?thesis 
   by (simp add: "10" "2" "7" "8" "9")
qed

lemma StatePiUntil:
 "\<turnstile> (init w) \<Pi> f = (init (\<not> w)) \<U> ( (init w) \<and> (init w) \<Pi> f )"
using StatePiUntilsem by blast

lemma StateAndPiEmpty:
 "\<turnstile>  ((init w) \<and> (init w) \<Pi> empty) = (w \<and> empty) \<frown> (wnext (\<box> (init (\<not> w))))"
proof -
 have 1: "\<turnstile>  ((init w) \<and> (init w) \<Pi> empty) = 
             ((init w) \<and> (init (\<not> w)) \<U> ( (init w) \<and> (wnext (\<box> (init (\<not> w))))) ) "
   using PiEmpty by fastforce 
 have 2: "\<turnstile> ((init w) \<and> (init (\<not> w)) \<U> ( (init w) \<and> (wnext (\<box> (init (\<not> w))))) )
             = ((init w) \<and> (wnext (\<box> (init (\<not> w))))) "
   by (auto simp add: until_d_def Valid_def init_defs ndropn_nfirst)
      (metis ndropn_0 ndropn_nfirst neq0_conv,
       metis gr_implies_not_zero ndropn_0 ndropn_nfirst zero_enat_def zero_le)
 have 3: "\<turnstile> ((init w) \<and> (wnext (\<box> (init (\<not> w))))) = (w \<and> empty) \<frown> (wnext (\<box> (init (\<not> w))))"
   proof -
    have "\<And>p pa. \<turnstile>((p::'a nellist \<Rightarrow> bool) \<and> empty) \<frown> pa = (init p \<and> pa)"
      by (metis InitAndEmptyEqvAndEmpty StateAndEmptySChop inteq_reflection)
    then show ?thesis
      by (simp add: Prop11)
   qed
 show ?thesis
   by (metis "1" "2" "3" inteq_reflection) 
qed

lemma PiFPowerExpandsem:
 " (\<sigma> \<Turnstile> (\<exists>k. (init w) \<Pi> (fpower f k))) = 
   (\<sigma> \<Turnstile> (init w) \<Pi> empty \<or> (\<exists>k. (init w) \<Pi> (f\<frown>(fpower f (k))))) "
proof -
 have 1: "(\<sigma> \<Turnstile> (\<exists>k. (init w) \<Pi> (fpower f k))) =
          (\<exists> k. (\<sigma> \<Turnstile> (init w) \<Pi> (fpower f k)))  "
   by simp 
 have 2: "(\<exists> k. (\<sigma> \<Turnstile> (init w) \<Pi> (fpower f k))) =
          ( (\<sigma> \<Turnstile> (init w) \<Pi> (fpower f 0)) \<or>     (\<exists> k. 1 \<le> k \<and> (\<sigma> \<Turnstile> (init w) \<Pi> (fpower f (k))))) "
   by (metis One_nat_def diff_Suc_1 le_SucE le_add1 plus_1_eq_Suc)
 have 3: "(\<sigma> \<Turnstile> (init w) \<Pi> (fpower f 0)) = (\<sigma> \<Turnstile> (init w) \<Pi> empty)" 
   by (simp add: itl_def fpower_d_def)
 have 4: "(\<exists> k. 1 \<le> k \<and> (\<sigma> \<Turnstile> (init w) \<Pi> (fpower f (k)))) =
          (\<exists> k.  (\<sigma> \<Turnstile> (init w) \<Pi> (fpower f (Suc k))))" 
   by (metis le_add1 ordered_cancel_comm_monoid_diff_class.add_diff_inverse plus_1_eq_Suc)
 have 5: "(\<exists> k.  (\<sigma> \<Turnstile> (init w) \<Pi> (fpower f (Suc k)))) =
          (\<sigma> \<Turnstile> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))))" 
   by simp
 have 6: " ((\<sigma> \<Turnstile> (init w) \<Pi> empty) \<or> (\<sigma> \<Turnstile> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))))) = 
           (\<sigma> \<Turnstile> (init w) \<Pi> empty \<or> (\<exists>k. (init w) \<Pi> (f\<frown>fpower f (k))) )  "
  unfolding fpower_d_def by (auto simp add: schop_d_def) 
 show ?thesis using "1" "2" "3" "4" "5" "6" by blast
qed

lemma PiFPowerExpandsem1:
 " \<forall> \<sigma>. \<sigma> \<Turnstile>  (\<exists>k. (init w) \<Pi> (fpower f k)) = 
     ( (init w) \<Pi> empty   \<or> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))  ) )     "
proof 
  fix \<sigma>
  show "\<sigma> \<Turnstile>  (\<exists>k. (init w) \<Pi> (fpower f k)) = 
              ( (init w) \<Pi> empty   \<or> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))  ) )" 
   proof -
     have 1: "(\<sigma> \<Turnstile>  (\<exists>k. (init w) \<Pi> (fpower f k)) = 
              ( (init w) \<Pi> empty   \<or> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))  ) )) 
               = ( (\<sigma> \<Turnstile>  (\<exists>k. (init w) \<Pi> (fpower f k))) = 
                   (\<sigma> \<Turnstile> (init w) \<Pi> empty   \<or> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))  ) ) )
              " 
       by auto
     have 2: "( (\<sigma> \<Turnstile>  (\<exists>k. (init w) \<Pi> (fpower f k))) = 
                   (\<sigma> \<Turnstile> (init w) \<Pi> empty   \<or> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))  ) ) )" 
       using PiFPowerExpandsem[of "w" "f" "\<sigma>"]  
                 unfolding fpower_d_def by (auto simp add: schop_d_def) 
     show ?thesis using "1" "2" by blast
   qed
qed

lemma PiFPowerExpand:
 " \<turnstile>  (\<exists>k. (init w) \<Pi> (fpower f k)) = 
     ( (init w) \<Pi> empty   \<or> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))  ) )     "
using PiFPowerExpandsem1[of "w" "f" ] by (auto simp add: Valid_def PiFPowerExpandsem1)

lemma exists_expand_sem:
 "(\<sigma> \<Turnstile> (\<exists>k. (fpower ((init w) \<Pi> f \<and> fin w) k))) = 
  ((\<sigma> \<Turnstile> (fpower ((init w) \<Pi> f \<and> fin w) 0)) \<or> 
   (\<sigma> \<Turnstile> (\<exists>k. (fpower ((init w) \<Pi> f \<and> fin w) (Suc k)))))"
by (metis (no_types, lifting) not0_implies_Suc unl_Rex)  

lemma exists_expand:
 "\<turnstile>  (\<exists>k. (fpower ((init w) \<Pi> f \<and> fin w) k)) = 
     ((fpower ((init w) \<Pi> f \<and> fin w) 0) \<or> (\<exists>k. (fpower ((init w) \<Pi> f \<and> fin w) (Suc k))))  "
using exists_expand_sem Valid_def by fastforce



subsubsection \<open>TruePiEqv\<close>

lemma TruePiEqvsem:
 "\<sigma> \<Turnstile> #True \<Pi> f = f"
by (simp add: pi_d_def pifilt_true) (metis zero_enat_def zero_le)

lemma TruePiEqv:
 "\<turnstile> (#True) \<Pi> f = f"
using  TruePiEqvsem by  (auto simp add: Valid_def)

subsubsection \<open>BoxImpEqvPi\<close>

lemma BoxImpEqvPi:
 "\<turnstile> \<box> f \<longrightarrow> g = f \<Pi> g "
 proof (simp add: Valid_def itl_defs  pi_d_def pifilt_def sfxfilt_def )
 show "\<forall>w. (\<forall>n. enat n \<le> nlength w \<longrightarrow> f (ndropn n w)) \<longrightarrow> 
       g w = ((\<exists>i. enat i \<le> nlength w \<and> f (ndropn i w)) \<and> 
              g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns w)))) " 
   proof 
    fix w
    show "(\<forall>n. enat n \<le> nlength w \<longrightarrow> f (ndropn n w)) \<longrightarrow> 
              g w = ((\<exists>i. enat i \<le> nlength w \<and> f (ndropn i w)) \<and> 
                     g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns w))))" 
     proof 
      assume a0: "\<forall>n. enat n \<le> nlength w \<longrightarrow> f (ndropn n w)"
      show "g w = ((\<exists>i. enat i \<le> nlength w \<and> f (ndropn i w)) \<and> 
                   g (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns w))))" 
        proof - 
          have 1: "(nfilter f (ndropns w)) = (ndropns w)" 
            by (metis (mono_tags, lifting) a0 ile0_eq in_nset_ndropns le_cases nfilter_id_conv 
                zero_enat_def)
          have 2: "w = (nmap (\<lambda>s. nnth s 0) (nfilter f (ndropns w)))"
            by (simp add: "1" nmap_first_ndropns)
          show ?thesis by (metis "1" a0 nmap_first_ndropns zero_enat_def zero_le)
        qed
     qed
   qed
qed

subsubsection \<open>PiEqvDiamondUPi\<close>

lemma PiEqvDiamondUPi:
 "\<turnstile> f \<Pi> g = ( \<diamond> f \<and> f \<Pi>\<^sup>u g)"
by (simp add: Valid_def upi_d_def itl_defs pi_d_def,blast)

subsubsection \<open>PiEqvUntilPi\<close>

lemma PiEqvUntilPi:
 "\<turnstile> (init w) \<Pi> g = (init (\<not> w)) \<U> ((init w) \<Pi> g)"
by (metis StatePiUntil UntilUntil int_eq)



subsubsection \<open>UPiEqvBoxOrPi\<close>

lemma UPiEqvBoxOrPi:
 "\<turnstile> f \<Pi>\<^sup>u g = (\<box> (\<not> f) \<or> f \<Pi> g)"
by (simp add: Valid_def upi_d_def itl_defs pi_d_def,blast)

subsection \<open>Theorems\<close>

lemma UPiImpRule:
 assumes "\<turnstile> g1 \<longrightarrow> g2"
 shows   "\<turnstile> f \<Pi>\<^sup>u g1 \<longrightarrow> f \<Pi>\<^sup>u g2"
using assms 
by (meson MP PiK PiN)

lemma UPiEqvRule:
 assumes "\<turnstile> g1 = g2"
 shows   "\<turnstile> f \<Pi>\<^sup>u g1 = f \<Pi>\<^sup>u g2"
proof -
 have 1: "\<turnstile>  g1 \<longrightarrow> g2"
   using assms by (simp add: int_iffD1)
 have 2: "\<turnstile> f \<Pi>\<^sup>u g1 \<longrightarrow> f \<Pi>\<^sup>u g2"
   using "1" UPiImpRule by blast
 have 3: "\<turnstile> g2 \<longrightarrow> g1"
   using assms by (simp add: int_iffD2)
 have 4: "\<turnstile> f \<Pi>\<^sup>u g2 \<longrightarrow> f \<Pi>\<^sup>u g1"
   using "3" UPiImpRule by blast
 from 3 4 show ?thesis 
 by (simp add: "2" int_iffI)
qed

lemma PiEqvNotUPiNot:
 "\<turnstile>  f \<Pi> g = (\<not> (f \<Pi>\<^sup>u (\<not> g)))"
by (simp add: upi_d_def)

lemma NotPiEqvNotUPi:
 "\<turnstile>  f \<Pi> (\<not> g) = (\<not> (f \<Pi>\<^sup>u g))"
by (simp add: upi_d_def)

lemma UPiEqvNotPiNot:
 "\<turnstile> f \<Pi>\<^sup>u g = (\<not> (f \<Pi> (\<not> g)))"
by (simp add: upi_d_def)

lemma NotUPiEqvNotPi:
 "\<turnstile> f \<Pi>\<^sup>u (\<not>g) = (\<not> (f \<Pi> g))"
by (simp add: upi_d_def)

lemma PiImpRule:
 assumes "\<turnstile> g1 \<longrightarrow> g2"
 shows   "\<turnstile> f \<Pi> g1 \<longrightarrow> f \<Pi> g2"
proof -
 have 1: "\<turnstile> \<not> g2 \<longrightarrow> \<not> g1"
   by (simp add: assms) 
 have 2: "\<turnstile> f \<Pi>\<^sup>u (\<not> g2) \<longrightarrow> f \<Pi>\<^sup>u (\<not>g1)"
   using 1 UPiImpRule by blast
 have 3: "\<turnstile> \<not>(f \<Pi>\<^sup>u (\<not>g1)) \<longrightarrow> \<not>(f \<Pi>\<^sup>u (\<not>g2))"
   using 2 by fastforce
 from 3 show ?thesis using PiEqvNotUPiNot by fastforce
qed

lemma PiEqvRule:
 assumes "\<turnstile> g1 = g2"
 shows   "\<turnstile> f \<Pi> g1 = f \<Pi> g2"
proof -
 have 1: "\<turnstile> g1 \<longrightarrow> g2"
   using assms by (simp add: int_iffD1)
 have 2: "\<turnstile> f \<Pi> g1 \<longrightarrow> f \<Pi> g2"
   using "1" PiImpRule by blast
 have 3: "\<turnstile> g2 \<longrightarrow> g1 "
   using assms by (simp add: int_iffD2)
 have 4:"\<turnstile> f \<Pi> g2 \<longrightarrow> f \<Pi> g1"
   using "3" PiImpRule by blast
 from 2 4 show ?thesis by (simp add: int_iffI)
qed

lemma UPiAndPiImpPiAnd:
 "\<turnstile> f1 \<Pi>\<^sup>u f \<and> f1 \<Pi> (\<not> g) \<longrightarrow> f1 \<Pi> (f \<and> \<not>g)"
proof -
 have 1: "\<turnstile> (\<not>(f \<longrightarrow> g)) = (f \<and> \<not>g)"
   by fastforce
 have 2: "\<turnstile> (\<not> (f1 \<Pi>\<^sup>u (f \<longrightarrow>g))) = f1 \<Pi> (\<not>(f \<longrightarrow> g))"
   by (simp add: NotPiEqvNotUPi int_iffD1 int_iffD2 int_iffI)
 have 3: "\<turnstile>  \<not>(  f1 \<Pi>\<^sup>u f \<longrightarrow> f1 \<Pi>\<^sup>u g ) \<longrightarrow> \<not> (f1 \<Pi>\<^sup>u (f \<longrightarrow>g))"
   by (simp add: PiK) 
 have 4: "\<turnstile> (\<not>(  f1 \<Pi>\<^sup>u f \<longrightarrow> f1 \<Pi>\<^sup>u g ) ) = (f1 \<Pi>\<^sup>u f \<and> f1 \<Pi> (\<not> g))"
   using NotPiEqvNotUPi[of "f1" "g"] by fastforce  
 have 5:"\<turnstile> f1 \<Pi> (\<not>(f \<longrightarrow> g)) = f1 \<Pi> (f \<and> \<not>g)" 
   using 1 by (simp add: PiEqvRule)
 from 1 2 3 4 5 show ?thesis by fastforce
qed
 
lemma UPiAndPiImpPiAndA:
 "\<turnstile> f1 \<Pi>\<^sup>u f \<and> f1 \<Pi> g \<longrightarrow> f1 \<Pi> (f \<and> g)"
using UPiAndPiImpPiAnd[of "f1" "f" "LIFT(\<not>g)"] by fastforce

lemma PiAndPiImpPiAnd:
 "\<turnstile> f1 \<Pi> f \<and> f1 \<Pi> g \<longrightarrow> f1 \<Pi> (f \<and> g)"
proof -
 have 1: "\<turnstile>  f1 \<Pi>\<^sup>u f \<and> f1 \<Pi> g \<longrightarrow> f1 \<Pi> (f \<and> g) "
   using UPiAndPiImpPiAndA by fastforce
 have 2: "\<turnstile> f1 \<Pi> f \<longrightarrow> f1 \<Pi>\<^sup>u f"
   using PiDc by blast 
 from 1 2 show ?thesis by fastforce
qed

lemma PiAnd:
 "\<turnstile> f \<Pi> (g1 \<and> g2) = (f \<Pi> g1 \<and> f \<Pi> g2)"
proof - 
 have 1: "\<turnstile> f \<Pi> (g1 \<and> g2) \<longrightarrow> f \<Pi> g1"
   by (meson PiImpRule Prop12 int_iffD1 lift_and_com)
 have 2: "\<turnstile> f \<Pi> (g1 \<and> g2) \<longrightarrow> f \<Pi> g2"
   by (meson PiImpRule Prop12 int_iffD1 lift_and_com)
 have 3: "\<turnstile> f \<Pi> (g1 \<and> g2) \<longrightarrow> f \<Pi> g1 \<and> f \<Pi> g2" 
   using 1 2 by fastforce
 have 4: "\<turnstile> f \<Pi> g1 \<and> f \<Pi> g2 \<longrightarrow> f \<Pi> (g1 \<and> g2)"
   by (simp add: PiAndPiImpPiAnd)
 from  3 4 show ?thesis by fastforce
qed

lemma UPiAnd:
 "\<turnstile> f \<Pi>\<^sup>u (g1 \<and> g2) = ( f \<Pi>\<^sup>u g1 \<and> f \<Pi>\<^sup>u g2 )"
proof -
 have 1: "\<turnstile> f \<Pi> (\<not>g1 \<or> \<not> g2) = (f \<Pi> (\<not>g1) \<or> f \<Pi> (\<not> g2)) "
   by (simp add: PiOr)
 have 2: "\<turnstile> (\<not>(f \<Pi> (\<not>g1 \<or> \<not> g2))) = (\<not>(f \<Pi> (\<not>g1) \<or> f \<Pi> (\<not> g2)))"
   using 1 by fastforce
 have 3: "\<turnstile> (\<not>(f \<Pi> (\<not>g1 \<or> \<not> g2))) = f \<Pi>\<^sup>u (\<not>(\<not>g1 \<or> \<not> g2))"
   by (meson NotUPiEqvNotPi Prop11) 
 have 4: "\<turnstile> (\<not>(\<not>g1 \<or> \<not> g2)) = (g1 \<and> g2)"
   by fastforce
 have 5: "\<turnstile> f \<Pi>\<^sup>u (\<not>(\<not>g1 \<or> \<not> g2)) = f \<Pi>\<^sup>u (g1 \<and> g2)"
   using 4 by (simp add: UPiEqvRule)
 have 6: "\<turnstile> (\<not>(f \<Pi> (\<not>g1) \<or> f \<Pi> (\<not> g2))) = (\<not>(f \<Pi> (\<not>g1)) \<and> \<not>(f \<Pi>(\<not>g2)))"
   by fastforce
 have 7: "\<turnstile>\<not>(f \<Pi> (\<not>g1)) = f \<Pi>\<^sup>u g1"
   by (simp add: NotPiEqvNotUPi)  
 have 8: "\<turnstile> \<not>(f \<Pi> (\<not>g2)) = f \<Pi>\<^sup>u g2 "
   by (simp add: NotPiEqvNotUPi)  
 have 9: "\<turnstile> (\<not>(f \<Pi> (\<not>g1)) \<and> \<not>(f \<Pi>(\<not>g2))) = (f \<Pi>\<^sup>u g1 \<and> f \<Pi>\<^sup>u g2)"
   using 6 7 8 by fastforce
 show ?thesis by (metis "2" "3" "5" "6" "9" inteq_reflection) 
qed 

lemma UPiOr: 
 "\<turnstile> f \<Pi>\<^sup>u (g1 \<or> g2) = ( f \<Pi>\<^sup>u g1 \<or> f \<Pi>\<^sup>u g2) " 
proof -
 have 1: "\<turnstile> f \<Pi> (\<not>g1 \<and> \<not>g2) = (f \<Pi> (\<not>g1) \<and> f \<Pi> (\<not>g2)) " 
   by (simp add: PiAnd)
 have 2: "\<turnstile> (\<not>(f \<Pi> (\<not>g1 \<and> \<not>g2)))  = (\<not>(f \<Pi> (\<not>g1) \<and> f \<Pi> (\<not>g2))) " 
   using 1 by fastforce
 have 3: "\<turnstile> (\<not>(f \<Pi> (\<not>g1 \<and> \<not>g2))) = f \<Pi>\<^sup>u (\<not>(\<not>g1 \<and> \<not> g2)) " 
     by (meson NotUPiEqvNotPi Prop11)
 have 4: "\<turnstile> (\<not>(\<not>g1 \<and> \<not> g2)) = (g1 \<or> g2) "
    by fastforce
 have 5: "\<turnstile> f \<Pi>\<^sup>u (\<not>(\<not>g1 \<and> \<not> g2)) = f \<Pi>\<^sup>u (g1 \<or> g2)"
   using "4" UPiEqvRule by blast
 have 6: "\<turnstile> (\<not>(f \<Pi> (\<not>g1) \<and> f \<Pi> (\<not>g2))) = (\<not>(f \<Pi> (\<not>g1)) \<or> \<not>(f \<Pi> (\<not>g2))) " 
   by fastforce
 have 7: "\<turnstile> (\<not>(f \<Pi> (\<not>g1))) = f \<Pi>\<^sup>u g1 "
     by (simp add: upi_d_def)
 have 8: "\<turnstile> (\<not>(f \<Pi> (\<not>g2))) = f \<Pi>\<^sup>u g2 "
   by (simp add: upi_d_def)
 have 9: "\<turnstile> (\<not>(f \<Pi> (\<not>g1)) \<or> \<not>(f \<Pi> (\<not>g2))) = ( f \<Pi>\<^sup>u g1 \<or> f \<Pi>\<^sup>u g2)"
   using 7 8 by fastforce
 show ?thesis 
 by (metis "2" "3" "4" "6" "9" inteq_reflection )
qed


lemma UpiAndImp:
 "\<turnstile> f \<Pi>\<^sup>u (g1 \<longrightarrow> g2) \<and> f \<Pi> g1 \<longrightarrow> f \<Pi> g2"
proof -
 have 2: "\<turnstile> f \<Pi>\<^sup>u ((\<not>g2) \<longrightarrow> (\<not>g1)) \<longrightarrow> ( f \<Pi>\<^sup>u (\<not> g2) \<longrightarrow> f \<Pi>\<^sup>u (\<not> g1) )"
   using PiK by blast
 have 3: "\<turnstile> (f \<Pi>\<^sup>u (\<not>g2) \<longrightarrow> f \<Pi>\<^sup>u (\<not>g1)) = ((\<not> (f \<Pi>\<^sup>u (\<not>g1))) \<longrightarrow> (\<not>(f \<Pi>\<^sup>u (\<not>g2))))"
   by auto
 have 4: "\<turnstile> (\<not> (f \<Pi>\<^sup>u (\<not>g2))) = f \<Pi> g2" 
   by (simp add: upi_d_def)
 have 5: "\<turnstile> (\<not> (f \<Pi>\<^sup>u (\<not>g1))) = f \<Pi> g1" 
   by (simp add: upi_d_def)
 have 6: "\<turnstile> f \<Pi>\<^sup>u ((\<not>g2) \<longrightarrow> (\<not>g1)) = f \<Pi>\<^sup>u (g1 \<longrightarrow> g2)"
   by simp
 have 7: "\<turnstile> (f \<Pi>\<^sup>u (g1 \<longrightarrow> g2) \<and> f \<Pi> g1 \<longrightarrow> f \<Pi> g2) =
            (f \<Pi>\<^sup>u (g1 \<longrightarrow> g2)  \<longrightarrow> (f \<Pi> g1 \<longrightarrow> f \<Pi> g2))  "
   by auto
 show ?thesis 
   using "2" "4" "5" by fastforce
qed

lemma BoxImpUPiBox:
 "\<turnstile> \<box> (init w) \<longrightarrow> f \<Pi>\<^sup>u (\<box> (init w))"
proof -
 have 1: "\<turnstile> f \<Pi> (\<diamond> (init (\<not>w))) \<longrightarrow> \<diamond> (init (\<not>w))"
   by (simp add: PiDiamondImp) 
 have 2: "\<turnstile> \<not>  \<diamond> (init (\<not>w)) \<longrightarrow> \<not> (f \<Pi> (\<diamond> (init (\<not>w))))" 
   using "1" by auto
 have 3: "\<turnstile> (\<not>  \<diamond> (init (\<not>w))) = \<box> (init w)" 
   by (metis "2" Initprop(2) Prop10 always_d_def inteq_reflection)
 have 4: "\<turnstile> (\<not> (f \<Pi> (\<diamond> (init (\<not>w))))) = f \<Pi>\<^sup>u  (\<box> (init w))" 
   by (simp add: upi_d_def)
      (metis "3" int_simps(4) inteq_reflection) 
 show ?thesis 
 using "2" "3" "4" by fastforce
qed

lemma WPrevPi:
 "\<turnstile> (init w) \<Pi> f = (wprev (\<box> (init (\<not> w))))\<frown> ((init w) \<and> (init w) \<Pi> f)"
   using StatePiUntil StatePiUntil1 by fastforce

lemma  EmptyAndSChopAndMoreEqvAndSChop: 
 "\<turnstile> (w \<and> empty)\<frown>(f \<and> more) = ((w \<and> empty)\<frown>f \<and> more) "
proof -
 have 1: "\<turnstile> (w \<and> empty)\<frown>(f \<and> more) \<longrightarrow> (w \<and> empty)\<frown>f "
   by (simp add: SChopAndA) 
 have 2: "\<turnstile> (w \<and> empty)\<frown>(f \<and> more) \<longrightarrow> more"
   by (meson SChopAndB SChopMoreImpMore lift_imp_trans) 
 have 3: "\<turnstile>((w \<and> empty)\<frown>f \<and> more) \<longrightarrow> (w \<and> empty)\<frown>(f \<and> more)"
   by (metis (no_types, opaque_lifting) InitAndEmptyEqvAndEmpty Prop11 Prop12 StateAndEmptySChop 
       int_simps(1) inteq_reflection)
 show ?thesis
   by (simp add: "1" "2" "3" Prop12 int_iffI)
qed  
   

lemma PiInfImpInf:
 "\<turnstile> f \<Pi> inf \<longrightarrow> inf" 
unfolding Valid_def pi_d_def init_defs infinite_defs
by auto
   (metis enat_ile nfinite_conv_nlength_enat pifilt_nlength_bound)


lemma DiamondWPrevBoxSChop: 
 "\<turnstile> \<diamond> (init w) = (wprev (\<box> (init (\<not> w))))\<frown> (init w)" 
by (metis Initprop(2) StateUntilEqvWPrevChop UntilRule inteq_reflection)



lemma PiChopDist1:
 "\<turnstile> ((init w) \<Pi> (f;g) \<or> ((init w) \<Pi> (f \<and> finite) \<and> inf)) = 
    (((init w) \<Pi> f);((init w) \<and>  (init w) \<Pi> g) )" 
proof -
 have 1: "\<turnstile> f;g = (f\<frown>g \<or> (f \<and> inf))"
   by (simp add: ChopSChopdef) 
 have 2: "\<turnstile> (init w) \<Pi> (f;g) = ((init w) \<Pi> (f\<frown>g) \<or> (init w) \<Pi> (f \<and> inf))" 
   by (metis "1" PiOr int_eq)
 have 3: "\<turnstile> (init w) \<Pi> (f\<frown>g) = ((init w) \<Pi> f)\<frown>((init w) \<and> (init w) \<Pi> g)" 
   by (simp add: PiSChopDist)
 have 4: "\<turnstile> ((init w) \<Pi> f);((init w) \<and>  (init w) \<Pi> g) =
            (((init w) \<Pi> f)\<frown>((init w) \<and> (init w) \<Pi> g) \<or> ((init w) \<Pi> f \<and> inf)) " 
  by (simp add: ChopSChopdef)
 have 41: "\<turnstile> f = (f \<and> (finite \<or> inf)) "
   unfolding finite_d_def by fastforce
 have 5: "\<turnstile> (init w) \<Pi> f =  (init w) \<Pi> (f \<and> (finite \<or> inf))"
    by (simp add: "41" PiEqvRule)
 have 6: "\<turnstile> ((init w) \<Pi> f \<and> inf) = 
            (((init w) \<Pi> (f \<and> finite) \<and> inf) \<or> ((init w) \<Pi> (f \<and> inf) \<and> inf))"
    by (metis AndInfEqvChopFalse OrChopEqvRule OrFiniteInf PiOr inteq_reflection)
 have 7: "\<turnstile> ((init w) \<Pi> (f \<and> inf) \<and> inf) = ((init w) \<Pi> (f \<and> inf))"
   by (metis EmptySChop PiImpRule PiInfImpInf Prop01 Prop04 Prop05 Prop10 int_iffD1 lift_imp_trans)
 show ?thesis 
 using "2" "3" "4" "6" "7" by fastforce
qed


lemma PiChopDist2:
 "\<turnstile> ((init w) \<Pi> (f;g)) = 
    (((init w) \<Pi> (f \<and> finite) )\<frown>((init w) \<and> (init w) \<Pi> g) \<or> ((init w) \<Pi> (f \<and> inf) \<and> inf))"
proof -
 have 1: "\<turnstile> (init w) \<Pi> (f;g) = ((init w) \<Pi> (f\<frown>g) \<or> (init w) \<Pi> (f \<and> inf))" 
  by (metis ChopSChopdef PiOr inteq_reflection)
 have 2: "\<turnstile> (init w) \<Pi> (f\<frown>g) = ((init w) \<Pi> f)\<frown>((init w) \<and> (init w) \<Pi> g)"
   by (simp add: PiSChopDist) 
 have 3: "\<turnstile> ((init w) \<Pi> f)\<frown>((init w) \<and> (init w) \<Pi> g) =
            ((init w) \<Pi> f \<and> finite);((init w) \<and> (init w) \<Pi> g)"
   by (simp add: schop_d_def) 
 have 31: "\<turnstile> f = ( (f \<and> finite) \<or> (f \<and> inf)) " 
   unfolding finite_d_def by fastforce
 have 32: "\<turnstile> (init w) \<Pi> f = (init w) \<Pi> ( (f \<and> finite) \<or> (f \<and> inf)) "
   using "31" PiEqvRule by blast
 have 33: "\<turnstile> (init w) \<Pi> ( (f \<and> finite) \<or> (f \<and> inf)) = ((init w) \<Pi> (f \<and> finite) \<or> (init w) \<Pi> (f \<and> inf))"
   by (simp add: PiOr)
 have 34: "\<turnstile> \<not>((init w) \<Pi> (f \<and> inf) \<and> finite)  "
   unfolding finite_d_def using PiAnd[of "LIFT init w" f "LIFT inf" ] PiInfImpInf[of "LIFT init w"]
    by fastforce
 have 4: "\<turnstile> ((init w) \<Pi> f \<and> finite) = ((init w) \<Pi> (f \<and> finite) \<and> finite) "
   using 32 33 34 by fastforce
 have 5: "\<turnstile> (init w) \<Pi> (f \<and> inf) = ((init w) \<Pi> (f \<and> inf) \<and> inf)"
   by (metis PiAnd PiInfImpInf Prop10 Prop12 int_eq int_iffD2) 
 show ?thesis 
 by (metis "1" "2" "4" "5" int_eq schop_d_def)
qed
 
lemma InfEqvNotForallNotLen: 
 "\<turnstile> inf = (\<forall>n. \<not> (len n))" 
proof -
 have 1: "\<turnstile> finite = (\<exists>n. len n)" 
  by (simp add: Finite_exist_len)
 have 2: "\<turnstile> (\<not> finite) = (\<not>(\<exists>n. len n))"  
  using 1 
  by (metis NotEqvYieldsMore int_eq)
 have 3: "\<turnstile> (\<not>(\<exists>n. len n)) = (\<forall>n. \<not> (len n))"
   by fastforce
 show ?thesis using 2 3 by (metis InfEqvNotFinite int_eq)
qed

lemma PiEx: 
 "\<turnstile> f \<Pi> (\<exists>n. g n) = (\<exists>n. f \<Pi> (g n)) " 
unfolding Valid_def by (auto simp add: pi_d_def)

lemma PiAll:
 "\<turnstile> f \<Pi> (\<forall>n. g n) = (\<forall>n. f \<Pi> (g n)) " 
unfolding Valid_def by (auto simp add: pi_d_def)




lemma PiLenSuc:
 "\<turnstile> (init w) \<Pi> (len (Suc n)) = ((init w) \<Pi> skip)\<frown>( (init w) \<and> (init w) \<Pi> (len n) ) " 
proof -
 have 1: "\<turnstile> len (Suc n) = skip\<frown>(len n) " 
     by (metis NextSChopdef len_d_def next_d_def wpow_Suc)
 have 2: "\<turnstile> (init w) \<Pi> (skip\<frown>(len n)) = ((init w) \<Pi> skip)\<frown>( (init w) \<and> (init w) \<Pi> (len n) )"
  by (simp add: PiSChopDist)
 show ?thesis by (metis "1" "2" int_eq)
qed


lemma PiImpDiamond: 
 "\<turnstile> f \<Pi> g \<longrightarrow> \<diamond> f" 
by (meson PiEqvDiamondUPi Prop12 int_iffD1)




lemma PiMPA: 
assumes "\<turnstile> f \<Pi> g1"
        "\<turnstile> f \<Pi> (g1 \<longrightarrow> g2)"
shows   "\<turnstile> f \<Pi> g2" 
using assms
by (meson MP PiDc Prop09 UpiAndImp) 

lemma PiMPB: 
assumes "\<turnstile> f \<Pi> g1"
        "\<turnstile> g1 \<longrightarrow> g2"
shows   "\<turnstile> f \<Pi> g2" 
using assms 
by (metis MP PiAnd Prop10 Prop12 int_iffD2 inteq_reflection)


lemma PiMPBC: 
 assumes "\<turnstile> f1 \<longrightarrow> f2 \<Pi> g1" 
         "\<turnstile> g1 \<longrightarrow> g2 " 
 shows "\<turnstile> f1 \<longrightarrow> f2 \<Pi> g2" 
using assms
by (meson PiImpRule lift_imp_trans) 


lemma SlideInitSFin: 
 "\<turnstile> f\<frown>( (init w) \<and> g) = (f \<and> sfin w)\<frown>g "
proof -
 have 1: "\<turnstile> sfin (init w) = sfin ( w)" 
   by (metis InitAndEmptyEqvAndEmpty SFinEqvTrueSChopAndEmpty int_eq)
 show ?thesis
 by (metis "1" AndSFinSChopEqvStateAndSChop inteq_reflection) 
qed

lemma UPiSChop: 
 "\<turnstile> (init w) \<Pi>\<^sup>u (f\<frown>g) = (\<box>(init (\<not>w)) \<or> ((init w) \<Pi> f \<and> sfin w)\<frown>( (init w) \<Pi> g)) " 
proof -
 have 1: "\<turnstile> (init w) \<Pi>\<^sup>u (f\<frown>g) = (\<box> (init (\<not>w)) \<or> (init w) \<Pi> (f\<frown>g)) " 
   by (metis Initprop(2) UPiEqvBoxOrPi int_eq)
 have 2: "\<turnstile> (init w) \<Pi> (f\<frown>g) = ((init w) \<Pi> f)\<frown>( (init w) \<and> (init w) \<Pi> g) " 
   by (simp add: PiSChopDist) 
 have 3: "\<turnstile> ((init w) \<Pi> f)\<frown>( (init w) \<and> (init w) \<Pi> g) =
            ((init w) \<Pi> f \<and> sfin w)\<frown>((init w) \<Pi> g) " 
  by (simp add: SlideInitSFin)
 show ?thesis 
 using "1" "2" "3" by (meson Prop06)
qed


lemma PiUPiSChop: 
 "\<turnstile> ((init w) \<Pi> f \<and> sfin w)\<frown>( (init w) \<Pi> g) = ((init w) \<Pi>\<^sup>u f \<and> sfin w)\<frown>( (init w) \<Pi>\<^sup>u g) "
proof -
 have 1: "\<turnstile> ((init w) \<Pi> f) = (\<diamond> (init w) \<and> (init w) \<Pi>\<^sup>u f) "
   by (simp add: PiEqvDiamondUPi)
 have 2: "\<turnstile> ((init w) \<Pi> f \<and> sfin w) = (\<diamond> (init w) \<and> (init w) \<Pi>\<^sup>u f \<and> sfin w) " 
    using 1 by auto
 have 3: "\<turnstile> (\<diamond> (init w) \<and> sfin w) = sfin w " 
     by (metis InitAndEmptyEqvAndEmpty Prop10 SChopAndA SFinEqvTrueSChopAndEmpty TrueSChopEqvDiamond 
          inteq_reflection lift_and_com)
 have 4: "\<turnstile> (\<diamond> (init w) \<and> (init w) \<Pi>\<^sup>u f \<and> sfin w) = ( (init w) \<Pi>\<^sup>u f \<and> sfin w) "
   using 3 by auto
 have 5: "\<turnstile> ( (init w) \<Pi> g) =  (\<diamond> (init w) \<and> (init w) \<Pi>\<^sup>u g) "
    by (simp add: PiEqvDiamondUPi)
 have 6: "\<turnstile> (init w \<and>  (init w) \<Pi> g) =  (init w \<and> \<diamond> (init w) \<and> (init w) \<Pi>\<^sup>u g) "
    using 5 by auto
 have 7: "\<turnstile> (init w \<and> \<diamond> (init w)) = init w " 
    by (meson NowImpDiamond Prop10 Prop11)
 have 8: "\<turnstile> (init w \<and> \<diamond> (init w) \<and> (init w) \<Pi>\<^sup>u g) = (init w \<and> (init w) \<Pi>\<^sup>u g) " 
  using 7 by auto
 show ?thesis 
 by (metis "2" "4" "6" "8" SlideInitSFin inteq_reflection)
qed




lemma PiFiniteAbsorb: 
"\<turnstile> (g \<Pi> (f \<and> finite  ) \<and> finite) = (g \<Pi> f \<and> finite) " 
proof -
 have 1: "\<turnstile> (g \<Pi> (f \<and> finite  ) \<and> finite) \<longrightarrow> (g \<Pi> f \<and> finite)"
   by (metis FiniteImp PiAnd PiAndPiImpPiAnd Prop01 Prop05 Prop12 inteq_reflection lift_and_com)
 have 31: "\<turnstile> f = ( (f \<and> finite) \<or> (f \<and> inf)) " 
   unfolding finite_d_def by fastforce
 have 32: "\<turnstile> g \<Pi> f = g  \<Pi> ( (f \<and> finite) \<or> (f \<and> inf)) "
   using "31" PiEqvRule by blast
 have 33: "\<turnstile> g \<Pi> ( (f \<and> finite) \<or> (f \<and> inf)) = (g \<Pi> (f \<and> finite) \<or> g \<Pi> (f \<and> inf))"
   by (simp add: PiOr)
 have 34: "\<turnstile> \<not>(g \<Pi> (f \<and> inf) \<and> finite)  "
   unfolding finite_d_def using PiAnd[of g f "LIFT inf" ] PiInfImpInf[of g]
    by fastforce
 have 4: "\<turnstile> (g \<Pi> f \<and> finite) = (g \<Pi> (f \<and> finite) \<and> finite) "
   using 32 33 34 by fastforce
 have 2: "\<turnstile> (g \<Pi> f \<and> finite) \<longrightarrow> (g \<Pi> (f \<and> finite  ) \<and> finite) "
  by (simp add: "4" int_iffD1)
 show ?thesis using 1 2 by fastforce
qed

lemma PiInfAbsorb: 
"\<turnstile> (g \<Pi> (f \<and> inf  ) \<and> inf) = (g \<Pi> (f \<and> inf)) "
proof -
 have 1: "\<turnstile> (g \<Pi> (f \<and> inf  ) \<and> inf) \<longrightarrow> (g \<Pi> (f \<and> inf)) "
   by fastforce
 have 2: "\<turnstile> (g \<Pi> (f \<and> inf)) \<longrightarrow> (g \<Pi> (f \<and> inf  ) \<and> inf)"
   by (metis PiAnd PiInfImpInf Prop10 Prop12 int_iffD1 lift_imp_trans)
 show ?thesis using 1 2 by fastforce
qed



lemma PiMoreImpMore: 
 "\<turnstile> (g \<Pi>  more ) \<longrightarrow> more"
unfolding Valid_def pi_d_def itl_defs pifilt_def sfxfilt_def 
by auto 
  (metis enat_min_eq_0_iff length_nfilter_le min_def ndropns_nlength)


lemma PiMoreAbsorb: 
"\<turnstile> (g \<Pi> (f \<and> more  ) \<and> more) = (g \<Pi> (f \<and> more) ) " 
proof -
 have 1: "\<turnstile> (g \<Pi> (f \<and> more  ) \<and> more) \<longrightarrow> (g \<Pi> (f \<and> more) ) "
   by auto
 have 2: "\<turnstile> (g \<Pi> (f \<and> more) ) \<longrightarrow> (g \<Pi> (f \<and> more  ) \<and> more) "
   by (metis PiMoreImpMore PiAnd Prop10 Prop12 int_iffD1 inteq_reflection)
 show ?thesis using 1 2 by fastforce
qed


lemma EmptyImpPiEmpty: 
 "\<turnstile> (g \<Pi> f \<and> empty ) \<longrightarrow> (g \<Pi> empty) "
unfolding Valid_def pi_d_def itl_defs pifilt_def sfxfilt_def 
by auto 
   (metis ile0_eq length_nfilter_le ndropns_nlength)

lemma PiEmptyAbsorb: 
"\<turnstile> (g \<Pi> (f \<and> empty) \<and> empty) = (g \<Pi> f \<and> empty ) "
proof -
 have 1: "\<turnstile>  (g \<Pi> (f \<and> empty) \<and> empty) \<longrightarrow> (g \<Pi> f \<and> empty ) "
    using PiAnd by fastforce
 have 2: "\<turnstile> (g \<Pi> f \<and> empty ) \<longrightarrow> (g \<Pi> (f \<and> empty) \<and> empty) "
   using EmptyImpPiEmpty PiAnd by fastforce
 show ?thesis using 1 2 by fastforce
qed



lemma SlideInitSFinVar1: 
 "\<turnstile> (init w) \<Pi> (((F \<and> more) \<and> finite);X) = 
     (((init w) \<Pi> (F \<and> more) \<and> sfin w) \<and> finite);((init w) \<Pi> X)" 
proof -
 have 1: "\<turnstile> (init w) \<Pi> (((F \<and> more) \<and> finite);X) =
            ((init w) \<Pi> (F \<and> more) \<and> finite);((init w) \<and> (init w) \<Pi> X) " 
    by (metis PiSChopDist schop_d_def)
 have 2: "\<turnstile> ((init w) \<Pi> (F \<and> more) \<and> finite);((init w) \<and> (init w) \<Pi> X) =
            (((init w) \<Pi> (F \<and> more) \<and> sfin w) \<and> finite);( (init w) \<Pi> X) " 
  using SlideInitSFin[of  "LIFT (init w) \<Pi> (F \<and> more)" w "LIFT (init w) \<Pi> X"] unfolding schop_d_def  
    by blast
 show ?thesis using 1 2 by (metis int_eq)
qed
 


lemma UPiAbsorp: 
 "\<turnstile> (((init w) \<Pi> (F \<and> more) \<and> sfin w) \<and> finite);((init w) \<Pi>\<^sup>u X) =
    (((init w) \<Pi> (F \<and> more) \<and> sfin w) \<and> finite);((init w) \<Pi> X) " 
proof -
 have 0: "\<turnstile> ((init w) \<Pi>\<^sup>u X) = (\<box> (init (\<not>w)) \<or>  (init w) \<Pi> X) "
 by (metis Initprop(2) UPiEqvBoxOrPi inteq_reflection)
 have 1: "\<turnstile> (((init w) \<Pi> (F \<and> more) \<and> sfin w) \<and> finite);((init w) \<Pi>\<^sup>u X) =
            (((init w) \<Pi> (F \<and> more) \<and> sfin w) \<and> finite);(\<box> (init (\<not>w)) \<or>  (init w) \<Pi> X) "
            using "0" RightChopEqvChop by blast 
 have 2: "\<turnstile> (((init w) \<Pi> (F \<and> more) \<and> sfin w) \<and> finite);(\<box> (init (\<not>w)) \<or>  (init w) \<Pi> X) =
            (((init w) \<Pi> (F \<and> more)) \<and> finite);((init w) \<and> (\<box> (init (\<not>w)) \<or>  (init w) \<Pi> X)) " 
   using SlideInitSFin[of  "LIFT (init w) \<Pi> (F \<and> more)" w "LIFT (\<box> (init (\<not>w)) \<or>  (init w) \<Pi> X)" ]
    unfolding schop_d_def 
    by (metis inteq_reflection)
 have 3: "\<turnstile> ((init w) \<and> (\<box> (init (\<not>w)) \<or>  (init w) \<Pi> X)) =
            ((init w) \<and> (  (init w) \<Pi> X)) " 
    using BoxElim Initprop(2) by fastforce
 have 4: "\<turnstile> (((init w) \<Pi> (F \<and> more)) \<and> finite);((init w) \<and> (\<box> (init (\<not>w)) \<or>  (init w) \<Pi> X)) =
            (((init w) \<Pi> (F \<and> more)) \<and> finite);((init w) \<and>   (init w) \<Pi> X) "
     using "3" RightChopEqvChop by blast
 have 5: "\<turnstile> (((init w) \<Pi> (F \<and> more)) \<and> finite);((init w) \<and>   (init w) \<Pi> X) =
            (((init w) \<Pi> (F \<and> more) \<and> sfin w) \<and> finite);((init w) \<Pi> X) "
     using SlideInitSFin[of "LIFT ((init w) \<Pi> (F \<and> more))" w "LIFT (init w) \<Pi> X"]
      unfolding schop_d_def 
      by auto
 show ?thesis 
 by (metis "1" "2" "4" "5" inteq_reflection)
qed



lemma PiStateFinite: 
 "\<turnstile> (init w) \<Pi> finite = \<diamond> ((init w) \<and> (wnext (\<box> (init (\<not>w)))))"
proof -
 have 1: "\<turnstile>  (init w) \<Pi> finite = (init w) \<Pi> (#True\<frown>empty)"
  by (metis DiamondEmptyEqvFinite DiamondSChopdef PiEqvRule int_eq)    
 have 2: "\<turnstile>  (init w) \<Pi> (#True\<frown>empty) = ((init w) \<Pi> #True) \<frown>((init w) \<and> (init w) \<Pi> empty) " 
   by (simp add: PiSChopDist)       
 have 3: "\<turnstile> ((init w) \<and> (init w) \<Pi> empty) = ((init w) \<and> wnext (\<box> (init (\<not>w)))) " 
    by (metis InitAndEmptyEqvAndEmpty StateAndEmptySChop StateAndPiEmpty inteq_reflection)
 have 4: "\<turnstile> (init w) \<Pi> #True = \<diamond> (init w) " 
     by (simp add: PiTrueEqvDiamond) 
 have 5: "\<turnstile> ((init w) \<Pi> #True) \<frown>((init w) \<and> (init w) \<Pi> empty) = 
            ( \<diamond> (init w))\<frown> ((init w) \<and> wnext (\<box> (init (\<not>w))))"
   by (simp add: "3" "4" SChopEqvSChop)
 have 6: "\<turnstile> ( \<diamond> (init w))\<frown> ((init w) \<and> wnext (\<box> (init (\<not>w)))) = 
            ( \<diamond> (init w) \<and> sfin w)\<frown>(wnext (\<box> (init (\<not>w)))) " 
   by (simp add: SlideInitSFin)
 have 7: "\<turnstile> ( \<diamond> (init w) \<and> sfin w) = sfin w "
    by (metis DiamondSChopdef InitAndEmptyEqvAndEmpty Prop10 SChopAndA SFinEqvTrueSChopAndEmpty 
        inteq_reflection lift_and_com)
 have 8: "\<turnstile> ( \<diamond> (init w) \<and> sfin w)\<frown>(wnext (\<box> (init (\<not>w)))) =
            (sfin w) \<frown>(wnext (\<box> (init (\<not>w)))) "
     using "7" LeftSChopEqvSChop by blast
 have 9: "\<turnstile> (sfin w) \<frown>(wnext (\<box> (init (\<not>w)))) = #True \<frown> ((init w) \<and> (wnext (\<box> (init (\<not>w))))) " 
     by (metis SlideInitSFin int_eq int_simps(17))
 show ?thesis 
 by (metis "1" "2" "3" "4" "6" "7" "9" TrueSChopEqvDiamond inteq_reflection)
qed


lemma PiStateInf: 
 "\<turnstile> (init w) \<Pi> inf = ( \<diamond> (init w) \<and> \<box> ( (init w) \<longrightarrow> \<circle> (\<diamond> (init w))))"
proof -
 have 1: "\<turnstile> (init w) \<Pi> inf = (init w) \<Pi> (\<not> finite) " 
   by (simp add: InfEqvNotFinite PiEqvRule)
 have 2: "\<turnstile> (init w) \<Pi> (\<not> finite) = (\<not> ( (init w) \<Pi>\<^sup>u finite)) " 
   by (simp add: NotPiEqvNotUPi)
 have 3: "\<turnstile> ( (init w) \<Pi>\<^sup>u finite) = (\<box> (init (\<not> w)) \<or> (init w) \<Pi> finite) " 
   by (metis Initprop(2) UPiEqvBoxOrPi inteq_reflection)
 have 4: "\<turnstile> (\<box> (init (\<not> w)) \<or> (init w) \<Pi> finite) = 
            (\<box> (init (\<not> w)) \<or> \<diamond> ((init w) \<and> (wnext (\<box> (init (\<not>w)))))) "
   using PiStateFinite by fastforce
 have 5: "\<turnstile> (\<not> ( (init w) \<Pi>\<^sup>u finite)) = 
            (\<not> (\<box> (init (\<not> w)) \<or> \<diamond> ((init w) \<and> (wnext (\<box> (init (\<not>w))))))) " 
    using "3" "4" by fastforce
 have 6: "\<turnstile> (\<not> (\<box> (init (\<not> w)) \<or> \<diamond> ((init w) \<and> (wnext (\<box> (init (\<not>w))))))) =
            ( \<not>(\<box> (init (\<not> w))) \<and> \<not>(\<diamond> ((init w) \<and> (wnext (\<box> (init (\<not>w))))))) "
   by force
 have 7: "\<turnstile> (\<not>(\<box> (init (\<not> w)))) = \<diamond> (init w) " 
    unfolding always_d_def 
    by (metis Initprop(2) int_simps(4) inteq_reflection)
 have 8: "\<turnstile> (\<not>(\<diamond> ((init w) \<and> (wnext (\<box> (init (\<not>w))))))) =
            \<box>( \<not> ((init (w)) \<and> (wnext (\<box> (init (\<not>w)))))) "
      by (simp add: always_d_def)
 have 9: "\<turnstile> ( \<not> ((init (w)) \<and> (wnext (\<box> (init (\<not>w)))))) =
            (\<not>(init ( w)) \<or> \<not> (wnext (\<box> (init (\<not>w)))))" 
      by force
 have 10: "\<turnstile> (\<not>(init ( w))) = (init (\<not> w)) " 
    by (simp add: Initprop(2))
 have 11: "\<turnstile> (\<not> (wnext (\<box> (init (\<not>w))))) = \<circle>(\<not> ((\<box> (init (\<not>w)))))"
   by (simp add: wnext_d_def)
 have 12: "\<turnstile> \<circle>(\<not> ((\<box> (init (\<not>w))))) = \<circle> (\<diamond> (init w)) " 
    by (simp add: "7" NextEqvNext)
 have 13: "\<turnstile> (\<not>(init ( w)) \<or> \<not> (wnext (\<box> (init (\<not>w))))) = ( (init  w) \<longrightarrow> \<circle> (\<diamond> (init w))) "
   using "10" "11" "12" by fastforce
 have 14: "\<turnstile> \<box>( \<not> ((init (w)) \<and> (wnext (\<box> (init (\<not>w)))))) =
             \<box> ( (init w) \<longrightarrow> \<circle> (\<diamond> (init w))) "
     by (metis "13" "8" "9" inteq_reflection)
 have 15: "\<turnstile> ( \<not>(\<box> (init (\<not> w))) \<and> \<not>(\<diamond> ((init w) \<and> (wnext (\<box> (init (\<not>w))))))) =
             ( \<diamond> (init w) \<and> \<box> ( (init  w) \<longrightarrow> \<circle> (\<diamond> (init w)))) " 
      by (metis "13" "6" "7" "8" "9" inteq_reflection)
 show ?thesis
 by (metis "1" "15" "2" "5" "6" int_eq)
qed 


lemma UPiPiState: 
 "\<turnstile> (init w) \<Pi>\<^sup>u ( (init w) \<Pi> f) = (init w) \<Pi>\<^sup>u f " 
proof -
 have 1: "\<turnstile> (init w) \<Pi>\<^sup>u ( (init w) \<Pi> f) = (\<box>(init (\<not>w)) \<or> (init w) \<Pi> ( (init w) \<Pi> f)) " 
   by (metis Initprop(2) UPiEqvBoxOrPi int_eq)
 have 2: "\<turnstile> (init w) \<Pi> ( (init w) \<Pi> f) =  ( (init w) \<Pi> f) " 
   by (metis PiAssoc Prop10 StateEqvBi StateImpBi inteq_reflection)
 have 3: "\<turnstile> (\<box>(init (\<not>w)) \<or> (init w) \<Pi> ( (init w) \<Pi> f)) =
            (\<box>(init (\<not>w)) \<or>  ( (init w) \<Pi> f)) "
   using 2 by auto
 have 4: "\<turnstile> (\<box>(init (\<not>w)) \<or>  ( (init w) \<Pi> f)) = (init w) \<Pi>\<^sup>u f " 
    by (metis Initprop(2) UPiEqvBoxOrPi int_eq)
 show ?thesis 
 by (metis "1" "3" "4" int_eq)
qed



lemma PiUPiState: 
 "\<turnstile> (init w) \<Pi> ( (init w) \<Pi>\<^sup>u f) = (init w) \<Pi> f " 
proof -
 have 1: "\<turnstile> (init w) \<Pi> ( (init w) \<Pi>\<^sup>u f) = (\<diamond>(init w) \<and> (init w) \<Pi>\<^sup>u ( (init w) \<Pi>\<^sup>u f)) " 
    by (simp add: PiEqvDiamondUPi)
 have 2: "\<turnstile> (init w) \<Pi>\<^sup>u ( (init w) \<Pi>\<^sup>u f) =  ( (init w) \<Pi>\<^sup>u f) " 
    by (metis (no_types, lifting) NotUPiEqvNotPi UPiPiState inteq_reflection upi_d_def)
 have 3: "\<turnstile> (\<diamond> (init w) \<and> (init w) \<Pi>\<^sup>u ( (init w) \<Pi>\<^sup>u f)) =
            (\<diamond>(init w) \<and>  ( (init w) \<Pi>\<^sup>u f)) "
   using 2 by auto
 have 4: "\<turnstile> (\<diamond>(init w) \<and>  ( (init w) \<Pi>\<^sup>u f)) = (init w) \<Pi> f " 
    by (meson PiEqvDiamondUPi Prop11)
 show ?thesis 
 by (metis "1" "3" "4" int_eq)
qed


lemma PiStateFiniteAndFinite: 
 "\<turnstile> ((init w) \<Pi> finite \<and> finite) = (\<diamond> (init w) \<and> finite) " 
proof -
 have 1: "\<turnstile> ((init w) \<Pi> finite \<and> finite) \<longrightarrow> (\<diamond> (init w) \<and> finite) "
   using PiImpDiamond by fastforce
 have 2: "\<turnstile> (\<diamond> (init w) \<and> finite) \<longrightarrow> ((init w) \<Pi> finite \<and> finite) "
   by (metis PiFiniteAbsorb PiTrueEqvDiamond TrueW int_simps(13) int_simps(17) inteq_reflection)
 show ?thesis 
 using "1" "2" int_iffI by blast
qed

lemma PiStateState: 
 "\<turnstile> (init w) \<Pi> (init w) = (init w) \<Pi> #True" 
by (metis BoxElim PiImpDiamond PiMPBC PiTrueEqvDiamond Prop11 StatePiBoxState int_simps(17) inteq_reflection)


lemma StateAndPiState: 
 "\<turnstile> ((init w) \<and> (init w) \<Pi> (empty \<and> w)) = ((init w) \<and> wnext (\<box> (init (\<not>w)))) " 
proof -
 have 2: "\<turnstile> ((init w) \<and> init w \<Pi> empty) =   (init w \<and> wnext (\<box> (init (\<not> w)))) "
    by (metis InitAndEmptyEqvAndEmpty StateAndEmptySChop StateAndPiEmpty inteq_reflection)
 have 5: "\<turnstile> (init w) \<Pi> #True = \<diamond> (init w) " 
      by (simp add: PiTrueEqvDiamond)
 show ?thesis 
 by (metis "2" "5" InitAndEmptyEqvAndEmpty PiAnd PiImpDiamond PiStateState Prop10 inteq_reflection 
     lift_and_com)
qed    
     
       
lemma PiStateFiniteSfin: 
 "\<turnstile> ((init w) \<Pi> finite \<and> sfin w) = sfin w " 
proof -
 have 1: "\<turnstile> ((init w) \<Pi> finite \<and> sfin w) =
            (((init w) \<Pi> finite \<and> finite) \<and> sfin w) "
   by (metis AndSFinEqvSChopAndEmpty SChopAndEmptyEqvSChopAndEmpty SFinEqvTrueSChopAndEmpty 
       inteq_reflection lift_and_com)
 have 2: "\<turnstile> (((init w) \<Pi> finite \<and> finite) \<and> sfin w) = ((\<diamond> (init w) \<and> finite) \<and> sfin w) "
    using PiStateFiniteAndFinite by fastforce
 have 3: "\<turnstile> ((\<diamond> (init w) \<and> finite) \<and> sfin w) = ((\<diamond> (init w)) \<and> sfin w) "
   by (metis AndSFinEqvSChopAndEmpty SChopAndEmptyEqvSChopAndEmpty SFinEqvTrueSChopAndEmpty 
      inteq_reflection lift_and_com)
 have 4: "\<turnstile>  ((\<diamond> (init w)) \<and> sfin w) = sfin w" 
   by (metis "3" InitAndEmptyEqvAndEmpty Prop11 Prop12 SChopAndA SFinEqvTrueSChopAndEmpty 
         TrueSChopEqvDiamond inteq_reflection)   
 show ?thesis 
 by (metis "1" "2" "3" "4" inteq_reflection)
qed


lemma PiStateSFin: 
 "\<turnstile> (init w) \<Pi> (sfin w) = ( sfin w);( wnext (\<box> (init (\<not>w)))) "
proof -
 have 1: "\<turnstile> (sfin w) = finite;(empty \<and> w) " 
  by (metis ChopAndCommute SFinEqvTrueSChopAndEmpty TrueSChopEqvDiamond inteq_reflection sometimes_d_def)
 have 2: "\<turnstile> (init w) \<Pi> (finite;(empty \<and> w)) =
            ((init w) \<Pi> finite \<and> finite);((init w) \<and> (init w) \<Pi> (empty \<and> w)) "
  using PiSChopDist[of w "LIFT finite" "LIFT (empty \<and> w)" ] unfolding schop_d_def 
     by auto  
 have 3: "\<turnstile> ((init w) \<Pi> finite \<and> finite);((init w) \<and> (init w) \<Pi> (empty \<and> w)) =
            ((init w) \<Pi> finite \<and> finite);((init w) \<and> wnext (\<box> (init (\<not>w)))) "
   using RightChopEqvChop StateAndPiState by blast
 have 4: "\<turnstile> (sfin w \<and> finite) = sfin w" 
    by (metis "1" ChopAndA DiamondEmptyEqvFinite Prop10 inteq_reflection sometimes_d_def)
 have 5: "\<turnstile> ((init w) \<Pi> finite \<and> finite);((init w) \<and> wnext (\<box> (init (\<not>w)))) =
            ((init w) \<Pi> finite \<and> sfin w);( wnext (\<box> (init (\<not>w)))) " 
   using SlideInitSFin[of "LIFT (init w) \<Pi> finite" w "LIFT wnext (\<box> (init (\<not>w)))" ] 4
    unfolding schop_d_def 
    by (metis Prop10 Prop12 int_eq int_iffD2 lift_and_com)
 have 6: "\<turnstile> ((init w) \<Pi> finite \<and> sfin w);( wnext (\<box> (init (\<not>w)))) = ( sfin w);( wnext (\<box> (init (\<not>w)))) " 
     using LeftChopEqvChop PiStateFiniteSfin by blast  
 show ?thesis 
 by (metis "1" "2" "3" "5" "6" int_eq)
qed

 
subsection \<open>SChopstar and Pi\<close>


lemma PiChopstarhelp2a:
 "\<turnstile>  (w \<and> empty)\<frown>(fpower ( (f\<frown>(w \<and> empty)) \<and> more) k) =
            (fpower (((w \<and> empty)\<frown>f) \<and> more) k ) \<frown>(w \<and> empty)"
proof (induction k)
case 0
then show ?case 
proof -
 have 1: "\<turnstile> (w \<and> empty) \<frown> empty = empty \<frown> (w \<and> empty)"
    by (metis EmptySChop InitAndEmptyEqvAndEmpty StateAndEmptySChop int_eq) 
 show ?thesis 
 by (metis 1 fpower_d_def wpow_0)
qed 
next
case (Suc k)
then show ?case 
  proof -
   have 1: "\<turnstile> (w \<and> empty)\<frown>fpower (f\<frown>(w \<and> empty) \<and> more) (Suc k)  =
               (w \<and> empty)\<frown>((f\<frown>(w \<and> empty) \<and> more)\<frown> fpower (f\<frown>(w \<and> empty) \<and> more) k) "
      unfolding fpower_d_def 
     by (simp add: schop_d_def)  
   have 11: "\<turnstile> (f \<and> more)\<frown>(w \<and> empty) = (sfin w \<and> (f \<and> more)) "
     by (metis SChopAndEmptyEqvSChopAndEmpty SFinEqvTrueSChopAndEmpty int_eq)
   have 12: "\<turnstile> (f\<frown>(w \<and> empty)) = (sfin w \<and> f)"
     by (metis SChopAndEmptyEqvSChopAndEmpty SFinEqvTrueSChopAndEmpty inteq_reflection) 
   have 13: "\<turnstile> (f\<frown>(w \<and> empty) \<and> more) = ((sfin w \<and> f) \<and> more)" 
     using "12" by auto
   have 14: "\<turnstile> ((sfin w \<and> f) \<and> more) = (sfin w \<and> (f \<and> more))"
     by fastforce
   have 2: "\<turnstile> (f\<frown>(w \<and> empty) \<and> more) = (f \<and> more)\<frown>(w \<and> empty)"
     using "11" "13" by fastforce
   have 20: "\<turnstile>((f\<frown>(w \<and> empty) \<and> more)\<frown> fpower (f\<frown>(w \<and> empty) \<and> more) k) =
              ((f \<and> more)\<frown>(w \<and> empty))\<frown> fpower (f\<frown>(w \<and> empty) \<and> more) k"
     by (simp add: "2" LeftSChopEqvSChop)      
   have 21: "\<turnstile> ((f \<and> more)\<frown>(w \<and> empty))\<frown> fpower (f\<frown>(w \<and> empty) \<and> more) k =
               (f \<and> more)\<frown>((w \<and> empty)\<frown> fpower (f\<frown>(w \<and> empty) \<and> more) k)"
          by (meson Prop11 SChopAssoc)
   have 22: "\<turnstile> (f \<and> more)\<frown>((w \<and> empty)\<frown> fpower (f\<frown>(w \<and> empty) \<and> more) k) =
              ((f \<and> more)\<frown> (fpower ((w \<and> empty)\<frown>f \<and> more) k\<frown>(w \<and> empty))) "
     by (simp add: RightSChopEqvSChop Suc.IH)
   have 23: "\<turnstile> (w \<and> empty)\<frown>((f\<frown>(w \<and> empty) \<and> more)\<frown> fpower (f\<frown>(w \<and> empty) \<and> more) k) =
             (w \<and> empty)\<frown>(((f \<and> more) \<frown> (w \<and> empty))\<frown> fpower (f\<frown>(w \<and> empty) \<and> more) k)"
     using "20" RightSChopEqvSChop by blast 
   have 24: "\<turnstile> (w \<and> empty)\<frown>(((f \<and> more) \<frown> (w \<and> empty))\<frown> fpower (f\<frown>(w \<and> empty) \<and> more) k) =
               (w \<and> empty)\<frown>((f \<and> more) \<frown> ((w \<and> empty)\<frown> fpower (f\<frown>(w \<and> empty) \<and> more) k))"
     using "21" RightSChopEqvSChop by blast    
   have 25: "\<turnstile> (w \<and> empty)\<frown>((f \<and> more) \<frown> ((w \<and> empty)\<frown> fpower (f\<frown>(w \<and> empty) \<and> more) k)) =
               (w \<and> empty)\<frown>((f \<and> more)\<frown> (fpower ((w \<and> empty)\<frown>f \<and> more) k\<frown>(w \<and> empty))) " 
     using 23 22 RightSChopEqvSChop by blast
   have 3: "\<turnstile> (w \<and> empty)\<frown>((f\<frown>(w \<and> empty) \<and> more)\<frown> fpower (f\<frown>(w \<and> empty) \<and> more) k) =
              (w \<and> empty)\<frown>((f \<and> more)\<frown> (fpower ((w \<and> empty)\<frown>f \<and> more) k\<frown>(w \<and> empty)))  "
     using 23 24 25 by fastforce
   have 4: "\<turnstile> (w \<and> empty)\<frown>((f \<and> more)\<frown> (fpower ((w \<and> empty)\<frown>f \<and> more) k\<frown>(w \<and> empty))) =
              ((w \<and> empty)\<frown>(f \<and> more))\<frown> ((fpower ((w \<and> empty)\<frown>f \<and> more) k\<frown>(w \<and> empty)))"       
     using SChopAssoc by blast  
   have 6: "\<turnstile> ((w \<and> empty)\<frown>(f \<and> more))\<frown> ((fpower ((w \<and> empty)\<frown>f \<and> more) k\<frown>(w \<and> empty))) =
             ((w \<and> empty)\<frown>f \<and> more)\<frown> ((fpower ((w \<and> empty)\<frown>f \<and> more) k\<frown>(w \<and> empty))) "
     by (meson EmptyAndSChopAndMoreEqvAndSChop LeftSChopEqvSChop)
   have 7: "\<turnstile> ((w \<and> empty)\<frown>f \<and> more)\<frown> ((fpower ((w \<and> empty)\<frown>f \<and> more) k\<frown>(w \<and> empty))) =
              ((fpower ((w \<and> empty)\<frown>f \<and> more) (Suc k)\<frown>(w \<and> empty)))" 
         by (metis SChopAssoc fpower_d_def schop_d_def wpow_Suc)
   show ?thesis using 1 3 4 6 7
     by (metis inteq_reflection)
   qed
qed

lemma PiChopstarhelp2:
 "\<turnstile> (w \<and> empty)\<frown>(schopstar(f\<frown>(w \<and> empty))) = (schopstar((w \<and> empty)\<frown>f))\<frown>(w \<and> empty)"
proof -
 have 1: "\<turnstile> (w \<and> empty)\<frown>(schopstar(f\<frown>(w \<and> empty))) = 
            (w \<and> empty)\<frown>(\<exists>k. fpower ( (f\<frown>(w \<and> empty)) \<and> more) k)  " 
   by (simp add: schopstar_d_def fpowerstar_d_def)
 have 2: "\<turnstile> (w \<and> empty)\<frown>(\<exists>k. fpower ( (f\<frown>(w \<and> empty)) \<and> more) k) =
             (\<exists>k. (w \<and> empty)\<frown>(fpower ( (f\<frown>(w \<and> empty)) \<and> more) k))     "
   using SChopExist by fastforce 
 have 3: "\<turnstile> (\<exists>k. (w \<and> empty)\<frown>(fpower ( (f\<frown>(w \<and> empty)) \<and> more) k)) =
            (\<exists>k. (fpower ((w \<and> empty)\<frown>f \<and> more) k)\<frown>(w \<and> empty)) " 
   by (simp add: ExEqvRule PiChopstarhelp2a)
 have 4: "\<turnstile> (\<exists>k. (fpower ((w \<and> empty)\<frown>f \<and> more) k)\<frown>(w \<and> empty)) =
            (\<exists>k. (fpower ((w \<and> empty)\<frown>f \<and> more) k))\<frown>(w \<and> empty)"
   using ExistSChop by fastforce
 have 5: "\<turnstile> (\<exists>k. (fpower ((w \<and> empty)\<frown>f \<and> more) k))\<frown>(w \<and> empty) =
            (schopstar((w \<and> empty)\<frown>f))\<frown>(w \<and> empty)" 
   by (simp add: schopstar_d_def fpowerstar_d_def)
 show ?thesis
   by (metis "1" "2" "3" "4" "5" int_eq)
qed  

lemma PiFPowerSuca:
 "\<turnstile> (init w) \<Pi> (fpower f (Suc k)) = ((init w) \<Pi> f)\<frown>((init w) \<and> (init w) \<Pi> (fpower f k))"
unfolding fpower_d_def by (metis PiSChopDist schop_d_def wpow_Suc)


lemma PiFPowerSucb:
 "\<turnstile> (init w) \<Pi> (fpower f (Suc k)) = ((init w) \<Pi> f \<and> sfin w)\<frown>((init w) \<and>  (init w) \<Pi> (fpower f k))"
proof -
 have 0: "\<turnstile> (init w) \<Pi> (fpower f (Suc k)) = ((init w) \<Pi> f)\<frown>((init w) \<and> (init w) \<Pi> (fpower f k))"
   by (meson PiFPowerSuca)
 have 1: "\<turnstile>((init w) \<and>  (init w) \<Pi> (fpower f k)) = (w \<and> empty)\<frown> ((init w) \<and>  (init w) \<Pi> (fpower f k))"
    by (metis InitAndEmptyEqvAndEmpty Prop10 Prop12 StateAndEmptySChop int_iffD2 inteq_reflection lift_and_com)
 have 2: "\<turnstile>((init w) \<Pi> f)\<frown>((init w) \<and> (init w) \<Pi> (fpower f k)) =
           ((init w) \<Pi> f)\<frown>((w \<and> empty)\<frown> ((init w) \<and>  (init w) \<Pi> (fpower f k))) "
   using "1" RightSChopEqvSChop by blast   
 have 3: "\<turnstile> ((init w) \<Pi> f)\<frown>((w \<and> empty)\<frown> ((init w) \<and>  (init w) \<Pi> (fpower f k))) =
            (((init w) \<Pi> f)\<frown> (w \<and> empty))\<frown>((init w) \<and>  (init w) \<Pi> (fpower f k))"
   by (simp add: SChopAssoc)
 have 4: "\<turnstile> (((init w) \<Pi> f)\<frown> (w \<and> empty)) = ((init w) \<Pi> f \<and> sfin w)"
   by (metis SChopAndEmptyEqvSChopAndEmpty SFinEqvTrueSChopAndEmpty inteq_reflection lift_and_com)
 have 5: "\<turnstile> (((init w) \<Pi> f)\<frown> (w \<and> empty))\<frown>((init w) \<and>  (init w) \<Pi> (fpower f k)) =
          ((init w) \<Pi> f \<and> sfin w)\<frown>((init w) \<and>  (init w) \<Pi> (fpower f k)) "
   by (simp add: "4" LeftSChopEqvSChop) 
 show ?thesis
   using "0" "2" "3" "5" by fastforce
qed     
     
lemma PiFPowerSucc:
  "\<turnstile> (init w) \<Pi> (fpower f (Suc k)) = 
     (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<and> (init w) \<Pi> empty)"
proof (induction k)
case 0
then show ?case 
  using PiFPowerSucb[of w f 0]  
unfolding schop_d_def fpower_d_def wpow_0 wpow_Suc
proof -
assume a1: "\<turnstile> init w \<Pi> ((f \<and> finite);empty) = 
            ((init w \<Pi> f \<and> sfin w) \<and> finite);(init w \<and> init w \<Pi> empty)"
have "\<turnstile>(((init w \<Pi> f \<and> sfin w) \<and> finite);empty \<and> finite) = ((init w \<Pi> f \<and> sfin w) \<and> finite)"
  by (metis AndChopB ChopEmpty Prop10 inteq_reflection)
then show "\<turnstile> init w \<Pi> ((f \<and> finite);empty) = 
             (((init w \<Pi> f \<and> sfin w) \<and> finite);empty \<and> finite);(init w \<and> init w \<Pi> empty)"
  by (metis a1 int_eq)
qed
next
case (Suc k)
  then show ?case 
  proof -
    have 3: "\<turnstile> (init w \<Pi> f) \<frown> 
               (init w \<and> fpower (init w \<Pi> f \<and> sfin w) (Suc k) \<frown> (init w \<and> init w \<Pi> empty)) =
               ((init w \<Pi> f \<and> sfin w) \<frown> fpower (init w \<Pi> f \<and> sfin w) (Suc k)) \<frown> 
               (init w \<and> init w \<Pi> empty)"
      by (metis (no_types, lifting) AndSFinSChopEqvStateAndSChop InitAndEmptyEqvAndEmpty SChopAssoc
          SFinEqvTrueSChopAndEmpty inteq_reflection)
    have 4:  "\<turnstile>(init w \<Pi> f) \<frown> (init w \<and> init w \<Pi> (f \<frown> fpower f k)) = 
               ((init w \<Pi> f \<and> sfin w) \<frown> 
                ((init w \<Pi> f \<and> sfin w) \<frown> fpower (init w \<Pi> f \<and> sfin w) k)) \<frown>
               (init w \<and> init w \<Pi> empty)"
      using 3  Suc.IH unfolding fpower_d_def wpow_Suc schop_d_def by (metis int_eq) 
    show ?thesis 
      by (metis "4" PiFPowerSuca fpower_d_def inteq_reflection schop_d_def wpow_Suc)
  qed
qed

lemma PiFPowerSucd:
  "\<turnstile> (init w) \<Pi> (fpower f (Suc k)) = (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<Pi> empty)"
proof (induction k)
case 0
then show ?case unfolding fpower_d_def wpow_0 wpow_Suc 
by (metis (no_types, opaque_lifting) InitAndEmptyEqvAndEmpty PiSChopDist SChopAndEmptyEqvSChopAndEmpty 
    SChopAssoc SFinEqvTrueSChopAndEmpty StateAndEmptySChop inteq_reflection lift_and_com schop_d_def)
next
case (Suc k)
then show ?case 
  proof -
   have 1: "\<turnstile>  init w \<Pi> fpower f (Suc (Suc k)) = (init w) \<Pi> (f\<frown>(fpower f (Suc k)))"
      by (metis PiFPowerSuca PiSChopDist inteq_reflection) 
   have 2: "\<turnstile> (init w) \<Pi> (f\<frown>(fpower f (Suc k))) = ((init w) \<Pi> f)\<frown>((init w) \<and> (init w) \<Pi> (fpower f (Suc k)))"
     by (simp add: PiSChopDist) 
   have 3: "\<turnstile> ((init w) \<Pi> f)\<frown>((init w) \<and> (init w) \<Pi> (fpower f (Suc k))) =
               ((init w) \<Pi> f)\<frown>((init w) \<and> (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<and> (init w) \<Pi> empty))"
     by (metis "2" PiFPowerSucc inteq_reflection)               
   have 4: "\<turnstile> ((init w) \<Pi> f)\<frown>((init w) \<and> (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<and> (init w) \<Pi> empty)) =
                ((init w) \<Pi> f \<and> sfin w)\<frown>( (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<and> (init w) \<Pi> empty)) " 
      by (metis AndSFinSChopEqvStateAndSChop InitAndEmptyEqvAndEmpty SFinEqvTrueSChopAndEmpty inteq_reflection)
   have 5: "\<turnstile> ((init w) \<Pi> f \<and> sfin w)\<frown>( (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<and> (init w) \<Pi> empty)) =
              ( (fpower ((init w) \<Pi> f \<and> sfin w) (Suc (Suc k)))\<frown>((init w) \<and> (init w) \<Pi> empty)) " 
     by (metis "1" "2" "4" PiFPowerSucc inteq_reflection)
   have 6: "\<turnstile> ( (fpower ((init w) \<Pi> f \<and> sfin w) (Suc (Suc k)))\<frown>((init w) \<and> (init w) \<Pi> empty)) =
              ( (fpower ((init w) \<Pi> f \<and> sfin w) (Suc (Suc k)))\<frown>( (init w) \<Pi> empty)) " 
      by (metis (no_types, lifting) PiFPowerSucc SChopAssoc Suc fpower_d_def inteq_reflection schop_d_def wpow_Suc)
   show ?thesis 
   by (metis "1" "2" "3" "4" "5" "6" inteq_reflection)
  qed
qed

lemma SFin_SChop:
 "\<turnstile> (f \<and> sfin w)\<frown> g = (f \<and> fin w)\<frown>g"
proof -
 have 1: "\<turnstile> (f \<and> fin w)\<frown> g  = (f \<and> (fin w \<and> finite)) \<frown> g" 
   by (metis (no_types, lifting) LeftChopEqvChop Prop11 Prop12 lift_and_com schop_d_def)
 have 2: "\<turnstile> (fin w \<and> finite) = (sfin w \<and> finite)"
   by (metis (no_types, lifting) EmptyImpFinite FinAndEmpty FinEqvTrueChopAndEmpty Finprop(5)
       Prop10 SChopAndB SChopImpFinite SFinAndEmpty SFinEqvTrueSChopAndEmpty inteq_reflection 
       lift_imp_trans sfin_d_def)
 have 3: "\<turnstile> (f \<and> sfin w)\<frown> g  = (f \<and> (sfin w \<and> finite)) \<frown> g" 
   by (metis AndSChopCommute DiamondEmptyEqvFinite Prop10 SChopAndB SFinEqvTrueSChopAndEmpty 
       TrueSChopEqvDiamond int_eq)
 show ?thesis 
   using "1" "2" "3" by (metis inteq_reflection) 
qed

lemma PiChopstar:
 "\<turnstile> (init w) \<Pi> (schopstar f) = 
    (init (\<not> w)) \<U> ((init w) \<and> (schopstar(((init w) \<Pi> f) \<and> sfin w))\<frown>wnext(\<box> (init (\<not> w))))"
proof -
 have 1: "\<turnstile> (init w) \<Pi> (schopstar f) = (init w) \<Pi> (\<exists>k. fpower f k)"
    by (metis FPowerstardef PiEqvRule SChopstar_FPowerstar inteq_reflection)
 have 2: "\<turnstile> (init w) \<Pi> (\<exists>k. fpower f k) = (\<exists>k. (init w) \<Pi> (fpower f k))"
   by (simp add: Valid_def pi_d_def) 
 have 3: "\<turnstile> (\<exists>k. (init w) \<Pi> (fpower f k)) = 
            ( (init w) \<Pi> empty \<or> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))))"
   using PiFPowerExpand by auto
 have 4: "\<turnstile> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))) =
            (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<Pi> empty))" 
   by (meson ExEqvRule PiFPowerSucd)
 have 5: "\<turnstile> (init w) \<Pi> empty = empty \<frown> ((init w) \<Pi> empty)"
   by (simp add: EmptySChop int_iffD1 int_iffD2 int_iffI)
 have 6: "\<turnstile> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<Pi> empty)) =
            (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k)))\<frown>((init w) \<Pi> empty)"
   by (simp add: ExistSChop) 
 have 7: "\<turnstile> ( (init w) \<Pi> empty \<or> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<Pi> empty)))=
            ( empty\<frown> ((init w) \<Pi> empty) \<or> 
              (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k)))\<frown>((init w) \<Pi> empty))"
   using "5" "6" by fastforce
 have 8: "\<turnstile> ( empty\<frown> ((init w) \<Pi> empty) \<or> 
              (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k)))\<frown>((init w) \<Pi> empty)) =
            ( empty \<or> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))))\<frown>((init w) \<Pi> empty)"
   by (meson OrSChopEqv Prop11)
 have 9: "\<turnstile> empty = (fpower ((init w) \<Pi> f \<and> sfin w) 0)"
   unfolding fpower_d_def by simp 
 have 10: "\<turnstile> ( empty \<or> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k)))) = 
             ( (fpower ((init w) \<Pi> f \<and> sfin w) 0) \<or> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))))"
    unfolding fpower_d_def by simp
 have 11: "\<turnstile> ( (fpower ((init w) \<Pi> f \<and> sfin w) 0) \<or> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k)))) =
             (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) k))"
   using exists_expand[of "w" "f"] unfolding fpower_d_def 
   by (metis (mono_tags) Valid_def inteq_reflection wpow_0 wpowersem1)  
 have 12: "\<turnstile> ( (init w) \<Pi> empty \<or> 
               (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<Pi> empty))) =
            (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) ( k)))\<frown> ((init w) \<Pi> empty)"
   by (metis "11" "7" "8" "9" inteq_reflection)
 have 13: "\<turnstile> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) ( k)))\<frown> ((init w) \<Pi> empty) =
             (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>  ((init w) \<Pi> empty)" 
     by (metis FPowerstardef LeftSChopEqvSChop SChopstar_FPowerstar inteq_reflection)
 have 14: "\<turnstile> (schopstar ((init w) \<Pi> f \<and> sfin w))\<frown>  ((init w) \<Pi> empty) =
             ( ((init w) \<Pi> empty) \<or> 
               (((init w) \<Pi> f \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> f \<and> sfin w)))\<frown>((init w) \<Pi> empty))"
      by (metis "5" OrSChopEqvRule SChopstar_unfoldl_eq inteq_reflection)
 have 15: "\<turnstile> ((init w) \<Pi> empty) =  (init (\<not> w)) \<U> ( (init w) \<and> wnext (\<box> (init (\<not> w))))"
   by (simp add: PiEmpty) 
 have 16:"\<turnstile> ( ((init w) \<Pi> f \<and> sfin w) \<frown> (schopstar ((init w) \<Pi> f \<and> sfin w)) )\<frown>((init w) \<Pi> empty) =
            ( ((init w) \<Pi> f \<and> sfin w) \<frown> (schopstar ((init w) \<Pi> f \<and> sfin w)) )\<frown> wnext (\<box> (init (\<not> w)))"
  proof -
    have 161: "\<turnstile> ( ((init w) \<Pi> f \<and> sfin w) \<frown> (schopstar ((init w) \<Pi> f \<and> sfin w)) )\<frown>((init w) \<Pi> empty) =
               ( ((init w) \<Pi> f \<and> sfin w) \<frown> ((schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>((init w) \<Pi> empty))) "
       by (meson Prop11 SChopAssoc)
    have 162: "\<turnstile> ((init w) \<Pi> f \<and> sfin w) = ((init w) \<Pi> f)\<frown>(w \<and> empty)  "
      by (metis SChopAndEmptyEqvSChopAndEmpty SFinEqvTrueSChopAndEmpty inteq_reflection lift_and_com) 
    have 163: "\<turnstile> (schopstar ((init w) \<Pi> f \<and> sfin w)) = (schopstar (((init w) \<Pi> f)\<frown> (w \<and> empty)))" 
      by (metis "162" SChopstardef inteq_reflection)
    have 164: " \<turnstile> ((init w) \<Pi> f \<and> sfin w) \<frown> (schopstar ((init w) \<Pi> f \<and> sfin w))  =
                  ( (init w) \<Pi> f)\<frown>((w \<and> empty)\<frown>(schopstar ( ((init w) \<Pi> f) \<frown> (w \<and> empty)))) "
      using "162" SChopAssoc int_eq by metis  
    have 165: "\<turnstile> ( (init w) \<Pi> f)\<frown>((w \<and> empty)\<frown>(schopstar ( ((init w) \<Pi> f) \<frown> (w \<and> empty)))) =
                 ( (init w) \<Pi> f)\<frown> ((schopstar ( (w \<and> empty)\<frown> ( (init w) \<Pi> f)))\<frown>(w \<and> empty))  "
      by (simp add: PiChopstarhelp2 RightSChopEqvSChop) 
    have 166: "\<turnstile> (( (init w) \<Pi> f)\<frown> ((schopstar ( (w \<and> empty)\<frown> ( (init w) \<Pi> f)))\<frown>(w \<and> empty)))\<frown>((init w) \<Pi> empty) =
                 ( (( (init w) \<Pi> f)\<frown> (schopstar ( (w \<and> empty)\<frown> ( (init w) \<Pi> f)))) \<frown>( (init w) \<and> ((init w) \<Pi> empty) ) ) "
      by (metis (no_types, lifting) InitAndEmptyEqvAndEmpty SChopAssoc StateAndEmptySChop int_eq)
    have 167: "\<turnstile> ( (( (init w) \<Pi> f)\<frown> (schopstar ( (w \<and> empty)\<frown> ( (init w) \<Pi> f)))) \<frown>( (init w) \<and> ((init w) \<Pi> empty) ) ) =
                 ( (( (init w) \<Pi> f)\<frown> (schopstar ( (w \<and> empty)\<frown> ( (init w) \<Pi> f)))) \<frown>((w \<and> empty) \<frown> wnext (\<box> (init (\<not> w))) ) )"
      by (simp add: RightSChopEqvSChop StateAndPiEmpty)
    have 168: "\<turnstile> ( (( (init w) \<Pi> f)\<frown> (schopstar ( (w \<and> empty)\<frown> ( (init w) \<Pi> f)))) \<frown>((w \<and> empty) \<frown> wnext (\<box> (init (\<not> w))) ) ) =
                 ( (( (init w) \<Pi> f)\<frown> (schopstar ( (w \<and> empty)\<frown> ( (init w) \<Pi> f)))) \<frown>(w \<and> empty)) \<frown> wnext (\<box> (init (\<not> w)))  "
      by (simp add: SChopAssoc)
    have 169: "\<turnstile> ( (( (init w) \<Pi> f)\<frown> (schopstar ( (w \<and> empty)\<frown> ( (init w) \<Pi> f)))) \<frown>(w \<and> empty)) =
                 ( ((init w) \<Pi> f \<and> sfin w) \<frown> (schopstar ((init w) \<Pi> f \<and> sfin w)) ) "
      using "164" "165" SChopAssoc by fastforce
    show ?thesis by (metis "164" "165" "166" "167" "168" "169" int_eq)
   qed
 have 17: "\<turnstile> ((init w) \<Pi> f \<and> sfin w) = ((init w) \<Pi> f)\<frown>((init w) \<and> empty)"
   by (metis InitAndEmptyEqvAndEmpty SChopAndEmptyEqvSChopAndEmpty SFinEqvTrueSChopAndEmpty 
       inteq_reflection lift_and_com)
 have 18: "\<turnstile> ((init w) \<Pi> f) = wprev(\<box> (init (\<not> w)))\<frown>((init w) \<and> (init w) \<Pi> f)"
   by (simp add: WPrevPi)
 have 19: "\<turnstile> wprev(\<box> (init (\<not> w)))\<frown>((init w) \<and> (init w) \<Pi> f) = 
             wprev(\<box> (init (\<not> w)))\<frown>(((init w) \<and> empty) \<frown> ((init w) \<Pi> f))"
   by (meson RightSChopImpSChop StateAndEmptySChop int_iffD1 int_iffD2 int_iffI)
 have 20: "\<turnstile> wprev(\<box> (init (\<not> w)))\<frown>(((init w) \<and> empty) \<frown> ((init w) \<Pi> f)) =
             (init (\<not> w)) \<U> ( (init w) \<and> ((init w) \<Pi> f)) "
   using "18" "19" StatePiUntil by fastforce   
 have 21: "\<turnstile> (((init w) \<Pi> f \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> f \<and> sfin w)))\<frown> wnext (\<box> (init (\<not> w))) =
             (init (\<not> w)) \<U> (
                       (init w) \<and> 
                       ((((init w) \<Pi> f) \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> f \<and> sfin w)))\<frown>wnext (\<box> (init (\<not> w)))
                             )" 
   proof -
   have "\<turnstile>(init w \<Pi> f) \<frown> (init w \<and> empty) = (init w \<Pi> f \<and> sfin w)"
     by (metis "17" inteq_reflection)
   then show ?thesis using StatePiUntil[of w f] 
     UntilChopDist[of w "LIFT((init w \<and> empty))" "LIFT((schopstar ((init w) \<Pi> f \<and> sfin w)))\<frown>wnext (\<box> (init (\<not> w)))"]
     by (metis (no_types, lifting) AndSFinSChopEqvStateAndSChop SChopAssoc StateUntilEqvWPrevChop int_eq) 
   qed 
 have 22: "\<turnstile> ((init (\<not> w)) \<U> ( (init w) \<and> wnext (\<box> (init (\<not> w)))) \<or>
             (init (\<not> w)) \<U> ( 
                    (init w) \<and> 
                    ((((init w) \<Pi> f) \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> f \<and> sfin w)))\<frown>wnext (\<box> (init (\<not> w)))
                            ) ) =
              (init (\<not> w)) \<U>  ( 
         ((init w) \<and> wnext (\<box> (init (\<not> w)))) \<or>
         ( (init w) \<and> ((((init w) \<Pi> f) \<and> sfin w)\<frown>(schopstar((init w) \<Pi> f \<and> sfin w)))\<frown>wnext (\<box> (init (\<not> w))))
                               ) "
   using UntilOrDist by fastforce
 have 23: "\<turnstile> ( 
        ((init w) \<and> wnext (\<box> (init (\<not> w)))) \<or>
        ( (init w) \<and> ((((init w) \<Pi> f) \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> f \<and> sfin w)))\<frown>wnext (\<box> (init (\<not> w))))
             ) =
             ( (init w) \<and> (schopstar (((init w) \<Pi> f) \<and> sfin w))\<frown>wnext(\<box> (init (\<not> w))))"
    by (metis (mono_tags, lifting) EmptyOrSChopEqv Prop11 SChopOrEqvRule SChopstar_unfoldl_eq StateAndEmptySChop int_eq)
 have 24: "\<turnstile> (init w) \<Pi> (schopstar f) = ( (init w) \<Pi> empty \<or> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))))" 
   using "1" "2" "3" by fastforce
 have 25: "\<turnstile> ( (init w) \<Pi> empty \<or> (\<exists>k. (init w) \<Pi> (fpower f (Suc k)))) =
             ( (init w) \<Pi> empty \<or> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<Pi> empty)))" 
   using "4" by fastforce
 have 26: "\<turnstile> ( (init w) \<Pi> empty \<or> 
               (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<Pi> empty))) =
             (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>  ((init w) \<Pi> empty)"
   using "12" "13" by fastforce
 have 27: "\<turnstile> (schopstar ((init w) \<Pi> f \<and> sfin w))\<frown>  ((init w) \<Pi> empty) = 
           ((init (\<not> w)) \<U> ( (init w) \<and> wnext (\<box> (init (\<not> w)))) \<or>
            ( ((init w) \<Pi> f \<and> sfin w) \<frown> (schopstar ((init w) \<Pi> f \<and> sfin w)) )\<frown> wnext (\<box> (init (\<not> w))))"
   using "14" "15" "16" by fastforce     
 have 28:"\<turnstile> (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>  ((init w) \<Pi> empty) =
            (init (\<not> w)) \<U> ((init w) \<and> (schopstar (((init w) \<Pi> f) \<and> sfin w))\<frown>wnext(\<box> (init (\<not> w))))"
   using 27  21 22 23
   by (metis inteq_reflection)
 show ?thesis
   by (metis "24" "25" "26" "28" inteq_reflection) 
qed 


lemma PiChopstar_var1: 
  "\<turnstile> (init w) \<Pi> (schopstar f) = 
     (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>  ((init w) \<Pi> empty) " 
proof -
 have 1: "\<turnstile> (init w) \<Pi> (schopstar f) = (init w) \<Pi> (\<exists>k. fpower f k)"
    by (metis FPowerstardef PiEqvRule SChopstar_FPowerstar inteq_reflection)
 have 2: "\<turnstile> (init w) \<Pi> (\<exists>k. fpower f k) = (\<exists>k. (init w) \<Pi> (fpower f k))"
   by (simp add: Valid_def pi_d_def) 
 have 3: "\<turnstile> (\<exists>k. (init w) \<Pi> (fpower f k)) = 
            ( (init w) \<Pi> empty \<or> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))))"
  using PiFPowerExpand by auto
 have 4: "\<turnstile> (\<exists>k. (init w) \<Pi> (fpower f (Suc k))) =
            (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<Pi> empty))" 
   by (meson ExEqvRule PiFPowerSucd)
 have 5: "\<turnstile> (init w) \<Pi> empty = empty \<frown> ((init w) \<Pi> empty)"
   by (simp add: EmptySChop int_iffD1 int_iffD2 int_iffI)
 have 6: "\<turnstile> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<Pi> empty)) =
            (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k)))\<frown>((init w) \<Pi> empty)"
   by (simp add: ExistSChop) 
 have 7: "\<turnstile> ( (init w) \<Pi> empty \<or> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<Pi> empty)))=
            ( empty\<frown> ((init w) \<Pi> empty) \<or> 
              (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k)))\<frown>((init w) \<Pi> empty))"
   using "5" "6" by fastforce
 have 8: "\<turnstile> ( empty\<frown> ((init w) \<Pi> empty) \<or> 
              (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k)))\<frown>((init w) \<Pi> empty)) =
            ( empty \<or> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))))\<frown>((init w) \<Pi> empty)"
   by (meson OrSChopEqv Prop11)
 have 9: "\<turnstile> empty = (fpower ((init w) \<Pi> f \<and> sfin w) 0)"
   unfolding fpower_d_def by simp 
 have 10: "\<turnstile> ( empty \<or> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k)))) = 
             ( (fpower ((init w) \<Pi> f \<and> sfin w) 0) \<or> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))))"
    unfolding fpower_d_def by simp
 have 11: "\<turnstile> ( (fpower ((init w) \<Pi> f \<and> sfin w) 0) \<or> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k)))) =
             (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) k))"
   using exists_expand[of "w" "f"] unfolding fpower_d_def 
   by (metis (mono_tags) Valid_def inteq_reflection wpow_0 wpowersem1)  
 have 12: "\<turnstile> ( (init w) \<Pi> empty \<or> 
               (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) (Suc k))\<frown>((init w) \<Pi> empty))) =
            (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) ( k)))\<frown> ((init w) \<Pi> empty)"
   by (metis "11" "7" "8" "9" inteq_reflection)
 have 13: "\<turnstile> (\<exists>k. (fpower ((init w) \<Pi> f \<and> sfin w) ( k)))\<frown> ((init w) \<Pi> empty) =
             (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>  ((init w) \<Pi> empty)" 
     by (metis FPowerstardef LeftSChopEqvSChop SChopstar_FPowerstar inteq_reflection)
 show ?thesis 
 by (metis "1" "12" "13" "2" "3" "4" inteq_reflection)
qed

lemma PiChopstar_var2: 
  "\<turnstile> (init w) \<Pi> (schopstar f) = 
      ((init w) \<Pi> empty \<or> 
       (schopstar ((init w) \<Pi> f \<and> sfin w))\<frown>(((init w) \<Pi> f \<and> sfin w)\<frown>(wnext (\<box> (init (\<not>w))))))
   " 
proof -
 have 1: "\<turnstile> (schopstar ((init w) \<Pi> f \<and> sfin w))  =
            (empty \<or> (schopstar ((init w) \<Pi> f \<and> sfin w))\<frown> ((init w) \<Pi> f \<and> sfin w)) " 
    by (metis (no_types, lifting) DiamondEmptyEqvFinite Prop10 Prop12 SCSEqvOrChopSCSB SChopAndB 
        SFinEqvTrueSChopAndEmpty TrueSChopEqvDiamond int_eq int_iffD2 lift_and_com)
 have 2: "\<turnstile> (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>  ((init w) \<Pi> empty) =
            (  ((init w) \<Pi> empty) \<or> ((schopstar ((init w) \<Pi> f \<and> sfin w))\<frown> ((init w) \<Pi> f \<and> sfin w))\<frown>((init w) \<Pi> empty)) "
     using "1" EmptyOrSChopEqvRule by blast
 have 3: "\<turnstile> ((schopstar ((init w) \<Pi> f \<and> sfin w))\<frown> ((init w) \<Pi> f \<and> sfin w))\<frown>((init w) \<Pi> empty) =
            (schopstar ((init w) \<Pi> f \<and> sfin w))\<frown> (((init w) \<Pi> f \<and> sfin w)\<frown>((init w) \<Pi> empty)) " 
    by (meson Prop11 SChopAssoc)
 have 4: "\<turnstile> (((init w) \<Pi> f \<and> sfin w)\<frown>((init w) \<Pi> empty)) =
            (((init w) \<Pi> f \<and> sfin w)\<frown>(wnext (\<box> (init (\<not>w))))) " 
   by (metis (no_types, lifting) InitAndEmptyEqvAndEmpty SlideInitSFin StateAndEmptySChop StateAndPiEmpty inteq_reflection)
 have 5: "\<turnstile>  (schopstar ((init w) \<Pi> f \<and> sfin w))\<frown> (((init w) \<Pi> f \<and> sfin w)\<frown>((init w) \<Pi> empty)) =
             (schopstar ((init w) \<Pi> f \<and> sfin w))\<frown>(((init w) \<Pi> f \<and> sfin w)\<frown>(wnext (\<box> (init (\<not>w))))) " 
     by (metis "4" RightChopEqvChop schop_d_def)
 show ?thesis 
 by (metis "2" "3" "5" PiChopstar_var1 int_eq)
qed




lemma PiChopstar_var3: 
  "\<turnstile> (init w) \<Pi> (f\<frown>(schopstar f) ) = 
     (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>(((init w) \<Pi> f \<and> sfin w)\<frown>(wnext (\<box> (init (\<not>w))))) " 
proof -
 have 1: "\<turnstile> (init w) \<Pi> (f\<frown>(schopstar f)) =
            ((init w) \<Pi> f \<and> sfin w) \<frown> ( (init w) \<Pi> (schopstar f)) " 
    by (metis PiSChopDist SlideInitSFin inteq_reflection)
 have 2: "\<turnstile> ((init w) \<Pi> (schopstar f)) = (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>  ((init w) \<Pi> empty) "
   using PiChopstar_var1 by auto
 have 3: "\<turnstile> ((init w) \<Pi> f \<and> sfin w) \<frown> ( (init w) \<Pi> (schopstar f)) =
             ((init w) \<Pi> f \<and> sfin w) \<frown>((schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>  ((init w) \<Pi> empty))" 
   using "2" RightSChopEqvSChop by blast
 have 4: "\<turnstile> ((init w) \<Pi> f \<and> sfin w) \<frown>((schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>  ((init w) \<Pi> empty)) =
            (((init w) \<Pi> f \<and> sfin w) \<frown>(schopstar ((init w) \<Pi> f \<and> sfin w))) \<frown>  ((init w) \<Pi> empty) " 
       by (simp add: SChopAssoc)
 have 5: "\<turnstile>  (((init w) \<Pi> f \<and> sfin w) \<frown>(schopstar ((init w) \<Pi> f \<and> sfin w))) =
             (schopstar ((init w) \<Pi> f \<and> sfin w))\<frown> (((init w) \<Pi> f \<and> sfin w) \<and> finite)" 
     by (simp add: SChopplusCommute)
 have 6: "\<turnstile> (((init w) \<Pi> f \<and> sfin w) \<and> finite) = (((init w) \<Pi> f \<and> sfin w)) "
   using SFinEQvFinAndFinite by fastforce
 have 7: "\<turnstile> (((init w) \<Pi> f \<and> sfin w))\<frown>((init w) \<Pi> empty) = 
            ((init w) \<Pi> f \<and> sfin w)\<frown> (wnext (\<box> (init (\<not>w)))) " 
     by (metis (no_types, lifting) InitAndEmptyEqvAndEmpty SlideInitSFin StateAndEmptySChop 
         StateAndPiEmpty inteq_reflection)
 show ?thesis 
 by (metis (no_types, lifting) "1" "2" "5" "6" "7" SChopAssoc inteq_reflection)
qed

lemma PiChopstar_var4: 
  "\<turnstile> (init w) \<Pi> (f\<frown>(schopstar f) ) = 
     (((init w) \<Pi> f \<and> sfin w)\<frown> (schopstar ((init w) \<Pi> f \<and> sfin w))) \<frown>(wnext (\<box> (init (\<not>w)))) "
proof -
 have 1: "\<turnstile>  (init w) \<Pi> (f\<frown>(schopstar f) ) =
             (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>(((init w) \<Pi> f \<and> sfin w)\<frown>(wnext (\<box> (init (\<not>w))))) "
    using PiChopstar_var3 by blast
 have 2: "\<turnstile> (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>(((init w) \<Pi> f \<and> sfin w)\<frown>(wnext (\<box> (init (\<not>w))))) =
            ((schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>((init w) \<Pi> f \<and> sfin w))\<frown>(wnext (\<box> (init (\<not>w))))"
   by (simp add: SChopAssoc)
 have 3: "\<turnstile> ((init w) \<Pi> f \<and> sfin w) = (((init w) \<Pi> f \<and> sfin w) \<and> finite) " 
   using SFinEQvFinAndFinite by fastforce
 have 4: "\<turnstile> (schopstar ((init w) \<Pi> f \<and> sfin w))\<frown> (((init w) \<Pi> f \<and> sfin w) \<and> finite) =
            (((init w) \<Pi> f \<and> sfin w) \<frown>(schopstar ((init w) \<Pi> f \<and> sfin w))) "
      by (simp add: SChopstar_slide_var)
 show ?thesis 
 by (metis "1" "2" "3" "4" int_eq)
qed

lemma PiChopstar_var5: 
  "\<turnstile> (init w) \<Pi> (f\<frown>(schopstar f) ) = 
     (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>((init w) \<Pi> (f \<and> finite)) "
proof -
 have 1: "\<turnstile> (init w) \<Pi> (f\<frown>(schopstar f)) =
            ((init w) \<Pi> f \<and> sfin w) \<frown> ( (init w) \<Pi> (schopstar f)) " 
    by (metis PiSChopDist SlideInitSFin inteq_reflection)
 have 2: "\<turnstile> ((init w) \<Pi> (schopstar f)) = 
            (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>  ((init w) \<Pi> empty) "
   using PiChopstar_var1 by auto
 have 3: "\<turnstile> ((init w) \<Pi> f \<and> sfin w) \<frown> ( (init w) \<Pi> (schopstar f)) =
           ((init w) \<Pi> f \<and> sfin w) \<frown> ( (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>  ((init w) \<Pi> empty)) "
    using "2" RightSChopEqvSChop by blast
 have 4: "\<turnstile> ((init w) \<Pi> f \<and> sfin w) \<frown> ( (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>  ((init w) \<Pi> empty)) =
            (((init w) \<Pi> f \<and> sfin w) \<frown>  (schopstar ((init w) \<Pi> f \<and> sfin w))) \<frown>  ((init w) \<Pi> empty)"
    using SChopAssoc by blast
 have 5: "\<turnstile>  (((init w) \<Pi> f \<and> sfin w) \<frown>(schopstar ((init w) \<Pi> f \<and> sfin w))) =
             (schopstar ((init w) \<Pi> f \<and> sfin w))\<frown> (((init w) \<Pi> f \<and> sfin w) \<and> finite)" 
     by (simp add: SChopplusCommute)
 have 6: "\<turnstile> (((init w) \<Pi> f \<and> sfin w) \<and> finite) = (((init w) \<Pi> f \<and> sfin w)) "
   using SFinEQvFinAndFinite by fastforce
 have 7: "\<turnstile> (((init w) \<Pi> f \<and> sfin w)) \<frown> ((init w) \<Pi> empty) =
            (init w) \<Pi> (f\<frown>empty) " 
     by (metis PiSChopDist SlideInitSFin inteq_reflection)
 have 8: "\<turnstile> (init w) \<Pi> (f\<frown>empty) = (init w) \<Pi> (f \<and> finite) "
   by (simp add: ChopEmpty PiEqvRule schop_d_def)
 have 9: "\<turnstile> (((init w) \<Pi> f \<and> sfin w) \<frown>  (schopstar ((init w) \<Pi> f \<and> sfin w))) \<frown>  ((init w) \<Pi> empty) =
            ( (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>(((init w) \<Pi> f \<and> sfin w))) \<frown> ((init w) \<Pi> empty)"
   by (metis "4" "5" "6" inteq_reflection)
 have 10: "\<turnstile> ( (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>(((init w) \<Pi> f \<and> sfin w))) \<frown> ((init w) \<Pi> empty) =
             (schopstar ((init w) \<Pi> f \<and> sfin w)) \<frown>((((init w) \<Pi> f \<and> sfin w)) \<frown> ((init w) \<Pi> empty))" 
    by (meson Prop11 SChopAssoc)
 show ?thesis 
 by (metis "1" "10" "3" "4" "7" "8" "9" int_eq)
qed

   

lemma PiChopstar_var6: 
 "\<turnstile> (init w) \<Pi> (schopstar f) = 
     ((  ((init w) \<Pi> (f \<or> empty) \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> (f \<or> empty) \<and> sfin w)))
     \<frown>(wnext(\<box>(init (\<not>w))))) "
proof -
 have 1: "\<turnstile> (schopstar f) = (schopstar (f \<or> empty)) "
   by (metis SChopstar_star2 SChopstar_swap inteq_reflection)
 have 1: "\<turnstile> (init w) \<Pi> (schopstar (f \<or> empty)) = 
            (schopstar ((init w) \<Pi> (f \<or> empty) \<and> sfin w)) \<frown>  ((init w) \<Pi> empty) " 
  by (simp add: PiChopstar_var1)
 have 2: "\<turnstile> (schopstar ((init w) \<Pi> (f \<or> empty) \<and> sfin w)) =
            (empty \<or> ((init w) \<Pi> (f \<or> empty) \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> (f \<or> empty) \<and> sfin w))) " 
     by (meson Prop06 SCSEqvOrChopSCSB SChopstar_slide_var)
 have 3: "\<turnstile> (empty \<or> ((init w) \<Pi> (f \<or> empty) \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> (f \<or> empty) \<and> sfin w)))\<frown> ((init w) \<Pi> empty)
            = (((init w) \<Pi> empty) \<or> (((init w) \<Pi> (f \<or> empty) \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> (f \<or> empty) \<and> sfin w)))\<frown> ((init w) \<Pi> empty) ) 
         "
      using EmptyOrSChopEqv by blast   
 have 4: "\<turnstile>  ((init w) \<Pi> (f \<or> empty) \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> (f \<or> empty) \<and> sfin w)) =
             (schopstar ((init w) \<Pi> (f \<or> empty) \<and> sfin w))\<frown> (((init w) \<Pi> (f \<or> empty) \<and> sfin w) \<and> finite) " 
    by (simp add: SChopplusCommute)
 have 5: "\<turnstile> (((init w) \<Pi> (f \<or> empty) \<and> sfin w) \<and> finite) =
            (((init w) \<Pi> (f \<or> empty) \<and> sfin w)) " 
    using SFinEQvFinAndFinite by fastforce
 have 6: "\<turnstile> (((init w) \<Pi> (f \<or> empty) \<and> sfin w)) \<frown> ((init w) \<Pi> empty) =
             (((init w) \<Pi> (f \<or> empty) \<and> sfin w)) \<frown> (wnext(\<box>(init (\<not>w)))) "
     by (metis (no_types, lifting) InitAndEmptyEqvAndEmpty SlideInitSFin StateAndEmptySChop StateAndPiEmpty int_eq)
 have 7: "\<turnstile> (((init w) \<Pi> (f \<or> empty) \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> (f \<or> empty) \<and> sfin w)))\<frown> ((init w) \<Pi> empty) =
            (((init w) \<Pi> (f \<or> empty) \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> (f \<or> empty) \<and> sfin w)))\<frown> (wnext(\<box>(init (\<not>w)))) " 
   by (metis (no_types, lifting) "4" "5" "6" SChopAssoc inteq_reflection)
 have 8: "\<turnstile> (((init w) \<Pi> empty) \<or> (((init w) \<Pi> (f \<or> empty) \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> (f \<or> empty) \<and> sfin w)))\<frown> ((init w) \<Pi> empty) ) =
            ( (((init w) \<Pi> (f \<or> empty) \<and> sfin w)\<frown>(schopstar ((init w) \<Pi> (f \<or> empty) \<and> sfin w)))\<frown> ((init w) \<Pi> empty) ) " 
  by (metis "1" "2" "3" "7" FiniteAndEmptyEqvEmpty PiChopstar_var4 Prop05 SChopstar_sup_id_star1 int_iffD1 inteq_reflection)
 show ?thesis
 by (meson PiChopstar_var4 PiEqvRule Prop04 SChopstar_star2 SChopstar_sup_id_star1 
     SChopstar_swap UntilImpOr UntilIntro lift_imp_trans)
qed


subsection \<open>Omega and Pi\<close>

lemma OmegaImpDiamond: 
 "\<turnstile> (omega ((init w) \<Pi> (f \<and> more) \<and> sfin w)) \<longrightarrow> \<diamond> (init w) " 
proof -
 have 1: "\<turnstile> (omega ((init w) \<Pi> (f \<and> more) \<and> sfin w)) = 
            ((((init w) \<Pi> (f \<and> more) \<and> sfin w) \<and> more) \<and> finite);(omega ((init w) \<Pi> (f \<and> more) \<and> sfin w)) "
   using OmegaUnroll[of "LIFT ((init w) \<Pi> (f \<and> more) \<and> sfin w)"]
   by (simp add: schop_d_def) 
 have 2: "\<turnstile> ((((init w) \<Pi> (f \<and> more) \<and> sfin w) \<and> more) \<and> finite) \<longrightarrow> \<diamond> (init w) "
    using PiImpDiamond by fastforce
 have 3: "\<turnstile> ((((init w) \<Pi> (f \<and> more) \<and> sfin w) \<and> more) \<and> finite) =
            ((((init w) \<Pi> (f \<and> more) \<and> more)  \<and>  sfin w) \<and> finite) "
      by auto  
 have 4: "\<turnstile> ((((init w) \<Pi> (f \<and> more) \<and> sfin w) \<and> more) \<and> finite);(omega ((init w) \<Pi> (f \<and> more) \<and> sfin w)) =
            ((((init w) \<Pi> (f \<and> more)  \<and> more)  \<and> sfin w) \<and> finite);(omega ((init w) \<Pi> (f \<and> more) \<and> sfin w)) " 
   using "3" LeftChopEqvChop by blast
 have 5: "\<turnstile> ((((init w) \<Pi> (f \<and> more) \<and> more)  \<and>  sfin w) \<and> finite );(omega ((init w) \<Pi> (f \<and> more) \<and> sfin w)) 
            = (((init w) \<Pi> (f \<and> more) \<and> more) \<and> finite);((init w) \<and> omega ((init w) \<Pi> (f \<and> more) \<and> sfin w)) "
   using SlideInitSFin[of "LIFT ((init w) \<Pi> (f \<and> more) \<and> more)" w
                          "LIFT (omega ((init w) \<Pi> (f \<and> more) \<and> sfin w))"] unfolding schop_d_def 
    by (metis inteq_reflection) 
 have 6: "\<turnstile>  (((init w) \<Pi> (f \<and> more) \<and> more) \<and> finite);((init w) \<and> omega ((init w) \<Pi> (f \<and> more) \<and> sfin w)) 
             \<longrightarrow> \<diamond> (init w) " 
     by (metis ChopAndB FiniteChopImpDiamond inteq_reflection lift_and_com lift_imp_trans)
 show ?thesis using 1 3 4 5 6 by (metis int_eq)
qed


lemma PiStateOmegaUnroll: 
 "\<turnstile> (init w) \<Pi> (omega f) = ( ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> more) \<and> sfin w); ((init w) \<Pi> (omega f)) "
proof -
 have 1: "\<turnstile> (init w) \<Pi> (omega f) = (init w) \<Pi> (((f \<and> more) \<and> finite);(omega f)) "
 by (metis OmegaUnroll PiEqvRule schop_d_def) 
 have 2: "\<turnstile> (init w) \<Pi> (((f \<and> more) \<and> finite);(omega f)) =
            ((init w) \<Pi> (f \<and> more) \<and> finite);((init w) \<and> (init w) \<Pi> (omega f)) " 
  using PiSChopDist[of w "LIFT (f \<and> more)" "LIFT (omega f)"]  unfolding schop_d_def 
   by simp
 have 3: "\<turnstile> ((init w) \<Pi> (f \<and> more) \<and> finite) = ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> finite) " 
    by (meson PiFiniteAbsorb Prop11)
 have 4: "\<turnstile> ((init w) \<Pi> (f \<and> more) \<and> finite);((init w) \<and> (init w) \<Pi> (omega f)) =
            ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> finite);((init w) \<and> (init w) \<Pi> (omega f)) "
    by (simp add: "3" LeftChopEqvChop)
 have 5: "\<turnstile> ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> finite);((init w) \<and> (init w) \<Pi> (omega f)) =
            (((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w) \<and> finite);( (init w) \<Pi> (omega f)) "
   using SlideInitSFin[of "LIFT (init w) \<Pi> ((f \<and> more) \<and> finite)" w "LIFT (init w) \<Pi> (omega f)"]
    unfolding schop_d_def  by blast
 have 6: "\<turnstile> (((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w) \<and> finite) =
            (((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w)) " 
   using SFinEQvFinAndFinite by fastforce
 have 7: "\<turnstile> (init w) \<Pi> ((f \<and> more) \<and> finite) = ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> more) " 
   by (metis PiAnd PiAndPiImpPiAnd PiMoreAbsorb Prop10 Prop12 inteq_reflection)
 have 8: "\<turnstile> (((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w)) =
            ((((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> more)  \<and> sfin w)) " 
    using "7" by auto
 show ?thesis 
 by (metis "1" "2" "4" "5" "6" "8" int_eq)
qed


lemma  PiAOmega_sfin: 
assumes "\<not> nfinite l"
        " nnth l 0 = 0 "
        " nidx l"
        "  (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>)" 
shows   " ((nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> sfin w) =
           w (NNil (nnth \<sigma> (nnth l (Suc i))))  " 
using assms nidx_expand[of l] 
by (simp add: sfin_defs )
   (metis diff_add less_imp_le_nat linorder_le_cases ndropn_nlast nfinite_ntaken nsubn_def1 ntaken_all 
    ntaken_ndropn_nlast)  

lemma  PiAOmega_sfin_exist: 
assumes "\<not> nfinite l"
        " nnth l 0 = 0 "
        " nidx l"
        "  (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>)" 
        "w (NNil (nnth \<sigma> (nnth l (Suc i))))"
shows "(\<exists>x\<in>nset (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) . w (NNil x))" 
using assms nidx_expand[of l ] in_nset_conv_nnth[of _ "(nsubn \<sigma> (nnth l i) (nnth l (Suc i)))"]
unfolding nsubn_def1 
by (metis diff_add less_imp_le_nat linorder_le_cases nfinite_ntaken nset_nlast ntaken_all ntaken_ndropn_nlast)


lemma PiAOmega_help2: 
"(\<sigma> \<Turnstile> (aomega ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w))) =
 (\<exists>l. \<not> nfinite l \<and>
         nnth l 0 = 0 \<and>
         nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i.  ( 
                 ((nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<Turnstile> f) \<and>
                 0< nlength (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<and>
                 nfinite (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) 
                 \<and> w (NNil (nnth \<sigma> (nnth l (Suc i)))) 
               )   
         ) ) 
 " 
proof -
 have 1: " (\<sigma> \<Turnstile> (aomega ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w))) = 
          (\<exists>l. \<not> nfinite l \<and>
         nnth l 0 = 0 \<and>
         nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i. nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> init w \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w))"
 unfolding aomega_d_def by blast
 have 2: " (\<exists>l. \<not> nfinite l \<and>
         nnth l 0 = 0 \<and>
         nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i. nsubn \<sigma> (nnth l i) (nnth l (Suc i)) \<Turnstile> init w \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w)) =
         (\<exists>l. \<not> nfinite l \<and>
         nnth l 0 = 0 \<and>
         nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i.  ( (\<exists>x\<in>nset (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) . w (NNil x)) \<and> 
                 (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> (f \<and> more) \<and> finite)
                 \<and> ((nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> sfin w)
               )   
         ) ) "
  using Pistate[of w "LIFT ((f \<and> more) \<and> finite)"]  by auto
 have 3: "(\<exists>l. \<not> nfinite l \<and>
         nnth l 0 = 0 \<and>
         nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i.  ( (\<exists>x\<in>nset (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) . w (NNil x)) \<and> 
                 (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> (f \<and> more) \<and> finite)
                 \<and> ((nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> sfin w)
               )   
         ) ) = 
         (\<exists>l. \<not> nfinite l \<and>
         nnth l 0 = 0 \<and>
         nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i.  ( (\<exists>x\<in>nset (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) . w (NNil x)) \<and> 
                 ((nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<Turnstile> f) \<and>
                 0< nlength (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<and>
                 nfinite (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) 
                 \<and> ((nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> sfin w)
               )   
         ) )"
     unfolding more_defs finite_defs by simp
 have 4: "(\<exists>l. \<not> nfinite l \<and>
         nnth l 0 = 0 \<and>
         nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i.  ( (\<exists>x\<in>nset (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) . w (NNil x)) \<and> 
                 ((nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<Turnstile> f) \<and>
                 0< nlength (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<and>
                 nfinite (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) 
                 \<and> ((nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> sfin w)
               )   
         ) ) =
         (\<exists>l. \<not> nfinite l \<and>
         nnth l 0 = 0 \<and>
         nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i.  ( (\<exists>x\<in>nset (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) . w (NNil x)) \<and> 
                 ((nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<Turnstile> f) \<and>
                 0< nlength (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<and>
                 nfinite (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) 
                 \<and> w (NNil (nnth \<sigma> (nnth l (Suc i)))) 
               )   
         ) )" 
       using PiAOmega_sfin[of _  \<sigma> w] by blast
  have 5: "(\<exists>l. \<not> nfinite l \<and>
         nnth l 0 = 0 \<and>
         nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i.  ( (\<exists>x\<in>nset (nsubn \<sigma> (nnth l i) (nnth l (Suc i))) . w (NNil x)) \<and> 
                 ((nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<Turnstile> f) \<and>
                 0< nlength (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<and>
                 nfinite (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) 
                 \<and> w (NNil (nnth \<sigma> (nnth l (Suc i)))) 
               )   
         ) ) =
         (\<exists>l. \<not> nfinite l \<and>
         nnth l 0 = 0 \<and>
         nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i.  ( 
                 ((nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<Turnstile> f) \<and>
                 0< nlength (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<and>
                 nfinite (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) 
                 \<and> w (NNil (nnth \<sigma> (nnth l (Suc i)))) 
               )   
         ) )" 
     using PiAOmega_sfin_exist[of _ \<sigma> w] by blast
 show ?thesis 
 using "1" "2" "3" "4" "5" by presburger
qed


primcorec oml :: "('a \<Rightarrow> bool) \<Rightarrow> 'a nellist \<Rightarrow> nat nellist \<Rightarrow> nat \<Rightarrow> nat nellist "
 where 
  " oml P xs l x = 
     NCons (case x of 0 \<Rightarrow> 0 | 
                 Suc j \<Rightarrow> (the_enat (nlength(nfilter P (ntaken (nnth l x) xs)))))
          (oml P xs l (Suc x)) "


lemma oml_nnth_zero: 
 " nnth (oml P xs l 0) 0 = 0 " 
using oml.disc_iff oml.simps(2) nhd_conv_nnth
by (metis old.nat.simps(4))

lemma oml_nnth_one: 
assumes "\<not> nfinite l" 
        "\<not> nfinite xs " 
shows " nnth (oml P xs l 0) 1 = (the_enat (nlength(nfilter P (ntaken (nnth l 1) xs)))) " 
using assms 
oml.code oml.disc_iff oml.simps(2) nhd_conv_nnth nnth_Suc_NCons
by (metis One_nat_def old.nat.simps(5))

lemma oml_nnth: 
assumes "\<not> nfinite l" 
        "\<not> nfinite xs " 
shows " nnth (oml P xs l x) i = 
        (if x = 0 then 
         (if i = 0 then 0 else (the_enat (nlength(nfilter P (ntaken (nnth l i) xs)))))
         else (the_enat (nlength(nfilter P (ntaken (nnth l (i+x)) xs)))))  "
using assms
proof (induct i arbitrary: x)
case 0
then show ?case 
  by (metis Nitpick.case_nat_unfold ndropn_0 ndropn_nnth nhd_conv_nnth oml.disc_iff oml.simps(2))
next
case (Suc i)
then show ?case 
 by (metis One_nat_def Zero_not_Suc add.commute add_Suc_shift nnth_Suc_NCons oml.code plus_1_eq_Suc) 
qed 

lemma oml_infinite: 
 assumes "\<not> nfinite l" 
        "\<not> nfinite xs " 
shows " \<not> nfinite (oml P xs l x) " 
proof 
 assume " nfinite (oml P xs l x) "
 thus False
  using assms 
   proof (induct zs\<equiv>" (oml P xs l x) " arbitrary: x rule: nfinite_induct)
   case (NNil y)
   then show ?case by (metis nellist.disc(1) oml.disc_iff)
   next
   case (NCons x nell)
   then show ?case by (metis nellist.sel(5) oml.simps(3))
   qed
qed 


lemma PiAOmegaImpSem: 
 assumes "(\<sigma> \<Turnstile> (aomega ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w))) "
 shows  "(\<sigma> \<Turnstile> (init w) \<Pi> (aomega f))"
proof -
 have 1: "(\<exists>l. \<not> nfinite l \<and>
         nnth l 0 = 0 \<and>
         nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i.  ( 
                 ((nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<Turnstile> f) \<and>
                 0< nlength (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<and>
                 nfinite (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) 
                 \<and> w (NNil (nnth \<sigma> (nnth l (Suc i)))) 
               )   
         ) ) "
  using assms  PiAOmega_help2[of w f \<sigma>] by blast
 obtain l where 2: "\<not> nfinite l \<and>
         nnth l 0 = 0 \<and>
         nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i.  ( 
                 ((nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<Turnstile> f) \<and>
                 0< nlength (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) \<and>
                 nfinite (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))) 
                 \<and> w (NNil (nnth \<sigma> (nnth l (Suc i)))) 
               )   
         )"
   using 1 by auto
 have 3: "\<And>j. 0<j \<longrightarrow> w  (NNil (nnth \<sigma> (nnth l j))) "
   using 2 by (metis Suc_diff_1)
 have 31: "\<And>j. 0<j \<longrightarrow> nfinite (nfilter (\<lambda>y. w (NNil y)) (ntaken (nnth l j) \<sigma>))  "
    by (metis "2" "3" nfilter_chop1_ntaken nfinite_ntaken ntaken_nlast) 
 have 32: "\<And>j. (nnth l j) < (nnth l (Suc j))"
    by (metis "2" linorder_le_cases nfinite_ntaken nidx_expand ntaken_all)
 have 4: "\<And>j. 0<j \<longrightarrow> 
               (nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> (nnth l j) (nnth l (Suc j)))) =
               (nsubn (nfilter (\<lambda>y. w (NNil y)) \<sigma> )
                      (the_enat (nlength(nfilter (\<lambda>y. w (NNil y)) (ntaken (nnth l j) \<sigma>))))
                      (the_enat (nlength(nfilter (\<lambda>y. w (NNil y)) (ntaken (nnth l (Suc j)) \<sigma>))))
               )
               " 
   using 2 3 32 nfilter_nsubn[of _ \<sigma> "(\<lambda>y. w (NNil y))"] 
   by (metis "31" less_imp_le_nat nfinite_nlength_enat the_enat.simps zero_less_Suc)
 have 5: "(nfilter (\<lambda>y. w (NNil y)) (nsubn \<sigma> 0 (nnth l (Suc 0)))) =
           (nsubn (nfilter (\<lambda>y. w (NNil y)) \<sigma> )
                      0
                      (the_enat (nlength(nfilter (\<lambda>y. w (NNil y)) (ntaken (nnth l (Suc 0)) \<sigma>))))
               )"
   by (simp add: "2" nfilter_nsubn_zero)
 have 6: "0< (the_enat (nlength(nfilter (\<lambda>y. w (NNil y)) (ntaken (nnth l (Suc 0)) \<sigma>))))" 
  by (metis "2" "5" gr0I i0_less nlength_NNil nsubn_def1 ntaken_0 zero_less_diff)
 have 7: "\<And>j. 0<j \<longrightarrow> (the_enat (nlength(nfilter (\<lambda>y. w (NNil y)) (ntaken (nnth l j) \<sigma>)))) <
                       (the_enat (nlength(nfilter (\<lambda>y. w (NNil y)) (ntaken (nnth l (Suc j)) \<sigma>))))"
   by (metis "2" "4" bot_nat_0.not_eq_extremum i0_less nlength_NNil nsubn_def1 ntaken_0 zero_less_diff)
 have 8: "\<not> nfinite (oml (\<lambda>y. w (NNil y)) \<sigma> l 0) " 
   using "2" infinite_nidx_imp_infinite_interval oml_infinite by blast
 have 9: "nnth (oml (\<lambda>y. w (NNil y)) \<sigma> l 0) 0 = 0 "
    by (simp add: oml_nnth_zero) 
 have 10: "\<And>j. 0< j \<longrightarrow> nnth (oml (\<lambda>y. w (NNil y)) \<sigma> l 0) j = 
                (the_enat (nlength(nfilter (\<lambda>y. w (NNil y)) (ntaken (nnth l j) \<sigma>)))) " 
   by (metis "2" infinite_nidx_imp_infinite_interval less_not_refl2 oml_nnth)
 have 11: "\<And>j . nnth (oml (\<lambda>y. w (NNil y)) \<sigma> l 0) j < nnth (oml (\<lambda>y. w (NNil y)) \<sigma> l 0) (Suc j)" 
   by (metis "10" "6" "7" "9" bot_nat_0.not_eq_extremum zero_less_Suc)
 have 12: "nidx (oml (\<lambda>y. w (NNil y)) \<sigma> l 0)"
   using nidx_expand[of "(oml (\<lambda>y. w (NNil y)) \<sigma> l 0)"] 
   using "11" by blast
 have 13: "(\<forall>i. enat (nnth (oml (\<lambda>y. w (NNil y)) \<sigma> l 0) i) \<le> nlength (nfilter (\<lambda>y. w (NNil y)) \<sigma>))"   
    by (metis "10" "3" bot_nat_0.not_eq_extremum enat_ile le_zero_eq linorder_le_cases 
        nfilter_chop1_ntaken ntaken_all ntaken_nlast oml_nnth_zero the_enat.simps zero_enat_def)
 have 14: "(\<forall>i. f (nsubn (nfilter (\<lambda>y. w (NNil y)) \<sigma>) 
                         (nnth (oml (\<lambda>y. w (NNil y)) \<sigma> l 0) i) 
                         (nnth (oml (\<lambda>y. w (NNil y)) \<sigma> l 0) (Suc i)))) " 
  by (metis "10" "2" "4" "5" "9" bot_nat_0.not_eq_extremum zero_less_Suc)
 have 15: "(\<exists>x\<in>nset \<sigma>. w (NNil x))" 
   by (meson "2" in_nset_conv_nnth)
 have 16: "((\<exists>x\<in>nset \<sigma>. w (NNil x)) \<and>
   (\<exists>l. \<not> nfinite l \<and>
        nnth l 0 = 0 \<and>
        nidx l \<and>
        (\<forall>i. enat (nnth l i) \<le> nlength (nfilter (\<lambda>y. w (NNil y)) \<sigma>)) \<and>
        (\<forall>i. f (nsubn (nfilter (\<lambda>y. w (NNil y)) \<sigma>) (nnth l i) (nnth l (Suc i))))))"
     using "12" "13" "14" "15" "8" oml_nnth_zero by blast
 have 17: "(\<sigma> \<Turnstile> (init w) \<Pi> (aomega f))"
  using 16  
   using Pistate[of w "LIFT aomega f" \<sigma>] unfolding aomega_d_def by blast
 show ?thesis using 17 by auto
qed
 
lemma PiAOmegaImp:
 "\<turnstile> (aomega ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w)) \<longrightarrow> (init w) \<Pi> (aomega f)"
unfolding Valid_def using PiAOmegaImpSem 
using unl_lift2 by blast


lemma PiOmegaImpSem: 
 assumes "(\<sigma> \<Turnstile> (omega ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w))) "
 shows  "(\<sigma> \<Turnstile> (init w) \<Pi> (omega f))"
by (metis OmegaEqvAOmega PiAOmegaImpSem assms inteq_reflection)

lemma PiOmegaImp:
 "\<turnstile> (omega ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w)) \<longrightarrow> (init w) \<Pi> (omega f)"
unfolding Valid_def using PiOmegaImpSem  using unl_lift2 by blast


lemma PiOmega:
 "\<turnstile> (init w) \<Pi> (omega f) =  (omega ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w)) " 
proof -
 have 0: "\<turnstile> (((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> more) \<and> sfin w) =
            ((((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w) \<and> more) \<and> finite) "
    using SFinEQvFinAndFinite by fastforce
 have 1: "\<turnstile> (init w) \<Pi> (omega f) \<longrightarrow> inf"
    by (meson PiImpRule PiInfImpInf lift_imp_trans omega_imp_inf)
 have 2: "\<turnstile> (init w) \<Pi> (omega f) \<longrightarrow> 
            inf \<and> (((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w) \<and> fmore);((init w) \<Pi> (omega f))"
   by (metis "1" AndMoreSChopEqvAndFmoreChop 0 PiStateOmegaUnroll Prop12 int_eq int_iffD1 schop_d_def)
 have 3: "\<turnstile> (init w) \<Pi> (omega f) \<longrightarrow> (omega ((init w) \<Pi> ((f \<and> more) \<and> finite) \<and> sfin w))"
   using OmegaIntro by (metis "2" Prop12)
 have 4: "\<turnstile> (omega ((init w) \<Pi> ((f \<and> more) \<and> finite)  \<and> sfin w)) \<longrightarrow> (init w) \<Pi> (omega f)" 
    by (simp add: PiOmegaImp)
 show ?thesis 
 by (simp add: "3" "4" int_iffI)
qed


end