(*  Title     : A Shallow Encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)


theory Omega
imports  Chopstar
begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="
  cvc4 z3 e spass vampire verit
"]
*)

text \<open>
This theory  defines the omega operator for infinite ITL and  provides a library of lemmas.
We also define a weak version womega that corresponds to the omega operator from Omega Algebra
\cite{Georg:Klealg}. We also provide a semantic version aomega and provide lemmas that express 
various relationships between them.  
We also ported the numerous omega algebra lemmas from \cite{Georg:Klealg} to ITL. 
\<close>

section \<open>Omega and variants\<close>

subsection \<open>Definitions\<close>

subsubsection \<open>Weak Omega\<close>

lemma MoreChopSemMono [mono]:
 "(w \<Turnstile>  (f \<and> more) ;g) =
  (  (\<exists>n. 0<n \<and> enat n \<le> nlength w \<and> f ( (ntaken n w))  \<and> g  (ndropn n  w)) \<or>
    (\<not>nfinite w \<and> f w)) "
by (simp add: itl_defs)
 (metis bot_nat_0.not_eq_extremum le_zero_eq nlength_eq_enat_nfiniteD the_enat.simps zero_enat_def)

coinductive womega_d :: "('a::world) formula \<Rightarrow> 'a formula" 
 for F where 
 " (s \<Turnstile>  (F \<and> more);(womega_d F)) \<Longrightarrow>  (s \<Turnstile> (womega_d F))  "



syntax 
 "_womega_d" :: "lift \<Rightarrow> lift"        ("(_\<^sup>\<W>)" [85] 85)

syntax (ASCII)
 "_womega_d" :: "lift \<Rightarrow> lift"        ("(womega _)" [85] 85)

translations
 "_womega_d"        \<rightleftharpoons> "CONST womega_d" 


lemma WOmegaIntros:
 " \<turnstile>  (f \<and> more);(womega f) \<longrightarrow> (womega f)"
using womega_d.intros using unl_lift2 by blast

lemma WOmegaCases:
 " (w \<Turnstile> (womega F)) \<Longrightarrow> 
   (w \<Turnstile>   (F \<and> more);(womega F) \<Longrightarrow> P) \<Longrightarrow> P" 
using  womega_d.cases[of F w P] by auto

lemma  WOmegaUnrollSem:
 "(s\<Turnstile> (womega f)) = (s \<Turnstile> (f \<and> more);(womega f))" 
 using womega_d.cases[of f ]
by (metis womega_d.simps)

lemma  WOmegaUnroll:
 "\<turnstile> (womega f) = (f \<and> more);(womega f) " 
using WOmegaUnrollSem 
using unl_lift2 by blast

lemma WOmegaCoinductSem:
 assumes "\<And>x.  X x \<Longrightarrow> x \<Turnstile>  (F \<and> more);(X  \<or> womega F)" 
 shows   " x  \<Turnstile> X   \<longrightarrow> womega F "
using assms womega_d.coinduct[of X x F]
by (auto simp add: chop_defs)

lemma WOmegaCoinduct:
  assumes "\<turnstile> X \<longrightarrow>  (F \<and> more);(X \<or> (womega F))"
  shows   "\<turnstile> X  \<longrightarrow> (womega F)"
using assms WOmegaCoinductSem[of X F]
by (simp add: Valid_def) 

lemma WOmegaWeakCoinductSem:
 assumes "\<And>x.  x \<Turnstile> X \<Longrightarrow> x \<Turnstile>  (F \<and> more);X" 
 shows " x  \<Turnstile> X   \<longrightarrow> womega F "
using assms womega_d.coinduct[of X x F]
by (auto simp add: chop_defs) 
  
lemma WOmegaWeakCoinduct:
  assumes "\<turnstile> X \<longrightarrow>  (F \<and> more);X "
  shows   "\<turnstile> X  \<longrightarrow> (womega F)"
using assms WOmegaWeakCoinductSem[of X F]
by (simp add: Valid_def) 


subsubsection \<open>Omega\<close>

lemma FMoreSem_var [ mono]:
 "(w \<Turnstile>  ((f \<and> more) \<and> finite);g) =
  ((0< nlength w  \<and> (\<exists>n. f ( (ntaken n w)) \<and> 0<n \<and> g  (ndropn n  w))) )"
by  (simp add: itl_defs)
    (metis enat_0_iff(1) linorder_not_less ndropn_all neq0_conv 
     nfinite_nlength_enat nfinite_ntaken ntaken_all order_le_less)

definition omega_d :: "('a::world) formula \<Rightarrow> 'a formula" 
 where "omega_d F \<equiv> LIFT(womega(F \<and> finite)) " 
(*
coinductive omega_d :: "('a::world) formula \<Rightarrow> 'a formula" 
 for F where 
 " (s \<Turnstile>  ((F \<and> more) \<and> finite);(omega_d F)) \<Longrightarrow>  (s \<Turnstile> (omega_d F))  "
*)

syntax 
 "_omega_d" :: "lift \<Rightarrow> lift"        ("(_\<^sup>\<omega>)" [85] 85)

syntax (ASCII)
 "_omega_d" :: "lift \<Rightarrow> lift"        ("(omega _)" [85] 85)

translations
 "_omega_d"        \<rightleftharpoons> "CONST omega_d" 

lemma Chop_more_finite_swap: 
 "\<turnstile> ((f \<and> more) \<and> finite) = ((f \<and> finite) \<and> more)"
by fastforce 

lemma OmegaIntros:
 " \<turnstile>  (f \<and> more)\<frown>(omega f) \<longrightarrow> (omega f)"
unfolding omega_d_def schop_d_def 
by (meson Chop_more_finite_swap LeftChopImpChop Prop11 WOmegaIntros lift_imp_trans)

lemma OmegaCases:
 " (w \<Turnstile> (omega F)) \<Longrightarrow> 
   (w \<Turnstile>   (F \<and> more)\<frown>(omega F) \<Longrightarrow> P) \<Longrightarrow> P" 
unfolding omega_d_def schop_d_def using womega_d.cases[of "LIFT (F \<and> finite)" w]
by (metis Chop_more_finite_swap inteq_reflection)

lemma OmegaUnrollSem:
 "(s\<Turnstile> (omega f)) = (s \<Turnstile> (f \<and> more)\<frown>(omega f))"
unfolding  omega_d_def schop_d_def
by (metis Chop_more_finite_swap WOmegaUnroll inteq_reflection)

lemma OmegaCoinductSem:
 assumes "\<And>x.  X x \<Longrightarrow> x \<Turnstile>  (F \<and> more)\<frown>(X  \<or> omega F)" 
 shows   " x  \<Turnstile> X   \<longrightarrow> omega F "
using assms unfolding omega_d_def schop_d_def 
by (metis Chop_more_finite_swap WOmegaCoinductSem inteq_reflection)

lemma OmegaWeakCoinductSem:
 assumes "\<And>x.  X x \<Longrightarrow> x \<Turnstile>  (F \<and> more)\<frown>X" 
 shows " x  \<Turnstile> X   \<longrightarrow> omega F "
using assms unfolding omega_d_def schop_d_def 
by (metis Chop_more_finite_swap WOmegaWeakCoinductSem inteq_reflection)

lemma OmegaUnroll:
    "\<turnstile> f\<^sup>\<omega> = (f \<and> more)\<frown>f\<^sup>\<omega>"
using OmegaUnrollSem unl_lift2 by blast

lemma OmegaCoinduct:
  assumes "\<turnstile> X \<longrightarrow>  (F \<and> more)\<frown>(X \<or> (omega F))"
  shows   "\<turnstile> X  \<longrightarrow> (omega F)"
using assms OmegaCoinductSem[of X F]
by (simp add: Valid_def) 

lemma OmegaWeakCoinduct:
  assumes "\<turnstile> X \<longrightarrow>  (F \<and> more)\<frown>X "
  shows   "\<turnstile> X  \<longrightarrow> (omega F)"
using assms OmegaWeakCoinductSem[of X F]
by (simp add: Valid_def) 


subsubsection \<open>Alternative definition for Omega\<close>

 lemma infinite_nidx_imp_infinite_interval: 
assumes "\<not> nfinite l "
        "nidx l"
        "(nnth l 0) = 0"
        "\<forall>i. (nnth l i) \<le> nlength s"
shows  "\<not> nfinite s" 
proof 
 assume "nfinite s"
 thus False
   using assms
  proof (induct s  rule: nfinite_induct)
  case (NNil y)
  then show ?case
  by (metis dual_order.antisym enat_ord_simps(1) linorder_linear nfinite_ntaken nidx_gr_first nlength_NNil 
      not_gr_zero ntaken_all zero_le zero_less_Suc) 
  next
  case (NCons x nell)
  then show ?case 
    proof -
     have 1: "\<And>j. (nnth l j) < (nnth l (Suc j)) "
       by (metis assms(1) assms(2) linorder_le_cases nfinite_ntaken nidx_expand ntaken_all)
     have 2: "\<forall>i. enat (nnth l i) \<le> nlength (NCons x nell) \<Longrightarrow> False" 
      by (metis "1" NCons.hyps(2) assms(1) assms(2) assms(3) dual_order.strict_iff_order 
          enat_ord_simps(1) iless_Suc_eq linorder_not_le nlength_NCons)
     show ?thesis 
     using "2" NCons.prems(4) by auto
  qed 
 qed
qed

definition aomega_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "aomega_d F \<equiv> 
       (\<lambda>s. 
          (\<exists>(l:: nat nellist). 
              \<not>nfinite l \<and> (nnth l 0) = 0 \<and> nidx l \<and> 
              (\<forall>i. (nnth l i) \<le> nlength s) \<and>
              (\<forall>i. ( (nsubn s (nnth l i) (nnth l (Suc i))) \<Turnstile> F) ) 
           )
        )"


syntax 
 "_aomega_d"        :: "lift \<Rightarrow> lift"        ("(aomega _)" [85] 85) 

syntax (ASCII)
 "_aomega_d"        :: "lift \<Rightarrow> lift"        ("(aomega _)" [85] 85) 

translations
 "_aomega_d"        \<rightleftharpoons> "CONST aomega_d" 


lemma aomega_unroll_chain_a:
 assumes "(\<exists>l. \<not> nfinite l \<and> (nnth l 0) = 0 \<and> nidx l \<and> 
           (\<forall>i. (nnth l i) \<le> nlength \<sigma>) \<and>
           (\<forall>i. f ( (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))))) "
 shows "(\<exists>n. enat n \<le> nlength \<sigma> \<and>
        f ( (ntaken n \<sigma>)) \<and>
        0 < n \<and>
        enat 0 < nlength \<sigma> \<and>
        (\<exists>l.
            \<not> nfinite l \<and> (nnth l 0) = 0 \<and> nidx l  \<and> 
            (\<forall>i. (nnth l i) \<le> nlength \<sigma> - enat n) \<and>
            (\<forall>i. f ( (nsubn (ndropn n \<sigma>) (nnth l i) (nnth l (Suc i)))))
        )
   ) " 
proof -
 obtain l where 1: "\<not>nfinite l \<and>  (nnth l 0) = 0 \<and> nidx  l \<and> 
                    (\<forall>i. (nnth l i) \<le> nlength \<sigma>) \<and> (\<forall>i. f ( (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))))" 
 using assms by auto
 have 2: "l = NCons (nnth l 0) (ndropn 1 l)  "
     by (metis "1" One_nat_def gr_zeroI ndropn_0 ndropn_Suc_conv_ndropn nlength_eq_enat_nfiniteD 
         zero_enat_def)
 have 3: "(nnth l 0) = 0"
  using "1"  by blast   
 have 4: "(nnth l 0) < (nnth l 1)" 
  using "1" 
  by (metis One_nat_def Suc_ile_eq enat_defs(1) nidx_gr_first nlength_eq_enat_nfiniteD 
      not_gr_zero zero_less_one)
 have 5: " nidx (ndropn 1 l) "
  using "1"  nidx_expand[of l ] nidx_expand[of "(ndropn 1 l)"] 
  by (metis dual_order.order_iff_strict enat_defs(1) ndropn_all ndropn_nnth nfinite_ndropn_b 
      nlength_NNil nlength_eq_enat_nfiniteD not_le_imp_less plus_1_eq_Suc)
 have 6: "f ( (nsubn \<sigma> (nnth l 0) (nnth l 1)))"
    by (metis "1" One_nat_def)
 have 7: "(\<forall>i. f ( (nsubn \<sigma>  (nnth (ndropn 1 l) i) (nnth (ndropn 1 l) (Suc i)))))"
   by (simp add: "1") 
 have 8: "f ( (ntaken (nnth l 1) \<sigma>))"
   by (metis "1" "4" One_nat_def Suc_diff_1 Suc_diff_Suc ndropn_0 nsubn_def1)
 have 81: "\<not> nfinite (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l)) "
     by (simp add: "1")
 have 82: "\<And>j. 0 < j \<longrightarrow> (nnth l 1) < nnth (ndropn 1 l) j "
    by (metis "1" Suc_diff_1 enat_defs(1) linorder_le_cases ndropn_all ndropn_nnth nfinite_ndropn_b 
        nidx_less nlength_NNil nlength_eq_enat_nfiniteD  plus_1_eq_Suc)
 have 83: "\<And>j. nnth (nmap (\<lambda>e. e - nnth l 1) (ndropn 1 l)) j =
               (nnth l (Suc j)) - (nnth l 1) "
     by (metis "81" le_cases ndropn_nnth nfinite_ntaken nlength_nmap nnth_nmap  ntaken_all plus_1_eq_Suc) 
 have 84: "\<And>j. nnth (nmap (\<lambda>e. e - nnth l 1) (ndropn 1 l)) (Suc j) =
               (nnth l (Suc (Suc j))) - (nnth l 1) "
    using "83" by blast
 have 840: " \<And>j. (nnth l 1) \<le> (nnth l (Suc j))"
    by (metis "1" diff_add_zero diff_is_0_eq linorder_le_cases nfinite_ntaken nidx_less_eq ntaken_all 
        plus_1_eq_Suc)
 have 85: "\<And>j. (nnth l (Suc j)) - (nnth l 1) < (nnth l (Suc (Suc j))) - (nnth l 1) "
    by (metis "1" "840" diff_less_mono linorder_le_cases nfinite_ntaken nidx_expand ntaken_all)
 have 86: "(\<forall>i. enat (Suc i) \<le> nlength (nmap (\<lambda>e. e - nnth l 1) (ndropn 1 l)) \<longrightarrow>
            nnth (nmap (\<lambda>e. e - nnth l 1) (ndropn 1 l)) i < 
            nnth (nmap (\<lambda>e. e - nnth l 1) (ndropn 1 l)) (Suc i))"
       using "83" "85" by presburger
 have 9: " nidx  (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l))   "
     using nidx_expand[of "(nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l))"]   
      using "83" "85" by presburger
 have 91: "\<And>j. (nnth (nmap (\<lambda>e. e + (nnth l 1)) (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l))) j) =
                 (nnth l (Suc j))  "
   by (metis  "81" "82" "83" One_nat_def  add_Suc diff_Suc_1 diff_Suc_eq_diff_pred 
      diff_add diff_is_0_eq le_refl linorder_le_cases ndropn_nnth nfinite_ntaken nlength_nmap 
       nnth_nmap not_le_imp_less ntaken_all order_less_imp_le plus_1_eq_Suc) 
 have 10: "(\<forall>i. f ( (nsubn \<sigma>  
                    (nnth (nmap (\<lambda>e. e + (nnth l 1)) (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l))) i)
                    (nnth (nmap (\<lambda>e. e + (nnth l 1)) (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l))) (Suc i)) ))) "
     using "1" "91" by presburger
 have 11: "(\<forall>i. f ( (nsubn \<sigma>  
                    ( (nnth (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l)) i)  + (nnth l 1) )
                    ( (nnth (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l)) (Suc i))  + (nnth l 1) )
                     ) ) )"
    using "7" "83" "840" by fastforce
 have 12: "(\<forall>i. f ( (nsubn (ndropn (nnth l 1) \<sigma>)  
                    ( (nnth (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l)) i)  )
                    ( (nnth (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l)) (Suc i))  ) 
                      ) ) )"
     by (metis "11" "83" "85" nsubn_ndropn)
 have 121: " nnth (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l)) 0 = 0 "
     using "83" One_nat_def by presburger
 have 122: " \<not> nfinite (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l)) \<and> 
             nnth (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l)) 0 = 0 \<and>
             nidx  (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l)) \<and>
             (\<forall>i. f ( (nsubn (ndropn (nnth l 1) \<sigma>)  
                    ( (nnth (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l)) i)  )
                    ( (nnth (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l)) (Suc i))  ) 
                      ) ) )"
      using "12" "121" "81" "9" by blast
 have 123: " (\<forall>i. (nnth (nmap (\<lambda> e. e - (nnth l 1)) (ndropn 1 l)) i) \<le> nlength (ndropn (nnth l 1) \<sigma>)) "
      by (meson "1" enat_ile infinite_nidx_imp_infinite_interval le_cases nfinite_ndropn_b 
      nlength_eq_enat_nfiniteD)
 have 13: " (\<exists>ls.
            \<not> nfinite ls \<and> (nnth ls 0) = 0 \<and> nidx ls  \<and> 
            (\<forall>i. (nnth ls i) \<le> nlength (ndropn (nnth l 1) \<sigma>)) \<and>
            (\<forall>i. f ( (nsubn (ndropn (nnth l 1) \<sigma>) (nnth ls i) (nnth ls (Suc i)))))
        )"
    using "122" 123 by blast
 from 13 show ?thesis using "4" "8" 
 by (metis "1" enat_ord_simps(1) linorder_not_less ndropn_nlength not_gr_zero zero_enat_def)
qed   


lemma aomega_unroll_chain_b:
 assumes "(\<exists>n. enat n \<le> nlength \<sigma> \<and>
             f ( (ntaken n \<sigma>)) \<and>
            0 < n \<and>
            enat 0 < nlength \<sigma> \<and>
        (\<exists>l.
            \<not>nfinite l \<and> (nnth l 0) = 0 \<and> nidx l \<and> 
            (\<forall>i. (nnth l i) \<le> nlength \<sigma> - enat n) \<and>
            (\<forall>i. f ( (nsubn (ndropn n \<sigma>) (nnth l i) (nnth l (Suc i)))))
        )
   ) "
 shows "(\<exists>l. \<not> nfinite l \<and> (nnth l 0) = 0 \<and> nidx l \<and> 
         (\<forall>i. (nnth l i) \<le> nlength \<sigma>) \<and> 
         (\<forall>i. f ( (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))))) " 
proof -
 obtain n where 1: "enat n \<le> nlength \<sigma> \<and> f ( (ntaken n \<sigma>)) \<and>
                    0 < n \<and>  enat 0 < nlength \<sigma> \<and>
        (\<exists>l.
            \<not>nfinite l \<and> (nnth l 0) = 0 \<and> nidx l \<and> 
            (\<forall>i. (nnth l i) \<le> nlength \<sigma> - enat n) \<and>
            (\<forall>i. f ( (nsubn (ndropn n \<sigma>) (nnth l i) (nnth l (Suc i)))))
        )" 
  using assms by auto
 have 2: "(\<exists>l.
            \<not>nfinite l \<and> (nnth l 0) = 0 \<and> nidx l \<and> 
            (\<forall>i. (nnth l i) \<le> nlength (ndropn n \<sigma>)) \<and>
            (\<forall>i. f ( (nsubn (ndropn n \<sigma>) (nnth l i) (nnth l (Suc i)))))
        )" 
 using 1 by auto
 obtain l where 3: "\<not>nfinite l \<and> (nnth l 0) = 0 \<and> nidx l \<and> 
            (\<forall>i. (nnth l i) \<le> nlength (ndropn n \<sigma>)) \<and>
            (\<forall>i. f ( (nsubn (ndropn n \<sigma>) (nnth l i) (nnth l (Suc i)))))" 
  using 2 by auto
 have 4: "nidx (nmap (\<lambda>e. e + n) l) "
   using "3" 
   by (simp add: Suc_ile_eq nidx_expand) 
 have 42: "\<And>j.  nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) j = 
                   (case j of  0 \<Rightarrow> 0 |
                     (Suc k) \<Rightarrow> nnth (nmap (\<lambda>e. e + n) l) k ) " 
       by (simp add: nnth_NCons)
 have 43: "\<And>j. nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) (Suc j) = 
                nnth (nmap (\<lambda>e. e + n) l) j " 
       by simp
 have 44: " 0 < nnth (nmap (\<lambda>e. e + n) l) 0    " 
    using "1" enat_defs(1) by auto
 have 45: "\<And>j.  nnth (nmap (\<lambda>e. e + n) l) j < nnth (nmap (\<lambda>e. e + n) l) (Suc j) "
    by (metis "3" "4" add.right_neutral le_cases nfinite_ntaken nidx_less nlength_nmap ntaken_all)
 have 46: "(\<forall>i. enat (Suc i) \<le> nlength (NCons 0 (nmap (\<lambda>e. e + n) l)) \<longrightarrow>
                nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) i < 
                nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) (Suc i))"
   by (metis "43" "44" "45" lessI less_Suc_eq_0_disj nnth_0)
 have 5: "nidx (NCons 0 (nmap (\<lambda>e. e + n) l))"
   using "46" nidx_expand by blast
 have 6: "( (ntaken n \<sigma>)) =
          ( (nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) 0) 
                     (nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) (Suc 0)) )) "
     by (metis "3" "43" "44" One_nat_def Suc_diff_1 Suc_diff_Suc add_0 ndropn_0 nnth_0 nnth_nmap 
         nsubn_def1 zero_enat_def zero_le)
 have 7: "(\<forall>i. f ( (nsubn (ndropn n \<sigma>) (nnth l i) (nnth l (Suc i)))))" 
  using 3 by auto
 have 8: "(\<forall>i. f ( (nsubn \<sigma> ((nnth l i)+n) ((nnth l (Suc i)) +n) )))" 
   using 7 
   by (simp add: add.commute ndropn_ndropn nsubn_def1)
 have 9: "(\<forall>i. f ( (nsubn \<sigma> ( (nnth (nmap (\<lambda>e. e + n) l) i)) 
                            ( (nnth (nmap (\<lambda>e. e + n) l)  (Suc i))) ) ))  "
   using 8 
   by (metis "45" Suc_ile_eq dual_order.order_iff_strict linorder_le_cases nat_neq_iff nlength_nmap 
       nnth_beyond nnth_nmap)
 have 10: "f ( (nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) 0)
                        (nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) (Suc 0)))) "
  using "1" "6" by auto 
 have 11: "(\<forall>i. i>0 \<longrightarrow>
        f ( (nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) i)
                     (nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) (Suc i)))))"
    by (metis "9" Suc_diff_1 nnth_Suc_NCons)  
 have 12: "(\<forall>i. 
             f ( (nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) i)
                          (nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) (Suc i)))))" 
  using 10 11 neq0_conv by blast
 have 13: "\<forall>i. (nnth (NCons 0 (nmap (\<lambda>e. e + n) l)) i) \<le> nlength \<sigma> " 
      by (metis "3" add.commute enat_ile infinite_nidx_imp_infinite_interval linorder_le_cases ndropn_all 
          ndropn_ndropn nfinite_ndropn_b nlength_NNil nlength_eq_enat_nfiniteD zero_le)
 show ?thesis using "12" "5" 
 by (metis "13" "3" nfinite_NCons nfinite_nmap nnth_0)
qed 

lemma aomega_unroll_chain:
 "(\<exists>l. \<not> nfinite l \<and> (nnth l 0) = 0 \<and> nidx l \<and> 
     (\<forall>i. (nnth l i) \<le> nlength \<sigma>) \<and>
     (\<forall>i. f ( (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))))) 
  =
   (\<exists>n. enat n \<le> nlength \<sigma> \<and>
        f ((ntaken n \<sigma>)) \<and>
        0 < n \<and>
        enat 0 < nlength \<sigma> \<and>
        (\<exists>l.
            \<not> nfinite l \<and> (nnth l 0) = 0 \<and> nidx l \<and> 
            (\<forall>i. (nnth l i) \<le> nlength \<sigma> - enat n) \<and>
            (\<forall>i. f ( (nsubn (ndropn n \<sigma>) (nnth l i) (nnth l (Suc i)))))
        )
   ) 
    "
using aomega_unroll_chain_a[of \<sigma> f]  aomega_unroll_chain_b[of \<sigma> f] 
by blast
      
lemma aomega_unroll_sem:
 "(\<sigma> \<Turnstile> ((f \<and> more) \<and> finite);(aomega f) = (aomega f))"
proof 
 (simp add: itl_defs zero_enat_def aomega_d_def) 
 show "((\<exists>n. enat n \<le> nlength \<sigma> \<and>
          f (ntaken n \<sigma>) \<and>
          0 < n \<and>
          enat 0 < nlength \<sigma> \<and>
          (\<exists>l. \<not> nfinite l \<and> nnth l 0 = 0 \<and> nidx l \<and> 
              (\<forall>i. enat (nnth l i) \<le> nlength \<sigma> - enat n) \<and> 
              (\<forall>i. f (nsubn (ndropn n \<sigma>) (nnth l i) (nnth l (Suc i)))))) \<or>
     \<not> nfinite \<sigma> \<and> f \<sigma> \<and> enat 0 < nlength \<sigma> \<and> nfinite \<sigma>) =
    (\<exists>l. \<not> nfinite l \<and> nnth l 0 = 0 \<and> nidx l \<and> (\<forall>i. enat (nnth l i) \<le> nlength \<sigma>) \<and> 
           (\<forall>i. f (nsubn \<sigma> (nnth l i) (nnth l (Suc i)))))  " 
   using aomega_unroll_chain[of \<sigma> f] by blast
qed

lemma AOmegaUnroll: 
 "\<turnstile> (aomega f) = ((f \<and> more) \<and> finite);(aomega f)"
unfolding Valid_def 
using aomega_unroll_sem by auto 

lemma AOmegaInductSem_help:
 "(\<sigma> \<Turnstile>  inf \<and> g \<and> \<box>(g \<longrightarrow> ((f \<and> more) \<and> finite);g)) =
  ( \<not> nfinite \<sigma> \<and> g \<sigma> \<and>
               (\<forall>n. g ( (ndropn n \<sigma>)) \<longrightarrow>
                       (\<exists>na. f ( (nsubn \<sigma> n (na+n))) \<and> 
                            0 < na \<and> g ( (ndropn (n + na) \<sigma>))))
   ) 
  "
by (simp add: itl_defs zero_enat_def min_def ndropn_ndropn nsubn_def1)
   (metis linorder_le_cases ndropn_all ndropn_nlength nfinite_NNil nfinite_ndropn_b )


primrec cpoint :: " ('a::world) formula  \<Rightarrow> 'a formula \<Rightarrow> nat \<Rightarrow> 'a intervals \<Rightarrow> nat"
 where "cpoint f g 0 \<sigma> = 0"
       | "cpoint f g (Suc n) \<sigma> =
          (\<some> x. (\<exists> m. ( (nsubn \<sigma> (cpoint f g n \<sigma>) (m+(cpoint f g n \<sigma>))  ) \<Turnstile> f) 
                 \<and> m>0 \<and>((ndropn (m+(cpoint f g n \<sigma>)) \<sigma>) \<Turnstile> g) \<and>
                 x=m+(cpoint f g n \<sigma>)   
                )
          )"



lemma cpoint_expand_0:
 "(cpoint f g 0 \<sigma>) =0"
by simp 

lemma cpoint_expand_1:
 "(cpoint f g 1 \<sigma>) = 
   (SOME x. (\<exists> m. f ( (nsubn \<sigma> 0 (m) ))
                  \<and> m>0 \<and> g ( (ndropn (m) \<sigma>))
                  \<and> x=m))"
by (simp add: itl_defs)


lemma cpoint_expand_n:
 "(cpoint f g (Suc n) \<sigma>) = 
  (SOME x. (\<exists> m. f ( (nsubn \<sigma> (cpoint f g n \<sigma>) (m+(cpoint f g n \<sigma>)))) 
                 \<and> m>0 \<and> g ((ndropn (m+(cpoint f g n \<sigma>)) \<sigma>))
                 \<and> x=m+(cpoint f g n \<sigma>)) 
  )"
by (simp add: itl_defs)


lemma cpoint_0:
 assumes "\<not>nfinite \<sigma> \<and> g \<sigma> \<and>
               (\<forall>k. g ( (ndropn k \<sigma>)) \<longrightarrow>
                       (\<exists>m. f ( (nsubn \<sigma> k (m+k))) \<and> 
                            0 < m \<and> g ( (ndropn (m+k) \<sigma>))))
         "
 shows " g ((ndropn (cpoint f g i \<sigma>) \<sigma>)) " 
proof
 (induct i)
 case 0
 then show ?case  by (simp add: assms)
 next
 case (Suc i)
 then show ?case 
 proof -
  have 1: "g ((ndropn (cpoint f g i \<sigma>) \<sigma>)) "
  by (simp add: Suc.hyps)
  have 2: "g ((ndropn (cpoint f g i \<sigma>) \<sigma>)) \<longrightarrow>
           (\<exists>m. f ( (nsubn \<sigma> (cpoint f g i \<sigma>) (m+(cpoint f g i \<sigma>)))) \<and> 
                            0 < m \<and> g ((ndropn (m+(cpoint f g i \<sigma>)) \<sigma>))) "
  using assms by blast
  have 3: "(\<exists>m. f ( (nsubn \<sigma> (cpoint f g i \<sigma>) (m+(cpoint f g i \<sigma>)))) \<and> 
                            0 < m \<and> g ( (ndropn (m+(cpoint f g i \<sigma>)) \<sigma>))) "
  using 1 2 by auto
  have 4: "(cpoint f g (Suc i) \<sigma>) =
           (SOME x. (\<exists> m. f ( (nsubn \<sigma> (cpoint f g i \<sigma>) (m+(cpoint f g i \<sigma>)))) 
                 \<and> m>0 \<and> g ((ndropn (m+(cpoint f g i \<sigma>)) \<sigma>))
                 \<and> x=m+(cpoint f g i \<sigma>)))  "
  by simp
  have 5: "g ((ndropn ((cpoint f g (Suc i) \<sigma>)) \<sigma>)) "
  using 3 4 someI_ex[of "\<lambda>x. (\<exists> m. f ( (nsubn \<sigma> (cpoint f g i \<sigma>) (m+(cpoint f g i \<sigma>)))) 
                 \<and> m>0 \<and> g ((ndropn (m+(cpoint f g i \<sigma>)) \<sigma>))
                 \<and> x=m+(cpoint f g i \<sigma>))"] by auto
  from 5 show ?thesis by auto
  qed
 qed



lemma cpoint_1:
 assumes "\<not> nfinite \<sigma> \<and> g  \<sigma> \<and>
               (\<forall>k. g ( (ndropn k \<sigma>)) \<longrightarrow>
                       (\<exists>m. f ( (nsubn \<sigma> k (m+k))) \<and> 
                            0 < m \<and> g ( (ndropn (m+k) \<sigma>))))
         "
 shows   "( g ((ndropn (cpoint f g i \<sigma>) \<sigma>)) 
              \<Longrightarrow> g ((ndropn (cpoint f g (Suc i) \<sigma>) \<sigma>)))"
using assms cpoint_0 by blast

lemma cpoint_2:
 assumes "\<not>nfinite \<sigma> \<and> g  \<sigma> \<and>
               (\<forall>k. g ( (ndropn k \<sigma>)) \<longrightarrow>
                       (\<exists>m. f ( (nsubn \<sigma> k (m+k))) \<and> 
                            0 < m \<and> g ( (ndropn (m+k) \<sigma>))))
         "
 shows   "f ( (nsubn \<sigma> (cpoint f g i \<sigma>) (cpoint f g (Suc i) \<sigma>)))
  "
proof 
 (induct i)
 case 0
 then show ?case 
 proof -
  have 1: "g ((ndropn 0 \<sigma>)) "
       using assms cpoint_0 cpoint_expand_0 by force
  have 2: "(\<exists>m. f ( (nsubn \<sigma> (cpoint f g 0 \<sigma>) (m+(cpoint f g 0 \<sigma>)))) \<and> 
                            0 < m \<and> g ( (ndropn (m+(cpoint f g 0 \<sigma>)) \<sigma>))) "
       using assms 1 by (metis cpoint_expand_0)
  have 3: "(cpoint f g 1 \<sigma>) =
           (SOME x. (\<exists> m. f ((nsubn \<sigma> (cpoint f g 0 \<sigma>) (m+(cpoint f g 0 \<sigma>)) ))
                          \<and> m>0 \<and> g ( (ndropn (m+(cpoint f g 0 \<sigma>)) \<sigma>))
                          \<and> x=m+(cpoint f g 0 \<sigma>))
                 )
          "
       by simp
  have 4: "f ( (nsubn \<sigma> (cpoint f g 0 \<sigma>) ((cpoint f g 1 \<sigma>)) )) "
       using 2 3 someI_ex[of "\<lambda>x. (\<exists> m. f ( (nsubn \<sigma> (cpoint f g 0 \<sigma>) 
                                                              (m+(cpoint f g 0 \<sigma>)) ))
                          \<and> m>0 \<and> g ( (ndropn (m+(cpoint f g 0 \<sigma>)) \<sigma>))
                          \<and> x=m+(cpoint f g 0 \<sigma>))"] by auto
  from 4 show ?thesis by auto
  qed
 next
 case (Suc i)
 then show ?case 
 proof -
  have n1: "g ( (ndropn (cpoint f g (Suc i) \<sigma>) \<sigma>)) " 
       using assms cpoint_0 by blast
  have n2: "(\<exists>m. f ( (nsubn \<sigma> (cpoint f g (Suc i) \<sigma>) (m+(cpoint f g (Suc i) \<sigma>)))) \<and> 
                            0 < m \<and> g ( (ndropn (m+(cpoint f g (Suc i) \<sigma>)) \<sigma>))) "
       using assms n1 by auto
  have n3: "(cpoint f g (Suc (Suc i)) \<sigma>) = 
           (SOME x. (\<exists> m. f ( (nsubn \<sigma> (cpoint f g (Suc i) \<sigma>) (m+(cpoint f g (Suc i) \<sigma>)))) 
                 \<and> m>0 \<and> g ((ndropn (m+(cpoint f g (Suc i) \<sigma>)) \<sigma>))
                 \<and> x=m+(cpoint f g (Suc i) \<sigma>)) 
              ) "
       using cpoint_expand_n by blast
  have n4: "f ( (nsubn \<sigma> (cpoint f g (Suc i) \<sigma>) ((cpoint f g (Suc (Suc i)) \<sigma>)))) " 
       using n2 n3 someI_ex[of "\<lambda>x. (\<exists> m. f ( (nsubn \<sigma> (cpoint f g (Suc i) \<sigma>) 
                                                                (m+(cpoint f g (Suc i) \<sigma>)))) 
                 \<and> m>0 \<and> g ((ndropn (m+(cpoint f g (Suc i) \<sigma>)) \<sigma>))
                 \<and> x=m+(cpoint f g (Suc i) \<sigma>)) "] by auto 
  from n4 show ?thesis by auto
  qed
qed
   
lemma cpoint_3a:
 "m>0 \<and> x=m+(cpoint f g (Suc i) \<sigma>) \<Longrightarrow> (cpoint f g (Suc i) \<sigma>) < x"
by auto


lemma cpoint_3:
 assumes "\<not>nfinite \<sigma> \<and> g  \<sigma> \<and>
               (\<forall>k. g ( (ndropn k \<sigma>)) \<longrightarrow>
                       (\<exists>m. f ( (nsubn \<sigma> k (m+k))) \<and> 
                            0 < m \<and> g ( (ndropn (m+k) \<sigma>))))
         "
 shows   "(cpoint f g i \<sigma>) < (cpoint f g (Suc i) \<sigma>) 
  "
proof 
 (induct i)
 case 0
 then show ?case 
 proof -
  have 1: "g ((ndropn 0 \<sigma>)) "
       using assms cpoint_0 cpoint_expand_0 by force
  have 2: "(\<exists>m. f ( (nsubn \<sigma> (cpoint f g 0 \<sigma>) (m+(cpoint f g 0 \<sigma>)))) \<and> 
                            0 < m \<and> g ( (ndropn (m+(cpoint f g 0 \<sigma>)) \<sigma>))) "
       using assms 1 by (metis cpoint_expand_0)
  have 3: "(cpoint f g 1 \<sigma>) =
           (SOME x. (\<exists> m. f ( (nsubn \<sigma> (cpoint f g 0 \<sigma>) (m+(cpoint f g 0 \<sigma>)) ))
                          \<and> m>0 \<and> g ( (ndropn (m+(cpoint f g 0 \<sigma>)) \<sigma>))
                          \<and> x=m+(cpoint f g 0 \<sigma>))
                 )
          "
       by simp
  have 4: "(cpoint f g 0 \<sigma>) < (cpoint f g 1 \<sigma>) "
       using 2 3 someI_ex[of "\<lambda>x. (\<exists> m. f ( (nsubn \<sigma> (cpoint f g 0 \<sigma>) (m+(cpoint f g 0 \<sigma>)) ))
                          \<and> m>0 \<and> g ( (ndropn (m+(cpoint f g 0 \<sigma>)) \<sigma>))
                          \<and> x=m+(cpoint f g 0 \<sigma>))"] by auto 
  from 4 show ?thesis by auto
  qed
 next
 case (Suc i)
 then show ?case 
 proof -
  have n1: "g ( (ndropn (cpoint f g (Suc i) \<sigma>) \<sigma>)) " 
       using assms cpoint_0 by blast
  have n2: "(\<exists>m. f ( (nsubn \<sigma> (cpoint f g (Suc i) \<sigma>) (m+(cpoint f g (Suc i) \<sigma>)))) \<and> 
                            0 < m \<and> g ( (ndropn (m+(cpoint f g (Suc i) \<sigma>)) \<sigma>))) "
       using assms n1 by auto
  have n3: "(cpoint f g (Suc (Suc i)) \<sigma>) = 
           (SOME x. (\<exists> m. f ( (nsubn \<sigma> (cpoint f g (Suc i) \<sigma>) (m+(cpoint f g (Suc i) \<sigma>)))) 
                 \<and> m>0 \<and> g ((ndropn (m+(cpoint f g (Suc i) \<sigma>)) \<sigma>))
                 \<and> x=m+(cpoint f g (Suc i) \<sigma>)) 
              ) "
       using cpoint_expand_n by blast
  have n4: "(\<exists> m. f ( (nsubn \<sigma> (cpoint f g (Suc i) \<sigma>) (m+(cpoint f g (Suc i) \<sigma>)))) 
                 \<and> m>0 \<and> g ((ndropn (m+(cpoint f g (Suc i) \<sigma>)) \<sigma>))
                 \<and> (cpoint f g (Suc (Suc i)) \<sigma>)=m+(cpoint f g (Suc i) \<sigma>))  " 
       using n2 n3 someI_ex[of "\<lambda>x. (\<exists> m. f ( (nsubn \<sigma> (cpoint f g (Suc i) \<sigma>) (m+(cpoint f g (Suc i) \<sigma>)))) 
                 \<and> m>0 \<and> g ((ndropn (m+(cpoint f g (Suc i) \<sigma>)) \<sigma>))
                 \<and> x=m+(cpoint f g (Suc i) \<sigma>)) "] by auto
  have n5: "(cpoint f g (Suc i) \<sigma>) < (cpoint f g (Suc (Suc i)) \<sigma>)  "
  using n4 using cpoint_3a by blast
  from n5 show ?thesis by auto
  qed
qed 


primcorec cpl :: "('a::world) formula  \<Rightarrow> 'a formula \<Rightarrow> nat \<Rightarrow> 'a intervals \<Rightarrow> nat nellist" 
 where " cpl f g x \<sigma> = NCons (cpoint f g x \<sigma> ) (cpl f g (Suc x) \<sigma>)" 

lemma
 "nnth (cpl f g 0 \<sigma>) 0 = 0 " 
by (metis cpl.disc_iff cpl.simps(2) cpoint_expand_0 nhd_conv_nnth) 

lemma
 "nnth (cpl f g 0 \<sigma>) 1 = cpoint f g 1 \<sigma>" 
by (metis One_nat_def cpl.code cpl.disc_iff cpl.simps(2) nhd_conv_nnth nnth_Suc_NCons)

lemma nnth_cpl: 
 "nnth (cpl f g x \<sigma>) i = cpoint f g (x+i) \<sigma> "
proof (induct i arbitrary: x)
case 0
then show ?case by (simp add: nnth_0_conv_nhd)
next
case (Suc i)
then show ?case by (metis add_Suc_shift cpl.simps(3) nnth_ntl)
qed



lemma cpl_infinite: 
 "\<not>nfinite (cpl f g x \<sigma>) " 
proof 
 assume "nfinite (cpl f g x \<sigma>) " 
 thus False
 proof (induct zs\<equiv>"(cpl f g x \<sigma>)" arbitrary: x rule: nfinite_induct)
 case (NNil y)
 then show ?case by (metis cpl.disc_iff nellist.disc(1))
 next
 case (NCons x nell)
 then show ?case by (metis cpl.simps(3) nellist.sel(5))
 qed  
qed

lemma AOmegaInductSem:
 "( w \<Turnstile> (inf \<and> g \<and> \<box>(g \<longrightarrow> ((f \<and> more) \<and> finite);g)) \<longrightarrow> aomega f)"
proof -
   have 1: "(w \<Turnstile> (inf \<and> g \<and> \<box>(g \<longrightarrow> ((f \<and> more) \<and> finite);g))) = 
               (\<not>nfinite w \<and> g w \<and>
               (\<forall>k. g ( (ndropn  k w)) \<longrightarrow>
                       (\<exists>m. f ( (nsubn w k (m+k))) \<and> 
                            0 < m \<and> g ( (ndropn (m+k) w))))) "
        using AOmegaInductSem_help[of g f w ] 
        by (simp add: add.commute)
   have 2: "(\<not> nfinite w \<and> g w \<and>
               (\<forall>k. g ( (ndropn k w)) \<longrightarrow>
                       (\<exists>m. f ( (nsubn w k (m+k))) \<and> 
                            0 < m \<and> g ( (ndropn (m+k) w))))) \<longrightarrow>
                     nidx  (cpl f g 0 w) "
    using nidx_expand[of "(cpl f g 0 w)"] nnth_cpl[of f g 0 w]
     by (metis add_0 cpoint_3)
   have 3: "(\<not>nfinite w \<and> g w \<and>
               (\<forall>k. g ( (ndropn k w)) \<longrightarrow>
                       (\<exists>m. f ( (nsubn w k (m+k))) \<and> 
                            0 < m \<and> g ( (ndropn (m+k) w))))) \<longrightarrow>
                              (\<forall>i. f ( (nsubn w (nnth (cpl f g 0 w) i) 
                                                         (nnth (cpl f g 0 w)  (Suc i))))) "
   using 1 cpoint_2 by (metis add_0 nnth_cpl)
   have 31: " (\<not>nfinite w \<and> g w \<and>
               (\<forall>k. g ( (ndropn k w)) \<longrightarrow>
                       (\<exists>m. f ( (nsubn w k (m+k))) \<and> 
                            0 < m \<and> g ( (ndropn (m+k) w))))) \<longrightarrow>
                        (\<forall>i. (nnth (cpl f g 0 w) i) \<le> nlength w) " 
        by (simp add: nfinite_conv_nlength_enat)
   have 4: "(\<not> nfinite w \<and> g w \<and>
               (\<forall>k. g ( (ndropn k w)) \<longrightarrow>
                       (\<exists>m. f ( (nsubn w k (m+k))) \<and> 
                            0 < m \<and> g ( (ndropn (m+k) w))))) \<longrightarrow>
                             \<not>nfinite w \<and> \<not>nfinite (cpl f g 0 w) \<and> nnth (cpl f g 0 w) 0 = 0 \<and>  
                              nidx  (cpl f g 0 w)  \<and>
                             (\<forall>i. (nnth (cpl f g 0 w) i) \<le> nlength w) \<and>
                             (\<forall>i. f ( (nsubn w (nnth (cpl f g 0 w)  i) 
                                               (nnth (cpl f g 0 w)  (Suc i)))))"
    using 2 3 31
    by (simp add: cpl_infinite nnth_0_conv_nhd)
   have 5: "(\<not>nfinite w \<and> g w \<and>
               (\<forall>k. g ( (ndropn k w)) \<longrightarrow>
                       (\<exists>m. f ( (nsubn w k (m+k))) \<and> 
                            0 < m \<and> g ( (ndropn (m+k) w))))) \<longrightarrow>
                             \<not>nfinite w \<and> 
                             (\<exists>(ls).  \<not>nfinite ls \<and> nnth ls 0 = 0 \<and>  nidx ls \<and> 
                                 (\<forall>i. (nnth ls i) \<le> nlength w) \<and>
                                 (\<forall>i. f ( (nsubn w (nnth ls  i) 
                                               (nnth ls  (Suc i))))) \<and> 
                                                    ls = (cpl f g 0 w))"
    using "4" by auto
   have 6: "(\<not>nfinite w \<and> g w \<and>
               (\<forall>k. g ( (ndropn k w)) \<longrightarrow>
                       (\<exists>m. f ( (nsubn w k (m+k))) \<and> 
                            0 < m \<and> g ( (ndropn (m+k) w))))) \<longrightarrow> (w \<Turnstile> aomega f) "
    using "3" "5" unfolding aomega_d_def by blast
   have 7: "(w \<Turnstile> (inf \<and> g \<and> \<box>(g \<longrightarrow> ((f \<and> more) \<and> finite);g)) \<longrightarrow> (aomega f)  ) "
   using "6" "1" by auto
   from 7 show ?thesis  by blast
qed

lemma AOmegaInduct:
 " \<turnstile> (inf \<and> g \<and> \<box>(g \<longrightarrow> ((f \<and> more) \<and> finite);g)) \<longrightarrow> aomega f" 
unfolding Valid_def using AOmegaInductSem[of g f ] by blast
 

lemma AOmegaWeakCoInduct:
 assumes "\<turnstile> g \<longrightarrow> ((f \<and> more) \<and> finite);g "
 shows   "\<turnstile> inf \<and> g \<longrightarrow> aomega f" 
proof -
 have 1: "\<turnstile> \<box>(g \<longrightarrow> ((f \<and> more) \<and> finite);g) " 
  using assms by (simp add: BoxGen)
 show ?thesis using assms AOmegaInduct[of g f]
  using "1" by fastforce
qed

 




lemma Omega_WOmega:
 "\<turnstile> (omega f) = (womega (f \<and> finite)) " 
unfolding omega_d_def by simp

lemma WOmegaInducthelp:
 "\<turnstile> ( g \<and> \<box>(g \<longrightarrow> f;g)) 
            \<longrightarrow>
             f;( ( g \<and> \<box>(g \<longrightarrow> f;g)) )" 
proof -
 have 1: "\<turnstile> ( g \<and> \<box>(g \<longrightarrow> f;g)) = ( g \<and> (g \<longrightarrow> f;g) \<and> \<box>(g \<longrightarrow> f;g)) "
   using BoxEqvAndBox[of "LIFT g \<longrightarrow> f;g" ] by fastforce
 have 2: "\<turnstile> ( g \<and> (g \<longrightarrow> f;g) \<and> \<box>(g \<longrightarrow> f;g)) = ( g \<and> ( f;g) \<and> \<box>(g \<longrightarrow> f;g)) "
     by fastforce
 have 4: "\<turnstile> \<box>(g \<longrightarrow> f;g) = \<box> (\<box> (g \<longrightarrow> f;g)) " 
    by (simp add: BoxEqvBoxBox)
 have 5: "\<turnstile> ( g \<and> ( f;g) \<and> \<box>(g \<longrightarrow> f;g)) = ( g \<and> ( f;g) \<and>\<box>( \<box>(g \<longrightarrow> f;g))) "
   using 4 by fastforce 
 have 6: "\<turnstile> ( g \<and> ( f;g) \<and> \<box>(\<box>(g \<longrightarrow> f;g))) \<longrightarrow> f;( ( g \<and> \<box>(g \<longrightarrow> f;g)) )" 
    using ChopAndBoxImport[of f g "LIFT \<box>(g \<longrightarrow> f;g)"]
    by (meson Prop01 Prop05)
 show ?thesis 
 by (metis "1" "2" "5" "6" int_eq)
qed


lemma OmegaInducthelp:
 "\<turnstile> ( g \<and> \<box>(g \<longrightarrow> ((f \<and> more))\<frown>g)) 
            \<longrightarrow>
             ((f \<and> more))\<frown>( ( g \<and> \<box>(g \<longrightarrow> ((f \<and> more))\<frown>g)) )" 
using WOmegaInducthelp unfolding schop_d_def by blast

lemma WOmegaInduct:
    "\<turnstile> ( g \<and> \<box>(g \<longrightarrow> (f \<and> more);g)) \<longrightarrow> womega f"
by (simp add: WOmegaInducthelp WOmegaWeakCoinduct)

lemma OmegaInduct:
    "\<turnstile> ( g \<and> \<box>(g \<longrightarrow> ((f \<and> more))\<frown>g)) \<longrightarrow> omega f"
by (simp add: OmegaInducthelp OmegaWeakCoinduct)


lemma WOmega_coinduct: 
 assumes "\<turnstile> g \<longrightarrow> h \<or> (f \<and> more);g"
 shows   "\<turnstile> g \<longrightarrow> (womega f) \<or> (wpowerstar (f \<and> more));h" 
proof -
 have 1: "\<turnstile> (wpowerstar (f \<and> more));h = (empty \<or> (f \<and> more);(wpowerstar (f \<and> more)));h "
   by (simp add: LeftChopEqvChop WPowerstarEqv)
 have 2: "\<turnstile> (empty \<or> (f \<and> more);(wpowerstar (f \<and> more)));h = (h \<or> ((f \<and> more);(wpowerstar (f \<and> more)));h)" 
    by (simp add: EmptyOrChopEqv)
 have 3: "\<turnstile> (\<not>(h \<or> ((f \<and> more);(wpowerstar (f \<and> more)));h)) = (\<not> h \<and> \<not>((f \<and> more);((wpowerstar (f \<and> more));h))) " 
    using ChopAssoc by fastforce
 have 31: "\<turnstile> (\<not>( (wpowerstar (f \<and> more));h)) = (\<not> h \<and> \<not>((f \<and> more);((wpowerstar (f \<and> more));h)))" 
    by (metis "1" "2" "3" int_eq)
 have 32: "\<turnstile> ((\<not> h \<and> \<not>((f \<and> more);((wpowerstar (f \<and> more));h))) \<and> (h \<or> (f \<and> more);g)) \<longrightarrow>
              (\<not>((f \<and> more);((wpowerstar (f \<and> more));h)) \<and> (f \<and> more);g) " 
     by force
 have 33: "\<turnstile>  g \<and> \<not>( (wpowerstar (f \<and> more));h) \<longrightarrow> ((\<not> h \<and> \<not>((f \<and> more);((wpowerstar (f \<and> more));h))) \<and> (h \<or> (f \<and> more);g)) "
       using assms 31 by fastforce
 have 4: "\<turnstile> g \<and> \<not>( (wpowerstar (f \<and> more));h) \<longrightarrow> \<not>((f \<and> more);((wpowerstar (f \<and> more));h)) \<and> (f \<and> more);g "
    using assms 31  32 33 
    using lift_imp_trans by blast
 have 5: "\<turnstile> \<not>((f \<and> more);((wpowerstar (f \<and> more));h)) \<and> (f \<and> more);g \<longrightarrow> (f \<and> more);(g \<and> \<not> wpowerstar (f \<and> more);h)"
       by (metis ChopAndNotChopImp int_eq lift_and_com)
 have 6: "\<turnstile> g \<and> \<not>( (wpowerstar (f \<and> more));h) \<longrightarrow> (f \<and> more);(g \<and> \<not> wpowerstar (f \<and> more);h)" 
   using "4" "5" lift_imp_trans by blast
 have 7: "\<turnstile> g \<and> \<not>((wpowerstar (f \<and> more));h) \<longrightarrow> (womega f)" 
   using WOmegaWeakCoinduct[of "LIFT g \<and> \<not>((wpowerstar (f \<and> more));h)" f] 6 
      by blast
 show ?thesis using 7 by fastforce
qed

lemma WOmega_coinduct_var: 
 assumes "\<turnstile> g \<longrightarrow> h \<or> (f \<and> more);g"
 shows   "\<turnstile> g \<longrightarrow> (womega f) \<or> (wpowerstar f);h" 
using assms 
by (metis WOmega_coinduct WPowerstar_more_absorb int_eq)

lemma WOmega_coinduct_no_more: 
 assumes "\<turnstile> g \<longrightarrow> f;g"
 shows   "\<turnstile> g \<longrightarrow> (womega f) \<or> (wpowerstar f);init f" 
proof -
 have 1: "\<turnstile> ( f) = ( (f \<and> empty) \<or> (f \<and> more)) " 
  using EmptyOrMoreSplit by auto
 have 2: "\<turnstile>  ( f;g) = ( (f \<and> empty);g \<or> (f \<and> more);g) " 
     by (simp add: "1" OrChopEqvRule)
 have 3: "\<turnstile> ((f \<and> empty);g) = (g \<and> init f) " 
    by (meson InitAndEmptyEqvAndEmpty LeftChopEqvChop Prop04 StateAndEmptyChop lift_and_com)
 have 4: "\<turnstile> g  \<longrightarrow> init f \<or> (f \<and> more);g " 
   using 2  3 assms by fastforce 
 show ?thesis using 4 WOmega_coinduct_var by blast
qed

lemma WOmega_coinduct_no_more_var: 
 assumes "\<turnstile> g \<longrightarrow> h \<or> f;g"
 shows   "\<turnstile> g \<longrightarrow> (womega f) \<or> (wpowerstar f);(h \<or> init f)" 
proof -
  have 1: "\<turnstile> ( f) = ( (f \<and> empty) \<or> (f \<and> more)) " 
  using EmptyOrMoreSplit by auto
  have 2: "\<turnstile>  ( f;g) = ( (f \<and> empty);g \<or> (f \<and> more);g) " 
     by (simp add: "1" OrChopEqvRule)
  have 3: "\<turnstile> ((f \<and> empty);g) = (g \<and> init f) " 
    by (meson InitAndEmptyEqvAndEmpty LeftChopEqvChop Prop04 StateAndEmptyChop lift_and_com)
  have 4: "\<turnstile> g \<longrightarrow> (h \<or> init f) \<or> (f \<and> more);g " 
    using assms "2" "3" by fastforce
  show ?thesis using 4 WOmega_coinduct_var by blast
qed  

lemma Omega_coinduct: 
 assumes "\<turnstile> g \<longrightarrow> h \<or> (f \<and> more)\<frown>g"
 shows   "\<turnstile> g \<longrightarrow> (omega f) \<or> (schopstar f)\<frown>h" 
using assms WOmega_coinduct[of g h "LIFT (f  \<and> finite)" ]
Omega_WOmega[of f] SChopstar_WPowerstar[of "LIFT (f \<and> more)" ] 
by (metis Chop_more_finite_swap Prop10 SChopstar_WPowerstar SChopstar_finite WPowerstar_more_absorb inteq_reflection schop_d_def)

subsection \<open>Omega algebra\<close>

lemma WOmega_coinduct_var1: 
 assumes "\<turnstile> f \<longrightarrow> empty \<or>   (g \<and> more);f"
 shows   "\<turnstile> f  \<longrightarrow> (womega g) \<or> (wpowerstar g)"
using assms  WOmega_coinduct_var[of f "LIFT empty" g] 
by (metis ChopEmpty inteq_reflection)

lemma WOmega_coinduct_no_more_var1: 
 assumes "\<turnstile> f \<longrightarrow> empty \<or>   g;f"
 shows   "\<turnstile> f  \<longrightarrow> (womega g) \<or> (wpowerstar g);(empty \<or> init g)"
using assms WOmega_coinduct_no_more_var[of f "LIFT empty" g] by blast

lemma Omega_coinduct_var1: 
 assumes "\<turnstile> f \<longrightarrow> empty \<or> (  (g \<and> more)\<frown>f)"
 shows   "\<turnstile> f  \<longrightarrow> (omega g) \<or> (schopstar g)"
using assms 
by (metis Omega_coinduct Prop08 SChop_SChopstar_Closure SChopstar_imp_empty SChopstardef int_iffD1 schopstar_d_def)



lemma WOmega_coinduct_eq: 
 assumes "\<turnstile> f = ( h \<or>   (g \<and> more);f)"
 shows   "\<turnstile> f  \<longrightarrow> (womega g) \<or> (wpowerstar g);h"
using assms WOmega_coinduct_var int_iffD1 by blast

lemma WOmega_coinduct_no_more_eq: 
 assumes "\<turnstile> f = ( h \<or>   g;f)"
 shows   "\<turnstile> f  \<longrightarrow> (womega g) \<or> (wpowerstar g);(h \<or> init g)"
using assms by (simp add: Prop11 WOmega_coinduct_no_more_var)

lemma Omega_coinduct_eq: 
 assumes "\<turnstile> f = ( h \<or>   (g \<and> more)\<frown>f)"
 shows   "\<turnstile> f  \<longrightarrow> (omega g) \<or> (schopstar g)\<frown>h"
using assms by (simp add: Omega_coinduct Prop11)

lemma WOmega_coinduct_eq_var1: 
 assumes "\<turnstile> f = ( empty \<or>   (g \<and> more);f)"
 shows   "\<turnstile> f  \<longrightarrow> (womega g) \<or> (wpowerstar g)"
using assms WOmega_coinduct_var1 int_iffD1 by blast

lemma WOmega_coinduct_no_more_eq_var1: 
 assumes "\<turnstile> f = ( empty \<or>   g;f)"
 shows   "\<turnstile> f  \<longrightarrow> (womega g) \<or> (wpowerstar g);(empty \<or> init g)"
using assms WOmega_coinduct_no_more_eq by blast

lemma Omega_coinduct_eq_var1: 
 assumes "\<turnstile> f = ( empty \<or>   (g \<and> more)\<frown>f)"
 shows   "\<turnstile> f  \<longrightarrow> (omega g) \<or> (schopstar g)"
using assms by (simp add: Omega_coinduct_var1 Prop11)

lemma WOmega_coinduct_eq_var2: 
 assumes "\<turnstile> f = (g \<and> more);f"
 shows   "\<turnstile> f   \<longrightarrow> (womega g) "
using assms by (simp add: Prop11 WOmegaWeakCoinduct)

lemma WOmega_coinduct_no_more_eq_var2: 
 assumes "\<turnstile> f = g;f"
 shows   "\<turnstile> f   \<longrightarrow> (womega g) \<or> (wpowerstar g);init g"
using assms WOmega_coinduct_no_more int_iffD1 by blast

lemma Omega_coinduct_eq_var2: 
 assumes "\<turnstile> f = (g \<and> more)\<frown>f"
 shows   "\<turnstile> f   \<longrightarrow> (omega g) "
using assms OmegaWeakCoinduct int_iffD1 by blast

lemma WOmega_subdist: 
 "\<turnstile> (womega f) \<longrightarrow> (womega (f \<or> g)) " 
proof -
 have 0: "\<turnstile> ((f \<or> g) \<and> more) = (( f \<and> more \<or> g \<and> more)) " 
   by fastforce
 have 1: "\<turnstile> (womega f) \<longrightarrow> ((f \<and> more \<or> g \<and> more));(womega f) "
   using OrChopEqv WOmegaCases by fastforce
 have 15: "\<turnstile> ((f \<and> more \<or> g \<and> more));(womega f) = ((f \<or> g) \<and> more);(womega f) " 
    using 0 by (metis LeftChopEqvChop int_eq)
 have 2: "\<turnstile> (womega f) \<longrightarrow>  ((f \<or> g) \<and> more);(womega f) " 
   by (metis "0" "1" int_eq)
 show ?thesis by (simp add: "2" WOmegaWeakCoinduct)
qed

lemma Omega_subdist: 
 "\<turnstile> (omega f) \<longrightarrow> (omega (f \<or> g)) " 
by (metis FiniteOr WOmega_subdist int_eq omega_d_def)


lemma WOmega_iso: 
assumes "\<turnstile> f \<longrightarrow> g"
shows   "\<turnstile> (womega f) \<longrightarrow> (womega g) " 
using assms WOmega_subdist[of f g] 
by (metis Prop02 Prop03 Prop11 int_simps(20) inteq_reflection)

lemma Omega_iso: 
assumes "\<turnstile> f \<longrightarrow> g"
shows   "\<turnstile> (omega f) \<longrightarrow> (omega g) " 
using assms WOmega_iso[of "LIFT (f \<and> finite)" "LIFT (g \<and> finite)" ]
unfolding omega_d_def 
by (metis (no_types, opaque_lifting) AndChopA AndChopB ChopEmpty Prop12 inteq_reflection
 lift_imp_trans omega_d_def)


lemma WOmega_subdist_var: 
 "\<turnstile> (womega f) \<or> (womega g) \<longrightarrow> (womega (f \<or> g)) "
by (meson EmptyChop Prop02 Prop04 Prop05 Prop11 WOmega_iso WOmega_subdist) 

lemma Omega_subdist_var: 
 "\<turnstile> (omega f) \<or> (omega g) \<longrightarrow> (omega (f \<or> g)) " 

by (meson Omega_iso Omega_subdist Prop02 Prop04 Prop05 int_iffD2 int_simps(26))


lemma WOmega_zero:
 "\<turnstile> \<not>(womega #False)  " 
by (metis AndInfChopEqvAndInf TrueW WOmegaUnroll int_simps(19) int_simps(3) inteq_reflection)

lemma Omega_zero:
 "\<turnstile> \<not>(omega #False)  " 
using Omega_WOmega[of "LIFT #False"]  WOmega_zero  by fastforce

lemma WOmega_star_1: 
 "\<turnstile> (wpowerstar f);(womega f) = (womega f) " 
proof -
 have 1: "\<turnstile> (f \<and> more); (womega f) \<longrightarrow> (womega f)" 
   by (simp add: WOmegaUnroll int_iffD2)
 have 2: "\<turnstile> (wpowerstar f);(womega f) \<longrightarrow> (womega f) " 
    by (meson "1" LeftChopImpChop WPowerstar_inductL_var_equiv WPowerstar_more_absorb int_iffD2 lift_imp_trans)
 have 3: "\<turnstile> (womega f) \<longrightarrow> (wpowerstar f);(womega f)"
    by (metis "2" Prop03 Prop10 WPowerstarChopEqvChopOrRule inteq_reflection)
 show ?thesis 
 by (simp add: "2" "3" int_iffI)
qed

lemma Omega_star_1: 
 "\<turnstile> (schopstar f)\<frown>(omega f) = (omega f) " 
by (meson OmegaUnroll Prop11 SChopstarMore_induct_lvar_eq2)

lemma WOmega_max_element: 
 "\<turnstile> f \<longrightarrow> (womega empty) \<or> wpowerstar empty;init empty" 
by (metis EmptyChop WOmega_coinduct_no_more_eq_var2 inteq_reflection)

lemma WOmega_empty_zero:
 "\<turnstile> \<not>(womega empty) " 
by (metis Prop10 Prop12 WOmegaUnroll WOmegaWeakCoinduct WOmega_zero empty_d_def int_iffD2 int_simps(21) inteq_reflection lift_and_com)

lemma Omega_empty_zero:
 "\<turnstile> \<not>(omega empty) " 
by (metis FiniteAndEmptyEqvEmpty WOmega_empty_zero int_eq lift_and_com omega_d_def)
 


lemma WOmega_1: 
 "\<turnstile> (womega f); g \<longrightarrow> (womega f)"
by (meson ChopAssoc LeftChopEqvChop Prop11 WOmegaUnroll WOmegaWeakCoinduct lift_imp_trans)

lemma Omega_1: 
 "\<turnstile> (omega f); g \<longrightarrow> (omega f) " 
by (simp add: WOmega_1 omega_d_def)

lemma WOmega_sup_id: 
assumes "\<turnstile> empty \<longrightarrow> g"
 shows  "\<turnstile> (womega f);g = (womega f) " 
by (meson ChopEmpty Prop11 RightChopImpChop WOmega_1 assms lift_imp_trans)

lemma Omega_sup_id: 
assumes "\<turnstile> empty \<longrightarrow> g"
 shows  "\<turnstile> (omega f);g = (omega f) " 
using assms 
by (metis Omega_WOmega WOmega_sup_id int_eq)



lemma WOmega_simulation: 
assumes " \<turnstile> h; (f \<and> more) \<longrightarrow>  (g \<and> more) ; h "
shows  " \<turnstile> h ; (womega f) \<longrightarrow> (womega g) " 
proof -
 have 1: "\<turnstile> h ; (womega f) = h ; ((f \<and> more); (womega f))" 
   by (simp add: RightChopEqvChop WOmegaUnroll)
 have 2: "\<turnstile> h ; ((f \<and> more); (womega f)) \<longrightarrow> (g \<and> more) ; (h; (womega f))"
    by (metis AndChopB ChopAssoc Prop10 assms inteq_reflection)
 have 3: "\<turnstile> h ; (womega f)  \<longrightarrow>  (g \<and> more) ; (h ; (womega f) ) "
    by (metis "1" "2" int_eq)
 show ?thesis by (simp add: "3" WOmegaWeakCoinduct)
qed

lemma Omega_simulation: 
assumes " \<turnstile> (h)\<frown> (f \<and> more) \<longrightarrow>  (g \<and> more) \<frown> (h) "
shows  " \<turnstile> (h) \<frown> (omega f) \<longrightarrow> (omega g) " 
using assms 
by (metis AndSChopA OmegaUnroll OmegaWeakCoinduct Prop10 SChopAssoc inteq_reflection lift_and_com)


lemma WOmega_WOmega: 
 "\<turnstile> (womega (womega f)) \<longrightarrow> (womega f) " 
by (meson AndChopA Prop11 WOmegaUnroll WOmega_1 lift_imp_trans)


lemma WOmega_eqv: 
 assumes "\<turnstile> f = g "
 shows "\<turnstile> womega f = womega g"
by (meson Prop11 WOmega_iso assms) 

lemma WOmega_More_absorb: 
 " \<turnstile> womega f = womega (f \<and> more) " 
proof -
 have 1: "\<turnstile> f\<^sup>\<W> \<longrightarrow> (f \<and> more)\<^sup>\<W>" 
using WOmega_coinduct_eq_var2[of "LIFT  womega f" "LIFT (f \<and> more)" ] 
 by (metis LeftChopImpChop Prop12 TrueW WOmegaUnroll WOmega_simulation int_simps(13)
   inteq_reflection)
 have 2: " \<turnstile> (f \<and> more)\<^sup>\<W> \<longrightarrow> f\<^sup>\<W>" 
   using WOmega_coinduct_eq_var2[of "LIFT womega (f \<and> more)" f] 
   by (metis AndChopB WOmegaUnroll WOmega_simulation inteq_reflection lift_and_com)
 show ?thesis using 1 2 by fastforce
qed

  

lemma WOmega_more_eqv: 
 assumes "\<turnstile> f = g "
 shows "\<turnstile> womega (f \<and> more) = womega g"
by (meson Prop04 WOmega_More_absorb WOmega_eqv assms)



lemma WOmega_Wagner_1: 
 "\<turnstile> (womega (wpowerstar f)) = (womega f) " 
proof -   
 have 1: "\<turnstile> (womega ((f \<and> more);wpowerstar f)) = 
            (((f \<and> more);(wpowerstar f));((f \<and> more);(wpowerstar f)));(womega ((f \<and> more);wpowerstar f))"
   by (metis (no_types, lifting) ChopAssoc WOmegaUnroll WOmega_star_1 WPowerstar_more_absorb WpowerstarAndMoreEqvAndMoreChop inteq_reflection)

 have 2: "\<turnstile> (((f \<and> more);(wpowerstar f));((f \<and> more);(wpowerstar f)));(womega ((f \<and> more);wpowerstar f)) =
             (f \<and> more); (womega ((f \<and> more);wpowerstar f))"
             by (metis (no_types, lifting) AndChopB ChopAssoc Prop11 WOmegaUnroll WPowerstar_inductL_var_equiv WPowerstar_invol WPowerstar_more_absorb WpowerstarAndMoreEqvAndMoreChop inteq_reflection lift_and_com) 
 have 3: "\<turnstile> (womega ((f \<and> more);wpowerstar f)) \<longrightarrow> (womega f) "
 by (metis "1" "2" WOmega_coinduct_eq_var2 inteq_reflection)
 have 4: "\<turnstile> (womega f) \<longrightarrow> (womega ((f \<and> more);wpowerstar f)) "
 by (metis WOmega_More_absorb WOmega_iso WPowerstar_ext WpowerstarAndMoreEqvAndMoreChop
   inteq_reflection)
 have 5: "\<turnstile> (wpowerstar f \<and> more) = (f \<and> more);powerstar f"
 by (metis ChopstarEqvPowerstar Chopstar_WPowerstar WPowerstar_more_absorb WpowerstarAndMoreEqvAndMoreChop inteq_reflection)   
 have 6: "\<turnstile> (womega (wpowerstar f)) =  (womega ((f \<and> more);wpowerstar f)) "
  by (metis "5" ChopstarEqvPowerstar Chopstar_WPowerstar WOmega_More_absorb WPowerstar_more_absorb
    inteq_reflection)
 show ?thesis using "3" "4" "6" by fastforce 
qed

lemma Omega_Wagner_1: 
 "\<turnstile> (omega (schopstar f)) = (omega f) "
 by (metis Prop10 SChopstar_WPowerstar SChopstar_finite WOmega_Wagner_1 inteq_reflection omega_d_def) 

lemma Omega_Eqv_rule: 
 assumes "\<turnstile> f = g "
 shows   "\<turnstile> omega f = omega g" 
using assms
by (simp add: Omega_iso Prop11) 

lemma Omega_Wagner_1_var2: 
 "\<turnstile> (omega ((f \<and> more)\<frown>schopstar f)) = (omega f) "
proof -
 have 1: "\<turnstile> ((f \<and> more)\<frown>schopstar f \<and> finite \<and> more) = ((f \<and> more)\<frown>schopstar f) "
 by (metis Prop10 Prop11 Prop12 SCSAndMoreEqvAndMoreSChop SChopImpFinite SChopstar_finite)
 have 2: "\<turnstile> ((schopstar f \<and> finite) \<and> more) = (f \<and> more)\<frown>(schopstar f) "
 by (metis Prop10 SCSAndMoreEqvAndMoreSChop SChopstar_finite inteq_reflection)
 have 3: "\<turnstile> (omega ((f \<and> more)\<frown>schopstar f)) = (omega (schopstar f)) "
 by (metis "2" Omega_Wagner_1 SChopstar_and_more SChopstar_finite_absorb inteq_reflection) 
 show ?thesis
 by (metis "3" Omega_Wagner_1 inteq_reflection)
qed


lemma WOmega_Wagner_2_var:
 "\<turnstile> (f \<and> more);(womega ((g \<and> more);(f \<and> more))) \<longrightarrow> (womega ((f \<and> more);(g \<and> more))) " 
proof -
 have 1: "\<turnstile>  (f \<and> more);((g \<and> more);(f \<and> more) \<and> more) \<longrightarrow> ((f \<and> more);(g \<and> more) \<and> more);(f \<and> more)  "
 by (metis (no_types, lifting) ChopAssoc Prop10 Prop12 RightChopImpMoreRule int_iffD1 inteq_reflection lift_and_com) 
   
 show ?thesis using  WOmega_simulation[of "LIFT (f \<and> more)" "LIFT ((g \<and> more);(f \<and> more))" "LIFT ((f \<and> more);(g \<and> more))"]
 using "1" by blast
qed

lemma Omega_Wagner_2_var:
 "\<turnstile> (f \<and> more)\<frown>(omega ((g \<and> more)\<frown>(f \<and> more))) \<longrightarrow> (omega ((f \<and> more)\<frown>(g \<and> more))) "
using WOmega_Wagner_2_var[of "LIFT ((f \<and> more) \<and> finite)" "LIFT ((g \<and> more) \<and> finite)"]   
Omega_WOmega[of "LIFT (((g \<and> more) \<and> finite);((f \<and> more) \<and> finite))" ] 
Omega_WOmega[of "LIFT (((f \<and> more) \<and> finite);((g \<and> more) \<and> finite))"]
by (metis (no_types, lifting) AndMoreSChopAndMoreEqvAndMoreSChop Omega_simulation SChopAssoc 
    int_eq int_iffD1)


lemma WOmega_Wagner_2: 
 "\<turnstile> (womega ((f \<and> more);(g \<and> more))) = (f \<and> more) ; (womega ((g \<and> more);(f \<and> more))) " 
proof -
 have 1: "\<turnstile> (f \<and> more) ; (womega ((g \<and> more);(f \<and> more))) \<longrightarrow> (womega ((f \<and> more);(g \<and> more)))   "
   by (simp add: WOmega_Wagner_2_var)
 have 2: "\<turnstile> (womega ((f \<and> more);(g \<and> more))) = ((f \<and> more);(g \<and> more)); (womega ((f \<and> more);(g \<and> more))) "
  by (meson AndChopA LeftChopImpChop Prop11 WOmegaUnroll WOmega_star_1 WPowerstar_ext lift_imp_trans)
 have 3: "\<turnstile> (womega ((f \<and> more);(g \<and> more))) \<longrightarrow> (f \<and> more) ; (womega ((g \<and> more);(f \<and> more)))"
 by (metis (no_types, lifting) "2" ChopAndB ChopAssoc Prop10 WOmega_Wagner_2_var inteq_reflection)
 show ?thesis 
 using "1" "3" int_iffI by blast
qed

lemma Omega_Wagner_2: 
 "\<turnstile> (omega ((f \<and> more)\<frown>(g \<and> more))) = (f \<and> more) \<frown> (omega ((g \<and> more)\<frown>(f \<and> more))) " 
proof -
 have 1: "\<turnstile> (f \<and> more) \<frown> (omega ((g \<and> more)\<frown>(f \<and> more))) \<longrightarrow>
            (omega ((f \<and> more)\<frown>(g \<and> more)))   "
   by (simp add: Omega_Wagner_2_var)
 have 2: "\<turnstile> (omega ((f \<and> more)\<frown>(g \<and> more))) = ((f \<and> more)\<frown>(g \<and> more))\<frown> (omega ((f \<and> more)\<frown>(g \<and> more))) "
 by (metis AndMoreSChopAndMoreEqvAndMoreSChop OmegaUnroll int_eq )
 have 3: "\<turnstile> (omega ((f \<and> more)\<frown>(g \<and> more))) \<longrightarrow> (f \<and> more) \<frown> (omega ((g \<and> more)\<frown>(f \<and> more)))"
   by (metis (no_types, lifting) "2" Omega_Wagner_2_var RightSChopImpSChop SChopAssoc inteq_reflection)
 show ?thesis 
 using "1" "3" int_iffI by blast
qed

lemma WOmega_Wagner_2_var1: 
 "\<turnstile> (womega ((f);(g \<and> more))) = (f) ; (womega ((g \<and> more);(f))) "
proof -
 have 1: "\<turnstile>  (f) ; (womega ((g \<and> more);(f))) \<longrightarrow> (womega ((f);(g \<and> more))) "
    using WOmega_simulation[of f "LIFT (g \<and> more) ; f" "LIFT f ; (g \<and> more)" ]
    by (metis (no_types, lifting) AndChopB ChopAndB ChopAssoc ChopMoreImpMore MoreChopImpMore 
       Prop10 int_iffD1 inteq_reflection lift_imp_trans) 
 have 2: "\<turnstile> (womega ((f);(g \<and> more))) = ((f);(g \<and> more)); (womega ((f);(g \<and> more))) "
 by (metis AndChopA Prop11 WOmegaUnroll WOmega_star_1 WPowerstar_inductL_var_equiv inteq_reflection)
 have 25: "\<turnstile>  (g \<and> more);womega (f;(g \<and> more)) = ((g \<and> more);f \<and> more); ((g \<and> more);womega (f;(g \<and> more))) " 
   by (metis (no_types, lifting) "2" AndChopB ChopAssoc MoreChopImpMore Prop10 int_eq lift_imp_trans)
 have 3: "\<turnstile> (womega ((f);(g \<and> more))) \<longrightarrow> (f) ; (womega ((g \<and> more);(f))) " 
    using 2 25 WOmega_coinduct_eq_var2[of "LIFT (g \<and> more) ; (womega ((f);(g \<and> more))) " "LIFT ((g \<and> more);(f))" ]
     by (metis (no_types, opaque_lifting) ChopAssoc RightChopImpChop inteq_reflection)
 show ?thesis 
 using "1" "3" int_iffI by blast
qed

lemma Omega_Wagner_2_var1: 
 "\<turnstile> (omega ((f)\<frown>(g \<and> more))) = (f) \<frown> (omega ((g \<and> more)\<frown>(f))) "
proof -
 have 1: "\<turnstile>  (f) \<frown> (omega ((g \<and> more)\<frown>(f))) \<longrightarrow> (omega ((f)\<frown>(g \<and> more))) "
    using Omega_simulation[of f "LIFT (g \<and> more) \<frown> f" "LIFT f \<frown> (g \<and> more)" ] 
    by (metis (no_types, lifting) AndMoreSChopAndMoreEqvAndMoreSChop Prop10 SChopAndB SChopAssoc 
        SChopMoreImpMore int_iffD1 inteq_reflection lift_imp_trans) 
 have 2: "\<turnstile> (omega ((f)\<frown>(g \<and> more))) = ((f)\<frown>(g \<and> more))\<frown> (omega ((f)\<frown>(g \<and> more))) "
 by (metis AndSChopA OmegaUnroll SChopstarMore_induct_lvar_eq SChopstar_inductL_var_equiv int_eq int_iffI ) 
 have 3: "\<turnstile> (omega ((f)\<frown>(g \<and> more))) \<longrightarrow> (f) \<frown> (omega ((g \<and> more)\<frown>(f))) " 
    using 2 Omega_coinduct_eq_var2[of "LIFT (g \<and> more) \<frown> (omega ((f)\<frown>(g \<and> more))) " "LIFT ((g \<and> more)\<frown>(f))" ]  
     by (metis (no_types, lifting) AndMoreSChopAndMoreEqvAndMoreSChop RightSChopImpSChop SChopAssoc inteq_reflection)
 show ?thesis 
 using "1" "3" int_iffI by blast
qed

lemma WOmega_Wagner_2_var2: 
 "\<turnstile> (womega ((f \<and> more);(g))) = (f \<and> more) ; (womega ((g);(f \<and> more))) "
proof -
 have 1: "\<turnstile>  (f \<and> more) ; (womega ((g);(f \<and> more))) \<longrightarrow> (womega ((f \<and> more);(g))) "
 by (metis (no_types, lifting) ChopAssoc WOmega_Wagner_2_var1 WOmega_star_1 WPowerstar_inductL_var_equiv int_eq int_iffD1)
 have 2: "\<turnstile>  (womega ((f \<and> more);(g))) = ((f \<and> more);(g)); (womega ((f \<and> more);(g)))"
 by (metis (no_types, lifting) "1" AndChopA ChopAssoc WOmegaUnroll WOmega_Wagner_2_var1 int_iffI inteq_reflection)
 have 3: " \<turnstile> (womega ((f \<and> more);(g))) \<longrightarrow> (f \<and> more) ; (womega ((g);(f \<and> more))) "
  using 1 2 WOmega_Wagner_2_var1[of _ f]
 by (metis "1" "2" ChopAssoc inteq_reflection)
show ?thesis 
 using "1" "3" int_iffI by blast
qed 

lemma Omega_Wagner_2_var2: 
 "\<turnstile> (omega ((f \<and> more)\<frown>(g))) = (f \<and> more) \<frown> (omega ((g)\<frown>(f \<and> more))) "
proof -
 have 1: "\<turnstile>  (f \<and> more) \<frown> (omega ((g)\<frown>(f \<and> more))) \<longrightarrow> (omega ((f \<and> more)\<frown>(g))) "
 by (metis (no_types, lifting) AndMoreSChopAndMoreEqvAndMoreSChop Omega_simulation SChopAndA SChopAssoc int_eq)
 have 2: "\<turnstile>  (omega ((f \<and> more)\<frown>(g))) = ((f \<and> more)\<frown>(g))\<frown> (omega ((f \<and> more)\<frown>(g)))"
 by (metis AndMoreSChopAndMoreEqvAndMoreSChop OmegaUnroll inteq_reflection )
 have 3: " \<turnstile> (omega ((f \<and> more)\<frown>(g))) \<longrightarrow> (f \<and> more) \<frown> (omega ((g)\<frown>(f \<and> more))) "
 by (meson "2" Omega_Wagner_2_var1 Prop04 RightSChopEqvSChop SChopAssoc int_iffD1 int_iffD2 int_iffI)
show ?thesis 
 using "1" "3" int_iffI by blast
qed 

lemma WOmega_Wagner_3: 
 assumes "\<turnstile> ((f \<and> more) ; (womega (f \<or> g)) \<or> h) = (womega (f \<or> g)) "
 shows   "\<turnstile> (womega (f \<or> g)) = ((womega f) \<or> (wpowerstar f);h) "
proof -
 have 1: "\<turnstile> (womega (f \<or> g)) \<longrightarrow> ((womega f) \<or> (wpowerstar (f \<and> more));h) "
   using WOmega_coinduct[of "LIFT (womega (f \<or> g))" h f]  assms
   by fastforce
 have 11: "\<turnstile> (wpowerstar (f \<and> more));h = (wpowerstar f);h "
   by (simp add: LeftChopEqvChop WPowerstar_more_absorb) 
 have 2: "\<turnstile> (wpowerstar f);h  \<longrightarrow> (womega (f \<or> g))"
   by (metis ChopImpChop Prop03 WOmega_star_1 WPowerstar_subdist assms inteq_reflection)
 have 3: "\<turnstile> ((womega f) \<or> (wpowerstar f);h) \<longrightarrow> (womega (f \<or> g))"
   by (meson "2" Prop02 WOmega_subdist)
 show ?thesis 
 using "1" "3" int_iffI
 by (metis "11" inteq_reflection) 
qed

lemma Omega_Wagner_3: 
 assumes "\<turnstile> ((f \<and> more) \<frown> (omega (f \<or> g)) \<or> (h)) = (omega (f \<or> g)) "
 shows   "\<turnstile> (omega (f \<or> g)) = ((omega f) \<or> (schopstar f)\<frown>(h)) "
proof -
 have 1: "\<turnstile> (omega (f \<or> g)) \<longrightarrow> ((omega f) \<or> (schopstar f)\<frown>h) "
   using assms Omega_coinduct[of "LIFT (omega (f \<or> g))" h f] 
   by fastforce
 have 2: "\<turnstile> (schopstar f)\<frown>h  \<longrightarrow> (omega (f \<or> g))" 
   by (meson Omega_star_1 Prop03 Prop11 SChopImpSChop SChopstar_subdist assms lift_imp_trans)
 have 3: "\<turnstile> ((omega f) \<or> (schopstar f)\<frown>h) \<longrightarrow> (omega (f \<or> g))"
   by (meson "2" Prop02 Omega_subdist)
 show ?thesis using 1 3 by fastforce
qed


lemma WOmega_Wagner_1_var:
"\<turnstile> (womega ((wpowerstar f);(f \<and> more))) = (womega f) "
by (metis WOmega_More_absorb WOmega_Wagner_1 WOmega_Wagner_2_var1 WOmega_star_1
  WpowerstarAndMoreEqvAndMoreChop inteq_reflection)


lemma Omega_Wagner_1_var3:
"\<turnstile> (omega ((schopstar f)\<frown>(f \<and> more))) = (omega f) " 
by (metis Omega_Wagner_1_var2 Omega_Wagner_2_var1 Omega_star_1 inteq_reflection)

lemma WOmega_star_4:
 "\<turnstile> (wpowerstar (womega f)) = (empty \<or> (womega f)) "
 by (simp add: WOmega_1 WPowerstar_boffa_var) 


lemma WOmega_star_5:
 "\<turnstile> (womega f);(wpowerstar (womega f)) = (womega f) " 
proof -
 have 1: "\<turnstile> (womega f);(wpowerstar (womega f)) \<longrightarrow> (womega f) "
   by (simp add: WOmega_1)
 have 2: "\<turnstile> (womega f) \<longrightarrow> (womega f);(wpowerstar (womega f))"
  by (simp add: WOmega_sup_id WPowerstar_imp_empty int_iffD2)
 show ?thesis 
 by (simp add: "1" "2" int_iffI)
qed

lemma WOmega_or_unfold: 
 "\<turnstile> ((womega f) \<or> ((wpowerstar f);(g \<and> more)); (womega (f \<or> g))) = (womega (f \<or> g)) " 
proof -
 have 10: "\<turnstile> (womega (f \<or> g)) = ((f \<or> g) \<and> more);(womega (f \<or> g)) "
 by (simp add: WOmegaUnroll) 
 have 11: "\<turnstile> ((f \<or> g) \<and> more) = ((f \<and> more) \<or> (g \<and> more)) "
    by fastforce
 have 1: "\<turnstile> (womega (f \<or> g)) = ((f \<and> more);(womega (f \<or> g)) \<or> (g \<and> more);(womega (f \<or> g))) " 
    using 10 11 by (metis OrChopEqv inteq_reflection) 
 have 2: "\<turnstile>((wpowerstar f);(g \<and> more)); (womega (f \<or> g)) = (wpowerstar f);((g \<and> more); (womega (f \<or> g)))"
   by (meson ChopAssoc int_iffD1 int_iffD2 int_iffI) 
 show ?thesis using 1 2 WOmega_Wagner_3[of f g "LIFT (g \<and> more);(womega (f \<or> g))" ] 
 by (metis int_eq)
qed

lemma Omega_or_unfold: 
 "\<turnstile> ((omega f) \<or> ((schopstar f)\<frown>(g \<and> more))\<frown> (omega (f \<or> g))) = (omega (f \<or> g)) " 
proof -
 have 10: "\<turnstile> (omega (f \<or> g)) = ((f \<or> g) \<and> more)\<frown>(omega (f \<or> g)) "
 by (simp add: OmegaUnroll) 
 have 11: "\<turnstile> ((f \<or> g) \<and> more) = ((f \<and> more) \<or> (g \<and> more)) "
    by fastforce
 have 1: "\<turnstile> (omega (f \<or> g)) = ((f \<and> more)\<frown>(omega (f \<or> g)) \<or> (g \<and> more)\<frown>(omega (f \<or> g))) " 
    using 10 11 by (metis OrSChopEqv inteq_reflection) 
 have 2: "\<turnstile>((schopstar f)\<frown>(g \<and> more))\<frown> (omega (f \<or> g)) = (schopstar f)\<frown>((g \<and> more)\<frown> (omega (f \<or> g)))"
   by (meson SChopAssoc int_iffD1 int_iffD2 int_iffI) 
  show ?thesis using 1 2 Omega_Wagner_3[of f g "LIFT (g \<and> more)\<frown>(omega (f \<or> g))" ] 
   by (metis int_eq)
qed
  
lemma WOmega_or_unfold_coinduct: 
 "\<turnstile> (womega (f \<or> g)) \<longrightarrow> (womega (((wpowerstar f);(g \<and> more)) )) 
    \<or> (wpowerstar ((wpowerstar f);(g \<and> more)  ));(womega f)       "
using WOmega_or_unfold[of f g]    
      WOmega_coinduct_eq[of "LIFT (womega (f \<or> g))" "LIFT (womega f)"
       "LIFT (wpowerstar f);(g \<and> more)"  ]
      by (metis Prop10 Prop11 Prop12 RightChopImpMoreRule inteq_reflection lift_and_com)

lemma Omega_or_unfold_coinduct: 
 "\<turnstile> (omega (f \<or> g)) \<longrightarrow> (omega (((schopstar f)\<frown>(g \<and> more)) )) 
    \<or> (schopstar ((schopstar f)\<frown>(g \<and> more)  ))\<frown>(omega f)       "
using Omega_or_unfold[of f g]    
      Omega_coinduct_eq[of "LIFT (omega (f \<or> g))" "LIFT (omega f)"
       "LIFT (schopstar f)\<frown>(g \<and> more)"  ]
   by (metis Prop10 Prop12 RightSChopImpMoreRule int_eq int_iffD2 lift_and_com)

lemma WOmega_or_unfold_induct: 
 "\<turnstile> (wpowerstar ((wpowerstar f);(g \<and> more)));(womega f) \<longrightarrow> (womega (f \<or> g))  " 
using WOmega_or_unfold[of f g] 
using WPowerstar_induct_leq by blast

lemma Omega_or_unfold_induct: 
 "\<turnstile> (schopstar ((schopstar f)\<frown>(g \<and> more)))\<frown>(omega f) \<longrightarrow> (omega (f \<or> g))  " 
using Omega_or_unfold[of f g] 
by (metis AndChopA SChopstar_WPowerstar WPowerstar_induct_leq inteq_reflection lift_imp_trans schop_d_def)


lemma WOmega_Wagner_1_gen: 
 "\<turnstile> (womega ((f \<and> more);(wpowerstar g) )) \<longrightarrow> (womega (f \<or> g)) " 
proof -
 have 01: "\<turnstile> f \<longrightarrow> (f \<or> g) " 
     by auto
 have 02: "\<turnstile> (wpowerstar g) \<longrightarrow> (wpowerstar (f \<or> g)) " 
         by (metis WPowerstar_subdist WPowerstar_swap inteq_reflection)
 have 03: "\<turnstile> ((f \<and> more);(wpowerstar g) ) \<longrightarrow> ((f \<or> g) \<and> more);(wpowerstar (f \<or> g)) "
       using "01" "02" ChopImpChop
       by (metis Prop10 Prop12 int_iffD1 inteq_reflection lift_and_com) 
 have 1: "\<turnstile> (womega ((f \<and> more);(wpowerstar g) )) \<longrightarrow>
            (womega (((f \<or> g) \<and> more);(wpowerstar (f \<or> g)))) " 
    by (simp add: "03" WOmega_iso)
 show ?thesis using WOmega_Wagner_1
 by (metis "03" Prop12 WOmega_iso WpowerstarAndMoreEqvAndMoreChop inteq_reflection)  
qed

lemma Omega_Wagner_1_gen: 
 "\<turnstile> (omega ((f \<and> more)\<frown>(schopstar g) )) \<longrightarrow> (omega (f \<or> g)) " 

proof -
 have 1: "\<turnstile> ((f \<and> more) \<and> finite \<or> (g \<and> more) \<and> finite) =(((f \<or> g) \<and> more) \<and> finite) " 
    by fastforce
 show ?thesis 
  using Omega_WOmega[of "LIFT ((f \<and> more)\<frown>(schopstar g) )"] Omega_WOmega[of "LIFT (f \<or> g)"] 
  WOmega_Wagner_1_gen[of "LIFT ((f \<and> more) \<and> finite)" "LIFT ((g \<and> more) \<and> finite)"] 
  SChopstar_WPowerstar[of "LIFT (g \<and> more)" ] SChopstar_WPowerstar_more[of g] 1
  by (metis (no_types, lifting) Omega_Wagner_1 Prop10 Prop12 SChopImpFinite SChopstar_WPowerstar_more SChopstar_finite int_iffD1 inteq_reflection itl_def(9) lift_and_com omega_d_def)
qed

lemma WOmega_swap: 
"\<turnstile> (womega (f \<or> g)) = (womega (g \<or> f)) "
proof -
 have 1: "\<turnstile> (f \<or> g) = (g \<or> f)"
   by fastforce
 show ?thesis 
 by (metis "1" WOmega_star_5 int_eq)
qed

lemma Omega_swap:
"\<turnstile> (omega (f \<or> g)) = (omega (g \<or> f)) " 
proof -
 have 1: "\<turnstile> (f \<or> g) = (g \<or> f)" 
  by fastforce
 show ?thesis 
 by (metis "1" Omega_star_1 inteq_reflection)
qed

lemma WOmega_Wagner_1_var_gen: 
 "\<turnstile> (womega ((wpowerstar f);(g \<and> more))) \<longrightarrow> (womega (f \<or> g)) " 
proof -
 have 1: "\<turnstile> (womega ((wpowerstar f);(g \<and> more))) =
            (wpowerstar f);(womega ((g \<and> more);(wpowerstar f)))"
            by (simp add: WOmega_Wagner_2_var1)
 have 2: "\<turnstile> (womega ((g \<and> more);(wpowerstar f))) \<longrightarrow> (womega (g \<or> f)) "
    using WOmega_Wagner_1_gen[of g f]  by blast
 have 5: "\<turnstile> (wpowerstar f);(womega ((g \<and> more);(wpowerstar f)))
            \<longrightarrow> (wpowerstar f);(womega (f \<or> g)) "
     by (metis "2" RightChopImpChop WOmega_swap int_eq)
 have 6: "\<turnstile> (wpowerstar f) \<longrightarrow> (wpowerstar (f \<or> g)) " 
   by (simp add: WPowerstar_subdist)
 have 7: "\<turnstile> (wpowerstar f);(womega (f \<or> g))  \<longrightarrow> (wpowerstar (f \<or> g));(womega (f \<or> g)) "
   by (simp add: "6" LeftChopImpChop) 
 have 8: "\<turnstile> (wpowerstar (f \<or> g));(womega (f \<or> g)) \<longrightarrow>  (womega (f \<or> g)) "
   by (simp add: WOmega_star_1 int_iffD1)
 show ?thesis 
 using "1" "5" "7" "8" 
 by (metis int_eq lift_imp_trans)
qed

lemma Omega_Wagner_1_var_gen: 
 "\<turnstile> (omega ((schopstar f)\<frown>(g \<and> more))) \<longrightarrow> (omega (f \<or> g)) " 
proof -
 have 1: "\<turnstile> (omega ((schopstar f)\<frown>(g \<and> more))) =
            (schopstar f)\<frown>(omega ((g \<and> more)\<frown>(schopstar f)))"
  by (simp add: Omega_Wagner_2_var1) 
 have 2: "\<turnstile> (omega ((g \<and> more)\<frown>(schopstar f))) \<longrightarrow> (omega (g \<or> f)) "
    using Omega_Wagner_1_gen[of g f]  by blast
 have 3: "\<turnstile> (g \<or> f) = (f \<or> g) "
     by fastforce
 have 4: "\<turnstile> (omega (g \<or> f)) = (omega (f \<or> g)) " 
  by (simp add: Omega_swap)
 have 5: "\<turnstile> (schopstar f)\<frown>(omega ((g \<and> more)\<frown>(schopstar f)))
            \<longrightarrow> (schopstar f)\<frown>(omega (f \<or> g)) " 
   by (metis "2" "3" RightSChopImpSChop inteq_reflection)
 have 6: "\<turnstile> (schopstar f) \<longrightarrow> (schopstar (f \<or> g)) " 
   by (metis "3" MP Prop03 SBaGen SBaSCSImpSCS int_eq)
 have 7: "\<turnstile> (schopstar f)\<frown>(omega (f \<or> g))  \<longrightarrow> (schopstar (f \<or> g))\<frown>(omega (f \<or> g)) " 
   by (simp add: "6" LeftSChopImpSChop)
 have 8: "\<turnstile> (schopstar (f \<or> g))\<frown>(omega (f \<or> g)) \<longrightarrow>  (omega (f \<or> g)) "
   by (simp add: Omega_star_1 int_iffD1)
 show ?thesis 
 using "1" "5" "7" "8" by fastforce
qed

lemma WOmega_Denest: 
"\<turnstile> (womega (f \<or> g)) = 
    ((womega ((wpowerstar f);(g \<and> more)))  \<or>
     (wpowerstar ((wpowerstar f);(g \<and> more)));(womega f))"
proof -
 have 1: "\<turnstile> (womega (f \<or> g)) \<longrightarrow> (womega (((wpowerstar f);(g \<and> more)) )) 
    \<or> (wpowerstar ((wpowerstar f);(g \<and> more)));(womega f)  "
   by (simp add: WOmega_or_unfold_coinduct)
 have 2: " \<turnstile> (womega ((wpowerstar f);(g \<and> more))) \<longrightarrow> (womega (f \<or> g)) "
   by (simp add: WOmega_Wagner_1_var_gen)
 have 3: "\<turnstile> (wpowerstar ((wpowerstar f);(g \<and> more)));(womega f) \<longrightarrow> (womega (f \<or> g)) "
   by (simp add: WOmega_or_unfold_induct)
 show ?thesis 
 by (meson "1" "2" "3" Prop02 int_iffI)
qed 

lemma Omega_Denest: 
"\<turnstile> (omega (f \<or> g)) = 
    ((omega ((schopstar f)\<frown>(g \<and> more)))  \<or>
     (schopstar ((schopstar f)\<frown>(g \<and> more)))\<frown>(omega f))"
proof -
 have 1: "\<turnstile> (omega (f \<or> g)) \<longrightarrow> (omega (((schopstar f)\<frown>(g \<and> more)) )) 
    \<or> (schopstar ((schopstar f)\<frown>(g \<and> more)  ))\<frown>(omega f)  "
   by (simp add: Omega_or_unfold_coinduct)
 have 2: " \<turnstile> (omega ((schopstar f)\<frown>(g \<and> more))) \<longrightarrow> (omega (f \<or> g)) "
   by (simp add: Omega_Wagner_1_var_gen)
 have 3: "\<turnstile> (schopstar ((schopstar f)\<frown>(g \<and> more)))\<frown>(omega f) \<longrightarrow> (omega (f \<or> g)) "
   by (simp add: Omega_or_unfold_induct)
 show ?thesis 
 by (meson "1" "2" "3" Prop02 int_iffI)
qed 

lemma WOmega_or_refine: 
assumes "\<turnstile> (g \<and> more); (f \<and> more) \<longrightarrow> (f \<and> more) ; (wpowerstar (f \<or> g)) "
shows "\<turnstile> (womega (f \<or> g)) = ((womega f) \<or> (wpowerstar f);(womega g))"
proof -
 have 1: "\<turnstile> (wpowerstar g);(f \<and> more) \<longrightarrow> (f \<and> more);(wpowerstar (f \<or> g)) " 
   using  assms WPowerstar_invol WPowerstar_sim1[of "LIFT (g \<and> more )"  "LIFT (f \<and> more)" "LIFT powerstar (f \<or> g)"] 
    
    by (metis (no_types, opaque_lifting) ChopstarEqvPowerstar Chopstar_WPowerstar WPowerstar_more_absorb
      inteq_reflection)
 have 2: "\<turnstile> (womega (f \<or> g)) = ((womega g) \<or> ((wpowerstar g);(f \<and> more));(womega (f \<or> g))) " 
    by (metis WOmega_swap WOmega_or_unfold inteq_reflection)
 have 3: "\<turnstile> ((womega g) \<or> ((wpowerstar g);(f \<and> more));(womega (f \<or> g))) \<longrightarrow>
              (womega g) \<or> ((f \<and> more);(wpowerstar (f \<or> g)));(womega (f \<or> g)) "
    by (metis "1" "2" AndChopB Prop08 Prop10 int_iffD2 inteq_reflection)
 have 40: "\<turnstile> (womega g) \<longrightarrow> (f \<and> more);(womega (f \<or> g)) \<or> (womega g) "
    by (simp add: Prop05) 
 have 41: "\<turnstile> ((f \<and> more);(wpowerstar (f \<or> g)));(womega (f \<or> g)) \<longrightarrow>
             (f \<and> more);(womega (f \<or> g)) \<or> (womega g) "
    by (metis (mono_tags, lifting) ChopAssoc WOmega_star_1 intI inteq_reflection unl_lift2)        
 have 4: "\<turnstile> (womega g) \<or> ((f \<and> more);(wpowerstar (f \<or> g)));(womega (f \<or> g)) \<longrightarrow>
             (f \<and> more);(womega (f \<or> g)) \<or> (womega g) "
    using "40" "41" Prop02 by blast 
 have 5: "\<turnstile> (womega (f \<or> g)) \<longrightarrow> ((womega f) \<or> (wpowerstar f);(womega g)) "
 by (metis "2" "3" ChopAssoc WOmega_coinduct_var WOmega_star_1 inteq_reflection) 
 have 6: "\<turnstile> ((womega f) \<or> (wpowerstar f);(womega g)) \<longrightarrow>
             (womega (f \<or> g)) \<or> (wpowerstar (f \<or> g));(womega (f \<or> g)) "
   by (metis ChopImpChop Prop02 Prop05 WOmega_star_1 WOmega_subdist WOmega_swap WPowerstar_subdist int_eq) 
 have 7: "\<turnstile> ((womega f) \<or> (wpowerstar f);(womega g)) \<longrightarrow> (womega (f \<or> g))"
   using "6" WOmega_star_1 
   by (metis Prop02 int_iffD1 inteq_reflection lift_imp_trans)
 show ?thesis 
 using "5" "7" int_iffI by blast
qed



lemma Omega_or_refine: 
assumes "\<turnstile> (g \<and> more)\<frown> ((f \<and> more) \<and> finite) \<longrightarrow> (f \<and> more) \<frown> (schopstar (f \<or> g)) "
shows "\<turnstile> (omega (f \<or> g)) = ((omega f) \<or> (schopstar f)\<frown>(omega g))"
proof -
 have 1: "\<turnstile> (schopstar (g ))\<frown>((f \<and> more) \<and> finite) \<longrightarrow> (f \<and> more)\<frown>(schopstar (f \<or> g)) " 
  using assms 
  using SChopstar_QuasicommMore_var by blast
 have 2: "\<turnstile> (omega (f \<or> g)) = ((omega g) \<or> ((schopstar g)\<frown>(f \<and> more))\<frown>(omega (f \<or> g))) " 
    by (metis Omega_swap Omega_or_unfold inteq_reflection)
 have 20: "\<turnstile> ((schopstar g)\<frown>(f \<and> more))\<frown>(omega (f \<or> g)) =
             ((schopstar g)\<frown>((f \<and> more) \<and> finite))\<frown>(omega (f \<or> g)) " 
   by (metis (no_types, lifting) AndMoreAndFiniteEqvAndFmore Prop10 Prop12 SChopAssoc 
       int_eq int_iffD2 itl_def(9))
 have 25: "\<turnstile> ((schopstar g)\<frown>(f \<and> more))\<frown>(omega (f \<or> g)) \<longrightarrow>
             ((f \<and> more)\<frown>(schopstar (f \<or> g)))\<frown>(omega (f \<or> g)) "
    using 1 20 by (metis LeftSChopImpSChop inteq_reflection) 
 have 3: "\<turnstile> ((omega g) \<or> ((schopstar g)\<frown>(f \<and> more))\<frown>(omega (f \<or> g))) \<longrightarrow>
              (omega g) \<or> ((f \<and> more)\<frown>(schopstar (f \<or> g)))\<frown>(omega (f \<or> g)) "
   using "25" by auto
 have 4: "\<turnstile> (omega g) \<or> ((f \<and> more)\<frown>(schopstar (f \<or> g)))\<frown>(omega (f \<or> g)) \<longrightarrow>
             (omega g) \<or>  (f \<and> more)\<frown>(omega (f \<or> g))  "   
    by (metis (mono_tags, lifting) Omega_star_1 SChopAssoc intI int_eq intensional_rews(3))
 have 5: "\<turnstile> (omega (f \<or> g)) \<longrightarrow> ((omega f) \<or> (schopstar f)\<frown>(omega g)) " 
   by (metis "2" "3" Omega_coinduct Omega_star_1 SChopAssoc inteq_reflection)   
 have 50: "\<turnstile> (omega f) \<longrightarrow> (omega (f \<or> g))" 
    by (simp add: Omega_subdist)
 have 51: "\<turnstile> (omega g) \<longrightarrow> (omega (f \<or> g)) "
      by (metis Omega_swap Omega_subdist inteq_reflection)
 have 52: "\<turnstile> (schopstar f) \<longrightarrow> (schopstar (f \<or> g)) "
    by (simp add: SChopstar_subdist)
 have 6: "\<turnstile> ((omega f) \<or> (schopstar f)\<frown>(omega g)) \<longrightarrow>
             (omega (f \<or> g)) \<or> (schopstar (f \<or> g))\<frown>(omega (f \<or> g)) " 
   by (metis "50" "51" "52" Omega_star_1 Prop02 Prop05 SChopImpSChop inteq_reflection)   
 have 7: "\<turnstile> ((omega f) \<or> (schopstar f)\<frown>(omega g)) \<longrightarrow> (omega (f \<or> g))"
   using "6" Omega_star_1 by fastforce
 show ?thesis 
 using "5" "7" int_iffI by blast
qed


lemma WOmega_bachmair_dershowitz:
assumes "\<turnstile> (g \<and> more);(f \<and> more) \<longrightarrow> (f \<and> more);(wpowerstar (f \<or> g)) "
shows   "(\<turnstile> (womega (f \<or> g)) = #False) = (\<turnstile> ((womega f) \<or> (womega g)) = #False) " 
proof -
 have 1: "(\<turnstile> (womega (f \<or> g)) = #False) \<Longrightarrow> (\<turnstile> ((womega f) \<or> (womega g)) = #False)"
   using assms 
   by (metis Prop10 WOmega_subdist_var int_eq int_simps(18))
 have 2: "(\<turnstile> ((womega f) \<or> (womega g)) = #False)  \<Longrightarrow>  (\<turnstile> (womega (f \<or> g)) = #False)"
   using assms WOmega_or_refine[of g f] 
   by (metis Prop03 Prop10 WOmega_star_1 int_simps(18) int_simps(25) int_simps(26) inteq_reflection)
 show ?thesis 
 using "1" "2" by blast
qed


lemma Omega_bachmair_dershowitz:
assumes "\<turnstile> (g \<and> more)\<frown>((f \<and> more) \<and> finite) \<longrightarrow> (f \<and> more)\<frown>(schopstar (f \<or> g)) "
shows   "(\<turnstile> (omega (f \<or> g)) = #False) = (\<turnstile> ((omega f) \<or> (omega g)) = #False) " 
proof -
 have 1: "(\<turnstile> (omega (f \<or> g)) = #False) \<Longrightarrow> (\<turnstile> ((omega f) \<or> (omega g)) = #False)"
   using assms 
   by (metis Prop10 Omega_subdist_var int_eq int_simps(18))
 have 2: "(\<turnstile> ((omega f) \<or> (omega g)) = #False)  \<Longrightarrow>  (\<turnstile> (omega (f \<or> g)) = #False)"
   using assms Omega_or_refine[of g f] 
    by (metis Prop03 SChopRightFalse int_simps(14) int_simps(16) int_simps(21) int_simps(9) inteq_reflection)
 show ?thesis 
 using "1" "2" by blast
qed


lemma Omega_and_more_imp_more: 
 "\<turnstile> (omega (f \<and> more)) \<longrightarrow> more "
 by (metis AndSChopB MoreSChopImpMore OmegaUnroll inteq_reflection lift_imp_trans)

lemma WOmega_and_more_imp_more: 
 "\<turnstile> (womega (f \<and> more)) \<longrightarrow> more "
by (metis ChopImpDi DiAndB DiMoreEqvMore WOmegaUnroll int_eq lift_imp_trans)

lemma Omega_and_fmore_imp_more: 
 "\<turnstile> (omega ((f \<and> more) \<and> finite)) \<longrightarrow> more "
by (metis Omega_and_more_imp_more Omega_subdist OrFiniteInf inteq_reflection lift_imp_trans)

lemma WOmega_and_fmore_imp_more: 
 "\<turnstile> (womega ((f \<and> more) \<and> finite)) \<longrightarrow> more "
by (metis LeftChopImpMoreRule Prop12 WOmegaUnroll int_iffD1 inteq_reflection lift_and_com)


lemma womega_len: 
 "\<turnstile> womega skip = (len k);womega skip " 
proof (induct k)
case 0
then show ?case 
by (metis EmptyChop inteq_reflection wpow_0 wpower_len)
next
case (Suc k)
then show ?case 
 proof -
 have 1: "\<turnstile> len Suc k = len k ; skip "
 by (metis LenNPlusOneB Suc_eq_plus1)
  have 2: "\<turnstile> len Suc k ; womega skip = len k;(skip;womega skip) "
  by (metis "1" ChopAssoc inteq_reflection) 
  have 3: "\<turnstile> skip;womega skip = womega skip "
  by (metis AndChopA Prop11 WOmegaUnroll WPowerstar_inductL_var_equiv WPowerstar_more_absorb inteq_reflection) 
  show ?thesis
  by (metis "2" "3" Suc inteq_reflection) 
 qed
qed


lemma omega_len: 
 "\<turnstile> omega skip = (len k)\<frown>omega skip " 
proof (induct k)
case 0
then show ?case 
by (metis EmptyChop FiniteAndEmptyEqvEmpty inteq_reflection lift_and_com schop_d_def wpow_0 wpower_len)
next
case (Suc k)
then show ?case 
 proof -
  have 1: "\<turnstile> len Suc k = len k \<frown> skip "
    by (simp add: len_k_schop)
  have 2: "\<turnstile> len Suc k \<frown> omega skip = len k\<frown>(skip\<frown>omega skip) " 
    by (metis "1" SChopAssoc inteq_reflection)
  have 3: "\<turnstile> skip\<frown>omega skip = omega skip "
  by (metis FiniteChopSkipEqvFiniteAndMore FiniteChopSkipEqvSkipChopFinite FmoreEqvSkipChopFinite OmegaUnroll Omega_Wagner_1 Omega_star_1 SChopAssoc inteq_reflection schopstar_skip_finite schopstar_skip_fmore) 
  show ?thesis 
  by (metis "2" "3" Suc inteq_reflection)
 qed
qed

lemma womega_exist_len: 
 "\<turnstile> (womega skip) = (\<exists>k. len k; (womega skip))"
proof -
 have 1: "\<turnstile> (womega skip) = (wpowerstar skip);(womega skip)"
 by (meson Prop11 WOmega_star_1) 
 have 2: "\<turnstile> (wpowerstar skip) = (\<exists> k. len k) " 
   unfolding wpowerstar_d_def 
     by (simp add: ExEqvRule wpower_len)
 have 3: "\<turnstile> (\<exists> k. len k);(womega skip) = (\<exists>k. len k; (womega skip))"
 by (metis ExistChop int_eq)
 show ?thesis using 1 2 3 
 by (metis int_eq)
qed

lemma omega_exist_len: 
 "\<turnstile> (omega skip) = (\<exists>k. len k\<frown> (omega skip))"
proof -
 have 1: "\<turnstile> (omega skip) = (schopstar skip)\<frown>(omega skip)" 
   by (meson Omega_star_1 Prop11)
 have 2: "\<turnstile> (schopstar skip) = (\<exists> k. len k) " 
   by (metis ExEqvRule int_eq schopstar_skip_finite wpower_len wpowerstar_d_def wpowerstar_skip_finite)
 have 3: "\<turnstile> (\<exists> k. len k)\<frown>(omega skip) = (\<exists>k. len k\<frown> (omega skip))"
   by (metis ExistSChop inteq_reflection)
 show ?thesis using 1 2 3 
  by (metis int_eq)
qed


lemma not_len_and_womega: 
" \<turnstile> \<not> (len k \<and> (womega skip))"
using womega_len[of "Suc k" ] 
using len_len_suc_not by fastforce 

lemma not_len_and_omega: 
" \<turnstile> \<not> (len k \<and> (omega skip))"
using omega_len[of "Suc k" ] 
len_len_suc_not_schop by fastforce 

lemma not_len_and_womega_var: 
 "\<turnstile> (womega skip) \<longrightarrow> \<not>(len k) " 
using not_len_and_womega by fastforce

lemma not_len_and_omega_var: 
 "\<turnstile> (omega skip) \<longrightarrow> \<not>(len k) " 
using not_len_and_omega by fastforce


lemma womega_skip_inf:
 "\<turnstile> (womega skip) \<longrightarrow> inf " 
proof -
 have 1: "\<turnstile> finite = (\<exists>k. len k) " 
   by (simp add: intI itl_defs len_defs  nfinite_conv_nlength_enat)
 have 2: "\<turnstile> inf = (\<forall>k. \<not> len k) " 
   using 1 unfolding finite_d_def by fastforce
 have 3: "\<turnstile> (womega skip) \<longrightarrow> (\<forall>k. \<not> len k)"
    using not_len_and_womega_var by fastforce
 show ?thesis using 2 3 
 by (metis int_eq) 
qed

lemma omega_skip_inf:
 "\<turnstile> (omega skip) \<longrightarrow> inf " 
proof -
 have 1: "\<turnstile> finite = (\<exists>k. len k) " 
   by (simp add: intI itl_defs  len_defs nfinite_conv_nlength_enat)
 have 2: "\<turnstile> inf = (\<forall>k. \<not> len k) " 
   using 1 unfolding finite_d_def by fastforce
 have 3: "\<turnstile> (omega skip) \<longrightarrow> (\<forall>k. \<not> len k)"
    using not_len_and_omega_var by fastforce
 show ?thesis using 2 3 
  by fastforce
qed

lemma womega_fmore_inf: 
 "\<turnstile> (womega fmore) \<longrightarrow> inf " 
using wpowerstar_skip_fmore
by (metis SkipChopFiniteImpFinite WOmega_Wagner_1 WOmega_iso inteq_reflection lift_imp_trans womega_skip_inf wpowerstar_skip_finite)

lemma omega_fmore_inf: 
 "\<turnstile> (omega fmore) \<longrightarrow> inf "
 by (metis AndChopA ChopEmpty WOmega_iso inteq_reflection lift_imp_trans omega_d_def womega_fmore_inf)  

lemma womega_and_fmore_inf: 
"\<turnstile> (womega (f \<and> fmore)) \<longrightarrow> inf "
by (meson Prop12 WOmega_iso int_iffD1 lift_and_com lift_imp_trans womega_fmore_inf)

lemma omega_and_fmore_inf: 
"\<turnstile> (omega (f \<and> fmore)) \<longrightarrow> inf "
by (meson Prop12 Omega_iso int_iffD1 lift_and_com lift_imp_trans omega_fmore_inf)


lemma womega_and_empty: 
 "\<turnstile> \<not>(womega (f \<and> empty)) "
 by (metis WOmega_Wagner_1 WOmega_empty_zero WPowerstar_and_empty inteq_reflection)

lemma omega_and_empty:
 "\<turnstile> \<not>(omega (f \<and> empty))"
 by (meson MP Omega_empty_zero Omega_iso Prop11 Prop12 lift_and_com lift_imp_neg)

lemma womega_split_empty_more: 
 "\<turnstile> (womega f) =  (womega (f \<and> more)) "
 by (metis WOmega_Wagner_1 WPowerstar_more_absorb inteq_reflection) 


lemma omega_split_empty_more: 
 "\<turnstile> (omega f) = ( (omega (f \<and> more)) ) "
 by (metis Omega_Wagner_1 SChopstar_and_more inteq_reflection)


lemma omega_split_empty_more_var: 
 "\<turnstile> (omega (f \<and> more)) = (omega (f \<and> fmore)) "
by (metis AndMoreSChopEqvAndFmoreChop ChopEmpty EmptySChop Omega_Wagner_2_var1 inteq_reflection) 

lemma omega_imp_inf: 
"\<turnstile> (omega f) \<longrightarrow> inf" 
by (metis int_eq omega_and_fmore_inf omega_split_empty_more omega_split_empty_more_var)

lemma inf_imp_omega_skip: 
"\<turnstile> inf \<longrightarrow> (omega skip)"
proof -
 have 1: "\<turnstile> ((skip \<and> more) \<and> finite) = skip "
    by (metis DiIntro DiSkipEqvMore Prop10 WPowerstar_ext WPowerstar_skip_finite inteq_reflection)
 have 2: "\<turnstile> inf \<longrightarrow> skip;inf" 
   by (metis AndInfEqvChopFalse ChopAndInf MoreAndInfEqvInf MoreEqvSkipChopTrue infinite_d_def int_eq int_iffD1)
 have 3: "\<turnstile> inf \<longrightarrow> ((skip \<and> more) \<and> finite);inf"
   using 1 2 by (metis int_eq) 
 show ?thesis
 by (simp add: "3" OmegaWeakCoinduct schop_d_def)  
qed

 

lemma Omega_schopstar_max_element: 
 "\<turnstile> f \<longrightarrow> (omega skip) \<or> (schopstar skip) " 
proof -
 have 1: "\<turnstile> (schopstar skip) = finite " 
  by (meson Prop11 schopstar_skip_finite)
 have 2: "\<turnstile> (omega skip) = inf " 
   by (simp add: inf_imp_omega_skip int_iffI omega_skip_inf)
 show ?thesis using 1 2 unfolding finite_d_def by fastforce
qed



lemma Omega_fmore_absorb: 
 " \<turnstile> (omega f) = (omega (f \<and> fmore))"
by (metis int_eq omega_split_empty_more omega_split_empty_more_var)


lemma Omega_top:
 "\<turnstile> (omega f);(omega empty) = (omega f)"
proof -
 have 1: "\<turnstile> (omega f) \<longrightarrow> inf "
    by (simp add: omega_imp_inf)
  have 2: "\<turnstile> (omega f);(omega empty) \<longrightarrow> (omega f) "
    by (simp add: Omega_1)
  have 3: "\<turnstile> (omega f) \<longrightarrow> (omega f);(omega empty) " 
    by (metis "1" AndInfChopEqvAndInf Prop10 int_iffD2 inteq_reflection)
  show ?thesis 
  by (simp add: "2" "3" int_iffI)
qed


lemma womega_and_inf: 
 "\<turnstile> (womega (f \<and> inf)) = (f \<and> inf) "
 by (metis (no_types, lifting) AndInfChopEqvAndInf AndMoreAndInfEqvAndInf Prop10 Prop12 WOmegaUnroll int_iffD1 inteq_reflection) 

lemma womega_split_finite_inf: 
 "\<turnstile> (womega f) =
    (wpowerstar (f \<and> finite);(f \<and> inf) \<or> (womega (f \<and> finite))) "
proof -
 have 0: "\<turnstile> (womega f) = (womega ((f \<and> finite) \<or> (f \<and> inf))) " 
    by (metis OrFiniteInf int_eq)
 have 1: "\<turnstile> (womega ((f \<and> finite) \<or> (f \<and> inf))) =
     (womega (wpowerstar (f \<and> finite);((f \<and> inf) \<and> more)) \<or> 
      wpowerstar (wpowerstar (f \<and> finite);((f \<and> inf) \<and> more));womega (f \<and> finite)) " 
  using WOmega_Denest[of "LIFT f \<and> finite" "LIFT (f \<and> inf)"]
 by blast
 have 15: "\<turnstile> ((f \<and> inf) \<and> more) = (f \<and> inf) "
 using MoreAndInfEqvInf by auto    
 have 2: "\<turnstile> (womega (wpowerstar (f \<and> finite);((f \<and> inf) \<and> more))) = wpowerstar (f \<and> finite);(f \<and> inf) "
 by (metis "15" AndInfChopEqvAndInf WOmega_Wagner_2_var1 inteq_reflection)
 have 3: "\<turnstile> wpowerstar (wpowerstar (f \<and> finite);((f \<and> inf) \<and> more)) = 
           (empty \<or>  (wpowerstar (f \<and> finite);(f \<and> inf))) "
           by (metis "15" "2" WOmega_star_4 inteq_reflection)
 have 4: "\<turnstile> (empty \<or>  (wpowerstar (f \<and> finite);(f \<and> inf))); womega (f \<and> finite) =
    (womega (f \<and> finite) \<or> (wpowerstar (f \<and> finite);(f \<and> inf)))  "
    by (meson AndInfChopEqvAndInf ChopAssoc EmptyOrChopEqv Prop04 Prop06 RightChopEqvChop)
 have 5: "\<turnstile> (womega f) = 
          (wpowerstar (f \<and> finite);(f \<and> inf) \<or> 
            (empty \<or>  (wpowerstar (f \<and> finite);(f \<and> inf)));womega (f \<and> finite)) " 
    by (metis "0" "1" 2 "3" int_eq)
 have 6: "\<turnstile> (wpowerstar (f \<and> finite);(f \<and> inf) \<or> 
            (empty \<or>  (wpowerstar (f \<and> finite);(f \<and> inf)));womega (f \<and> finite)) =
            ((wpowerstar (f \<and> finite);(f \<and> inf)) \<or> 
            (womega (f \<and> finite)) \<or> (wpowerstar (f \<and> finite);(f \<and> inf))) " 
    using 4 
    by (metis "5" int_eq)
  have 7: "\<turnstile> ((wpowerstar (f \<and> finite);(f \<and> inf)) \<or> 
            (womega (f \<and> finite)) \<or> (wpowerstar (f \<and> finite);(f \<and> inf))) =
            ((wpowerstar (f \<and> finite);(f \<and> inf)) \<or> 
            (womega (f \<and> finite)) ) "
     by auto 
 show ?thesis using 5 6 7 
 by (metis int_eq)
qed

lemma womega_imp_inf: 
"\<turnstile> (womega f) \<longrightarrow> inf" 
proof -
 have 1: "\<turnstile> (womega f) = (wpowerstar (f \<and> finite);(f \<and> inf) \<or> (womega (f \<and> finite)))" 
  using womega_split_finite_inf by blast
 have 2: "\<turnstile> (womega (f \<and> finite)) \<longrightarrow> inf "
  by (metis omega_d_def omega_imp_inf)
 have 3: "\<turnstile> wpowerstar (f \<and> finite);(f \<and> inf) \<longrightarrow> inf "
 by (meson AndChopB ChopAndB FiniteChopInfEqvInf Prop11 WPowerstar_inductL_var_equiv lift_imp_trans)
 show ?thesis using 1 2 3 by fastforce
qed 

lemma WOmega_Omega_skip: 
 "\<turnstile> womega skip = omega skip "
 by (metis FiniteChopSkipEqvFiniteAndMore FiniteChopSkipEqvSkipChopFinite Omega_Wagner_1 Prop10 SkipChopFiniteImpFinite WOmega_Wagner_1 WPowerstar_skip_finite inteq_reflection omega_d_def omega_split_empty_more schopstar_skip_finite womega_split_empty_more) 

lemma WOmega_Omega: 
 "\<turnstile> (womega f) = ( (schopstar f);( (f \<and> inf)) \<or> (omega f) ) "
 by (metis SChopstar_WPowerstar inteq_reflection omega_d_def womega_split_finite_inf) 



lemma womega_and_more_inf: 
 " \<turnstile> (womega (f \<and> more)) \<longrightarrow> inf "
 by (simp add: womega_imp_inf)


lemma womega_fmore_inf_eq: 
"\<turnstile> womega fmore = inf "
by (metis FiniteChopSkipEqvFiniteAndMore FiniteChopSkipEqvSkipChopFinite Prop11 WOmega_Omega_skip 
    WOmega_Wagner_1 WPowerstar_more_absorb inf_imp_omega_skip inteq_reflection womega_fmore_inf
 wpowerstar_skip_finite wpowerstar_skip_fmore)


lemma womega_skip_inf_eq:
 "\<turnstile> womega skip = inf "
 by (metis Prop11 WOmega_Omega_skip inf_imp_omega_skip inteq_reflection womega_skip_inf)


lemma womega_more_inf:
 "\<turnstile> womega more \<longrightarrow> inf "
 by (simp add: womega_imp_inf) 

lemma womega_more_inf_eq:
 "\<turnstile> womega more = inf " 
by (metis OrFiniteInf WOmega_subdist fmore_d_def int_eq int_iffI womega_fmore_inf_eq womega_more_inf)

lemma womega_true: 
 "\<turnstile> womega #True = inf"
 by (metis AndInfEqvChopFalse MoreAndInfEqvInf Prop11 RightChopImpChop WOmegaWeakCoinduct int_simps(11) int_simps(17) inteq_reflection womega_imp_inf) 

lemma OmegaChopOmega: 
 "\<turnstile> (omega f);g = (omega f) " 
by (metis AndInfChopEqvAndInf Prop10 inteq_reflection omega_imp_inf)

lemma WOmegaChopWOmega: 
 "\<turnstile> (womega f);g = (womega f) "
 by (metis ChopSChopdef Prop04 Prop05 Prop12 WOmega_1 WPowerstar_tc int_iffD2 int_iffI lift_imp_trans womega_imp_inf) 

lemma WOmegYieldsEvvNotWOmega: 
 "\<turnstile> (womega f) yields g = (\<not>(womega f)) "
 by (simp add: WOmegaChopWOmega WOmega_1 int_iffD2 int_iffI yields_d_def) 

subsection \<open>Properties of Omega\<close>
 

lemma NotOmegaInf:
 "\<turnstile> \<not>(omega (inf))"
proof -
 have 1: "\<turnstile> omega (inf) = ((inf \<and> more) \<and> finite);omega inf"
 by (metis OmegaUnroll schop_d_def) 
 have 2: "\<turnstile> ((inf \<and> more) \<and> finite) = #False"
   unfolding finite_d_def by auto
 have 3: "\<turnstile> #False;omega inf = #False"
   by (metis AndInfChopEqvAndInf int_eq int_simps(22))
 from 1 2 3 show ?thesis 
   by (metis TrueW int_simps(3) inteq_reflection)
qed

lemma InfImpOmegaLenPlusOne:
 "\<turnstile> inf \<longrightarrow> omega (len(Suc n))"
proof -
 have 1: "\<turnstile> inf \<longrightarrow> ((len Suc n \<and> more) \<and> finite);inf " 
   by (auto simp add: Valid_def itl_defs len_defs nfinite_conv_nlength_enat zero_enat_def)
 show ?thesis 
using OmegaWeakCoinduct[of "LIFT inf" "LIFT len (Suc n)" ] 1
by (metis schop_d_def) 
qed

lemma OmegaLenPlusOneEqvInf:
 "\<turnstile> omega (len(Suc n)) = inf" 
 by (simp add: InfImpOmegaLenPlusOne int_iffI omega_imp_inf)


lemma OmegaSkipEqvInf:
 "\<turnstile> omega skip = inf"
proof -
 have 1: "\<turnstile> skip = (len 1)" 
   by (simp add: len_d_def wpower_d_def) 
     (metis  ChopEmpty inteq_reflection) 
 have 2: "\<turnstile> omega skip = omega (len 1)"
   using 1 by (metis OmegaUnroll inteq_reflection)
 from 2 show ?thesis using OmegaLenPlusOneEqvInf by fastforce
qed


lemma InfImpOmegaTrue:
 "\<turnstile> inf \<longrightarrow> omega #True"
proof -
 have 1: "\<turnstile> inf \<longrightarrow> ((#True \<and> more) \<and> finite);inf "
  by (auto simp add: Valid_def itl_defs nfinite_conv_nlength_enat zero_enat_def)
 show ?thesis 
  using OmegaWeakCoinduct[of "LIFT inf" "LIFT #True" ] 1
  by (metis schop_d_def)
qed

lemma OmegaTrueEqvInf:
 "\<turnstile> omega #True = inf"
by (simp add: InfImpOmegaTrue int_iffI omega_imp_inf)


lemma InfImpOmegaMore:
 "\<turnstile> inf \<longrightarrow> omega more"
using OmegaWeakCoinduct[of "LIFT inf" "LIFT more" ]
proof -
 have 1: " \<turnstile> inf \<longrightarrow> ((more \<and> more) \<and> finite);inf "
   by (auto simp add: Valid_def itl_defs nfinite_conv_nlength_enat zero_enat_def)
 show ?thesis using OmegaWeakCoinduct[of "LIFT inf" "LIFT more" ] 1  by (metis schop_d_def)
qed
 
lemma OmegaMoreEqvInf:
 "\<turnstile> omega more = inf"
by (simp add: InfImpOmegaMore int_iffI omega_imp_inf)

lemma InfImpOmegaFinite:
 "\<turnstile> inf \<longrightarrow> omega finite"
proof -
 have 1: "\<turnstile> inf \<longrightarrow> ((finite \<and> more) \<and> finite);inf"
 by (auto simp add: Valid_def itl_defs nfinite_conv_nlength_enat zero_enat_def)
 show ?thesis using OmegaWeakCoinduct[of "LIFT inf" "LIFT finite" ] 1  by (metis schop_d_def)
qed

lemma OmegaFiniteEqvInf:
 "\<turnstile> omega finite = inf"
by (simp add: InfImpOmegaFinite int_iffI omega_imp_inf)


lemma BoxStateAndInfImpWOmegaBoxState:
 "\<turnstile>  \<box>(init w)  \<and>  inf \<longrightarrow> womega (\<box>(init w))"
using WOmegaWeakCoinduct[of "LIFT (\<box>(init w))" "LIFT \<box>(init w)"]
by (metis BoxStateChopBoxAndInfImpBox BoxStateChopBoxEqvBox WOmega_iso inteq_reflection womega_and_inf)


lemma BoxStateAndInfImpOmegaBoxState:
 "\<turnstile>  \<box>(init w) \<and> inf \<longrightarrow> omega (\<box>(init w))"
proof -
 have 1: "\<turnstile> ((\<box> (init w) \<and> more) \<and> finite);(\<box> (init w) \<and> inf) =
             (\<box> (init w) \<and> (more \<and> finite);inf) "  
     using BoxStateAndChopEqvChop[of w "LIFT (more \<and> finite)" "LIFT inf" ]
       by (metis AndMoreAndFiniteEqvAndFmore fmore_d_def int_eq) 
 have 2: "\<turnstile> (more \<and> finite);inf  = inf"
 by (metis AndInfEqvChopFalse ChopAssoc FiniteChopMoreEqvMore FmoreEqvSkipChopFinite MoreAndInfEqvInf MoreEqvSkipChopTrue fmore_d_def infinite_d_def inteq_reflection) 
 have 3: " \<turnstile> \<box> (init w) \<and> inf \<longrightarrow> ((\<box> (init w) \<and> more) \<and> finite);(\<box> (init w) \<and> inf) "
   by (metis "1" "2" Prop11 int_eq)
 show ?thesis using OmegaWeakCoinduct[of "LIFT (\<box>(init w) \<and> inf)" "LIFT \<box>(init w)" ] 3
 by (metis schop_d_def)
qed

lemma WOmegaBoxStateImpBoxState:
 "\<turnstile> womega (\<box>(init w) ) \<longrightarrow>  \<box>(init w)  "
proof -
 have 1: "\<turnstile> womega (\<box>(init w)) \<longrightarrow> init w "
 by (meson BoxElim Prop11 StateChopExportA WOmegaUnroll WOmega_iso lift_imp_trans) 
 have 2: "\<turnstile> womega (\<box>(init w)) \<longrightarrow> (\<box>(init w) \<and> more); womega (\<box>(init w)) "
    by (simp add: WOmegaUnroll int_iffD1)
 have 21: " \<turnstile> ((\<box>(init w) \<and> more)) \<longrightarrow> \<circle>(\<box>(init w))"
    by (metis BoxImpNowAndWeakNext Prop01 Prop12 WnextEqvEmptyOrNext empty_d_def inteq_reflection 
        lift_and_com)
 have 22: "\<turnstile> womega (\<box>(init w)) \<longrightarrow> more"
 by (meson Prop11 WOmega_and_more_imp_more lift_imp_trans womega_split_empty_more)
 have 23: "\<turnstile> \<circle>(\<box>(init w)) \<longrightarrow> \<circle> ( wpowerstar (\<box> (init w))) "
 by (simp add: NextImpNext WPowerstar_ext)
 have 24: "\<turnstile>  womega (\<box>(init w)) \<longrightarrow> (\<circle>(wpowerstar (\<box> (init w) ))); (womega (\<box>(init w))) "
 by (metis "21" "23" LeftChopImpChop WOmegaUnroll int_eq lift_imp_trans)
 have 10: "\<turnstile>  womega (\<box>(init w) ) \<longrightarrow> \<circle> (womega (\<box>(init w) ))"
  using WOmega_star_1[of "LIFT (\<box>(init w))"]
  by (metis "24" NextChop inteq_reflection)
 show ?thesis using BoxIntro[of "LIFT womega (\<box>(init w) )" "LIFT (init w)"]
 by (simp add: "1" "10" Prop01 Prop05 Prop12 womega_imp_inf)   
qed

lemma OmegaBoxStateImpBoxState:
 "\<turnstile> omega (\<box>(init w)) \<longrightarrow>  \<box>(init w) "
 by (meson Prop03 WOmegaBoxStateImpBoxState WOmega_Omega lift_imp_trans)



lemma OmegaIntro:
 assumes "\<turnstile> h \<longrightarrow> (f \<and> fmore);h"
 shows   "\<turnstile> h  \<longrightarrow> omega f"
using assms
by (metis AndMoreSChopEqvAndFmoreChop OmegaWeakCoinduct inteq_reflection) 

lemma WOmegaEqvRule:
 assumes "\<turnstile> f = g"
 shows   "\<turnstile> womega f = womega g"
using assms using int_eq by force

lemma OmegaEqvRule:
 assumes "\<turnstile> f = g"
 shows   "\<turnstile> omega f = omega g"
using assms using int_eq by force

lemma AndWOmegaA:
 "\<turnstile> womega (f \<and> g) \<longrightarrow> womega f"
by (meson WOmega_iso Prop12 int_iffD2 lift_and_com)

lemma AndOmegaA:
 "\<turnstile> omega (f \<and> g) \<longrightarrow> omega f"
by (meson Omega_iso Prop12 int_iffD2 lift_and_com)

lemma AndWOmegaB:
 "\<turnstile> womega (f \<and> g) \<longrightarrow> womega g"
by (meson WOmega_iso Prop12 int_iffD2 lift_and_com)

lemma AndOmegaB:
 "\<turnstile> omega (f \<and> g) \<longrightarrow> omega g"
by (meson Omega_iso Prop12 int_iffD2 lift_and_com)


lemma BaWOmegaImpWOmega:
 "\<turnstile> ba (f \<longrightarrow> g) \<longrightarrow> womega f \<longrightarrow> womega g"
proof -
 have 1: "\<turnstile> ba (f \<longrightarrow> g) \<and> (f \<and> more);womega f \<longrightarrow>((f \<longrightarrow> g) \<and> (f \<and> more) );((f \<longrightarrow> g) \<and> womega f) "
   using BaAndChopImport[of "LIFT (f \<longrightarrow> g)" "LIFT  (f \<and> more)" "LIFT womega f" ]
     by blast
 have 2: "\<turnstile> (f \<longrightarrow> g) \<and> (f \<and> more) \<longrightarrow> ( g \<and> more)"
   by auto
 have 3: "\<turnstile> (f \<longrightarrow> g) \<and> womega f \<longrightarrow> womega f"
   by auto
 have 4: "\<turnstile> ba (f \<longrightarrow> g) \<and> (f \<and> more);womega f \<longrightarrow> (g \<and> more); womega f" 
   using 1 2 3 
   by (metis (no_types, lifting) AndChopB ChopAndB Prop10 int_eq lift_imp_trans)
 have 5: "\<turnstile> ba (f \<longrightarrow> g) \<and> (f \<and> more);womega f \<longrightarrow> ((f \<longrightarrow> g) \<and> (f \<and> more));(ba(f \<longrightarrow> g) \<and> womega f) " 
   using BaAndChopImportB by blast
 have 6: "\<turnstile> ((f \<longrightarrow> g) \<and> (f \<and> more));(ba(f \<longrightarrow> g) \<and> womega f) \<longrightarrow>
            (g \<and> more);(ba(f \<longrightarrow> g) \<and> womega f)"
            using "2" LeftChopImpChop by blast
 have 61: "\<turnstile>  womega f = (f \<and> more);womega f "
     by (simp add: WOmegaUnroll)
 have 62: "\<turnstile> (ba (f \<longrightarrow> g) \<and> womega f) \<longrightarrow> (ba (f \<longrightarrow> g) \<and> (f \<and> more);womega f) "
     by (metis "61" ChopEmpty Prop11 int_eq)
 have 7: "\<turnstile> (ba (f \<longrightarrow> g) \<and> womega f) \<longrightarrow>(g \<and> more);(ba (f \<longrightarrow> g) \<and> womega f)"
   using  62 5 6
   by (meson lift_imp_trans)
 have 8: "\<turnstile> (ba (f \<longrightarrow> g) \<and> womega f)  \<longrightarrow> womega g"
   using 7 
   using WOmegaWeakCoinduct by blast
 from 8 show ?thesis by fastforce
qed


lemma BaOmegaImpOmega:
 "\<turnstile> ba (f \<longrightarrow> g) \<longrightarrow> omega f \<longrightarrow> omega g"
proof -
 have 1: "\<turnstile> ba (f \<longrightarrow> g) \<and> (f \<and> fmore);omega f \<longrightarrow>((f \<longrightarrow> g) \<and> (f \<and> fmore) );((f \<longrightarrow> g) \<and> omega f) "
   using BaAndChopImport[of "LIFT (f \<longrightarrow> g)" "LIFT  (f \<and> fmore)" "LIFT omega f" ]
     by blast
 have 2: "\<turnstile> (f \<longrightarrow> g) \<and> (f \<and> fmore) \<longrightarrow> ( g \<and> fmore)"
   by auto
 have 3: "\<turnstile> (f \<longrightarrow> g) \<and> omega f \<longrightarrow> omega f"
   by auto
 have 4: "\<turnstile> ba (f \<longrightarrow> g) \<and> (f \<and> fmore);omega f \<longrightarrow> (g \<and> fmore); omega f" 
   using 1 2 3 
   by (metis (no_types, lifting) AndChopB ChopAndB Prop10 int_eq lift_imp_trans)
 have 5: "\<turnstile> ba (f \<longrightarrow> g) \<and> (f \<and> fmore);omega f \<longrightarrow> ((f \<longrightarrow> g) \<and> (f \<and> fmore));(ba(f \<longrightarrow> g) \<and> omega f) " 
   using BaAndChopImportB by blast
 have 6: "\<turnstile> ((f \<longrightarrow> g) \<and> (f \<and> fmore));(ba(f \<longrightarrow> g) \<and> omega f) \<longrightarrow>
            (g \<and> fmore);(ba(f \<longrightarrow> g) \<and> omega f)"
   using "2" LeftChopImpChop by blast
 have 61: "\<turnstile>  omega f = (f \<and> fmore);omega f "
 by (metis AndMoreAndFiniteEqvAndFmore OmegaUnroll inteq_reflection schop_d_def)
 have 62: "\<turnstile> (ba (f \<longrightarrow> g) \<and> omega f) \<longrightarrow> (ba (f \<longrightarrow> g) \<and> (f \<and> fmore);omega f) "
     by (metis "61" ChopEmpty Prop11 int_eq)
 have 7: "\<turnstile> (ba (f \<longrightarrow> g) \<and> omega f) \<longrightarrow>(g \<and> fmore);(ba (f \<longrightarrow> g) \<and> omega f)"
   using  62 5 6
   by (meson lift_imp_trans)
 have 8: "\<turnstile> (ba (f \<longrightarrow> g) \<and> omega f)  \<longrightarrow> omega g"
   using 7 OmegaIntro by blast
 from 8 show ?thesis by fastforce
qed


lemma BaWOmegaEqvWOmega:
 "\<turnstile> ba (f = g) \<longrightarrow> (womega f = womega g)"
proof -
 have 0: "\<turnstile> (f = g) = ( (f \<longrightarrow> g) \<and> (g \<longrightarrow> f)) "  by fastforce
 have 1: "\<turnstile> ba (f = g) = (ba (f \<longrightarrow> g) \<and> ba (g \<longrightarrow> f))  " by (metis "0" BaAndEqv int_eq)
 have 2: "\<turnstile> ba (f \<longrightarrow> g) \<longrightarrow> (womega f \<longrightarrow> womega g)" using BaWOmegaImpWOmega by blast
 have 3: "\<turnstile> ba (g \<longrightarrow> f) \<longrightarrow> (womega g \<longrightarrow> womega f)" using BaWOmegaImpWOmega by blast
 have 4: "\<turnstile> ba (f = g)  \<longrightarrow> (womega f \<longrightarrow> womega g) \<and> (womega g \<longrightarrow> womega f)" using "1" "2" "3" by fastforce
 have 5: "\<turnstile> ((womega f \<longrightarrow> womega g) \<and> (womega g \<longrightarrow> womega f)) = (womega f = womega g) " by auto
 from 4 5 show ?thesis by auto
qed

lemma BaOmegaEqvOmega:
 "\<turnstile> ba (f = g) \<longrightarrow> (omega f = omega g)"
proof -
 have 0: "\<turnstile> (f = g) = ( (f \<longrightarrow> g) \<and> (g \<longrightarrow> f)) "  by fastforce
 have 1: "\<turnstile> ba (f = g) = (ba (f \<longrightarrow> g) \<and> ba (g \<longrightarrow> f))  " by (metis "0" BaAndEqv int_eq)
 have 2: "\<turnstile> ba (f \<longrightarrow> g) \<longrightarrow> (omega f \<longrightarrow> omega g)" using BaOmegaImpOmega by blast
 have 3: "\<turnstile> ba (g \<longrightarrow> f) \<longrightarrow> (omega g \<longrightarrow> omega f)" using BaOmegaImpOmega by blast
 have 4: "\<turnstile> ba (f = g)  \<longrightarrow> (omega f \<longrightarrow> omega g) \<and> (omega g \<longrightarrow> omega f)" using "1" "2" "3" by fastforce
 have 5: "\<turnstile> ((omega f \<longrightarrow> omega g) \<and> (omega g \<longrightarrow> omega f)) = (omega f = omega g) " by auto
 from 4 5 show ?thesis by auto
qed

lemma BaAndWOmegaImport:
 "\<turnstile> ba f \<and> womega g  \<longrightarrow> womega (f \<and> g)"
proof -
 have 1: "\<turnstile> f \<longrightarrow> (g \<longrightarrow> (f \<and> g))" by auto
 hence 2: "\<turnstile> ba f \<longrightarrow> ba (g \<longrightarrow> f \<and> g)" by (rule BaImpBa)
 have 3: "\<turnstile>  ba  (g \<longrightarrow> f \<and> g) \<longrightarrow> womega g \<longrightarrow> womega (f \<and> g) " by (rule BaWOmegaImpWOmega)
 from 2 3 show ?thesis by fastforce
qed

lemma BaAndOmegaImport:
 "\<turnstile> ba f \<and> omega g  \<longrightarrow> omega (f \<and> g)"
proof -
 have 1: "\<turnstile> f \<longrightarrow> (g \<longrightarrow> (f \<and> g))" by auto
 hence 2: "\<turnstile> ba f \<longrightarrow> ba (g \<longrightarrow> f \<and> g)" by (rule BaImpBa)
 have 3: "\<turnstile>  ba  (g \<longrightarrow> f \<and> g) \<longrightarrow> omega g \<longrightarrow> omega (f \<and> g) " by (rule BaOmegaImpOmega)
 from 2 3 show ?thesis by fastforce
qed

lemma InfAndChop:
 "\<turnstile> (inf \<and> (f \<and> fmore);g) = (f \<and> fmore);(g \<and> inf) "
by (meson ChopAndInf Prop04 lift_and_com)

lemma InfImportBox:
 "\<turnstile> (inf \<and> \<box> f) = \<box> (inf \<and> f)"
by (metis BoxAndBoxEqvBoxRule FiniteChopEqvDiamond FiniteChopFiniteEqvFinite InfEqvNotFinite 
     NotDiamondNotEqvBox OrFiniteInf finite_d_def int_eq)
 
lemma InfImportBoxImp:
  "\<turnstile> (inf \<and> \<box> (f \<longrightarrow> g) ) = \<box> ( (inf \<longrightarrow> f) \<longrightarrow> (inf \<and> g))"
proof -
 have 1: "\<turnstile> (inf \<and> (f \<longrightarrow> g)) = ((inf \<longrightarrow> f) \<longrightarrow> (inf \<and> g))"
   by auto
 show ?thesis 
   by (metis "1" InfImportBox inteq_reflection)
qed

lemma OmegaImpAOmega: 
 "\<turnstile> omega f \<longrightarrow> aomega f" 
using AOmegaWeakCoInduct[of "LIFT omega f" f]
by (metis OmegaUnroll Prop10 Prop11 inteq_reflection lift_and_com omega_imp_inf schop_d_def)


lemma AOmegaImpOmega: 
"\<turnstile> aomega f \<longrightarrow> omega f "
using OmegaWeakCoinduct[of "LIFT aomega f" f]
by (simp add: AOmegaUnroll int_iffD1 schop_d_def)

lemma OmegaEqvAOmega: 
"\<turnstile> omega f = aomega f " 
using OmegaImpAOmega AOmegaImpOmega 
by (metis int_iffI)

lemma WOmega_Finite_Import: 
 "\<turnstile> \<not>((womega f) \<and> finite) "
 by (metis MP NotEmptyAndInf Prop05 Prop13 finite_d_def int_eq int_simps(32) womega_imp_inf) 


lemma WOmega_Inf_Import: 
 "\<turnstile> ((womega f) \<and> inf) = ((f;(wpowerstar f) \<and> inf) \<or> ((womega (f \<and> finite)) \<and> inf )) " 
proof -
 have 1: "\<turnstile> (womega f) = (wpowerstar (f \<and> finite);(f \<and> inf) \<or> (womega (f \<and> finite)))"
  using womega_split_finite_inf by blast
 have 2: "\<turnstile> (f;(wpowerstar f) \<and> inf) = ((wpowerstar f) \<and> finite);(f \<and> inf)" 
   by (metis WPowerstar_And_Inf_Eqv_WPowerplus_And_Inf WPowerstar_Inf_Import WPowerstar_slide_var inteq_reflection) 
 have 4: "\<turnstile> ((womega f) \<and> inf) = 
            ((wpowerstar (f \<and> finite);(f \<and> inf) \<and> inf) \<or> ((womega (f \<and> finite)) \<and> inf)) "
  using 1 by fastforce
 have 5: "\<turnstile> (wpowerstar (f \<and> finite);(f \<and> inf) \<and> inf) = wpowerstar (f \<and> finite);(f \<and> inf) "
 by (metis ChopAndInf Prop10 inteq_reflection womega_and_inf womega_imp_inf)
 show ?thesis 
 by (metis "2" "4" "5" WPowerstar_import_finite inteq_reflection)
qed 

lemma WOmega_And_Finite_Inf_Import: 
"\<turnstile>  (womega (f \<and> finite) \<and> inf) = (womega ((f \<and> finite)))"
by (meson Prop10 Prop11 womega_imp_inf) 


lemma WOmega_Inf_Import_var_1: 
 "\<turnstile> ((womega f) \<and> inf) = (wpowerstar (f \<and> finite);((f \<and> inf)) \<or> womega ((f \<and> finite) )) "
 by (meson Prop04 Prop10 womega_imp_inf womega_split_finite_inf) 


lemma WOmega_Inf_Import_var_2: 
 "\<turnstile> ((womega (f)) \<and> inf) = (womega (f)) "
 by (meson Prop10 Prop11 womega_imp_inf) 

lemma womega_4: 
 "\<turnstile> (womega ( (g \<or> (womega (f))) )) = 
    ( (womega (g )) \<or> wpowerstar g;(womega (f)) ) "  
using WOmega_or_unfold[of g "LIFT (womega (f))"]
by (metis (no_types, lifting) ChopAssoc Prop10 WOmegaChopWOmega WOmega_and_more_imp_more inteq_reflection womega_split_empty_more) 

lemma womega_5: 
 "\<turnstile> (womega ( (g \<or> h;(womega (f ))) )) = 
    ( (womega (g )) \<or> wpowerstar g;(h;(womega (f))) )" 
proof -
 have 1: "\<turnstile> (g\<^sup>\<W> \<or>
      (wpowerstar g;(h;f\<^sup>\<W> \<and> more));(g \<or> h;f\<^sup>\<W>)\<^sup>\<W>) =
     (g \<or> h;f\<^sup>\<W>)\<^sup>\<W>" 
   using WOmega_or_unfold[of g "LIFT (h;(womega (f )))" ]  by blast
 have 2: "\<turnstile> (wpowerstar g;(h;f\<^sup>\<W> \<and> more));(g \<or> h;f\<^sup>\<W>)\<^sup>\<W> = (wpowerstar g);((h;f\<^sup>\<W> \<and> more);((g \<or> h;f\<^sup>\<W>)\<^sup>\<W>))"
 by (meson ChopAssoc Prop11)
 have 3: "\<turnstile>  ((h;f\<^sup>\<W> \<and> more)) = (h;f\<^sup>\<W>) "
 by (metis MoreAndInfEqvInf Prop10 Prop12 RightChopImpMoreRule inteq_reflection womega_imp_inf) 
 have 4: "\<turnstile> (h;f\<^sup>\<W>);((g \<or> h;f\<^sup>\<W>)\<^sup>\<W>) =  (h;f\<^sup>\<W>)"
 by (metis ChopAssoc WOmegaChopWOmega inteq_reflection) 
 show ?thesis
 by (metis "1" "2" "3" "4" inteq_reflection) 
qed
 

lemma womega_6: 
 "\<turnstile> womega ((womega (f)) ) = womega (f ) "
 by (meson Prop04 WOmegaEqvRule WOmega_Inf_Import_var_2 womega_and_inf)

lemma womega_7:
"\<turnstile> (womega ((g;(womega (f))) )) = g;(womega (f )) "
proof -
 have 1: "\<turnstile> (womega ((g;(womega (f))) )) = (g;(womega (f))) ; (womega ((g;(womega (f))) ))" 
   by (meson AndChopA LeftChopImpChop Prop11 WOmegaUnroll WOmega_star_1 WPowerstar_ext lift_imp_trans)
 have 2: "\<turnstile> (g;(womega (f))) ; (womega ((g;(womega (f))) )) = (g;(womega (f)))" 
   by (metis ChopAssoc WOmegaChopWOmega inteq_reflection)
 show ?thesis using 1 2 by fastforce
qed

lemma womega_8:
"\<turnstile> (womega (f ));(wpowerstar (womega (f))) = womega (f) " 
using WOmega_star_5 by auto

lemma womega_9:
"\<turnstile> (g;(womega (f )));(wpowerstar (g;(womega (f )))) = g;womega (f ) "
by (metis inteq_reflection womega_7 womega_8)


lemma womega_10: 
 "\<turnstile> ( (g \<or> (womega (f ))));(wpowerstar ( (g \<or> (womega (f ))))) = 
    ( g;wpowerstar g \<or> wpowerstar g;(womega (f )) ) "  
proof -
 have 1: "\<turnstile> (wpowerstar ( (g \<or> (womega (f ))))) =  
            wpowerstar (wpowerstar g;wpowerstar ((f )\<^sup>\<W>))"
  using WPowerstar_denest[of g "LIFT (womega (f ))"]  by auto
 have 2: "\<turnstile> wpowerstar ((f )\<^sup>\<W>) = (empty \<or> ((f )\<^sup>\<W>)) " 
   by (simp add: WOmega_star_4)
 have 3: "\<turnstile>  (wpowerstar g;(empty \<or> ((f )\<^sup>\<W>)) ) =
             (wpowerstar g \<or> (wpowerstar g);((f )\<^sup>\<W>))"
    by (metis ChopEmpty ChopOrEqv inteq_reflection) 
 have 4: "\<turnstile>  wpowerstar (wpowerstar g \<or> (wpowerstar g);((f )\<^sup>\<W>)) =
             wpowerstar (wpowerstar g;wpowerstar ((wpowerstar g);((f )\<^sup>\<W>))) " 
  using WPowerstar_denest   by (metis WPowerstar_invol inteq_reflection)   
 have 5: "\<turnstile> wpowerstar (wpowerstar g;wpowerstar ((wpowerstar g);((f )\<^sup>\<W>))) =
            wpowerstar g;wpowerstar((wpowerstar g);((f )\<^sup>\<W>))" 
  by (metis (no_types, opaque_lifting) ChopAssoc WOmegaChopWOmega WPowerstar_denest WPowerstar_denest_var inteq_reflection) 
 have 6: "\<turnstile> wpowerstar((wpowerstar g);((f )\<^sup>\<W>)) =
            (empty \<or> wpowerstar g;(wpowerstar ((f )\<^sup>\<W>;wpowerstar g);(f )\<^sup>\<W>)) " 
   using WPowerstar_star_prod_unfold[of "LIFT (wpowerstar g)" "LIFT ((f )\<^sup>\<W>)"]
    by auto
 have 7: "\<turnstile> ( wpowerstar g;(wpowerstar ((f )\<^sup>\<W>;wpowerstar g);(f )\<^sup>\<W>)) =
            ( wpowerstar g;((empty \<or> ((f )\<^sup>\<W>));((f )\<^sup>\<W>))) "
   by (metis WOmegaChopWOmega WOmega_star_4 int_simps(1) inteq_reflection)
 have 8: "\<turnstile> ( wpowerstar g;((empty \<or> ((f )\<^sup>\<W>));((f )\<^sup>\<W>))) =
            (  wpowerstar g;((f )\<^sup>\<W>) ) "
     by (metis "2" RightChopEqvChop WOmegaChopWOmega WPowerstar_induct_lvar_eq2 inteq_reflection)
 have 9: "\<turnstile>  wpowerstar((wpowerstar g);((f )\<^sup>\<W>)) = (empty \<or> (  wpowerstar g;((f )\<^sup>\<W>) ))"
    by (meson Prop06 WPowerstarEqv womega_9) 
 have 10: "\<turnstile> (wpowerstar ( (g \<or> (womega (f ))))) = 
             wpowerstar g;(empty \<or> (  wpowerstar g;((f )\<^sup>\<W>) ))"
  by (metis "1" "2" "3" "4" "5" "6" "7" "8" inteq_reflection)
 have 11: "\<turnstile>  wpowerstar g;(empty \<or> (  wpowerstar g;((f )\<^sup>\<W>) )) =  
             (wpowerstar g \<or> (wpowerstar g;((f )\<^sup>\<W>) )) "
   by (metis "10" "2" "3" WOmegaChopWOmega WPowerstar_denest_var inteq_reflection)
 have 12: "\<turnstile> ( (g \<or> (womega (f )))); (wpowerstar g \<or> (wpowerstar g;((f )\<^sup>\<W>) )) =
             ( g;wpowerstar g \<or> g;(wpowerstar g;((f )\<^sup>\<W>) ) \<or> (womega (f )))  "
    by (meson ChopOrEqv OrChopOr Prop04 Prop06 WOmegaChopWOmega int_simps(20))
 have 13: "\<turnstile> (g;(wpowerstar g;((f )\<^sup>\<W>) ) \<or> (womega (f ))) =
             (wpowerstar g;((f )\<^sup>\<W>) ) " 
     by (metis LeftChopEqvChop Prop04 WPowerstarChopEqvChopOrRule WPowerstar_tc WPowerstar_trans)
 have 14: "\<turnstile> ( g;wpowerstar g \<or> g;(wpowerstar g;((f )\<^sup>\<W>) ) \<or> (womega (f))) =
             ( g;wpowerstar g \<or> (wpowerstar g;((f )\<^sup>\<W>) )) "
  using "13" by auto
 show ?thesis by (metis "10" "11" "12" "14" inteq_reflection)
qed

lemma womega_11: 
 "\<turnstile> ( (g \<or> h;(womega (f ))));(wpowerstar ( (g \<or> h;(womega (f ))))) = 
    ( g;wpowerstar g \<or> wpowerstar g;(h;(womega (f ))) ) "  
proof -
 have 1: "\<turnstile> (wpowerstar ( (g \<or> h;(womega (f ))))) =  
            wpowerstar (wpowerstar g;wpowerstar (h;(f )\<^sup>\<W>))"
  using WPowerstar_denest[of g "LIFT h;(womega (f ))"]  by auto
 have 2: "\<turnstile> wpowerstar (h;(f )\<^sup>\<W>) = (empty \<or> h;((f )\<^sup>\<W>)) " 
  by (meson Prop06 WPowerstarEqv womega_9)
 have 3: "\<turnstile>  (wpowerstar g;(empty \<or> h;((f )\<^sup>\<W>)) ) =
             (wpowerstar g \<or> (wpowerstar g);(h;(f )\<^sup>\<W>))"
    by (metis ChopEmpty ChopOrEqv inteq_reflection) 
 have 4: "\<turnstile>  wpowerstar (wpowerstar g \<or> (wpowerstar g);(h;(f )\<^sup>\<W>)) =
             wpowerstar (wpowerstar g;wpowerstar ((wpowerstar g);(h;(f )\<^sup>\<W>))) " 
  using WPowerstar_denest   by (metis WPowerstar_invol inteq_reflection)   
 have 5: "\<turnstile> wpowerstar (wpowerstar g;wpowerstar ((wpowerstar g);(h;(f )\<^sup>\<W>))) =
            wpowerstar g;wpowerstar((wpowerstar g);(h;(f )\<^sup>\<W>))" 
  by (metis (no_types, opaque_lifting) ChopAssoc WOmegaChopWOmega WPowerstar_denest WPowerstar_denest_var inteq_reflection) 
 have 6: "\<turnstile> wpowerstar((wpowerstar g);(h;(f )\<^sup>\<W>)) =
            (empty \<or> wpowerstar g;(wpowerstar ((h;(f )\<^sup>\<W>);wpowerstar g);(h;(f )\<^sup>\<W>))) " 
   using WPowerstar_star_prod_unfold[of "LIFT (wpowerstar g)" "LIFT (h;(f )\<^sup>\<W>)"]
    by auto
 have 7: "\<turnstile> ( wpowerstar g;(wpowerstar ((h;(f )\<^sup>\<W>);wpowerstar g);(h;(f )\<^sup>\<W>))) =
            ( wpowerstar g;((empty \<or> (h;(f )\<^sup>\<W>));(h;(f )\<^sup>\<W>))) "
    by (metis "2" ChopAssoc WOmegaChopWOmega inteq_reflection)
 have 8: "\<turnstile> ( wpowerstar g;((empty \<or> (h;(f )\<^sup>\<W>));(h;(f )\<^sup>\<W>))) =
            (  wpowerstar g;(h;(f )\<^sup>\<W>) ) "
    by (metis "2" RightChopEqvChop WPowerstar_slide_var inteq_reflection womega_9)
 have 9: "\<turnstile>  wpowerstar((wpowerstar g);(h;(f )\<^sup>\<W>)) = 
             (empty \<or> (  wpowerstar g;(h;(f )\<^sup>\<W>) ))"
   by (meson "6" "7" "8" Prop06) 
 have 10: "\<turnstile> (wpowerstar ( (g \<or> h;(womega (f ))))) = 
             wpowerstar g;(empty \<or> (  wpowerstar g;(h;(f )\<^sup>\<W>) ))"
  by (metis "1" "2" "3" "4" "5" "6" "7" "8" inteq_reflection)
 have 11: "\<turnstile>  wpowerstar g;(empty \<or> (  wpowerstar g;(h;(f )\<^sup>\<W>) )) =  
             (wpowerstar g \<or> (wpowerstar g;(h;(f )\<^sup>\<W>) )) "
  by (metis (no_types, lifting) "10" "3" WOmegaChopWOmega WPowerstarEqv WPowerstar_denest_var inteq_reflection womega_7)
 have 12: "\<turnstile> ( (g \<or> h;(womega (f )))); (wpowerstar g \<or> (wpowerstar g;(h;(f )\<^sup>\<W>) )) =
             ( g;wpowerstar g \<or> g;(wpowerstar g;(h;(f )\<^sup>\<W>) ) \<or> h;(womega (f )))  "
      by (metis (no_types, lifting) "10" "11" WPowerstarChopEqvChopOrRule inteq_reflection womega_10 womega_7)
 have 13: "\<turnstile> (g;(wpowerstar g;(h;(f )\<^sup>\<W>) ) \<or> h;(womega (f ))) =
             (wpowerstar g;(h;(f )\<^sup>\<W>) ) " 
     by (metis LeftChopEqvChop Prop04 WPowerstarChopEqvChopOrRule WPowerstar_tc WPowerstar_trans)
 have 14: "\<turnstile> ( g;wpowerstar g \<or> g;(wpowerstar g;(h;(f )\<^sup>\<W>) ) \<or> h;(womega (f ))) =
             ( g;wpowerstar g \<or> (wpowerstar g;(h;(f )\<^sup>\<W>) )) "
  using "13" by auto
 show ?thesis by (metis "10" "11" "12" "14" inteq_reflection)
qed

lemma WPowerstar_imp_finite_or_finite_chop_and_inf: 
 "\<turnstile> wpowerstar f \<longrightarrow> finite \<or> finite;(f \<and> inf) " 
proof -
 have 1: "\<turnstile> wpowerstar f \<longrightarrow> finite;(empty \<or> (f \<and> inf))"
    by (meson LeftChopImpChop Prop11 SChopstar_finite WPowerstar_SChopstar lift_imp_trans)
 have 2: "\<turnstile> finite;(empty \<or> (f \<and> inf)) = (finite \<or> finite;(f \<and> inf))"
   by (metis ChopEmpty ChopOrEqv int_eq)
 show ?thesis using 1 2 by fastforce
qed

lemma Inf_and_always_imp_not_wpowerstar: 
 "\<turnstile> inf \<and> \<box> (f \<longrightarrow> finite) \<longrightarrow> \<not>(wpowerstar f) " 
proof -
 have 1: "\<turnstile> (\<not> (f \<longrightarrow> \<not> inf)) = (f \<and> inf)" 
    by fastforce
 have 2: "\<turnstile> (\<not> (\<not> inf);(\<not> (f \<longrightarrow> \<not> inf)) ) = (\<not>((\<not> inf);(f \<and> inf)))"
    by (metis "1" int_simps(15) inteq_reflection)
 show ?thesis
using WPowerstar_imp_finite_or_finite_chop_and_inf unfolding always_d_def sometimes_d_def finite_d_def
using 2 by fastforce
qed

lemma WOmega_And_empty: 
 "\<turnstile> \<not>( womega f \<and> empty)  " 
proof - 
 have 1: "\<turnstile> ( womega f \<and> empty) =
            ( ( (schopstar f);( (f \<and> inf)) \<or> (omega f) ) \<and> empty) " 
  using WOmega_Omega by (metis int_simps(1) inteq_reflection)
 have 2: "\<turnstile> ( ( (schopstar f);(  (f \<and> inf)) \<or> (omega f) ) \<and> empty) =
            ( ((schopstar f);(  (f \<and> inf)) \<and> empty) \<or>
              (omega f \<and> empty)) " 
    by fastforce
 have 3: "\<turnstile> \<not>(omega f \<and> empty) " 
    using NotEmptyAndInf omega_imp_inf by fastforce
 have 5: "\<turnstile> \<not>((schopstar f);(f \<and> inf) \<and> empty)" 
   by (metis (no_types, opaque_lifting) AndInfChopEqvAndInf ChopAndEmptyEqvEmptyChopEmpty ChopAndInf 
       InitAndEmptyEqvAndEmpty NotEmptyAndInf StateAndEmptyChop inteq_reflection lift_and_com)
 show ?thesis using 1 2 3  5  by fastforce
qed

lemma WOmega_And_More: 
 "\<turnstile> ( womega f \<and> more) =  ( womega f )"
 by (metis FiniteChopSkipEqvFiniteAndMore FiniteChopSkipEqvSkipChopFinite FmoreEqvSkipChopFinite 
     Prop10 WOmega_and_more_imp_more inteq_reflection lift_imp_trans womega_fmore_inf_eq womega_imp_inf) 


end