(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

(*  based on       Terminated coinductive list
    by      Andreas Lochbihler
*)

section \<open>Non-empty coinductive lists\<close>

text \<open>
Coinductive lists are formalised by Andreas  Lochbihler in \cite{Coinductive-AFP}. 
We define coinductive non-empty lists \<open>'a nellist\<close> as a subtype of coinductive lists using the 
quotient type construction. 
 The usual operators, like take, drop, length, nth, filter etc. are defined
for \<open>'a nellist\<close>.  The formalisation is based on terminated coinductive list defined 
by Andreas Lochbihler.  
\<close>

theory NELList imports
  LList_Extras
begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="cvc4 z3 e spass vampire " ]
*)

text \<open>
  Coinductive non-empty lists \<open>'a nellist\<close> are the codatatype defined by the constructors
  \<open>NNil\<close> of type \<open>'a \<Rightarrow> 'a nellist\<close> and
  \<open>NCons\<close> of type \<open>'a \<Rightarrow> 'a nellist \<Rightarrow> 'a nellist\<close>.
\<close>


subsection \<open>Type definition\<close>

consts nlast0 :: "'a"

codatatype (nset: 'a) nellist =
    NNil (nlast : 'a)
  | NCons (nhd : 'a) (ntl : "'a nellist")
for
  map: nmap
  rel: nellist_all2
where
  "nhd (NNil _) = undefined"
| "ntl (NNil b) = NNil b"
| "nlast (NCons _ nell) = nlast0 nell"


overloading
  nlast0 == "nlast0::'a nellist \<Rightarrow> 'a"
begin

partial_function (tailrec) nlast0 
where "nlast0 nell = (case nell of (NNil x) \<Rightarrow> x | (NCons y nell') \<Rightarrow> nlast0 nell')"

end

lemma nlast0_nlast [simp]: "nlast0 = nlast"
proof -
 have 1: "\<And>x. nlast0 x = nlast x"
 by (simp add: nlast0.simps nlast_def) 
 show ?thesis  using 1 by (rule ext)
qed

lemmas nlast_NNil [code, nitpick_simp] = nellist.sel(1)

lemma nlast_NCons [simp, code, nitpick_simp]: "nlast (NCons x nell) = nlast nell"
by simp

declare nellist.sel(2) [simp del]

definition nfirst :: "'a nellist \<Rightarrow> 'a"
 where " nfirst nell = (case nell of (NNil b) \<Rightarrow> b | (NCons b nell') \<Rightarrow> b)"

primcorec unfold_nellist :: "('a \<Rightarrow> bool) \<Rightarrow> ('a \<Rightarrow> 'b) \<Rightarrow> ('a \<Rightarrow> 'b) \<Rightarrow> ('a \<Rightarrow> 'a) \<Rightarrow> 'a \<Rightarrow> 'b nellist" 
 where
  "p a \<Longrightarrow> unfold_nellist p g1 g21 g22 a = NNil (g1 a)" |
  "_ \<Longrightarrow> unfold_nellist p g1 g21 g22 a =
     NCons (g21 a) (unfold_nellist p g1 g21 g22 (g22 a))"

declare
  unfold_nellist.ctr(1) [simp]
  nellist.corec(1) [simp]

subsection \<open>Code generator setup\<close>

(*
text \<open>Test quickcheck setup\<close>

lemma "nell = NNil x"
quickcheck[random, expect=counterexample]
quickcheck[exhaustive, expect=counterexample]
oops

lemma "NCons x nell = NCons x nell"
quickcheck[narrowing, expect=no_counterexample]
oops
*)

text \<open>More lemmas about generated constants\<close>

lemma ntl_unfold_nellist:
  "ntl (unfold_nellist IS_NNIL NNIL NHD NTL a) = 
  (if IS_NNIL a then NNil (NNIL a) else unfold_nellist IS_NNIL NNIL NHD NTL (NTL a))"
by(simp)

lemma is_NNil_ntl [simp]: 
  "is_NNil nell \<Longrightarrow> is_NNil (ntl nell)"
by(cases nell) simp_all

lemma nlast_ntl [simp]: "nlast (ntl nell) = nlast nell"
by(cases nell) simp_all

lemma unfold_nellist_eq_NNil [simp]:
  "unfold_nellist IS_NNIL NNIL NHD NTL a = NNil b \<longleftrightarrow> IS_NNIL a \<and> b = NNIL a"
by(auto simp add: unfold_nellist.code)

lemma NNil_eq_unfold_nellist [simp]:
  "NNil b = unfold_nellist IS_NNIL NNIL NHD NTL a \<longleftrightarrow> IS_NNIL a \<and> b = NNIL a"
by(auto simp add: unfold_nellist.code)

lemma nmap_is_NNil: 
  "is_NNil nell \<Longrightarrow> nmap f  nell = NNil (f (nlast nell))"
by(clarsimp simp add: is_NNil_def)

declare nellist.map_sel(2)[simp]

lemma ntl_nmap [simp]:  
  "ntl (nmap f nell) = nmap f (ntl nell)"
by(cases nell) simp_all

lemma nmap_eq_NNil_conv:
  "nmap f nell = NNil y \<longleftrightarrow> (\<exists>y'. nell = NNil y' \<and> f y' = y)"
by(cases nell) simp_all

lemma NNil_eq_nmap_conv:
  "NNil y = nmap f nell \<longleftrightarrow> (\<exists>y'. nell = NNil y' \<and> f y' = y)"
by(cases nell) auto

declare nellist.set_sel(1)[simp]

lemma nset_ntl: "nset (ntl nell) \<subseteq> nset nell"
by(cases nell) auto

lemma in_nset_ntlD: "x \<in> nset (ntl nell) \<Longrightarrow> x \<in> nset nell"
using nset_ntl[of nell] by auto

theorem nellist_set_induct[consumes 1, case_names findnil find step]:
  assumes "x \<in> nset nell" 
  and "\<And>nell. is_NNil nell \<Longrightarrow> P (nlast nell) nell"  
  and "\<And>nell. \<not> is_NNil nell \<Longrightarrow> P (nhd nell) nell"
  and "\<And>nell y. \<lbrakk>\<not> is_NNil nell; y \<in> nset (ntl nell); P y (ntl nell)\<rbrakk> \<Longrightarrow> P y nell"
  shows "P x nell"
using assms 
proof (induct)
case (NNil z)
then show ?case by force
next
case (NCons1 z1 z2)
then show ?case by (fastforce simp del: nellist.disc(2) iff: nellist.disc(2))
next
case (NCons2 z1 z2 xa)
then show ?case by auto
qed

lemma nset_induct' [consumes 1, case_names findnil find step]: 
 assumes major: "x \<in> nset nell" 
 and 0: "\<And>nell. is_NNil nell \<Longrightarrow> P (nlast nell)" 
 and 1: "\<And>nell. P (NCons x nell)" 
 and 2: "\<And> x' nell. \<lbrakk> x \<in> nset nell; P nell \<rbrakk> \<Longrightarrow> P (NCons x' nell)"
shows " P nell"
using major 
proof (induct y\<equiv>"x" nell rule:  nellist_set_induct)
case (findnil nell)
then show ?case using 0 1 2 by (metis nellist.collapse(1) nellist.map_disc_iff nellist.map_sel(1))
next
case (find nell)
then show ?case by (metis "1" nellist.collapse(2))
next
case (step nell)
then show ?case by (metis "2" nellist.collapse(2))
qed

lemma nset_induct [consumes 1, case_names findnil find step, induct set: nset]: 
 assumes major: "x \<in> nset nell" 
 and findnil: "\<And>nell. is_NNil nell \<Longrightarrow> P (nlast nell)" 
 and find: "\<And>nell. P (NCons x nell)" 
 and step: "\<And> x' nell. \<lbrakk> x \<in> nset nell; x \<noteq> x'; P nell \<rbrakk> \<Longrightarrow> P (NCons x' nell)"
 shows " P nell"
using major
proof (induct rule: nset_induct')
case (findnil nell)
then show ?case by (simp add: assms(2))
next
case (find nell)
then show ?case by (simp add: assms(3))
next
case (step x' nell)
then show ?case by (metis assms(4) find)
qed



subsection \<open>Connection with @{typ "'a llist"}\<close>

primcorec llist_of_nellist :: "'a nellist \<Rightarrow> 'a llist"
where "llist_of_nellist nell = (case nell of NNil b \<Rightarrow> LCons b LNil |
                                      NCons x nell' \<Rightarrow> LCons x (llist_of_nellist nell'))"

context 
fixes 
b :: 'a 
begin
primcorec nellist_of_llist_a :: "'a llist \<Rightarrow> 'a nellist" where
  "nellist_of_llist_a ll = (case ll of LNil \<Rightarrow> NNil b | 
                                LCons x ll' \<Rightarrow>  NCons x (nellist_of_llist_a ll'))"
end

abbreviation "nellist_of_llist == (\<lambda> ll. nellist_of_llist_a (llast ll) (lbutlast ll))"

simps_of_case nellist_of_llist_a_simps [simp, code, nitpick_simp]: nellist_of_llist_a.code

lemmas nellist_of_llist_a_LNil = nellist_of_llist_a_simps(1)
  and nellist_of_llist_a_LCons = nellist_of_llist_a_simps(2)

lemma nlast_nellist_of_llist_a_lnull [simp]:
  "lnull ll \<Longrightarrow> nlast (nellist_of_llist_a b ll) = b"
unfolding lnull_def by simp


declare nellist_of_llist_a.sel(1)[simp del]

lemma lhd_LNil: 
  "lhd LNil = undefined"
by(simp add: lhd_def)

lemma nhd_NNil: 
  "nhd (NNil b) = undefined"
by(simp add: nhd_def)

lemma nhd_nellist_of_llist_a [simp]: 
   "nhd (nellist_of_llist_a b ll) = lhd ll"
by (cases ll)
   (simp_all add: lhd_LNil nhd_NNil)

lemma ntl_nellist_of_llist_a [simp]: 
  "ntl (nellist_of_llist_a b ll) = nellist_of_llist_a b (ltl ll)"
by(cases ll) simp_all

lemma llist_of_nellist_eq_LNil:
  "llist_of_nellist nell = LCons (nlast nell) LNil \<longleftrightarrow> is_NNil nell"
by (simp add: nellist.case_eq_if llist_of_nellist.code)

simps_of_case llist_of_nellist_simps [simp, code, nitpick_simp]: llist_of_nellist.code

lemmas llist_of_nellist_NNil  = llist_of_nellist_simps(1)
  and  llist_of_nellist_NCons = llist_of_nellist_simps(2)

declare llist_of_nellist.sel [simp del]

lemma lhd_llist_of_nellist [simp]: 
  "\<not> is_NNil nell \<Longrightarrow> lhd (llist_of_nellist nell) = nhd nell"
by(cases nell) simp_all

lemma lhd_llist_of_nellist1 [simp]: 
  " is_NNil nell \<Longrightarrow> lhd (llist_of_nellist nell) = nlast nell"
by (cases nell) simp_all

lemma lhd_llist_of_nellist2 [simp]: 
  " (case nell of (NNil b) \<Rightarrow> lhd LNil | (NCons b nell') \<Rightarrow> lhd (llist_of_nellist nell)) = nhd nell"
by (cases nell) (simp_all add: lhd_LNil nhd_NNil)

lemma ltl_llist_of_nellist [simp]:
  "\<not> is_NNil nell \<Longrightarrow> ltl (llist_of_nellist nell) = llist_of_nellist (ntl nell)"
by(cases nell) simp_all

lemma ltl_llist_of_nellist1 [simp]:
  " is_NNil nell \<Longrightarrow> ltl (llist_of_nellist nell) = LNil"
by(cases nell) simp_all

lemma ltl_llist_of_nellist2 [simp]:
  "(case nell of (NNil b) \<Rightarrow> (LCons b LNil) | 
                 (NCons b nell') \<Rightarrow> ltl (llist_of_nellist nell)) = llist_of_nellist (ntl nell) "
by (simp add: llist_of_nellist.code nellist.case_eq_if)

lemma nellist_of_llist_a_cong [cong]:
  assumes "ll = ll'" "lfinite ll' \<Longrightarrow> b = b'"
  shows "nellist_of_llist_a b ll = nellist_of_llist_a b' ll'"
proof(unfold \<open>ll = ll'\<close>)
  from assms have "lfinite ll' \<longrightarrow> b = b'" by simp
  thus "nellist_of_llist_a b ll' = nellist_of_llist_a b' ll'"
  by(coinduction arbitrary: ll') auto
qed


primcorec snocn :: "'a nellist \<Rightarrow> 'a \<Rightarrow> 'a nellist"
 where " snocn nell a =
         (case nell of (NNil x) \<Rightarrow> NCons x (NNil a) |
                 (NCons x nell') \<Rightarrow> NCons x (snocn nell' a))"

simps_of_case snocn_code [code, simp, nitpick_simp]: snocn.code

lemma snocn_simps [simp]:
 shows nhd_snocn: "nhd(snocn nell a) = nfirst nell"
 and   ntl_snocn: "ntl(snocn nell a) = (if is_NNil nell then (NNil a) else snocn (ntl nell) a)"
by (case_tac [!] nell)
   (auto simp add: nfirst_def)

lemma is_NNil_snocn:
 "is_NNil(snocn nell a) \<longleftrightarrow> False"
by (auto simp add: snocn_def)

lemma nmap_snocn_distrib:
 " nmap f (snocn nell a) = snocn (nmap f nell) (f a)"
proof (coinduction arbitrary: nell rule: nellist.coinduct_strong)
case (Eq_nellist nella)
then show ?case 
by (auto simp add: nellist.case_eq_if nellist.map_sel(1))
qed

definition nfinite :: "'a nellist \<Rightarrow> bool"
where "nfinite nell \<equiv> lfinite (llist_of_nellist nell)"

lemma nfinite_induct [consumes 1, case_names NNil NCons]:
  assumes "nfinite nell"
  and "\<And>y. P (NNil y)"
  and "\<And>x nell. \<lbrakk>nfinite nell; P nell\<rbrakk> \<Longrightarrow> P (NCons x nell)"
  shows "P nell"
using assms
unfolding nfinite_def
proof (induct ll\<equiv>"llist_of_nellist nell" arbitrary: nell rule: lfinite_induct)
case LNil
then show ?case by simp
next
case LCons
then show ?case by (metis lfinite.cases lnull_def ltl_llist_of_nellist ltl_simps(2) 
                    nellist.collapse(1) nellist.exhaust_sel)
qed

lemma 
 shows nfinite_NNil: " nfinite (NNil x)"
 and nfinite_NConsI: " nfinite nell \<Longrightarrow> nfinite (NCons x nell)"
unfolding nfinite_def 
by auto

declare nfinite_NNil [iff] 

lemma is_NNil_imp_nfinite [simp]:
 " is_NNil nell \<Longrightarrow> nfinite nell"
using lfinite.simps llist_of_nellist_eq_LNil by (auto simp add: nfinite_def )



lemma nfinite_NCons [simp]:
 " nfinite (NCons x nell) = nfinite nell"
by (simp add: nfinite_def)

lemma nfinite_ntl [simp]:
 " nfinite (ntl nell) = nfinite nell"
by (cases nell) simp_all

lemma nfinite_code [code]:
 " nfinite (NNil x) = True"
 " nfinite (NCons x nell) = nfinite nell"
by simp_all

lemma nfinite_imp_finite_nset:
 assumes "nfinite nell"
 shows   "finite (nset nell)"
using assms 
by (induct nell rule:nfinite_induct) simp_all

lemma nfinite_snocn [simp]:
 "nfinite(snocn nell a) \<longleftrightarrow> nfinite nell"
(is "?lhs \<longleftrightarrow> ?rhs") 
proof
 assume ?lhs thus ?rhs
 proof (induct zs\<equiv>"snocn nell a" arbitrary: nell rule: nfinite_induct)
 case (NNil y)
 then show ?case 
 by (metis is_NNil_snocn nellist.disc(1))
 next
 case (NCons x nell)
 then show ?case 
   by (cases nell) simp_all
 qed
next
 assume ?rhs thus ?lhs
  by (induct rule: nfinite_induct) auto
qed

lemma snocn_inf:
 "\<not> nfinite nell \<Longrightarrow> snocn nell a = nell"
proof (coinduction arbitrary: nell)
case (Eq_nellist nella)
then show ?case 
   proof -
     have 1: " is_NNil (snocn nella a) = is_NNil nella" 
       using Eq_nellist by auto
     have 2: "(is_NNil (snocn nella a) \<longrightarrow> is_NNil nella \<longrightarrow> nlast (snocn nella a) = nlast nella)" 
       by auto
     have 3: "(\<not> is_NNil (snocn nella a) \<longrightarrow> \<not> is_NNil nella \<longrightarrow> 
               nhd (snocn nella a) = nhd nella \<and> 
               (\<exists>nell. ntl (snocn nella a) = snocn nell a \<and> ntl nella = nell \<and> \<not> nfinite nell)) "
       by (simp add: Eq_nellist nellist.case_eq_if)
     from 1 2 3 show ?thesis by blast
   qed
qed

lemma nfinite_nmap [simp]:
 "nfinite (nmap f nell) = nfinite nell" (is "?lhs \<longleftrightarrow> ?rhs")
proof
 assume ?lhs thus ?rhs
  proof (induct zs\<equiv>"nmap f nell" arbitrary: nell rule: nfinite_induct)
  case (NNil y)
  then show ?case by (metis nellist.disc(1) nellist.map_disc_iff is_NNil_imp_nfinite)
  next
  case (NCons x nell)
  then show ?case by (metis nellist.sel(5) nfinite_ntl ntl_nmap)
  qed
next
 assume ?rhs thus ?lhs
   by (induct rule: nfinite_induct) simp_all
qed

lemma nset_snocn_nfinite [simp]: 
 " nfinite nell \<Longrightarrow> nset(snocn nell a) = nset nell \<union> {a}"
by (induct rule: nfinite_induct) auto

lemma nset_snocn1:
 "nset (snocn nell a) \<subseteq> nset nell \<union> {a} "
proof (cases "nfinite nell")
case True
then show ?thesis by simp
next
case False
then show ?thesis by (auto simp add: snocn_inf)
qed

lemma nset_snocn_conv:
 "nset (snocn nell a) = (if nfinite nell then nset nell \<union> {a} else nset nell)"
by (simp add: snocn_inf)

lemma in_nset_snocn_iff:
 " x \<in> nset (snocn nell a) \<longleftrightarrow> x \<in> nset nell \<or> nfinite nell \<and> x = a"
by (metis Un_iff empty_iff insert_iff nset_snocn_conv)

lemma llist_of_nellist_inverse_1:
 assumes "\<not> nfinite nell"
 shows " nellist_of_llist_a b (llist_of_nellist nell) = snocn nell b"
using assms
proof (coinduction arbitrary: nell)
case (Eq_nellist nella)
then show ?case 
    proof -
     have 1: "is_NNil (nellist_of_llist_a b (llist_of_nellist nella)) = is_NNil (snocn nella b)"
       by auto
     have 2: "(is_NNil (nellist_of_llist_a b (llist_of_nellist nella)) \<longrightarrow> 
               is_NNil (snocn nella b)  \<longrightarrow> 
               nlast (nellist_of_llist_a b (llist_of_nellist nella)) = nlast (snocn nella b))"  
       by simp
     have 3: "(\<not> is_NNil (nellist_of_llist_a b (llist_of_nellist nella)) \<longrightarrow> 
               \<not> is_NNil (snocn nella b) \<longrightarrow>
               nhd (nellist_of_llist_a b (llist_of_nellist nella)) = nhd (snocn nella b) \<and>
              (\<exists>nell.
                 ntl (nellist_of_llist_a b (llist_of_nellist nella)) = 
                   nellist_of_llist_a b (llist_of_nellist nell) \<and>
                 ntl (snocn nella b) = snocn nell b \<and> \<not> nfinite nell)) "
       by (metis Eq_nellist lhd_llist_of_nellist ltl_llist_of_nellist nfinite_ntl 
            nhd_nellist_of_llist_a ntl_nellist_of_llist_a snocn_inf)
     from 1 2 3 show ?thesis by blast
  qed   
qed

lemma llist_of_nellist_inverse_2:
 assumes "nfinite nell"
 shows  " nellist_of_llist_a b (llist_of_nellist nell) = snocn nell b"
using assms
by (induct rule: nfinite_induct) simp_all

lemma llist_of_nellist_inverse [simp]:
 shows  " nellist_of_llist_a b (llist_of_nellist nell) = snocn nell b"
using llist_of_nellist_inverse_1 llist_of_nellist_inverse_2 by fastforce

lemma llist_of_nellist_inverse_3: 
assumes "\<not> nfinite nell"
shows  "nellist_of_llist_a (nlast nell) (lbutlast (llist_of_nellist nell)) = nell   "
using assms
proof (coinduction arbitrary: nell)
case (Eq_nellist nella)
then show ?case 
    proof -
     have 1: "is_NNil (nellist_of_llist_a (nlast nella) (lbutlast (llist_of_nellist nella))) = 
             is_NNil nella"
       by (metis Eq_nellist is_NNil_imp_nfinite lbutlast.disc(2) llist_of_nellist.disc_iff
           ltl_llist_of_nellist nellist_of_llist_a.disc(2))
     have 2: "(is_NNil (nellist_of_llist_a (nlast nella) (lbutlast (llist_of_nellist nella))) \<longrightarrow>
               is_NNil nella \<longrightarrow> 
               nlast (nellist_of_llist_a (nlast nella) 
                     (lbutlast (llist_of_nellist nella))) = nlast nella)" 
       using Eq_nellist is_NNil_imp_nfinite by blast 
     have 3: "(\<not> is_NNil (nellist_of_llist_a (nlast nella) (lbutlast (llist_of_nellist nella))) \<longrightarrow>
               \<not> is_NNil nella \<longrightarrow>
                  nhd (nellist_of_llist_a (nlast nella) (lbutlast (llist_of_nellist nella))) =
                  nhd nella \<and>
                  (\<exists>nell.
                    ntl (nellist_of_llist_a (nlast nella) (lbutlast (llist_of_nellist nella))) =
                    nellist_of_llist_a (nlast nell) (lbutlast (llist_of_nellist nell)) \<and>
                    ntl nella = nell \<and> \<not> nfinite nell)) "
       by (auto simp add: llist.case_eq_if Eq_nellist)
     from 1 2 3 show ?thesis by blast
   qed
qed

lemma llist_of_nellist_inverse_4: 
assumes "nfinite nell"
shows  "nellist_of_llist_a (nlast nell) (lbutlast (llist_of_nellist nell)) = nell   "
using assms
by (induct rule: nfinite_induct) (simp_all  add: llist_of_nellist.code)

lemma llist_of_nellist_inverse_a [simp]: 
shows  "nellist_of_llist_a (nlast nell) (lbutlast (llist_of_nellist nell)) = nell   "
using llist_of_nellist_inverse_3 llist_of_nellist_inverse_4 by fastforce

lemma nlast_llast:
 assumes "nfinite nell"
 shows   "nlast nell = llast(llist_of_nellist nell)"
using assms 
by (induct rule: nfinite_induct)
   (simp_all add: llist_of_nellist.code)

lemma llist_of_nellist_inverse_b [simp]: 
shows  "nellist_of_llist (llist_of_nellist nell) = nell   "
by (metis lappend_inf lbutlast_snoc llist_of_nellist_inverse llist_of_nellist_inverse_a 
    nfinite_def snocn_inf nlast_llast)

lemma nellist_of_llist_a_eq [simp]: 
  "nellist_of_llist_a b' ll = NNil b \<longleftrightarrow> b = b' \<and> ll = LNil"
by(cases ll) auto

lemma NNil_eq_nellist_of_llist_a [simp]: 
  "NNil b = nellist_of_llist_a b' ll \<longleftrightarrow> b = b' \<and> ll = LNil"
by(cases ll) auto

lemma nellist_of_llist_a_inject [simp]:
  "nellist_of_llist_a b llx = nellist_of_llist_a c lly \<longleftrightarrow> llx = lly \<and> (lfinite lly \<longrightarrow> b = c)"
  (is "?lhs \<longleftrightarrow> ?rhs")
proof(intro iffI conjI impI)
  assume ?rhs
  thus ?lhs by(auto intro: nellist_of_llist_a_cong)
next
  assume ?lhs
  thus "llx = lly"
    by(coinduction arbitrary: llx lly)(auto simp add: lnull_def neq_LNil_conv)
  assume "lfinite lly"
  thus "b = c" using \<open>?lhs\<close>
    unfolding \<open>llx = lly\<close> by(induct) simp_all
qed  

lemma nellist_of_llist_a_inverse_1:
assumes "\<not> lfinite ll" 
shows  " llist_of_nellist (nellist_of_llist_a b ll) = lappend ll (LCons b LNil)"
using assms 
by (coinduction arbitrary: ll) auto

lemma nellist_of_llist_a_inverse_2:
assumes " lfinite ll" 
shows  " llist_of_nellist (nellist_of_llist_a b ll) = lappend ll (LCons b LNil)"
using assms 
proof (induct ll rule: lfinite_induct)
 case (LNil xs)
 then show ?case by (simp add: lnull_def)
 next
 case (LCons xs)
 then show ?case 
   by (metis nellist_of_llist_a.disc(2) lappend_code(2) lhd_LCons_ltl lhd_llist_of_nellist 
       llist_of_nellist.code llist_of_nellist.simps(2) llist_of_nellist.simps(3) 
       ltl_llist_of_nellist nhd_nellist_of_llist_a ntl_nellist_of_llist_a)
qed

lemma nellist_of_llist_a_inverse [simp]:
shows  " llist_of_nellist (nellist_of_llist_a b ll) = lappend ll (LCons b LNil)"
using nellist_of_llist_a_inverse_1 nellist_of_llist_a_inverse_2 by metis

lemma nellist_of_llist_inverse [simp]:
assumes "\<not> lnull ll"
shows  " llist_of_nellist (nellist_of_llist  ll) = ll"
using assms by simp

lemma nmap_nellist_of_llist_a:
  "nmap f (nellist_of_llist_a b ll) = nellist_of_llist_a (f b) (lmap f ll)"
by  (coinduction arbitrary: ll) (auto simp add: nmap_is_NNil)

lemma nmap_nellist_of_llist:
assumes "\<not> lnull ll"
shows " nmap f (nellist_of_llist ll) = nellist_of_llist (lmap f ll)" 
using assms 
by (metis lappend_inf lbutlast_snoc lfinite_lmap llast_lmap lmap_lbutlast nellist_of_llist_a_cong 
    nmap_nellist_of_llist_a)

lemma lmap_llist_of_nellist: 
  "lmap f (llist_of_nellist nell) = llist_of_nellist (nmap f nell)  " 
by (metis llist.map_disc_iff llist_of_nellist.disc_iff llist_of_nellist_inverse_b 
      nellist_of_llist_inverse nmap_nellist_of_llist)



definition cr_nellist :: "'a llist \<Rightarrow> 'a nellist \<Rightarrow> bool" 
 where "cr_nellist = (\<lambda> ll nell.  llist_of_nellist nell = ll)"

lemma llist_of_nellist_not_lnull:
 " \<not> (lnull (llist_of_nellist nell))"
by simp

lemma not_lnull_eq_lappend_lbutlast_llast:
 " \<not>(lnull ll) \<longleftrightarrow>  ll = lappend (lbutlast ll)  (LCons (llast ll) LNil)"
using llist.collapse(1) by fastforce

lemma Domainp_help:
 "\<not> lnull ll \<Longrightarrow> \<exists>nell. llist_of_nellist nell = ll"
using nellist_of_llist_inverse by blast

lemma not_lnull_conv_llist_of_nellist:
 " \<not> lnull ll \<longleftrightarrow> (\<exists>nell. llist_of_nellist nell = ll)" 
using Domainp_help llist_of_nellist_not_lnull by blast

lemma Domainp_cr_nellist [transfer_domain_rule]: 
  " Domainp cr_nellist = (\<lambda>ll. \<not>(lnull ll))"
unfolding cr_nellist_def Domainp_iff[abs_def] 
using Domainp_help by fastforce

lemma bi_unique_cr_nellist_help:
 "llist_of_nellist nelly = llist_of_nellist nellz \<Longrightarrow> nelly = nellz"
by (coinduction arbitrary: nelly nellz)
   (metis llist_of_nellist_inverse_b)

lemma quotient_help_2:
"( \<not> lnull (llist_of_nellist nell) \<and> nellist_of_llist  (llist_of_nellist nell) = nell)"
by (simp add: bi_unique_cr_nellist_help)

lemma quotient_help_nellist_1:
 " cr_nellist ll nell \<longrightarrow> nellist_of_llist ll = nell"
by (metis cr_nellist_def llist_of_nellist_inverse_b)

lemma quotient_help_nellist_2:
 "( cr_nellist (llist_of_nellist nell) nell)" 
by (simp add: cr_nellist_def)

lemma quotient_help_3_nellist: 
  "( (\<not> lnull llx \<and>  llx = lly) =
           (cr_nellist llx (nellist_of_llist llx) \<and>
            cr_nellist lly (nellist_of_llist lly) \<and>
            nellist_of_llist llx = 
            nellist_of_llist lly))"
by (metis Domainp.DomainI Domainp_cr_nellist cr_nellist_def nellist_of_llist_inverse)

lemma Quotient_nellist:
  "Quotient (\<lambda> llx lly. \<not> lnull llx \<and> llx = lly)
     nellist_of_llist llist_of_nellist cr_nellist"
unfolding Quotient_alt_def 
using quotient_help_3_nellist quotient_help_nellist_1 quotient_help_nellist_2 by blast


setup_lifting Quotient_nellist

context includes lifting_syntax
begin

lemma bi_unique_cr_nellist [transfer_rule]:
 " bi_unique cr_nellist "
unfolding cr_nellist_def bi_unique_def 
by (auto simp add: bi_unique_cr_nellist_help)

lemma right_total_cr_nellist [transfer_rule]:
 " right_total cr_nellist"
unfolding cr_nellist_def right_total_def
by simp

lemma NNil_transfer [transfer_rule]:
  "(A ===> pcr_nellist A) (\<lambda>b. LCons b LNil) NNil"
by (auto simp add: pcr_nellist_def OO_def cr_nellist_def rel_fun_def)

lemma NCons_transfer [transfer_rule]:
  "(A ===> pcr_nellist A ===> pcr_nellist A) LCons NCons"
by (auto simp add: pcr_nellist_def OO_def cr_nellist_def rel_fun_def)

lemma nmap_transfer [transfer_rule]:
  "((=) ===> pcr_nellist (=) ===> pcr_nellist (=)) lmap nmap"
by (auto simp add: cr_nellist_def nellist.pcr_cr_eq rel_fun_def lmap_llist_of_nellist)


lemma is_NNil_transfer [transfer_rule]:
  "(pcr_nellist (=)  ===> (=)) is_lfirst is_NNil"
by ( simp add: cr_nellist_def nellist.pcr_cr_eq rel_fun_def )
   (metis lbutlast_simps(2) llast_singleton llist.disc(2) llist_of_nellist_eq_LNil 
    llist_of_nellist_inverse_a nellist_of_llist_inverse)

lemma is_NNil_nellist_of_llist_conv_is_lfirst: 
assumes "\<not> lnull lx" 
shows "is_NNil(nellist_of_llist lx) \<longleftrightarrow> is_lfirst lx " 
using assms
by (cases lx) 
   (simp, metis lbutlast.ctr(1) lbutlast.disc_iff(2) lbutlast_eq_LNil_conv llist.disc(1) 
    nellist_of_llist_a.disc_iff(2))

lemma nfirst_transfer_a [transfer_rule]:
 "(pcr_nellist (=) ===> (=)) lhd nfirst "
by ( simp add: cr_nellist_def nellist.pcr_cr_eq nfirst_def rel_fun_def llist_of_nellist.simps(2))

lemma nhd_transfer_a1 [transfer_rule]:
 "(pcr_nellist (=) ===> (=)) (\<lambda> ll. if is_lfirst ll then lhd LNil else lhd ll)  nhd" 
by ( simp add: cr_nellist_def nellist.pcr_cr_eq  rel_fun_def OO_def)
   (metis lbutlast_simps(2) lhd_llist_of_nellist llist_of_nellist_eq_LNil nhd_nellist_of_llist_a 
    quotient_help_2)

lemma ntl_transfer [transfer_rule]:
 " ( pcr_nellist A ===> pcr_nellist A) (\<lambda> ll. if is_lfirst ll then ll else ltl ll) ntl"
proof (auto simp add: pcr_nellist_def cr_nellist_def OO_def rel_fun_def intro!: llist_all2_ltlI
       lfinite_LConsI dest: llist_all2_lnullD) 
show "\<And>b y. llist_all2 A (LCons b LNil) (llist_of_nellist y) \<Longrightarrow> 
                llist_all2 A (LCons b LNil) (llist_of_nellist (ntl y))"  
using llist_all2_ltlI
by (metis eq_LConsD is_NNil_ntl llist_all2_LNil1 llist_of_nellist.code llist_of_nellist.simps(3) 
       llist_of_nellist_eq_LNil ltl_llist_of_nellist nlast_ntl)
next
show "\<And>x y. \<forall>b. x \<noteq> LCons b LNil \<Longrightarrow> llist_all2 A x (llist_of_nellist y) \<Longrightarrow> 
                   llist_all2 A (ltl x) (llist_of_nellist (ntl y))" 
by (metis lhd_LCons_ltl llist_all2_LNil2 llist_all2_lnullD llist_all2_ltlI 
    llist_of_nellist.disc_iff ltl_llist_of_nellist ltl_llist_of_nellist1)
qed

lemma nfinite_transfer [transfer_rule]:
 "(pcr_nellist (=) ===> (=)) lfinite nfinite"
by (auto simp add: nellist.pcr_cr_eq cr_nellist_def nfinite_def rel_fun_def)

lemma llist_of_nellist_transfer [transfer_rule]:
  "(pcr_nellist (=)  ===> (=)) id llist_of_nellist"
by (simp add: pcr_nellist_def cr_nellist_def OO_def rel_fun_def llist.rel_eq)

lemma nellist_of_llist_a_transfer [transfer_rule]:
  "((=) ===> (=) ===> pcr_nellist (=)) (\<lambda>b ll. lappend ll (LCons b LNil)) nellist_of_llist_a"
by (auto simp add: pcr_nellist_def cr_nellist_def OO_def rel_fun_def)

lemma nlast_nellist_of_llist_a_lfinite [simp]:
  "lfinite ll \<Longrightarrow> nlast (nellist_of_llist_a b ll) = b"
by(induct rule: lfinite.induct) simp_all

lemma snocn_transfer [transfer_rule]:
 " (pcr_nellist (A)  ===>  (A)  ===> pcr_nellist (A)) (\<lambda> ll a. lappend ll (LCons a LNil)) snocn"
unfolding rel_fun_def
by (auto simp add: pcr_nellist_def nellist.pcr_cr_eq cr_nellist_def OO_def)
   (metis LNil_transfer llist.rel_intros(2) llist_all2_lappendI llist_of_nellist_inverse 
    nellist_of_llist_a_inverse)

lemma nellist_all2_help_a:
 "llist_all2 P (llist_of_nellist nella) (llist_of_nellist nellb) \<Longrightarrow> nellist_all2 P nella nellb"
by (coinduction arbitrary: nella nellb)
   (metis lhd_llist_of_nellist llist.disc(1) llist_all2_LCons_LCons llist_all2_LNil1 
    llist_all2_LNil2 llist_all2_lhdD llist_all2_ltlI llist_of_nellist.disc_iff 
    llist_of_nellist_eq_LNil ltl_llist_of_nellist ltl_simps(2))

lemma nellist_all2_help_b:
 "nellist_all2 P nella nellb \<Longrightarrow> llist_all2 P (llist_of_nellist nella) (llist_of_nellist nellb)"
proof (coinduction arbitrary: nella nellb)
 case LNil
 then show ?case 
 by simp
 next
 case LCons
 then show ?case 
 by auto
    (metis llist_of_nellist.simps(2) nellist.case_eq_if nellist.rel_sel,
     metis llist_all2_LNil1 ltl_llist_of_nellist ltl_llist_of_nellist1 nellist.rel_sel)
qed

lemma nellist_all2_transfer [transfer_rule]:
  "((=) ===> pcr_nellist (=)  ===> pcr_nellist (=)  ===> (=))  llist_all2 nellist_all2"
unfolding nellist.pcr_cr_eq cr_nellist_def
using nellist_all2_help_a nellist_all2_help_b by blast

lift_definition nappend :: "'a nellist \<Rightarrow> 'a nellist \<Rightarrow> 'a nellist"
is "(\<lambda> llx lly. lappend llx lly)  " 
by auto


lemma llist_all2_llist_of_nellist_1:
assumes "\<not> lnull y " 
        "llist_all2 (\<lambda>x z. llist_of_nellist z = x) llist1 y " 
        "llist_all2 (\<lambda>x z. llist_of_nellist z = x) llist2 y "  
shows "     llist1 =  llist2"
proof -
 have 1: "llist_all2 (\<lambda>x z. llist_of_nellist z = x) llist1 y = 
          llist_all2 (=) llist1 (lmap llist_of_nellist y) "
     using llist_all2_lmap2[of "(=)" llist1 llist_of_nellist y]
        using llist_all2_mono by fastforce 
 have 2: "llist_all2 (\<lambda>x z. llist_of_nellist z = x) llist2 y =
          llist_all2 (=) llist2 (lmap llist_of_nellist y)" 
   using llist_all2_lmap2[of "(=)" llist2 llist_of_nellist y]
        using llist_all2_mono by fastforce 
 show ?thesis using assms
 by (metis (full_types) "1" "2" llist.rel_eq)
qed

lemma llist_all2_llist_of_nellist_2:
assumes "\<not> lnull y " 
        "llist_all2 (\<lambda>z x. llist_of_nellist z = x) y llist1  " 
        "llist_all2 (\<lambda>z x. llist_of_nellist z = x) y llist2  "  
shows "     llist1 =  llist2"
proof -
 have 1: "llist_all2 (\<lambda>z x. llist_of_nellist z = x) y llist1  = 
          llist_all2 (=)  (lmap llist_of_nellist y) llist1"
     using llist_all2_lmap1[of "(=)" ]
        using llist_all2_mono by fastforce
 have 2: "llist_all2 (\<lambda>z x. llist_of_nellist z = x) y llist2  =
          llist_all2 (=) (lmap llist_of_nellist y) llist2" 
   using llist_all2_lmap1[of "(=)" ]
        using llist_all2_mono by fastforce 
 show ?thesis using assms
 by (metis (full_types) "1" "2" llist.rel_eq)
qed



lift_definition nconcat :: "'a nellist nellist \<Rightarrow> 'a nellist "
 is "(\<lambda> lxs. lconcat lxs) " 
apply ( simp add: pcr_nellist_def cr_nellist_def OO_def rel_fun_def llist.rel_eq  )
using llist.rel_cases mem_Collect_eq llist_all2_llist_of_nellist_1 by fastforce

lift_definition nfilter :: " ('a \<Rightarrow> bool) \<Rightarrow> 'a nellist \<Rightarrow> 'a nellist"
is "\<lambda> P ll.   (if lnull(lfilter P ll) then ll else lfilter P ll)   "
by auto

lift_definition lappendn :: "'a llist \<Rightarrow> 'a nellist \<Rightarrow> 'a nellist"
is " lappend"
by auto

lift_definition nzip :: "'a nellist \<Rightarrow> 'b nellist \<Rightarrow> ('a \<times> 'b) nellist"
is "(\<lambda> llx lly. lzip llx lly)"
by auto

lift_definition niterates :: "('a \<Rightarrow> 'a) \<Rightarrow> 'a \<Rightarrow> 'a nellist"
is "(\<lambda> f a . iterates f a)"
by auto

lift_definition ndistinct :: "'a nellist \<Rightarrow> bool"
 is " ldistinct "
by auto

lift_definition nnth :: "'a nellist \<Rightarrow> nat \<Rightarrow> 'a"
is "\<lambda> ll n. lnth ll (the_enat (min (enat n) ((epred (llength ll)))))"
by blast

lift_definition nlength :: "'a nellist \<Rightarrow> enat"
is "\<lambda> ll. epred(llength ll) " 
by auto
 
lift_definition ndropn :: "nat \<Rightarrow> 'a nellist \<Rightarrow> 'a nellist"
is "\<lambda> n ll.  ldropn (the_enat (min (enat n) ((epred(llength ll)))))  ll" 
by auto
  (metis co.enat.exhaust_sel enat_the_enat iless_Suc_eq infinity_ileE leD llength_eq_0 
         min.cobounded1 min.cobounded2)

lift_definition ntake :: "enat \<Rightarrow> 'a nellist \<Rightarrow> 'a nellist"
is "\<lambda> n ll. ltake (eSuc n) ll" 
by auto

lift_definition ntaken :: "nat \<Rightarrow> 'a nellist \<Rightarrow> 'a nellist"
 is "\<lambda> n ll. ltake (Suc n) ll"
using enat_0_iff(2) by auto





subsection \<open>The nlast element @{term "nlast"}\<close>

lemma nlast_not_nfinite:
  assumes "\<not> nfinite nell"
  shows "nlast nell = undefined" 
unfolding nlast0_nlast[symmetric]
using assms
by (rule contrapos_np)
   (induct nell rule: nlast0.raw_induct[rotated 1, OF refl, consumes 1], 
    auto split: nellist.split_asm) 

lemma nlast_nellist_of_llist_a:
  "nlast (nellist_of_llist_a y ll) = (if lfinite ll then y else undefined)"
by (simp add: nfinite_def nlast_not_nfinite)

lemma nlast_transfer [transfer_rule]:
  "(pcr_nellist (=) ===> (=)) (\<lambda>ll. if lfinite ll then llast ll else undefined) nlast"
by (auto simp add: cr_nellist_def pcr_nellist_def nlast_nellist_of_llist_a OO_def 
         dest: llist_all2_lfiniteD)
   (simp add: llist.rel_eq nfinite_def nlast_llast nlast_not_nfinite rel_funI)

lemma nlast_nmap [simp]: 
  "nfinite nell \<Longrightarrow> nlast (nmap f  nell) = f (nlast nell)"
  by (induct rule: nfinite_induct) 
     (auto simp add: nellist.map_sel(1))

lemma nset_nlast:
 " nfinite nell \<Longrightarrow> nlast nell \<in> nset nell"
 by (induct rule: nfinite_induct)
    (simp_all add: nellist.set_sel(3))

subsection \<open>@{term "nset"}\<close>

lemma lset_llist_of_nellist_1:
assumes "nfinite nell"
shows  "lset (lbutlast (llist_of_nellist nell)) \<union> {nlast nell}= nset nell" (is "?lhs = ?rhs")
proof(intro set_eqI iffI)
  fix x
  assume "x \<in> ?lhs"
  thus "x \<in> ?rhs"
    proof -
     have 1: " nlast nell = x \<Longrightarrow> x \<in> nset nell"
       using assms nset_nlast by blast
     have 2: " nlast nell = x \<Longrightarrow> x  \<in> nset nell"
       unfolding nlast0_nlast[symmetric] by (simp add: "1")
     have 3: "x \<in> lset (lbutlast(llist_of_nellist nell)) \<Longrightarrow> x \<in> nset nell"       
       proof (induct "lbutlast (llist_of_nellist nell)" arbitrary: nell rule: llist_set_induct)
       case find
       then show ?case
          by (metis lbutlast.disc(1) llist.disc(1) llist_of_nellist_inverse_b ltl_llist_of_nellist1
              nellist.set_sel(2) nhd_nellist_of_llist_a)
       next
       case (step y)
       then show ?case 
         by (metis lbutlast.disc(1) lbutlast_ltl llist.disc(1) ltl_llist_of_nellist ltl_llist_of_nellist1
             nellist.disc(1) nellist.exhaust_sel nellist.set_intros(3))
       qed
     show ?thesis 
       using "2" "3" \<open>x \<in> lset (lbutlast (llist_of_nellist nell)) \<union> {nlast nell}\<close> by blast
  qed
next
  fix x
  assume "x \<in> ?rhs"
  thus "x \<in> ?lhs"
  proof(induct rule: nellist_set_induct)
    case (findnil nell)
    then show ?case 
     by (cases nell) auto
    next
    case (find nell)
    thus ?case 
     by (metis UnI1 lbutlast.disc_iff(2) llist.set_sel(1) ltl_llist_of_nellist 
         nhd_nellist_of_llist_a quotient_help_2)
    next
    case step
    thus ?case 
     by (metis Un_iff in_lset_ltlD lbutlast_ltl ltl_llist_of_nellist nlast_ntl)
  qed
qed

lemma lset_llist_of_nellist_2:
assumes "\<not>nfinite nell"
shows  "lset (lbutlast (llist_of_nellist nell)) = nset nell" (is "?lhs = ?rhs")
proof(intro set_eqI iffI)
  fix x
  assume "x \<in> ?lhs"
  thus "x \<in> ?rhs"
    proof -
     have 3: "x \<in> lset (lbutlast (llist_of_nellist nell)) \<Longrightarrow> x \<in> nset nell"       
       proof (induct "lbutlast (llist_of_nellist nell)" arbitrary: nell rule: llist_set_induct)
       case find
       then show ?case 
        by (metis lbutlast.disc(1) llist.disc(1) llist_of_nellist_inverse_a ltl_llist_of_nellist1 
            nellist.set_sel(2) nhd_nellist_of_llist_a)
       next
       case (step y)
       then show ?case 
         by (metis lbutlast.disc(1) lbutlast_ltl llist.disc(1) ltl_llist_of_nellist 
             ltl_llist_of_nellist1 nellist.collapse(2) nellist.set_intros(3))
       qed  
     show ?thesis 
       using  "3" \<open>x \<in> lset (lbutlast (llist_of_nellist nell))\<close> by blast
  qed
next
  fix x
  assume "x \<in> ?rhs"
  thus "x \<in> ?lhs"
  using assms
  proof(induct rule: nellist_set_induct)
    case (findnil nell)
    then show ?case 
     by (cases nell) auto
    next
    case (find nell)
    thus ?case 
      by (metis lbutlast_not_lfinite lhd_llist_of_nellist llist.set_sel(1) llist_of_nellist_not_lnull
          nfinite_def)
    next
    case step
    thus ?case 
     by (metis in_lset_ltlD lbutlast_ltl ltl_llist_of_nellist nfinite_ntl)
  qed
qed

lemma lset_llist_of_nellist [simp]: 
 "(if nfinite nell then lset (lbutlast(llist_of_nellist nell)) \<union> {nlast nell} 
   else lset (lbutlast (llist_of_nellist nell))) = nset nell" 
using lset_llist_of_nellist_1 lset_llist_of_nellist_2 by auto

lemma lset_llist_of_nellist_a [simp]:
 " lset(llist_of_nellist nell) = nset nell"
proof (cases "nfinite nell" )
 case True
 then show ?thesis 
 by (metis lappend_lbutlast_llast_id_lfinite lbutlast_lfinite llist.simps(19) 
     llist_of_nellist_not_lnull lset_LNil lset_lappend_lfinite lset_llist_of_nellist_1 nfinite_def 
     nlast_llast)
 next
 case False
 then show ?thesis  by (metis lbutlast_not_lfinite lset_llist_of_nellist_2 nfinite_def)
qed
 
lemma nset_nellist_of_llist_a [simp]:
 shows "nset (nellist_of_llist_a b ll) = (if lfinite ll then lset ll \<union> {b} else lset ll)"
proof (cases "lfinite ll")
 case True
 then show ?thesis 
 by (metis llist.simps(19) lset_LNil lset_lappend_lfinite lset_llist_of_nellist_a 
     nellist_of_llist_a_inverse)
 next
 case False
 then show ?thesis 
 by (metis lappend_inf lset_llist_of_nellist_a nellist_of_llist_a_inverse)
qed

lemma nset_transfer [transfer_rule]:
  "(pcr_nellist (=)  ===> (=)) lset nset"
by(auto simp add: cr_nellist_def nellist.pcr_cr_eq)

end

subsection \<open>@{term "nmap"}\<close>

lemma nmap_eq_NCons_conv:
  "nmap f  nellx = NCons y nelly \<longleftrightarrow>
  (\<exists>z nellz. nellx = NCons z nellz \<and> f z = y \<and> nmap f nellz = nelly)"
by(cases nellx) simp_all

lemma NCons_eq_nmap_conv:
  "NCons y nelly = nmap f  nellx \<longleftrightarrow>
  (\<exists>z nellz. nellx = NCons z nellz \<and> f z = y \<and> nmap f nellz = nelly)"
by(cases nellx) auto


subsection \<open>Appending two nonempty lazy lists @{term "nappend" }\<close>

lemma nappend_NNil [simp, code, nitpick_simp]:
  "nappend (NNil b) nell = (NCons b nell)"
by transfer auto

lemma nappend_NCons [simp, code, nitpick_simp]:
  "nappend (NCons a nellx) nelly = NCons a (nappend nellx nelly)"
by transfer auto 

lemma nhd_nappend [simp]:
 " nhd(nappend nellx nelly) = (if is_NNil nellx then nlast nellx else nhd nellx)"
by (cases nellx) auto

lemma ntl_nappend [simp]:
 " ntl(nappend nellx nelly) = (if is_NNil nellx then nelly else nappend (ntl nellx) nelly)"
by (cases nellx) auto

lemma is_NNil_nappend:
 " is_NNil(nappend nellx nelly) \<longleftrightarrow> False"
by (cases nellx) auto

lemma nappend_assoc: 
  "nappend (nappend nellx nelly) nellz = nappend nellx (nappend nelly nellz)"
by transfer (auto simp add: split_beta lappend_assoc)

lemma nmap_nappend_distrib:
 " nmap f (nappend nellx nelly) = nappend (nmap f nellx) (nmap f nelly)"
by transfer (auto simp add: split_beta lmap_lappend_distrib)

lemma nlast_nappend:
  "nlast (nappend nellx nelly) = (if nfinite nellx then nlast nelly else nlast nellx)"
by transfer (auto simp add: llast_lappend)

lemma nfinite_nappend: 
 "nfinite (nappend nellx nelly) \<longleftrightarrow> nfinite nellx \<and> nfinite nelly"
by transfer auto

lemma nappend_inf: 
   "\<not> nfinite nellx \<Longrightarrow> nappend nellx nelly = nellx"
by transfer (auto simp add: lappend_inf)

lemma nappend_snocn_inf:
 assumes " \<not> nfinite nell"
 shows   " nappend nell (NNil a) = snocn nell a"
using assms 
by (simp add: nappend_inf snocn_inf)

lemma nappend_snocn_finite:
 assumes "  nfinite nell"
 shows   " nappend nell (NNil a) = snocn nell a"
using assms 
by (induct rule: nfinite_induct) simp_all

lemma nappend_snocn:
 " nappend nell (NNil a) = snocn nell a"
by (meson nappend_snocn_finite nappend_snocn_inf)

lemma split_nellist_first:
 assumes "x \<in> nset nell"
 shows   "nell = (NNil x) \<or> (\<exists> ys. nell = nappend (NNil x) ys) \<or> 
          ( \<exists>ys. nell = nappend ys (NNil x) \<and> nfinite ys \<and> x \<notin> nset ys) \<or>
         ( \<exists>ys zs. nell = nappend ys (NCons x zs) \<and> nfinite ys \<and> x \<notin> nset ys)"
using assms 
by transfer
   (auto,
    metis eq_LConsD lhd_lappend llist.disc(1) llist.expand split_llist_first)

lemma split_nellist:
 assumes " x \<in> nset nell"
 shows " nell = (NNil x) \<or> (\<exists> ys. nell = nappend (NNil x) ys) \<or> 
        ( \<exists> ys zs. nell = nappend ys (NCons x zs) \<and> nfinite ys) \<or>
        ( \<exists> ys. nell = nappend ys (NNil x) \<and> nfinite ys) "
using assms 
by (meson split_nellist_first)

lemma split_nellist_a:
 assumes " nell = (NNil x) \<or> (\<exists> ys. nell = nappend (NNil x) ys) \<or> 
        ( \<exists> ys zs. nell = nappend ys (NCons x zs) \<and> nfinite ys) \<or>
        ( \<exists> ys. nell = nappend ys (NNil x) \<and> nfinite ys)"
 shows " x \<in> nset nell "
proof -
 have 1: "nell = (NNil x) \<Longrightarrow> x \<in> nset nell "
   by simp
 have 2: "(\<exists> ys. nell = nappend (NNil x) ys) \<Longrightarrow> x \<in> nset nell" 
   by auto
 have 3: "( \<exists> ys zs. nell = nappend ys (NCons x zs) \<and> nfinite ys) \<Longrightarrow>  x \<in> nset nell" 
   by transfer  auto
 have 4: " ( \<exists> ys. nell = nappend ys (NNil x) \<and> nfinite ys) \<Longrightarrow> x \<in> nset nell" 
   by transfer  auto
 show ?thesis using 1 2 3 4 assms by blast
qed


subsection \<open>Appending a nonempty lazy list to a lazy list @{term "lappendn"}\<close>

lemma lappendn_LNil [simp, code, nitpick_simp]: 
 "lappendn LNil nell = nell"
by transfer auto

lemma lappendn_LCons [simp, code, nitpick_simp]:
  "lappendn (LCons x ll) nell = NCons x (lappendn ll nell )"
by transfer auto

lemma nlast_lappendn_lfinite [simp]:
  "lfinite ll \<Longrightarrow> nlast (lappendn ll nell) = nlast nell"
by transfer
   (auto simp add: llast_lappend)

lemma nset_lappendn_lfinite [simp]:
  "lfinite ll \<Longrightarrow> nset (lappendn ll nell) = lset ll \<union> nset nell"
by transfer auto

lemma nlength_nappend [simp]:
 " nlength (nappend nellx nelly) = nlength nellx + nlength nelly +1"
by transfer
   (auto, metis co.enat.exhaust_sel epred_iadd1 iadd_Suc_right llength_eq_0 plus_1_eSuc(2))

lemma nfinite_nlength_enat:
 assumes "nfinite nell"
 shows " \<exists> n. nlength nell = enat n"
using assms
by transfer (metis epred_conv_minus idiff_enat_enat lfinite_llength_enat one_enat_def)

lemma nlength_eq_enat_nfiniteD:
 " nlength nell = enat n \<Longrightarrow> nfinite nell"
by transfer (metis epred_Infty llength_eq_enat_lfiniteD not_lfinite_llength)
 
lemma nfinite_conv_nlength_enat:
 " nfinite nell \<longleftrightarrow> (\<exists> n. nlength nell = enat n)"
using nfinite_nlength_enat nlength_eq_enat_nfiniteD by blast



subsection \<open>The length of a nonempty lazy list @{term "nlength"}\<close>

lemma [simp, nitpick_simp]:
  shows nlength_NNil: "nlength (NNil b) = 0"
  and nlength_NCons: "nlength (NCons x nell) = eSuc (nlength nell)"
by (transfer, simp) (transfer, auto)

lemma llength_llist_of_nellist [simp]: 
  "epred(llength (llist_of_nellist nell)) = nlength nell"
by transfer auto

lemma nlength_nmap [simp]: 
  "nlength (nmap f  nell) = nlength nell"
by transfer simp

definition gen_nlength :: "nat \<Rightarrow> 'a nellist \<Rightarrow> enat"
where "gen_nlength n nell = enat n + nlength nell"

lemma gen_nlength_code [code]:
  "gen_nlength n (NNil b) = enat n"
  "gen_nlength n (NCons x nell) = gen_nlength (n + 1) nell"
by (simp_all add: gen_nlength_def iadd_Suc eSuc_enat[symmetric] iadd_Suc_right)

lemma nlength_code [code]: 
  "nlength = gen_nlength 0"
by(simp add: gen_nlength_def fun_eq_iff zero_enat_def)


subsection \<open>The nth element of a nonempty  lazy list @{term "nnth"}\<close>

lemma nnth_NNil [nitpick_simp]:
  "nnth (NNil b) n = b"
by transfer simp

lemma nnth_NCons:
  "nnth (NCons x nell) n = (case n of 0 \<Rightarrow> x | Suc n' \<Rightarrow> nnth nell n')"
by (transfer fixing: n)
   (auto simp add: lnth_LCons Nitpick.case_nat_unfold zero_enat_def min_enat1_conv_enat,
    metis enat_0_iff(1) less_not_refl3 llength_eq_0 min_def min_enat1_conv_enat,
    metis enat_min_eq_0_iff min_enat1_conv_enat not_gr_zero the_enat.simps the_enat_0,
    metis One_nat_def epred_enat epred_min min_enat1_conv_enat the_enat.simps)

lemma nnth_code [simp, nitpick_simp, code]:
  shows nnth_0: "nnth (NCons x nell) 0 = x"
  and nnth_Suc_NCons: "nnth (NCons x nell) (Suc n) = nnth nell n"
by(simp_all add: nnth_NCons)

lemma lnth_llist_of_nellist [simp]:
  " lnth (llist_of_nellist nell)  (the_enat (min (enat n) ((epred(llength (llist_of_nellist nell)))))) 
    = nnth nell n"
by transfer auto

lemma nnth_nmap [simp]: 
  "enat n \<le> nlength nell \<Longrightarrow> nnth (nmap f  nell) n = f (nnth nell n)"
by transfer
   (metis co.enat.exhaust_sel iless_Suc_eq llength_eq_0 llength_lmap lnth_lmap min.orderE 
    the_enat.simps)

lemma nhd_conv_nnth:
 " \<not> is_NNil nell \<Longrightarrow> nhd nell = nnth nell 0"
by (metis nellist.collapse(2) nnth_0)

lemmas nnth_0_conv_nhd = nhd_conv_nnth[symmetric]

lemma nnth_ntl:
 " nnth (ntl nell) n = nnth nell (Suc n)"
by (metis nellist.exhaust_sel nellist.sel(4) nnth_NNil nnth_Suc_NCons)

lemma in_nset_conv_nnth:
 " x \<in> nset nell \<longleftrightarrow> (\<exists> n. enat n \<le> nlength nell \<and> nnth nell n = x)"
by  transfer
    (metis eSuc_epred iless_Suc_eq in_lset_conv_lnth llength_eq_0 min_absorb1 the_enat.simps)

lemma nnth_beyond:
 " nlength nell < enat n \<Longrightarrow> nnth nell n = nlast nell  "
by transfer
 (metis co.enat.exhaust_sel epred_llength less_enatE lfinite_ltl llast_conv_lnth llength_eq_0 
  llength_eq_enat_lfiniteD min.absorb4 the_enat.simps)

lemma exists_Pred_nnth_nset:
" (\<exists> x \<in> nset nell. P x) = (\<exists> n. n\<le> nlength nell \<and> P (nnth nell n))"
by (metis in_nset_conv_nnth)

lemma nset_conv_nnth:
 " nset nell = {nnth nell n| n. enat n \<le> nlength nell}"
by (auto simp add: in_nset_conv_nnth)

lemma nnth_nappend1:
 " enat n \<le> nlength nellx \<Longrightarrow> nnth (nappend nellx nelly) n = nnth nellx n"
proof (induct n arbitrary: nellx)
 case 0
 then show ?case
 by (metis is_NNil_def is_NNil_nappend nellist.sel(1) nhd_nappend nnth_0_conv_nhd nnth_NNil) 
 next
 case (Suc n)
 then show ?case 
 proof (cases nellx)
 case (NNil x1)
 then show ?thesis
 using Suc.prems enat_0_iff(1) by auto
 next
 case (NCons x21 x22)
 then show ?thesis
 using Suc.hyps Suc.prems Suc_ile_eq by auto
 qed
qed

lemma nnth_nappend2:
 " \<lbrakk>nlength nellx = enat k; k<n\<rbrakk> \<Longrightarrow> nnth (nappend nellx nelly) n = nnth nelly (n -Suc k)"
proof (induct n arbitrary: nellx k)
 case 0
 then show ?case by blast
 next
 case (Suc n)
 then show ?case 
 by (cases nellx)
    (auto simp add: eSuc_def zero_enat_def split: enat.split_asm)
qed

lemma nnth_nappend:
 " nnth (nappend nellx nelly) n =
   (if enat n \<le> nlength nellx then nnth nellx n else nnth nelly (n- Suc(the_enat(nlength nellx))))"
by (cases "nlength nellx")
   (auto simp add: nnth_nappend1 nnth_nappend2)

lemma nnth_nlast:
 " nfinite nell \<Longrightarrow> nlast nell = nnth nell (the_enat (nlength nell))"
by transfer
   (simp,
    metis co.enat.exhaust_sel enat_the_enat ile_eSuc infinity_ileE lfinite_llength_enat 
    llast_conv_lnth llength_eq_0 min.idem)



subsection \<open>@{term "ntake"}\<close>

lemma ntake_NNil [simp, code, nitpick_simp]:
 " ntake n (NNil b) = (NNil b)"
by transfer auto

lemma ntake_0 [simp]:
 " ntake 0 nell = (NNil (nfirst nell))"
by transfer (auto simp add: ltake.ctr(2))

lemma ntake_Suc_NCons [simp]:
 " ntake (eSuc n) (NCons x nell) = (NCons x (ntake n nell))"
by transfer auto

lemma ntake_Suc:
 " ntake (eSuc n) nell = 
   (case nell of (NNil b) \<Rightarrow> (NNil b) | (NCons x nell') \<Rightarrow> (NCons x (ntake n nell')))"
by (cases nell) simp_all

lemma is_NNil_ntake [simp]:
 " is_NNil(ntake n nell) \<longleftrightarrow> is_NNil nell \<or> n=0"
proof (cases nell)
 case (NNil x1)
 then show ?thesis by simp
 next
 case (NCons x nell1)
 then show ?thesis 
  proof (cases n)
   case (enat nat)
   then show ?thesis 
   by (metis NCons enat_coexhaust nellist.disc(1) nellist.disc(2) ntake_0 ntake_Suc_NCons)
   next
   case infinity
   then show ?thesis 
   by (metis NCons eSuc_infinity i0_ne_infinity nellist.disc(2) ntake_Suc_NCons)
  qed
  qed
lemma ntake_eq_NNil_iff [simp]:
 " ntake n nell = (NNil x) \<longleftrightarrow> nell = (NNil x) \<or> (n = 0 \<and> nfirst nell = x) "
proof (cases nell)
 case (NNil x1)
 then show ?thesis 
 using ntake_0 by fastforce
 next
 case (NCons x nell1)
 then show ?thesis 
 proof (cases n)
  case (enat nat)
  then show ?thesis 
  by (metis NCons is_NNil_ntake nellist.disc(1) nellist.disc(2) nellist.inject(1) ntake_0)
  next
  case infinity
  then show ?thesis 
  by (metis NCons eSuc_infinity infinity_ne_i0 nellist.distinct(1) ntake_Suc_NCons)
 qed
qed
 
lemma NNil_eq_ntake_iff [simp]:
 " (NNil x) = ntake n nell   \<longleftrightarrow> nell = (NNil x) \<or> (n = 0 \<and> nfirst nell = x) "
by (metis ntake_eq_NNil_iff)

lemma ntake_NCons [code, nitpick_simp]:
 " ntake n (NCons x nell) = (case n of 0 \<Rightarrow> (NNil x) | (eSuc n') \<Rightarrow> (NCons x (ntake n' nell)) )"
by (simp add: co.enat.case_eq_if) 
   (metis  eSuc_epred nellist.simps(6) nfirst_def ntake_Suc_NCons) 

lemma nhd_ntake [simp]:
 " n \<noteq> 0 \<Longrightarrow> nhd(ntake n nell) = nhd nell"
unfolding nhd_def 
by (simp add:  nellist.case_eq_if ) 
   (metis (no_types, lifting) co.enat.case_eq_if  nellist.collapse(2) nellist.sel(3)  ntake_NCons)

lemma ntl_ntake:
 " n \<noteq> 0 \<Longrightarrow> ntl(ntake n nell) = ntake (epred n) (ntl nell)"
by (cases nell) (simp, metis eSuc_epred nellist.sel(5) ntake_Suc_NCons)

lemma ntl_ntake_0:
 " ntl(ntake 0 nell) = (NNil (nfirst nell))"
by simp

lemma ntake_ntl:
 " ntake n (ntl nell) = ntl(ntake (Suc n) nell)"
by (simp add: enat_0_iff(1) ntl_ntake)

lemma nlength_ntake [simp]:
 " nlength (ntake n nell) = min n (nlength nell)"
by transfer simp

lemma ntake_nmap [simp]:
 " ntake n (nmap f nell) = nmap f (ntake n nell)"
by transfer simp

lemma ntake_ntake [simp]:
 " ntake n (ntake m nell) = ntake (min n m) nell"
by transfer simp

lemma nset_ntake:
 " nset (ntake n nell) \<subseteq> nset nell"
by transfer (simp add: lset_ltake)

lemma ntake_all:
 "nlength nell \<le> m \<Longrightarrow>  ntake m nell = nell"
by transfer (auto, metis eSuc_epred eSuc_ile_mono llength_eq_0 ltake_all)

lemma nfinite_ntake [simp]:
 " nfinite (ntake n nell) \<longleftrightarrow> nfinite nell \<or> n < \<infinity>"
by transfer (metis Extended_Nat.eSuc_mono eSuc_infinity lfinite_ltake)

lemma ntake_nappend1:
 " n \<le> nlength nellx \<Longrightarrow> ntake n (nappend nellx nelly) = ntake n nellx"
by transfer (auto, metis eSuc_epred eSuc_ile_mono llength_eq_0 ltake_lappend1)

lemma ntake_nappend2:
assumes " nlength nellx < n "
shows "ntake n (nappend nellx nelly) = nappend nellx (ntake (n - nlength nellx -1) nelly) " 
proof (cases "nellx")
 case (NNil x1)
 then show ?thesis using assms 
 by (metis eSuc_le_iff eSuc_minus_1 idiff_0_right ileI1 nappend_NNil nlength_NNil ntake_Suc_NCons)
 next
 case (NCons x21 x22)
 then show ?thesis 
 proof (cases "nfinite nellx")
  case True
  then show ?thesis 
   using assms 
   proof  (transfer)
    fix llxa :: "'a llist" 
    fix na
    fix llya :: "'a llist" 
    assume a0: "\<not> lnull llxa \<and> llxa = llxa" 
    assume a1: " lfinite llxa " 
    assume a2: "epred (llength llxa) < na "
    assume a3: "\<not> lnull llya \<and> llya = llya " 
    show "\<not> lnull (ltake (eSuc na) (lappend llxa llya)) \<and>
       ltake (eSuc na) (lappend llxa llya) = 
       lappend llxa (ltake (eSuc (na - epred (llength llxa) - 1)) llya)  "  
    proof -
     have 1: "\<not> lnull (ltake (eSuc na) (lappend llxa llya))"
       using a0 by force
     have 2: " ltake (eSuc na) (lappend llxa llya) =
               lappend (ltake (eSuc na) llxa) (ltake ((eSuc na) - llength llxa) llya)"
       by (meson ltake_lappend) 
     have 21: "llength llxa \<le> (eSuc na)" 
       by (metis a0 a2 co.enat.exhaust_sel eSuc_ile_mono leD le_cases llength_eq_0)
     have 3: "lappend (ltake (eSuc na) llxa) (ltake ((eSuc na) - llength llxa) llya) =
              lappend llxa (ltake (eSuc na - llength llxa) llya)"
       using ltake_lappend2[of llxa "(eSuc na)" llya] 21 2 by auto
     have 4: "(eSuc na - llength llxa) = (eSuc (na - epred (llength llxa) - 1))"
       by (metis a0 a1 a2 canonically_ordered_monoid_add_class.lessE co.enat.exhaust_sel eSuc_infinity 
           eSuc_minus_1 eSuc_minus_eSuc enat_add_sub_same llength_eq_0 llength_eq_infty_conv_lfinite)
     have 5: "(ltake (eSuc na - llength llxa) llya) = 
              (ltake (eSuc (na - epred (llength llxa) - 1)) llya)" 
       using "4" by auto
    show ?thesis using "1" "2" "3" "5" by fastforce
   qed
  qed            
  next
  case False
  then show ?thesis using assms 
  by (simp add: nappend_inf ntake_all)
 qed
qed

lemma ntake_eq_ntake_antimono:
 " \<lbrakk> ntake n nellx = ntake n nelly; m \<le> n \<rbrakk> \<Longrightarrow> ntake m nellx = ntake m nelly"
by (metis min.orderE ntake_ntake)

lemma ntake_nnth:
 assumes " enat m \<le> n"
 shows  " (nnth (ntake n nell) m) = (nnth nell m)"
using assms 
proof (induct m arbitrary: nell n)
 case 0
 then show ?case 
  proof (cases n rule: enat_coexhaust)
   case 0
   then show ?thesis
   using "0.prems" 
   by (metis nellist.case(2) nellist.collapse(1) nellist.exhaust_sel nfirst_def 
        nnth_0_conv_nhd nnth_NNil ntake_eq_NNil_iff)
   next
   case (eSuc n')
   then show ?thesis 
   by (simp add: nellist.case_eq_if nnth_0_conv_nhd ntake_Suc)
  qed
next
case (Suc m)
then show ?case 
  proof (cases n rule: enat_coexhaust)
   case 0
   then show ?thesis 
   using Suc.prems by (simp add: enat_0_iff(1))
   next
   case (eSuc n')
   then show ?thesis 
   proof (cases nell)
    case (NNil x1)
    then show ?thesis by simp
    next
    case (NCons x21 x22)
    then show ?thesis 
    using Suc.hyps Suc.prems Suc_ile_eq eSuc by force
   qed
  qed
qed


subsection \<open>@{term "ntaken"}\<close>

lemma ntaken_NNil [simp, code, nitpick_simp]:
 " ntaken n (NNil b) = (NNil b)"
by transfer
   (metis eSuc_enat llist.disc(2) ltake_LNil ltake_eSuc_LCons)

lemma ntaken_0 [simp]:
 " ntaken 0 nell = (NNil (nfirst nell))"
proof (cases nell)
 case (NNil x1)
 then show ?thesis by (metis ntake_0 ntake_NNil ntaken_NNil)
 next
 case (NCons x21 x22)
 then show ?thesis 
 by transfer (simp, (metis One_nat_def ltake_0 ltake_eSuc_LCons one_eSuc one_enat_def zero_neq_one))
qed

lemma ntaken_Suc_NCons [simp]:
 " ntaken (Suc n) (NCons x nell) = (NCons x (ntaken n nell))"
by transfer (auto simp add: zero_enat_def, metis eSuc_enat ltake_eSuc_LCons)

lemma ntaken_Suc:
 " ntaken (Suc n) nell = 
   (case nell of (NNil b) \<Rightarrow> (NNil b) | (NCons x nell') \<Rightarrow> (NCons x (ntaken n nell')))"
by (cases nell) simp_all

lemma is_NNil_ntaken [simp]:
 " is_NNil(ntaken n nell) \<longleftrightarrow> is_NNil nell \<or> n=0"
proof (cases nell)
 case (NNil x1)
 then show ?thesis 
 by simp
 next
 case (NCons x nell1)
 then show ?thesis 
 proof (cases n)
  case 0
  then show ?thesis 
  by simp
  next
  case (Suc nat)
  then show ?thesis 
  by (simp add: NCons)
 qed
qed

lemma ntaken_eq_NNil_iff [simp]:
 " ntaken n nell = (NNil x) \<longleftrightarrow> nell = (NNil x) \<or> (n = 0 \<and> nfirst nell = x) "
proof (cases nell)
 case (NNil x1)
 then show ?thesis 
 by (metis ntake_0 ntake_NNil ntaken_NNil)
 next
 case (NCons x nell1)
 then show ?thesis 
 proof (cases n)
 case 0
 then show ?thesis 
 by (simp add: NCons)
 next
 case (Suc nat)
 then show ?thesis 
 using NCons by force
 qed
qed

lemma NNil_eq_ntaken_iff [simp]:
 " (NNil x) = ntaken n nell   \<longleftrightarrow> nell = (NNil x) \<or> (n = 0 \<and> nfirst nell = x) "
by (metis ntaken_eq_NNil_iff)

lemma ntaken_NCons [code, nitpick_simp]:
 " ntaken n (NCons x nell) = (case n of 0 \<Rightarrow> (NNil x) | (Suc n') \<Rightarrow> (NCons x (ntaken n' nell)) )"
by (cases n) (auto simp add: nfirst_def)

lemma nhd_ntaken [simp]:
 " n \<noteq> 0 \<Longrightarrow> nhd(ntaken n nell) = nhd nell"
by (cases nell)
   (simp_all add: Nitpick.case_nat_unfold ntaken_NCons)

lemma ntl_ntaken:
 " n \<noteq> 0 \<Longrightarrow> ntl(ntaken n nell) = ntaken (n-1) (ntl nell)"
by simp_all
   (metis Suc_pred nellist.exhaust_sel nellist.sel(4) nellist.sel(5) ntaken_NNil ntaken_Suc_NCons)

lemma ntl_ntaken_0:
 " ntl(ntaken 0 nell) = (NNil (nfirst nell))"
by simp

lemma ntaken_ntl:
 " ntaken n (ntl nell) = ntl(ntaken (Suc n) nell)"
by (simp add: enat_0_iff(1) ntl_ntaken)

lemma ntaken_nlength [simp]:
 " nlength (ntaken n nell) = min n (nlength nell)"
by transfer simp

lemma ntaken_nmap [simp]:
 " ntaken n (nmap f nell) = nmap f (ntaken n nell)"
using enat_0_iff(2) by transfer auto

lemma ntaken_ntaken [simp]:
 " ntaken n (ntaken m nell) = ntaken (min n m) nell"
using enat_0_iff(2) by transfer auto

lemma nset_ntaken:
 " nset (ntaken n nell) \<subseteq> nset nell"
by transfer (simp add: lset_ltake)

lemma ntaken_all:
 "nlength nell \<le> m \<Longrightarrow>  ntaken m nell = nell"
by transfer
   (auto simp add: zero_enat_def, metis eSuc_enat eSuc_epred eSuc_ile_mono llength_eq_0 ltake_all)

lemma ntaken_nnth:
 shows  " (nnth (ntaken m nell) k) = (nnth nell (min k m))"
apply transfer
by (auto simp add: min_def lnth_ltake ltake_all) 
   (metis co.enat.sel(2) enat_eSuc_iff enat_ord_simps(1) epred_le_epredI order_trans, 
    metis Suc_ile_eq co.enat.exhaust_sel iless_Suc_eq llength_eq_0,
    metis Suc_ile_eq co.enat.exhaust_sel iless_Suc_eq llength_eq_0,
    meson dual_order.strict_iff_order enat_ord_simps(2) linorder_not_less order_less_le_trans,
    metis Suc_ile_eq co.enat.exhaust_sel iless_Suc_eq llength_eq_0)


lemma nfinite_ntaken [simp]:
 " nfinite (ntaken n nell) "
by transfer simp

lemma ntaken_nlast:
 " nlast (ntaken n nell) = nnth nell n"
using nnth_nlast[of "ntaken n nell"] ntaken_nlength[of n nell] 
by (metis min.absorb3 min.idem min.orderI nfinite_ntaken nnth_beyond not_less_iff_gr_or_eq 
    ntaken_all ntaken_nnth the_enat.simps)

lemma ntaken_nfirst:
 " nfirst (ntaken n nell) = nfirst nell" 
by transfer (simp add: enat_0_iff(1))

lemma ntaken_nappend1:
 " n \<le> nlength nellx \<Longrightarrow> ntaken n (nappend nellx nelly) = ntaken n nellx"
by transfer
   (simp add: zero_enat_def, metis eSuc_enat eSuc_epred eSuc_ile_mono llength_eq_0 ltake_lappend1)

lemma ntaken_nappend2:
 " nlength nellx < (enat n) \<Longrightarrow>
   ntaken n (nappend nellx nelly) = nappend nellx (ntaken (n - (the_enat(nlength nellx)) -1) nelly) "
proof (induct n arbitrary: nellx nelly)
case 0
then show ?case using zero_enat_def by auto
next
case (Suc n)
then show ?case 
 proof (cases nellx)
 case (NNil x1)
 then show ?thesis 
 by simp
 next
 case (NCons x21 x22)
 then show ?thesis using Suc by simp
  (metis Extended_Nat.eSuc_mono eSuc_enat enat_ord_code(4) order_less_imp_not_less the_enat_eSuc) 
 qed
qed
     
lemma ntaken_eq_ntaken_antimono:
 " \<lbrakk> ntaken n nellx = ntaken n nelly; m \<le> n \<rbrakk> \<Longrightarrow> ntaken m nellx = ntaken m nelly"
by (metis min.orderE ntaken_ntaken)

lemma ntake_eq_ntaken: 
assumes " (enat k) = m "  
shows "ntake m nell = ntaken k nell" 
using assms apply transfer
using eSuc_enat by auto



subsection \<open>Concatenating a nonempty lazy list of nonempty lazy lists @{term "nconcat"}\<close>

lemma nconcat_NNil [simp]: 
 " nconcat (NNil nell) = nell " 
by transfer auto

lemma nconcat_NCons [simp]: 
 " nconcat (NCons nell nells) = nappend nell (nconcat nells) " 
by transfer auto

lemma nconcat_def2: 
"nconcat = nellist_of_llist \<circ> lconcat \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist) " 
by (simp add: map_fun_def nconcat_def)

lemma not_null_lconcat: 
 "\<not>lnull((lconcat \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist)) nells)" 
by (simp add: llist_of_nellist.code)

lemma nconcat_def3:
" ((llist_of_nellist \<circ> nconcat) nells) =
  ((lconcat \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist)) nells) " 
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "llist_of_nellist \<circ> nconcat = 
          llist_of_nellist \<circ> nellist_of_llist \<circ> lconcat \<circ> ?n2l" 
   by (simp add: nconcat_def2 rewriteL_comp_comp)
 have 2: " \<not>lnull((lconcat \<circ> ?n2l) nells) "
     using not_null_lconcat by blast
 have 3: "(llist_of_nellist \<circ> nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells =
          (lconcat \<circ> ?n2l) nells" 
   using 2 nellist_of_llist_inverse by simp
 show ?thesis 
 by (metis "1" "3")
qed  

lemma nmap_lmap_inverse: 
 "nmap nellist_of_llist (nellist_of_llist  (lmap llist_of_nellist (llist_of_nellist nells))) = nells "
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nmap nellist_of_llist (nellist_of_llist  (?n2l nells)) = 
          nmap nellist_of_llist (nmap llist_of_nellist nells)" 
    by  (simp add: lmap_llist_of_nellist)
 have 2: "nmap nellist_of_llist (nmap llist_of_nellist nells) =
          nmap (nellist_of_llist\<circ>llist_of_nellist) nells"
    by (simp add: nellist.map_comp) 
 have 3: "(nellist_of_llist\<circ>llist_of_nellist) = id"
   by auto
 show ?thesis 
 by (metis "1" "2" "3" comp_apply nellist.map_id)
qed

lemma lmap_nmap_inverse: 
assumes "\<not> lnull nells "
        "\<forall>nell\<in>lset nells. \<not> lnull nell"  
 shows "(lmap llist_of_nellist (llist_of_nellist (nmap nellist_of_llist (nellist_of_llist nells)))) = nells"
proof -
 let ?l2n = "nmap nellist_of_llist \<circ> nellist_of_llist" 
 have 0:" (nmap nellist_of_llist (nellist_of_llist nells)) =
          nellist_of_llist (lmap nellist_of_llist nells)" 
    using nmap_nellist_of_llist assms  by simp
 have 00: "(llist_of_nellist (nellist_of_llist (lmap nellist_of_llist nells))) =
           (lmap nellist_of_llist nells)" 
    using assms by simp
 have 1: "(lmap llist_of_nellist (llist_of_nellist (?l2n nells)) ) = 
          (lmap llist_of_nellist ( (  (lmap nellist_of_llist nells))))  "
   using "0" "00" by auto
 have 2: "(lmap llist_of_nellist ( (  (lmap nellist_of_llist nells)))) =
          (lmap (llist_of_nellist\<circ>nellist_of_llist) nells)" 
    using llist.map_comp by blast
 have 3: "\<forall>nell \<in> lset nells.  (llist_of_nellist\<circ>nellist_of_llist) nell  = nell"
    using assms by simp
 have 4: "(lmap (llist_of_nellist\<circ>nellist_of_llist) nells) = nells"
    using 3  by auto
 show ?thesis 
 using "1" "2" "4" "0" "00" by presburger
qed


lemma nconcat_expand: 
 " nconcat nells = 
  (if is_NNil nells then nfirst nells else nappend (nfirst nells) (nconcat (ntl nells))) " 
by (metis nconcat_NCons nconcat_NNil nellist.case_eq_if nellist.collapse(1) nellist.collapse(2) 
    nfirst_def)

lemma lmap_llist_of_nellist_nmap: 
"(lmap (lmap f) (lmap llist_of_nellist (llist_of_nellist nells))) = 
 (lmap llist_of_nellist (lmap (nmap f) (llist_of_nellist nells)))" 
proof -
 have 5: "(lmap (lmap f) (lmap llist_of_nellist (llist_of_nellist nells))) =
          (lmap ((lmap f) \<circ> llist_of_nellist) (llist_of_nellist nells))"
      using llist.map_comp by blast 
 have 6: "(lmap ((lmap f) \<circ> llist_of_nellist) (llist_of_nellist nells)) =
          (lmap (llist_of_nellist \<circ> (nmap f)) (llist_of_nellist nells))" 
    by (simp add: lmap_llist_of_nellist)
 have 7: "(lmap (llist_of_nellist \<circ> (nmap f)) (llist_of_nellist nells)) =
          (lmap llist_of_nellist (lmap (nmap f) (llist_of_nellist nells)))  "
      using  llist.map_comp[of llist_of_nellist "(nmap f)" "(llist_of_nellist nells)"] 
    by presburger
 show ?thesis 
 using "5" "6" "7" by presburger
qed

lemma nmap_nconcat :
 " nmap f (nconcat nells) = nconcat (nmap (nmap f) nells) " 
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nmap f (nconcat nells) = 
         nmap f ((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells) " 
   by (simp add: nconcat_def2)
 have 2: " nmap f ((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells) =
           nellist_of_llist (lmap f ((lconcat \<circ> ?n2l) nells))"
  using nmap_nellist_of_llist 
    using not_null_lconcat by fastforce
 have 3: "(lmap f ((lconcat \<circ> ?n2l) nells)) =
          lconcat (lmap (lmap f) (?n2l nells))"
    by (simp add: lmap_lconcat) 
 have 4: "(nconcat (nmap ( nmap f ) nells)) =
          (nellist_of_llist \<circ> lconcat \<circ> ?n2l) (nmap ( nmap f ) nells)"
    by (simp add: nconcat_def2) 
 have 8: "(lmap (lmap f) (?n2l nells)) = 
          (lmap llist_of_nellist (lmap (nmap f) (llist_of_nellist nells)))" 
  using lmap_llist_of_nellist_nmap by auto
 show ?thesis 
 using "1" "2" "3" "4" "8" 
   by (metis (no_types, opaque_lifting) comp_eq_dest_lhs llist.map_disc_iff llist_of_nellist_inverse_b
       llist_of_nellist_not_lnull lmap_lconcat lmap_llist_of_nellist_nmap nconcat_def3 
       nellist_of_llist_inverse nmap_nellist_of_llist)
qed

lemma nconcat_eq_NNil: 
  "nconcat nells = (NNil x) \<longleftrightarrow> nells = (NNil (NNil x))"
by (metis is_NNil_def is_NNil_nappend nconcat_NNil nconcat_expand)

lemma nhd_nconcat [simp]:
  "\<lbrakk> \<not> is_NNil nells; \<not> is_NNil (nhd nells) \<rbrakk> \<Longrightarrow> nhd (nconcat nells) = nhd (nhd nells)"
by (metis nconcat_NCons nellist.collapse(2) nhd_nappend)

lemma ntl_nconcat [simp]:
  "\<lbrakk> \<not> is_NNil nells; \<not> is_NNil (nhd nells) \<rbrakk> \<Longrightarrow>
      ntl (nconcat nells) = nappend (ntl (nhd nells)) (nconcat (ntl nells))"
by (metis nconcat_NCons nellist.collapse(2) ntl_nappend)

lemma nconcat_nappend [simp]:
  assumes "nfinite nells"
  shows "nconcat (nappend nells nells1) = nappend (nconcat nells) (nconcat nells1)"
using assms
by (induct rule:nfinite_induct) (simp_all add: nappend_assoc)


lemma nconcat_eq_NCons_conv:
  "nconcat nells = NCons x nell \<longleftrightarrow> 
     nells = (NNil (NCons x nell)) \<or>
     (\<exists> nells'. nells = (NCons (NNil x) nells') \<and> nell = nconcat nells') \<or>
     (\<exists> nell' nells''. nells = (NCons (NCons x nell') nells'') \<and> nell = nappend nell' (nconcat nells'') ) " 
proof (cases nells)
case (NNil nell1)
then show ?thesis by simp
next
case (NCons nell nell2)
then show ?thesis 
 proof (cases "is_NNil nell")
 case True
 then show ?thesis 
  using NCons by simp
  (metis nappend_NCons nappend_NNil nellist.collapse(1) nellist.sel(3) nellist.sel(5))
 next
 case False
 then show ?thesis 
  using NCons by simp 
  (metis nappend_NCons nellist.collapse(2) nellist.distinct(1) nellist.sel(3) nellist.sel(5))
 qed
qed

lemma nlength_nconcat:
  shows "nlength (nconcat nells) = 
         (case nells of (NNil nell) \<Rightarrow> nlength nell |   
                (NCons nell nells1) \<Rightarrow> eSuc(nlength nell) + nlength (nconcat nells1))   "
proof (cases nells)
case (NNil nell1)
then show ?thesis by simp
next
case (NCons nell nell2)
then show ?thesis by (simp add: eSuc_plus plus_1_eSuc(2))
qed


lemma nlength_nconcat_nfinite_conv_sum:
  assumes "nfinite nells"
  shows "nlength (nconcat nells) = 
         nlength nells + (\<Sum>i = 0.. (the_enat (nlength nells)). nlength (nnth nells i))"
using assms
proof(induct rule: nfinite_induct)
case (NNil y)
then show ?case by (simp add: zero_enat_def) (simp add: enat_0_iff(2) nnth_NNil)
next
case (NCons nell nells)
then show ?case 
   proof -
    have 1: "nlength (nconcat (NCons nell nells)) = 1+ nlength nell + nlength (nconcat nells)"
      by simp
    have 2: "nlength (nconcat nells) = 
             nlength nells + (\<Sum>i = 0.. (the_enat (nlength nells)). nlength (nnth nells i))"
       using NCons.hyps(2) by blast 
    have 3: "nlength (NCons x nells) = 1+ nlength nells "
     by (simp add: plus_1_eSuc(1)) 
    have 4: "(\<Sum>i = 0..the_enat (nlength (NCons nell nells)). nlength (nnth (NCons nell nells) i)) =
             nlength (nnth (NCons nell nells) 0) + 
             (\<Sum>i = 1..the_enat (nlength (NCons nell nells)). nlength (nnth (NCons nell nells) i))" 
       by (simp add: sum.atLeast_Suc_atMost)
    have 5: "(\<Sum>i = 1..the_enat (nlength (NCons nell nells)). nlength (nnth (NCons nell nells) i)) =
             (\<Sum>i = 1..(Suc (the_enat (nlength ( nells)))). nlength (nnth (NCons nell nells) i))" 
        using NCons.hyps(1) eSuc_enat nfinite_nlength_enat by fastforce
    have 6: "(\<Sum>i = 1..(Suc (the_enat (nlength ( nells)))). nlength (nnth (NCons nell nells) i)) =
             (\<Sum>i = 0..( (the_enat (nlength ( nells)))). nlength (nnth (NCons nell nells) (Suc i)))" 
       using sum.shift_bounds_cl_nat_ivl[of "\<lambda> i. nlength (nnth (NCons nell nells) i)" 0 1  
       "( (the_enat (nlength ( nells))))"]
          by simp
    have 7: "(\<Sum>i = 0..( (the_enat (nlength ( nells)))). nlength (nnth (NCons nell nells) (Suc i))) =
             (\<Sum>i = 0..( (the_enat (nlength ( nells)))). nlength (nnth ( nells) ( i))) "
       by auto
    show ?thesis 
    using "2" "3" "4" "5" "6" by force  
  qed    
qed

lemma nlength_nconcat_nfinite_conv_sum_alt: 
 assumes "nfinite nells"
 shows " nlength nells + (\<Sum>i = 0.. (the_enat (nlength nells)). nlength (nnth nells i)) =
          epred(\<Sum>i = 0.. (the_enat (nlength nells)). eSuc (nlength (nnth nells i))) "
using assms
proof(induct rule: nfinite_induct)
case (NNil y)
then show ?case by simp
next
case (NCons nell nells)
then show ?case 
  proof -
   have 1: "(\<Sum>i = 0..the_enat (nlength (NCons nell nells)). nlength (nnth (NCons nell nells) i)) =
             nlength (nnth (NCons nell nells) 0) + 
             (\<Sum>i = 1..the_enat (nlength (NCons nell nells)). nlength (nnth (NCons nell nells) i))" 
       by (simp add: sum.atLeast_Suc_atMost)
   have 2: "(\<Sum>i = 1..the_enat (nlength (NCons nell nells)). nlength (nnth (NCons nell nells) i)) =
             (\<Sum>i = 1..(Suc (the_enat (nlength ( nells)))). nlength (nnth (NCons nell nells) i))" 
      using NCons.hyps(1) eSuc_enat nfinite_nlength_enat by fastforce
   have 3: "(\<Sum>i = 1..(Suc (the_enat (nlength ( nells)))). nlength (nnth (NCons nell nells) i)) =
             (\<Sum>i = 0..( (the_enat (nlength ( nells)))). nlength (nnth (NCons nell nells) (Suc i)))" 
       using sum.shift_bounds_cl_nat_ivl[of "\<lambda> i. nlength (nnth (NCons nell nells) i)" 0 1  
          "( (the_enat (nlength ( nells))))"]
          by simp
   have 4: "(\<Sum>i = 0..( (the_enat (nlength ( nells)))). nlength (nnth (NCons nell nells) (Suc i))) =
             (\<Sum>i = 0..( (the_enat (nlength ( nells)))). nlength (nnth ( nells) ( i))) "
       by auto
   have 5: "(\<Sum>i = 0.. (the_enat (nlength (NCons nell nells))). eSuc (nlength (nnth (NCons nell nells) i))) =
            eSuc (nlength (nnth (NCons nell nells) 0)) +
            (\<Sum>i = 1.. (the_enat (nlength (NCons nell nells))). eSuc (nlength (nnth (NCons nell nells) i))) " 
     by (simp add: sum.atLeast_Suc_atMost)
   have 6: "(\<Sum>i = 1.. (the_enat (nlength (NCons nell nells))). eSuc (nlength (nnth (NCons nell nells) i))) =
            (\<Sum>i = 1.. (Suc (the_enat (nlength ( nells)))). eSuc (nlength (nnth (NCons nell nells) i))) "
       using NCons.hyps(1) eSuc_enat nfinite_nlength_enat by fastforce
   have 7: " (\<Sum>i = 1.. (Suc (the_enat (nlength ( nells)))). eSuc (nlength (nnth (NCons nell nells) i))) =
             (\<Sum>i = 0.. ( (the_enat (nlength ( nells)))). eSuc (nlength (nnth (NCons nell nells) (Suc i)))) "
      using sum.shift_bounds_cl_nat_ivl[of "\<lambda> i. eSuc(nlength (nnth (NCons nell nells) i))" 0 1 
         "( (the_enat (nlength ( nells))))"]
         by simp
   have 8: "(\<Sum>i = 0.. ( (the_enat (nlength ( nells)))). eSuc (nlength (nnth (NCons nell nells) (Suc i)))) =
            (\<Sum>i = 0.. ( (the_enat (nlength ( nells)))). eSuc (nlength (nnth ( nells) ( i)))) "
      by auto 
   have 9: "(\<Sum>i = 0.. (the_enat (nlength (NCons nell nells))). eSuc (nlength (nnth (NCons nell nells) i))) =
            ( eSuc(nlength nell) + 
                   (\<Sum>i = 0.. ( (the_enat (nlength ( nells)))). eSuc (nlength (nnth ( nells) ( i))))) "
       by (metis "5" "6" "7" "8" nnth_0)  
   have 10: "epred(\<Sum>i = 0.. (the_enat (nlength (NCons nell nells))). eSuc (nlength (nnth (NCons nell nells) i))) =
            epred( eSuc(nlength nell) + 
                   (\<Sum>i = 0.. ( (the_enat (nlength ( nells)))). eSuc (nlength (nnth ( nells) ( i)))))" 
       using "9" by presburger
   have 11: "epred( eSuc(nlength nell) + 
                   (\<Sum>i = 0.. ( (the_enat (nlength ( nells)))). eSuc (nlength (nnth ( nells) ( i))))) =
             eSuc(nlength nell) + 
             epred(\<Sum>i = 0.. ( (the_enat (nlength ( nells)))). eSuc (nlength (nnth ( nells) ( i)))) " 
       by (metis add.commute eSuc_ne_0 epred_iadd1 le_add1 le_add_same_cancel1 sum.last_plus 
           zero_eq_add_iff_both_eq_0)
   show ?thesis 
   using "1" "11" "2" "3" "9" NCons.hyps(2) eSuc_plus by fastforce
 qed   
qed
 

lemma nset_nconcat_nfinite:
assumes  "\<forall>xs \<in> nset nells. nfinite xs"
shows " nset (nconcat nells) = (\<Union>xs\<in>nset nells. nset xs)"
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nset (nconcat nells) = 
          nset (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)) " 
   by (simp add: nconcat_def2)
 have 2: "nset (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)) =
          lset (llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)))"
    by (metis lset_llist_of_nellist_a)
 have 3: "(llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells))) =
            ( ((( lconcat \<circ> ?n2l) nells))) "
    using nellist_of_llist_inverse not_null_lconcat by fastforce 
 have 4: "lset (llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells))) =
          lset ( ((( lconcat \<circ> ?n2l) nells)))" 
   using "3" by presburger
 have 5: "\<forall>nell \<in> lset (?n2l nells) . lfinite nell" 
     using assms nfinite_def by fastforce
 have 6: "lset ( ((( lconcat \<circ> ?n2l) nells))) = (\<Union>nell\<in> lset (?n2l nells). lset nell) "
    by (simp add: "5" lset_lconcat_lfinite)
 have 7: "(\<Union>nell\<in> lset (?n2l nells). lset nell) = \<Union> (nset ` nset nells)" 
   by simp
  show ?thesis 
  by (metis "6" "7" lset_llist_of_nellist_a nconcat_def3 o_apply)
qed


lemma nconcat_ntake:
shows  "nconcat (ntake (enat n) nells) = 
        ntake (n +(\<Sum>i=0..n. nlength (nnth nells i))) (nconcat nells)"
proof(induct n arbitrary: nells)
  case 0 thus ?case 
     proof (cases nells)
     case (NNil nell)
     then show ?thesis by (simp add: zero_enat_def[symmetric]  nnth_NNil ntake_all)
     next
     case (NCons nell nell2)
     then show ?thesis by (simp add: zero_enat_def[symmetric])
         (metis dual_order.refl nlast_NNil nnth_0 ntake_all ntake_nappend1 ntaken_0 ntaken_nlast)
     qed
next
  case (Suc n)
  show ?case
  proof (cases nells)
  case (NNil nell)
  then show ?thesis 
    by (metis (no_types, lifting) dual_order.trans enat_le_plus_same(1) enat_le_plus_same(2) nconcat_NNil
        nfinite_NNil nlast_NNil nlength_NNil nnth_nlast ntake_NNil ntake_all sum.atLeast0_atMost_Suc_shift
        the_enat_0)
  next
  case (NCons nell nells1)
  then show ?thesis 
     proof -
      have 1: "nconcat (ntake (enat (Suc n)) nells) =
               nappend nell (nconcat (ntake (enat n) nells1)) " 
          by (metis NCons eSuc_enat nconcat_NCons ntake_Suc_NCons)
      have 2: "(nconcat (ntake (enat n) nells1)) =
               ntake (enat n + (\<Sum>i = 0..n. nlength (nnth nells1 i))) (nconcat nells1)"
         using NCons Suc.hyps Suc.prems Suc_ile_eq by auto
      have 3: " nlength (nappend nell (nconcat nells1)) =
                nlength nell + 1 + nlength (nconcat nells1)"
         by simp 
      have 4: "nappend nell (ntake (enat n + (\<Sum>i = 0..n. nlength (nnth nells1 i))) (nconcat nells1)) =
               ntake (nlength nell + 1 + (enat n + (\<Sum>i = 0..n. nlength (nnth nells1 i)))) (nappend nell (nconcat nells1))"
        proof (cases "nlength nell = \<infinity>")
        case True
        then show ?thesis by (metis enat_le_plus_same(2) ntake_all ntake_nappend1 plus_enat_simps(2))
        next
        case False
        then show ?thesis by (simp add: ab_semigroup_add_class.add_ac(1) enat_0_iff(1) ntake_nappend2)
        qed  
      have 5: "(\<Sum>i = 0..Suc n. nlength (nnth nells i)) =
               nlength (nnth nells 0) + (\<Sum>i = 1..Suc n. nlength (nnth nells i))"  
             by (simp add: sum.atLeast_Suc_atMost) 
      have 6: "nlength (nnth nells 0) = nlength nell"
        by (simp add: NCons) 
      have 7: "(\<Sum>i = 1..Suc n. nlength (nnth nells i)) =
               (\<Sum>i = 0..n. nlength (nnth nells (Suc i)))"
         using sum.shift_bounds_cl_nat_ivl[of "\<lambda>i . nlength (nnth nells i)" 0 1 n]
          by simp
      have 8: "(\<Sum>i = 0..n. nlength (nnth nells (Suc i))) =
               (\<Sum>i = 0..n. nlength (nnth nells1 ( i)))"
        using NCons by auto  
      have 9: "nlength nell + 1 + (enat n + (\<Sum>i = 0..n. nlength (nnth nells1 i))) =
               (enat (Suc n) + (\<Sum>i = 0..Suc n. nlength (nnth nells i)))"
        by (metis (no_types, lifting) "5" "6" "7" "8" ab_semigroup_add_class.add_ac(1) 
           add.left_commute eSuc_enat plus_1_eSuc(2)) 
     show ?thesis 
     by (metis "1" "2" "4" "9" NCons nconcat_NCons)
   qed
  qed
qed

lemma nnth_nconcat_conv:
  assumes "enat n \<le> nlength (nconcat nells)"
  shows "\<exists>m n'. nnth (nconcat nells) n = nnth (nnth nells m) n' \<and> enat n' \<le> nlength (nnth nells m) \<and>
                enat m \<le> nlength nells \<and>
                enat n =  (\<Sum> i<m . eSuc(nlength (nnth nells i))) + enat n'"
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "\<And>nell. nlength nell = epred (llength (llist_of_nellist nell))" 
   unfolding nlength_def by auto
 have 2: "\<And>nell j. j\<le> nlength nell \<longrightarrow> nnth nell j = lnth (llist_of_nellist nell) j"
   unfolding nnth_def  by auto 
 have 3: "nlength (nconcat nells) =
          nlength (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)) " 
   by (simp add: nconcat_def2)
 have 4: "nlength (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)) =
          epred (llength (llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)))) "
    using nlength.rep_eq by blast
 have 5:  "(llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells))) =
            ( ((( lconcat \<circ> ?n2l) nells))) "
    using nellist_of_llist_inverse not_null_lconcat by fastforce 
 have 6: " epred (llength (llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)))) =
           epred (llength ((( lconcat \<circ> ?n2l) nells)))"
     using "5" by presburger  
 have 7: "enat n < (llength ((( lconcat \<circ> ?n2l) nells)))"
   by (metis (no_types, lifting) "3" "4" "5" assms co.enat.collapse iless_Suc_eq llength_eq_0 
       llist_of_nellist_not_lnull)
 have 8: "((( lconcat \<circ> ?n2l) nells)) = 
          lconcat (?n2l nells)" 
    by simp
 have 9: "\<exists>m n'.
           lnth (lconcat (?n2l nells)) n = lnth (lnth (?n2l nells) m) n' \<and>
            enat n' < llength (lnth (?n2l nells) m) \<and> enat m < llength (?n2l nells) \<and>
            enat n = (\<Sum>i<m. llength (lnth (?n2l nells) i)) + enat n'" 
   using 7 lnth_lconcat_conv[of n "(?n2l nells)"]
     by fastforce
 obtain m n' where 10: "lnth (lconcat (?n2l nells)) n = lnth (lnth (?n2l nells) m) n' \<and>
         enat n' < llength (lnth (?n2l nells) m) \<and> enat m < llength (?n2l nells) \<and>
         enat n = (\<Sum>i<m. llength (lnth (?n2l nells) i)) + enat n'"
    using 9 by blast
 have 11: "lnth (lconcat (?n2l nells)) n = nnth (nconcat nells) n"
    by (metis "2" "5" "8" assms nconcat_def2)  
 have 12: "lnth (lnth (?n2l nells) m) n' = nnth (nnth nells m) n'"
    by (metis "10" "2" comp_apply co.enat.collapse iless_Suc_eq llength_eq_0 llength_lmap 
        llist_of_nellist_not_lnull lnth_lmap nlength.rep_eq) 
 have 13: "llength (lnth (?n2l nells) m) = eSuc (nlength (nnth nells m)) "     
    by (metis "10" "2" comp_apply co.enat.collapse iless_Suc_eq llength_eq_0 llength_lmap 
        llist_of_nellist_not_lnull lnth_lmap nlength.rep_eq)
 have 14: "llength (?n2l nells) = eSuc(nlength nells)" 
   by (metis comp_apply co.enat.exhaust_sel llength_eq_0 llength_llist_of_nellist llength_lmap 
       llist_of_nellist_not_lnull)
 have 15: "\<And>i. i<m \<Longrightarrow> llength (lnth (?n2l nells) i) = eSuc(nlength(nnth nells i))"
     proof -
       fix i
       assume a: "i<m" 
       show "llength (lnth (?n2l nells) i) = eSuc(nlength(nnth nells i))" 
       proof -
        have 151: "(lnth (?n2l nells) i) = llist_of_nellist (nnth nells i)"  
        using a  10 14 2
        by (metis comp_apply enat_ord_simps(2) iless_Suc_eq llength_lmap lnth_lmap order_less_trans)
        have 152: "llength (llist_of_nellist (nnth nells i)) = eSuc(nlength(nnth nells i))" 
          using epred_inject by force
        show ?thesis using "151" "152" by presburger
        qed
      qed
 have 16: "(\<Sum>i<m. llength (lnth (?n2l nells) i)) = (\<Sum>i<m. eSuc(nlength(nnth nells i)))"
   by (meson "15" lessThan_iff sum.cong) 
 show ?thesis 
 using "10" "11" "12" "13" "14" "16" by (metis iless_Suc_eq)
qed

lemma nnth_nconcat_ntake:
  assumes "enat w \<le> nlength (nconcat (ntake (enat n) nells))"
  shows "nnth (nconcat (ntake (enat n) nells)) w = nnth (nconcat nells) w"
using assms by (simp add: nconcat_ntake ntake_nnth)

lemma nfinite_nconcat [simp]:
  "nfinite (nconcat nells) \<longleftrightarrow> nfinite nells \<and> (\<forall>nell \<in> nset nells. nfinite nell)"
  (is "?lhs \<longleftrightarrow> ?rhs")
proof
 assume "?lhs"
  thus "?rhs" (is "?concl nells")
  proof(induct "nconcat nells" arbitrary: nells rule: nfinite_induct)
  case (NNil y)
  then show ?case 
  by (metis is_NNil_imp_nfinite nconcat_eq_NNil nellist.discI(1) nellist.simps(20) singleton_iff)
  next
  case (NCons x nell)
  then show ?case 
    proof (cases nells)
    case (NNil nell1)
    then show ?thesis using NCons.hyps by auto
    next
    case (NCons nell1 nells1)
    then show ?thesis using NCons.hyps by simp 
     (metis nconcat_NCons nellist.sel(5) nellist.set_intros(3) nfinite_NCons nfinite_nappend ntl_nappend)
    qed
  qed
next
  assume "?rhs"
  then obtain "nfinite ( nells)"
    and "\<forall>nell\<in>nset nells. nfinite nell" ..
  thus ?lhs
  proof(induct nells rule: nfinite_induct)
  case (NNil nell)
  then show ?case
  by simp
  next
  case (NCons nell nells)
  then show ?case 
  by (simp add: nfinite_nappend)
  qed
qed

lemma nfilter_nconcat_nfinite_help: 
assumes "(\<forall>nell \<in> nset nells. (\<exists> x \<in> nset nell. P x))"
shows  "(\<exists>nell \<in> nset (nconcat nells). P nell) " 
proof (cases nells)
case (NNil nell)
then show ?thesis using assms by simp 
next
case (NCons nell nells1)
then show ?thesis using assms 
  by simp (metis dual_order.trans enat_le_plus_same(1) in_nset_conv_nnth nlength_nappend nnth_nappend1)
qed

lemma nfilter_nconcat_nfinite:
assumes  "\<forall>nell\<in>nset nells. nfinite nell "
         "\<forall>nell \<in> nset nells. (\<exists> x \<in> nset nell. P x)" 
 shows "  nfilter P (nconcat nells) = nconcat (nmap (nfilter P) nells)"
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 0: "\<exists>nell \<in> nset (nconcat nells). P nell " 
   using assms nfilter_nconcat_nfinite_help by blast
 have 1: "nset nells = lset(llist_of_nellist nells) " 
   by simp
 have 2: "\<And>nell. nell \<in> lset(llist_of_nellist nells)\<longrightarrow> lfinite (llist_of_nellist nell)"
  using "1" assms nfinite_def by blast 
 have 3: "\<And>nell Q. (\<exists>x\<in>nset nell. Q x) \<longrightarrow>
                   nfilter Q nell = nellist_of_llist (lfilter Q (llist_of_nellist nell)) " 
     unfolding nfilter_def by simp
 have 4: "nfilter P (nconcat nells) =
          nfilter P (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)) "
     by (simp add: nconcat_def2)
 have 5: "nfilter P (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)) =
          nellist_of_llist (lfilter P (llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells))))"
    by (metis "3" 0 nconcat_def2) 
 have 6: "(llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells))) =
          ( ((( lconcat \<circ> ?n2l) nells))) " 
   using nellist_of_llist_inverse not_null_lconcat by fastforce
 have 7: "(lfilter P (llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)))) =
          (lfilter P ( ((( lconcat \<circ> ?n2l) nells))))" 
   using "6" by presburger
 have 8: "((( lconcat \<circ> ?n2l) nells)) = lconcat (?n2l nells)"
       by simp 
 have 9: "\<forall>nell\<in>lset (?n2l nells). lfinite nell" 
    by (simp add: "2")
 have 10: " (lfilter P ( ((( lconcat \<circ> ?n2l) nells)))) = lconcat (lmap (lfilter P)  (  ?n2l nells))" 
   by (simp add: "9" lfilter_lconcat_lfinite)
 have 11: "nconcat (nmap (nfilter P) nells) =
           (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) (nmap (nfilter P) nells)))"
     by (simp add: nconcat_def2) 
 have 12: "\<And>nell f. nmap f nell = nellist_of_llist (lmap f (llist_of_nellist nell)) "
   by (metis llist_of_nellist_inverse_b llist_of_nellist_not_lnull nmap_nellist_of_llist)
 have 13: " (nmap (nfilter P) nells) = nellist_of_llist (lmap (nfilter P)  (llist_of_nellist nells))"
   using "12" by blast
 have 14: "nellist_of_llist (lmap (nfilter P)  (llist_of_nellist nells)) =
           nellist_of_llist (lmap (\<lambda>ys. nellist_of_llist (lfilter P (llist_of_nellist ys)))  (llist_of_nellist nells))" 
  by (metis (mono_tags, lifting) "1" "3" assms(2) llist.map_cong)
 have 15: "lmap (lfilter P) (lmap llist_of_nellist (llist_of_nellist nells)) =
           lmap ((lfilter P)\<circ> llist_of_nellist)  (llist_of_nellist nells) " 
   using llist.map_comp by blast
 have 16: "( (?n2l (nmap (nfilter P) nells))) =
           ( (?n2l (nellist_of_llist (lmap (nfilter P)  (llist_of_nellist nells)))))" 
    using "13" by presburger
 have 17: "( (?n2l (nellist_of_llist (lmap (nfilter P)  (llist_of_nellist nells))))) =
           ( (?n2l (nellist_of_llist 
              (lmap  (\<lambda>ys. nellist_of_llist (lfilter P (llist_of_nellist ys)))  (llist_of_nellist nells)))))"
    using "14" by presburger
 have 18: "( (?n2l (nellist_of_llist 
               (lmap  (\<lambda>ys. nellist_of_llist (lfilter P (llist_of_nellist ys)))  (llist_of_nellist nells))))) =
           ( (lmap llist_of_nellist 
               ( ( (lmap  (\<lambda>ys. nellist_of_llist (lfilter P (llist_of_nellist ys)))  (llist_of_nellist nells))))))" 
   by simp
 have 19: "( (lmap llist_of_nellist 
              ( ( (lmap  (\<lambda>ys. nellist_of_llist (lfilter P (llist_of_nellist ys)))  (llist_of_nellist nells)))))) =
           ( (lmap (llist_of_nellist\<circ>(\<lambda>ys. nellist_of_llist (lfilter P (llist_of_nellist ys)))) (llist_of_nellist nells))) "
     using llist.map_comp by blast 
 have 20: "\<And> nell. nell \<in> lset(llist_of_nellist nells) \<longrightarrow> \<not>lnull (lfilter P (llist_of_nellist nell))" 
   by (simp add: assms(2))
 have 21: "( (lmap (llist_of_nellist\<circ>(\<lambda>ys. nellist_of_llist (lfilter P (llist_of_nellist ys)))) (llist_of_nellist nells))) =
           ( (lmap ((\<lambda>ys. (lfilter P (llist_of_nellist ys)))) (llist_of_nellist nells)))" 
     by (metis (no_types, lifting) "20" comp_def llist.map_cong0 nellist_of_llist_inverse)
 have 22: "( (lmap llist_of_nellist (llist_of_nellist (nmap (nfilter P) nells)))) =
           (lmap (lfilter P) ( (((  (lmap llist_of_nellist \<circ> llist_of_nellist)) nells)))) "
   using "13" "14" "15" "19" "21" by force
 have 23: "nellist_of_llist (lconcat (lmap (lfilter P)  (  ?n2l nells))) =
           nconcat (nmap (nfilter P) nells)"
  by (simp add: "22" nconcat_def2)  
 show ?thesis 
 by (metis "10" "23" "5" "6" nconcat_def2)
qed

lemma nconcat_nmap_singleton [simp]: 
 "nconcat (nmap (\<lambda>x. NNil (f x) ) nell) = nmap f nell"
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nconcat (nmap (\<lambda>x. NNil (f x) ) nell) =
          (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) (nmap (\<lambda>x. NNil (f x) ) nell)))" 
    by (simp add: nconcat_def2)
 have 2: "(nmap (\<lambda>x. NNil (f x) ) nell) =
          nellist_of_llist (lmap (\<lambda>x. NNil (f x) ) (llist_of_nellist nell))" 
    by (simp add: lmap_llist_of_nellist)
 have 3: "nellist_of_llist (lmap (\<lambda>x. NNil (f x) ) (llist_of_nellist nell)) =
          nellist_of_llist (lmap (\<lambda>x. nellist_of_llist (LCons (f x) LNil) ) (llist_of_nellist nell))" 
    by force
 have 4: "nmap f nell = nellist_of_llist (lmap f (llist_of_nellist nell))"
   by (simp add: lmap_llist_of_nellist) 
 have 5: "lconcat (?n2l (nellist_of_llist 
                     (lmap (\<lambda>x. nellist_of_llist (LCons (f x) LNil) ) (llist_of_nellist nell)))) =
          lconcat (lmap llist_of_nellist (lmap (\<lambda>z. NNil (f z)) (llist_of_nellist nell)))"
   by simp 
 have 6: "(lmap llist_of_nellist (lmap (\<lambda>z. NNil (f z)) (llist_of_nellist nell))) =
          (lmap (llist_of_nellist\<circ>(\<lambda>z. NNil (f z))) (llist_of_nellist nell))  "
   using llist.map_comp by metis
 have 7: "(lmap (llist_of_nellist\<circ>(\<lambda>z. NNil (f z))) (llist_of_nellist nell)) =
          (lmap (\<lambda>z. LCons (f z) LNil) (llist_of_nellist nell))" 
     by auto
 have 8: "lconcat (?n2l (nellist_of_llist 
                     (lmap (\<lambda>x. nellist_of_llist (LCons (f x) LNil) ) (llist_of_nellist nell)))) 
          = (lmap f (llist_of_nellist nell))"
    using "6" by force
 have 9: "(((nellist_of_llist \<circ> lconcat \<circ> ?n2l) (nmap (\<lambda>x. NNil (f x) ) nell))) =
          nellist_of_llist (lmap f (llist_of_nellist nell))"
   using "2" "8" by auto 
 show ?thesis
 using "1" "4" "9" by presburger
qed

lemma nset_nconcat_subset: 
 "nset (nconcat nells) \<subseteq> (\<Union>nell\<in>nset nells. nset nell)"
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "nset (nconcat nells) = 
          nset (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)) " 
   by (simp add: nconcat_def2)
 have 2: "nset (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)) =
          lset (llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)))"
    by (metis lset_llist_of_nellist_a)
 have 3: "(llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells))) =
            ( ((( lconcat \<circ> ?n2l) nells))) "
    using nellist_of_llist_inverse not_null_lconcat by fastforce 
 have 4: "lset (llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells))) =
          lset ( ((( lconcat \<circ> ?n2l) nells)))" 
   using "3" by presburger
 have 5: "lset ( ((( lconcat \<circ> ?n2l) nells))) \<subseteq>
         (\<Union>nell\<in>lset (?n2l nells). lset nell)  " 
    using lset_lconcat_subset by fastforce
 have 6: "(\<Union>nell\<in>lset (?n2l nells). lset nell) =
          \<Union> (nset ` nset nells) "
   by simp
 show ?thesis 
 by (metis "1" "2" "3" "5" "6")
qed

lemma ndistinct_nconcat:
 assumes "ndistinct nells"
         " \<And>nell. nell \<in> nset nells \<Longrightarrow> ndistinct nell "
         " \<And>nell nell1. \<lbrakk> nell \<in> nset nells; nell1 \<in> nset nells; nell \<noteq> nell1 \<rbrakk> \<Longrightarrow> nset nell \<inter> nset nell1 = {} "
 shows  "   ndistinct (nconcat nells)"
proof -
 let ?n2l = "(lmap llist_of_nellist \<circ> llist_of_nellist)" 
 have 1: "ldistinct (llist_of_nellist nells)"
   using assms(1) ndistinct.rep_eq by auto
 have 2: "\<And>nell. nell \<in> lset (llist_of_nellist nells) \<Longrightarrow> ldistinct (llist_of_nellist nell)"
   using assms(2) ndistinct.rep_eq by auto 
 have 3: "\<And>nell nell1. \<lbrakk> nell \<in> lset (llist_of_nellist nells); nell1 \<in> lset (llist_of_nellist nells); nell \<noteq> nell1 \<rbrakk> 
          \<Longrightarrow> lset (llist_of_nellist nell) \<inter> lset (llist_of_nellist nell1) = {}" 
   using assms(3) by simp
 have 4: "ndistinct (nconcat nells) =
          ndistinct (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells))"
      by (simp add: nconcat_def2)
 have 5: "ndistinct (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)) =
          ldistinct (llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells)))"
    using ndistinct.rep_eq by blast
 have 6: "(llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells))) =
            ( ((( lconcat \<circ> ?n2l) nells))) "
    using nellist_of_llist_inverse not_null_lconcat by fastforce
 have 7: "ldistinct (llist_of_nellist (((nellist_of_llist \<circ> lconcat \<circ> ?n2l) nells))) =
          ldistinct ( ((( lconcat \<circ> ?n2l) nells)))"
    using "6" by presburger
 have 8: "ldistinct (?n2l nells)"
      by (metis "1" bi_unique_cr_nellist_help comp_eq_dest_lhs inj_on_def ldistinct_lmap) 
 have 9: "\<And>nell. nell \<in> lset (?n2l nells) \<Longrightarrow> ldistinct nell" 
    using "2" by auto
 have 10: " \<And>nell nell1. \<lbrakk> nell \<in> lset (?n2l nells); 
                      nell1 \<in> lset (?n2l nells); nell \<noteq> nell1\<rbrakk> \<Longrightarrow> 
              lset nell \<inter> lset nell1 = {}" 
   by (metis (no_types, lifting) assms(3) comp_def imageE lset_llist_of_nellist_a lset_lmap)
 have 11: "ldistinct (lconcat (?n2l nells)) "
   using "10" "8" "9" ldistinct_lconcat by blast 
 show ?thesis 
 using "11" "4" "5" "6" by fastforce
qed


subsection \<open>@{term "nellist_all2"}\<close>

lemmas nellist_all2_NNil = nellist.rel_inject(1)
lemmas nellist_all2_NCons = nellist.rel_inject(2)

lemma nellist_all2_NNil1: 
  "nellist_all2 Q (NNil b) nell \<longleftrightarrow> (\<exists>b'. nell = NNil b' \<and> Q b b')"
using nellist.rel_cases by fastforce

lemma nellist_all2_NNil2: 
  "nellist_all2 Q nell (NNil b') \<longleftrightarrow> (\<exists>b. nell = NNil b \<and> Q b b')"
using nellist.rel_sel
by (metis is_NNil_def nellist_all2_NNil) 

lemma nellist_all2_NCons1: 
  "nellist_all2 P (NCons x nell) nell' \<longleftrightarrow> 
     (\<exists>x' nell''. nell' = NCons x' nell'' \<and> P x x' \<and> nellist_all2 P nell nell'')"
using nellist.rel_sel
by (metis nellist.collapse(2) nellist.disc(2) nellist.sel(3) nellist.sel(5)) 

lemma nellist_all2_NCons2: 
  "nellist_all2 P nell' (NCons x nell) \<longleftrightarrow> 
     (\<exists>x' nell''. nell' = NCons x' nell'' \<and> P x' x \<and> nellist_all2 P  nell'' nell)"
by (metis nellist.collapse(2) nellist.disc(2) nellist.rel_sel nellist.sel(3) nellist.sel(5))

lemma nellist_all2_coinduct [consumes 1, case_names ilist_all2, 
                             case_conclusion nellist_all2 is_NNil NNil NCons, 
                             coinduct pred: nellist_all2]:
  assumes "X nellx nelly"
  and "\<And>nellix nelliy. 
    X nellix nelliy \<Longrightarrow>
  (is_NNil nellix = is_NNil nelliy) \<and>
  (is_NNil nellix \<longrightarrow> is_NNil nelliy \<longrightarrow> P (nlast nellix) (nlast nelliy)) \<and>
  (\<not> is_NNil nellix \<longrightarrow> \<not> is_NNil nelliy \<longrightarrow> P (nhd nellix) (nhd nelliy) \<and> 
         (X (ntl nellix) (ntl nelliy) \<or> nellist_all2 P  (ntl nellix) (ntl nelliy)))"
  shows "nellist_all2 P nellx nelly"
using assms 
nellist.rel_coinduct[of "(\<lambda> nelx nely. X nelx nely \<or> nellist_all2 P nelx nely)" nellx nelly P] 
by (metis nellist.rel_sel)

lemma nellist_all2_cases[consumes 1, case_names NNil NCons, cases pred]:
  assumes "nellist_all2 P nellx nelly"
  obtains (NNil) b b' where "nellx = NNil b" "nelly = NNil b'" "P b b'"
  | (NCons) x nellx' y nelly'
    where "nellx = NCons x nellx'" and "nelly = NCons y nelly'" 
    and "P x y" and "nellist_all2 P nellx' nelly'"
using assms
using nellist.rel_cases by blast

lemma nellist_all2_nmap:
  "nellist_all2 P (nmap f nellx) nelly \<longleftrightarrow> nellist_all2  (\<lambda>x y. P (f x) y) nellx nelly"
using nellist.rel_map(1) by blast

lemma nellist_all2_nmap2:
  "nellist_all2 P nellx (nmap f nelly) \<longleftrightarrow> nellist_all2 (\<lambda>x y. P x (f y)) nellx nelly"
using nellist.rel_map(2) by blast

lemma nellist_all2_mono:
  "\<lbrakk> nellist_all2 P nellx nelly; \<And>x y. P x y \<Longrightarrow> P' x y \<rbrakk>
  \<Longrightarrow> nellist_all2 P' nellx nelly"
using nellist.rel_mono_strong by blast

lemma nellist_all2_nlengthD: 
  "nellist_all2 P nellx nelly \<Longrightarrow> nlength nellx = nlength nelly"
by(transfer)(auto dest: llist_all2_llengthD)

lemma nellist_all2_nfiniteD: 
  "nellist_all2 P nellx nelly \<Longrightarrow> nfinite nellx = nfinite nelly"
by transfer
   (auto dest: llist_all2_lfiniteD)

lemma nellist_all2_nfinite1_nlastD:
  "\<lbrakk> nellist_all2 P nellx nelly; nfinite nellx \<rbrakk> \<Longrightarrow> P (nlast nellx) (nlast nelly)"
by (frule nellist_all2_nfiniteD)  
   (transfer,
    auto simp add: llist_all2_conv_all_lnth,
    metis Suc_ile_eq eSuc_enat lfinite.simps lfinite_llength_enat llast_conv_lnth llength_LCons 
    llist.discI(1) order_refl)

lemma nellist_all2_nfinite2_nlastD:
  "\<lbrakk> nellist_all2 P nellx nelly; nfinite nelly \<rbrakk> \<Longrightarrow> P (nlast nellx) (nlast nelly)"
by(metis nellist_all2_nfinite1_nlastD nellist_all2_nfiniteD)

lemma nellist_all2D_llist_all2_llist_of_nellist:
  "nellist_all2 P nellx nelly \<Longrightarrow> llist_all2 P (llist_of_nellist nellx) (llist_of_nellist nelly)"
by transfer
   (simp add: nellist_all2_help_b)

lemma nellist_all2_is_NNilD:
  "nellist_all2 P nellx nelly \<Longrightarrow> is_NNil nellx \<longleftrightarrow> is_NNil nelly"
by (cases nellx) (auto simp add: nellist_all2_NNil1 nellist_all2_NCons1)

lemma nellist_all2_nhdD:
  "\<lbrakk> nellist_all2 P nellx nelly; \<not> is_NNil nellx \<or> \<not> is_NNil nelly \<rbrakk> \<Longrightarrow> P (nhd nellx) (nhd nelly)"
by (cases nellx) (auto simp add: nellist_all2_NNil1 nellist_all2_NCons1)

lemma nellist_all2_ntlI:
  "\<lbrakk> nellist_all2 P nellx nelly; \<not> is_NNil nellx \<or> \<not> is_NNil nelly \<rbrakk> \<Longrightarrow> 
    nellist_all2 P (ntl nellx) (ntl nelly)"
by (cases nellx) (auto simp add: nellist_all2_NNil1 nellist_all2_NCons1)

lemma nellist_all2_refl:
  "nellist_all2 P nell nell \<longleftrightarrow> 
   (\<forall>x \<in> nset nell. P x x) \<and> (nfinite nell \<longrightarrow> P (nlast nell) (nlast nell))"
by transfer
   (auto, metis in_lset_lappend_iff lappend_lbutlast_llast_id_lfinite lfinite_lappend 
          llist.set_intros(1))

lemma nellist_all2_reflI:
  "\<lbrakk> \<And>x. x \<in> nset nell \<Longrightarrow> P x x; nfinite nell \<Longrightarrow> P (nlast nell) (nlast nell) \<rbrakk>
  \<Longrightarrow> nellist_all2 P nell nell"
by(simp add: nellist_all2_refl)

lemma nellist_all2_conv_all_nnth_help1:
 " \<not> lnull nellx \<Longrightarrow> \<not> lnull nelly \<Longrightarrow> lfinite nelly \<Longrightarrow> llist_all2 P nellx nelly \<Longrightarrow> 
   P (llast nellx) (llast nelly)"
proof -
assume a1: "lfinite nelly"
assume a2: "llist_all2 P nellx nelly"
assume a3: "\<not> lnull nellx"
assume a4: "\<not> lnull nelly"
have f5: "lfinite (lappend (lbutlast nellx) (LCons (llast nellx) LNil))"
using a3 a2 a1 by (simp add: llist_all2_lfiniteD) 
have f6: "llength (ltl nellx) = epred (llength nelly)"
using a2 by (metis (no_types) epred_llength llist_all2_llengthD) 
have f7: "lappend (lbutlast nelly) (LCons (llast nelly) LNil) = nelly"
using a4 by (meson lappend_lbutlast_llast_id) 
have "llength (lbutlast nellx) = llength (ltl nellx)"
using epred_llength by auto 
then show ?thesis
using f7 f6 f5 a3 a2 a1 
by (metis (no_types) lappend_eq_lappend_conv lappend_lbutlast_llast_id lappend_ltake_ldrop 
    lbutlast_conv_ltake lfinite_lappend lhd_LCons llist.disc(2) llist_all2_lappend1D(2) 
    llist_all2_lhdD2) 
qed

lemma nellist_all2_conv_all_nnth:
  "nellist_all2 P nellx nelly \<longleftrightarrow> 
  nlength nellx = nlength nelly \<and> 
  (\<forall>n. enat n \<le> nlength nellx \<longrightarrow> P (nnth nellx n) (nnth nelly n))"
by transfer
    (auto simp add: llist_all2_llengthD, 
     metis eSuc_epred iless_Suc_eq llength_eq_0 llist_all2_lnthD2,
     metis eSuc_epred iless_Suc_eq llength_eq_0 llist_all2_conv_all_lnth)

lemma nellist_all2_True [simp]: 
  "nellist_all2 (\<lambda>_ _. True) nellx nelly \<longleftrightarrow> nlength nellx = nlength nelly"
by(simp add: nellist_all2_conv_all_nnth)


lemma nellist_all2_nnthD:
  "\<lbrakk> nellist_all2 P nellx nelly; enat n \<le> nlength nellx \<rbrakk> \<Longrightarrow> P (nnth nellx n) (nnth nelly n)"
by (simp add: nellist_all2_conv_all_nnth)

lemma nellist_all2_nnthD2:
  "\<lbrakk> nellist_all2 P nellx nelly; enat n \<le> nlength nelly \<rbrakk> \<Longrightarrow> P (nnth nellx n) (nnth nelly n)"
by (simp add: nellist_all2_conv_all_nnth)

lemmas nellist_all2_eq = nellist.rel_eq

lemma nmap_eq_nmap_conv_nellist_all2:
  "nmap f nellx = nmap f'  nelly \<longleftrightarrow>
  nellist_all2 (\<lambda>x y. f x = f' y) nellx nelly"
by transfer
   (clarsimp simp add: lmap_eq_lmap_conv_llist_all2)

lemma nellist_all2_trans:
  "\<lbrakk> nellist_all2 P nellx nelly; nellist_all2 P nelly nellz; transp P \<rbrakk>
   \<Longrightarrow> nellist_all2 P nellx nellz"
by transfer (auto elim: llist_all2_trans dest: llist_all2_lfiniteD transpD)

lemma nellist_all2_nappendI:
  "\<lbrakk> nellist_all2 P nellx nelly;
     \<lbrakk> nfinite nellx; nfinite nelly; P (nlast nellx) (nlast nelly) \<rbrakk>
     \<Longrightarrow> nellist_all2 P nellx' nelly' \<rbrakk>
  \<Longrightarrow> nellist_all2 P (nappend nellx nellx') (nappend nelly nelly')"
by transfer
 (auto simp add: lnull_def lappend_eq_lappend_conv nellist_all2_conv_all_nnth_help1
   intro: llist_all2_lappendI)
   
lemma nested_nellist_all2_nested_llist_all2: 
 " nellist_all2 (nellist_all2 A) nells nells1 =
   llist_all2 (llist_all2 A) (lmap llist_of_nellist (llist_of_nellist nells))
                             (lmap llist_of_nellist (llist_of_nellist nells1)) " 
proof -
 have 1: "nellist_all2 (nellist_all2 A) nells nells1 =
          llist_all2 (nellist_all2 A) (llist_of_nellist nells) (llist_of_nellist nells1)"
    using nellist_all2_help_a nellist_all2_help_b by blast
 have 2: "(\<lambda> xx yy. nellist_all2 A xx yy) = 
          (\<lambda> xx yy. llist_all2 A (llist_of_nellist xx) (llist_of_nellist yy)) "
   by (meson nellist_all2D_llist_all2_llist_of_nellist nellist_all2_help_a) 
 have 3: "llist_all2 (nellist_all2 A) (llist_of_nellist nells) (llist_of_nellist nells1) =
           llist_all2 (\<lambda> xx yy. llist_all2 A (llist_of_nellist xx) (llist_of_nellist yy))
                      (llist_of_nellist nells) (llist_of_nellist nells1) "
    using "2" by auto
 have 6: "llist_all2 (\<lambda> xx yy. llist_all2 A (llist_of_nellist xx) (llist_of_nellist yy))
                      (llist_of_nellist nells) (llist_of_nellist nells1) =
          llist_all2 (llist_all2 A) (lmap llist_of_nellist (llist_of_nellist nells))
                                    (lmap llist_of_nellist (llist_of_nellist nells1))"
  using llist_all2_lmap1[of "(llist_all2 A)" llist_of_nellist "(llist_of_nellist nells)" 
           "(lmap llist_of_nellist (llist_of_nellist nells1))"]  
        llist_all2_lmap2[of "(llist_all2 A)" _ llist_of_nellist "(llist_of_nellist nells1)"] 
      by (simp add: llist_all2_conv_all_lnth)   
 show ?thesis
 using "1" "3" "6" by blast 
qed

lemma nellist_all2_nconcatI:
assumes  "nellist_all2 (nellist_all2 A) nells nells1"
shows    "  nellist_all2 A (nconcat nells) (nconcat nells1)"
proof -
 have 3: "nellist_all2 A (nconcat nells) (nconcat nells1) =
          llist_all2 A (llist_of_nellist (nconcat nells)) (llist_of_nellist (nconcat nells1)) " 
   using nellist_all2_help_a nellist_all2_help_b by blast
 have 4: "(llist_of_nellist (nconcat nells)) =
          ((lconcat \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist)) nells)"
      using nconcat_def3 by auto 
 have 5: "(llist_of_nellist (nconcat nells1)) =
          ((lconcat \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist)) nells1)"
     using nconcat_def3 by auto 
 have 7: " llist_all2 (llist_all2 A) (lmap llist_of_nellist (llist_of_nellist nells))
                                    (lmap llist_of_nellist (llist_of_nellist nells1)) \<Longrightarrow>
          llist_all2 A ((lconcat \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist)) nells)
                       ((lconcat \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist)) nells1) "
    using llist_all2_lconcatI[of A "(lmap llist_of_nellist (llist_of_nellist nells))" 
         "(lmap llist_of_nellist (llist_of_nellist nells1))" ]
    by simp
 have 8: "
          llist_all2 A ((lconcat \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist)) nells)
                       ((lconcat \<circ> (lmap llist_of_nellist \<circ> llist_of_nellist)) nells1) 
           = nellist_all2 A (nconcat nells) (nconcat nells1) "
   using "3" "4" "5" by presburger 
 have 9: "nellist_all2 (nellist_all2 A) nells nells1 =
          llist_all2 (llist_all2 A) (lmap llist_of_nellist (llist_of_nellist nells))
                                    (lmap llist_of_nellist (llist_of_nellist nells1))" 
    using nested_nellist_all2_nested_llist_all2 by blast
 show ?thesis using assms 7 8 9 
 by blast 
qed   


lemma nlength_nconcat_eqI:
  fixes nells :: "'a nellist nellist" and nells1 :: "'b nellist nellist"
  assumes "nellist_all2 (\<lambda>xs ys. nlength xs = nlength ys) nells nells1"
  shows "nlength (nconcat nells) = nlength (nconcat nells1)"
proof -
 have  "nellist_all2 (nellist_all2 (\<lambda>a b. True)) nells nells1"
  using assms nellist.rel_mono_strong nellist_all2_True by blast
 then show ?thesis  using nellist_all2_nconcatI nellist_all2_nlengthD by blast
qed    


lemma llist_all2_nellist_of_llistI:
  "nellist_all2 A nellx nelly \<Longrightarrow> 
   llist_all2 A (lbutlast(llist_of_nellist nellx)) (lbutlast(llist_of_nellist nelly))"
proof (coinduction arbitrary: nellx nelly)
  case LNil
  then show ?case
   by (metis lbutlast.disc_iff(1) llist.disc(1) llist_of_nellist_inverse_a ltl_llist_of_nellist1
       nellist_all2_is_NNilD nellist_of_llist_a.disc(1))
  next
  case (LCons nell1 nell2)
  then show ?case 
    proof -
     assume a0: "nellist_all2 A nell1 nell2" 
     assume a1: "\<not> lnull (lbutlast (llist_of_nellist nell1))" 
     assume a2: "\<not> lnull (lbutlast (llist_of_nellist nell2))" 
     have 1: "A (lhd (lbutlast (llist_of_nellist nell1))) (lhd (lbutlast (llist_of_nellist nell2)))"
         using a0 a1 a2   
         by (metis lbutlast.disc(1) llist.disc(1) llist_of_nellist_inverse_a ltl_llist_of_nellist1 
                nellist_all2_nhdD nhd_nellist_of_llist_a)
    have 2: "((\<exists>nellx nelly.
         ltl (lbutlast (llist_of_nellist nell1)) = lbutlast (llist_of_nellist nellx) \<and>
         ltl (lbutlast (llist_of_nellist nell2)) = lbutlast (llist_of_nellist nelly) \<and> 
         nellist_all2 A nellx nelly) \<or>
         llist_all2 A (ltl (lbutlast (llist_of_nellist nell1))) (ltl (lbutlast (llist_of_nellist nell2))))" 
     by (metis a0 a1 lbutlast.ctr(1) lbutlast_ltl llist.discI(1) ltl_llist_of_nellist ltl_llist_of_nellist1 
         nellist.rel_sel)
    show ?thesis using 1 2 by blast
  qed 
qed

lemma nellist_all2_nellist_of_llist_a [simp]:
  "nellist_all2 A (nellist_of_llist_a b llx) (nellist_of_llist_a c lly) \<longleftrightarrow>
  llist_all2 A llx lly \<and> (lfinite llx \<longrightarrow> A b c)"
proof (cases "lfinite llx")
  case True
  then show ?thesis 
  using llist_all2_nellist_of_llistI
  by (auto simp add: llist_all2_lappendI nellist_all2_help_a, fastforce,
      metis lbutlast_lfinite lbutlast_snoc llist_all2_lfiniteD nellist_all2_nfinite1_nlastD 
      nellist_of_llist_a_inverse nfinite_def nlast_nellist_of_llist_a_lfinite)
  next
  case False
  then show ?thesis 
  by (metis lbutlast_snoc llist_all2_lappendI llist_all2_nellist_of_llistI nellist_all2_help_a
       nellist_of_llist_a_inverse)
qed


subsection \<open>From a nonempty lazy list to a lazy list @{term "llist_of_nellist"}\<close>

lemma llist_of_nellist_nmap [simp]:
  "llist_of_nellist (nmap f nell) = lmap f (llist_of_nellist nell)"
by (simp add: lmap_llist_of_nellist)

lemma llist_of_nellist_nappend:
  "llist_of_nellist (nappend nellx nelly) = lappend (llist_of_nellist nellx) (llist_of_nellist nelly)"
by (transfer) auto

lemma llist_of_nellist_lappendn [simp]:
  "llist_of_nellist (lappendn ll nell) = lappend ll (llist_of_nellist nell)"
by transfer auto

lemma llist_of_nellist_nconcat [simp]: 
 " llist_of_nellist (nconcat nell) = lconcat ((lmap llist_of_nellist \<circ> llist_of_nellist) nell)  " 
using nconcat_def3 by fastforce


lemma llist_of_nellist_nfilter [simp]:
assumes "\<exists> x \<in> nset nell. P x"
 shows "llist_of_nellist (nfilter P nell) = lfilter P (llist_of_nellist nell)"
using assms
by transfer auto



subsection \<open>@{term "ndropn"}\<close>

lemma ndropn_0 [simp, code, nitpick_simp]: 
  "ndropn 0 nell = nell"
 using zero_enat_def  by transfer auto

lemma ndropn_NNil [simp, code]: 
  "ndropn n (NNil b) = (NNil b)"
by transfer auto

lemma ndropn_Suc_NCons [simp, code]: 
 "ndropn (Suc n) (NCons x nell) = ndropn n nell"
 proof (cases "nfinite nell")
 case True
 then show ?thesis 
 by transfer
    (auto simp add: min_def not_lnull_conv Suc_ile_eq llength_eq_infty_conv_lfinite the_enat_eSuc ,
     metis Extended_Nat.eSuc_mono eSuc_enat iless_Suc_eq leD,
     metis antisym eSuc_enat enat_the_enat ile_eSuc llength_eq_infty_conv_lfinite n_not_Suc_n 
     the_enat.simps)
 next
 case False
 then show ?thesis 
 by transfer
    (simp,
     metis co.enat.sel(2) eSuc_infinity infinity_ileE ldropn_Suc_LCons llength_eq_infty_conv_lfinite 
     min.cobounded1 min_def the_enat.simps)
 qed
   
lemma ndropn_Suc [nitpick_simp]: 
  "ndropn (Suc n) nell = (case nell of NNil b \<Rightarrow> NNil b | NCons x nell' \<Rightarrow> ndropn n nell')"
by(cases nell) simp_all

lemma ltl_power_NNil_help:
 "((\<lambda>ll. if \<exists>b. ll = LCons b LNil then ll else ltl ll) ^^ n) (LCons b LNil) = LCons b LNil"
by (induction n) simp_all

lemma ntl_power_NNil:
 " (ntl ^^ n) (NNil b) = (NNil b)"
by transfer (auto simp add: lnull_def ltl_power_NNil_help)

lemma ntl_power_NCons:
 "ntl ((ntl ^^ n) (NCons x nell)) = (ntl ^^ n) nell"
by (induction n) ( transfer, auto)

lemma ntl_power_Suc [simp]:
 " (ntl ^^ (Suc n)) nell = (case nell of NNil b \<Rightarrow> NNil b | NCons x nell' \<Rightarrow> (ntl ^^ n) nell')"
 by (cases nell)
    (simp_all add: ntl_power_NNil ntl_power_NCons)

lemma llist_of_nellist_ndropn [simp]:
  "llist_of_nellist (ndropn n nell) = 
   ldropn (the_enat (min (enat n) ((epred(llength((llist_of_nellist nell))))))) 
          (llist_of_nellist nell) "
by transfer auto

lemma ndropn_Suc_conv_ndropn:
  "enat n < nlength nell \<Longrightarrow> NCons (nnth nell n) (ndropn (Suc n) nell) = ndropn n nell" 
  proof (induct n arbitrary: nell)
  case 0
  then show ?case 
  proof (cases nell)
  case (NNil x1)
  then show ?thesis using "0.prems" by auto
  next
  case (NCons x nell1)
  then show ?thesis by simp
  qed
  next
  case (Suc n)
  then show ?case 
  proof (cases nell)
  case (NNil x1)
  then show ?thesis using Suc.prems by auto
  next
  case (NCons x nell1)
  then show ?thesis using Suc 
  by (metis One_nat_def add.commute add_left_mono gen_nlength_code(2) gen_nlength_def leD 
      ndropn_Suc_NCons nlength_code nnth_Suc_NCons not_le_imp_less of_nat_Suc of_nat_eq_enat 
      one_enat_def plus_1_eq_Suc)
  qed
qed

lemma ndropn_nlength [simp]:
   "nlength (ndropn n nell) = nlength nell - enat n"
proof (induct n arbitrary: nell)
 case 0
 then show ?case by simp
 next
 case (Suc n)
 then show ?case 
   proof (cases nell)
    case (NNil x1)
    then show ?thesis by simp
    next
    case (NCons x nell1)
    then show ?thesis using Suc
     by (metis eSuc_enat eSuc_minus_eSuc ndropn_Suc_NCons nlength_NCons)
   qed
qed

lemma ndropn_nnth [simp]: 
  " nnth (ndropn n nell) m = nnth nell (n+m)"  
proof (induct n arbitrary: m nell)
  case 0
  then show ?case by simp
  next
  case (Suc n)
  then show ?case  
  proof (cases nell)
   case (NNil x1)
   then show ?thesis by (simp add: nnth_NNil)
   next
   case (NCons x nell1)
   then show ?thesis by (simp add: Suc.hyps)
  qed
qed

lemma ndropn_nnth_a:
assumes " nlength nell \<le> enat(n+m)"
shows "  nnth (ndropn n nell) m =  (nlast nell)"
 proof -
 have 1: "nfinite nell"
 using assms enat_ile nfinite_conv_nlength_enat by auto
 have 2: "nfinite nell \<Longrightarrow> nlength nell \<le> enat(n+m) \<Longrightarrow> nnth (ndropn n nell) m =  (nlast nell)"
 proof (induct arbitrary: n m rule: nfinite_induct)
 case (NNil y)
 then show ?case by (simp add: nnth_NNil) 
 next
 case (NCons x nell)
 then show ?case 
 proof (cases n)
 case 0
 then show ?thesis 
 by (metis NCons(2) NCons(3) Suc_ile_eq Suc_pred add_cancel_right_left enat_le_plus_same(1) 
     gen_nlength_def iless_Suc_eq leD le_add1 ndropn_0 nellist.sel(2) nlast0_nlast nlength_NCons
      nlength_code nnth_Suc_NCons not_le_imp_less order.not_eq_order_implies_strict)
 next
 case (Suc nat)
 then show ?thesis 
 by (metis NCons(2) NCons(3) add_Suc eSuc_enat epred_eSuc epred_le_epredI ndropn_Suc_NCons
     nlast_NCons nlength_NCons)
 qed
 qed
 show ?thesis using "1" "2" assms by auto  
 qed
 
lemma ndropn_ntl :
 " ndropn n nell = (ntl ^^ n) nell "
proof (induction n arbitrary: nell)
case 0
then show ?case by simp
next
case (Suc n)
then show ?case 
  proof (cases nell)
  case (NNil x1)
  then show ?thesis 
     by (simp add: ntl_power_NNil)
  next
  case (NCons x nell)
  then show ?thesis 
  by (metis Suc.IH funpow_Suc_right ndropn_Suc_NCons nellist.sel(5) o_apply)
  qed
qed

lemma ndropn_is_NNil:
 " is_NNil nell \<Longrightarrow> ndropn n nell = nell"
proof (induct n arbitrary: nell)
 case 0
 then show ?case by auto
 next
 case (Suc n)
 then show ?case by (simp add: ndropn_Suc nellist.case_eq_if)
qed
 
lemma is_NNil_ndropn:
 " is_NNil(ndropn n nell) \<longleftrightarrow> nlength nell \<le> (enat n)"
proof (induct n arbitrary: nell)
 case 0
 then show ?case 
  proof (cases nell)
  case (NNil x1)
  then show ?thesis by simp
  next
  case (NCons x nell)
  then show ?thesis using zero_enat_def by auto
  qed
 next
 case (Suc n)
 then show ?case 
  proof (cases nell)
  case (NNil x1)
  then show ?thesis by simp
  next
  case (NCons x nell)
  then show ?thesis 
  by (metis One_nat_def Suc.hyps add.commute add_left_mono co.enat.sel(2) eSuc_enat epred_le_epredI
       gen_nlength_code(2) gen_nlength_def ndropn_Suc_NCons nlength_NCons nlength_code of_nat_Suc 
       of_nat_eq_enat one_enat_def plus_1_eq_Suc)
  qed
qed

lemma ndropn_eq_NNil:
 " ndropn n nell = (NNil b) \<longleftrightarrow> nnth nell (the_enat(nlength nell)) = b \<and> nlength nell \<le> (enat n)"
proof -
 have 1: "nfinite nell \<Longrightarrow> ndropn n nell = NNil b \<Longrightarrow> nnth nell (the_enat (nlength nell)) = b"
    by (metis add_cancel_left_right enat_le_plus_same(2) gen_nlength_def is_NNil_ndropn ndropn_NNil 
        ndropn_nnth ndropn_nnth_a nfinite_nlength_enat nlast_NNil nlength_NNil nlength_code 
        plus_enat_simps(1) the_enat.simps zero_enat_def)
 have 2: "nfinite nell \<Longrightarrow> ndropn n nell = NNil b \<Longrightarrow> nlength nell \<le> enat n"
   by (metis is_NNil_ndropn nellist.disc(1))
 have 3: "nfinite nell \<Longrightarrow> nlength nell \<le> enat n \<Longrightarrow> b = nnth nell (the_enat (nlength nell)) \<Longrightarrow> 
          ndropn n nell = NNil (nnth nell (the_enat (nlength nell)))" 
     by (metis add.right_neutral is_NNil_ndropn ndropn_nnth_a nellist.collapse(1) nfinite_NNil 
         nlength_NNil nnth_nlast the_enat_0)
 have 4: "\<not>nfinite nell \<Longrightarrow>     
           ndropn n nell = (NNil b) \<longleftrightarrow> 
           nnth nell (the_enat(nlength nell)) = b \<and> nlength nell \<le> (enat n)"
   by (metis enat_ile is_NNil_ndropn nellist.disc(1) nfinite_conv_nlength_enat)         
 show ?thesis
   using "1" "2" "3" "4" by fastforce
qed

lemma ntl_ndropn:
 " ntl(ndropn n nell) = ndropn n (ntl nell)"
by (simp add: funpow_swap1 ndropn_ntl)

lemma nfinite_ndropn_a:
assumes "nfinite nell "
shows   " nfinite(ndropn n nell) "
using assms
proof (induct n arbitrary: nell)
 case 0
 then show ?case by auto
 next
 case (Suc n)
 then show ?case by (simp add: ndropn_ntl)
qed

lemma nfinite_ndropn_b:
assumes "nfinite(ndropn n nell)"
shows   " nfinite nell  "
using assms 
proof (induct ys\<equiv>"ndropn n nell" arbitrary: n nell rule: nfinite_induct)
 case (NNil y)
 then show ?case by (metis enat_ile ndropn_eq_NNil nfinite_conv_nlength_enat)
 next
 case (NCons x nell)
 then show ?case by (metis nellist.sel(5) nfinite_ntl ntl_ndropn)
qed

lemma nfinite_ndropn[simp]:
 " nfinite(ndropn n nell) = nfinite nell"
using nfinite_ndropn_a nfinite_ndropn_b by blast

lemma ndropn_ndropn:
 " ndropn m (ndropn n nell) = ndropn (n+m) nell"
proof (induct n arbitrary: nell)
 case 0
 then show ?case by simp
 next
 case (Suc n)
 then show ?case
 by (metis add_Suc ndropn_NNil ndropn_Suc nellist.case_eq_if)
qed

lemma ndropn_nlast:
 "nfinite nell \<Longrightarrow>  ndropn (the_enat(nlength nell)) nell = (NNil (nlast nell)) "
 by (metis add.left_neutral enat.simps(3) enat_the_enat ndropn_eq_NNil ndropn_nnth ndropn_nnth_a
      nfinite_conv_nlength_enat order_refl) 

lemma ndropn_nfirst:
 "nfirst (ndropn n nell) = (nnth nell n) "
by transfer
   (metis lhd_ldropn llist_of_nellist_ndropn lnull_ldropn not_le_imp_less 
    not_lnull_conv_llist_of_nellist)

lemma ndropn_all:
 "nlength nell \<le> enat n \<Longrightarrow> ndropn n nell = (NNil (nlast nell)) "
by (metis enat_ile ndropn_eq_NNil ndropn_nlast nlength_eq_enat_nfiniteD)

lemma ndropn_nappend1:
 " nfinite nellx \<Longrightarrow> n \<le> nlength nelly \<Longrightarrow> 
   ndropn (Suc(the_enat (nlength nellx) + n)) (nappend nellx nelly) = ndropn n nelly "
proof (induct arbitrary: nelly n rule: nfinite_induct)
 case (NNil y)
 then show ?case by simp
 next
 case (NCons x nell)
 then show ?case 
 by (metis ab_semigroup_add_class.add_ac(1) eSuc_enat nappend_NCons ndropn_Suc_NCons 
     nfinite_nlength_enat nlength_NCons plus_1_eq_Suc the_enat.simps)
qed
 
lemma ndropn_nappend2:
 "enat n \<le> (nlength nellx) \<Longrightarrow>  ndropn n (nappend nellx nelly) = nappend (ndropn n nellx) nelly " 
proof (induct n arbitrary: nellx nelly)
case 0
then show ?case by simp
next
case (Suc n)
then show ?case 
  proof (cases nellx)
  case (NNil x1)
  then show ?thesis 
  using Suc.prems enat_0_iff(1) by auto
  next
  case (NCons x21 x22)
  then show ?thesis 
  by (metis Suc.hyps Suc.prems eSuc_enat eSuc_ile_mono nappend_NCons ndropn_Suc_NCons nlength_NCons)
  qed
qed
 

lemma ndropn_nappend3:
" nlength nellx < enat n \<Longrightarrow> 
  ndropn n (nappend nellx nelly) = ndropn (n - (the_enat (eSuc(nlength nellx))) ) nelly "
proof (induct n arbitrary: nellx nelly)
case 0
then show ?case using zero_enat_def by auto
next
case (Suc n)
then show ?case 
 proof (cases nellx)
 case (NNil x1)
 then show ?thesis 
 by (metis add_diff_cancel_left' nappend_NNil ndropn_Suc_NCons nlength_NNil one_eSuc 
     one_enat_def plus_1_eq_Suc the_enat.simps)
 next
 case (NCons x21 x22)
 then show ?thesis 
 by (metis Extended_Nat.eSuc_mono Suc.hyps Suc.prems add_diff_cancel_left' diff_Suc_eq_diff_pred 
     eSuc_enat enat_ord_code(4) nappend_NCons ndropn_Suc_NCons nlength_NCons 
     order_less_imp_not_less plus_1_eq_Suc the_enat_eSuc)
 qed
qed
  
lemma nset_ndropn:
 " nset (ndropn n nell) \<subseteq> nset nell"
by transfer (simp add: lset_ldropn_subset)

lemma ndropn_nmap:
 "ndropn n (nmap f nell) = nmap f (ndropn n nell)" 
by transfer
   (auto,
    metis eSuc_epred enat_the_enat iless_Suc_eq infinity_ileE leD llength_eq_0 min.cobounded1 
    min.cobounded2)

lemma nappend_ntaken_ndropn:
 assumes " (Suc k) \<le> nlength nell"
 shows   " nappend (ntaken k nell) (ndropn (Suc k) nell) = nell"
using assms
by transfer  (simp add: min_absorb1)   


lemma nfirst_eq_nnth_zero:
 " nfirst nell = nnth nell 0"
 by (metis ndropn_0 ndropn_nfirst) 

subsection \<open>@{term "nzip"}\<close>

lemma nzip_nhd:
 " \<not>is_NNil nellx \<and> \<not>is_NNil nelly \<Longrightarrow> nhd (nzip nellx nelly) = (nhd nellx, nhd nelly)"
by transfer
    (auto simp add: lzip_eq_LCons_conv,
     metis  lzip_eq_LNil_conv)  

lemma nzip_ntl:
 "\<not>is_NNil nellx \<and> \<not>is_NNil nelly \<Longrightarrow> ntl(nzip nellx nelly) = nzip (ntl nellx) (ntl nelly)"
by transfer
   (auto,
    metis lhd_LCons_ltl ltl_lzip ltl_simps(2) lzip_eq_LNil_conv,
    metis (full_types) lhd_LCons_ltl lnull_def,
    metis lhd_LCons_ltl llist.collapse(1))

lemma nzip_simps [simp, code, nitpick_simp]:
 "nzip (NNil b) nelly = (NNil (b, (nnth nelly 0)))"
 "nzip nellx (NNil b) = (NNil ((nnth nellx 0), b))"
 "nzip (NCons x nellx) (NCons y nelly) = NCons (x, y) (nzip nellx nelly)"
apply transfer
apply (auto simp add: not_lnull_conv)
apply (metis lnth_0   min_enat_simps(3) the_enat_0 zero_enat_def)
apply transfer
apply (auto simp add: not_lnull_conv)
apply (metis lnth_0   min_enat_simps(3) the_enat_0 zero_enat_def)
apply transfer
by (auto simp add: not_lnull_conv)

lemma is_NNil_nzip [simp]:
 "is_NNil (nzip nellx nelly) \<longleftrightarrow> (is_NNil nellx) \<or> (is_NNil nelly)"
by transfer
   (auto simp add: lzip_eq_LCons_conv not_lnull_conv,
    metis  lzip_eq_LNil_conv)

lemma nzip_eq_NNil_conv:
 " nzip nellx nelly = (NNil (x, y)) \<longleftrightarrow> 
   ((is_NNil nellx) \<or> (is_NNil nelly)) \<and> (nnth nellx 0) = x \<and> (nnth nelly 0) = y"
by auto
   (metis is_NNil_nzip nellist.disc(1),
    metis Pair_inject is_NNil_nzip nellist.collapse(1) nellist.disc(1) nlast_NNil nzip_simps(1) 
    nzip_simps(2),
    metis Pair_inject is_NNil_def is_NNil_nzip nellist.inject(1) nzip_simps(1) nzip_simps(2),
    metis nellist.collapse(1) nnth_NNil nzip_simps(1),
    metis nellist.collapse(1) nnth_NNil nzip_simps(2))

lemma nzip_eq_NCons_conv:
 " nzip nellx nelly = (NCons z zs) \<longleftrightarrow>
   (\<exists> x nellx' y nelly'. nellx = (NCons x nellx') \<and> nelly = (NCons y nelly') \<and>
        z = (x, y) \<and> zs = (nzip nellx' nelly') )"
by (cases nellx nelly rule: nellist.exhaust[case_product nellist.exhaust])
    auto

lemma nzip_nappend:
 "nlength nellx = nlength nellu
  \<Longrightarrow> nzip (nappend nellx nelly) (nappend nellu nellv) =
      nappend (nzip nellx nellu) (nzip nelly nellv)"
by transfer
   (simp,
    meson co.enat.expand llength_eq_0 lzip_lappend)

lemma nlength_nzip [simp]:
 " nlength (nzip nellx nelly) = (min (nlength nellx) (nlength nelly))"
by transfer simp
 
lemma ntake_nzip:
 "ntake n (nzip nellx nelly) = nzip (ntake n nellx) (ntake n nelly)"
by transfer 
   (simp add: ltake_lzip) 

lemma ntaken_nzip:
 "ntaken n (nzip nellx nelly) = nzip (ntaken n nellx) (ntaken n nelly)"
by transfer
   (simp add: enat_0_iff(1) ltake_lzip)

lemma ndropn_nzip [simp]:
 " n \<le> nlength nellx \<and> n \<le> nlength nelly  \<Longrightarrow>
   ndropn n (nzip nellx nelly) = nzip (ndropn n nellx) (ndropn n nelly)"
by transfer
   (auto simp add: min_def,
    metis co.enat.exhaust_sel iless_Suc_eq leD llength_eq_0,
    metis co.enat.exhaust_sel iless_Suc_eq leD llength_eq_0)

lemma nzip_niterates:
 "nzip (niterates f x) (niterates g y) = niterates (\<lambda>(x,y). (f x, g y)) (x, y)"
by transfer
   (simp add: lzip_iterates)

lemma nnth_nzip:
 assumes "n \<le> nlength nellx"
         "n \<le> nlength nelly"
 shows   " nnth (nzip nellx nelly) n = (nnth nellx n, nnth nelly n)"
using assms
by transfer
   (auto simp add: min_def,
    metis co.enat.exhaust_sel iless_Suc_eq llength_eq_0 lnth_lzip)
 
lemma nset_nzip:
"nset (nzip nellx nelly) =
 { (nnth nellx n, nnth nelly n) | n. n \<le> min (nlength nellx) (nlength nelly)}"
by transfer
   (auto simp add: in_lset_conv_lnth,
    metis Pair_inject co.enat.exhaust_sel iless_Suc_eq llength_eq_0 lnth_lzip min.orderE 
    the_enat.simps,
    metis eSuc_epred iless_Suc_eq llength_eq_0 lnth_lzip)

lemma nset_nzipD1:
 "(x, y) \<in> nset (nzip nellx nelly) \<Longrightarrow> x \<in> nset nellx"
by transfer
   (meson lset_lzipD1) 

lemma nset_nzipD2:
 "(x, y) \<in> nset (nzip nellx nelly) \<Longrightarrow> y \<in> nset nelly"
by transfer
   (meson lset_lzipD2)

lemma nset_nzip_same [simp]:
 " nset (nzip nellx nellx) = (\<lambda> x. (x, x)) ` nset nellx"
by transfer
   simp

lemma nfinite_nzip [simp]:
 "nfinite (nzip nellx nelly) \<longleftrightarrow> nfinite nellx \<or> nfinite nelly "
by transfer
   simp

lemma nzip_eq_nappend_conv:
 assumes eq: " nzip nellx nelly = nappend nellu nellv"
 shows   "\<exists> nellx' nellx'' nelly' nelly''.
           nellx = nappend nellx' nellx'' \<and> nelly = nappend nelly' nelly'' \<and>
           nlength nellx' = nlength nelly' \<and>
           nellu = nzip nellx' nelly' \<and> nellv = nzip nellx'' nelly''"
using assms
apply transfer
using lzip_eq_lappend_conv
by (auto simp add: lzip_eq_lappend_conv)
   fastforce

lemma nzip_nmap [simp]:
 "nzip (nmap f nellx) (nmap g nelly) = nmap (\<lambda>(x, y). (f x, g y)) (nzip nellx nelly)"
by transfer  auto

lemma nzip_nmap1:
 "nzip (nmap f nellx) nelly = nmap (\<lambda>(x, y). (f x, y)) (nzip nellx nelly)"
by transfer
   (simp add: lzip_lmap1)

lemma nzip_nmap2:
 "nzip  nellx (nmap f nelly) = nmap (\<lambda>(x, y). (x, f y)) (nzip nellx nelly)"
by transfer
   (simp add:  lzip_lmap2)

lemma nmap_fst_nzip_conv_ntake:
 "nmap fst (nzip nellx nelly) = ntake (min (nlength nellx) (nlength nelly)) nellx"
by transfer
  (auto,
   metis co.enat.exhaust_sel llength_eq_0 lmap_fst_lzip_conv_ltake min_eSuc_eSuc)

lemma nmap_snd_nzip_conv_ntake:
 " nmap snd (nzip nellx nelly) = ntake (min (nlength nellx) (nlength nelly)) nelly"
by transfer
   (auto,
    metis co.enat.exhaust_sel llength_eq_0 lmap_snd_lzip_conv_ltake min_eSuc_eSuc)

lemma nzip_conv_nzip_ntake_min_nlength:
 "nzip nellx nelly =
  nzip (ntake (min (nlength nellx) (nlength nelly)) nellx)
       (ntake (min (nlength nellx) (nlength nelly)) nelly) "
by transfer
   (auto,
    metis co.enat.exhaust_sel epred_min i0_lb llength_lzip ltake_all ltake_lzip order_refl)

lemma nellist_all2_conv_nzip:
 " nellist_all2 P nellx nelly \<longleftrightarrow> 
   nlength nellx = nlength nelly \<and> (\<forall>(x, y) \<in> nset(nzip nellx nelly). P x y) "
using nset_nzip[of nellx nelly]  
by (auto simp add: nellist_all2_conv_all_nnth)
   blast

lemma nellist_all2_all_nnthI:
assumes " nlength nellx = nlength nelly"
        " \<And>n. enat n \<le> nlength nellx \<Longrightarrow> P (nnth nellx n) (nnth nelly n)"
 shows  " nellist_all2 P nellx nelly"
using assms by (simp add:  nellist_all2_conv_all_nnth) 

lemma nellist_all2_nsetD1:
assumes " nellist_all2 P nellx nelly"
        " x \<in> nset nellx"
 shows  " \<exists>y \<in> nset nelly. P x y" 
using assms
by (metis in_nset_conv_nnth nellist_all2_conv_all_nnth) 

lemma nellist_all2_nsetD2:
assumes " nellist_all2 P nellx nelly"
        " y \<in> nset nelly"
 shows  " \<exists>x \<in> nset nellx. P x y"
using assms
by (metis in_nset_conv_nnth nellist_all2_conv_all_nnth) 

lemma nellist_all2_nzipI:
 assumes " nellist_all2 P nellx nelly"
         " nellist_all2 P' nellx' nelly'"
 shows   " nellist_all2 (rel_prod P P') (nzip nellx nellx') (nzip nelly nelly')"
using assms 
proof (coinduction arbitrary: nellx nellx' nelly nelly')
case (ilist_all2 nx nx' ny ny') 
then show ?case   
   proof -
    have 1: "is_NNil (nzip nx nx') = is_NNil (nzip ny ny')"
      by (metis ilist_all2(1) ilist_all2(2) is_NNil_nzip nellist_all2_is_NNilD)
    have 2: "(is_NNil (nzip nx nx') \<Longrightarrow>
              is_NNil (nzip ny ny') \<Longrightarrow>
              rel_prod P P' (nlast (nzip nx nx')) (nlast (nzip ny ny')))"
      by (metis (no_types, lifting) enat_min_eq_0_iff ilist_all2(1) ilist_all2(2) 
          is_NNil_nzip min_def_raw nellist.sel(1) nellist_all2_conv_all_nnth 
          nzip_eq_NNil_conv order_refl rel_prod_inject zero_enat_def)
    have 3: "(\<not> is_NNil (nzip nx nx') \<Longrightarrow>
              \<not> is_NNil (nzip ny ny') \<Longrightarrow>
            ((\<exists>nellx nellx' nelly nelly'.
               ntl (nzip nx nx') = nzip nellx nellx' \<and>
               ntl (nzip ny ny') = nzip nelly nelly' \<and>
               nellist_all2 P nellx nelly \<and> nellist_all2 P' nellx' nelly') \<or>
               nellist_all2 (rel_prod P P') (ntl (nzip nx nx')) (ntl (nzip ny ny'))))"
      by (auto simp add: nzip_eq_NCons_conv dest: nellist_all2_nhdD intro: nellist_all2_ntlI)
         (meson ilist_all2(1) ilist_all2(2) nellist_all2_ntlI nzip_ntl) 
    have 4: "\<not> is_NNil (nzip nx nx') \<longrightarrow>
             \<not> is_NNil (nzip ny ny') \<longrightarrow>
             rel_prod P P' (nhd (nzip nx nx')) (nhd (nzip ny ny'))"
      by (simp add: ilist_all2(1) ilist_all2(2) nellist_all2_nhdD nzip_nhd)         
    have 5: "(\<not> is_NNil (nzip nx nx') \<longrightarrow>
              \<not> is_NNil (nzip ny ny') \<longrightarrow>
               rel_prod P P' (nhd (nzip nx nx')) (nhd (nzip ny ny')) \<and>
               ((\<exists>nellx nellx' nelly nelly'.
                  ntl (nzip nx nx') = nzip nellx nellx' \<and>
                  ntl (nzip ny ny') = nzip nelly nelly' \<and>
                  nellist_all2 P nellx nelly \<and> nellist_all2 P' nellx' nelly') \<or>
                  nellist_all2 (rel_prod P P') (ntl (nzip nx nx')) (ntl (nzip ny ny'))))"
      using "3" "4" by blast
    show ?thesis
      using "1" "2" "3" "4" by presburger
  qed
qed

lemma ndistinct_nzipI1:
 " ndistinct nellx \<Longrightarrow> ndistinct (nzip nellx nelly)"
by transfer
   (simp add: ldistinct_lzipI1)

lemma ndistinct_nzipI2:
 " ndistinct nelly \<Longrightarrow> ndistinct (nzip nellx nelly)"
by transfer
   (simp add: ldistinct_lzipI2)


subsection \<open>@{term "niterates"}\<close>

lemma niterates_not_is_NNil [nitpick_simp, simp]:
 " \<not> is_NNil (niterates f x)"
by transfer
   (metis lfinite_LConsI lfinite_code(1) lfinite_iterates)

lemma nhd_niterates [code, simp, nitpick_simp]:
 " nhd(niterates f x) = x"
by transfer
   (metis lfinite_LConsI lfinite_LNil lfinite_iterates lhd_iterates)

lemma ntl_niterates [code, simp, nitpick_simp]:
 " ntl(niterates f x) = niterates f (f x)"
by transfer
   (simp,
    metis lfinite_LConsI lfinite_LNil lfinite_iterates)

lemma nfinite_niterates [iff]:
 " \<not> nfinite (niterates f x)"
by transfer  simp

lemma niterates_nmap:
 " niterates f x = NCons x (nmap f (niterates f x))"
by transfer
   (meson iterates.disc_iff iterates_lmap)

lemma [simp]:
  fixes f :: "'a \<Rightarrow> 'a"
  shows is_NNil_funpow_nmap: "is_NNil ((nmap f ^^ n) nellx) \<longleftrightarrow> is_NNil nellx"
  and nhd_funpow_nmap: "\<not> is_NNil nellx \<Longrightarrow> nhd ((nmap f ^^ n) nellx) = (f ^^ n) (nhd nellx)"
  and ntl_funpow_nmap: "\<not> is_NNil nellx \<Longrightarrow> ntl ((nmap f ^^ n) nellx) = (nmap f ^^ n) (ntl nellx)"
by (induct n) simp_all

lemma niterates_equality:
 assumes h: " \<And>x. h x = NCons x (nmap f (h x))"
 shows   " h = niterates f"
proof -
 { fix x
   have "\<not> is_NNil (h x)" "nhd (h x) = x"  "ntl (h x) = nmap f (h x)"
    by (subst h, simp)+ }
 note [simp] = this
 { fix x
   define n :: nat where "n = 0" 
   have "(nmap f ^^ n) (h x) = (nmap f ^^ n) (niterates f x) "
     proof (coinduction arbitrary: n)
     case (Eq_nellist nn) 
     then show ?case 
       proof -
        have 1: "is_NNil ((nmap f ^^ nn) (h x)) = is_NNil ((nmap f ^^ nn) (niterates f x))"
          by auto
        have 2: "(is_NNil ((nmap f ^^ nn) (h x)) \<longrightarrow>
                  is_NNil ((nmap f ^^ nn) (niterates f x)) \<longrightarrow>
                  nlast ((nmap f ^^ nn) (h x)) = nlast ((nmap f ^^ nn) (niterates f x)))"
          by simp
        have 3: "(\<not> is_NNil ((nmap f ^^ nn) (h x)) \<longrightarrow>
                  \<not> is_NNil ((nmap f ^^ nn) (niterates f x)) \<longrightarrow>
                  nhd ((nmap f ^^ nn) (h x)) = nhd ((nmap f ^^ nn) (niterates f x)))"
          by simp   
        have 4: "(\<not> is_NNil ((nmap f ^^ nn) (h x)) \<longrightarrow>
                  \<not> is_NNil ((nmap f ^^ nn) (niterates f x)) \<longrightarrow>
                  ( ntl ((nmap f ^^ nn) (h x)) = (nmap f ^^ (Suc nn)) (h x) \<and>
                       ntl ((nmap f ^^ nn) (niterates f x)) = (nmap f ^^ (Suc nn)) (niterates f x))) " 
          by (metis \<open>\<And>x. \<not> is_NNil (h x)\<close> \<open>\<And>x. ntl (h x) = nmap f (h x)\<close> funpow_simps_right(2) 
              nellist.sel(5) niterates_nmap niterates_not_is_NNil ntl_funpow_nmap o_apply)
        show ?thesis using "1" "2" "3" "4" by blast
       qed
     qed
    }
   thus ?thesis by auto
qed
           
lemma nlength_niterates [simp]:
 "nlength (niterates f x) = \<infinity>"
by transfer auto 

lemma ndropn_niterates:
 " ndropn n (niterates f x) = niterates f ((f ^^ n) x)"
by transfer
   (simp add: ldropn_iterates)

lemma nnth_niterates [simp]:
 " nnth (niterates f x) n = (f ^^ n) x"
by transfer auto

lemma nset_niterates:
 " nset (niterates f x) = { (f ^^ n) x|n. True}"
by transfer
   (metis lset_iterates)

lemma nnth_niterates_Suc: 
 "nnth (niterates Suc 0) i = i "
proof (induct i)
case 0
then show ?case 
by force
next
case (Suc i)
then show ?case by simp 
qed


subsection \<open>Filtering non-empty lazy  lists @{term "nfilter"}\<close>

lemma nfilter_NNil [simp]:
shows  "nfilter  P (NNil b) = NNil b"
by transfer auto

lemma nfilter_True [simp]:
 shows "nfilter (\<lambda>x. True) nell = nell"
by transfer auto

lemma nfilter_False_finite:
assumes "nfinite nell" 
 shows " nfilter (\<lambda> x. False) nell =  nell"
using assms by transfer auto

lemma nfilter_NCons [simp]:
assumes " (\<exists> x \<in> nset nell. P x)"  
 shows "nfilter P (NCons x nell) = (if P x then NCons x (nfilter P nell) else nfilter P nell)"
using assms by transfer auto

lemma nfilter_NCons_a [simp]:
assumes " \<not>(\<exists> x \<in> nset nell. P x)"  
        "P x" 
 shows "nfilter P (NCons x nell) = (NNil x)"
using assms by transfer auto

lemma nfilter_expand:
assumes "\<exists> x \<in> nset nell. P x"
shows " nfilter P nell =
        (if is_NNil nell then nell 
         else 
         (if (\<exists>x \<in> nset(ntl nell). P x) then 
           (if P (nhd nell) then (NCons (nhd nell) (nfilter P (ntl nell))) 
                            else (nfilter P (ntl nell) ) )
          else (NNil (nhd nell)) )) "
using assms by (cases nell) auto

lemma nset_nfilter:
 assumes "\<exists> x \<in> nset nell. P x " 
 shows "nset (nfilter P nell) = nset nell \<inter> {xa. P xa}"
using assms
by transfer auto

lemma exist_conj:
assumes "\<exists> x \<in> nset (nfilter Q nell). P x"
        "\<exists> x \<in> nset nell. Q x"
shows "\<exists> x \<in> nset nell. P x \<and> Q x"
 using assms
by transfer (auto split: if_split_asm) 
  
lemma nfilter_nfilter [simp]:
assumes "\<exists> x \<in> nset (nfilter Q nell). P x"
        "\<exists> x \<in> nset nell. Q x"
shows "nfilter P (nfilter Q nell) = nfilter (\<lambda>x. P x \<and> Q x) nell"
using assms
proof (transfer fixing: P Q)
fix xsa :: "'a llist"
assume "\<not> lnull xsa \<and> xsa = xsa"
assume a1: "Bex (lset (if lnull (lfilter Q xsa) then xsa else lfilter Q xsa)) P"
assume "Bex (lset xsa) Q"
then have "\<not> lnull (lfilter Q xsa)"
by simp
then have "lfilter P (if lnull (lfilter Q xsa) then xsa else lfilter Q xsa) = lfilter P (lfilter Q xsa) \<and> 
           \<not> lnull (lfilter P (lfilter Q xsa)) \<and> 
           \<not> lnull (if lnull (lfilter P (if lnull (lfilter Q xsa) then xsa else lfilter Q xsa))
                    then if lnull (lfilter Q xsa) then xsa else lfilter Q xsa 
                    else lfilter P (if lnull (lfilter Q xsa) then xsa else lfilter Q xsa))"
using a1 by (auto split: if_split_asm)   
then show "\<not> lnull (if lnull (lfilter P (if lnull (lfilter Q xsa) then xsa else lfilter Q xsa))
                    then if lnull (lfilter Q xsa) then xsa else lfilter Q xsa 
                     else lfilter P (if lnull (lfilter Q xsa) then xsa else lfilter Q xsa)) \<and> 
            (if lnull (lfilter P (if lnull (lfilter Q xsa) then xsa else lfilter Q xsa)) 
             then if lnull (lfilter Q xsa) then xsa else lfilter Q xsa 
              else lfilter P (if lnull (lfilter Q xsa) then xsa else lfilter Q xsa)) = 
            (if lnull (lfilter (\<lambda>a. P a \<and> Q a) xsa) then xsa else lfilter (\<lambda>a. P a \<and> Q a) xsa)"
using lfilter_lfilter by auto
qed

lemma length_nfilter_le [simp]: 
 "nlength (nfilter P nell) \<le> nlength nell"
by transfer (simp add: epred_le_epredI llength_lfilter_ile)

lemma nfilter_nnth:
 assumes "(\<exists> x \<in> nset nell. P x)"
         "i\<le> nlength (nfilter P nell)"
 shows   "(\<exists> k \<le> nlength nell. nnth(nfilter P nell) i = nnth nell k)"
using assms nset_nfilter[of nell P]
using in_nset_conv_nnth
by (metis Int_iff)

lemma nfilter_nappend1:
assumes "\<forall>x\<in>nset nell. \<not> P x"
        "nfinite nell"
        "\<exists> y \<in> nset nell1. P y"
shows   "nfilter P (nappend nell  nell1) = nfilter P nell1"
using assms by transfer auto

lemma nfilter_nappend2:
assumes "\<forall>x\<in>nset nell. \<not> P x"
        "\<exists>y\<in>nset nell1. P y"
shows   "nfilter P (nappend nell1  nell) = nfilter P nell1"
using assms
by transfer  
   (auto split: if_split_asm,
    meson in_lset_lappend_iff,
    metis lappend_LNil2 lappend_inf lfilter_empty_conv lfilter_lappend_lfinite) 

lemma nfilter_nappend [simp]:
assumes "(\<exists> x \<in> nset (nappend nell  nell1). P x)"
        "(\<exists> x \<in> nset nell. P x)"
        "(\<exists> x \<in> nset nell1. P x)"
shows "nfilter P (nappend nell nell1) = 
       (if nfinite nell then nappend (nfilter P nell)  (nfilter P nell1) 
        else  (nfilter P nell) )"
proof (cases "nfinite nell")
case True
then show ?thesis using assms by transfer auto
next
case False
then show ?thesis using assms by transfer (auto simp add: lappend_inf)
qed

lemma nfilter_nmap:
shows   " nfilter P (nmap f nell) = nmap f (nfilter (P\<circ>f) nell)"
by transfer  (auto simp add: lfilter_lmap)

lemma nlength_nfilter_nmap[simp]:
shows  "nlength (nfilter P (nmap f nell)) = nlength(nfilter (P \<circ> f) nell)"
by (simp add: nfilter_nmap)

lemma nfilter_is_subset [simp]:
 assumes "\<exists> x \<in> nset nell. P x"
 shows   "nset (nfilter P nell) \<le> nset nell"
using assms by (simp add: nset_nfilter)

lemma nfilter_cong[fundef_cong]:
 assumes "nell = nell1"
         " (\<And>x. x \<in> nset nell1 \<Longrightarrow> P x = Q x) "
 shows   " nfilter P nell = nfilter Q nell1"
using assms by transfer auto   


lemma nset_nappend:
 " nset (nappend nell nell1) = (if nfinite nell then nset nell \<union> nset nell1 else nset nell)"
by transfer (simp add: lappend_inf)

lemma split_nellist_nappend:
assumes "\<exists> i \<le> nlength nell. nnth nell i = x \<and> P (nnth nell i) \<and> 
                           (\<forall> j. j \<noteq> i \<and> j \<le> nlength nell \<longrightarrow> \<not> P(nnth nell j))"         
shows "P x \<and> 
      ( nell = (NNil x) \<or> 
        (\<exists> vs. nell = (NCons x vs) \<and> (\<forall> v \<in> nset vs. \<not> P v)) \<or>
        (\<exists> us. nell = nappend us  (NNil x) \<and> nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u)) \<or>
        (\<exists> us vs . nell = nappend us  (NCons x  vs) \<and> nfinite us \<and> 
                   (\<forall> u \<in> nset us. \<not> P u) \<and> (\<forall> v \<in> nset vs. \<not> P v))
      ) "
proof -
 obtain i where 1: " i \<le> nlength nell \<and> nnth nell i = x \<and> P (nnth nell i) \<and> 
                     (\<forall> j. j \<noteq> i \<and> j \<le> nlength nell \<longrightarrow> \<not> P(nnth nell j))"
   using assms by auto
 have 01: "(\<forall> j. (j<i \<or> i<j) \<and> j \<le> nlength nell \<longrightarrow> \<not> P(nnth nell j))" 
   using 1 
   by auto
 have 2: "i=0 \<and> nlength nell = 0 \<Longrightarrow> P x \<and> nell = (NNil x)"
      by (metis "1" ndropn_0 ndropn_eq_NNil the_enat_0 zero_enat_def)
 have 3: "i=0 \<and>nlength nell > 0 \<Longrightarrow> P x \<and>  nell = (NCons x (ntl nell))  \<and> (\<forall> v \<in> nset (ntl nell). \<not> P v)" 
   using 1 in_nset_conv_nnth[of _ "ntl nell"] 
   by (metis Suc_ile_eq iless_Suc_eq nat.simps(3) nellist.disc(2) nellist.exhaust_sel nlength_NCons 
       nlength_NNil nnth_0_conv_nhd nnth_ntl not_less_iff_gr_or_eq)
 have 4: " i=0 \<and>nlength nell > 0 \<Longrightarrow> P x \<and> (\<exists> vs. nell = (NCons x vs) \<and> (\<forall> v \<in> nset vs. \<not> P v))"
   using 3 by auto
 have 5: "i >0 \<and> i = nlength nell \<and> nfinite nell \<Longrightarrow> P x \<and> nell = nappend (ntaken (i-1) nell) (NNil x)  " 
   using 1  
   by transfer 
      (auto,
       metis eSuc_epred lappend_lbutlast_llast_id_lfinite lbutlast_conv_ltake llast_conv_lnth 
       llength_eq_0 the_enat.simps)
 have 6: "i >0 \<and> i = nlength nell \<and> nfinite nell \<Longrightarrow>  (\<forall> u \<in> nset (ntaken (i-1) nell). \<not> P u) "
   using 1 
   by  transfer
       (auto,
        metis in_lset_conv_lnth lbutlast_conv_ltake less_imp_le llength_lbutlast lnth_ltake 
        min.strict_order_iff)
 have 7: " i >0 \<and> i = nlength nell \<and> nfinite nell \<Longrightarrow> P x \<and> (\<exists> us. nell = nappend us  (NNil x) \<and> 
           nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u)) "
   using 5 6 by auto
 have 8: " i >0 \<and> i < nlength nell \<Longrightarrow> 
            P x \<and> nell = nappend (ntaken (i-1) nell) (NCons x (ndropn (i+1) nell)) " 
   using 1   
   by transfer
      (auto,
       metis co.enat.collapse eSuc_enat ileI1 iless_Suc_eq lappend_ltake_enat_ldropn 
       ldropn_Suc_conv_ldropn llength_eq_0 min.absorb1 the_enat.simps)
 have 9: " i >0 \<and> i < nlength nell \<Longrightarrow> nfinite (ntaken (i-1) nell)  "
   using enat_ord_code(4) nfinite_ntaken by blast
 have 10: "i >0 \<and> i < nlength nell \<Longrightarrow> (\<forall> u \<in> nset (ntaken (i-1) nell). \<not> P u) "
   using 01 in_nset_conv_nnth[of _ "(ntaken (i-1) nell)" ] by simp
   (metis Suc_pred le_imp_less_Suc min.orderE ntaken_nnth) 
 have 11: "i >0 \<and> i < nlength nell \<Longrightarrow> (\<forall> v \<in> nset (ndropn (i+1) nell). \<not> P v) "   
   using 1 01 in_nset_conv_nnth[of _ "(ndropn (i+1) nell)" ] 
   by simp
     (metis Suc_ile_eq iless_Suc_eq is_NNil_ndropn le_add1 le_imp_less_Suc ndropn_Suc_conv_ndropn 
       ndropn_ndropn ndropn_nlength nlength_NCons not_less)
 have 12: " i >0 \<and> i < nlength nell \<Longrightarrow> (\<exists> us vs . nell = nappend us  (NCons x  vs) \<and> nfinite us \<and> 
            (\<forall> u \<in> nset us. \<not> P u) \<and> (\<forall> v \<in> nset vs. \<not> P v))"
   using 8 9 10 11 by blast
 show ?thesis 
   by (metis "1" "12" "2" "4" "7" dual_order.order_iff_strict neq0_conv nlength_eq_enat_nfiniteD 
       zero_enat_def)        
qed         

lemma nellist_split_2_first:
 assumes " 0 < nlength nell" 
shows " nell = (NCons (nnth nell 0) (ntl nell))" 
using assms by (metis ndropn_0 ndropn_Suc_conv_ndropn nellist.sel(5) zero_enat_def)

lemma nellist_split_2_last:
 assumes "0< i"
         " i = nlength nell"
         "  nfinite nell "
 shows " nell = nappend (ntaken (i-1) nell) (NNil (nnth nell i)) "
using assms 
by transfer
   (simp,
    metis Suc_ile_eq co.enat.exhaust_sel eSuc_enat llength_eq_0 ltake_Suc_conv_snoc_lnth ltake_all 
    order_refl the_enat.simps)

lemma nellist_split_3:
 assumes " 0 < i"
         " i < nlength nell" 
 shows   " nell = nappend (ntaken (i-1) nell) (NCons (nnth nell i) (ndropn (i+1) nell)) "
using assms by transfer
  (auto, 
   metis eSuc_enat eSuc_epred ileI1 iless_Suc_eq lappend_ltake_enat_ldropn ldropn_Suc_conv_ldropn 
   less_imp_le llength_eq_0 min_def the_enat.simps)

lemma NNil_eq_nfilterD:
 assumes " \<exists> x \<in> nset nell.  P x"
         " (NNil x) = nfilter P nell  "
 shows   "(\<exists> us vs . (nell = (NNil x) \<or>
         (nell = (NCons x vs) \<and> (\<forall> v \<in> nset vs. \<not> P v)) \<or>
         (nell = nappend us  (NNil x) \<and> nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u)) \<or>
         (nell = nappend us  (NCons x  vs) \<and> nfinite us \<and> 
            (\<forall> u \<in> nset us. \<not> P u) \<and> (\<forall> v \<in> nset vs. \<not> P v))
        ) \<and> P x)"
proof -
 have 1: " (\<exists> i \<le> nlength nell. P (nnth nell i)) "
   by (metis assms(1) in_nset_conv_nnth)
 obtain i where 2: " i \<le> nlength nell \<and> P (nnth nell i)"
   using 1 by auto 
 have 3: "i = 0 \<and> nlength nell = 0 \<Longrightarrow> P x \<and> nell = (NNil x)"
   by (metis "2" assms(2) ndropn_0 ndropn_eq_NNil nfilter_NNil the_enat_0 zero_enat_def)
 have 4: " i=0 \<and>nlength nell > 0 \<Longrightarrow>P x \<and>  nell = (NCons x (ntl nell)) \<and> (\<forall> v \<in> nset (ntl nell). \<not> P v)"
   using 2 assms nellist_split_2_first[of nell] nfilter_expand[of nell P] 
   by (metis nellist.distinct(1) nellist.sel(3) nlast_NNil) 
 have 5: "i=0 \<and>nlength nell > 0 \<Longrightarrow> P x \<and> (\<exists> vs. nell = (NCons x vs) \<and> (\<forall> v \<in> nset vs. \<not> P v)) " 
   using 4 by auto
 have 60: "0< i  \<and> i = nlength nell \<and> nfinite nell \<Longrightarrow> x= (nnth nell i)"
   using 2 assms nellist_split_2_last[of i nell] 
   proof  simp
   assume a1: "nell = nappend (ntaken (i - Suc 0) nell) (NNil (nnth nell i))"
   assume a2: "P (nnth nell i)"
   assume a3: "NNil x = nfilter P nell"
   assume a4: "\<exists>x\<in>nset nell. P x"
   have f5: "\<forall>as p asa. ((\<exists>a. (a::'a) \<in> nset as \<and> p a) \<or> \<not> nfinite as \<or> (\<forall>a. a \<notin> nset asa \<or> \<not> p a))
               \<or> nfilter p (nappend as asa) = nfilter p asa"
   by (metis (full_types) nfilter_nappend1)
   have f7: "\<exists>a. a \<in> nset (NNil (nnth nell i)) \<and> P a"
   using a2 by auto
   have f8: "nfilter P (nappend (ntaken (i - Suc 0) nell) (NNil (nnth nell i))) \<noteq> 
             nappend (nfilter P (ntaken (i - Suc 0) nell)) (nfilter P (NNil (nnth nell i)))"
   using a3 a1 by (metis (full_types) is_NNil_nappend nellist.disc(1))
   have f9: "nfinite (ntaken (i - Suc 0) nell)"
   by simp 
   obtain aaa :: 'a where
   f9: "aaa \<in> nset nell \<and> P aaa"
   using a4 by blast
   then have "aaa \<in> nset (nappend (ntaken (i - Suc 0) nell) (NNil (nnth nell i)))"
   using a1 by presburger
   then have f10: "\<forall>a. a \<notin> nset (ntaken (i - Suc 0) nell) \<or> \<not> P a"
   using f9 f8 f7 by (meson nfilter_nappend nfinite_ntaken)
   then show ?thesis
   using f9 f7 f5 a3 a1 
    proof -
    have "NNil (nnth nell i) = NNil x"
    using nfilter_NNil[of P "(nnth nell i)" ] by (metis (no_types) f10 a1 a3 f5 f7 nfinite_ntaken)
    then show ?thesis
    by blast
    qed
   qed 
 have 6: " 0< i  \<and> i = nlength nell \<and> nfinite nell \<Longrightarrow> P x \<and>  nell = nappend (ntaken (i-1) nell) (NNil x)"
   using 2 60 assms nellist_split_2_last[of i nell] 
   by blast
 have 7: "i >0 \<and> i = nlength nell \<and> nfinite nell \<Longrightarrow> (\<forall> u \<in> nset (ntaken (i-1) nell). \<not> P u)"
   using assms 6 nfilter_nappend[of "(ntaken (i-1) nell)" _ P] 
   by (metis is_NNil_nappend nellist.disc(1) nfinite_ntaken split_nellist_a)      
 have 8: "i >0 \<and> i = nlength nell \<and> nfinite nell \<Longrightarrow> P x \<and> (\<exists> us. nell = nappend us  (NNil x) \<and> 
          nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u))"
   using 6 7 by auto     
 have 9: "i >0 \<and> i < nlength nell \<Longrightarrow>  
           P x \<and> nell = nappend (ntaken (i-1) nell) (NCons x (ndropn (i+1) nell))"
   using 2 assms nellist_split_3[of i nell] 
     nfilter_nappend1[of "(ntaken (i-1) nell)" P "(NCons x (ndropn (i+1) nell))"]
     nfilter_nappend[of "(ntaken (i - 1) nell)" _ P]  
     proof simp
     assume a1: "enat i \<le> nlength nell \<and> P (nnth nell i)"
     assume a2: "NNil x = nfilter P nell"
     assume a3: "\<exists>x\<in>nset nell. P x"
     assume a4: "nell = nappend (ntaken (i - Suc 0) nell) (NCons (nnth nell i) (ndropn (Suc i) nell))"
     have f5: "P (nnth (nappend (ntaken (i - Suc 0) nell) (NCons (nnth nell i) (ndropn (Suc i) nell))) i)"
     using a1 a4 by auto
     have f6: "NNil x = nfilter P (nappend (ntaken (i - Suc 0) nell) (NCons (nnth nell i) (ndropn (Suc i) nell)))"
     using a2 a4 by auto
     have f7: "\<forall>as p asa. ((\<exists>a. (a::'a) \<in> nset as \<and> p a) \<or> \<not> nfinite as \<or> (\<forall>a. a \<notin> nset asa \<or> \<not> p a)) \<or> 
                            nfilter p (nappend as asa) = nfilter p asa"
     by (metis (full_types) nfilter_nappend1)
     have "nfilter P (nappend (ntaken (i - Suc 0) nell) (NCons (nnth nell i) (ndropn (Suc i) nell))) = 
           nappend (nfilter P (ntaken (i - Suc 0) nell)) (nfilter P (NCons (nnth nell i) (ndropn (Suc i) nell)))
           \<longrightarrow> 
           nappend (nfilter P (ntaken (i - Suc 0) nell)) (nfilter P (NCons (nnth nell i) (ndropn (Suc i) nell))) =
           NNil x"
     using f6 by presburger
     then have "nfilter P (nappend (ntaken (i - Suc 0) nell) (NCons (nnth nell i) (ndropn (Suc i) nell))) \<noteq> 
                nappend (nfilter P (ntaken (i - Suc 0) nell)) (nfilter P (NCons (nnth nell i) (ndropn (Suc i) nell)))"
     by (metis (no_types) is_NNil_nappend nellist.disc(1))
     then have f9: "(\<forall>a. a \<notin> nset (NCons (nnth nell i) (ndropn (Suc i) nell)) \<or> \<not> P a) \<or> 
                     nfilter P (nappend (ntaken (i - Suc 0) nell) (NCons (nnth nell i) (ndropn (Suc i) nell))) = 
                     nfilter P (NCons (nnth nell i) (ndropn (Suc i) nell)) \<or> 
                    (\<forall>a. a \<notin> nset (nappend (ntaken (i - Suc 0) nell) (NCons (nnth nell i) (ndropn (Suc i) nell))) 
                        \<or> \<not> P a)"
     using nfilter_nappend[of "(ntaken (i - Suc 0) nell)" "(NCons (nnth nell i) (ndropn (Suc i) nell))" P]  
           nfinite_ntaken[of "(i - Suc 0)" nell]
     by (metis f7) 
     obtain aaa :: 'a where f10: "aaa \<in> nset nell \<and> P aaa"
     using a3 by blast
     then have f11: "aaa \<in> nset (nappend (ntaken (i - Suc 0) nell) (NCons (nnth nell i) (ndropn (Suc i) nell)))"
     using a4 by presburger
     have f12: "nnth (nappend (ntaken (i - Suc 0) nell) (NCons (nnth nell i) (ndropn (Suc i) nell))) i \<in>
                nset (NCons (nnth nell i) (ndropn (Suc i) nell))"
     using a4 by fastforce
     then have "NNil x = nfilter P (NCons (nnth nell i) (ndropn (Suc i) nell))"
     using f11 f10 f9 f6 f5 by (metis (no_types))
     then have "NNil x = nfilter P (NCons (nnth (nappend (ntaken (i - Suc 0) nell) 
                                                         (NCons (nnth nell i) (ndropn (Suc i) nell)))
                                                 i) (ndropn (Suc i) nell))"
     using a4 by presburger
     then have f13: "\<forall>a. a \<notin> nset (ndropn (Suc i) nell) \<or> \<not> P a"
     using f5 by (metis (full_types) nellist.distinct(1) nfilter_NCons)
     have "nfilter P (NCons (nnth (nappend (ntaken (i - Suc 0) nell)
                                           (NCons (nnth nell i) (ndropn (Suc i) nell))) 
                                   i) (ndropn (Suc i) nell)) = NNil x"
     using \<open>NNil x = nfilter P (NCons (nnth nell i) (ndropn (Suc i) nell))\<close> a4 by presburger
     then have "x = nnth (nappend (ntaken (i - Suc 0) nell) (NCons (nnth nell i) (ndropn (Suc i) nell))) i"
     using f13 f5 by simp
     then have f14: "x = nnth nell i"
     using a4 by presburger
     then have "nell = nappend (ntaken (i - Suc 0) nell) (NCons x (ndropn (Suc i) nell))"
     using a4 by blast
     then show "P x \<and> nell = nappend (ntaken (i - Suc 0) nell) (NCons x (ndropn (Suc i) nell))"
     using f14 a1 by blast
   qed  
 have 10: "i >0 \<and> i < nlength nell \<Longrightarrow> (\<forall> u \<in> nset (ntaken (i-1) nell). \<not> P u)"
   using 9 assms  nfilter_nappend[of "(ntaken (i-1) nell)" _ P] 
   by (metis is_NNil_nappend nellist.disc(1) nellist.set_intros(2) nfinite_ntaken)
 have 11: "i >0 \<and> i < nlength nell \<Longrightarrow> (\<forall> v \<in> nset (ndropn (i+1) nell). \<not> P v)"
   using 9 10 assms  nfilter_nappend[of "(ntaken (i-1) nell)" _ P] 
                     nfilter_nappend1[of  "(ntaken (i-1) nell)" P "(NCons x (ndropn (i+1) nell))"]
   by (metis nellist.distinct(1) nellist.set_intros(2) nfilter_NCons nfinite_ntaken)
 have 12: "i >0 \<and> i < nlength nell \<Longrightarrow> 
           P x \<and> (\<exists> us vs . nell = nappend us  (NCons x  vs) \<and> nfinite us \<and> 
           (\<forall> u \<in> nset us. \<not> P u) \<and> (\<forall> v \<in> nset vs. \<not> P v))  "
   using 9 10 11 using assms(1) by fastforce 
 show ?thesis 
  by (metis "12" "2" "3" "5" "8" dual_order.order_iff_strict neq0_conv nlength_eq_enat_nfiniteD 
      zero_enat_def)
qed

lemma nfilter_eq_NNilD:
 assumes " \<exists> x \<in> nset nell.  P x"
         "  nfilter P nell  = (NNil x) "
 shows   "(\<exists> us vs . (nell = (NNil x) \<or>
         (nell = (NCons x vs) \<and> (\<forall> v \<in> nset vs. \<not> P v)) \<or>
         (nell = nappend us  (NNil x) \<and> nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u)) \<or>
         (nell = nappend us  (NCons x  vs) \<and> nfinite us \<and> 
            (\<forall> u \<in> nset us. \<not> P u) \<and> (\<forall> v \<in> nset vs. \<not> P v))
        ) \<and> P x)"
using assms NNil_eq_nfilterD[of nell P x] by simp

lemma nfilter_eq_NNil_iff:
  assumes " \<exists> x \<in> nset nell.  P x"
  shows " (nfilter P nell  = (NNil x)) \<longleftrightarrow>
          (\<exists> us vs . (nell = (NNil x) \<or>
         (nell = (NCons x vs) \<and> (\<forall> v \<in> nset vs. \<not> P v)) \<or>
         (nell = nappend us  (NNil x) \<and> nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u)) \<or>
         (nell = nappend us  (NCons x  vs) \<and> nfinite us \<and> 
         (\<forall> u \<in> nset us. \<not> P u) \<and> (\<forall> v \<in> nset vs. \<not> P v))
        ) \<and> P x)"
proof -
 have 1: "(nfilter P nell  = (NNil x)) \<Longrightarrow>
          (\<exists> us vs . (nell = (NNil x) \<or>
         (nell = (NCons x vs) \<and> (\<forall> v \<in> nset vs. \<not> P v)) \<or>
         (nell = nappend us  (NNil x) \<and> nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u)) \<or>
         (nell = nappend us  (NCons x  vs) \<and> nfinite us \<and> 
           (\<forall> u \<in> nset us. \<not> P u) \<and> (\<forall> v \<in> nset vs. \<not> P v))
        ) \<and> P x)"
   using assms  nfilter_eq_NNilD[of nell P x] by simp
 have 2: "(\<exists> us vs . (nell = (NNil x) \<or>
         (nell = (NCons x vs) \<and> (\<forall> v \<in> nset vs. \<not> P v)) \<or>
         (nell = nappend us  (NNil x) \<and> nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u)) \<or>
         (nell = nappend us  (NCons x  vs) \<and> nfinite us \<and> 
           (\<forall> u \<in> nset us. \<not> P u) \<and> (\<forall> v \<in> nset vs. \<not> P v))
        ) \<and> P x) \<Longrightarrow> (nfilter P nell  = (NNil x))"
   using nfilter_NCons_a nfilter_NNil[of P x] 
   by (auto simp add: nfilter_nappend1)
 show ?thesis
   using "1" "2" by blast 
qed

lemma NCons_eq_nfilterD_help:
assumes "\<not> lnull ys " 
        "\<not> lnull xs "
        "LCons x xs = lfilter P ys "
        " xa \<in> lset ys  "
        "P xa " 
shows "\<exists>us. \<not> lnull us \<and>
            (\<exists>vs. (\<exists>x\<in>lset vs. P x) \<and>
                  ((\<exists>x\<in>lset vs. P x) \<longrightarrow>
                   \<not> lnull vs \<and> (ys = LCons x vs \<or> ys = lappend us (LCons x vs) \<and> lfinite us \<and> 
                                  (\<forall>u\<in>lset us. \<not> P u)) \<and> P x \<and> xs = lfilter P vs)) "
unfolding lnull_def using assms lfilter_eq_LConsD[of P ys x xs]
by (metis diverge_lfilter_LNil lappend_code(1) lfilter_LNil llist.disc(1)) 

lemma NCons_eq_nfilterD:
 assumes " \<exists> x \<in> nset nell.  P x"
         " (NCons x nell1) = nfilter P nell " 
 shows   " (\<exists> us vs. 
            ( nell = (NCons x vs) \<or>
              nell = nappend us (NCons x vs) \<and> nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u))  \<and>
              (\<exists> x \<in> nset vs. P x) \<and> P x \<and> nell1 = nfilter P vs ) "
using assms by transfer (auto  split: if_split_asm simp add: NCons_eq_nfilterD_help)   

lemma nfilter_eq_NConsD:
 assumes " \<exists> x \<in> nset nell.  P x"
         "   nfilter P nell = (NCons x nell1)" 
 shows   " (\<exists> us vs. 
            ( nell = (NCons x vs) \<or>
              nell = nappend us (NCons x vs) \<and> nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u))  \<and>
              (\<exists> x \<in> nset vs. P x) \<and> P x \<and> nell1 = nfilter P vs ) "
using assms  NCons_eq_nfilterD[of nell P x nell1] by simp

lemma nfilter_eq_NCons_iff:
  assumes " \<exists> x \<in> nset nell.  P x"
  shows "    nfilter P nell = (NCons x nell1) \<longleftrightarrow> 
             (\<exists> us vs. 
            ( nell = (NCons x vs) \<or>
              nell = nappend us (NCons x vs) \<and> nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u))  \<and>
              (\<exists> x \<in> nset vs. P x) \<and> P x \<and> nell1 = nfilter P vs ) "
proof -
 have 1: "  nfilter P nell = (NCons x nell1) \<Longrightarrow>
           (\<exists> us vs. 
            ( nell = (NCons x vs) \<or>
              nell = nappend us (NCons x vs) \<and> nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u))  \<and>
              (\<exists> x \<in> nset vs. P x) \<and> P x \<and> nell1 = nfilter P vs )"
   using assms nfilter_eq_NConsD[of nell P x nell1] by simp
 have 2: "(\<exists> us vs. 
            ( nell = (NCons x vs) \<or>
              nell = nappend us (NCons x vs) \<and> nfinite us \<and> (\<forall> u \<in> nset us. \<not> P u))  \<and>
              (\<exists> x \<in> nset vs. P x) \<and> P x \<and> nell1 = nfilter P vs ) \<Longrightarrow> nfilter P nell = (NCons x nell1)"
   using nfilter_nappend1[of _ P]  nfilter_NCons[of _  P x]
   by (auto, meson nfilter_NCons, metis)  
 show ?thesis using 1 2 by blast
qed

lemma nfilter_id_conv:
assumes " (\<exists> x \<in> nset nell. P x) "
 shows "(nfilter P nell = nell) = (\<forall>x\<in>nset nell. P x)" (is "?lhs = ?rhs")
using assms by transfer (auto simp add: lfilter_id_conv)

lemma nfilter_idem:
 assumes " (\<exists> x \<in> nset nell. P x) "
 shows   " nfilter P (nfilter P nell) = nfilter P nell"
using assms by transfer auto
             
lemma nfilter_ndropn_nlength:
 assumes "k \<le> nlength nell"
         "\<exists> x \<in> nset ( (ndropn k nell)). P x"
 shows   " nlength(nfilter P ( (ndropn k nell))) \<le> nlength (nfilter P ( nell))"
using assms
by transfer
 (auto simp add: min_def dest: in_lset_ldropnD, 
  metis co.enat.exhaust_sel epred_le_epredI iless_Suc_eq lfilter_ldropn_llength llength_eq_0)




subsection \<open>Setup for Lifting/Transfer\<close>

subsubsection \<open>Relator and predicator properties\<close>

abbreviation "nellist_all == pred_nellist"

subsubsection \<open>Transfer rules for the Transfer package\<close>

context includes lifting_syntax
begin

lemma set1_pre_nellist_transfer [transfer_rule]:
  "(rel_pre_nellist A C ===> rel_set A) set1_pre_nellist set1_pre_nellist"
by(auto simp add: rel_pre_nellist_def vimage2p_def rel_fun_def set1_pre_nellist_def rel_set_def 
   collect_def sum_set_defs prod_set_defs elim: rel_sum.cases split: sum.split_asm)

lemma set2_pre_nellist_transfer [transfer_rule]:
  "(rel_pre_nellist A  C ===> rel_set C) set2_pre_nellist set2_pre_nellist"
by (auto simp add:  rel_pre_nellist_def vimage2p_def rel_fun_def set2_pre_nellist_def 
    rel_set_def collect_def sum_set_defs prod_set_defs elim: rel_sum.cases split: sum.split_asm)

lemma NCons_transfer2 [transfer_rule]:
  "(A ===> nellist_all2 A  ===> nellist_all2 A ) NCons NCons"
unfolding rel_fun_def by simp
declare NCons_transfer [transfer_rule]

lemma case_nellist_transfer [transfer_rule]:
  "((A ===> C) ===> (A ===> nellist_all2 A ===> C) ===> nellist_all2 A ===> C)
    case_nellist case_nellist"
unfolding rel_fun_def
by (simp add: nellist_all2_NNil1 nellist_all2_NNil2 split: nellist.split)

lemma unfold_nellist_transfer [transfer_rule]:
  "((A ===> (=)) ===> (A ===> C) ===> (A ===> C) ===> (A ===> A) ===> A ===> nellist_all2 C)
     unfold_nellist unfold_nellist"
proof(rule rel_funI)+
  fix IS_NNIL1 :: "'a \<Rightarrow> bool" and IS_NNIL2
    NLAST1 NLAST2 NHD1 NHD2 NTL1 NTL2 x y
  assume rel: "(A ===> (=)) IS_NNIL1 IS_NNIL2" "(A ===> C) NLAST1 NLAST2"
    "(A ===> C) NHD1 NHD2" "(A ===> A) NTL1 NTL2"
    and "A x y"
  show "nellist_all2 C  (unfold_nellist IS_NNIL1 NLAST1 NHD1 NTL1 x) 
                        (unfold_nellist IS_NNIL2 NLAST2 NHD2 NTL2 y)"
    using \<open>A x y\<close> using rel by (coinduction arbitrary: x y) (auto 4 4 elim: rel_funE)
qed

lemma corec_nellist_transfer [transfer_rule]:
  "((A ===> (=)) ===> (A ===> C) ===> (A ===> C) ===> (A ===> (=)) ===> (A ===> nellist_all2 C)
    ===> (A ===> A) ===> A ===> nellist_all2 C) corec_nellist corec_nellist"
by (simp add: nellist.corec_transfer)

lemma ntl_transfer2 [transfer_rule]:
  "(nellist_all2 A  ===> nellist_all2 A) ntl ntl"
  unfolding ntl_def[abs_def] by transfer_prover
declare ntl_transfer [transfer_rule]

lemma nset_transfer2 [transfer_rule]:
  "(nellist_all2 A  ===> rel_set A) nset nset"
by (simp add: nellist.set_transfer)

lemma nmap_transfer4 [transfer_rule]:
  "((A ===> B) ===> nellist_all2 A ===> nellist_all2 B) nmap nmap"
by (simp add: nellist.map_transfer)
declare nmap_transfer [transfer_rule]

lemma is_NNil_transfer2 [transfer_rule]:
  "(nellist_all2 A  ===> (=)) is_NNil is_NNil"
by(auto dest: nellist_all2_is_NNilD)
declare is_NNil_transfer [transfer_rule]

lemma snocn_transfer2 [transfer_rule]:
 " (nellist_all2 A  ===> A  ===> nellist_all2 A ) snocn snocn"
unfolding rel_fun_def
by (metis nappend_snocn nellist_all2_NNil nellist_all2_nappendI)
declare snocn_transfer[transfer_rule]

lemma nappend_transfer [transfer_rule]:
  "(nellist_all2 A ===> ( nellist_all2 A) ===> nellist_all2 A) nappend nappend"
by(auto intro: nellist_all2_nappendI elim: rel_funE)
declare nappend.transfer [transfer_rule]

lemma lappendn_transfer [transfer_rule]:
  "(llist_all2 A ===> nellist_all2 A ===> nellist_all2 A) lappendn lappendn"
unfolding rel_fun_def
by transfer(auto intro: llist_all2_lappendI)
declare lappendn.transfer [transfer_rule]

lemma nellist_of_llist_a_transfer2 [transfer_rule]:
  "(A ===> llist_all2 A ===> nellist_all2 A) nellist_of_llist_a nellist_of_llist_a"
by (simp add: rel_funI)
declare nellist_of_llist_a_transfer [transfer_rule]

lemma nlength_transfer [transfer_rule]:
  "(nellist_all2 A ===> (=)) nlength nlength"
by(auto dest: nellist_all2_nlengthD)
declare nlength.transfer [transfer_rule]

lemma ndropn_transfer [transfer_rule]:
  "((=) ===> nellist_all2 A ===> nellist_all2 A) ndropn ndropn"
unfolding rel_fun_def
by transfer (auto intro: llist_all2_ldropnI simp add: llist_all2_ldropnI llist_all2_llengthD )
declare ndropn.transfer [transfer_rule]

lemma ntake_transfer [transfer_rule]:
  "((=) ===> nellist_all2 A ===> nellist_all2 A) ntake ntake"
unfolding rel_fun_def
by transfer (auto simp add: llist_all2_ltakeI)
declare ntake.transfer [transfer_rule]

lemma ntaken_transfer [transfer_rule]:
  "((=) ===> nellist_all2 A ===> nellist_all2 A) ntaken ntaken"
unfolding rel_fun_def
by transfer (auto simp add: llist_all2_ltakeI)
declare ntaken.transfer [transfer_rule]

lemma nzip_transfer [transfer_rule]:
 " (nellist_all2 A ===> nellist_all2 B ===> nellist_all2 (rel_prod A B)) nzip nzip"
by (auto intro: nellist_all2_nzipI) 


lemma niterates_transfer [transfer_rule]:
 " ((A ===> A) ===>  A ===> nellist_all2 A) niterates niterates"
unfolding rel_fun_def
apply transfer
using iterates_transfer unfolding rel_fun_def
by blast

lemma nfilter_transfer [transfer_rule]:
  "( (A ===> (=)) ===> nellist_all2 A ===> nellist_all2 A) nfilter nfilter"
unfolding rel_fun_def
by transfer
   (auto intro: llist_all2_lfilterI  dest: llist_all2_lfiniteD llist_all2_lsetD1,
    meson llist_all2_lsetD2)
declare nfilter.transfer [transfer_rule]

lemma nconcat_transfer [transfer_rule]:
  "( nellist_all2 (nellist_all2 A)  ===> nellist_all2 A ) nconcat nconcat"
unfolding rel_fun_def
using nellist_all2_nconcatI 
by auto
declare nconcat.transfer [transfer_rule]

lemma nellist_all2_rsp:
  assumes R1: "\<forall>x y. R1 x y \<longrightarrow> (\<forall>a b. R1 a b \<longrightarrow> S x a = T y b)"
  and R2: "\<forall>x y. R2 x y \<longrightarrow> (\<forall>a b. R2 a b \<longrightarrow> S' x a = T' y b)"
  and xsys: "nellist_all2 R1 xs ys"
  and xs'ys': "nellist_all2 R1 xs' ys'"
  shows "nellist_all2 S xs xs' = nellist_all2 T ys ys'"
proof
  assume "nellist_all2 S xs xs'"
  with xsys xs'ys' show "nellist_all2 T ys ys'"
  proof(coinduction arbitrary: ys ys' xs xs')
    case (ilist_all2 ys ys' xs xs')
    thus ?case
      by cases (auto 4 4 simp add: nellist_all2_NCons1 nellist_all2_NCons2 nellist_all2_NNil1
                nellist_all2_NNil2 dest: R1[rule_format] R2[rule_format])
  qed
next
  assume "nellist_all2 T  ys ys'"
  with xsys xs'ys' show "nellist_all2 S xs xs'"
  proof(coinduction arbitrary: xs xs' ys ys')
    case (ilist_all2 xs xs' ys ys')
    thus ?case
      by cases (auto 4 4 simp add: nellist_all2_NCons1 nellist_all2_NCons2 nellist_all2_NNil1 
                nellist_all2_NNil2 dest: R1[rule_format] R2[rule_format])
  qed
qed

lemma nellist_all2_transfer2 [transfer_rule]:
  "((R1 ===> R1 ===> (=) ) ===>
    nellist_all2 R1 ===> nellist_all2 R1 ===> (=)) nellist_all2 nellist_all2"
by (simp add: nellist_all2_rsp rel_fun_def)
declare nellist_all2_transfer [transfer_rule]

end

text \<open>
  Delete lifting rules for @{typ "'a nellist"} 
  because the parametricity rules take precedence over
  most of the transfer rules. They can be restored by 
  including the bundle \<open>nellist.lifting\<close>.
\<close>

lifting_update nellist.lifting
lifting_forget nellist.lifting

end
