(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)
chapter \<open>Model: Intervals\<close>
text \<open>
Intervals are defined in terms of non empty coinductive lists. 
\<close>


section \<open>Extra operations on coinductive lists\<close>

text \<open>
the operations  kfilter, lleast, lbutlast, lidx, ridx, lfirst, lfuse, lsub, lsubc,
	llastlfirst, lltl, llbutlast, is\_lfirst, lfusecat  and lrev on coinductive lists are defined together with 
a library of lemmas. 
\<close>

theory LList_Extras 

 imports 
  "Coinductive.Coinductive_List"
begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="cvc4 z3 e spass vampire " ]
*)

definition kfilter :: " ('a \<Rightarrow> bool) \<Rightarrow> nat \<Rightarrow> 'a llist  \<Rightarrow> nat llist"
 where " kfilter P n xs  = lmap snd (lfilter (P\<circ>fst) (lzip xs (iterates Suc n)))"

definition lleast :: "('a \<Rightarrow> bool) \<Rightarrow> 'a llist \<Rightarrow> nat " 
 where "lleast P xs = (LEAST na. na < llength xs \<and> P (lnth xs na)) "

primcorec lbutlast :: "'a llist \<Rightarrow> 'a llist"
 where "lbutlast xs =
        (case xs of LNil \<Rightarrow> LNil |
           (LCons x xs') \<Rightarrow> 
            (case xs' of LNil \<Rightarrow> LNil |
               (LCons x1 xs1) \<Rightarrow> (LCons x (lbutlast xs'))))"

simps_of_case lbutlast_simps [simp, code, nitpick_simp]: lbutlast.code


definition ridx :: "('a \<Rightarrow> 'a \<Rightarrow> bool) \<Rightarrow> 'a llist \<Rightarrow> bool"
 where " ridx R xs = (\<forall> i. (Suc i)<llength xs \<longrightarrow> R (lnth xs i) (lnth xs (Suc i))) " 


definition lidx :: "  nat llist \<Rightarrow> bool"
 where " lidx  xs \<longleftrightarrow>  (\<forall>n. (Suc n)<llength xs \<longrightarrow> lnth xs n < lnth xs (Suc n)) "

definition lfirst :: "'a llist \<Rightarrow> 'a"
 where "lfirst xs = lhd xs " 

definition lfuse :: "'a llist \<Rightarrow> 'a llist \<Rightarrow> 'a llist "
 where "lfuse xs ys = 
        (if \<not> lnull xs \<and> \<not> lnull ys then lappend xs (ltl ys) else lappend xs ys)" 


definition lsub :: "nat \<Rightarrow> nat \<Rightarrow> 'a llist \<Rightarrow> 'a llist "
where "lsub k n xs = (ltake (eSuc (n - k)) (ldrop k xs)) "

definition lsubc :: "nat \<Rightarrow> nat \<Rightarrow> 'a llist \<Rightarrow> 'a llist "
where "lsubc k n xs = (ltake (eSuc (n - k)) (ldrop ((min k (epred (llength xs)))) xs)) " 

definition llastlfirst :: "'a llist llist \<Rightarrow> bool"
 where "llastlfirst xss = (\<forall>i. (Suc i)<llength xss \<longrightarrow> llast (lnth xss i) = lfirst(lnth xss (Suc i))) " 

definition  lltl :: " 'a llist llist \<Rightarrow> 'a llist llist "
 where " lltl xss = lmap ltl xss" 

definition llbutlast :: " 'a llist llist \<Rightarrow> 'a llist llist "
 where " llbutlast xss = lmap lbutlast xss " 

abbreviation "is_lfirst \<equiv>  (\<lambda>xs. \<exists>b. xs = (LCons b LNil))"

definition lrev :: "'a llist \<Rightarrow> 'a llist"
where "lrev xs = (if lfinite xs then llist_of (rev (list_of xs)) else xs)" 



subsection \<open>Auxiliary lemmas\<close>

lemma enat_min:
  assumes "m \<ge> enat n'"
    and "enat n < m - enat n'"
  shows "enat n + enat n' < m" 
using assms 
by (metis add.commute enat.simps(3) enat_add_mono enat_add_sub_same le_iff_add)

lemma enat_min_eq:
  assumes "m \<ge> enat n'"
    and "enat n \<le> m - enat n'"
  shows "enat n + enat n' \<le> m" 
using assms
by (metis add.commute enat.simps(3) enat_add_sub_same enat_min le_iff_add le_less)
  
lemma llist_eq_lnth_eq:
 " (xs = ys) \<longleftrightarrow> (llength xs = llength ys \<and> (\<forall> i < llength xs. lnth xs i = lnth ys i))"
proof auto
 show "llength xs = llength ys \<Longrightarrow> \<forall>i. enat i < llength ys \<longrightarrow> lnth xs i = lnth ys i \<Longrightarrow> xs = ys" 
 proof (coinduction arbitrary: xs ys)
 case (Eq_llist xsa ysa)
 then show ?case 
    proof  -
     assume a: "llength xsa = llength ysa" 
     assume b: "\<forall>i. enat i < llength ysa \<longrightarrow> lnth xsa i = lnth ysa i" 
     have 1: " lnull xsa = lnull ysa" 
       using a by auto
     have 2: "\<not> lnull xsa \<longrightarrow>
              \<not> lnull ysa \<longrightarrow>
              lhd xsa = lhd ysa" 
       using b lhd_conv_lnth zero_enat_def by force
     have 3: "\<not> lnull xsa \<longrightarrow>
              \<not> lnull ysa \<longrightarrow>
                (\<exists>xs ys. ltl xsa = xs \<and> ltl ysa = ys \<and> llength xs = llength ys \<and> 
                         (\<forall>i. enat i < llength ys \<longrightarrow> lnth xs i = lnth ys i))"
       by (metis Extended_Nat.eSuc_mono a b eSuc_enat eSuc_epred llength_eq_0 llength_ltl lnth_ltl)
    show ?thesis using 1 2 3 by blast
   qed
 qed
qed

lemma exist_lset_lnth:
 " (\<exists> x \<in> lset xs. P x) \<longleftrightarrow> (\<exists> i<llength xs. P (lnth xs i))"
by (metis in_lset_conv_lnth)

lemma exist_llength_gr_zero:
 assumes "(\<exists> x \<in> lset xs. P x)"
 shows   " 0< llength xs"
 by (metis assms gr_implies_not_zero gr_zeroI in_lset_conv_lnth)


subsection \<open>lbutlast\<close>


lemma lbutlast_snoc [simp]:
 " lbutlast (lappend xs (LCons x LNil)) = xs"
proof (cases "lfinite xs")
case True
then show ?thesis 
  proof (induct rule: lfinite_induct)
  case (LNil xs)
  then show ?case using llist.collapse(1) by fastforce
  next
  case (LCons xs)
  then show ?case
     proof (cases "xs=LNil")
     case True
     then show ?thesis by simp
     next
     case False
     then show ?thesis 
      proof -
       have 1: "\<exists> y ys. xs =(LCons y ys)"
         using False llist.exhaust_sel by blast 
       obtain y ys where 2: " xs =(LCons y ys)" 
          using 1 by blast
       have 3: " lbutlast (lappend xs (LCons x LNil)) = lbutlast (lappend (LCons y ys) (LCons x LNil))" 
          by (simp add: "2")
       have 4: "lbutlast (lappend (LCons y ys) (LCons x LNil)) = lbutlast (LCons y (lappend ys (LCons x LNil))) " 
         by simp
       have 5: "lnull ys \<Longrightarrow> ?thesis " 
         by (simp add: "2" lnull_def)
       have 6: "\<not>lnull ys \<Longrightarrow> ?thesis"
          by (metis "2" LCons.hyps(3) eq_LConsD lappend_code(2) lbutlast_simps(3) lhd_LCons_ltl)
       show ?thesis 
       using "5" "6" by blast
      qed
     qed
  qed
next
case False
then show ?thesis 
 proof (coinduction arbitrary: xs)
 case (Eq_llist xsa)
 then show ?case 
     proof -
      have 1: "lnull (lbutlast (lappend xsa (LCons x LNil))) = lnull xsa"
        by (metis Eq_llist lappend_inf lbutlast.disc(2) lfinite.simps lhd_LCons_ltl lnull_imp_lfinite)
      have 2: "\<not> lnull (lbutlast (lappend xsa (LCons x LNil))) \<longrightarrow>
               \<not> lnull xsa \<longrightarrow>
               lhd (lbutlast (lappend xsa (LCons x LNil))) = lhd xsa" 
           by (metis Eq_llist eq_LConsD lappend_inf lbutlast.disc(1) lbutlast_simps(3) not_lnull_conv)
      have 3: "\<not> lnull (lbutlast (lappend xsa (LCons x LNil))) \<longrightarrow>
               \<not> lnull xsa \<longrightarrow> 
              (\<exists>xs. ltl (lbutlast (lappend xsa (LCons x LNil))) = 
                           lbutlast (lappend xs (LCons x LNil)) \<and> ltl xsa = xs \<and> \<not> lfinite xs)" 
        by (metis Eq_llist eq_LConsD lappend_inf lbutlast_simps(3) lfinite_ltl lhd_LCons_ltl lnull_imp_lfinite)
      show ?thesis using 1 2 3 by auto
   qed
 qed
qed



lemma lbutlast_ltl:
 " lbutlast (ltl xs) = ltl (lbutlast xs)"
proof (cases "lfinite xs")
case True
then show ?thesis 
  proof  (induct rule: lfinite_induct)
  case (LNil xs)
  then show ?case by (simp add: lbutlast.ctr(1))
  next
  case (LCons xs)
  then show ?case 
     by (simp add: lbutlast.code llist.case_eq_if)
  qed
next
case False
then show ?thesis 
   proof  (coinduction arbitrary: xs)
   case (Eq_llist xsa)
   then show ?case 
     proof -
      have 1: "lnull (lbutlast (ltl xsa)) = lnull (ltl (lbutlast xsa))" 
        by (metis Eq_llist lappend_inf lbutlast_snoc lfinite_ltl)
      have 2: "\<not> lnull (lbutlast (ltl xsa)) \<longrightarrow>
               \<not> lnull (ltl (lbutlast xsa)) \<longrightarrow>
               lhd (lbutlast (ltl xsa)) = lhd (ltl (lbutlast xsa))" 
        by (metis Eq_llist lappend_inf lbutlast_snoc lfinite_ltl)
      have 3: "\<not> lnull (lbutlast (ltl xsa)) \<longrightarrow>
               \<not> lnull (ltl (lbutlast xsa)) \<longrightarrow>
               (\<exists>xs. ltl (lbutlast (ltl xsa)) = lbutlast (ltl xs) \<and> 
                     ltl (ltl (lbutlast xsa)) = ltl (lbutlast xs) \<and> \<not> lfinite xs)" 
         by (metis Eq_llist lappend_inf lbutlast_snoc lfinite_ltl)
      show ?thesis using 1 2 3 by auto
    qed
   qed
qed

lemma lbutlast_not_lfinite:
 assumes "\<not> lfinite xs"
 shows   "lbutlast xs = xs"
 using assms 
by (coinduction arbitrary: xs)
   (metis lappend_inf lbutlast_snoc lfinite_ltl)

lemma lbutlast_lfinite:
 "lfinite (lbutlast xs) \<longleftrightarrow> lfinite xs"
proof 
 show "lfinite (lbutlast xs) \<Longrightarrow> lfinite xs"
  proof (induct zs\<equiv>"lbutlast xs" rule: lfinite_induct)
  case LNil
  then show ?case using lbutlast_not_lfinite by fastforce
  next
  case LCons
  then show ?case using lbutlast_not_lfinite by fastforce
  qed
 show "lfinite xs \<Longrightarrow> lfinite (lbutlast xs) "
  proof (induct rule: lfinite_induct)
  case (LNil xs)
  then show ?case by simp
  next
  case (LCons xs)
  then show ?case by (simp add: lbutlast_ltl)
  qed 
qed

lemma llength_lbutlast [simp]:
 " llength (lbutlast xs) = epred (llength xs)"
by (coinduction arbitrary: xs rule: enat_coinduct)
   (simp,
   metis epred_enat_unfold  lbutlast_ltl llength_def llength_eq_0 llength_ltl )

lemma lbutlast_lappend:
 " lbutlast (lappend xs ys) = (if ys = LNil then lbutlast xs else lappend xs (lbutlast ys))"
proof (cases "lfinite xs")
case True
then show ?thesis 
   proof (induct arbitrary: ys rule: lfinite_induct)
   case (LNil xs)
   then show ?case by (simp add: lnull_def)
   next
   case (LCons xs)
   then show ?case 
       proof (cases "lnull ys" )
       case True
       then show ?thesis by (metis lappend_LNil2 lnull_def)
       next
       case False
       then show ?thesis 
         proof (cases "lnull xs")
         case True
         then show ?thesis 
         using LCons.hyps(2) by auto
         next
         case False
         then show ?thesis using LCons by (auto simp add: llist.case_eq_if not_lnull_conv)
             (metis lappend.ctr(2) lbutlast_simps(3) llist.collapse(1) ltl_simps(2))               
         qed
       qed
   qed
next
case False
then show ?thesis by (metis lappend_inf lbutlast_snoc)
qed

lemma lappend_lbutlast_llast_id_lfinite:
 assumes "lfinite xs"
         " \<not> lnull xs"
 shows   "(lappend (lbutlast xs) (LCons (llast xs) LNil)) = xs"
using assms 
proof (induct rule: lfinite_induct)
case (LNil xs)
then show ?case by simp
next
case (LCons xs)
then show ?case 
   proof (cases xs)
   case lfinite_LNil
   then show ?thesis using LCons by simp
   next
   case (lfinite_LConsI xs x)
   then show ?thesis using LCons
   by (auto simp add: llast_LCons)
      (metis lappend_code(1) lbutlast.ctr(1) llist.collapse(1) ltl_simps(2),  
       metis lappend_code(2) lbutlast_simps(3) lhd_LCons_ltl)
qed
qed
    
lemma lappend_lbutlast_llast_id_not_lfinite:
 assumes "\<not>lfinite xs"
         " \<not> lnull xs"
 shows   "(lappend (lbutlast xs) (LCons (llast xs) LNil)) = xs"
using assms 
proof (coinduction arbitrary: xs)
case (Eq_llist xsa)
then show ?case 
    by (auto simp add: lbutlast_ltl llast_linfinite)
       (metis lfinite_LNil lfinite_ltl llist.collapse(1), 
        metis lbutlast.simps(3) lbutlast_not_lfinite)
qed
     
lemma lappend_lbutlast_llast_id [simp]:
shows " \<not> lnull xs \<Longrightarrow> (lappend (lbutlast xs) (LCons (llast xs) LNil)) = xs"
using lappend_lbutlast_llast_id_lfinite lappend_lbutlast_llast_id_not_lfinite by blast


lemma lbutlast_eq_LNil_conv: 
 " lbutlast xs = LNil \<longleftrightarrow> xs =LNil \<or> (\<exists>x. xs = (LCons x LNil)) " 
by (metis lbutlast.disc_iff(1) lbutlast_simps(2) lhd_LCons_ltl llist.collapse(1) llist.disc(1))

lemma lbutlast_eq_LCons_conv: 
 " lbutlast xs = (LCons x ys) \<longleftrightarrow>  xs = (LCons x (lappend ys (LCons (llast xs) LNil))) " 
by (metis eq_LConsD  lappend_code(2) lappend_lbutlast_llast_id lbutlast.ctr(1) lbutlast_snoc)


lemma lbutlast_conv_ltake:
 " lbutlast xs = ltake (epred (llength xs)) xs"
proof (cases "lfinite xs")
case True
then show ?thesis 
   proof (induct rule: lfinite_induct)
   case (LNil xs)
   then show ?case by simp
   next
   case (LCons xs)
   then show ?case 
     by (metis enat_le_plus_same(2) gen_llength_def lappend_lbutlast_llast_id_lfinite llength_code 
          llength_lbutlast ltake_all ltake_lappend1) 
   qed
next
case False
then show ?thesis 
  proof (coinduction arbitrary: xs)
  case (Eq_llist xsa)
  then show ?case 
   by (auto simp add: not_lfinite_llength lbutlast_not_lfinite)
      (metis epred_Infty epred_llength lbutlast_not_lfinite llength_eq_infty_conv_lfinite ltake_epred_ltl)
  qed
qed

lemma lmap_lbutlast:
 " lmap f (lbutlast xs) = lbutlast (lmap f xs)"
by (simp add: lbutlast_conv_ltake)
 
lemma snocs_eq_iff_lbutlast:
 " lappend xs (LCons x LNil) = ys \<longleftrightarrow> 
   (( lfinite ys \<and> \<not> lnull ys \<and> lbutlast ys = xs \<and> llast ys = x) 
   \<or> (\<not> lfinite ys \<and> lbutlast xs = ys))"
by (metis lappend.disc(2) lappend_inf lappend_lbutlast_llast_id lbutlast.ctr(1) lbutlast_snoc 
     lfinite_LNil llast_lappend llast_singleton llist.discI(2))

lemma in_lset_lbutlastD:
 " x \<in> lset(lbutlast xs) \<Longrightarrow> x \<in> lset xs"
by (metis in_lset_lappend_iff lappend_ltake_ldrop lbutlast_conv_ltake)

lemma in_lset_lbutlast_lappendI:
 " x \<in> lset (lbutlast xs) \<or> (lfinite xs \<and> x \<in> lset(lbutlast ys)) \<Longrightarrow>
   x \<in> lset (lbutlast (lappend xs ys))"
by (metis empty_iff in_lset_lappend_iff in_lset_lbutlastD lbutlast_lappend lset_LNil)

lemma lnth_lbutlast:
 assumes "n< llength(lbutlast xs)"
 shows   " lnth (lbutlast xs) n = lnth xs n"
proof (cases xs)
case LNil
then show ?thesis by simp
next
case (LCons x21 x22)
then show ?thesis by (metis assms lbutlast_conv_ltake llength_lbutlast lnth_ltake)
qed


subsection \<open>lfuse\<close>


lemma lfuse_conv_lnull:
 " lnull (lfuse xs ys)  \<longleftrightarrow> lnull xs  \<and> lnull ys  " 
by (simp add: lfuse_def)


lemma lfuse_LNil_1 [simp]:
 "lfuse LNil ys =   ys " 
by (simp add: lfuse_def)

lemma lfuse_LNil_2 [simp]:
 "lfuse xs LNil = xs " 
by (metis lappend_lnull2 lfuse_def llist.discI(1))

lemma lfuse_One_a [simp]: 
 assumes " \<not> lnull xs "
 shows   " lfuse (LCons (lhd xs) LNil) xs = xs "
using assms by (simp add: lfuse_def)

lemma lfuse_One_b [simp]:
 assumes "\<not> lnull xs" 
 shows   "lfuse xs (LCons y LNil) = xs " 
using assms 
by (simp add: lfuse_def)

lemma lfuse_One_a_var: 
 shows "lfuse (LCons x LNil) ys = (if \<not> lnull ys then (LCons x (ltl ys)) else (LCons x LNil))" 
unfolding lfuse_def by simp

lemma lfuse_One_b_var: 
 shows " lfuse xs (LCons y LNil) = (if \<not>lnull xs then xs else (LCons y LNil))" 
unfolding lfuse_def by (simp add: lappend_lnull1)

lemma lfuse_LCons_a [simp]:
 " lfuse (LCons x xs) ys = 
     (if lnull xs then (LCons x (ltl ys)) else (LCons x (lfuse xs ys)))"
by (metis lappend_code(2) lappend_lnull1 lfuse_def llist.collapse(1) llist.disc(2) ltl_simps(1))

lemma lfuse_LCons_b [simp]:
 "lfuse xs (LCons y ys) =
   (if \<not> lnull xs then lappend xs ys else (LCons y ys)) " 
unfolding lfuse_def by (simp add: lappend_lnull1)

lemma lfuse_simps [simp]:
  shows lhd_lfuse: "lhd (lfuse xs ys) = (if lnull xs then lhd ys else lhd xs)"
  and ltl_lfuse: "ltl (lfuse xs ys) =
                    (if lnull xs 
                     then ltl ys 
                     else (if lnull (ltl ys) 
                           then ltl xs
                           else lappend (ltl xs) (ltl ys)))"
by (auto simp add: lfuse_def lappend_lnull2)

lemma lfuse_lbutlast: 
 assumes " llast xs = lfirst ys" 
  shows  " lfuse xs ys = (if lnull ys then xs else lappend (lbutlast xs) ys)" 
using assms
by (metis lappend_lbutlast_llast_id lappend_snocL1_conv_LCons2 lbutlast.ctr(1) lfirst_def 
    lfuse_LNil_2 lfuse_def lhd_LCons_ltl llist.collapse(1))

lemma lfuse_llength:
 shows "llength (lfuse xs ys) =  (llength xs) + (if lnull xs then llength ys else epred(llength ys)) " 
unfolding lfuse_def by ( simp add: llist.case_eq_if epred_llength)

lemma not_lnull_llength:
 " \<not> lnull xs \<longleftrightarrow> 1 \<le> llength xs "
by (metis gr_zeroI ileI1 llength_eq_0 not_one_le_zero one_eSuc)

lemma lfuse_llength_atmost_one: 
 shows "llength (lfuse xs ys) \<le> 1 \<longleftrightarrow> llength xs \<le>1 \<and> llength ys \<le> 1" 
proof (cases "lnull xs")
case True
then show ?thesis 
by (metis lfuse_LNil_1 llength_LNil lnull_def zero_le)
next
case False
then show ?thesis unfolding lfuse_llength 
by  auto
   (meson dual_order.trans enat_le_plus_same(1),
    metis add.commute add_increasing2 co.enat.exhaust_sel not_lnull_llength order_antisym_conv 
    order_refl plus_1_eSuc(2) zero_le, 
    metis add_decreasing2 epred_1 epred_le_epredI)
qed

lemma lfuse_llength_less_a: 
 assumes "1 < llength xs "
         "1 < llength ys"
         "llast xs = lfirst ys "
         "lfinite xs" 
 shows   " llength xs < llength (lfuse xs ys)" 
unfolding lfuse_def
using assms 
by (auto simp add: lfinite_llength_enat) 
   (metis co.enat.exhaust_sel epred_llength linorder_neq_iff llength_eq_0 one_eSuc)

lemma lfuse_llength_less_b: 
 assumes "1 < llength xs "
         "1 < llength ys"
         "llast xs = lfirst ys "
         "lfinite ys" 
 shows   " llength ys < llength (lfuse xs ys)" 
proof -
 have 1: "lfuse xs ys = lappend (lbutlast xs) ys"
  by (metis assms(2) assms(3) lfuse_lbutlast llength_lnull not_less_zero)
 have 2: " llength (lappend (lbutlast xs) ys) = epred(llength xs) + llength ys"
   by simp
 have 3: "0 < epred(llength xs)" 
   by (metis assms(1) co.enat.exhaust_sel gr_zeroI less_numeral_extra(4) not_one_less_zero one_eSuc)
 have 4: "llength ys < epred(llength xs) + llength ys"
   using "3" assms(4) lfinite_llength_enat by auto
 show ?thesis 
 using "1" "2" "4" by presburger 
qed           

lemma lfuse_llength_le_a: 
 "llength xs \<le> llength (lfuse xs ys)" 
by (simp add: lfuse_llength)

lemma lfuse_llength_le_b: 
 assumes "llast xs = lfirst ys"
 shows   "llength ys \<le> llength (lfuse xs ys)" 
by (simp add: assms lfuse_lbutlast)

lemma lfuse_lnth_a:
 assumes "(enat i) < llength xs "
 shows   "lnth (lfuse xs ys) i = lnth xs i " 
using assms
by (simp add: lfuse_def lnth_lappend1)

lemma lfuse_lnth_b:
 assumes " llength xs \<le> (enat i) "
         " (enat i) < llength (lfuse xs ys) "        
 shows   "lnth (lfuse xs ys) i = (lnth ys (i - (the_enat(epred(llength xs)))))"
proof (cases "\<not> lnull xs \<and> \<not>lnull ys")
case True
then show ?thesis 
   proof -
    have 1: "lfinite xs"
       using assms(1) lfinite_ldropn lnull_imp_lfinite lnull_ldropn by blast
    obtain n where 15: "(enat n) = llength xs" using 1 by (metis assms(1) enat_ile)
    have 16: "(the_enat(epred(llength xs))) = n-1" 
        by (metis "15" epred_enat the_enat.simps)
    have 17: "0<n" 
       by (metis "15" True gr0I llength_eq_0 zero_enat_def)
    have 2: "lfuse xs ys = lappend xs (ltl ys)"
       unfolding lfuse_def using assms True by presburger
    have 3: "n-1 \<le> i"
         by (metis "15" "17" Suc_pred' assms(1) enat_ord_simps(2) leD le_imp_less_Suc wlog_linorder_le)
    have 4: "(Suc (i - n)) =  (i - (n-1)) " 
         by (metis "15" "17" Suc_diff_eq_diff_pred Suc_diff_le assms(1) enat_ord_simps(1))
    have 5: "lnth (ltl ys) (i - the_enat (llength xs)) = (lnth ys (i - (the_enat(epred(llength xs)))))" 
       using True 3 lnth_ltl[of ys "(i - the_enat (llength xs))"] by (metis "15" "16" "4" the_enat.simps)
   show ?thesis   using lnth_lappend[of xs "(ltl ys)" ] by (simp add: "2" "5" assms(1) leD)
  qed  
next
case False
then show ?thesis  using assms unfolding lfuse_def
using lappend_lnull1 by fastforce
qed

lemma lfuse_lnth_c:
 assumes " epred(llength xs) \<le> (enat i) "
         " (enat i) < llength (lfuse xs ys) "  
         "llast xs = lfirst ys "       
 shows   "lnth (lfuse xs ys) i = 
          (if lnull ys then lnth xs (the_enat (epred(llength xs))) 
           else   lnth ys (i - (the_enat (epred(llength xs))))) "
proof (cases "\<not> lnull xs \<and> \<not>lnull ys")
case True
then show ?thesis 
  using assms 
  by (simp add: leD lfuse_lbutlast lnth_lappend)
next
case False
then show ?thesis 
   proof (cases "lnull xs")
   case True
   then show ?thesis 
      using assms 
     by (metis epred_0 gr_implies_not_zero lfuse_conv_lnull lfuse_lnth_b llength_lnull)
   next
   case False
   then show ?thesis 
      using assms 
      by (metis co.enat.exhaust_sel iless_Suc_eq leD lfuse_lbutlast llength_eq_0 
          llength_lbutlast lnth_lappend nle_le the_enat.simps) 
   qed
qed

 
lemma lfirst_lfuse_1:
 shows   " lfirst (lfuse xs ys) = (if \<not> lnull xs then lfirst xs else lfirst ys)" 
by (simp add: lfirst_def)

lemma llast_lfuse:
 assumes " \<not> lnull xs "
         " \<not> lnull ys "
         "lfinite xs "
         "lfinite ys"
         "llast xs = lfirst ys " 
 shows   "llast(lfuse xs ys) = llast ys " 
using assms 
by (metis lfirst_def lfuse_def lhd_LCons_ltl llast_LCons llast_lappend)

lemma lfuse_assoc:
assumes "\<not> lnull xs "
        "\<not> lnull ys "
        "\<not> lnull zs "
shows   " (lfuse xs (lfuse ys zs)) = (lfuse (lfuse xs ys) zs) " 
using assms 
by (metis lappend_assoc lappend_ltl lfuse_def lfuse_conv_lnull)

lemma lfirst_llast:
 assumes " i< llength xs" 
 shows   " llast (ltake (Suc i) xs) = lfirst (ldropn i xs) "
using assms 
by (simp add: lfirst_def lhd_ldropn ltake_Suc_conv_snoc_lnth)

lemma ltake_lfuse:
 shows   " ltake (llength xs) (lfuse xs ys) = xs "
by (metis dual_order.refl lappend_lnull2 lfuse_def ltake_all ltake_lappend1)

lemma llast_lfirst_LNil: 
 "llast LNil = lfirst LNil" 
by (simp add: lfirst_def lhd_def llast_LNil)

lemma ldrop_lappend_either_LNil: 
 assumes "lnull xs \<or> lnull ys"
 shows   " ldrop (llength xs) (lappend xs ys) = 
            (if lfinite xs then ys else LNil) "
proof -
 have 1: "lnull xs \<Longrightarrow> ?thesis "
   by (simp add: lappend_lnull1)
 have 2: "lnull ys \<Longrightarrow> ?thesis"
    by (metis dual_order.refl lappend_LNil2 ldrop_eq_LNil lnull_def) 
 show ?thesis
 using "1" "2" assms by blast    
qed

lemma ldrop_lfuse:
 assumes " llast xs = lfirst ys "
 shows   " ldrop (if \<not> lnull xs \<and> \<not> lnull ys then epred(llength xs) else (llength xs)) (lfuse xs ys) = 
          (if lfinite xs then ys else LNil) "  
proof -
 have 3: "lfuse xs ys = (if \<not> lnull xs \<and> \<not> lnull ys 
                         then lappend (lbutlast xs) ys  
                         else lappend xs ys) " 
    by (meson assms lfuse_def lfuse_lbutlast)
 have 4: "ldrop ((if \<not> lnull xs \<and> \<not> lnull ys then epred(llength xs) else (llength xs)))
                (if \<not> lnull xs \<and> \<not> lnull ys 
                         then lappend (lbutlast xs) ys  
                         else lappend xs ys) = (if lfinite xs then ys else LNil) "   
  proof (cases "lfinite xs")
  case True
  then show ?thesis 
      by (simp add: ldrop_lappend lfinite_llength_enat)
  next
  case False
  then show ?thesis  
     by simp
       (metis epred_Infty llength_eq_infty_conv_lfinite lnull_imp_lfinite)
  qed
 show ?thesis 
 by (simp add: "3" "4")
qed

lemma ldrop_lfuse_a:
 assumes "\<not> lnull xs"
         "\<not> lnull ys" 
         " llast xs = lfirst ys "
 shows   " ldrop (epred(llength xs)) (lfuse xs ys) = 
          (if lfinite xs then ys else LNil) " 
using assms 
using ldrop_lfuse by force


lemma lfuse_ltake_ldrop:
 assumes " i < llength xs " 
 shows   "lfuse (ltake (eSuc i) xs) (ldrop i xs) = xs" 
using assms 
by (metis lappend_ltake_ldrop ldrop_eSuc_conv_ltl ldrop_lnull leD lfuse_def lnull_ldrop 
    ltake.disc(2) zero_ne_eSuc)


lemma lset_lfuse:
 shows "lset (lfuse xs ys) = 
        (if lfinite xs then
           (if \<not> lnull xs \<and> \<not> lnull ys then lset xs \<union> lset (ltl ys) else lset xs \<union> lset ys)
         else lset xs) " 
by (simp add: lappend_inf lfuse_def)
 



lemma mono_llist_lfuse2 [partial_function_mono]:
  "mono_llist A \<Longrightarrow> mono_llist (\<lambda>f. lfuse xs (A f))"
by (auto intro!: monotoneI lprefix_lappend_sameI simp add: lfuse_def lnull_lprefix 
      llist.case_eq_if fun_ord_def lprefix_ltlI monotone_def dest: monotoneD)
   (metis llist.collapse(1) lprefix_ltlI ltl_simps(1))


lemma mono2mono_lfuse2 [THEN llist.mono2mono, cont_intro, simp]:
  shows monotone_lfuse2: "monotone (lprefix) (lprefix) (lfuse xs)"
by (rule monotoneI)
   (metis Coinductive_List.finite_lprefix_def Coinductive_List.finite_lprefix_nitpick_simps(1) 
    lfuse_def lprefix_LNil lprefix_lappend_sameI ltl_simps(1) monotoneD monotone_ltl)

lemma silys: 
 assumes "f = g" 
 shows   "mcont lSup (lprefix) lSup (lprefix) f =
          mcont lSup (lprefix) lSup (lprefix) g"
using assms by auto

 
lemma lfuse_LNil_eq_id: 
 "lfuse LNil = id " 
by (simp add: fun_eq_iff llist.case_eq_if)    


lemma mcont2mcont_lfuse2 [THEN llist.mcont2mcont, cont_intro, simp]:
  shows mcont_lfuse2: "mcont lSup (lprefix) lSup (lprefix) (lfuse xs)"
proof(cases "lfinite xs")
  case True
  thus ?thesis
 proof induct
  case lfinite_LNil
  then show ?case  using lfuse_LNil_eq_id by simp
  next
  case (lfinite_LConsI xs x)
  then show ?case by (simp add:monotone_lfuse2)
 qed
next
  case False
  hence "lfuse xs = (\<lambda>_. xs)" 
     by (simp add: fun_eq_iff lappend_inf lfuse_def) 
  thus ?thesis by(simp add: ccpo.cont_const[OF llist_ccpo])
qed


lemma lfuse_inf: "\<not> lfinite xs \<Longrightarrow> lfuse xs ys = xs"
by (simp add: lappend_inf lfuse_def)

lemma llist_all2_lfuseI:
  assumes 1: "llist_all2 P xs ys"
  and 2: "\<lbrakk> lfinite xs; lfinite ys \<rbrakk> \<Longrightarrow> llist_all2 P xs' ys'"
  shows "llist_all2 P (lfuse xs xs') (lfuse ys ys')"
proof(cases "lfinite xs")
  case True
  with 1 have "lfinite ys" by(auto dest: llist_all2_lfiniteD)
  from 1 2[OF True this] show ?thesis
    proof (coinduction arbitrary: xs ys)
    case LNil
    then show ?case by (simp add: lfuse_conv_lnull llist_all2_lnullD)
    next
    case LCons
    then show ?case
    by (metis (no_types, lifting) lfuse_def llist.rel_sel llist_all2_lappendI)
    qed
 next
  case False
  with 1 have "\<not> lfinite ys" by(auto dest: llist_all2_lfiniteD)
  with False 1 show ?thesis
  by (simp add: lfuse_inf)  
qed


subsection \<open>ridx and lidx\<close>

lemma ridx_lidx: 
 "ridx (<) xs = lidx xs" 
unfolding lidx_def ridx_def 
by simp

lemma ridx_expand_1:
 " ridx R xs \<longleftrightarrow> lnull xs \<or> llength xs = 1 \<or> 
               (1 < llength xs \<and> (\<forall>n. (Suc n)<llength xs \<longrightarrow> R (lnth xs n)  (lnth xs (Suc n))))"
unfolding ridx_def
by (metis Zero_not_Suc enat_0_iff(1) gr_implies_not_zero iless_eSuc0 linorder_cases llength_eq_0 
    one_eSuc)

lemma lidx_expand_1:
 " lidx xs \<longleftrightarrow> lnull xs \<or> llength xs = 1 \<or> 
               (1 < llength xs \<and> (\<forall>n. (Suc n)<llength xs \<longrightarrow> lnth xs n < lnth xs (Suc n)))"
using ridx_lidx ridx_expand_1 by blast

lemma ridx_LCons:
 " ridx R (LCons x xs) \<longleftrightarrow> 
     lnull xs \<or> 
     (0 < llength xs \<and> 
      (\<forall>n. (Suc n)< eSuc (llength xs)  \<longrightarrow> R (lnth (LCons x xs) n) (lnth (LCons x xs) (Suc n))))"
unfolding ridx_def
using enat_0_iff(1) by force 

lemma lidx_LCons:
 " lidx (LCons x xs) \<longleftrightarrow> 
     lnull xs \<or> 
     (0 < llength xs \<and> 
      (\<forall>n. (Suc n)< eSuc (llength xs)  \<longrightarrow> lnth (LCons x xs) n < lnth (LCons x xs) (Suc n)))" 
using ridx_lidx[of "(LCons x xs)"] ridx_LCons[of "(<)" x xs] by blast

lemma ridx_LCons_conv:
 "(0 < llength xs \<and> 
   (\<forall>n. (Suc n)< eSuc (llength xs)  \<longrightarrow> R (lnth (LCons x xs) n) (lnth (LCons x xs) (Suc n))))
  \<longleftrightarrow> (0 < llength xs \<and> 
           R x (lhd xs) \<and> 
           (\<forall>n. 0\<le> n \<and> (Suc n)<  (llength xs)  \<longrightarrow> R (lnth xs n)  (lnth xs (Suc n))))"
proof -
 have 1: "(0 < llength xs \<and> 
           (\<forall>n. (Suc n)< eSuc (llength xs)  \<longrightarrow> R (lnth (LCons x xs) n)  (lnth (LCons x xs) (Suc n)))) \<longleftrightarrow>
          (0 < llength xs \<and> 
           (\<forall>n. 0\<le> n \<and> (Suc n)< eSuc (llength xs)  \<longrightarrow> R (lnth (LCons x xs) n)  (lnth (LCons x xs) (Suc n)))) " 
       by blast
 have 2: "(0 < llength xs \<and> 
           (\<forall>n. 0\<le> n \<and> (Suc n)< eSuc (llength xs)  \<longrightarrow> R (lnth (LCons x xs) n) (lnth (LCons x xs) (Suc n)))) \<longleftrightarrow>
          (0 < llength xs \<and> 
           R x  (lhd xs) \<and> 
           (\<forall>n. 1\<le> n \<and> (Suc n)< eSuc (llength xs)  \<longrightarrow> R (lnth (LCons x xs) n) (lnth (LCons x xs) (Suc n))))"
           by (metis Extended_Nat.eSuc_mono One_nat_def eSuc_enat gr_implies_not_zero ldropn_0 less_Suc_eq 
               lhd_ldropn lnth_0 lnth_Suc_LCons not_le_imp_less zero_enat_def) 
 have 3: "(0 < llength xs \<and> 
           R x (lhd xs) \<and> 
           (\<forall>n. 1\<le> n \<and> (Suc n)< eSuc (llength xs)  \<longrightarrow> R (lnth (LCons x xs) n)  (lnth (LCons x xs) (Suc n)))) \<longleftrightarrow>
         (0 < llength xs \<and> 
           R x (lhd xs) \<and> 
           (\<forall>n. 1\<le> n \<and> (n)<  (llength xs)  \<longrightarrow> R (lnth (LCons x xs) n)  (lnth (LCons x xs) (Suc n))))" 
    using Suc_ile_eq by auto
 have 4: "(0 < llength xs \<and> 
           R x (lhd xs) \<and> 
           (\<forall>n. 1\<le> n \<and> (n)<  (llength xs)  \<longrightarrow> R (lnth (LCons x xs) n)  (lnth (LCons x xs) (Suc n)))) \<longleftrightarrow>
          (0 < llength xs \<and> 
           R x (lhd xs) \<and> 
           (\<forall>n. 1\<le> n \<and> (n)<  (llength xs)  \<longrightarrow> R (lnth xs (n -1)) ( lnth xs ( n))))" 
    by (metis le_add_diff_inverse llist.disc(2) lnth_ltl ltl_simps(2) plus_1_eq_Suc)
 have 5: "(0 < llength xs \<and> 
           R x (lhd xs) \<and> 
           (\<forall>n. 1\<le> n \<and> (n)<  (llength xs)  \<longrightarrow> R (lnth xs (n -1)) ( lnth xs ( n)))) \<longleftrightarrow>
          (0 < llength xs \<and> 
           R x  (lhd xs) \<and> 
           (\<forall>n. 0\<le> (n-1) \<and> (Suc (n-1))<  (llength xs)  \<longrightarrow> R (lnth xs (n -1))  (lnth xs (Suc (n -1)))))"
         by (metis diff_Suc_1 le0 le_add1 le_add_diff_inverse plus_1_eq_Suc) 
 have 6: "(0 < llength xs \<and> 
           R x (lhd xs) \<and> 
           (\<forall>n. 0\<le> (n-1) \<and> (Suc (n-1))<  (llength xs)  \<longrightarrow> R (lnth xs (n -1))  (lnth xs (Suc (n -1))))) \<longleftrightarrow>
          (0 < llength xs \<and> 
           R x (lhd xs) \<and> 
           (\<forall>n. 0\<le> n \<and> (Suc n)<  (llength xs)  \<longrightarrow> R (lnth xs n)  (lnth xs (Suc n))))"
         by (metis diff_Suc_1) 
  show ?thesis 
  using "2" "3" "4" "5" "6" by auto  
qed


lemma lidx_LCons_conv:
 "(0 < llength xs \<and> (\<forall>n. (Suc n)< eSuc (llength xs)  \<longrightarrow> lnth (LCons x xs) n < lnth (LCons x xs) (Suc n)))
  \<longleftrightarrow> (0 < llength xs \<and> 
           x< lhd xs \<and> 
           (\<forall>n. 0\<le> n \<and> (Suc n)<  (llength xs)  \<longrightarrow> lnth xs n < lnth xs (Suc n)))"
using ridx_LCons_conv 
by metis

lemma ridx_LCons_1 [simp]:
 "ridx R (LCons x xs) \<longleftrightarrow> lnull xs \<or> (0 < llength xs \<and> R x (lhd xs) \<and> ridx R xs )"
unfolding ridx_def
using ridx_LCons_conv[of xs R x]
by (auto simp add: enat_0_iff(1))


lemma lidx_LCons_1 [simp]:
 "lidx (LCons x xs) \<longleftrightarrow> lnull xs \<or> (0 < llength xs \<and> x< lhd xs \<and> lidx xs )"
using ridx_lidx[of "LCons x xs" ] ridx_lidx[of xs]ridx_LCons_1[of "(<)" x xs]  by blast

lemma ridx_less:
 assumes " ridx R xs "
         " Suc(n+k) < llength xs"
         "transp R" 
 shows   " R (lnth xs n)  (lnth xs (Suc(n+k))) "
using assms  unfolding ridx_def
proof (induct k)
case 0
then show ?case by simp
next
case (Suc k)
then show ?case
by (metis Suc_ile_eq add_Suc_right order_less_imp_le transpE)
qed

lemma lidx_less:
 assumes " lidx xs "
         " Suc(n+k) < llength xs"
 shows   " lnth xs n < lnth xs (Suc(n+k)) "
using assms  ridx_lidx ridx_less[of "(<)" xs n k]  transp_on_less by blast
  
lemma ridx_less_eq:
 assumes " ridx R xs "
         " k \<le> j"
         " j < llength xs "
         "transp R" 
         "reflp R" 
 shows " R (lnth xs k)  (lnth xs j)" 
proof (cases "k=j")
case True
then show ?thesis using assms by (simp add: reflp_def)
next
case False
then show ?thesis using assms 
by (metis less_iff_Suc_add order_neq_le_trans ridx_less)
qed

lemma lidx_less_eq:
 assumes " lidx  xs "
         " k \<le> j"
         " j < llength xs "
 shows " lnth xs k \<le> lnth xs j" 
using assms ridx_lidx ridx_less_eq[of "(<)" xs k j ] 
by (metis dual_order.order_iff_strict less_iff_Suc_add lidx_less) 

lemma ridx_gr_first:
 assumes "ridx R xs"
         " 0<i"
         " i < llength xs"
         "transp R" 
 shows "R (lnth xs 0)  (lnth xs i) "
using assms ridx_less[of R xs 0 "i-1" ] unfolding ridx_def by simp

lemma lidx_gr_first:
 assumes "lidx xs"
         " 0<i"
         " i < llength xs"
 shows "(lnth xs 0) < lnth xs i "
using assms lidx_less[of  xs 0 "i-1" ] unfolding lidx_def by simp

lemma ridx_ltake_a: 
 assumes "ridx R xs"
         "n \<le> llength xs"
 shows   "ridx R (ltake n xs) " 
using assms
unfolding ridx_def 
by simp
  (metis Suc_ile_eq dual_order.strict_trans1 lnth_ltake order_less_imp_le)

lemma lidx_ltake_a: 
 assumes "lidx  xs"
         "n \<le> llength xs"
 shows   "lidx  (ltake n xs) " 
using assms
using ridx_lidx ridx_ltake_a by blast

lemma ridx_lappend_lfinite: 
assumes " lfinite xs "
shows   " ridx R (lappend xs ys) \<longleftrightarrow> 
          ridx R xs \<and> ((lnull xs \<or> lnull ys) \<or> R (llast xs)  (lhd ys)) \<and> ridx R ys " 
using assms
proof (induction xs arbitrary: ys)
case lfinite_LNil
then show ?case  by (simp add: ridx_expand_1)
next
case (lfinite_LConsI xs x)
then show ?case
   proof (cases "lnull xs")
   case True
   then show ?thesis  
   by (metis lappend_code(1) lappend_code(2) llast_singleton llength_LCons llist.collapse(1)
       one_eSuc order_neq_le_trans ridx_LCons_1 ridx_expand_1 zero_le)
   next
   case False
   then show ?thesis 
   by (simp add: lfinite_LConsI.IH llast_LCons)
   qed
qed

lemma lidx_lappend_lfinite: 
 assumes "lfinite xs" 
 shows   "lidx (lappend xs ys) \<longleftrightarrow> 
          lidx xs \<and> ((lnull xs \<or> lnull ys) \<or> (llast xs) <  (lhd ys)) \<and> lidx ys " 
using assms by (metis ridx_lappend_lfinite ridx_lidx)

lemma ridx_ldrop: 
 assumes "ridx R xs"
         "n \<le> llength xs" 
 shows   "ridx R (ldrop n xs) " 
proof -
 have 1: "xs = lappend (ltake n xs) (ldrop n xs)"
   by (simp add: lappend_ltake_ldrop) 
 have 2: "\<not> lfinite (ltake n xs) \<Longrightarrow> ?thesis"
    by (simp add: ridx_expand_1)     
 have 3: "lfinite (ltake n xs) \<Longrightarrow> ridx R (lappend (ltake n xs) (ldrop n xs)) \<longleftrightarrow>
           ridx R (ltake n xs) \<and> 
           ((lnull (ltake n xs) \<or> lnull (ldrop n xs)) \<or> R (llast (ltake n xs))  (lhd (ldrop n xs))) \<and>
           ridx R (ldrop n xs)" 
     using ridx_lappend_lfinite[of "(ltake n xs)" R "(ldrop n xs)" ] by blast
 have 4: "lfinite (ltake n xs) \<Longrightarrow> ?thesis" 
    using 1 3 assms by metis
 show ?thesis using 2 4 by blast
qed       
 
lemma lidx_ldrop: 
 assumes "lidx xs"
         "n \<le> llength xs" 
 shows   "lidx (ldrop n xs) " 
using assms ridx_ldrop ridx_lidx by blast

lemma ridx_ltake_all: 
 assumes "\<And>n. n\<le> llength xs \<Longrightarrow> ridx R (ltake (enat n) xs) "
 shows   "ridx R xs" 
using assms
unfolding ridx_def 
by auto
  (metis Suc_ile_eq dual_order.refl eSuc_enat ile_eSuc lessI lnth_ltake)

lemma lidx_ltake_all: 
 assumes "\<And>n. n\<le> llength xs \<Longrightarrow> lidx (ltake (enat n) xs) "
 shows   "lidx xs" 
using assms ridx_ltake_all  ridx_lidx by blast

lemma ridx_ltake: 
 assumes "ridx R (ltake n xs) "
         "n \<le> llength xs" 
         "k \<le> n"
 shows   "ridx R (ltake (enat k) xs)" 
using assms
using ridx_ltake_a by fastforce

lemma lidx_ltake: 
 assumes "lidx (ltake n xs) "
         "n \<le> llength xs" 
         "k \<le> n"
 shows   "lidx (ltake (enat k) xs)" 
using assms ridx_ltake ridx_lidx by blast

lemma lidx_imp_lsorted:
assumes "lidx xs"
shows   "lsorted xs"
using assms 
by (metis (no_types, lifting) less_imp_le lhd_LCons_ltl lidx_LCons_1 lsorted_coinduct')

lemma lidx_imp_ldistinct:
 assumes "lidx  xs"
 shows  "ldistinct xs"
using assms 
proof (coinduction arbitrary: xs)
case (ldistinct xsa)
then show ?case 
   proof -
     have 1: "lhd xsa \<notin> lset (ltl xsa)"
       by (metis empty_iff lappend_code(1) lappend_lnull2 ldistinct(1) ldistinct(2) leD lhd_LCons_ltl 
     lidx_LCons_1 lidx_imp_lsorted lmember_code(2) lset_LNil lset_lmember lsortedD)        
     have 2: "((\<exists>xs. ltl xsa = xs \<and> lidx  xs) \<or> ldistinct (ltl xsa))" 
       unfolding lidx_def 
       by (metis Extended_Nat.eSuc_mono eSuc_enat ldistinct(1) ldistinct(2) lhd_LCons_ltl lidx_def 
           llength_LCons lnth_ltl)
     show ?thesis 
       using "1" "2" by auto    
  qed
qed

lemma ldistinct_Ex1:
 assumes "ldistinct xs"
         " x \<in> lset xs"
 shows   " \<exists>!i. i< llength xs \<and> (lnth xs i) = x"
using assms 
by (metis in_lset_conv_lnth ldistinct_conv_lnth)

lemma lidx_lset_eq:
 assumes "lidx  xs"
         "lidx  ys"
         "lset xs = lset ys"
 shows " xs = ys"
using assms 
 by (simp add: lidx_imp_ldistinct lidx_imp_lsorted lsorted_ldistinct_lset_unique)

lemma ridx_lfuse_lfirst_llast: 
 assumes " ridx R ys "
         " (lnth ys 0 ) = (0::nat) "
         " ridx R zs "
         "(lnth zs 0 ) = 0 " 
         "lfinite ys "
         "lfinite xs "
         "\<not> lnull xs " 
         "\<not> lnull zs"
         " llast ys = cp "
 shows " llast ys = lfirst(lmap (\<lambda>x. x+cp) zs) " 
using assms unfolding lfirst_def
by (simp add: lnth_0_conv_lhd) 

lemma lidx_lfuse_lfirst_llast: 
 assumes " lidx ys "
         " (lnth ys 0 ) = 0 "
         " lidx zs "
         "(lnth zs 0 ) = 0 " 
         "lfinite ys "
         "lfinite xs "
         "\<not> lnull xs " 
         "\<not> lnull zs"
         " llast ys = cp "
 shows " llast ys = lfirst(lmap (\<lambda>x. x+cp) zs) "
using assms 
by (simp add: lfirst_def lnth_0_conv_lhd)

lemma ridx_lfuse_lnth_cp: 
 assumes " ridx R ys "
         " (lnth ys 0 ) = 0 "
         " ridx R zs "
         "(lnth zs 0 ) = 0 " 
         "lfinite ys "
         "lfinite zs" 
         "lfinite xs "
         "\<not> lnull xs " 
         "\<not> lnull zs"
         "\<not> lnull ys" 
         " llast ys = cp "
         " llast zs = the_enat(epred (llength xs)) - cp "
         " i < (llength zs)" 
         "cp < llength xs " 
 shows "lnth (lfuse ys (lmap (\<lambda>x. x+cp) zs)) (the_enat(epred(llength ys)) + i) = cp + (lnth zs i) "
proof -
 have 1: "llast ys = lfirst(lmap (\<lambda>x. x+cp) zs) "
    by (metis assms(11) assms(4) assms(9) lfirst_def llist.map_sel(1) lnth_0_conv_lhd plus_nat.add_0)
 have 3: "lfirst(lmap (\<lambda>x. x+cp) zs) = cp + (lnth zs 0)" 
   using "1" assms(11) assms(4) by force
 have 8: "i\<le> epred(llength zs) "
    using assms 
     by (metis  co.enat.exhaust_sel iless_Suc_eq llength_eq_0)
 have 81: "enat (the_enat (epred (llength ys)) + i) \<le> 
           enat (the_enat (epred (llength ys)) + (the_enat (epred(llength zs)))) " 
   by (metis "8" add_mono_thms_linordered_semiring(2) assms(6) assms(9) co.enat.exhaust_sel 
       enat_ord_simps(1) enat_ord_simps(4) enat_the_enat ile_eSuc iless_Suc_eq lfinite_llength_enat 
       llength_eq_0 order_le_less_trans)
 have 9: "epred(llength zs) < llength zs" 
   by (metis assms(6) assms(9) co.enat.exhaust_sel ile_eSuc iless_Suc_eq lfinite_llength_enat 
       llength_eq_0 order_neq_le_trans)
 have 10: "epred (llength ys) \<le> enat (the_enat (epred (llength ys)) + (i::nat))"
   by (metis assms(10) assms(5) co.enat.exhaust_sel enat_ord_simps(1) enat_the_enat ile_eSuc 
       infinity_ileE le_add1 lfinite_llength_enat llength_eq_0)
 have 83: " enat (the_enat (epred (llength ys)) + (the_enat (epred(llength zs)))) =
            enat (the_enat (epred (llength ys) + epred(llength zs))) "
      by (metis "10" "9" enat_ord_code(4) enat_the_enat leD order_less_imp_le plus_enat_simps(1))
 have 84: "enat (the_enat (epred (llength ys) + epred(llength zs))) =
            ( (epred (llength ys) + epred(llength zs)))" 
    by (metis "10" "9" enat_ord_code(4) enat_the_enat leD order_less_imp_not_less plus_enat_simps(1))
 have 85: " epred(llength ys) < llength ys"
   by (metis "10" assms(10) co.enat.exhaust_sel enat_ord_code(4) enat_the_enat ile_eSuc iless_Suc_eq 
       leD llength_eq_0 order_neq_le_trans) 
 have 86: " llength (lfuse ys (lmap (\<lambda>x::nat. x + cp) zs)) = llength ys + epred(llength zs)"
     by (simp add: assms(10) lfuse_llength)
 have 87: "( (epred (llength ys) + epred(llength zs))) < llength ys + epred(llength zs)"
    by (metis "83" "84" "85" add.commute enat_le_plus_same(2) enat_less_enat_plusI2 enat_the_enat 
        infinity_ileE)
 have 82: "enat (the_enat (epred (llength ys)) + (the_enat (epred(llength zs)))) <
           llength (lfuse ys (lmap (\<lambda>x::nat. x + cp) zs))"
    using "83" "84" "86" "87" by presburger
 have 11: "enat (the_enat (epred (llength ys)) + i) < llength (lfuse ys (lmap (\<lambda>x::nat. x + cp) zs))" 
   using "81" "82" order_le_less_trans by blast
 have 12: "(the_enat (epred (llength ys)) + i - the_enat (epred (llength ys))) = i" 
   by auto
 have 4: "lnth (lfuse ys (lmap (\<lambda>x. x+cp) zs)) (the_enat(epred(llength ys)) + i ) = 
          (lnth (lmap (\<lambda>x. x+cp) zs)  i)"
   by (simp add: "1" "10" "11" assms(13) assms(9) lfuse_lnth_c)
 have 5: "(lnth (lmap (\<lambda>x. x+cp) zs)  i) = cp + (lnth zs i)" 
    by (simp add: assms)
 show ?thesis 
 using "4" "5" by presburger
qed

lemma lidx_lfuse_lnth_cp: 
 assumes " lidx ys "
         " (lnth ys 0 ) = 0 "
         " lidx zs "
         "(lnth zs 0 ) = 0 " 
         "lfinite ys "
         "lfinite zs" 
         "lfinite xs "
         "\<not> lnull xs " 
         "\<not> lnull zs"
         "\<not> lnull ys" 
         " llast ys = cp "
         " llast zs = the_enat(epred (llength xs)) - cp "
         " i < (llength zs)" 
         "cp < llength xs " 
 shows "lnth (lfuse ys (lmap (\<lambda>x. x+cp) zs)) (the_enat(epred(llength ys)) + i) = cp + (lnth zs i) "
using assms ridx_lidx ridx_lfuse_lnth_cp by blast

lemma ridx_lfuse_lnth_cp_a: 
 assumes " ridx R ys "
         " (lnth ys 0 ) = 0 "
         " ridx R zs "
         "(lnth zs 0 ) = 0 " 
         "lfinite ys "
         "lfinite zs" 
         "lfinite xs "
         "\<not> lnull xs " 
         "\<not> lnull zs"
         "\<not> lnull ys" 
         " llast ys = cp "
         " llast zs = the_enat(epred (llength xs)) - cp "
         " i < (epred(llength ys)) + (llength zs)" 
         " the_enat(epred(llength ys)) \<le> i " 
         "cp < llength xs" 
 shows "lnth (lfuse ys (lmap (\<lambda>x. x+cp) zs)) i = cp + (lnth zs (i -the_enat(epred(llength ys)))) "
proof -
 have 1: "i = (the_enat (epred (llength ys)) + (i - the_enat (epred (llength ys))))"
   using assms by fastforce 
 have 2: "enat (i - the_enat (epred (llength ys))) < llength zs"
  using assms 
 by (metis "1" enat_add_mono epred_enat lfinite_llength_enat plus_enat_simps(1) the_enat.simps) 
 show ?thesis 
 using 1 2 assms ridx_lfuse_lnth_cp[of R ys zs xs cp "(i -the_enat(epred(llength ys)))"]
 by presburger 
qed

lemma lidx_lfuse_lnth_cp_a: 
 assumes " lidx ys "
         " (lnth ys 0 ) = 0 "
         " lidx zs "
         "(lnth zs 0 ) = 0 " 
         "lfinite ys "
         "lfinite zs" 
         "lfinite xs "
         "\<not> lnull xs " 
         "\<not> lnull zs"
         "\<not> lnull ys" 
         " llast ys = cp "
         " llast zs = the_enat(epred (llength xs)) - cp "
         " i < (epred(llength ys)) + (llength zs)" 
         " the_enat(epred(llength ys)) \<le> i " 
         "cp < llength xs" 
 shows "lnth (lfuse ys (lmap (\<lambda>x. x+cp) zs)) ( i) = cp + (lnth zs (i -the_enat(epred(llength ys)))) "
using assms ridx_lidx ridx_lfuse_lnth_cp_a by blast

lemma ridx_lfuse_lnth_cp_llast: 
 assumes " ridx R ys "
         " (lnth ys 0 ) = 0 "
         " ridx R zs "
         "(lnth zs 0 ) = 0 " 
         "lfinite ys "
         "lfinite zs" 
         "lfinite xs "
         "\<not> lnull xs " 
         "\<not> lnull zs"
         "\<not> lnull ys" 
         " llast ys = cp "
         " llast zs = the_enat(epred (llength xs)) - cp "
         " i < (llength zs)" 
         " cp < llength xs" 
shows " llast (lfuse ys (lmap (\<lambda>x. x+cp) zs)) = (the_enat (epred(llength xs))) " 
proof -
 have 1: "llast (lfuse ys (lmap (\<lambda>x. x+cp) zs)) = llast (lmap (\<lambda>x. x+cp) zs)" 
   using assms
  by (metis add_cancel_left_left lfinite_lmap lfirst_def llast_lfuse llist.map_disc_iff
       llist.map_sel(1) lnth_0_conv_lhd)
 have 2: "llast (lmap (\<lambda>x. x+cp) zs) = cp + (llast zs)" 
    by (simp add: assms(6) assms(9) llast_lmap)
 have 3: "cp + (llast zs) = (the_enat (epred(llength xs)))"
    using assms 
    by (metis add_diff_inverse_nat co.enat.exhaust_sel enat_ord_simps(2) enat_the_enat ileI1
        ile_eSuc infinity_ileE leD lfinite_conv_llength_enat llength_eq_0) 
 show ?thesis using 1 2 3 by auto
qed

lemma lidx_lfuse_lnth_cp_llast: 
 assumes " lidx ys "
         " (lnth ys 0 ) = 0 "
         " lidx zs "
         "(lnth zs 0 ) = 0 " 
         "lfinite ys "
         "lfinite zs" 
         "lfinite xs "
         "\<not> lnull xs " 
         "\<not> lnull zs"
         "\<not> lnull ys" 
         " llast ys = cp "
         " llast zs = the_enat(epred (llength xs)) - cp "
         " i < (llength zs)" 
         " cp < llength xs" 
shows " llast (lfuse ys (lmap (\<lambda>x. x+cp) zs)) = (the_enat (epred(llength xs))) "
using assms ridx_lidx ridx_lfuse_lnth_cp_llast by blast
 
lemma ridx_lfuse_lnth_cp_infinite: 
 assumes " ridx R ys "
         " (lnth ys 0 ) = (0::nat) "
         " ridx R zs "
         "(lnth zs 0 ) = 0 " 
         "lfinite ys "
         "\<not>lfinite zs" 
         "\<not>lfinite xs "
         "\<not> lnull ys" 
         " llast ys = cp "
 shows "lnth (lfuse ys (lmap (\<lambda>x. x+cp) zs)) (the_enat(epred(llength ys)) + i) = cp + (lnth zs i) "
proof -
 have 1: "llast ys = lfirst(lmap (\<lambda>x. x+cp) zs) "
   by (metis assms(4) assms(6) assms(9) lfirst_def llist.map_sel(1) lnth_0_conv_lhd 
       lnull_imp_lfinite plus_nat.add_0)
 have 3: "lfirst(lmap (\<lambda>x. x+cp) zs) = cp + (lnth zs 0)" 
   using "1" assms by auto
 have 8: "i\<le> epred(llength zs) "
     by (metis assms(6) enat.simps(3) ldrop_eq_LNil lfinite_ldrop lfinite_ltl linorder_le_cases llength_ltl)
 have 10: "epred (llength ys) \<le> enat (the_enat (epred (llength ys)) + (i::nat))"
  by (metis add.right_neutral add_le_same_cancel1 assms(5) assms(8) co.enat.exhaust_sel 
      enat_ord_simps(1) enat_the_enat ile_eSuc infinity_ileE le_zero_eq lfinite_llength_enat 
      linorder_le_cases llength_eq_0)
 have 11: "enat (the_enat (epred (llength ys)) + i) < llength (lfuse ys (lmap (\<lambda>x::nat. x + cp) zs))" 
   using assms 
   by (metis "1"  enat_ile ldrop_lfuse lfinite_conv_llength_enat lfinite_ldrop 
       lfinite_lmap not_le_imp_less)
 have 12: "(the_enat (epred (llength ys)) + i - the_enat (epred (llength ys))) = i" 
   by auto
 have 4: "lnth (lfuse ys (lmap (\<lambda>x. x+cp) zs)) (the_enat(epred(llength ys)) + i ) = 
          (lnth (lmap (\<lambda>x. x+cp) zs)  i)"
   using assms 
   by (metis "1" "10" "11" "12"  lfuse_lnth_c llist.map_disc_iff lnull_imp_lfinite)
 have 5: "(lnth (lmap (\<lambda>x. x+cp) zs)  i) = cp + (lnth zs i)" 
  using assms 
   by (metis "8" add.commute co.enat.exhaust_sel iless_Suc_eq llength_eq_0 lnth_lmap 
       lnull_imp_lfinite)
 show ?thesis 
 using "4" "5" by presburger
qed

lemma lidx_lfuse_lnth_cp_infinite: 
 assumes " lidx ys "
         " (lnth ys 0 ) = 0 "
         " lidx zs "
         "(lnth zs 0 ) = 0 " 
         "lfinite ys "
         "\<not>lfinite zs" 
         "\<not>lfinite xs "
         "\<not> lnull ys" 
         " llast ys = cp "
 shows "lnth (lfuse ys (lmap (\<lambda>x. x+cp) zs)) (the_enat(epred(llength ys)) + i) = cp + (lnth zs i) "
using assms ridx_lidx ridx_lfuse_lnth_cp_infinite by blast

lemma lidx_lfuse_lidx:
 assumes "lidx  ys"
         "lnth ys 0 = 0 " 
         "lidx  zs"
         "lnth zs 0 = 0 " 
         "lfinite ys"  
         "\<not> lnull ys" 
         "llast ys  = cp"
         "lfinite zs" 
         "\<not> lnull zs" 
         "lfinite xs" 
         "llast zs  = the_enat(epred(llength xs)) -cp"
         "cp < llength xs"
         "i < llength zs"
         "cp < llength xs" 
 shows   "lidx (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) \<and> (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) 0) = 0"
proof -
 have 1: "llast ys = lfirst(lmap (\<lambda>x. x+ cp) zs)"
   by (metis assms(4) assms(7) assms(9) cancel_comm_monoid_add_class.diff_cancel diff_add
       dual_order.refl lfirst_def llist.map_sel(1) lnth_0_conv_lhd) 
 have 2: "lfirst (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) = lfirst ys" 
   by (simp add: assms(6) assms(9) lfirst_lfuse_1)
 have 3: "llength (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) = llength ys + epred(llength zs)" 
  by (simp add: assms(13) assms(6) lfuse_llength)
 have 4: "\<And>j. j<llength ys \<Longrightarrow> lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j= lnth ys j"
   using lfuse_lnth_a by blast 
 have 40: "\<exists> k1. llength ys = (enat k1) " 
   by (simp add: assms(5) lfinite_llength_enat)
 obtain k1 where 41: "llength ys = (enat k1)" 
   using 40 by blast
 have 42: "(enat (k1 -1)) = epred(llength ys) " 
     using "41" epred_enat by presburger
 have 43: "0<k1" 
   using assms 41 
   by (metis gr0I llength_eq_0 zero_enat_def)
 have 44: "the_enat(epred(llength ys)) = (k1 -1)"
   by (metis "42" the_enat.simps) 
 have 5: "\<And>j.  epred(llength ys) \<le>j \<and> j < epred(llength ys) + llength zs \<Longrightarrow>
          lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j = 
          cp + (lnth zs (j- the_enat(epred(llength ys)))) "   
    using assms lidx_lfuse_lnth_cp_a[of ys zs xs cp] 
     by (metis  enat_ord_simps(1) enat_the_enat gr_implies_not_zero 
          infinity_ileE  llength_eq_0)
 have 45: "\<And>j.  k1-1 \<le>j \<and> j < (enat (k1-1)) + llength zs \<Longrightarrow>
          lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j = 
          cp + (lnth zs (j- (k1-1)))"
    using 5 44 by (metis "42" enat_ord_simps(1)) 
 have 50: "llength(lfuse ys (lmap (\<lambda>x. x+ cp) zs)) = epred(llength ys) + llength zs" 
   using assms 
   by (metis "3" add.commute epred_iadd1 llength_eq_0)
 have 51: "\<And>j. enat (Suc j) < llength ys \<Longrightarrow>
               (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j) <
               (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) (Suc j))"
     by (metis assms(1) assms(5) eSuc_enat iless_Suc_eq lfinite_conv_llength_enat lfuse_lnth_a 
         lidx_def not_le_imp_less not_less_iff_gr_or_eq) 
 have 52: "\<And>j.  k1-1 \<le>j \<and> (Suc j) < enat (k1-1) + llength zs \<Longrightarrow>
                cp + (lnth zs (j- (k1-1))) <
                cp + (lnth zs ((Suc j)- (k1-1)))"
        using assms(3) unfolding lidx_def  
        by simp
         (metis Suc_diff_le add_Suc_right assms(8) enat_ord_simps(2) leD lfinite_llength_enat 
          nat_add_left_cancel_le not_le_imp_less ordered_cancel_comm_monoid_diff_class.add_diff_inverse 
          plus_enat_simps(1))
 have 6: "\<And>j. enat (Suc j) < llength(lfuse ys (lmap (\<lambda>x. x+ cp) zs)) \<Longrightarrow>
               (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j) <
               (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) (Suc j))"
      proof -
         fix j 
         assume a: "enat (Suc j) < llength(lfuse ys (lmap (\<lambda>x. x+ cp) zs))"
         show "(lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j) < (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) (Suc j))"  
         proof -
          have 61: "enat (Suc j) < llength ys \<Longrightarrow>
               (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j) < (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) (Suc j))" 
            using "51" by blast
          have 62: "k1-1 \<le>j \<and> (Suc j) < llength(lfuse ys (lmap (\<lambda>x. x+ cp) zs)) \<Longrightarrow>
               (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j) < (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) (Suc j))" 
            by (metis "42" "45" "50" "52" Suc_ile_eq nle_le not_less_eq_eq order_less_imp_le)
          show ?thesis
          by (metis "41" "43" "61" "62" Suc_diff_1 Suc_mono a enat_ord_simps(2) leI)
         qed
     qed
 show ?thesis unfolding lidx_def 
 by (simp add: "41" "43" "6" assms(13) assms(2) lfuse_lnth_a)
qed

lemma lidx_lfuse_lidx_infinite:
 assumes "lidx  ys"
         "lnth ys 0 = 0 " 
         "lidx  zs"
         "lnth zs 0 = 0 " 
         "lfinite ys"  
         "\<not> lnull ys" 
         "llast ys  = cp"
         "\<not>lfinite zs" 
         "\<not>lfinite xs" 
 shows   "lidx (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) \<and> (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) 0) = 0"
proof -
 have 1: "llast ys = lfirst(lmap (\<lambda>x. x+ cp) zs)"
    by (metis assms(4) assms(7) assms(8) lfirst_def llist.map_sel(1) lnth_0_conv_lhd 
        lnull_imp_lfinite plus_nat.add_0)
 have 2: "lfirst (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) = lfirst ys" 
   by (simp add: assms(6) lfirst_lfuse_1)
 have 4: "\<And>j. j<llength ys \<Longrightarrow> lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j= lnth ys j"
   using lfuse_lnth_a by blast
 have 40: "\<exists> k1. llength ys = (enat k1) " 
   by (simp add: assms(5) lfinite_llength_enat)
 obtain k1 where 41: "llength ys = (enat k1)" 
   using 40 by blast
 have 42: "(enat (k1 -1)) = epred(llength ys) " 
     using "41" epred_enat by presburger
 have 43: "0<k1" 
   using assms 41 
   by (metis gr0I llength_eq_0 zero_enat_def)
 have 44: "the_enat(epred(llength ys)) = (k1 -1)"
   by (metis "42" the_enat.simps) 
 have 5: "\<And>j.  epred(llength ys) \<le>j \<and> j < epred(llength ys) + llength zs \<Longrightarrow>
          lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j = 
          cp + (lnth zs (j- the_enat(epred(llength ys)))) "   
    using assms lidx_lfuse_lnth_cp_infinite[of ys zs xs cp  ]
      by (metis "42" "44" enat_ord_simps(1) ordered_cancel_comm_monoid_diff_class.add_diff_inverse)
 have 45: "\<And>j.  k1-1 \<le>j \<and> j < (enat (k1-1)) + llength zs \<Longrightarrow>
          lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j = 
          cp + (lnth zs (j- (k1-1)))"
    using 5 44 by (metis "42" enat_ord_simps(1))
 have 46: "llength ys + epred (llength zs) = epred (llength ys) + llength zs"
     by (metis add.commute assms(6) assms(8) epred_iadd1 llength_eq_0 lnull_imp_lfinite)
 have 50: "llength(lfuse ys (lmap (\<lambda>x. x+ cp) zs)) = epred(llength ys) + llength zs"
  using lfuse_llength[of ys "(lmap (\<lambda>x::nat. x + cp) zs)" ] 
        llength_lmap[of "(\<lambda>x::nat. x + cp)" zs] 
   using "46" assms(6) by presburger
 have 51: "\<And>j. enat (Suc j) < llength ys \<Longrightarrow>
               (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j) <
               (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) (Suc j))"
     by (metis assms(1) assms(5) eSuc_enat iless_Suc_eq lfinite_conv_llength_enat lfuse_lnth_a 
         lidx_def not_le_imp_less not_less_iff_gr_or_eq) 
 have 52: "\<And>j.  k1-1 \<le>j \<and> (Suc j) < enat (k1-1) + llength zs \<Longrightarrow>
                cp + (lnth zs (j- (k1-1))) <
                cp + (lnth zs ((Suc j)- (k1-1)))"
        using assms(3) unfolding lidx_def by simp
      (metis Nat.add_diff_assoc assms(8) enat_ile lfinite_conv_llength_enat not_le_imp_less
           plus_1_eq_Suc)
 have 53: "\<And>j.  k1-1 \<le>j \<and> (Suc j) < enat (k1-1) + llength zs \<Longrightarrow>
                 lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j <
                 lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) (Suc j)" 
       by (metis "45" "52" Suc_ile_eq nle_le not_less_eq_eq order_less_imp_le)
 have 6: "\<And>j. enat (Suc j) < llength(lfuse ys (lmap (\<lambda>x. x+ cp) zs)) \<Longrightarrow>
               (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) j) <
               (lnth (lfuse ys (lmap (\<lambda>x. x+ cp) zs)) (Suc j))"
    by (metis "41" "42" "43" "50" "51" "53" Suc_diff_1 Suc_eq_plus1 add_less_cancel_right 
        enat_ord_simps(2) leI)
 show ?thesis unfolding lidx_def 
 by (simp add: "41" "43" "6" assms lfuse_lnth_a)
qed
   

subsection \<open>lsub\<close>

lemma lsub_eq_lsubc: 
 assumes "k\<le> n"
         "n < llength xs "
 shows "lsub k n xs = lsubc k n xs" 
using assms
unfolding lsub_def lsubc_def
by (auto simp add: min_def)
   (metis co.enat.exhaust_sel enat_ord_simps(1) iless_Suc_eq ldrop_lnull llength_eq_0 
    order_le_less_trans)

lemma lsub_same: 
 assumes " (enat k)<llength xs" 
 shows   "lsub k k xs = (LCons (lnth xs k ) LNil) " 
using assms 
unfolding lsub_def 
by (metis LNil_eq_ltake_iff diff_is_0_eq' enat_0_iff(1) ldrop_enat ldropn_Suc_conv_ldropn
    ltake_eSuc_LCons order.order_iff_strict)

lemma lsubc_same: 
 assumes " (enat k)<llength xs" 
 shows   "lsubc k k xs = (LCons (lnth xs k ) LNil) " 
using assms 
lsub_eq_lsubc[of k k xs]  lsub_same[of k xs]
by fastforce

lemma llength_lsub: 
 assumes "k\<le> n"
         "n < llength xs " 
 shows "llength (lsub k n xs) = (eSuc (n-k))" 
proof -
 have 1: "llength (ldrop (enat k) xs) = (llength xs - enat k)" 
   by (simp add: ldrop_enat)
 have 2: " min (eSuc (enat (n - k))) (llength xs - enat k) = (eSuc (n-k)) " 
   by (metis "1" Suc_diff_le assms(1) assms(2) eSuc_enat enat_llength_ldropn ileI1 ldrop_enat min.absorb1)
 have 3: "llength (ltake (eSuc (enat (n - k))) (ldrop (enat k) xs)) = (eSuc (n-k))"
   by (simp add: "1" "2")
 show ?thesis 
   by (metis "3" idiff_enat_enat lsub_def)
qed

lemma llength_lsubc: 
 assumes "k\<le> n"
         "n < llength xs " 
 shows "llength (lsubc k n xs) = (eSuc (n-k))" 
using assms lsub_eq_lsubc[of k n xs] llength_lsub[of k n xs] by presburger

lemma llength_lsub_a: 
 shows "llength (lsub k n xs) = 
         min (eSuc (n - k)) 
             (if k = \<infinity> then 0::enat else llength xs - k)"
unfolding lsub_def 
using llength_ltake[of "eSuc (n-k)"  "(ldrop k xs)"]  llength_ldrop[of k xs ] 
using idiff_enat_enat by presburger

lemma llength_lsubc_a: 
 shows "llength (lsubc k n xs) = 
        min (eSuc (n - k)) 
            (if min k (epred (llength xs)) = \<infinity> 
             then 0::enat 
             else llength xs - min k (epred (llength xs)))"
unfolding lsubc_def 
using llength_ldrop[of "(min k (epred (llength xs)))" xs]
using llength_ltake[of "(eSuc (n - k))" "(ldrop (min k (epred (llength xs))) xs)"]
using idiff_enat_enat by presburger
   
lemma lsub_not_lnull: 
 assumes "k\<le> n"
         "n < llength xs " 
 shows "\<not>lnull (lsub k n xs)"
using assms 
by (metis eSuc_enat idiff_enat_enat llength_LNil llength_lsub llist.collapse(1) zero_ne_eSuc)

lemma lsub_llength_gr_one:
 assumes "k<n"
         "n < llength xs " 
 shows "1< llength (lsub k n xs) " 
using llength_lsub_a[of k n xs] assms
by (auto simp add: min_def one_eSuc zero_enat_def llength_lsub)

lemma lsub_lfinite: 
 assumes "k\<le> n"
         "n < llength xs " 
 shows "lfinite (lsub k n xs)"
using assms
by (simp add: eSuc_enat llength_eq_enat_lfiniteD llength_lsub)


lemma lnth_lsub: 
 assumes "n < llength xs " 
         " k+j \<le> n " 
 shows "lnth (lsub k n xs) j = (lnth xs (k+j))"
proof -
 have 1: "enat (j::nat) < eSuc (enat ((n::nat) - (k::nat)))" 
   using assms(2) by auto
 have 2: "lnth (ltake (eSuc (enat (n - k))) (ldrop (enat k) (xs::'a llist))) j = 
          lnth (ldrop (enat k) xs) j" 
  using "1" lnth_ltake by blast
 have 3: "lnth (ldrop (enat k) xs) j = (lnth xs (k+j))"
    by (metis add.commute assms(1) assms(2) enat_ord_simps(1) ldrop_enat lnth_ldropn order_le_less_trans)
 show ?thesis unfolding lsub_def 
 using "2" "3" by presburger
qed

lemma lnth_lsub_a: 
 assumes "n < llength xs " 
         " m \<le> n " 
         " k \<le> m" 
 shows "lnth (lsub k n xs) (m-k) = (lnth xs m)"
using assms by (simp add: lnth_lsub)

lemma ltake_lsub: 
 assumes "n < llength xs " 
         "m + k \<le>  n " 
 shows " ltake (eSuc m) (lsub k n xs) = lsub k (m+k) xs"
proof -
 have 1: "ltake (eSuc m) (ltake (eSuc (n - k)) (ldrop k xs)) =
          ltake (eSuc m) (ldrop k xs)"
     using ltake_ltake[of "(eSuc m)" "(eSuc (n - k))" "(ldrop k xs)"] 
    using Nat.le_diff_conv2 assms(2) by auto
 have 2: "ltake (eSuc (m + k - k)) (ldrop k xs) = ltake (eSuc m) (ldrop k xs)"
    by simp
  show ?thesis 
   unfolding lsub_def using 1 2 by presburger
qed

lemma ldrop_lsub: 
assumes "n < llength xs " 
         "m + k \<le>  n " 
 shows " ldrop m (lsub k n xs) = lsub (m+k) n xs"
proof -
 have 1: "(ltake (eSuc (n - k)) (ldrop k xs)) = 
          ldrop k (ltake (eSuc n) xs)"
      by (metis Suc_diff_le add_leD2 assms(2) eSuc_enat idiff_enat_enat ldrop_ltake)
 have 2: "ldrop m (ldrop k (ltake (eSuc n) xs)) =
          ldrop (m +k) (ltake (eSuc n) xs)"
   by simp   
 show ?thesis unfolding lsub_def using 1 2 
   by (simp add: Suc_diff_le assms(2) eSuc_enat ldrop_ltake)
qed
 
lemma ltl_lsub: 
 assumes " n < llength xs"
         " k \<le> n "
 shows   "ltl(lsub k n xs) = (if k=n then LNil else lsub k (n-1) (ltl xs))"
proof -
 have 1: "(lsub k n xs) = (ltake (eSuc (n - k)) (ldrop k xs))"
 unfolding lsub_def by simp
 have 2: "  k=n \<Longrightarrow> ?thesis "
   by (simp add: assms(1) lsub_same)
 have 25: "k \<noteq> n \<Longrightarrow> (epred (eSuc (n - k))) = (eSuc (enat (n - (1::nat) - k)))"
  by (metis Suc_diff_1 assms(2) co.enat.sel(2) diff_commute eSuc_enat order_neq_le_trans zero_less_diff)
  have 3: "k \<noteq> n \<Longrightarrow> ?thesis" 
  using assms ltl_ltake[of "(eSuc (n - k))" "(ldrop k xs)"]
    ltl_ldrop[of k xs]
   unfolding lsub_def 
     using "25" by presburger
 show ?thesis 
 using "2" "3" by blast
qed

lemma ltl_ldrop_one:
 " ltl xs = ldrop 1 xs" 
by (metis ldrop_0 ldrop_ltl one_eSuc)

lemma lappend_lsub_ltl_lsub: 
assumes " k \<le> n"
         " n \<le> m"
         " m < llength xs"
shows  "lappend (lsub k n xs) (ltl (lsub n m xs)) = lsub k m xs " 
proof -
 have 0: "(ltl (lsub n m xs)) = ldrop 1 (lsub n m xs)"
   by (metis ldrop_0 ldrop_ltl one_eSuc) 
 have 1: "(lsub k n xs) = (ltake (eSuc (n - k)) (ldrop k xs))"
     unfolding lsub_def by simp
 have 2: "(lsub n m xs) = (ltake (eSuc (m - n)) (ldrop n xs))"
     unfolding lsub_def by simp
 have 3: "lsub k m xs = (ltake (eSuc (m - k)) (ldrop k xs))"
    unfolding lsub_def by simp
 have 8: "ltake (eSuc (enat ((n::nat) - (k::nat))) + enat ((m::nat) - n)) (ldrop (enat k) (xs::'a llist)) =
         (ltake (eSuc (m - k)) (ldrop k xs)) " 
    by (simp add: assms(1) assms(2) eSuc_enat)
 have 9: "(ltake (enat (m - n)) (ldrop (eSuc (enat (n - k))) (ldrop (enat k) xs))) =
          ltl(ltake (eSuc (m - n)) (ldrop n xs)) " 
   by (metis "0" "2" add.commute assms(1) eSuc_minus_1 ldrop_ldrop ldrop_ltake le_add_diff_inverse 
       plus_1_eSuc(1) plus_enat_simps(1))
 have 10: "ltake (eSuc (enat ((n::nat) - (k::nat))) + enat ((m::nat) - n)) (ldrop (enat k) (xs::'a llist)) =
          lappend (ltake (eSuc (enat (n - k))) (ldrop (enat k) xs)) 
                  (ltake (enat (m - n)) (ldrop (eSuc (enat (n - k))) (ldrop (enat k) xs)))" 
  using ltake_plus_conv_lappend[of "(eSuc (n - k))" "m-n"  "(ldrop k xs)" ] by blast
 show ?thesis 
 using "1" "10" "2" "3" "8" "9" by fastforce
qed

lemma lsub_lfuse:
 assumes 
         " k \<le> n"
         " n \<le> m"
         " m < llength xs"
 shows " lfuse (lsub k n xs) (lsub n m xs) = lsub k m xs "
using assms 
by (simp add: lappend_lsub_ltl_lsub lfuse_def lsub_not_lnull order_le_less_subst2)

lemma llast_lsub: 
assumes "lfinite xs"
        "\<not> lnull xs"
        " k \<le> n "
        "n < llength xs"
shows   "llast (lsub k n xs) = (lnth xs n) " 
using assms
using lnth_lsub[of n xs k ]
by (metis  enat_ord_simps(1) idiff_enat_enat llast_conv_lnth llength_lsub 
not_le_imp_less order_less_irrefl ordered_cancel_comm_monoid_diff_class.add_diff_inverse) 

lemma lfirst_lsub:
 assumes "\<not> lnull xs"
         " k \<le> n "
        "n < llength xs"
 shows  " lfirst (lsub k n xs) = (lnth xs k) " 
using assms
by (simp add: ldrop_enat lfirst_def lhd_ldropn lsub_def order_le_less_subst2)


lemma lsub_lfuse_lidx:
 assumes "lidx ls" 
         "lfinite ls"
         "lfinite xs "
         "\<not> lnull xs" 
         "llast ls = epred(llength xs) "
         " (Suc i) < (llength ls) "
 shows   " lfuse (lsub (lnth ls i) (lnth ls (Suc i)) xs) (lsub (lnth ls (Suc i)) (llast ls) xs) =
           (lsub (lnth ls i) (llast ls) xs) " 
proof -
 have 1: "(lnth ls i) \<le> (lnth ls (Suc i))"
   by (simp add: assms(1) assms(6) lidx_less_eq)     
 have 2: " llast ls = (lnth ls (the_enat(epred (llength ls)))) " 
   using llast_conv_lnth 
   by (metis assms(2) assms(6) co.enat.collapse enat.simps(3) enat_ord_simps(4) enat_the_enat 
       gr_implies_not_zero ile_eSuc lfinite_llength_enat not_less_iff_gr_or_eq order_neq_le_trans)
 have 3: "1 < llength ls " 
    by (metis assms(1) assms(6) enat_ord_simps(1) gr_implies_not_zero leD le_add1 lidx_expand_1 
        llength_eq_0 one_enat_def plus_1_eq_Suc) 
 have 4: "(lnth ls (Suc i)) \<le> (llast ls)"
    using "2" assms(1) assms(2) assms(6) lfinite_llength_enat lidx_less_eq by fastforce
 have 5: "enat (lnth ls (Suc i)) < llength xs"
   by (metis "4" assms(4) assms(5) co.enat.exhaust_sel eSuc_enat enat_ord_simps(2) le_imp_less_Suc 
       llength_eq_0)
 have 6: "llast (lsub (lnth ls i) (lnth ls (Suc i)) xs) = (lnth xs (lnth ls (Suc i)))" 
   using llast_lsub[of xs "(lnth ls i)" "(lnth ls (Suc i))"]   
     using "1" "5" assms(3) assms(4) enat_ord_simps(1) by blast
 have 7: "lfirst (lsub (lnth ls (Suc i)) (llast ls) xs) = (lnth xs (lnth ls (Suc i))) "
   by (metis "4" assms(4) assms(5) co.enat.exhaust_sel enat_ord_simps(1) ile_eSuc iless_Suc_eq 
       lfirst_lsub llength_eq_0 order_less_le) 
 show ?thesis
 by (metis "1" "4" assms(4) assms(5) co.enat.exhaust_sel enat_ord_simps(1) ile_eSuc iless_Suc_eq 
     llength_eq_0 lsub_lfuse order_neq_le_trans) 
qed

subsection \<open>llastlfirst\<close>

lemma llastlfirst_ridx: 
 "llastlfirst xss = ridx (\<lambda> a b. llast a = lfirst b) xss" 
by (simp add: llastlfirst_def ridx_def)
           
lemma llastlfirst_LNil[simp]:
 " llastlfirst LNil"
unfolding llastlfirst_def by simp

lemma llastlfirst_LOne[simp]:
 " llastlfirst (LCons xs LNil)" 
unfolding llastlfirst_def by (simp add: zero_enat_def)

lemma llastlfirst_LCons[simp]: 
 assumes "\<not>lnull xss"     
 shows "llastlfirst (LCons xs xss) \<longleftrightarrow> llast xs = lfirst (lfirst xss) \<and> llastlfirst xss" 
using assms unfolding llastlfirst_def 
by auto
   (metis One_nat_def lfirst_def lhd_LCons_ltl lnth_LCons' not_lnull_llength one_enat_def, 
    metis Suc_ile_eq lnth_Suc_LCons,
    metis One_nat_def Suc_diff_le Suc_ile_eq add_diff_cancel_left' le_Suc_eq le_add1 lfirst_def 
    llist.disc(2) lnth_0 lnth_0_conv_lhd lnth_ltl ltl_simps(2) plus_1_eq_Suc)

lemma llastlfirst_lappend_lfinite:
assumes "lfinite xss" 
shows " llastlfirst (lappend xss yss) \<longleftrightarrow> 
        (llastlfirst xss \<and> llastlfirst yss \<and> 
        (if lnull xss \<or> lnull yss then True else llast(llast xss) = lfirst(lfirst yss)) )"
using assms
by (metis (mono_tags, lifting) lfirst_def llastlfirst_ridx ridx_lappend_lfinite)


subsection \<open>lfusecat\<close>

context notes [[function_internals]]
begin

partial_function (llist) lfusecat :: "'a llist llist \<Rightarrow> 'a llist"
where "lfusecat xss = (case xss of LNil \<Rightarrow> LNil | LCons xs xss' \<Rightarrow> lfuse xs (lfusecat xss'))"

end

lemma lfusecat_simps [simp, code]:
  shows lfusecat_LNil: "lfusecat LNil = LNil"
  and lfusecat_LCons: "lfusecat (LCons xs xss) = lfuse xs (lfusecat xss)"
by (simp_all add: lfusecat.simps)  

declare lfusecat.mono[cont_intro]

lemma mono2mono_lfusecat[THEN llist.mono2mono, cont_intro, simp]:
  shows monotone_lfusecat: "monotone (lprefix) (lprefix) lfusecat"
by(rule llist.fixp_preserves_mono1[OF lfusecat.mono lfusecat_def]) simp


lemma mcont2mcont_lfusecat[THEN llist.mcont2mcont, cont_intro, simp]:
  shows mcont_lfusecat: "mcont lSup (lprefix) lSup (lprefix) lfusecat"
by (rule llist.fixp_preserves_mcont1[OF lfusecat.mono lfusecat_def]) 
   simp

lemmas lfusecat_fixp_parallel_induct =
  parallel_fixp_induct_1_1[OF llist_partial_function_definitions llist_partial_function_definitions
    lfusecat.mono lfusecat.mono lfusecat_def lfusecat_def, case_names adm LNil step]


lemma llist_all2_lfusecatI:
  "llist_all2 (llist_all2 A) xss yss
  \<Longrightarrow> llist_all2 A (lfusecat xss) (lfusecat yss)"
proof(induct arbitrary: xss yss rule: lfusecat_fixp_parallel_induct)
case adm
then show ?case by (auto split: llist.split intro: llist_all2_lappendI)
next
case LNil
then show ?case by (auto split: llist.split intro: llist_all2_lappendI)
next
case (step f g)
then show ?case 
using llist_all2_lfuseI by (auto split: llist.split intro: llist_all2_lappendI) blast
qed


lemma not_lnull_lset_conv_a: 
 " \<not> lnull xss \<and> (\<forall> xs \<in> lset xss. \<not> lnull xs) \<longleftrightarrow> lset xss \<noteq> {} \<and> LNil \<notin> lset xss " 
using llist.collapse(1) llist.disc(1) lset_eq_empty by blast

lemma not_lnull_lset_conv_b: 
 " \<not> lnull xss \<and> (\<forall> xs \<in> lset xss. \<not> lnull xs) \<longleftrightarrow> lset xss \<noteq> {} \<and> lset xss \<inter> {LNil} = {} " 
using llist.collapse(1) by force

lemma lfusecat_eq_LNil: 
"lfusecat xss = LNil \<longleftrightarrow> lset xss \<subseteq> {LNil}  "
proof (induction xss)
case adm
then show ?case 
  by simp
next
case LNil
then show ?case 
by simp
next
case (LCons xs xss)
then show ?case 
by simp 
   (metis (no_types, lifting)  lfuse_conv_lnull  llist.disc(1) llist.expand )
qed 


lemma lfusecat_lfilter_neq_LNil:
  "lfusecat (lfilter (\<lambda>xs. \<not> lnull xs) xss) = lfusecat xss"
proof (induct xss)
case adm
then show ?case by simp
next
case LNil
then show ?case by simp
next
case (LCons xs xss)
then show ?case by (simp add: lappend_lnull1 lfuse_def)
qed

lemma lprefix_lfusecatI:
  "lprefix xss  yss \<Longrightarrow>lprefix  (lfusecat xss) (lfusecat yss)"
by (meson mcont_lfusecat mcont_monoD)

lemma lset_lltl_llength: 
 "(\<forall> xs \<in> lset xss. llength xs \<le> 1) \<longleftrightarrow> (\<forall> xs \<in> lset (lltl xss). lnull xs)" 
unfolding lltl_def
by (auto simp add: ltl_ldrop_one)


lemma lset_lltl_llength_var: 
 " (\<forall> xs \<in> lset xss. llength xs \<le> 1) \<longleftrightarrow> lset(lltl xss) \<subseteq> {LNil}"
using lset_lltl_llength 
by (metis all_not_in_conv insert_iff lnull_def subsetI subset_singletonD)

lemma lset_lltl_llength_var2: 
" (\<forall> xs \<in> lset xss. llength xs > 1) \<Longrightarrow> lset (lltl xss) \<inter> {LNil} = {} " 
unfolding lltl_def
by auto
   (metis co.enat.exhaust_sel epred_llength less_numeral_extra(4) llength_LNil not_one_less_zero 
    one_eSuc)


lemma lfusecat_all_empty_or_LNil_a: 
 assumes "lset(lltl xss) \<subseteq> {LNil}" 
 shows   "llength (lfusecat xss) \<le> 1" 
using assms
proof (induction xss)
case adm
then show ?case unfolding lltl_def by simp
next
case LNil
then show ?case by simp
next
case (LCons xs xss)
then show ?case 
  unfolding lltl_def 
   by simp
      (metis LCons.prems lfuse_llength_atmost_one  llist.set_intros(1)  lset_lltl_llength_var)
qed

lemma lfusecat_all_empty_or_LNil_b: 
 assumes "llength (lfusecat xss) \<le> 1" 
 shows   "lset(lltl xss) \<subseteq> {LNil}" 
using assms
proof (induction xss)
case adm
then show ?case unfolding lltl_def  by simp
next
case LNil
then show ?case unfolding lltl_def by simp
next
case (LCons xs xss)
then show ?case 
unfolding lltl_def
by (simp add: lfuse_llength_atmost_one ltl_ldrop_one)
qed

lemma lfusecat_all_empty_or_LNil:
 shows " llength (lfusecat xss) \<le> 1 \<longleftrightarrow> lset(lltl xss) \<subseteq> {LNil}" 
using lfusecat_all_empty_or_LNil_a lfusecat_all_empty_or_LNil_b by blast


lemma lfusecat_not_lnull:
 " \<not>lnull (lfusecat xss) \<longleftrightarrow> \<not>lnull xss \<and> (\<exists>xs \<in> lset ( xss). \<not>lnull( xs)) " 
by (metis lconcat_eq_LNil lfusecat_LNil lfusecat_eq_LNil lnull_def lnull_lconcat mem_Collect_eq 
    subsetD subsetI)

lemma lset_eq_forall_lnull: 
 " lnull xss \<or> (\<forall>xs \<in> lset ( xss). lnull( xs)) \<longleftrightarrow> lset xss \<subseteq> {LNil} " 
by (auto simp add: lset_lnull)

lemma lfusecat_not_lnull_var: 
 assumes "\<not>lnull xss"
         "\<forall>xs \<in> lset xss. \<not>lnull xs" 
 shows   "\<not>lnull(lfusecat xss) " 
using assms 
using lfusecat_not_lnull llist.set_sel(1) by blast
 
lemma lfirst_lfusecat_lfirst:
 assumes "\<not>lnull xss"
         " \<not>lnull (lfirst xss)" 
  shows " lfirst(lfusecat xss) = lfirst(lfirst xss) " 
proof (cases xss)
case LNil
then show ?thesis 
using assms(1) llist.disc(1) by blast
next
case (LCons x21 x22)
then show ?thesis 
by (metis assms(2) eq_LConsD lfirst_def lfirst_lfuse_1  lfusecat_LCons)
qed 

lemma lfirst_lfusecat:
 assumes "\<not> lnull xs" 
  shows " lfirst(lfusecat (LCons xs xss)) = lfirst xs "
using assms by (simp add: lfirst_def)


lemma ltl_lfusecat :
 assumes "\<not>lnull xss"
         " \<not>lnull (lfirst xss)" 
 shows   "ltl(lfusecat xss) = lappend (ltl (lhd xss)) (ltl (lfusecat (ltl xss))) "
using assms
unfolding lfirst_def
by (metis lappend_lnull2 lfusecat_LCons lhd_LCons_ltl ltl_lfuse)
 

lemma lfusecat_lappend: 
assumes "lfinite xss"     
shows "lfusecat (lappend xss yss) = lfuse (lfusecat xss) (lfusecat yss) " 
using assms
proof (induct xss arbitrary: yss)
case lfinite_LNil
then show ?case by auto
next
case (lfinite_LConsI xss xs)
then show ?case 
 proof -
   have 1: "lfusecat (lappend (LCons xs xss) yss) = lfuse xs (lfusecat (lappend xss yss))"
    by simp 
   have 2: "lfinite xss"
    by (simp add: lfinite_LConsI.hyps(1)) 
   have 3: "(lfusecat (lappend xss yss)) = lfuse (lfusecat xss) (lfusecat yss)" 
       by (simp add: lfinite_LConsI.hyps(2))
   have 4: "lfuse xs (lfuse (lfusecat xss) (lfusecat yss)) = 
            lfuse (lfusecat (LCons xs xss)) (lfusecat yss)" 
        by (metis lappend_lnull1 lfuse_LNil_2 lfuse_assoc lfuse_def lfusecat_LCons)
   show ?thesis 
   by (simp add: "3" "4")
 qed
qed

lemma lfusecat_split: 
shows "lfusecat xss = lfuse (lfusecat (ltake n xss)) (lfusecat (ldrop n xss)) " 
proof -
 have 1: " xss = lappend (ltake n xss) (ldrop n xss) "
   by (simp add: lappend_ltake_ldrop)
 show ?thesis using 1 lfusecat_lappend[of "(ltake n xss)" "(ldrop n xss)"] by force 
qed
 
lemma lfuse_lfinite:  
 shows   "lfinite (lfuse xs ys) \<longleftrightarrow> lfinite xs \<and> lfinite ys" 
by (metis lfinite_lappend lfinite_ltl lnull_imp_lfinite ltl_lfuse)

lemma lfusecat_lfinite_a:
 assumes "lfinite xss"        
         "\<forall>xs \<in> lset xss. lfinite xs " 
 shows " lfinite (lfusecat xss) " 
using assms
proof (induct xss)
case lfinite_LNil
then show ?case by auto
next
case (lfinite_LConsI xss xs)
then show ?case 
  proof (cases "xss=LNil")
  case True
  then show ?thesis by (simp add: lfinite_LConsI.prems)
  next
  case False
  then show ?thesis 
      proof -
    have 1: "lfinite xs"
      by (simp add: lfinite_LConsI.prems)
    have 5: "\<forall>xs \<in> lset xss. lfinite xs " 
     by (simp add: lfinite_LConsI.prems)
    have 6: "lfinite (lfusecat xss)"
      using "5" lfinite_LConsI.hyps(2) by blast
    have 7: "lfinite (lfuse xs (lfusecat xss))"
      by (simp add: "1" "6" lfuse_lfinite) 
    show ?thesis by (simp add: "7")
  qed
 qed
qed

lemma lfusecat_repeat_LNil [simp]: 
  "lfusecat (repeat LNil) = LNil"
by (simp add: lfusecat_eq_LNil)

lemma llastfirst_lfusecat_llast:
 assumes "lfinite xss" 
         "\<not>lnull xss"
         "\<forall>xs \<in> lset xss. \<not>lnull xs" 
         "llastlfirst xss"
         "\<forall>xs \<in> lset xss. lfinite xs " 
 shows   " llast(lfusecat xss) = llast(llast xss)"
using assms
proof (induction xss)
case lfinite_LNil
then show ?case
using llist.disc(1) by blast
next
case (lfinite_LConsI xss xs)
then show ?case 
  proof (cases "xss = LNil" )
  case True
  then show ?thesis 
  by simp
  next
  case False
  then show ?thesis 
    proof -
     have 1: "llastlfirst xss"
       using False lfinite_LConsI.prems(3) llastlfirst_LCons llist.collapse(1) by blast 
     have 2: "\<forall>xs \<in> lset xss. lfinite xs " 
       by (simp add: lfinite_LConsI.prems(4))
     have 3: "\<forall>xs \<in> lset xss. \<not>lnull xs" 
        using lfinite_LConsI.prems(2) by auto
     have 4: "lfinite xss"
       by (simp add: lfinite_LConsI.hyps) 
     have 5: "llast (lfusecat xss) = llast (llast xss)"
       using "1" "2" "3" False lfinite_LConsI.IH llist.collapse(1) by blast
     have 6: "llast (llast (LCons xs xss)) = llast (llast xss)"
        by (metis False llast_LCons llist.collapse(1)) 
     have 7: "(lfusecat (LCons xs xss)) = lfuse xs (lfusecat xss)"
       by simp
     have 8: "\<not> lnull xs" 
        by (simp add: lfinite_LConsI.prems(2))
     have 9: "lfinite xs" 
        by (simp add: lfinite_LConsI.prems(4))
     have 10: "\<not> lnull (lfusecat xss)"
        using "3" False lfusecat_not_lnull_var llist.collapse(1) by blast
     have 11: "lfinite (lfusecat xss)" 
        using "2" "4" lfusecat_lfinite_a by blast
     have 111: "lfirst (lfusecat xss) = lfirst (lfirst xss)" 
       by (metis "3" False lfirst_def lfirst_lfusecat_lfirst llist.collapse(1) llist.set_sel(1))
     have 12: "llast xs = lfirst (lfusecat xss)" 
       using "10" "111" lfinite_LConsI.prems(3) lfusecat_not_lnull llastlfirst_LCons by auto
     have 13: "llast (lfuse xs (lfusecat xss)) = llast (lfusecat xss)"  
         using llast_lfuse[of xs "(lfusecat xss)" ] 8 9 10 11 12 by blast
     have 14: "llast (lfusecat (LCons xs xss)) = llast (lfusecat xss)"
          by (simp add: "13")
     show ?thesis 
     using "13" "5" "6" by auto
   qed
  qed
qed


lemma lfusecat_llength_lNil:
 "llength (lfusecat LNil) = 0 "
by simp

lemma lfusecat_llength_lfinite:
 assumes "lfinite xss"
         "\<not>lnull xss"
         "\<forall>xs \<in> lset xss. \<not> lnull xs"
         "\<forall>xs \<in> lset xss. lfinite xs" 
         "llastlfirst xss" 
 shows  " llength(lfusecat xss) = 
          eSuc(\<Sum> i = 0 .. (the_enat(epred(llength xss))) . epred(llength (lnth xss i)))" 
using assms
proof (induction xss)
case lfinite_LNil
then show ?case by simp
next
case (lfinite_LConsI xss xs)
then show ?case 
  proof (cases "xss = LNil")
  case True
  then show ?thesis 
    proof -
     have 1: "lfusecat (LCons xs xss) = xs"
        by (simp add: True) 
     have 2: "llength (lfusecat (LCons xs xss) ) = llength xs"
       using "1" by auto 
     have 4: "(\<Sum> i = 0 .. (the_enat(epred(llength(LCons xs xss)))). 
                  epred (llength (lnth (LCons xs xss) i))) =
                      epred(llength (lnth (LCons xs xss) 0))" 
       using True  by simp 
     have 5: "eSuc(epred(llength (lnth (LCons xs xss) 0))) = llength xs"
       by (simp add: lfinite_LConsI.prems(2)) 
     show ?thesis 
     using "2" "4" "5" by presburger
    qed
  next
  case False
  then show ?thesis 
   proof - 
   have 6: "(lfusecat (LCons xs xss)) = lfuse xs (lfusecat xss)"
      by simp
   have 7: "llastlfirst (LCons xs xss)" 
     by (simp add: lfinite_LConsI.prems(4)) 
   have 71: "lfirst (lfusecat xss) = lfirst(lfirst xss)"
     by (metis False insert_iff lfinite_LConsI.prems(2) lfirst_def lfirst_lfusecat_lfirst 
         llist.collapse(1) llist.set_sel(1) llist.simps(19))
   have 8: "llast xs = lfirst (lfusecat xss)" 
     using "7" "71" False llastlfirst_LCons llist.collapse(1) by auto
   have 9: "llength (lfuse xs (lfusecat xss)) = llength xs + epred(llength(lfusecat xss)) "
     by (simp add: lfinite_LConsI.prems(2) lfuse_llength) 
   have 10: "(llength(lfusecat xss)) = 
             eSuc(\<Sum> i = 0 .. (the_enat(epred (llength xss))). epred(llength (lnth xss i)))"
      using "7" False lfinite_LConsI.IH lfinite_LConsI.prems(2) lfinite_LConsI.prems(3) llastlfirst_LCons 
      llist.collapse(1) by auto 
   have 11: "(the_enat(epred(llength (LCons xs xss)))) = (1+the_enat(epred(llength xss)))"
      using False 
      by simp
         (metis False co.enat.sel(2) eSuc_enat lfinite.cases lfinite_LConsI.hyps lfinite_llength_enat 
          llength_LCons  the_enat.simps)
   have 12: "(\<Sum>i = 0 .. (the_enat(epred(llength (LCons xs xss)))). epred (llength (lnth (LCons xs xss) i))) =
             epred(llength(lnth (LCons xs xss) 0)) +
             (\<Sum>i = 1 ..(1+the_enat(epred(llength xss))). epred (llength (lnth (LCons xs xss) i)))" 
      using 11  
      by (simp add: sum.atLeast_Suc_atMost)
   have 13: "epred(llength(lnth (LCons xs xss) 0)) = epred (llength xs)"
     by simp
   have 14: "(\<Sum>i = 1 ..(1+the_enat(epred(llength xss))). epred (llength (lnth (LCons xs xss) i))) =
             (\<Sum>i = 0 ..(the_enat(epred(llength xss))). epred (llength (lnth (LCons xs xss) (Suc i))))" 
       using sum.shift_bounds_cl_nat_ivl[of "\<lambda>k. epred (llength (lnth (LCons xs xss) k))" 0 1 
             "(the_enat(epred(llength xss)))"]
        by simp
   have 15: "(\<Sum>i = 0 ..(the_enat(epred(llength xss))). epred (llength (lnth (LCons xs xss) (Suc i)))) =
             (\<Sum>i = 0 ..(the_enat(epred(llength xss))). epred (llength (lnth (xss) (i))))" 
      by auto
   have 16: "epred (llength xs) + (\<Sum>i = 0 ..(the_enat(epred(llength xss))). epred (llength (lnth (xss) (i)))) =
             (\<Sum>i = 0 .. (the_enat(epred(llength (LCons xs xss)))). epred (llength (lnth (LCons xs xss) i)))"
       using "12" "13" "14" "15" by presburger 
   have 17: "llength xs + epred(llength(lfusecat xss)) = 
             eSuc(epred (llength xs) + (\<Sum>i = 0 ..(the_enat(epred(llength xss))). epred (llength (lnth (xss) (i)))))" 
         by (metis "10" eSuc_epred eSuc_plus epred_eSuc lfinite_LConsI.prems(2) llength_eq_0 lset_intros(1))
   show ?thesis
   using "16" "17" "6" "9" by presburger
  qed
 qed     
qed

lemma llastlfirst_ltake: 
 assumes "n\<le>llength xss"
         "llastlfirst xss"
 shows   "llastlfirst (ltake (enat n) xss)" 
using assms
proof (induction n arbitrary: xss)
case 0
then show ?case 
by (simp add: llastlfirst_def)
next
case (Suc n)
then show ?case 
 unfolding llastlfirst_def 
  by auto
    (metis Suc_ile_eq enat_ord_simps(2) less_Suc_eq lnth_ltake order_le_less_trans)
qed

lemma lfusecat_ltake:
 assumes"\<not> lnull xss"  
        " n \<le> llength xss" 
        "\<forall> xs \<in> lset xss. \<not> lnull xs"
        "\<forall> xs \<in> lset xss. lfinite xs" 
        "llastlfirst xss" 
  shows " lfusecat (ltake (enat n) xss) =
          ltake (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) 
                (lfusecat xss) " 
proof -
 have 1: "lfinite (ltake (enat n) xss)" 
     using enat_ord_code(4) lfinite_ltake by blast
 have 2: "llength (ltake (enat n) xss) = n" 
    by (simp add: assms(2))
 have 3: "(the_enat(epred(llength (ltake (enat n) xss)))) = n-1"
     using "2" epred_enat the_enat.simps by presburger 
 have 4: "\<forall> xs \<in> lset (ltake (enat n) xss). \<not> lnull xs"
     by (meson assms(3) lset_ltake subsetD)
 have 5: "\<forall> xs \<in> lset (ltake (enat n) xss). lfinite xs" 
     by (meson assms(4) lset_ltake subsetD)
 have 6: "llastlfirst (ltake (enat n) xss)"
    by (simp add: assms(2) assms(5) llastlfirst_ltake) 
 have 7: " \<And> j. 0< n \<and> j < n-1 \<longrightarrow>  
                epred(llength (lnth (ltake (enat n) xss) j)) =
                epred(llength (lnth xss j)) " 
      by (metis diff_le_self dual_order.strict_trans1 enat_ord_simps(2) lnth_ltake) 
 have 71: "llength (lfusecat (ltake (enat n) xss)) =
          (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth (ltake (enat n) xss) i))))"
       using lfusecat_llength_lfinite[of "(ltake (enat n) xss)" ]
       by (metis "1" "2" "3" "4" "5" "6" lfusecat_LNil llength_LNil llist.collapse(1) ltake_0 
       the_enat.simps zero_enat_def) 
 have 8: "(if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth (ltake (enat n) xss) i)))) =
          (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))"
   using 7 atLeastAtMost_iff[of _ 0 "n-1"] sum.cong[of "{0..n-1}" 
    "{0..n-1}" "\<lambda>i. epred(llength (lnth (ltake (enat n) xss) i)) " 
           "\<lambda>i. epred(llength (lnth xss i))" ]  
   by (simp add: Suc_ile_eq dual_order.refl lnth_ltake) 
 have 9: "llength (lfusecat (ltake (enat n) xss)) \<le> llength (lfusecat xss)"
     by (metis  lfuse_llength_le_a lfusecat_split)
 have 10: "llength (ltake (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) 
                (lfusecat xss)) =
             (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))   "
    using 71 "8" "9" by auto 
 have 11: "ltake (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
                 (lfuse (lfusecat (ltake n xss)) (lfusecat (ldrop n xss))) = 
                (lfusecat (ltake n xss))"
     using 71 8 ltake_lfuse[of "(lfusecat (ltake (enat n) xss))" "(lfusecat (ldrop n xss))"]
        by presburger 
 have 12: "(lfuse (lfusecat (ltake n xss)) (lfusecat (ldrop n xss))) = lfusecat xss"
    by (metis  lfusecat_split) 
 show ?thesis 
 using "11" "12" by auto
qed 

lemma lfusecat_ldrop:
 assumes"\<not> lnull xss"  
        " n < llength xss" 
        "\<forall> xs \<in> lset xss. \<not> lnull xs"
        "\<forall> xs \<in> lset xss. lfinite xs" 
        "llastlfirst xss" 
  shows " lfusecat (ldrop (enat n) xss) =
          ldrop (if n = 0 then 0 else (\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
                (lfusecat xss) " 
proof -
 have 1: "lfinite (ltake (enat n) xss)" 
     using enat_ord_code(4) lfinite_ltake by blast
 have 2: "llength (ltake (enat n) xss) = n" 
    by (simp add: assms(2))
 have 3: "(the_enat(epred(llength (ltake (enat n) xss)))) = n-1"
     using "2" epred_enat the_enat.simps by presburger 
 have 4: "\<forall> xs \<in> lset (ltake (enat n) xss). \<not> lnull xs"
     by (meson assms(3) lset_ltake subsetD)
 have 5: "\<forall> xs \<in> lset (ltake (enat n) xss). lfinite xs" 
     by (meson assms(4) lset_ltake subsetD)
 have 6: "llastlfirst (ltake (enat n) xss)"
    by (simp add: assms(2) assms(5) llastlfirst_ltake order.strict_implies_order)
 have 7: " \<And> j. 0< n \<and> j < n-1 \<longrightarrow>  
                epred(llength (lnth (ltake (enat n) xss) j)) =
                epred(llength (lnth xss j)) " 
      by (metis diff_le_self dual_order.strict_trans1 enat_ord_simps(2) lnth_ltake)
  have 71: "llength (lfusecat (ltake (enat n) xss)) =
          (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth (ltake (enat n) xss) i))))"
       using lfusecat_llength_lfinite[of "(ltake (enat n) xss)" ]
       by (metis "1" "2" "3" "4" "5" "6" lfusecat_LNil llength_LNil llist.collapse(1) ltake_0 
           the_enat.simps zero_enat_def) 
 have 8: "(if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth (ltake (enat n) xss) i)))) =
          (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))"
  using 7 atLeastAtMost_iff[of _ 0 "n-1"] sum.cong[of "{0..n-1}" "{0..n-1}" 
           "\<lambda>i. epred(llength (lnth (ltake (enat n) xss) i)) " 
           "\<lambda>i. epred(llength (lnth xss i))" ]  
   by (simp add: Suc_ile_eq dual_order.refl lnth_ltake) 
 have 9: "llength (lfusecat (ltake (enat n) xss)) \<le> llength (lfusecat xss)"
       by (metis  lfuse_llength_le_a lfusecat_split)
 have 10: "llength (ltake (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) 
                (lfusecat xss)) =
             (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))   "
    using 71 "8" "9" by auto 
 have 11: "ltake (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
                 (lfuse (lfusecat (ltake n xss)) (lfusecat (ldrop n xss))) = 
                (lfusecat (ltake n xss))"
     using 71 8 ltake_lfuse[of "(lfusecat (ltake (enat n) xss))" "(lfusecat (ldrop n xss))"]
        by presburger 
 have 12: "ldrop 0 (lfuse (lfusecat (ltake 0 xss)) (lfusecat (ldrop 0 xss))) =
              (lfusecat (ldrop 0 xss))"
      by simp
 have 13: "0< n \<Longrightarrow>\<not> lnull (lfusecat (ltake (enat (n::nat)) (xss::'a llist llist)))"
    using 71 "8" by force
 have 14: "0< n \<Longrightarrow> \<not> lnull (lfusecat (ldrop (enat n) xss))"
   by (meson assms(2) assms(3) in_lset_ldropD leD lfusecat_not_lnull_var lnull_ldrop) 
 have 15: "0<n \<Longrightarrow> llast (lfusecat (ltake (enat n) xss)) = lfirst (lfusecat (ldrop (enat n) xss))"
   proof -
     assume a: "0<n" 
    have 151: "llast (lfusecat (ltake (enat n) xss)) = llast(llast  (ltake (enat n) xss)) "
      using "1" "13" "4" "5" "6" a lfusecat_not_lnull llastfirst_lfusecat_llast by blast
    have 152: "lfirst (lfusecat (ldrop (enat n) xss)) = lfirst (lfirst (ldrop (enat n) xss))" 
       by (metis assms(2) assms(3) in_lset_conv_lnth ldrop_enat leD lfirst_def lfirst_lfusecat_lfirst 
           lhd_ldropn lnull_ldrop)
    have 153: "llast(llast  (ltake (enat n) xss)) = lfirst (lfirst (ldrop (enat n) xss))"    
      using assms a
      by (metis "1" "13" lappend_ltake_ldrop leD lfusecat_not_lnull llastlfirst_lappend_lfinite
           lnull_ldrop)       
    show ?thesis 
    by (simp add: "151" "152" "153")
  qed
 have 16: "0<n \<Longrightarrow> ldrop (epred (eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))))
                 (lfuse (lfusecat (ltake n xss)) (lfusecat (ldrop n xss))) = 
                (lfusecat (ldrop n xss)) " 
   using 71 8 13 14 15 ldrop_lfuse_a[of "(lfusecat (ltake (enat n) xss))" "(lfusecat (ldrop n xss))"]
   by (metis "1" "5" less_numeral_extra(3) lfusecat_lfinite_a)
 have 17: "ldrop (if n =0 then 0 else (\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
                 (lfuse (lfusecat (ltake n xss)) (lfusecat (ldrop n xss))) = 
                (lfusecat (ldrop n xss))" 
   using "11" "16" by auto
 show ?thesis 
 by (metis "17"  lfusecat_split)
qed     

 

lemma lfusecat_eq_LCons_conv:
shows  "lfusecat xss = LCons x xs \<longleftrightarrow>
  (\<exists>xs' xss' xss''. xss = lappend (llist_of xss') (LCons (LCons x xs') xss'') \<and>
                    xs = lappend xs' (ltl (lfusecat xss'')) \<and> set xss' \<subseteq> {xs. lnull xs})"
  (is "?lhs \<longleftrightarrow> ?rhs")
proof
  assume "?rhs"
  then obtain xs' xss' xss''
    where "xss = lappend (llist_of xss') (LCons (LCons x xs') xss'')"
    and "xs = ( (lappend xs' (ltl (lfusecat xss''))) )"
    and "set xss' \<subseteq> {xs. lnull xs}" by blast
  moreover from \<open>set xss' \<subseteq> {xs. lnull xs}\<close>
  have "lnull (lfusecat (llist_of xss'))"    
  by (metis lfusecat_not_lnull lset_llist_of mem_Collect_eq subset_eq)
  have 1: "lfusecat xss = lfuse (lfusecat (llist_of xss')) (lfusecat (LCons (LCons x xs') xss'')) "
    by (simp add: calculation(1) lfusecat_lappend) 
  have 2: "lfuse (lfusecat (llist_of xss')) (lfusecat (LCons (LCons x xs') xss'')) =
           (lfusecat (LCons (LCons x xs') xss''))" 
     by (simp add: \<open>lnull (lfusecat (llist_of (xss'::'a llist list)))\<close> lfuse_conv_lnull llist.expand)
  have 3: "(lfusecat (LCons (LCons x xs') xss'')) = lfuse (LCons x xs') (lfusecat xss'')"
    by simp 
  have 4: "lfuse (LCons x xs') (lfusecat xss'') = 
         (LCons x  (lappend xs' (ltl (lfusecat xss''))))"
    unfolding lfuse_def
   using llist.collapse(1) by force
  ultimately show ?lhs 
  using "1" "2" by auto
next
  assume "?lhs"
  hence "\<not> lnull (lfusecat xss)" by simp
  hence "\<not> lset xss \<subseteq> {xs. lnull xs}" using lfusecat_eq_LNil lnull_def by blast
  hence "\<not> lnull (lfilter (\<lambda>xs. \<not> lnull xs) xss)" by(auto)
  then obtain y ys where yss: "lfilter (\<lambda>xs. \<not> lnull xs) xss = LCons y ys"
    unfolding not_lnull_conv by auto
  from lfilter_eq_LConsD[OF this]
  obtain us vs where xss: "xss = lappend us (LCons y vs)"
    and "lfinite us"
    and "lset us \<subseteq> {xs. lnull xs}" "\<not> lnull y"
    and ys: "ys = lfilter (\<lambda>xs. \<not> lnull xs) vs" by blast
  from \<open>lfinite us\<close> obtain us' where [simp]: "us = llist_of us'"
    unfolding lfinite_eq_range_llist_of by blast
  from \<open>lset us \<subseteq> {xs. lnull xs}\<close> have us: "lnull (lfusecat us)"
    using lfusecat_eq_LNil lnull_def by blast
  from \<open>\<not> lnull y\<close> obtain y' ys' where y: "y = LCons y' ys'"
    unfolding not_lnull_conv by blast
  from \<open>?lhs\<close> us have [simp]: "y' = x" 
        "xs = (lappend ys' (ltl (lfusecat vs)))"
    unfolding xss y 
    by (metis \<open>\<not> lnull (y::'a llist)\<close> \<open>lfinite (us::'a llist llist)\<close> eq_LConsD lfusecat_LCons 
        lfusecat_lappend lhd_lfuse y)
       (metis \<open>\<not> lnull (y::'a llist)\<close> \<open>lfusecat (xss::'a llist llist) = LCons (x::'a) (xs::'a llist)\<close> 
        eq_LConsD lappend_lnull2 lfusecat_LCons lfusecat_lfilter_neq_LNil ltl_lfuse y ys yss)
  from \<open>lset us \<subseteq> {xs. lnull xs}\<close> ys show ?rhs unfolding xss y by simp blast
qed


lemma not_lnull_expand:
 " \<not> lnull xs \<longleftrightarrow> (\<exists> b. xs = (LCons b LNil)) \<or> (\<exists> b xs'. xs = (LCons b xs') \<and> \<not> lnull xs')"
by (metis lhd_LCons_ltl llist.disc(1) llist.disc(2) llist.expand)


lemma is_LMore_llength:
 " (\<exists> b xs'. xs = (LCons b xs') \<and> \<not> lnull xs') \<longleftrightarrow> 1 < llength xs"
by (metis dual_order.strict_implies_not_eq gr_implies_not_zero lhd_LCons_ltl llength_LCons
    llength_eq_0 lstrict_prefix_code(2) lstrict_prefix_code(4) lstrict_prefix_llength_less one_eSuc )


lemma is_LEmpty_llength:
 " (\<exists> b. xs = (LCons b LNil)) \<longleftrightarrow> llength xs =1"
by (metis epred_llength llength_LCons llength_eq_0 llist.disc(1) ltl_simps(2) not_lnull_expand 
     one_eSuc)

lemma lfusecat_all_llength_one_lfuse: 
 assumes "\<forall> ys \<in> lset (llist_of xss'). llength ys = 1" 
 shows   "lfuse (lfusecat (llist_of xss')) ys = 
          lfuse (if \<not> lnull (llist_of xss') then (LCons (lfirst (lfirst (llist_of xss'))) LNil) else LNil) ys
         " 
proof -
 have 1: "(lfusecat (llist_of xss')) = 
          (if \<not> lnull (llist_of xss') then (LCons (lfirst (lfirst (llist_of xss'))) LNil) else LNil)" 
    using assms
   proof (induction xss')
   case Nil
   then show ?case by simp
   next
   case (Cons as xss')
   then show ?case 
      proof (cases "xss'=Nil" )
      case True
      then show ?thesis using Cons 
       by simp (metis is_LEmpty_llength lfirst_def lhd_LCons)
      next
      case False
      then show ?thesis using Cons 
          by simp 
             (metis is_LEmpty_llength lfirst_def lhd_LCons llength_lnull zero_one_enat_neq(1))
      qed
    qed
   show ?thesis 
   using "1"  by presburger
qed

lemma lfusecat_eq_LCons_conv_all_lset_singleton: 
 assumes "\<forall> ys \<in> lset (llist_of xss). llength ys = 1" 
         "\<not> lnull (llist_of xss) " 
         "llastlfirst (llist_of xss)" 
shows " lset (llist_of xss) = {lfirst (llist_of xss) } " 
using assms
proof (induction xss)
case Nil
then show ?case by simp
next
case (Cons as xss)
then show ?case 
  proof -
   have 1: "xss= [] \<or> as \<in> lset (llist_of xss)" 
     by (metis Cons.prems(1) Cons.prems(3) List.set_insert insert_iff is_LEmpty_llength 
         le_numeral_extra(4) lfirst_def lhd_LCons_ltl llast_singleton llastlfirst_LCons llist.set_sel(1)
          llist_of.simps(2) lnull_llist_of lset_llist_of ltl_simps(2) not_in_set_insert not_lnull_llength)
   have 2: "xss= [] \<Longrightarrow> ?thesis"
     by (simp add: lfirst_def) 
   have 3: "as \<in> lset (llist_of xss) \<Longrightarrow> ?thesis"
   by (metis "2" Cons.IH Cons.prems(1) Cons.prems(3) insert_absorb2 insert_iff lfirst_def lhd_LCons 
       llastlfirst_LCons llist.simps(19) llist_of.simps(2) lnull_llist_of singletonD)
   show ?thesis
   using "1" "2" "3" by linarith
  qed
qed


lemma lfusecat_eq_LCons_conv_all_the_same: 
 assumes "\<forall> ys \<in> lset (llist_of xss'). llength ys = 1" 
         "\<not> lnull (llist_of xss') " 
         "llastlfirst (llist_of xss')" 
         "(llast (llist_of xss')) = (LCons x LNil)" 
         " ys \<in> lset (llist_of xss') " 
 shows   " ys = (LCons x LNil) " 
proof -
 have 1: "lset (llist_of xss') = { lfirst (llist_of xss')} "
   using assms lfusecat_eq_LCons_conv_all_lset_singleton by blast
 have 2: "llast (llist_of xss') = lnth (llist_of xss') (the_enat(epred(llength (llist_of xss'))))"
     by (metis assms(2) co.enat.exhaust_sel enat_the_enat ile_eSuc infinity_ileE llast_conv_lnth 
         llength_eq_0 llength_llist_of)
 have 3: "lnth (llist_of xss') (the_enat(epred(llength (llist_of xss')))) \<in> lset (llist_of xss')"
   by (metis "2" assms(2) co.enat.exhaust_sel ile_eSuc in_lset_lappend_iff infinity_ileE 
       lappend_lbutlast_llast_id llength_eq_0 llength_eq_infty_conv_lfinite llength_lbutlast 
       llength_llist_of lset_intros(1)) 
 have 4: "(LCons x LNil) \<in> lset (llist_of xss') "
    using "2" "3" assms(4) by force
 show ?thesis 
 using "1" "4" assms(5) by auto
qed    
  
lemma lfusecat_eq_LCons_conv_all_the_same_a: 
 assumes "lfinite uss" 
         "\<forall> ys \<in> lset (uss). llength ys = 1" 
         "\<not> lnull (uss) " 
         "llastlfirst (uss)" 
         "(llast (uss)) = (LCons x LNil)" 
         " ys \<in> lset (uss) " 
 shows   " ys = (LCons x LNil) " 
using assms lfusecat_eq_LCons_conv_all_the_same llist_of_list_of
by (metis (full_types))



lemma lfusecat_eq_LCons_conv_alt:
assumes "llastlfirst xss" 
        "\<forall> ys \<in> lset xss. \<not> lnull ys" 
       "\<not> lnull xs" 
shows  "lfusecat xss = LCons x xs \<longleftrightarrow>
  (\<exists>xs' xss' xss''. xss = lappend (llist_of xss') (LCons (LCons x xs') xss'') \<and> \<not>lnull xs' \<and>
                    xs = lfuse xs' (lfusecat xss'') \<and> set xss' \<subseteq> {xs. lnull (ltl xs)})"
  (is "?lhs \<longleftrightarrow> ?rhs")
proof
  assume "?rhs"
  then obtain xs' xss' xss''
    where "xss = lappend (llist_of xss') (LCons (LCons x xs') xss'')"  
    and "\<not>lnull xs' " 
    and "xs = (lfuse xs'  (lfusecat xss'')) "
    and "set xss' \<subseteq> {xs. lnull (ltl xs)}" by blast
  moreover from \<open>set xss' \<subseteq> {xs. lnull (ltl xs)}\<close>
  have 00: "llength (lfusecat (llist_of xss')) \<le>1"
       by (metis is_LMore_llength lfusecat_all_empty_or_LNil_a lset_llist_of lset_lltl_llength_var 
           ltl_simps(2) mem_Collect_eq not_le_imp_less subset_eq)  
  have 01: "\<forall> y \<in> lset (llist_of xss'). llength y = 1" 
     by (metis assms(2) calculation(1) calculation(4) dual_order.strict_iff_order is_LMore_llength 
         lset_lappend1 lset_llist_of ltl_simps(2) mem_Collect_eq not_lnull_llength subsetD)
  have 02: "lfinite (llist_of xss')" 
    using lfinite_llist_of by blast
  have 03: "llastlfirst (lappend (llist_of xss') (LCons (LCons x xs') xss''))" 
       using assms calculation(1) by blast
  have 04: "llastlfirst (llist_of xss')" 
     using assms(1) calculation(1) lfinite_llist_of llastlfirst_lappend_lfinite by blast
  have 05: "(if lnull (llist_of xss') \<or> lnull (LCons (LCons x xs') xss'') 
             then True 
             else llast (llast (llist_of xss')) = lfirst (lfirst (LCons (LCons x xs') xss'')))" 
     using 03 02 llastlfirst_lappend_lfinite[of "(llist_of xss')" "(LCons (LCons x xs') xss'')" ] 
     by blast 
  have 06: "\<not> lnull (LCons (LCons x xs') xss'')"
     by simp       
  have 07: "\<not> lnull (llist_of xss') \<Longrightarrow> 
             llast (llast (llist_of xss')) = lfirst (lfirst (LCons (LCons x xs') xss''))"
    using "05" "06" by presburger 
  have 08: "\<not> lnull (llist_of xss') \<Longrightarrow> (llast (llist_of xss')) = (LCons x LNil)" 
    using 07 lappend_lbutlast_llast_id[of "(llist_of xss')" ] 
    by (metis "01" "02" eq_LConsD in_lset_lappend_iff is_LEmpty_llength lbutlast_lfinite lfirst_def
         llast_singleton lset_intros(1))
  have 09: "\<not> lnull (llist_of xss') \<Longrightarrow> (\<forall> ys \<in> lset (llist_of xss'). ys = (LCons x LNil)) " 
     using lfusecat_eq_LCons_conv_all_the_same 
     by (metis "01" "04" "08")
  have 0: " (lfusecat (llist_of xss')) = LNil \<or> (lfusecat (llist_of xss')) = (LCons x LNil)"
     by (metis "00" "09" eq_LConsD is_LMore_llength lfirst_def lfirst_lfusecat lfusecat_LNil lhd_LCons_ltl 
         linorder_not_le llist.collapse(1) llist.set_sel(1))
  have 1: "lfusecat xss = lfuse (lfusecat (llist_of xss')) (lfusecat (LCons (LCons x xs') xss'')) "
    by (simp add: calculation(1) lfusecat_lappend) 
  have 2: "lfuse (lfusecat (llist_of xss')) (lfusecat (LCons (LCons x xs') xss'')) =
           (lfusecat (LCons (LCons x xs') xss''))" 
        using "0" by fastforce
  have 3: "(lfusecat (LCons (LCons x xs') xss'')) = lfuse (LCons x xs') (lfusecat xss'')"
    by simp 
  have 4: "lfuse (LCons x xs') (lfusecat xss'') = 
         (LCons x ( (lfuse xs' ( (lfusecat xss''))) 
         ))"
     unfolding lfuse_def 
     using calculation(2) by auto
  ultimately show ?lhs 
  using "1" "2" by auto
next
  assume "?lhs"
  hence "\<not> lnull (lfusecat xss)" by simp
  hence "\<not> lset xss \<subseteq> {xs. lnull (ltl xs)}"   
  by (metis \<open>lfusecat (xss::'a llist llist) = LCons (x::'a) (xs::'a llist)\<close> assms(3) 
      lfusecat_all_empty_or_LNil_a lnull_ldrop lset_lltl_llength_var ltl_ldrop_one ltl_simps(2) 
      mem_Collect_eq subsetD)
  hence "\<not> lnull (lfilter (\<lambda>xs. \<not> lnull (ltl xs)) xss)" by(auto)
  then obtain y ys where yss: "lfilter (\<lambda>xs. \<not> lnull (ltl xs)) xss = LCons y ys"
    unfolding not_lnull_conv by auto
  from lfilter_eq_LConsD[OF this]
  obtain us vs where xss: "xss = lappend us (LCons y vs)"
    and "lfinite us"
    and "lset us \<subseteq> {xs. lnull (ltl xs)}" "\<not> lnull (ltl y)"
    and ys: "ys = lfilter (\<lambda>xs. \<not> lnull (ltl xs)) vs"  using lnull_ltlI by blast
  from \<open>lfinite us\<close> obtain us' where [simp]: "us = llist_of us'"
    unfolding lfinite_eq_range_llist_of by blast
  from \<open>lset us \<subseteq> {xs. lnull (ltl xs)}\<close> have us: "llength (lfusecat us) \<le>1"
    using lfusecat_eq_LNil  
    by (metis lfusecat_all_empty_or_LNil_a lnull_ldrop lset_lltl_llength_var ltl_ldrop_one 
        mem_Collect_eq subset_eq)
  from \<open>\<not> lnull (ltl y)\<close> obtain y' ys' where y: "y = LCons y' ys'"
    unfolding not_lnull_conv 
    by (metis \<open>\<not> lnull (ltl y)\<close> lhd_LCons_ltl lnull_ltlI)
  have 100: "llastlfirst us"
     using \<open>lfinite us\<close> assms(1) llastlfirst_lappend_lfinite xss by blast 
  have 101: "\<not> lnull ys' "
     using \<open>\<not> lnull (ltl y)\<close> y by auto 
  have 102: "\<not> lnull (LCons y vs)"
     by simp 
  have 103: "\<not> lnull us \<Longrightarrow> llast (llast us) = lfirst (lfirst (LCons y vs))"
    using llastlfirst_lappend_lfinite[of us "(LCons y vs)"] 
    using assms(1) xss by auto
  have 104: "(\<forall> z \<in> lset us. llength z =1)"
     by (metis \<open>lset us \<subseteq> {xs. lnull (ltl xs)}\<close> assms(2) 
         dual_order.strict_iff_order eq_LConsD in_lset_lappend_iff is_LMore_llength mem_Collect_eq 
         not_lnull_llength subsetD xss)  
  have 1040: "\<not> lnull us \<Longrightarrow> llast us \<in> lset us"
     by simp
  have 1041: "\<not> lnull us \<Longrightarrow> llast us = LCons x LNil"
    using lappend_lbutlast_llast_id[of us] 100 104 1040
      lfusecat_eq_LCons_conv_all_the_same_a[of us x "llast us"]
      lfusecat_eq_LCons_conv_all_lset_singleton[of us' ] 
      by (metis  \<open>\<not> lnull (lfusecat xss)\<close> \<open>lfusecat xss = LCons x xs\<close> \<open>us = llist_of us'\<close> 
          eq_LConsD is_LEmpty_llength lfirst_def lfirst_lfusecat_lfirst  lfusecat_not_lnull 
          lhd_lappend lset_eq_empty not_lnull_lset_conv_a singletonD xss)  
  have 105: "\<not> lnull us \<Longrightarrow> (\<forall> z \<in> lset us. z= (LCons x LNil))"
   using lfusecat_eq_LCons_conv_all_the_same_a[of us x ]
   using "100" "104" "1041" \<open>lfinite us\<close> by blast 
  have 106: "\<not> lnull us \<Longrightarrow>  lfusecat us = (LCons x LNil)" 
     by (metis "100" "104" "1041" \<open>lfinite us\<close> dual_order.antisym is_LEmpty_llength lfinite_LConsI
         lfinite_LNil lfusecat_not_lnull_var llast_singleton llastfirst_lfusecat_llast llength_LNil
          lset_eq_empty not_lnull_llength not_lnull_lset_conv_a us zero_one_enat_neq(1))
  from \<open>?lhs\<close> us have [simp]: "y' = x" 
       by (metis "103" "1041" \<open>\<not> lnull (ltl y)\<close> eq_LConsD lappend_lnull1 lfirst_def lfirst_lfusecat
        llast_singleton lnull_ltlI xss y)
  from \<open>?lhs\<close> us have [simp]:  "xs = (  (lfuse ys' ( (lfusecat vs))) )"    
    using lfusecat_lappend[of us "(LCons y vs)" ]   lfusecat_LCons[of y vs] y xss
       "101" \<open>lfinite us\<close> \<open>lfusecat xss = LCons x xs\<close> 
       by (metis "103" "1041" "106" eq_LConsD lappend_code(1) lappend_lnull1 lbutlast_simps(2) lfirst_def
           lfuse_LCons_a lfuse_conv_lnull lfuse_lbutlast lnull_ltlI)
  from \<open>lset us \<subseteq> {xs. lnull (ltl xs)}\<close> ys show ?rhs unfolding xss y 
  using "101" by auto
qed


lemma lfusecat_eq_One_conv: 
 "lfusecat xss = LCons x LNil \<longleftrightarrow>
  (\<exists> xss' xss''. xss = lappend (llist_of xss') (LCons (LCons x LNil) xss'') \<and>
                    LNil =  (ltl (lfusecat xss'')) \<and> set xss' \<subseteq> {xs. lnull xs})"
by (metis lappend_eq_LNil_iff lfusecat_eq_LCons_conv)

lemma llength_lfusecat_eq_one_conv: 
 "llength (lfusecat xss) = 1  \<longleftrightarrow>
  (\<exists> x xss' xss''. xss = lappend (llist_of xss') (LCons (LCons x LNil) xss'') \<and>
                    LNil =  (ltl (lfusecat xss'')) \<and> set xss' \<subseteq> {xs. lnull xs}) "
by (metis is_LEmpty_llength lfusecat_eq_One_conv)


lemma lfusecat_lfinite_b:
 assumes "lfinite(lfusecat xss)"
         "\<forall>xs \<in> lset xss. \<not> lnull (ltl xs)"
         "\<forall>xs \<in> lset xss. lfinite xs" 
         "llastlfirst xss" 
 shows  "lfinite xss"
using assms
proof (induct zs\<equiv>"(lfusecat xss)" arbitrary: xss )
case lfinite_LNil
then show ?case  
by (metis lfusecat_not_lnull_var llist.disc(1) lnull_imp_lfinite lset_eq_empty ltl_simps(1) 
    not_lnull_lset_conv_a)
next
case (lfinite_LConsI xs x)
then show ?case 
     proof -
      have 1: "(\<exists>xs' xss' xss''. xss = lappend (llist_of xss') (LCons (LCons x xs') xss'') \<and>
                    xs = lappend xs' (ltl (lfusecat xss'')) \<and> set xss' \<subseteq> {xs. lnull xs})" 
         using lfinite_LConsI.hyps(3) lfusecat_eq_LCons_conv by fastforce
      obtain xs' xss' xss'' where 2: " xss = lappend (llist_of xss') (LCons (LCons x xs') xss'') \<and>
                    xs = lappend xs' (ltl (lfusecat xss'')) \<and> set xss' \<subseteq> {xs. lnull xs}"
        using 1 by blast
      have 3: "(llist_of xss') = LNil "
         by (metis "2" in_lset_lappend_iff lconcat_eq_LNil lfinite_LConsI.prems(1) llist.collapse(1)
             llist.set_sel(1) lnull_lconcat lset_eq_forall_lnull lset_llist_of ltl_simps(1))
      have 4: " xss = (LCons (LCons x xs') xss'')"
         by (simp add: "2" "3")    
      have 5: "llastlfirst xss" 
         by (simp add: lfinite_LConsI.prems(3))
      have 6: "\<not> lnull xs'" 
         using "4" lfinite_LConsI.prems(1) by force
      have 7: " lnull xss'' \<Longrightarrow> ?thesis" 
         by (simp add: "4")
      have 8: "\<not> lnull xss'' \<Longrightarrow> ?thesis" 
        proof -
          assume a: "\<not>lnull xss''"
          have 9: "llast (LCons x xs') = lfirst (lfirst (xss''))" 
           using a 6 "4" "5" llastlfirst_LCons by blast
          have 10: "\<forall>xs \<in> lset xss''. \<not> lnull (ltl xs)"
           by (simp add: "4" lfinite_LConsI.prems)
          have 11: "\<forall>xs \<in> lset xss''. lfinite xs" 
           by (simp add: "4" lfinite_LConsI.prems)
          have 12: "llastlfirst xss''" 
           using "4" "5" a by auto
          have 13: "lfinite(lfusecat xss'')" 
            using "2" lfinite_LConsI.hyps(1) by auto
          have 14: "xs = lappend (lbutlast xs')  (lfusecat xss'')"
            by (metis "10" "2" "6" "9" a lappend_lbutlast_llast_id lappend_snocL1_conv_LCons2 lfirst_def
                 lfirst_lfusecat_lfirst lfusecat_not_lnull_var lhd_LCons_ltl llast_LCons llist.disc(1) 
                llist.set_sel(1) lset_eq_empty ltl_simps(1) not_lnull_lset_conv_a)
          have 15: "xs= lfuse xs' (lfusecat xss'')"
            by (simp add: "2" "6" lappend_lnull2 lfuse_def)
          have 16: " xs = lfusecat (LCons xs' xss'')"  
            by (simp add: "15")
          have 17: "lnull (ltl xs') \<Longrightarrow> ?thesis " 
            by (metis "10" "11" "12" "14" "4" lappend_code(1) lbutlast.ctr(1) lfinite.lfinite_LConsI
                 lfinite_LConsI.hyps(2) llist.collapse(1))
          have 18: "\<not>lnull (ltl xs') \<Longrightarrow> (\<forall>xs \<in> lset (LCons xs' xss''). \<not> lnull (ltl xs))"
            by (simp add: "10")
          have 19: "\<forall>xs \<in> lset (LCons xs' xss''). lfinite xs" 
            by (metis "11" "2" insert_iff lappend_inf lfinite_LConsI.hyps(1) llist.simps(19))
          have 20: "llastlfirst (LCons xs' xss'')" 
            by (metis "12" "6" "9" a llast_LCons llastlfirst_LCons)
          have 21: "lfinite (LCons xs' xss'')"
            by (metis "16" "17" "18" "19" "20" "4" lfinite_LConsI.hyps(2) lfinite_code(2) )            
          show ?thesis 
          using "21" "4" by auto
        qed
     show ?thesis 
     using "7" "8" by blast
   qed
qed

lemma lfusecat_lfinite:
assumes "\<forall>xs \<in> lset xss. \<not> lnull (ltl xs)"
         "\<forall>xs \<in> lset xss. lfinite xs" 
         "llastlfirst xss" 
shows "lfinite (lfusecat xss) \<longleftrightarrow> lfinite xss " 
using assms lfusecat_lfinite_a lfusecat_lfinite_b by blast 

lemma ridx_lfusecat_ltake: 
 assumes "ridx R (lfusecat xss)"
         " n \<le> llength xss" 
 shows   " ridx R (lfusecat (ltake n xss))" 
using assms 
by (metis lfusecat_split ltake_all ltake_lfuse nle_le ridx_ltake_a)


lemma ridx_lfusecat_ldrop: 
 assumes "ridx R (lfusecat xss)"
         " (enat n) < llength xss" 
         "llastlfirst xss"
         "\<forall> xs \<in> lset xss. \<not> lnull xs"
         "\<forall> xs \<in> lset xss. lfinite xs" 
 shows   " ridx R (lfusecat (ldrop n xss))" 
proof -
 have 1: "(lfusecat (ldrop n xss)) = 
          ldrop (if n = 0 then 0 else (\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
                (lfusecat xss)"
     using assms gr_implies_not_zero lfusecat_ldrop llength_eq_0 by blast 
 have 2: " (if n = 0 then 0 else (\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) < llength (lfusecat xss)" 
    by (metis (no_types, lifting) "1" add.commute add.right_neutral assms(2) assms(4) in_lset_conv_lnth 
        lfusecat_not_lnull llist.set_sel(1) lnth_0_conv_lhd lnth_ldrop lnull_ldrop not_less_iff_gr_or_eq 
        order.order_iff_strict the_enat.simps zero_enat_def) 
 have 3: "\<exists> k. (enat k) = (if n = 0 then 0 else (\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))"
   proof (cases n)
   case 0
   then show ?thesis by (simp add: zero_enat_def)
   next
   case (Suc nat)
   then show ?thesis  by (metis (mono_tags, lifting) "2" enat_iless enat_less_imp_le leD)
   qed 
 have 4: "ridx R (ldrop (if n = 0 then 0 else (\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
                (lfusecat xss))"
   using ridx_ldrop[of R "(lfusecat xss)" ]
   using "2" assms(1) order.strict_implies_order by blast
 show ?thesis by (simp add: "1" "4")
qed   

lemma ridx_lfusecat_a:
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. \<not> lnull xs"
         "\<forall> xs \<in> lset xss. lfinite xs" 
         "ridx R (lfusecat xss)"
         "i < llength xss" 
 shows   "ridx R (lnth xss i) " 
proof -
 have 1: "lsub i i xss = (LCons (lnth xss i) LNil)"
    by (simp add: assms(5) lsub_same)
 have 2: "lsub i i xss  = ltake (eSuc (enat (i - i))) (ldrop (enat i) xss)" 
       by (simp add: lsub_def)
 have 3: "ltake (eSuc (enat (i - i))) (ldrop (enat i) xss) = ltake 1 (ldrop (enat i) xss)"
   by (simp add: eSuc_enat one_enat_def)
 have 4: "ridx R (lfusecat (ltake 1 (ldrop (enat i) xss)))"
    by (metis assms ltake_all nle_le ridx_lfusecat_ldrop ridx_lfusecat_ltake) 
 have 5: "(lfusecat (ltake 1 (ldrop (enat i) xss))) = (lnth xss i)"
     by (metis "1" "2" "3" lfuse_LNil_2 lfusecat_LCons lfusecat_LNil)
 show ?thesis 
 using "4" "5" by auto 
qed


lemma ridx_lfuse_lfinite: 
 assumes "lfinite l1" 
         "llast l1 = lfirst l2"
 shows   " ridx R (lfuse l1 l2) \<longleftrightarrow> ridx R l1 \<and> ridx R l2 " 
using assms  
proof (cases "\<not> lnull l1 \<and> \<not> lnull l2")
case True
then show ?thesis 
  proof (cases "\<not> lnull (ltl l2)")
  case True
  then show ?thesis 
    proof -
     have 1: "(lfuse l1 l2) = lappend l1 (ltl l2) " 
        unfolding lfuse_def using \<open>\<not> lnull l1 \<and> \<not> lnull l2\<close> by simp
     have 2: "ridx R (lappend l1 (ltl l2)) = (ridx R l1 \<and> ( R (llast l1) (lhd (ltl l2))) \<and> ridx R (ltl l2))" 
       using assms ridx_lappend_lfinite[of  l1 R "ltl l2" ]  True \<open>\<not> lnull l1 \<and> \<not> lnull l2\<close> 
         by auto
     have 3: "( R (llast l1) (lhd (ltl l2)) \<and> ridx R (ltl l2)) =  ridx R l2"
        using  True \<open>\<not> lnull l1 \<and> \<not> lnull l2\<close> ridx_LCons_1[of R "lfirst l2" "ltl l2"]
        by (simp add: assms lfirst_def) 
     show ?thesis 
     by (simp add: "1" "2" "3")
   qed
 next
  case False
  then show ?thesis 
    proof -
     have 4: "(lfuse l1 l2) = lappend l1 (ltl l2) " 
        unfolding lfuse_def using \<open>\<not> lnull l1 \<and> \<not> lnull l2\<close> by simp
     have 5: "ridx R (lappend l1 (ltl l2)) =  (ridx R l1 \<and> ridx R l2)" 
        using assms ridx_lappend_lfinite[of  l1 R "ltl l2" ] False \<open>\<not> lnull l1 \<and> \<not> lnull l2\<close>  
         ridx_expand_1 by (metis lhd_LCons_ltl ridx_LCons_1)
     show ?thesis by (simp add: "4" "5")
    qed    
  qed
next
case False
then show ?thesis 
by (metis lfuse_LNil_1 lfuse_LNil_2 llist.collapse(1) ridx_expand_1)
qed



lemma ridx_lfusecat_lfuse:
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. \<not> lnull xs"
         "\<forall> xs \<in> lset xss. lfinite xs" 
         "\<forall>i.  i < llength xss \<longrightarrow>  ridx R (lnth xss i) "
         " (Suc n) < llength xss"
 shows   " ridx R (lfuse (lnth xss n) (lnth xss (Suc n))) "  
using assms 
by (metis Suc_ile_eq  lfuse_inf llastlfirst_def order_less_imp_le ridx_lfuse_lfinite)


lemma ridx_lfusecat_ltake_a:
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. \<not> lnull xs"
         "\<forall> xs \<in> lset xss. lfinite xs" 
         "\<forall>i.  i < llength xss \<longrightarrow>  ridx R (lnth xss i) "
         "n \<le> llength xss"
 shows   "ridx R (lfusecat (ltake (enat n) xss)) "
using assms
proof (induct n arbitrary: xss)
case 0
then show ?case 
by (metis LNil_eq_ltake_iff gr_implies_not_zero lfusecat_LNil llength_eq_0 llist.disc(1) ridx_def 
    zero_enat_def)
next
case (Suc n)
then show ?case 
   proof -
   have 0: "n=0 \<Longrightarrow> ?thesis" 
     proof -
       assume a0: "n=0" 
      have 000: "\<not>lnull xss"
        using Suc.prems(5) a0 one_enat_def by force
      have 001: "(ltake (enat (Suc n)) xss) = (LCons (lnth xss 0) LNil) " 
          using a0 000
          by (metis One_nat_def lhd_LCons_ltl lnth_0_conv_lhd ltake.ctr(1) ltake_eSuc_LCons one_eSuc 
              one_enat_def)
      have 002: "lfusecat (ltake (enat (Suc n)) xss) =  (lnth xss 0) "
         using a0 000 001  lfusecat_LCons[of "(lnth xss 0)" LNil] by simp 
      show ?thesis using "002" Suc.prems(4) Suc.prems(5) Suc_ile_eq a0 by auto
     qed
   have 01: "n>0 \<Longrightarrow> ?thesis" 
     proof -
      assume a: "n>0"   
      have 1: "(lfusecat (ltake (enat (Suc n)) xss)) = 
            ltake (eSuc (\<Sum>i::nat = 0::nat..Suc n - (1::nat). epred (llength (lnth xss i)))) (lfusecat xss)" 
        using lfusecat_ltake[of  xss "( (Suc n))"] 
        using Suc.prems(1) Suc.prems(2) Suc.prems(3) Suc.prems(5) Suc_ile_eq by force
      have 2: "ridx R (lfusecat (ltake (enat n) xss))"
         using Suc Suc_ile_eq order.strict_implies_order by blast 
      have 20: "lfinite (ltake (enat n) xss)"
         by simp 
       have 21: "\<not> lnull (ltake (enat n) xss)"
        using Suc.prems(5) a enat_0_iff(2) by fastforce
       have 22: "\<forall>xs\<in>lset (ltake (enat n) xss). \<not> lnull xs" 
        by (meson Suc.prems(2) lset_ltake subsetD)
       have 23: "\<forall>xs\<in>lset (ltake (enat n) xss). lfinite xs" 
        by (meson Suc.prems(3) lset_ltake subsetD)
       have 24: "llastlfirst (ltake (enat n) xss)"
        using Suc.prems(1) Suc.prems(5) Suc_ile_eq less_imp_le llastlfirst_ltake by blast 
       have 25: "llast (lfusecat (ltake (enat n) xss)) = llast (llast (ltake (enat n) xss))"
         using "20" "21" "22" "23" "24" llastfirst_lfusecat_llast by blast 
       have 26: "lfinite (llast (ltake (enat n) xss))"
          by (metis Suc.prems(3) Suc.prems(5) Suc_ile_eq Suc_pred' a enat_ord_code(4) in_lset_conv_lnth 
              lfinite_ltake llast_lappend_LCons llast_singleton ltake_Suc_conv_snoc_lnth order_less_imp_le) 
       have 27: "n< llength xss" 
          using Suc.prems(5) Suc_ile_eq order_less_imp_le by blast
       have 271: "(llast (ltake (enat n) xss)) =  (lnth xss (n-1))" 
          by (metis "27" Suc_diff_1 Suc_ile_eq a enat_ord_code(4) lfinite_ltake llast_lappend_LCons 
              llast_singleton ltake_Suc_conv_snoc_lnth order.strict_implies_order)
       have 28: "llast (llast (ltake (enat n) xss)) = llast (lnth xss (n-1))"
         using "271" by auto
       have 29: "llast (lnth xss (n-1)) = lfirst (lnth xss n)"
          by (metis "27" Suc.prems(1) Suc_pred' a llastlfirst_def) 
       have 30: "ridx R (lfuse (lfusecat (ltake (enat n) xss)) (lnth xss ( n))) " 
        by (metis "2" "25" "27" "28" "29" Suc.prems(4) lfuse_inf ridx_lfuse_lfinite)
       have 31: "(lfuse (lfusecat (ltake (enat n) xss)) (lnth xss ( n))) =
                 (lfusecat (ltake (enat (Suc n)) xss))"
         by (simp add: "27" lfusecat_lappend ltake_Suc_conv_snoc_lnth) 
        show ?thesis 
       using "30" "31" by auto
     qed
   show ?thesis 
     using "0" "01" by fastforce
  qed
qed

lemma lfusecat_ltake_llength_less_than_llength_lfusecat: 
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. 1 < llength xs"
         "\<forall> xs \<in> lset xss. lfinite xs" 
         " (enat n)\<le> llength xss"
 shows " min 
            (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
             (llength (lfusecat xss)) = 
             (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))" 
proof -
 have 0: "\<forall> xs \<in> lset xss. \<not>lnull xs"
    using assms(2) by fastforce
 have 1: "lfinite xss \<Longrightarrow> \<not> lnull xss\<Longrightarrow> llength(lfusecat xss) =
          eSuc(\<Sum> i = 0 .. (the_enat(epred(llength xss))) . epred(llength (lnth xss i)))"  
 using lfusecat_llength_lfinite[of xss] assms 0 by fastforce
 have 2: "\<not> lfinite xss \<Longrightarrow> \<not>lfinite (lfusecat xss) " 
    using assms lfusecat_lfinite[of xss] 0 
    by (metis less_numeral_extra(4) lhd_LCons_ltl llength_LCons llength_LNil llist.collapse(1) one_eSuc)
 have 3: "\<not> lfinite xss \<Longrightarrow> llength (lfusecat xss) = \<infinity>" 
   using not_lfinite_llength 2 by blast
 have 50: " \<not> lnull xss\<Longrightarrow> lfusecat (ltake (enat n) xss) =
          ltake (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) 
                (lfusecat xss)" 
        using assms lfusecat_ltake[of xss n] 0 by fastforce
 have 51: "lfinite (lfusecat (ltake (enat n) xss)) "
    by (meson assms(3) enat_ord_code(4) lfinite_ltake lfusecat_lfinite_a lset_ltake subsetD)
 have 52: "lfinite (ltake (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) 
                (lfusecat xss))"
    using "50" "51" llist.collapse(1) by fastforce
 have 53: "\<exists>m. llength (ltake (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) 
                (lfusecat xss)) = (enat m)"
    using "52" lfinite_llength_enat by blast
 have 54: "llength (ltake (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) 
                (lfusecat xss)) \<le> llength (lfusecat xss)" 
       by auto
 have 55: "llength (ltake (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) 
                (lfusecat xss)) = 
            min 
            (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
            (llength (lfusecat xss))"
    using llength_ltake[of "(if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))" 
                           "(lfusecat xss)" ]  by auto
 have 56: " min 
            (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
            (llength (lfusecat xss)) < \<infinity>" 
     by (metis "53" "55" enat_ord_code(4))
 have 57: "(llength (lfusecat xss)) = llength (lfusecat (lappend (ltake n xss) (ldrop n xss)))"
     by (simp add: lappend_ltake_ldrop) 
 have 58: "llength (lfusecat (lappend (ltake n xss) (ldrop n xss))) =
           llength (lfuse (lfusecat (ltake n xss)) (lfusecat (ldrop n xss)))" 
   by (metis "57" lfusecat_split)
 have 59: "lnull (lfusecat (ltake (enat n) xss)) \<longleftrightarrow> n = 0"
    proof (cases xss)
    case LNil
    then show ?thesis using assms  enat_0_iff(2) by force
    next
    case (LCons x21 x22)
    then show ?thesis using "1" "2" "50" by force
    qed 
 have 60: "llength (lfuse (lfusecat (ltake n xss)) (lfusecat (ldrop n xss))) = 
           llength (lfusecat (ltake (enat n) xss)) + 
          (if n= 0 then llength (lfusecat (ldrop (enat n) xss)) 
           else epred (llength (lfusecat (ldrop (enat n) xss)))) "
    using lfuse_llength[of "(lfusecat (ltake n xss))" "(lfusecat (ldrop n xss))"]    
      using "59" by presburger
 have 62: "n= 0 \<Longrightarrow> llength (lfusecat (ldrop (enat n) xss)) = llength (lfusecat xss)"
     using "57" "58" "59" llist.collapse(1) by force 
 have 620: "n>0 \<Longrightarrow> n<llength xss \<Longrightarrow> \<not> lnull (ldrop (enat n) xss)" 
     using Suc_ile_eq assms by simp
 have 621: "n>0 \<Longrightarrow> n<llength xss \<Longrightarrow> (\<exists>xs\<in>lset (ldrop (enat n) xss). \<not> lnull xs)" 
   by (meson "0" "620" in_lset_ldropD lfusecat_not_lnull lfusecat_not_lnull_var)
 have 622: "n>0 \<Longrightarrow> n<llength xss \<Longrightarrow> (\<not> lnull (lfusecat (ldrop (enat n) xss)))"
    using 620 621  lfusecat_not_lnull[of "(ldrop (enat n) xss)" ] by blast
 have 623: "n>0 \<Longrightarrow>  \<not> lnull xss" 
   using assms "59" by force
 have 63: "n>0 \<Longrightarrow>  n<llength xss \<Longrightarrow> (llength (lfusecat (ldrop (enat n) xss))) =
                 (llength (ldrop ((\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) (lfusecat xss) )) " 
   using lfusecat_ldrop[of xss n] assms 
   using "623" llength_LNil not_one_less_zero 0 by presburger
 have 630: "n>0 \<Longrightarrow> n<llength xss \<Longrightarrow> epred(llength (lfusecat (ldrop (enat n) xss))) =
                 epred(llength (ldrop ((\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) (lfusecat xss) ))" 
     using "63" by presburger
 have 631: "n>0 \<Longrightarrow> n<llength xss \<Longrightarrow> 
            (llength (ldrop ((\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) (lfusecat xss) )) > 0"
    by (metis "622" "63" gr_zeroI llength_eq_0) 
 have 64: "llength (lfusecat (ltake (enat n) xss)) \<le>
           llength (lfuse (lfusecat (ltake n xss)) (lfusecat (ldrop n xss)))" 
     using lfuse_llength_le_a by blast
 have 65: "0 < n \<Longrightarrow> n<llength xss \<Longrightarrow> 0< (llength (lfusecat (ldrop (enat n) xss))) " 
    using "63" "631" by presburger
 have 66: "n>0 \<Longrightarrow> llength xss > 0 " 
       using "623" gr_zeroI llength_eq_0 by blast
 have 67: "n>0\<Longrightarrow> n-1 \<le>  (epred (llength xss))"
       using assms 66 
       by (metis epred_enat epred_le_epredI)
 have 68: " min 
            (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
             (llength (lfusecat xss)) = 
             (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))" 
       proof (cases "lfinite xss")
       case True
       then show ?thesis 
          proof -
           have 681: "n>0 \<Longrightarrow>n<llength xss \<Longrightarrow> eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))) \<le> 
                      eSuc (\<Sum>i = 0..the_enat (epred (llength xss)). epred (llength (lnth xss i)))"
             by (metis "1" "631" True gr_implies_not_zero ileI1 linorder_not_less llength_lnull lnull_ldrop)
           have 682: "n>0 \<Longrightarrow>n=llength xss \<Longrightarrow> eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))) \<le> 
                      eSuc (\<Sum>i = 0..the_enat (epred (llength xss)). epred (llength (lnth xss i)))"
             by (metis dual_order.refl epred_enat the_enat.simps)
           show ?thesis 
           using "1" "681" 682 True "623" assms(4) min.absorb1 order.order_iff_strict by auto
          qed
       next
       case False
       then show ?thesis using "3" min_enat_simps(4) by presburger
       qed
  show ?thesis 
  using "68" by blast
qed     




lemma lfusecat_ltake_llength_less_than_next: 
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. 1 < llength xs"
         "\<forall> xs \<in> lset xss. lfinite xs" 
         " (Suc n) < llength xss" 
 shows " (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
         < (if (Suc n) =0 then 0 else eSuc(\<Sum> i = 0 .. (n) . epred(llength (lnth xss i)))) "
proof -
 have 0: "n<\<infinity>"
    by simp
 have 1: "\<And>i. i< llength xss \<Longrightarrow> 1< llength (lnth xss i)"
   by (meson assms(2) in_lset_conv_lnth) 
 have 01: "\<And>i. i< llength xss \<Longrightarrow> \<not>lnull (lnth xss i)" 
     by (metis "1" llength_LNil llist.collapse(1) not_one_less_zero)
 have 02: "\<forall> xs \<in> lset xss. \<not>lnull xs"
     using assms(2) by fastforce 
 have 2: "\<And>i . i< llength xss \<Longrightarrow> 0<epred(llength (lnth xss i))" 
   by (metis "1" co.enat.exhaust_sel gr_zeroI less_numeral_extra(4) not_one_less_zero one_eSuc)
 have 3: "n= 0 \<Longrightarrow> ?thesis"
   by auto 
 have 4: "0<n \<Longrightarrow> (\<Sum> i = 0 .. (n) . epred(llength (lnth xss i))) =
           (\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))) +
            epred(llength (lnth xss n)) " 
    by (metis (no_types, lifting) Suc_diff_1 sum.atLeast0_atMost_Suc)
 have 5: "\<And>i. i< llength xss \<Longrightarrow> epred(llength (lnth xss i)) < \<infinity> " 
   using assms(3) 
     by (metis enat_ord_simps(4) epred_0 epred_Infty epred_inject in_lset_conv_lnth infinity_ne_i0 
         llength_eq_infty_conv_lfinite)
 have 50: "llength (lfusecat (ltake n xss)) \<le> llength(lfusecat xss)" 
    by (simp add: lprefix_lfusecatI lprefix_llength_le)
 
 have 6: "0< n \<Longrightarrow> eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))< \<infinity> "
      proof -
        assume a: "n>0" 
       have 60: " (lfusecat (ltake n xss)) = 
                (ltake (eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) (lfusecat xss))" 
         using lfusecat_ltake[of xss n]   using assms 02 a by simp 
          (metis Suc_ile_eq gr_implies_not_zero llength_eq_0 order.strict_implies_order)
       have 61: "llength (lfusecat (ltake n xss)) = 
                 llength (ltake (eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) (lfusecat xss))"
         using 60 by fastforce
       have 62: "llength (ltake (eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) (lfusecat xss)) =
                 min (eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
                     (llength (lfusecat xss))"
          using llength_ltake by blast  
       have 63: "min (eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
                     (llength (lfusecat xss)) = 
                 (eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))" 
         using lfusecat_ltake_llength_less_than_llength_lfusecat[of xss  n] a assms 
          using Suc_ile_eq order_less_imp_le by simp blast
       show ?thesis 
       by (metis "0" "60" "62" "63" assms(3) enat_ord_simps(4) lfinite_ltake lfusecat_lfinite_a 
           llength_eq_infty_conv_lfinite lset_ltake subsetD)      
     qed
  have 7: "0<n \<Longrightarrow> eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))) < 
          eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))) +
            epred(llength (lnth xss n)) "
        by (metis "2" "6" Suc_ile_eq add.right_neutral assms(4) enat_add_mono less_infinityE 
            order_less_imp_le) 
 have 8: "0<n \<Longrightarrow> ?thesis" 
   using "4" "7" eSuc_plus by auto
 show ?thesis 
 using "3" "8" by blast
qed

lemma ridx_lfusecat_b:
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. 1 < llength xs"
         "\<forall> xs \<in> lset xss. lfinite xs" 
         "\<forall>i.  i < llength xss \<longrightarrow>  ridx R (lnth xss i) "
 shows   "ridx R (lfusecat xss) "
proof -
 have 0: "\<forall> xs \<in> lset xss. \<not> lnull xs" 
   using assms(2) gr_implies_not_zero llength_LNil llist.collapse(1) by blast
 have 1: "\<And>n. n\<le> llength xss \<Longrightarrow> ridx R (lfusecat (ltake (enat n) xss))"
  using assms 0 ridx_lfusecat_ltake_a by blast
 have 2: "\<And>n. n\<le> llength xss \<Longrightarrow> 
           ridx R (ltake (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
                       (lfusecat xss)) "
  using lfusecat_ltake[of xss ] 1 assms using "0" llist.collapse(1) by fastforce
 have 30: "\<And>n . (enat n)\<le> llength xss \<Longrightarrow>
              (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) \<le>
              llength (lfusecat xss)"
       proof -
        fix n::nat
        assume a: "n\<le> llength xss" 
        show "(if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) \<le>
              llength (lfusecat xss)" 
     using lfusecat_ltake_llength_less_than_llength_lfusecat[of xss n ] 0 assms
      a min.order_iff[of "(if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))"
        "llength (lfusecat xss)" ] 
      by presburger   
    qed   
 have 3: "\<And>n . (enat n)\<le> llength xss \<Longrightarrow>
              (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) \<le>
              llength (lfuse (lfusecat (ltake n xss)) (lfusecat (ldrop n xss))) "
      by (metis "30" lfusecat_split)
 have 6: "\<And>n. (Suc n)< llength xss \<Longrightarrow> 
               (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) <
               (if (Suc n) =0 then 0 else eSuc(\<Sum> i = 0 .. (n) . epred(llength (lnth xss i))))" 
    using assms(1) assms(2) assms(3) assms(4) lfusecat_ltake_llength_less_than_next by blast
 have 61:  "\<And>i. i< llength xss \<Longrightarrow> epred(llength (lnth xss i)) < \<infinity> " 
    by (metis assms(3) enat_ord_simps(4) epred_0 epred_Infty epred_inject i0_ne_infinity 
        in_lset_conv_lnth llength_eq_infty_conv_lfinite)
 have 62: "\<And>n. n \<le> llength xss \<Longrightarrow> 
             (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))< \<infinity> "
     by (metis (no_types, lifting) 30 "6" add_diff_cancel_left' assms(3) eSuc_enat enat_ord_code(4) 
         enat_ord_simps(4) ileI1 leD lfusecat_lfinite_a llength_eq_infty_conv_lfinite plus_1_eq_Suc)
 have 7: "\<And>k n. k \<le> (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) 
                \<and>  n\<le> llength xss 
               \<Longrightarrow>
               ridx R (ltake (enat k) (lfusecat xss))"  
  proof -
   fix k n       
   show "k \<le> (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) 
                \<and>  n\<le> llength xss 
               \<Longrightarrow>ridx R (ltake (enat k) (lfusecat xss))"
    using ridx_ltake[of R "(if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))" 
     "(lfusecat xss)" k ]
    using "2" 30 by presburger 
  qed
  have 5: "\<And>k. k < llength xss  \<Longrightarrow>
               ridx R (ltake (enat k) (lfusecat xss))"
  proof - 
    fix k
    show "k < llength xss  \<Longrightarrow>
               ridx R (ltake (enat k) (lfusecat xss))"  
    proof (cases "lfinite xss")
     case True
     then show ?thesis 
       by (metis "1" lfinite_llength_enat linorder_le_cases ltake_all ridx_ltake_a)
     next  
     case False
       then show ?thesis   
        proof -
         have 51: "\<not>lfinite(lfusecat xss)" 
              using assms 0 
              by (metis False iless_Suc_eq lfusecat_lfinite_b lhd_LCons_ltl 
                  llength_LCons not_lnull_llength one_enat_def)
         have 52: "\<And>k. (\<exists>n. enat k < (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
                           \<and> n\<le>llength xss  \<and>  
                       ridx R (ltake (if n =0 then 0 else 
                                      eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) (lfusecat xss)))"
          proof -
           fix k  
           show "(\<exists>n. enat k < (if n =0 then 0 else eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
                           \<and> n\<le>llength xss  \<and>  
                       ridx R (ltake (if n =0 then 0 else 
                               eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) (lfusecat xss)))"
                  using 2 6
           proof (induct k)
            case 0
            then show ?case 
                 proof -
                  have 521: "(enat 0)< eSuc(\<Sum> i = 0 .. (1-1) . epred(llength (lnth xss i)))" 
                     using i0_iless_eSuc zero_enat_def by presburger
                  have 522: "(enat 1) \<le> llength xss"
                    using False 
                    using lnull_imp_lfinite not_lnull_llength one_enat_def by auto
                  have 523: "ridx R (ltake (if (enat 1) =0 then 0 else 
                               eSuc(\<Sum> i = 0 .. (1-1) . epred(llength (lnth xss i)))) (lfusecat xss))" 
                    using "0.prems"(1) "522" one_enat_def by fastforce
                  show ?thesis using "521" "522" "523" one_enat_def by force
                 qed
            next
            case (Suc k)
            then show ?case 
            by (metis (no_types, lifting) False antisym_conv2 diff_Suc_1 eSuc_enat enat_ord_code(4) 
                ileI1 llength_eq_infty_conv_lfinite)
          qed
         qed
        have 53: "\<And>k. (\<exists>m. (enat k) < m \<and>  ridx R(ltake (enat m) (lfusecat xss)))"   
         proof -
          fix k
          show "(\<exists>m. (enat k) < m \<and>  ridx R (ltake (enat m) (lfusecat xss)))"
          proof -   
           have 54: "(\<exists>n.   enat k < (if n =0 then 0 else 
                                      eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i))))
                           \<and> n\<le>llength xss  \<and>  
                          ridx R (ltake (if n =0 then 0 else 
                              eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) (lfusecat xss)))"
              using "52" by blast 
           obtain n where 55: "enat k < (if n =0 then 0 else 
                                         eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) \<and> 
                               n\<le>llength xss  \<and>  
                               ridx R (ltake (if n =0 then 0 else 
                                       eSuc(\<Sum> i = 0 .. (n-1) . epred(llength (lnth xss i)))) (lfusecat xss))"
              using 54 by blast
           show ?thesis 
              using "55" "62" by force
          qed
        qed
     show ?thesis 
     by (metis "51" "53" enat_ord_code(4) llength_eq_infty_conv_lfinite order.strict_implies_order 
         ridx_ltake)  
   qed
  qed
 qed
 show ?thesis 
  by (metis "1" "5" dual_order.refl enat_ord_code(4) lfinite_conv_llength_enat 
      llength_eq_infty_conv_lfinite ltake_all ridx_ltake_all)
qed

lemma ridx_lfusecat:
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. 1 < llength xs"
         "\<forall> xs \<in> lset xss. lfinite xs"         
 shows   "ridx R (lfusecat xss) \<longleftrightarrow> (\<forall>i.  i < llength xss \<longrightarrow>  ridx R (lnth xss i) )"
using ridx_lfusecat_a ridx_lfusecat_b assms
using gr_implies_not_zero llength_eq_0 by blast


lemma lfusecat_split_lsub: 
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. 1 < llength xs"
         "\<forall> xs \<in> lset xss. lfinite xs"   
 shows "(\<forall> i. i< llength xss \<longrightarrow> ridx (\<lambda> a b. f (lsub a b \<sigma>))  (lnth xss i)) \<longleftrightarrow>
        ridx (\<lambda> a b. f (lsub a b \<sigma>)) (lfusecat xss) " 
using assms ridx_lfusecat by blast

lemma lidx_lfuse_lfinite: 
 assumes "lfinite xs" 
         "llast xs = lfirst ys"
 shows   " lidx (lfuse xs ys) \<longleftrightarrow> lidx xs \<and> lidx ys " 
using assms 
using ridx_lfuse_lfinite ridx_lidx by blast

lemma lidx_lfusecat_ltake: 
 assumes "lidx (lfusecat xss)"
         " n \<le> llength xss" 
 shows   " lidx (lfusecat (ltake n xss))" 
using assms ridx_lfusecat_ltake ridx_lidx by blast

lemma lidx_lfusecat_ldrop: 
 assumes "lidx (lfusecat xss)"
         " (enat n) < llength xss" 
         "llastlfirst xss"
         "\<forall> xs \<in> lset xss. \<not> lnull xs"
         "\<forall> xs \<in> lset xss. lfinite xs" 
 shows   " lidx (lfusecat (ldrop n xss))" 
using assms ridx_lfusecat_ldrop ridx_lidx by blast

lemma lidx_lfusecat_a:
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. \<not> lnull xs"
         "\<forall> xs \<in> lset xss. lfinite xs" 
         "lidx (lfusecat xss) "
         " i < llength xss" 
 shows   "  lidx (lnth xss i) "
using assms ridx_lfusecat_a  ridx_lidx by blast


lemma lidx_lfusecat_lfuse:
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. \<not> lnull xs"
         "\<forall> xs \<in> lset xss. lfinite xs" 
         "\<forall>i.  i < llength xss \<longrightarrow>  lidx (lnth xss i) "
         " (Suc n) < llength xss"
 shows   " lidx (lfuse (lnth xss n) (lnth xss (Suc n))) " 
using assms ridx_lfusecat_lfuse ridx_lidx by blast
 
lemma lidx_lfusecat_ltake_a:
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. \<not> lnull xs"
         "\<forall> xs \<in> lset xss. lfinite xs" 
         "\<forall>i.  i < llength xss \<longrightarrow>  lidx (lnth xss i) "
         "n \<le> llength xss"
 shows   "lidx (lfusecat (ltake (enat n) xss)) "
using assms ridx_lfusecat_ltake_a ridx_lidx by blast


lemma lidx_lfusecat_b:
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. 1 < llength xs"
         "\<forall> xs \<in> lset xss. lfinite xs" 
         "\<forall>i.  i < llength xss \<longrightarrow>  lidx (lnth xss i) "
 shows   "lidx (lfusecat xss) "
using assms ridx_lfusecat_b ridx_lidx by blast



lemma lidx_lfusecat:
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. 1 < llength xs"
         "\<forall> xs \<in> lset xss. lfinite xs"         
 shows   "lidx (lfusecat xss) \<longleftrightarrow> (\<forall>i.  i < llength xss \<longrightarrow>  lidx (lnth xss i) )"
using assms ridx_lfusecat ridx_lidx by blast


lemma lnth_sum_expand: 
 assumes "\<not> lnull xss"
         "llastlfirst xss"
         "\<forall> xs \<in> lset xss. 1 < llength xs"
         "\<forall> xs \<in> lset xss. lfinite xs"   
         "(enat i)< llength xss" 
         "lfinite xss" 
 shows " (\<Sum> i = 0 .. (the_enat(epred(llength xss))) . epred(llength (lnth xss i))) =
         epred(llength (lnth xss i)) + 
        (\<Sum> j \<in> {k. k\<noteq>i  \<and> k\<le> (the_enat(epred(llength xss)))} . epred(llength (lnth xss j))) "
proof -
 have 1: "{k. k\<le> (the_enat(epred(llength xss)))} = insert i {k. k\<noteq>i  \<and> k\<le> (the_enat(epred(llength xss)))}" 
   using assms by auto 
    (metis eSuc_epred enat_ord_simps(1) epred_enat gr_implies_not_zero iless_Suc_eq 
     lfinite_llength_enat the_enat.simps)
 have 2: "{0.. (the_enat(epred(llength xss)))} = {k. k\<le> (the_enat(epred(llength xss)))}" 
    by auto 
 have 3: "(\<Sum> i = 0 .. (the_enat(epred(llength xss))) . epred(llength (lnth xss i))) = 
          (\<Sum> i \<in>{k. k\<le> (the_enat(epred(llength xss)))} . epred(llength (lnth xss i))) " 
       using "2" by presburger
 have 4: "(\<Sum> i \<in>{k. k\<le> (the_enat(epred(llength xss)))} . epred(llength (lnth xss i))) =
          (\<Sum> j \<in>(insert i {k. k\<noteq>i  \<and> k\<le> (the_enat(epred(llength xss)))}) . epred(llength (lnth xss j))) " 
      using "1" by presburger
 have 5: "(\<Sum> j \<in>(insert i {k. k\<noteq>i  \<and> k\<le> (the_enat(epred(llength xss)))}) . epred(llength (lnth xss j))) =
          epred(llength (lnth xss i)) + 
        (\<Sum> j \<in> {k. k\<noteq>i  \<and> k\<le> (the_enat(epred(llength xss)))} . epred(llength (lnth xss j)))" 
     by auto
 show ?thesis 
 using "3" "4" "5" by presburger
qed


lemma lfusecat_llength_a: 
 assumes "llastlfirst xss"
         "\<forall> xs \<in> lset xss. 1 < llength xs"
         "\<forall> xs \<in> lset xss. lfinite xs"   
         " i < llength xss"
         "(enat j) < llength (lnth xss i)"
 shows " (enat j) < llength (lfusecat xss)" 
proof (cases "lfinite xss")
case True
then show ?thesis 
  proof -
   have 1: "lnull xss \<Longrightarrow> ?thesis" 
       using assms(4) gr_implies_not_zero llength_eq_0 by blast
   have 2: "\<not> lnull xss \<Longrightarrow>
             llength (lfusecat xss) = 
             eSuc(\<Sum> i = 0 .. (the_enat(epred(llength xss))) . epred(llength (lnth xss i)))" 
    using True assms lfusecat_llength_lfinite by fastforce
   have 3: "\<not> lnull xss \<Longrightarrow>
            llength (lnth xss i) \<le> 
            eSuc(\<Sum> i = 0 .. (the_enat(epred(llength xss))) . epred(llength (lnth xss i)))"
      using lnth_sum_expand[of xss i] assms 
      by (metis (no_types, lifting) True co.enat.collapse eSuc_plus gr_implies_not_zero le_iff_add)  
   have 4: "\<not> lnull xss \<Longrightarrow> ?thesis " 
       using 2 3 
       by (metis assms(5) order_less_le_trans)
   show ?thesis 
   using "1" "4" by blast  
 qed
next
case False
then show ?thesis 
  proof -
   have 5: "\<not> lfinite (lfusecat xss) " 
      by (metis False assms(1) assms(2) assms(3) gr_implies_not_zero iless_Suc_eq lfusecat_lfinite_b 
          lhd_LCons_ltl llength_LCons llength_eq_0 not_lnull_llength one_enat_def)
   show ?thesis 
   using "5" enat_iless lfinite_conv_llength_enat not_less_iff_gr_or_eq by blast
  qed
qed

lemma lmap_lfusecat: 
 " lmap f (lfusecat xss) = (lfusecat (lmap ( lmap f ) xss))"
proof (induct xss)
case adm
then show ?case by simp
next
case LNil
then show ?case 
by simp
next
case (LCons xs xss)
then show ?case 
by (simp add: lfuse_def  )
   (metis lmap_eq_LNil lmap_lappend_distrib lnull_def ltl_lmap)
qed



lemma lfusecat_is_lfirst_conv: 
 assumes "\<forall>i. i< llength xss \<longrightarrow> is_lfirst(lnth xss i) " 
         "is_lfirst xss" 
 shows   "is_lfirst(lfusecat xss) " 
using assms
proof (induction xss)
case adm
  then show ?case 
   by (rule ccpo.admissibleI)  (auto, metis lnth_0 zero_enat_def)
next
case LNil
then show ?case by simp
next
case (LCons xs xss)
then show ?case 
by (simp add: zero_enat_def)
qed

subsection \<open>kfilter\<close>

lemma kfilter_code [simp, code]:
  shows kfilter_LNil: "kfilter P n LNil  = LNil"
  and   kfilter_LCons: "kfilter P n (LCons x xs)  = 
                        (if P x then LCons n (kfilter P (Suc n) xs) else kfilter P (Suc n) xs  )"
by (auto simp add: kfilter_def lzip.ctr(2))

lemma lmap_fst_imp_a: 
assumes "lnull (lfilter P xs)"
 shows  " lnull (lmap fst (lfilter (P \<circ> fst) (lzip xs (iterates Suc n)))) "
using assms lset_lzipD1 by fastforce

lemma lmap_fst_imp_b:
assumes "lnull (lmap fst (lfilter (P \<circ> fst) (lzip xs (iterates Suc n))))"
shows   "  lnull (lfilter P xs)"
proof -
 have 0: "lnull (lmap fst (lfilter (\<lambda> (x,k). P x) (lzip xs (iterates Suc n))))" 
   using assms by auto
 have 1: "lnull (lfilter (\<lambda> (x,k). P x) (lzip xs (iterates Suc n))) " 
   using 0   by auto
 have 2: "(\<forall> (x,k) \<in> lset(lzip xs (iterates Suc n)).  \<not> P x) " 
   using 1 by (simp add: prod.case_eq_if)
 have 3: "(\<forall> (x,k) \<in> { (lnth xs i , n+i) | i. enat i < llength xs}. \<not> P x)"
   using 2 by (simp add: lset_lzip) 
 have 4: "(\<forall> x \<in> { lnth xs i  | i. enat i < llength xs}. \<not> P x)" 
   using 3 by blast
  from 4 show ?thesis by (simp add: lset_conv_lnth)
qed

lemma lmap_snd_lnull:
 "lnull (lmap snd (lfilter (P \<circ> fst) (lzip xs (iterates Suc n)))) = lnull(lfilter P xs)"
by (metis llist.collapse(1) llist.disc(1) lmap_eq_LNil lmap_fst_imp_a lmap_fst_imp_b)

lemma kfilter_lnull_conv:
 " lnull (kfilter P n xs) \<longleftrightarrow> (\<forall> x \<in> lset xs. \<not> P x)"
unfolding kfilter_def using lmap_snd_lnull[of P xs] lnull_lfilter[of P xs] by blast 

lemma kfilter_not_lnull_conv:
 " \<not>lnull (kfilter P n xs) \<longleftrightarrow> (\<exists> x \<in> lset xs.  P x)"
by (simp add: kfilter_lnull_conv)

lemma lmap_fst_lfilter:
 " lfilter P (lmap fst (lzip xs (iterates Suc n))) = 
   lmap fst (lfilter (P \<circ> fst) (lzip xs (iterates Suc n))) "
using lfilter_lmap by blast

lemma lmap_fst_lzip:
 " (lmap fst (lzip xs (iterates Suc n))) = xs"
by (coinduction arbitrary: xs) (auto, simp add: lmap_fst_lzip_conv_ltake) 

lemma lfilter_kfilter_1: 
 " (lmap fst (lfilter (P\<circ>fst) (lzip xs (iterates Suc n)))) =
   (lfilter P xs)"
by (metis (mono_tags, lifting)  lmap_fst_lfilter lmap_fst_lzip)

lemma lfilter_kfilter_snd_llength:
 " llength (lmap fst (lfilter (P\<circ>fst) (lzip xs (iterates Suc n)))) =
   llength (lmap snd (lfilter (P\<circ>fst) (lzip xs (iterates Suc n)))) "
by simp

lemma kfilter_llength:
 " llength(kfilter P n xs) = llength(lfilter P xs)"
proof -
 have 1: "llength(kfilter P n xs) = llength (lmap snd (lfilter (P\<circ>fst) (lzip xs (iterates Suc n))))"
   by (simp add: kfilter_def) 
 have 2: "llength (lmap snd (lfilter (P\<circ>fst) (lzip xs (iterates Suc n)))) =
          llength (lmap fst (lfilter (P\<circ>fst) (lzip xs (iterates Suc n)))) " 
   using lfilter_kfilter_snd_llength by auto
 have 3: "llength (lmap fst (lfilter (P\<circ>fst) (lzip xs (iterates Suc n)))) =
          llength(lfilter P xs)"
   by (metis lfilter_kfilter_1)
 from 1 2 3 show ?thesis by auto
qed

lemma ldropWhile_LEAST:
assumes " \<exists> n<llength xs. (Not \<circ> P) (lnth xs n)"
shows " ldropWhile P xs = ldropn (LEAST n. n < llength xs \<and> (Not \<circ>P) (lnth xs n)) xs " 
proof -
 from assms obtain m where 
   "m< llength xs \<and> (Not \<circ> P) (lnth xs m)" 
   "\<And>n. n< llength xs \<longrightarrow> (Not \<circ>P) (lnth xs n) \<Longrightarrow> m \<le> n"
 and *: "(LEAST n. n < llength xs \<and> (Not \<circ> P) (lnth xs n)) = m" 
  by atomize_elim
    (metis (no_types, lifting) dual_order.strict_trans1 enat_ord_code(2) less_imp_le not_le_imp_less 
       not_less_Least wellorder_Least_lemma(1))
 thus ?thesis unfolding *
 proof (induct m arbitrary: xs)
 case 0
  then show ?case 
    by (cases xs) simp_all
 next
 case (Suc m)
 then show ?case 
    proof -
      have 1: "ldropWhile P (ltl xs) = ldropn m (ltl xs)" using Suc 
        by (cases xs)
           (simp,
            metis Suc_le_mono ldrop_eSuc_ltl ldropn_Suc_conv_ldropn ldropn_eq_LNil llist.simps(3) 
            lnth_Suc_LCons ltl_simps(2) not_le_imp_less)
      have 2: "P (lnth xs 0) "
        using Suc.prems(2) by auto
     show ?thesis 
      by (metis "1" "2" ldropWhile_LCons ldrop_eSuc_ltl lhd_LCons_ltl llist.collapse(1) 
          lnth_0_conv_lhd ltl_simps(1))
 qed
qed
qed

lemma ldropWhile_LEAST_not:
assumes " \<exists> n<llength xs. P (lnth xs n)"
shows " ldropWhile (Not \<circ> P) xs = ldropn (LEAST n. n < llength xs \<and> P (lnth xs n)) xs "
using assms ldropWhile_LEAST[of xs "Not \<circ> P"] by simp

lemma ltakeWhile_LEAST:
assumes " \<exists> n<llength xs. (Not \<circ> P) (lnth xs n)"
shows "(ltakeWhile P xs) = (ltake (lleast (Not \<circ> P) xs) xs)" 
proof-
 from assms obtain m where 
  "m< llength xs \<and> (Not \<circ> P) (lnth xs m)" 
  "\<And>n. n< llength xs \<longrightarrow> (Not \<circ>P) (lnth xs n) \<Longrightarrow> m \<le> n"
 and *: " (lleast (Not \<circ> P) xs) = m" unfolding lleast_def 
 by atomize_elim
 (metis (no_types, lifting) Least_le dual_order.trans enat_ord_code(2) not_less not_less_iff_gr_or_eq
   wellorder_Least_lemma(1))
 thus ?thesis unfolding *
 proof (induct m arbitrary: xs)
 case 0
 then show ?case 
 proof -
 have "ltakeWhile P (LCons (lnth xs 0) (ldropn (Suc 0) xs)) = LNil"
 using 0 by fastforce 
 then show ?thesis 
 by (metis (no_types) lappend.disc_iff(2) lappend_ltakeWhile_ldropWhile ldropn_0 
     ldropn_Suc_conv_ldropn llist.collapse(1) lnull_ldropn ltake.ctr(1) not_less zero_enat_def)
 qed
 next
 case (Suc m)
 then show ?case 
 proof -
   have 1: "ltakeWhile P (ltl xs) = ltake m (ltl xs)" 
     using Suc by (cases xs)
        (simp,
         metis Extended_Nat.eSuc_mono Suc_le_mono eSuc_enat llength_LCons lnth_Suc_LCons ltl_simps(2))
   have 2: "P (lnth xs 0) "
     using Suc by auto
   show ?thesis 
     by (metis "1" "2" eSuc_enat lhd_LCons_ltl lnth_0_conv_lhd ltake.ctr(1) ltakeWhile.code 
         ltake_eSuc_LCons)
  qed
 qed
qed

lemma llength_LEAST_not:
 assumes " \<exists> n < llength xs.  (Not \<circ> P) (lnth xs n)"
 shows "(lleast (Not \<circ> P) xs) < llength xs"
using assms  unfolding lleast_def
by (metis (no_types, lifting) LeastI)

lemma llength_LEAST:
 assumes " \<exists> n < llength xs.  P (lnth xs n)"
 shows "(lleast P xs) < llength xs"
using assms llength_LEAST_not[of xs "Not \<circ> P"] unfolding lleast_def by simp

lemma llength_ltakeWhile_LEAST:
assumes " \<exists> n< llength xs.  (Not \<circ> P) (lnth xs n)"
shows " (llength (ltakeWhile P xs)) = (lleast (Not \<circ> P) xs) "
using assms ltakeWhile_LEAST[of xs P] unfolding lleast_def
by (metis (no_types, lifting) dual_order.strict_trans1 enat_ord_simps(2) le_cases llength_ltake 
    min_def not_less_Least)

lemma llength_ltakeWhile_LEAST_not:
assumes " \<exists> n< llength xs.  P (lnth xs n)"
shows " (llength (ltakeWhile (Not \<circ>P) xs)) = (lleast P xs)"
using assms llength_ltakeWhile_LEAST[of xs "Not \<circ> P"]  unfolding lleast_def by simp

lemma lzip_ldropWhile_fst: 
assumes " llength (lzip xs ys) = llength xs"
shows " lzip (ldropWhile P xs) (lmap snd (ldropWhile (P \<circ> fst) (lzip xs ys)) ) = 
        ldropWhile (P \<circ> fst) (lzip xs ys)"
using assms 
proof -
have f1: "ldropWhile P xs = lmap fst (ldropWhile (P \<circ> fst) (lzip xs ys))"
by (metis (no_types) llength_lzip assms ldropWhile_lmap lmap_fst_lzip_conv_ltake ltake_all order_refl) 
have "ltake (min (llength (ldrop (llength (ltakeWhile (P \<circ> fst) (lzip xs ys))) xs))
                  (llength (ldrop (llength (ltakeWhile (P \<circ> fst) (lzip xs ys))) ys)))
            (ldrop (llength (ltakeWhile (P \<circ> fst) (lzip xs ys))) (lzip xs ys)) = 
      ldrop (llength (ltakeWhile (P \<circ> fst) (lzip xs ys))) (lzip xs ys)"
by (simp add: ltake_all) 
then show ?thesis
using f1 by (simp add: ldropWhile_eq_ldrop lmap_fst_lzip_conv_ltake lmap_snd_lzip_conv_ltake ltake_lzip) 
qed

lemma lzip_ldropWhile_fst_iterates:
 " ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n)) = 
   lzip (ldropWhile (Not \<circ>P) xs) (lmap snd (ldropWhile (Not \<circ>P \<circ> fst) (lzip xs (iterates Suc n)))  )"
by (metis llength_lmap lmap_fst_lzip lzip_ldropWhile_fst)

lemma ldropWhile_iterates_split:
assumes " \<exists> n<llength xs. P (lnth xs n)"
shows " ( (ldropWhile (Not \<circ>P \<circ> fst) (lzip xs (iterates Suc n)))) =
     (lzip  (ldropWhile (Not \<circ>P) xs)
                (iterates Suc (n+(lleast P xs))) )"
proof -
 have 1: "\<exists>na. enat na < llength (lzip xs (iterates Suc n)) \<and> 
           (Not\<circ> (Not\<circ> P) \<circ> fst) (lnth (lzip xs (iterates Suc n)) na)" 
   using lmap_fst_lzip[of xs n] 
   by (metis assms comp_apply llength_lmap lnth_lmap) 
 have 2: "(ldropWhile ((Not \<circ>P) \<circ> fst) (lzip xs (iterates Suc n))) =
          (ldropn (LEAST na. na < llength(lzip xs (iterates Suc n)) \<and>  
          ((Not \<circ> (Not \<circ>P\<circ> fst))) (lnth (lzip xs (iterates Suc n)) na)) (lzip xs (iterates Suc n)))" 
   using 1 ldropWhile_LEAST[of " (lzip xs (iterates Suc n))" "(Not \<circ>P) \<circ> fst"] by auto
 have 3: "(ldropn (LEAST na. na < llength(lzip xs (iterates Suc n)) \<and>
          ((Not \<circ> (Not \<circ>P\<circ> fst))) (lnth (lzip xs (iterates Suc n)) na)) (lzip xs (iterates Suc n))) =
          (ldropn (LEAST na. na < llength(lzip xs (iterates Suc n)) \<and> 
          (( (P\<circ> fst))) (lnth (lzip xs (iterates Suc n)) na)) (lzip xs (iterates Suc n)))" 
   by auto
 have 4: "(ldropn (LEAST na. na < llength(lzip xs (iterates Suc n)) \<and> 
                  (( (P\<circ> fst))) (lnth (lzip xs (iterates Suc n)) na)) (lzip xs (iterates Suc n))) =
          (lzip (ldropn (LEAST na. na < llength(lzip xs (iterates Suc n)) \<and> 
                        (( (P\<circ> fst))) (lnth (lzip xs (iterates Suc n)) na)) xs)
                (ldropn (LEAST na. na < llength(lzip xs (iterates Suc n)) \<and> 
                        (( (P\<circ> fst))) (lnth (lzip xs (iterates Suc n)) na)) (iterates Suc n)))" 
   using ldropn_lzip by blast
 have 5: "(ldropn (LEAST na. na < llength(lzip xs (iterates Suc n)) \<and> 
                  (( (P\<circ> fst))) (lnth (lzip xs (iterates Suc n)) na)) (iterates Suc n)) =
          (iterates Suc (n+(LEAST na. na < llength(lzip xs (iterates Suc n)) \<and> 
                  (( (P\<circ> fst))) (lnth (lzip xs (iterates Suc n)) na)))) " 
   by (metis (no_types, lifting) funpow_Suc_conv ldropn_iterates)
 have 6: "(LEAST na. na < llength(lzip xs (iterates Suc n)) \<and> 
          (( (P\<circ> fst))) (lnth (lzip xs (iterates Suc n)) na)) =
          (LEAST na. na < llength(lzip xs (iterates Suc n)) \<and> 
          (( (P \<circ>fst))) (lnth ( xs ) na, lnth (iterates Suc n) na))"
   by (metis (no_types, opaque_lifting) comp_def fst_conv lmap_fst_lzip lnth_lmap)
 have 7: "llength(lzip xs (iterates Suc n)) = llength xs" 
   by simp
 have 8: "(LEAST na. na < llength(lzip xs (iterates Suc n)) \<and> 
          (( (P \<circ>fst))) (lnth ( xs ) na, lnth (iterates Suc n) na)) =
          (LEAST na. na < llength xs \<and> P (lnth xs na)  )"
   by auto 
 have 9: "(lzip (ldropn (LEAST na. na < llength(lzip xs (iterates Suc n)) \<and> 
                        (( (P\<circ> fst))) (lnth (lzip xs (iterates Suc n)) na)) xs)
                (ldropn (LEAST na. na < llength(lzip xs (iterates Suc n)) \<and> 
                        (( (P\<circ> fst))) (lnth (lzip xs (iterates Suc n)) na)) (iterates Suc n))) =
          (lzip (ldropn (LEAST na. na < llength xs \<and> P (lnth xs na)  ) xs) 
                (iterates Suc (n+(LEAST na. na < llength xs  \<and>  P (lnth xs na) ))) )"
   using "5" "6" by auto
 show ?thesis unfolding lleast_def  
   using assms 2 3 4 9 ldropWhile_LEAST_not[of xs P] by presburger   
qed
 
lemma kfilter_ldropWhile :
 assumes "\<not> lnull(kfilter P n xs)"
 shows " kfilter P n xs =
         (LCons (n+  (lleast P xs)) 
                (lmap snd (lfilter (P\<circ>fst) 
                  (lzip (ldrop (Suc (lleast P xs)) ( xs))
                        (iterates Suc (Suc (n+(lleast P xs))))))))"
proof -
 let ?Least = "(lleast P xs)" 
 have 1: "lhd(lfilter (P\<circ>fst) (lzip xs (iterates Suc n))) =
          lhd (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n)))"
   by simp
 have 2: "ltl(lfilter (P\<circ>fst) (lzip xs (iterates Suc n))) =
           (lfilter (P\<circ>fst) (ltl (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n)))))"  
   by (simp add: ltl_lfilter)
 have 3: "lfilter (P\<circ>fst) (lzip xs (iterates Suc n)) =
          (LCons (lhd(lfilter (P\<circ>fst) (lzip xs (iterates Suc n))))
                 (ltl(lfilter (P\<circ>fst) (lzip xs (iterates Suc n))))) "
   by (metis assms kfilter_llength llength_eq_0 llength_lmap llist.disc(1) llist.exhaust_sel 
        lmap_fst_lfilter lmap_fst_lzip) 
 have 4: "kfilter P n xs =
          lmap  snd (lfilter (P\<circ>fst) (lzip xs (iterates Suc n)))"
   by (simp add: kfilter_def) 
 have 5: "ltl (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n))) =
         ltl((lzip (ldropWhile (Not \<circ>P) xs) 
                   (lmap snd (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n)))))) " 
   by (metis lzip_ldropWhile_fst_iterates)
 have 6: "ltl((lzip (ldropWhile (Not \<circ>P) xs) 
                    (lmap snd (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n)))))) =
         (lzip (ltl (ldropWhile (Not \<circ>P) xs)) 
               (ltl (lmap snd (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n))))))"
   using lzip_ldropWhile_fst_iterates[of P xs n] 
         ltl_lzip[of "(ldropWhile (Not \<circ> P) xs)" 
                     "(lmap snd (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n))))"]
   by force
 have 7: "lmap snd (
         (LCons (lhd (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n))))
                (lfilter (P\<circ>fst) (ltl (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n))))))) =
          (LCons (snd (lhd (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n)))))
                 (lmap snd (lfilter (P\<circ>fst) (ltl (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n)))))))"
   by simp 
 have 8: "\<exists>n. enat n < llength xs \<and> P (lnth xs n)" 
    by (metis assms in_lset_conv_lnth kfilter_llength llength_eq_0 lnull_lfilter)
 have 9: "(snd (lhd (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n))))) =
          (snd (lhd (lzip  (ldropWhile (Not \<circ>P) xs)
                (iterates Suc (n+?Least)) )))"
   using 8 ldropWhile_iterates_split[of xs P n] by simp
 have 10: "(snd (lhd (lzip  (ldropWhile (Not \<circ>P) xs)
                (iterates Suc (n+?Least)) ))) =
             lhd (iterates Suc (n+?Least))"
   by (metis (no_types, lifting) "3" comp_assoc iterates.disc_iff lfilter_eq_LCons lhd_lzip 
       llist.disc(2) lzip.disc(1) lzip_ldropWhile_fst_iterates snd_conv)   
 have 11: "lhd (iterates Suc (n+?Least)) = (n+?Least)"
   by simp   
 have 12: "ltl (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n))) =
           ltl (lzip  (ldropWhile (Not \<circ>P) xs)
                (iterates Suc (n+?Least)) )" 
   using 8 ldropWhile_iterates_split[of xs P n] by simp
 have 13: "ltl (lzip  (ldropWhile (Not \<circ>P) xs) (iterates Suc (n+?Least)) ) =
          (lzip (ltl (ldropWhile (Not \<circ>P) xs)) (ltl (iterates Suc (n+?Least))))"
   by (metis (no_types, lifting) "3" comp_assoc iterates.disc_iff lfilter_eq_LCons llist.discI(2) 
        ltl_lzip lzip.disc_iff(2) lzip_ldropWhile_fst_iterates)
 have 14: "(ltl (iterates Suc (n+?Least))) = (iterates Suc (Suc (n+?Least))) "
   by simp
 have 15: "ltl (ldropWhile (Not \<circ>P) xs) =  ltl (ldrop (llength (ltakeWhile (Not \<circ>P) xs)) xs)" 
   by (simp add: ldropWhile_eq_ldrop)
 have 16: "ltl (ldrop (llength (ltakeWhile (Not \<circ>P) xs)) xs) =
           ldrop (eSuc (llength (ltakeWhile (Not \<circ>P) xs))) ( xs)"
   by (simp add: ldrop_eSuc_conv_ltl) 
 have 17: "(llength (ltakeWhile (Not \<circ>P) xs)) = (llength (ltake ?Least xs))"
   using 8 ltakeWhile_LEAST[of xs "Not \<circ>P" ] unfolding lleast_def by auto
 have 18: "(llength (ltake ?Least xs)) = enat ?Least"
   using "17" "8" llength_ltakeWhile_LEAST_not unfolding lleast_def by fastforce
 have 19: "ldrop (eSuc (llength (ltakeWhile (Not \<circ>P) xs))) ( xs) = ldrop (Suc ?Least) ( xs)" 
   using "17" "18" by (simp add: eSuc_enat)
 have 20: "ltl (ldropWhile (Not \<circ> P \<circ> fst) (lzip xs (iterates Suc n))) =
           (lzip (ldrop (Suc ?Least) ( xs)) (iterates Suc (Suc (n+?Least))))"
   using "12" "13" "15" "16" "19"  by auto
 show ?thesis 
 using "2" "3" "4" "10" "20" "9" by auto
qed
   
lemma kfilter_eq_LCons:
 " kfilter P n xs = LCons x xs' \<Longrightarrow>
        x  = n+(lleast P xs) \<and> 
       xs' = (lmap snd (lfilter (P\<circ>fst) 
                                (lzip (ldrop (Suc (lleast P xs)) ( xs))
                                      (iterates Suc (Suc (n+(lleast P xs)))))))  "
using kfilter_ldropWhile[of P n xs] by auto

lemma kfilter_eq_LCons_1:
 " kfilter P n xs = LCons x xs' \<Longrightarrow>
     x   = (n+(lleast P xs)) \<and>
     xs' = kfilter P (Suc (n+(lleast P xs))) (ldrop (Suc (lleast P xs)) xs)"
using kfilter_eq_LCons[of P n xs x xs'] 
      kfilter_def[of P "(Suc (n + (lleast P xs)))" 
                       "(ldrop (Suc (lleast P xs)) xs)"]
by auto  

lemma kfilter_eq_conv:
 "kfilter P n xs = LNil \<or> 
    kfilter P n xs =
      LCons (n+(lleast P xs)) (kfilter P (Suc (n+(lleast P xs))) (ldrop (Suc (lleast P xs)) xs))"
proof (cases "kfilter P n xs")
case LNil
then show ?thesis by simp
next
case (LCons x21 x22)
then show ?thesis 
    proof -
       let ?Least = "(lleast P xs)" 
       have 1: "x21 = n+?Least"
         using  kfilter_eq_LCons_1[of P n xs x21 x22] by (meson LCons)
       have 2: "x22= (kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs))" 
         using  kfilter_eq_LCons_1[of P n xs x21 x22] by (meson LCons)
       show ?thesis 
         by (metis "1" "2" LCons)
    qed
qed

lemma kfilter_lnth_zero:
 assumes " \<not>lnull(kfilter P n xs)"
 shows   "lnth (kfilter P n xs) 0 = n+ (lleast P xs)"
using assms 
by (metis kfilter_eq_LCons_1 lhd_LCons_ltl lnth_0_conv_lhd)

lemma length_LEAST_a:
 assumes "\<not>lnull(kfilter P n xs)"
 shows   "lleast P xs< llength xs"
using assms exist_lset_lnth[of xs P]  kfilter_not_lnull_conv[of P n xs]  
llength_LEAST[of xs P]  by blast

lemma kfilter_upperbound:
assumes " i < llength(kfilter P n xs)"      
 shows " (lnth (kfilter P n xs) i) < n + llength xs"
proof (cases "lfinite (kfilter P n xs)")
case True
then show ?thesis using assms
   proof (induct zs\<equiv>"kfilter P n xs" arbitrary: xs n i rule: lfinite_induct)
   case (LNil xs)
   then show ?case 
   using gr_implies_not_zero llength_lnull by blast
   next
   case (LCons xs)
   then show ?case 
         proof -
          let ?Least = "(lleast P xs)" 
          have 1: "\<exists> x xs'. kfilter P n xs = LCons x xs'"
            using LCons.hyps(2) kfilter_ldropWhile by blast
          obtain x xs' where 2:"kfilter P n xs = LCons x xs'" 
            using 1 by auto
          have 3: "x = n+?Least"
            using kfilter_ldropWhile[of P n xs] by (simp add: "2")
          have 4: " i=0 \<Longrightarrow> (lnth (kfilter P n xs) i) = x"
            by (simp add: "2")      
          have 5: "enat n+ enat ?Least < enat n + llength xs "  
            using length_LEAST_a[of P n xs] LCons.hyps(2) enat_add_mono by blast
          have 6: "i=0 \<Longrightarrow> enat (lnth (kfilter P n xs) i) < enat n + llength xs"
            using "3" "4" "5" by auto   
          have 7: "xs' = kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs) "
            using kfilter_ldropWhile[of P n xs] "2" kfilter_eq_LCons_1 by blast
          have 8: "i>0 \<Longrightarrow> enat (lnth (kfilter P n xs) i) = enat (lnth xs' (i-1))"
            by (simp add: "2" lnth_LCons') 
          have 9: "i>0 \<and> lnull xs' \<Longrightarrow> enat (lnth (kfilter P n xs) i) < enat n + llength xs"
            by (metis "2" LCons.prems(1) One_nat_def enat_ord_simps(2) llength_LCons llength_LNil 
                llist.collapse(1) not_less_eq one_eSuc one_enat_def)
          have 10: "i>0 \<and> \<not>lnull xs' \<Longrightarrow> (i-1) < llength xs'" 
            using "2" LCons.prems(1) Suc_ile_eq by fastforce
          have 11: "i>0 \<and> \<not>lnull xs' \<Longrightarrow> 
                     enat (lnth xs' (i-1)) < enat (Suc (n+?Least)) + llength (ldrop (Suc ?Least) xs)"
            by (metis (no_types, lifting) "10" "2" "7" LCons.hyps(3) ltl_simps(2))   
          have 111: "lappend (ltake (Suc (?Least)) xs) (ldrop (Suc (?Least)) xs) = xs"
            using lappend_ltake_ldrop by blast
          have 112: "enat (Suc (n+?Least)) = n + (Suc ?Least)"
            by auto
          have 113: " Suc ?Least \<le> (llength xs) "
            using "5" Suc_ile_eq enat_add_mono by blast      
          have 114: "llength (ltake (Suc (?Least)) xs) = Suc ?Least"
            using llength_ltake[of "(Suc (?Least))" xs]   "113" by linarith    
          have 12: "enat (Suc (n+?Least)) + llength (ldrop (Suc ?Least) xs) \<le> enat n + llength xs "
            using llength_lappend[of "(ltake (Suc (?Least)) xs)" "(ldrop (Suc (?Least)) xs)"] 
            by (metis (no_types, lifting) "111" "112" "114" eq_refl group_cancel.add1 plus_enat_simps(1)) 
          have 14: "i>0 \<and> \<not>lnull xs' \<Longrightarrow> enat (lnth (kfilter P n xs) i) < enat n + llength xs"
            using "11" "12" "8" by auto
          have 15: "i>0 \<Longrightarrow> enat (lnth (kfilter P n xs) i) < enat n + llength xs" 
            using "14" "9" dual_order.strict_implies_order by blast         
          show ?thesis 
            using "15" "6" by blast
       qed
   qed
next
case False
then show ?thesis 
by (metis enat_ord_simps(4) iadd_le_enat_iff kfilter_llength leD leI llength_eq_enat_lfiniteD 
     llength_eq_infty_conv_lfinite llength_lfilter_ile)
qed

lemma kfilter_lowerbound:
assumes " i < llength(kfilter P n xs)"
 shows " n \<le>(lnth (kfilter P n xs) i) "
using assms
proof (induct i arbitrary: xs n)
case 0
then show ?case
using kfilter_ldropWhile zero_enat_def by fastforce
next
case (Suc i)
then show ?case 
  proof -
   let ?Least = "lleast P xs" 
   have 7: "\<exists> x xs'. (kfilter P n xs) = LCons x xs'"
     using Suc.prems gr_implies_not_zero kfilter_ldropWhile llength_lnull by blast
   obtain x xs' where 8: "(kfilter P n xs) = LCons x xs'"
     using 7 by auto
   have 9: "lnth (kfilter P n xs) (Suc i) = lnth xs' i"
     by (simp add: "8") 
   have 10: "xs' = kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs) "
     using kfilter_ldropWhile[of P n xs] "8" kfilter_eq_LCons_1 by blast
   have 11: "enat i < llength (kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs))"
     by (metis "10" "8" Extended_Nat.eSuc_mono Suc.prems(1) eSuc_enat llength_LCons)
   have 12: "lnull xs'  \<Longrightarrow> n \<le> lnth (kfilter P n xs) (Suc i)"
     using "10" "11" gr_implies_not_zero llength_eq_0 by blast 
   have 13: "\<not>lnull xs' \<Longrightarrow> (Suc (n+?Least)) \<le> lnth xs' i"
     using "10" "11" Suc.hyps by blast 
   have 14: "\<not> lnull xs'  \<Longrightarrow> n \<le> lnth (kfilter P n xs) (Suc i)" 
     using "13" "9" by linarith
   show ?thesis using "12" "14" by blast
 qed
qed

lemma kfilter_mono:
 assumes " (Suc i) < llength(kfilter P n xs)"
 shows " (lnth (kfilter P n xs) i) < (lnth (kfilter P n xs) (Suc i))"
using assms
 proof (induct i arbitrary: xs n)
 case 0
 then show ?case 
  proof -
   let ?Least = "lleast P xs"
   have 1: "\<exists> x xs'. kfilter P n xs = LCons x xs'"
     using "0.prems" gr_implies_not_zero kfilter_ldropWhile llength_lnull by blast
   obtain x xs' where 2:"kfilter P n xs = LCons x xs'" 
     using 1 by auto
   have 3: "x = n+?Least"
     using kfilter_ldropWhile[of P n xs] by (simp add: "2")
   have 4: "lnth (kfilter P n xs) 0 = n+?Least"
     by (simp add: "2" "3")
   have 5: "lnth (kfilter P n xs) (Suc 0)  = lnth xs' 0"
     by (simp add: "2") 
   have 6: "xs' = kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs) "
     using kfilter_ldropWhile[of P n xs] "2" kfilter_eq_LCons_1 by blast
   have 7: "n+?Least < lnth xs' 0" 
     by (metis "0.prems" "2" "6" Extended_Nat.eSuc_mono One_nat_def Suc_le_lessD kfilter_lowerbound 
         llength_LCons one_eSuc one_enat_def zero_enat_def)
   show ?thesis by (simp add: "4" "5" "7")
 qed
 next
 case (Suc i)
 then show ?case 
  proof -
   let ?Least = "lleast P xs"
   have 8: "\<exists> x xs'. kfilter P n xs = LCons x xs'"
     using Suc.prems gr_implies_not_zero kfilter_ldropWhile llength_lnull by blast
   obtain x xs' where 9:"kfilter P n xs = LCons x xs'" 
     using 8 by auto
   have 10: "xs' = kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs) "
     using kfilter_ldropWhile[of P n xs] "9" kfilter_eq_LCons_1 by blast
   have 11: "lnth (kfilter P n xs) (Suc (Suc i))  = lnth xs' (Suc i)"
     by (simp add: "9") 
   have 12: "lnth (kfilter P n xs) (Suc i) < lnth xs' (Suc i)"
     by (metis (no_types, lifting) "10" "9" Extended_Nat.eSuc_mono Suc(1) Suc(2) eSuc_enat 
         llength_LCons  lnth_Suc_LCons) 
   show ?thesis by (simp add: "11" "12")  
 qed
qed

lemma lfilter_kfilter:
 assumes "  i < llength(kfilter P n xs)"
 shows " (lnth xs ((lnth (kfilter P n xs) i)-n)) = (lnth (lfilter P xs) i)"
using assms
proof (induct i arbitrary: xs n)
case 0
then show ?case 
  proof -
    let ?Least = "lleast P xs" 
    have 1: "lnth (kfilter P n xs) 0 = n +?Least "
    using "0.prems" kfilter_ldropWhile zero_enat_def by fastforce
    have 2: "(lnth xs ((lnth (kfilter P n xs) 0)-n)) =
             (lnth xs ?Least)"
      by (simp add: "1")
    have 3: "(lnth (lfilter P xs) 0) = lhd (ldropWhile (Not \<circ> P) xs)" 
      by (metis (full_types) "0.prems" kfilter_llength lhd_conv_lnth lhd_lfilter llength_eq_0 not_iless0)
    have 4: " \<exists> n<llength xs. P (lnth xs n)" 
      by (metis "0.prems" dual_order.irrefl in_lset_conv_lnth kfilter_llength llength_eq_0 lnull_lfilter 
          zero_enat_def)
    have 5: " (ldropWhile (Not \<circ> P) xs) = ldropn ?Least xs"
      using ldropWhile_LEAST_not[of xs "P" ] "4" unfolding lleast_def by blast
    have 6: "lhd (ldropn ?Least xs) = (lnth xs ?Least)" 
      by (simp add: "4" lhd_ldropn llength_LEAST)
    show ?thesis using "2" "3" "5" "6" by auto
  qed           
next
case (Suc i)
then show ?case 
  proof -
   let ?Least= "lleast P xs"  
   have 7: "\<exists> x xs'. (kfilter P n xs) = LCons x xs'"
     using Suc.prems gr_implies_not_zero kfilter_ldropWhile llength_lnull by blast
   obtain x xs' where 8: "(kfilter P n xs) = LCons x xs'"
     using 7 by auto
   have 9: "lnth (kfilter P n xs) (Suc i) = lnth xs' i"
     by (simp add: "8")  
   have 10: "xs' = kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs) "
     using kfilter_ldropWhile[of P n xs] by (simp add: "8" kfilter_eq_LCons_1)
   have 11: "enat i < llength (kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs))"
     by (metis "10" "8" Extended_Nat.eSuc_mono Suc.prems(1) eSuc_enat llength_LCons)
   have 12: "lnull xs'  \<Longrightarrow>
             lnth xs (lnth (kfilter P n xs) (Suc i) - n) = lnth (lfilter P xs) (Suc i)  "
     by (metis "10" "11" llength_LNil llist.collapse(1) not_less_zero) 
   have 13: "\<not>lnull xs' \<Longrightarrow>  
              lnth (ldrop (Suc ?Least) xs) (lnth xs' i - (Suc (n+?Least))) = 
           lnth (lfilter P (ldrop (Suc ?Least) xs)) i " 
     by (metis (no_types, lifting) "10" "11" Suc.hyps)
   have 14: "(lnth (lfilter P xs) (Suc i)) = (lnth (ltl(lfilter P xs)) i)"
     by (metis "8" kfilter_not_lnull_conv llist.disc(2) lnth_ltl lnull_lfilter)
   have 15: "(ltl(lfilter P xs)) = lfilter P (ltl (ldropWhile (Not \<circ> P) xs))"
     using ltl_lfilter by blast 
   have 16: "(ltl (ldropWhile (Not \<circ> P) xs)) = 
            (ltl (ldropn (LEAST n. n < llength xs \<and> (Not \<circ>(Not\<circ> P)) (lnth xs n)  ) xs)) "
     using ldropWhile_LEAST[of xs "Not \<circ> P"]
     by (metis (no_types, lifting) "8" comp_apply in_lset_conv_lnth kfilter_lnull_conv llist.disc(2))  
   have 17: " (ltl (ldropn (LEAST n. n < llength xs \<and> (Not \<circ>(Not\<circ> P)) (lnth xs n)  ) xs)) =
            (ltl (ldropn ?Least xs))"
     unfolding lleast_def by auto     
   have 18: "(ltl (ldropn ?Least xs)) =
            ldropn (Suc ?Least) xs"
     by (simp add: ldropn_ltl ltl_ldropn)  
   have 19: "ldropn (Suc ?Least) xs =
            ldrop (Suc ?Least) xs"  
     by (simp add: ldrop_enat) 
   have 20: "lnth (lfilter P xs) (Suc i) = lnth (lfilter P (ldrop (Suc ?Least) xs)) i "
     using "14" "15" "16" "18" "19" unfolding lleast_def by auto
   have 21: "lnth xs (lnth (kfilter P n xs) (Suc i) - n) = lnth xs (lnth xs' i -n)"
     by (simp add: "10" "9")     
   have 22: "n\<le> lnth xs' i"
     using "9" Suc.prems(1) kfilter_lowerbound by fastforce 
   have 23: " (Suc (n+?Least)) \<le> lnth (kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs)) i "
     using  kfilter_lowerbound[of i P "Suc (n+?Least)" "(ldrop (Suc ?Least) xs) " ] "11" by force
   have 24: "(Suc ?Least) > llength (ltake ?Least xs)"
     by (simp add: min.strict_coboundedI1) 
   have 25: "(ldrop (Suc ?Least) (lappend (ltake ?Least xs) (ldrop ?Least xs))) =
             ldrop ((Suc ?Least) - llength(ltake ?Least xs) ) (ldrop ?Least xs)  "
     by (simp add: ldrop_lappend) 
   have 26: "lappend (ltake (Suc ?Least) xs) (ldrop (Suc ?Least) xs) = xs"
     by (simp add: lappend_ltake_ldrop) 
   have 27: " (Suc ?Least) \<le> (lnth xs' i -n)  "
     using "10" "23" by auto
   have 271: " lnth xs' i -n - (Suc ?Least) = lnth xs' i - (Suc (n+?Least)) "
     by auto 
   have 28: "lnth (lappend (ltake (Suc ?Least) xs) (ldrop (Suc ?Least) xs)) (lnth xs' i -n) =
              lnth (ldrop (Suc ?Least) xs) (lnth xs' i - (Suc (n+?Least))) "
     using lnth_lappend[of "(ltake (Suc ?Least) xs)" "(ldrop (Suc ?Least) xs)" "(lnth xs' i -n)"] 
           27 271 11 19
     by (metis (no_types, lifting) gr_implies_not_zero kfilter_LNil ldropn_eq_LNil 
          le_cases llength_lnull llength_ltake llist.disc(1) lnth_lappend2 min_def)        
   have 29: "lnth xs (lnth xs' i -n) =
              lnth (ldrop (Suc ?Least) xs) (lnth xs' i - (Suc (n+?Least)))" 
     using 28 by (metis (no_types, lifting) "26")
   have 30: " \<not>lnull xs' \<Longrightarrow> 
             lnth xs (lnth (kfilter P n xs) (Suc i) - n) = lnth (lfilter P xs) (Suc i) "
     by (metis "13" "20" "21" "29")    
   show ?thesis 
     using "12" "30" by blast
 qed
qed

lemma in_kfilter_lset:
 shows " x \<in> lset (kfilter P n xs) \<longleftrightarrow> x \<in> { n+i | i. i<llength xs \<and> P (lnth xs i) }" 
 (is "?lhs = ?rhs")
proof 
 assume ?lhs
  thus ?rhs
   proof (induct zs\<equiv>"kfilter P n xs" arbitrary: n xs rule:llist_set_induct)
   case (find)
   then show ?case 
    proof -
      let ?Least = "lleast P xs" 
      have 1: "lhd (kfilter P n xs) = n+?Least"
        using kfilter_ldropWhile[of P n xs] find by auto
      have 2: "P (lnth xs ?Least)" unfolding lleast_def 
        by (metis (mono_tags, lifting) LeastI exist_lset_lnth find.hyps kfilter_lnull_conv)
      have 3: "?Least < llength xs"
        by (metis find.hyps length_LEAST_a )
      have 4: "n+?Least \<in> {n + i |i. enat i < llength xs \<and> P (lnth xs i)}" 
        using "2" "3" by blast
      show ?thesis using "1" "4" by auto
    qed
 next
 case (step y)
 then show ?case 
    proof -
     let ?Least = "lleast P xs" 
     have 5: "ltl (kfilter P n xs) = kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs)"
       using kfilter_ldropWhile[of P n xs] 
       by (metis kfilter_def ltl_simps(2) step.hyps(1))
     have 6: "lnull (kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs)) \<Longrightarrow>
              y \<in> {n + i |i. enat i < llength xs \<and> P (lnth xs i)} "
       by (metis "5" gr_implies_not_zero in_lset_conv_lnth llength_eq_0 step.hyps(2))
     have 7: "\<not> lnull (kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs)) \<Longrightarrow>
               y \<in> {(Suc (n+?Least)) + i |i. enat i < llength (ldrop (Suc ?Least) xs) \<and> 
                                             P (lnth (ldrop (Suc ?Least) xs) i)}"
       using step.hyps using "5" by blast
     have 8: "(Suc (n+?Least)) = n + Suc ?Least"
       by auto 
     have 9: "\<not> lnull(kfilter P (Suc (n + ?Least)) (ldrop (enat (Suc ?Least)) xs)) \<Longrightarrow>
               \<exists>i. y = n + Suc (?Least) + i \<and> enat i < llength (ldrop (enat (Suc ?Least)) xs) \<and> 
                   P (lnth (ldrop (enat (Suc ?Least)) xs) i) \<Longrightarrow> 
                \<exists>i. y = n + i \<and> enat i < llength xs \<and> P (lnth xs i)" 
       proof -
        assume a0: "\<not> lnull(kfilter P (Suc (n + ?Least)) (ldrop (enat (Suc ?Least)) xs))"
        assume a1: "\<exists>i. y = n + Suc (?Least) + i \<and> enat i < llength (ldrop (enat (Suc ?Least)) xs) \<and> 
                          P (lnth (ldrop (enat (Suc ?Least)) xs) i)"
        show "\<exists>i. y = n + i \<and> enat i < llength xs \<and> P (lnth xs i)"
        proof -
         obtain i where 10: "y = n + Suc (?Least) + i \<and> enat i < llength (ldrop (enat (Suc ?Least)) xs) \<and>
                                P (lnth (ldrop (enat (Suc ?Least)) xs) i)"
           using a1 by auto
         have 11: "Suc (?Least) + i < llength xs"
           by (metis "10" add.commute ldrop_enat ldrop_ldrop leD le_less_linear lnull_ldropn 
               plus_enat_simps(1))    
         have 12: "P (lnth xs (Suc (?Least) + i))"
           by (metis "10" "11" add.commute ldrop_enat lnth_ldropn) 
         show ?thesis 
           using "10" "11" "12" ab_semigroup_add_class.add_ac(1) by blast
        qed
       qed
     have 13: "\<not> lnull (kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs)) \<Longrightarrow>
              y \<in> {n + i |i. enat i < llength xs \<and> P (lnth xs i)}" 
       using "7" "9" by auto
     show ?thesis 
       using "13" "6" by blast    
   qed
 qed
next
 assume ?rhs
 then obtain i where 15: "x = n+i \<and> enat i < llength xs \<and> P (lnth xs i)"
   by blast  
 thus ?lhs
  proof (induct i arbitrary: xs n)
  case 0
  then show ?case 
     proof -
      have 16:  "lhd (kfilter P n xs) = n+(lleast P xs)"
        using kfilter_ldropWhile[of P n xs] using "0.prems"
        by (metis in_lset_conv_lnth kfilter_lnull_conv lhd_LCons) 
      show ?thesis 
        by (metis "0.prems" add.right_neutral kfilter_LCons ldropn_0 ldropn_Suc_conv_ldropn 
            llist.set_intros(1))
     qed
  next
  case (Suc i)
  then show ?case 
     proof - 
      have 18: "lnull xs \<Longrightarrow> x \<in> lset (kfilter P n xs)" 
        using Suc(2) by auto
      have 19: "\<not> lnull xs \<Longrightarrow> P (lnth xs (Suc i)) = P (lnth (ltl xs) ( i)) " 
        by (simp add: lnth_ltl)
      have 20: "\<not> lnull xs \<Longrightarrow> x = (Suc n) +i \<and> enat i < llength (ltl xs) \<and> P (lnth (ltl xs) ( i))" 
        by (metis "19" Extended_Nat.eSuc_mono Suc.prems(1) add_Suc_shift eSuc_enat lhd_LCons_ltl 
            llength_LCons) 
      have 21: "\<not> lnull xs \<Longrightarrow> lnull (kfilter P n (ltl xs)) \<Longrightarrow> x \<in> lset (kfilter P n xs) "
        by (metis "20" in_lset_conv_lnth kfilter_lnull_conv)
      have 22: "\<not> lnull xs \<Longrightarrow> \<not>lnull (kfilter P n (ltl xs)) \<Longrightarrow> x \<in> lset (kfilter P (Suc n) (ltl xs))"
        by (metis "20" Suc.hyps) 
      have 23: "\<not>lnull xs \<Longrightarrow> x \<in> lset (kfilter P n xs)" 
        using kfilter_LCons[of P n "lhd xs" "ltl xs"]
              in_lset_ltlD lhd_LCons_ltl[of "(kfilter P n (ltl xs))"] 
        using "21" "22" by fastforce
      show ?thesis 
        using "18" "23" by blast
    qed
  qed
qed

lemma kfilter_lset:
 shows "  lset (kfilter P n xs) = { n+i | i. i<llength xs \<and> P (lnth xs i) }" 
using  in_kfilter_lset by blast 

lemma lfilter_lnth_exist:
 assumes 
         " i < llength (lfilter P xs)" 
 shows " (\<exists> k < llength xs. lnth (lfilter P xs) i = lnth xs k)"
using assms lset_lfilter[of P xs] 
by (metis (no_types, opaque_lifting) add.left_neutral diff_zero kfilter_llength kfilter_upperbound 
lfilter_kfilter zero_enat_def)

lemma ldistinct_kfilter:
 " ldistinct(kfilter P n xs)"
proof (coinduction arbitrary: n xs)
case (ldistinct n1 xs1)
then show ?case 
   proof -
    have 1: "lhd (kfilter P n1 xs1) \<notin> lset (ltl (kfilter P n1 xs1))"
      proof -
       have f1: "kfilter P n1 xs1 =
                 LCons (n1 + lleast P xs1)
                       (lmap snd 
                           (lfilter (P \<circ> fst) 
                                    (lzip 
                                         (ldrop (enat (Suc (lleast P xs1))) xs1) 
                                         (iterates Suc (Suc (n1 + lleast P xs1))))))"
         by (meson kfilter_ldropWhile ldistinct)
       then have "lmap snd 
                    (lfilter (P \<circ> fst) 
                            (lzip (ldrop (enat (Suc (lleast P xs1))) xs1) 
                                  (iterates Suc (Suc (n1 + lleast P xs1))))) = 
              kfilter P (Suc (n1 + lleast P xs1)) (ldrop (enat (Suc (lleast P xs1))) xs1)"
       using kfilter_eq_LCons_1 by blast
       then show ?thesis
       using f1 by (metis (no_types) in_lset_conv_lnth kfilter_lowerbound leD lessI lhd_LCons ltl_simps(2))
      qed
    have 2: "((\<exists>n xs. ltl (kfilter P n1 xs1) = kfilter P n xs) \<or> ldistinct (ltl (kfilter P n1 xs1)))"
      by (metis kfilter_eq_LCons_1 ldistinct llist.discI(1) llist.exhaust_sel)
    show ?thesis
      using "1" "2" by auto         
 qed
qed
   

lemma kfilter_llength_ltake:
 " llength(kfilter P n (ltake k xs)) \<le> llength(kfilter P n xs)"
by (simp add: kfilter_llength lprefix_lfilterI lprefix_llength_le)

lemma kfilter_ldropn_lset:
 assumes " k< llength xs"
 shows   "  lset(kfilter P n (ldropn k xs)) = 
            { n+i | i. i< llength xs - k \<and> P (lnth xs (k+i)) } "
using assms 
kfilter_lset[of P n "(ldropn k xs)" ] 
by auto
  (metis (no_types, lifting) add.commute enat_min lnth_ldropn order.strict_implies_order 
   plus_enat_simps(1),
   metis (no_types, lifting) add.commute dual_order.strict_implies_order enat_min lnth_ldropn 
   plus_enat_simps(1))

lemma kfilter_ldropn_lset_a:
 assumes " k< llength xs"
 shows "  lset(kfilter P n (ldropn k xs)) = 
          { n+(i-k) | i. k\<le>i \<and> i< llength xs \<and> P (lnth xs i) } "
proof -
 have 1: "\<And>x. x\<in> { n+i | i. i< llength xs - k \<and> P (lnth xs (k+i)) } \<longleftrightarrow> 
            x \<in> { n+(i-k) | i. k\<le>i \<and> i< llength xs \<and> P (lnth xs i) }" 
    proof auto
     show "\<And>i. enat i < llength xs - enat k \<Longrightarrow> 
               P (lnth xs (k + i)) \<Longrightarrow>
                \<exists>ia. i = ia - k \<and> k \<le> ia \<and> enat ia < llength xs \<and> P (lnth xs ia)" 
       using assms 
       by (metis add.commute add_diff_cancel_left' enat_min le_add1 less_imp_le plus_enat_simps(1))
     show "\<And>i. k \<le> i \<Longrightarrow>
                enat i < llength xs \<Longrightarrow> 
                P (lnth xs i) \<Longrightarrow> 
                \<exists>ia. n + i - k = n + ia \<and> enat ia < llength xs - enat k \<and> P (lnth xs (k + ia)) " 
       using assms ldropn_Suc_conv_ldropn[of _ xs] ldropn_eq_LConsD[of _ "ldropn k xs"] 
             ldropn_ldropn[of _ _ xs]
       by (metis Nat.add_diff_assoc le_add_diff_inverse le_add_diff_inverse2 llength_ldropn)
     qed 
  show ?thesis  using assms 1 kfilter_ldropn_lset[of k xs P n] by auto 
qed

lemma kfilter_ldropn_lset_b:
 assumes " k< llength xs"
 shows "  lset(kfilter P n (ldropn k xs)) = 
          { n+i | i. i< llength xs -k \<and> P (lnth xs (i+k)) } "
proof -
 have 1: "\<And>x. x\<in>{ n+i | i. i< llength xs - k \<and> P (lnth xs (k+i)) }  \<longleftrightarrow>
               x\<in> { n+i | i. i< llength xs -k \<and> P (lnth xs (i+k)) }" 
   by (auto simp add: add.commute)
 show ?thesis  using assms 1 kfilter_ldropn_lset[of k xs P n] by auto 
qed
  
lemma kfilter_llength_n_zero: 
shows " llength(kfilter P n xs) = llength(kfilter P  0 xs)"
 by (simp add: kfilter_llength)

lemma kfilter_lnth_n_zero_a:
 assumes " k < llength (kfilter P n xs)"
 shows " n \<le> (lnth (kfilter P n xs) k)"
using assms by (simp add: kfilter_lnull_conv kfilter_lowerbound)

lemma kfilter_lnth_n_zero:
 assumes " k < llength (kfilter P n xs)"
 shows "(lnth (kfilter P n xs) k) -n = (lnth (kfilter P 0 xs) k)"
using assms
proof (induct k arbitrary: xs n)
case 0
then show ?case by (cases "(kfilter P n xs)")
     (simp, 
      metis add.left_neutral add_left_cancel kfilter_ldropWhile kfilter_lnull_conv kfilter_lowerbound 
      le_add_diff_inverse llength_eq_0 lnth_0 not_gr_zero zero_enat_def)
next
case (Suc k)
then show ?case 
     proof -
      have 1: "lnull(kfilter P n xs) \<Longrightarrow> lnth (kfilter P n xs) (Suc k) - n = lnth (kfilter P 0 xs) (Suc k) "
        using Suc.prems gr_implies_not_zero llength_lnull by blast 
      have 2: "\<not>lnull(kfilter P n xs) \<Longrightarrow> lnth (kfilter P n xs) (Suc k) -n = lnth (kfilter P 0 xs) (Suc k)"
        proof -
         assume a0: "\<not>lnull(kfilter P n xs)"
         show "lnth (kfilter P n xs) (Suc k) -n = lnth (kfilter P 0 xs) (Suc k)"
          proof -
           let ?Least = "lleast P xs" 
           have 3: "\<exists> x xs'.(kfilter P n xs) = LCons x xs' "
             using a0 kfilter_ldropWhile by blast 
           obtain x xs' where 4: "(kfilter P n xs) = LCons x xs'"
             using 3 by auto
           have 5: "x = n+?Least"
             using kfilter_ldropWhile[of P n xs]
             by (simp add: "4")
           have 6: "\<not>lnull(kfilter P n xs) \<longleftrightarrow> \<not>lnull(kfilter P 0 xs)"
             by (simp add: kfilter_not_lnull_conv) 
           have 7: "\<exists> y ys'.(kfilter P 0 xs) = LCons y ys' "
             using "6" a0 kfilter_ldropWhile by blast
           obtain y ys' where 8: "(kfilter P 0 xs) = LCons y ys'" 
             using 7 by auto
           have 9: "y = ?Least" 
             using kfilter_ldropWhile[of P 0 xs] by (simp add: "8")
           have 10: "x-n = y"
             using "5" "9" diff_add_inverse by blast 
           have 11: "xs' = kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs) "
             using kfilter_ldropWhile[of P n xs]
             by (metis (no_types, lifting) "4" a0 kfilter_def ltl_simps(2))
           have 12: "ys' = kfilter P (Suc (0+?Least)) (ldrop (Suc ?Least) xs)" 
             using kfilter_ldropWhile[of P 0 xs]
             by (metis (no_types, lifting) "6" "8" a0 kfilter_def ltl_simps(2))
           have 13: "lnth (kfilter P n xs) (Suc k) - n  = lnth xs' k - n"
             by (simp add: "4") 
           have 14: "lnth (kfilter P 0 xs) (Suc k)   = lnth ys' k "
             by (simp add: "8") 
           have 15: "\<not>(\<exists> x \<in> lset (ldrop (Suc ?Least) xs). P x) \<Longrightarrow>
                      lnth (kfilter P n xs) (Suc k) -n = lnth (kfilter P 0 xs) (Suc k)  " 
             using 8 Suc by auto 
             (metis (full_types) gr_implies_not_zero kfilter_eq_conv kfilter_not_lnull_conv 
              ldropn_Suc_LCons ldropn_Suc_conv_ldropn ldropn_eq_LConsD llength_LNil llist.disc(2))   
           have 16: "enat k < llength (kfilter P (Suc (n+?Least)) (ldrop (Suc ?Least) xs)) "
             by (metis (no_types, lifting) "12" "8" Extended_Nat.eSuc_mono Suc.prems eSuc_enat 
                  kfilter_llength llength_LCons)
           have 17: "(\<exists> x \<in> lset (ldrop (Suc ?Least) xs). P x) \<Longrightarrow>
                   lnth xs' k  - (Suc (n+?Least)) = lnth (kfilter P 0 (ldrop (Suc ?Least) xs)) k"
             using Suc.hyps using "11" "16" by blast
           have 18: "enat k < llength (kfilter P (Suc (?Least)) (ldrop (Suc ?Least) xs)) "  
             by (metis "16" kfilter_llength)  
           have 19: "(\<exists> x \<in> lset (ldrop (Suc ?Least) xs). P x) \<Longrightarrow>
                     lnth ys' k - (Suc (0+?Least)) =
                     lnth (kfilter P 0 (ldrop (Suc ?Least) xs)) k" 
             using Suc.hyps by (simp add: "12" "18")
           have 20: "(\<exists> x \<in> lset (ldrop (Suc ?Least) xs). P x) \<Longrightarrow>
                        lnth (kfilter P n xs) (Suc k) -n = lnth xs' k -n"
             using "13" by blast
           have 21: "(\<exists> x \<in> lset (ldrop (Suc ?Least) xs). P x) \<Longrightarrow> 
                       lnth xs' k -n = lnth (kfilter P 0 (ldrop (Suc ?Least) xs)) k + (Suc (?Least)) "
             using "11" "16" "17" kfilter_lowerbound by fastforce 
           have 22: "(\<exists> x \<in> lset (ldrop (Suc ?Least) xs). P x) \<Longrightarrow> 
                       lnth (kfilter P 0 (ldrop (Suc ?Least) xs)) k + (Suc (?Least)) = lnth ys' k"
             using "12" "18" "19" kfilter_lowerbound by fastforce
           have 23: "(\<exists> x \<in> lset (ldrop (Suc ?Least) xs). P x) \<Longrightarrow>
                       lnth (kfilter P n xs) (Suc k) -n = lnth (kfilter P 0 xs) (Suc k)  "                
             using "14" "20" "21" "22" by linarith        
           show ?thesis 
             using "15" "23" by blast
         qed
       qed
     show ?thesis 
     using "1" "2" by blast
  qed
qed

lemma kfilter_n_zero:
 shows " (kfilter P n xs) = lmap (\<lambda>i. i+n) (kfilter P 0 xs)"
proof -
 have 1: "llength(kfilter P n xs) = llength (lmap (\<lambda>i. i+n) (kfilter P 0 xs)) "
   by (simp add: kfilter_llength)
 have 2: "\<And>k. k < llength(kfilter P n xs) \<longrightarrow> 
           lnth (kfilter P n xs) k = lnth (lmap (\<lambda>i. i+n) (kfilter P 0 xs)) k " 
   using kfilter_lnth_n_zero kfilter_lnth_n_zero_a
   by (metis "1"  le_add_diff_inverse2 llength_lmap lnth_lmap )
 show ?thesis 
   by (simp add: "1" "2" llist_eq_lnth_eq)   
qed

lemma kfilter_n_zero_a:
 shows " (kfilter P 0 xs) = lmap (\<lambda>i. i-n) (kfilter P n xs)"
proof -
 have 1: "llength(kfilter P 0 xs) = llength (lmap (\<lambda>i. i-n) (kfilter P n xs)) "
   by (simp add: kfilter_llength)
 have 2:  "\<And>k. k < llength(kfilter P 0 xs) \<longrightarrow> 
           lnth (kfilter P 0 xs) k = lnth (lmap (\<lambda>i. i-n) (kfilter P n xs)) k "
   by (simp add:  kfilter_llength kfilter_lnth_n_zero)
 show ?thesis 
   using "1" "2" llist_eq_lnth_eq by blast
qed

lemma kfilter_holds:
 assumes "y \<in> lset(kfilter P n xs)" 
 shows " P (lnth xs (y-n))"
using assms in_kfilter_lset[of y P n xs]  kfilter_lnull_conv[of P n xs]  
using lset_lnull by fastforce

lemma kfilter_holds_not:
 assumes "y \<in> ({i+n| i. i < llength xs} - (lset (kfilter P n xs)))" 
 shows "  \<not> P (lnth xs (y-n))"
using assms kfilter_lset[of P n xs] kfilter_lnull_conv[of P n xs]
by auto 

lemma kfilter_holds_a:
 assumes " i < llength xs "
         "(i+n) \<in> lset(kfilter P n xs) " 
 shows " P (lnth xs i) "
using assms kfilter_holds[of "i+n" P n xs]  by simp 

lemma kfilter_holds_not_a:
 assumes " i < llength xs "
         " P (lnth xs i)"
 shows " (i+n) \<in> lset(kfilter P n xs)  "   
using assms 
by (simp add: in_kfilter_lset kfilter_lnull_conv)

lemma kfilter_holds_b:
 assumes " i < llength xs"
 shows "(i+n) \<in> lset(kfilter P n xs) =   P (lnth xs i)"
using assms 
by (meson kfilter_holds_a kfilter_holds_not_a)    

lemma kfilter_holds_c: 
  assumes " n \<le> i"
          " i -n < llength xs"
  shows " i \<in> lset(kfilter P n xs) = P (lnth xs (i-n))"
using assms 
by (metis diff_add idiff_enat_enat kfilter_holds kfilter_holds_not_a)  

lemma kfilter_holds_not_b:
  assumes " n \<le> i"
          " i -n < llength xs"
  shows  " i \<notin> lset(kfilter P n xs) = (\<not> P (lnth xs (i-n)))"
using assms by (simp add: kfilter_holds_c)

lemma kfilter_disjoint_lset_coset: 
   shows "({i+n| i. i < llength xs} - (lset (kfilter P n xs))) \<inter> lset (kfilter P n xs) = {}  "
by blast 

lemma lidx_kfilter_expand:
 assumes " (Suc na) < llength(kfilter P n xs)" 
 shows " lnth (kfilter P n xs) na < lnth (kfilter P n xs) (Suc na)"
using assms kfilter_mono by force

lemma lidx_kfilter:
 shows " lidx  (kfilter P n xs)"
unfolding lidx_def 
using lidx_kfilter_expand by blast

lemma lidx_kfilter_gr_eq:
  assumes 
          "k\<le> j"
          " j < llength(kfilter P n xs)"
 shows    " lnth (kfilter P n xs) k \<le> lnth (kfilter P n xs) j"
using assms  
using lidx_kfilter lidx_less_eq by blast

lemma lidx_kfilter_gr:
  shows    "\<forall> j . k<j \<and> j < llength(kfilter P n xs) \<longrightarrow>  
             lnth (kfilter P n xs) k < lnth (kfilter P n xs) j"
using less_imp_Suc_add lidx_kfilter lidx_less by blast

lemma kfilter_not_before:
 assumes "0<llength(kfilter P 0 xs)" 
         " i< lnth (kfilter P 0 xs) 0 "
  shows " \<not> P (lnth xs i)"
proof -
 have 0: " (lnth (kfilter P 0 xs) 0)  < llength xs"
       by (metis assms(1) gen_llength_def kfilter_upperbound llength_code zero_enat_def)
 have 1: "\<not> lnull (kfilter P 0 xs)" 
    using assms(1) by auto
 have 2: " i \<in> lset (kfilter P 0 xs) \<Longrightarrow> 
           \<not> lnull (kfilter P 0 xs) \<Longrightarrow> 
           i< lnth (kfilter P 0 xs) 0 \<Longrightarrow> 
           False"
    proof (induct zs\<equiv>"(kfilter P 0 xs)" arbitrary: xs rule: lset_induct)
    case (find xs)
    then show ?case by (metis less_not_refl3 lhd_LCons lnth_0_conv_lhd)
    next
    case (step x' xs)
    then show ?case
    proof -
    have "\<forall>ns n. \<exists>na. ((n::nat) \<notin> lset ns \<or> lnth ns na = n) \<and> (n \<notin> lset ns \<or> enat na < llength ns)"
    by (meson in_lset_conv_lnth)
    then show ?thesis using step
    by (metis (no_types) leD less_nat_zero_code lidx_kfilter_gr_eq llist.set_intros(2) not_le_imp_less)
    qed 
    qed
 have 3: " i \<notin> lset (kfilter P 0 xs) \<and> i < llength xs \<longrightarrow> \<not> P (lnth xs i) "
   by (simp add: "1" in_kfilter_lset)      
 have 4: " i < llength xs" 
    using "0" assms(2) dual_order.strict_trans enat_ord_simps(2) by blast
 show ?thesis 
 using "1" "2" "3" "4" assms(2) by blast
qed

lemma kfilter_n_not_before:
 assumes "0 < llength (kfilter P n (ldropn n xs) ) "
         " n < llength xs" 
         " n \<le> i"
         " i< lnth (kfilter P n (ldropn n xs) ) 0 "
  shows " \<not> P (lnth xs i)"
proof -
 have 00: "\<not> lnull (kfilter P n (ldrop n xs)) " 
     by (metis assms(1) ldrop_enat less_numeral_extra(3) llength_LNil llist.collapse(1))
 have 0: "lnth (kfilter P n (ldrop n xs) ) 0 < llength xs" 
   using assms kfilter_upperbound[of 0 P n "(ldropn n xs)"]
   by (metis lappend_ltake_enat_ldropn ldrop_enat llength_lappend llength_ltake min.strict_order_iff 
       zero_enat_def) 
 have 1: "i \<in> lset (kfilter P n (ldropn n xs)) \<Longrightarrow> 
          \<not> lnull (kfilter P n (ldropn n xs)) \<Longrightarrow> 
          i< lnth (kfilter P n (ldropn n xs) ) 0 \<Longrightarrow> 
          n < llength xs  \<Longrightarrow> 
          n \<le> i\<Longrightarrow> 
          False" 
   proof (induct zs\<equiv>"(kfilter P n (ldropn n xs))" arbitrary: n xs rule: lset_induct)
   case (find xs)
   then show ?case 
   by (metis lnth_0 nat_less_le)
   next
   case (step x' xs)
   then show ?case 
   by (metis (no_types, lifting) in_lset_conv_lnth kfilter_eq_LCons_1 kfilter_lowerbound leD 
       less_SucI  lnth_0 )
   qed
 have 2: "i \<notin> lset (kfilter P n (ldropn n xs)) \<and> n \<le> i \<and> i < llength xs \<longrightarrow> \<not> P (lnth xs i) "
    using assms kfilter_holds_not_a[of i] 00 
       by (simp add: kfilter_ldropn_lset_a ldrop_enat)
 have 3: " n \<le> i \<and> i < llength xs"
     using assms 00 kfilter_ldropWhile[of P n "ldropn n xs" ] 
     by (metis "0" enat_ord_simps(2) ldrop_enat less_trans)    
 show ?thesis 
 using "1" "2" "3" assms(1) assms(2) assms(4) kfilter_not_lnull_conv by auto
qed 


lemma kfilter_not_after:
 assumes "0<llength(kfilter P 0 xs)" 
         " lnth (kfilter P 0 xs) (k-1) < i"
         " llength (kfilter P 0 xs) = (enat k)" 
         " i < llength xs" 
  shows " \<not> P (lnth xs i)"
proof -
 have 0: "\<not> lnull (kfilter P 0 xs)" 
     using assms(1) by auto
 have 01: "0<k" 
    using "0" assms(3) gr0I zero_enat_def by fastforce
 have 02: "i \<notin> lset (kfilter P 0 xs) "
 by (metis "01" One_nat_def Suc_pred assms(2) assms(3) diff_less enat_ord_simps(2) 
     in_lset_conv_lnth leD less_Suc_eq_le lidx_kfilter_gr_eq zero_less_one)   
 from 0 01 02 show ?thesis 
 by (metis add.right_neutral assms(4) kfilter_holds_not_a)
qed

lemma kfilter_n_not_after:
 assumes "0 < llength (kfilter P n (ldropn n xs))" 
         " n < llength xs" 
         " lnth (kfilter P n (ldropn n xs) ) (k-1) < i"
         " llength (kfilter P n (ldropn n xs) ) = (enat k)" 
         " i < llength xs" 
  shows " \<not> P (lnth xs i)"
proof -
 have 0: "\<not> lnull (kfilter P n (ldropn n xs))" 
      using assms(1) by auto
 have 1: "0<k"
   using assms(1) assms(4) by (simp add: zero_enat_def)
 have 2: "i \<notin> lset(kfilter P n (ldropn n xs))"  
     using assms
     by (metis "1" Suc_diff_1 enat_ord_simps(2) in_lset_conv_lnth leD less_Suc_eq_le 
         lidx_kfilter_gr_eq order_refl)       
 from  0 1 2 show ?thesis 
using assms kfilter_ldropn_lset_a[of n xs P n] kfilter_lowerbound[of _ P n "(ldropn n xs)"] 
  by auto (metis One_nat_def diff_less le_trans less_imp_le zero_less_one) 
qed

lemma kfilter_not_between:
 assumes " lnth (kfilter P 0 xs) (k) < i"
         " i<lnth (kfilter P 0 xs) (Suc k)" 
         " (Suc k) < llength (kfilter P 0 xs) " 
  shows " \<not> P (lnth xs i)"
proof -
 have 0: "\<exists> x \<in> lset xs. P x" 
     using assms(3) gr_implies_not_zero kfilter_not_lnull_conv by fastforce
 have 1: "\<not> lnull (kfilter P 0 xs)" 
    by (simp add: 0 kfilter_not_lnull_conv)
 have 2: "lnth (kfilter P 0 xs) (Suc k) \<le> llength xs"
    using kfilter_upperbound[of "Suc k" P 0 xs] 
    using "1" assms(3) zero_enat_def by auto 
 have 3: "i \<notin> lset(kfilter P 0 xs)"
  using assms
  by (metis Suc_ile_eq dual_order.strict_implies_order in_lset_conv_lnth leD lidx_kfilter_gr_eq 
      not_less_eq_eq) 
 from 1 2 3 show ?thesis 
 by (metis add.right_neutral assms(2) enat_ord_simps(2) kfilter_holds_not_a less_le_trans)
qed
       
lemma lfilter_kfilter_ltake_lidx_a:
 assumes " k < llength(lfilter P xs)"
 shows  " lidx (ltake k (kfilter P n xs))"
unfolding lidx_def
using assms 
by (metis Suc_ile_eq kfilter_mono less_imp_le  llength_ltake lnth_ltake min.strict_boundedE) 

lemma lfilter_kfilter_ldropn_lidx_a:
 assumes " k < llength(lfilter P xs)"
 shows  " lidx (ldropn k (kfilter P n xs))"
using assms unfolding lidx_def 
  proof auto
   fix na 
   assume a0: "enat k < llength (lfilter P xs)"
   assume a1: "enat (Suc na) < llength (kfilter P n xs) - enat k" 
   show "lnth (ldropn k (kfilter P n xs)) na < lnth (ldropn k (kfilter P n xs)) (Suc na) "
   proof -
    have 1: "enat (k + (Suc na)) < llength (kfilter P n xs)"
      proof -
       have "Suc na + k = k + Suc na"
         by presburger
       then show ?thesis
         by (metis (no_types) a1 ldropn_eq_LNil ldropn_ldropn leD leI llength_ldropn)
      qed
    have 2 : "enat (k + na) < llength (kfilter P n xs)"
      by (metis "1" Suc_ile_eq add_Suc_right less_imp_le)
    have 3: "lnth (kfilter P n xs) (k + (na)) <  lnth (kfilter P n xs) (k + (Suc na))"
      using "1" kfilter_mono by auto
    show ?thesis by (metis "1" "2" "3" add.commute lnth_ldropn) 
   qed
qed

lemma lfilter_kfilter_ldropn_lidx_b:
 assumes " k < llength(lfilter P xs)"
 shows  " lidx (kfilter P (lnth (kfilter P n xs) k) (ldropn (lnth (kfilter P n xs) k) xs))"
using assms using lidx_kfilter by blast

lemma ltake_lset:
 assumes "k < llength xs"
 shows   " lset (ltake k xs) = {(lnth xs i) | i. i < k}  "
using assms 
by (auto simp add: in_lset_conv_lnth lnth_ltake)
   (blast, meson less_trans lnth_ltake)

lemma ldropn_lset:
 assumes " k < llength xs"
 shows " lset (ldropn k xs) = {(lnth xs i) | i. k \<le> i \<and> i < llength xs}"
using assms 
proof (auto simp add: in_lset_conv_lnth )
  fix n :: nat
  assume a1: "enat n < llength xs - enat k"
  assume "enat k < llength xs"
  then have "enat k \<le> llength xs"
    by (meson dual_order.strict_implies_order)
  then have "enat n + enat k < llength xs"
    using a1 by (meson enat_min)
  then show "\<exists>na. lnth (ldropn k xs) n = lnth xs na \<and> k \<le> na \<and> enat na < llength xs"
    using a1 by (metis ldropn_ldropn le_add2 lhd_ldropn llength_ldropn plus_enat_simps(1))
next
  fix i :: nat
  assume a1: "enat i < llength xs"
  assume a2: "k \<le> i"
  have 1: " enat (i-k) < llength xs -enat k"
    by (metis a1 a2 diff_add ldropn_Suc_conv_ldropn ldropn_ldropn llength_ldropn llist.disc(2)
          lnull_ldropn not_le_imp_less) 
  have 2: "lnth (ldropn k xs) (i-k) = lnth xs i"
    by (simp add: a1 a2) 
  show "\<exists>n. enat n < llength xs - enat k \<and> lnth (ldropn k xs) n = lnth xs i"
    using 1 2 by blast
qed  
 
lemma lfilter_kfilter_ltake_lset_eq:
 assumes 
         " k < llength(lfilter P xs)"
 shows " lset (ltake k (kfilter P 0 xs)) =
         lset (kfilter P 0 (ltake ((lnth (kfilter P 0 xs) k)) xs))"
proof -
 have 1: "(lnth (kfilter P 0 xs) k) < llength xs"
   using kfilter_llength kfilter_upperbound
   by (metis assms gen_llength_def  kfilter_llength kfilter_upperbound llength_code )
 have 2: " \<exists> x \<in> lset xs . P x"
   using assms gr_implies_not_zero llength_eq_0 lnull_lfilter by blast
 have 3: "{i. i< llength(ltake ((lnth (kfilter P 0 xs) k)) xs) \<and>
           P (lnth (ltake ((lnth (kfilter P 0 xs) k)) xs) i)} =
          {i. i< (lnth (kfilter P 0 xs) k) \<and> P (lnth xs i)} "
   by (auto simp add: lnth_ltake "1" order_less_subst2)
 have 5: "{i. i< (lnth (kfilter P 0 xs) k) \<and> P (lnth xs i)} =
          {i. i< (lnth (kfilter P 0 xs) k) \<and> i \<in> lset(kfilter P 0 xs)} "
   by (auto simp add: "1" kfilter_holds_c order_less_subst2)
 have 6: "{i. i< (lnth (kfilter P 0 xs) k) \<and> i \<in> lset(kfilter P 0 xs)} =
          {i. i< (lnth (kfilter P 0 xs) k) \<and> i \<in> {(lnth (kfilter P 0 xs) j) | j. j < llength(kfilter P 0 xs)}} "
   by (simp add: lset_conv_lnth) 
 have 7: "{i. i< (lnth (kfilter P 0 xs) k) \<and> i \<in> {(lnth (kfilter P 0 xs) j) | j. j < llength(kfilter P 0 xs)}} =
            {(lnth (kfilter P 0 xs) j) | j. j<k}"
   by (auto simp add: assms lidx_kfilter_gr kfilter_llength)
      (metis kfilter_llength leD lidx_kfilter_gr_eq not_le_imp_less,
       metis assms enat_ord_simps(2) less_trans)        
 have 8: "k < llength(kfilter P 0 xs)"
   by (simp add: assms kfilter_llength) 
 have 9: "{(lnth (kfilter P 0 xs) j) | j. j<k} = lset(ltake k (kfilter P 0 xs))" 
   using ltake_lset[of k "(kfilter P 0 xs)"] 8  by auto
 have 10: "lset (kfilter P 0 (ltake ((lnth (kfilter P 0 xs) k)) xs)) =
           {i. i< (lnth (kfilter P 0 xs) k) \<and>
               P (lnth (ltake (lnth (kfilter P 0 xs) k) xs) i) }"
   by (auto simp add: in_kfilter_lset)
      (meson "1" enat_ord_simps(2) less_trans)
 have 11: "(lnth (kfilter P 0 xs) k) = llength(ltake (lnth (kfilter P 0 xs) k) xs) "
   by (simp add: "1" dual_order.strict_implies_order min_def)      
 have 12: "{i. i< (lnth (kfilter P 0 xs) k) \<and>
               P (lnth (ltake (lnth (kfilter P 0 xs) k) xs) i) } =
           {i. i< llength(ltake (lnth (kfilter P 0 xs) k) xs) \<and>
               P (lnth (ltake (lnth (kfilter P 0 xs) k) xs) i) }" 
   using "11" by auto
 show ?thesis 
   using "10" "12" "3" "5" "6" "7" "9" by auto
qed

lemma lfilter_kfilter_ldropn_lset_eq:
 assumes "  k < llength(lfilter P xs)"
 shows   " lset(kfilter P (lnth (kfilter P 0 xs) k) (ldropn (lnth (kfilter P 0 xs) k) xs)) =
           lset(ldropn  k (kfilter P 0 xs))"
proof -
 have 1: "(lnth (kfilter P 0 xs) k) < llength xs "
   using kfilter_llength kfilter_upperbound
   by (metis assms gen_llength_def  kfilter_llength kfilter_upperbound llength_code )
 have 2: " \<exists> x \<in> lset xs . P x"
   using assms gr_implies_not_zero llength_eq_0 lnull_lfilter by blast
 have 10: "lset(kfilter P (lnth (kfilter P 0 xs) k) (ldropn (lnth (kfilter P 0 xs) k) xs)) =
           {(lnth (kfilter P 0 xs) k) + i | i.  i< llength xs - (lnth (kfilter P 0 xs) k) \<and>
              P (lnth xs (i+(lnth (kfilter P 0 xs) k))) }"
   using 1 kfilter_ldropn_lset_b[of "(lnth (kfilter P 0 xs) k)" xs P "(lnth (kfilter P 0 xs) k)"] 
   by linarith
 have 5: "{(lnth (kfilter P 0 xs) k) + i | i.  i< llength xs - (lnth (kfilter P 0 xs) k) \<and>
              P (lnth xs (i+(lnth (kfilter P 0 xs) k))) } =
          {(lnth (kfilter P 0 xs) k) + i | i.  i< llength xs - (lnth (kfilter P 0 xs) k) \<and>
              (i+(lnth (kfilter P 0 xs) k)) \<in> lset(kfilter P 0 xs) }"
   using kfilter_holds_b[of _ xs 0 P] using "1" "2" 
   by auto 
      (metis ldropn_eq_LNil ldropn_ldropn leD llength_ldropn not_le_imp_less,
      metis gen_llength_def in_lset_conv_lnth kfilter_upperbound llength_code)
 have 51: "{(lnth (kfilter P 0 xs) k) + i | i.  i< llength xs - (lnth (kfilter P 0 xs) k) \<and>
              (i+(lnth (kfilter P 0 xs) k)) \<in> lset(kfilter P 0 xs) } =
           {(lnth (kfilter P 0 xs) k) + i | i.  
               (lnth (kfilter P 0 xs) k) \<le> i + (lnth (kfilter P 0 xs) k) \<and>
               i + (lnth (kfilter P 0 xs) k) < llength xs  \<and>
              (i+(lnth (kfilter P 0 xs) k)) \<in> lset(kfilter P 0 xs) }" 
   by auto
      (metis "1" enat_min less_imp_le plus_enat_simps(1),
       metis ldropn_eq_LNil ldropn_ldropn leD llength_ldropn not_le_imp_less)      
 have 52: "{(lnth (kfilter P 0 xs) k) + i | i.  
               (lnth (kfilter P 0 xs) k) \<le> i + (lnth (kfilter P 0 xs) k) \<and>
               i + (lnth (kfilter P 0 xs) k) < llength xs  \<and>
              (i+(lnth (kfilter P 0 xs) k)) \<in> lset(kfilter P 0 xs) } =
            {j. (lnth (kfilter P 0 xs) k) \<le>j \<and> j < llength xs \<and> j \<in> lset(kfilter P 0 xs)}"
   by (metis (no_types, lifting)  add.commute le_iff_add) 
 have 53: "{j. (lnth (kfilter P 0 xs) k) \<le>j \<and> j < llength xs \<and> j \<in> lset(kfilter P 0 xs)} =
            {j. (lnth (kfilter P 0 xs) k) \<le>j \<and> j < llength xs \<and> 
                j \<in> { (lnth (kfilter P 0 xs) jj)| jj. jj< llength(kfilter P 0 xs)}}"
   by (simp add: lset_conv_lnth) 
 have 54: "{j. (lnth (kfilter P 0 xs) k) \<le>j \<and> j < llength xs \<and> 
                j \<in> { (lnth (kfilter P 0 xs) jj)| jj. jj< llength(kfilter P 0 xs)}} =
          { (lnth (kfilter P 0 xs) j) | j. k\<le>j \<and> j < llength (kfilter P 0 xs)}"
   by auto
      (metis assms kfilter_llength lidx_kfilter_gr not_less,
       meson lidx_kfilter_gr_eq,
       metis gen_llength_def kfilter_upperbound llength_code)
 have 8: "k < llength(kfilter P 0 xs)"
   by (simp add: assms kfilter_llength) 
 have 9: "{ (lnth (kfilter P 0 xs) j) | j. k\<le>j \<and> j < llength (kfilter P 0 xs)} =
          lset(ldropn k (kfilter P 0 xs))"
   using ldropn_lset[of k "(kfilter P 0 xs)"] using "8" by blast 
 show ?thesis 
   using "10" "5" "51" "52" "53" "54" "9" by auto
qed

lemma kfilter_kfilter_ltake:
 assumes " k < llength(lfilter P xs)"
 shows   " (ltake k (kfilter P 0 xs)) = 
           (kfilter P 0 (ltake ((lnth (kfilter P 0 xs) k)) xs))"
using assms 
by (simp add: lfilter_kfilter_ltake_lidx_a lfilter_kfilter_ltake_lset_eq lidx_kfilter lidx_lset_eq)

lemma kfilter_kfilter_ldropn:
 assumes " k < llength(lfilter P xs)"
 shows  "(ldropn  k (kfilter P 0 xs)) =
           (kfilter P (lnth (kfilter P 0 xs) k) (ldropn (lnth (kfilter P 0 xs) k) xs))" 
using assms 
by (simp add: lfilter_kfilter_ldropn_lidx_a lfilter_kfilter_ldropn_lidx_b 
    lfilter_kfilter_ldropn_lset_eq lidx_lset_eq)

lemma kfilter_lmap_lfilter:
 shows   " lmap (\<lambda>n. (lnth xs n)) (kfilter P 0 xs) = lfilter P xs"
using lfilter_kfilter[of _ P 0 xs] 
by (metis (no_types, lifting) diff_zero  kfilter_llength llength_lmap 
    llist_eq_lnth_eq lnth_lmap)

lemma lfilter_kfilter_ltake:
 assumes "  k < llength(lfilter P xs)"
 shows   " ltake k (lfilter P xs) = 
           (lfilter P (ltake (lnth (kfilter P 0 xs) k) xs))"
proof -
 have 2: "lmap (\<lambda>n. (lnth xs n)) (kfilter P 0 xs) = lfilter P xs"
   using kfilter_lmap_lfilter by blast 
 have 3: "ltake k (lfilter P xs) = 
          ltake k (lmap (\<lambda>n. (lnth xs n)) (kfilter P 0 xs))" 
   by (simp add: "2")
 have 4: "ltake k (lmap (\<lambda>n. (lnth xs n)) (kfilter P 0 xs)) =
           lmap (\<lambda>s. lnth xs s) (ltake k (kfilter P 0 xs))"
   using ltake_lmap by blast 
 have 6: "(lfilter P (ltake (lnth (kfilter P 0 xs) k) xs)) =
          lmap (\<lambda>s. (lnth (ltake (lnth (kfilter P 0 xs) k) xs) s)) 
               (kfilter P 0 (ltake (lnth (kfilter P 0 xs) k) xs))"
   by (simp add: kfilter_lmap_lfilter) 
 have 7: "lmap (\<lambda>s. lnth xs s) (ltake k (kfilter P 0 xs)) =
          lmap (\<lambda>s. lnth xs s) (kfilter P 0 (ltake (lnth (kfilter P 0 xs) k) xs))"
   by (simp add:  assms kfilter_kfilter_ltake) 
 have 8: "lmap (\<lambda>s. lnth xs s) (kfilter P 0 (ltake (lnth (kfilter P 0 xs) k) xs)) =
          lmap (\<lambda>s. (lnth (ltake (lnth (kfilter P 0 xs) k) xs) s)) 
               (kfilter P 0 (ltake (lnth (kfilter P 0 xs) k) xs))"
   using kfilter_kfilter_ltake[of k P xs] 
   by (metis gen_llength_def kfilter_upperbound lappend_ltake_enat_ldropn ldistinct_Ex1 
       ldistinct_kfilter llength_code llist.map_cong0 lnth_lappend)
 show ?thesis 
   using "2" "4" "6" "7" "8" by auto
qed

lemma kfilter_lmap_shift_ldropn:
 shows   " lmap (\<lambda>s. lnth xs (s+(lnth (kfilter P 0 xs) k))) 
                (kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k) xs)) =
           lmap (\<lambda>s. lnth xs s) 
                (kfilter P (lnth (kfilter P 0 xs) k) (ldropn (lnth (kfilter P 0 xs) k) xs))"
proof -
 have 1: "llength (lmap (\<lambda>s. lnth xs (s+(lnth (kfilter P 0 xs) k))) 
                  (kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k) xs))) =
          llength (lmap (\<lambda>s. lnth xs s) 
                  (kfilter P (lnth (kfilter P 0 xs) k) (ldropn (lnth (kfilter P 0 xs) k) xs)))"
   by (simp add: kfilter_llength) 
 have 2: "\<And>i. i< llength (lmap (\<lambda>s. lnth xs (s+(lnth (kfilter P 0 xs) k))) 
                               (kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k) xs))) \<Longrightarrow>
             lnth (lmap (\<lambda>s. lnth xs (s+(lnth (kfilter P 0 xs) k))) 
                        (kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k) xs))) i =
             lnth xs ((lnth (kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k) xs)) i)+
                      (lnth (kfilter P 0 xs) k))  "
   by simp 
 have 3: "\<And>i. i< llength (lmap (\<lambda>s. lnth xs s) 
              (kfilter P (lnth (kfilter P 0 xs) k) (ldropn (lnth (kfilter P 0 xs) k) xs))) \<Longrightarrow>
              lnth (lmap (\<lambda>s. lnth xs s) 
              (kfilter P (lnth (kfilter P 0 xs) k) 
                         (ldropn (lnth (kfilter P 0 xs) k) xs))) i =
              lnth xs  (lnth (kfilter P (lnth (kfilter P 0 xs) k) 
                                        (ldropn (lnth (kfilter P 0 xs) k) xs)) i) "
   by simp 
 have 4: "\<And>i. i< llength (lmap (\<lambda>s. lnth xs (s+(lnth (kfilter P 0 xs) k))) 
                         (kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k) xs))) \<Longrightarrow>
               lnth xs ((lnth (kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k) xs)) i)+
                        (lnth (kfilter P 0 xs) k)) =
               lnth xs  (lnth (kfilter P (lnth (kfilter P 0 xs) k) 
                                         (ldropn (lnth (kfilter P 0 xs) k) xs)) i) "
   using kfilter_lnth_n_zero[of _ P "(lnth (kfilter P 0 xs) k)" "(ldropn (lnth (kfilter P 0 xs) k) xs)" ]  
         kfilter_lowerbound[of _ P "(lnth (kfilter P 0 xs) k)" "(ldropn (lnth (kfilter P 0 xs) k) xs)"]  
   using 1 diff_add by fastforce
 show ?thesis 
   using llist_eq_lnth_eq[of "lmap (\<lambda>s. lnth xs (s+(lnth (kfilter P 0 xs) k))) 
                (kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k) xs))"]
   using "1" "2" "3" "4" by presburger
qed

lemma lfilter_kfilter_ldropn:
 assumes "  k < llength(lfilter P xs)"
 shows " (ldropn k (lfilter P xs)) = 
         (lfilter P (ldropn (lnth (kfilter P 0 xs) k) xs))"
proof -
 have 1: "\<exists> x \<in> lset xs. P x"
   using assms gr_implies_not_zero llength_eq_0 lnull_lfilter by blast  
 have 2: "(lfilter P xs) = lmap (\<lambda>s. lnth xs s) (kfilter P 0 xs)" 
   by (simp add: kfilter_lmap_lfilter)
 have 3: "ldrop k (lfilter P xs) =
          ldrop k (lmap (\<lambda>s. lnth xs s) (kfilter P 0 xs)) "
   by (simp add: "2") 
 have 4: "(ldrop k (lmap (\<lambda>s. lnth xs s) (kfilter P 0 xs))) =
          (lmap (\<lambda>s. lnth xs s) (ldrop k (kfilter P 0 xs)))" 
   using ldrop_lmap by blast
 have 6: "(lfilter P (ldropn (lnth (kfilter P 0 xs) k) xs)) =
          lmap (\<lambda>s. lnth (ldropn (lnth (kfilter P 0 xs) k) xs) s) 
               (kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k ) xs))"
   by (simp add: kfilter_lmap_lfilter) 
 have 61: "\<And>z. z \<in> lset ((kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k ) xs))) 
            \<Longrightarrow> (\<lambda>s. lnth (ldropn (lnth (kfilter P 0 xs) k) xs) s) z = 
          (\<lambda>s. lnth xs (s + (lnth (kfilter P 0 xs) k)) ) z" 
   using assms in_lset_conv_lnth[of _ "((kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k ) xs)))"] 
   by simp
      (metis add.commute add.left_neutral kfilter_upperbound ldropn_eq_LNil ldropn_ldropn leD
       lnth_ldropn not_le_imp_less zero_enat_def)
 have 7: "lmap (\<lambda>s. lnth (ldropn (lnth (kfilter P 0 xs) k) xs) s) 
               (kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k ) xs)) =
          lmap (\<lambda>s. lnth xs (s + (lnth (kfilter P 0 xs) k)) )
              (kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k ) xs)) "
   using "61" by auto
 have 8: " lmap (\<lambda>s. lnth xs (s + (lnth (kfilter P 0 xs) k)) )
              (kfilter P 0 (ldropn (lnth (kfilter P 0 xs) k ) xs)) =
          lmap (\<lambda>s. lnth xs s) 
              (kfilter P (lnth (kfilter P 0 xs) k) (ldropn (lnth (kfilter P 0 xs) k) xs)) "
   by (simp add:  kfilter_lmap_shift_ldropn) 
 have 9: "lmap (\<lambda>s. lnth xs s) 
              (kfilter P (lnth (kfilter P 0 xs) k) (ldropn (lnth (kfilter P 0 xs) k) xs)) =
          lmap (\<lambda>s. lnth xs s) (ldropn k (kfilter P 0 xs))"
   by (simp add:  assms kfilter_kfilter_ldropn)
 show ?thesis 
   by (metis "4" "7" "8" "9" kfilter_lmap_lfilter ldrop_enat) 
qed

lemma lfilter_lnth_aa:
assumes "n < llength (lfilter P xs)"
 shows  " P (lnth (lfilter P xs) n)"
using assms 
by (meson in_lset_conv_lnth lfilter_id_conv lfilter_idem)

lemma exist_one_conv:
 "(\<exists>! i. i<llength xs \<and> P (lnth xs i)) \<longleftrightarrow>
  (\<exists> k < llength xs. P (lnth xs k) \<and> 
           (\<forall> j < llength xs. j\<noteq>k \<longrightarrow> \<not> P (lnth xs j)))"
by blast

lemma lfilter_llength_one_conv_a:
 assumes "llength(lfilter P xs) =1"
 shows   " \<exists> k < llength xs. P (lnth xs k) \<and> 
           (\<forall> j < llength xs. j\<noteq>k \<longrightarrow> \<not> P (lnth xs j))"
proof -
 have 1: "P (lnth xs (lnth (kfilter P 0 xs) 0))"
   by (metis assms(1)  kfilter_llength kfilter_lmap_lfilter less_numeral_extra(1) lfilter_lnth_aa 
       lnth_lmap zero_enat_def) 
 have 2: "(lnth (kfilter P 0 xs) 0) < llength xs"
   by (metis assms(1) gen_llength_def kfilter_llength  kfilter_upperbound less_numeral_extra(1) 
       llength_code zero_enat_def) 
 have 3: "(\<forall> j < llength xs. j\<noteq> (lnth (kfilter P 0 xs) 0) \<longrightarrow> \<not> P (lnth xs j)) "
   using assms kfilter_not_after[of P xs] kfilter_not_before[of P xs]
   by (metis One_nat_def diff_Suc_1 kfilter_llength linorder_neqE_nat one_enat_def zero_less_one) 
 show ?thesis 
   using "1" "2" "3" by blast
qed

lemma lfilter_llength_one_conv_c:
 " (\<exists> k < llength xs. P (lnth xs k) \<and> 
           (\<forall> j < llength xs. j\<noteq>k \<longrightarrow> \<not> P (lnth xs j))) \<longleftrightarrow>
   ( \<exists> k < llength xs. P (lnth xs k) \<and> 
           (\<forall> j < llength xs. j<k \<or> k<j \<longrightarrow> \<not> P (lnth xs j)))"
using antisym_conv3 by  auto

lemma lfilter_llength_one_conv_d:
 " ( \<exists> k < llength xs. P (lnth xs k) \<and> 
           (\<forall> j < llength xs. j<k \<or> k<j \<longrightarrow> \<not> P (lnth xs j))) \<longleftrightarrow>
   ( \<exists> k < llength xs. P (lnth xs k) \<and> 
      (\<forall> j . j<k  \<longrightarrow> \<not> P (lnth xs j)) \<and>
      (\<forall> j < llength xs. k<j \<longrightarrow> \<not> P (lnth xs j)))"
by (meson enat_ord_simps(2) less_trans)

lemma exist_one_lfilter_llength_one:
assumes "(\<exists>! i. i<llength xs \<and> P (lnth xs i))"
shows " llength(lfilter P xs) \<le>1 "
using assms 
proof auto
 fix i :: nat
 assume a1: "\<forall>y y'. enat y < llength xs \<and> P (lnth xs y) \<and> enat y' < llength xs \<and> P (lnth xs y') \<longrightarrow> 
             y = y'"
 show "llength (lfilter P xs) \<le> 1"
 proof -
 have f1: "\<forall>e ea. (e::enat) \<le> ea \<or> ea < e"
   by (meson not_le_imp_less)
 have f3: "llength (kfilter P 0 xs) = gen_llength 0 (lfilter P xs)"
   by (metis kfilter_llength llength_code)
 have f4: "enat 0 + llength xs = llength xs"
   by (metis gen_llength_def llength_code)
 have "llength xs = enat 0 + llength xs"
   by (metis (full_types) gen_llength_def llength_code)
 then have f5: "\<not> 1 < llength (lfilter P xs)"
  proof -
   have g1: "Suc 0 = 0 + Suc 0"
     by auto
   have g2: "\<And>i. enat i < llength (kfilter P 0 xs) \<Longrightarrow> 
                lnth xs (lnth (kfilter P 0 xs) i - 0) = lnth (lfilter P xs) i"
     using lfilter_kfilter[of _ P 0 xs] by auto
   have g3: "\<And>k. enat k < llength (kfilter P 0 xs) \<Longrightarrow> 
                lnth (kfilter P 0 xs) k - 0 = lnth (kfilter P 0 xs) k"
     using kfilter_mono[of _ P 0 xs] by auto
   have g4:  "\<And>n. enat n < llength (lfilter P xs) \<Longrightarrow> P (lnth (lfilter P xs) n)"
     using lfilter_lnth_aa[of _ P xs]   by auto
   have g5: "enat (0 + Suc 0) = 1" 
     using one_enat_def by auto
   have "\<not> 1 < gen_llength 0 (lfilter P xs) \<or> enat 0 < gen_llength 0 (lfilter P xs)"
     using zero_enat_def by fastforce
   then show ?thesis
     by (metis g1 g2 g3 g4 g5 a1 f3 f4 kfilter_upperbound ldistinct_conv_lnth ldistinct_kfilter 
         llength_code n_not_Suc_n)
   qed
 show ?thesis 
 using f5 not_le by blast
 qed
qed

lemma lfilter_llength_one_conv_b:
assumes " \<exists> k < llength xs. P (lnth xs k) \<and> 
           (\<forall> j < llength xs. j\<noteq>k \<longrightarrow> \<not> P (lnth xs j))"
shows " llength(lfilter P xs) =1 "
proof -
 have 1: "llength(lfilter P xs) \<le>1"
   by (metis assms exist_one_lfilter_llength_one)
 have 2: "0<llength(lfilter P xs)"
   by (metis assms gr_zeroI in_lset_conv_lnth llength_eq_0 lnull_lfilter) 
 show ?thesis
   by (metis "1" "2" One_nat_def Suc_ile_eq dual_order.antisym one_enat_def zero_enat_def) 
qed
              
lemma lfilter_llength_one_conv:
shows "(\<exists> k < llength xs. P (lnth xs k) \<and> 
           (\<forall> j < llength xs. j\<noteq>k \<longrightarrow> \<not> P (lnth xs j))) \<longleftrightarrow>
          llength(lfilter P xs) =1 "
using lfilter_llength_one_conv_a[of P xs] lfilter_llength_one_conv_b[of xs P] by blast 

lemma lfilter_llength_one_conv_1:
shows "(\<exists> k < llength xs. P (lnth xs k) \<and> 
           (\<forall> j < llength xs. j<k \<or> k<j \<longrightarrow> \<not> P (lnth xs j))) \<longleftrightarrow>
          llength(lfilter P xs) =1 "
using lfilter_llength_one_conv[of xs P] lfilter_llength_one_conv_c[of xs P]
by blast

lemma lfilter_llength_one_conv_2:
shows "( \<exists> k < llength xs. P (lnth xs k) \<and> 
      (\<forall> j . j<k  \<longrightarrow> \<not> P (lnth xs j)) \<and>
      (\<forall> j < llength xs. k<j \<longrightarrow> \<not> P (lnth xs j))) \<longleftrightarrow>
          llength(lfilter P xs) =1 "
using lfilter_llength_one_conv_1[of xs P] lfilter_llength_one_conv_d[of xs P]
by blast

lemma lfilter_lappend_ltake:
 assumes " k < llength xs"
 shows   " lfilter P (ltake k xs) = ltake (llength(lfilter P (ltake k xs))) (lfilter P xs)"
proof -
 have 1: " lfilter P xs = 
           lappend (lfilter P (ltake (the_enat k) xs)) (lfilter P (ldropn (the_enat k) xs)) " 
   by (metis enat_ord_code(4) lappend_ltake_enat_ldropn lfilter_lappend_lfinite lfinite_ltake)
 have 2: "ltake (llength(lfilter P (ltake k xs))) 
                (lappend (lfilter P (ltake (the_enat k) xs)) (lfilter P (ldropn (the_enat k) xs))) =
           lfilter P (ltake k xs) "
   by (metis assms enat_the_enat lfilter_idem lfinite_ltake llength_eq_infty_conv_lfinite 
       llength_lfilter_ile llength_ltake ltake_all ltake_lappend1 min.strict_order_iff) 
 show ?thesis using 1 2 by simp
qed

lemma kfilter_lappend_lfinite:
 "lfinite xs \<Longrightarrow>
    kfilter P n (lappend xs ys) = 
    lappend (kfilter P n xs) (kfilter P (n+ (the_enat(llength xs))) ys)"
unfolding kfilter_def
proof (induct arbitrary: n rule: lfinite.induct)
case lfinite_LNil
then show ?case by simp
next
case (lfinite_LConsI xs x)
then show ?case
  proof -
   have 1: "(lzip (lappend (LCons x xs) ys) (iterates Suc n)) =
            (LCons (x , n) (lzip (lappend xs ys) (iterates Suc (Suc n)))) "
     by (simp add: lzip.ctr(2))
   have 3: "lmap snd (lfilter (P \<circ> fst) (lzip (lappend (LCons x xs) ys) (iterates Suc n))) =
            (if P x then (LCons n (lmap snd (lfilter (P \<circ> fst) (lzip (lappend xs ys) (iterates Suc (Suc n))))))
                    else lmap snd (lfilter (P \<circ> fst) (lzip (lappend xs ys) (iterates Suc (Suc n))))) "
     using "1" by auto 
   have 4: "(if P x then (LCons n (lmap snd (lfilter (P \<circ> fst) (lzip (lappend xs ys) (iterates Suc (Suc n))))))
                    else lmap snd (lfilter (P \<circ> fst) (lzip (lappend xs ys) (iterates Suc (Suc n))))) =
            (if P x then 
              (LCons n (lappend (lmap snd (lfilter (P \<circ> fst) (lzip xs (iterates Suc (Suc n))))) 
                                (lmap snd (lfilter (P \<circ> fst) (lzip ys (iterates Suc ((Suc n) + the_enat (llength xs))))))))
             else (lappend (lmap snd (lfilter (P \<circ> fst) (lzip xs (iterates Suc (Suc n))))) 
                           (lmap snd (lfilter (P \<circ> fst) (lzip ys (iterates Suc ((Suc n) + the_enat (llength xs))))))))"
     using lfinite_LConsI.hyps(2) by auto 
   have 5: "(lmap snd (lfilter (P \<circ> fst) (lzip (LCons x xs) (iterates Suc n)))) =
            (if P x then (LCons n (lmap snd (lfilter (P \<circ> fst) (lzip ( xs) (iterates Suc (Suc n))))))
                    else (lmap snd (lfilter (P \<circ> fst) (lzip ( xs) (iterates Suc (Suc n))))))" 
              by (simp add: lzip.ctr(2))
   have 6: "(lmap snd (lfilter (P \<circ> fst) (lzip ys (iterates Suc (n + the_enat (llength (LCons x xs))))))) =
            (lmap snd (lfilter (P \<circ> fst) (lzip ys (iterates Suc ((Suc n) + the_enat (llength (xs)))))))" 
     using eSuc_enat lfinite_LConsI.hyps(1) llength_eq_infty_conv_lfinite by force
   have 7: "lappend (lmap snd (lfilter (P \<circ> fst) (lzip (LCons x xs) (iterates Suc n))))
            (lmap snd (lfilter (P \<circ> fst) (lzip ys (iterates Suc (n + the_enat (llength (LCons x xs))))))) =
            lappend (if P x then (LCons n (lmap snd (lfilter (P \<circ> fst) (lzip xs (iterates Suc (Suc n))))))
                            else (lmap snd (lfilter (P \<circ> fst) (lzip xs (iterates Suc (Suc n))))))
                    (lmap snd (lfilter (P \<circ> fst) (lzip ys (iterates Suc ((Suc n) + ( (the_enat (llength  xs))))))))
                        " 
     using "5" "6" by auto
   show ?thesis using "3" "4" "7" by auto
 qed           
qed

lemma kfilter_lappend_ltake:
 assumes " k < llength xs"
 shows   " kfilter P n (ltake k xs) = ltake (llength(kfilter P n (ltake k xs))) (kfilter P n xs)"
proof -
 have 1: " kfilter P n xs = lappend (kfilter P n (ltake (the_enat k) xs)) 
                                    (kfilter P (n+ (the_enat k)) (ldropn (the_enat k) xs)) " 
   using kfilter_lappend_lfinite[of "(ltake (the_enat k) xs)" P n "(ldropn (the_enat k) xs)" ]
   by (metis assms enat_iless enat_ord_code(4) lappend_ltake_ldrop ldrop_enat lfinite_ltake 
       llength_ltake min.strict_order_iff neq_iff the_enat.simps) 
 have 2: "ltake (llength(kfilter P n (ltake k xs))) 
           (lappend (kfilter P n (ltake (the_enat k) xs)) 
                    (kfilter P (n+ (the_enat k)) (ldropn (the_enat k) xs))) =
            kfilter P n (ltake k xs)"
   by (metis assms enat_the_enat lfinite_ltake llength_eq_infty_conv_lfinite llength_ltake 
       ltake_all ltake_lappend1 min.strict_order_iff order_refl)
 show ?thesis using 1 2 by simp
qed

lemma lfilter_ldropn_llength:
 assumes "k < llength xs"
 shows " llength (lfilter P ( (ldropn k xs))) \<le> llength (lfilter P ( xs))"
using assms 
proof (induct k arbitrary: xs)
case 0
then show ?case by simp
next
case (Suc k)
then show ?case 
 proof (cases xs)
 case LNil
 then show ?thesis by simp
 next
 case (LCons x21 x22)
 then show ?thesis using Suc Suc_ile_eq by auto
  (meson Suc_ile_eq dual_order.trans ile_eSuc)
 qed
qed

lemma lfilter_nlength_imp:
 shows   " llength (lfilter (\<lambda>x. P x \<and> Q x) xs) \<le> llength (lfilter P xs)"
proof -
 have 0: "lfilter (\<lambda>x. P x \<and> Q x) xs = lfilter (\<lambda>x. Q x \<and> P x) xs" 
   by meson
 have 1: "lfilter (\<lambda>x. Q x \<and> P x) xs = lfilter Q (lfilter P xs)"
   using lfilter_lfilter[of Q P xs] by auto
 have 2: "llength (lfilter Q (lfilter P xs)) \<le> llength (lfilter P xs)"
   using llength_lfilter_ile by blast 
 show ?thesis by (simp add: 1 2 0)
qed

lemma lnths_kfilter_lfilter:
 " lnths xs (lset(kfilter P 0 xs)) = lfilter P xs"
using lfilter_conv_lnths[of P xs]  kfilter_lset[of P 0 xs]
by simp 

subsection \<open>lrev\<close>

lemma lnull_lrev[simp]:
 "lnull (lrev xs) = (lnull xs)" 
unfolding lrev_def 
by (metis (full_types) llist_of_list_of lnull_llist_of rev.simps(1) rev_rev_ident)


lemma lrev_LNil[simp]:
 "lrev LNil = LNil" 
unfolding lrev_def 
by simp

lemma lrev_LCons[simp]:
" lrev (LCons x xs) = (if lfinite xs then (lappend (lrev xs) (LCons x LNil)) else (LCons x xs)) " 
unfolding lrev_def
by simp
 (metis lappend_llist_of_llist_of llist_of.simps(1) llist_of.simps(2))

lemma ltl_lrev[simp]: 
" \<not>lnull xs \<Longrightarrow> 
ltl (lrev xs) = 
 (if lfinite xs then  
     (if (\<exists>a. xs = (LCons a LNil)) then LNil else lappend (ltl (lrev (ltl xs))) (LCons (lhd xs) LNil))
  else ltl xs) " 
using ltl_lappend[of "(lrev (ltl xs))" "(LCons (lhd xs) LNil)" ] 
by (metis lfinite_ltl lhd_LCons_ltl llist.collapse(1) llist.disc(1) lnull_lrev lrev_LCons ltl_simps(2))

lemma lmap_lrev:
"lmap f (lrev xs) = lrev (lmap f xs) " 
unfolding lrev_def 
by (simp add: rev_map)

lemma lfinite_lrev [simp]:
 " lfinite (lrev xs) = lfinite xs" 
by (simp add:  lrev_def)

lemma lset_lrev:
 "lset (lrev xs) = lset xs" 
unfolding lrev_def
by simp

lemma llength_lrev[simp]: 
 "llength (lrev xs) = llength xs " 
unfolding lrev_def
by (simp add: length_list_of)

lemma lrev_llist_of [simp]: 
 " lrev (llist_of xs) = llist_of (rev xs) " 
unfolding lrev_def
by simp

lemma list_of_lrev [simp]: 
" lfinite xs \<Longrightarrow> list_of (lrev xs) = rev (list_of xs) " 
unfolding lrev_def
by simp

lemma lnth_lrev: 
 assumes "lfinite xs "
         " (enat i) < llength xs "
   shows"  (lnth (lrev xs) i) = (lnth xs ((the_enat (llength xs)) - (Suc i))) " 
using assms
unfolding lrev_def 
by simp
 (metis enat_ord_simps(2) length_list_of length_list_of_conv_the_enat nth_list_of rev_nth)

lemma lappend_lrev_lfinite: 
assumes "lfinite xs"
        "lfinite ys"
shows "lrev (lappend xs ys) = lappend (lrev ys) (lrev xs)"
using assms 
unfolding lrev_def 
by (simp add: lappend_llist_of_llist_of list_of_lappend)

lemma lappend_lrev_infa: 
assumes "lfinite xs"
        "\<not>lfinite ys"
shows "lrev (lappend xs ys) = lappend xs ys" 
using assms 
unfolding lrev_def by auto

lemma lappend_lrev__infb: 
assumes "\<not>lfinite xs"
shows "lrev (lappend xs ys) = lappend (lrev xs) (lrev ys)" 
using assms 
unfolding lrev_def by (simp add: lappend_inf)

lemma lrev_lrev: 
 "lrev (lrev xs) = xs " 
by (simp add: lrev_def)

lemma lrev_ltake: 
 assumes "lfinite xs" 
          "enat k < llength xs" 
 shows " lrev (ltake k xs) = ldrop (the_enat(llength xs) -k) (lrev xs)"
using assms 
unfolding lrev_def
by simp
 (metis ldrop_llist_of length_list_of_conv_the_enat rev_take take_list_of)

lemma lrev_ldrop: 
 assumes "lfinite xs" 
          "enat k < llength xs" 
 shows " lrev (ldrop k xs) = ltake (the_enat(llength xs) -k) (lrev xs) "
using assms 
unfolding lrev_def 
by simp
 (metis drop_list_of ldrop_enat length_list_of_conv_the_enat rev_drop)

lemma lrev_lsubc: 
 assumes "lfinite xs" 
         " enat i \<le> j "
         "enat j < llength xs" 
 shows " lrev (lsubc i j xs) = lsubc  ((the_enat(llength xs)) -(j+1)) ((the_enat(llength xs)) - (i+1)) (lrev xs) " 
proof -
 have 1: "lrev (lsubc i j xs) = lrev (ltake (eSuc(j-i)) (ldrop i xs))" 
   unfolding lsubc_def  min_def using assms apply simp
    by (metis assms(2) co.enat.exhaust_sel iless_Suc_eq not_less_zero order_le_less_trans) 
 have 20: " i<\<infinity>" 
     by simp
 have 21: "(llength (ldrop (enat i) xs)) = llength xs - i"
       by (simp add: ldrop_enat) 
 have 22: " (the_enat (llength xs - i)) - (j -i +1) =  (llength xs) -eSuc j "
    using assms apply simp 
    using eSuc_enat llength_eq_infty_conv_lfinite by fastforce 
 have 23: "(enat (the_enat (llength (ldrop (enat i) xs)) - (j - i + 1))) = 
            (llength xs) - eSuc j "
     using "21" "22" by presburger
 have 2: "lrev (ltake (eSuc(j-i)) (ldrop i xs)) = 
           ldrop ((llength xs) - eSuc j) (lrev (ldrop (enat i) xs))" 
   using lrev_ltake[of "(ldrop i xs)" "(j-i)+1" ]  using assms 23 apply auto
       by (metis "21" "22" add.commute diff_is_0_eq' eSuc_enat enat_ord_simps(1) enat_the_enat ldrop_enat ldropn_0 lfinite_ldrop llength_eq_infty_conv_lfinite ltake_all not_le_imp_less plus_1_eq_Suc)
 have 3: "(lrev (ldrop (enat i) xs)) = (ltake ((llength xs) - i) (lrev xs))"
      using lrev_ldrop[of xs i] using assms  order_le_less_trans 
      by (metis enat_the_enat idiff_enat_enat llength_eq_infty_conv_lfinite)
 have 30: "eSuc j \<le> llength xs"
     using assms(3) ileI1 by blast
 have 31: "llength (lrev (ldrop (enat i) xs)) = llength xs - i" 
    by (simp add: "21")
 have 32: " llength (ldrop  ((llength xs) - eSuc j) (lrev (ldrop (enat i) xs))) = 
            ( llength xs - i) -  ((llength xs) - eSuc j) "
       by (metis "23" "31" ldrop_enat llength_ldropn) 
 have 33: "( llength xs - i) -  ((llength xs) - eSuc j) = (eSuc j) - i"
     by (metis "1" "2" "32" add.commute assms(2) assms(3) eSuc_enat enat_ord_simps(1) idiff_enat_enat llength_lrev llength_lsubc ordered_cancel_comm_monoid_diff_class.diff_add_assoc2 plus_1_eq_Suc) 

 have 4: "ldrop  ((llength xs) - eSuc j) (lrev (ldrop (enat i) xs)) =
          ldrop ((llength xs)  - eSuc j) ( (ltake ((llength xs) - i) (lrev xs)))"    
    using 3 by presburger
 have 40: "(min (enat (the_enat (llength xs) - (j+1))) (epred (llength (lrev xs)))) =
            ((llength xs) - eSuc j)  " 
     by (metis add.commute assms(1) diff_diff_left eSuc_plus_1 enat_less_imp_le enat_ord_simps(2) epred_enat idiff_enat_enat less_imp_diff_less lfinite_llength_enat llength_lrev min.absorb1 one_enat_def plus_enat_simps(1) the_enat.simps)
 have 41: "min (enat (the_enat (llength xs) - (j + 1))) (epred (llength (lrev xs))) =
           (enat (the_enat (llength xs) - (j + 1))) " 
     by (metis "40" add.commute assms(1) eSuc_enat enat_the_enat idiff_enat_enat llength_eq_infty_conv_lfinite plus_1_eq_Suc)
 have 42: "eSuc (enat (the_enat (llength xs) - (i + 1)) - enat (the_enat (llength xs) - (j + 1))) =
           eSuc ( llength xs - (i+1) - (llength xs - (j+1)))" 
        using assms(1) llength_eq_infty_conv_lfinite by fastforce
 have 43: "eSuc ( llength xs - (i+1) - (llength xs - (j+1))) =
            eSuc ( j -i)"
         using "30" assms(1) eSuc_enat llength_eq_infty_conv_lfinite by fastforce 
 have 44: "eSuc ( j -i) + (enat (the_enat (llength xs) - (j + 1))) =
              llength xs -  i" 
      using "30" assms(1) assms(2) eSuc_enat llength_eq_infty_conv_lfinite by fastforce
 have 45: "(eSuc (enat (the_enat (llength xs) - (i + 1)) - enat (the_enat (llength xs) - (j + 1))) +
        (enat (the_enat (llength xs) - (j + 1))) ) =
          llength xs -  i " 
   using "42" "43" "44" by presburger
 have 5: "ldrop ((llength xs)  - eSuc j) ( (ltake ((llength xs) -  i) (lrev xs))) =
           lsubc  ((the_enat(llength xs)) -  (j+1)) ((the_enat(llength xs)) - (i+1)) (lrev xs)"
     unfolding lsubc_def ltake_ldrop
     using "40" "41" "42" "43" "44" by presburger
  show ?thesis 
  using "1" "2" "4" "5" by presburger 
qed

lemma llast_lrev: 
assumes "\<not> lnull xs" 
shows "llast (lrev xs) = (if lfinite xs then lfirst xs else llast xs  ) "
using assms
unfolding lrev_def 
by (simp add: last_rev lfirst_def)

lemma lfirst_lrev: 
assumes "\<not> lnull xs"
shows " lfirst (lrev xs)  = (if lfinite xs then llast xs else lfirst xs) "
using assms
unfolding lrev_def 
by simp
 (metis hd_rev lfirst_def lhd_llist_of llast_llist_of llist_of_list_of)

lemma llist_all2_lrevI:
 "llist_all2 P xs ys \<Longrightarrow> llist_all2 P (lrev xs) (lrev ys) " 
unfolding lrev_def
by simp
 (metis llist_all2_lfiniteD llist_all2_llist_of llist_of_list_of)

lemma llist_all2_lrevD_lfinite: 
assumes " llist_all2 P (lrev xs) (lrev ys) "
        "lfinite xs " 
        "lfinite ys" 
shows"  llist_all2 P xs ys" 
using assms
unfolding lrev_def
by simp
 (metis llist_all2_llist_of llist_of_list_of)

lemma llist_all2_lrevD_not_lfinite: 
assumes " llist_all2 P (lrev xs) (lrev ys) "
        "\<not>lfinite xs " 
        "\<not>lfinite ys" 
shows"  llist_all2 P xs ys" 
using assms by (simp add: lrev_def)

lemma llist_all2_lrevD: 
assumes " llist_all2 P (lrev xs) (lrev ys) "
shows"  llist_all2 P xs ys" 
by (metis assms lfinite_lrev llist_all2_lfiniteD llist_all2_lrevD_lfinite llist_all2_lrevD_not_lfinite)

lemma  llist_all2_lrev:
 "llist_all2 P (lrev xs) (lrev ys) \<longleftrightarrow> llist_all2 P xs ys" 
using llist_all2_lrevD llist_all2_lrevI by blast



subsection \<open>Transfer rules\<close>

context includes lifting_syntax
begin

lemma lbutlast_transfer [transfer_rule]:
 " (llist_all2 A ===> llist_all2 A) lbutlast lbutlast"
by (auto simp add: rel_fun_def lbutlast_conv_ltake llist_all2_llengthD llist_all2_ltakeI)

lemma lleast_transfer [transfer_rule]:
 " ((A ===> (=)) ===> llist_all2 A ===> (=)) lleast lleast "
unfolding lleast_def[abs_def] 
by (auto simp add: rel_fun_def)
   (metis (full_types, opaque_lifting) llist_all2_conv_all_lnth)

lemma lfuse_transfer [transfer_rule]:
  "(llist_all2 A ===> llist_all2 A ===> llist_all2 A) lfuse lfuse"
by(auto simp add: rel_fun_def intro: llist_all2_lfuseI)

lemma ridx_transfer [transfer_rule]: 
  "((R ===> R ===> (=)) ===> llist_all2 R ===> (=)) ridx ridx " 
by (simp add: llist_all2_rsp rel_fun_def ridx_def llist_all2_conv_all_lnth)
   (meson Suc_ile_eq order_less_imp_le)

lemma lsub_transfer [transfer_rule]:
 " ( (=) ===> (=) ===> llist_all2 A ===> llist_all2 A) lsub lsub " 
by (auto simp add: lsub_def rel_fun_def intro: llist_all2_ltakeI llist_all2_ldropI) 

lemma lsubc_transfer [transfer_rule]:
 " ( (=) ===> (=) ===> llist_all2 A ===> llist_all2 A) lsubc lsubc " 
by (auto simp add: lsubc_def rel_fun_def min_def llist_all2_llengthD 
         intro: llist_all2_ltakeI llist_all2_ldropI) 

lemma lfusecat_transfer [transfer_rule]:
  "(llist_all2 (llist_all2 A) ===> llist_all2 A) lfusecat lfusecat"
by(auto intro: llist_all2_lfusecatI)

lemma lrev_parametric [transfer_rule]:
  shows "(llist_all2 A ===> llist_all2 A) lrev lrev"
by (rule rel_funI)(rule llist_all2_lrevI)

end

end
