(*  Title     : A Shallow Encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

section \<open>Interval Temporal  Algebra\<close>  

text \<open>
We have given an algebraic axiom system for Interval Temporal Logic: Interval Temporal Algebra. 
The axiom system is a combination of a variant of Kleene algebra and Omega algebra plus axioms for 
linearity and conﬂuence.  Kleene algebra and Omega algebra have been defined by Alasdair Armstrong,
Georg Struth and Tjark Weber in \cite{Georg:Klealg}.  
\<close>

theory ITA
imports Semantics Chopstar Omega
begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="
  cvc4 z3 e spass vampire
"]
*)

subsection \<open>Definition of Set of intervals and Operations on them\<close>

type_synonym 'a iintervals = "'a nellist set "

definition lan:: "('a:: world) formula \<Rightarrow> 'a iintervals"
where "lan f = { \<sigma> . (\<sigma> \<Turnstile> f) }" 
 
definition fusion :: "'a iintervals \<Rightarrow> 'a iintervals \<Rightarrow> 'a iintervals" (infixl "\<cdot>" 70)
where "X\<cdot>Y = { \<sigma> . 
        (\<exists> \<sigma>1 \<sigma>2. \<sigma> = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and>
         (\<sigma>1 \<in> X) \<and> (\<sigma>2  \<in> Y)  \<and> (nlast \<sigma>1 = nfirst \<sigma>2)) 
        \<or> (\<not> nfinite \<sigma> \<and> \<sigma> \<in> X) }"

definition sempty :: "'a iintervals" ("SEmpty") 
where
  "SEmpty \<equiv> { \<sigma> .  nlength \<sigma> = 0  }"

definition smore :: "'a iintervals" ("SMore")
where
  "SMore \<equiv>  - SEmpty"

definition sskip :: "'a iintervals" ("SSkip")
where
  "SSkip \<equiv> -(SEmpty \<union> (SMore\<cdot>SMore))"

definition sfalse :: "'a iintervals" ("SFalse")
where
  "SFalse \<equiv> {}" 

definition strue :: "'a iintervals" ("STrue")
where
  "STrue \<equiv> -{}" 

definition sinit :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SInit _)" [85] 85)
where
  "SInit X \<equiv> (X \<inter> SEmpty)\<cdot>STrue"

definition sinf :: "'a iintervals" ("SInf")
where " SInf \<equiv> STrue \<cdot> SFalse "

definition sfinite ::  "'a iintervals" ("SFinite")
where " SFinite \<equiv> - SInf "

definition sfmore :: "'a iintervals" ("SFMore")
where " SFMore \<equiv> SFinite \<inter> SMore " 

definition sfin :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SFin _)" [85] 85)
where
  "SFin X \<equiv> SFinite\<cdot>(X \<inter> SEmpty)"

definition ssometime :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SSometime _)" [85] 85)
where
  "SSometime X \<equiv> SFinite\<cdot>X"

definition salways :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SAlways _)" [85] 85)
where
  "SAlways X \<equiv> -(SSometime (-X))"

definition sdi :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SDi _)" [85] 85)
where
  "SDi X \<equiv> X\<cdot>STrue"

definition sbi :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SBi _)" [85] 85)
where
  "SBi X \<equiv> -(SDi (-X))"

definition sda :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SDa _)" [85] 85)
where
  "SDa X \<equiv> SFinite\<cdot>X\<cdot>STrue"

definition sba :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SBa _)" [85] 85)
where
  "SBa X \<equiv> -(SDa (-X))"

definition snext :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SNext _)" [85] 85)
where 
  "SNext X \<equiv> SSkip\<cdot>X" 

definition swnext :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SWnext _)" [85] 85)
where 
  "SWnext X \<equiv> (-(SSkip\<cdot>(-X)))" 
 
definition sprev :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SPrev _)" [85] 85)
where 
  "SPrev X \<equiv> X\<cdot>SSkip" 

definition swprev :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SWprev _)" [85] 85)
where 
  "SWprev X \<equiv> (-((-X)\<cdot>SSkip))"
 
primrec spower :: "'a iintervals \<Rightarrow> nat \<Rightarrow> 'a iintervals" ("(SPower _ _)" [88,88] 87)
where
   pwr_0  : "SPower X 0       = SEmpty"
 | pwr_Suc: "SPower X (Suc n) = ((X \<inter> SFinite)\<cdot>(SPower X n))"

definition sfpowerstar :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SFPowerstar _)" [85] 85)
where
  "SFPowerstar X \<equiv> (\<Union> n. SPower X n) "

definition spowerstar :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SPowerstar _)" [85] 85)
where
  "SPowerstar X \<equiv> (SFPowerstar X)\<cdot>(SEmpty \<union> (X \<inter> SInf))"

definition sstar :: "'a iintervals \<Rightarrow> 'a iintervals" ("(SStar _)" [85] 85)
where
  "SStar X \<equiv> SPowerstar(X \<inter> SMore)" 


subsection \<open>Simplification Lemmas\<close>

lemma snot_elim :
 "x \<in> -X \<longleftrightarrow> x \<notin> X "
by simp

lemma sor_elim :
 "x\<in> (X \<union> Y) \<longleftrightarrow> (x \<in> X \<or> x\<in>Y) "
by simp

lemma sand_elim :
  "x\<in> (X \<inter> Y) \<longleftrightarrow> (x \<in> X \<and> x\<in>Y) "
by simp

lemma sfalse_elim :
 " \<sigma> \<notin> SFalse"
by (simp add: sfalse_def)

lemma strue_elim :
 " \<sigma> \<in> STrue "
by (simp add: strue_def)

lemma sempty_elim :
    " \<sigma> \<in> SEmpty \<longleftrightarrow> ( nlength \<sigma> = 0 )"
by (simp add: sempty_def)

lemma smore_elim :
    " \<sigma> \<in> SMore \<longleftrightarrow> 
      ( nlength \<sigma> > 0 )"
by (simp add: sempty_elim smore_def)

lemma fusion_iff:
    "\<sigma> \<in> X\<cdot>Y \<longleftrightarrow> 
     (
      (\<exists> \<sigma>1 \<sigma>2. \<sigma> = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and>
         (\<sigma>1 \<in> X) \<and> (\<sigma>2  \<in> Y)  \<and> (nlast \<sigma>1 = nfirst \<sigma>2)) 
        \<or> (\<not> nfinite \<sigma> \<and> \<sigma> \<in> X) ) 
 "
by (unfold fusion_def) auto

lemma fusion_iff_1:
    "\<sigma> \<in> X\<cdot>Y \<longleftrightarrow> 
    ( (\<exists>n \<le> nlength \<sigma>.  ( (ntaken n \<sigma>) \<in> X ) \<and> ( (ndropn n \<sigma>) \<in> Y) )
         \<or> (\<not> nfinite \<sigma> \<and> \<sigma> \<in> X)
       )"
using  fusion_iff[of "\<sigma>" "X" "Y"] nfuse_ntaken_ndropn[of _ \<sigma>]
by (simp add: chop_nfuse_2)

lemma smore_fusion_smore :
 " \<sigma> \<in> (SMore\<cdot>SMore) \<longleftrightarrow> ( (enat 1) < nlength \<sigma>)"
using fusion_iff_1[of \<sigma> SMore SMore] 
proof  (auto simp add:  smore_elim )
  show "\<And>n na.
       \<sigma> \<in> SMore \<cdot> SMore \<Longrightarrow>
       enat n \<le> nlength \<sigma> \<Longrightarrow>
       enat n \<noteq> 0 \<Longrightarrow>
       nlength \<sigma> - enat n \<noteq> 0 \<Longrightarrow> enat na \<le> nlength \<sigma> \<Longrightarrow> enat na \<noteq> 0 \<Longrightarrow> 
       nlength \<sigma> \<noteq> 0 \<Longrightarrow> nlength \<sigma> - enat na \<noteq> 0 \<Longrightarrow> 
       enat (Suc 0) < nlength \<sigma>" 
    by (metis One_nat_def antisym_conv3 enat_ord_code(4) idiff_self ileI1 le_less_trans less_le 
        not_iless0 one_eSuc one_enat_def)
  show "\<And>n. \<sigma> \<in> SMore \<cdot> SMore \<Longrightarrow> 
            enat n \<le> nlength \<sigma> \<Longrightarrow> 
            enat n \<noteq> 0 \<Longrightarrow> nlength \<sigma> - enat n \<noteq> 0 \<Longrightarrow> 
            \<not> nfinite \<sigma> \<Longrightarrow>
            nlength \<sigma> \<noteq> 0 \<Longrightarrow> 
           enat (Suc 0) < nlength \<sigma>" 
    by (metis Suc_ile_eq i0_less nlength_eq_enat_nfiniteD order.not_eq_order_implies_strict 
        zero_enat_def)
  show "\<And>n. \<sigma> \<in> SMore \<cdot> SMore \<Longrightarrow> 
            \<not> nfinite \<sigma> \<Longrightarrow> 
            enat n \<le> nlength \<sigma> \<Longrightarrow>  
            enat n \<noteq> 0 \<Longrightarrow> 
            nlength \<sigma> \<noteq> 0 \<Longrightarrow>
            nlength \<sigma> - enat n \<noteq> 0 \<Longrightarrow> 
          enat (Suc 0) < nlength \<sigma>" 
   by (metis Suc_ile_eq i0_less nlength_eq_enat_nfiniteD order.not_eq_order_implies_strict zero_enat_def)
 show "\<sigma> \<in> SMore \<cdot> SMore \<Longrightarrow> \<not> nfinite \<sigma> \<Longrightarrow> nlength \<sigma> \<noteq> 0 \<Longrightarrow> enat (Suc 0) < nlength \<sigma>" 
   by (metis One_nat_def ileI1 linorder_cases nlength_eq_enat_nfiniteD not_less_zero one_eSuc 
       one_enat_def order.not_eq_order_implies_strict)
 show "\<sigma> \<notin> SMore \<cdot> SMore \<Longrightarrow> 
       enat (Suc 0) < nlength \<sigma> \<Longrightarrow> 
       \<forall>n. enat n \<le> nlength \<sigma> \<longrightarrow> enat n = 0 \<or> nlength \<sigma> = 0 \<or> nlength \<sigma> - enat n = 0 \<Longrightarrow> 
       nfinite \<sigma> \<Longrightarrow>
       False " 
 proof -
  assume a1: "enat (Suc 0) < nlength \<sigma>"
  assume "nfinite \<sigma>"
  assume a2: "\<forall>n. enat n \<le> nlength \<sigma> \<longrightarrow> enat n = 0 \<or> nlength \<sigma> = 0 \<or> nlength \<sigma> - enat n = 0"
  have "enat (Suc 0) = 1"
    using One_nat_def one_enat_def by presburger
  then have f3: "enat 1 < nlength \<sigma>"
    using a1 one_enat_def by presburger
  have f4: "enat 1 \<le> enat 1"
    by blast
  have "enat 1 \<noteq> \<infinity>"
    by blast
  then show False
    using f4 f3 a2
    by (metis (no_types) dual_order.strict_iff_order enat_diff_cancel_left 
       gr_implies_not_zero idiff_self one_enat_def zero_neq_one)
  qed
qed

lemma sskip_elim :
   " \<sigma> \<in> SSkip \<longleftrightarrow> 
    (nlength \<sigma> = 1)"
using sskip_def smore_fusion_smore
by (metis One_nat_def Suc_ile_eq less_numeral_extra(4) one_enat_def order.not_eq_order_implies_strict 
    sempty_elim smore_def smore_elim snot_elim sor_elim zero_enat_def zero_one_enat_neq(1)) 

lemma sinfinite_elim:
 " \<sigma> \<in> SInf \<longleftrightarrow> 
   (\<not> nfinite \<sigma>)"
by (simp add: fusion_iff_1 sfalse_elim sinf_def strue_elim)

lemma sfinite_elim:
 " \<sigma> \<in> SFinite \<longleftrightarrow> 
   (nfinite \<sigma>)"
by (simp add: sfinite_def sinfinite_elim )

lemma ffusion_iff:
    "\<sigma> \<in> (X \<inter> SFinite)\<cdot>Y \<longleftrightarrow> 
     (
      (\<exists> \<sigma>1 \<sigma>2. \<sigma> = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and> 
         (\<sigma>1 \<in> X) \<and> (\<sigma>2  \<in> Y)  \<and> (nlast \<sigma>1 = nfirst \<sigma>2)) 
      ) "
unfolding fusion_def 
by (simp add: sfinite_elim) blast

lemma ffusion_iff_1:
    "\<sigma> \<in> (X \<inter> SFinite)\<cdot>Y \<longleftrightarrow> 
    (( (\<exists>n \<le> nlength \<sigma>.  ( (ntaken n \<sigma>) \<in> X ) \<and> ( (ndropn n \<sigma>) \<in> Y))
       )
  )"
using  fusion_iff_1[of "\<sigma>" "X \<inter> SFinite" Y ] 
by (meson nfinite_ntaken sand_elim sfinite_elim)

lemma sfmore_elim:
  " \<sigma> \<in> SFMore \<longleftrightarrow> 
    (nfinite \<sigma> \<and> 0 < nlength \<sigma>) "
by (simp add: sfinite_elim sfmore_def smore_elim )

lemma spower_elim_zero :
    " \<sigma> \<in> SPower X 0 \<longleftrightarrow> \<sigma> \<in> SEmpty  "
by simp

lemma spower_elim_suc :
    " \<sigma> \<in> SPower X (Suc n) \<longleftrightarrow> \<sigma> \<in> (X \<inter> SFinite)\<cdot>(SPower X n)"
by simp
  
lemma spower_elim_suc_1 :
    " \<sigma> \<in> (X \<inter> SFinite)\<cdot>(SPower X n) \<longleftrightarrow>
     ( (\<exists> \<sigma>1 \<sigma>2. \<sigma> = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and> 
            \<sigma>1 \<in> X  \<and> \<sigma>2  \<in> (SPower X n)  \<and>
             nlast \<sigma>1 = nfirst \<sigma>2 )
     )"
by (simp add: ffusion_iff)
    
lemma ffusionimp:
assumes "  Y0 \<subseteq> Y1"
 shows  " \<sigma> \<in> (X \<inter> SFinite)\<cdot>Y0  \<longrightarrow> \<sigma> \<in> (X \<inter> SFinite)\<cdot>Y1"
using assms unfolding ffusion_iff_1 by (auto)

lemma spower_finite:
  "  (SPower X n) \<subseteq> SFinite"
proof (induct n)
case 0
then show ?case
  by (metis nlength_eq_enat_nfiniteD sempty_elim sfinite_elim spower.simps(1) subsetI zero_enat_def) 
next
case (Suc n)
then show ?case 
  proof -
   have 1: " (SPower X (Suc n)) = (X \<inter> SFinite)\<cdot>(SPower X n) " 
     by simp
   have 2: " SPower X n \<subseteq> SFinite"
     using Suc.hyps by blast
   have 3: " (X \<inter> SFinite)\<cdot> SFinite \<subseteq> SFinite" 
     using ffusion_iff_1[ of _ X SFinite] 
     by (simp add: sfinite_elim) 
        (metis ComplI sfinite_def sinfinite_elim subsetI)
   have 4: " (X \<inter> SFinite)\<cdot>(SPower X n) \<subseteq> (X \<inter> SFinite)\<cdot> SFinite" 
     using Suc.hyps ffusionimp by blast
   show ?thesis 
     using "3" "4" by auto
 qed  
qed

lemma sfpowerstar_elim:
  "\<sigma> \<in> SFPowerstar X \<longleftrightarrow> (\<exists>n. \<sigma> \<in> SPower X n)"
by (simp add: sfpowerstar_def)

lemma sfpowerstar_elim_1:
  "(\<exists>n. \<sigma> \<in> SPower X n) \<longleftrightarrow> ( \<sigma> \<in> SPower X 0 \<or> (\<exists> n. \<sigma> \<in> SPower X (Suc n)))"
by (metis not0_implies_Suc)

lemma sfpowerstar_suc:
  "(\<exists> n. \<sigma> \<in> SPower X (Suc n)) \<longleftrightarrow> (\<exists> n. \<sigma> \<in> (X \<inter> SFinite)\<cdot>(SPower X n))"
by simp

lemma sfpowerstar_suc_1:
  " (\<exists> n. \<sigma> \<in> (X \<inter> SFinite)\<cdot>(SPower X n)) \<longleftrightarrow> \<sigma> \<in> (X \<inter> SFinite)\<cdot>(SFPowerstar X)"
unfolding fusion_iff
by (cases \<sigma>) (auto simp add: sfpowerstar_elim)

lemma sfpowerstar_eqv_sem:
 "\<sigma> \<in> SFPowerstar X \<longleftrightarrow> ( \<sigma> \<in> SEmpty \<or> \<sigma> \<in> (X \<inter> SFinite)\<cdot>(SFPowerstar X))" 
by (simp add: sfpowerstar_elim sfpowerstar_elim_1 sfpowerstar_suc_1)

lemma sfpowerstar_eqv:
 "SFPowerstar X  = SEmpty \<union> (X \<inter> SFinite)\<cdot>(SFPowerstar X)"
using sfpowerstar_eqv_sem by blast

lemma sfpowerstar_eqv_1:
 " (\<Union> n. SPower X n) = SEmpty \<union> (X \<inter> SFinite)\<cdot>(\<Union> n. SPower X n)"
using sfpowerstar_eqv by (simp add: sfpowerstar_def)


subsection \<open>Algebraic Laws\<close>

subsubsection \<open>Commutative Additive Monoid\<close>

lemma UnionCommute:
 " (X::'a iintervals) \<union> Y = Y \<union> X"
by (simp add: Un_commute)

lemma UnionSFalse:
 " X \<union> SFalse = X"
by (simp add: sfalse_def)

lemma UnionAssoc:
 " (X::'a iintervals) \<union> (Y \<union> Z) = (X \<union> Y) \<union> Z"
by (simp add: sup_assoc) 


subsubsection \<open> Boolean algebra\<close>

lemma Huntington:
  " (X::'a iintervals) = -(-X \<union> -Y) \<union> -(-X \<union> Y) "
by auto

lemma Morgan:
 " (X::'a iintervals) \<inter> Y = -(-X \<union> -Y)"
by auto

\<comment> \<open>identities\<close> 

lemma STrueTop:
 " STrue = X \<union> -X"
by (simp add: strue_def)

lemma SFalseBottom:
 " SFalse = X \<inter> -X"
by (simp add: sfalse_def)


subsubsection \<open>multiplicative monoid\<close>

lemma FusionSEmptyLsem:
 "\<sigma> \<in> SEmpty \<cdot> X  \<longleftrightarrow> \<sigma> \<in> X"
using fusion_iff_1[of \<sigma> SEmpty X]
by (auto simp add:  fusion_iff_1 sempty_elim  nlength_eq_enat_nfiniteD zero_enat_def )
   (metis le_numeral_extra(3) min.bounded_iff min_enat_simps(3) ndropn_0 zero_enat_def,
    metis le_numeral_extra(3) min.bounded_iff min_enat_simps(3) ndropn_0 zero_enat_def)

lemma FusionSEmptyL : 
    "SEmpty \<cdot> X = X"
using set_eqI[of "SEmpty\<cdot>X" "X"]  FusionSEmptyLsem
 by auto

lemma FusionSEmptyRsem : 
    "\<sigma> \<in> X \<cdot> SEmpty  \<longleftrightarrow> \<sigma> \<in> X" 
using fusion_iff_1[of \<sigma> X SEmpty]  
by (auto simp add:  fusion_iff_1 sempty_elim )
   (metis is_NNil_ndropn le_numeral_extra(3) ndropn_0 ndropn_nlength ntaken_all zero_enat_def,
    metis add.right_neutral le_iff_add ndropn_all ndropn_nlength nfinite_nlength_enat nlength_NNil 
    ntaken_all)

lemma FusionSEmptyR : 
    "X \<cdot> SEmpty  = X"
using set_eqI[of "X\<cdot>SEmpty" "X"]  FusionSEmptyRsem by auto

lemma FusionAssocA:
assumes "x \<in> X \<cdot> (Y \<cdot> Z)"
shows    "x \<in> (X \<cdot> Y) \<cdot> Z"
proof -
 have 1: "(\<exists>\<sigma>1 \<sigma>2. x = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and>  \<sigma>1 \<in> X \<and>  \<sigma>2 \<in> Y \<cdot> Z \<and> nlast \<sigma>1 = nfirst \<sigma>2) 
          \<or> (\<not> nfinite x \<and> x \<in> X)"
   using assms  fusion_iff[of "x" "X" "Y \<cdot> Z"] by auto
 have 2: "\<not> nfinite x \<and> x \<in> X \<Longrightarrow> x \<in> (X \<cdot> Y) \<cdot> Z"
        by (simp add: fusion_iff)     
 have 3: "(\<exists>\<sigma>1 \<sigma>2. x = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and>  \<sigma>1 \<in> X \<and> \<sigma>2 \<in> Y \<cdot> Z \<and> nlast \<sigma>1 = nfirst \<sigma>2) \<Longrightarrow>
            x \<in> (X \<cdot> Y) \<cdot> Z "
   proof -
     assume a0: "(\<exists>\<sigma>1 \<sigma>2. x = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and>  \<sigma>1 \<in> X \<and> \<sigma>2 \<in> Y \<cdot> Z \<and> nlast \<sigma>1 = nfirst \<sigma>2)"
     show "x \<in> (X \<cdot> Y) \<cdot> Z"
      proof -
       obtain "\<sigma>1" "\<sigma>2" where 5: 
            "x = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and>  \<sigma>1 \<in> X \<and> \<sigma>2 \<in> Y \<cdot> Z \<and> nlast \<sigma>1 = nfirst \<sigma>2"
         using a0  by auto
       have 6: "(\<exists>\<sigma>3 \<sigma>4. \<sigma>2 = nfuse \<sigma>3 \<sigma>4 \<and> nfinite \<sigma>3 \<and> \<sigma>3 \<in> Y \<and> \<sigma>4 \<in> Z \<and> nlast \<sigma>3 = nfirst \<sigma>4) \<or>
                  (\<not> nfinite \<sigma>2 \<and>  \<sigma>2 \<in> Y)"
         using fusion_iff[of " \<sigma>2" Y Z]  5 by simp   
       have 7: "(\<not> nfinite \<sigma>2 \<and> \<sigma>2 \<in> Y) \<Longrightarrow> x \<in> (X \<cdot> Y) \<cdot> Z"
         by (metis "5" fusion_iff ndropn_nfuse nfinite_ndropn) 
       have 8: "(\<exists>\<sigma>3 \<sigma>4. \<sigma>2 = nfuse \<sigma>3 \<sigma>4 \<and> nfinite \<sigma>3 \<and> \<sigma>3 \<in> Y \<and> \<sigma>4 \<in> Z \<and> nlast \<sigma>3 = nfirst \<sigma>4) \<Longrightarrow>
                 x \<in> (X \<cdot> Y) \<cdot> Z"
        proof -
         assume a1: "(\<exists>\<sigma>3 \<sigma>4. \<sigma>2 = nfuse \<sigma>3 \<sigma>4 \<and> nfinite \<sigma>3 \<and> \<sigma>3 \<in> Y \<and> \<sigma>4 \<in> Z \<and> nlast \<sigma>3 = nfirst \<sigma>4)"
         show   "x \<in> (X \<cdot> Y) \<cdot> Z"
          proof -
          obtain "\<sigma>3" "\<sigma>4" where 10: 
            "\<sigma>2 = nfuse \<sigma>3 \<sigma>4 \<and> nfinite \<sigma>3 \<and>  \<sigma>3 \<in> Y \<and>  \<sigma>4 \<in> Z \<and> nlast \<sigma>3 = nfirst \<sigma>4" 
            using a1 by auto
          have 11: "x= nfuse \<sigma>1 (nfuse \<sigma>3 \<sigma>4)"
            using "10" "5" by blast     
          have 12: "x = nfuse (nfuse \<sigma>1 \<sigma>3) \<sigma>4 "
            by (simp add: "10" "5" nfirst_nfuse nfuseassoc)
          have 13: "(nfuse \<sigma>1 \<sigma>3) \<in> X\<cdot>Y"
            using fusion_iff[of "(nfuse \<sigma>1 \<sigma>3)" X Y]
            using "10" "5" nfirst_nfuse by fastforce 
          show ?thesis using fusion_iff[of x "X\<cdot>Y" Z] 
            using "10" "12" "13" "5"
            by (metis nfuse_nlength nfinite_nlength_enat nfirst_nfuse nlength_eq_enat_nfiniteD
                plus_enat_simps(1) nlast_nfuse)  
          qed
         qed
      show ?thesis 
      using "6" "7" "8" by blast
    qed
   qed
   show ?thesis 
   using "1" "2" "3" by blast   
qed

lemma FusionAssocB:
assumes "x \<in> (X \<cdot> Y) \<cdot> Z"
shows    "x \<in> X \<cdot> (Y \<cdot> Z)"
proof -
   have 1: "(\<exists>\<sigma>1 \<sigma>2. x = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and> \<sigma>1 \<in> X \<cdot> Y \<and> \<sigma>2 \<in> Z \<and> nlast \<sigma>1 = nfirst \<sigma>2) \<or>
            (\<not> nfinite x \<and> x \<in> X \<cdot> Y)"
     using  assms fusion_iff[of x "X\<cdot>Y" Z]  by auto
   have 2: "(\<not> nfinite x \<and> x \<in> X \<cdot> Y) \<Longrightarrow> x \<in> X \<cdot> (Y \<cdot> Z)"
     by (metis fusion_iff_1 nfinite_ndropn_b)
   have 3: "(\<exists>\<sigma>1 \<sigma>2. x = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and>  \<sigma>1 \<in> X \<cdot> Y \<and> \<sigma>2 \<in> Z \<and> nlast \<sigma>1 = nfirst \<sigma>2) \<Longrightarrow>
            x \<in> X \<cdot> (Y \<cdot> Z)"
     proof - 
      assume a0: "(\<exists>\<sigma>1 \<sigma>2. x = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and>  \<sigma>1 \<in> X \<cdot> Y \<and> \<sigma>2 \<in> Z \<and> nlast \<sigma>1 = nfirst \<sigma>2)"
      show "x \<in> X \<cdot> (Y \<cdot> Z)" 
      proof -
       obtain "\<sigma>1" "\<sigma>2" where 4: 
         "x = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and>  \<sigma>1 \<in> X \<cdot> Y \<and> \<sigma>2 \<in> Z \<and> nlast \<sigma>1 = nfirst \<sigma>2"
         using a0 by auto
       have 5: "\<exists>\<sigma>3 \<sigma>4. \<sigma>1 = nfuse \<sigma>3 \<sigma>4 \<and> nfinite \<sigma>3 \<and>  \<sigma>3 \<in> X \<and> \<sigma>4 \<in> Y \<and> nlast \<sigma>3 = nfirst \<sigma>4"
         using 4  fusion_iff[of " \<sigma>1" X Y]
         using nfuse_nappend by fastforce 
       obtain "\<sigma>3" "\<sigma>4" where 6: 
          "\<sigma>1 = nfuse \<sigma>3 \<sigma>4 \<and> nfinite \<sigma>3 \<and>  \<sigma>3 \<in> X \<and> \<sigma>4 \<in> Y \<and> nlast \<sigma>3 = nfirst \<sigma>4"
         using 5 by auto
       have 7: "x = nfuse (nfuse \<sigma>3 \<sigma>4) \<sigma>2 "
         by (simp add: "4" "6")  
       have 8: "x = nfuse \<sigma>3 (nfuse \<sigma>4 \<sigma>2)"
         using "4" "6" nfuseassoc by metis
       have 9: " (nfuse \<sigma>4 \<sigma>2) \<in> Y\<cdot>Z" 
         using fusion_iff[of " (nfuse \<sigma>4 \<sigma>2)" Y Z] 
         by (metis "4" "6" nfuse_nappend nlast_nfuse) 
       have 10: "nlast \<sigma>3 = nfirst (nfuse \<sigma>4 \<sigma>2)" 
         using "4" "6" nfirst_nfuse nlast_nfuse by fastforce 
      show ?thesis 
         using  fusion_iff[of x X "Y\<cdot>Z"]  
         using "10" "6" "8" "9" by auto 
      qed
    qed
   show ?thesis 
   using "1" "2" "3"     by blast
qed

lemma FusionAssoc :
 " X\<cdot>(Y\<cdot>Z) = (X\<cdot>Y)\<cdot>Z"
using set_eqI[of "X\<cdot>(Y\<cdot>Z)" "(X\<cdot>Y)\<cdot>Z"]
FusionAssocA FusionAssocB  by blast

lemma FusionAssoc1:
 " (X\<cdot>Y)\<cdot>Z = X\<cdot>(Y\<cdot>Z)  "
by (simp add: FusionAssoc)


\<comment> \<open>left and right distributivity\<close>

lemma FusionUnionDistLsem:
 " x\<in>(X \<union> Y)\<cdot>Z \<longleftrightarrow> x \<in> (X\<cdot>Z) \<union> (Y\<cdot>Z)"
by (auto simp add: fusion_iff) 

lemma FusionUnionDistL:
 " (X \<union> Y)\<cdot>Z = (X\<cdot>Z) \<union> (Y\<cdot>Z)"
using FusionUnionDistLsem[of _ X Y Z] by fastforce

lemma FusionUnionDistRsem:
 " x\<in> X\<cdot>(Y \<union> Z) \<longleftrightarrow> x \<in> (X\<cdot>Y) \<union> (X\<cdot>Z)"
by (auto simp add: fusion_iff)

lemma FusionUnionDistR:
 " X\<cdot>(Y \<union> Z) = (X\<cdot>Y) \<union> (X\<cdot>Z)"
using FusionUnionDistRsem[of _ X Y Z] by fastforce


\<comment> \<open>left and right annihilation\<close>
lemma SFalseFusion:
  "SFalse \<cdot> X = SFalse"
by (simp add: fusion_def sfalse_def)

lemma FusionSFalse:
 "X\<cdot>SFalse = (X \<inter> SInf)"
by (auto simp add: fusion_def sfalse_def sinfinite_elim)


\<comment> \<open>idempotency\<close>
lemma UnionIdem:
 " (X::'a iintervals) \<union> X = X"
by simp


subsubsection \<open>Subsumption order\<close>

lemma Subsumption:
 "((X :: 'a iintervals) \<subseteq> Y) = (X \<union> Y = Y)"
by auto


subsubsection \<open>Helper lemmas\<close>

lemma FusionRuleR:
 assumes "X \<subseteq> Y"
 shows   " Z \<cdot> X \<subseteq> Z \<cdot> Y"
using assms FusionUnionDistR by (metis Subsumption)

lemma FusionRuleL:
 assumes "X \<subseteq> Y"
 shows   " X \<cdot> Z \<subseteq> Y \<cdot> Z"
using assms by (metis FusionUnionDistL subset_Un_eq)

lemma spower_commutes:
 "(X \<inter> SFinite)\<cdot>(SPower X n) = (SPower X n)\<cdot>(X \<inter> SFinite)"
proof (induct n)
case 0
then show ?case by (simp add: FusionSEmptyL FusionSEmptyR)
next
case (Suc n)
then show ?case by (simp add: FusionAssoc)
qed

lemma fusion_inductl:
 assumes "Y \<union> X\<cdot>Z \<subseteq> Z "
 shows   "(SPower X n)\<cdot>Y \<subseteq> Z"
using assms 
proof (induct n)
case 0
then show ?case by (simp add: FusionSEmptyL)
next
case (Suc n)
then show ?case
  proof -
   have f1: "X \<cdot> (SPower X n \<cdot> Y) \<subseteq> Z"
     using FusionRuleR Suc.hyps assms by blast
   have "X \<inter> SFinite \<subseteq> X"
     by blast 
   then show ?thesis
     using f1 by (metis (no_types) FusionAssoc FusionRuleL order_trans pwr_Suc)
  qed
qed

lemma fusion_inductl_finite:
 assumes "Y \<union> (X \<inter> SFinite)\<cdot>Z \<subseteq> Z "
 shows   "(SPower X n)\<cdot>Y \<subseteq> Z"
using assms 
proof (induct n)
case 0
then show ?case
by (simp add: FusionSEmptyL) 
next
case (Suc n)
then show ?case
   proof -
   have f1: "SPower X n \<cdot> Y \<subseteq> Z"
     using Suc.hyps assms by blast
   then have "SPower X n \<cdot> Y \<union> Z = Z"
     by blast
   then show ?thesis
     using f1
     by (metis FusionAssoc1 FusionUnionDistR assms le_supE pwr_Suc) 
  qed
qed

lemma fusion_inductl_fmore:
 assumes "Y \<union> (X \<inter> SFMore)\<cdot>Z \<subseteq> Z "
 shows   "(SPower X n)\<cdot>Y \<subseteq> Z"
using assms 
proof (induct n)
case 0
then show ?case
by (simp add: FusionSEmptyL) 
next
case (Suc n)
then show ?case 
 proof -
  have 1: "SPower X (Suc n) \<cdot> Y = ((X \<inter> SFinite)\<cdot>(SPower X n))\<cdot>Y"
    by simp 
  have 2: "((X \<inter> SFinite)\<cdot>(SPower X n))\<cdot> Y = (X \<inter> SFinite)\<cdot>((SPower X n)\<cdot>Y) "
    by (simp add: FusionAssoc) 
  have 3: "(X \<inter> SFinite)\<cdot>((SPower X n)\<cdot>Y) \<subseteq> (X \<inter> SFinite)\<cdot>Z"
    using FusionRuleR Suc.hyps assms by blast
  have 31: "X \<inter> SFinite = ((X \<inter> SFMore) \<union> ((X \<inter> SEmpty) \<inter> SFinite)) "
    by (simp add: sfmore_def smore_def)   blast
  have 4: "(X \<inter> SFinite)\<cdot>Z = ((X \<inter> SFMore)\<cdot>Z \<union> ((X \<inter> SEmpty) \<inter> SFinite)\<cdot>Z)"
    by (simp add: "31" FusionUnionDistL)
  have 5: "(X \<inter> SFMore)\<cdot>Z \<subseteq> Z"
    using assms by auto 
  have 6: "((X \<inter> SEmpty) \<inter> SFinite)\<cdot>Z \<subseteq> Z"
    by (metis FusionRuleL FusionSEmptyL inf_assoc inf_commute inf_le1)
  show ?thesis 
    using "1" "2" "3" "4" "5" "6" by blast
 qed
qed

lemma fusion_inductl_inf:
 assumes "Y \<union> X \<cdot>Z \<subseteq> Z "
 shows   "((SPower X n)\<cdot>(X \<inter> SInf))\<cdot>Y \<subseteq> Z"
using assms 
proof (induct n)
case 0
then show ?case 
by (metis FusionAssoc FusionRuleL FusionRuleR FusionSEmptyL FusionSFalse le_supE order_trans
    pwr_0 sfalse_elim subsetI)
next
case (Suc n)
then show ?case
proof -
have f2: "Z = Z \<union> (Y \<union> X \<cdot> Z)"
  using assms by blast
have "SPower X n \<cdot> (X \<inter> SInf) \<cdot> Y \<subseteq> Z"
  using Suc(1) assms by blast
then have "Z = Z \<union> SPower X n \<cdot> (X \<inter> SInf) \<cdot> Y"
  by blast
then have f3: "(X \<union> X \<inter> SFinite) \<cdot> (Z \<union> SPower X n \<cdot> (X \<inter> SInf) \<cdot> Y) = X \<cdot> Z"
  by force
have "SPower X (Suc n) \<cdot> (X \<inter> SInf) \<cdot> Y = X \<inter> SFinite \<cdot> (SPower X n \<cdot> (X \<inter> SInf) \<cdot> Y)"
  by (simp add: FusionAssoc)
then have "SPower X (Suc n) \<cdot> (X \<inter> SInf) \<cdot> Y \<union> X \<inter> SFinite \<cdot> (Z \<union> SPower X n \<cdot> (X \<inter> SInf) \<cdot> Y) =
           X \<inter> SFinite \<cdot> (Z \<union> SPower X n \<cdot> (X \<inter> SInf) \<cdot> Y)"
  using FusionUnionDistR by blast
then have "SPower X (Suc n) \<cdot> (X \<inter> SInf) \<cdot> Y \<union> X \<cdot> Z = X \<cdot> Z"
  using f3 FusionUnionDistL by blast
then show ?thesis
  using f2 by blast
qed 
qed

lemma fusion_inductl_infmore:
 assumes "Y \<union> X \<cdot>Z \<subseteq> Z "
 shows   "((SPower X n)\<cdot>((X \<inter> SMore) \<inter> SInf))\<cdot>Y \<subseteq> Z"
using assms 
proof (induct n)
case 0
then show ?case
by (metis (no_types, lifting) Diff_Compl Int_assoc Un_Diff_Int compl_inf double_compl fusion_inductl_inf 
    inf.absorb_iff2 pwr_0 sfinite_def smore_def spower_finite sup_left_idem) 
next
case (Suc n)
then show ?case
proof -
have f1: "\<And>n. SPower X n \<cdot> (X \<inter> SInf) \<cdot> Y \<subseteq> Z"
  by (meson assms fusion_inductl_inf)
have "X \<inter> SMore \<inter> SInf \<subseteq> X \<inter> SInf"
  by blast
then show ?thesis
  using f1 by (metis (no_types) FusionRuleL FusionRuleR inf.orderE le_inf_iff)
qed 
qed

lemma fusion_inductl_more:
 assumes "Y \<union> X\<cdot>Z \<subseteq> Z "
 shows   "(SPower (X \<inter> SMore)  n)\<cdot>Y \<subseteq> Z"
using assms 
proof (induct n)
case 0
then show ?case by (simp add: FusionSEmptyL)
next
case (Suc n)
then show ?case 
by (metis FusionUnionDistL fusion_inductl sup.boundedE sup.boundedI sup_inf_absorb)
qed

lemma fusion_inductr:
assumes "Y \<union> Z\<cdot>X \<subseteq> Z "
 shows   "Y\<cdot>(SPower X n) \<subseteq> Z"
using assms 
proof (induct n)
case 0
then show ?case by (simp add: FusionSEmptyR)
next
case (Suc n)
then show ?case 
proof -
have f1: "Z \<cdot> X \<subseteq> Z"
  using assms by auto 
have "\<forall>S. Y \<cdot> (SPower X n \<cdot> S) \<subseteq> Z \<cdot> S"
  using FusionAssoc FusionRuleL Suc.hyps assms by blast 
then show ?thesis
  using f1 by (metis (no_types) FusionRuleR Int_iff order_trans pwr_Suc spower_commutes subsetI) 
qed
qed

lemma SInfSFinite:
 "X = (X \<inter> SFinite) \<union> (X \<inter> SInf)"
using sfinite_def by auto

lemma fusion_inductr_inf:
assumes "Y \<union> Z\<cdot>X \<subseteq> Z "
 shows   "Y\<cdot>((SPower X n)\<cdot>(X \<inter> SInf)) \<subseteq> Z"
proof -
 have 1: "Y\<cdot>(SPower X n) \<subseteq> Z"
   using assms fusion_inductr by blast
 show ?thesis 
 proof -
 have f4: "Y \<cdot> SPower X n = Y \<cdot> SPower X n \<inter> Z"
   using "1" by blast
 have "Y \<cdot> SPower X n \<union> Z = Z"
   by (metis "1" subset_Un_eq) 
 then have f5: "Z \<cdot> (X \<inter> SInf) = Z \<cdot> (X \<inter> SInf) \<union> Y \<cdot> SPower X n \<inter> Z \<cdot> (X \<inter> SInf)"
   using f4 by (metis (no_types) FusionUnionDistL sup.commute) 
 have "Y \<cdot> SPower X n = Y \<cdot> SPower X n \<inter> Z"
   using f4 by auto
 then have "Y \<cdot> (SPower X n \<cdot> (X \<inter> SInf)) = Y \<cdot> SPower X n \<inter> Z \<cdot> (X \<inter> SInf)"
   by (simp add: FusionAssoc1) 
 then show ?thesis
   proof -
   have f1: "Y \<union> Z \<cdot> X \<union> Z = Z"
     by (meson Subsumption assms) 
   have f2: "Y \<cdot> SPower X n \<inter> Z \<cdot> X \<union> Z \<cdot> X = Z \<cdot> X"
     by (metis (no_types) FusionUnionDistL \<open>Y \<cdot> SPower X n \<union> Z = Z\<close> f4)
   have "Y \<cdot> SPower X n \<inter> Z \<cdot> X \<union> Y \<cdot> SPower X n \<inter> Z \<cdot> (X \<inter> SInf) = Y \<cdot> SPower X n \<inter> Z \<cdot> X"
     by (metis (no_types) FusionUnionDistR sup_inf_absorb)
   then show ?thesis
     using f2 f1 \<open>Y \<cdot> (SPower X n \<cdot> (X \<inter> SInf)) = Y \<cdot> SPower X n \<inter> Z \<cdot> (X \<inter> SInf)\<close> by auto
   qed
 qed
qed 
 
lemma fusion_inductr_more:
assumes "Y \<union> Z\<cdot>X \<subseteq> Z "
 shows   "Y\<cdot>(SPower (X \<inter> SMore)  n) \<subseteq> Z"
using assms 
proof (induct n)
case 0
then show ?case by (simp add: FusionSEmptyR) 
next
case (Suc n)
then show ?case
by (metis FusionUnionDistR fusion_inductr le_supE sup.boundedI sup_inf_absorb) 
qed

lemma FusionUnionSPowersem:
 " \<sigma> \<in> Y \<cdot> (\<Union>n.  (SPower X n)\<cdot>Z) \<longleftrightarrow> \<sigma> \<in> (\<Union>n. Y \<cdot> ((SPower X n)\<cdot>Z))"
by (simp add: fusion_iff) blast

lemma FusionUnionSPower:
 "  Y \<cdot> (\<Union>n.  (SPower X n)\<cdot>Z) = (\<Union>n. Y \<cdot> ((SPower X n)\<cdot>Z))"
using FusionUnionSPowersem by blast

lemma FusionUnionSPower1:
 "  Y \<cdot> (\<Union>n.  (SPower X n)) = (\<Union>n. Y \<cdot> ((SPower X n)))"
using FusionUnionSPower[of Y X SEmpty] 
by (metis (no_types, lifting) FusionSEmptyR Sup.SUP_cong)

lemma UnionFusionSPowersem:
 " \<sigma> \<in>  (\<Union>n.  Z\<cdot>(SPower X n))\<cdot>Y \<longleftrightarrow> \<sigma> \<in> (\<Union>n.  (Z\<cdot>(SPower X n))\<cdot>Y)"
by (simp add: fusion_iff) blast

lemma UnionFusionSPower:
 "  (\<Union>n.  Z\<cdot>(SPower X n))\<cdot>Y = (\<Union>n.  (Z\<cdot>(SPower X n))\<cdot>Y)"
using UnionFusionSPowersem by blast

lemma UnionFusionSPower1:
 "  (\<Union>n.  (SPower X n))\<cdot>Y = (\<Union>n.  ((SPower X n))\<cdot>Y)"
using UnionFusionSPower[of SEmpty X Y] 
by (metis (no_types, lifting) FusionSEmptyL Sup.SUP_cong)

lemma spowercommute:
 "(X \<inter> SFinite)\<cdot>(\<Union>n.  (SPower X n)) = (\<Union>n.  (SPower X n)\<cdot>(X \<inter> SFinite))"
by (metis (no_types, lifting) FusionUnionSPower1 Sup.SUP_cong spower_commutes)

lemma sstar_contl: 
  " Y \<cdot> (SStar X) = (\<Union>n. Y \<cdot> (SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> X \<inter> SMore \<inter> SInf))"
proof -
 have 1: " Y \<cdot> (SStar X) = Y \<cdot> ((\<Union>n. ( (SPower (X \<inter> SMore) n))) \<cdot> (SEmpty \<union> X \<inter> SMore \<inter> SInf))"
   by (simp add: sstar_def  spowerstar_def sfpowerstar_def)
 have 2: "Y \<cdot> ((\<Union>n. ( (SPower (X \<inter> SMore) n))) \<cdot> (SEmpty \<union> X \<inter> SMore \<inter> SInf)) =
          (Y \<cdot> (\<Union>n. (SPower (X \<inter> SMore) n)))\<cdot> (SEmpty \<union> X \<inter> SMore \<inter> SInf)"
   using FusionAssoc by blast
 have 3: "(Y \<cdot> (\<Union>n. (SPower (X \<inter> SMore) n))) = (\<Union>n. Y\<cdot> (SPower (X \<inter> SMore) n))" 
   using FusionUnionSPower1 by blast
 have 4: "(Y \<cdot> (\<Union>n. (SPower (X \<inter> SMore) n)))\<cdot> (SEmpty \<union> X \<inter> SMore \<inter> SInf) =
          (\<Union>n. Y\<cdot> (SPower (X \<inter> SMore) n))\<cdot> (SEmpty \<union> X \<inter> SMore \<inter> SInf) "
   by (simp add: "3") 
 have 5: "(\<Union>n. Y\<cdot> (SPower (X \<inter> SMore) n))\<cdot> (SEmpty \<union> X \<inter> SMore \<inter> SInf) =
          (\<Union>n. (Y\<cdot> (SPower (X \<inter> SMore) n))\<cdot> (SEmpty \<union> X \<inter> SMore \<inter> SInf))"
   using UnionFusionSPower[of Y "X \<inter> SMore" "(SEmpty \<union> X \<inter> SMore \<inter> SInf)"]
   by auto
  show ?thesis 
   by (simp add: "1" "2" "4" "5")
qed  

lemma sstar_contr: 
  " (SStar X)\<cdot>Y = (\<Union>n. ((SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> X \<inter> SMore \<inter> SInf))\<cdot>Y)"
proof -
 have 1: "(SStar X)\<cdot>Y = ((\<Union>n. ((SPower (X \<inter> SMore) n)))\<cdot>(SEmpty \<union> X \<inter> SMore \<inter> SInf)) \<cdot> Y "
   by (simp add: sstar_def spowerstar_def sfpowerstar_def)  
 have 2: "((\<Union>n. ((SPower (X \<inter> SMore) n)))\<cdot>(SEmpty \<union> X \<inter> SMore \<inter> SInf)) \<cdot> Y =
          (\<Union>n. ((SPower (X \<inter> SMore) n)))\<cdot>((SEmpty \<union> X \<inter> SMore \<inter> SInf) \<cdot> Y) " 
   using FusionAssoc by blast
 have 3: "(\<Union>n. ((SPower (X \<inter> SMore) n)))\<cdot>((SEmpty \<union> X \<inter> SMore \<inter> SInf) \<cdot> Y) =
          (\<Union>n. ((SPower (X \<inter> SMore) n))\<cdot>((SEmpty \<union> X \<inter> SMore \<inter> SInf) \<cdot> Y))"
   using UnionFusionSPower1[of "X \<inter> SMore" "((SEmpty \<union> X \<inter> SMore \<inter> SInf) \<cdot> Y)"]
   by auto
 have 4: "(\<Union>n. ((SPower (X \<inter> SMore) n))\<cdot>((SEmpty \<union> X \<inter> SMore \<inter> SInf) \<cdot> Y)) =
         (\<Union>n. ((SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> X \<inter> SMore \<inter> SInf))\<cdot>Y) " 
   using FusionAssoc by blast
 show ?thesis 
   by (simp add: "1" "2" "3" "4")
qed   

lemma FPowerstarInductL:
 assumes " Y \<union> (X \<inter> SFinite)\<cdot>Z \<subseteq> Z"
 shows   "(SFPowerstar X)\<cdot> Y \<subseteq> Z"
proof -
 have 1: "(SFPowerstar X)\<cdot> Y = (\<Union>n. (SPower X n))\<cdot>Y"
   by (simp add: sfpowerstar_def) 
 have 2: "(\<Union>n. (SPower X n))\<cdot>Y = (\<Union>n. (SPower X n)\<cdot>Y)"
   using UnionFusionSPower1 by fastforce  
 have 3: "\<And>n. (SPower X n)\<cdot>Y \<subseteq> Z" 
   using assms fusion_inductl_finite by blast 
 have 4: "(\<Union>n. (SPower X n)\<cdot>Y) \<subseteq> Z"
   using "3" by blast 
 show ?thesis 
   using "1" "2" "4" by blast
qed

lemma FPowerstarInductMoreL:
 assumes " Y \<union> ((X \<inter> SMore) \<inter> SFinite)\<cdot>Z \<subseteq> Z"
 shows   "(SFPowerstar X)\<cdot> Y \<subseteq> Z"
proof -
 have 1: "(SFPowerstar X)\<cdot> Y = (\<Union>n. (SPower X n))\<cdot>Y"
   by (simp add: sfpowerstar_def)
 have 2: "(\<Union>n. (SPower X n))\<cdot>Y = (\<Union>n. (SPower X n)\<cdot>Y)"
   using UnionFusionSPower1 by fastforce
 have 3: "\<And>n. (SPower X n)\<cdot>Y \<subseteq> Z"
   by (metis Int_assoc Int_commute assms fusion_inductl_fmore sfmore_def)
 have 4: "(\<Union>n. (SPower X n)\<cdot>Y) \<subseteq> Z"
   using "3" by blast
 show ?thesis 
   using "1" "2" "4" by blast
qed 

lemma PowerstarInductL:
 assumes " Y \<union> X\<cdot>Z \<subseteq> Z"
 shows  "(SPowerstar X)\<cdot>Y \<subseteq> Z"
proof -
 have 1: "(SPowerstar X)\<cdot> Y = ((\<Union>n. (SPower X n))\<cdot>(SEmpty \<union> X \<inter> SInf)) \<cdot>Y"
   by (simp add: spowerstar_def sfpowerstar_def) 
 have 2: "((\<Union>n. (SPower X n))\<cdot>(SEmpty \<union> X \<inter> SInf)) \<cdot>Y =
          (\<Union>n. ((SPower X n)))\<cdot>((SEmpty \<union> X \<inter> SInf) \<cdot>Y)"
   using FusionAssoc1[of "(\<Union>n. ((SPower X n)))" "(SEmpty \<union> X \<inter> SInf)" Y]
   by blast
 have 3: "(\<Union>n. ((SPower X n)))\<cdot>((SEmpty \<union> X \<inter> SInf) \<cdot>Y) =
          (\<Union>n. ((SPower X n))\<cdot>((SEmpty \<union> X \<inter> SInf) \<cdot>Y))"
   using UnionFusionSPower1[of X "((SEmpty \<union> X \<inter> SInf) \<cdot>Y)"]
   by auto 
 have 4: "\<And>n. (SPower X n)\<cdot>Y \<subseteq> Z"
   using assms fusion_inductl by blast 
 have 5: "\<And>n. ((SPower X n)\<cdot>(X \<inter> SInf))\<cdot>Y \<subseteq> Z " 
   using assms fusion_inductl_inf by blast
 have 6: "\<And>n. (SPower X n)\<cdot>Y \<union> ((SPower X n)\<cdot>(X \<inter> SInf))\<cdot>Y =
               ((SPower X n)\<cdot>(SEmpty \<union> (X \<inter> SInf)))\<cdot>Y " 
   by (simp add: FusionSEmptyR FusionUnionDistL FusionUnionDistR)
 have 7: "\<And>n. ((SPower X n)\<cdot>(SEmpty \<union> (X \<inter> SInf)))\<cdot>Y \<subseteq> Z" 
   using "4" "5" "6" by blast
 have 8: "(\<Union>n. ((SPower X n)\<cdot>(SEmpty \<union> (X \<inter> SInf))) \<cdot>Y) \<subseteq> Z"
   using 7 by blast 
 show ?thesis 
   by (metis (no_types, lifting) "1" "3" "8" FusionAssoc Sup.SUP_cong)
qed

lemma SStarInductMoreL:
 assumes " Y \<union> (X \<inter> SMore)\<cdot>Z \<subseteq> Z"
 shows  "(SStar X)\<cdot>Y \<subseteq> Z"
proof -
 have 1: "(SStar X)\<cdot> Y = ((\<Union>n. (SPower (X \<inter> SMore) n))\<cdot>(SEmpty \<union> (X \<inter> SMore) \<inter> SInf)) \<cdot>Y"
   by (simp add: sstar_def spowerstar_def sfpowerstar_def) 
 have 2: "((\<Union>n. (SPower (X \<inter> SMore)  n))\<cdot>(SEmpty \<union> (X \<inter> SMore) \<inter> SInf)) \<cdot>Y =
          (\<Union>n. ((SPower (X \<inter> SMore) n)))\<cdot>((SEmpty \<union> (X \<inter> SMore) \<inter> SInf) \<cdot>Y)"
   using FusionAssoc1[of "(\<Union>n. ((SPower (X \<inter> SMore) n)))" "(SEmpty \<union> (X \<inter> SMore) \<inter> SInf)" Y]
   by blast
 have 3: "(\<Union>n. ((SPower (X \<inter> SMore) n)))\<cdot>((SEmpty \<union> (X \<inter> SMore) \<inter> SInf) \<cdot>Y) =
          (\<Union>n. ((SPower (X \<inter> SMore) n))\<cdot>((SEmpty \<union> (X \<inter> SMore) \<inter> SInf) \<cdot>Y))"
   using UnionFusionSPower1[of "X \<inter> SMore" "((SEmpty \<union> (X \<inter> SMore) \<inter> SInf) \<cdot>Y)"]
   by auto 
 have 4: "\<And>n. (SPower (X \<inter> SMore) n)\<cdot>Y \<subseteq> Z"
   using assms fusion_inductl by blast   
 have 5: "\<And>n. ((SPower (X \<inter> SMore) n)\<cdot>((X \<inter> SMore) \<inter> SInf))\<cdot>Y \<subseteq> Z " 
   using assms fusion_inductl_inf by blast
 have 6: "\<And>n. (SPower (X \<inter> SMore) n)\<cdot>Y \<union> ((SPower (X \<inter> SMore) n)\<cdot>((X \<inter> SMore) \<inter> SInf))\<cdot>Y =
               ((SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> ((X \<inter> SMore) \<inter> SInf)))\<cdot>Y " 
   by (simp add: FusionSEmptyR FusionUnionDistL FusionUnionDistR)
 have 7: "\<And>n. ((SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> ( (X\<inter> SMore) \<inter> SInf)))\<cdot>Y \<subseteq> Z" 
   using "4" "5" "6" by blast
 have 8: "(\<Union>n. ((SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> ((X \<inter> SMore) \<inter> SInf))) \<cdot>Y) \<subseteq> Z"
   using 7 by blast 
 show ?thesis 
   by (metis (no_types, lifting) "1" "3" "8" FusionAssoc Sup.SUP_cong)
qed

lemma SFPowerstarInductR:
 assumes "Y \<union> Z\<cdot>X \<subseteq> Z"
 shows   "Y\<cdot>(SFPowerstar X) \<subseteq> Z "
proof -
 have 1: "Y\<cdot>(SFPowerstar X) = Y\<cdot>(\<Union>n. (SPower X n))"
   by (simp add: sfpowerstar_def)
 have 2: "Y\<cdot>(\<Union>n. (SPower X n)) = (\<Union>n. Y\<cdot>(SPower X n))" 
   using FusionUnionSPower1 by blast 
 have 3: "\<And>n. Y\<cdot>(SPower X n) \<subseteq> Z"
   using assms fusion_inductr by blast 
 have 4: "(\<Union>n. Y\<cdot>(SPower X n)) \<subseteq> Z"
   using "3" by blast 
 show ?thesis 
   by (simp add: "1" "2" "4")
qed

lemma SPowerstarInductR:
 assumes "Y \<union> Z\<cdot>X \<subseteq> Z"
 shows   "Y\<cdot>(SPowerstar X) \<subseteq> Z "
proof -
 have 1: "Y\<cdot>(SPowerstar X) = Y\<cdot>((\<Union>n. (SPower X n))\<cdot> (SEmpty \<union> (X \<inter> SInf)))"
   by (simp add: spowerstar_def sfpowerstar_def)
 have 11: "Y\<cdot>((\<Union>n. (SPower X n))\<cdot> (SEmpty \<union> (X \<inter> SInf))) =
           Y\<cdot>((\<Union>n. (SPower X n)\<cdot> (SEmpty \<union> (X \<inter> SInf))))"
   by (metis UnionFusionSPower1)
 have 12: "Y\<cdot>((\<Union>n. (SPower X n)\<cdot> (SEmpty \<union> (X \<inter> SInf)))) =
           ((\<Union>n. Y\<cdot>(SPower X n)\<cdot> (SEmpty \<union> (X \<inter> SInf))))" 
   using FusionUnionSPower[of Y X "(SEmpty \<union> X \<inter> SInf)"]   
   by (metis (no_types, lifting) FusionAssoc Sup.SUP_cong)
 have 3: "\<And>n. Y\<cdot>(SPower X n) \<subseteq> Z"
   using assms fusion_inductr by blast 
 have 31: "\<And>n. Y\<cdot>(SPower X n)\<cdot>(X \<inter> SInf) \<subseteq> Z"
   using FusionAssoc assms fusion_inductr_inf by blast 
 have 32: "\<And>n. Y\<cdot>(SPower X n)\<cdot>(SEmpty \<union> (X \<inter> SInf)) \<subseteq> Z" 
   by (simp add: "3" "31" FusionSEmptyR FusionUnionDistR)
 have 4: "(\<Union>n. Y\<cdot>(SPower X n)\<cdot>(SEmpty \<union> (X \<inter> SInf))) \<subseteq> Z"
   using "32" by blast 
 show ?thesis 
   by (simp add: "1" "11" "12" "4")
qed

lemma SStarInductMoreR:
 assumes "Y \<union> Z\<cdot>(X \<inter> SMore) \<subseteq> Z"
 shows   "Y\<cdot>(SStar X) \<subseteq> Z "
proof -
 have 1: "Y\<cdot>(SStar X) = Y\<cdot>((\<Union>n. (SPower (X \<inter> SMore) n))\<cdot> (SEmpty \<union> ((X \<inter> SMore) \<inter> SInf)))"
   by (simp add: spowerstar_def sfpowerstar_def sstar_def)
 have 11: "Y\<cdot>((\<Union>n. (SPower (X \<inter> SMore) n))\<cdot> (SEmpty \<union> ((X \<inter> SMore) \<inter> SInf))) =
           Y\<cdot>((\<Union>n. (SPower (X \<inter> SMore) n)\<cdot> (SEmpty \<union> ((X \<inter> SMore) \<inter> SInf))))"
   by (metis UnionFusionSPower1)
 have 12: "Y\<cdot>((\<Union>n. (SPower (X \<inter> SMore) n)\<cdot> (SEmpty \<union> ((X \<inter> SMore) \<inter> SInf)))) =
           ((\<Union>n. Y\<cdot>(SPower (X \<inter> SMore) n)\<cdot> (SEmpty \<union> ((X \<inter> SMore) \<inter> SInf))))" 
   using FusionUnionSPower[of Y "(X \<inter> SMore)" "(SEmpty \<union> ((X \<inter> SMore) \<inter> SInf))"] 
   by (metis (no_types, lifting) FusionAssoc Sup.SUP_cong)
 have 3: "\<And>n. Y\<cdot>(SPower (X \<inter> SMore) n) \<subseteq> Z"
   using assms fusion_inductr by blast
 have 31: "\<And>n. Y\<cdot>(SPower (X \<inter> SMore) n)\<cdot>((X \<inter> SMore) \<inter> SInf) \<subseteq> Z"
   using assms FusionAssoc fusion_inductr_inf by blast
 have 32: "\<And>n. Y\<cdot>(SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> ((X \<inter> SMore) \<inter> SInf)) \<subseteq> Z" 
   by (simp add: "3" "31" FusionSEmptyR FusionUnionDistR)
 have 4: "(\<Union>n. Y\<cdot>(SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> ((X \<inter> SMore) \<inter> SInf))) \<subseteq> Z"
   using "32" by blast 
 show ?thesis 
   by (simp add: "1" "11" "12" "4")
qed

lemma Powerstarhelp2:
 "(X \<inter> SInf) = (X \<inter> SInf)\<cdot> Z"
by (metis FusionAssoc FusionSFalse SFalseFusion)

lemma Powerstarhelp3:
 "((X \<inter> SInf)\<cdot>Y \<union> (X \<inter> SFinite)\<cdot>Y) = X\<cdot>Y  "
by (metis Diff_Compl FusionUnionDistL Un_Diff_Int sfinite_def) 

lemma Powerstareqv:
 "(SPowerstar X) = SEmpty \<union> X\<cdot>(SPowerstar X)"
proof -
 have 1: "(SPowerstar X) = (\<Union>n. (SPower X n))\<cdot>(SEmpty \<union> (X \<inter> SInf))"
   by (simp add: sfpowerstar_def spowerstar_def)
 have 2: "(\<Union>n. (SPower X n))\<cdot>(SEmpty \<union> (X \<inter> SInf)) =
          (SEmpty \<union> (X \<inter> SFinite)\<cdot>(\<Union>n. (SPower X n)))\<cdot> (SEmpty \<union> (X \<inter> SInf)) "
   by (metis sfpowerstar_eqv_1)
 have 3: "(SEmpty \<union> (X \<inter> SFinite)\<cdot>(\<Union>n. (SPower X n)))\<cdot> (SEmpty \<union> (X \<inter> SInf)) =
          SEmpty\<cdot>(SEmpty \<union> (X \<inter> SInf)) \<union> (X \<inter> SFinite)\<cdot>(\<Union>n. (SPower X n))\<cdot> (SEmpty \<union> (X \<inter> SInf))"
   by (simp add: FusionUnionDistL) 
 have 4: "SEmpty\<cdot>(SEmpty \<union> (X \<inter> SInf)) \<union> (X \<inter> SFinite)\<cdot>(\<Union>n. (SPower X n))\<cdot> (SEmpty \<union> (X \<inter> SInf)) =
          SEmpty \<union> (X \<inter> SInf) \<union> (X \<inter> SFinite)\<cdot>(\<Union>n. (SPower X n))\<cdot> (SEmpty \<union> (X \<inter> SInf))"
   by (simp add: FusionSEmptyL)
 have 5: "SEmpty \<union> (X \<inter> SInf) \<union> (X \<inter> SFinite)\<cdot>(\<Union>n. (SPower X n))\<cdot> (SEmpty \<union> (X \<inter> SInf)) =
          SEmpty \<union> (X \<inter> SInf)\<cdot>(\<Union>n. (SPower X n))\<cdot> (SEmpty \<union> (X \<inter> SInf)) \<union> 
                   (X \<inter> SFinite)\<cdot>(\<Union>n. (SPower X n))\<cdot> (SEmpty \<union> (X \<inter> SInf))"
   by (metis Powerstarhelp2)
 have 6: "SEmpty \<union> (X \<inter> SInf)\<cdot>(\<Union>n. (SPower X n))\<cdot> (SEmpty \<union> (X \<inter> SInf)) \<union> 
                   (X \<inter> SFinite)\<cdot>(\<Union>n. (SPower X n))\<cdot> (SEmpty \<union> (X \<inter> SInf)) =
          SEmpty \<union> X \<cdot>((\<Union>n. (SPower X n))\<cdot> (SEmpty \<union> (X \<inter> SInf)))"
   by (metis FusionAssoc Powerstarhelp3 UnionAssoc) 
 show ?thesis
   using "1" "2" "3" "4" "5" "6" by presburger 
qed 

lemma SStareqv:
  "SStar X = SEmpty \<union> (X \<inter> SMore)\<cdot>(SStar X)"
by (metis Powerstareqv sstar_def)
   
lemma SSkipSFinite:
 " SSkip \<inter> SFinite = SSkip"
proof -
 have 1: "SSkip \<inter> SFinite \<subseteq> SSkip " 
   by simp
 have 2: " SInf \<subseteq>  SEmpty \<union> (- SEmpty \<cdot> - SEmpty) "
   by (metis Diff_Compl Diff_subset_conv Powerstarhelp2 Powerstarhelp3 double_compl inf_commute sup_ge1) 
 have 3: " SSkip \<subseteq> SSkip \<inter> SFinite" 
   using 2 by (simp add: sskip_def smore_def sfinite_def )  blast
 show ?thesis 
   using "3" by blast
qed

lemma spower_sskip_elim :
 "(\<sigma> \<in> SPower SSkip n) \<longleftrightarrow> 
   ( nlength \<sigma> = n )"
proof (induct n arbitrary: \<sigma>)
case 0
then show ?case
by (simp add: sempty_elim zero_enat_def) 
next
case (Suc n)
then show ?case
 proof -
  have 1: "(\<sigma> \<in> SPower SSkip (Suc n)) \<longleftrightarrow>  \<sigma> \<in> (SFinite \<inter> SSkip)\<cdot> (SPower SSkip n)" 
    by (simp add: inf_commute)
  have 2: " \<sigma> \<in> (SFinite \<inter> SSkip) \<longleftrightarrow> \<sigma> \<in> SSkip"
    using SSkipSFinite by auto 
  have 3: "\<sigma> \<in> (SFinite \<inter> SSkip)\<cdot> (SPower SSkip n) \<longleftrightarrow>
           (\<exists>\<sigma>1 \<sigma>2. \<sigma> = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and> \<sigma>1 \<in> SSkip \<and> \<sigma>2 \<in> SPower SSkip n \<and> nlast \<sigma>1 = nfirst \<sigma>2) " 
    using ffusion_iff[of \<sigma> "SSkip" "SPower SSkip n"] by (simp add: inf_commute)
  have 4: "(\<exists>\<sigma>1 \<sigma>2. \<sigma> = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and> \<sigma>1 \<in> SSkip \<and> \<sigma>2 \<in> SPower SSkip n \<and> nlast \<sigma>1 = nfirst \<sigma>2) \<longleftrightarrow>
           (nlength \<sigma> = enat (Suc n))"
    by (auto simp add: Suc.hyps nfuse_nlength one_enat_def sskip_elim)
       (metis One_nat_def add.commute eSuc_enat enat.simps(3) enat_add_sub_same enat_le_plus_same(2)  
        min.orderE ndropn_nfirst ndropn_nlength nfinite_ntaken nfuse_ntaken_ndropn ntaken_nlast 
        ntaken_nlength one_enat_def plus_1_eSuc(2))
  show ?thesis 
    using "1" "3" "4" by presburger
 qed
qed


subsubsection \<open>Kleene Algebra\<close>

\<comment> \<open>left unfold\<close>
lemma UnfoldL:
 "SEmpty \<union> X\<cdot>(SStar X) = (SStar X)"
proof -
 have 1: "(SStar X) = SEmpty \<union> (X \<inter> SMore)\<cdot>(SStar X) " 
   by (meson Un_iff set_eqI SStareqv)
 have 2: " (X \<inter> SMore)\<cdot>(SStar X) \<subseteq> X\<cdot>(SStar X)" 
   by (simp add: FusionRuleL)
 have 3: " (SStar X) \<subseteq> SEmpty \<union> X\<cdot>(SStar X)" 
   using "1" "2" by blast
 have 4: " SEmpty \<subseteq> (SStar X)" 
   using "1" by auto
 have 5: " X \<subseteq> SEmpty \<union> (X \<inter> SMore)" 
   by (simp add: Un_Int_distrib smore_def)
 have 6: " X\<cdot>(SStar X) \<subseteq> (SStar X) \<union> (X \<inter> SMore)\<cdot>(SStar X)" 
   using "5" by (metis FusionRuleL FusionUnionDistL FusionSEmptyL)
 have 7: " (SStar X) \<subseteq> SEmpty \<union> (X \<inter> SMore)\<cdot>(SStar X)"
   using "1" by auto
 have 8: "X\<cdot>(SStar X) \<subseteq> SEmpty \<union> (X \<inter> SMore)\<cdot>(SStar X) "
   using "6" "7" by blast
 hence 9: " X\<cdot>(SStar X) \<subseteq> (SStar X)"
   using "1" by auto
 have 10: "SEmpty \<union> X\<cdot>(SStar X) \<subseteq> (SStar X) "
   using "9" "4" by simp
 from 3 10 show ?thesis by auto
qed


\<comment> \<open>Left induction\<close>

lemma SStarInductL:
 assumes " Y \<union> X\<cdot>Z \<subseteq> Z"
 shows  "(SStar X)\<cdot>Y \<subseteq> Z"
proof -
 have 1: "(SStar X)\<cdot> Y = ((\<Union>n. (SPower (X \<inter> SMore) n))\<cdot>(SEmpty \<union> (X \<inter> SMore) \<inter> SInf)) \<cdot>Y"
   by (simp add: sstar_def spowerstar_def sfpowerstar_def) 
 have 2: "((\<Union>n. (SPower (X \<inter> SMore)  n))\<cdot>(SEmpty \<union> (X \<inter> SMore) \<inter> SInf)) \<cdot>Y =
          (\<Union>n. ((SPower (X \<inter> SMore) n)))\<cdot>((SEmpty \<union> (X \<inter> SMore) \<inter> SInf) \<cdot>Y)"
   using FusionAssoc1[of "(\<Union>n. ((SPower (X \<inter> SMore) n)))" "(SEmpty \<union> (X \<inter> SMore) \<inter> SInf)" Y]
   by blast
 have 3: "(\<Union>n. ((SPower (X \<inter> SMore) n)))\<cdot>((SEmpty \<union> (X \<inter> SMore) \<inter> SInf) \<cdot>Y) =
          (\<Union>n. ((SPower (X \<inter> SMore) n))\<cdot>((SEmpty \<union> (X \<inter> SMore) \<inter> SInf) \<cdot>Y))"
   using UnionFusionSPower1[of "X \<inter> SMore" "((SEmpty \<union> (X \<inter> SMore) \<inter> SInf) \<cdot>Y)"]
   by auto 
 have 31: "Y \<union> (X \<inter> SMore)\<cdot> Z \<subseteq> Z"
   by (meson FusionRuleL Un_mono assms dual_order.trans inf.cobounded1 subset_refl) 
 have 4: "\<And>n. (SPower (X \<inter> SMore) n)\<cdot>Y \<subseteq> Z"
   using assms fusion_inductl_more by blast
 have 5: "\<And>n. ((SPower (X \<inter> SMore) n)\<cdot>((X \<inter> SMore) \<inter> SInf))\<cdot>Y \<subseteq> Z " 
   using "31" fusion_inductl_inf by blast
 have 6: "\<And>n. (SPower (X \<inter> SMore) n)\<cdot>Y \<union> ((SPower (X \<inter> SMore) n)\<cdot>((X \<inter> SMore) \<inter> SInf))\<cdot>Y =
               ((SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> ((X \<inter> SMore) \<inter> SInf)))\<cdot>Y " 
   by (simp add: FusionSEmptyR FusionUnionDistL FusionUnionDistR)
 have 7: "\<And>n. ((SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> ( (X\<inter> SMore) \<inter> SInf)))\<cdot>Y \<subseteq> Z" 
   using "4" "5" "6" by blast
 have 8: "(\<Union>n. ((SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> ((X \<inter> SMore) \<inter> SInf))) \<cdot>Y) \<subseteq> Z"
   using 7 by blast 
 show ?thesis 
   by (metis (no_types, lifting) "1" "3" "8" FusionAssoc Sup.SUP_cong)
qed


\<comment> \<open>Right induction\<close>
lemma SStarInductR:
 assumes "Y \<union> Z\<cdot>X \<subseteq> Z"
 shows   "Y\<cdot>(SStar X) \<subseteq> Z "
proof -
 have 1: "Y\<cdot>(SStar X) = Y\<cdot>((\<Union>n. (SPower (X \<inter> SMore) n))\<cdot> (SEmpty \<union> ((X \<inter> SMore) \<inter> SInf)))"
   by (simp add: spowerstar_def sfpowerstar_def sstar_def)
 have 11: "Y\<cdot>((\<Union>n. (SPower (X \<inter> SMore) n))\<cdot> (SEmpty \<union> ((X \<inter> SMore) \<inter> SInf))) =
           Y\<cdot>((\<Union>n. (SPower (X \<inter> SMore) n)\<cdot> (SEmpty \<union> ((X \<inter> SMore) \<inter> SInf))))"
   by (metis UnionFusionSPower1)
 have 12: "Y\<cdot>((\<Union>n. (SPower (X \<inter> SMore) n)\<cdot> (SEmpty \<union> ((X \<inter> SMore) \<inter> SInf)))) =
           ((\<Union>n. Y\<cdot>(SPower (X \<inter> SMore) n)\<cdot> (SEmpty \<union> ((X \<inter> SMore) \<inter> SInf))))" 
   using FusionUnionSPower[of Y "(X \<inter> SMore)" "(SEmpty \<union> ((X \<inter> SMore) \<inter> SInf))" ]   
   by (metis (no_types, lifting) FusionAssoc Sup.SUP_cong)
 have 21: "Y \<union> Z\<cdot>(X \<inter> SMore) \<subseteq> Z" 
   by (meson FusionRuleR Un_mono assms dual_order.trans inf.cobounded1 subset_refl)
 have 3: "\<And>n. Y\<cdot>(SPower (X \<inter> SMore) n) \<subseteq> Z"
   using "21" fusion_inductr by blast
 have 31: "\<And>n. Y\<cdot>(SPower (X \<inter> SMore) n)\<cdot>((X \<inter> SMore) \<inter> SInf) \<subseteq> Z"
   using "21" FusionAssoc fusion_inductr_inf by blast
 have 32: "\<And>n. Y\<cdot>(SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> ((X \<inter> SMore) \<inter> SInf)) \<subseteq> Z" 
   by (simp add: "3" "31" FusionSEmptyR FusionUnionDistR)
 have 4: "(\<Union>n. Y\<cdot>(SPower (X \<inter> SMore) n)\<cdot>(SEmpty \<union> ((X \<inter> SMore) \<inter> SInf))) \<subseteq> Z"
   using "32" by blast 
 show ?thesis 
   by (simp add: "1" "11" "12" "4")
qed


subsubsection \<open>Omega Algebra\<close>

lemma SFMoresem [ mono]:
 " x \<in> ( (((F \<inter> SMore) \<inter> SFinite)\<cdot>G))  \<longleftrightarrow>  
 ((\<exists>\<sigma>1 \<sigma>2.  x = nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and> \<sigma>1 \<in> F \<and> 0<(nlength \<sigma>1) \<and> \<sigma>2 \<in> G \<and> nlast \<sigma>1 = nfirst \<sigma>2))
  " 
by (simp add: fusion_iff sinfinite_elim smore_elim sfinite_elim) blast 

lemma monoomega[ mono]:
 "mono (\<lambda>p x.  x \<in> ((F \<inter> SMore) \<inter> SFinite \<cdot> p))" 
by (auto simp add: mono_def sinfinite_elim fusion_iff sfmore_elim)

coinductive_set somega :: "'a iintervals \<Rightarrow> 'a iintervals"
   for F where 
     "((\<exists>\<sigma>1 \<sigma>2. x=nfuse \<sigma>1 \<sigma>2 \<and> nfinite \<sigma>1 \<and> \<sigma>1 \<in> F \<and> 0<(nlength \<sigma>1) \<and> 
       \<sigma>2 \<in> (somega F) \<and> nlast \<sigma>1 = nfirst \<sigma>2)) 
        \<Longrightarrow> x \<in> (somega F)" 

lemma SOmegaIntros:
 "  (((F \<inter> SMore) \<inter> SFinite)\<cdot>(somega F)) \<subseteq> (somega F)" 
using somega.intros[of _ F]  SFMoresem[of _ F "(somega F)"]
by (meson subsetI) 

lemma SOmegaCases:
assumes "x \<in> somega F" 
        "x \<in> ( (((F \<inter> SMore) \<inter> SFinite)\<cdot>(somega F))) \<Longrightarrow> P"
 shows  "P"
using assms somega.cases[of x F P] SFMoresem by blast 
 
lemma SOmegaUnrollsem:
 " x\<in> (somega F)  \<longleftrightarrow>  x\<in> (((F \<inter> SMore) \<inter> SFinite)\<cdot>(somega F))"
proof auto
  show "x \<in> somega F \<Longrightarrow> x \<in> (F \<inter> SMore)\<inter>SFinite \<cdot> somega F"
    using SOmegaCases by blast
  show "x \<in> (F \<inter> SMore)\<inter> SFinite \<cdot> somega F \<Longrightarrow> x \<in> somega F" 
   by (metis (no_types, lifting)   SOmegaIntros inf_commute   subset_eq)
qed

lemma SOmegaUnroll:
 " (somega F)  =  (((F \<inter> SMore)\<inter> SFinite)\<cdot>(somega F))"
using SOmegaUnrollsem by blast

lemma SOmegaCoinductsem:
 assumes " \<And> x. x \<in> X \<Longrightarrow> x \<in> ( (((F \<inter> SMore)\<inter>SFinite)\<cdot>(X \<union> (somega F)))) "
 shows " x \<in> X \<Longrightarrow> x \<in> (somega F)"
using assms somega.coinduct[of "\<lambda>x. x \<in> X" "x" F] SFMoresem[of  _ F "(X \<union> somega F)"]  by auto

lemma SOmegaCoinduct:
 assumes " X \<subseteq>  ( (((F \<inter> SMore)\<inter>SFinite)\<cdot>(X \<union> (somega F)))) "
 shows " X \<subseteq>  (somega F)"
using assms SOmegaCoinductsem[of X F] by blast 

lemma SOmegaWeakCoinductsem:
 assumes " \<And> x. x \<in> X \<Longrightarrow> x \<in> ( (((F \<inter> SMore)\<inter>SFinite)\<cdot>X )) "
 shows " x \<in> X \<Longrightarrow> x \<in> (somega F)"
using assms somega.coinduct[of "\<lambda>x. x \<in> X" "x" F] 
by (metis SFMoresem)

lemma SOmegaWeakCoinduct:
 assumes " X \<subseteq>  ( (((F \<inter> SMore)\<inter>SFinite)\<cdot>X )) "
 shows " X \<subseteq>  (somega F)"
using assms SOmegaWeakCoinductsem[of X F] by blast 


subsubsection \<open>ITL specific Laws\<close>

lemma PwrFusionInterLsem:
 " x \<in> ((((SPower SSkip n) \<inter> X)\<cdot>V) \<inter> (((SPower SSkip n) \<inter> Y)\<cdot>W)) \<longleftrightarrow>
  x \<in> (((SPower SSkip n) \<inter> X \<inter> Y)\<cdot>(V \<inter> W)) "
by (auto simp add:  spower_sskip_elim fusion_iff_1 min_absorb1 nlength_eq_enat_nfiniteD)

lemma PwrFusionInterL:
 "((((SPower SSkip n) \<inter> X)\<cdot>V) \<inter> (((SPower SSkip n) \<inter> Y)\<cdot>W)) = 
  (((SPower SSkip n) \<inter> X \<inter> Y)\<cdot>(V \<inter> W)) "
using PwrFusionInterLsem by blast

lemma PwrFusionInterRsem :
 " x \<in> ((V\<cdot>((SPower SSkip n) \<inter> X)) \<inter> ((W\<cdot>((SPower SSkip n) \<inter> Y)))) \<longleftrightarrow>
   x \<in> ((V \<inter> W)\<cdot>((SPower SSkip n) \<inter> X \<inter> Y)) "
by (simp add: spower_sskip_elim fusion_iff_1 )
   (rule,
    metis enat.simps(3) enat_add2_eq enat_add_sub_same le_iff_add ndropn_nlength nfinite_ndropn 
    nlength_eq_enat_nfiniteD the_enat.simps,
    metis enat.simps(3)  enat_add_sub_same le_iff_add ndropn_nlength nfinite_ndropn_b 
    nlength_eq_enat_nfiniteD)
  
lemma PwrFusionInterR :
 " ((V\<cdot>((SPower SSkip n) \<inter> X)) \<inter> ((W\<cdot>((SPower SSkip n) \<inter> Y)))) =
    ((V \<inter> W)\<cdot>((SPower SSkip n) \<inter> X \<inter> Y)) "
using PwrFusionInterRsem by blast

lemma SSkipFusionImpSMore:
 " SSkip\<cdot>STrue \<subseteq> SMore" 
using subsetI[of "SSkip\<cdot>STrue" "SMore"]  
by (auto simp add:  fusion_iff_1 sskip_elim smore_elim strue_elim)

lemma SMoreImpSSkipFusion:
 " SMore \<subseteq> SSkip\<cdot>STrue  " 
using subsetI[of "SMore" "SSkip\<cdot>STrue"]  
by (auto simp add: fusion_iff_1 sskip_elim smore_elim strue_elim)
   (metis i0_less ileI1  one_eSuc one_enat_def,
    metis i0_less ileI1  one_eSuc one_enat_def)

lemma SSkipSMore:
 "SSkip \<inter> SMore = SSkip"
by (simp add: inf.absorb1 smore_def sskip_def) 
   
lemma SPowerSkip:
  " (\<Union>n. (SPower SSkip n)) = SFinite"
proof -
 have 1: "(\<Union>n. (SPower SSkip n)) \<subseteq> SFinite"
   by (simp add: SUP_least spower_finite)
 have 2: " \<And>x.  x \<in> SFinite \<Longrightarrow> x \<in> (\<Union>n. (SPower SSkip n))" 
   by (auto simp add: sfinite_elim spower_sskip_elim nfinite_nlength_enat)          
 have 3: "SFinite \<subseteq> (\<Union>n. (SPower SSkip n))"
   using 2 by blast 
 from 1 3 show ?thesis by blast
qed
   
lemma SStarSkip:
 " (SStar SSkip) = SFinite"
by (simp add: sstar_def SSkipSMore spowerstar_def sfpowerstar_def SPowerSkip) 
   (metis Compl_disjoint2 FusionSEmptyR SSkipSFinite UnionSFalse inf.assoc inf_bot_right
     sfalse_def sfinite_def)

(* 
 sledgehammer (del: sskip_elim fusion_iff fusion_iff_1 smore_fusion_smore spower_commutes 
               fusion_inductl fusion_inductr sstar_contl sstar_contr )
 
*)  

subsection \<open>Derived Laws\<close>


subsubsection \<open>Helper Lemmas\<close>

lemma B01:
 assumes "(X:: 'a iintervals) \<subseteq> Y"
 shows   " -Y \<subseteq> -X"
using assms by auto

lemma B04:
 " ((X:: 'a iintervals) = Y ) \<longleftrightarrow> (X \<subseteq> Y) \<and> (Y \<subseteq> X)"
by auto

lemma B09:
 assumes "-X \<union> Y = STrue"
 shows   "(X:: 'a iintervals) \<subseteq> Y"
using assms using strue_def by auto

lemma B20:
 "(X:: 'a iintervals) \<subseteq> Y \<union> Z \<longleftrightarrow> X \<inter> -Y \<subseteq> Z"
by auto

lemma B28:
 " ((X:: 'a iintervals) \<inter> Y) \<union> (X \<inter> Z) = X \<inter> (Y \<union> Z) "
by auto
  
lemma CH01:
 "STrue\<cdot>STrue = STrue"
by (metis FusionSEmptyR FusionUnionDistR Int_commute  STrueTop  inf_sup_absorb )

lemma CH07:
 "(((SSkip \<inter> X)\<cdot>V) \<inter> ((SSkip \<inter> Y)\<cdot>W)) = ((SSkip \<inter> X \<inter> Y)\<cdot>(V \<inter> W))   "
using PwrFusionInterL[ of "1" "X" "V" "Y" "W"] 
by (simp add: FusionSEmptyR SSkipSFinite)

lemma CH08:
 "((V\<cdot>(SSkip \<inter> X)) \<inter> ((W\<cdot>(SSkip \<inter> Y)))) = ((V \<inter> W)\<cdot>(SSkip \<inter> X \<inter> Y)) "
using PwrFusionInterR[of "V" "1" "X" "W" "Y"] 
by (simp add: FusionSEmptyR SSkipSFinite)

lemma CH09:
 "(((X \<inter> SEmpty)\<cdot>V) \<inter> ((Y \<inter> SEmpty)\<cdot>W)) = (((X \<inter> Y) \<inter> SEmpty)\<cdot>(V \<inter> W))   "
using PwrFusionInterL[ of "0" "X" "V" "Y" "W"] 
by (metis (no_types, lifting) inf_assoc inf_commute pwr_0)

lemma CH10:
 "((V\<cdot>(X \<inter> SEmpty)) \<inter> ((W\<cdot>(Y \<inter> SEmpty)))) = ((V \<inter> W)\<cdot>((X \<inter> Y) \<inter> SEmpty)) "
using PwrFusionInterR[of "V" "0" "X" "W" "Y"]
by (metis (no_types, lifting) inf_assoc inf_commute pwr_0)

lemma ST13:
 " ((X \<inter> SEmpty)\<cdot>Z) \<inter> ((Y \<inter> SEmpty)\<cdot>Z) = ((X \<inter> Y) \<inter> SEmpty)\<cdot>Z"
by (simp add: CH09)

lemma ST15:
 "(SStar (X \<inter> SEmpty)) = SEmpty"
using SStareqv[of "(X \<inter> SEmpty)"]  
by (metis Compl_disjoint2 Powerstarhelp2 SFalseBottom UnionSFalse inf_assoc inf_bot_right sfalse_def 
    smore_def)

lemma ST21:
 "((-X) \<inter> SEmpty) \<union> (X \<inter> SEmpty) = SEmpty "
by blast 

lemma ST24:
 " (SInit X) \<inter> (SInit Y) = (SInit (X \<inter> Y))"
by (simp add: ST13 sinit_def)

lemma ST25:
 " (SInit STrue) = STrue"
by (simp add: sinit_def strue_def FusionSEmptyL)

lemma ST26:
 " (SInit (-X)) \<union> (SInit X) = STrue"
by (metis Compl_disjoint2 ST21 ST25 Un_Int_distrib compl_bot_eq FusionUnionDistL 
          sinit_def strue_def sup_bot.right_neutral sup_top_right) 

lemma ST28:
 " (SDi (SInit X)) = (SInit X)"
by (metis compl_bot_eq FusionAssoc FusionUnionDistR FusionSEmptyR sdi_def 
          sinit_def strue_def sup_top_right UnionCommute)

lemma ST33:
 " (STrue \<inter> SEmpty)\<cdot>SEmpty = SEmpty"
by (simp add: strue_def FusionSEmptyL)

lemma ST36:
 " (SInit (-X)) \<subseteq> -(SInit X)"
unfolding sinit_def
by (metis SFalseBottom SFalseFusion ST24 disjoint_eq_subset_Compl inf.commute inf_compl_bot_left2 
    sinit_def) 

lemma ST37:
 "-(SInit X) \<subseteq> (SInit (-X)) "
using B09 ST26 by auto

lemma ST38:
 " -(SInit X) = (SInit (-X))"
using ST37 ST36 by auto

lemma ST47:
 " X \<union> Y\<cdot>X = (SEmpty \<union> Y)\<cdot>X"
by (simp add: FusionUnionDistL FusionSEmptyL)

lemma SStar01:
 assumes "X\<cdot>(SStar Y) \<union> SEmpty \<subseteq> (SStar Y) "
 shows   "(SStar X) \<subseteq> (SStar Y)"
using assms 
by (metis Un_commute FusionSEmptyR SStarInductL) 

lemma SStar03:
 "(SStar X)\<cdot>(SStar X) \<subseteq> (SStar X) "
by (metis SStarInductL Un_absorb UnfoldL sup.absorb_iff1 sup.right_idem)
 
lemma SStar05:
 assumes "((SStar X)\<cdot>(SStar X)) \<union> SEmpty \<subseteq> (SStar X)"
 shows   " (SStar (SStar X)) \<subseteq> (SStar X) "
using assms 
by (simp add: SStar01) 

lemma SStar12:
 " (SEmpty \<union> (X\<cdot>(SStar X))) \<subseteq> (SStar X)"
using UnfoldL by blast

lemma SStar06:
 "((SStar X)\<cdot>(SStar X)) \<union> SEmpty \<subseteq> (SStar X) "
using SStar03 SStar12 by force

lemma SStar07:
 " (SStar X) \<subseteq> (SStar (SStar X))"
proof -
 have "SStar X \<cdot> SEmpty \<union> SStar X \<cdot> (SStar X \<cdot> SStar SStar X) = SStar X \<cdot> SStar SStar X"
   by (metis (full_types) FusionUnionDistR UnfoldL)
 then have "SStar X \<cdot> SEmpty \<union> SStar X \<cdot> (SStar X \<cdot> SStar SStar X) \<subseteq> SStar SStar X"
   using UnfoldL by auto
 then show ?thesis
   by (simp add: FusionSEmptyR)
qed

lemma SStar08:
 " (SStar X) = (SStar (SStar X))"
by (meson B04 SStar05 SStar06 SStar07)

lemma SStar15:
 " SEmpty \<subseteq> (SStar SSkip)"
using UnfoldL by fastforce

lemma SStar16:
 " SSkip \<subseteq> (SStar SSkip)"
using SSkipSFinite SStarSkip by blast

lemma SStar22:
 " (SEmpty \<inter> X)\<cdot>(SStar (SEmpty \<inter> X)) = (SEmpty \<inter> X)"
by (metis ST15 FusionSEmptyR inf_commute)

lemma SStar23:
 " (SStar (SEmpty \<inter> X)) = SEmpty"
using SStar22 UnfoldL by auto

lemma SStar25:
 " (SStar STrue) = STrue"
by (metis FusionRuleR FusionSEmptyR Un_absorb2 UnfoldL  UnionCommute compl_bot_eq 
    strue_def  sup.right_idem sup_ge2 sup_top_right)

lemma SStar28:
 " (SStar X)\<cdot>X \<subseteq> X\<cdot>(SStar X)"
by (metis B04 FusionUnionDistR FusionSEmptyR UnfoldL SStarInductL)

lemma SStar29:
 " X\<cdot>(SStar X) \<subseteq> (SStar X)\<cdot>X"
by (metis B04 SStar28 SStarInductR UnfoldL FusionRuleL ST47 sup.mono)

lemma SStar17:
 " (SStar SSkip)\<cdot>SSkip \<subseteq> SSkip\<cdot>(SStar SSkip)"
by (simp add: SStar28)

lemma SStar18:
 " SSkip\<cdot>(SStar SSkip) \<subseteq> (SStar SSkip)\<cdot>SSkip"
by (simp add: SStar29)

lemma SStar19:
 " (SStar SSkip)\<cdot>SSkip = SSkip\<cdot>(SStar SSkip)"
using SStar17 SStar18 by auto

lemma SStar30:
 " X\<cdot>(SStar X) = (SStar X)\<cdot>X"
using SStar28 SStar29 by auto

lemma SStar34:
 assumes "SEmpty \<union> (X \<union> Y)\<cdot>((SStar X)\<cdot>(SStar (Y\<cdot>(SStar X))) ) \<subseteq> (SStar X)\<cdot>(SStar (Y\<cdot>(SStar X))) "
 shows  "(SStar (X \<union> Y)) \<subseteq> (SStar X)\<cdot>(SStar (Y\<cdot>(SStar X))) "
by (metis assms FusionSEmptyR SStarInductL)

lemma SStar35:
 "SEmpty \<union> (X \<union> Y)\<cdot>((SStar X)\<cdot>(SStar (Y\<cdot>(SStar X))) ) \<subseteq> (SStar X)\<cdot>(SStar (Y\<cdot>(SStar X)))" 
by (simp add: FusionAssoc FusionUnionDistL ST47 UnfoldL UnionAssoc UnionCommute)

lemma SStar36:
 " (SStar (X \<union> Y)) \<subseteq> (SStar X)\<cdot>(SStar (Y\<cdot>(SStar X)))"
using SStar34 SStar35 by blast

lemma SStar46:
 " (SStar X)\<cdot>(SStar (Y\<cdot>(SStar X))) \<subseteq> (SStar (X \<union> Y))"
proof -
 have "(SEmpty \<union> SStar (X \<union> Y) \<cdot> Y) \<cdot> SStar X \<subseteq> SStar (X \<union> Y)"
   by (metis (no_types) FusionUnionDistR SStar12 SStar30 SStarInductR sup.bounded_iff) 
 then show ?thesis by (simp add: SStarInductR ST47 FusionAssoc) 
qed

lemma SStar47:
 " (SStar Z) = (SStar (Z \<inter> SMore))"
proof -
 have 1: "(SStar Z) = (SStar ((SEmpty \<inter> Z) \<union> (SMore \<inter> Z))) " 
   by (metis Int_Un_distrib2 compl_bot_eq inf_top.left_neutral smore_def strue_def STrueTop)
 have 2: "(SStar ((SEmpty \<inter> Z) \<union> (SMore \<inter> Z))) =
          (SStar (SEmpty \<inter> Z))\<cdot>(SStar ((SMore \<inter> Z)\<cdot>(SStar (SEmpty \<inter> Z)))) " 
   by (simp add: SStar36 SStar46 subset_antisym)
 have 3: "(SStar (SEmpty \<inter> Z))\<cdot>(SStar ((SMore \<inter> Z)\<cdot>(SStar (SEmpty \<inter> Z)))) =
          (SStar (Z \<inter> SMore)) "  
   by (simp add: FusionSEmptyL FusionSEmptyR SStar23 inf_commute) 
 from 1 2 3 show ?thesis by auto
qed

lemma SStar48:
 " (SStar SMore) = STrue"
by (metis Compl_Un Compl_disjoint2 SStar25 SStar47 ST21 ST33 FusionSEmptyR 
          inf.right_idem smore_def strue_def)

lemma SStar50:
assumes "SSkip\<cdot>((-X) \<union> ((SStar SSkip)\<cdot>(X \<inter> (SSkip\<cdot>(-X))))) \<union> (-X)
         \<subseteq> (-X) \<union> (SStar SSkip)\<cdot>(X \<inter> (SSkip\<cdot>(-X))) "
shows " ((SStar SSkip)\<cdot>(-X)) \<subseteq> ((-X) \<union> ((SStar SSkip)\<cdot>(X \<inter> (SSkip\<cdot>(-X)))))"
using SStarInductL assms by blast

lemma SStar51:
 " SSkip\<cdot>((-X) \<union> ((SStar SSkip)\<cdot>(X \<inter> (SSkip\<cdot>(-X))))) \<union> (-X)
         \<subseteq> (-X) \<union> (SStar SSkip)\<cdot>(X \<inter> (SSkip\<cdot>(-X)))"
using FusionUnionDistR[of "SSkip"  ] UnfoldL[of "SSkip"] 
proof -
have f1: "- X \<union> (SEmpty \<union> SSkip \<cdot> SStar SSkip) \<cdot> (X \<inter> (SSkip \<cdot> - X)) \<subseteq> 
          - X \<union> SStar SSkip \<cdot> (X \<inter> (SSkip \<cdot> - X))"
  by (simp add: \<open>SEmpty \<union> SSkip \<cdot> SStar SSkip = SStar SSkip\<close>) (* 0.6 ms *)
have f2: "SSkip \<cdot> - X \<subseteq> - X \<union> SStar SSkip \<cdot> (X \<inter> (SSkip \<cdot> - X))"
  by (metis B20 Morgan ST47 UnionIdem \<open>SEmpty \<union> SSkip \<cdot> SStar SSkip = SStar SSkip\<close> 
    inf_commute inf_idem sup.cobounded1) (* 48 ms *)
have "SSkip \<cdot> (SStar SSkip \<cdot> (X \<inter> (SSkip \<cdot> - X))) \<subseteq> - X \<union> SStar SSkip \<cdot> (X \<inter> (SSkip \<cdot> - X))"
  using f1 by (metis (no_types) FusionAssoc ST47 Un_subset_iff) (* 22 ms *)
then show ?thesis
  using f2 by (simp add: \<open>\<And>Z Y. SSkip \<cdot> (Y \<union> Z) = SSkip \<cdot> Y \<union> SSkip \<cdot> Z\<close>) (* 1 ms *)
qed

lemma SStar52:
 " (SStar X) \<subseteq> SEmpty \<union> (X \<inter> SMore)\<cdot>(SStar X)"
by (metis B04 SStar47 UnfoldL)

lemma SStar53:
 " SEmpty \<union> (X \<inter> SMore)\<cdot>(SStar X) \<subseteq> (SStar X)"
by (metis SStar12 SStar47)

lemma BD45:
 "(SBi ((-X) \<union> X1)) \<inter> (X\<cdot>Y) \<subseteq> X1\<cdot>Y  "
proof -
 have 1: "(SBi ((-X) \<union> X1)) = -(-((-X) \<union> X1)\<cdot>(Y \<union> (-Y)))  " 
   by (metis sbi_def sdi_def STrueTop)
 have 2: "-(-((-X) \<union> X1)\<cdot>(Y \<union> (-Y))) \<inter> (X\<cdot>Y) \<subseteq>
           -(-((-X) \<union> X1)\<cdot>(Y)) \<inter> (X\<cdot>Y)" 
   using FusionUnionDistR by fastforce
  have 3: "-(-((-X) \<union> X1)\<cdot>(Y)) \<inter> (X\<cdot>Y) \<subseteq>  (((-X) \<union> X1) \<inter> X)\<cdot>Y  " 
   by (metis (no_types, opaque_lifting) B20 FusionRuleL FusionUnionDistL Morgan UnionCommute 
          double_compl order_refl)
 have 4: " (((-X) \<union> X1) \<inter> X)\<cdot>Y \<subseteq> X1\<cdot>Y" 
   by (metis B20 double_compl FusionRuleL inf.right_idem inf_le1)
 from 1 2 3 4 show ?thesis by blast
qed

lemma BD46:
 "(SAlways ((-Y) \<union> Y1)) \<inter> (X1\<cdot>Y) \<subseteq> (X1\<cdot>Y1)  "
proof -
 have 1: "(SAlways ((-Y) \<union> Y1)) = -((SFinite \<inter> (X1 \<union> -X1))\<cdot> (-((-Y) \<union> Y1 ) ) ) " 
   by (simp add: salways_def ssometime_def) 
 have 2: " -((SFinite \<inter> (X1 \<union> -X1))\<cdot>(-((-Y) \<union> Y1 ) ) ) =
           -(( (SFinite \<inter> X1) \<union> (SFinite \<inter> -X1))\<cdot>(-((-Y) \<union> Y1 ) ) ) " 
   by (simp add: B28)
 have 3: "-(( (SFinite \<inter> X1) \<union> (SFinite \<inter> -X1))\<cdot>(-((-Y) \<union> Y1 ) ) ) =
         -( (SFinite \<inter> X1)\<cdot>(-((-Y) \<union> Y1 )) \<union> (SFinite \<inter> -X1)\<cdot>(-((-Y) \<union> Y1 )) ) "
   by (simp add: FusionUnionDistL) 
 have 4: "-( (SFinite \<inter> X1)\<cdot>(-((-Y) \<union> Y1 )) \<union> (SFinite \<inter> -X1)\<cdot>(-((-Y) \<union> Y1 )) ) =
          -((SFinite \<inter> X1)\<cdot>(-((-Y) \<union> Y1 ))) \<inter> -((SFinite \<inter> -X1)\<cdot>(-((-Y) \<union> Y1 ))) "
   by blast
 have 5: "(X1\<cdot>Y) = ((SFinite \<inter> X1)\<cdot>Y) \<union> (SInf \<inter> X1 )"
   by (metis Powerstarhelp2 Powerstarhelp3 UnionCommute inf_commute)
 have 6: "-((SFinite \<inter> X1)\<cdot>(-((-Y) \<union> Y1 ))) = 
          -((SFinite \<inter> X1)\<cdot>(Y \<inter> -Y1 ))"
   by simp   
 have 7: "-((SFinite \<inter> X1)\<cdot>(-((-Y) \<union> Y1 )) ) \<inter> ((SFinite \<inter> X1)\<cdot>Y) \<subseteq> 
          (SFinite \<inter> X1)\<cdot>( ((-Y) \<union> Y1) \<inter> Y) "
   by (metis (no_types) B20 FusionRuleR FusionUnionDistR double_compl inf_sup_aci(1) order_refl)
 have 8: "(SFinite \<inter> X1)\<cdot>( ((-Y) \<union> Y1) \<inter> Y) \<subseteq> (SFinite \<inter> X1)\<cdot>Y1"
         by (metis B20 FusionRuleR double_compl inf.cobounded1 inf.right_idem)
 have 9: "(SFinite \<inter> X1)\<cdot>Y1 \<subseteq> X1 \<cdot> Y1"
   by (simp add: FusionRuleL) 
 have 10: "-((SFinite \<inter> X1)\<cdot>(-((-Y) \<union> Y1 )) ) \<inter> (SInf \<inter> X1) \<subseteq> ((SFinite \<inter> X1)\<cdot>Y1) \<union> (SInf \<inter> X1) "
   by blast 
 have 11: " ((SFinite \<inter> X1)\<cdot>Y1) \<union> (SInf \<inter> X1) \<subseteq> X1 \<cdot> Y1"
   by (metis B04 Powerstarhelp2 Powerstarhelp3 UnionCommute inf_commute) 
 have 12: "-((SFinite \<inter> (X1 \<union> -X1))\<cdot> (-((-Y) \<union> Y1 ) ) ) \<inter> (X1\<cdot>Y) =
           (-((SFinite \<inter> (X1 \<union> -X1))\<cdot> (-((-Y) \<union> Y1 ) ) ) \<inter> ((SFinite \<inter> X1)\<cdot>Y)) \<union>
           (-((SFinite \<inter> (X1 \<union> -X1))\<cdot> (-((-Y) \<union> Y1 ) ) ) \<inter> (SInf \<inter> X1)) "
   by (simp add: "5" B28)
 have 13: "(-((SFinite \<inter> (X1 \<union> -X1))\<cdot> (-((-Y) \<union> Y1 ) ) ) \<inter> ((SFinite \<inter> X1)\<cdot>Y)) \<subseteq> X1 \<cdot> Y1"
   using 7 8 9 2 3 by blast
 have 14: " (-((SFinite \<inter> (X1 \<union> -X1))\<cdot> (-((-Y) \<union> Y1 ) ) ) \<inter> (SInf \<inter> X1)) \<subseteq>  X1 \<cdot> Y1"
   using 10 11 by blast  
 have 15: "(-((SFinite \<inter> (X1 \<union> -X1))\<cdot> (-((-Y) \<union> Y1 ) ) ) \<inter> ((SFinite \<inter> X1)\<cdot>Y)) \<union>
           (-((SFinite \<inter> (X1 \<union> -X1))\<cdot> (-((-Y) \<union> Y1 ) ) ) \<inter> (SInf \<inter> X1)) \<subseteq> X1 \<cdot> Y1" 
   using 13 14 by blast
 show ?thesis 
   using "1" "12" "15" by blast
qed 

subsubsection \<open>ITL Axioms derived\<close>

lemma SBoxGen:
 assumes  "X=STrue"
 shows    "(SAlways X) = STrue"
using assms 
by (metis Compl_disjoint2 FusionSFalse double_compl salways_def sfalse_def sfinite_def 
    ssometime_def strue_def)

lemma SBiGen:
 assumes  "X=STrue"
 shows    "(SBi X) = STrue"
using assms 
by (metis double_compl sbi_def sdi_def sfalse_def SFalseFusion strue_def)

lemma SMP:
 assumes "X \<subseteq> Y"
 assumes "X=STrue"
 shows   " Y=STrue"
using assms using strue_def by blast

lemma SChopAssoc:
 "X\<cdot>(Y\<cdot>Z) = (X\<cdot>Y)\<cdot>Z"
by (simp add: FusionAssoc)

lemma SOrChopImp:
 " (X \<union> Y)\<cdot>Z \<subseteq> (X\<cdot>Z) \<union> (Y\<cdot>Z)"
by (simp add: FusionUnionDistL)

lemma SChopOrImp:
 " X\<cdot>(Y \<union> Z) \<subseteq> (X\<cdot>Y) \<union> (X\<cdot>Z)"
by (simp add: FusionUnionDistR)

lemma SEmptyChop:
 " SEmpty\<cdot>X = X"
by (simp add: FusionSEmptyL)

lemma SChopEmpty:
 " X\<cdot>SEmpty = X"
by (simp add: FusionSEmptyR)

lemma SStateImpBi:
 "(SInit X) \<subseteq> (SBi (SInit X)) " 
by (simp add: ST28 ST38 sbi_def)

lemma SNextImpNotNextNot:
 "(SNext X) \<subseteq> -(SNext (-X))"
proof -
 have 1: "((SNext X) \<subseteq> -(SNext (-X))) = (((SNext X) \<inter> (SNext (-X))) \<subseteq> SFalse)" 
   by (simp add: disjoint_eq_subset_Compl sfalse_def)
 have 2: " ((SNext X) \<inter> (SNext (-X))) = SSkip\<cdot>(X \<inter> (-X)) " 
   by (metis CH07 SStar16 inf.orderE snext_def) 
 have 3: "(SSkip)\<cdot>(X \<inter> (-X)) = SSkip\<cdot>SFalse " 
   by (simp add: sfalse_def)
 have 4: " SSkip\<cdot>SFalse = SFalse" 
   using FusionSFalse SStar16 SStarSkip sfalse_def sfinite_def by fastforce
 from 1 2 3 4 show ?thesis by auto
qed

lemma SBiBoxChopImpChop:
 " (SBi ((-X) \<union> X1)) \<inter> (SAlways ((-Y) \<union> Y1))  \<inter> (X\<cdot>Y) \<subseteq> (X1\<cdot>Y1)"
using BD45 BD46 by blast

lemma SBoxInduct:
 "(SAlways (-X \<union> (SWnext X))) \<inter> X \<subseteq> (SAlways X)"
proof -
 have 1: " ((SAlways (-X \<union> (SWnext X))) \<inter> X \<subseteq> (SAlways X)) =
           ((SSometime (-X)) \<subseteq> ((-X) \<union> (SSometime (X \<inter> (SNext (-X))) ))) " 
   by (simp add: salways_def snext_def swnext_def)
   blast
 have 2: "((SSometime (-X)) \<subseteq> ((-X) \<union> (SSometime (X \<inter> (SNext (-X))) ))) =
          ( ((SStar SSkip)\<cdot>(-X)) \<subseteq> ((-X) \<union> ((SStar SSkip)\<cdot>(X \<inter> (SSkip\<cdot>(-X)))))  ) " 
   by (simp add: SStarSkip snext_def ssometime_def)
 have 3: "( ((SStar SSkip)\<cdot>(-X)) \<subseteq> ((-X) \<union> ((SStar SSkip)\<cdot>(X \<inter> (SSkip\<cdot>(-X)))))  ) " 
   using SStar51 SStar50 by blast
 from 1 2 3 show ?thesis by auto
qed

lemma SChopstarEqv:
 " (SStar X) = SEmpty \<union> (X \<inter> SMore)\<cdot>(SStar X)"
using SStar52 SStar53 by blast


subsection \<open>Extra Laws\<close>

subsubsection \<open>Boolean Laws\<close>

lemma B02:
 assumes " -Y \<subseteq> -X"
 shows   "(X:: 'a iintervals) \<subseteq> Y"
using assms by auto

lemma B03:
 "((X:: 'a iintervals) = Y ) \<longleftrightarrow> (-X = -Y) "
by auto

lemma B05:
 assumes "(X:: 'a iintervals) \<union> Y \<subseteq> Z "
 shows   "X \<subseteq> Z \<and> Y \<subseteq> Z "
using assms by auto

lemma B06:
 assumes "X \<subseteq> Z \<and> Y \<subseteq> Z "
 shows   "(X:: 'a iintervals) \<union> Y \<subseteq> Z "
using assms by auto

lemma B07:
 "(X:: 'a iintervals) \<union> Y \<subseteq> Z \<longleftrightarrow> X \<subseteq> Z \<and> Y \<subseteq> Z"
by auto

lemma B08:
 assumes "(X:: 'a iintervals) \<subseteq> Y"
 shows   "-X \<union> Y = STrue"
using assms 
using strue_def by auto

lemma B10:
 "(X:: 'a iintervals) \<subseteq> Y \<longleftrightarrow> -X \<union> Y = STrue"
using strue_def by auto

lemma B11:
 assumes "(X:: 'a iintervals) \<subseteq> Y "
 shows   " X \<inter> -Y = SFalse"
using assms sfalse_def by auto

lemma B12:
 assumes " X \<inter> -Y = SFalse"
 shows   "(X:: 'a iintervals) \<subseteq> Y "
using assms sfalse_def by auto

lemma B13:
 "(X:: 'a iintervals) \<subseteq> Y \<longleftrightarrow> X \<inter> -Y = SFalse"
using sfalse_def by auto

lemma B14:
 assumes "(X:: 'a iintervals) \<subseteq> Y"
 shows   "X \<inter> Y =X"
using assms by auto

lemma B15:
 assumes "(X:: 'a iintervals) \<subseteq> Y \<inter> Z"
 shows   "X \<subseteq> Y \<and> X \<subseteq> Z "
using assms by auto

lemma B16:
 assumes "X \<subseteq> Y \<and> X \<subseteq> Z "
 shows   "(X:: 'a iintervals) \<subseteq> Y \<inter> Z"
using assms by auto

lemma B17:
 "(X:: 'a iintervals) \<subseteq> Y \<inter> Z \<longleftrightarrow> X \<subseteq> Y \<and> X \<subseteq> Z "
by auto

lemma B18:
 assumes "(X:: 'a iintervals) \<subseteq> Y \<union> Z "
 shows   "X \<inter> -Y \<subseteq> Z"
using assms by auto

lemma B19:
 assumes "X \<inter> -Y \<subseteq> Z"
 shows   "(X:: 'a iintervals) \<subseteq> Y \<union> Z "
using assms by auto

lemma B21:
 "(X:: 'a iintervals) \<union> (Y \<inter> Z) \<subseteq> (X \<union> Y) \<inter> (X \<union> Z) \<longleftrightarrow>
  X \<union> (Y \<inter> Z) \<subseteq> (X \<union> Y) \<and> X \<union> (Y \<inter> Z) \<subseteq> (X \<union> Z)"
by auto

lemma B22:
  " (X:: 'a iintervals) \<union> (Y \<inter> Z) \<subseteq> X \<union> Y"
by auto

lemma B23:
 " (X:: 'a iintervals) \<union> (Y \<inter> Z) \<subseteq> X \<union> Z"
by auto

lemma B24:
 "((X:: 'a iintervals) \<union> Y) \<inter> (X \<union> Z) \<subseteq> X \<union> (Y \<inter> Z) \<longleftrightarrow>
  ((X \<union> Y) \<inter> (X \<union> Z)) \<inter> -X \<subseteq> Y \<inter> Z "
by auto

lemma B25:
 "(((X:: 'a iintervals) \<union> Y) \<inter> (X \<union> Z)) \<inter> -X \<subseteq> Y \<inter> Z \<longleftrightarrow>
  ((X \<union> Y) \<inter> (X \<union> Z)) \<inter> -X \<subseteq> Y \<and>
  ((X \<union> Y) \<inter> (X \<union> Z)) \<inter> -X \<subseteq> Z"
by auto

lemma B26:
 "(((X:: 'a iintervals) \<union> Y) \<inter> (X \<union> Z)) \<inter> -X \<subseteq> Y "
by auto

lemma B27:
 "(((X:: 'a iintervals) \<union> Y) \<inter> (X \<union> Z)) \<inter> -X \<subseteq> Z"
by auto

lemma B29:
 "(X:: 'a iintervals) \<union> (Y \<inter> Z) = (X \<union> Y) \<inter> (X \<union> Z) "
by auto


subsubsection \<open>Chop\<close>

lemma CH02:
  " X\<cdot>Y \<inter> -(X\<cdot>Z) \<subseteq> X\<cdot>(Y \<inter> -Z)"
by (metis B20 FusionRuleR FusionUnionDistR inf.right_idem inf_le1)

lemma CH03:
 " X\<cdot>Y \<inter> -(Z\<cdot>Y) \<subseteq> (X \<inter> -Z)\<cdot>Y"
by (metis B20 FusionRuleL FusionUnionDistL inf.right_idem inf_le1)

lemma CH04:
 " X\<cdot>Y \<inter> -(X\<cdot>-Z) \<subseteq> X\<cdot>(Y \<inter> Z)"
using CH02 by fastforce

lemma CH05:
 " X\<cdot>Y \<inter> -(-Z\<cdot>Y) \<subseteq> (X \<inter> Z)\<cdot>Y"
using CH03 by fastforce

lemma CH06:
 assumes "X\<subseteq>X1"
         "Y\<subseteq>Y1"
 shows   "X\<cdot>Y\<subseteq>X1\<cdot>Y1"
using assms
by (metis FusionRuleL FusionRuleR order_trans)
  
lemma CH11:
 " ((X \<inter> (SPower SSkip n))\<cdot>STrue) \<inter> ((SPower SSkip n)\<cdot>Y) = (X \<inter> (SPower SSkip n))\<cdot>Y"
using PwrFusionInterL[of "n" "X" "STrue" "STrue" "Y"] 
by (simp add: inf_commute strue_def) 

lemma CH12:
 " (STrue\<cdot>(X \<inter> (SPower SSkip n))) \<inter> (Y\<cdot>(SPower SSkip n)) = (Y\<cdot>(X \<inter> (SPower SSkip n))) "
using PwrFusionInterR[of "STrue" "n" "X" "Y" "STrue"] 
by (metis STrueTop inf_commute inf_sup_absorb) 

lemma CH13:
 " (SPower SSkip n) \<cdot> (SPower SSkip m) = (SPower SSkip (n+m))"
proof
 (induct n arbitrary: m)
 case 0
 then show ?case by (simp add: FusionSEmptyL)
 next
 case (Suc n)
 then show ?case 
 by (metis FusionAssoc add_Suc pwr_Suc)
qed


subsubsection \<open>Next\<close>

lemma N01:
 " (SNext SEmpty) = SSkip"
by (simp add: FusionSEmptyR snext_def)

lemma N02:
 " (SNext SFalse) = SFalse "
by (metis FusionSFalse SStar16 SStarSkip disjoint_eq_subset_Compl sfalse_def sfinite_def snext_def)

lemma N03:
 " (SNext X)\<cdot>Y = (SNext (X\<cdot>Y))"
by (simp add: snext_def FusionAssoc)

lemma N04:
 " (SNext (X \<union> Y)) = (SNext X) \<union> (SNext Y)"
by (simp add: FusionUnionDistR snext_def)

lemma N05:
 " (SNext (X \<inter> Y)) = (SNext X) \<inter> (SNext Y)"
by (metis CH07 SStar16 inf.orderE snext_def)

lemma N06:
 assumes "X \<subseteq> Y "
 shows   "(SNext X) \<subseteq> (SNext Y)"
using assms 
by (metis FusionUnionDistR Subsumption snext_def)

lemma N07:
 " (SNext ((-X) \<union> Y)) = (SNext (-X)) \<union> (SNext Y)"
by (simp add: N04)

lemma N08:
 " SMore \<subseteq> SSkip\<cdot>STrue"
using SMoreImpSSkipFusion by blast

lemma N23:
 "(SWprev X) \<subseteq>  (SEmpty \<union>  (SPrev X)) "
proof -
have 1: "(X \<inter> SFinite) \<cdot> SSkip \<union> (- X \<inter> SFinite) \<cdot> SSkip = SStar SSkip \<cdot> SSkip"
  by (metis B28 FusionUnionDistL SStarSkip STrueTop boolean_algebra_cancel.inf0 compl_bot_eq 
     inf_commute strue_def)
have 2: "(SEmpty) \<union> STrue \<cdot> SSkip = STrue"
  by (metis FusionSFalse FusionUnionDistL Powerstarhelp2 SStar19 SStarSkip UnfoldL UnionAssoc
      compl_bot_eq compl_sup_top sfinite_def sinf_def strue_def)
have 3: "- SWprev X \<union> (SEmpty \<union> SPrev X) = STrue"
  unfolding swprev_def sprev_def
  by (metis "2" FusionUnionDistL compl_bot_eq compl_sup_top double_compl inf_sup_aci(7) strue_def)
then show ?thesis by (meson B09) 
qed

lemma N24:
 " (SEmpty) \<subseteq> (SWprev X)"
proof -
 have 1: "(- SEmpty \<inter> - (- SEmpty \<cdot> - SEmpty)) \<subseteq> SMore"
   by (simp add: smore_def) 
 have 2: " -X \<cdot> SMore \<subseteq> SMore" 
   by (metis CH01 FusionAssoc1 FusionUnionDistL SMoreImpSSkipFusion SSkipFusionImpSMore 
       SStar30 SStar48 STrueTop subset_antisym sup.orderI sup.right_idem)
 have 3: " - X \<cdot> (- SEmpty \<inter> - (- SEmpty \<cdot> - SEmpty))  \<subseteq> SMore "
   using "1" "2" FusionRuleR by blast
 show ?thesis 
   by (metis "3" compl_le_swap1 compl_sup smore_def sskip_def swprev_def)
qed

lemma N25:
 " (SPrev X) \<subseteq> (SWprev X)"
proof -
 have 1: "((SPrev X) \<subseteq> (SWprev X) ) = (((SPrev X) \<inter> (SPrev (-X)))\<subseteq> SFalse )" 
   by (simp add: B10 sfalse_def sprev_def swprev_def)
 have 2: "((SPrev X) \<inter> (SPrev (-X))) = (X \<inter> (-X))\<cdot>SSkip " 
   by (metis CH08 SStar16 inf.orderE sprev_def)
 have 3: " (X \<inter> (-X))\<cdot>SSkip = SFalse\<cdot>SSkip" 
   by (simp add: sfalse_def)
 have 4: "SFalse\<cdot>SSkip = SFalse " 
   by (simp add: SFalseFusion)
 from 1 2 3 4 show ?thesis by auto
qed

lemma N26:
 "(SWprev X) =  (SEmpty \<union>  (SPrev X)) "
using N23 N24 N25 by blast

lemma N09:
 " SSkip \<union> SMore\<cdot>SSkip \<subseteq> SMore"
proof -
 have 1: " SSkip \<subseteq> SMore" 
   by (simp add: smore_def sskip_def)
 have 2: " SMore\<cdot>SSkip \<subseteq> SMore" 
   by (metis N26 UnionCommute compl_le_swap1 smore_def sup_ge2 swprev_def)
 from 1 2 show ?thesis by auto
qed

lemma N10:
 assumes "SSkip \<union> SMore\<cdot>SSkip \<subseteq> SMore "
 shows   "SSkip\<cdot>(SStar SSkip) \<subseteq> SMore "
using assms SStarInductR N09 by blast

lemma N11:
 " SSkip \<cdot> STrue \<subseteq> SMore"
by (simp add: SSkipFusionImpSMore)

lemma N12:
 " (SNext X) = -(SWnext (-X))"
by (simp add: snext_def swnext_def)

lemma N13:
 " SMore\<cdot>STrue = SMore "
by (metis CH01 FusionAssoc1 SMoreImpSSkipFusion SSkipFusionImpSMore subset_antisym)

lemma N14:
 " STrue\<cdot>SSkip \<subseteq> SMore"
by (metis N09 ST47 STrueTop smore_def)

lemma N15:
 " SMore \<subseteq> STrue\<cdot>SSkip"
proof -
 have 1: "SMore \<subseteq> SSkip \<cdot> STrue" 
   by (simp add: SMoreImpSSkipFusion)
 have 2: "SSkip \<cdot> STrue \<subseteq>  STrue\<cdot>SSkip"
   proof -
    have f3: "SSkip \<cdot> SFinite \<subseteq> STrue \<cdot> SSkip"
      by (metis B19 FusionRuleL SStar19 SStarSkip STrueTop inf_sup_ord(2))
    have "SSkip \<cdot> SInf \<subseteq> STrue \<cdot> SSkip"
      by (metis (no_types, opaque_lifting) B04 Compl_disjoint2 FusionRuleR SChopAssoc SMoreImpSSkipFusion 
          SSkipFusionImpSMore ST47 boolean_algebra_cancel.inf0 compl_inf double_compl 
          inf_commute inf_le1 sfalse_def sinf_def strue_def)
    then show ?thesis
      using f3 by (metis (no_types) FusionUnionDistR STrueTop le_sup_iff sfinite_def)
   qed
 show ?thesis 
   using "2" SMoreImpSSkipFusion by blast
qed
  
lemma N19:
 "(SWnext X) \<subseteq> (SEmpty \<union> (SNext X)) "
proof -
  have "STrue \<subseteq> SSkip \<cdot> (- X) \<union> (SEmpty \<union> SSkip \<cdot> X)"
    by (metis B04 FusionUnionDistR N13 SMoreImpSSkipFusion SSkipFusionImpSMore SStar48 UnfoldL 
        compl_bot_eq compl_sup_top inf_sup_aci(7) strue_def)
  then show ?thesis
    by (simp add: B13 snext_def strue_def swnext_def)
qed

lemma N20:
 " (SEmpty ) \<subseteq> (SWnext X)"
proof -
 have 1: "((SEmpty ) \<subseteq> (SWnext X) ) = (-(SWnext X) \<subseteq> SMore)" 
   by (simp add: smore_def)
 have 2: "(-(SWnext X) \<subseteq> SMore) = ((SNext (-X)) \<subseteq> SMore) " 
   by (simp add: snext_def swnext_def)
 have 3: "(SNext (-X)) \<subseteq> SSkip\<cdot>STrue " 
   by (metis FusionUnionDistR STrueTop snext_def sup.orderI sup.right_idem)
 hence 4: "(SNext (-X)) \<subseteq> SMore " using SSkipFusionImpSMore by auto
 from 1 2 4 show ?thesis by auto
qed

lemma N21:
 " (SEmpty \<union> (SNext X)) \<subseteq> (SWnext X)"
using N20 
by (metis B06 SNextImpNotNextNot snext_def swnext_def)

lemma N22:
 "(SWnext X) = (SEmpty \<union> (SNext X)) "
using N21 N19 by blast

lemma N16:
 " (SNext X) = SMore \<inter> (SWnext X)"
using N12 N22 smore_def by blast

lemma N17:
 " (SWnext (X \<inter> Y)) = (SWnext X) \<inter> (SWnext Y)"
by (simp add: N05 N22 Un_Int_distrib)

lemma N18:
 " (SWnext (X \<union> Y)) = (SWnext X) \<union> (SWnext Y)"
by (simp add: swnext_def)
   (metis CH07 SSkipSFinite compl_inf)

lemma N27:
 " (SNext ((-X) \<union> Y)) \<subseteq> (-(SNext X) \<union> (SNext Y)) "
using N04 SNextImpNotNextNot by blast

lemma N28:
 "(SPrev ((-X) \<union> Y)) \<subseteq> ( -(SPrev X) \<union> (SPrev Y))"
unfolding sprev_def 
proof -
have "\<And>I. (SEmpty) \<union> I \<cdot> SSkip = - (- I \<cdot> SSkip)"
  using N26 sprev_def swprev_def by blast (* 7 ms *)
then have "(Y \<union> - X) \<cdot> SSkip \<subseteq> Y \<cdot> SSkip \<union> - (X \<cdot> SSkip)"
  using FusionUnionDistL by blast (* 2 ms *)
then show "(- X \<union> Y) \<cdot> SSkip \<subseteq> - (X \<cdot> SSkip) \<union> Y \<cdot> SSkip"
  by (simp add: UnionCommute) (* 0.9 ms *)
qed

lemma N29:
 "(SPrev X) = -(SWprev (-X) )"
by (simp add: sprev_def swprev_def)


subsubsection \<open>SInit\<close>

lemma ST01:
 "(X \<inter> SEmpty)\<cdot>Y \<subseteq> Y "
by (metis Int_commute FusionUnionDistL FusionSEmptyL le_iff_sup sup_inf_absorb UnionCommute)

lemma ST02:
 " (X \<inter> SEmpty)\<cdot>Y \<subseteq> (X \<inter> SEmpty)\<cdot>STrue"
by (simp add: FusionRuleR strue_def)

lemma ST03:
 "(X \<inter> SEmpty)\<cdot>(X \<inter> SEmpty) \<subseteq> (X \<inter> SEmpty) "
using ST01 by auto

lemma ST04:
 "(X \<inter> SEmpty) \<subseteq> (X \<inter> SEmpty)\<cdot>(X \<inter> SEmpty) "
proof -
 have "\<forall>S Sa Sb Sc. (Sc) \<cdot> (Sb \<inter> SEmpty) \<inter> (Sa \<cdot> (S \<inter> SEmpty)) = Sc \<inter> Sa \<cdot> (Sb \<inter> (S \<inter> SEmpty))"
   by (simp add: CH10 inf_assoc)
 then have "\<forall>S Sa. (Sa) \<cdot> (S \<inter> SEmpty) \<cdot> (S \<inter> SEmpty) = Sa \<cdot> (S \<inter> SEmpty)"
   by (metis FusionSEmptyR inf.idem inf_commute)
 then show ?thesis
   by (metis FusionSEmptyL subset_refl)
qed

lemma ST05:
 "(X \<inter> SEmpty) \<subseteq> -((-X) \<inter> SEmpty) "
by blast

lemma ST06:
 "((-X) \<inter> SEmpty) \<subseteq> -(X \<inter> SEmpty) "
by auto

lemma ST07:
 "(X \<inter> SEmpty) \<inter> (Y \<inter> SEmpty) \<subseteq> (X \<inter> SEmpty)\<cdot>STrue "
using ST02 FusionSEmptyR by blast

lemma ST08:
 "(X \<inter> SEmpty) \<inter> (Y \<inter> SEmpty) \<subseteq> (STrue \<inter> SEmpty)\<cdot>(Y \<inter> SEmpty) "
by (metis FusionSEmptyL FusionSEmptyR ST33 inf.cobounded2)

lemma ST09:
 "((X \<inter> SEmpty)\<cdot>STrue) \<inter> (STrue \<inter> SEmpty)\<cdot>(Y \<inter> SEmpty) \<subseteq> (X \<inter> SEmpty)\<cdot>(Y \<inter> SEmpty)"
by (metis compl_bot_eq eq_refl FusionSEmptyR inf.commute inf_top.left_neutral CH09 strue_def)

lemma ST10:
 "(X \<inter> SEmpty)\<cdot>(Y \<inter> SEmpty) \<subseteq> (X \<inter> SEmpty) "
by (metis FusionRuleR FusionSEmptyR inf_le2)

lemma ST11:
 "(X \<inter> SEmpty)\<cdot>(Y \<inter> SEmpty) \<subseteq> (Y \<inter> SEmpty) "
using ST01 by blast

lemma ST12:
 "(X \<inter> SEmpty) \<inter> (Y \<inter> SEmpty) = (X \<inter> SEmpty)\<cdot>SEmpty \<inter> (Y \<inter> SEmpty)\<cdot>SEmpty "
by (simp add: FusionSEmptyR)

lemma ST14:
 " ((X \<inter> Y) \<inter> SEmpty)\<cdot>SEmpty = ((X \<inter> Y) \<inter> SEmpty) "
by (simp add: FusionSEmptyR)

lemma ST16:
 " (X \<inter> SEmpty) \<inter> (Y \<inter> SEmpty) \<subseteq> SEmpty"
by (simp add: le_infI2)

lemma ST17:
 " (X \<inter> SEmpty)\<cdot>(Y \<inter> SEmpty) \<subseteq> SEmpty"
using ST10 by auto

lemma ST18:
 "-((X \<inter> SEmpty) \<union> (Y \<inter> SEmpty)) = -(X \<inter> SEmpty) \<inter> -(Y \<inter> SEmpty)"
by auto

lemma ST19:
 " (X \<inter> SEmpty)\<cdot>((-X) \<inter> SEmpty) \<subseteq> (X \<inter> SEmpty) "
using ST10 by blast

lemma ST20:
 " (X \<inter> SEmpty)\<cdot>((-X) \<inter> SEmpty) \<subseteq> ((-X) \<inter> SEmpty)"
using ST01 by auto

lemma ST22:
 " ((X \<inter> SEmpty)\<cdot>SSkip)\<cdot>(Y \<inter> SEmpty) \<subseteq> (X \<inter> SEmpty)\<cdot>SSkip"
using FusionRuleR FusionSEmptyR by blast

lemma ST23:
 " ((X \<inter> SEmpty)\<cdot>SSkip)\<cdot>(Y \<inter> SEmpty) \<subseteq> SSkip\<cdot>(Y \<inter> SEmpty)"
by (simp add: ST01 FusionRuleL)

lemma ST27:
 " (SInit X) \<inter> (Y\<cdot>Z) \<subseteq> ((SInit X) \<inter> Y)\<cdot>Z"
by (metis B04 compl_bot_eq FusionAssoc FusionSEmptyL inf_commute inf_top.left_neutral 
          CH09 sinit_def strue_def)

lemma ST29:
 " (SInit X)\<cdot>Y \<subseteq> (SInit X) "
using ST02 FusionAssoc sinit_def by fastforce

lemma ST30:
 " (SInit X) \<inter> (SDi Y) = (SDi ( (SInit X) \<inter> Y))" 
unfolding sdi_def sinit_def strue_def 
by (metis CH09 FusionAssoc FusionSEmptyL compl_bot_eq inf_top.left_neutral)

lemma ST31:
 " (X\<cdot>(STrue \<inter> SEmpty)) \<inter> (STrue\<cdot>(Y \<inter> SEmpty)) = X\<cdot>(Y \<inter> SEmpty) "
by (metis Int_commute compl_bot_eq inf_top.right_neutral CH10 strue_def)

lemma ST32:
 "(STrue \<inter> SEmpty)\<cdot>SEmpty \<inter> (SInit X) = (X \<inter> SEmpty) "
proof -
 have "\<forall>S Sa. (Sa) \<inter> (S \<inter> Sa) = S \<inter> Sa"
   by fastforce
 then have f1: "\<forall>S Sa. (Sa) \<inter> (S \<inter> SEmpty) \<subseteq> Sa \<inter> SEmpty \<cdot> STrue"
   by (metis (no_types) ST07 inf_assoc)
 have f4: "\<forall>S. - (- (S::'a iintervals)) = S"
   by blast
 have f6: "\<forall>S. (SEmpty) \<inter> (S \<inter> (S \<inter> SEmpty \<cdot> STrue)) = S \<inter> SEmpty"
   using f1 by auto
 have f7: "\<forall>S. (SEmpty) \<inter> (SEmpty \<inter> S \<cdot> STrue) \<subseteq> S"
   using  f4 by (metis CH11 FusionSEmptyR inf_aci(1) inf_le2 pwr_0)
 have "\<forall>S. (SEmpty) \<inter> (S \<inter> (SEmpty \<inter> S \<cdot> STrue)) = SEmpty \<inter> S"
   using f6 by (simp add: inf_commute)
 then have "SEmpty \<inter> (SEmpty \<inter> X \<cdot> STrue) = SEmpty \<inter> X"
   using f7 by auto
 then show ?thesis
   by (metis (no_types) ST33 inf_commute sinit_def)
qed

lemma ST34:
 "((X \<inter> SEmpty)\<cdot>Y) = (SInit X) \<inter> Y "
by (metis FusionSEmptyL Int_commute CH09 compl_bot_eq inf_top_right sinit_def strue_def)

lemma ST35:
 "((SInit X) \<inter> Y)\<cdot>Z \<subseteq> (SInit X) \<inter> (Y\<cdot>Z) "
by (metis B04 ST34 FusionAssoc)
 
lemma ST39:
 "SEmpty \<inter> (SInit X) \<subseteq> (X \<inter> SEmpty) "
using ST32 by blast

lemma ST40:
 " (X \<inter> SEmpty) \<subseteq> SEmpty \<inter> (SInit X)"
using ST32 by auto

lemma ST41:
 " SEmpty \<inter> (SInit X) = (X \<inter> SEmpty)"
using ST40 ST39 by auto

lemma ST42:
 " (X \<inter> SEmpty) \<subseteq> ((X \<union> Y) \<inter> SEmpty)"
by blast

lemma ST43:
 " (Y \<inter> SEmpty) \<subseteq> ((X \<union> Y) \<inter> SEmpty)"
by blast

lemma ST44:
 " (X \<inter> SEmpty) \<inter> ((-X) \<inter> SEmpty) = SFalse"
by (simp add: sfalse_def) 

lemma ST45:
 "((X \<union> Y) \<inter> SEmpty) \<subseteq> (X \<inter> SEmpty) \<union> (Y \<inter> SEmpty)"
by auto

lemma ST46:
 " (SInit X) \<union> (SInit Y) = (SInit (X \<union> Y))"
by (simp add: Int_Un_distrib2 FusionUnionDistL sinit_def)

lemma ST48:
 " -(STrue\<cdot>(X \<inter> SEmpty)) \<subseteq> STrue\<cdot>((-X) \<inter> SEmpty)"
by (metis B09 FusionSEmptyR FusionUnionDistR ST21 double_compl)


subsubsection \<open>SStar\<close>

lemma SStar02:
 assumes " X \<subseteq> Y "
 shows   "X\<cdot>(SStar Y) \<union> SEmpty \<subseteq> (SStar Y)"
using assms UnfoldL[of Y] 
by (metis B05 B06 FusionRuleL Subsumption sup.cobounded2)   

lemma SStar04:
 " (SStar X) \<subseteq> (SStar X)\<cdot>(SStar X)"
by (metis Un_absorb UnfoldL UnionAssoc ST47 sup.absorb_iff2)

lemma SStar09:
 assumes " (X\<cdot>(SEmpty \<union> (X\<cdot>(SStar X)))) \<union> SEmpty \<subseteq> (SEmpty \<union> (X\<cdot>(SStar X))) "
 shows   " (SStar X) \<subseteq> SEmpty \<union> (X\<cdot>(SStar X))"
using assms 
by (simp add: UnfoldL) 

lemma SStar10:
 " (X\<cdot>(SEmpty \<union> (X\<cdot>(SStar X)))) \<subseteq> (SEmpty \<union> (X\<cdot>(SStar X)))"
by (metis UnfoldL sup_ge2)

lemma SStar11:
 " SEmpty \<subseteq> (SEmpty \<union> (X\<cdot>(SStar X)))"
by auto

lemma SStar13:
 " (SStar SSkip) = SFinite"
by (simp add: SStarSkip)

lemma SStar14:
 " (SSometime X) = (SStar SSkip)\<cdot>X"
by (simp add: SStarSkip ssometime_def)

lemma SStar20:
 " (SStar SEmpty) = SEmpty"
by (metis FusionSEmptyR ST15 ST33)

lemma SStar21:
 " (SStar (SEmpty \<inter> X))\<cdot>(SEmpty \<inter> X) = (SEmpty \<inter> X)"
by (metis ST15 FusionSEmptyL inf_commute)

lemma SStar24:
 " (SStar SFalse) = SEmpty"
by (metis SStar20 SStar47 inf_compl_bot sfalse_def smore_def)

lemma SStar26:
 " X \<subseteq> (SStar X)"
using UnfoldL[of X] 
by (metis B05 FusionSEmptyR FusionUnionDistR SStar10)

lemma SStar27:
 " SEmpty \<subseteq> (SStar X)"
using UnfoldL by blast

lemma SStar31:
 assumes "X \<union> (X\<cdot>Y)\<cdot>(X\<cdot>(SStar (Y\<cdot>X))) \<subseteq> X\<cdot>(SStar (Y\<cdot>X))"
 shows   "(SStar (X\<cdot>Y))\<cdot>X \<subseteq> X\<cdot>(SStar (Y\<cdot>X)) "
using assms SStarInductL by blast

lemma SStar32:
 "X \<union> (X\<cdot>Y)\<cdot>(X\<cdot>(SStar (Y\<cdot>X))) \<subseteq> X\<cdot>(SStar (Y\<cdot>X)) "
by (metis B06 SStar10 SStar11 FusionAssoc FusionRuleR FusionSEmptyR UnfoldL)

lemma SStar33:
 "(SStar (X\<cdot>Y))\<cdot>X \<subseteq> X\<cdot>(SStar (Y\<cdot>X))"
 by (meson SStar32 SStarInductL)

lemma SStar37:
 assumes "X\<cdot>Z \<subseteq> Z\<cdot>Y "
 shows   "(SStar X)\<cdot>Z \<subseteq> Z\<cdot>(SStar Y) "
proof -
have "Z \<cdot> SStar Y = Z \<cdot> SEmpty \<union> Z \<cdot> (Y \<cdot> SStar Y)"
  by (metis FusionUnionDistR UnfoldL) 
then have "Z \<cdot> SStar Y \<union> (Z \<union> X \<cdot> (Z \<cdot> SStar Y)) = Z \<union> Z \<cdot> Y \<cdot> SStar Y \<union> X \<cdot> Z \<cdot> SStar Y"
  using FusionAssoc FusionSEmptyR by blast 
then have "Z \<cdot> SStar Y \<union> (Z \<union> X \<cdot> (Z \<cdot> SStar Y)) = Z \<cdot> SStar Y"
  by (metis (no_types) FusionAssoc FusionSEmptyR FusionUnionDistL FusionUnionDistR UnfoldL UnionAssoc
      assms sup.absorb_iff1) 
then show ?thesis
  by (meson SStarInductL sup.absorb_iff1) 
qed

lemma SStar38:
 assumes " Z\<cdot>X \<subseteq> Y\<cdot>Z"
 shows   "Z\<cdot>(SStar X) \<subseteq> (SStar Y)\<cdot>Z "
using assms 
proof -
have f1: "Z \<union> SStar Y \<cdot> Y \<cdot> Z = SStar Y \<cdot> Z"
  by (metis (no_types) SStar30 ST47 UnfoldL) 
have "SStar Y \<cdot> Y \<cdot> Z = SStar Y \<cdot> Z \<cdot> X \<union> SStar Y \<cdot> Y \<cdot> Z"
  by (metis FusionAssoc FusionUnionDistR assms subset_Un_eq) 
then have "Z \<union> SStar Y \<cdot> Z \<cdot> X \<subseteq> SStar Y \<cdot> Z"
  using f1 by blast 
then show ?thesis
  by (simp add: SStarInductR) 
qed

lemma SStar39:
 " Y\<cdot>(SStar ((SStar X)\<cdot>Y)) \<subseteq> (SStar (Y\<cdot>(SStar X)))\<cdot>Y"
by (simp add: SStar38 FusionAssoc)

lemma SStar40:
 " (SStar (Y\<cdot>(SStar X)))\<cdot>Y \<subseteq> Y\<cdot>(SStar ((SStar X)\<cdot>Y))"
by (simp add: SStar33)

lemma SStar41:
 "Y\<cdot>(SStar ((SStar X)\<cdot>Y)) = (SStar (Y\<cdot>(SStar X)))\<cdot>Y "
using SStar39 SStar40 by blast

lemma SStar42:
 " Z\<cdot>(SStar (Y\<cdot>Z)) \<subseteq> (SStar (Z\<cdot>Y))\<cdot>Z"
by (simp add: SStar38 FusionAssoc)

lemma SStar43:
 " (SStar (Z\<cdot>Y))\<cdot>Z \<subseteq> Z\<cdot>(SStar (Y\<cdot>Z))"
by (simp add: SStar33)

lemma SStar44:
 " Z\<cdot>(SStar (Y\<cdot>Z)) = (SStar (Z\<cdot>Y))\<cdot>Z"
using SStar42 SStar43 by blast

lemma SStar49:
 " (SStar X) = SEmpty \<union> (SStar X)\<cdot>X"
using SStar30 UnfoldL  by blast


subsubsection \<open>Box and Diamond\<close>

lemma BD01:
 " (SSometime SEmpty) = SFinite"
by (simp add: ssometime_def FusionSEmptyR)

lemma BD02:
 " X \<subseteq> (SSometime X)"
unfolding ssometime_def 
by (metis SStar15 SStarSkip ST47 subset_Un_eq)

lemma BD03: 
 " (SNext (SSometime X)) \<subseteq> (SSometime X)"
by (metis FusionUnionDistL N03 SStar16 SStar03 SStar04 SStarSkip snext_def ssometime_def 
          sup.absorb_iff2 sup.orderE) 
lemma BD04:
 " (SSometime (SNext X)) \<subseteq> (SSometime X)"
by (metis BD03 FusionAssoc SStar28 SStar29 SStarSkip inf_sup_aci(5) snext_def ssometime_def sup.orderE)

lemma BD05:
 " (SSometime X) \<union> (SSometime Y) = (SSometime (X \<union> Y))"
by (simp add: FusionUnionDistR ssometime_def)

lemma BD06:
 " (SSometime STrue) = STrue"
using BD02 SMP by blast

lemma BD07:
 " (SSometime (X \<inter> Y)) \<subseteq> (SSometime X) \<inter> (SSometime Y)"
by (simp add: FusionRuleR ssometime_def)

lemma BD08:
 " (SAlways STrue) = STrue"
by (simp add: SBoxGen)

lemma BD09:
 " -(SAlways X) = (SSometime (-X))"
by (simp add: salways_def)

lemma BD10:
 " (SAlways X) \<subseteq> (SSometime X)"
by (metis B02 BD02 BD09 set_rev_mp subsetI)

lemma BD11:
 " (SSometime (SSometime X)) = (SSometime X)"
by (metis FusionAssoc SStar03 SStar04 SStarSkip ssometime_def subset_antisym)

lemma BD12:
 " (SAlways X) \<subseteq> X"
by (simp add: B02 BD02 BD09)

lemma BD13:
 " (SDi STrue) = STrue"
by (simp add: CH01 sdi_def)

lemma BD14:
 " (SDi SEmpty) = STrue"
by (simp add: sdi_def FusionSEmptyL)

lemma BD15:
 " (SBi STrue) = STrue"
by (simp add: SBiGen) 

lemma BD16:
 " (SDi (X \<union> Y)) = (SDi X) \<union> (SDi Y)"
by (simp add: FusionUnionDistL sdi_def)

lemma BD17:
 assumes " X \<subseteq> Y"
 shows   "(SDi X) \<subseteq> (SDi Y)"
using assms 
by (metis FusionUnionDistL Subsumption sdi_def)

lemma BD18:
 " (SDi (SDi X)) = (SDi X)"
by (metis CH01 FusionAssoc sdi_def)

lemma BD19:
 " (SDa SEmpty) = STrue"
by (metis BD01 BD06 sda_def ssometime_def)

lemma BD20:
 " (SDa STrue) = STrue"
by (metis BD06 CH01 sda_def ssometime_def)

lemma BD21:
 " (SBa STrue) = STrue"
by (metis BD15 BD08 BD09 sba_def sbi_def sda_def sdi_def ssometime_def)

lemma BD22:
 " (SDa (X \<union> Y)) = (SDa X) \<union> (SDa Y)"
by (simp add: FusionUnionDistL FusionUnionDistR sda_def) 

lemma BD23:
 assumes " X \<subseteq> Y" 
 shows   " (SDa X) \<subseteq> (SDa Y)"
using assms 
by (metis BD22 Subsumption)

lemma BD24:
 assumes " X \<subseteq> Y"
 shows   " (SDa (-Y)) \<subseteq> (SDa (-X))"
using assms 
by (simp add: BD23)

lemma BD25:
 " (SDi X) \<subseteq> (SDa X)"
by (metis BD02 FusionAssoc sda_def sdi_def ssometime_def)

lemma BD26:
 " (SSometime X) \<subseteq> (SDa X)"
by (metis B08 B12 FusionRuleR FusionSEmptyR SFalseBottom double_compl inf.absorb_iff2 inf_le1 
     sda_def ssometime_def sup_inf_absorb)

lemma BD27:
 " (SBa X) \<subseteq> (SBi X)"
by (simp add: BD25 sba_def sbi_def)

lemma BD28:
 " (SBa X) \<subseteq> (SAlways X)"
by (simp add: B02 BD26 BD09 sba_def) 

lemma BD29:
 " (SAlways X) \<inter> (SAlways Y) = (SAlways (X \<inter> Y)) "
proof -
 have 1: "(SAlways X) \<inter> (SAlways Y) = - SSometime (- X) \<inter> - SSometime (- Y)" 
   by (simp add : salways_def) 
 have 2: " SSometime (- X) \<union>  SSometime (- Y) = SSometime(-X \<union> -Y)" 
   using BD05 by blast
 have 3: "- SSometime(-X \<union> -Y) = (SAlways (X \<inter> Y))" 
   by (simp add : salways_def) 
 show ?thesis using "1" "2" "3" by auto
qed

lemma BD30:
 " (SAlways X) \<union> (SAlways Y) \<subseteq> (SAlways (X \<union> Y))"
using BD07 
by (metis B02 BD09 compl_sup)

lemma BD31:
 " (SDi (X \<inter> Y)) \<subseteq> (SDi X) \<inter> (SDi Y)"
by (simp add: BD17)

lemma BD32:
 " (SBi X) \<union> (SBi Y) \<subseteq> (SBi (X \<union> Y))"
using  BD31 
by (metis (mono_tags, lifting) B02 compl_sup double_compl sbi_def)

lemma BD33:
 " (SDa (X \<inter> Y)) \<subseteq> (SDa X) \<inter> (SDa Y)"
by (simp add: BD23)

lemma BD34:
 " (SBa X) \<union> (SBa Y) \<subseteq> (SBa (X \<union> Y))"
using BD33 
by (metis (mono_tags, lifting) B02 compl_sup double_compl sba_def)

lemma BD35:
 " (SAlways SEmpty) = SEmpty"
by (metis B08 BD08 BD12 FusionSEmptyR N20 SBoxInduct ST33 sup.absorb_iff2 sup.orderE)

lemma BD36:
 " (SBi SEmpty) = SEmpty"
proof -
 have 1: "- (- SEmpty \<cdot> STrue) \<subseteq> SEmpty"
   by (metis B04 N13 double_compl smore_def) 
 have 2: "SEmpty \<subseteq> - (- SEmpty \<cdot> STrue)"
   using N13 smore_def by fastforce
 show ?thesis
   by (simp add: "1" "2" B04 sbi_def sdi_def)
qed

lemma BD37:
 " (SBa SEmpty) = SEmpty"
by (metis BD09 BD35 BD36 sba_def sbi_def sda_def sdi_def ssometime_def)

lemma BD38:
 assumes " X \<subseteq> Y"
 shows   " (SAlways X) \<subseteq> (SAlways Y)"
using assms 
by (simp add: BD29 inf.absorb_iff2)

lemma BD39:
 assumes " X \<subseteq> Y"
 shows   " (SBi X) \<subseteq> (SBi Y)"
using assms 
by (simp add: BD17 sbi_def)

lemma BD40:
 assumes " X \<subseteq> Y"
 shows   " (SBa X) \<subseteq> (SBa Y)"
using assms 
by (simp add: BD24 sba_def)

lemma BD41:
 " (SBi (SBi X)) = (SBi X)"
by (simp add: BD18 sbi_def)

lemma BD42:
 " (SAlways (SAlways X)) = (SAlways X)"
by (simp add: BD11 salways_def)

lemma BD43:
 " (SDa (SDa X)) = (SDa X)"
by (metis BD11 CH01 FusionAssoc sda_def ssometime_def)

lemma BD44:
 " (SBa (SBa X)) = (SBa X)"
by (simp add: BD43 sba_def)

lemma BD47:
 " (SAlways ( (-X) \<union> Y)) \<subseteq> ( -(SAlways X) \<union> (SAlways Y)) "
by (metis B20 BD12 BD29 BD38 BD42 double_compl)

lemma BD48:
 " (SAlways X) \<subseteq> X \<inter> (SWnext (SAlways X))"
by (metis B02 B16 BD03 BD09 BD12 N12 salways_def)

lemma BD49:
 " (SBi ( (-X) \<union> Y)) \<subseteq> (-(SBi X) \<union> (SBi Y) )"
by (metis B20 BD45 Un_commute double_complement sbi_def sdi_def)

lemma BD50: 
 " (SPrev (SDi X)) \<subseteq> (SDi X)"
by (simp add: FusionAssoc1 FusionRuleR sdi_def sprev_def strue_elim subsetI)

lemma BD51: 
 " -(SBi X) = (SDi (-X))"
by (simp add: sbi_def)

lemma BD52:
 " X \<subseteq> (SDi X)"
by (metis FusionSEmptyR FusionUnionDistR ST33 Subsumption UnionCommute sdi_def sup_inf_absorb)

lemma BD53:
 " (SBi X) \<subseteq> X"
by (simp add: B02 BD51 BD52)

lemma BD54:
 " (SBi X) \<subseteq> X \<inter> (SWprev (SBi X))"
by (metis B02 B16 BD50 BD51 BD53 N29 sbi_def)

lemma BD55:
 " (SBi (SMore \<union> X)) = (SInit X)"
by (metis (no_types, lifting) ST38 compl_sup double_complement inf_commute sbi_def sdi_def
     sinit_def smore_def)

lemma BD56:
 " (SAlways (SMore \<union> X)) = STrue\<cdot>(X \<inter> SEmpty)"
proof -
 have 1: " ((SFinite\<cdot>(X \<inter> SEmpty)) \<union> SInf) = (STrue\<cdot>(X \<inter> SEmpty))  "
   by (metis Powerstarhelp2 Powerstarhelp3 UnionCommute boolean_algebra_cancel.inf0 compl_bot_eq 
      inf_commute strue_def)
 have 11: "SInf \<union> SFinite \<cdot> (SEmpty \<inter> X \<union> SEmpty \<inter> - X) = STrue"
   by (simp add: B28 FusionSEmptyR sfinite_def strue_def)
 have 2: "(SAlways (SMore \<union> X)) \<inter> SFinite \<subseteq> SFinite\<cdot>(X \<inter> SEmpty)"
   using 11 
   by (simp add: B09 BD09 FusionUnionDistR UnionCommute inf_commute inf_sup_aci(7) 
       smore_def ssometime_def sfinite_def)
 have 3: "(SAlways (SMore \<union> X)) \<subseteq> ((SFinite\<cdot>(X \<inter> SEmpty)) \<union> SInf)"
   using "2" sfinite_def by fastforce
 have 4: "(SAlways (SMore \<union> X)) \<subseteq>  STrue\<cdot>(X \<inter> SEmpty)" 
   using "1" "3" by blast
 have 5: "SFinite\<cdot>(X \<inter> SEmpty) \<subseteq> (SAlways (SMore \<union> X))" 
   unfolding salways_def ssometime_def  smore_def 
   using CH10[of SFinite X SFinite] FusionSFalse 
   by (metis (no_types, opaque_lifting) SFalseBottom compl_sup disjoint_eq_subset_Compl double_compl 
        inf_commute inf_le2 sfinite_def) 
 have 6: "SInf \<subseteq> (SAlways (SMore \<union> X))"
   by (metis B05 BD30 FusionSEmptyR double_compl salways_def sfinite_def smore_def ssometime_def)   
 show ?thesis 
   using "1" "3" "5" "6" by auto  
qed

lemma BD57:
 " (SAlways H) \<subseteq> SAlways (-G \<union> ((SAlways H)\<inter> G))"
proof -
 have 1: " (SAlways H) \<subseteq> (-G \<union> ((SAlways H)\<inter> G))"
   by blast
 have 2: "SAlways (SAlways H) \<subseteq> SAlways (-G \<union> ((SAlways H)\<inter> G))" 
   by (simp add: "1" BD38)
 have 3: "SAlways (SAlways H) = (SAlways H)" 
   by (simp add: BD42)
 show ?thesis using "2" "3" by blast
qed

lemma BD58:
 " ((SAlways H) \<inter> (F\<cdot>G)) \<subseteq> (F\<cdot>((SAlways H) \<inter> G)) "
proof -
 have 1: "SAlways (-G \<union> ((SAlways H)\<inter> G)) \<inter> (F\<cdot>G) \<subseteq> (F\<cdot>((SAlways H) \<inter> G))"
   using BD46 by blast 
 show ?thesis using "1" BD57 by blast
qed


subsubsection \<open>Finite and Infinite\<close>   
 
lemma FI01:
 "SFinite\<cdot> SFinite = SFinite"
by (metis BD01 BD11 ssometime_def) 

lemma FI02:
 "(X \<inter> SFinite)\<cdot>(Y \<inter> SFinite) \<subseteq> (X\<cdot>Y) \<inter> SFinite"
 by (metis B16 CH06 FI01 inf.cobounded1 inf_le2)

lemma FI03:
 "(X\<cdot>Y) \<inter> SFinite \<subseteq> (X \<inter> SFinite)\<cdot>(Y \<inter> SFinite)"
proof -
 have 1: "(X\<cdot>Y) = (X \<inter> SFinite)\<cdot>(Y \<inter> SFinite) \<union> (X \<inter> SFinite)\<cdot>(Y \<inter> SInf) 
           \<union> (X \<inter> SInf)\<cdot>(Y \<inter> SFinite) \<union> (X \<inter> SInf)\<cdot>(Y \<inter> SInf)"
   by (metis FusionUnionDistR Powerstarhelp2 Powerstarhelp3 SInfSFinite UnionCommute sup.right_idem)
 have 2: "((X \<inter> SFinite)\<cdot>(Y \<inter> SFinite)) \<inter> SFinite \<subseteq> (X \<inter> SFinite)\<cdot>(Y \<inter> SFinite)"
   by auto
 have 3: "(X \<inter> SFinite)\<cdot>(Y \<inter> SInf) \<inter> SFinite \<subseteq> (X \<inter> SFinite)\<cdot>(Y \<inter> SFinite)"
   by (metis (no_types, lifting) B20 FusionAssoc FusionSFalse inf_le2 sfinite_def subset_trans sup_ge1)
 have 4: "(X \<inter> SInf)\<cdot>(Y \<inter> SFinite)  \<inter> SFinite \<subseteq> (X \<inter> SFinite)\<cdot>(Y \<inter> SFinite)"
   using Powerstarhelp2 sfinite_def by fastforce
 have 5: "(X \<inter> SInf)\<cdot>(Y \<inter> SInf) \<inter> SFinite \<subseteq> (X \<inter> SFinite)\<cdot>(Y \<inter> SFinite)"
   by (metis "4" Powerstarhelp2) 
 have 6: "(X\<cdot>Y) \<inter> SFinite = ((X \<inter> SFinite)\<cdot>(Y \<inter> SFinite) \<inter> SFinite) \<union>
          ((X \<inter> SFinite)\<cdot>(Y \<inter> SInf)  \<inter> SFinite) \<union> 
          ((X \<inter> SInf)\<cdot>(Y \<inter> SFinite) \<inter> SFinite) \<union>
          ((X \<inter> SInf)\<cdot>(Y \<inter> SInf) \<inter> SFinite)"
   using 1 by auto 
 from 6 2 3 4 5 show ?thesis by auto
qed

lemma FI04:
 "(X \<inter> SFinite)\<cdot>(Y \<inter> SFinite) = (X\<cdot>Y) \<inter> SFinite "
using FI03 FI02 by fastforce
 
lemma FI05:
 "SAlways SMore \<subseteq> SInf"
by (metis B04 BD56 Compl_disjoint2 sfalse_def sinf_def smore_def sup.orderE) 

lemma FI06:
 "SEmpty \<subseteq> SFinite"
using BD01 BD02 by auto

lemma FI07:
 "SSkip\<cdot>SFinite \<subseteq> SFinite"
 using SStarSkip UnfoldL by fastforce

lemma FI08:
 "SFinite\<cdot>SSkip \<subseteq> SFinite"
by (metis FI07 SStar19 SStarSkip)

lemma FI09:
 "SFinite\<cdot>SSkip = SFinite \<inter> SMore"
proof -
 have 1: "SFinite\<cdot>SSkip \<subseteq> SFinite \<inter> SMore" 
   by (metis B16 FI07 N09 N10 SStar19 SStarSkip)
 have 2: " SFinite \<inter> SMore \<subseteq> SFinite\<cdot>SSkip" 
   by (metis B20 FI01 FI02 SStar19 SStarSkip UnfoldL inf.idem smore_def)
 show ?thesis using 1 2 by blast
qed

lemma FI10:
 "SFinite\<cdot> SSkip = SSkip \<cdot>SFinite"
by (metis SStar19 SStarSkip) 

lemma FI11:
 "SFinite \<inter> SEmpty = SEmpty"
by (simp add: FI06 Int_absorb2 inf_sup_aci(1))

lemma FI12:
 "SInf\<cdot>SInf = SInf"
by (metis FusionSFalse Powerstarhelp2 sinf_def)

lemma FI13:
 "SFinite\<cdot>SInf = SInf"
 by (metis BD06 FusionAssoc1 sinf_def ssometime_def)

lemma FI14:
 "SInf \<cdot> SFinite = SInf"
by (metis FusionSFalse Powerstarhelp2 sinf_def)

lemma FI15:
 "SInf = - SFinite"
 by (simp add: sfinite_def)

lemma FI16:
 "SFinite = - SInf"
 by (simp add: sfinite_def)

lemma FI17:
 "((F\<cdot>STrue) \<inter> SFinite) = (F \<inter> SFinite)\<cdot>SFinite"
 by (metis FI04 boolean_algebra_cancel.inf0 compl_bot_eq inf_commute strue_def)

lemma FI18:
 "((STrue\<cdot>F) \<inter> SFinite) = SFinite\<cdot>(F \<inter> SFinite)"
 by (metis BD19 FI01 FI04 FI17 FusionSEmptyR inf.idem sda_def)

lemma FI19:
 " SFinite\<cdot> SMore = SMore"
 by (metis BD06 FusionAssoc1 N14 N15 ssometime_def subset_antisym)

lemma FI20:
 " SInf \<union> SFinite = STrue"
 using STrueTop sfinite_def by auto

lemma FI21:
 "SFMore = SSkip\<cdot>SFinite"
 using FI09 FI10 sfmore_def by auto

lemma FI22:
 " (F \<inter> SFinite \<subseteq> G) = ((F \<inter> SFinite) \<subseteq> (G \<inter> SFinite))"
 by simp

lemma FI23:
 " (F\<cdot>G) \<inter> SInf = F \<cdot> (G \<inter> SInf)"
 by (metis FusionAssoc FusionSFalse)

lemma FI24:
 " ((F\<cdot>G) \<inter> SInf) = ((F \<inter> SInf) \<union> ((F \<inter> SFinite)\<cdot>(G \<inter> SInf)))"
 using FI23[of F G] Powerstarhelp2 Powerstarhelp3 by fastforce

lemma FI25:
 "SMore \<inter> SInf = SInf"
 using SStar15 SStarSkip sfinite_def smore_def by fastforce

lemma FI26:
 "(F \<inter> SInf)\<cdot>(F \<inter> SInf) = (F \<inter> SInf)"
 using Powerstarhelp2 by blast

lemma FI27:
 "(F \<inter> SInf)\<cdot>G = (F \<inter> SInf)"
 using Powerstarhelp2 by blast

lemma FI28:
 "((F \<inter> SMore) \<inter> SInf) = (F \<inter> SInf)"
 using FI25 by blast 

lemma FI29:
 "((F \<inter> SMore) \<inter> SFinite) = (F \<inter> SFMore)"
 by (metis inf_assoc inf_commute sfmore_def)

lemma FI30: 
 " (SEmpty \<inter> SFMore) = SFalse"
 by (simp add: sfalse_def sfmore_def smore_def) 

lemma FI31:
 " ((F \<inter> SInf) \<inter> SFMore) = SFalse"
 by (simp add: sfalse_def sfinite_def sfmore_def)

lemma FI32: 
 " (SEmpty \<inter> SInf) = SFalse"
 using FI25 sfalse_def smore_def by auto 

lemma FI33:
 " (F \<inter> SInf) = F\<cdot>SFalse"
 by (simp add: FusionSFalse)

lemma FI34:
 "(F \<inter> SFinite)\<cdot>G \<subseteq> SSometime G"
 by (simp add: FusionRuleL ssometime_def)

lemma FI35:
 assumes " F \<subseteq> SNext F"
 shows   " SFinite \<subseteq> -F"
using assms
by (metis B01 FusionSEmptyR N12 N22 SStarInductL SStarSkip double_compl snext_def)

lemma FI36:
 assumes "F \<inter> -G \<subseteq> SNext F"  
 shows   " F \<inter> SFinite \<subseteq> SSometime G"
using assms 
proof -
 have 1: "F \<inter> -G \<subseteq> SNext F " 
   using assms by auto
 have 2: "F \<inter> -G \<inter> SAlways (-G) \<subseteq> SNext F \<inter> SAlways (-G)"
   using assms by blast
 have 3: "SAlways (-G) \<subseteq> -G"
   by (simp add: BD12)
 have 4: "SAlways (-G) = SAlways (-G) \<inter> -G"
   using "3" by blast
 have 5: "F \<inter> SAlways (-G) \<subseteq> SNext F \<inter> SAlways (-G)"
   using "2" "4" by blast 
 have 51: " (SFinite \<cdot> G) =  G \<union> (SSkip \<cdot> (SFinite \<cdot> G))" 
   by (metis FusionAssoc1 SStarSkip ST47 UnfoldL)
 have 6: "SAlways (-G) = -G \<inter> SWnext (SAlways (-G))"
   using 51 by (simp add: salways_def ssometime_def swnext_def)   blast
 have 7: "SNext F \<inter> SAlways (-G) \<subseteq> SNext F \<inter> SWnext (SAlways (-G))"
   using "6" by blast  
 have 8: "F \<inter> SAlways (-G) \<subseteq> SNext F \<inter> SWnext (SAlways (-G))"
   using "5" "7" by blast
 have 9: "F \<inter> SAlways (-G) \<subseteq> SMore \<inter> SWnext F \<inter> SWnext (SAlways (-G))"
   using "8" N16 by blast  
 have 10: "F \<inter> SAlways (-G) \<subseteq> SWnext F \<inter> SWnext (SAlways (-G))"
   using "8" N16 by fastforce 
 have 11: "F \<inter> SAlways (-G) \<subseteq> SWnext (F \<inter> SAlways (-G))"
   using "10" N17 by blast 
 have 12: "SAlways (F \<inter> SAlways (-G)) \<subseteq> SWnext (F \<inter> SAlways (-G))"
   by (metis "10" B15 BD12 N17 inf.absorb_iff2) 
 have 13: "SAlways (F \<inter> SAlways (-G)) \<subseteq> 
           SWnext (F \<inter> SAlways (-G)) \<inter> F \<inter> (-(SAlways (-G)) \<union> SAlways( F \<inter> SAlways (-G)))"
   using "12" BD12 by auto
 have 14: "(F \<inter> SAlways (-G)) \<subseteq> SAlways( F \<inter> SAlways (-G))" 
   using 12 13
   by (metis "10" B08 BD08 N17 SBoxInduct boolean_algebra_cancel.inf0 compl_bot_eq inf_commute strue_def)
 have 15: "SAlways( F \<inter> SAlways (-G)) \<subseteq> ( F \<inter> SAlways (-G))"
   using BD12 by blast 
 have 16: "SAlways (F \<inter> SAlways (-G)) = ( F \<inter> SAlways (-G))"
   using "14" "15" by blast 
 have 17: "( F \<inter> SAlways (-G)) \<subseteq> SMore"
   using "9" by blast 
 have 18: "SAlways (F \<inter> SAlways (-G)) \<subseteq> SAlways SMore"
   by (simp add: "17" BD38) 
 have 19: "SFinite = -(SAlways SMore)"
   by (simp add: BD01 salways_def smore_def) 
 have 20: "SFinite \<subseteq> -(F \<inter>  SAlways (-G))"
   using "16" "18" "19" by blast 
 have 21: "SFinite \<subseteq> -F \<union> -(SAlways (-G))"
   using "20" by blast
 have 22: "-(SAlways (-G)) = SSometime G"
   by (simp add: BD09)
 show ?thesis
   using "20" "22" by blast 
qed

lemma FI37:
 assumes "F \<inter> -G \<subseteq> SNext (F \<inter> -G)"
 shows "F \<inter> SFinite \<subseteq> SSometime G"
using assms
by (metis B15 FI36 N05)  

lemma FI38:
 assumes " F \<inter> -G \<subseteq> (SNext F) \<inter> -(SNext G)"
 shows "F \<inter> SFinite \<subseteq> G"
proof -
 have 1: "F \<inter> -G \<subseteq> SNext((F \<inter> -G))"
   using N17[of F "-G"] 
   by (metis (no_types, lifting) N12 N16 assms double_compl inf.semigroup_axioms semigroup.assoc) 
 have 2: "SFinite \<subseteq> -((F \<inter> -G))"
   using "1" FI35 by auto
 show ?thesis
   using "2" by blast 
qed

lemma FI39:
 assumes "SWnext (SSometime F) \<subseteq> F "
 shows  " SFinite \<subseteq> F"
proof -
 have 1: " -F \<subseteq> SNext (-F)"
   by (metis BD02 N12 N18 Subsumption assms compl_le_swap2 double_complement sup.coboundedI1) 
 from 1 show ?thesis using FI35 by blast 
qed

lemma FI40:
 assumes " SEmpty \<subseteq> F"
         " SNext F \<subseteq> F"
 shows   " SFinite \<subseteq> F"
proof -
 have 1: "-F \<subseteq> SNext (-F) "
   using N12 N19 assms(1) assms(2) by blast 
 from 1 show ?thesis using FI35 by blast 
qed 
  
lemma FI41:
 assumes " SEmpty \<inter> F \<subseteq> G"
         " SNext (-F \<union> G) \<inter> F \<subseteq> G"
 shows   " F \<inter> SFinite \<subseteq> G"   
proof -
 have 1: " (F \<inter> -G) \<subseteq> SNext (F \<inter> -G)"
   using N19[of "(-F \<union> G)"] 
   using assms(1) assms(2) snext_def swnext_def by fastforce
 have 2: " SFinite \<subseteq> - (F \<inter> -G)"
   using "1" FI35 by auto 
 from 2 show ?thesis by blast 
qed
   
lemma FI42:
 assumes " F \<subseteq> SFMore\<cdot> F"
 shows   " SFinite \<subseteq> -F"
proof - 
 have 1: "SSometime F \<subseteq> SNext (SSometime F)"
   by (simp add: ssometime_def snext_def)
     (metis FI21 SChopAssoc SStarSkip ST47 UnfoldL assms order_refl subset_Un_eq)  
 have 2: " SFinite \<subseteq> -(SSometime F)"
   by (simp add: "1" FI35) 
 show ?thesis
   by (metis "2" B01 FI21 SStarSkip ST47 UnfoldL assms le_iff_sup ssometime_def subset_trans) 
qed       
    
lemma FI43:
 assumes " F \<inter> -G \<subseteq> SFMore\<cdot>(F \<inter> -G)"
 shows   "F \<inter> SFinite \<subseteq> G"    
proof -
 have 1: "SFinite \<subseteq> -( F \<inter> -G )"
   using FI42 assms by blast
 from 1 show ?thesis by blast
qed

lemma FI44:
 assumes " F \<inter> SFinite \<subseteq> SFMore \<cdot> F "
 shows   " SFinite \<subseteq> -F"
proof - 
 have 1: "F \<inter> SFinite \<subseteq> SFMore \<cdot> (F \<inter> SFinite)"
   by (metis FI04 FI21 FI22 SSkipSFinite assms inf.idem) 
 show ?thesis 
   by (metis "1" B01 B11 B12 FI43 inf.right_idem sfinite_def)
qed  

lemma FI45:
 assumes " F \<inter> SFinite \<subseteq> SMore \<cdot> F "
 shows   " SFinite \<subseteq> -F"
proof - 
 have 1: "F \<inter> SFinite \<subseteq> SFMore \<cdot> (F \<inter> SFinite)"
   by (metis FI04 FI22 assms inf_commute sfmore_def)
 show ?thesis 
   by (metis "1" B01 B11 B12 FI43 inf.right_idem sfinite_def)
qed 

lemma FI46:
 assumes " F  \<subseteq> SMore \<cdot> F "
 shows   " SFinite \<subseteq> -F"
proof -
 have 1: "F \<inter> SFinite \<subseteq> SFMore \<cdot> (F \<inter> SFinite)"
   using B11 FI45 assms by auto
 show ?thesis 
   by (metis "1" B01 B11 B12 FI43 inf.right_idem sfinite_def)
qed 

lemma FI47:
 assumes " (F \<inter> -G)  \<inter> SFinite \<subseteq> SFMore \<cdot> (F \<inter> -G) "
 shows   " F \<inter> SFinite \<subseteq> G"
proof -
 have 1: "SFinite \<subseteq> -( F \<inter> -G)"
   using FI44 assms by blast 
 from 1 show ?thesis by blast
qed 

lemma FI48:
assumes " (F \<inter> -G)  \<subseteq> SMore \<cdot> (F \<inter> -G) "
 shows   " F \<inter> SFinite \<subseteq> G" 
proof -
 have 1: "SFinite \<subseteq> -( F \<inter> -G)"
   using FI45 assms by blast
from 1 show ?thesis by blast
qed  

lemma FI49:
 assumes " F \<subseteq> G \<cdot> F"
         " G \<subseteq> SFMore"
 shows   " SFinite \<subseteq> -F"
proof -
 have 1: " G\<cdot>F \<subseteq> SFMore \<cdot> F"
   by (simp add: FusionRuleL assms(2)) 
 have 2: "F \<subseteq> SFMore \<cdot> F"
   using "1" assms(1) by auto 
 from 2 show ?thesis
   by (simp add: FI42)   
qed

lemma FI50:
 assumes " F \<subseteq> G \<cdot> F"
         " G \<subseteq> SMore"
 shows   " SFinite \<subseteq> -F"
proof -
 have 1: " G\<cdot>F \<subseteq> SMore \<cdot> F"
   by (simp add: FusionRuleL assms(2)) 
 have 2: "F \<subseteq> SMore \<cdot> F"
   using "1" assms(1) by auto 
 from 2 show ?thesis
   by (simp add: FI46)  
qed

lemma FI51:
 assumes " F \<inter> -G \<subseteq> (H\<cdot>F) \<inter> -(H\<cdot>G)"
         " H \<subseteq> SFMore"
 shows   " F \<inter> SFinite \<subseteq> G"
proof -
 have 1: "H\<cdot>(F \<inter> -G) \<subseteq> SFMore \<cdot> (F \<inter> -G) "
   by (simp add: FusionRuleL assms(2)) 
 have 2: " F \<inter> -G \<subseteq> SFMore \<cdot> (F \<inter> -G)"
   using "1" CH02 assms(1) by blast  
 from 2 show ?thesis
   by (simp add: FI43) 
qed

lemma FI52:
 assumes " F \<inter> -G \<subseteq> (H\<cdot>F) \<inter> -(H\<cdot>G)"
         " H \<subseteq> SMore"
 shows   " F \<inter> SFinite \<subseteq> G"
proof -
 have 1: "H\<cdot>(F \<inter> -G) \<subseteq> SMore \<cdot> (F \<inter> -G) "
   by (simp add: FusionRuleL assms(2)) 
 have 2: " F \<inter> -G \<subseteq> SMore \<cdot> (F \<inter> -G)"
   using "1" CH02 assms(1) by blast 
 from 2 show ?thesis
   by (simp add: FI48)
qed

lemma FI53:
 "SAlways SInf = SInf"
by (metis BD01 BD35 BD42 FI15 salways_def)

  
subsubsection \<open>Omega\<close>
  
lemma OA01:
 "(somega SSkip) = SSkip\<cdot>(somega SSkip)"
by (metis SOmegaUnroll SSkipSFinite SSkipSMore) 

lemma OA02:
 "(somega SEmpty) = SFalse"
by (metis FI30 SFalseFusion SOmegaUnroll inf_assoc inf_commute sfmore_def)

lemma OA03:
 "(somega SFalse) = SFalse" 
by (metis Compl_disjoint2 Powerstarhelp2 SOmegaUnroll  inf_assoc inf_compl_bot_right sfalse_def)

lemma OA04:
 "(somega SInf) = SFalse" 
by (metis FI25 SFalseBottom SFalseFusion SOmegaUnroll inf_commute sfinite_def)

lemma BD59:
 " SInf \<inter> G \<inter> SAlways ( -G \<union> (((F \<inter> SMore)\<inter>SFinite)\<cdot>G)) \<subseteq>
   ((F \<inter> SMore)\<inter>SFinite)\<cdot>( SInf \<inter> G \<inter> SAlways(-G \<union> (((F \<inter> SMore)\<inter>SFinite)\<cdot>G)))"
proof -
 have 1: "SInf \<inter> SAlways ( -G \<union> (((F \<inter> SMore)\<inter>SFinite)\<cdot>G)) \<subseteq> (-G \<union> (((F \<inter> SMore)\<inter>SFinite)\<cdot>G))  "
   using BD12 by blast 
 have 2: "SInf \<inter> G \<inter> SAlways ( -G \<union> (((F \<inter> SMore)\<inter>SFinite)\<cdot>G)) \<subseteq> 
          SInf \<inter> (((F \<inter> SMore)\<inter>SFinite)\<cdot>G)  \<inter> SAlways ( -G \<union> (((F \<inter> SMore)\<inter>SFinite)\<cdot>G))"
   using "1" by blast
 have 3: "SInf \<inter> (((F \<inter> SMore)\<inter>SFinite)\<cdot>G)  \<inter> SAlways ( -G \<union> (((F \<inter> SMore)\<inter>SFinite)\<cdot>G)) \<subseteq>
          ((F \<inter> SMore)\<inter>SFinite)\<cdot>( SInf \<inter> G \<inter> SAlways(-G \<union> (((F \<inter> SMore)\<inter>SFinite)\<cdot>G)))"
   by (metis BD58 FI23 inf_commute)
 show ?thesis using "2" "3" by blast
qed 

lemma OA06:
 "SInf \<inter> G \<inter> SAlways ( -G \<union> (((F \<inter> SMore)\<inter>SFinite)\<cdot>G)) \<subseteq> (somega F)"
by (metis (no_types, lifting) BD59 SOmegaWeakCoinduct) 

lemma FI54:
 "SAlways SFMore \<subseteq> SInf"
   by (metis BD38 BD56 Compl_disjoint2 SMoreImpSSkipFusion SSkipFusionImpSMore 
       inf_le2 sfalse_def sfmore_def sinf_def smore_def subset_antisym sup.orderE)

lemma FI55:
 "SAlways SFinite = SFinite"
using FI13 FI15 salways_def ssometime_def by fastforce

lemma FI56:
 " (SAlways SFMore) = SFalse "
using BD12 FI31 FI54 by blast
  
lemma OA07:
 " (somega ((SPower SSkip (Suc n)))) \<subseteq> SInf "
by (metis FI15 FI42 FusionRuleL SOmegaUnroll boolean_algebra_cancel.inf0 compl_le_swap1 inf_commute 
    inf_le2 inf_mono sfmore_def)

lemma OA08:
 "SInf \<subseteq> (somega ((SPower SSkip (Suc n))))"
proof -
 have 1: "SInf \<inter> STrue \<inter> SAlways ( -STrue \<union> ((((SPower SSkip (Suc n)) \<inter> SMore)\<inter>SFinite)\<cdot>STrue)) \<subseteq> 
         (somega (SPower SSkip (Suc n)))"
   by (simp add: OA06) 
 have 2: "- STrue \<union> (SPower SSkip (Suc n) \<inter> SMore)\<inter>SFinite \<cdot> STrue = 
            (SPower SSkip (Suc n) \<inter> SMore)\<inter>SFinite \<cdot> STrue"
   by (simp add: strue_def)
 have 21: "((((SPower SSkip (Suc n)) \<inter> SMore)\<inter>SFinite)\<cdot>STrue) \<subseteq> SMore" 
   by (metis N13 N14 N15 SStar25 SStarInductR inf_le2 le_sup_iff subset_antisym sup_inf_absorb)
 have 3: " SAlways ( ((((SPower SSkip (Suc n)) \<inter> SMore)\<inter>SFinite)\<cdot>STrue)) \<subseteq> SAlways (SMore)"
   using BD38 21 by auto
 have 4: "SAlways (SMore) \<subseteq> SInf" 
   using FI05 by blast
 have 5: "SAlways ( -STrue \<union> ((((SPower SSkip (Suc n)) \<inter> SMore)\<inter>SFinite)\<cdot>STrue)) \<subseteq> SInf" 
   by (metis "2" "3" FI05 subset_trans)
 have 6: " SInf \<subseteq> SAlways (SMore)" 
   by (simp add: BD01 FI15 salways_def smore_def)
 have 7: "(SPower SSkip (Suc n)) \<inter> SMore \<inter> SFinite\<subseteq> SMore \<inter> SFinite" 
   using FI07 FI21 by blast
 have 8: " SMore \<subseteq> (SMore \<inter> SFinite)\<cdot> STrue "
    by (metis FI19 FI21 ITA.SChopAssoc SMoreImpSSkipFusion SSkipFusionImpSMore SStar19 SStarSkip 
        inf_commute sfmore_def subset_antisym)
 have 9: "SInf \<inter> ((((SPower SSkip (Suc n)) \<inter> SMore)\<inter>SFinite)\<cdot>STrue) = 
          ((((SPower SSkip (Suc n)) \<inter> SMore)\<inter>SFinite)\<cdot>SInf)"
   by (metis FI23 boolean_algebra_cancel.inf0 compl_bot_eq inf_commute strue_def)
 have 10: "SInf \<inter> SAlways ( ((((SPower SSkip (Suc n)) \<inter> SMore)\<inter>SFinite)\<cdot>STrue)) =
           SAlways ( ((((SPower SSkip (Suc n)) \<inter> SMore)\<inter>SFinite)\<cdot>SInf))"
   by (metis "9" BD29 FI53)
 have 11: "((((SPower SSkip (Suc n)) \<inter> SMore)\<inter>SFinite)\<cdot>SInf) = SInf"
    proof (induct n)
    case 0
    then show ?case
    proof -
    have f1: "(SFinite) \<inter> SSkip = SSkip"
      using SSkipSFinite by blast
    have f2: "\<forall>S. S \<cdot> SSkip \<subseteq> (SMore \<inter> SFinite) \<or> \<not> S \<subseteq> SFinite"
      using FI10 FI21 FusionRuleL  by (metis inf_commute sfmore_def)
    have f3: "(SEmpty) \<cdot> SSkip = SSkip"
      using f1 by (metis FusionSEmptyR inf_commute pwr_0 spower_commutes)
    have f4: "(SEmpty) \<subseteq> SFinite"
      using pwr_0 spower_finite by blast
    have f5: "(SSkip) \<subseteq> SFinite"
      using f1 by blast
    have f6: "(SSkip) \<cdot> (STrue \<cdot> (SFalse \<cdot> SFalse)) = SInf"
      by (metis FI25 FusionAssoc1 FusionSFalse Powerstarhelp2 SMoreImpSSkipFusion SSkipFusionImpSMore
          subset_antisym)
    have f7: "(SSkip) \<inter> SMore \<inter> SFinite = SSkip"
      using f4 f3 f2 by blast
    have "(SSkip) \<cdot> SInf = SInf"
      using f6 f5 f4 f3 f2 by (simp add: SFalseFusion sinf_def)
    then show ?thesis
      by (simp add: f7 FusionSEmptyR SSkipSFinite)
    qed 
   next
    case (Suc n)
    then show ?case
    proof -
    have f1: "SSkip \<cdot> STrue = SMore"
      using SMoreImpSSkipFusion SSkipFusionImpSMore by blast
    have f2: "\<forall>S. STrue \<inter> S = S"
      by (simp add: strue_def)
    have f3: "\<forall>S. SMore \<inter> (SSkip \<cdot> S) = SSkip \<cdot> S"
      using f1 
      by (metis (no_types) B14 CH06 boolean_algebra_cancel.inf0 compl_bot_eq inf_aci(1) inf_le2 strue_def)
    then have f0: "\<forall>S. SMore \<inter> SFinite \<inter> (SSkip \<cdot> S) = SFinite \<inter> (SSkip \<cdot> S)"
      by blast
    then show ?thesis
      using f3 f2 f1
    proof -
     have f4: "SPower (SSkip) (Suc (Suc n)) \<inter> SMore  \<inter> SFinite= SFinite \<inter> (SSkip \<inter> SFinite \<cdot> SPower SSkip (Suc n))"
       by (metis SSkipSFinite f3 inf_sup_aci(1) pwr_Suc)
     then have f5: "SPower (SSkip) (Suc (Suc n)) \<inter> SMore \<inter> SFinite = SSkip \<inter> SFinite \<cdot> SPower SSkip (Suc n)"
      proof -
       have "(SSkip) \<inter> SFinite \<cdot> SPower SSkip (Suc n) \<subseteq> SFinite"
         using pwr_Suc spower_finite by blast
       then show ?thesis
         using f4 by blast
      qed
    show ?thesis 
    by (metis B14 CH01 FI12 FI23 FI25 FusionAssoc1 Powerstarhelp3 SSkipSFinite Suc f1 f3 
        inf_commute pwr_Suc sinf_def spower_finite sup_inf_absorb)
    qed
   qed
  qed
 have 12: "SInf \<subseteq> SAlways ( ((((SPower SSkip (Suc n)) \<inter> SMore) \<inter>SFinite)\<cdot>SInf))" 
   by (metis "11" B04 FI53)
 show ?thesis 
   using "11" SOmegaWeakCoinduct by (metis "12" FI53)
qed

lemma OA09:
 " (somega ((SPower SSkip (Suc n)))) = SInf "
 using OA07 OA08 by blast

lemma OA10:
 " (somega SSkip) = SInf"
 by (metis FusionSEmptyR OA09 SSkipSFinite pwr_0 spower.simps(2))

lemma OA11:
 " (somega STrue) \<subseteq> SInf"
by (metis FI15 FI42 SOmegaUnroll boolean_algebra.compl_zero boolean_algebra_cancel.inf0 
    compl_le_swap1 dual_order.refl inf_commute sfmore_def strue_def)



lemma OA12:
 " SInf \<subseteq> (somega STrue)"
proof -
 have 1: "(SAlways (SFalse \<union> ((STrue \<inter> SMore)\<inter>SFinite)\<cdot>STrue)) \<subseteq> SInf "
    by (metis B04 BD01 BD09 BD35 FI10 FI15 FI19 FI21 ITA.SChopAssoc SMoreImpSSkipFusion 
        SSkipFusionImpSMore boolean_algebra.compl_zero boolean_algebra.de_Morgan_disj 
        boolean_algebra_cancel.inf0 inf_commute salways_def sfalse_def sfmore_def smore_def strue_def)
 have 2: " SInf \<subseteq> (SAlways (SFalse \<union> ((STrue \<inter> SMore)\<inter>SFinite)\<cdot>STrue))"
   by (metis "1" B15 BD01 BD09 BD12 BD35 FI10 FI15 FI19 FI21 FI56 ITA.SChopAssoc SMoreImpSSkipFusion 
       SSkipFusionImpSMore boolean_algebra.compl_zero boolean_algebra_cancel.inf0 inf.absorb_iff2 
       inf_commute salways_def sfmore_def smore_def strue_def sup.absorb2)
 have 3: "SInf \<inter> STrue \<inter> (SAlways (SFalse \<union> ((STrue \<inter> SMore)\<inter>SFinite)\<cdot>STrue)) \<subseteq> (somega STrue)"
   by (metis OA06 compl_bot_eq compl_top_eq sfalse_def strue_def)
 show ?thesis
   using "2" "3" strue_def by fastforce 
qed

lemma OA13:
 " (somega STrue) = SInf"
 by (simp add: B04 OA11 OA12)

lemma OA14:
  " (somega SMore) \<subseteq> SInf"
by (metis Int_absorb1 Int_lower2 OA13 SOmegaUnroll SOmegaWeakCoinduct compl_bot_eq inf_top_left  strue_def)

lemma OA15:
 " SInf \<subseteq> (somega SMore)"
proof -
 have 1: "(SAlways (SFalse \<union> ((SMore \<inter> SMore)\<inter>SFinite)\<cdot>STrue)) \<subseteq> SInf "
   by (metis FI05 FI10 FI19 FI21 FusionSEmptyR ITA.SChopAssoc SMoreImpSSkipFusion SSkipFusionImpSMore
       ST33 boolean_algebra.compl_one boolean_algebra.compl_zero boolean_algebra.de_Morgan_conj 
       inf.idem inf_commute sfalse_def sfmore_def smore_def strue_def subset_antisym)
 have 2:  " SInf \<subseteq> (SAlways (SFalse \<union> ((SMore \<inter> SMore)\<inter>SFinite)\<cdot>STrue))"
   by (metis BD38 FI10 FI19 FI21 FI25 FI53 FusionSEmptyR ITA.SChopAssoc SMoreImpSSkipFusion 
       SSkipFusionImpSMore ST33 boolean_algebra.compl_one boolean_algebra.compl_zero 
       boolean_algebra.de_Morgan_conj inf.idem inf_commute inf_le1 sfalse_def sfmore_def 
       smore_def strue_def subset_antisym)
 have 3: "SInf \<inter> STrue \<inter> (SAlways (SFalse \<union> ((SMore \<inter> SMore)\<inter>SFinite)\<cdot>STrue)) \<subseteq> (somega STrue)"
   using OA13 by blast
 show ?thesis
   by (metis "1" "2" Int_absorb1 OA06 boolean_algebra_cancel.inf0 compl_bot_eq compl_top_eq sfalse_def
      strue_def subset_antisym)
qed

lemma OA16:
  " (somega SMore) = SInf"
  using OA14 OA15 by blast 

lemma OA17:
 " (somega SFinite) \<subseteq> SInf"
by (metis FI07 FI15 FI21 FI42 SOmegaUnroll compl_le_swap1 dual_order.refl inf.orderE sfmore_def)

lemma OA18:
 "SInf \<subseteq> (somega SFinite) "
proof -
 have 1: "(SAlways (SFalse \<union> ((SFinite \<inter> SMore)\<inter>SFinite)\<cdot>STrue)) \<subseteq> SInf "
 by (metis B14 FI05 FI07 FI10 FI19 FI21 FusionSEmptyR ITA.SChopAssoc SMoreImpSSkipFusion
     SSkipFusionImpSMore ST33 boolean_algebra.compl_one boolean_algebra.compl_zero 
     boolean_algebra.de_Morgan_conj sfalse_def sfmore_def smore_def strue_def subset_antisym)
 have 2: " SInf \<subseteq> (SAlways (SFalse \<union> ((SFinite \<inter> SMore)\<inter>SFinite)\<cdot>STrue))"
 by (metis B14 BD38 FI10 FI19 FI21 FI25 FI53 FusionSEmptyR ITA.SChopAssoc SMoreImpSSkipFusion 
     SSkipFusionImpSMore ST33 boolean_algebra.compl_one boolean_algebra.compl_zero 
     boolean_algebra.de_Morgan_conj inf_le1 sfalse_def sfmore_def smore_def strue_def subset_antisym)
 have 3: "SInf \<inter> STrue \<inter> (SAlways (SFalse \<union> ((SFinite \<inter> SMore)\<inter>SFinite)\<cdot>STrue)) \<subseteq> (somega STrue)"
   using OA13 by blast
 show ?thesis
  by (metis "1" "2" Int_absorb1 OA06 boolean_algebra_cancel.inf0 compl_bot_eq compl_top_eq sfalse_def 
      strue_def subset_antisym)
qed

lemma OA19:
 " (somega SFinite) = SInf"
 by (simp add: B04 OA17 OA18) 

lemma OA20:
 assumes "H \<subseteq> ((F \<inter> SMore)\<inter>SFinite)\<cdot>H "
 shows   " H \<inter> SInf \<subseteq> (somega F)"
using assms
using SOmegaWeakCoinduct by blast

lemma OA21:
 assumes "F \<subseteq> G"
 shows   " (somega F) \<inter> SInf \<subseteq> (somega G)"
using assms
by (metis FusionRuleL OA20 SOmegaUnroll inf_mono subset_refl)   

lemma OA22:
 assumes "F = G"
 shows   " (somega F)  = (somega G)"
using assms by auto   

lemma OA23:
 " (somega (F \<inter> G)) \<inter> SInf \<subseteq> (somega F)"
 by (simp add: OA21)

lemma OA24:
  " (somega (F \<inter> G)) \<inter> SInf \<subseteq> (somega G)"
 by (simp add: OA21)

lemma BD60: 
 "(SBi F) \<inter> (G0\<cdot>G1) \<subseteq> (F \<inter> G0)\<cdot>G1"
proof -
 have 1: " F \<subseteq> (-G0 \<union> (F \<inter> G0))"
   by blast
 have 2: "SBi F \<subseteq> SBi(-G0 \<union> (F \<inter> G0))"
   by (simp add: "1" BD39) 
 have 3: "SBi(-G0 \<union> (F \<inter> G0)) \<inter> (G0\<cdot>G1) \<subseteq> (F \<inter> G0)\<cdot>G1"
   by (simp add: BD45) 
 show ?thesis
   using "2" "3" by blast 
qed 
   
lemma BD61: 
 "(SAlways H) \<inter> (F\<cdot>G) \<subseteq> F\<cdot>(H \<inter> G)"
proof -
 have 1: " H \<subseteq> (-G \<union> (H \<inter> G))"
   by blast
 have 2: "SAlways H \<subseteq> SAlways (-G \<union> (H \<inter> G))"
   by (simp add: "1" BD38) 
 have 3: "SAlways (-G \<union> (H \<inter> G)) \<inter> (F\<cdot>G) \<subseteq> F\<cdot>(H \<inter> G)"
   using BD46 by blast
 show ?thesis
   using "2" "3" by blast 
qed
  
lemma BD62:
 " (SBa F) \<inter> (G0\<cdot>G1) \<subseteq> (F \<inter> G0)\<cdot>(F \<inter> G1)"
proof -
 have 1: "(SBa F) \<subseteq> (SBi F)"
   by (simp add: BD27)
 have 2: "(SBi F) \<inter> (G0\<cdot>G1) \<subseteq> (F \<inter> G0)\<cdot>G1"
   by (simp add: BD60) 
 have 3: "(SBa F) \<subseteq> (SAlways F)"
   by (simp add: BD28)
 have 4: "(SAlways F) \<inter> ((F \<inter> G0)\<cdot>G1) \<subseteq> (F \<inter> G0)\<cdot>(F \<inter> G1)"
   using BD61 by blast
 show ?thesis
   using "1" "2" "3" "4" by blast 
qed

lemma BD63:
 "(SBa F) \<inter> (G \<cdot> G1) \<subseteq> (F \<inter> G) \<cdot>((SBa F)  \<inter> G1)"
proof -
 have 1: "(SBa F) = (SBa (SBa F))"
   using BD44 by blast
 have 2: "(SBa (SBa F))  \<inter> (G \<cdot> G1) \<subseteq> G\<cdot>((SBa F) \<inter> G1)"
   using BD28 BD61 by blast
 have 3: "(SBa F) \<inter> (G\<cdot>((SBa F) \<inter> G1)) \<subseteq> (F \<inter> G) \<cdot>((SBa F)  \<inter> G1)"
   using BD62 FusionRuleR by blast
 show ?thesis
   using "1" "2" "3" by blast 
qed
 
lemma OA25:
 " SBa (-F \<union> G) \<inter> SInf \<inter> (somega F) \<subseteq> (somega G)"
proof -
 have 1: "SBa (-F \<union> G) \<inter> (((F \<inter> SMore)\<inter>SFinite)\<cdot>(somega F)) \<subseteq> 
         ((-F \<union> G) \<inter> ((F \<inter> SMore)\<inter>SFinite))\<cdot>((-F \<union> G) \<inter> (somega F))"
   by (simp add: BD62)
 have 2: "(-F \<union> G) \<inter> ((F \<inter> SMore)\<inter>SFinite) \<subseteq> ((G \<inter> SMore)\<inter>SFinite)"
   by auto
 have 3: "(-F \<union> G) \<inter> (somega F) \<subseteq> (somega F)"
   by auto
 have 4: "SBa (-F \<union> G) \<inter> (((F \<inter> SMore)\<inter>SFinite)\<cdot>(somega F)) \<subseteq> ((G \<inter> SMore)\<inter>SFinite)\<cdot> (somega F)"
   using "1" "2" "3" CH06 by blast
 have 5: "SBa (-F \<union> G) \<inter> (((F \<inter> SMore)\<inter>SFinite)\<cdot>(somega F)) \<subseteq> 
          ((-F \<union> G) \<inter> ((F \<inter> SMore)\<inter>SFinite))\<cdot>(SBa (-F \<union> G) \<inter> (somega F))"
   using BD63 by blast
 have 6: "((-F \<union> G) \<inter> ((F \<inter> SMore)\<inter>SFinite))\<cdot>(SBa (-F \<union> G) \<inter> (somega F)) \<subseteq>
          ((G \<inter> SMore)\<inter>SFinite)\<cdot>(SBa (-F \<union> G) \<inter> (somega F))"
   using "2" FusionRuleL by blast    
 have 7: "(SBa (-F \<union> G) \<inter> (somega F)) \<subseteq> ((G \<inter> SMore)\<inter>SFinite)\<cdot>(SBa (-F \<union> G) \<inter> (somega F)) "
   using "5" "6" SOmegaUnroll by blast  
 have 8: "(SBa (-F \<union> G) \<inter> (somega F)) \<inter> SInf \<subseteq> (somega G)"
   by (simp add: "7" OA20)  
 show ?thesis
   using "8" by auto  
qed
  
lemma OA26:
 " SBa ((-F \<union> G) \<inter> (-G \<union> F))   \<inter> SInf  \<subseteq> (-(somega G) \<union> (somega F)) \<inter> (-(somega F) \<union> (somega G))"
proof -
 have 1: "SBa ((-F \<union> G) \<inter> (-G \<union> F)) = SBa (-F \<union> G) \<inter> SBa (-G \<union> F)"
   by (simp add: BD22 sba_def)   
 have 2: " SBa (-F \<union> G) \<inter> SInf \<subseteq> (-(somega F) \<union> (somega G))"
   by (simp add: B19 OA25)  
 have 3:  " SBa (-G \<union> F) \<inter> SInf \<subseteq> (-(somega G) \<union> (somega F))"
   by (simp add: B19 OA25) 
 show ?thesis
   using "1" "2" "3" by blast 
qed

lemma OA27:
 "SBa F \<inter> (somega G) \<inter> SInf \<subseteq> somega (F \<inter> G) "
 by (metis (no_types, lifting) BD63 OA20 SOmegaUnroll inf_assoc)

lemma FI57:
 "SInf \<inter> (((F \<inter> SMore)\<inter>SFinite)\<cdot>G) = ((F \<inter> SMore)\<inter>SFinite)\<cdot>(G \<inter> SInf)"
 using FI23 by blast

lemma FI58:
 "SInf \<inter> (SAlways F) = SAlways (F \<inter> SInf)"
 using BD29 FI53 by blast

lemma FI59:
 "SInf \<inter> (SAlways (-F \<union> G)) = SAlways ( (-F \<inter> SInf) \<union> (G \<inter> SInf)  )"
 by (simp add: FI58 inf_sup_distrib2)


subsection \<open>Link between Set of Intervals and ITL\<close>

lemma interval_lan [simp]:
 " \<sigma> \<in> (lan f) \<longleftrightarrow> (\<sigma> \<Turnstile> f)"
by (simp add: lan_def)

lemma valid_lan_eqv :
 "( (lan f) = (lan g) ) \<longleftrightarrow> ( \<turnstile> f = g ) "
using interval_lan lan_def Valid_def by fastforce

lemma valid_lan_imp :
 " ( (lan f) \<subseteq> (lan g) ) \<longleftrightarrow> (\<turnstile> f \<longrightarrow> g) "
using interval_lan lan_def Valid_def 
by (simp add: Valid_def lan_def Collect_mono_iff)

lemma valid_strue :
 " ( (lan f) = STrue) \<longleftrightarrow> (\<turnstile> f) "
using strue_def by fastforce

lemma strue_true:
 " \<sigma> \<in> STrue \<longleftrightarrow> (\<sigma> \<Turnstile> #True)"
by (simp add: strue_elim)

lemma strue_true_1:
 " STrue = (lan (LIFT #True))"
using lan_def strue_true by fastforce

lemma sfalse_false:
 " \<sigma> \<in> SFalse \<longleftrightarrow> (\<sigma> \<Turnstile> #False)"
by (simp add: sfalse_def)

lemma sfalse_false_1:
 " SFalse = (lan (LIFT(#False)))"
using sfalse_false using lan_def by fastforce

lemma not_negation:
 " \<sigma> \<in> (-(lan f)) \<longleftrightarrow> (\<sigma> \<Turnstile> \<not> f)"
by simp
 
lemma not_negation_1:
 "-(lan f ) = (lan (LIFT(\<not> f))) "
using interval_lan lan_def by fastforce

lemma inter_and:
 "(\<sigma> \<in> ( (lan f) \<inter> (lan g) )) \<longleftrightarrow> (\<sigma> \<Turnstile> f \<and> g) "
by (simp add: lan_def)

lemma inter_and_1:
 " ( (lan f) \<inter> (lan g) ) = (lan (LIFT(f \<and> g))) "
using inter_and lan_def by fastforce

lemma union_or:
 "(\<sigma> \<in> ( (lan f) \<union> (lan g) )) \<longleftrightarrow> (\<sigma> \<Turnstile> f \<or> g) "
by (simp add: lan_def)

lemma union_or_1:
 " ( (lan f) \<union> (lan g) ) = (lan (LIFT(f \<or> g))) "
using union_or lan_def by fastforce

lemma subset_impl:
 " (\<sigma> \<in> ( -(lan f) \<union> (lan g))) \<longleftrightarrow> (\<sigma> \<Turnstile> f \<longrightarrow> g)"
by simp

lemma subset_impl_1:
 " ( -(lan f) \<union> (lan g)) = (lan (LIFT(f \<longrightarrow> g)))"
using subset_impl lan_def by fastforce

lemma fusion_chop:
 "(\<sigma> \<in> ((lan f)\<cdot>(lan g))) \<longleftrightarrow> (\<sigma> \<Turnstile> f;g)   "
by (auto simp add: fusion_iff chop_nfuse) 

lemma fusion_chop_1:
 " ((lan f)\<cdot>(lan g)) = (lan (LIFT(f;g)))   "
using fusion_chop lan_def by blast

lemma sempty_empty:
 "  \<sigma> \<in> SEmpty \<longleftrightarrow> (\<sigma> \<Turnstile> empty)  "
by (simp add: itl_defs sempty_elim zero_enat_def)

lemma sempty_empty_1:
 " SEmpty = (lan (LIFT empty))"
using sempty_empty lan_def by fastforce

lemma smore_more:
 "\<sigma> \<in> SMore \<longleftrightarrow> (\<sigma> \<Turnstile> more)  "
using zero_enat_def by (auto simp add: itl_defs sempty_elim smore_def )

lemma smore_more_1:
 " SMore = (lan (LIFT more))  "
using smore_more lan_def by fastforce

lemma sskip_skip:
 " \<sigma> \<in> SSkip = (\<sigma> \<Turnstile> skip)"
by (simp add: one_enat_def itl_defs sskip_elim)

lemma sskip_skip_1:
 " SSkip = (lan (LIFT skip))"
using sskip_skip lan_def by fastforce

lemma snext_next:
 " \<sigma> \<in> (SNext (lan f)) \<longleftrightarrow> (\<sigma> \<Turnstile> \<circle> f)"
by (metis snext_def fusion_chop next_d_def sskip_skip_1)

lemma snext_next_1:
 " (SNext (lan f)) = (lan (LIFT(\<circle> f)))"
using snext_next lan_def by fastforce

lemma swnext_wnext:
 " \<sigma> \<in> (SWnext (lan f)) \<longleftrightarrow> (\<sigma> \<Turnstile> wnext f)"
by (simp add: swnext_def fusion_chop_1 next_d_def not_negation_1 sskip_skip_1 wnext_d_def)

lemma swnext_wnext_1:
 " (SWnext (lan f)) = (lan (LIFT(wnext f)))"
using swnext_wnext lan_def by fastforce

lemma sprev_prev:
 " \<sigma> \<in> (SPrev (lan f)) \<longleftrightarrow> (\<sigma> \<Turnstile> prev f)"
by (metis fusion_chop prev_d_def sprev_def sskip_skip_1)

lemma sprev_prev_1:
 " (SPrev (lan f)) = (lan (LIFT(prev f)))"
using sprev_prev lan_def by fastforce

lemma swprev_wprev:
 " \<sigma> \<in> (SWprev (lan f)) \<longleftrightarrow> (\<sigma> \<Turnstile> wprev f)"
by (simp add: fusion_chop_1 not_negation_1 prev_d_def sskip_skip_1 swprev_def wprev_d_def)

lemma swprev_wprev_1:
 " (SWprev (lan f)) = (lan (LIFT(wprev f)))"
using swprev_wprev lan_def by fastforce

lemma sinit_init:
 " \<sigma> \<in> SInit (lan f) \<longleftrightarrow> (\<sigma> \<Turnstile> init f)"
by (simp add: Int_commute fusion_chop_1 init_d_def inter_and_1 sempty_empty_1 sinit_def strue_true_1)

lemma sinit_init_1:
 " SInit (lan f) = (lan (LIFT(init f)))"
using sinit_init lan_def by fastforce

lemma sfinite:
 " \<sigma> \<in> SFinite \<longleftrightarrow> (\<sigma> \<Turnstile> finite) "
by (simp add: sfinite_def sinf_def finite_d_def infinite_d_def fusion_chop sfalse_false_1 strue_true_1)

lemma sfinite_1:
 " SFinite = lan(LIFT(finite)) "
using sfinite lan_def by fastforce

lemma and_inter_finite:
 " \<sigma> \<in> (((lan f) \<inter> SFinite)) \<longleftrightarrow> (\<sigma> \<Turnstile> (f \<and> finite) ) "
using sfinite inter_and by auto

lemma and_inter_finite_1:
 " (((lan f) \<inter> SFinite)) = lan(LIFT (f \<and> finite) ) "
by (simp add: inter_and_1 sfinite_1)

lemma and_inter_more:
 " \<sigma> \<in> (((lan f) \<inter> SMore)) \<longleftrightarrow> (\<sigma> \<Turnstile> (f \<and> more) ) "
using smore_more inter_and by auto

lemma and_inter_more_1:
 " \<sigma> \<in> (((lan f) \<inter> SMore)) \<longleftrightarrow> (\<sigma> \<in> (lan (LIFT(f \<and> more)) )) "
using and_inter_more lan_def by (simp add: smore_more_1)

lemma and_inter_more_2:
 "  ((lan f) \<inter> SMore) =  (lan (LIFT(f \<and> more)) ) "
using and_inter_more_1 by blast

lemma and_chop:
 "\<sigma> \<in> (((lan f) \<inter> SMore)\<cdot> (lan g)) \<longleftrightarrow> (\<sigma> \<Turnstile> (f \<and> more);g)  "
by (metis fusion_chop inter_and_1 smore_more_1)

lemma and_chop_1:
 " (((lan f) \<inter> SMore)\<cdot> (lan g)) = (lan (LIFT((f \<and> more);g)))  "
using and_chop lan_def by blast

lemma spower_chop_power:
 "  (SPower (lan f) n) = (lan (LIFT(fpower f n)))"
proof (induct n)
case 0
then show ?case 
by (simp add: sempty_empty_1 fpower_d_def)
next
case (Suc n)
then show ?case 
by (simp add: and_inter_finite_1 fusion_chop_1 fpower_d_def)
qed

lemma spowerstar:
 " \<sigma> \<in> SPowerstar (lan f) \<longleftrightarrow> \<sigma> \<in> SFPowerstar (lan (f)) \<cdot> (SEmpty \<union> ((lan f) \<inter> SInf))   "
 by (simp add: spowerstar_def)

lemma sstar_spowerstar:
 " \<sigma> \<in> SStar (lan f) \<longleftrightarrow> \<sigma> \<in> SPowerstar ((lan f) \<inter> SMore)"
by (simp add: sstar_def)

lemma union_exists:
 " \<sigma> \<in> (\<Union> n. SPower (lan f) n) \<longleftrightarrow> \<sigma> \<in> lan(LIFT(\<exists>n. fpower f n))"
  by (simp add: spower_chop_power)
 
lemma union_exists_1:
 " (\<Union> n. SPower (lan f) n) = lan(LIFT(\<exists>n. fpower f n))"
using union_exists lan_def by blast

lemma sstar_chopstar:
 " \<sigma> \<in> (SStar (lan f)) \<longleftrightarrow> \<sigma> \<in> (lan (LIFT(f\<^sup>\<star>))) "
proof -
 have 1: "\<sigma> \<in> (SStar (lan f)) \<longleftrightarrow> \<sigma> \<in> SPowerstar ((lan f) \<inter> SMore)  " 
   using  sstar_spowerstar by blast
 have 2: " \<sigma> \<in> SPowerstar ((lan f) \<inter> SMore) \<longleftrightarrow> 
           \<sigma> \<in> SFPowerstar (lan (f) \<inter> SMore) \<cdot> (SEmpty \<union> (((lan f) \<inter> SMore) \<inter> SInf)) "
   by (simp add: spowerstar_def)
 have 3: "SFPowerstar (lan (f) \<inter> SMore) = SFPowerstar(lan(LIFT(f \<and> more)))"
   by (simp add: and_inter_more_2) 
 have 31: "\<And>n. SPower (lan(LIFT(f \<and> more))) n = lan(LIFT(fpower (f \<and> more) n))"
   using spower_chop_power by blast
 have 32: "SFPowerstar(lan(LIFT(f \<and> more))) = lan(LIFT(fpowerstar (f \<and> more)))"
   using union_exists_1 by (auto simp add: sfpowerstar_def fpowerstar_d_def)
 have 4: "(SEmpty \<union> (((lan f) \<inter> SMore) \<inter> SInf)) = 
          lan(LIFT(empty \<or> ((f \<and> more) \<and> inf)))"
   by (metis Morgan and_inter_more_2 finite_d_def inter_and_1 not_negation_1 sempty_empty_1 
       sfinite_1 sfinite_def union_or_1) 
 have 5: "SFPowerstar (lan (f) \<inter> SMore) \<cdot> (SEmpty \<union> (((lan f) \<inter> SMore) \<inter> SInf)) =
          lan(LIFT(powerstar ( f \<and> more))) "
   by (simp add: powerstar_d_def  "3" "32" "4" fpowerstar_d_def fusion_chop_1)
 have 6: "lan(LIFT(powerstar ( f \<and> more))) =
          lan(LIFT(chopstar f))"
   by (simp add: chopstar_d_def) 
 show ?thesis 
   by (simp add: "1" "2" "5" "6")
qed

lemma chopstar_sstar_1:
 " (SStar (lan f)) = (lan (LIFT(f\<^sup>\<star>)))"
using sstar_chopstar lan_def by blast

lemma chopstar_seqv:
 " \<sigma> \<in> (lan (LIFT(f\<^sup>\<star>))) \<longleftrightarrow> \<sigma> \<in> (lan (LIFT(empty \<or>  (f \<and> more); f\<^sup>\<star>)))"
by (metis (no_types, lifting) SChopstarEqv chopstar_sstar_1 fusion_chop_1 inter_and_1 
    sempty_empty_1 smore_more_1 union_or_1)

lemma chopstar_seqv_1:
 " (lan (LIFT(f\<^sup>\<star>))) = (lan (LIFT(empty \<or>  (f \<and> more); f\<^sup>\<star>)))"
using  chopstar_seqv lan_def by blast

lemma sinf:
 " \<sigma> \<in> SInf \<longleftrightarrow> (\<sigma> \<Turnstile> inf) "
by (simp add: fusion_chop infinite_d_def sfalse_false_1 sinf_def strue_true_1)

lemma sinf_1:
 "SInf = lan(LIFT(inf))"
using sinf by fastforce

lemma fmore:
 " \<sigma> \<in> SFMore \<longleftrightarrow> (\<sigma> \<Turnstile> fmore) "
 by (metis fmore_d_def inf_commute inter_and_1 interval_lan sfinite_1 sfmore_def smore_more_1)

lemma fmore_1:
 "SFMore = lan(LIFT(fmore))" 
using fmore by fastforce

lemma omega_induct_sem:
 "x \<in> - (lan g) \<union> ( (( ((lan f) \<inter> SMore)\<inter>SFinite)\<cdot>(lan g)))  \<longleftrightarrow>
  (x \<Turnstile> g \<longrightarrow> ((f \<and> more)\<and>finite);g) "
  by (simp add: fusion_chop_1 inter_and_1 sfinite_1 smore_more_1)

lemma  omega_induct:
 " - (lan g) \<union> ( (( ((lan f) \<inter> SMore)\<inter>SFinite)\<cdot>(lan g))) =
    lan(LIFT(g \<longrightarrow>  ((f \<and> more)\<and> finite);g))"
using omega_induct_sem[of _ g f]  by fastforce

lemma somega_omega_sem_1:
 assumes "x \<Turnstile> f\<^sup>\<omega>"
  shows  "x \<in> somega (lan f)"
proof -
 have 1: "x\<Turnstile> f\<^sup>\<omega> \<longrightarrow>  ((f \<and> more)\<and>finite);f\<^sup>\<omega> "
 by (metis OmegaUnroll Valid_def int_iffD1 schop_d_def)
 have 2: "x\<in> -(lan (LIFT(f\<^sup>\<omega>))) \<union> ((( (lan f) \<inter> SMore)\<inter>SFinite)\<cdot>(lan(LIFT(f\<^sup>\<omega>))))   "
 using "1" omega_induct_sem by blast
 have 3: "\<And>x . x \<in> (lan (LIFT(f\<^sup>\<omega>))) \<Longrightarrow> x \<in> ( ((((lan f) \<inter> SMore)\<inter>SFinite)\<cdot>(lan (LIFT(f\<^sup>\<omega>))) )) "
 by (metis (mono_tags, opaque_lifting) OmegaUnroll and_inter_finite_1 and_inter_more_2 fusion_chop_1 int_eq itl_def(9))
 show ?thesis using "3" SOmegaWeakCoinductsem assms interval_lan by blast
qed

lemma somega_omega_sem_2:
 assumes "x \<in> somega (lan f)" 
  shows  "x \<Turnstile> f\<^sup>\<omega>"
proof -
 have 1: "x \<in> - (somega (lan f)) \<union> ( (( ((lan f) \<inter> SMore)\<inter>SFinite)\<cdot>(somega (lan f)))) "
   using SOmegaCases by blast
 have 2: "\<And>x. (\<lambda> x. x\<in> somega (lan f)) x \<Longrightarrow> 
               x \<in> ( (( ((lan f) \<inter> SMore)\<inter>SFinite)\<cdot>(somega (lan f)))) "
   using SOmegaCases by blast 
 have 3: "\<And>x. (\<lambda> x. x\<in> somega (lan f)) x \<Longrightarrow> 
               ( (\<exists>n. f ( (ntaken n  x)) \<and> n>0 \<and>
                (\<lambda> x. x\<in> somega (lan f)) ( (ndropn n ( x)))) ) 
                 "
   using interval_lan[of _ f] somega.cases[of _ "(lan f)" ] 
   by (metis enat_ord_simps(2) ndropn_nfuse nfinite_nlength_enat ntaken_nfuse the_enat.simps 
         zero_enat_def) 
 have 4: "\<And>x. (\<lambda> x. x\<in> somega (lan f)) x \<Longrightarrow> 
               x \<Turnstile>  ((f \<and> more)\<and>finite);(\<lambda> x. x\<in> somega (lan f)) 
                 "
   by (metis "3" FMoreSem_var i0_less min_enat_simps(2) ntaken_nfuse ntaken_nlength somega.simps)            
 have 5: "(\<lambda> x. x\<in> somega (lan f)) x" 
   by (simp add: assms)
 show ?thesis using 4 5 OmegaWeakCoinductSem[of "(\<lambda> x. x\<in> somega (lan f))" f]
 by (simp add: schop_d_def) 
qed

lemma somega_omega:
 "x \<in> somega (lan f) \<longleftrightarrow> (x \<Turnstile> f\<^sup>\<omega>)"
using somega_omega_sem_1 somega_omega_sem_2 by blast 

lemma somega_omega_1:
 "somega (lan f) = lan(LIFT( f\<^sup>\<omega>))"
using  somega_omega by fastforce


end