(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)

section \<open>First occurrence operator\<close>

theory First
 imports
   Omega 
begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="z3 cvc4 e spass vampire " ]

*)
(*declare [[smt_timeout=120]]*)

(*declare [[show_types]]*)


text \<open>
Runtime verification (RV) has gained significant interest in recent years. 
The behaviour of a program can be verified in real time by analysing its evolving trace. 
This approach has two significant benefits over static verification techniques such as model 
checking. Firstly, it is only necessary to verify actual execution paths rather than all 
possible paths. Secondly, it is possible to react at runtime should the program diverge from 
its specified behaviour. RV does not replace traditional verification techniques but it 
does provide an extra layer of security.

Linear Temporal Logic (LTL) is a popular formalism for writing specifications from which RV 
monitors can be derived automatically. By contrast, Interval Temporal Logic (ITL) has not 
been as widely represented in this field despite being more expressive and 
compositional. The principal issue is efficiency. ITL uses non-deterministic operators to 
IConstruct sequential and iterative specifications (chop and chop-star, respectively) and 
these introduce combinatorial complexity. Approaches to mitigate this include using a 
deterministic subset of ITL or adapting the semantics to include a deterministic chop operator.
 This work proposes an alternative approach, wholly within existing ITL, and based upon a new,
 derived operator called “first occurrence”.

A theory of first occurrence is developed and used to derive an algebra of RV monitors.

\<close>

subsection \<open>Definitions\<close>

subsubsection \<open>Definitions Strict Initial and Final\<close>


definition fprev_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "fprev_d F \<equiv> LIFT(F\<frown>skip)"

syntax
  "_fprev_d"      :: "lift \<Rightarrow> lift" ("(fprev _)" [88] 87)

syntax (ASCII)
  "_fprev_d"      :: "lift \<Rightarrow> lift" ("(fprev _)" [88] 87)

translations 
  "_fprev_d"      \<rightleftharpoons> "CONST fprev_d"

definition wfprev_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "wfprev_d F \<equiv> LIFT(\<not>(fprev(\<not>F)))"

syntax
  "_wfprev_d"      :: "lift \<Rightarrow> lift" ("(wfprev _)" [88] 87)

syntax (ASCII)
  "_wfprev_d"      :: "lift \<Rightarrow> lift" ("(wfprev _)" [88] 87)

translations 
  "_wfprev_d"      \<rightleftharpoons> "CONST wfprev_d"



definition bs_d :: "('a::world) formula \<Rightarrow> 'a formula"
where 
  "bs_d f \<equiv> LIFT(empty \<or> ((bi f) \<frown> skip))" 

syntax 
 "_bs_d" :: "lift \<Rightarrow> lift" ("(bs _)" [88] 87)

syntax (ASCII)
 "_bs_d" :: "lift \<Rightarrow> lift" ("(bs _)" [88] 87)

translations
 "_bs_d" \<rightleftharpoons> "CONST bs_d"


definition ds_d :: "('a::world) formula \<Rightarrow> 'a formula"
where 
  "ds_d f \<equiv> LIFT (\<not> (bs (\<not> f)))"  


syntax
 "_ds_d" :: "lift \<Rightarrow> lift" ("(ds _)" [88] 87)

syntax (ASCII)
 "_ds_d" :: "lift \<Rightarrow> lift" ("(ds _)" [88] 87)

translations
 "_ds_d" \<rightleftharpoons> "CONST ds_d"

subsubsection \<open>Definition First and Last Operators\<close>

definition first_d :: "('a::world) formula \<Rightarrow> 'a formula"
where   
  "first_d f \<equiv>  LIFT (f \<and>  (bs (\<not> f)))"


syntax
 "_first_d" :: "lift \<Rightarrow> lift" ("(\<rhd> _)" [88] 87)

syntax (ASCII)
 "_first_d" :: "lift \<Rightarrow> lift" ("(first _)" [88] 87)

translations
 "_first_d" \<rightleftharpoons> "CONST first_d"



subsection \<open>Semantic Theorems\<close>

subsection \<open>Bi induction\<close>


lemma finite_ntaken:  
 "nfinite w \<Longrightarrow> f w = f (ntaken (the_enat (nlength w))  w) " 
by (metis ndropn_eq_NNil ndropn_nlast ntaken_all)


lemma biinduct_help: 
"(\<forall>n. enat n \<le> nlength w \<longrightarrow> f (ntaken n w) \<longrightarrow> n = 0 \<or> f (ntaken (n - Suc 0) w)) \<Longrightarrow> 
   f w \<Longrightarrow> 
   nfinite w \<Longrightarrow> 
   enat j \<le> nlength w \<Longrightarrow> 
   enat k = nlength w \<Longrightarrow>
    f (ntaken j w)"
proof (induct k arbitrary: w j)
case 0
then show ?case 
by (metis le_zero_eq ntaken_all zero_enat_def)
next
case (Suc k)
then show ?case 
  proof -
   have 1: "f (ntaken (k+1)  w)" 
     by (simp add: Suc.prems(2) Suc.prems(5) ntaken_all)
   have 2: "f (ntaken (k)  w)" 
    by (metis "1" One_nat_def Suc.prems(1) Suc.prems(5) Suc_eq_plus1 Zero_not_Suc diff_Suc_1 nle_le)
   have 3: "w = nappend (ntaken k w) (NNil (nlast(w)) )"
      by (metis Suc.prems(5) nappend_ntaken_ndropn ndropn_all order_refl) 
   have 4: "\<forall>n. enat n \<le> nlength (ntaken (k)  w) \<longrightarrow> f (ntaken n (ntaken (k)  w)) \<longrightarrow> 
      n = 0 \<or> f (ntaken (n - Suc 0) (ntaken (k)  w))" 
     by (metis "3" Suc.prems(1) diff_le_self min.bounded_iff ntaken_eq_ntaken_antimono 
          ntaken_nappend1 ntaken_nlength)
   have 5: " j = nlength w \<Longrightarrow> ?thesis" 
     by (simp add: Suc.prems(2) ntaken_all)
   have 6: "j< nlength w  \<Longrightarrow> ?thesis" 
    using 2 3 4 Suc.hyps[of "(ntaken k w)"] 
    by (metis (no_types, lifting) Suc.prems(5) Suc_ile_eq linorder_not_less min.orderE 
        nfinite_ntaken nle_le ntaken_nappend1 ntaken_nlength) 
    show ?thesis 
   using "5" "6" Suc.prems(4) order.order_iff_strict by blast
  qed
qed



lemma BiInduct:
 "\<turnstile> bi(f \<longrightarrow> wprev f) \<and> f \<and> finite \<longrightarrow> bi f "
proof -
 have 1: "\<And>j w. 
   (\<forall>n. enat n \<le> nlength w \<longrightarrow> f (ntaken n w) \<longrightarrow> n = 0 \<or> f (ntaken (n - Suc 0) w)) \<Longrightarrow> 
   f w \<Longrightarrow> 
   nfinite w \<Longrightarrow> 
   enat j \<le> nlength w \<Longrightarrow> 
    f (ntaken j w)"
   by (metis biinduct_help nfinite_conv_nlength_enat)
 have 2: "(\<turnstile> bi(f \<longrightarrow> wprev f) \<and> f \<and> finite \<longrightarrow> bi f) =
        (\<forall> w n. (\<forall>n. enat n \<le> nlength w \<longrightarrow> f (ntaken n w) \<longrightarrow> n = 0 \<or> f (ntaken (n - Suc 0) w)) \<longrightarrow>
           f w \<longrightarrow> nfinite w \<longrightarrow> enat n \<le> nlength w \<longrightarrow> f (ntaken n w))" 
   by (auto simp add: Valid_def itl_defs) 
      
  show ?thesis  using 1 2 by blast
qed

subsubsection \<open>Semantics First  Operator\<close>

lemma FstAndBfsem:
 "(nlength \<sigma> >0 \<and> (\<sigma> \<Turnstile> f) \<and> ( \<sigma> \<Turnstile>  bi (\<not>f)\<frown>skip)) =
  (nlength \<sigma> >0 \<and> nfinite \<sigma> \<and> (\<sigma> \<Turnstile> f) \<and> (\<forall>ia<nlength (\<sigma>). (ntaken ia  \<sigma> \<Turnstile> \<not>f)) ) "
proof -
 have " (nlength \<sigma> >0 \<and> (\<sigma> \<Turnstile> f) \<and> ( \<sigma> \<Turnstile>  bi (\<not>f)\<frown>skip)) =
        (0 < nlength \<sigma> \<and> nfinite \<sigma> \<and> (\<sigma> \<Turnstile> f) \<and>
           (\<exists>i. (i \<le> nlength \<sigma> \<longrightarrow> (\<forall>ia\<le>i. \<not> (ntaken ia (ntaken i \<sigma>) \<Turnstile> f)) \<and>
           nlength \<sigma> - i = Suc 0) \<and> i \<le> nlength \<sigma>)
          )"
     unfolding itl_defs  by simp 
    (metis enat_ord_simps(1) ndropn_nlength nfinite_ndropn_b nlength_eq_enat_nfiniteD order_subst2)  
 also have  " ... = 
          (0 < nlength \<sigma> \<and> nfinite \<sigma> \<and> (\<sigma> \<Turnstile> f) \<and>
           (\<exists>i. (i \<le> nlength \<sigma> \<longrightarrow> (\<forall>ia\<le>i. \<not> (ntaken ia (ntaken i \<sigma>) \<Turnstile> f)) \<and>
           i = nlength \<sigma> - Suc 0) \<and> i \<le> nlength \<sigma>)
          )" 
    by (auto  simp add: nfinite_conv_nlength_enat) 
      (metis Suc_diff_Suc cancel_comm_monoid_add_class.diff_cancel diff_zero lessI ntaken_all 
       order.order_iff_strict zero_less_iff_neq_zero)
 also have "... =
          (0 < nlength \<sigma> \<and> nfinite \<sigma> \<and> (\<sigma> \<Turnstile> f) \<and>
           (\<forall>ia. enat ia\<le>(nlength \<sigma> - Suc 0)\<longrightarrow> \<not> (ntaken ia (ntaken (the_enat (nlength \<sigma> - Suc 0)) \<sigma>) \<Turnstile> f) )
          )" 
    using diff_le_self by (auto simp add: min_def nfinite_conv_nlength_enat, blast)
 also have "... =
          (nlength \<sigma> >0 \<and> nfinite \<sigma> \<and> (\<sigma> \<Turnstile> f) \<and>
            (\<forall>ia. ia<nlength ( \<sigma>) \<longrightarrow> \<not> (ntaken ia (ntaken (the_enat (nlength \<sigma> - Suc 0)) \<sigma>) \<Turnstile> f))
          ) " 
       by (metis Suc_pred enat_ord_simps(1) enat_ord_simps(2) idiff_enat_enat less_Suc_eq_le
            nfinite_conv_nlength_enat zero_enat_def)
 also have "... =
          (nlength \<sigma> >0 \<and> nfinite \<sigma> \<and> (\<sigma> \<Turnstile> f) \<and>
           (\<forall>ia. ia<nlength ( \<sigma>) \<longrightarrow> (ntaken ia (ntaken (the_enat (nlength \<sigma> - Suc 0)) \<sigma>) \<Turnstile> \<not>f))
          ) " 
      by auto
 also have "... = 
          (nlength \<sigma> >0 \<and> nfinite \<sigma> \<and> (\<sigma> \<Turnstile> f) \<and> (\<forall>ia<nlength (\<sigma>). (ntaken ia  \<sigma> \<Turnstile> \<not>f)) ) " 
          by (metis Suc_pred enat_ord_simps(2) idiff_enat_enat less_Suc_eq_le min_def 
              nfinite_conv_nlength_enat ntaken_ntaken the_enat.simps zero_enat_def) 
 finally show " (nlength \<sigma> >0  \<and> (\<sigma> \<Turnstile> f) \<and> ( \<sigma> \<Turnstile>  bi (\<not>f)\<frown>skip)) =
               (nlength \<sigma> >0 \<and> nfinite \<sigma> \<and> (\<sigma> \<Turnstile> f) \<and> (\<forall>ia<nlength (\<sigma>). (ntaken ia  \<sigma> \<Turnstile> \<not>f)) ) " .
qed
     
lemma Fstsem_0:
 "(\<sigma> \<Turnstile> \<rhd>f) = 
  ( 
   ( \<sigma> \<Turnstile> f \<and> empty) \<or> (nlength \<sigma> >0  \<and> (\<sigma> \<Turnstile> f) \<and> ( \<sigma> \<Turnstile>  bi (\<not>f)\<frown>skip)) 
  )  "
using empty_defs by  (auto simp add: first_d_def bs_d_def)

lemma Emptysem:
 "(\<sigma> \<Turnstile> f \<and> empty) = ((\<sigma> \<Turnstile> f) \<and> nlength \<sigma> = 0) "
using empty_defs by auto

lemma Fstsem:
 "(\<sigma> \<Turnstile> \<rhd>f) =
  (  
   ( (\<sigma> \<Turnstile> f) \<and> nlength \<sigma> = 0) \<or>
   ( nlength \<sigma>>0 \<and> nfinite \<sigma> \<and> (\<sigma> \<Turnstile> f) \<and> (\<forall>ia<nlength (\<sigma>). (ntaken ia  \<sigma> \<Turnstile> \<not>f)))
  ) "
using Fstsem_0 Emptysem FstAndBfsem by metis


subsubsection \<open>Various Semantic Lemmas\<close>

lemma DiLensem:
 "(\<sigma> \<Turnstile> di (f \<and> len(i))) =
   ( (ntaken i \<sigma> \<Turnstile> f) \<and> i\<le>nlength \<sigma>) "
using nlength_eq_enat_nfiniteD by (auto simp add: itl_defs len_defs)

lemma DfLensem:
 "(\<sigma> \<Turnstile> df (f \<and> len(i))) =
   ( (ntaken i \<sigma> \<Turnstile> f) \<and> i\<le>nlength \<sigma>) "
by (auto simp add: itl_defs len_defs)

lemma PrefixFstsem:
 "( (ntaken i \<sigma> \<Turnstile> \<rhd>f) \<and> i\<le>nlength \<sigma>) =
  ( i\<le>nlength \<sigma> \<and>
   ( 
    ( (ntaken i \<sigma> \<Turnstile> f) \<and> i = 0) \<or>
    ( i>0 \<and> (ntaken i \<sigma> \<Turnstile> f) \<and> (\<forall>ia<i. (ntaken ia \<sigma> \<Turnstile> \<not>f)))
   ) 
 ) "
proof -
 have 1: "( ((ntaken i \<sigma>) \<Turnstile> \<rhd>f)) =
  (  
   ( ((ntaken i \<sigma>) \<Turnstile> f) \<and> nlength (ntaken i \<sigma>) = 0) \<or>
   ( nlength (ntaken i \<sigma>)>0 \<and> ((ntaken i \<sigma>) \<Turnstile> f) \<and> 
     (\<forall>ia<nlength (ntaken i \<sigma>). (ntaken ia  (ntaken i \<sigma>) \<Turnstile> \<not>f)) ) 
  ) " 
       using Fstsem by (metis nfinite_ntaken)
 hence 2: "( ((ntaken i \<sigma>) \<Turnstile> \<rhd>f) \<and> i\<le>nlength \<sigma>) =
  (  i\<le>nlength \<sigma> \<and> (
   ( ((ntaken i \<sigma>) \<Turnstile> f) \<and> nlength (ntaken i \<sigma>) = 0) \<or>
   ( nlength (ntaken i \<sigma>)>0 \<and> ((ntaken i \<sigma>) \<Turnstile> f) \<and> 
     (\<forall>ia<nlength (ntaken i \<sigma>). (ntaken ia  (ntaken i \<sigma>) \<Turnstile> \<not>f)) )
    )
  ) " 
       by auto
 hence 3: "( ((ntaken i \<sigma>) \<Turnstile> \<rhd>f) \<and> i\<le>nlength \<sigma>) =
  (  i\<le>nlength \<sigma> \<and> (
   ( ((ntaken i \<sigma>) \<Turnstile> f) \<and> i = 0) \<or>
   ( i>0 \<and> ((ntaken i \<sigma>) \<Turnstile> f) \<and> (\<forall>ia<i. (ntaken ia  (ntaken i \<sigma>) \<Turnstile> \<not>f)))
    )
  ) " 
     by (metis diff_zero enat_ord_simps(2) gr_zeroI less_numeral_extra(3) min.orderE ndropn_0 
         nlength_NNil nsubn_def1 nsubn_nlength_gr_one ntaken_0 ntaken_nlength)
 hence 4: "( ((ntaken i \<sigma>) \<Turnstile> \<rhd>f) \<and> i\<le>nlength \<sigma>) =
  (  i\<le>nlength \<sigma> \<and> (
   ( ((ntaken i \<sigma>) \<Turnstile> f) \<and> i = 0) \<or>
   ( i>0 \<and> ((ntaken i \<sigma>) \<Turnstile> f) \<and> (\<forall>ia<i. (ntaken ia \<sigma> \<Turnstile> \<not>f)))
    )
  ) " 
       using less_imp_add_positive by fastforce
 from 4 show ?thesis by auto
qed

lemma PrefixFstAndsem:
 "( (ntaken i \<sigma> \<Turnstile> \<rhd>f \<and> g) \<and> i\<le>nlength \<sigma>) =
  ( i\<le>nlength \<sigma> \<and>
   ( 
    ( (ntaken i \<sigma> \<Turnstile> f \<and> g ) \<and> i = 0) \<or>
    ( i>0 \<and> (ntaken i \<sigma> \<Turnstile> f \<and> g) \<and> (\<forall>ia<i. (ntaken ia \<sigma> \<Turnstile> \<not>f)))
   ) 
 ) "
using PrefixFstsem[of "f" "i" "\<sigma>"]    by (metis unl_lift2)

lemma DiLenFstsem:
 "(\<sigma> \<Turnstile> di (\<rhd>f \<and> len(i))) =
   ( i\<le>nlength \<sigma> \<and>
   ( 
    ( (ntaken i \<sigma> \<Turnstile> f) \<and> i = 0) \<or>
    ( i>0 \<and> (ntaken i \<sigma> \<Turnstile> f) \<and> (\<forall>ia<i. (ntaken ia \<sigma> \<Turnstile> \<not>f)))
   ) 
 ) "
by (simp add: DiLensem PrefixFstsem)

lemma DfLenFstsem:
 "(\<sigma> \<Turnstile> df (\<rhd>f \<and> len(i))) =
   ( i\<le>nlength \<sigma> \<and>
   ( 
    ( (ntaken i \<sigma> \<Turnstile> f) \<and> i = 0) \<or>
    ( i>0 \<and> (ntaken i \<sigma> \<Turnstile> f) \<and> (\<forall>ia<i. (ntaken ia \<sigma> \<Turnstile> \<not>f)))
   ) 
 ) "
by (simp add: DfLensem PrefixFstsem)


lemma DiLenFstAndsem:
 "(\<sigma> \<Turnstile> di ((\<rhd>f \<and> g) \<and> len(i))) =
   ( i\<le>nlength \<sigma> \<and>
   ( 
    ( (ntaken i \<sigma> \<Turnstile> f \<and> g) \<and> i = 0) \<or>
    ( i>0 \<and> (ntaken i \<sigma> \<Turnstile> f \<and> g) \<and> (\<forall>ia<i. (ntaken ia \<sigma> \<Turnstile> \<not>f)))
   ) 
 ) "
using DiLensem PrefixFstAndsem by metis

lemma DfLenFstAndsem:
 "(\<sigma> \<Turnstile> df ((\<rhd>f \<and> g) \<and> len(i))) =
   ( i\<le>nlength \<sigma> \<and>
   ( 
    ( (ntaken i \<sigma> \<Turnstile> f \<and> g) \<and> i = 0) \<or>
    ( i>0 \<and> (ntaken i \<sigma> \<Turnstile> f \<and> g) \<and> (\<forall>ia<i. (ntaken ia \<sigma> \<Turnstile> \<not>f)))
   ) 
 ) "
using DfLensem PrefixFstAndsem by metis

lemma FstLenSamesem:
 "( ( i\<le>nlength \<sigma> \<and>
   ( 
    ( (ntaken i \<sigma> \<Turnstile> f) \<and> i = 0) \<or>
    ( i>0 \<and> (ntaken i \<sigma> \<Turnstile> f) \<and> (\<forall>ia<i. (ntaken ia \<sigma> \<Turnstile> \<not>f)))
   ) 
 ) \<and>
  ( j\<le>nlength \<sigma> \<and>
   ( 
    ( (ntaken j \<sigma> \<Turnstile> f) \<and> j = 0) \<or>
    ( j>0 \<and> (ntaken j \<sigma> \<Turnstile> f) \<and> (\<forall>ia<j. (ntaken ia \<sigma> \<Turnstile> \<not>f)))
   ) 
 )
) \<longrightarrow> (i=j)
"
by (metis not_less_iff_gr_or_eq unl_lift)


subsection \<open>Theorems\<close>

subsubsection \<open>Fixed length intervals\<close>

lemma LenZeroEqvEmpty:
 "\<turnstile> len(0) = empty"
by (simp add: len_d_def)


lemma LenOneEqvSkip:
 "\<turnstile> len(1) = skip "
by (simp add: len_d_def ChopEmpty) 

lemma LenNPlusOneA:
 "\<turnstile>  len(n+1) = skip;len(n)"
by (simp add: len_d_def)

lemma SkipFiniteEqvFiniteSkip: 
 "\<turnstile> skip;finite = finite;skip "
using FiniteChopSkipEqvSkipChopFinite by auto

lemma TrueEqvFiniteOrInfinite: 
  "\<turnstile> #True = finite \<or> inf" 
by (simp add: FiniteOrInfinite)

lemma SkipInfEqvinfSkip: 
 "\<turnstile> skip;inf = inf;skip"
 by (metis ChopAndInf MoreAndInfEqvInf MoreEqvSkipChopTrue PowerstarEqvSemhelp2 int_simps(16) inteq_reflection lift_and_com) 

lemma SkipTrueEqvTrueSkip: 
 "\<turnstile> skip;#True = #True;skip "
proof -
 have 1: "\<turnstile> skip;#True = (skip;finite \<or> skip;inf)" 
  by (meson ChopOrEqv FiniteOrInfinite Prop04 RightChopEqvChop int_eq_true)
 have 2: "\<turnstile>  (finite;skip \<or> inf;skip) = #True;skip" 
  by (meson FiniteOrInfinite LeftChopEqvChop OrChopEqv Prop04 int_eq_true)
 have 3: "\<turnstile> (skip;finite \<or> skip;inf) = (finite;skip \<or> inf;skip) "
   by (metis "2" SkipFiniteEqvFiniteSkip SkipInfEqvinfSkip inteq_reflection)
 show ?thesis 
 by (metis "1" "2" "3" int_eq)
qed

lemma AndExistsLen:
 "\<turnstile> (f \<and> finite) = (f \<and> (\<exists>k. len(k)))"
using Finite_exist_len by fastforce


lemma AndExistsLenChop:
 "\<turnstile> (f\<frown>g) =  (\<exists>k. (f \<and> len(k));g)  "
by (simp add: Valid_def len_defs chop_defs schop_defs )
   (meson min.absorb1 nlength_eq_enat_nfiniteD)

lemma AndExistsLenSChop:
 "\<turnstile> (f\<frown>g) =  (\<exists>k. (f \<and> len(k))\<frown>g)  "
by (simp add: Valid_def len_defs chop_defs schop_defs )
   (meson min.absorb1 nlength_eq_enat_nfiniteD)

lemma AndExistsLenChopR:
 "\<turnstile> (f;(g \<and> finite)) = (\<exists>k. f;(g \<and> len(k))) "
by (simp add: Valid_def len_defs chop_defs finite_defs )
   (metis ndropn_nlength nfinite_conv_nlength_enat nfinite_ndropn)

lemma AndExistsLenSChopR:
 "\<turnstile> (f\<frown>(g \<and> finite)) = (\<exists>k. f\<frown>(g \<and> len(k))) "
by (simp add: Valid_def len_defs schop_defs finite_defs )
   (metis ndropn_nlength nfinite_conv_nlength_enat nfinite_ndropn)

lemma LFixedAndDistr:
 "\<turnstile> ((f0 \<and> len(k));g0 \<and> (f1 \<and> len(k));g1) = ((f0 \<and> f1) \<and> len(k));(g0 \<and> g1) "
by (auto simp add: Valid_def len_defs chop_defs nlength_eq_enat_nfiniteD)


lemma LFixedAndDistrS:
 "\<turnstile> ((f0 \<and> len(k))\<frown>g0 \<and> (f1 \<and> len(k))\<frown>g1) = ((f0 \<and> f1) \<and> len(k))\<frown>(g0 \<and> g1) "
by (auto simp add: Valid_def len_defs schop_defs)

lemma RFixedAndDistr:
 "\<turnstile> (f0;(g0 \<and> len(k)) \<and> f1;(g1 \<and> len(k))) = (f0 \<and> f1);((g0 \<and> g1) \<and> len(k))"
unfolding Valid_def itl_defs len_defs
by simp  
 (metis enat.simps(3) enat_add2_eq enat_add_sub_same le_iff_add ndropn_nlength
   nfinite_ndropn_b nlength_eq_enat_nfiniteD the_enat.simps)

lemma RFixedAndDistrS:
 "\<turnstile> (f0\<frown>(g0 \<and> len(k)) \<and> f1\<frown>(g1 \<and> len(k))) = (f0 \<and> f1)\<frown>((g0 \<and> g1) \<and> len(k))"
by (auto simp add: Valid_def len_defs schop_defs) 
   (metis enat.simps(3) enat_add2_eq enat_add_sub_same le_iff_add the_enat.simps)

lemma LFixedChopEqvFixedSChop:
 "\<turnstile> (f \<and> len(k));g = (f \<and> len(k))\<frown>g" 
by (auto simp add: Valid_def len_defs schop_defs chop_defs nlength_eq_enat_nfiniteD) 

lemma LFixedAndDistrA:
 "\<turnstile> ((f0 \<and> len(k));g0 \<and> (f1 \<and> len(k));g0) = ((f0 \<and> f1) \<and> len(k));g0 "
proof -
 have 1: "\<turnstile> ((f0 \<and> len(k));g0 \<and> (f1 \<and> len(k));g0) = ((f0 \<and> f1) \<and> len(k));(g0 \<and> g0) " 
      by (rule LFixedAndDistr)
 have 2: "\<turnstile> ((f0 \<and> f1) \<and> len(k));(g0 \<and> g0) = ((f0 \<and> f1) \<and> len(k));g0" 
      by auto
 from 1 2 show ?thesis by fastforce
qed

lemma LFixedAndDistrB:
 "\<turnstile> ((f0 \<and> len(k));g0 \<and> (f0 \<and> len(k));g1) = (f0 \<and> len(k));(g0 \<and> g1)"
proof -
 have 1: "\<turnstile> ((f0 \<and> len(k));g0 \<and> (f0 \<and> len(k));g1) = ((f0 \<and> f0) \<and> len(k));(g0 \<and> g1) " 
      by (rule LFixedAndDistr)
 have 2: "\<turnstile> ((f0 \<and> f0) \<and> len(k));(g0 \<and> g1) = (f0 \<and> len(k));(g0 \<and> g1)" 
     by auto
 from 1 2 show ?thesis by fastforce
qed

lemma LFixedAndDistrB1:
 "\<turnstile> (len(k);f \<and> len(k);g) = len(k);(f \<and> g) "
proof -
 have 1: "\<turnstile> len(k);f = (#True \<and> len(k));f " 
      by auto
 have 2: "\<turnstile> len(k);g = (#True \<and> len(k));g " 
      by auto
 have 3: "\<turnstile> (len(k);f \<and> len(k);g) = ((#True \<and> len(k));f \<and> (#True \<and> len(k));g)" 
      using "1" "2" by auto
 have 4: "\<turnstile> ((#True \<and> len(k));f \<and> (#True \<and> len(k));g) = (#True \<and> len(k));(f \<and> g)" 
      using LFixedAndDistrB by blast
 have 5: "\<turnstile> (#True \<and> len(k));(f \<and> g) = (len(k));(f \<and> g)" 
      by auto
 from 1 2 3 4 5 show ?thesis by auto
qed

lemma RFixedAndDistrA:
 "\<turnstile> (f0;(g0 \<and> len(k)) \<and> f0;(g1 \<and> len(k))) = f0;((g0 \<and> g1) \<and> len(k)) "
proof -
 have 1: "\<turnstile> (f0;(g0 \<and> len(k)) \<and> f0;(g1 \<and> len(k))) = (f0 \<and> f0);((g0 \<and> g1) \<and> len(k)) " 
      by (rule RFixedAndDistr)
 have 2: "\<turnstile> (f0 \<and> f0);((g0 \<and> g1) \<and> len(k)) = f0;((g0 \<and> g1) \<and> len(k)) " 
      by auto
from 1 2 show ?thesis by fastforce
qed

lemma RFixedAndDistrAS:
 "\<turnstile> (f0\<frown>(g0 \<and> len(k)) \<and> f0\<frown>(g1 \<and> len(k))) = f0\<frown>((g0 \<and> g1) \<and> len(k)) "
proof -
 have 1: "\<turnstile> (f0\<frown>(g0 \<and> len(k)) \<and> f0\<frown>(g1 \<and> len(k))) = (f0 \<and> f0)\<frown>((g0 \<and> g1) \<and> len(k)) " 
      by (rule RFixedAndDistrS)
 have 2: "\<turnstile> (f0 \<and> f0)\<frown>((g0 \<and> g1) \<and> len(k)) = f0\<frown>((g0 \<and> g1) \<and> len(k)) " 
      by auto
from 1 2 show ?thesis by fastforce
qed

lemma RFixedAndDistrB:
 "\<turnstile> (f0;(g0 \<and> len(k)) \<and> f1;(g0 \<and> len(k))) = (f0 \<and> f1);(g0 \<and> len(k)) "
proof -
 have 1: "\<turnstile> (f0;(g0 \<and> len(k)) \<and> f1;(g0 \<and> len(k))) = (f0 \<and> f1);((g0 \<and> g0) \<and> len(k)) " 
      by (rule RFixedAndDistr)
 have 2: "\<turnstile> (f0 \<and> f1);((g0 \<and> g0) \<and> len(k)) = (f0 \<and> f1);(g0 \<and> len(k)) " 
      by auto
from 1 2 show ?thesis by fastforce
qed 

lemma RFixedAndDistrBS:
 "\<turnstile> (f0\<frown>(g0 \<and> len(k)) \<and> f1\<frown>(g0 \<and> len(k))) = (f0 \<and> f1)\<frown>(g0 \<and> len(k)) "
proof -
 have 1: "\<turnstile> (f0\<frown>(g0 \<and> len(k)) \<and> f1\<frown>(g0 \<and> len(k))) = (f0 \<and> f1)\<frown>((g0 \<and> g0) \<and> len(k)) " 
      by (rule RFixedAndDistrS)
 have 2: "\<turnstile> (f0 \<and> f1)\<frown>((g0 \<and> g0) \<and> len(k)) = (f0 \<and> f1)\<frown>(g0 \<and> len(k)) " 
      by auto
from 1 2 show ?thesis by fastforce
qed 

lemma ChopSkipAndChopSkip:
 "\<turnstile> (f0;skip \<and> f1;skip) = (f0 \<and> f1);skip"
proof -
 have 1: "\<turnstile> (f0;(#True \<and> len(1)) \<and> f1;(#True \<and> len(1))) = (f0 \<and> f1);(#True \<and> len(1)) " 
      by (rule RFixedAndDistrB)
 have 2: "\<turnstile> (#True \<and> len(1)) = skip " 
      using LenOneEqvSkip by fastforce
 hence 3: "\<turnstile> f0;(#True \<and> len(1)) = f0;skip " 
      using RightChopEqvChop by blast
 have 4: "\<turnstile> f1;(#True \<and> len(1)) = f1;skip " 
      using "2" RightChopEqvChop by blast
 have 5: "\<turnstile> (f0;(#True \<and> len(1)) \<and> f1;(#True \<and> len(1))) = (f0;skip \<and> f1;skip) " 
      using "3" "4" by fastforce
 have 6: "\<turnstile> (f0 \<and> f1);(#True \<and> len(1)) = (f0 \<and> f1);skip" 
      using "2" RightChopEqvChop by blast
 from 1 5 6 show ?thesis by fastforce
qed

lemma SChopSkipAndSChopSkip:
 "\<turnstile> (f0\<frown>skip \<and> f1\<frown>skip) = (f0 \<and> f1)\<frown>skip"
proof -
 have 1: "\<turnstile> (f0\<frown>(#True \<and> len(1)) \<and> f1\<frown>(#True \<and> len(1))) = (f0 \<and> f1)\<frown>(#True \<and> len(1)) " 
      by (rule RFixedAndDistrBS)
 have 2: "\<turnstile> (#True \<and> len(1)) = skip " 
      using LenOneEqvSkip by fastforce
 hence 3: "\<turnstile> f0\<frown>(#True \<and> len(1)) = f0\<frown>skip " 
      using RightSChopEqvSChop by blast
 have 4: "\<turnstile> f1\<frown>(#True \<and> len(1)) = f1\<frown>skip " 
      using "2" RightSChopEqvSChop by blast
 have 5: "\<turnstile> (f0\<frown>(#True \<and> len(1)) \<and> f1\<frown>(#True \<and> len(1))) = (f0\<frown>skip \<and> f1\<frown>skip) " 
      using "3" "4" by fastforce
 have 6: "\<turnstile> (f0 \<and> f1)\<frown>(#True \<and> len(1)) = (f0 \<and> f1)\<frown>skip" 
      using "2" RightSChopEqvSChop by blast
 from 1 5 6 show ?thesis by fastforce
qed

lemma BiAndChopSkipEqv:
 "\<turnstile> (bi (f \<and> g));skip = ((bi f);skip \<and> (bi g);skip)"
proof -
 have 1: "\<turnstile> bi (f \<and> g) = ((bi f) \<and> (bi g)) " 
      by (auto simp add: bi_defs Valid_def)
 hence 2: "\<turnstile> (bi (f \<and> g));skip = (bi f \<and> bi g);skip" 
      by (rule LeftChopEqvChop)
 have 3: "\<turnstile> (bi f \<and> bi g);skip = ((bi f);skip \<and> (bi g);skip)" 
      using ChopSkipAndChopSkip  by fastforce
 from 2 3 show ?thesis by fastforce
qed

lemma BfAndSChopSkipEqv:
 "\<turnstile> (bf (f \<and> g))\<frown>skip = ((bf f)\<frown>skip \<and> (bf g)\<frown>skip)"
proof -
 have 1: "\<turnstile> bf (f \<and> g) = ((bf f) \<and> (bf g)) " 
      by (auto simp add: bf_defs Valid_def)
 hence 2: "\<turnstile> (bf (f \<and> g))\<frown>skip = (bf f \<and> bf g)\<frown>skip" 
      by (rule LeftSChopEqvSChop)
 have 3: "\<turnstile> (bf f \<and> bf g)\<frown>skip = ((bf f)\<frown>skip \<and> (bf g)\<frown>skip)" 
      using SChopSkipAndSChopSkip  by fastforce
 from 2 3 show ?thesis by fastforce
qed


lemma DiAndChopSkipEqv:
 "\<turnstile> (di (f \<and> g));skip \<longrightarrow> (di f);skip \<and> (di g);skip"
proof -
 have 1: "\<turnstile> di (f \<and> g) \<longrightarrow> (di f) \<and> (di g) " 
      by (simp add: DiAndImpAnd)
 hence 2: "\<turnstile> (di (f \<and> g));skip \<longrightarrow> (di f \<and> di g);skip" 
      by (rule LeftChopImpChop)
 have 3: "\<turnstile> (di f \<and> di g);skip = ((di f);skip \<and> (di g);skip)" 
      using ChopSkipAndChopSkip by fastforce
 from 2 3 show ?thesis by fastforce
qed

lemma DfAndSChopSkipEqv:
 "\<turnstile> (df (f \<and> g))\<frown>skip \<longrightarrow> (df f)\<frown>skip \<and> (df g)\<frown>skip"
proof -
 have 1: "\<turnstile> df (f \<and> g) \<longrightarrow> (df f) \<and> (df g) " 
      by (simp add: DfAndImpAnd)
 hence 2: "\<turnstile> (df (f \<and> g))\<frown>skip \<longrightarrow> (df f \<and> df g)\<frown>skip" 
      by (rule LeftSChopImpSChop)
 have 3: "\<turnstile> (df f \<and> df g)\<frown>skip = ((df f)\<frown>skip \<and> (df g)\<frown>skip)" 
      using SChopSkipAndSChopSkip by fastforce
 from 2 3 show ?thesis by fastforce
qed

lemma ChopEmptyAndEmpty:
 "\<turnstile> (f;g \<and> empty) = (f \<and> g \<and> empty)"
by (simp add: Valid_def itl_defs)
   (metis enat_0_iff(2) le_zero_eq ndropn_0 nfinite_ntaken ntaken_all)  

lemma SChopEmptyAndEmpty:
 "\<turnstile> (f\<frown>g \<and> empty) = (f \<and> g \<and> empty)"
by (simp add: Valid_def itl_defs)
   (metis is_NNil_ndropn le_numeral_extra(3) ndropn_0 ndropn_is_NNil ntaken_all zero_enat_def)


lemma ChopSkipImpMore:
 "\<turnstile> f;skip \<longrightarrow> more"
by (metis DiIntro DiSkipEqvMore RightChopImpMoreRule inteq_reflection)

lemma SChopSkipImpMore:
 "\<turnstile> f\<frown>skip \<longrightarrow> more"
by (metis DiIntro DiSkipEqvMore RightSChopImpMoreRule inteq_reflection)

lemma MoreEqvMoreChopTrue:
 "\<turnstile> more = more;#True"
proof -
 have 1: "\<turnstile> more = skip;#True" 
      using MoreEqvSkipChopTrue by blast
 have 2: "\<turnstile> #True = #True;#True" 
     by (auto simp add: Valid_def chop_defs)
        (metis zero_enat_def zero_le)
 hence 3: "\<turnstile> skip;#True = skip;(#True;#True)" 
      using RightChopEqvChop by blast
 have 4: "\<turnstile> skip;(#True;#True) = (skip;#True);#True" 
      using ChopAssoc by blast
 have 5: "\<turnstile> (skip;#True);#True = more;#True" 
      using MoreEqvSkipChopTrue by (simp add: more_d_def next_d_def)
 from 1 3 4 5 show ?thesis by fastforce
qed

lemma MoreEqvMoreSChopTrue:
 "\<turnstile> more = more\<frown>#True"
by (metis MoreEqvSkipSChopTrue SChopAssoc TrueEqvTrueSChopTrue inteq_reflection)

lemma NotNotChopSkip:
 "\<turnstile> (\<not>((\<not> f) ;skip)) = (empty \<or> (f;skip)) "
by (metis WprevEqvEmptyOrPrev prev_d_def wprev_d_def)


lemma NotSChopFixed:
 "\<turnstile> (\<not>(f\<frown>(g \<and> len(k)))) = (\<not>(\<diamond>(g \<and> len(k))) \<or> ((\<not>f)\<frown>(g \<and> len(k))))"
by (auto simp add: itl_defs Valid_def len_defs)
  (metis diff_diff_cancel enat_ord_simps(1) idiff_enat_enat ndropn_nlength 
     nfinite_conv_nlength_enat nfinite_ndropn the_enat.simps)

lemma NotNotSChopSkip:
 "\<turnstile> (\<not>((\<not> f) \<frown>skip)) =  (empty \<or> inf \<or> (f\<frown> skip))"
proof -
 have 1: "\<turnstile> ( (\<not> f) \<frown>skip \<or> f \<frown> skip) = (\<not>f \<or> f)\<frown>skip" 
 by (meson OrSChopEqv Prop11)
 have 2: "\<turnstile> (\<not>f \<or> f) = #True " 
   by simp
 have 3: "\<turnstile> more \<and> finite \<longrightarrow> #True\<frown>skip " 
   by (metis DiamondSChopdef FmoreEqvSkipChopFinite SkipFiniteEqvFiniteSkip fmore_d_def 
       int_simps(13) int_simps(9) inteq_reflection sometimes_d_def) 
 have 4: "\<turnstile> more \<and> finite \<longrightarrow> ((\<not> f) \<frown>skip) \<or> (f \<frown> skip)" 
   using "1" "3" by fastforce
 have 5: "\<turnstile> more \<and> finite \<and> \<not>((f \<frown> skip)) \<longrightarrow> ((\<not> f) \<frown>skip)" 
    using 4 by fastforce 
 have 6: "\<turnstile> (\<not>((\<not> f) \<frown>skip)) \<longrightarrow> empty \<or> inf \<or> (f\<frown> skip)"
    using 5 unfolding empty_d_def finite_d_def by fastforce
 have 7: "\<turnstile> ((\<not> f) \<frown>skip) \<longrightarrow> finite " 
   by (metis LenOneEqvSkip SChopImpFinite inteq_reflection len_k_finite)
 have 8: "\<turnstile> ((\<not> f) \<frown>skip) \<longrightarrow> more " 
  by (simp add: SChopSkipImpMore)
 have 9: "\<turnstile> ((\<not> f) \<frown>skip) \<longrightarrow> \<not>((f \<frown> skip)) " 
  by (meson NotSChopSkipEqvFmoreAndNotSChopSkip Prop11 Prop12)
 have 10 :"\<turnstile> ((\<not> f) \<frown>skip) \<longrightarrow> more \<and> finite \<and> \<not>((f \<frown> skip)) " 
   by (simp add: "7" "8" "9" Prop12)
 have 11: "\<turnstile> (empty \<or> inf \<or> (f\<frown> skip)) \<longrightarrow> (\<not>((\<not> f) \<frown>skip)) "
   using 10 unfolding finite_d_def empty_d_def by fastforce
 show ?thesis 
 by (meson "11" "6" Prop11)
qed

lemma  NotFixedSChop: 
 "\<turnstile> (\<not>((g \<and> len(k))\<frown>f)) = (\<not>(df(g \<and> len(k))) \<or> ((g \<and> len(k))\<frown>(\<not>f)))"
proof -
 have 1: "\<turnstile> (g \<and> len(k))\<frown>f = (g \<and> len(k));f " 
    by (meson LFixedChopEqvFixedSChop Prop11)
 have 2: "\<turnstile> (g \<and> len(k))\<frown>(\<not>f) = ((g \<and> len(k));(\<not>f)) " 
   by (meson LFixedChopEqvFixedSChop Prop11)
 have 3: "\<turnstile> df(g \<and> len(k)) = di(g \<and> len(k)) "
   by (metis LFixedChopEqvFixedSChop df_d_def di_d_def inteq_reflection)
 show ?thesis using 1 2 3 
 by (metis NotFixedChop inteq_reflection)
qed

lemma NotChopNotSkip:
 "\<turnstile> (\<not>(f;skip)) = (empty \<or> ((\<not> f);skip))"
proof -
 have 1: "\<turnstile> (\<not>((\<not>(\<not>f));skip)) = (empty \<or> ((\<not> f);skip))" using NotNotChopSkip by blast
 have 2: "\<turnstile> (\<not>((\<not>(\<not>f));skip)) = (\<not>(f;skip))" by auto
 from 1 2 show ?thesis by auto
qed

lemma NotSChopNotSkip:
 "\<turnstile> (\<not>(f\<frown>skip)) = (empty \<or> inf \<or> ((\<not> f)\<frown>skip))"
by (metis NotNotSChopSkip int_simps(4) inteq_reflection)


lemma NotNotSChopInf: 
 "\<turnstile> (\<not> ( (\<not> f)\<frown>inf)) = (finite \<or> (bf f))"
by (auto simp add: Valid_def itl_defs)
 

subsubsection \<open>Additional ITL theorems\<close>


lemma DiAndFiniteEqvDfAndFinite: 
 "\<turnstile> (di f \<and> finite) = (df f \<and> finite) " 
by (auto simp add: Valid_def di_defs df_defs finite_defs) 


lemma DiEqvDfOrInf: 
 "\<turnstile> di f = (df f \<or> (f \<and> inf))" 
by (simp add: ChopSChopdef df_d_def di_d_def)

lemma BiEqvBfAndfinite:
" \<turnstile> bi f = (bf f \<and> (inf \<longrightarrow> f))" 
proof -
have 1: " \<turnstile> di (\<not> f) = (df (\<not> f) \<or> \<not> f \<and> inf)" 
  using DiEqvDfOrInf[of "LIFT \<not> f"]  by blast
have 2: "\<turnstile> (\<not>di (\<not> f)) = (\<not>df (\<not> f) \<and> (inf \<longrightarrow> f)) " 
 using 1 by fastforce 
show ?thesis 
unfolding bf_d_def bi_d_def by (simp add: "2")
qed

lemma DfEqvDiAndFinite: 
 "\<turnstile> df f = di (f \<and> finite) " 
by (simp add: df_d_def di_d_def schop_d_def)

lemma BfEqvBiOrInf: 
 "\<turnstile> bf f = bi (f \<or> inf) " 
proof -
 have 1: "\<turnstile> df (\<not> f) = di (\<not> f \<and> finite) " 
   by (simp add: DfEqvDiAndFinite)
 have 2: "\<turnstile> (\<not>df (\<not> f)) = (\<not>di (\<not> f \<and> finite)) " 
     using "1" by auto
 have 3: "\<turnstile>  (\<not>di (\<not> f \<and> finite)) = (bi (\<not> (\<not> f \<and> finite))) "
     by (simp add: NotDiEqvBiNot)
 have 4: "\<turnstile> (\<not> (\<not> f \<and> finite)) = (f \<or> inf) " 
   unfolding finite_d_def by fastforce
 have 5: "\<turnstile> bi (\<not> (\<not> f \<and> finite)) = bi (f \<or> inf) " 
   by (metis "3" "4" inteq_reflection)
 have 6: "\<turnstile> bf f = (\<not> (df (\<not> f))) " 
   by (simp add: bf_d_def)
 show ?thesis 
 by (metis "2" "3" "5" "6" inteq_reflection)
qed


lemma BiOrBiImpBiOr:
 "\<turnstile> bi f \<or> bi g \<longrightarrow> bi(f \<or> g) "
proof -
 have 1: "\<turnstile> f \<longrightarrow> f \<or> g" by auto
 hence 2: "\<turnstile> bi f \<longrightarrow> bi(f \<or> g)" by (rule BiImpBiRule)
 have 3: "\<turnstile> g \<longrightarrow> f \<or> g" by auto
 hence 4: "\<turnstile> bi g \<longrightarrow> bi(f \<or> g)" by (rule BiImpBiRule)
 from 2 4 show ?thesis by fastforce
qed

lemma BfOrBfImpBfOr:
 "\<turnstile> bf f \<or> bf g \<longrightarrow> bf(f \<or> g) "
proof -
 have 1: "\<turnstile> f \<longrightarrow> f \<or> g" by auto
 hence 2: "\<turnstile> bf f \<longrightarrow> bf(f \<or> g)" by (rule BfImpBfRule)
 have 3: "\<turnstile> g \<longrightarrow> f \<or> g" by auto
 hence 4: "\<turnstile> bf g \<longrightarrow> bf(f \<or> g)" by (rule BfImpBfRule)
 from 2 4 show ?thesis by fastforce
qed


lemma MoreAndBiImpBiChopSkip:
 "\<turnstile> more \<and> bi f \<longrightarrow> (bi f);skip "
proof -
 have 1: "\<turnstile>  (bi f);skip = ((\<not>(di (\<not> f)));skip) " by (simp add: bi_d_def)
 have 2: "\<turnstile> (\<not>( (\<not>(di (\<not> f)));skip)) = (empty \<or> (di (\<not> f));skip)" by (rule NotNotChopSkip)
 have 3: "\<turnstile> empty \<longrightarrow> empty \<or> di (\<not> f) " by auto
 have 4: "\<turnstile> (di (\<not> f));skip \<longrightarrow> di (\<not> f)" using ChopImpDi DiEqvDiDi by fastforce
 hence 5: "\<turnstile> (di (\<not> f));skip \<longrightarrow> empty \<or> di (\<not> f)" by (rule Prop05)
 have 6: "\<turnstile> \<not>( (\<not>(di (\<not> f)));skip) \<longrightarrow> empty \<or> di (\<not> f)" using "2" "3" "5" by fastforce
 hence 7: "\<turnstile> \<not>(empty \<or> di (\<not> f)) \<longrightarrow> \<not>(\<not>( (\<not>(di (\<not> f)));skip)) "  by fastforce
 have 8: "\<turnstile> (\<not>( \<not>( (\<not>(di (\<not> f)));skip))) = ((\<not>(di (\<not> f)));skip)" by auto
 have 9: "\<turnstile> (\<not>(empty \<or> di (\<not> f))) = (more \<and> \<not>( di (\<not> f))) "
 unfolding empty_d_def by force
 have 10: "\<turnstile> (more \<and> \<not>( di (\<not> f))) = (more \<and> bi f)" by (simp add: bi_d_def)
 from 1 6 7 8 9 10 show ?thesis by (metis int_eq)
qed

lemma MoreAndBiImpBiSChopSkip:
 "\<turnstile> more \<and> finite \<and> bi f \<longrightarrow> (bi f)\<frown>skip "
proof -
 have 1: "\<turnstile>  (bi f)\<frown>skip = ((\<not>(di (\<not> f)))\<frown>skip) " by (simp add: bi_d_def)
 have 2: "\<turnstile> (\<not>( (\<not>(di (\<not> f)))\<frown>skip)) = (empty \<or> inf \<or> (di (\<not> f))\<frown>skip)" by (rule NotNotSChopSkip)
 have 3: "\<turnstile> empty \<longrightarrow> empty \<or> di (\<not> f) " by auto
 have 03: "\<turnstile> skip \<longrightarrow> #True" by simp
 have 04: "\<turnstile> (di (\<not> f))\<frown>skip \<longrightarrow> di (\<not> f)\<frown>#True " by (simp add: RightSChopImpSChop)
 have 4: "\<turnstile> (di (\<not> f))\<frown>skip \<longrightarrow> df (\<not> f)" unfolding di_d_def df_d_def 
 by (metis (no_types, opaque_lifting) ChopAssoc ChopImpDi ChopTrueAndFiniteEqvAndFiniteChopFinite
     di_d_def inteq_reflection schop_d_def)
 hence 5: "\<turnstile> (di (\<not> f))\<frown>skip \<longrightarrow> empty \<or> df (\<not> f)" by (rule Prop05)
 have 6: "\<turnstile> \<not>( (\<not>(di (\<not> f)))\<frown>skip) \<longrightarrow> empty \<or> inf \<or> df (\<not> f)" using "2" "3" "5" by fastforce
 hence 7: "\<turnstile> \<not>(empty \<or> inf \<or> df (\<not> f)) \<longrightarrow> \<not>(\<not>( (\<not>(di (\<not> f)))\<frown>skip)) "  by fastforce
 have 8: "\<turnstile> (\<not>( \<not>( (\<not>(di (\<not> f)))\<frown>skip))) = ((\<not>(di (\<not> f)))\<frown>skip)" by auto
 have 9: "\<turnstile> (\<not>(empty \<or> inf \<or> df (\<not> f))) = (more \<and> finite \<and>\<not>( df (\<not> f))) "
 unfolding empty_d_def finite_d_def by force
 have 10: "\<turnstile> (more \<and> finite \<and> \<not>( df (\<not> f))) = (more \<and> finite \<and> bf f)"
 by (simp add: bf_d_def)
 have 11: "\<turnstile> (more \<and> finite \<and> bf f) = (more \<and> finite \<and> bi f) "
   using BfElim BiEqvBfAndfinite by fastforce 
 from 1 6 7 8 9 10 11 show ?thesis by (metis inteq_reflection)
qed

lemma MoreAndBfImpBfSChopSkip:
 "\<turnstile> more \<and> bf f \<longrightarrow> inf \<or> (bf f)\<frown>skip "
proof -
 have 1: "\<turnstile>  (bf f)\<frown>skip = ((\<not>(df (\<not> f)))\<frown>skip) " by (simp add: bf_d_def)
 have 2: "\<turnstile> (\<not>( (\<not>(df (\<not> f)))\<frown>skip)) = (empty \<or> inf \<or>(df (\<not> f))\<frown>skip)" by (rule NotNotSChopSkip)
 have 3: "\<turnstile> empty \<longrightarrow> empty \<or> inf \<or> df (\<not> f) " by auto
 have 4: "\<turnstile> (df (\<not> f))\<frown>skip \<longrightarrow> df (\<not> f)" 
 by (metis DfEqvDfDf SChopImpDf inteq_reflection)
 hence 5: "\<turnstile> (df (\<not> f))\<frown>skip \<longrightarrow> empty \<or> inf \<or> df (\<not> f)"
   by (simp add: Prop05)
 have 6: "\<turnstile> \<not>( (\<not>(df (\<not> f)))\<frown>skip) \<longrightarrow> empty \<or> inf \<or> df (\<not> f)" using "2" "3" "5" by fastforce
 hence 7: "\<turnstile> \<not>(empty \<or> inf \<or> df (\<not> f)) \<longrightarrow> \<not>(\<not>( (\<not>(df (\<not> f)))\<frown>skip)) "  by fastforce
 have 8: "\<turnstile> (\<not>( \<not>( (\<not>(df (\<not> f)))\<frown>skip))) = ((\<not>(df (\<not> f)))\<frown>skip)" by auto
 have 9: "\<turnstile> (\<not>(empty \<or> inf \<or> df (\<not> f))) = (more \<and> finite \<and>\<not>( df (\<not> f))) "
 unfolding empty_d_def unfolding finite_d_def by fastforce
 have 10: "\<turnstile> (more \<and> finite \<and> \<not>( df (\<not> f))) = (more \<and> finite \<and> bf f)" by (simp add: bf_d_def)
 from 1 6 7 8 9 10 show ?thesis unfolding finite_d_def 
 using NotNotSChopSkip Prop11 by fastforce
qed

lemma DiChopEqvChopDi: 
 "\<turnstile> di(f;g) = f;(di g) "
by (metis ChopAssoc Prop11 di_d_def)

lemma DiChopImpDiB:
 "\<turnstile> di(f;g) \<longrightarrow> di f "
proof -
 have 1: "\<turnstile> f ; (g;#True) \<longrightarrow> di f" by (rule ChopImpDi)
 have 2: "\<turnstile> f ; (g;#True) = (f;g);#True " by (rule ChopAssoc)
 from 1 2 show ?thesis by (metis di_d_def int_eq)
qed

lemma DfSChopEqvSChopDf: 
 "\<turnstile> df(f;g) = f\<frown>(df g) "
by (metis (no_types, opaque_lifting) ChopAndFiniteDist ChopEmpty SChopAssoc 
    int_eq itl_def(15) itl_def(9))

lemma DfChopEqvDfSchop: 
 "\<turnstile> df(f;g) = df(f\<frown>g) " 
by (metis DfSChopEqvSChopDf Prop04 Prop11 SChopAssoc df_d_def)

lemma DfSChopImpDfB:
 "\<turnstile> df(f\<frown>g) \<longrightarrow> df f "
proof -
 have 1: "\<turnstile> f \<frown> (g\<frown>#True) \<longrightarrow> df f" by (simp add: SChopImpDf)
 have 2: "\<turnstile> f \<frown> (g\<frown>#True) = (f\<frown>g)\<frown>#True " by (rule SChopAssoc)
 from 1 2 show ?thesis by (metis df_d_def int_eq)
qed

lemma BiBiOrImpBi:
 "\<turnstile> bi ( bi f \<or> bi g) \<longrightarrow> bi f \<or> bi g"
using BiElim by auto

lemma BfBfOrImpBf:
 "\<turnstile> bf ( bf f \<or> bf g) \<longrightarrow> bf f \<or> bf g"
proof -
 have 1: "\<turnstile> bf ( bf f \<or> bf g) \<and> finite \<longrightarrow> bf f \<or> bf g" 
 using BfElim by fastforce
 have 2: "\<turnstile> bf ( bf f \<or> bf g) \<and> inf \<longrightarrow> bf f \<or> bf g" 
   by (auto simp add: Valid_def itl_defs)
      (meson nle_le)
 show ?thesis using 1 2 unfolding finite_d_def 
 by (meson "1" OrFiniteInf Prop02 Prop11 lift_imp_trans)
qed    

lemma BiImpBiBiOr:
 "\<turnstile> bi f  \<longrightarrow> bi ( bi f \<or> bi g)"
proof -
 have 1: "\<turnstile> bi f \<longrightarrow> bi f \<or> bi g" by auto
 hence 2: "\<turnstile> bi (bi f) \<longrightarrow> bi(bi f \<or> bi g)  " using BiImpBiRule by blast
 have 3: "\<turnstile> bi (bi f) = bi f" using BiEqvBiBi by fastforce
 from 2 3 show ?thesis by fastforce
qed

lemma BfImpBfBfOr:
 "\<turnstile> bf f  \<longrightarrow> bf ( bf f \<or> bf g)"
proof -
 have 1: "\<turnstile> bf f \<longrightarrow> bf f \<or> bf g" by auto
 hence 2: "\<turnstile> bf (bf f) \<longrightarrow> bf(bf f \<or> bf g)  " using BfImpBfRule by blast
 have 3: "\<turnstile> bf (bf f) = bf f" using BfEqvBfBf by fastforce
 from 2 3 show ?thesis by fastforce
qed

lemma BiImpBiBiOrB:
 "\<turnstile> bi g  \<longrightarrow> bi ( bi f \<or> bi g)"
proof -
 have 1: "\<turnstile> bi g \<longrightarrow> bi f \<or> bi g" by auto
 hence 2: "\<turnstile> bi (bi g) \<longrightarrow> bi(bi f \<or> bi g)  " using BiImpBiRule by blast
 have 3: "\<turnstile> bi (bi g) = bi g" using BiEqvBiBi by fastforce
 from 2 3 show ?thesis by fastforce
qed

lemma BfImpBfBfOrB:
 "\<turnstile> bf g  \<longrightarrow> bf ( bf f \<or> bf g)"
proof -
 have 1: "\<turnstile> bf g \<longrightarrow> bf f \<or> bf g" by auto
 hence 2: "\<turnstile> bf (bf g) \<longrightarrow> bf(bf f \<or> bf g)  " using BfImpBfRule by blast
 have 3: "\<turnstile> bf (bf g) = bf g" using BfEqvBfBf by fastforce
 from 2 3 show ?thesis by fastforce
qed


lemma BiBiOrEqvBi:
 "\<turnstile> bi ( bi f \<or> bi g) = (bi f \<or> bi g) "
proof -
 have 1: "\<turnstile> bi ( bi f \<or> bi g) \<longrightarrow> bi f \<or> bi g " by (rule BiBiOrImpBi)
 have 2: "\<turnstile> bi f  \<longrightarrow> bi ( bi f \<or> bi g) " by (rule BiImpBiBiOr)
 have 3: "\<turnstile> bi g  \<longrightarrow> bi ( bi f \<or> bi g) " by (rule BiImpBiBiOrB)
 have 4: "\<turnstile> bi f  \<or> bi g \<longrightarrow> bi ( bi f \<or> bi g) " using "2" "3" by fastforce
 from 1 4 show ?thesis by fastforce
qed

lemma BfBfOrEqvBf:
 "\<turnstile> bf ( bf f \<or> bf g) = (bf f \<or> bf g) "
by (meson BfBfOrImpBf BfImpBfBfOr BfImpBfBfOrB Prop02 int_iffI)

lemma DiEqvOrDiChopSkipA:
 "\<turnstile> di f = (f \<or> di(f;skip))"
proof -
 have 1: "\<turnstile> di f = f ;#True " by (simp add: di_d_def)
 hence 2: "\<turnstile> di f = f; ( empty \<or> more)" by (simp add: empty_d_def)
 hence 3: "\<turnstile>f; ( empty \<or> more) = (f;empty \<or> f;more) " using ChopOrEqv by blast
 have 4: "\<turnstile> f;empty = f" by (rule ChopEmpty)
 have 5: "\<turnstile> more = skip;#True" using MoreEqvSkipChopTrue by blast
 hence 6: "\<turnstile> f;more = f;(skip;#True)" using RightChopEqvChop by blast
 have 7: "\<turnstile> f;(skip;#True) = (f;skip);#True" by (rule ChopAssoc)
 from 2 3 4 6 7 show ?thesis by (metis di_d_def int_eq)
qed 

lemma DfEqvOrDfSChopSkipA:
 "\<turnstile> df f = ((f \<and> finite) \<or> df(f\<frown>skip))"
proof -
 have 1: "\<turnstile> df f = f \<frown>#True " by (simp add: df_d_def)
 hence 2: "\<turnstile> df f = f \<frown> ( empty \<or> more)" by (simp add: empty_d_def)
 hence 3: "\<turnstile>f\<frown> ( empty \<or> more) = (f\<frown>empty \<or> f\<frown>more) " using SChopOrEqv by blast
 have 4: "\<turnstile> f\<frown>empty = (f \<and> finite)" by (simp add: ChopEmpty schop_d_def)
 have 5: "\<turnstile> more = skip\<frown>#True " using MoreEqvSkipSChopTrue by simp
 hence 6: "\<turnstile> f\<frown>more = f\<frown>(skip\<frown>#True)" using RightSChopEqvSChop by blast
 have 7: "\<turnstile> f\<frown>(skip\<frown>#True) = (f\<frown>skip)\<frown>#True" by (rule SChopAssoc)
 from 2 3 4 6 7 show ?thesis by (metis df_d_def inteq_reflection)
qed 

lemma DiEqvOrDiChopSkipB:
 "\<turnstile> di f = (f \<or> (di f);skip)"
proof -
 have 1: "\<turnstile> (di f) = (f \<or> di(f;skip))" by (rule DiEqvOrDiChopSkipA)
 have 2: "\<turnstile>  di(f;skip) = (f;skip);#True " by (simp add: di_d_def)
 have 3: "\<turnstile> (f;skip);#True = f;(skip;#True)" 
   by (meson ChopAssoc Prop11)
 have 4: "\<turnstile> di(f;skip) = f;(skip;#True) " using "2" "3" by fastforce
 have 5: "\<turnstile> skip;#True = #True;skip"  
   by (simp add: SkipTrueEqvTrueSkip)
 hence 6: "\<turnstile> f;(skip;#True) = f;(#True;skip)" using RightChopEqvChop by blast
 have 7: "\<turnstile> di(f;skip) = f;(#True;skip)" using "4" "6" by fastforce
 have 8: "\<turnstile> f;(#True;skip) = (f;#True);skip" by (rule ChopAssoc)
 have 9: "\<turnstile> (f;#True);skip = (di f);skip" by (simp add: di_d_def)
 have 10 : "\<turnstile> di(f;skip) =(di f);skip " using "7" "8" "9" by fastforce
 hence 11: "\<turnstile> (f \<or> di(f;skip)) = (f \<or> (di f);skip) " by auto
 from 1 11 show ?thesis by fastforce
qed

lemma DfEqvOrDfSChopSkipB:
 "\<turnstile> df f = ((f \<and> finite) \<or> (df f)\<frown>skip \<or> (df f) \<frown> inf)"
proof -
 have 1: "\<turnstile> (df f) = ((f \<and> finite) \<or> df(f\<frown>skip))" by (rule DfEqvOrDfSChopSkipA)
 have 2: "\<turnstile>  df(f\<frown>skip) = (f\<frown>skip)\<frown>#True " by (simp add: df_d_def)
 have 3: "\<turnstile> (f\<frown>skip)\<frown>#True = f\<frown>(skip\<frown>#True)" 
   by (meson SChopAssoc Prop11)
 have 4: "\<turnstile> df(f\<frown>skip) = f\<frown>(skip\<frown>#True) " using "2" "3" by fastforce
 have 5: "\<turnstile> skip\<frown>#True = (#True\<frown>skip \<or> #True\<frown>inf)"
   by (metis FiniteChopInfEqvInf FmoreEqvSkipChopFinite MoreAndInfEqvInf MoreEqvSkipSChopTrue
        OrFiniteInf SkipFiniteEqvFiniteSkip fmore_d_def int_simps(17) inteq_reflection schop_d_def)
 hence 6: "\<turnstile> f\<frown>(skip\<frown>#True) = f\<frown>(#True\<frown>skip \<or> #True\<frown>inf)" using RightSChopEqvSChop by blast
 have 7: "\<turnstile> df(f\<frown>skip) = f\<frown>(#True\<frown>skip \<or> #True\<frown>inf)" using "4" "6" by fastforce
 have 71: "\<turnstile> f\<frown>(#True\<frown>skip \<or> #True\<frown>inf) = ( f\<frown>(#True\<frown>skip) \<or> f\<frown>(#True\<frown>inf))"
    by (simp add: SChopOrEqv) 
 have 8: "\<turnstile> f\<frown>(#True\<frown>skip) = (f\<frown>#True)\<frown>skip" by (rule SChopAssoc)
 have 81: "\<turnstile> f\<frown>(#True\<frown>inf) = (f\<frown>#True)\<frown>inf" by (rule SChopAssoc) 
 have 9: "\<turnstile> (f\<frown>#True)\<frown>skip = (df f)\<frown>skip" by (simp add: df_d_def)
 have 91: "\<turnstile> (f\<frown>#True)\<frown>inf = (df f)\<frown>inf"  by (simp add: df_d_def)
 have 10 : "\<turnstile> df(f\<frown>skip) = ((df f)\<frown>skip \<or> (df f) \<frown> inf)" 
    using 8 81 9 91 7 71 by (metis int_eq)
 hence 11: "\<turnstile> ((f \<and> finite) \<or> df(f\<frown>skip)) = ((f \<and> finite) \<or> (df f)\<frown>skip  \<or> (df f)\<frown> inf) " by auto
 from 1 11 show ?thesis by fastforce
qed


  
lemma BiEqvAndEmptyOrBiChopSkip:
 "\<turnstile> bi f = (f \<and> (empty \<or> (bi f);skip)) "
proof -
 have 1: "\<turnstile> di (\<not> f) = (\<not> f \<or> (di (\<not> f);skip))" by (rule DiEqvOrDiChopSkipB)
 have 2: "\<turnstile> di (\<not> f) = (\<not>( bi f ))" by (rule DiNotEqvNotBi)
 have 3: "\<turnstile> (\<not>( bi f )) =  (\<not> f \<or> (di (\<not> f);skip)) " using "1" "2" by fastforce
 hence 4: "\<turnstile> bi f = (\<not>(\<not> f \<or> (di (\<not> f);skip))) " by auto
 have 5: "\<turnstile> (\<not>(\<not> f \<or> (di (\<not> f);skip))) = (f \<and> \<not>(di (\<not> f);skip)) " by auto
 have 6: "\<turnstile> di (\<not> f);skip = ((\<not>(bi f));skip) " by (simp add: "2" LeftChopEqvChop)
 hence 7: "\<turnstile> (\<not>(di (\<not> f);skip)) = (\<not>((\<not>(bi f));skip)) " by auto
 have 8: "\<turnstile> (\<not>((\<not>(bi f));skip)) = (empty \<or> (bi f);skip) " using NotNotChopSkip by blast
 hence 9: "\<turnstile> (f \<and> \<not>(di (\<not> f);skip)) = (f \<and> (empty \<or> (bi f);skip)) " using "7" "8" by fastforce
 from 4 5 9 show ?thesis by fastforce
qed


lemma BfEqvAndEmptyOrBfSChopSkip:
 "\<turnstile> bf f =  ( (finite  \<longrightarrow> f) \<and> (empty \<or> inf \<or> (bf f)\<frown>skip) \<and> (finite \<or> bf f) )  "
proof -
 have 1: "\<turnstile> (df (\<not> f)) = ((\<not> f \<and> finite)\<or> (df (\<not> f)\<frown>skip) \<or> (df (\<not> f)\<frown>inf))" 
   by (rule DfEqvOrDfSChopSkipB)
 have 2: "\<turnstile> (df (\<not> f)) = (\<not>( bf f ))" by (rule DfNotEqvNotBf)
 have 3: "\<turnstile> (\<not>( bf f )) =  ( (\<not> f \<and> finite) \<or> (df (\<not> f)\<frown>skip) \<or> (df (\<not> f)\<frown>inf)) " 
   using "1" "2" by fastforce
 hence 4: "\<turnstile> (bf f) = (\<not>((\<not> f \<and> finite) \<or> (df (\<not> f)\<frown>skip) \<or> (df (\<not> f) \<frown>inf))) " by auto
 have 5: "\<turnstile> (\<not>((\<not> f \<and> finite) \<or> (df (\<not> f)\<frown>skip) \<or> (df (\<not> f)\<frown>inf))) = 
            ( (finite \<longrightarrow> f) \<and> (\<not>(df (\<not> f)\<frown>skip)) \<and> (\<not>(df (\<not> f)\<frown>inf))) " by auto
 have 50: "\<turnstile> (bf f) = ( (finite \<longrightarrow> f) \<and> \<not>(df (\<not> f)\<frown>skip) \<and> \<not>(df (\<not> f)\<frown>inf))" 
     using "4" by auto
 have 6: "\<turnstile> df (\<not> f)\<frown>skip = ((\<not>(bf f))\<frown>skip) " by (simp add: "2" LeftSChopEqvSChop)
 hence 7: "\<turnstile> (\<not>(df (\<not> f)\<frown>skip)) = (\<not>((\<not>(bf f))\<frown>skip)) " by auto
 have 8: "\<turnstile> (\<not>((\<not>(bf f))\<frown>skip)) = (empty \<or> inf \<or> (bf f)\<frown>skip) " using NotNotSChopSkip by blast
 have 61: "\<turnstile> df (\<not> f)\<frown>inf = ((\<not>(bf f))\<frown>inf)" by (simp add: "2" LeftSChopEqvSChop)
 hence 71: "\<turnstile> (\<not>(df (\<not> f)\<frown>inf)) = (\<not>((\<not>(bf f))\<frown>inf)) " by auto
 have 81: "\<turnstile> (\<not>(df (\<not> f)\<frown>inf)) = (finite \<or> (bf (bf f)))" 
    by (metis "2" NotNotSChopInf int_eq)
 have 82: "\<turnstile> (bf (bf f)) = bf f"
    by (simp add: BfEqvBfBf BfImpBfBf int_iffD2 int_iffI)
 have 83: "\<turnstile> (finite \<or> (bf (bf f))) = (finite \<or> bf f)" 
    using "82" by auto      
 hence 9: "\<turnstile> ((finite \<longrightarrow> f) \<and> \<not>(df (\<not> f)\<frown>skip) \<and>  \<not>(df (\<not> f)\<frown>inf)) = 
             ((finite  \<longrightarrow> f) \<and> (empty \<or> inf \<or> (bf f)\<frown>skip) \<and> (finite \<or> bf f) ) "
    by (metis "2" "8" "81" inteq_reflection lift_and_com)
 from 4 5 9  show ?thesis 
 by (metis inteq_reflection)
qed


lemma DiDiAndEqvDi:
 "\<turnstile> di ( di f \<and> di g) = (di f \<and> di g) "
proof -
 have 1: "\<turnstile> bi ( bi (\<not> f) \<or> bi (\<not> g)) = (bi (\<not>f) \<or> bi (\<not> g)) " 
      by (meson BiBiOrImpBi BiImpBiBiOr BiImpBiBiOrB Prop02 int_iffI)
 have 2: "\<turnstile> bi (\<not>f) = (\<not> (di f)) " 
      by (simp add: bi_d_def)
 have 3: "\<turnstile> bi (\<not>g) = (\<not> (di g)) " 
      by (simp add: bi_d_def)
 have 4: "\<turnstile> (bi (\<not>f) \<or> bi (\<not> g)) = (\<not> (di f) \<or> \<not> (di g)) " 
      using "2" "3" by fastforce
 have 5: "\<turnstile> (\<not> (di f) \<or> \<not> (di g)) = (\<not>(di f \<and> di g)) " 
      by auto
 have 6: "\<turnstile> bi ( bi (\<not> f) \<or> bi (\<not> g)) = (\<not>(di f \<and> di g)) " 
      using "1" "5" "4" by fastforce
 hence 7: "\<turnstile> (\<not>(bi ( bi (\<not> f) \<or> bi (\<not> g)))) = (di f \<and> di g)  " 
      by auto
 have 8 : "\<turnstile> (\<not>(bi ( bi (\<not> f) \<or> bi (\<not> g)))) = di ( \<not>(bi (\<not> f) \<or> bi (\<not> g)))  " 
      using DiNotEqvNotBi  by fastforce
 have 9 : "\<turnstile> (\<not>(bi (\<not> f) \<or> bi (\<not> g))) = (di f \<and> di g) " 
      using "1" "7" by fastforce
 hence 10: "\<turnstile> di ( \<not>(bi (\<not> f) \<or> bi (\<not> g))) = di ( di f \<and> di g) " 
      using DiEqvDi by blast
 from 7 8 10 show ?thesis by fastforce 
qed

lemma DfDfAndEqvDf:
 "\<turnstile> df ( df f \<and> df g) = (df f \<and> df g) "
proof -
 have 1: "\<turnstile> bf ( bf (\<not> f) \<or> bf (\<not> g)) = (bf (\<not>f) \<or> bf (\<not> g)) " 
      by (meson BfBfOrImpBf BfImpBfBfOr BfImpBfBfOrB Prop02 int_iffI)
 have 2: "\<turnstile> bf (\<not>f) = (\<not> (df f)) " 
      by (simp add: bf_d_def)
 have 3: "\<turnstile> bf (\<not>g) = (\<not> (df g)) " 
      by (simp add: bf_d_def)
 have 4: "\<turnstile> (bf (\<not>f) \<or> bf (\<not> g)) = (\<not> (df f) \<or> \<not> (df g)) " 
      using "2" "3" by fastforce
 have 5: "\<turnstile> (\<not> (df f) \<or> \<not> (df g)) = (\<not>(df f \<and> df g)) " 
      by auto
 have 6: "\<turnstile> bf ( bf (\<not> f) \<or> bf (\<not> g)) = (\<not>(df f \<and> df g)) " 
      using "1" "5" "4" by fastforce
 hence 7: "\<turnstile> (\<not>(bf ( bf (\<not> f) \<or> bf (\<not> g)))) = (df f \<and> df g)  " 
      by auto
 have 8 : "\<turnstile> (\<not>(bf ( bf (\<not> f) \<or> bf (\<not> g)))) = df ( \<not>(bf (\<not> f) \<or> bf (\<not> g)))  " 
      using DfNotEqvNotBf  by fastforce
 have 9 : "\<turnstile> (\<not>(bf (\<not> f) \<or> bf (\<not> g))) = (df f \<and> df g) " 
      using "1" "7" by fastforce
 hence 10: "\<turnstile> df ( \<not>(bf (\<not> f) \<or> bf (\<not> g))) = df ( df f \<and> df g) " 
      using DfEqvDf by blast
 from 7 8 10 show ?thesis by fastforce
qed


lemma BiInductLen:
 "\<turnstile> bi(f \<longrightarrow> wprev f) \<and> f \<and> (len k) \<longrightarrow> bi f "
proof -
 have 1: "\<turnstile> len k \<longrightarrow> finite" 
   by (simp add: len_k_finite)
 have 2: "\<turnstile> bi(f \<longrightarrow> wprev f) \<and> f \<and> finite \<longrightarrow> bi f " 
   by (simp add: BiInduct)
 have 3: "\<turnstile> finite = (\<exists>k. len k) " 
   by (simp add: Finite_exist_len) 
 have 4: "\<turnstile>bi(f \<longrightarrow> wprev f) \<and> f \<and> (\<exists>k. len k) \<longrightarrow> bi f " 
   by (metis "2" "3" inteq_reflection)
 show ?thesis using 4 by fastforce
qed


lemma  NotFinitePrevEqv: 
 "\<turnstile> (\<not>((f \<and> finite);skip)) = (empty \<or> (\<not> f \<or> inf);skip) " 
proof -
 have 1: "\<turnstile> (\<not>((f \<and> finite);skip)) = (empty \<or> (\<not>(f \<and> finite));skip) "
   by (simp add: NotChopNotSkip)
 have 2: "\<turnstile> (\<not>(f \<and> finite)) = (\<not>f \<or> inf) " 
   unfolding finite_d_def by fastforce
 have 3: "\<turnstile> (empty \<or> (\<not>(f \<and> finite));skip) = (empty \<or> (\<not>f \<or> inf);skip) " 
   by (metis "1" "2" inteq_reflection)
 show ?thesis 
 by (metis "1" "3" inteq_reflection)
qed

lemma WPrevAndFiniteEqv: 
 "\<turnstile> wprev (f \<and> finite) = (empty \<or> f\<frown>skip) " 
unfolding wprev_d_def prev_d_def schop_d_def
by (simp add: NotNotChopSkip)


lemma BiInductB:
 "\<turnstile> bi(f \<longrightarrow> wprev (f \<and> finite)) \<and> f \<and> finite \<longrightarrow> bi f "
unfolding Valid_def itl_defs
by simp
 (metis biinduct_help nfinite_conv_nlength_enat)

lemma BfInduct:
 "\<turnstile> bf(f \<longrightarrow> wprev f) \<and> f  \<and> finite \<longrightarrow> bf f "
by (auto simp add: Valid_def itl_defs)
 (metis biinduct_help nfinite_conv_nlength_enat)

lemma PrevLoop:
 assumes "\<turnstile> f \<longrightarrow> prev f"
 shows   "\<turnstile> finite \<longrightarrow> \<not> f"
proof -
 have  1: "\<turnstile> f \<longrightarrow> prev f " using assms by auto
 hence 2: "\<turnstile> f \<longrightarrow> ( more \<and> wprev f) " 
   by (metis ChopSkipImpMore Prop05 Prop12 WprevEqvEmptyOrPrev inteq_reflection 
       lift_imp_trans prev_d_def)
 hence 3: "\<turnstile> f \<longrightarrow> wprev f " by auto
 hence 4: "\<turnstile> bi(f \<longrightarrow> wprev f)" by (rule BiGen)
 have  5: "\<turnstile> bi(f \<longrightarrow> wprev f) \<and> f \<and> finite \<longrightarrow> bi f " by (rule BiInduct)
 hence 6: "\<turnstile> bi(f \<longrightarrow> wprev f) \<longrightarrow> (f \<and> finite\<longrightarrow> bi f)" by fastforce
 have  7: "\<turnstile> (f \<and> finite \<longrightarrow> bi f)" using "4" "6" MP by blast
 have  8: "\<turnstile> bi f \<longrightarrow> f" by (rule BiElim)
 have  9: "\<turnstile> finite \<longrightarrow> f  = bi f" using "7" "8" by fastforce
 have 10: "\<turnstile> f \<longrightarrow> more" using "2" by auto
 hence 11: "\<turnstile> bi f \<longrightarrow>  bi more" using BiImpBiRule by blast
 have  12: "\<turnstile> \<not>(bi more)" using DiEmpty bi_d_def empty_d_def by (simp add: bi_d_def empty_d_def)
 from 7 9 11 12 show ?thesis using MP by fastforce 
qed

lemma FinitePrevLoop:
 assumes "\<turnstile> f \<longrightarrow> prev (f \<and> finite)"
 shows   "\<turnstile> finite \<longrightarrow> \<not> f"
by (metis AndChopA PrevLoop assms lift_imp_trans prev_d_def)



lemma PrevImpNotPrevNot:
  "\<turnstile> prev f \<longrightarrow> \<not>(prev (\<not> f)) "
by (metis NotNotChopSkip Prop03 prev_d_def)

lemma BiEqvAndWprevBi:
 "\<turnstile> bi f = (f \<and> wprev(bi f))"
by (metis BiEqvAndEmptyOrBiChopSkip WprevEqvEmptyOrPrev inteq_reflection prev_d_def) 

 

lemma DiIntroB:
 assumes "\<turnstile> (f \<and> \<not> g) \<longrightarrow> prev f"
 shows "\<turnstile> f \<and> finite \<longrightarrow> di g"
proof -
 have   1: "\<turnstile> f \<and> \<not>  g \<longrightarrow> prev f " 
   using assms by auto
 hence  2: "\<turnstile> f \<and> \<not>  g \<and> (bi (\<not> g))\<longrightarrow> (prev f) \<and> (bi (\<not> g))" 
   by auto
 have   3: "\<turnstile> (bi (\<not> g)) \<longrightarrow> \<not>  g" 
   by (rule BiElim)
 hence  4: "\<turnstile> bi (\<not> g) = ((bi (\<not> g)) \<and> \<not>  g)" 
    by fastforce
 have   5: "\<turnstile> f \<and> (bi (\<not> g))\<longrightarrow> prev f \<and> bi (\<not> g) " 
   using "2" "4"  by fastforce
 have   6: "\<turnstile> bi (\<not> g) = ((\<not> g) \<and> wprev(bi (\<not> g)))  "  
    using BiEqvAndWprevBi by blast
 have   7: "\<turnstile> prev f \<and> bi (\<not> g) \<longrightarrow>  prev f \<and> wprev(bi (\<not> g)) " 
   using "6" by auto
 have   8: "\<turnstile> f \<and> (bi (\<not> g))\<longrightarrow> prev f \<and> wprev(bi (\<not> g)) " 
   using "5" "7" using lift_imp_trans by blast
 hence  9: "\<turnstile> f \<and> (bi (\<not> g))\<longrightarrow> more \<and> wprev f \<and> wprev(bi (\<not> g))" 
   using zero_enat_def by (auto simp: Valid_def itl_defs)
     (metis le_zero_eq nlength_eq_enat_nfiniteD)
 hence 10: "\<turnstile> f \<and> (bi (\<not> g))\<longrightarrow>wprev f \<and> wprev(bi (\<not> g)) " 
   by auto
 hence 11: "\<turnstile> f \<and> (bi (\<not> g))\<longrightarrow> wprev (f \<and> bi (\<not> g)) " 
   by (auto simp: Valid_def itl_defs)
 hence 12: "\<turnstile> bi(f \<and> (bi (\<not> g)) \<longrightarrow> wprev (f \<and> bi (\<not> g))) " 
   by (rule BiGen)
 have  13: "\<turnstile> bi(f \<and> (bi (\<not> g)) \<longrightarrow> wprev (f \<and> bi (\<not> g))) \<and> 
              (f  \<and> (bi (\<not> g))) \<and> finite \<longrightarrow> bi(f \<and> (bi (\<not> g))) " 
   using BiInduct by auto
 hence 14: "\<turnstile> bi(f \<and> (bi (\<not> g)) \<longrightarrow> wprev (f \<and> bi(\<not> g))) \<longrightarrow> 
              ((f \<and> (bi(\<not> g))) \<and> finite \<longrightarrow> bi(f \<and> (bi (\<not> g)))) " 
   by fastforce
 have  15: "\<turnstile> ((f \<and> (bi (\<not> g))) \<and> finite \<longrightarrow> bi(f \<and> (bi (\<not> g)))) " 
   using "12" "14" MP by blast
 have  16: "\<turnstile> bi(f \<and> (bi (\<not> g))) \<longrightarrow> (f \<and> (bi (\<not> g))) " 
   by (rule BiElim)
 have  17: "\<turnstile> finite \<longrightarrow> bi(f \<and> (bi (\<not> g))) = (f \<and> (bi (\<not> g)))" 
   using "16" "15" by fastforce
 have  18: "\<turnstile> (f \<and> (bi (\<not> g))) \<longrightarrow> more" 
   using "9" by auto
 hence 19: "\<turnstile> bi(f \<and> (bi (\<not> g))) \<longrightarrow> bi more" 
    by (simp add: BiImpBiRule)
 have 191: "\<turnstile> (\<not> more) = empty" 
    by (simp add: empty_d_def)
 have  20: "\<turnstile> (\<not>(bi more)) "
   unfolding bi_d_def di_d_def 
   by (metis EmptyChop TrueW empty_d_def int_simps(2) int_simps(3) inteq_reflection)
 have  21: "\<turnstile>  finite \<longrightarrow> \<not>(f \<and> (bi (\<not> g)))" 
   using "17" "19" "20" by fastforce
 hence 22: "\<turnstile>  finite \<longrightarrow> \<not> f \<or> \<not> (bi (\<not> g)) " 
   by auto
 have  23: "\<turnstile> (\<not> (bi (\<not> g))) = di g" 
   by (auto simp: bi_d_def)
  show ?thesis using 22 23 by fastforce
qed

lemma DiIntroC:
 assumes "\<turnstile> (f \<and> \<not> g) \<longrightarrow> prev (f \<and> finite)"
 shows "\<turnstile> f \<and> finite \<longrightarrow> di g"
using assms 
by (metis AndChopA DiIntroB lift_imp_trans prev_d_def)


lemma DfIntroB:
 assumes "\<turnstile> (f \<and> \<not> g) \<longrightarrow> prev f"
 shows "\<turnstile> f \<and> finite \<longrightarrow> df g"
proof -
 have 1: "\<turnstile> f \<and> finite \<longrightarrow> di g" 
   by (simp add: DiIntroB assms)
 have 2: "\<turnstile> di g = (df g \<or> (g \<and> inf)) "
   by (simp add: DiEqvDfOrInf) 
 have 3: "\<turnstile> f \<and> finite \<longrightarrow> finite \<and> (df g \<or> (g \<and> inf))"
   by (metis "1" ChopSChopdef FiniteImp df_d_def di_d_def int_eq lift_and_com)  
 have 4: "\<turnstile> finite \<and> (df g \<or> (g \<and> inf)) \<longrightarrow> df g" 
    unfolding finite_d_def by fastforce
 show ?thesis 
 using "3" "4" lift_imp_trans by blast
qed

lemma DfIntroC:
 assumes "\<turnstile> (f \<and> \<not> g) \<longrightarrow> prev (f \<and> finite)"
 shows "\<turnstile> f \<and> finite \<longrightarrow> df g"
using assms 
by (metis AndChopA DfIntroB lift_imp_trans prev_d_def)

lemma DiEqvOrChopMore:
 "\<turnstile> di f = (f \<or> f;more)"
proof -
 have 1: "\<turnstile> di f = f;#True" by (simp add: di_d_def)
 hence 2: "\<turnstile> di f = f; (empty \<or> more) " by (simp add: empty_d_def)
 have  3: "\<turnstile> f; (empty \<or> more) = (f;empty \<or> f;more) " by (simp add: ChopOrEqv)
 have  4: "\<turnstile> f;empty = f " by (rule ChopEmpty)
 from 2 3 4 show ?thesis by fastforce
qed

lemma DfEqvOrSChopMore:
 "\<turnstile> df f = ((f \<and> finite) \<or> f\<frown>more)"
proof -
 have 1: "\<turnstile> df f = f\<frown>#True" by (simp add: df_d_def)
 hence 2: "\<turnstile> df f = f\<frown> (empty \<or> more) " by (simp add: empty_d_def)
 have  3: "\<turnstile> f\<frown> (empty \<or> more) = (f\<frown>empty \<or> f\<frown>more) " by (simp add: SChopOrEqv)
 have  4: "\<turnstile> f\<frown>empty = (f \<and> finite) " by (simp add: ChopEmpty schop_d_def)
 from 2 3 4 show ?thesis by fastforce
qed


lemma DiAndDiEqvDiAndDiOrDiAndDi:
 "\<turnstile> (di f \<and> di g) = (di(f \<and> di g) \<or> di(g \<and> di f)) "
unfolding Valid_def 
apply auto
unfolding itl_defs
apply simp
apply (metis min.absorb1 nle_le)
apply simp
apply (metis)
apply simp
apply (metis min.absorb1)
apply simp
apply (metis min.absorb1)
apply simp
apply (metis)
done


lemma DfAndDfEqvDiAndDfOrDfAndDf:
 "\<turnstile> (df f \<and> df g) = (df(f \<and> df g) \<or> df(g \<and> df f)) "
unfolding Valid_def itl_defs
using linorder_le_cases by fastforce

lemma BoxStateEqvBiFinState:
 "\<turnstile> \<box> (init w) = bi (finite \<longrightarrow>fin (init w))"
proof -
  have 1: "\<turnstile> \<diamond> (\<not> (init w)) = finite ; (\<not>(init w))" 
       by (simp add: sometimes_d_def)
  have 2: "\<turnstile> \<diamond> (init(\<not> w)) = finite ; init (\<not> w)" 
       by (simp add: sometimes_d_def)
  have 3: "\<turnstile> di (finite \<and> fin (init (\<not> w))) = finite ; init (\<not> w)" 
    by (metis DiAndFinEqvChopState NowImpDiamond Prop10 inteq_reflection lift_and_com)
  have 4: "\<turnstile> \<diamond> (init(\<not> w)) = di (finite \<and> fin (init (\<not> w))) " 
       using 1 2 3 by fastforce
  have 5: "\<turnstile> (\<not> (\<diamond> (init(\<not> w)))) = (\<not> (di (finite \<and> fin (init (\<not> w)))))" 
       using 4 by fastforce
  have 51: "\<turnstile> (\<not>(finite \<and> fin (init (\<not> w)))) = (finite \<longrightarrow>\<not> (fin (init (\<not> w))))" 
    by fastforce
  have 6: "\<turnstile> \<box> (init w) = (\<not> (di (finite \<and> fin (init (\<not> w)))))" 
       using 5 always_d_def Initprop(2) by (metis int_eq)
  have 7: "\<turnstile> \<box> (init w) = bi (finite \<longrightarrow>\<not> (fin (init (\<not> w))))" 
       using 6 51 unfolding bi_d_def
       by (metis int_simps(4) inteq_reflection)
  have 8: "\<turnstile> init (\<not> w) = (\<not> (init w))" 
       using Initprop(2) by fastforce
  have 9: "\<turnstile> fin (init (\<not> w)) = fin (\<not> (init w))" 
       using 8 FinEqvFin by blast
  have 10: "\<turnstile> finite \<longrightarrow>fin (init (\<not> w)) = (\<not> (fin (init w)))" 
       using 8 FinNotStateEqvNotFinState FinEqvFin  by fastforce
  have 11: "\<turnstile> finite \<longrightarrow> ((\<not> (fin (init (\<not> w)))) = (fin (init w)))" 
       using 10  by fastforce
  have 111: "\<turnstile> (finite \<longrightarrow> \<not> (fin (init (\<not> w)))) = (finite \<longrightarrow>fin (init w))" 
    using 11 by fastforce
  have 12: "\<turnstile> bi (finite \<longrightarrow> \<not> (fin (init (\<not> w)))) = bi (finite \<longrightarrow>fin (init w))" 
       using 111  by (simp add: BiEqvBi)
  have 13: "\<turnstile> \<box> (init w) = bi (finite \<longrightarrow>fin (init w))" 
       using 7 12 by fastforce
  from 13 show ?thesis by simp
qed

lemma BoxStateEqvBfFinState:
 "\<turnstile> \<box> (init w) = bf (fin (init w))"
proof -
 have 1: "\<turnstile> bi (finite \<longrightarrow>fin (init w)) = 
           (bf (finite \<longrightarrow>fin (init w)) \<and> (inf \<longrightarrow> finite \<longrightarrow>fin (init w))) " 
  by (simp add: BiEqvBfAndfinite)
 have 2: "\<turnstile> (bf (finite \<longrightarrow>fin (init w)) \<and> (inf \<longrightarrow> finite \<longrightarrow>fin (init w))) =
            bf (finite \<longrightarrow>fin (init w))" 
   unfolding finite_d_def by fastforce
 have 3: "\<turnstile> \<box> (init w) = bi (finite \<longrightarrow> fin (init w))"
   by (simp add: BoxStateEqvBiFinState)
 have 4: "\<turnstile> bf (finite \<longrightarrow>fin (init w)) = bf (fin (init w)) " 
    by (simp add: FiniteImpBfEqvRule intI)
 show ?thesis 
 by (metis "1" "2" "3" "4" inteq_reflection)
qed


lemma DiamondStateEqvDiFinState:
 "\<turnstile> \<diamond> (init w) = di (finite \<and> fin (init w))"
proof -
  have 1: "\<turnstile> \<box> (init (\<not> w)) = bi (finite\<longrightarrow>fin (init (\<not> w)))" 
       using BoxStateEqvBiFinState by blast
  have 2: "\<turnstile> (\<not> (\<box> (init (\<not> w)))) = (\<not> (bi (finite \<longrightarrow>fin (init (\<not> w)))))" 
       using 1 by auto
  have 3: "\<turnstile> \<diamond> (\<not> (init (\<not> w))) = di (finite \<and> \<not> (fin (init (\<not> w))))" 
       using 2 
       by (metis (no_types, opaque_lifting) FinChopEqvDiamond FinNotStateEqvNotFinState 
        Initprop(2) di_d_def int_simps(17) inteq_reflection lift_and_com)
  have 4: "\<turnstile> \<diamond> (init  w) = di (finite \<and> \<not> (fin (init (\<not> w))))" 
       by (metis "3" DiEqvNotBiNot DiState Initprop(2) StateEqvBi int_eq)
  have 5: "\<turnstile> \<diamond> (init  w) = di (finite \<and> fin (init w))" using 4 FinNotStateEqvNotFinState 
    by (metis int_simps(4) inteq_reflection lift_and_com)
  from 1 2 3 4 5 show ?thesis by simp
qed

lemma DiamondStateEqvDfFinState:
 "\<turnstile> \<diamond> (init w) = df (fin (init w))"
by (metis DiamondStateEqvDiFinState df_d_def di_d_def inteq_reflection 
    lift_and_com schop_d_def)


lemma OrDiEqvDi:
 "\<turnstile> (f \<or> di f) = di f"
by (simp add: Prop05 Prop11 DiIntro)

lemma OrDfEqvDf:
 "\<turnstile> ((f \<and> finite) \<or> df f) = df f"
by (simp add: AndFiniteImpDf Prop05 Prop11)

lemma AndDiEqv:
 "\<turnstile> (f \<and> di f) = f"
proof -
  have 1: "\<turnstile> f \<longrightarrow> di f" using DiIntro by blast
 from 1 show ?thesis by auto
qed

lemma AndDfEqv:
 "\<turnstile> ((f \<and> finite) \<and> df f) = (f \<and> finite)"
by (meson AndFiniteImpDf Prop10 Prop11)


lemma BiEmptyEqvEmpty:
 "\<turnstile> bi empty = empty"
proof -
 have 1: "\<turnstile> bi empty = (\<not> (di (\<not> empty)))" by (simp add: bi_d_def)
 have 2: "\<turnstile> (\<not> (di (\<not> empty))) = (\<not> ((\<not> empty);#True))" by (simp add: di_d_def)
 have 3: "\<turnstile> (\<not> ((\<not> empty);#True)) = (\<not> (more;#True))" by (simp add: empty_d_def)
 have 4: "\<turnstile> more;#True =  more" using MoreEqvMoreChopTrue by auto
 hence 5: "\<turnstile> (\<not>(more;#True)) =  (\<not> more)" by fastforce
 from 1 2 3 5 show ?thesis by (metis empty_d_def inteq_reflection)
qed

lemma BfEmptyEqvEmpty:
 "\<turnstile> bf empty = empty"
by (metis MoreEqvMoreSChopTrue empty_d_def int_eq int_simps(4) itl_def(15) itl_def(23))

lemma EmptyChopSkipInduct:
 assumes "\<turnstile> empty \<longrightarrow> f"
         "\<turnstile> prev f \<longrightarrow> f"
 shows   "\<turnstile> finite \<longrightarrow> f"
proof -
 have 1: "\<turnstile> empty  \<longrightarrow> f" using assms(1) by auto
 have 2: "\<turnstile> prev f \<longrightarrow> f " using assms(2) by blast
 have 3: "\<turnstile> (empty \<or> prev f) \<longrightarrow> f" using "1" "2"  by fastforce
 have 4: "\<turnstile> wprev f = (empty \<or> prev f)" by (simp add: WprevEqvEmptyOrPrev)
 hence 5: "\<turnstile> wprev f \<longrightarrow> f" using 3 by fastforce 
 hence 6: "\<turnstile> \<not>f \<longrightarrow> \<not> (wprev f) " by fastforce
 hence 7: "\<turnstile> \<not>f \<longrightarrow> prev (\<not> f) " by (simp add: wprev_d_def)
 hence 8: "\<turnstile> finite \<longrightarrow>\<not> \<not> f " by (rule PrevLoop)
 from 8 show ?thesis by auto
qed

lemma PrevOrInfEqv: 
 "\<turnstile> prev (f \<or> inf) = (prev (f \<and> finite) \<or> inf)" 
proof -
 have 1: "\<turnstile> prev (f \<or> inf)  = (f \<or> inf);skip " 
 unfolding prev_d_def by auto
 have 2: "\<turnstile> (f \<or> inf);skip = (f;skip \<or> inf;skip) " 
   by (simp add: OrChopEqv)
 have 3: "\<turnstile> f;skip = ((f \<and> finite);skip \<or> (f \<and> inf);skip)" 
   by (simp add: OrChopEqvRule OrFiniteInf)
 have 4: "\<turnstile> (f \<and> inf);skip = (f \<and> inf)"
   by (simp add: AndInfChopEqvAndInf)
 have 5: "\<turnstile> inf;skip = inf " 
  by (metis AndInfChopEqvAndInf MoreAndInfEqvInf int_eq)
 have 6: "\<turnstile> ((f \<and> inf) \<or> inf) = ( inf) " 
      by fastforce
 have 7: "\<turnstile> (f;skip \<or> inf;skip) = ((f \<and> finite);skip \<or> (f \<and> inf) \<or> inf)" 
   using 3 4 5 6 by fastforce
 have 8 : "\<turnstile> ((f \<and> finite);skip \<or> (f \<and> inf) \<or> inf) =
             ((f \<and> finite);skip \<or>  inf) " 
    using 6 by fastforce
 show ?thesis 
 by (metis "2" "7" "8" inteq_reflection prev_d_def)
qed


lemma EmptySChopSkipInduct:
 assumes "\<turnstile> empty \<longrightarrow> f"
         "\<turnstile> prev (f \<or>  inf) \<longrightarrow> f"
 shows   "\<turnstile> finite \<longrightarrow> f"
proof -
 have 1: "\<turnstile> empty  \<longrightarrow> f" using assms(1) by auto
 have 2: "\<turnstile> prev (f \<or> inf) \<longrightarrow> f " using assms(2) by blast
 have 3: "\<turnstile> (empty \<or> prev (f \<or> inf)) \<longrightarrow> f" using "1" "2"  by fastforce
 have 4: "\<turnstile> wprev (f \<or> inf) = (empty \<or> prev (f \<or> inf))" by (simp add: WprevEqvEmptyOrPrev)
 hence 5: "\<turnstile> wprev (f \<or> inf) \<longrightarrow> f" using 3 by fastforce
 hence 6: "\<turnstile> \<not>f \<longrightarrow> \<not> (wprev (f \<or> inf)) " by fastforce
 hence 7: "\<turnstile> \<not>f \<longrightarrow> prev (\<not> (f \<or> inf)) " by (simp add: wprev_d_def)
 have 8 : "\<turnstile> (\<not> (f \<or> inf)) = (\<not> f \<and> finite) " unfolding finite_d_def by fastforce
 hence 9: "\<turnstile> finite \<longrightarrow>\<not> \<not> f "  using FinitePrevLoop[of "LIFT \<not> f"]  
 by (metis "7" inteq_reflection)
 from 9 show ?thesis by auto
qed

lemma EmptySChopSkipInductB:
 assumes "\<turnstile> empty \<longrightarrow> f"
         "\<turnstile> prev (f \<and> finite) \<longrightarrow> f"
 shows   "\<turnstile> finite \<longrightarrow> f"
by (metis EmptyChopSkipInduct EmptyImpFinite Prop12 SChopImpFinite WPowerstar_ext 
    WPowerstar_skip_finite assms(1) assms(2) int_eq itl_def(14) itl_def(9))



lemma MoreImpImpChopSkipEqv:
 "\<turnstile> more \<longrightarrow> ( (f\<longrightarrow>g);skip = ((f;skip)\<longrightarrow>(g;skip)) )  "
proof -
 have 01: "\<turnstile> (f\<longrightarrow>g) = (\<not> f \<or> g)" by auto
 hence 02: "\<turnstile> (f\<longrightarrow>g);skip = (\<not> f \<or> g);skip " by (simp add: LeftChopEqvChop)
 hence 1: "\<turnstile> (more \<and> (f\<longrightarrow>g);skip) = (more \<and>  (\<not> f \<or> g);skip)" by fastforce
 have 2: "\<turnstile> (\<not> f \<or> g);skip = ((\<not> f);skip \<or> g;skip)" 
      using OrChopEqv by auto
 hence 3: "\<turnstile> (more \<and>  (\<not> f \<or> g);skip) = (more \<and> ((\<not> f);skip \<or> g;skip))" 
      by auto
 have 4: "\<turnstile> (\<not>((\<not> f);skip)) = (empty \<or> (f;skip)) " 
      using NotNotChopSkip by blast
 hence 5: "\<turnstile> ((\<not> f);skip) = (\<not>(empty \<or> (f;skip)))" 
      by fastforce
 have 6: "\<turnstile> \<not>(empty \<or> (f;skip)) = (more \<and> \<not>(f;skip))" 
      using "5" NotChopSkipEqvMoreAndNotChopSkip by fastforce
 have 7: "\<turnstile> ((\<not> f);skip \<or> g;skip) = ( (more \<and> \<not>(f;skip)) \<or> g;skip)" 
      using "5" "6" by fastforce
 hence 8: "\<turnstile> (more \<and>(\<not> f;skip \<or> g;skip)) = (more \<and> ( (more \<and> \<not>(f;skip)) \<or> g;skip)) " 
      by auto
 have 9: "\<turnstile> (more \<and> ( (more \<and> \<not>(f;skip)) \<or> g;skip)) = (more \<and> (\<not>(f;skip) \<or> g;skip))" 
      by auto
 have 10: "\<turnstile> (more \<and> (\<not>(f;skip) \<or> g;skip)) = (more \<and> ((f;skip)\<longrightarrow>(g;skip)))  " 
      by auto
 have 11: "\<turnstile> (more \<and> (f\<longrightarrow>g);skip) = (more \<and> ((f;skip)\<longrightarrow>(g;skip)))" 
      using "1" "2" "3" "8" "9" "10"  "7" by fastforce
 from 11 show ?thesis using MP by fastforce
qed

lemma MoreImpImpSChopSkipEqv:
 "\<turnstile> more \<and> finite \<longrightarrow> ( (f\<longrightarrow>g)\<frown>skip = ((f\<frown>skip)\<longrightarrow>(g\<frown>skip)) )  "
proof -
 have 01: "\<turnstile> (f\<longrightarrow>g) = (\<not> f \<or> g)" by auto
 hence 02: "\<turnstile> (f\<longrightarrow>g)\<frown>skip = (\<not> f \<or> g)\<frown>skip " by (simp add: LeftSChopEqvSChop)
 have 03: "\<turnstile> (f\<longrightarrow>g)\<frown>skip \<longrightarrow> finite" 
   by (metis LenOneEqvSkip SChopImpFinite inteq_reflection len_k_finite)
 have 04: "\<turnstile> (more  \<and> (f\<longrightarrow>g)\<frown>skip) = (more \<and> finite \<and> (f\<longrightarrow>g)\<frown>skip) " 
   using "03" by auto
 have 1: "\<turnstile> (more \<and> finite \<and> (f\<longrightarrow>g)\<frown>skip) = (more \<and>  finite \<and> (\<not> f \<or> g)\<frown>skip)"
      by (metis "01" "04" inteq_reflection)
 have 2: "\<turnstile> (\<not> f \<or> g)\<frown>skip = ((\<not> f)\<frown>skip \<or> g\<frown>skip)" 
      using OrSChopEqv by auto
 hence 3: "\<turnstile> (more \<and>  finite \<and> (\<not> f \<or> g)\<frown>skip) = (more \<and> finite \<and> ((\<not> f)\<frown>skip \<or> g\<frown>skip))" 
      by auto
 have 4: "\<turnstile> (\<not>((\<not> f)\<frown>skip)) = (empty \<or> inf \<or> (f\<frown>skip)) " 
      using NotNotSChopSkip by blast
 hence 5: "\<turnstile> ((\<not> f)\<frown>skip) = (\<not>(empty \<or> inf \<or> (f\<frown>skip)))" 
      by fastforce
 have 6: "\<turnstile> \<not>(empty \<or> inf \<or>(f\<frown>skip)) = (more \<and> finite \<and> \<not>(f\<frown>skip))" 
      using "5" unfolding finite_d_def empty_d_def by fastforce
 have 7: "\<turnstile> ((\<not> f)\<frown>skip \<or> g\<frown>skip) = ( (more \<and> finite \<and> \<not>(f\<frown>skip)) \<or> g\<frown>skip)" 
      using "5" "6" by fastforce
 hence 8: "\<turnstile> (more \<and> finite \<and> (\<not> (f\<frown>skip) \<or> g\<frown>skip)) = 
             (more \<and> finite \<and> ( (more \<and> finite \<and> \<not>(f\<frown>skip)) \<or> g\<frown>skip)) " 
    by fastforce      
 have 9: "\<turnstile> (more \<and> finite \<and> ( (more \<and> finite \<and> \<not>(f\<frown>skip)) \<or> g\<frown>skip)) = 
            (more \<and> finite \<and> (\<not>(f\<frown>skip) \<or> g\<frown>skip))" 
      by auto
 have 10: "\<turnstile> (more \<and> finite \<and> (\<not>(f\<frown>skip) \<or> g\<frown>skip)) = 
             (more \<and> finite \<and> ((f\<frown>skip)\<longrightarrow>(g\<frown>skip)))  " 
      by auto
 have 11: "\<turnstile> (more \<and> finite \<and> (f\<longrightarrow>g)\<frown>skip) = (more \<and> finite \<and> ((f\<frown>skip)\<longrightarrow>(g\<frown>skip)))" 
      using "1" "2" "3" "8" "9" "10"  "7" 
      by (metis inteq_reflection)   
 from 11 show ?thesis using MP by fastforce
qed



lemma MoreImpImpPrevEqv:
 "\<turnstile> more \<longrightarrow> ( prev(f\<longrightarrow>g) = (prev f\<longrightarrow> prev g) )"
by (simp add: MoreImpImpChopSkipEqv prev_d_def)

lemma BiBoxNotEqvNotFiniteChopChopTrue:
 "\<turnstile> bi(\<box> (\<not> f)) = (\<not>((finite;f);#True )) "
by (simp add: bi_d_def always_d_def di_d_def sometimes_d_def)

lemma BfBoxNotEqvNotTrueSChopSChopTrue:
 "\<turnstile> bf(\<box> (\<not> f)) = (\<not>((#True\<frown>f)\<frown>#True )) "
by (simp add: bf_d_def always_d_def df_d_def sometimes_d_def schop_d_def)

(* changed proof *)
lemma DiAndEmptyEqvAndEmpty:
 "\<turnstile> (di f \<and> empty) = (f  \<and> empty)"
by (metis ChopEmptyAndEmpty di_d_def int_simps(17) inteq_reflection)


lemma DfAndEmptyEqvAndEmpty:
 "\<turnstile> (df f \<and> empty) = (f  \<and> empty)"
by (metis ChopEmptyAndEmpty DiAndEmptyEqvAndEmpty Prop04 SChopEmptyAndEmpty df_d_def di_d_def)


lemma FiniteImpBfEqvBi: 
"\<turnstile> finite \<longrightarrow> bf f = bi f" 
proof -
 have 1: "\<turnstile> bi f = (bf f \<and> (inf \<longrightarrow> f)) " 
  by (simp add: BiEqvBfAndfinite)
 have 2: "\<turnstile> finite \<longrightarrow> (bf f \<and> (inf \<longrightarrow> f)) = bf f"
   unfolding finite_d_def by fastforce
 show ?thesis 
 using "1" "2" by fastforce 
qed

lemma FiniteImpDfEqvDi: 
"\<turnstile> finite \<longrightarrow> df f = di f" 
proof -
 have 1: "\<turnstile> di f = (df f \<or> (f \<and> inf)) " 
   by (simp add: DiEqvDfOrInf)
 have 2: "\<turnstile> finite \<longrightarrow> (df f \<or> (f \<and> inf)) = df f"
    unfolding finite_d_def by fastforce
 show ?thesis using 1 2 by fastforce
qed 

subsubsection \<open>Strict initial intervals\<close>


lemma BsEqvEmptyOrBiSChopSkip: 
 "\<turnstile> bs f = (empty \<or> ((bi f)\<frown>skip))"
unfolding bs_d_def by simp

lemma DsMoreDi:
 "\<turnstile> ds f = (more \<and> (finite \<longrightarrow> (di f)\<frown>skip))"
proof -
 have 1: "\<turnstile> ds f = (\<not>(bs (\<not> f))) " 
      by (simp add: ds_d_def)
 have 2: "\<turnstile> (\<not>(bs (\<not> f))) = (\<not>(empty \<or> (bi (\<not> f))\<frown>skip)) " 
      by (simp add: bs_d_def)
 have 3: "\<turnstile> (\<not>(empty \<or> (bi (\<not> f))\<frown>skip)) = (\<not>empty \<and> \<not>((bi (\<not>f))\<frown>skip)) " 
      by auto
 have 4: "\<turnstile> (\<not>empty \<and> \<not>((bi (\<not>f))\<frown>skip)) = (more \<and> \<not>((bi (\<not>f))\<frown>skip)) " 
     unfolding empty_d_def by auto
 have 40: "\<turnstile> (bi (\<not>f)) = (\<not>(di f)) " 
   by (meson NotDiEqvBiNot Prop11)
 have 41: "\<turnstile> (\<not>((bi (\<not>f))\<frown>skip)) = (empty \<or> inf \<or> ((di f)\<frown>skip))"
   using 40 NotNotSChopSkip[of "LIFT (di f)"] by (metis inteq_reflection)
 have 5: "\<turnstile> (more \<and> \<not>((bi (\<not>f))\<frown>skip)) = (more \<and> (empty \<or> inf \<or> (di f)\<frown>skip)) " 
    using "41" by auto
 have 6: "\<turnstile> (more \<and> (empty \<or> inf \<or> (di f)\<frown>skip)) = 
            (more \<and> (inf \<or> (di f)\<frown>skip))" 
     unfolding empty_d_def by fastforce
 show ?thesis unfolding finite_d_def using 1 2 3 4 5 
 by fastforce  
qed

lemma DsDi:
 "\<turnstile> ds f = (finite \<longrightarrow> (di f)\<frown>skip)"
proof -
 have 1: "\<turnstile> ds f = (more \<and> (finite  \<longrightarrow> (di f)\<frown>skip))" by (rule DsMoreDi)
 have 2: "\<turnstile> (di f)\<frown>skip \<longrightarrow> more " by (simp add: SChopSkipImpMore)
 hence 3: "\<turnstile> (more \<and> (finite \<longrightarrow> (di f)\<frown>skip)) = 
    ( (more \<and> inf) \<or> (more \<and> (di f)\<frown>skip))" 
    unfolding finite_d_def by fastforce
 have 4: "\<turnstile> ( (more \<and> inf) \<or> (more \<and> (di f)\<frown>skip)) =
            ( inf \<or>  (di f)\<frown>skip)" 
   by (metis "2" "3" MoreAndInfEqvInf Prop10 inteq_reflection lift_and_com)
 from 1 2 4 show ?thesis unfolding finite_d_def 
 by fastforce
qed

lemma DsDf:
 "\<turnstile> ds f = (finite \<longrightarrow> (df f)\<frown>skip)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow> (di f) = (df f) " 
  using FiniteImpDfEqvDi by fastforce
 have 2: "\<turnstile> ds f = (finite \<longrightarrow> (di f)\<frown>skip) "
   by (simp add: DsDi)
 have 3: "\<turnstile> (finite \<longrightarrow> (di f)\<frown>skip) = (finite \<longrightarrow> (df f)\<frown>skip) "
   by (metis "1" "2" FiniteImpAnd inteq_reflection schop_d_def)
 show ?thesis
 by (metis "2" "3" inteq_reflection)
qed

lemma BsEqvNotDsNot:
 "\<turnstile> bs f = (\<not>(ds (\<not> f)))"
proof -
 have 1: "\<turnstile> ds (\<not> f)  = (more \<and> (finite \<longrightarrow> di (\<not> f)\<frown>skip))" 
   using  DsMoreDi[of "LIFT \<not> f"]   by blast
 hence 2: "\<turnstile> (\<not>(ds (\<not> f)))  = (\<not>(more \<and> (finite \<longrightarrow> di (\<not> f)\<frown>skip)))" 
      by auto
 have 3: "\<turnstile> (\<not>(more \<and> (finite \<longrightarrow> di (\<not> f)\<frown>skip))) = 
            (empty \<or> \<not>(finite \<longrightarrow> (di (\<not> f))\<frown>skip))" 
   unfolding empty_d_def by fastforce
 have 31: "\<turnstile> di (\<not> f) = (\<not>(bi f)) " 
    by (simp add: DiNotEqvNotBi)
 have 4: "\<turnstile> (empty \<or> \<not>(finite \<longrightarrow> (di (\<not> f))\<frown>skip)) = 
            (empty \<or> ( finite \<and>  \<not>((di (\<not> f))\<frown>skip)))" 
     by fastforce
 have 41: "\<turnstile> (\<not>((di (\<not> f))\<frown>skip)) = (\<not>((\<not>(bi f))\<frown> skip))"
     by (metis "31" int_simps(15) inteq_reflection) 
 have 5: "\<turnstile>  (\<not>((\<not>(bi f))\<frown>skip)) = (empty \<or> inf \<or> (bi f)\<frown>skip) " 
      by (rule NotNotSChopSkip)
 have 51: "\<turnstile>  (empty \<or> ( finite \<and>  \<not>((di (\<not> f))\<frown>skip))) =
              (empty \<or> ( finite \<and> (empty \<or> inf \<or> (bi f)\<frown>skip) )) "
    by (metis "31" "4" "5" inteq_reflection) 
 have 52: "\<turnstile> ( finite \<and> (empty \<or> inf \<or> (bi f)\<frown>skip) ) =
             ( finite \<and> (empty \<or> (bi f)\<frown>skip) )" 
    unfolding finite_d_def by fastforce
 have 53: "\<turnstile> ( finite \<and> (empty \<or> (bi f)\<frown>skip) ) =
             ( empty \<or> (finite \<and> (bi f)\<frown>skip) ) " 
    by (metis (no_types, opaque_lifting) ChopEmpty EmptyOrSChopEqv EmptySChop 
        FiniteImportSChopRight int_eq itl_def(9))
 have 54: "\<turnstile> (finite \<and> (bi f)\<frown>skip) = (bi f)\<frown>skip" 
   by (meson Prop10 Prop11 SChopImpFinite WPowerstar_ext WPowerstar_skip_finite lift_and_com lift_imp_trans)
 have 55: "\<turnstile> (empty \<or> ( finite \<and> (empty \<or> inf \<or> (bi f)\<frown>skip) )) =
             ( empty \<or> (bi f)\<frown>skip) " 
     using "52" "54" by fastforce
 have 6: "\<turnstile> (empty \<or> ( finite \<and>  \<not>((di (\<not> f))\<frown>skip))) =
            ( empty \<or> (bi f)\<frown>skip) " 
    by (metis "51" "55" inteq_reflection)    
 from 2 3 4 6 show ?thesis  
 by (simp add: ds_d_def)
qed

lemma NotBsEqvDsNot:
 "\<turnstile> (\<not>(bs f)) = ds (\<not> f)"
proof -
 have 1: "\<turnstile> bs f = (\<not>(ds (\<not> f)))" by (rule BsEqvNotDsNot)
 hence 2: "\<turnstile> (\<not>( bs f)) = (\<not>\<not>(ds (\<not> f))) " by auto
 from 2 show ?thesis by auto
qed

lemma NotDsEqvBsNot:
 "\<turnstile> (\<not> (ds f)) = bs (\<not> f)"
proof -
 have 1: "\<turnstile> (\<not>(ds  f)) = (\<not>\<not>(bs (\<not>f)))" by (simp add: ds_d_def)
 from 1 show ?thesis by auto
qed

lemma NotDsAndEmpty:
 "\<turnstile> \<not>(ds f \<and> empty)"
proof -
 have 1: "\<turnstile> ds f = (more \<and> (finite \<longrightarrow>(di f)\<frown>skip))" by (rule DsMoreDi)
 have 2: "\<turnstile> (more \<and> (finite \<longrightarrow>(di f)\<frown>skip)) \<and> empty \<longrightarrow> #False" 
  unfolding empty_d_def by fastforce
 from 1 2 show ?thesis by fastforce
qed

lemma BsMoreEqvEmpty:
 "\<turnstile> bs more = empty"
proof -
 have 1: "\<turnstile> bs more = (empty \<or> (bi more)\<frown>skip)"
 by (simp add: bs_d_def)
 have 2: "\<turnstile> bi more \<longrightarrow> #False"
   by (metis EmptyChop TrueW bi_d_def di_d_def empty_d_def int_simps(14) int_simps(4) 
       inteq_reflection)
 hence 3: "\<turnstile> (bi more)\<frown>skip \<longrightarrow> #False\<frown>skip " 
   using LeftSChopImpSChop by blast
 have 31: "\<turnstile> #False\<frown>skip \<longrightarrow> #False "
 by (simp add: Valid_def skip_defs schop_defs)
 have 32: "\<turnstile> (bi more)\<frown>skip \<longrightarrow> #False " using "3" "31" by fastforce 
 hence 4: "\<turnstile> (empty \<or> ((bi more)\<frown>skip)) = empty"  by fastforce
 from 1 4 show ?thesis by fastforce
qed

lemma  BsAndEqv:
 "\<turnstile> (bs f \<and> bs g) = bs(f \<and> g)"
proof -
 have 1: "\<turnstile> bs f = (empty \<or> (bi f) \<frown>skip)" 
      by (simp add: bs_d_def)
 have 2: "\<turnstile> bs g = (empty \<or> (bi g) \<frown>skip)" 
      by (simp add: bs_d_def)
 have 3: "\<turnstile> (bs f \<and> bs g) = ((empty \<or> (bi f) \<frown>skip) \<and> (empty \<or> (bi g) \<frown>skip)) " 
      using "1" "2" by fastforce
 have 4: "\<turnstile> ((empty \<or> (bi f) \<frown>skip) \<and> (empty \<or> (bi g) \<frown>skip)) = 
             (empty \<or> ((bi f) \<frown>skip \<and> (bi g) \<frown>skip)) " 
      by auto
 have 5: "\<turnstile> (((bi f) \<frown>skip \<and> (bi g) \<frown>skip)) = bi(f \<and> g)\<frown>skip " 
   by (metis BiAndEqv SChopSkipAndSChopSkip inteq_reflection)
 hence 6: "\<turnstile> (empty \<or> ((bi f) \<frown>skip \<and> (bi g) \<frown>skip)) = (empty \<or> bi(f \<and> g)\<frown>skip) " 
      by auto
 from 3 4 6 show ?thesis by (metis bs_d_def inteq_reflection)
qed


lemma BsEqvRule:
 assumes "\<turnstile> f = g"
 shows   "\<turnstile> bs f = bs g"
proof -
 have 1: "\<turnstile> f = g"  using assms by auto
 hence 2: "\<turnstile> bi(f) = bi(g)" by (simp add: BiEqvBi)
 hence 3: "\<turnstile> bi(f)\<frown>skip  = bi(g)\<frown>skip" by (simp add: LeftSChopEqvSChop)
 hence 4: "\<turnstile> (empty \<or> bi(f)\<frown>skip)  = (empty \<or> bi(g)\<frown>skip)" by auto
 hence 5: "\<turnstile> bs(f) = bs(g)" by (simp add: bs_d_def)
 from 1 2 3 4 5 show ?thesis by auto
qed

lemma DsEqvRule:
 assumes "\<turnstile> f = g"
 shows "\<turnstile> ds f = ds g"
using assms using int_eq by force

lemma DsOrEqv:
 "\<turnstile> (ds f \<or> ds g) = ds (f \<or> g)"
proof -
 have 1: "\<turnstile> ds f = (\<not>(bs (\<not> f)))" by (simp add: ds_d_def)
 have 2: "\<turnstile> ds g = (\<not>(bs (\<not> g)))" by (simp add: ds_d_def)
 have 3: "\<turnstile> (ds f \<or> ds g) = (\<not>(bs (\<not> f)) \<or> \<not>(bs (\<not> g)))" using "1" "2" by fastforce
 have 4: "\<turnstile> (\<not>(bs (\<not> f)) \<or> \<not>(bs (\<not> g))) = (\<not>(bs (\<not> f) \<and> bs (\<not> g)))" by auto
 have 5: "\<turnstile> (bs (\<not> f) \<and> bs (\<not> g)) = bs( \<not> f \<and> \<not>g)" by (rule BsAndEqv)
 hence 6: "\<turnstile> (\<not>(bs (\<not> f) \<and> bs (\<not> g))) = (\<not>(bs( \<not> f \<and> \<not>g))) " by auto
 have 7: "\<turnstile> (\<not>(bs( \<not> f \<and> \<not>g))) = ds (\<not>(\<not> f \<and> \<not>g)) " by (rule NotBsEqvDsNot)
 have 8: "\<turnstile> (\<not>(\<not> f \<and> \<not>g)) = (f \<or> g)" by auto
 hence 9: "\<turnstile> ds(\<not>(\<not> f \<and> \<not>g)) = ds (f \<or> g) " by (rule DsEqvRule)
 from 3 4 6 7 9 show ?thesis by fastforce
qed

lemma BsOrImp:
 "\<turnstile> bs f \<or> bs g \<longrightarrow> bs(f \<or> g)"
proof -
 have 1: "\<turnstile> bi f \<or> bi g \<longrightarrow> bi(f \<or> g)" 
      by (rule BiOrBiImpBiOr)
 hence 2: "\<turnstile> (bi f \<or> bi g)\<frown>skip \<longrightarrow> (bi(f \<or> g))\<frown>skip" 
      by (rule LeftSChopImpSChop)
 have 3: "\<turnstile> (bi f)\<frown>skip \<or> (bi g)\<frown>skip \<longrightarrow> (bi(f \<or> g))\<frown>skip" 
      using "1" OrSChopEqv "2" by fastforce
 hence 4: "\<turnstile> empty \<or> (bi f)\<frown>skip \<or> (bi g)\<frown>skip \<longrightarrow>  empty \<or> (bi(f \<or> g))\<frown>skip" 
      by auto
 hence 5: "\<turnstile> (empty \<or> (bi f)\<frown>skip) \<or> (empty \<or> (bi g)\<frown>skip) \<longrightarrow>  empty \<or> (bi(f \<or> g))\<frown>skip " 
      by auto
 from 5 show ?thesis by (simp add: bs_d_def)
qed

lemma DsAndImp:
 "\<turnstile> ds (f \<and> g) \<longrightarrow> ds f \<and> ds g "
proof -
 have 1: "\<turnstile> bs (\<not>f) \<or> bs (\<not>g) \<longrightarrow> bs(\<not>f \<or> \<not>g)" by (rule BsOrImp)
 have 2: "\<turnstile> (\<not>f \<or> \<not>g) = (\<not>(f \<and> g))" by auto
 hence 3: "\<turnstile> bs(\<not>f \<or> \<not>g) = bs (\<not>(f \<and> g))" by (rule BsEqvRule)
 have 4: "\<turnstile> bs (\<not>f) \<or> bs (\<not>g) \<longrightarrow> bs (\<not>(f \<and> g))" using "1" "3" by fastforce
 have 5: "\<turnstile> bs (\<not>f) = (\<not>(ds f))" using NotDsEqvBsNot by fastforce
 have 6: "\<turnstile> bs (\<not>g) = (\<not>(ds g))" using NotDsEqvBsNot by fastforce
 have 7: "\<turnstile> bs (\<not>(f \<and> g)) =  (\<not>(ds (f \<and> g)))" using NotDsEqvBsNot by fastforce
 have 8: "\<turnstile>  \<not>(ds f) \<or>  \<not>(ds g) \<longrightarrow>  \<not>(ds (f \<and> g))" using "4" "5" "6" "7" by fastforce
 hence 9: "\<turnstile> \<not>(ds f \<and> ds g)  \<longrightarrow>  \<not>(ds (f \<and> g))" by auto
 from 9 show ?thesis by auto
qed

lemma DsAndImpElimL:
 "\<turnstile> ds (f \<and> g) \<longrightarrow> ds f "
using DsAndImp by fastforce

lemma DsAndImpElimR:
 "\<turnstile> ds (f \<and> g) \<longrightarrow> ds g "
using DsAndImp by fastforce

lemma BiImpBs:
 "\<turnstile> bi f \<and> finite \<longrightarrow> bs f  "
proof -
 have 1: "\<turnstile> empty \<longrightarrow> empty \<or> (bi f)\<frown>skip" by auto
 hence 10: "\<turnstile> empty \<and> bi f \<longrightarrow> empty \<or> (bi f)\<frown>skip" by auto
 have 11: "\<turnstile>  more \<and> bi f \<longrightarrow> (f \<and> inf) \<or> (more \<and> bi f \<and> finite)" 
    using BiElim[of f] OrFiniteInf[of "LIFT bi f"] by fastforce 
 have 12: "\<turnstile> more \<and> bi f \<and> finite \<longrightarrow> (bi f)\<frown>skip "
    using MoreAndBiImpBiSChopSkip by fastforce
 have 2: "\<turnstile> more \<and> bi f \<longrightarrow> inf \<or> (bi f)\<frown>skip" 
    using 12 unfolding finite_d_def by fastforce
 have 3: "\<turnstile> more \<and> bi f \<and> finite \<longrightarrow> empty \<or>  (bi f)\<frown>skip " 
   by (metis "12"   Prop05)
 have 4: "\<turnstile> (bi f  \<and> finite)= ((bi f \<and> empty \<and> finite) \<or> (bi f \<and> more \<and> finite))"
   by  (auto simp add: empty_d_def) 
 have 41: "\<turnstile> (bi f \<and> empty \<and> finite) = (bi f \<and> empty )" 
   using FiniteAndEmptyEqvEmpty by fastforce
 have 5: "\<turnstile> (empty \<or> (bi f)\<frown>skip) = bs f " by (simp add: bs_d_def)
 have 6: "\<turnstile> (bi f \<and> finite) = (bf f \<and> finite) " 
    by (meson FiniteImpAnd FiniteImpBfEqvBi Prop11)
 from 2 3 4 5 show ?thesis by fastforce
qed


lemma BfImpBs:
 "\<turnstile> bf f \<and> finite \<longrightarrow> bs f  "
proof -
 have 1: "\<turnstile> empty \<longrightarrow> empty \<or> (bi f)\<frown>skip" by auto
 hence 2: "\<turnstile> empty \<and> bi f \<longrightarrow> empty \<or> (bi f)\<frown>skip" by auto
 have 20: "\<turnstile> more \<and> bi f \<longrightarrow> inf \<or> (bi f)\<frown>skip"
  using MoreAndBiImpBiSChopSkip unfolding finite_d_def by fastforce
 have 21: "\<turnstile> more \<and> bi f \<and> finite \<longrightarrow> (bi f)\<frown>skip "
   unfolding finite_d_def using 20 by fastforce
 have 3: "\<turnstile> more \<and> bi f \<and> finite \<longrightarrow> empty \<or>  (bi f)\<frown>skip " using 21 by auto
 have 4: "\<turnstile> (bi f \<and> finite) = ((bi f \<and> empty \<and> finite) \<or> (bi f \<and> more \<and> finite))" 
 by  (auto simp add: empty_d_def) 
 have 41: "\<turnstile> (bi f \<and> empty \<and> finite) = (bi f \<and> empty )" 
   using FiniteAndEmptyEqvEmpty by fastforce
 have 5: "\<turnstile> (empty \<or> (bi f)\<frown>skip) = bs f " by (simp add: bs_d_def)
 from 2 3 4 5 show ?thesis 
 by (meson BiImpBs FiniteImpAnd FiniteImpBfEqvBi Prop11 lift_imp_trans)
qed

lemma BiFiniteEqvFinite: 
 "\<turnstile> bi finite = finite " 
by (auto simp add: Valid_def itl_defs)


lemma BsImpBsBs:
 "\<turnstile> bs f \<longrightarrow> bs ( bs f)"
proof -
 have 1: "\<turnstile> bi f \<and> finite \<longrightarrow> bs f"
  using BiImpBs by auto
 hence 2: "\<turnstile> bi (bi f \<and> finite) \<longrightarrow> bi(bs f)" by (rule BiImpBiRule)
 have 21: "\<turnstile> bi (bi f \<and> finite) = (bi f \<and> finite) " 
   by (metis BiAndEqv BiEqvBiBi BiFiniteEqvFinite inteq_reflection)
 have 3: "\<turnstile> (bi f) \<and> finite \<longrightarrow> bi(bs f)" using "2" "21" by fastforce
 have 4: "\<turnstile> (bi f)\<frown>skip \<longrightarrow> (bi(bs f))\<frown>skip"
   by (metis (no_types, opaque_lifting) "3" LeftChopImpChop Prop12 SChopstar_ext 
       SChopstar_finite lift_imp_trans schop_d_def)
 have 5: "\<turnstile> empty \<or> (bi f)\<frown>skip \<longrightarrow> empty \<or> (bi(bs f))\<frown>skip" 
   using "4" by fastforce
 have 6: "\<turnstile> (bi(bs f))\<frown>skip = (bf(bs f))\<frown>skip " 
    by (meson BfEqvImpSChopEqvSChop FiniteBfGen FiniteImpBfEqvBi MP Prop11)
 from 5 6 show ?thesis 
 by (metis BsEqvEmptyOrBiSChopSkip inteq_reflection)
qed

lemma DsImpDi:
 "\<turnstile> ds f \<longrightarrow> inf \<or> di f"
proof -
 have 1: "\<turnstile> bi (\<not> f) \<and> finite \<longrightarrow> bs (\<not> f)" by (rule BiImpBs)
 hence 2: "\<turnstile> \<not>(bs (\<not> f)) \<longrightarrow> inf \<or> \<not>(bi (\<not> f)) " unfolding finite_d_def by fastforce
 from 2 show ?thesis
 using NotBsEqvDsNot DiEqvNotBiNot by fastforce
qed

lemma DsImpDf:
 "\<turnstile> ds f \<longrightarrow> inf \<or> df f"
proof -
 have 1: "\<turnstile> bf (\<not> f) \<and> finite \<longrightarrow> bs (\<not> f)" by (rule BfImpBs)
 hence 2: "\<turnstile> \<not>(bs (\<not> f)) \<longrightarrow> inf \<or>\<not>(bf (\<not> f)) " unfolding finite_d_def by fastforce
 from 2 show ?thesis by (simp add: bf_d_def ds_d_def)
qed

lemma BsImpBsRule:
 assumes "\<turnstile> f \<longrightarrow> g"
 shows "\<turnstile> bs f  \<longrightarrow> bs g"
proof -
 have 1: "\<turnstile> f \<longrightarrow> g" using assms by auto
 hence 2: "\<turnstile> bi f \<longrightarrow> bi g " by (rule BiImpBiRule)
 hence 3: "\<turnstile> (bi f)\<frown>skip \<longrightarrow> (bi g)\<frown>skip " by (rule LeftSChopImpSChop)
 hence 4: "\<turnstile> empty \<or> (bi f)\<frown>skip \<longrightarrow> empty \<or> (bi g)\<frown>skip" by auto
 from 4 show ?thesis by (simp add: bs_d_def)
qed
 
lemma DsChopImpDsB:
 "\<turnstile> ds (f;g) \<longrightarrow> ds f"
proof -
 have 1: "\<turnstile> di(f;g) \<longrightarrow> di f" by (rule DiChopImpDiB)
 hence 2: "\<turnstile> (di(f;g))\<frown>skip \<longrightarrow> (di f)\<frown>skip " by (rule LeftSChopImpSChop)
 from 2 show ?thesis using DsDi 
 by (metis ChopImpDi DiEqvDiDi DsAndImpElimR Prop10 inteq_reflection)
qed

lemma DsSChopImpDsB:
 "\<turnstile> ds (f\<frown>g) \<longrightarrow> ds f"
by (metis DsAndImpElimL DsChopImpDsB lift_imp_trans schop_d_def)

lemma NotBsImpBsNotChop:
 "\<turnstile> bs (\<not> f) \<longrightarrow> bs ( \<not>(f;g))"
proof -
 have 1: "\<turnstile> ds (f;g) \<longrightarrow> ds f" by (rule DsChopImpDsB)
 hence 2: "\<turnstile> \<not>(ds f) \<longrightarrow> \<not>(ds (f;g))" by fastforce
 from 2 show ?thesis using NotDsEqvBsNot by fastforce
qed

(* proof has changed notice the crucial lemma BiBiOrEqvBi that is needed!*)
lemma BsOrBsEqvBsBiOrBi:
 "\<turnstile> (bs f \<or> bs g) = bs(bi f \<or> bi g) "
proof -
 have 1: "\<turnstile> (bs f \<or> bs g) = ((empty \<or> (bi f)\<frown>skip) \<or> (empty \<or> (bi g)\<frown>skip))" 
      by (simp add: bs_d_def)
 have 2: "\<turnstile> ((empty \<or> (bi f)\<frown>skip) \<or> (empty \<or> (bi g)\<frown>skip)) = (empty \<or> (bi f)\<frown>skip \<or> (bi g)\<frown>skip)" 
      by auto
 have 3: "\<turnstile> ((bi f)\<frown>skip \<or> (bi g)\<frown>skip) = (bi f \<or> bi g)\<frown>skip" 
    by (meson OrSChopEqv Prop11)
 hence 4: "\<turnstile> (empty \<or> (bi f)\<frown>skip \<or> (bi g)\<frown>skip) = (empty \<or> (bi f \<or> bi g)\<frown>skip)" 
      by auto
 have 5: "\<turnstile> (bi f \<or> bi g) = bi ( bi f \<or> bi g)  " 
     by (meson BiBiOrEqvBi Prop11)
 hence 6: "\<turnstile> (bi f \<or> bi g)\<frown>skip = bi ( bi f \<or> bi g)\<frown>skip  " 
   by (simp add: LeftSChopEqvSChop)
 hence 7: "\<turnstile> (empty \<or> bi ( bi f \<or> bi g)\<frown>skip) = (empty \<or> (bi f \<or> bi g)\<frown>skip)" 
      by auto
 have 8: "\<turnstile> (empty \<or> (bi f \<or> bi g)\<frown>skip) = bs(bi f \<or> bi g)"  
   by (metis BiBiOrEqvBi BsEqvEmptyOrBiSChopSkip inteq_reflection)
 from 1 2 4 8  show ?thesis by (metis inteq_reflection) 
qed

lemma BsOrBsEqvBsBfOrBf:
 "\<turnstile> (bs f \<or> bs g) = bs(bf f \<or> bf g) "
proof -
 have 1: "\<turnstile> (bs f \<or> bs g) = bs(bi f \<or> bi g) " 
   by (simp add: BsOrBsEqvBsBiOrBi)
 have 2: "\<turnstile> (bi f \<or> bi g)\<frown>skip = (bf f \<or> bf g)\<frown>skip " 
   by (metis FiniteImpAnd FiniteImpBfEqvBi OrSChopEqv inteq_reflection schop_d_def)
 from 1 2 show ?thesis unfolding bs_d_def 
 by (metis BfBfOrEqvBf BiBiOrEqvBi FiniteImpAnd FiniteImpBfEqvBi inteq_reflection schop_d_def)
qed



lemma DiOrDsEqvDi:
 "\<turnstile> di f \<or> (ds f \<and> finite) = di f"
proof -
 have 1: "\<turnstile> di f \<longrightarrow> di f \<or> ds f" by auto
 have 2: "\<turnstile> di f \<longrightarrow> di f" by auto
 have 3: "\<turnstile> ds f \<longrightarrow> inf \<or> di f" by (rule DsImpDi)
 have 4: "\<turnstile> di f \<or> ds f \<longrightarrow> inf \<or> di f" using "2" "3" by auto
 from 1 4 show ?thesis unfolding finite_d_def by fastforce
qed

lemma DiAndDsEqvDs:
 "\<turnstile> (di f \<and> (ds f \<and> finite)) = (ds f \<and> finite)"
proof -
 have 1: "\<turnstile> di f \<and> ds f \<longrightarrow> ds f" by auto
 have 2: "\<turnstile> ds f \<longrightarrow> ds f " by auto
 have 3: "\<turnstile> ds f \<longrightarrow> inf \<or> di f" by (rule DsImpDi)
 have 4: "\<turnstile> (ds f \<and> finite)\<longrightarrow> di f \<and> (ds f \<and> finite)" 
  using "2" "3" unfolding finite_d_def by auto
 from 1 4 show ?thesis by auto
qed

lemma DiFiniteOrInf: 
 "\<turnstile> di f = ((f \<and> finite) \<or> (di f)\<frown>skip \<or> (di f \<and> inf) )"
proof -
 have 1: "\<turnstile> (di f) = ((di f \<and> finite) \<or> (di f \<and> inf)) " 
   by (simp add: OrFiniteInf)
 have 2: "\<turnstile> (di f \<and> finite) = (di f)\<frown>finite " 
   by (metis ChopTrueAndFiniteEqvAndFiniteChopFinite DiEqvDiDi di_d_def inteq_reflection schop_d_def)
 have 3: "\<turnstile> (more \<and> finite ) = (#True\<frown>skip) " 
   by (metis DiamondSChopdef FmoreEqvSkipChopFinite SkipFiniteEqvFiniteSkip fmore_d_def 
       inteq_reflection sometimes_d_def)
 have 4: "\<turnstile> finite = (#True\<frown>skip \<or> empty) " 
  using 3 unfolding empty_d_def 
  by (metis DiamondEmptyEqvFinite FmoreEqvSkipChopFinite WPowerstarChopEqvChopOrRule 
      WPowerstar_skip_finite empty_d_def fmore_d_def int_eq sometimes_d_def)
 have 5: "\<turnstile> (di f)\<frown>finite = (di f)\<frown>(#True\<frown>skip \<or> empty) " 
   by (simp add: "4" RightSChopEqvSChop)
 have 6: "\<turnstile> (di f)\<frown>(#True\<frown>skip) = (di f)\<frown>skip " 
   unfolding di_d_def 
   by (metis "2" "3" ChopAssoc FmoreEqvSkipChopFinite SkipFiniteEqvFiniteSkip di_d_def fmore_d_def 
       inteq_reflection schop_d_def)
 have 7: "\<turnstile> (di f) \<frown> finite = (f \<frown> finite)\<frown>finite " 
   by (metis (no_types, lifting) "4" ChopTrueAndFiniteEqvAndFiniteChopFinite Prop10 Prop11
       SChopImpFinite di_d_def inteq_reflection schop_d_def)    
 have 8: "\<turnstile> (f \<frown> finite)\<frown>finite = (f \<frown> finite) " 
   by (metis "2" "7" ChopTrueAndFiniteEqvAndFiniteChopFinite Prop04 di_d_def schop_d_def)
 have 9: "\<turnstile> (f \<frown> finite) = f \<frown> (#True\<frown>skip \<or> empty) "
   by (simp add: "4" RightSChopEqvSChop)
 have 10: "\<turnstile> f \<frown> (#True\<frown>skip) =  (di f)\<frown>skip " 
   by (metis (no_types, opaque_lifting) "3" ChopAssoc ChopTrueAndFiniteEqvAndFiniteChopFinite
       FmoreEqvSkipChopFinite SkipFiniteEqvFiniteSkip di_d_def fmore_d_def inteq_reflection schop_d_def)
 have 11: "\<turnstile> f\<frown>empty = (f \<and> finite) " 
    by (simp add: ChopEmpty schop_d_def)
 have 12: "\<turnstile> f \<frown> (#True\<frown>skip \<or> empty) = (f \<frown> (#True\<frown>skip) \<or> f\<frown>empty) " 
   by (simp add: SChopOrEqv)
 have 13: "\<turnstile> (di f \<and> finite) = ((f \<and> finite) \<or> (di f)\<frown>skip)"
   by (metis "10" "11" "2" "3" "7" "8" FmoreEqvSkipChopFinite SChopOrEqvRule WPowerstarEqv 
       WPowerstar_skip_finite fmore_d_def int_eq) 
 have 14: "\<turnstile> ((di f \<and> finite) \<or> (di f \<and> inf)) = 
             ((f \<and> finite) \<or> (di f)\<frown>skip \<or> (di f \<and> inf)) " 
   using 13 by fastforce
 show ?thesis using 1 14 by fastforce
qed

lemma DiFiniteEqv: 
"\<turnstile> (di f \<and> finite) = ((f \<and> finite) \<or> (di f)\<frown>skip)"
proof -
 have 1: "\<turnstile> (di f \<and> finite) = 
          (((f \<and> finite) \<or> (di f)\<frown>skip \<or> (di f \<and> inf) ) \<and> finite) " 
  using DiFiniteOrInf by (metis int_simps(1) inteq_reflection)
 have 2: "\<turnstile> (((f \<and> finite) \<or> (di f)\<frown>skip \<or> (di f \<and> inf) ) \<and> finite) =
            ((f \<and> finite) \<or> ((di f)\<frown>skip \<and> finite))" 
   by (metis (no_types, opaque_lifting) ChopAndFiniteDist ChopEmpty DiEqvOrDiChopSkipB 
       DiFiniteOrInf OrSChopEqv SChopAssoc inteq_reflection itl_def(9))
 have 3: "\<turnstile> ((di f)\<frown>skip \<and> finite) = ((di f)\<frown>skip) "
   by (metis Prop10 SChopImpFinite WPowerstar_ext WPowerstar_skip_finite inteq_reflection)
 show ?thesis 
 by (metis "1" "2" "3" inteq_reflection)
qed


lemma OrDsEqvDi:
 "\<turnstile> ((f \<and> finite) \<or>  (ds f \<and> finite)) = (di f \<and> finite)"
proof -
 have 1: "\<turnstile> (ds f \<and> finite)=  ((finite \<longrightarrow>(di f)\<frown>skip) \<and> finite)"
  using DsDi by fastforce
 have 11: "\<turnstile> ((finite \<longrightarrow>(di f)\<frown>skip) \<and> finite) = (((di f)\<frown>skip) \<and> finite)"
   by fastforce
 have 12: "\<turnstile> (((di f)\<frown>skip) \<and> finite) = (di f)\<frown>skip "
    by (metis Prop10 SChopImpFinite WPowerstar_ext WPowerstar_skip_finite inteq_reflection)
 have 2: "\<turnstile> ((f \<and> finite)  \<or> (ds f \<and> finite)) = 
             ((f \<and> finite) \<or> (di f)\<frown>skip)" 
             by (metis "1" "11" "12" DiFiniteEqv inteq_reflection)
 have 3: "\<turnstile> ((f \<and> finite) \<or> (di f)\<frown>skip) = (di f \<and> finite)" 
   by (meson DiFiniteEqv Prop11)
 from 2 3 show ?thesis 
 by (metis int_eq)
qed 



lemma BiFiniteEqv: 
 "\<turnstile> (bi f \<and> finite) = ((f \<and> finite) \<and> (empty \<or> (bi f)\<frown>skip))"
by (metis (no_types, opaque_lifting) BiAndEqv BiEqvAndWprevBi BiFiniteEqvFinite 
    WPrevAndFiniteEqv inteq_reflection)

lemma AndSChopSkipEqVFiniteAndSChopSkip: 
 "\<turnstile> (f \<and> g\<frown>skip) = ((f \<and> finite) \<and> g\<frown>skip) " 
proof -
 have 1: "\<turnstile> g\<frown>skip \<longrightarrow> finite" 
   by (metis LenOneEqvSkip SChopImpFinite inteq_reflection len_k_finite) 
 from 1 show ?thesis 
 by fastforce
qed


lemma BiFiniteEQvB: 
 " \<turnstile> (bi f \<and> finite) = (f \<and> (empty \<or> (bi f)\<frown>skip)) " 
proof -
 have 1: "\<turnstile> ((f \<and> finite) \<and> empty) = (f \<and> empty) "
   using FiniteAndEmptyEqvEmpty by auto
 have 2: "\<turnstile> ((f \<and> finite) \<and> (bi f)\<frown>skip) =  (f \<and> (bi f)\<frown>skip)"
   by (meson AndSChopSkipEqVFiniteAndSChopSkip Prop11)
 have 4: "\<turnstile> (((f \<and> finite) \<and> empty) \<or> ((f \<and> finite) \<and> (bi f)\<frown>skip)) =
            ((f \<and> empty) \<or> (f \<and> (bi f)\<frown>skip))" 
   using "1" "2" by fastforce
 have 3: "\<turnstile> (bi f \<and> finite) = ((f \<and> finite) \<and> (empty \<or> (bi f)\<frown>skip))" 
   by (simp add: BiFiniteEqv)
 show ?thesis using 3 4 by fastforce
qed

lemma AndBsEqvBi:
 "\<turnstile> (f \<and> bs f) = (bi f \<and> finite)"
by (metis BiFiniteEQvB BsEqvEmptyOrBiSChopSkip inteq_reflection)

lemma BsEqvBsBi:
 "\<turnstile> bs f = bs (bi f) "
proof -
 have 1: "\<turnstile> bs f = (empty \<or> (bi f)\<frown>skip)" by (simp add: bs_d_def)
 have 2: "\<turnstile> bi f = bi ( bi f) " by (rule BiEqvBiBi)
 hence 3: "\<turnstile> (bi f)\<frown>skip = bi (bi f)\<frown>skip " using LeftSChopEqvSChop by blast
 hence 4: "\<turnstile>  (empty \<or> (bi f)\<frown>skip) =  (empty \<or> bi (bi f)\<frown>skip) " by auto
 from 1 4  show ?thesis unfolding bs_d_def by auto
qed

lemma StateImpBs:
 "\<turnstile> init w \<and> finite \<longrightarrow> bs (init w)"
proof -
 have 1: "\<turnstile> init w = bi (init w) "  by (rule StateEqvBi)
 have 2: "\<turnstile> bi (init w) \<and> finite \<longrightarrow> bs (init w) " by (rule BiImpBs)
 from 1 2 show ?thesis using StateImpBi by fastforce
qed

lemma DsAndDsEqvDsAndDiOrDsAndDi:
 "\<turnstile> (ds f \<and> ds g) =( ds (f \<and> di g) \<or> ds(g \<and> di f))"
proof -
 have 1: "\<turnstile> (di f \<and> di g) = (di(f \<and> di g) \<or> di(g \<and> di f))" 
      by (rule DiAndDiEqvDiAndDiOrDiAndDi)
 hence 2: "\<turnstile> (di f \<and> di g)\<frown>skip = (di(f \<and> di g) \<or> di(g \<and> di f))\<frown>skip" 
      by (rule LeftSChopEqvSChop)
 have 3: "\<turnstile> (di f \<and> di g)\<frown>skip = ((di f)\<frown>skip \<and> (di g)\<frown>skip) " 
      using SChopSkipAndSChopSkip by fastforce
 have 4: "\<turnstile> ((di f)\<frown>skip \<and> (di g)\<frown>skip) = (di(f \<and> di g) \<or> di(g \<and> di f))\<frown>skip " 
      using "2" "3" by fastforce
 have 5: "\<turnstile> (di(f \<and> di g) \<or> di(g \<and> di f))\<frown>skip = (di(f \<and> di g)\<frown>skip \<or> di(g \<and> di f)\<frown>skip)" 
      using OrSChopEqv by blast
 have 6: "\<turnstile> ds f = (finite \<longrightarrow>(di f)\<frown>skip)  " 
      using DsDi by blast
 have 7: "\<turnstile> ds g = (finite \<longrightarrow>(di g)\<frown>skip) " 
      using DsDi by blast
 have 8: "\<turnstile>  ((finite\<longrightarrow>(di f)\<frown>skip) \<and> (finite \<longrightarrow>(di g)\<frown>skip)) = (ds f \<and>  ds g)" 
      using "6" "7" by fastforce
 have 81: " \<turnstile> ((finite \<longrightarrow> di f \<frown> skip) \<and> (finite \<longrightarrow> di g \<frown> skip)) = 
              (finite \<longrightarrow> di f \<frown> skip \<and> di g \<frown> skip) "
      by fastforce
 have 9: "\<turnstile> ds(f \<and> di g) =  (finite \<longrightarrow> di(f \<and> di g)\<frown>skip)" 
      using DsDi by blast
 have 10: "\<turnstile> ds(g \<and> di f) = (finite \<longrightarrow>di(g \<and> di f)\<frown>skip)  " 
      using DsDi by blast
 have 11: "\<turnstile> ((finite \<longrightarrow> di(f \<and> di g)\<frown>skip) \<or> (finite \<longrightarrow> di(g \<and> di f)\<frown>skip)) = 
        (ds(f \<and> di g) \<or> ds(g \<and> di f))" 
      using "9" "10" by fastforce
 have 12: "\<turnstile> ((finite \<longrightarrow> di(f \<and> di g)\<frown>skip) \<or> (finite \<longrightarrow> di(g \<and> di f)\<frown>skip)) =
             (finite \<longrightarrow> di(f \<and> di g)\<frown>skip \<or> di(g \<and> di f)\<frown>skip) "
       by fastforce
 from 4 5 8 11 12 81 show ?thesis by (metis inteq_reflection)
qed

lemma SChopFiniteEqvSChopTrueAndFinite: 
 "\<turnstile> g\<frown>finite = (g\<frown>#True \<and> finite) " 
by (meson FiniteImportSChopRight Prop04 RightSChopEqvSChop int_simps(17) lift_and_com)

lemma BsEqvBiMoreImpChop:
 "\<turnstile> (bs f) = (bi(more \<and> finite \<longrightarrow> f\<frown>skip) \<and> finite) "
proof -
 have 1: "\<turnstile> (bs f \<or> inf) = (empty \<or> inf \<or> (bi f\<frown>skip))" 
   unfolding bs_d_def by fastforce
 have 2: "\<turnstile> (empty \<or> inf \<or> (bi f\<frown>skip)) = ((\<not>(\<not>(bi f))\<frown>skip))  " 
      using NotNotSChopSkip by fastforce
 have 3: "\<turnstile> \<not>((\<not>(bi f))\<frown>skip) = (\<not>(di (\<not> f)\<frown>skip))" 
      by (simp add: bi_d_def)
 have 4: "\<turnstile> (\<not>(di (\<not> f)\<frown>skip)) = (\<not>(((\<not> f) \<frown>#True)\<frown>skip)) " 
    by (metis FiniteImpAnd FiniteImpDfEqvDi LeftChopImpChop Prop11 itl_def(15) lift_imp_neg schop_d_def)
 have 5: "\<turnstile> (\<not>(((\<not> f) \<frown>#True)\<frown>skip)) = (\<not>((\<not> f) \<frown>(#True\<frown>skip)))" 
    by (meson Prop11 SChopAssoc lift_imp_neg)
 have 05: "\<turnstile> #True\<frown>skip = skip \<frown> finite " 
   by (metis DiamondSChopdef Prop10 SkipFiniteEqvFiniteSkip WPowerstar_ext WPowerstar_skip_finite
        inteq_reflection itl_def(10) itl_def(9))
 have 6: "\<turnstile> (\<not>((\<not> f) \<frown>(#True\<frown>skip))) = (\<not>((\<not> f) \<frown>(skip\<frown>finite)))"
      by (metis "05" "5" inteq_reflection)
 have 7: "\<turnstile> (\<not>((\<not> f) \<frown>(skip\<frown>finite))) = (\<not>(((\<not> f) \<frown>skip)\<frown>finite)) " 
      using SChopAssoc by fastforce
 have 07: "\<turnstile> (\<not>(((\<not> f) \<frown>skip)\<frown>finite)) = ((\<not>(((\<not> f) \<frown>skip)\<frown>#True)) \<or> inf) " 
    using InfEqvNotFinite SChopFiniteEqvSChopTrueAndFinite by fastforce
 have 8: "\<turnstile> (\<not>(((\<not> f) \<frown>skip)\<frown>finite)) = ((\<not>(di ((\<not> f)\<frown>skip)))  \<or> inf) "
   by (metis "07" FiniteImportSChopRight Prop10 WPowerstar_ext WPowerstar_skip_finite di_d_def
       inteq_reflection lift_and_com schop_d_def)
 have 9: "\<turnstile> (\<not>(di ((\<not> f)\<frown>skip))) = bi (\<not>((\<not> f) \<frown>skip))  " 
      using NotDiEqvBiNot by blast
 have 10: "\<turnstile> bi (\<not>((\<not> f) \<frown>skip)) = bi( empty \<or> inf \<or> (f\<frown>skip))" 
      using NotNotSChopSkip using BiEqvBi by blast
 have 010: "\<turnstile> ( empty \<or> inf \<or> (f\<frown>skip)) = ( \<not> (more \<and> finite) \<or> (f\<frown>skip)) "
     unfolding empty_d_def finite_d_def by fastforce
 have 11: "\<turnstile> bi( empty \<or> inf \<or> (f\<frown>skip)) = bi( \<not> (more \<and> finite) \<or> (f\<frown>skip))" 
    by (simp add: "010" BiEqvBi)
 have 12: "\<turnstile> ( \<not> (more \<and> finite) \<or> (f\<frown>skip)) = (more \<and> finite\<longrightarrow> f\<frown>skip) " by auto
 have 13: "\<turnstile> bi( \<not> (more \<and> finite) \<or> (f\<frown>skip)) = bi(more \<and> finite \<longrightarrow> f\<frown>skip) " 
   using "12" using BiEqvBi by blast
 have 14: "\<turnstile> (bs f \<or> inf) = (\<not> (((\<not> f)\<frown>skip)\<frown>finite)) " using  "1" "2" "3" "4" "5" "6" "7" 
    by (metis NotSChopNotSkip bi_d_def inteq_reflection)
 have 15: "\<turnstile> (\<not> (((\<not> f)\<frown>skip)\<frown>finite)) = (bi(more \<and> finite\<longrightarrow> f\<frown>skip) \<or> inf) " 
  using "8" "9" "10" "11" "13"  by (metis inteq_reflection)
 have 16: "\<turnstile> (bs f \<or> inf) = (bi(more \<and> finite \<longrightarrow> f\<frown>skip) \<or> inf)" 
   using 14 15 by fastforce
 have 17: "\<turnstile> bs f \<longrightarrow> finite"
    by (metis AndChopB EmptyImpFinite FiniteChopSkipImpFinite Prop02 bs_d_def lift_imp_trans schop_d_def)
 have 18: "\<turnstile> ((bs f \<or> inf) \<and> finite) = bs f" 
    using 17 unfolding finite_d_def  by fastforce
 have 19: "\<turnstile> ((bi(more \<and> finite \<longrightarrow> f\<frown>skip) \<or> inf) \<and> finite) =
             (bi(more \<and> finite \<longrightarrow> f\<frown>skip) \<and> finite) " 
       unfolding finite_d_def by fastforce
 have 20: "\<turnstile> ((bs f \<or> inf) \<and> finite) = ((bi(more \<and> finite \<longrightarrow> f\<frown>skip) \<or> inf) \<and> finite) " 
    using 16 by fastforce
  show ?thesis using 16 18 19 20 by (metis inteq_reflection)
qed

lemma NotSChopInf: 
 "\<turnstile> (\<not> ( f\<frown>inf)) = (finite \<or> \<not>(f\<frown>#True))"
proof -
 have 1: "\<turnstile> (\<not> ( f\<frown>inf)) = (finite \<or> (bf (\<not> f))) " 
   by (metis NotNotSChopInf int_simps(4) inteq_reflection)
 have 2: "\<turnstile> bf (\<not> f) = (\<not> (f \<frown> #True))" 
   unfolding bf_d_def df_d_def by auto   
 show ?thesis using "1" "2" Prop06 by blast
qed

lemma BoxMoreStateEqvBsFinState:
 "\<turnstile> (\<box>(more \<longrightarrow> \<not> (init w)) \<and> finite) = bs(\<not>(fin(init w))) "
proof -
 have 1: "\<turnstile> \<box>(more \<longrightarrow> \<not> (init w)) = (\<not>(\<diamond>(\<not>(more \<longrightarrow> \<not> (init w))))) " 
      by (simp add: always_d_def)
 have 01: "\<turnstile> (\<not>(more \<longrightarrow> \<not> (init w))) = (init w \<and>  more)" by auto  
 hence 2: "\<turnstile> \<not>(\<diamond>(\<not>(more \<longrightarrow> \<not> (init w)))) = (\<not>(#True\<frown>(init w \<and>  more)))" 
   by (metis DiamondSChopdef TrueW int_simps(3) int_simps(6) inteq_reflection)
 have 3: "\<turnstile> more = (#True\<frown> skip \<or> inf)" 
   by (metis FmoreEqvSkipChopFinite MoreAndInfEqvInf OrFiniteInf SkipFiniteEqvFiniteSkip 
       fmore_d_def int_simps(17) inteq_reflection schop_d_def)
 have 4: "\<turnstile> (init w \<and>  more) = (init w \<and> ((#True\<frown> skip \<or> inf))) " 
      using "3" by auto
 have 5: "\<turnstile> (init w \<and> (#True\<frown> skip \<or> inf))   = 
            ((init w \<and> empty)\<frown>(#True\<frown>skip) \<or> (init w \<and> empty)\<frown>inf)" 
     by (meson Prop04 SChopOrEqv StateAndEmptySChop)
 have 6: "\<turnstile> (init w \<and>  more) = ((init w \<and> empty)\<frown>(#True\<frown>skip) \<or> (init w \<and> empty)\<frown>inf) " 
      using "4" "5" by fastforce
 have 7: "\<turnstile>  (#True\<frown>(init w \<and>  more)) = 
             (#True\<frown>((init w \<and> empty)\<frown>(#True\<frown>skip) \<or> (init w \<and> empty)\<frown>inf) )" 
      using "6" by (simp add: RightSChopEqvSChop)
 have 8: "\<turnstile> (#True\<frown>((init w \<and> empty)\<frown>(#True\<frown>skip)) ) =
            (((#True\<frown>(init w \<and> empty))\<frown>(#True\<frown>skip)) ) " 
      using SChopAssoc by blast
 have 9: "\<turnstile> (((#True\<frown>(init w \<and> empty))\<frown>(#True\<frown>skip)) ) = 
            ((((#True\<frown>(init w \<and> empty))\<frown>#True)\<frown>skip) )" 
      using SChopAssoc by blast
 have 10: "\<turnstile> (#True\<frown>(init w \<and>  more)) = 
             ((((#True\<frown>(init w \<and> empty))\<frown>#True)\<frown>skip) \<or> #True\<frown>((init w \<and> empty)\<frown>inf) )" 
      using "7" "8" "9" by (metis SChopOrEqv inteq_reflection)
 hence 11: "\<turnstile> (\<not>(#True\<frown>(init w \<and>  more))) = 
              (\<not>((((#True\<frown>(init w \<and> empty))\<frown>#True)\<frown>skip) \<or> #True\<frown>((init w \<and> empty)\<frown>inf) ))" 
      by auto
 have 011: "\<turnstile> (\<not>((((#True\<frown>(init w \<and> empty))\<frown>#True)\<frown>skip) \<or> #True\<frown>((init w \<and> empty)\<frown>inf) )) =
              (\<not>((((#True\<frown>(init w \<and> empty))\<frown>#True)\<frown>skip) ) \<and> \<not>( #True\<frown>((init w \<and> empty)\<frown>inf))) " 
    by fastforce
 have 12: "\<turnstile> (\<not>((((#True\<frown>(init w \<and> empty))\<frown>#True)\<frown>skip) )) =
             (empty \<or> inf \<or> (\<not>((#True\<frown>(init w \<and> empty))\<frown>#True)\<frown>skip))" 
      using NotSChopNotSkip by fastforce
 have 012: "\<turnstile> (\<not>( #True\<frown>((init w \<and> empty)\<frown>inf))) = (finite \<or> (\<not>((#True\<frown>(init w \<and> empty)))\<frown>#True)) " 
    using NotSChopInf SChopAssoc by fastforce
 have 0130: "\<turnstile> (finite;(init w \<and> empty)) = finite; (w \<and> empty)" 
   by (simp add: InitAndEmptyEqvAndEmpty RightChopEqvChop)
 have 0131: "\<turnstile> ((init w \<and> empty) \<and> finite) = (w \<and> empty)"
 by (metis ChopEmpty InitAndEmptyEqvAndEmpty StateAndEmptySChop inteq_reflection schop_d_def) 
 have o132: "\<turnstile> #True\<frown>((init w \<and> empty) \<and> finite) = finite;(w \<and> empty)"
 by (simp add: "0131" ChopEqvChop schop_d_def)   
 have 013: "\<turnstile> (finite;(init w \<and> empty)) = #True\<frown>((init w \<and> empty) \<and> finite)"
 using "0130" o132 by fastforce
 have 13: "\<turnstile> (\<not>((#True\<frown>(init w \<and> empty))\<frown>#True)) =
              bi(\<box> (\<not>(init w \<and> empty)))" 
   using BiBoxNotEqvNotFiniteChopChopTrue[of "LIFT ((init w \<and> empty))" ]  
    by (metis (no_types, lifting) "013" ChopAssoc SChopAssoc inteq_reflection schop_d_def)
 hence 14: "\<turnstile> (\<not>((#True\<frown>(init w \<and> empty))\<frown>#True))\<frown>skip = 
              (bi(\<box> (\<not>(init w \<and> empty))))\<frown>skip" using LeftSChopEqvSChop by blast
 hence 15: "\<turnstile> (empty \<or> inf \<or> (\<not>((#True\<frown>(init w \<and> empty))\<frown>#True))\<frown>skip) = 
              (empty \<or> inf \<or>(bi(\<box> (\<not>(init w \<and> empty))))\<frown>skip)" 
      by auto
 have 015: "\<turnstile> (finite \<or> (\<not>((#True\<frown>(init w \<and> empty)))\<frown>#True)) = 
              (finite \<or> bi(\<box> (\<not>(init w \<and> empty)))) "
    using "13" by auto
 have 16: "\<turnstile> (\<not>((((#True\<frown>(init w \<and> empty))\<frown>#True)\<frown>skip) \<or> #True\<frown>((init w \<and> empty)\<frown>inf) )) =
              ((empty \<or> inf \<or>(bi(\<box> (\<not>(init w \<and> empty)))\<frown>skip)) \<and> (finite \<or> bi(\<box> (\<not>(init w \<and> empty))))) "
    by (metis "011" "012" "13" NotSChopNotSkip inteq_reflection) 
 have 171: "\<turnstile> (\<not>(init w \<and> empty)) = (\<not>(init w) \<or> \<not>empty)" 
      by auto
 hence 172: "\<turnstile> \<box>(\<not>(init w \<and> empty)) = \<box>(\<not>(init w) \<or> \<not>empty) " 
      by (simp add: BoxEqvBox)
 hence 173: "\<turnstile> bi(\<box>(\<not>(init w \<and> empty))) = bi(\<box>(\<not>(init w) \<or> \<not>empty)) " 
      by (simp add: BiEqvBi)
 hence 174: "\<turnstile> bi(\<box>(\<not>(init w \<and> empty)))\<frown>skip = bi(\<box>(\<not>(init w) \<or> \<not>empty))\<frown>skip "
      using LeftSChopEqvSChop by blast
 hence 17: "\<turnstile> (empty \<or> inf \<or> (bi(\<box> (\<not>(init w \<and> empty)))\<frown>skip)) =
             (empty \<or> inf \<or> (bi(\<box> (\<not>(init w) \<or> \<not>empty))\<frown>skip))" 
      by auto
 have 175: "\<turnstile> (finite \<or> bi(\<box> (\<not>(init w \<and> empty)))) = 
              (finite \<or> bi(\<box> (\<not>(init w) \<or> \<not>empty)))" 
   using 173 by fastforce
 have 181: "\<turnstile>  (\<not>(init w) \<or> \<not>empty) =  (\<not>empty \<or> \<not>(init w)  )" 
      by auto
 hence 18: "\<turnstile> \<box> (\<not>(init w) \<or> \<not>empty) = \<box> (\<not>empty \<or> \<not>(init w)  )" 
      by (simp add: BoxEqvBox)
 have 191: "\<turnstile>  (\<not>empty \<or> \<not>(init w)  ) = (empty \<longrightarrow> \<not>(init w) )" 
      by auto
 hence 19: "\<turnstile> \<box> (\<not>empty \<or> \<not>(init w)  ) = \<box>(empty \<longrightarrow> \<not>(init w) )" 
      by (simp add: BoxEqvBox)
 have 20: "\<turnstile> \<box>(empty \<longrightarrow> \<not>(init w) ) = fin (\<not>(init w) )" 
      by (simp add: fin_d_def)
 have 21: "\<turnstile> (fin (\<not>(init w) ) \<and> finite) = ((\<not>(fin (init w)) \<and> finite)) " 
   by (metis FinNotStateEqvNotFinState Initprop(2) inteq_reflection)
 have 22: "\<turnstile> (bi(\<box> (\<not>(init w) \<or> \<not>empty)) \<and> finite) = (bi (\<not>(fin (init w))) \<and> finite)"
   by (metis "181" "191" "20" "21" BiAndEqv BiFiniteEqvFinite inteq_reflection) 
 hence 23: "\<turnstile> (bi(\<box> (\<not>(init w) \<or> \<not>empty)))\<frown>skip = (bi (\<not>(fin (init w))))\<frown>skip " 
  by (simp add: LeftChopEqvChop schop_d_def)
 hence 24: "\<turnstile> (empty \<or> inf \<or> (bi(\<box> (\<not>(init w) \<or> \<not>empty)))\<frown>skip) = 
              (empty \<or> inf \<or> (bi (\<not>(fin (init w))))\<frown>skip)" 
      by auto
 have 241:"\<turnstile> (bi (\<not>(fin (init w))))\<frown>skip = bf (\<not> fin (init w)) \<frown> skip " 
    by (metis (no_types, opaque_lifting) EmptyChop FiniteImpAnd FiniteImpBfEqvBi LeftChopEqvChop Prop04 schop_d_def)
 hence 25: "\<turnstile> (empty \<or> inf \<or> (bi (\<not>(fin (init w))))\<frown>skip) = (inf \<or> bs(\<not>(fin (init w)))) " 
      unfolding bs_d_def by auto
 have 26:"\<turnstile> (\<box>(more \<longrightarrow> \<not> (init w)) \<and> finite) = 
              (((empty \<or> inf \<or>(bi(\<box> (\<not>(init w \<and> empty)))\<frown>skip)) \<and> (finite \<or> bi(\<box> (\<not>(init w \<and> empty)))))
               \<and> finite)"
    by (metis (no_types, opaque_lifting) "01" "10" "16" TrueSChopEqvDiamond always_d_def int_simps(4) inteq_reflection)
 have 27: "\<turnstile> (((empty \<or> inf \<or>(bi(\<box> (\<not>(init w \<and> empty)))\<frown>skip)) \<and> (finite \<or> bi(\<box> (\<not>(init w \<and> empty)))))
               \<and> finite) =
              (((empty \<or> inf \<or>(bi(\<box> (\<not>(init w \<and> empty)))\<frown>skip)) )
               \<and> finite)" 
     by fastforce
 have 28: "\<turnstile> (bi(\<box> (\<not>(init w \<and> empty)))\<frown>skip) \<longrightarrow> finite " 
          by (metis LenOneEqvSkip SChopImpFinite inteq_reflection len_k_finite)
 have 29:"\<turnstile> empty \<longrightarrow> finite" 
    by (simp add: EmptyImpFinite)
 have 29: "\<turnstile>  (((empty \<or> inf \<or>(bi(\<box> (\<not>(init w \<and> empty)))\<frown>skip)) ) \<and> finite) = 
              (((empty \<or>  (bi(\<box> (\<not>(init w \<and> empty)))\<frown>skip)) )) "     
    using 28 29 unfolding finite_d_def by fastforce
 have 30: "\<turnstile> (((empty \<or>  (bi(\<box> (\<not>(init w \<and> empty)))\<frown>skip)) )) = bs(\<not>(fin (init w))) " 
    by (metis "171" "23" "241" "29" bs_d_def inteq_reflection)
  show ?thesis 
  by (metis "26" "27" "29" "30" inteq_reflection)
qed

lemma BsFalseEqvEmpty:
 "\<turnstile> bs #False = empty"
proof -
 have 1: "\<turnstile> bs #False = (empty \<or> bi #False\<frown>skip) " 
      by (simp add: bs_d_def)
 have 2: "\<turnstile> \<not>(bi #False\<frown>skip)"  
  by (metis BiFiniteEQvB MoreEqvSkipChopTrue NotChopSkipEqvMoreAndNotChopSkip SkipTrueEqvTrueSkip 
      TrueW int_simps(19) int_simps(2) int_simps(21) int_simps(3) inteq_reflection schop_d_def)
 from 1 2 show ?thesis by fastforce
qed

subsubsection \<open>First occurrence\<close>


lemma FstEqvRule:
 assumes "\<turnstile> f = g"
 shows   "\<turnstile> \<rhd>f = \<rhd>g "
proof -
 have 1: "\<turnstile> f = g" using assms by auto
 hence 2: "\<turnstile> (\<not> f) = (\<not> g)" by auto
 hence 3: "\<turnstile> bs(\<not> f) = bs(\<not> g)" by (simp add: BsEqvRule)
 hence 4: "\<turnstile> (f \<and> bs(\<not> f)) = (g \<and> bs(\<not> g)) " using "1" by fastforce
 from 4 show ?thesis by (simp add:first_d_def)
qed

lemma FstImpFinite:
 "\<turnstile> \<rhd> f \<longrightarrow> finite" 
unfolding first_d_def using BsEqvBiMoreImpChop by fastforce

lemma FstWithAndImp:
 "\<turnstile> \<rhd>f \<and> g \<longrightarrow> \<rhd> ( f \<and> g)"
proof -
 have 1: "\<turnstile> (\<rhd>f \<and> g) = ((f \<and> (bs (\<not>f))) \<and> g) " 
      by (simp add: first_d_def) 
 have 2: "\<turnstile> ((f \<and> (bs (\<not>f))) \<and> g) = (f \<and> \<not>(ds f) \<and> g)" 
      using NotDsEqvBsNot by fastforce
 have 3: "\<turnstile> \<not>(ds f) \<longrightarrow> \<not>(ds(f \<and> g))" 
      using DsAndImpElimL by fastforce
 hence 4: "\<turnstile> f \<and> \<not>(ds f) \<and> g \<longrightarrow> f \<and> g \<and> \<not>(ds(f \<and> g)) " 
      by auto
 have  5: "\<turnstile> (f \<and> g \<and> \<not>(ds(f \<and> g))) = ((f \<and> g) \<and> (bs (\<not>(f \<and> g))))  " 
      using NotDsEqvBsNot by fastforce
 have  6: "\<turnstile> ((f \<and> g) \<and> (bs (\<not>(f \<and> g)))) = \<rhd>(f \<and> g) " 
      by (simp add: first_d_def)
 from 1 2 4 5 6 show ?thesis by fastforce
qed
 
lemma FstWithOrEqv:
 "\<turnstile> \<rhd>(f \<or> g) = ((\<rhd>f \<and> bs (\<not>g)) \<or> (\<rhd>g \<and> bs (\<not>f)))" 
proof -
 have 1: "\<turnstile> \<rhd>(f \<or> g) = ((f \<or> g) \<and> bs (\<not>(f \<or> g)))" 
      by (simp add: first_d_def)
 have 2: "\<turnstile>  (\<not>(f \<or> g)) =  (\<not>f \<and> \<not>g)" 
      by auto
 hence 3: "\<turnstile> bs (\<not>(f \<or> g)) =  bs (\<not>f \<and> \<not>g)" 
      using BsEqvRule by blast
 have  4: "\<turnstile>  bs (\<not>f \<and> \<not>g) = (bs (\<not> f) \<and> bs (\<not> g))" 
      using BsAndEqv  by fastforce
 have  5: "\<turnstile> ((f \<or> g) \<and> bs (\<not>(f \<or> g))) = ((f \<or> g) \<and> bs (\<not> f) \<and> bs (\<not> g))" 
      using "3" "4" by fastforce
 have  6: "\<turnstile> ((f \<or> g) \<and> bs (\<not> f) \<and> bs (\<not> g)) = 
             (((f \<and> bs (\<not> f)) \<and> bs (\<not> g)) \<or> (g \<and> bs (\<not> f) \<and> bs (\<not> g))) " 
      by auto
 have  7: "\<turnstile> ((f \<and> bs (\<not> f)) \<and> bs (\<not> g)) = (\<rhd>f \<and>  bs (\<not> g))  " 
      by (simp add: first_d_def)
 have  8: "\<turnstile> (g \<and> bs (\<not> f) \<and> bs (\<not> g)) = ((g \<and> bs (\<not> g)) \<and> bs (\<not> f))" 
      by auto
 have  9: "\<turnstile> ((g \<and> bs (\<not> g)) \<and> bs (\<not> f)) = (\<rhd>g \<and>  bs (\<not> f))" 
      by (simp add: first_d_def)
 have 10: "\<turnstile> ((f \<and> bs (\<not> f)) \<and> bs (\<not> g)) \<or> (g \<and> bs (\<not> f) \<and> bs (\<not> g)) = 
            (\<rhd>f \<and>  bs (\<not> g)) \<or> (\<rhd>g \<and>  bs (\<not> f)) " 
      using "7" "8" "9" by fastforce
 from 1 5 6 10 show ?thesis by (metis "7" "8" "9" int_eq)
qed

lemma FstFstAndEqvFstAnd:
 "\<turnstile> \<rhd>(\<rhd>f \<and> g) = (\<rhd>f \<and> g)"
proof -
  have 1: "\<turnstile> (\<rhd>f \<and> g) = ((f \<and> (bs (\<not>f))) \<and> g)" by (simp add: first_d_def)
  hence 2: "\<turnstile> \<rhd>f \<and> g \<longrightarrow> (bs (\<not>f))" by auto
  hence 3: "\<turnstile> \<rhd>f \<and> g \<longrightarrow> \<rhd>f \<and> g \<and> (bs (\<not>f)) " by auto
  have 4: "\<turnstile> \<not> f \<longrightarrow> \<not> f \<or> \<not>(bs (\<not>f)) \<or> \<not> g" by auto
  hence 5: "\<turnstile> bs (\<not> f) \<longrightarrow> bs(\<not> f \<or> \<not>(bs (\<not>f)) \<or> \<not> g)" using BsImpBsRule by blast
  have 6: "\<turnstile> (\<not> f \<or> \<not>(bs (\<not>f)) \<or> \<not> g) = (\<not>(f \<and> bs (\<not> f) \<and> g))" by auto
  hence 7: "\<turnstile> bs(\<not> f \<or> \<not>(bs (\<not>f)) \<or> \<not> g) = bs(\<not>(f \<and> bs (\<not> f) \<and> g))" using BsEqvRule by blast
  have 8: "\<turnstile> ((f \<and> bs (\<not> f)) \<and> g) = (\<rhd>f \<and> g)  "  by (simp add: first_d_def)
  hence 9: "\<turnstile> (\<not>(f \<and> bs (\<not> f) \<and> g)) = (\<not>(\<rhd>f \<and> g))" by auto
  hence 10: "\<turnstile> bs (\<not>(f \<and> bs (\<not> f) \<and> g)) = bs (\<not>(\<rhd>f \<and> g))" using BsEqvRule by blast
  have 11: "\<turnstile> \<rhd>f \<and> g \<longrightarrow> (\<rhd>f \<and> g) \<and>   bs (\<not>(\<rhd>f \<and> g))" using "3" "5" "7" "10" by fastforce
  hence 12: "\<turnstile> \<rhd>f \<and> g \<longrightarrow> \<rhd>(\<rhd>f \<and> g)" by (simp add: first_d_def)
  have 13: "\<turnstile> \<rhd>(\<rhd>f \<and> g) = ((\<rhd>f \<and> g) \<and>   bs (\<not>(\<rhd>f \<and> g))) " by (simp add: first_d_def)
  hence 14: "\<turnstile> \<rhd>(\<rhd>f \<and> g) \<longrightarrow> \<rhd>f \<and> g" by auto
 from 12 14 show ?thesis by fastforce
qed

lemma FstTrue:
 "\<turnstile> \<rhd> #True = empty"
proof -
 have 1: "\<turnstile> \<rhd> #True = (#True \<and> bs (\<not> #True))" 
      by (simp add: first_d_def)
 have 2: "\<turnstile> bs (\<not> #True) = (empty \<or> (bi (\<not> #True))\<frown>skip) " 
      by (simp add: bs_d_def)
 have 3: "\<turnstile> \<not>(bi (\<not> #True)) " 
   using BiElim by fastforce
 have 4: "\<turnstile> \<not>((bi (\<not> #True))\<frown>skip)" 
     by (metis (no_types, opaque_lifting) BiEqvBiBi DiSkipEqvMore NotChopSkipEqvMoreAndNotChopSkip
         SkipTrueEqvTrueSkip TrueEqvTrueSChopTrue di_d_def int_simps(17) int_simps(19) 
         int_simps(21) int_simps(7) int_simps(9) inteq_reflection itl_def(18) schop_d_def)
 have 5: "\<turnstile> bs (\<not> #True) = empty " 
      using "2" "4" by fastforce
 from 1 5 show ?thesis by fastforce
qed

lemma FstFalse:
  "\<turnstile> \<not>(\<rhd> #False) "
proof -
 have 1: "\<turnstile> \<rhd> #False = (#False \<and> bs #True)" by (simp add: first_d_def)
 from 1 show ?thesis by auto
qed

lemma FstChopFalseEqvFalse:
 "\<turnstile> \<not>(\<rhd>f \<frown> #False)"
by (simp add:Valid_def itl_defs first_d_def bs_d_def )

lemma FstEmpty:
 "\<turnstile> \<rhd> empty = empty"
proof -
 have 1: "\<turnstile> \<rhd> empty = (empty \<and> bs (\<not> empty))" by (simp add: first_d_def)
 have 2: "\<turnstile> bs (\<not> empty) = (empty \<or> bi (\<not> empty)\<frown>skip)" by (simp add: bs_d_def)
 from 1 2 show ?thesis by fastforce
qed

lemma FstAndEmptyEqvAndEmpty:
 "\<turnstile> (\<rhd>f \<and> empty) = (f \<and> empty)"
proof -
 have 1: "\<turnstile> (\<rhd>f \<and> empty) = ((f \<and> bs (\<not> f)) \<and> empty)" by (simp add: first_d_def)
 have 2: "\<turnstile> bs (\<not> f) = (empty \<or> bi (\<not>f)\<frown>skip)" by (simp add: bs_d_def)
 from 1 2 show ?thesis by fastforce
qed

lemma FstEmptyOrEqvEmpty:
 "\<turnstile> \<rhd>(empty \<or> f) = empty"
proof -
 have 1: "\<turnstile> \<rhd>(empty \<or> f) = ((\<rhd>empty \<and> bs (\<not> f)) \<or> (\<rhd>f \<and> bs (\<not> empty)))" using FstWithOrEqv by blast
 have 2: "\<turnstile>  (\<not> empty) = more"  by (simp add: empty_d_def)
 hence 3: "\<turnstile> bs (\<not> empty) = bs more" using BsEqvRule by blast
 have 4: "\<turnstile> bs more = empty" using BsMoreEqvEmpty by blast
 have 5: "\<turnstile> (\<rhd>f \<and> bs (\<not> empty)) = (\<rhd>f \<and> empty)" using "3" "4" by fastforce
 have  6: "\<turnstile> \<rhd>empty = empty" using FstEmpty by blast
 hence 7: "\<turnstile> (\<rhd>empty \<and> bs (\<not> f)) = (empty \<and> bs (\<not> f))" by auto
 have 8: "\<turnstile> (empty \<and>  bs (\<not> f)) = (empty \<and>(empty \<or> bi (\<not> f)\<frown>skip))" by (simp add:bs_d_def)
 have 9: "\<turnstile> (empty \<and>(empty \<or> bi (\<not> f)\<frown>skip)) = empty" by auto
 have 10: "\<turnstile> (empty \<and>  bs (\<not> f)) = empty " using "8" "9" by auto
 have 11: "\<turnstile> ((\<rhd>empty \<and> bs (\<not> f)) \<or> (\<rhd>f \<and> bs (\<not> empty))) =
          (empty \<or> (\<rhd>f \<and> empty))" using "7" "10" "5" by fastforce
 have 12: "\<turnstile> (empty \<or> (\<rhd>f \<and> empty)) = empty" by auto
 from 1 11 12 show ?thesis by fastforce
qed

lemma FstChopEmptyEqvFstChopFstEmpty:
 "\<turnstile> (\<rhd>f\<frown>g \<and> empty) = (\<rhd>f\<frown>\<rhd>g \<and> empty)"
proof -
 have 1: "\<turnstile> (\<rhd>f\<frown>g \<and> empty) = (\<rhd>f \<and> g \<and> empty)" using SChopEmptyAndEmpty by blast
 have 2: "\<turnstile> (\<rhd>g \<and> empty) = (g \<and> empty)  " using FstAndEmptyEqvAndEmpty by blast
 hence 3: "\<turnstile> (\<rhd>f \<and> g \<and> empty) = (\<rhd>f \<and> \<rhd>g \<and> empty)" by auto
 have 4: "\<turnstile> (\<rhd>f\<frown>\<rhd>g \<and> empty) = (\<rhd>f \<and> \<rhd>g \<and> empty)" using SChopEmptyAndEmpty by blast
 from 1 3 4 show ?thesis by fastforce
qed

lemma FstMoreEqvSkip:
 "\<turnstile> \<rhd> more = skip "
proof -
 have 1: "\<turnstile> \<rhd> more = (more \<and> bs (\<not> more)) " by (simp add: first_d_def)
 have 2: "\<turnstile> (more \<and> bs (\<not> more)) = (more \<and> (empty \<or> bi (\<not> more)\<frown>skip)) " by (simp add:bs_d_def) 
 have 3: "\<turnstile> (more \<and> (empty \<or> bi (\<not> more)\<frown>skip)) = (more \<and> bi (\<not> more)\<frown>skip)"
  unfolding empty_d_def by fastforce
 have 4: "\<turnstile> (more \<and> ((bi (\<not> more))\<frown>skip)) = ((bi (\<not> more))\<frown>skip)" 
  using SChopSkipImpMore by fastforce
 have 5: "\<turnstile> ((bi (\<not> more))\<frown>skip) = bi empty\<frown>skip" by (simp add: empty_d_def)
 have 6: "\<turnstile> bi empty = empty" using BiEmptyEqvEmpty by auto
 hence 7: "\<turnstile> bi empty\<frown>skip = empty\<frown>skip"
 using LeftSChopEqvSChop by blast
 have 8: "\<turnstile> empty\<frown>skip = skip " using EmptySChop by blast
 from 1 2 3 4 5 7 8 show ?thesis by (metis int_eq)
qed

lemma FstEqvBsNotAndDi:
 "\<turnstile> \<rhd>f = (bs (\<not> f) \<and> di f)"
proof -
 have 1: "\<turnstile> bs (\<not> f)  = (\<not>(ds f)) " by (simp add: ds_d_def)
 hence 2: "\<turnstile> (bs (\<not> f)  \<and> di f) = (\<not>(ds f) \<and> di f)" by auto
 have 3: "\<turnstile>  (di f \<and> finite) = ((ds f \<and> finite)\<or> (f \<and> finite))" 
  using OrDsEqvDi by fastforce
 have 03: "\<turnstile> (\<not>(ds f)) \<longrightarrow> finite" 
    by (metis BsEqvBiMoreImpChop Prop11 Prop12 ds_d_def int_simps(4) lift_imp_trans)
 have 4 : "\<turnstile> (\<not>(ds f)  \<and> di f) = (\<not>(ds f)  \<and> ((ds f \<and> finite) \<or> (f \<and> finite)))"
    using "03" "3" by fastforce
 have 5 : "\<turnstile> (\<not>(ds f)  \<and> ((ds f \<and> finite)\<or> (f \<and> finite))) = 
             (\<not>(ds f)  \<and> f \<and> finite)" by auto
 have 6 : "\<turnstile> (\<not>(ds f)  \<and> f \<and> finite) = (f \<and>  bs (\<not> f))" using 1 
 using "03" by fastforce
 from 2 4 5 6 show ?thesis by (metis first_d_def int_eq)
qed

lemma FstOrDiEqvDi:
 "\<turnstile> (\<rhd>f \<or> di f) = di f"
proof -
 have 1: "\<turnstile> (\<rhd>f \<or> di f) = ((f \<and> bs (\<not>f)) \<or> di f)" by (simp add: first_d_def)
 have 2: "\<turnstile> ((f \<and> bs (\<not>f)) \<or> di f) = ((f \<or> di f) \<and> (bs (\<not> f) \<or> di f))" by auto
 have 3: "\<turnstile> (f \<or> di f) = di f" by (simp add: OrDiEqvDi)
 hence 4: "\<turnstile> ((f \<or> di f) \<and> (bs (\<not> f) \<or> di f)) = (di f \<and> (bs (\<not> f) \<or> di f))" by auto
 have 5: "\<turnstile> (di f \<and> (bs (\<not> f) \<or> di f)) = di f" by auto
 from 1 2 4 5 show ?thesis by fastforce
qed

lemma FstAndDiEqvFst:
  "\<turnstile> (\<rhd>f \<and> di f) = \<rhd>f"
proof -
 have 1: "\<turnstile> (\<rhd>f \<and> di f) = ((f \<and> bs (\<not>f)) \<and> di f)" by (simp add: first_d_def)
 have 2: "\<turnstile> (f \<and> di f) = f" by (meson DiIntro Prop10 Prop11)
 hence 3: "\<turnstile> (f \<and> bs (\<not>f) \<and> di f) = (f \<and> bs (\<not>f))" by auto
 from 1 3 show ?thesis by (metis first_d_def int_iffD2 int_iffI Prop12)
qed

lemma WPrevFiniteEqv: 
 "\<turnstile> wprev(f \<and> finite) = (\<not>prev(\<not>f \<or> inf)) " 
proof -
 have 1: "\<turnstile> (\<not> (f \<and> \<not> inf)) = (\<not>f \<or> inf)"
    by fastforce
 show ?thesis unfolding wprev_d_def finite_d_def using 1
  by (metis int_simps(10) inteq_reflection)
qed


lemma FPrevEqv: 
 "\<turnstile> f\<frown>skip = (more \<and>  wprev(f \<and> finite))" 
proof -
 have 1:  "\<turnstile> wprev(f \<and> finite) = (empty \<or> f\<frown>skip)"
   using WPrevAndFiniteEqv by blast
 have 2: "\<turnstile> (more \<and>  wprev(f \<and> finite)) = f\<frown>skip"
  using 1  unfolding empty_d_def 
  by (simp add: Prop01 Prop03 Prop11 Prop12 SChopSkipImpMore)
 show ?thesis
 using "2" by auto
qed


lemma DiEqvDiFst:
 "\<turnstile> finite \<longrightarrow> di (f) = di (\<rhd> f)"
proof -
 have 1: "\<turnstile>  di (\<rhd> f) = di (f \<and> bs (\<not>f)) " 
      by (simp add: first_d_def)
 have 2: "\<turnstile> di (f \<and> bs (\<not>f)) \<longrightarrow> di f \<and> di (bs (\<not> f))" 
      using DiAndImpAnd by auto
 hence 3: "\<turnstile> di (f \<and> bs (\<not>f)) \<longrightarrow> di f" 
      by auto
 have 03: "\<turnstile> di (f \<and> bs (\<not>f)) \<longrightarrow> di (f) "
    by (metis AndDiEqv DiImpDi Prop11 Prop12 inteq_reflection)
 have 4: "\<turnstile> di (\<rhd> f) \<longrightarrow> di (f)" using "1" "03" 
      by fastforce

 have 5: "\<turnstile> (di (f) \<and> empty) = (f \<and> empty)" 
   by (simp add: DiAndEmptyEqvAndEmpty)
 have 6: "\<turnstile> (\<rhd> f \<and> empty) = (f \<and> empty) " 
      using FstAndEmptyEqvAndEmpty by auto
 have 7: "\<turnstile> di (f) \<and> empty \<longrightarrow> \<rhd>f" 
      using "5" "6" by fastforce
 have 8: "\<turnstile> \<rhd>f \<longrightarrow> di (\<rhd> f)" 
      using DiIntro by auto
 have 9: "\<turnstile>  di (f) \<and> empty \<longrightarrow> di (\<rhd> f)" 
      using "7" "8" using lift_imp_trans by blast
 hence 10: "\<turnstile> empty \<longrightarrow> (di (f) \<longrightarrow> di (\<rhd> f)) " 
      by auto

 have 11: "\<turnstile>  ( di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip \<longrightarrow> more \<and> finite" 
    by (metis FstImpFinite FstMoreEqvSkip Prop12 SChopImpFinite SChopSkipImpMore inteq_reflection)
 have 12: "\<turnstile> more \<and> finite \<longrightarrow>
             (  ( di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip = 
             ( ( di (f))\<frown>skip \<longrightarrow> (di (\<rhd>f))\<frown>skip )) " 
    by (simp add: MoreImpImpSChopSkipEqv)
 have 13: "\<turnstile> (more \<and> finite \<and> ( di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip ) = 
             (more \<and> finite \<and> ((di (f))\<frown>skip \<longrightarrow> (di (\<rhd>f))\<frown>skip))" 
      using "12" by fastforce
 have 14: "\<turnstile>  ( di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip = 
             (more \<and> finite \<and> ((di (f))\<frown>skip \<longrightarrow> (di (\<rhd>f))\<frown>skip))" 
      using "11" "13" by fastforce
 have 15: "\<turnstile> (di (f) \<and> finite) = ((f \<and> finite) \<or> (ds f \<and> finite))" 
      using OrDsEqvDi by (metis (no_types, opaque_lifting)   int_eq  lift_and_com)
 have 16: "\<turnstile> di (f) = (di (f) \<and> (bs (\<not> f) \<or> \<not>(bs (\<not> f))))  " 
      by auto
 have 17: "\<turnstile> (di (f) \<and> (bs (\<not> f) \<or> \<not>(bs (\<not> f)))) = 
             ((di (f) \<and> bs (\<not> f)) \<or> (di (f) \<and> \<not>(bs (\<not> f)))) " 
      by auto
 have 171: "\<turnstile> bs (\<not> f) \<longrightarrow> finite" 
    using BsEqvBiMoreImpChop by fastforce
 have 18: "\<turnstile> (di (f) \<and> bs (\<not> f)) = ((f \<or> ds f) \<and> bs (\<not> f))" 
      using "15" 171 by fastforce
 have 19: "\<turnstile> ((f \<or> ds f) \<and> bs (\<not> f)) = ((f \<and> bs (\<not> f)) \<or> (ds f \<and> bs (\<not> f))) " 
      by auto
 have 20: "\<turnstile> \<not>(ds f \<and> bs (\<not> f))" 
      by (simp add: ds_d_def)
 have 21: "\<turnstile> ((f \<and> bs (\<not> f)) \<or> (ds f \<and> bs (\<not> f))) = (f \<and> bs (\<not> f))" 
      using "20" by auto
 have 22: "\<turnstile> (di (f) \<and> bs (\<not> f)) = (f \<and> bs (\<not> f))" 
      using "18" "19" "21" by fastforce 
 have 23: "\<turnstile> (f \<and> bs (\<not> f)) = \<rhd>f" 
      by (simp add: first_d_def)
 have 24: "\<turnstile> (\<rhd>f) \<longrightarrow> di (\<rhd>f)" 
      using DiIntro by auto
 have 25: "\<turnstile> (f \<and> bs (\<not> f)) \<longrightarrow> di (\<rhd>f)" 
      using "23" "24" by fastforce
 have 26: "\<turnstile> (di (f) \<and> bs (\<not> f)) \<longrightarrow> di (\<rhd>f)" 
      using "25" "22" by fastforce
 hence 27: "\<turnstile> (di (f) \<and> bs (\<not> f) \<and> ( (di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip )) \<longrightarrow> di (\<rhd>f) " 
      by auto

 have 28: "\<turnstile> (di (f) \<and> \<not>(bs (\<not> f))) = (di (f) \<and> ds f) " 
      by (simp add: ds_d_def)
 hence 29: "\<turnstile> (di (f) \<and> \<not>(bs (\<not> f)) \<and> ( (di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip ))  = 
              (di (f) \<and> ds f \<and> finite \<and> ( (di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip ))" 
      using "11" by fastforce
 have 030: "\<turnstile> finite \<longrightarrow> (di f)\<frown>skip = (di f);skip   " 
     by (auto simp add: Valid_def itl_defs) 
 have 031: "\<turnstile> finite \<longrightarrow> (di f);skip = (di (f \<and> finite));skip " 
     by (auto simp add: Valid_def itl_defs) 
 have 032: "\<turnstile> finite \<longrightarrow> (di f)\<frown>skip = (di(f \<and> finite));skip " 
      by (auto simp add: Valid_def itl_defs) 
 have 30: "\<turnstile>  ds f = (finite \<longrightarrow> prev(di f))" 
      using DsDi 030 unfolding prev_d_def by fastforce
 have 033: "\<turnstile> ds f = (finite \<longrightarrow> (di (f))\<frown>skip) " 
    by (simp add: DsDi)
 hence 31: "\<turnstile> (di (f) \<and> ds f \<and> finite \<and> ( (di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip)) =  
              (di (f) \<and> finite \<and> ( (di (f))\<frown>skip) \<and>  ( (di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip))" 
   by auto
 have 32: "\<turnstile>  ( di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip \<longrightarrow>  ((di (f))\<frown>skip \<longrightarrow> (di (\<rhd>f))\<frown>skip ) " 
      using "14" by auto
 hence 33: "\<turnstile> di (f) \<and> finite \<and> ((di (f))\<frown>skip ) \<and>  ( di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip \<longrightarrow> 
              di (f) \<and> finite \<and> ((di (f))\<frown>skip) \<and> ((di (f))\<frown>skip \<longrightarrow> (di (\<rhd>f))\<frown>skip )" 
       by auto
 have 34: "\<turnstile> di (f) \<and> finite \<and> ((di (f))\<frown>skip) \<and> 
            ((di (f))\<frown>skip \<longrightarrow> (di (\<rhd>f))\<frown>skip) \<longrightarrow> (di (\<rhd>f))\<frown>skip " 
      by fastforce
 have 36: "\<turnstile> (di (\<rhd>f))\<frown>skip \<longrightarrow> di(di (\<rhd>f))" 
      by (metis AndChopA ChopImpDi lift_imp_trans schop_d_def)
 have 37: "\<turnstile> di(di (\<rhd>f)) = di (\<rhd>f)" 
      using DiEqvDiDi  by fastforce
 have 38: "\<turnstile> di (f) \<and> finite \<and> (di f)\<frown>skip \<and> ((di (f))\<frown>skip \<longrightarrow> (di (\<rhd>f))\<frown>skip) \<longrightarrow> di (\<rhd>f)" 
      using "37" "36" 
      by (metis  "34"  inteq_reflection lift_imp_trans )
 have 39: "\<turnstile> di (f) \<and> finite \<and> \<not>(bs (\<not> f)) \<and> ( (di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip ) \<longrightarrow> di (\<rhd>f)" 
      using "29" "31" "33" "38" "36" "37" by fastforce
 hence 40: "\<turnstile> \<not>(bs (\<not> f)) \<and> finite \<and>( (di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip )   \<longrightarrow> (di (f) \<longrightarrow> di (\<rhd>f))" 
      by fastforce
 have 41: "\<turnstile> bs (\<not> f) \<and> ( (di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip  ) \<longrightarrow>(di (f)  \<longrightarrow> di (\<rhd>f))" 
      using "27" by fastforce
 have 42: "\<turnstile> (\<not>(bs (\<not> f)) \<or> bs (\<not> f)) \<and> ( (di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip ) \<longrightarrow>(di (f)  \<longrightarrow> di (\<rhd>f)) " 
      using "40" "41" "29" by fastforce
 have 43: "\<turnstile> (\<not>(bs (\<not> f)) \<or> bs (\<not> f))" 
      by auto
 have 44: "\<turnstile> ( (di (f) \<longrightarrow> di (\<rhd> f))\<frown>skip ) \<longrightarrow> (di (f)  \<longrightarrow> di (\<rhd>f))" 
      using "42" "43" by fastforce
 have 45: "\<turnstile> finite \<longrightarrow> di (f)  \<longrightarrow> di (\<rhd>f) " 
      using "10" "44" EmptySChopSkipInductB[of "LIFT di (f) \<longrightarrow> di (\<rhd> f)" ] 
       unfolding prev_d_def schop_d_def  by fast
 from 4 45 show ?thesis by fastforce
qed

lemma FiniteImpDiEqvImpDfEqv: 
assumes "\<turnstile> finite \<longrightarrow> di f = di g"
shows "\<turnstile> df f = df g " 
proof -
 have 1: "\<turnstile> (di f \<and> finite) = (di g \<and> finite) " 
   using assms by fastforce
 have 2: "\<turnstile> (di f \<and> finite) = f \<frown> finite " 
   unfolding di_d_def 
   by (simp add: ChopTrueAndFiniteEqvAndFiniteChopFinite schop_d_def) 
 have 3: "\<turnstile> (di g \<and> finite) = g \<frown> finite " 
   unfolding di_d_def 
   by (simp add: ChopTrueAndFiniteEqvAndFiniteChopFinite schop_d_def) 
 have 4: "\<turnstile> ((f \<frown> finite) = (g \<frown> finite)) \<longrightarrow> (f\<frown>#True = g\<frown>#True) "
    by (metis (no_types, lifting) "1" "2" "3" Prop11 SChopAssoc SChopFiniteEqvSChopTrueAndFinite 
        TrueEqvTrueSChopTrue int_simps(1) inteq_reflection schop_d_def)
 show ?thesis unfolding df_d_def by (metis "1" "2" "3" "4" MP inteq_reflection)
qed

lemma DfEqvDfFst: 
 "\<turnstile>    df f = df (\<rhd> f)"
by (simp add: DiEqvDiFst FiniteImpDiEqvImpDfEqv)


lemma FstDiEqvFst:
 "\<turnstile> \<rhd>(di f) = \<rhd>f"
proof -
 have 1: "\<turnstile> \<rhd>(di f) = (di f \<and> bs (\<not> (di f)))" by (simp add: first_d_def)
 have 2: "\<turnstile>  (\<not> (di f)) = bi (\<not> f)  " by (simp add: NotDiEqvBiNot)
 hence 3: "\<turnstile> bs (\<not> (di f)) = bs (bi (\<not> f))" using BsEqvRule by blast
 have 4: "\<turnstile> bs (bi (\<not> f)) = bs ( \<not> f)" using BsEqvBsBi by fastforce
 hence 5: "\<turnstile> (di f \<and> bs (\<not> (di f))) = (di f \<and> bs ( \<not> f))" using "3" by fastforce
 have 05: "\<turnstile> bs (\<not> (di f)) \<longrightarrow> finite" using BsEqvBiMoreImpChop by fastforce
 have 06: "\<turnstile> bs ( \<not> f) \<longrightarrow> finite" using "05" "3" "4" by fastforce
 have 6 : "\<turnstile> (di f \<and> finite)= ((f \<or> ds f) \<and> finite)" using OrDsEqvDi by (metis FiniteOr inteq_reflection)
 have 07: "\<turnstile> (di f \<and> bs ( \<not> f)) = (di f \<and> finite \<and> bs ( \<not> f)) " using "06" by fastforce
 have 08: "\<turnstile>  (di f \<and> finite \<and> bs ( \<not> f)) = ((f \<or> ds f) \<and> finite \<and> bs ( \<not> f)) " using 6 by fastforce
 have 09: "\<turnstile> ((f \<or> ds f) \<and> finite \<and> bs ( \<not> f)) = ((f \<or> ds f) \<and>  bs ( \<not> f)) " using "06" by fastforce
 have 7: "\<turnstile> (di f \<and> bs ( \<not> f)) = ((f \<or> ds f) \<and> bs ( \<not> f)) " by (metis "07" "08" "09" inteq_reflection)
 have 8: "\<turnstile> ((f \<or> ds f) \<and> bs ( \<not> f)) = ((f \<and> bs ( \<not> f)) \<or> (ds f \<and> bs ( \<not> f)))" by auto
 have 9: "\<turnstile> \<not>(ds f \<and> bs ( \<not> f)) " by (simp add: ds_d_def)
 have 10: "\<turnstile> (f \<and> bs ( \<not> f)) = \<rhd>f" by (simp add: first_d_def)
 have 11: "\<turnstile> ((f \<and> bs ( \<not> f)) \<or> (ds f \<and> bs ( \<not> f))) = \<rhd>f" using "9" "10" by fastforce
 from 1 5 7 8 11 show ?thesis by (metis int_eq)
qed

lemma FstDfEqvFst:
 "\<turnstile> \<rhd>(df f) = \<rhd>f"
by (metis (no_types, opaque_lifting) DfEqvDfFst DfEqvDiAndFinite FstDiEqvFst FstFstAndEqvFstAnd
    FstImpFinite Prop10 int_eq)

lemma DiAndFstOrEqvFstOrDiAnd:
 "\<turnstile> (di f \<and> (\<rhd>f \<or> g)) = (\<rhd>f \<or> (di f \<and> g))"
proof -
 have 1: "\<turnstile> (di f \<and> (\<rhd>f \<or> g)) = (\<rhd>f \<and> di f) \<or> (di f \<and> g)" by auto
 have 2: "\<turnstile> (\<rhd>f \<and> di f) = \<rhd>f"  using FstAndDiEqvFst by blast
 from 1 2 show ?thesis by auto
qed

lemma DiOrFstAndEqvDi:
 "\<turnstile> di f \<or> (\<rhd>f \<and> g) = di f"
proof -
 have 1: "\<turnstile> (di f \<or> (\<rhd>f \<and> g)) = ((\<rhd>f \<or> di f) \<and> (di f \<or> g)) " by auto
 have 2: "\<turnstile> (\<rhd>f \<or> di f) = di f" using FstOrDiEqvDi by blast
 from 1 2 show ?thesis by auto
qed

lemma FstDiAndDiEqv:
 "\<turnstile> \<rhd>(di f \<and> di g) = ((\<rhd>f \<and> di g) \<or> (\<rhd>g \<and> di f))"
proof -
 have 1: "\<turnstile> \<rhd>(di f \<and> di g) = ((di f \<and> di g) \<and> bs (\<not>(di f \<and> di g)))" by (simp add: first_d_def)
 have 2: "\<turnstile> (\<not>(di f \<and> di g)) = (bi (\<not>f) \<or> bi (\<not>g)) " by (auto simp add: bi_d_def) 
 hence 3: "\<turnstile> bs (\<not>(di f \<and> di g)) = bs(bi (\<not>f) \<or> bi (\<not>g))" using BsEqvRule by blast
 hence 4: "\<turnstile>  ((di f \<and> di g) \<and> bs (\<not>(di f \<and> di g))) =
              (di f \<and> di g \<and> bs(bi (\<not>f) \<or> bi (\<not>g))) " by auto
 have 5: "\<turnstile> (bs (\<not>f) \<or> bs (\<not>g)) = bs(bi (\<not>f) \<or> bi (\<not>g))" using BsOrBsEqvBsBiOrBi by blast
 hence 6: "\<turnstile> (di f \<and> di g \<and> bs(bi (\<not>f) \<or> bi (\<not>g))) =
             (di f \<and> di g \<and>(bs (\<not>f) \<or> bs (\<not>g)))" by auto
 have 7: "\<turnstile> (di f \<and> di g \<and>(bs (\<not>f) \<or> bs (\<not>g))) = 
            ((  bs (\<not>f) \<and> di f \<and> di g) \<or> (di f \<and> bs (\<not>g) \<and> di g  ))" by auto
 have 8: "\<turnstile> \<rhd>f =  (bs (\<not>f)  \<and> di f)" using FstEqvBsNotAndDi by blast
 hence 9: "\<turnstile> (bs (\<not>f) \<and> di f \<and> di g) = (\<rhd>f \<and> di g)" by auto
 have 10: "\<turnstile> \<rhd>g =  (bs (\<not>g)  \<and> di g)" using FstEqvBsNotAndDi by blast
 hence 11: "\<turnstile> (di f \<and> bs (\<not>g) \<and> di g) = (di f \<and> \<rhd>g)" by auto
  show ?thesis
  by (metis "11" "4" "6" "7" "9" first_d_def inteq_reflection lift_and_com)
qed

lemma BiNotFstEqvBiNot: 
 "\<turnstile> finite \<longrightarrow> bi (\<not> (\<rhd>f)) = bi (\<not> f)"
proof -
 have 1: "\<turnstile> finite \<longrightarrow>di f = di (\<rhd>f)" using DiEqvDiFst by blast
 hence 2: "\<turnstile> finite \<longrightarrow>(\<not>(di f)) = (\<not>( di (\<rhd>f)))" by auto
 from 1 2 show ?thesis using NotDiEqvBiNot by fastforce
qed

lemma BsNotFstEqvBsNot:
 "\<turnstile> bs (\<not> (\<rhd>f)) = bs (\<not> f)"
proof -
 have 1: "\<turnstile> bs (\<not> (\<rhd>f)) = (empty \<or> bi (\<not> (\<rhd>f))\<frown>skip) " by (simp add: bs_d_def)
 have 2: "\<turnstile> finite\<longrightarrow>bi (\<not> (\<rhd>f)) = bi (\<not> f) " using BiNotFstEqvBiNot  by fastforce
 hence 3: "\<turnstile> bi (\<not> (\<rhd>f))\<frown>skip = bi (\<not> f)\<frown>skip" using LeftSChopEqvSChop 
 by (simp add: FiniteImpAnd LeftChopEqvChop schop_d_def)
 hence 4: "\<turnstile> (empty \<or> bi (\<not> (\<rhd>f))\<frown>skip) = (empty \<or> bi (\<not> f)\<frown>skip)" by auto
 from 1 4  show ?thesis by (metis BsEqvEmptyOrBiSChopSkip inteq_reflection)
qed

lemma FstState:
 "\<turnstile> \<rhd> (init w) = (empty \<and> init w)"
proof -
 have 1: "\<turnstile> \<rhd> (init w) = (init w \<and> bs (\<not>(init w)))" by (simp add: first_d_def)
 hence 2: "\<turnstile> \<rhd> (init w) \<longrightarrow> init w" by auto
 have 3: "\<turnstile> init w \<and> finite \<longrightarrow> bs (init w)" using StateImpBs by auto
 have 4: "\<turnstile> \<rhd> (init w) \<longrightarrow> bs (init w)" using "2" "3" 
 using FstImpFinite by fastforce
 have 5: "\<turnstile> \<rhd> (init w) \<longrightarrow> bs (\<not>(init w))" using "1" by auto
 have 6: "\<turnstile> \<rhd> (init w) \<longrightarrow> bs (init w) \<and> bs (\<not>(init w)) " using "4" "5" by fastforce
 have 7: "\<turnstile> (bs (init w) \<and> bs (\<not>(init w))) = (bs((init w) \<and> \<not>(init w))) " using BsAndEqv by blast
 have 8: "\<turnstile> ((init w) \<and> \<not>(init w)) =  #False"  by auto
 hence 9: "\<turnstile> (bs((init w) \<and> \<not>(init w))) =  bs #False" using BsEqvRule by blast
 have 10: "\<turnstile> bs #False = empty" using BsFalseEqvEmpty by auto
 have 11: "\<turnstile> \<rhd> (init w) \<longrightarrow> empty" using "10" "9" "7" "6" by fastforce
 have 12: "\<turnstile> \<rhd> (init w) \<longrightarrow> empty \<and> init w" using "11" "2" by fastforce
 have 13: "\<turnstile> empty \<and> init w \<longrightarrow> empty" by auto
 hence 14: "\<turnstile> empty \<and> init w \<longrightarrow> empty \<or> bi (\<not>(init w))\<frown>skip" by auto
 hence 15: "\<turnstile> empty \<and> init w \<longrightarrow> bs (\<not>(init w))" by (simp add: bs_d_def)
 have 16: "\<turnstile> empty \<and> init w \<longrightarrow> init w  " by auto
 have 17: "\<turnstile> empty \<and> init w \<longrightarrow> init w \<and> bs (\<not>(init w))" using "16" "15" by auto
 hence 18: "\<turnstile> empty \<and> init w \<longrightarrow> \<rhd>(init w)" by (simp add: first_d_def)
 from 12 18 show ?thesis by fastforce
qed

lemma FstStateAndBsNotEmpty:
 "\<turnstile> (\<rhd> (init w) \<and> bs (\<not> empty)) = \<rhd> (init w)"
proof -
 have 1: "\<turnstile> (\<rhd> (init w) \<and> bs (\<not> empty)) = (\<rhd> (init w) \<and> bs more) " 
  by (simp add: empty_d_def)
 have 2: "\<turnstile> (\<rhd> (init w) \<and> bs more) = (\<rhd> (init w) \<and> empty)" 
      using BsMoreEqvEmpty by fastforce
 have 3: "\<turnstile> \<rhd> (init w)  =  (empty \<and> (init w))  " 
      using FstState by blast
 hence 4: "\<turnstile> (\<rhd> (init w)  \<and> empty) =  (empty \<and> (init w)  \<and> empty) " 
      by auto
 have 5: "\<turnstile> (empty \<and> (init w)  \<and> empty) = (empty \<and> (init w))  " 
      by auto
 have 6: "\<turnstile> (empty \<and> (init w)) = \<rhd>(init w)" 
      using FstState by fastforce
 from 1 2 4 5 6 show ?thesis by fastforce
qed

lemma FstStateImpFstStateOr:
 "\<turnstile> \<rhd>(init w) \<longrightarrow> \<rhd>(init w \<or> f)"
proof -
 have 1: "\<turnstile> \<rhd>(init w) = (empty \<and> init w)" 
      using FstState by blast
 have 2: "\<turnstile>  (empty \<and> init w) =  (empty \<and> (empty \<or> bi (\<not> f)\<frown>skip) \<and> init w)" 
      by auto
 have 3: "\<turnstile>  (empty \<and> (empty \<or> bi (\<not> f)\<frown>skip) \<and> init w) = 
             (empty \<and> bs (\<not> f) \<and> init w)" 
      by (simp add: bs_d_def)
 have 4: "\<turnstile>  (empty \<and> bs (\<not> f) \<and> init w) = (empty \<and> init w \<and> bs (\<not> f)) " 
      by auto
 have 5: "\<turnstile> (empty \<and> init w)  = \<rhd> (init w) " 
      using FstState  by fastforce
 hence 6: "\<turnstile> (empty \<and> init w \<and> bs (\<not> f)) = (\<rhd> (init w) \<and> bs (\<not> f))" 
      by auto
 have 7: "\<turnstile> \<rhd> (init w) \<and> bs (\<not> f) \<longrightarrow> (\<rhd> (init w) \<and> bs (\<not> f)) \<or> (\<rhd>f \<and> bs (\<not>(init w)))" 
      by auto
 have 8: "\<turnstile> \<rhd>(init w \<or> f) = ((\<rhd> (init w) \<and> bs (\<not> f)) \<or> (\<rhd>f \<and> bs (\<not>(init w))))" 
      using FstWithOrEqv by blast
  show ?thesis using "1" "3" "8" by fastforce 
qed


lemma FstLenSame:
  "(\<forall> \<sigma>. (\<sigma> \<Turnstile>  di (\<rhd>f \<and> len(i)) \<and> di (\<rhd>f \<and> len(j))) \<longrightarrow> (i=j))"
by (simp add: DiLenFstsem FstLenSamesem)

lemma FstLenSame_1:
  "\<turnstile>  di (\<rhd>f \<and> len(i)) \<and> di (\<rhd>f \<and> len(j)) \<longrightarrow> (#i=#j)"
using FstLenSame Valid_def by fastforce

lemma FstAndLenSame:
  "(\<forall> \<sigma>. (\<sigma> \<Turnstile>  di ((\<rhd>f \<and> g1) \<and> len(i)) \<and> di ((\<rhd>f \<and> g2) \<and> len(j))) \<longrightarrow> (i=j))"
using linorder_neqE_nat by (simp  add: DiLenFstAndsem) blast 

lemma FstAndLenSame_1:
  "\<turnstile>  di ((\<rhd>f \<and> g1) \<and> len(i)) \<and> di ((\<rhd>f \<and> g2) \<and> len(j)) \<longrightarrow> (#i=#j)"
using FstAndLenSame Valid_def by fastforce

lemma FstLenSameChop:
  "(\<forall> \<sigma>. (\<sigma> \<Turnstile>  ((\<rhd>f \<and> g1) \<and> len(i));h1 \<and>  ((\<rhd>f \<and> g2) \<and> len(j));h2) \<longrightarrow> (i=j))"
proof
  fix \<sigma>
  show "(\<sigma> \<Turnstile>  ((\<rhd>f  \<and> g1) \<and> len(i));h1 \<and>  ((\<rhd>f \<and> g2) \<and> len(j));h2) \<longrightarrow> (i=j) "
  proof 
  assume 0: "(\<sigma> \<Turnstile>  ((\<rhd>f \<and> g1) \<and> len(i));h1 \<and>  ((\<rhd>f \<and> g2) \<and> len(j));h2)"
   have 1: "(\<sigma> \<Turnstile>  ((\<rhd>f \<and> g1) \<and> len(i));h1) " using "0" by auto
   have 2: "(\<sigma> \<Turnstile>  ((\<rhd>f \<and> g1) \<and> len(i));h1) \<longrightarrow>
            (\<sigma> \<Turnstile>  ((\<rhd>f \<and> g1) \<and> len(i));#True) " by (metis ChopImpDi Valid_def di_d_def unl_lift2)
   have 3: "(\<sigma> \<Turnstile>  di((\<rhd>f \<and> g1) \<and> len(i))) " using "1" "2" by (simp add: di_d_def)
   have 4: "(\<sigma> \<Turnstile>  ((\<rhd>f \<and> g2) \<and> len(j));h2) " using "0" by auto
   have 5: "(\<sigma> \<Turnstile>  ((\<rhd>f \<and> g2) \<and> len(j));h2) \<longrightarrow>
            (\<sigma> \<Turnstile>  ((\<rhd>f \<and> g2) \<and> len(j));#True) " by (metis ChopImpDi Valid_def di_d_def unl_lift2)
   have 6: "(\<sigma> \<Turnstile>  di((\<rhd>f \<and> g2) \<and> len(j))) " using "4" "5" by (simp add: di_d_def)
   have 7: "(\<sigma> \<Turnstile>  di((\<rhd>f \<and> g1) \<and> len(i)) \<and> di((\<rhd>f \<and> g2) \<and> len(j))) " using "3" "6" by auto
   thus "(i=j)" using FstAndLenSame by blast
   qed
qed

lemma FstLenSameChop_1:
  "\<turnstile>  ((\<rhd>f \<and> g1) \<and> len(i));h1 \<and>  ((\<rhd>f \<and> g2) \<and> len(j));h2 \<longrightarrow> (#i=#j)"
using FstLenSameChop Valid_def by fastforce

lemma DiImpExistsOneDiLenAndFst:
 assumes " nfinite \<sigma> "
         " (\<sigma>\<Turnstile> di f) "
   shows "(\<exists>! k. (\<sigma> \<Turnstile> di(  \<rhd>f \<and> len(k))))"
proof -
  have 1: "(\<sigma> \<Turnstile> di (\<rhd> f)) " 
       using assms DiEqvDiFst  
       by (metis (no_types, lifting) FiniteImpAnd finite_defs  inteq_reflection unl_lift2)
  have 2: "(\<sigma> \<Turnstile> \<rhd> f) = ( (\<sigma>\<Turnstile>\<rhd> f) \<and> (\<exists>k. (\<sigma> \<Turnstile> len(k))))  " 
       using AndExistsLen[of "TEMP \<rhd> f"] 
       using assms len_defs nfinite_conv_nlength_enat by blast
  have 3: "((\<sigma>\<Turnstile>\<rhd> f) \<and> (\<exists>k. (\<sigma> \<Turnstile> len(k)))) =
           (\<exists>k.  (\<sigma>\<Turnstile>\<rhd> f) \<and>  (\<sigma> \<Turnstile> len(k))) " 
       by auto
  have 4: "(\<sigma> \<Turnstile> di(\<rhd> f)) = (\<exists>k. (\<sigma> \<Turnstile> di(\<rhd>f \<and> len(k))))" 
   using assms by (metis  DiLensem itl_defs(10))
  have 5: "(\<exists>k. (\<sigma> \<Turnstile> di(\<rhd>f \<and> len(k)))) " 
       using "1" using "4" by auto
  then obtain i where 6: "(\<sigma> \<Turnstile> di(\<rhd>f \<and> len(i))) " by blast
  from 5 obtain j where 7: "(\<sigma> \<Turnstile> di(\<rhd>f \<and> len(j))) " by blast
  have 8: "(\<sigma> \<Turnstile> di(\<rhd>f \<and> len(i))) \<and> (\<sigma> \<Turnstile> di(\<rhd>f \<and> len(j))) " 
       using "6" "7" by auto
  hence 9: "(\<sigma> \<Turnstile> di(\<rhd>f \<and> len(i)) \<and> di(\<rhd>f \<and> len(j)))  " 
       by simp
  hence 10: "i=j " 
       using FstLenSame by blast
  have 11: "\<And>j. (\<sigma> \<Turnstile> di(\<rhd>f \<and> len(j))) \<longrightarrow> (j=i) " 
       using "9" "10" using FstLenSame by auto
  thus " (\<exists>!k. (\<sigma> \<Turnstile> di (\<rhd> f \<and> len k)) )"  
       using "11" "5" by blast  
qed


lemma DiImpExistsOneDiLenAndFst_1:
 "\<turnstile> finite \<longrightarrow> di f \<longrightarrow> (\<exists>! k. ( di( \<rhd>f \<and> len(k))))"
using  DiImpExistsOneDiLenAndFst unfolding Valid_def finite_defs by fastforce

lemma LFstAndDist_help:
 "(\<sigma> \<Turnstile>((\<rhd>f \<and> g1) \<and> len(k));h1  \<and> ((\<rhd>f \<and> g2) \<and> len(k));h2) =
  (\<sigma> \<Turnstile>(((\<rhd>f \<and> g1) \<and> (\<rhd>f \<and> g2)) \<and> len(k));(h1 \<and> h2) )"
using LFixedAndDistr by fastforce

lemma LFstAndDist_help_1:
 "(\<exists> k. (\<sigma> \<Turnstile>((\<rhd>f \<and> g1) \<and> len(k));h1  \<and> ((\<rhd>f \<and> g2) \<and> len(k));h2)) =
            (\<exists> k. (\<sigma> \<Turnstile>(((\<rhd>f \<and> g1) \<and> (\<rhd>f \<and> g2)) \<and> len(k));(h1 \<and> h2) )) "
proof 
  assume 0: "\<exists>k. \<sigma> \<Turnstile>  ((\<rhd> f \<and> g1) \<and> len k) ; h1  \<and>  ((\<rhd> f \<and> g2) \<and> len k) ; h2"
  obtain k where 1: " \<sigma> \<Turnstile>  ((\<rhd> f \<and> g1) \<and> len k) ; h1  \<and>  ((\<rhd> f \<and> g2) \<and> len k) ; h2" 
  using "0" by auto
  hence 2: "(\<sigma> \<Turnstile>(((\<rhd>f \<and> g1) \<and> (\<rhd>f \<and> g2)) \<and> len(k));(h1 \<and> h2)) " 
  using  LFstAndDist_help by blast
  show "(\<exists> k. (\<sigma> \<Turnstile>(((\<rhd>f \<and> g1) \<and> (\<rhd>f \<and> g2)) \<and> len(k));(h1 \<and> h2) )) "  
  using "2" by auto
  next
  assume 3: "(\<exists> k. (\<sigma> \<Turnstile>(((\<rhd>f \<and> g1) \<and> (\<rhd>f \<and> g2)) \<and> len(k));(h1 \<and> h2) )) "
  obtain k where 4: "(\<sigma> \<Turnstile>(((\<rhd>f \<and> g1) \<and> (\<rhd>f \<and> g2)) \<and> len(k));(h1 \<and> h2) ) " 
  using "3" by auto
  hence 5: "(\<sigma> \<Turnstile>((\<rhd>f \<and> g1) \<and> len(k));h1  \<and> ((\<rhd>f \<and> g2) \<and> len(k));h2) " 
  using  LFstAndDist_help by blast
  show "(\<exists> k. (\<sigma> \<Turnstile>((\<rhd>f \<and> g1) \<and> len(k));h1  \<and> ((\<rhd>f \<and> g2) \<and> len(k));h2)) " 
  using "5" by auto
qed

lemma LFstAndDistrsem:
 "( (\<sigma> \<Turnstile>((\<rhd>f \<and> g1);h1 \<and> (\<rhd>f \<and> g2);h2) = (\<rhd>f \<and> g1 \<and> g2);(h1 \<and> h2)))"
proof -
  have 01: " (\<sigma> \<Turnstile>(\<rhd>f \<and> g1);h1) = (\<sigma> \<Turnstile>(\<rhd>f \<and> g1)\<frown>h1) " 
    using FstImpFinite schop_d_def 
     by (metis (no_types, lifting) FstFstAndEqvFstAnd Prop10 inteq_reflection)
  have 02: " (\<sigma> \<Turnstile>(\<rhd>f \<and> g2);h2) = (\<sigma> \<Turnstile>(\<rhd>f \<and> g2)\<frown>h2) "
      using FstImpFinite schop_d_def 
     by (metis (no_types, lifting) FstFstAndEqvFstAnd Prop10 inteq_reflection)
  have 1: "(\<sigma> \<Turnstile>(\<rhd>f \<and> g1)\<frown>h1) = (\<exists> i. (\<sigma> \<Turnstile>((\<rhd>f \<and> g1) \<and> len(i));h1) ) " 
  using AndExistsLenChop[of "TEMP (\<rhd> f \<and> g1)"] by fastforce
  have 2: "(\<sigma> \<Turnstile>(\<rhd>f \<and> g2)\<frown>h2) = (\<exists> j. (\<sigma> \<Turnstile>((\<rhd>f \<and> g2) \<and> len(j));h2) ) " 
  using AndExistsLenChop[of "TEMP (\<rhd>f \<and> g2)"] by fastforce
  have 3: "(\<sigma> \<Turnstile>(\<rhd>f \<and> g1)\<frown>h1 \<and> (\<rhd>f \<and> g2)\<frown>h2) =
          ( (\<exists> i j. (\<sigma> \<Turnstile>((\<rhd>f \<and> g1) \<and> len(i));h1  \<and> ((\<rhd>f \<and> g2) \<and> len(j));h2) )
          ) " 
  using "1" "2" by auto
  have 4: "( (\<exists> i j. (\<sigma> \<Turnstile>((\<rhd>f \<and> g1) \<and> len(i));h1  \<and> ((\<rhd>f \<and> g2) \<and> len(j));h2) )
          ) =
          ( (\<exists> k. (\<sigma> \<Turnstile>((\<rhd>f \<and> g1) \<and> len(k));h1  \<and> ((\<rhd>f \<and> g2) \<and> len(k));h2) )
          )  " 
  using FstLenSameChop by blast
  have 5: "(\<exists> k. (\<sigma> \<Turnstile>((\<rhd>f \<and> g1) \<and> len(k));h1  \<and> ((\<rhd>f \<and> g2) \<and> len(k));h2)) =
            (\<exists> k. (\<sigma> \<Turnstile>(((\<rhd>f \<and> g1) \<and> (\<rhd>f \<and> g2)) \<and> len(k));(h1 \<and> h2) ))  " 
  using LFstAndDist_help_1 by blast 
  have 6 : " (\<exists> k. (\<sigma> \<Turnstile>(((\<rhd>f \<and> g1) \<and> (\<rhd>f \<and> g2)) \<and> len(k));(h1 \<and> h2) )) =
             (\<sigma> \<Turnstile>((\<rhd>f \<and> g1) \<and> (\<rhd>f \<and> g2))\<frown>(h1 \<and> h2)) "   
  using AndExistsLenChop[of "TEMP ((\<rhd> f \<and> g1) \<and> \<rhd> f \<and> g2)"] by fastforce
  have 7 : "(\<sigma> \<Turnstile>((\<rhd>f \<and> g1) \<and> (\<rhd>f \<and> g2));(h1 \<and> h2)) =
            (\<sigma> \<Turnstile>(\<rhd>f \<and> g1 \<and> g2);(h1 \<and> h2)) " 
  by (auto simp add: chop_defs)
  have 8: " (\<sigma> \<Turnstile>(\<rhd>f \<and> g1 \<and> g2)\<frown>(h1 \<and> h2)) = (\<sigma> \<Turnstile>(\<rhd>f \<and> g1 \<and> g2);(h1 \<and> h2)) " 
     using FstImpFinite schop_d_def 
     by (metis (no_types, lifting) FstFstAndEqvFstAnd Prop10 inteq_reflection)
  have 9: "(\<sigma> \<Turnstile>(\<rhd>f \<and> g1)\<frown>h1 \<and> (\<rhd>f \<and> g2)\<frown>h2) = 
            (\<sigma> \<Turnstile>(\<rhd>f \<and> g1 \<and> g2);(h1 \<and> h2))"  
        using 01 02 3 4 5 6 7 8 
        by (metis (no_types, lifting) FstFstAndEqvFstAnd FstImpFinite Prop10 inteq_reflection schop_d_def)
  show ?thesis
  unfolding schop_d_def using "01" "02" "9" by auto
qed


lemma LFstAndDistr:
 "\<turnstile> ((\<rhd>f \<and> g1);h1 \<and> (\<rhd>f \<and> g2);h2) = (\<rhd>f \<and> g1 \<and> g2);(h1 \<and> h2) "
using  LFstAndDistrsem by fastforce

lemma LFstAndDistrA:
 "\<turnstile> ((\<rhd>f \<and> g1);h \<and> (\<rhd>f \<and> g2);h) = (\<rhd>f \<and> g1 \<and> g2);h "
proof -
 have 1: "\<turnstile> ((\<rhd>f \<and> g1);h \<and> (\<rhd>f \<and> g2);h) = (\<rhd>f \<and> g1 \<and> g2);(h \<and> h) " using LFstAndDistr by blast
 have 2: "\<turnstile> (\<rhd>f \<and> g1 \<and> g2);(h \<and> h) = (\<rhd>f \<and> g1 \<and> g2);h " by auto
 from 1 2 show ?thesis by auto
qed

lemma LFstAndDistrB:
 "\<turnstile> ((\<rhd>f \<and> g);h1 \<and> (\<rhd>f \<and> g);h2) = (\<rhd>f \<and> g);(h1 \<and> h2) " 
proof -
 have 1: "\<turnstile> ((\<rhd>f \<and> g);h1 \<and> (\<rhd>f \<and> g);h2) = (\<rhd>f \<and> g \<and> g);(h1 \<and> h2) " using LFstAndDistr by blast
 have 2: "\<turnstile> (\<rhd>f \<and> g \<and> g);(h1 \<and> h2) = (\<rhd>f \<and> g);(h1 \<and> h2) " by auto
 from 1 2 show ?thesis by auto
qed

lemma LFstAndDistrC:
  "\<turnstile> ((\<rhd>f );h1 \<and> (\<rhd>f );h2) = (\<rhd>f);(h1 \<and> h2)"
proof -
 have 1: "\<turnstile> ((\<rhd>f \<and> #True);h1 \<and> (\<rhd>f \<and> #True);h2) = (\<rhd>f \<and> #True \<and> #True);(h1 \<and> h2) " 
      using LFstAndDistr by blast
 have 2: "\<turnstile> (\<rhd>f \<and> #True);h1 = (\<rhd>f );h1 " 
      by auto
 have 3: "\<turnstile> (\<rhd>f \<and> #True);h2 = (\<rhd>f );h2 " 
      by auto
 have 4: "\<turnstile> (\<rhd>f \<and> #True \<and> #True);(h1 \<and> h2) = (\<rhd>f);(h1 \<and> h2)  " 
      by auto
 from 1 2 3 4 show ?thesis by auto
qed

lemma LFstAndDistrD:
 "\<turnstile> (di(\<rhd>f \<and> g1) \<and> di(\<rhd>f \<and> g2)) = di(\<rhd>f \<and> g1 \<and> g2) "
proof -
 have 1: "\<turnstile> ((\<rhd>f \<and> g1);#True \<and> (\<rhd>f \<and> g2);#True) = (\<rhd>f \<and> g1 \<and> g2);(#True \<and> #True) " 
      using LFstAndDistr by blast
 have 2: "\<turnstile> (\<rhd>f \<and> g1);#True = di(\<rhd>f \<and> g1) " 
      by (simp add: di_d_def)
 have 3: "\<turnstile> (\<rhd>f \<and> g2);#True = di(\<rhd>f \<and> g2) " 
      by (simp add: di_d_def)
 have 4: "\<turnstile> (\<rhd>f \<and> g1 \<and> g2);(#True \<and> #True) = di(\<rhd>f \<and> g1 \<and> g2) "  
      by (simp add: di_d_def)
 from 1 2 3 4 show ?thesis by fastforce
qed


lemma NotFstChop:
 "\<turnstile> (\<not>(\<rhd>f ;g)) = (\<not>(di (\<rhd>f)) \<or> (\<rhd>f;(\<not>g)))"
proof -
 have  1: "\<turnstile> g \<longrightarrow> #True " by auto
 hence 2: "\<turnstile> \<rhd>f;g \<longrightarrow> \<rhd>f;#True " using RightChopImpChop by blast
 hence 3: "\<turnstile> \<rhd>f;g \<longrightarrow> di(\<rhd>f)" by (simp add:di_d_def)
 hence 4: "\<turnstile> \<not>(di(\<rhd>f)) \<longrightarrow> \<not>(\<rhd>f;g) " by auto
 have  5: "\<turnstile> (\<rhd>f; (\<not>g) \<longrightarrow> \<not>(\<rhd>f;g)) = ((\<rhd>f; (\<not>g)) \<and> (\<rhd>f;g) \<longrightarrow> #False ) " by auto
 have  6: "\<turnstile> ((\<rhd>f; (\<not>g)) \<and> (\<rhd>f;g)) = \<rhd>f;(\<not>g \<and> g) " using LFstAndDistrC by blast
 have 06: "\<turnstile> \<rhd>f \<longrightarrow> finite" by (simp add: FstImpFinite)
 have  7: "\<turnstile> \<not>(\<rhd>f;(\<not>g \<and> g)) " using 06 by (auto simp add: Valid_def itl_defs)  
 have  8: "\<turnstile> \<rhd>f; (\<not>g) \<longrightarrow> \<not>(\<rhd>f;g) " using "5" "6" "7" by fastforce
 have  9: "\<turnstile> \<not>(di(\<rhd>f)) \<or> (\<rhd>f; (\<not>g)) \<longrightarrow> \<not>(\<rhd>f;g) " using "4" "8" by fastforce
 have 10: "\<turnstile> di(\<rhd>f) \<or> \<not>(di(\<rhd>f))" by auto
 hence 11: "\<turnstile> (\<rhd>f;#True) \<or> \<not>(di(\<rhd>f)) " by (simp add: di_d_def)
 hence 12: "\<turnstile> (\<rhd>f;(g \<or> \<not>g)) \<or> \<not>(di(\<rhd>f))" by  auto
 have 13: "\<turnstile> (\<rhd>f;(g \<or> \<not>g)) = ((\<rhd>f;g) \<or> (\<rhd>f;(\<not>g)))  "  using ChopOrEqv by fastforce
 have 14: "\<turnstile> ((\<rhd>f;g) \<or> (\<rhd>f;(\<not>g))) \<or> \<not>(di(\<rhd>f))  " using "12" "13" by fastforce
 hence 15: "\<turnstile> \<not>(\<rhd>f;g) \<longrightarrow> \<not>(di(\<rhd>f)) \<or> (\<rhd>f; (\<not>g))"  by auto
 from 9 15 show ?thesis by fastforce
qed

lemma BsNotFstChop:
 "\<turnstile> bs(\<not>(\<rhd>f;g)) = (empty \<or> (finite \<and> \<not>(di (\<rhd>f))) \<or>  \<rhd>f;(bs(\<not>g)))"
proof -
 have 01: "\<turnstile> (\<rhd>f;g) = (\<rhd>f\<frown>g) "
    by (simp add: FstImpFinite LeftChopEqvChop Prop10 schop_d_def)
 have 1: "\<turnstile> bs(\<not>(\<rhd>f;g))= (empty \<or> bi(\<not>(\<rhd>f\<frown>g))\<frown>skip) " 
    using 01 unfolding bs_d_def  by (metis WPrevAndFiniteEqv inteq_reflection)
 have 000: "\<turnstile> (empty \<or> bi(\<not>(\<rhd>f\<frown>g))\<frown>skip) = (empty \<or> bf(\<not>(\<rhd>f\<frown>g))\<frown>skip)"
    by (metis FiniteImpAnd FiniteImpBfEqvBi WPrevAndFiniteEqv inteq_reflection)
 have 2: "\<turnstile> (empty \<or> bf(\<not>(\<rhd>f\<frown>g))\<frown>skip) = (empty \<or> (\<not>(df(\<rhd>f\<frown>g)))\<frown>skip)" 
  by (simp add: bf_d_def)
 have 3: "\<turnstile> (empty \<or> (\<not>(df(\<rhd>f\<frown>g)))\<frown>skip) = (empty \<or> (\<not>((\<rhd>f\<frown>g)\<frown>#True))\<frown>skip)" 
      by (simp add: df_d_def)
 have 4: "\<turnstile>  (\<not>((\<rhd>f\<frown>g)\<frown>#True))\<frown>skip =  (\<not>(\<rhd>f\<frown>(g\<frown>#True)))\<frown>skip" 
    by (metis LeftSChopEqvSChop SChopAssoc int_simps(14) inteq_reflection)
 hence 5: "\<turnstile> (empty \<or> (\<not>((\<rhd>f\<frown>g)\<frown>#True))\<frown>skip) =  (empty \<or> (\<not>(\<rhd>f\<frown>(g\<frown>#True)))\<frown>skip)" 
      by auto
 have 6: "\<turnstile> (empty \<or> (\<not>(\<rhd>f\<frown>(g\<frown>#True)))\<frown>skip) = (empty \<or> (\<not>(\<rhd>f\<frown>df(g)))\<frown>skip)" 
      by (simp add: df_d_def)
 have 7: "\<turnstile> (empty \<or> (\<not>(\<rhd>f\<frown>df(g)))\<frown>skip) = (empty \<or> \<not>(\<not>((\<not>(\<rhd>f\<frown>df(g)))\<frown>skip)))  " 
      by auto
 have 8: "\<turnstile> (\<not>(\<not>((\<not>(\<rhd>f\<frown>df(g)))\<frown>skip))) =  (\<not>(empty \<or> inf \<or> (\<rhd>f\<frown>df(g))\<frown>skip )) " 
      using NotNotSChopSkip  by fastforce
 hence 9: "\<turnstile> (empty \<or> \<not>(\<not>((\<not>(\<rhd>f\<frown>df(g)))\<frown>skip))) =  (empty \<or> \<not>(empty \<or> inf \<or> (\<rhd>f\<frown>df(g))\<frown>skip ))" 
      by auto
 have 09: "\<turnstile> (\<not>(empty \<or> inf \<or> (\<rhd>f\<frown>df(g))\<frown>skip )) = (more \<and> finite \<and>  \<not>((\<rhd>f\<frown>df(g))\<frown>skip )) " 
     unfolding empty_d_def finite_d_def by fastforce
 have 10: "\<turnstile> (empty \<or> \<not>(empty \<or> inf \<or> (\<rhd>f\<frown>df(g))\<frown>skip )) = (empty \<or> (more \<and> finite \<and>  \<not>((\<rhd>f\<frown>df(g))\<frown>skip )))" 
     using 6 7 8 9 09 by auto
 have 11: "\<turnstile> (empty \<or> (more \<and> finite \<and> \<not>((\<rhd>f\<frown>df(g))\<frown>skip ))) = (empty \<or> (finite \<and> \<not>((\<rhd>f\<frown>df(g))\<frown>skip )))" 
      by (auto simp add: empty_d_def)
 have 12: "\<turnstile>  (empty \<or> (finite \<and> \<not>((\<rhd>f\<frown>df(g))\<frown>skip) )) =  (empty \<or> (finite \<and> \<not>(\<rhd>f\<frown>(df(g)\<frown>skip)) ))" 
   by (metis "11" SChopAssoc inteq_reflection)
 have 012: "\<turnstile> (\<not>(\<rhd>f\<frown>(inf \<or> df(g)\<frown>skip))) = (\<not>(\<rhd>f\<frown>inf) \<and> \<not>(\<rhd>f\<frown>(df(g)\<frown>skip))) "
     using SChopOrEqv by fastforce 
 have 013: "\<turnstile> (\<not>(\<rhd>f\<frown>inf)) = (\<not> di (\<rhd> f) \<or> \<rhd> f\<frown>(\<not> inf))" 
     by (metis FstImpFinite NotFstChop Prop10 inteq_reflection schop_d_def)
 have 014:"\<turnstile> (empty \<or> (finite \<and> \<not>(\<rhd>f\<frown>(df(g)\<frown>skip)))) = 
     (empty \<or> (finite \<and> (\<not>(di (\<rhd>f)) \<or> (\<rhd>f\<frown>(\<not>(df(g)\<frown>skip))))) ) " 
     using NotFstChop 
     by (metis FstImpFinite Prop10 int_simps(1) inteq_reflection schop_d_def)
 have 015: "\<turnstile> (\<not>(df(g)\<frown>skip)) = (empty \<or> inf \<or> bf(\<not>g)\<frown>skip) " 
     by (metis NotDfEqvBfNot NotSChopNotSkip inteq_reflection)
 have 16: "\<turnstile> (\<rhd>f\<frown>(\<not>(df(g)\<frown>skip))) = (\<rhd>f\<frown>(empty \<or> inf \<or> bf(\<not>g)\<frown>skip)) " 
   by (simp add: "015" RightSChopEqvSChop)
 have 0160: "\<turnstile> bf(\<not>g)\<frown>skip = bi(\<not>g)\<frown>skip "
     by (simp add: FiniteImpAnd FiniteImpBfEqvBi LeftChopEqvChop schop_d_def)
 have 016: "\<turnstile> (empty \<or> inf \<or> bf(\<not>g)\<frown>skip) = (inf \<or> bs (\<not>g)) " 
   unfolding bs_d_def using 0160 by fastforce
 have 017: "\<turnstile> (\<rhd>f\<frown>(\<not>(df(g)\<frown>skip))) = (\<rhd>f\<frown> (inf \<or> bs (\<not>g))) "
   by (metis "016" "16" inteq_reflection)
 have 018:"\<turnstile> (\<rhd>f\<frown> (inf \<or> bs (\<not>g))) = (\<rhd>f\<frown>inf \<or> \<rhd>f\<frown>(bs(\<not>g))) " 
   by (simp add: SChopOrEqv)
 have 019: "\<turnstile> (finite \<and> (\<not>(di (\<rhd>f)) \<or> (\<rhd>f\<frown>(\<not>(df(g)\<frown>skip))))) =
              ((finite \<and> \<not>(di (\<rhd>f))) \<or> (finite \<and> (\<rhd>f\<frown>(\<not>(df(g)\<frown>skip))))) " 
    by force
 have 020: "\<turnstile> (finite \<and> (\<rhd>f\<frown>(\<not>(df(g)\<frown>skip)))) = (finite \<and> (\<rhd>f\<frown>inf \<or> \<rhd>f\<frown>(bs(\<not>g)))) " 
    using "017" "018" by fastforce
 have 021:"\<turnstile> \<rhd>f\<frown>inf \<longrightarrow> inf"
    by (metis AndChopB FiniteChopInfEqvInf Prop11 lift_imp_trans schop_d_def)
 have 022: "\<turnstile> (finite \<and> (\<rhd>f\<frown>inf \<or> \<rhd>f\<frown>(bs(\<not>g)))) = (finite \<and> \<rhd>f\<frown>(bs(\<not>g))) " 
   using 021 unfolding finite_d_def by fastforce
 have 023: "\<turnstile> \<rhd>f\<frown>(bs(\<not>g)) \<longrightarrow> finite " 
   by (metis AndChopB EmptyImpFinite FiniteChopSkipImpFinite Prop02 SChopImpFinite bs_d_def
    lift_imp_trans schop_d_def)
 have 024: "\<turnstile> (finite \<and> \<rhd>f\<frown>(bs(\<not>g))) =  \<rhd>f\<frown>(bs(\<not>g))" 
   using 023 by fastforce
 have 025: "\<turnstile> (empty \<or> (finite \<and> (\<not>(di (\<rhd>f)) \<or> (\<rhd>f\<frown>(\<not>(df(g)\<frown>skip))))) ) =
              (empty \<or> (finite \<and> \<not>(di (\<rhd>f))) \<or>  \<rhd>f\<frown>(bs(\<not>g))) " 
  by (metis "014" "019" "020" "022" "024" inteq_reflection) 
  have 026: "\<turnstile>  bs(\<not>(\<rhd>f;g)) =  (empty \<or> (finite \<and> (\<not>(di (\<rhd>f)) \<or> (\<rhd>f\<frown>(\<not>(df(g)\<frown>skip))))) ) "
    using "014" "09" "1" "11" "12" "2" "3" "5" "6" "7" "8"
    by (metis "000" inteq_reflection)
  show ?thesis using 025 026 by (metis FstImpFinite Prop10 inteq_reflection schop_d_def)
qed

lemma FstFstChopEqvFstChopFst:
 "\<turnstile> \<rhd>(\<rhd>f;g) = \<rhd>f;\<rhd>g "
proof -
 have 1: "\<turnstile> \<rhd>(\<rhd>f;g) = ((\<rhd>f;g) \<and> bs (\<not>(\<rhd>f;g))) " 
      by (simp add: first_d_def)
 have 2: "\<turnstile> bs (\<not>(\<rhd>f;g)) = (empty \<or> (finite \<and> \<not>(di(\<rhd>f))) \<or> (\<rhd>f;bs(\<not>g))) " 
      using BsNotFstChop by auto
 hence 3: "\<turnstile> ((\<rhd>f;g) \<and> bs (\<not>(\<rhd>f;g))) = ((\<rhd>f;g) \<and> (empty \<or> (finite \<and> \<not>(di(\<rhd>f))) \<or> (\<rhd>f;bs(\<not>g))))" 
      by auto
 have 4: "\<turnstile> ((\<rhd>f;g) \<and> (empty \<or> (finite \<and> \<not>(di(\<rhd>f))) \<or> (\<rhd>f;bs(\<not>g)))) =
            (((\<rhd>f;g) \<and> empty) \<or> ((\<rhd>f;g) \<and> \<not>(di(\<rhd>f)) \<and> finite) \<or> ((\<rhd>f;g) \<and> (\<rhd>f;bs(\<not>g)))) " 
      by auto
 have 5: "\<turnstile> \<not>((\<rhd>f;g) \<and> \<not>(di(\<rhd>f)) \<and> finite)" 
      using ChopImpDi by fastforce
 hence 6: "\<turnstile> (((\<rhd>f;g) \<and> empty) \<or> ((\<rhd>f;g) \<and> \<not>(di(\<rhd>f)) \<and> finite) \<or> ((\<rhd>f;g) \<and> (\<rhd>f;bs(\<not>g)))) =
             (((\<rhd>f;g) \<and> empty) \<or> ((\<rhd>f;g) \<and> (\<rhd>f;bs(\<not>g))))" 
      by auto
 have 7: "\<turnstile>  ((\<rhd>f;g) \<and> (\<rhd>f;(bs(\<not>g)))) = ((\<rhd>f;(g \<and> (bs(\<not>g)))))" 
      using LFstAndDistrC by blast
 hence 8: "\<turnstile> (((\<rhd>f;g) \<and> empty) \<or> ((\<rhd>f;g) \<and> (\<rhd>f;(bs(\<not>g))))) = 
             (((\<rhd>f;g) \<and> empty) \<or> ((\<rhd>f;(g \<and> (bs(\<not>g)))))) " 
      by auto
 have 9: "\<turnstile> (((\<rhd>f;g) \<and> empty) \<or> ((\<rhd>f;(g \<and> (bs(\<not>g)))))) = (((\<rhd>f;g) \<and> empty) \<or> \<rhd>f;\<rhd>g)" 
      by (simp add: first_d_def)
 have 10: "\<turnstile> ((\<rhd>f;g) \<and> empty) = ((\<rhd>f;\<rhd>g) \<and> empty)" 
      using FstChopEmptyEqvFstChopFstEmpty 
      by (metis ChopEmptyAndEmpty SChopEmptyAndEmpty inteq_reflection)
 hence 11: "\<turnstile> (((\<rhd>f;g) \<and> empty) \<or> \<rhd>f;\<rhd>g) = (((\<rhd>f;\<rhd>g) \<and> empty) \<or> \<rhd>f;\<rhd>g) " 
      by auto
 have 12: "\<turnstile> (((\<rhd>f;\<rhd>g) \<and> empty) \<or> \<rhd>f;\<rhd>g) = \<rhd>f;\<rhd>g" 
      by auto
 from 1 3 4 6 8 9 11 12 show ?thesis by (metis inteq_reflection)
qed

lemma FstFixFst:
 "\<turnstile>\<rhd>(\<rhd>f) = \<rhd>f"
proof -
 have 1: "\<turnstile> \<rhd>f = (\<rhd>f);empty " using ChopEmpty by (metis int_eq) 
 hence 2: "\<turnstile>\<rhd>( \<rhd>f ) = \<rhd>((\<rhd>f);empty) " using FstEqvRule by blast
 have 3: "\<turnstile> \<rhd>((\<rhd>f);empty) = \<rhd>f;\<rhd>empty" using FstFstChopEqvFstChopFst by auto
 have 4: "\<turnstile> \<rhd>f;\<rhd>empty = \<rhd>f;empty" using FstEmpty using RightChopEqvChop by blast
 have 5: "\<turnstile> \<rhd>f;empty = \<rhd>f" using ChopEmpty by blast
 from 2 3 4 5 show ?thesis by fastforce
qed

lemma FstCSEqvEmpty:
 "\<turnstile> \<rhd>(f\<^sup>\<star>) = empty"
proof -
 have 1: "\<turnstile> \<rhd>(f\<^sup>\<star>) = \<rhd>(empty \<or> ((f \<and> more);f\<^sup>\<star>))" using ChopstarEqv FstEqvRule by blast
 from 1 show ?thesis using FstEmptyOrEqvEmpty by fastforce
qed

lemma FstIterFixFst:
 "\<turnstile> fpower (\<rhd> f) n = \<rhd>(fpower (\<rhd> f) n)"
proof 
 (induct n)
 case 0
 then show ?case 
 proof -
  have 1: "\<turnstile> fpower (\<rhd> f) 0 = empty" 
  by (simp add: fpower_d_def)
  have 2: "\<turnstile> empty = \<rhd> empty" using FstEmpty by auto
  have 3: "\<turnstile> \<rhd> empty = \<rhd>(fpower (\<rhd>f) 0)" 
  by (metis "1" "2" inteq_reflection)
  from 1 2 3 show ?thesis 
  by fastforce
 qed
 next
  case (Suc n)
  then show ?case 
  proof -
   have 4: "\<turnstile> (fpower (\<rhd> f) (Suc n)) = (\<rhd> f) ;(fpower (\<rhd>f) n) " 
     by (metis FstImpFinite LeftChopEqvChop Prop10 Prop11 fpower_d_def wpow_Suc)
   have 5: "\<turnstile> (\<rhd> f) ;(fpower (\<rhd>f) n) = (\<rhd> f) ; \<rhd> (fpower (\<rhd> f) n) " 
   using RightChopEqvChop Suc.hyps by blast
   have 6: "\<turnstile> (\<rhd> f) ; \<rhd> (fpower (\<rhd> f) n) = \<rhd>(\<rhd>f;(fpower (\<rhd> f) n))" 
   using FstFstChopEqvFstChopFst by fastforce
   have 7: "\<turnstile> \<rhd>(\<rhd>f;(fpower (\<rhd> f) n)) = \<rhd>(fpower (\<rhd> f) (Suc n))" 
      by (metis "4" "6" inteq_reflection)
   from 4 5 6 7 show ?thesis by fastforce
  qed
qed 

lemma DsImpNotFst:
 "\<turnstile> ds f \<longrightarrow> (\<not>(\<rhd>f))"
proof -
 have 1: "\<turnstile> (ds f \<and> \<rhd>f) = (ds f \<and> (f \<and> bs (\<not> f)))" by (simp add: first_d_def)
 have 2: "\<turnstile> (ds f \<and> (f \<and> bs (\<not> f))) = (ds f \<and> f \<and> \<not>(ds f)) " using NotDsEqvBsNot by fastforce
 from 1 2 show ?thesis by fastforce
qed

lemma FstLenAndEqvLenAnd:
 "\<turnstile> \<rhd>(len(k) \<and> f) = (len(k) \<and> f)"
proof -
 have 1: "\<turnstile> len(k) \<and> f \<and> ds(len(k) \<and> f) \<longrightarrow> ds (len(k))  " 
      using DsAndImpElimL by fastforce
 have 01:"\<turnstile> ds(len(k) \<and> f) = (finite \<longrightarrow> di(len(k) \<and> f)\<frown>skip) " 
   by (simp add: DsDi)
 have 02:"\<turnstile> len k \<longrightarrow> finite "
   by (simp add: len_k_finite)
 have 03:"\<turnstile> di(len(k) \<and> f)\<frown>skip \<longrightarrow> (di(len(k)))\<frown>skip  " 
      by (simp add: DiAndA LeftSChopImpSChop)
 hence 2: "\<turnstile> len(k) \<and> f \<and> ds(len(k) \<and> f) \<longrightarrow> (di(len(k)))\<frown>skip " 
      using 01 02 03 by fastforce
 hence 3: "\<turnstile> len(k) \<and> f \<and> ds(len(k)\<and> f) \<longrightarrow> ((len(k)\<frown>#True))\<frown>skip" 
   unfolding di_d_def 
   by (metis Prop10 inteq_reflection len_k_finite schop_d_def)
 hence 4: "\<turnstile> len(k) \<and> f \<and> ds(len(k)\<and> f) \<longrightarrow> (len(k)\<frown>(#True\<frown>skip))" 
   using SChopAssoc by fastforce
 hence 5: "\<turnstile> len(k) \<and> f \<and> ds(len(k)\<and> f) \<longrightarrow> (len(k)\<frown>(skip\<frown>finite))"
    by (metis Prop10 SkipFiniteEqvFiniteSkip WPowerstar_ext WPowerstar_skip_finite int_simps(17)
    inteq_reflection schop_d_def)
 hence 6: "\<turnstile> len(k) \<and> f \<and> ds(len(k)\<and> f) \<longrightarrow> (len(k)\<frown>(skip\<frown>finite)) \<and> len(k) " 
      by auto
 hence 7: "\<turnstile> len(k) \<and> f \<and> ds(len(k)\<and> f) \<longrightarrow> (len(k)\<frown>(skip\<frown>finite)) \<and> len(k)\<frown>empty" 
   by (metis ChopEmpty Prop12 inteq_reflection len_k_finite lift_imp_trans schop_d_def)
 hence 8: "\<turnstile> len(k) \<and> f \<and> ds(len(k)\<and> f) \<longrightarrow> (len(k)\<frown>((skip\<frown>finite) \<and> empty)) "  
      by (metis LenSChopAnd Prop11 lift_imp_trans)
 have 08: "\<turnstile> \<not>((skip\<frown>finite) \<and> empty) " 
   by (metis ChopEmptyAndEmpty FmoreEqvSkipChopFinite NotFmoreAndEmpty SChopEmptyAndEmpty 
      inteq_reflection lift_and_com)
 have 09:"\<turnstile> ((skip\<frown>finite) \<and> empty) = #False" 
    using 08    by (simp add: Prop11)
 have 010:"\<turnstile> len(k)\<frown>((skip\<frown>finite) \<and> empty) = #False " 
  by (metis "09" SChopRightFalse inteq_reflection)
 have 9: "\<turnstile> \<not>(len(k)\<frown>((skip\<frown>finite) \<and> empty))" 
    using 010     by auto 
 have 10: "\<turnstile> len(k) \<and> f \<longrightarrow> \<not>(ds(len(k)\<and> f)) " 
      using "8" "9" by fastforce
 hence 11: "\<turnstile> len(k) \<and> f \<longrightarrow> bs (\<not>(len(k)\<and> f))" 
      using NotDsEqvBsNot by fastforce
 hence 12: "\<turnstile> len(k) \<and> f \<longrightarrow> (len(k) \<and> f) \<and> bs (\<not>(len(k)\<and> f))  " 
      by auto
 hence 13: "\<turnstile> len(k) \<and> f \<longrightarrow> \<rhd>(len(k) \<and> f)" 
      by (simp add: first_d_def)
 have 14: "\<turnstile> \<rhd>(len(k) \<and> f) \<longrightarrow> len(k) \<and> f" 
      by (auto simp add: first_d_def)
 from 13 14 show ?thesis by fastforce
qed

lemma FstAndElimL:
 "\<turnstile> \<rhd>f \<longrightarrow> f"
by (auto simp add: first_d_def)

lemma FstImpNotDiChopSkip:
 "\<turnstile> \<rhd>f \<longrightarrow> \<not>(di f\<frown>skip)"
proof -
 have 1: "\<turnstile> \<rhd>f \<longrightarrow> bs (\<not> f)" by (auto simp add: first_d_def)
 hence 2: "\<turnstile> \<rhd>f \<longrightarrow> \<not>(ds f)" using NotDsEqvBsNot by fastforce
 have 3: "\<turnstile> ds f =  (finite \<longrightarrow> di f \<frown> skip)" using DsDi by blast
 hence 4: "\<turnstile> (\<not>(ds f)) = (finite \<and> \<not>(di f\<frown>skip)) " by auto
 from 2 4 show ?thesis by fastforce
qed

lemma FstImpNotDiChopSkipB:
 "\<turnstile> \<rhd>f \<longrightarrow> \<not>(di (f\<frown>skip))"
proof -
 have 1: "\<turnstile> \<rhd>f \<longrightarrow> bs (\<not> f)" 
      by (auto simp add: first_d_def)
 hence 2: "\<turnstile> \<rhd>f \<longrightarrow> \<not>(ds f)" 
      using NotDsEqvBsNot by fastforce
 have 3: "\<turnstile> ds f =  (finite \<longrightarrow> di f \<frown> skip)" 
      using DsDi by blast
 have 4: "\<turnstile> di f \<frown> skip = (f\<frown>#True)\<frown>skip" 
    by (metis ChopTrueAndFiniteEqvAndFiniteChopFinite SChopFiniteEqvSChopTrueAndFinite
   di_d_def int_simps(1) inteq_reflection schop_d_def)
 have 5: "\<turnstile>(f\<frown>#True)\<frown>skip = f\<frown>(#True\<frown>skip) " 
   by (meson Prop11 SChopAssoc)
 have 6: "\<turnstile> f\<frown>(#True\<frown>skip) = f\<frown>(skip\<frown>finite)" 
    by (metis ChopTrueAndFiniteEqvAndFiniteChopFinite DiSkipEqvMore DiamondSChopdef
        FmoreEqvSkipChopFinite RightSChopEqvSChop SkipFiniteEqvFiniteSkip di_d_def fmore_d_def
         inteq_reflection schop_d_def sometimes_d_def)
 have 7: "\<turnstile> f\<frown>(skip\<frown>finite) = (f\<frown>skip)\<frown>finite" 
      using SChopAssoc by blast
 have 8: "\<turnstile> (f\<frown>skip)\<frown>finite = (di(f\<frown>skip) \<and> finite)" 
    by (metis ChopTrueAndFiniteEqvAndFiniteChopFinite Prop11 di_d_def schop_d_def)
 have 9: "\<turnstile>  (\<not>(ds f)) = (\<not>(di(f\<frown>skip)) \<and> finite) " 
      using "3" "4" "5" "6" "7" "8" by fastforce
 from 2 9 show ?thesis by fastforce
qed

lemma FstImpDiEqv:
 "\<turnstile> \<rhd>f \<longrightarrow> (di f = f)"
proof -
 have 1: "\<turnstile> \<rhd>f  \<longrightarrow> \<not>(di f\<frown>skip)" using FstImpNotDiChopSkip by blast
 have 2: "\<turnstile> di f \<and> finite \<longrightarrow> f \<or> (di f\<frown>skip) " using DiEqvOrDiChopSkipB 
 using DiFiniteEqv by fastforce
 have 3: "\<turnstile> \<rhd>f  \<and> di f \<longrightarrow> (f \<or> (di f\<frown>skip)) \<and> \<not>(di f\<frown>skip) " using "1" "2" 
 using FstAndElimL by fastforce
 have 4: "\<turnstile> ((f \<or> (di f\<frown>skip)) \<and> \<not>(di f\<frown>skip)) = (f \<and> \<not>(di f\<frown>skip))" by auto 
 have 5: "\<turnstile>  \<rhd>f  \<and> di f \<longrightarrow> f \<and> \<not>(di f\<frown>skip) " using "3" "4" by fastforce
 hence 6: "\<turnstile>  \<rhd>f  \<and> di f \<longrightarrow> f " by fastforce
 hence 7: "\<turnstile> \<rhd>f \<longrightarrow> (di f \<longrightarrow> f)" using FstAndElimL by fastforce
 have 8: "\<turnstile> f  \<longrightarrow> di f " using DiIntro by auto
 hence 9: "\<turnstile> \<rhd>f \<longrightarrow> (f  \<longrightarrow> (di f))" by auto
 from 7 9 show ?thesis by fastforce
qed

lemma FstAndDiFstAndEqvFstAnd:
 "\<turnstile> (\<rhd>f \<and>  di(\<rhd>f \<and> g)) = (\<rhd>f \<and> g) "
proof -
 have 1: "\<turnstile> \<rhd>f \<and>  di(\<rhd>f \<and> g) \<longrightarrow> \<rhd> f " 
      by auto
 have 2: "\<turnstile> \<rhd>f \<and>  di(\<rhd>f \<and> g) \<longrightarrow> di(\<rhd>f \<and> g)" 
      by auto
 have 3: "\<turnstile> finite \<longrightarrow> di(\<rhd>f \<and> g) = ((\<rhd>f \<and> g) \<or> di((\<rhd>f \<and> g)\<frown>skip))" 
      using DiEqvOrDiChopSkipA 
      by (metis (no_types, lifting) DiEqvDiFst FstFstAndEqvFstAnd FstImpFinite Prop10 inteq_reflection schop_d_def)
 have 4: "\<turnstile> finite \<longrightarrow> di((\<rhd>f \<and> g)\<frown>skip) = ((\<rhd>f \<and> g)\<frown>skip)\<frown>finite " 
   unfolding di_d_def schop_d_def 
   using ChopTrueAndFiniteEqvAndFiniteChopFinite by fastforce 
 have 04:"\<turnstile> \<rhd>f \<longrightarrow> finite" 
   by (simp add: FstImpFinite)
 have 5: "\<turnstile> \<rhd>f \<and>  di(\<rhd>f \<and> g) \<longrightarrow> (\<rhd>f \<and> g) \<or> ((\<rhd>f \<and> g)\<frown>skip)\<frown>finite " 
      using "2" "3" "4" 04 by fastforce
 have 6: "\<turnstile> \<rhd>f \<and> g \<longrightarrow> f" 
      using FstAndElimL by fastforce
 hence 7: "\<turnstile> ((\<rhd>f \<and> g)\<frown>skip)\<frown>finite  \<longrightarrow> (f\<frown>skip)\<frown>finite" 
      by (simp add: LeftSChopImpSChop)
 hence 8: "\<turnstile> ((\<rhd>f \<and> g)\<frown>skip)\<frown>finite  \<longrightarrow> di(f\<frown>skip)" 
     by (metis AndChopA ChopImpDi lift_imp_trans schop_d_def)
 have  9: "\<turnstile> \<rhd>f \<longrightarrow> \<not>(di(f\<frown>skip))" 
      using FstImpNotDiChopSkipB by  blast
 have 10: "\<turnstile> \<rhd>f \<and>  di(\<rhd>f \<and> g) \<longrightarrow> ((\<rhd>f \<and> g) \<or> di(f\<frown>skip)) " 
      using "5" "8"  by fastforce
 have 11: "\<turnstile> \<rhd>f \<and>  di(\<rhd>f \<and> g) \<longrightarrow> \<not>(di(f\<frown>skip)) \<and> ((\<rhd>f \<and> g) \<or> di(f\<frown>skip))" 
      using "9" "10" "1" by fastforce
 have 12: "\<turnstile> (\<not>(di(f\<frown>skip)) \<and> ((\<rhd>f \<and> g) \<or> di(f\<frown>skip))) = (\<not>(di(f\<frown>skip)) \<and> ((\<rhd>f \<and> g))) " 
      by auto
 have 13: "\<turnstile> \<rhd>f \<and>  di(\<rhd>f \<and> g) \<longrightarrow> (\<rhd>f \<and> g)"  
      using "11" "12" by auto
 have 14: "\<turnstile> (\<rhd>f \<and> g) \<longrightarrow> \<rhd>f" 
      by auto
 hence 15: "\<turnstile> (\<rhd>f \<and> g) \<longrightarrow> di(\<rhd>f \<and> g ) " 
      using DiIntro by auto
 have 16: "\<turnstile> (\<rhd>f \<and> g) \<longrightarrow> \<rhd>f \<and> di(\<rhd>f \<and> g)"  
      using "14" "15" by auto
 from 13 16 show ?thesis by fastforce
qed

lemma FstAndDiImpBsNotAndDi:
 "\<turnstile> (\<rhd>f \<and> di g) \<longrightarrow> (bs (\<not>(di f \<and> g))) "
proof -
 have 1: "\<turnstile> (\<rhd>f \<and> di g) \<and> \<not>(bs (\<not>(di f \<and> g))) \<longrightarrow> ds(di f \<and> g)  " 
      by (auto simp add: ds_d_def)
 hence 2: "\<turnstile> (\<rhd>f \<and> di g) \<and> \<not>(bs (\<not>(di f \<and> g))) \<longrightarrow> ds(di f)  " 
      using DsAndImp by fastforce
 hence 3: "\<turnstile> (\<rhd>f \<and> di g) \<and> \<not>(bs (\<not>(di f \<and> g))) \<longrightarrow> di(di f)\<frown>skip" 
      using DsDi FstImpFinite by fastforce
 hence 4: "\<turnstile> (\<rhd>f \<and> di g) \<and> \<not>(bs (\<not>(di f \<and> g))) \<longrightarrow> di f\<frown>skip" 
      using DiEqvDiDi by (metis int_eq)
 hence 5: "\<turnstile> (\<rhd>f \<and> di g) \<and> \<not>(bs (\<not>(di f \<and> g))) \<longrightarrow> ds f" 
      using DsDi by fastforce
 hence 6: "\<turnstile>  (\<rhd>f \<and> di g) \<and> \<not>(bs (\<not>(di f \<and> g))) \<longrightarrow>\<not> (\<rhd> f) " 
      using DsImpNotFst by fastforce
 from 6 show ?thesis by auto
qed
 
lemma FstFstOrEqvFstOrL:
 "\<turnstile> \<rhd>(\<rhd>f \<or> g) = \<rhd>(f \<or> g)"
proof -
 have 1: "\<turnstile> \<rhd>(f \<or> g) = ((f \<or> g) \<and> bs (\<not>(f \<or> g)))" 
      by (simp add: first_d_def)
 have 2: "\<turnstile>  (\<not>(f \<or> g)) =  (\<not>f \<and> \<not> g)" 
      by auto
 hence 3: "\<turnstile>  bs(\<not>(f \<or> g)) =  bs (\<not>f \<and> \<not> g)" 
      using BsEqvRule by blast
 have 4: "\<turnstile> bs (\<not>f \<and> \<not> g) = (bs (\<not> f) \<and> bs (\<not> g))" 
      using BsAndEqv by fastforce
 hence 5: "\<turnstile> ((f \<or> g) \<and> bs(\<not>(f \<or> g))) =  ((f \<or> g) \<and> bs (\<not> f) \<and> bs (\<not> g)) " 
      using "4" "3" by fastforce
 have 6: "\<turnstile> ((f \<or> g) \<and> bs (\<not> f) \<and> bs (\<not> g)) =
            (((f \<and> bs (\<not> f)) \<or> (g \<and> bs (\<not> f))) \<and> bs (\<not> g))" 
      by auto
 have 7: "\<turnstile> (((f \<and> bs (\<not> f)) \<or> (g \<and> bs (\<not> f))) \<and> bs (\<not> g)) =
            ((\<rhd>f \<or> (g \<and> bs (\<not> f))) \<and> bs (\<not> g)) " 
      by (simp add: first_d_def)
 have 8: "\<turnstile> ((\<rhd>f \<or> (g \<and> bs (\<not> f))) \<and> bs (\<not> g)) =
            (((\<rhd>f \<or> g) \<and> (\<rhd>f \<or> bs (\<not>f))) \<and> bs (\<not> g)) " 
      by auto
 have 9: "\<turnstile> (((\<rhd>f \<or> g) \<and> (\<rhd>f \<or> bs (\<not>f))) \<and> bs (\<not> g)) =
            (((\<rhd>f \<or> g) \<and> ((f \<and> bs (\<not> f)) \<or> bs (\<not>f))) \<and> bs (\<not> g))" 
      by (simp add: first_d_def)
 have 10: "\<turnstile> (((\<rhd>f \<or> g) \<and> ((f \<and> bs (\<not> f)) \<or> bs (\<not>f))) \<and> bs (\<not> g)) =
             ((\<rhd>f \<or> g) \<and> bs (\<not> f) \<and> bs (\<not> g))" 
      by auto
 have 11: "\<turnstile> ((\<rhd>f \<or> g) \<and> bs (\<not> f) \<and> bs (\<not> g)) = 
              ((\<rhd>f \<or> g) \<and> bs(\<not>(\<rhd>f))\<and> bs (\<not> g))  " 
      using BsNotFstEqvBsNot by fastforce
 have 12: "\<turnstile> ((\<rhd>f \<or> g) \<and> bs(\<not>(\<rhd>f))\<and> bs (\<not> g)) =
             ((\<rhd>f \<or> g) \<and> bs (\<not>(\<rhd>f) \<and> \<not> g))" 
      using BsAndEqv by fastforce
 have 13: "\<turnstile>  (\<not>(\<rhd>f) \<and> \<not> g) = (\<not>(\<rhd>f \<or> g))" 
      by auto
 hence 14: "\<turnstile> bs (\<not>(\<rhd>f) \<and> \<not> g) = bs (\<not>(\<rhd>f \<or> g))" 
      using BsEqvRule by blast
 hence 15: "\<turnstile> ((\<rhd>f \<or> g) \<and> bs (\<not>(\<rhd>f) \<and> \<not> g)) = ((\<rhd>f \<or> g) \<and> bs (\<not>(\<rhd>f \<or> g)))" 
      by auto
 have 16: " \<turnstile> ((\<rhd>f \<or> g) \<and> bs (\<not>(\<rhd>f \<or> g))) = \<rhd>(\<rhd>f \<or> g)" 
      by (simp add: first_d_def)
 from 16 15 12 11 10 9 8 7 6 5 1 show ?thesis by (metis int_eq)
qed

lemma FstFstOrEqvFstOrR:
 "\<turnstile> \<rhd>(f \<or> \<rhd>g)= \<rhd>(f \<or> g)"
proof -
 have 1: "\<turnstile> (f \<or> \<rhd>g)= (\<rhd>g \<or> f)" by auto
 hence 2: "\<turnstile> \<rhd> (f \<or> \<rhd>g)= \<rhd>(\<rhd>g \<or> f) " using FstEqvRule by blast
 have 3: "\<turnstile> \<rhd>(\<rhd>g \<or> f) = \<rhd>(g \<or> f)" using FstFstOrEqvFstOrL by blast
 have 4: "\<turnstile> (g \<or> f) = (f \<or> g)" by auto
 hence 5: "\<turnstile> \<rhd>(g \<or> f) = \<rhd>(f \<or> g)"  using FstEqvRule by blast
 from 2 3 5 show ?thesis by fastforce
qed

lemma FstFstOrEqvFstOr:
 "\<turnstile> \<rhd>(\<rhd>f \<or> \<rhd>g) = \<rhd>(f \<or> g)"
proof -
 have 1: "\<turnstile>  \<rhd>(\<rhd>f \<or> \<rhd>g) = \<rhd>(f \<or> \<rhd>g)" using FstFstOrEqvFstOrL by blast
 have 2: "\<turnstile> \<rhd>(f \<or> \<rhd>g) = \<rhd>(f \<or> g)" using FstFstOrEqvFstOrR by blast
 from 1 2 show ?thesis by fastforce
qed

lemma FstLenEqvLen:
 "\<turnstile> \<rhd>( len(k)) = len(k) "
proof -
 have 1: "\<turnstile> \<rhd>( len(k) \<and> #True) = (len(k) \<and> #True)" using FstLenAndEqvLenAnd by blast
 have 2: "\<turnstile> (len(k) \<and> #True) = len(k)" by auto
 hence 3: "\<turnstile> \<rhd>( len(k) \<and> #True) = \<rhd>(len(k)) " using FstEqvRule by blast
 from 1 2 3 show ?thesis by auto
qed

lemma FstSkip:
 "\<turnstile> \<rhd> skip = skip"
proof -
 have 1: "\<turnstile> skip = len(1)  " using LenOneEqvSkip by fastforce
 hence 2: "\<turnstile> \<rhd>skip = \<rhd> (len(1))" using FstEqvRule by blast
 have 3: "\<turnstile> \<rhd>( len(1)) = len(1)" using FstLenEqvLen by blast
 from 1 2 3 show ?thesis using LenOneEqvSkip by fastforce
qed


lemma HaltStateEqvFstFinState:
 "\<turnstile> (halt (init w) \<and> finite) = \<rhd> (fin (init w))"
proof -
 have 1: "\<turnstile> halt (init w) = \<box>(empty = (init w))" by (simp add: halt_d_def)
 have 21: "\<turnstile> (empty = (init w)) = (((empty \<longrightarrow> (init w)) \<and> ((init w) \<longrightarrow> empty)))"  
      by auto
 hence 2: "\<turnstile> \<box>(empty = (init w)) = (\<box>((empty \<longrightarrow> (init w)) \<and> ((init w) \<longrightarrow> empty)))"
      by (simp add: BoxEqvBox)
 have 3: "\<turnstile> (\<box>((empty \<longrightarrow> (init w)) \<and> ((init w) \<longrightarrow> empty))) = 
            (\<box>((empty \<longrightarrow> (init w))) \<and> \<box>((init w) \<longrightarrow> empty))" 
      by (metis "21" BoxAndBoxEqvBoxRule int_eq)
 have 4: "\<turnstile> ((init w) \<longrightarrow> empty) = (more \<longrightarrow> \<not>(init w))" 
      by (auto simp add: empty_d_def)
 hence 5: "\<turnstile> \<box> ((init w) \<longrightarrow> empty) = \<box> (more \<longrightarrow> \<not>(init w))" using BoxEqvBox by blast
 have 6: "\<turnstile> (\<box> (more \<longrightarrow> \<not>(init w)) \<and> finite) = bs(\<not>(fin(init w)))" using BoxMoreStateEqvBsFinState by blast
 have 7: "\<turnstile> \<box>((empty \<longrightarrow> (init w))) = fin(init w)" by (simp add: fin_d_def)
 have 8: "\<turnstile>  (\<box>((empty \<longrightarrow> (init w))) \<and> \<box>((init w) \<longrightarrow> empty) \<and> finite) =
             (fin(init w) \<and> bs(\<not>(fin(init w))))" using "5" "6" "7" by fastforce
 from 1 2 3 8 show ?thesis unfolding first_d_def 
 by fastforce
qed

lemma FstLenEqvLenFst:
 "\<turnstile> \<rhd>(len k ; f) = len k ; \<rhd> f"
proof -
  have 1: "\<turnstile> len k ; f = \<rhd>(len k) ; f " using FstLenEqvLen LeftChopEqvChop by fastforce
  have 2: "\<turnstile> \<rhd>(len k ; f) = \<rhd> (\<rhd>(len k) ; f) " using 1 FstEqvRule by blast
  have 3: "\<turnstile> \<rhd> (\<rhd>(len k) ; f) = \<rhd>(len k) ; \<rhd>f" using FstFstChopEqvFstChopFst by blast
  have 4: "\<turnstile> \<rhd>(len k) ; \<rhd>f = len k ; \<rhd>f" using FstLenEqvLen LeftChopEqvChop by fastforce
  from 2 3 4 show ?thesis by fastforce
qed

lemma FstNextEqvNextFst:
 "\<turnstile> \<rhd>(\<circle> f) = \<circle>( \<rhd> f)"
proof -
  have 1: "\<turnstile> \<rhd>(\<circle> f) = \<rhd>(skip ; f) " using FstEqvRule by (simp add: next_d_def)
  have 2: "\<turnstile> skip ; f = \<rhd>skip ; f " using FstSkip using LeftChopEqvChop by fastforce
  have 3: "\<turnstile> \<rhd>(skip ; f) = \<rhd> (\<rhd>skip ; f) " using 2 FstEqvRule LeftChopEqvChop by blast
  have 4: "\<turnstile> \<rhd> (\<rhd>skip ; f) = \<rhd>skip ; \<rhd>f " using 3 FstFstChopEqvFstChopFst by blast
  have 5: "\<turnstile> \<rhd>skip ; \<rhd>f = skip ; \<rhd>f " using 4 FstSkip LeftChopEqvChop by blast
  have 6: "\<turnstile> skip ; \<rhd>f  =  \<circle>( \<rhd> f)" by (simp add: next_d_def)
  from 1 2 3 4 5 6 show ?thesis by fastforce
qed

lemma FstDiamondStateEqvHalt:
 "\<turnstile> \<rhd> (\<diamond> (init w)) = (halt (init w) \<and> finite)"
proof -
  have 1: "\<turnstile> \<diamond> (init w) = \<diamond> ((init w) \<and> #True)" by simp
  have 2: "\<turnstile> fin (init w) \<frown> #True = \<diamond> ((init w) \<and> #True)" using 1 FinChopEqvDiamond 
  by (metis schop_d_def)
  have 3: "\<turnstile> fin (init w) \<frown> #True = df (fin (init w))" unfolding df_d_def by simp
  have 4: "\<turnstile> (\<diamond> (init w)) = (df (fin (init w)))" using 1 2 3 by fastforce
  have 5: "\<turnstile> \<rhd> (\<diamond> (init w)) = \<rhd> (df (fin (init w)))" using 4 FstEqvRule by blast
  hence 6: "\<turnstile> \<rhd> (\<diamond> (init w)) = \<rhd> (fin (init w))" using FstDfEqvFst by fastforce
  hence 7: "\<turnstile> \<rhd> (\<diamond> (init w)) = (halt (init w) \<and> finite)" using HaltStateEqvFstFinState by fastforce
  from 7 show ?thesis by simp
qed


lemma DiAndFiniteEqvDiFst: 
 "\<turnstile> di (f \<and> finite) = di(\<rhd> f) " 
proof -
 have 1: "\<turnstile> (\<rhd> f \<and> finite) = \<rhd> f" 
   by (meson FstImpFinite Prop10 Prop11)
 have 2: "\<turnstile> (f \<and> finite);#True = (\<rhd> f \<and> finite);#True" 
   using DfEqvDfFst FstImpFinite unfolding df_d_def schop_d_def by auto
 show ?thesis 
 by (metis "1" "2" di_d_def inteq_reflection)
qed   

lemma FstEqvFstAndFinite: 
 "\<turnstile> \<rhd> f = \<rhd> (f \<and> finite) " 
by (metis FstDfEqvFst FstDiEqvFst di_d_def inteq_reflection itl_def(15) schop_d_def)
 

lemma FstDiAndDiEqvFstDfAndDf: 
 "\<turnstile> \<rhd>(di f \<and> di g) = \<rhd>(df f \<and> df g) " 
proof -
 have 1: "\<turnstile>  \<rhd>(di f \<and> di g) = \<rhd> ((di f \<and> di g) \<and> finite) " 
   by (simp add: FstEqvFstAndFinite)
 have 2: "\<turnstile> ((di f \<and> di g) \<and> finite) = ((df f \<and> df g) \<and> finite) " 
  using DiAndFiniteEqvDfAndFinite[of f] DiAndFiniteEqvDfAndFinite[of g]
    by fastforce
 have 3: "\<turnstile> \<rhd>((di f \<and> di g) \<and> finite) = \<rhd>((df f \<and> df g) \<and> finite) "
   by (simp add: "2" FstEqvRule)
 have 4: "\<turnstile> \<rhd>((df f \<and> df g) \<and> finite) = \<rhd>(df f \<and> df g) "
  by (meson FstEqvFstAndFinite Prop11)
 show ?thesis 
 by (metis "1" "3" "4" inteq_reflection)
qed



lemma FstBoxStateEqvStateAndEmpty:
 "\<turnstile> \<rhd> (\<box> (init w)) = ((init w) \<and> empty)"
proof -
  have 1: "\<turnstile> ((init w) \<and> (\<box> (init w))\<^sup>\<star>) = \<box> (init w)" 
  using BoxCSEqvBox by blast
  have 2: "\<turnstile> \<box> (init w) = ((init w) \<and> (\<box> (init w))\<^sup>\<star>)" 
  using 1 by auto
  hence 3: "\<turnstile> \<box> (init w) = ((init w) \<and> (\<box> (init w))\<^sup>\<star>)" 
  by blast
  have 4: "\<turnstile> ((init w) \<and> empty) ; (\<box> (init w))\<^sup>\<star> = ((init w) \<and> (\<box> (init w))\<^sup>\<star>)" 
  using StateAndEmptyChop by blast
  have 5: "\<turnstile> ((init w) \<and> (\<box> (init w))\<^sup>\<star>)  =  ((init w) \<and> empty) ; (\<box> (init w))\<^sup>\<star>" 
  using 4 by fastforce
  have 6: "\<turnstile> \<box> (init w) = ((init w) \<and> empty) ; (\<box> (init w))\<^sup>\<star> " 
  using 3 5 by fastforce
  have 7: "\<turnstile> ((init w) \<and> empty) ; (\<box> (init w))\<^sup>\<star> = \<rhd> (init w) ; (\<box> (init w))\<^sup>\<star> " 
  using FstState by (metis AndChopCommute int_eq)
  have 8: "\<turnstile> \<box> (init w) = \<rhd> (init w) ; (\<box> (init w))\<^sup>\<star> " 
  using 6 7 by fastforce
  have 9: "\<turnstile> \<rhd> (\<box> (init w)) = \<rhd> (\<rhd> (init w) ; (\<box> (init w))\<^sup>\<star>) " 
  using 8 FstEqvRule by blast
  have 10: "\<turnstile> \<rhd> (\<rhd> (init w) ; (\<box> (init w))\<^sup>\<star>) = \<rhd> (init w) ;  \<rhd> ((\<box> (init w))\<^sup>\<star>) " 
  using FstFstChopEqvFstChopFst by blast
  have 11: "\<turnstile> \<rhd> (init w) ;  \<rhd> ((\<box> (init w))\<^sup>\<star>) = \<rhd> (init w) ; empty" 
  using RightChopEqvChop FstCSEqvEmpty by blast
  have 12: "\<turnstile> \<rhd> (init w) ; empty = \<rhd> (init w)" 
  using RightChopEqvChop ChopEmpty by blast
  have 13: "\<turnstile> \<rhd> (init w) = ((init w) \<and> empty)" 
  using FstState by fastforce
  from 9 10 11 12 13 show ?thesis by fastforce
qed

lemma FstAndFstStarEqvFst:
 "\<turnstile> (schopstar (\<rhd>f) \<and> (\<rhd>f)) = \<rhd>f "
proof -
  have 1: "\<turnstile> (schopstar (\<rhd>f)) = (empty \<or> (\<rhd>f)\<frown>(schopstar (\<rhd>f))) " 
    by (meson Prop11 SChopstar_unfoldl_eq)
  have 2: "\<turnstile> ((schopstar (\<rhd>f)) \<and> \<rhd>f) = ((empty \<or> (\<rhd>f)\<frown>(schopstar (\<rhd>f))) \<and> \<rhd>f)" 
       using 1 by fastforce
  have 3: "\<turnstile> ((empty \<or> (\<rhd>f)\<frown>(schopstar (\<rhd>f))) \<and> \<rhd>f) = 
             ((empty \<and> \<rhd>f) \<or> ((\<rhd>f)\<frown>(schopstar (\<rhd>f)) \<and> \<rhd>f))" 
       by auto
  have 4: "\<turnstile> ((schopstar (\<rhd>f)) \<and> \<rhd>f) = ((empty \<and> \<rhd>f) \<or> ((\<rhd>f)\<frown>(schopstar (\<rhd>f)) \<and> \<rhd>f))" 
       using 2 3 by fastforce
  have 5: "\<turnstile> ((\<rhd>f)\<frown>(schopstar (\<rhd>f)) \<and> \<rhd>f) = ((\<rhd>f)\<frown>(schopstar (\<rhd>f)) \<and> \<rhd>f\<frown>empty)" 
       by (metis ChopEmpty FstImpFinite Prop10 inteq_reflection schop_d_def)
  have 6: "\<turnstile> ((\<rhd>f)\<frown>(schopstar (\<rhd>f)) \<and> \<rhd>f\<frown>empty) = (\<rhd>f)\<frown>((schopstar (\<rhd>f)) \<and> empty)"  
       by (simp add: LFstAndDistrB schop_d_def)
  have 7: "\<turnstile> (schopstar (\<rhd>f) \<and> empty) = empty" 
   using SChopstar_imp_empty by fastforce

  have 8: "\<turnstile> (\<rhd>f)\<frown>(schopstar (\<rhd>f) \<and> empty) = \<rhd>f" 
       using 7 ChopEmpty 
       by (metis FstImpFinite Prop11 Prop12 inteq_reflection schop_d_def)
  have 9: "\<turnstile> ((\<rhd>f)\<frown>(schopstar (\<rhd>f)) \<and> \<rhd>f) = \<rhd>f" 
       using 5 6 8 by fastforce
  have 10: "\<turnstile> (schopstar (\<rhd>f) \<and> \<rhd>f) = ((empty \<and> \<rhd>f) \<or> \<rhd>f)" 
       using 4 9 by fastforce
  have 11: "\<turnstile> ((empty \<and> \<rhd>f) \<or> \<rhd>f) = \<rhd>f" 
       by auto
  have 12:  "\<turnstile> (schopstar (\<rhd>f) \<and> \<rhd>f) = \<rhd>f" 
       using 10 11 by fastforce
  from 12 show ?thesis by auto
qed

lemma HaltStateEqvFstHaltState:
   "\<turnstile> (halt(init(w)) \<and> finite) = \<rhd>(halt(init(w)))"
proof -
  have 1:  "\<turnstile> (halt (init w) \<and> finite)= \<rhd> (fin (init w))"
       by (simp add: HaltStateEqvFstFinState)
  have 2:  "\<turnstile> \<rhd> (fin (init w)) = \<rhd> (\<rhd> (fin (init w)))"
    using FstEqvRule FstFixFst by fastforce
  have 3:  "\<turnstile> \<rhd> (\<rhd> (fin (init w))) = \<rhd>(halt(init(w)) \<and> finite)"
      using FstEqvRule HaltStateEqvFstFinState by (metis inteq_reflection)
  have 4: "\<turnstile> \<rhd>(halt(init(w)) \<and> finite) = \<rhd>(halt(init(w))) " 
    by (metis FstDfEqvFst FstDiEqvFst Prop04 di_d_def itl_def(15) schop_d_def)
    from 1 2 3 4 show ?thesis by fastforce
qed


lemma DiHaltAndDiHaltAndEqvDiHaltAndAnd:
   "\<turnstile> (df(halt (init w) \<and> f) \<and> df(halt (init w) \<and> g)) = df(halt (init w) \<and> f \<and> g)"
proof -
  
  have 01: "\<turnstile> (halt (init w) \<and> f \<and> finite) = (\<rhd>(fin (init w)) \<and> f \<and> finite)" 
    using HaltStateEqvFstFinState by fastforce
  have 02: "\<turnstile> (halt (init w) \<and> g \<and> finite) = (\<rhd>(fin (init w)) \<and> g \<and> finite)" 
      using HaltStateEqvFstFinState by fastforce
  have 03: "\<turnstile> (halt (init w) \<and> f \<and> finite)\<frown>#True = (\<rhd>(fin (init w)) \<and> f \<and> finite)\<frown>#True" 
      by (simp add: "01" LeftSChopEqvSChop)
  have 04: "\<turnstile> (halt (init w) \<and> g \<and> finite)\<frown>#True = (\<rhd>(fin (init w)) \<and> g \<and> finite)\<frown>#True" 
      by (simp add: "02" LeftSChopEqvSChop)
  have 05: "\<turnstile> (\<rhd>(fin (init w)) \<and> f \<and> finite)\<frown>#True = (\<rhd>(fin (init w)) \<and> f)\<frown>#True" 
     by (metis (no_types, lifting) FstFstAndEqvFstAnd FstImpFinite LeftSChopEqvSChop Prop11 Prop12 inteq_reflection)
  have 06: "\<turnstile> (\<rhd>(fin (init w)) \<and> g \<and> finite)\<frown>#True = (\<rhd>(fin (init w)) \<and> g)\<frown>#True" 
     by (metis (no_types, lifting) FstFstAndEqvFstAnd FstImpFinite LeftSChopEqvSChop Prop11 Prop12 inteq_reflection)
  have 07: "\<turnstile> (halt (init w) \<and> f \<and> finite)\<frown>#True = (halt (init w) \<and> f)\<frown>#True  " 
     unfolding schop_d_def 
     by (metis (no_types, lifting) LeftChopEqvChop Prop11 Prop12 lift_and_com)
  have 08: "\<turnstile> (halt (init w) \<and> g \<and> finite)\<frown>#True = (halt (init w) \<and> g)\<frown>#True  " 
     unfolding schop_d_def 
     by (metis (no_types, lifting) LeftChopEqvChop Prop11 Prop12 lift_and_com)
  have 1: "\<turnstile> (df(halt (init w) \<and> f) \<and> df(halt (init w) \<and> g)) = 
             (df(\<rhd>(fin (init w)) \<and> f) \<and> df (\<rhd>(fin (init w)) \<and> g))" 
      unfolding df_d_def 
      by (metis "01" "02" "05" "06" "07" "08" inteq_reflection lift_and_com)
  have 2: "\<turnstile> (df(\<rhd>(fin (init w)) \<and> f) \<and> df(\<rhd>(fin (init w)) \<and> g)) = 
             df(\<rhd>(fin (init w)) \<and> f \<and> g)" 
       using LFstAndDistrD 
       by (metis (no_types, lifting) FstFstAndEqvFstAnd FstImpFinite Prop10 df_d_def di_d_def inteq_reflection schop_d_def)
  have 08: "\<turnstile>  (\<rhd>(fin (init w)) \<and> f \<and> g) = (halt (init w) \<and> f \<and> g \<and> finite) " 
     using HaltStateEqvFstFinState by fastforce
  have 09: "\<turnstile> (\<rhd>(fin (init w)) \<and> f \<and> g)\<frown>#True = (halt (init w) \<and> f \<and> g \<and> finite)\<frown>#True "
    by (simp add: "08" LeftSChopEqvSChop) 
  have 010: "\<turnstile> (halt (init w) \<and> f \<and> g \<and> finite)\<frown>#True = (halt (init w) \<and> f \<and> g)\<frown>#True " 
    unfolding schop_d_def 
    by (metis (no_types, lifting) LeftChopEqvChop Prop11 Prop12 lift_and_com)
  have 3: "\<turnstile> df(\<rhd>(fin (init w)) \<and> f \<and> g)  = df(halt (init w) \<and> f \<and> g)" 
     unfolding df_d_def 
         by (metis "010" "08" inteq_reflection)
  from 1 2 3 show ?thesis using int_eq by metis
qed

(* This doesn't work and is not a theorem.
      see    counter_ex_lhs and counter_ex_rhs                                                    
lemma FstAndFstChopEqvFstChopAndFstChop:
   "\<turnstile> ((\<rhd>f \<and> \<rhd>g) ; h) = ((\<rhd>f ; h) \<and> (\<rhd>g ; h))"
*)

lemma counter_ex_lhs:
 "\<turnstile> ((\<rhd>(len(5)) \<and> \<rhd>(len(2))) ; (len(5) \<or> len(2))) = #False "
proof -
 have 1: "\<turnstile> ((\<rhd>(len(5)) \<and> \<rhd>(len(2))) ; (len(5) \<or> len(2))) =
            (len(5) \<and> len(2)); (len(5) \<or> len(2))" 
       by (metis FstLenAndEqvLenAnd FstLenEqvLen LeftChopEqvChop inteq_reflection)
 have 2: "\<turnstile> (len(5) \<and> len(2)) = #False"
      by (simp add: Valid_def len_defs) 
 have 3: "\<turnstile> ((len(5) \<and> len(2)); (len(5) \<or> len(2))) = (#False;(len(5) \<or> len(2))) "  
      by (simp add: "2" LeftChopEqvChop)  
 have 4: "\<turnstile> (#False;(len(5) \<or> len(2))) = #False" 
      by (simp add: Valid_def chop_defs)
 from 1 3 4 show ?thesis by fastforce
qed

lemma counter_ex_rhs:
 "\<turnstile> ((\<rhd> (len(5)) ; (len(5) \<or> len(2))) \<and> (\<rhd> (len(2)) ; (len(5) \<or> len(2)))) = len(7)"  
proof -
 have 1: "\<turnstile> (\<rhd> (len(5)) ; (len(5) \<or> len(2))) = 
            len(5);(len(5) \<or> len(2)) " 
      using FstLenEqvLen LeftChopEqvChop by blast
 have 2: "\<turnstile> (\<rhd> (len(2)) ; (len(5) \<or> len(2))) =
            len(2) ;(len(5) \<or> len(2))" 
      using FstLenEqvLen LeftChopEqvChop by blast
 have 3: "\<turnstile> len(5);(len(5) \<or> len(2)) =
            ((len(5);len(5)) \<or> (len(5);len(2)))" 
      by (simp add: ChopOrEqv)
 have 4: "\<turnstile> ((len(5);len(5)) \<or> (len(5);len(2))) =
            (len(10) \<or> len(7))" 
      using LenEqvLenChopLen inteq_reflection by fastforce
 have 5: "\<turnstile> len(2) ;(len(5) \<or> len(2)) =
            ((len(2);len(5)) \<or> (len(2);len(2)))" 
      by (simp add: ChopOrEqv)
 have 6: "\<turnstile> ((len(2);len(5)) \<or> (len(2);len(2))) =
            (len(7) \<or> len(4))" 
      using LenEqvLenChopLen inteq_reflection by fastforce
 have 7: "\<turnstile> ((len(10) \<or> len(7)) \<and> (len(7) \<or> len(4))) = 
            ((len(7) \<or> len(10)) \<and> (len(7) \<or> len(4)))" 
      by fastforce
 have 8: "\<turnstile> ((len(7) \<or> len(10)) \<and> (len(7) \<or> len(4))) = 
             (len(7) \<or> (len(10) \<and> len(4))) "  
      by fastforce
 have 9: "\<turnstile> (len(10) \<and> len(4)) = #False" 
      by (simp add: Valid_def len_defs) 
 have 10 : "\<turnstile> (len(7) \<or> (len(10) \<and> len(4))) = len(7)" 
      using "9" by auto
 have 11: "\<turnstile> ((\<rhd> (len(5)) ; (len(5) \<or> len(2))) \<and> (\<rhd> (len(2)) ; (len(5) \<or> len(2)))) =
             (len(5);(len(5) \<or> len(2)) \<and> len(2) ;(len(5) \<or> len(2)))" 
      using "1" "2" by fastforce
 have 12: "\<turnstile> (len(5);(len(5) \<or> len(2)) \<and> len(2) ;(len(5) \<or> len(2))) = len(7) "
      by (metis "10" "4" "6" "7" "8" ChopOrEqv inteq_reflection)
 from 11 12 show ?thesis by fastforce
qed



end
