(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)
section \<open>Extended Integers\<close>

theory Extended_Int

imports
  "HOL-Library.Extended_Nat"

begin
(*
sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                    provers="cvc4 z3 e spass vampire " ]
*)

(*
declare [[show_types]]

declare [[show_consts]]
*)

datatype eint = eint int | PInfty | MInfty
(*
print_theorems
*)
lemma eint_cong:
 "x= y \<Longrightarrow> eint x = eint y " 
by simp

instantiation eint :: uminus
begin

fun uminus_eint where
 " -(eint i) = eint (- i) " 
| " -PInfty = MInfty"
| " -MInfty = PInfty " 

instance ..

end

instantiation eint :: infinity
begin

definition "(\<infinity> :: eint) = PInfty" 
instance ..

end

declare [[coercion "eint :: int \<Rightarrow> eint " ]]

lemma eint_uminus_uminus[simp]:
fixes a :: eint
shows "- ( - a) = a"
by (cases a) simp_all
 

lemma
  shows PInfty_eq_infinity[simp]: "PInfty = \<infinity>"
    and MInfty_eq_minfinity[simp]: "MInfty = - \<infinity>"
    and MInfty_neq_PInfty[simp]: "\<infinity> \<noteq> - (\<infinity>::eint)" "- \<infinity> \<noteq> (\<infinity>::eint)"
    and MInfty_neq_eint[simp]: "eint r \<noteq> - \<infinity>" "- \<infinity> \<noteq> eint r"
    and PInfty_neq_eint[simp]: "eint r \<noteq> \<infinity>" "\<infinity> \<noteq> eint r"
    and PInfty_cases[simp]: "(case \<infinity> of eint r \<Rightarrow> f r | PInfty \<Rightarrow> y | MInfty \<Rightarrow> z) = y"
    and MInfty_cases[simp]: "(case - \<infinity> of eint r \<Rightarrow> f r | PInfty \<Rightarrow> y | MInfty \<Rightarrow> z) = z"
  by (simp_all add: infinity_eint_def)


declare
  PInfty_eq_infinity[code_post]
  MInfty_eq_minfinity[code_post]

lemma [code_unfold]:
  "\<infinity> = PInfty"
  "- PInfty = MInfty"
  by simp_all

lemma inj_eint[simp]: " inj_on eint A" 
unfolding inj_on_def by auto

lemma eint_cases[cases type: eint]:
obtains (int) i where "x = eint i" 
| (PInf) " x = \<infinity>"
| (MInf) " x = -\<infinity>" 
by (cases x) auto

lemmas eint2_cases = eint_cases[case_product eint_cases]
lemmas eint3_cases = eint2_cases[case_product eint_cases]

lemma eint_all_split: "\<And>P. (\<forall>x::eint. P x) \<longleftrightarrow> P \<infinity> \<and> (\<forall>x. P (eint x)) \<and> P (-\<infinity>)"
  by (metis eint_cases)

lemma eint_ex_split: "\<And>P. (\<exists>x::eint. P x) \<longleftrightarrow> P \<infinity> \<or> (\<exists>x. P (eint x)) \<or> P (-\<infinity>)"
  by (metis eint_cases)


lemma eint_uminus_eq_iff[simp]:
  fixes a b :: eint
  shows "-a = -b \<longleftrightarrow> a = b"
  by (cases rule: eint2_cases[of a b]) simp_all

function int_of_eint :: "eint \<Rightarrow> int" where
  "int_of_eint (eint r) = r"
| "int_of_eint \<infinity> = 0"
| "int_of_eint (-\<infinity>) = 0"
  by (auto intro: eint_cases)
termination by standard  (rule wf_on_bot)

lemma int_of_eint[simp]:
  "int_of_eint (- x :: eint) = - (int_of_eint x)"
  by (cases x) simp_all

lemma range_eint[simp]: "range eint = UNIV - {\<infinity>, -\<infinity>}"
proof safe
  fix x
  assume "x \<notin> range eint" "x \<noteq> \<infinity>"
  then show "x = -\<infinity>"
    by (cases x) auto
qed auto

lemma eint_range_uminus[simp]: "range uminus = (UNIV::eint set)"
proof safe
  fix x :: eint
  show "x \<in> range uminus"
    by (intro image_eqI[of _ _ "-x"]) auto
qed auto

instantiation eint :: abs
begin

function abs_eint where
  "\<bar>eint i\<bar> = eint \<bar>i\<bar>"
| "\<bar>-\<infinity>\<bar> = (\<infinity>::eint)"
| "\<bar>\<infinity>\<bar> = (\<infinity>::eint)"
by (auto intro: eint_cases)
termination proof qed 
(rule wf_on_bot )

instance ..

end

lemma abs_eq_infinity_cases[elim!]:
  fixes x :: eint
  assumes "\<bar>x\<bar> = \<infinity>"
  obtains "x = \<infinity>" | "x = -\<infinity>"
  using assms by (cases x) auto

lemma abs_neq_infinity_cases[elim!]:
  fixes x :: eint
  assumes "\<bar>x\<bar> \<noteq> \<infinity>"
  obtains r where "x = eint r"
  using assms by (cases x) auto


lemma abs_eint_uminus[simp]:
  fixes x :: eint
  shows "\<bar>- x\<bar> = \<bar>x\<bar>"
  by (cases x) auto



lemma eint_infinity_cases:
  fixes a :: eint
  shows "a \<noteq> \<infinity> \<Longrightarrow> a \<noteq> -\<infinity> \<Longrightarrow> \<bar>a\<bar> \<noteq> \<infinity>"
  by auto


subsubsection "Addition"

instantiation eint :: "{one,comm_monoid_add,zero_neq_one}"
begin

definition "0 = eint 0"
definition "1 = eint 1"

function plus_eint where
  "eint r + eint p = eint (r + p)"
| "\<infinity> + a = (\<infinity>::eint)"
| "a + \<infinity> = (\<infinity>::eint)"
| "eint r + -\<infinity> = - \<infinity>"
| "-\<infinity> + eint p = -(\<infinity>::eint)"
| "-\<infinity> + -\<infinity> = -(\<infinity>::eint)"
proof goal_cases
  case prems: (1 P x)
  then obtain a b where "x = (a, b)"
    by (cases x) auto
  with prems show P
   by (cases rule: eint2_cases[of a b]) auto
qed auto
termination by standard (rule wf_on_bot )

lemma Infty_neq_0[simp]:
  "(\<infinity>::eint) \<noteq> 0" "0 \<noteq> (\<infinity>::eint)"
  "-(\<infinity>::eint) \<noteq> 0" "0 \<noteq> -(\<infinity>::eint)"
  by (simp_all add: zero_eint_def)

lemma eint_eq_0[simp]:
  "eint r = 0 \<longleftrightarrow> r = 0"
  "0 = eint r \<longleftrightarrow> r = 0"
  unfolding zero_eint_def by simp_all


lemma eint_eq_1[simp]:
  "eint r = 1 \<longleftrightarrow> r = 1"
  "1 = eint r \<longleftrightarrow> r = 1"
  unfolding one_eint_def by simp_all

instance
proof
  fix a b c :: eint
  show "0 + a = a"
    by (cases a) (simp_all add: zero_eint_def)
  show "a + b = b + a"
    by (cases rule: eint2_cases[of a b]) simp_all
  show "a + b + c = a + (b + c)"
    by (cases rule: eint3_cases[of a b c]) simp_all
  show "0 \<noteq> (1::eint)"
    by (simp add: one_eint_def zero_eint_def)
qed

end

lemma eint_0_plus [simp]: "eint 0 + x = x"
  and plus_eint_0 [simp]: "x + eint 0 = x"
by(simp_all flip: zero_eint_def)


instance eint :: numeral ..


lemma int_of_eint_0[simp]: "int_of_eint (0::eint) = 0"
  unfolding zero_eint_def by simp

lemma abs_eint_zero[simp]: "\<bar>0\<bar> = (0::eint)"
  unfolding zero_eint_def abs_eint.simps by simp

lemma eint_uminus_zero[simp]: "- 0 = (0::eint)"
  by (simp add: zero_eint_def)

lemma eint_uminus_zero_iff[simp]:
  fixes a :: eint
  shows "-a = 0 \<longleftrightarrow> a = 0"
  by (cases a) simp_all

lemma eint_plus_eq_PInfty[simp]:
  fixes a b :: eint
  shows "a + b = \<infinity> \<longleftrightarrow> a = \<infinity> \<or> b = \<infinity>"
  by (cases rule: eint2_cases[of a b]) auto

lemma eint_plus_eq_MInfty[simp]:
  fixes a b :: eint
  shows "a + b = -\<infinity> \<longleftrightarrow> (a = -\<infinity> \<or> b = -\<infinity>) \<and> a \<noteq> \<infinity> \<and> b \<noteq> \<infinity>"
  by (cases rule: eint2_cases[of a b]) auto

lemma eint_add_cancel_left:
  fixes a b :: eint
  assumes "a \<noteq> -\<infinity>"
  shows "a + b = a + c \<longleftrightarrow> a = \<infinity> \<or> b = c"
  using assms by (cases rule: eint3_cases[of a b c]) auto

lemma eint_add_cancel_right:
  fixes a b :: eint
  assumes "a \<noteq> -\<infinity>"
  shows "b + a = c + a \<longleftrightarrow> a = \<infinity> \<or> b = c"
  using assms by (cases rule: eint3_cases[of a b c]) auto

lemma eint_int: "eint (int_of_eint x) = (if \<bar>x\<bar> = \<infinity> then 0 else x)"
  by (cases x) simp_all

lemma int_of_eint_add:
  fixes a b :: eint
  shows "int_of_eint (a + b) =
    (if (\<bar>a\<bar> = \<infinity>) \<and> (\<bar>b\<bar> = \<infinity>) \<or> (\<bar>a\<bar> \<noteq> \<infinity>) \<and> (\<bar>b\<bar> \<noteq> \<infinity>) then int_of_eint a + int_of_eint b else 0)"
  by (cases rule: eint2_cases[of a b]) auto

subsubsection "Linear order on \<^typ>\<open>eint\<close>"

instantiation eint :: linorder
begin

function less_eint
where
  "   eint x < eint y     \<longleftrightarrow> x < y"
| "(\<infinity>::eint) < a           \<longleftrightarrow> False"
| "         a < -(\<infinity>::eint) \<longleftrightarrow> False"
| "eint x    < \<infinity>           \<longleftrightarrow> True"
| "        -\<infinity> < eint r     \<longleftrightarrow> True"
| "        -\<infinity> < (\<infinity>::eint) \<longleftrightarrow> True"
proof goal_cases
  case prems: (1 P x)
  then obtain a b where "x = (a,b)" by (cases x) auto
  with prems show P by (cases rule: eint2_cases[of a b]) auto
qed simp_all
termination by (relation "{}") simp

definition "x \<le> (y::eint) \<longleftrightarrow> x < y \<or> x = y"

lemma eint_infty_less[simp]:
  fixes x :: eint
  shows "x < \<infinity> \<longleftrightarrow> (x \<noteq> \<infinity>)"
    "-\<infinity> < x \<longleftrightarrow> (x \<noteq> -\<infinity>)"
  by (cases x, simp_all) (cases x, simp_all)

lemma eint_infty_less_eq[simp]:
  fixes x :: eint
  shows "\<infinity> \<le> x \<longleftrightarrow> x = \<infinity>"
    and "x \<le> -\<infinity> \<longleftrightarrow> x = -\<infinity>"
  by (auto simp add: less_eq_eint_def)

lemma eint_less[simp]:
  "eint r < 0 \<longleftrightarrow> (r < 0)"
  "0 < eint r \<longleftrightarrow> (0 < r)"
  "eint r < 1 \<longleftrightarrow> (r < 1)"
  "1 < eint r \<longleftrightarrow> (1 < r)"
  "0 < (\<infinity>::eint)"
  "-(\<infinity>::eint) < 0"
  by (simp_all add: zero_eint_def one_eint_def)

lemma eint_less_eq[simp]:
  "x \<le> (\<infinity>::eint)"
  "-(\<infinity>::eint) \<le> x"
  "eint r \<le> eint p \<longleftrightarrow> r \<le> p"
  "eint r \<le> 0 \<longleftrightarrow> r \<le> 0"
  "0 \<le> eint r \<longleftrightarrow> 0 \<le> r"
  "eint r \<le> 1 \<longleftrightarrow> r \<le> 1"
  "1 \<le> eint r \<longleftrightarrow> 1 \<le> r"
  by (auto simp add: less_eq_eint_def zero_eint_def one_eint_def)

lemma eint_infty_less_eq2:
  "a \<le> b \<Longrightarrow> a = \<infinity> \<Longrightarrow> b = (\<infinity>::eint)"
  "a \<le> b \<Longrightarrow> b = -\<infinity> \<Longrightarrow> a = -(\<infinity>::eint)"
  by simp_all

instance
proof
  fix x y z :: eint
  show "x \<le> x"
    by (cases x) simp_all
  show "x < y \<longleftrightarrow> x \<le> y \<and> \<not> y \<le> x"
    by (cases rule: eint2_cases[of x y]) auto
  show "x \<le> y \<or> y \<le> x "
    by (cases rule: eint2_cases[of x y]) auto
  {
    assume "x \<le> y" "y \<le> x"
    then show "x = y"
      by (cases rule: eint2_cases[of x y]) auto
  }
  {
    assume "x \<le> y" "y \<le> z"
    then show "x \<le> z"
      by (cases rule: eint3_cases[of x y z]) auto
  }
qed

end

instance eint :: ordered_comm_monoid_add
proof
  fix a b c :: eint
  assume "a \<le> b"
  then show "c + a \<le> c + b"
    by (cases rule: eint3_cases[of a b c]) auto
qed


lemma eint_one_not_less_zero_eint[simp]: "\<not> 1 < (0::eint)"
  by (simp add: zero_eint_def)

lemma int_of_eint_positive_mono:
  fixes x y :: eint
  shows "0 \<le> x \<Longrightarrow> x \<le> y \<Longrightarrow> y \<noteq> \<infinity> \<Longrightarrow> int_of_eint x \<le> int_of_eint y"
  by (cases rule: eint2_cases[of x y]) auto

lemma eint_MInfty_lessI[intro, simp]:
  fixes a :: eint
  shows "a \<noteq> -\<infinity> \<Longrightarrow> -\<infinity> < a"
  by (cases a) auto

lemma eint_less_PInfty[intro, simp]:
  fixes a :: eint
  shows "a \<noteq> \<infinity> \<Longrightarrow> a < \<infinity>"
  by (cases a) auto

lemma eint_less_eint_Ex:
  fixes a b :: eint
  shows "x < eint r \<longleftrightarrow> x = -\<infinity> \<or> (\<exists>p. p < r \<and> x = eint p)"
  by (cases x) auto

lemma less_PInf_Ex_of_nat: "x \<noteq> \<infinity> \<longleftrightarrow> (\<exists>n::nat. x < eint (int n))"
proof (cases x)
  case (int r)
  then show ?thesis 
  by (metis gt_ex int_ops(1) less_eint.simps(1) less_eint.simps(2) linorder_less_linear of_nat_less_iff zero_less_imp_eq_int)
qed simp_all

lemma eint_add_strict_mono2:
  fixes a b c d :: eint
  assumes "a < b" and "c < d"
  shows "a + c < b + d"
  using assms
  by (cases a; force simp add: elim: less_eint.elims)

lemma eint_minus_le_minus[simp]:
  fixes a b :: eint
  shows "- a \<le> - b \<longleftrightarrow> b \<le> a"
  by (cases rule: eint2_cases[of a b]) auto

lemma eint_minus_less_minus[simp]:
  fixes a b :: eint
  shows "- a < - b \<longleftrightarrow> b < a"
  by (cases rule: eint2_cases[of a b]) auto

lemma eint_le_int_iff:
  "x \<le> int_of_eint y \<longleftrightarrow> (\<bar>y\<bar> \<noteq> \<infinity> \<longrightarrow> eint x \<le> y) \<and> (\<bar>y\<bar> = \<infinity> \<longrightarrow> x \<le> 0)"
  by (cases y) auto

lemma int_le_eint_iff:
  "int_of_eint y \<le> x \<longleftrightarrow> (\<bar>y\<bar> \<noteq> \<infinity> \<longrightarrow> y \<le> eint x) \<and> (\<bar>y\<bar> = \<infinity> \<longrightarrow> 0 \<le> x)"
  by (cases y) auto

lemma eint_less_int_iff:
  "x < int_of_eint y \<longleftrightarrow> (\<bar>y\<bar> \<noteq> \<infinity> \<longrightarrow> eint x < y) \<and> (\<bar>y\<bar> = \<infinity> \<longrightarrow> x < 0)"
  by (cases y) auto

lemma int_less_eint_iff:
  "int_of_eint y < x \<longleftrightarrow> (\<bar>y\<bar> \<noteq> \<infinity> \<longrightarrow> y < eint x) \<and> (\<bar>y\<bar> = \<infinity> \<longrightarrow> 0 < x)"
  by (cases y) auto

text \<open>
  To help with inferences like \<^prop>\<open>a < eint x \<Longrightarrow> x < y \<Longrightarrow> a < eint y\<close>,
  where x and y are int.
\<close>

lemma le_eint_le: "a \<le> eint x \<Longrightarrow> x \<le> y \<Longrightarrow> a \<le> eint y"
  using eint_less_eq(3) order.trans by blast

lemma le_eint_less: "a \<le> eint x \<Longrightarrow> x < y \<Longrightarrow> a < eint y"
  by (simp add: le_less_trans)

lemma less_eint_le: "a < eint x \<Longrightarrow> x \<le> y \<Longrightarrow> a < eint y"
  using eint_less_eint_Ex by auto

lemma eint_le_le: "eint y \<le> a \<Longrightarrow> x \<le> y \<Longrightarrow> eint x \<le> a"
  by (simp add: order_subst2)

lemma eint_le_less: "eint y \<le> a \<Longrightarrow> x < y \<Longrightarrow> eint x < a"
  by (simp add: dual_order.strict_trans1)

lemma eint_less_le: "eint y < a \<Longrightarrow> x \<le> y \<Longrightarrow> eint x < a"
  using eint_less_eq(3) le_less_trans by blast

lemma int_of_eint_pos:
  fixes x :: eint
  shows "0 \<le> x \<Longrightarrow> 0 \<le> int_of_eint x" by (cases x) auto

lemmas int_of_eint_ord_simps =
  eint_le_int_iff int_le_eint_iff eint_less_int_iff int_less_eint_iff

lemma abs_eint_ge0[simp]: "0 \<le> x \<Longrightarrow> \<bar>x :: eint\<bar> = x"
  by (cases x) auto

lemma abs_eint_less0[simp]: "x < 0 \<Longrightarrow> \<bar>x :: eint\<bar> = -x"
  by (cases x) auto

lemma abs_eint_pos[simp]: "0 \<le> \<bar>x :: eint\<bar>"
  by (cases x) auto

lemma eint_abs_leI:
  fixes x y :: eint
  shows "\<lbrakk> x \<le> y; -x \<le> y \<rbrakk> \<Longrightarrow> \<bar>x\<bar> \<le> y"
by(cases x y rule: eint2_cases)(simp_all)

lemma eint_abs_add:
  fixes a b::eint
  shows "abs(a+b) \<le> abs a + abs b"
by (cases rule: eint2_cases[of a b]) (auto)

lemma int_of_eint_le_0[simp]: "int_of_eint (x :: eint) \<le> 0 \<longleftrightarrow> x \<le> 0 \<or> x = \<infinity>"
  by (cases x) auto

lemma abs_int_of_eint[simp]: "\<bar>int_of_eint (x :: eint)\<bar> = int_of_eint \<bar>x\<bar>"
  by (cases x) auto

lemma zero_less_int_of_eint:
  fixes x :: eint
  shows "0 < int_of_eint x \<longleftrightarrow> 0 < x \<and> x \<noteq> \<infinity>"
  by (cases x) auto

lemma eint_0_le_uminus_iff[simp]:
  fixes a :: eint
  shows "0 \<le> - a \<longleftrightarrow> a \<le> 0"
  by (cases rule: eint2_cases[of a]) auto

lemma eint_uminus_le_0_iff[simp]:
  fixes a :: eint
  shows "- a \<le> 0 \<longleftrightarrow> 0 \<le> a"
  by (cases rule: eint2_cases[of a]) auto

lemma eint_add_strict_mono:
  fixes a b c d :: eint
  assumes "a \<le> b"
    and "0 \<le> a"
    and "a \<noteq> \<infinity>"
    and "c < d"
  shows "a + c < b + d"
  using assms
  by (cases rule: eint3_cases[case_product eint_cases, of a b c d]) auto

lemma eint_less_add:
  fixes a b c :: eint
  shows "\<bar>a\<bar> \<noteq> \<infinity> \<Longrightarrow> c < b \<Longrightarrow> a + c < a + b"
  by (cases rule: eint2_cases[of b c]) auto

lemma eint_add_nonneg_eq_0_iff:
  fixes a b :: eint
  shows "0 \<le> a \<Longrightarrow> 0 \<le> b \<Longrightarrow> a + b = 0 \<longleftrightarrow> a = 0 \<and> b = 0"
  by (cases a b rule: eint2_cases) auto

lemma eint_uminus_eq_reorder: "- a = b \<longleftrightarrow> a = (-b::eint)"
  by auto

lemma eint_uminus_less_reorder: "- a < b \<longleftrightarrow> -b < (a::eint)"
  by (subst (3) eint_uminus_uminus[symmetric]) (simp only: eint_minus_less_minus)

lemma eint_less_uminus_reorder: "a < - b \<longleftrightarrow> b < - (a::eint)"
  by (subst (3) eint_uminus_uminus[symmetric]) (simp only: eint_minus_less_minus)

lemma eint_uminus_le_reorder: "- a \<le> b \<longleftrightarrow> -b \<le> (a::eint)"
  by (subst (3) eint_uminus_uminus[symmetric]) (simp only: eint_minus_le_minus)

lemmas eint_uminus_reorder =
  eint_uminus_eq_reorder eint_uminus_less_reorder eint_uminus_le_reorder

lemma eint_bot:
  fixes x :: eint
  assumes "\<And>B. x \<le> eint B"
  shows "x = - \<infinity>"
proof (cases x)
  case (int r)
  with assms[of "r - 1"] show ?thesis
    by auto
next
  case PInf
  with assms[of 0] show ?thesis
    by auto
next
  case MInf
  then show ?thesis
    by simp
qed

lemma eint_top:
  fixes x :: eint
  assumes "\<And>B. x \<ge> eint B"
  shows "x = \<infinity>"
proof (cases x)
  case (int r)
  with assms[of "r + 1"] show ?thesis
    by auto
next
  case MInf
  with assms[of 0] show ?thesis
    by auto
next
  case PInf
  then show ?thesis
    by simp
qed

lemma
  shows eint_max[simp]: "eint (max x y) = max (eint x) (eint y)"
    and eint_min[simp]: "eint (min x y) = min (eint x) (eint y)"
  by (simp_all add: min_def max_def)

lemma eint_max_0: "max 0 (eint r) = eint (max 0 r)"
  by (auto simp: zero_eint_def)

lemma
  fixes f :: "nat \<Rightarrow> eint"
  shows eint_incseq_uminus[simp]: "incseq (\<lambda>x. - f x) \<longleftrightarrow> decseq f"
    and eint_decseq_uminus[simp]: "decseq (\<lambda>x. - f x) \<longleftrightarrow> incseq f"
  unfolding decseq_def incseq_def by auto

lemma incseq_eint: "incseq f \<Longrightarrow> incseq (\<lambda>x. eint (f x))"
  unfolding incseq_def by auto

lemma sum_eint[simp]: "(\<Sum>x\<in>A. eint (f x)) = eint (\<Sum>x\<in>A. f x)"
proof (cases "finite A")
  case True
  then show ?thesis by induct auto
next
  case False
  then show ?thesis by simp
qed

lemma sum_list_eint [simp]: "sum_list (map (\<lambda>x. eint (f x)) xs) = eint (sum_list (map f xs))"
  by (induction xs) simp_all

lemma sum_Pinfty:
  fixes f :: "'a \<Rightarrow> eint"
  shows "(\<Sum>x\<in>P. f x) = \<infinity> \<longleftrightarrow> finite P \<and> (\<exists>i\<in>P. f i = \<infinity>)"
proof safe
  assume *: "sum f P = \<infinity>"
  show "finite P"
  proof (rule ccontr)
    assume "\<not> finite P"
    with * show False
      by auto
  qed
  show "\<exists>i\<in>P. f i = \<infinity>"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    then have "\<And>i. i \<in> P \<Longrightarrow> f i \<noteq> \<infinity>"
      by auto
    with \<open>finite P\<close> have "sum f P \<noteq> \<infinity>"
      by induct auto
    with * show False
      by auto
  qed
next
  fix i
  assume "finite P" and "i \<in> P" and "f i = \<infinity>"
  then show "sum f P = \<infinity>"
  proof induct
    case (insert x A)
    show ?case using insert by (cases "x = i") auto
  qed simp
qed

lemma sum_Inf:
  fixes f :: "'a \<Rightarrow> eint"
  shows "\<bar>sum f A\<bar> = \<infinity> \<longleftrightarrow> finite A \<and> (\<exists>i\<in>A. \<bar>f i\<bar> = \<infinity>)"
proof
  assume *: "\<bar>sum f A\<bar> = \<infinity>"
  have "finite A"
    by (rule ccontr) (insert *, auto)
  moreover have "\<exists>i\<in>A. \<bar>f i\<bar> = \<infinity>"
  proof (rule ccontr)
    assume "\<not> ?thesis"
    then have "\<forall>i\<in>A. \<exists>r. f i = eint r"
      by auto
    from bchoice[OF this] obtain r where "\<forall>x\<in>A. f x = eint (r x)" ..
    with * show False
      by auto
  qed
  ultimately show "finite A \<and> (\<exists>i\<in>A. \<bar>f i\<bar> = \<infinity>)"
    by auto
next
  assume "finite A \<and> (\<exists>i\<in>A. \<bar>f i\<bar> = \<infinity>)"
  then obtain i where "finite A" "i \<in> A" and "\<bar>f i\<bar> = \<infinity>"
    by auto
  then show "\<bar>sum f A\<bar> = \<infinity>"
  proof induct
    case (insert j A)
    then show ?case
      by (cases rule: eint3_cases[of "f i" "f j" "sum f A"]) auto
  qed simp
qed

lemma sum_int_of_eint:
  fixes f :: "'i \<Rightarrow> eint"
  assumes "\<And>x. x \<in> S \<Longrightarrow> \<bar>f x\<bar> \<noteq> \<infinity>"
  shows "(\<Sum>x\<in>S. int_of_eint (f x)) = int_of_eint (sum f S)"
proof -
  have "\<forall>x\<in>S. \<exists>r. f x = eint r"
  proof
    fix x
    assume "x \<in> S"
    from assms[OF this] show "\<exists>r. f x = eint r"
      by (cases "f x") auto
  qed
  from bchoice[OF this] obtain r where "\<forall>x\<in>S. f x = eint (r x)" ..
  then show ?thesis
    by simp
qed

lemma sum_eint_0:
  fixes f :: "'a \<Rightarrow> eint"
  assumes "finite A"
    and "\<And>i. i \<in> A \<Longrightarrow> 0 \<le> f i"
  shows "(\<Sum>x\<in>A. f x) = 0 \<longleftrightarrow> (\<forall>i\<in>A. f i = 0)"
proof
  assume "sum f A = 0" with assms show "\<forall>i\<in>A. f i = 0"
  proof (induction A)
    case (insert a A)
    then have "f a = 0 \<and> (\<Sum>a\<in>A. f a) = 0"
      by (subst eint_add_nonneg_eq_0_iff[symmetric]) (simp_all add: sum_nonneg)
    with insert show ?case
      by simp
  qed simp
qed auto

subsubsection "Multiplication"

instantiation eint :: "{comm_monoid_mult,sgn}"
begin

function sgn_eint :: "eint \<Rightarrow> eint" where
  "sgn (eint r) = eint (sgn r)"
| "sgn (\<infinity>::eint) = 1"
| "sgn (-\<infinity>::eint) = -1"
by (auto intro: eint_cases)
termination by standard (rule wf_on_bot)

function times_eint where
  "eint r * eint p = eint (r * p)"
| "eint r * \<infinity> = (if r = 0 then 0 else if r > 0 then \<infinity> else -\<infinity>)"
| "\<infinity> * eint r = (if r = 0 then 0 else if r > 0 then \<infinity> else -\<infinity>)"
| "eint r * -\<infinity> = (if r = 0 then 0 else if r > 0 then -\<infinity> else \<infinity>)"
| "-\<infinity> * eint r = (if r = 0 then 0 else if r > 0 then -\<infinity> else \<infinity>)"
| "(\<infinity>::eint) * \<infinity> = \<infinity>"
| "-(\<infinity>::eint) * \<infinity> = -\<infinity>"
| "(\<infinity>::eint) * -\<infinity> = -\<infinity>"
| "-(\<infinity>::eint) * -\<infinity> = \<infinity>"
proof goal_cases
  case prems: (1 P x)
  then obtain a b where "x = (a, b)"
    by (cases x) auto
  with prems show P
    by (cases rule: eint2_cases[of a b]) auto
qed simp_all
termination by (relation "{}") simp

instance
proof
  fix a b c :: eint
  show "1 * a = a"
    by (cases a) (simp_all add: one_eint_def)
  show "a * b = b * a"
    by (cases rule: eint2_cases[of a b]) simp_all
  show "a * b * c = a * (b * c)"
    by (cases rule: eint3_cases[of a b c])
       (simp_all add: zero_eint_def zero_less_mult_iff)
qed

end

lemma [simp]:
  shows eint_1_times: "eint 1 * x = x"
  and times_eint_1: "x * eint 1 = x"
by(simp_all flip: one_eint_def)

lemma one_not_le_zero_eint[simp]: "\<not> (1 \<le> (0::eint))"
  by (simp add: one_eint_def zero_eint_def)

lemma int_eint_1[simp]: "int_of_eint (1::eint) = 1"
  unfolding one_eint_def by simp

lemma int_of_eint_le_1:
  fixes a :: eint
  shows "a \<le> 1 \<Longrightarrow> int_of_eint a \<le> 1"
  by (cases a) (auto simp: one_eint_def)

lemma abs_eint_one[simp]: "\<bar>1\<bar> = (1::eint)"
  unfolding one_eint_def by simp

lemma eint_mult_zero[simp]:
  fixes a :: eint
  shows "a * 0 = 0"
  by (cases a) (simp_all add: zero_eint_def)

lemma eint_zero_mult[simp]:
  fixes a :: eint
  shows "0 * a = 0"
  by (cases a) (simp_all add: zero_eint_def)

lemma eint_m1_less_0[simp]: "-(1::eint) < 0"
  by (simp add: zero_eint_def one_eint_def)

lemma eint_times[simp]:
  "1 \<noteq> (\<infinity>::eint)" "(\<infinity>::eint) \<noteq> 1"
  "1 \<noteq> -(\<infinity>::eint)" "-(\<infinity>::eint) \<noteq> 1"
  by (auto simp: one_eint_def)

lemma eint_plus_1[simp]:
  "1 + eint r = eint (r + 1)"
  "eint r + 1 = eint (r + 1)"
  "1 + -(\<infinity>::eint) = -\<infinity>"
  "-(\<infinity>::eint) + 1 = -\<infinity>"
  unfolding one_eint_def by auto

lemma eint_zero_times[simp]:
  fixes a b :: eint
  shows "a * b = 0 \<longleftrightarrow> a = 0 \<or> b = 0"
  by (cases rule: eint2_cases[of a b]) auto

lemma eint_mult_eq_PInfty[simp]:
  "a * b = (\<infinity>::eint) \<longleftrightarrow>
    (a = \<infinity> \<and> b > 0) \<or> (a > 0 \<and> b = \<infinity>) \<or> (a = -\<infinity> \<and> b < 0) \<or> (a < 0 \<and> b = -\<infinity>)"
  by (cases rule: eint2_cases[of a b]) auto

lemma eint_mult_eq_MInfty[simp]:
  "a * b = -(\<infinity>::eint) \<longleftrightarrow>
    (a = \<infinity> \<and> b < 0) \<or> (a < 0 \<and> b = \<infinity>) \<or> (a = -\<infinity> \<and> b > 0) \<or> (a > 0 \<and> b = -\<infinity>)"
  by (cases rule: eint2_cases[of a b]) auto

lemma eint_abs_mult: "\<bar>x * y :: eint\<bar> = \<bar>x\<bar> * \<bar>y\<bar>"
  by (cases x y rule: eint2_cases) (auto simp: abs_mult)

lemma eint_0_less_1[simp]: "0 < (1::eint)"
  by (simp_all add: zero_eint_def one_eint_def)

lemma eint_mult_minus_left[simp]:
  fixes a b :: eint
  shows "-a * b = - (a * b)"
  by (cases rule: eint2_cases[of a b]) auto

lemma eint_mult_minus_right[simp]:
  fixes a b :: eint
  shows "a * -b = - (a * b)"
  by (cases rule: eint2_cases[of a b]) auto

lemma eint_mult_infty[simp]:
  "a * (\<infinity>::eint) = (if a = 0 then 0 else if 0 < a then \<infinity> else - \<infinity>)"
  by (cases a) auto

lemma eint_infty_mult[simp]:
  "(\<infinity>::eint) * a = (if a = 0 then 0 else if 0 < a then \<infinity> else - \<infinity>)"
  by (cases a) auto

lemma eint_mult_strict_right_mono:
  assumes "a < b"
    and "0 < c"
    and "c < (\<infinity>::eint)"
  shows "a * c < b * c"
  using assms
  by (cases rule: eint3_cases[of a b c]) (auto simp: zero_le_mult_iff)

lemma eint_mult_strict_left_mono:
  "a < b \<Longrightarrow> 0 < c \<Longrightarrow> c < (\<infinity>::eint) \<Longrightarrow> c * a < c * b"
  using eint_mult_strict_right_mono
  by (simp add: mult.commute[of c])

lemma eint_mult_right_mono:
  fixes a b c :: eint
  assumes "a \<le> b" "0 \<le> c"
  shows "a * c \<le> b * c"
proof (cases "c = 0")
  case False
  with assms show ?thesis
    by (cases rule: eint3_cases[of a b c]) auto
qed auto

lemma eint_mult_left_mono:
  fixes a b c :: eint
  shows "a \<le> b \<Longrightarrow> 0 \<le> c \<Longrightarrow> c * a \<le> c * b"
  using eint_mult_right_mono
  by (simp add: mult.commute[of c])

lemma eint_mult_mono:
  fixes a b c d::eint
  assumes "b \<ge> 0" "c \<ge> 0" "a \<le> b" "c \<le> d"
  shows "a * c \<le> b * d"
by (metis eint_mult_right_mono mult.commute order_trans assms)

lemma eint_mult_mono':
  fixes a b c d::eint
  assumes "a \<ge> 0" "c \<ge> 0" "a \<le> b" "c \<le> d"
  shows "a * c \<le> b * d"
by (metis eint_mult_right_mono mult.commute order_trans assms)

lemma eint_mult_mono_strict:
  fixes a b c d::eint
  assumes "b > 0" "c > 0" "a < b" "c < d"
  shows "a * c < b * d"
proof -
  have "c < \<infinity>" using \<open>c < d\<close> by auto
  then have "a * c < b * c" by (metis eint_mult_strict_left_mono[OF assms(3) assms(2)] mult.commute)
  moreover have "b * c \<le> b * d" using assms(2) assms(4) by (simp add: assms(1) eint_mult_left_mono less_imp_le)
  ultimately show ?thesis by simp
qed

lemma eint_mult_mono_strict':
  fixes a b c d::eint
  assumes "a > 0" "c > 0" "a < b" "c < d"
  shows "a * c < b * d"
  using assms eint_mult_mono_strict by auto

lemma zero_less_one_eint[simp]: "0 \<le> (1::eint)"
  by (simp add: one_eint_def zero_eint_def)

lemma eint_0_le_mult[simp]: "0 \<le> a \<Longrightarrow> 0 \<le> b \<Longrightarrow> 0 \<le> a * (b :: eint)"
  by (cases rule: eint2_cases[of a b]) auto

lemma eint_right_distrib:
  fixes r a b :: eint
  shows "0 \<le> a \<Longrightarrow> 0 \<le> b \<Longrightarrow> r * (a + b) = r * a + r * b"
  by (cases rule: eint3_cases[of r a b]) (simp_all add: field_simps)

lemma eint_left_distrib:
  fixes r a b :: eint
  shows "0 \<le> a \<Longrightarrow> 0 \<le> b \<Longrightarrow> (a + b) * r = a * r + b * r"
  by (cases rule: eint3_cases[of r a b]) (simp_all add: field_simps)

lemma eint_mult_le_0_iff:
  fixes a b :: eint
  shows "a * b \<le> 0 \<longleftrightarrow> (0 \<le> a \<and> b \<le> 0) \<or> (a \<le> 0 \<and> 0 \<le> b)"
  by (cases rule: eint2_cases[of a b]) (simp_all add: mult_le_0_iff)

lemma eint_zero_le_0_iff:
  fixes a b :: eint
  shows "0 \<le> a * b \<longleftrightarrow> (0 \<le> a \<and> 0 \<le> b) \<or> (a \<le> 0 \<and> b \<le> 0)"
  by (cases rule: eint2_cases[of a b]) (simp_all add: zero_le_mult_iff)

lemma eint_mult_less_0_iff:
  fixes a b :: eint
  shows "a * b < 0 \<longleftrightarrow> (0 < a \<and> b < 0) \<or> (a < 0 \<and> 0 < b)"
  by (cases rule: eint2_cases[of a b]) (simp_all add: mult_less_0_iff)

lemma eint_zero_less_0_iff:
  fixes a b :: eint
  shows "0 < a * b \<longleftrightarrow> (0 < a \<and> 0 < b) \<or> (a < 0 \<and> b < 0)"
  by (cases rule: eint2_cases[of a b]) (simp_all add: zero_less_mult_iff)

lemma eint_left_mult_cong:
  fixes a b c :: eint
  shows  "c = d \<Longrightarrow> (d \<noteq> 0 \<Longrightarrow> a = b) \<Longrightarrow> a * c = b * d"
  by (cases "c = 0") simp_all

lemma eint_right_mult_cong:
  fixes a b c :: eint
  shows "c = d \<Longrightarrow> (d \<noteq> 0 \<Longrightarrow> a = b) \<Longrightarrow> c * a = d * b"
  by (cases "c = 0") simp_all

lemma eint_distrib:
  fixes a b c :: eint
  assumes "a \<noteq> \<infinity> \<or> b \<noteq> -\<infinity>"
    and "a \<noteq> -\<infinity> \<or> b \<noteq> \<infinity>"
    and "\<bar>c\<bar> \<noteq> \<infinity>"
  shows "(a + b) * c = a * c + b * c"
  using assms
  by (cases rule: eint3_cases[of a b c]) (simp_all add: field_simps)

lemma numeral_eq_eint [simp]: "numeral w = eint (numeral w)"
proof (induct w rule: num_induct)
  case One
  then show ?case
    by simp
next
  case (inc x)
  then show ?case
    by (simp add: inc numeral_inc)
qed

lemma distrib_left_eint_nn:
  "c \<ge> 0 \<Longrightarrow> (x + y) * eint c = x * eint c + y * eint c"
  by(cases x y rule: eint2_cases)(simp_all add: ring_distribs)

lemma sum_eint_right_distrib:
  fixes f :: "'a \<Rightarrow> eint"
  shows "(\<And>i. i \<in> A \<Longrightarrow> 0 \<le> f i) \<Longrightarrow> r * sum f A = (\<Sum>n\<in>A. r * f n)"
  by (induct A rule: infinite_finite_induct)  (auto simp: eint_right_distrib sum_nonneg)

lemma sum_eint_left_distrib:
  "(\<And>i. i \<in> A \<Longrightarrow> 0 \<le> f i) \<Longrightarrow> sum f A * r = (\<Sum>n\<in>A. f n * r :: eint)"
  using sum_eint_right_distrib[of A f r] by (simp add: mult_ac)

lemma sum_distrib_right_eint:
  "c \<ge> 0 \<Longrightarrow> sum f A * eint c = (\<Sum>x\<in>A. f x * c :: eint)"
by(subst sum_comp_morphism[where h="\<lambda>x. x * eint c", symmetric])(simp_all add: distrib_left_eint_nn)



lemma eint_le_int:
  fixes x y :: eint
  assumes "\<And>z. x \<le> eint z \<Longrightarrow> y \<le> eint z"
  shows "y \<le> x"
  by (metis assms eint_bot eint_cases eint_infty_less_eq(2) eint_less_eq(1) linorder_le_cases)

    


lemma prod_eint_0:
  fixes f :: "'a \<Rightarrow> eint"
  shows "(\<Prod>i\<in>A. f i) = 0 \<longleftrightarrow> finite A \<and> (\<exists>i\<in>A. f i = 0)"
proof (cases "finite A")
  case True
  then show ?thesis by (induct A) auto
qed auto

lemma prod_eint_pos:
  fixes f :: "'a \<Rightarrow> eint"
  assumes pos: "\<And>i. i \<in> I \<Longrightarrow> 0 \<le> f i"
  shows "0 \<le> (\<Prod>i\<in>I. f i)"
proof (cases "finite I")
  case True
  from this pos show ?thesis
    by induct auto
qed auto

lemma prod_PInf:
  fixes f :: "'a \<Rightarrow> eint"
  assumes "\<And>i. i \<in> I \<Longrightarrow> 0 \<le> f i"
  shows "(\<Prod>i\<in>I. f i) = \<infinity> \<longleftrightarrow> finite I \<and> (\<exists>i\<in>I. f i = \<infinity>) \<and> (\<forall>i\<in>I. f i \<noteq> 0)"
proof (cases "finite I")
  case True
  from this assms show ?thesis
  proof (induct I)
    case (insert i I)
    then have pos: "0 \<le> f i" "0 \<le> prod f I"
      by (auto intro!: prod_eint_pos)
    from insert have "(\<Prod>j\<in>insert i I. f j) = \<infinity> \<longleftrightarrow> prod f I * f i = \<infinity>"
      by auto
    also have "\<dots> \<longleftrightarrow> (prod f I = \<infinity> \<or> f i = \<infinity>) \<and> f i \<noteq> 0 \<and> prod f I \<noteq> 0"
      using prod_eint_pos[of I f] pos
      by (cases rule: eint2_cases[of "f i" "prod f I"]) auto
    also have "\<dots> \<longleftrightarrow> finite (insert i I) \<and> (\<exists>j\<in>insert i I. f j = \<infinity>) \<and> (\<forall>j\<in>insert i I. f j \<noteq> 0)"
      using insert by (auto simp: prod_eint_0)
    finally show ?case .
  qed simp
qed auto

lemma prod_eint: "(\<Prod>i\<in>A. eint (f i)) = eint (prod f A)"
proof (cases "finite A")
  case True
  then show ?thesis
    by induct (auto simp: one_eint_def)
next
  case False
  then show ?thesis
    by (simp add: one_eint_def)
qed


subsubsection \<open>Power\<close>

lemma eint_power[simp]: "(eint x) ^ n = eint (x^n)"
  by (induct n) (auto simp: one_eint_def)

lemma eint_power_PInf[simp]: "(\<infinity>::eint) ^ n = (if n = 0 then 1 else \<infinity>)"
  by (induct n) (auto simp: one_eint_def)

lemma eint_power_uminus[simp]:
  fixes x :: eint
  shows "(- x) ^ n = (if even n then x ^ n else - (x^n))"
  by (induct n) (auto simp: one_eint_def)

lemma eint_power_numeral[simp]:
  "(numeral num :: eint) ^ n = eint (numeral num ^ n)"
  by (induct n) (auto simp: one_eint_def)

lemma zero_le_power_eint[simp]:
  fixes a :: eint
  assumes "0 \<le> a"
  shows "0 \<le> a ^ n"
  using assms by (induct n) (auto simp: eint_zero_le_0_iff)


subsubsection \<open>Subtraction\<close>

lemma eint_minus_minus_image[simp]:
  fixes S :: "eint set"
  shows "uminus ` uminus ` S = S"
  by (auto simp: image_iff)

lemma eint_uminus_lessThan[simp]:
  fixes a :: eint
  shows "uminus ` {..<a} = {-a<..}"
proof -
  {
    fix x
    assume "-a < x"
    then have "- x < - (- a)"
      by (simp del: eint_uminus_uminus)
    then have "- x < a"
      by simp
  }
  then show ?thesis
    by force
qed

lemma eint_uminus_greaterThan[simp]: "uminus ` {(a::eint)<..} = {..<-a}"
  by (metis eint_uminus_lessThan eint_uminus_uminus eint_minus_minus_image)

instantiation eint :: minus
begin

definition "x - y = x + -(y::eint)"
instance ..

end

lemma eint_minus[simp]:
  "eint r - eint p = eint (r - p)"
  "-\<infinity> - eint r = -\<infinity>"
  "eint r - \<infinity> = -\<infinity>"
  "(\<infinity>::eint) - x = \<infinity>"
  "-(\<infinity>::eint) - \<infinity> = -\<infinity>"
  "x - -y = x + y"
  "x - 0 = x"
  "0 - x = -x"
  by (simp_all add: minus_eint_def)

lemma eint_x_minus_x[simp]: "x - x = (if \<bar>x\<bar> = \<infinity> then \<infinity> else 0::eint)"
  by (cases x) simp_all

lemma eint_eq_minus_iff:
  fixes x y z :: eint
  shows "x = z - y \<longleftrightarrow>
    (\<bar>y\<bar> \<noteq> \<infinity> \<longrightarrow> x + y = z) \<and>
    (y = -\<infinity> \<longrightarrow> x = \<infinity>) \<and>
    (y = \<infinity> \<longrightarrow> z = \<infinity> \<longrightarrow> x = \<infinity>) \<and>
    (y = \<infinity> \<longrightarrow> z \<noteq> \<infinity> \<longrightarrow> x = -\<infinity>)"
  by (cases rule: eint3_cases[of x y z]) auto

lemma eint_eq_minus:
  fixes x y z :: eint
  shows "\<bar>y\<bar> \<noteq> \<infinity> \<Longrightarrow> x = z - y \<longleftrightarrow> x + y = z"
  by (auto simp: eint_eq_minus_iff)

lemma eint_less_minus_iff:
  fixes x y z :: eint
  shows "x < z - y \<longleftrightarrow>
    (y = \<infinity> \<longrightarrow> z = \<infinity> \<and> x \<noteq> \<infinity>) \<and>
    (y = -\<infinity> \<longrightarrow> x \<noteq> \<infinity>) \<and>
    (\<bar>y\<bar> \<noteq> \<infinity>\<longrightarrow> x + y < z)"
  by (cases rule: eint3_cases[of x y z]) auto

lemma eint_less_minus:
  fixes x y z :: eint
  shows "\<bar>y\<bar> \<noteq> \<infinity> \<Longrightarrow> x < z - y \<longleftrightarrow> x + y < z"
  by (auto simp: eint_less_minus_iff)

lemma eint_le_minus_iff:
  fixes x y z :: eint
  shows "x \<le> z - y \<longleftrightarrow> (y = \<infinity> \<longrightarrow> z \<noteq> \<infinity> \<longrightarrow> x = -\<infinity>) \<and> (\<bar>y\<bar> \<noteq> \<infinity> \<longrightarrow> x + y \<le> z)"
  by (cases rule: eint3_cases[of x y z]) auto

lemma eint_le_minus:
  fixes x y z :: eint
  shows "\<bar>y\<bar> \<noteq> \<infinity> \<Longrightarrow> x \<le> z - y \<longleftrightarrow> x + y \<le> z"
  by (auto simp: eint_le_minus_iff)

lemma eint_minus_less_iff:
  fixes x y z :: eint
  shows "x - y < z \<longleftrightarrow> y \<noteq> -\<infinity> \<and> (y = \<infinity> \<longrightarrow> x \<noteq> \<infinity> \<and> z \<noteq> -\<infinity>) \<and> (y \<noteq> \<infinity> \<longrightarrow> x < z + y)"
  by (cases rule: eint3_cases[of x y z]) auto

lemma eint_minus_less:
  fixes x y z :: eint
  shows "\<bar>y\<bar> \<noteq> \<infinity> \<Longrightarrow> x - y < z \<longleftrightarrow> x < z + y"
  by (auto simp: eint_minus_less_iff)

lemma eint_minus_le_iff:
  fixes x y z :: eint
  shows "x - y \<le> z \<longleftrightarrow>
    (y = -\<infinity> \<longrightarrow> z = \<infinity>) \<and>
    (y = \<infinity> \<longrightarrow> x = \<infinity> \<longrightarrow> z = \<infinity>) \<and>
    (\<bar>y\<bar> \<noteq> \<infinity> \<longrightarrow> x \<le> z + y)"
  by (cases rule: eint3_cases[of x y z]) auto

lemma eint_minus_le:
  fixes x y z :: eint
  shows "\<bar>y\<bar> \<noteq> \<infinity> \<Longrightarrow> x - y \<le> z \<longleftrightarrow> x \<le> z + y"
  by (auto simp: eint_minus_le_iff)

lemma eint_minus_eq_minus_iff:
  fixes a b c :: eint
  shows "a - b = a - c \<longleftrightarrow>
    b = c \<or> a = \<infinity> \<or> (a = -\<infinity> \<and> b \<noteq> -\<infinity> \<and> c \<noteq> -\<infinity>)"
  by (cases rule: eint3_cases[of a b c]) auto

lemma eint_add_le_add_iff:
  fixes a b c :: eint
  shows "c + a \<le> c + b \<longleftrightarrow>
    a \<le> b \<or> c = \<infinity> \<or> (c = -\<infinity> \<and> a \<noteq> \<infinity> \<and> b \<noteq> \<infinity>)"
  by (cases rule: eint3_cases[of a b c]) (simp_all add: field_simps)

lemma eint_add_le_add_iff2:
  fixes a b c :: eint
  shows "a + c \<le> b + c \<longleftrightarrow> a \<le> b \<or> c = \<infinity> \<or> (c = -\<infinity> \<and> a \<noteq> \<infinity> \<and> b \<noteq> \<infinity>)"
by(cases rule: eint3_cases[of a b c])(simp_all add: field_simps)

lemma eint_mult_le_mult_iff:
  fixes a b c :: eint
  shows "\<bar>c\<bar> \<noteq> \<infinity> \<Longrightarrow> c * a \<le> c * b \<longleftrightarrow> (0 < c \<longrightarrow> a \<le> b) \<and> (c < 0 \<longrightarrow> b \<le> a)"
  by (cases rule: eint3_cases[of a b c]) (simp_all add: mult_le_cancel_left)

lemma eint_minus_mono:
  fixes A B C D :: eint assumes "A \<le> B" "D \<le> C"
  shows "A - C \<le> B - D"
  using assms
  by (cases rule: eint3_cases[case_product eint_cases, of A B C D]) simp_all

lemma eint_mono_minus_cancel:
  fixes a b c :: eint
  shows "c - a \<le> c - b \<Longrightarrow> 0 \<le> c \<Longrightarrow> c < \<infinity> \<Longrightarrow> b \<le> a"
  by (cases a b c rule: eint3_cases) auto

lemma int_of_eint_minus:
  fixes a b :: eint
  shows "int_of_eint (a - b) = (if \<bar>a\<bar> = \<infinity> \<or> \<bar>b\<bar> = \<infinity> then 0 else int_of_eint a - int_of_eint b)"
  by (cases rule: eint2_cases[of a b]) auto

lemma int_of_eint_minus': "\<bar>x\<bar> = \<infinity> \<longleftrightarrow> \<bar>y\<bar> = \<infinity> \<Longrightarrow> int_of_eint x - int_of_eint y = int_of_eint (x - y :: eint)"
by(subst int_of_eint_minus) auto

lemma eint_diff_positive:
  fixes a b :: eint shows "a \<le> b \<Longrightarrow> 0 \<le> b - a"
  by (cases rule: eint2_cases[of a b]) auto

lemma eint_between:
  fixes x e :: eint
  assumes "\<bar>x\<bar> \<noteq> \<infinity>"
    and "0 < e"
  shows "x - e < x"
    and "x < x + e"
  using assms  by (cases x, cases e, auto)+

lemma eint_minus_eq_PInfty_iff:
  fixes x y :: eint
  shows "x - y = \<infinity> \<longleftrightarrow> y = -\<infinity> \<or> x = \<infinity>"
  by (cases x y rule: eint2_cases) simp_all

lemma eint_diff_add_eq_diff_diff_swap:
  fixes x y z :: eint
  shows "\<bar>y\<bar> \<noteq> \<infinity> \<Longrightarrow> x - (y + z) = x - y - z"
  by(cases x y z rule: eint3_cases) simp_all

lemma eint_diff_add_assoc2:
  fixes x y z :: eint
  shows "x + y - z = x - z + y"
  by(cases x y z rule: eint3_cases) simp_all

lemma eint_add_uminus_conv_diff: fixes x y z :: eint shows "- x + y = y - x"
  by(cases x y rule: eint2_cases) simp_all

lemma eint_minus_diff_eq:
  fixes x y :: eint
  shows "\<lbrakk> x = \<infinity> \<longrightarrow> y \<noteq> \<infinity>; x = -\<infinity> \<longrightarrow> y \<noteq> - \<infinity> \<rbrakk> \<Longrightarrow> - (x - y) = y - x"
  by(cases x y rule: eint2_cases) simp_all

lemma ediff_le_self [simp]: "x - y \<le> (x :: enat)"
  by(cases x y rule: enat.exhaust[case_product enat.exhaust]) simp_all

lemma eint_abs_diff:
  fixes a b::eint
  shows "abs(a-b) \<le> abs a + abs b"
  by (cases rule: eint2_cases[of a b]) (auto)





subsection "Distributive lattice"

instantiation eint :: distrib_lattice
begin

definition [simp]: "sup x y = (max x y :: eint)"
definition [simp]: "inf x y = (min x y :: eint)"
instance by standard 
(auto simp add: inf_int_def sup_int_def max_min_distrib2)

end




lemma min_PInf [simp]: "min (\<infinity>::eint) x = x"
 by simp


lemma min_PInf2 [simp]: "min x (\<infinity>::eint) = x"
by simp


lemma max_PInf [simp]: "max (\<infinity>::eint) x = \<infinity>"
by simp

lemma max_PInf2 [simp]: "max x (\<infinity>::eint) = \<infinity>"
by simp

lemma min_MInf [simp]: "min (-\<infinity>::eint) x = -\<infinity>"
by simp

lemma min_MInf2 [simp]: "min x (-\<infinity>::eint) = -\<infinity>"
by simp
 

lemma max_MInf [simp]: "max (-\<infinity>::eint) x = x"
by simp


lemma max_MInf2 [simp]: "max x (-\<infinity>::eint) = x"
by simp
 

instantiation eint :: "{order_bot, order_top}"
begin

definition bot_eint :: eint where "bot_eint = -\<infinity>"
definition top_eint :: eint where "top_eint = \<infinity>"

instance
  by standard (simp_all add: bot_eint_def top_eint_def)

end


lemma finite_eint_bounded_below_above:
  assumes le_fin: "\<And>y. y \<in> A \<Longrightarrow> eint m \<le> y \<and> y \<le> eint n  "
  shows "finite A"
proof (rule finite_subset)
  show "finite (eint ` {m..n})" by blast
  have "A \<subseteq> {eint m..eint n}" using le_fin by fastforce
  also have "\<dots> \<subseteq> eint ` {m..n}"
    apply (rule subsetI)
    subgoal for x by (cases x) auto
    done
  finally show "A \<subseteq> eint ` {m..n}" .
qed

lemma infinite_eint_bounded_above: 
 "infinite (eint ` {..n}) "  
by (meson finite_imageD infinite_Iic inj_eint)

lemma infinite_eint_bounded_below: 
 "infinite (eint ` {n..}) "
by (simp add: finite_image_iff infinite_Ici)

lemma infinite_eint_unbounded_below_or_above: 
 assumes "infinite A"
 shows " (\<exists> y \<in> A. y > eint n) \<or> (\<exists> y\<in>A. eint m > y)"
using assms  finite_eint_bounded_below_above[of A m n] 
by fastforce


subsection \<open>Extended int intervals\<close>

lemma int_greaterThanLessThan_infinity_eq:
  "int_of_eint ` {N::eint<..<\<infinity>} =
    (if N = \<infinity> then {} else if N = -\<infinity> then UNIV else {int_of_eint N<..})"
  by (force simp: int_less_eint_iff intro!: image_eqI[where x="eint _"] elim!: less_eint.elims)

lemma int_greaterThanLessThan_minus_infinity_eq:
  "int_of_eint ` {-\<infinity><..<N::eint} =
    (if N = \<infinity> then UNIV else if N = -\<infinity> then {} else {..<int_of_eint N})"
proof -
  have "int_of_eint ` {-\<infinity><..<N::eint} = uminus ` int_of_eint ` {-N<..<\<infinity>}"
    by (auto simp: eint_uminus_less_reorder intro!: image_eqI[where x="-x" for x])
  also note int_greaterThanLessThan_infinity_eq
  finally show ?thesis
  using eint_uminus_eq_reorder int_greaterThanLessThan_infinity_eq by auto
qed

lemma int_greaterThanLessThan_inter:
  "int_of_eint ` {N<..<M::eint} = int_of_eint ` {-\<infinity><..<M} \<inter> int_of_eint ` {N<..<\<infinity>}"
  by (force elim!: less_eint.elims)

lemma int_atLeastGreaterThan_eq: "int_of_eint ` {N<..<M::eint} =
   (if N = \<infinity> then {} else
   if N = -\<infinity> then
    (if M = \<infinity> then UNIV
    else if M = -\<infinity> then {}
    else {..< int_of_eint M})
  else if M = - \<infinity> then {}
  else if M = \<infinity> then {int_of_eint N<..}
  else {int_of_eint N <..< int_of_eint M})"
proof (cases "M = -\<infinity> \<or> M = \<infinity> \<or> N = -\<infinity> \<or> N = \<infinity>")
  case True
  then show ?thesis
    by (auto simp: int_greaterThanLessThan_minus_infinity_eq int_greaterThanLessThan_infinity_eq )
next
  case False
  then obtain p q where "M = eint p" "N = eint q"
    by (metis MInfty_eq_minfinity eint.distinct(3) uminus_eint.elims)
  moreover have "\<And>x. \<lbrakk>q < x; x < p\<rbrakk> \<Longrightarrow> x \<in> int_of_eint ` {eint q<..<eint p}"
    by (metis greaterThanLessThan_iff imageI less_eint.simps(1) int_of_eint.simps(1))
  ultimately show ?thesis 
    by (auto elim!: less_eint.elims)
qed

lemma int_image_eint_ivl:
  fixes a b::eint
  shows
  "int_of_eint ` {a<..<b} =
  (if a < b then (if a = - \<infinity> then if b = \<infinity> then UNIV else {..<int_of_eint b}
  else if b = \<infinity> then {int_of_eint a<..} else {int_of_eint a <..< int_of_eint b}) else {})"
  by (cases a; cases b; simp add: int_atLeastGreaterThan_eq not_less)

lemma fixes a b c::eint
  shows not_inftyI: "a < b \<Longrightarrow> b < c \<Longrightarrow> abs b \<noteq> \<infinity>"
  by force

lemma
  interval_neqs:
  fixes r s t::int
  shows "{r<..<s} \<noteq> {t<..}"
    and "{r<..<s} \<noteq> {..<t}"
    and "{r<..<ra} \<noteq> UNIV"
    and "{r<..} \<noteq> {..<s}"
    and "{r<..} \<noteq> UNIV"
    and "{..<r} \<noteq> UNIV"
    and "{} \<noteq> {r<..}"
    and "{} \<noteq> {..<r}"
  subgoal
    by (metis dual_order.strict_trans greaterThanLessThan_iff greaterThan_iff gt_ex not_le order_refl)
  subgoal
    by (metis finite_greaterThanLessThan_int infinite_Iio)
  subgoal by force
  subgoal
    by (metis greaterThan_iff lessThan_iff lt_ex order_less_le order_less_trans)
  subgoal by force
  subgoal by force
  subgoal using greaterThan_non_empty by blast
  subgoal using lessThan_non_empty by blast
  done

lemma int_greaterThanLessThan_empty_iff: 
 fixes a b :: int
 shows "({a<..<b} = {}) \<longleftrightarrow> a+1 \<ge> b "
 using atLeastPlusOneLessThan_greaterThanLessThan_int
 by fastforce
 


lemma greaterThanLessThan_eq_iff:
  fixes r s t u::int
  shows "({r<..<s} = {t<..<u}) = (r+1 \<ge> s \<and> u \<le> t+1 \<or> r = t \<and> s = u)"
   using int_greaterThanLessThan_empty_iff[of r s]  int_greaterThanLessThan_empty_iff[of t u]
    apply auto
     apply (metis greaterThanLessThan_empty greaterThanLessThan_iff leD order.strict_iff_order verit_la_generic)
     apply (metis greaterThanLessThan_empty greaterThanLessThan_iff leD order.strict_iff_order verit_la_generic)
     apply (metis greaterThanLessThan_iff linorder_neqE_linordered_idom order_less_irrefl order_less_trans)
     by (metis dual_order.strict_trans greaterThanLessThan_iff linorder_neqE_linordered_idom order_less_irrefl)
    


lemma int_of_eint_image_greaterThanLessThan_iff:
  "int_of_eint ` {a <..< b} = int_of_eint ` {c <..< d} \<longleftrightarrow> (a+1 \<ge> b \<and> c+1 \<ge> d \<or> a = c \<and> b = d)"
  unfolding int_atLeastGreaterThan_eq
  apply (cases a; cases b; cases c; cases d;
    simp add: greaterThanLessThan_eq_iff interval_neqs interval_neqs[symmetric])
   using int_greaterThanLessThan_empty_iff by blast+
  
   
lemma uminus_image_int_of_eint_image_greaterThanLessThan:
  "uminus ` int_of_eint ` {l <..< u} = int_of_eint ` {-u <..< -l}"
  by (force simp: algebra_simps eint_less_uminus_reorder
    eint_uminus_less_reorder intro: image_eqI[where x="-x" for x])

lemma add_image_int_of_eint_image_greaterThanLessThan:
  "(+) c ` int_of_eint ` {l <..< u} = int_of_eint ` {c + l <..< c + u}"
  apply safe
  subgoal for x
    using eint_less_add[of c]
    by (force simp: int_of_eint_add add.commute)
  subgoal for _ x
    by (force simp: add.commute int_of_eint_minus eint_minus_less eint_less_minus
      intro: image_eqI[where x="x - c"])
  done

lemma add2_image_int_of_eint_image_greaterThanLessThan:
  "(\<lambda>x. x + c) ` int_of_eint ` {l <..< u} = int_of_eint ` {l + c <..< u + c}"
  using add_image_int_of_eint_image_greaterThanLessThan[of c l u]
  by (metis add.commute image_cong)

lemma minus_image_int_of_eint_image_greaterThanLessThan:
  "(-) c ` int_of_eint ` {l <..< u} = int_of_eint ` {c - u <..< c - l}"
  (is "?l = ?r")
proof -
  have "?l = (+) c ` uminus ` int_of_eint ` {l <..< u}" by auto
  also note uminus_image_int_of_eint_image_greaterThanLessThan
  also note add_image_int_of_eint_image_greaterThanLessThan
  finally show ?thesis by (simp add: minus_eint_def)
qed

lemma int_eint_bound_lemma_up:
  assumes "s \<in> int_of_eint ` {a<..<b}"
  assumes "t \<notin> int_of_eint ` {a<..<b}"
  assumes "s \<le> t"
  shows "b \<noteq> \<infinity>"
proof (cases b)
  case PInf
  then show ?thesis
    using assms
    apply clarsimp
    by (metis UNIV_I assms(1) eint_less_PInfty greaterThan_iff less_eq_eint_def less_le_trans int_image_eint_ivl)
qed auto

lemma int_eint_bound_lemma_down:
  assumes s: "s \<in> int_of_eint ` {a<..<b}"
  and t: "t \<notin> int_of_eint ` {a<..<b}"
  and "t \<le> s"
  shows "a \<noteq> - \<infinity>"
proof (cases b)
  case (int r)
  then show ?thesis
    using assms int_greaterThanLessThan_minus_infinity_eq by force
next
  case PInf
  then show ?thesis
    using t int_greaterThanLessThan_infinity_eq by auto
next
  case MInf
  then show ?thesis
    using s by auto
qed


lemma eint_less_le_alt: 
 " eint i < j \<Longrightarrow> eint (i+1) \<le> j " 
using eint_less_eq(3)[of "i+1" ]
by (metis eint_less_eint_Ex eint_less_le less_iff_succ_less_eq linorder_not_less
  order.asym) 

lemma exists_eint_int: 
assumes " i \<noteq> -\<infinity> " 
        " i \<noteq> \<infinity>" 
shows " \<exists>n. i = eint n "
using assms 
by (metis PInfty_eq_infinity eint.exhaust uminus_eint.simps(2))
 



subsection "Relation to \<^typ>\<open>enat\<close>"

definition "eint_of_enat n = (case n of enat n \<Rightarrow> eint (int n) | \<infinity> \<Rightarrow> \<infinity>)"

declare [[coercion "eint_of_enat :: enat \<Rightarrow> eint"]]
declare [[coercion "(\<lambda>n. eint (int n)) :: nat \<Rightarrow> eint"]]

lemma eint_of_enat_simps[simp]:
  "eint_of_enat (enat n) = eint n"
  "eint_of_enat \<infinity> = \<infinity>"
  by (simp_all add: eint_of_enat_def)

lemma eint_of_enat_le_iff[simp]: 
"eint_of_enat m \<le> eint_of_enat n \<longleftrightarrow> m \<le> n"
  by (cases m n rule: enat2_cases) auto

lemma eint_of_enat_less_iff[simp]: 
"eint_of_enat m < eint_of_enat n \<longleftrightarrow> m < n"
  by (cases m n rule: enat2_cases) auto

lemma numeral_le_eint_of_enat_iff[simp]: 
"numeral m \<le> eint_of_enat n \<longleftrightarrow> numeral m \<le> n"
by (cases n)  auto


lemma numeral_less_eint_of_enat_iff[simp]: 
"numeral m < eint_of_enat n \<longleftrightarrow> numeral m < n"
by (cases n)  auto


lemma eint_of_enat_ge_zero_cancel_iff[simp]: 
"0 \<le> eint_of_enat n \<longleftrightarrow> 0 \<le> n"
  by (cases n) (auto simp flip: enat_0)

lemma eint_of_enat_gt_zero_cancel_iff[simp]:
 "0 < eint_of_enat n \<longleftrightarrow> 0 < n"
  by (cases n) (auto simp flip: enat_0)

lemma eint_of_enat_zero[simp]: 
"eint_of_enat 0 = 0"
  by (auto simp flip: enat_0)

lemma eint_of_enat_inf[simp]:
 "eint_of_enat n = \<infinity> \<longleftrightarrow> n = \<infinity>"
  by (cases n) auto

lemma eint_of_enat_add: 
"eint_of_enat (m + n) = eint_of_enat m + eint_of_enat n"
  by (cases m n rule: enat2_cases) auto

lemma eint_of_enat_sub:
  assumes "n \<le> m"
  shows "eint_of_enat (m - n) = eint_of_enat m - eint_of_enat n "
  using assms by (cases m n rule: enat2_cases) auto

lemma eint_of_enat_mult:
  "eint_of_enat (m * n) = eint_of_enat m * eint_of_enat n"
  by (cases m n rule: enat2_cases) auto

lemmas eint_of_enat_pushin = eint_of_enat_add eint_of_enat_sub eint_of_enat_mult
lemmas eint_of_enat_pushout = eint_of_enat_pushin[symmetric]

lemma eint_of_enat_nonneg: "eint_of_enat n \<ge> 0"
by(cases n) simp_all


end
