(*  title     : An encoding of ITL in Isabelle/HOL
    Authors   : Antonio Cau     <cau.researcher at gmail.com>
                Ben Moszkowski
                David Smallwood <drs at dmu.ac.uk>
    Maintainer: Antonio Cau     <cau.researcher at gmail.com>        
    License   : BSD
*)
section \<open>Chopstar and variants\<close>

theory Chopstar
 imports
   SChopTheorems
begin
(*
 sledgehammer_params [minimize=true,preplay_timeout=10,timeout=60,verbose=true,
                        isar_proofs=false,
                    provers="z3 cvc4 e spass vampire" ]
*)
text \<open>
This theory defines the chopstar operator for infinite ITL and provides a library of lemmas. 
We also define the strong version schopstar, the weak version wpowerstar and a semantic version
achopstar. The wpowerstar corresponds to the Kleene star operator from Kleene Algebra 
\cite{Georg:Klealg}. We provide lemmas that express various relationships between them. 
We also ported the numerous Kleene algebra lemmas from \cite{Georg:Klealg} to ITL. 
\<close>
subsection \<open>Definitions\<close>

primrec wpower_d :: "('a::world) formula \<Rightarrow> nat \<Rightarrow> 'a formula "
where  wpow_0  : "(wpower_d F 0) = LIFT(empty)"
     | wpow_Suc: "(wpower_d F (Suc n)) = LIFT((F);(wpower_d F n)) "

syntax
 "_wpower_d"      :: "[lift,nat] \<Rightarrow> lift"       ("(wpower _ _)" [88,88] 87) 

syntax (ASCII)
 "_wpower_d"      :: "[lift,nat] \<Rightarrow> lift"       ("(wpower _ _)" [88,88] 87) 

translations
 "_wpower_d"     \<rightleftharpoons> "CONST wpower_d"

definition fpower_d :: "('a::world) formula \<Rightarrow> nat \<Rightarrow> 'a formula "
where 
  "fpower_d F n \<equiv> LIFT(wpower (F \<and> finite) n) "


syntax
 "_fpower_d"      :: "[lift,nat] \<Rightarrow> lift"      ("(fpower _ _)" [88,88] 87) 

syntax (ASCII)
 "_fpower_d"      :: "[lift,nat] \<Rightarrow> lift"      ("(fpower _ _)" [88,88] 87)

translations
 "_fpower_d"     \<rightleftharpoons> "CONST fpower_d"


definition len_d :: "nat \<Rightarrow> ('a::world) formula "
where "len_d n \<equiv> LIFT(wpower skip n)"

definition wpowerstar_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "wpowerstar_d F \<equiv> LIFT((\<exists> k. wpower F k))"

definition fpowerstar_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "fpowerstar_d F \<equiv> LIFT(\<exists> k. fpower F k)"


syntax
 "_len_d"          :: "nat \<Rightarrow> lift"         ("(len _)" [88] 87) 
 "_wpowerstar_d"   :: "lift \<Rightarrow> lift"        ("(wpowerstar _)" [85] 85)
 "_fpowerstar_d"   :: "lift \<Rightarrow> lift"        ("(fpowerstar _)" [85] 85)


syntax (ASCII)
 "_len_d"          :: "nat \<Rightarrow> lift"         ("(len _)" [88] 87) 
 "_wpowerstar_d"   :: "lift \<Rightarrow> lift"        ("(wpowerstar _)" [85] 85)
 "_fpowerstar_d"   :: "lift \<Rightarrow> lift"        ("(fpowerstar _)" [85] 85)


translations
 "_len_d"          \<rightleftharpoons> "CONST len_d" 
 "_wpowerstar_d"   \<rightleftharpoons> "CONST wpowerstar_d"
 "_fpowerstar_d"   \<rightleftharpoons> "CONST fpowerstar_d"



definition powerstar_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "powerstar_d F \<equiv> LIFT(fpowerstar F;(empty \<or> (F \<and> inf)))"

syntax
 "_powerstar_d"    :: "lift \<Rightarrow> lift"        ("(powerstar _)" [85] 85)

syntax (ASCII)
 "_powerstar_d"    :: "lift \<Rightarrow> lift"        ("(powerstar _)" [85] 85)

translations
 "_powerstar_d"    \<rightleftharpoons> "CONST powerstar_d"

definition chopstar_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "chopstar_d F \<equiv> LIFT(powerstar (F \<and> more))"

definition schopstar_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "schopstar_d F \<equiv> LIFT(fpowerstar (F \<and> more))"


syntax
 "_chopstar_d"     :: "lift \<Rightarrow> lift"        ("(_\<^sup>\<star>)" [85] 85)
 "_schopstar_d"     ::"lift \<Rightarrow> lift"        ("(schopstar _)" [85] 85)


syntax (ASCII)
 "_chopstar_d"     :: "lift \<Rightarrow> lift"        ("(chopstar _)" [85] 85)
 "_schopstar_d"     ::"lift \<Rightarrow> lift"        ("(schopstar _)" [85] 85)


translations
 "_chopstar_d"     \<rightleftharpoons> "CONST chopstar_d"
 "_schopstar_d"    \<rightleftharpoons> "CONST schopstar_d"


definition while_d :: "('a::world) formula  \<Rightarrow> 'a formula \<Rightarrow> 'a formula "
where "while_d F G \<equiv> LIFT( ( F \<and> G)\<^sup>\<star> \<and> (fin ((\<not>F))) ) "

syntax
 "_while_d"  :: "[lift,lift] \<Rightarrow> lift" ("(while _ do _  )"  [88,88] 87)

syntax (ASCII)
 "_while_d"  :: "[lift,lift] \<Rightarrow> lift" ("(while _ do _  )"  [88,88] 87)

translations
 "_while_d"  \<rightleftharpoons> "CONST while_d"


definition swhile_d :: "('a::world) formula  \<Rightarrow> 'a formula \<Rightarrow> 'a formula "
where "swhile_d F G \<equiv> LIFT( schopstar( F \<and> G) \<and> (sfin ((\<not>F))) ) "

definition repeat_d :: "('a::world) formula  \<Rightarrow> 'a formula \<Rightarrow> 'a formula "
where "repeat_d F G \<equiv> LIFT(F;while (\<not> G) do F ) "

syntax
 "_swhile_d"  :: "[lift,lift] \<Rightarrow> lift" ("(swhile _ do _  )"  [88,88] 87)
 "_repeat_d"  :: "[lift,lift] \<Rightarrow> lift" ("(repeat _ until _  )"  [88,88] 87)

syntax (ASCII)
 "_swhile_d"  :: "[lift,lift] \<Rightarrow> lift" ("(swhile _ do _  )"  [88,88] 87)
 "_repeat_d"  :: "[lift,lift] \<Rightarrow> lift" ("(repeat _ until _  )"  [88,88] 87)

translations
 "_swhile_d" \<rightleftharpoons> "CONST swhile_d"
 "_repeat_d" \<rightleftharpoons> "CONST repeat_d"

definition srepeat_d :: "('a::world) formula  \<Rightarrow> 'a formula \<Rightarrow> 'a formula "
where "srepeat_d F G \<equiv> LIFT(F\<frown>swhile (\<not> G) do F ) "

syntax
 "_srepeat_d"            :: "[lift,lift] \<Rightarrow> lift" ("(srepeat _ until _  )"  [88,88] 87)

syntax (ASCII)
 "_srepeat_d"            :: "[lift,lift] \<Rightarrow> lift" ("(srepeat _ until _  )"  [88,88] 87)

translations
 "_srepeat_d"            \<rightleftharpoons> "CONST srepeat_d"

definition aschopstar_d :: "('a::world) formula \<Rightarrow> 'a formula"
where "aschopstar_d F \<equiv> 
        \<lambda>s. (\<exists> (l:: nat nellist). 
                   (nnth l 0) = 0 \<and> nfinite l \<and> nidx l  \<and> 
                   (enat (nlast l)) = (nlength s) \<and> nfinite s \<and> 
                   (\<forall> (i::nat) . ( (enat i)< (nlength l)) \<longrightarrow> 
                               ((nsubn s (nnth l i) (nnth l (Suc i)) ) \<Turnstile> F) )
             )"

syntax
 "_aschopstar_d"     :: "lift \<Rightarrow> lift"        ("(aschopstar _)" [85] 85)

syntax (ASCII)
 "_aschopstar_d"     :: "lift \<Rightarrow> lift"        ("(aschopstar _)" [85] 85)

translations
 "_aschopstar_d"    \<rightleftharpoons> "CONST aschopstar_d"
 
lemma FChopSem_var [ mono]:
 "(w \<Turnstile>  f;g) =
  (( (\<exists>n. enat n \<le> nlength w \<and> f ( (ntaken n w)) \<and> g  (ndropn n  w)) \<or>
          (\<not> nfinite w \<and> f w)) )"
by  (simp add: itl_defs)

inductive istar_d :: "('a::world) formula \<Rightarrow> 'a formula" 
 for F where 
 " (s \<Turnstile> empty) \<Longrightarrow> (s \<Turnstile> (istar_d F)) "
| " (s \<Turnstile>  F;(istar_d F)) \<Longrightarrow>  (s \<Turnstile> (istar_d F))  "


syntax 
 "_istar_d" :: "lift \<Rightarrow> lift"        ("(istar _)" [85] 85)

syntax (ASCII)
 "_istar_d" :: "lift \<Rightarrow> lift"        ("(istar _)" [85] 85)

translations
 "_istar_d"        \<rightleftharpoons> "CONST istar_d" 




subsection \<open>Semantic lemmas\<close>


lemma ChopExist:
 "\<turnstile>  (\<exists> k. f;g k) = f;(\<exists> k. g k)"
by (auto simp add: itl_defs Valid_def)

lemma SChopExist:
 "\<turnstile>  (\<exists> k. f\<frown>g k) = f\<frown>(\<exists> k. g k)"
by (auto simp add: itl_defs Valid_def)

lemma ExistChop:
 "\<turnstile> (\<exists> k. (g k);f) =  (\<exists> k. g k);f"
by (auto simp add: itl_defs Valid_def)

lemma ExistSChop:
 "\<turnstile> (\<exists> k. (g k)\<frown>f) =  (\<exists> k. g k)\<frown>f"
by (auto simp add: itl_defs Valid_def)


lemma wpowersem1:
 "(\<sigma> \<Turnstile>  (\<exists> k. wpower f k) = ( empty \<or> (\<exists> k. wpower f (Suc k)))) "
proof (auto) 
 show "\<And>x. \<sigma> \<Turnstile> (wpower f x)  \<Longrightarrow> \<forall>k. \<not> (\<sigma> \<Turnstile>  f;wpower f k)  \<Longrightarrow> \<sigma> \<Turnstile> empty" 
  by (metis not0_implies_Suc wpow_0 wpow_Suc)
 show " \<sigma> \<Turnstile> empty  \<Longrightarrow> \<exists>x. \<sigma> \<Turnstile> (wpower f x)  "
  by (metis wpow_0)
 show " \<And>k. \<sigma> \<Turnstile> (f;wpower f k)  \<Longrightarrow> \<exists>x. \<sigma> \<Turnstile> (wpower f x)   "
  by (metis wpow_Suc)
qed

lemma fpowersem1:
 "(\<sigma> \<Turnstile>  (\<exists> k. fpower f k) = ( empty \<or> (\<exists> k. fpower f (Suc k)))) "
unfolding fpower_d_def using wpowersem1[of "LIFT (f \<and> finite)" \<sigma>] 
by blast



lemma wpowersem:
 " \<turnstile> (\<exists> k. wpower f k) = (  empty \<or> f;(\<exists> k. (wpower f k)))"
proof -
 have 1: "\<turnstile> (\<exists> k. wpower f k) = ( empty \<or> (\<exists> k. wpower f (Suc k))) "
   using wpowersem1 by blast
 have 2: "\<turnstile> (\<exists> k. wpower f (Suc k)) = (\<exists> k. f;wpower f k) "
   by simp
 have 3: "\<turnstile> (\<exists> k. f;(wpower f k)) = f;(\<exists> k. (wpower f k)) " 
   using ChopExist by blast
 from 1 2 3 show ?thesis by fastforce
qed


lemma fpowersem:
 " \<turnstile> (\<exists> k. fpower f k) = (  empty \<or> (f \<and> finite);(\<exists> k. (fpower f k)))"
unfolding fpower_d_def using wpowersem[of "LIFT (f \<and> finite)"] 
by blast


lemma finite_nidx_bounded_nlast: 
assumes "nfinite l"
        "nidx l"
        "(enat (nlast l)) = (nlength s)"
        "(enat i)\<le> (nlength l)" 
shows "  (nnth l i) \<le> nlength s"
using assms 
by (metis enat_ord_simps(1) nfinite_nlength_enat nidx_less_eq nnth_nlast 
    the_enat.simps verit_comp_simplify1(2))

lemma aschopstar_wpower_chain_a:
assumes "(\<exists> (l:: nat nellist). 
                 (nlength l) = (Suc n) \<and> nfinite l \<and> nidx l \<and> (nnth l 0) = 0 \<and>
                  (enat (nlast l)) = (nlength \<sigma>) \<and> nfinite \<sigma> \<and>
                  (\<forall>(i::nat). i< (nlength l) \<longrightarrow> 
                               ((nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> f)
                  )
                )" 
shows "(\<exists> k. 0 \<le>k \<and> k \<le> nlength \<sigma> \<and> 0< k \<and> (nsubn \<sigma> 0 k  \<Turnstile> f) \<and> 
               (\<exists> ls. nfinite ls \<and> (nlength ls) = n \<and> nidx ls \<and> (nnth ls 0) = 0 \<and>  
                      (enat (nlast ls)) = (nlength (ndropn k \<sigma>)) \<and> nfinite(ndropn k \<sigma>) \<and>
                      (\<forall>(i::nat).  i< (nlength ls) \<longrightarrow> 
                               ((nsubn (ndropn k \<sigma>) (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)
                    ))
         )"
proof -
 obtain l where 1: "(nlength l) = (Suc n) \<and> nfinite l \<and>  nidx l \<and> (nnth l 0) = 0 \<and> 
                    (enat (nlast l))  = (nlength \<sigma>) \<and> nfinite \<sigma> \<and>
                           (\<forall> (i::nat).  i< (nlength l) \<longrightarrow> 
                               ((nsubn \<sigma> (nnth l i) (nnth l (Suc i)) ) \<Turnstile> f)
                           )" 
   using assms by auto
 have 2: "nlength l >0"
   using 1 by (simp add: enat_0_iff(1))
 have 3: "l = NCons (nnth l 0)  (ndropn 1 l)"
   using "2" by (simp add: ndropn_Suc_conv_ndropn zero_enat_def)
 have 4: "nlength (ndropn 1 l) = n" 
  by (simp add: "1")
 have 5: " 0\<le>(nnth l 0) "
  by simp
 have 6: "(nnth l 0) \<le> nlength \<sigma>"
  using "1" using i0_lb zero_enat_def by presburger
 have 7: "(nnth l 0) = 0"
  using "1" by blast
 have 71: " (nnth (ndropn 1 l) 0) = (nnth l 1)"
  by auto
 have 8: "nnth l 0 < nnth (ndropn 1 l) 0"
     by (metis "1" "71" One_nat_def eSuc_enat enat_ord_simps(2) ileI1 nidx_gr_first zero_less_Suc)
 have 9: "nidx  (ndropn 1 l)"
  by (metis "1" "2" One_nat_def Suc_eq_plus1 Suc_ile_eq enat_min_eq ndropn_nlength ndropn_nnth 
      nidx_expand plus_1_eq_Suc plus_enat_simps(1) zero_enat_def) 
 have 10: "(enat (nlast (ndropn 1 l)))  = (nlength \<sigma>) "
   using "1" "3" by (metis nlast_NCons)
 have 101: "\<And>j. j \<le> nlength (ndropn 1 l) \<longrightarrow>
                 nnth (nmap (\<lambda>x. x-  (nnth l 1)) (ndropn 1 l)) j =
                   (nnth l (Suc j)) - (nnth l 1)"
    by simp
 have 102: "\<And>j. j \<le> nlength (ndropn 1 l) \<longrightarrow>
                 (nnth l 1) \<le> (nnth l (Suc j)) "
          by (simp add: "1" nidx_less_eq)
 have 103: "\<And>j. j< nlength (ndropn 1 l) \<longrightarrow>
                  (nnth l (Suc j)) - (nnth l 1) <
                  (nnth l (Suc (Suc j))) - (nnth l 1) "
      using 1 nidx_expand[of l]  
      by (metis "102" "4" diff_less_mono eSuc_enat ileI1 iless_Suc_eq order_less_imp_le)
 have 11: "nidx (nmap (\<lambda>x. x-  (nnth l 1)) (ndropn 1 l))" 
    using 103 101 nidx_expand[of "(nmap (\<lambda>x. x-  (nnth l 1)) (ndropn 1 l))"]
      using Suc_ile_eq by force 
 have 12: "nlength (nmap (\<lambda>x. x-  (nnth l 1)) (ndropn 1 l)) = n" 
  using "4" by auto
 have 13: "(enat (nlast (nmap (\<lambda>x. x-  (nnth l 1)) (ndropn 1 l))))  
            = 
           (nlength (ndropn (nnth l 1) \<sigma>)) " 
     by (metis "1" "10" idiff_enat_enat ndropn_nlength nfinite_ndropn nlast_nmap)
 have 14: "(nsubn \<sigma>  0 (nnth l 1)  \<Turnstile> f)" 
   by (metis "1" One_nat_def enat_ord_simps(2) zero_less_Suc)
 have 15: "(\<forall> (i::nat). i < nlength (ndropn 1 l) \<longrightarrow>
              (nsubn \<sigma> (nnth (ndropn 1 l) i) (nnth (ndropn 1 l) (Suc i)) \<Turnstile> f))"
   using "1" "3" eSuc_enat ileI1 iless_Suc_eq ndropn_nnth nlength_NCons plus_1_eq_Suc by metis
 
 have 16: "(\<forall> (i::nat). i< nlength (nmap (\<lambda>x. x-  (nnth l 1)) (ndropn 1 l)) \<longrightarrow> 
            (nsubn \<sigma> ((nnth (nmap (\<lambda>x. x-  (nnth l 1)) (ndropn 1 l)) i) + (nnth l 1))
                     ((nnth (nmap (\<lambda>x. x-  (nnth l 1)) (ndropn 1 l)) (Suc i)) + (nnth l 1))\<Turnstile> f)) "
   using "102" "12" "15" "4" by force
 have 17: "(\<forall> (i::nat). i< nlength (nmap (\<lambda>x. x-  (nnth l 1)) (ndropn 1 l)) \<longrightarrow> 
               ((nsubn (ndropn (nnth l 1) \<sigma>)  
                      (nnth (nmap (\<lambda>x. x-  (nnth l 1)) (ndropn 1 l)) i) 
                      (nnth (nmap (\<lambda>x. x-  (nnth l 1)) (ndropn 1 l)) (Suc i))) \<Turnstile> f))  "
   by (metis "11" "16" eSuc_enat ileI1 nidx_expand nsubn_ndropn)
 have 18: "nfinite (nmap (\<lambda>x. x-  (nnth l 1)) (ndropn 1 l))"
    using "12" nlength_eq_enat_nfiniteD by blast
 have 19: "nfinite (ndropn (nnth l 1) \<sigma>)"
    using "1" nfinite_ndropn_a by blast
 have 20: "(\<exists> ls. nfinite ls \<and> (nlength ls) = n \<and> nidx ls \<and> (nnth ls 0) = 0 \<and>  
                      (enat (nlast ls)) = (nlength (ndropn (nnth l 1) \<sigma>)) \<and> nfinite(ndropn (nnth l 1) \<sigma>) \<and>
                  (\<forall>(i::nat).  i< (nlength ls) \<longrightarrow> 
                               ((nsubn (ndropn (nnth l 1) \<sigma>) (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)
                    ))" 
  by (metis "11" "12" "13"  "17" "18" "19" "71" diff_self_eq_0 enat_le_plus_same(1) gen_nlength_def 
      nlength_code nnth_nmap)
 show ?thesis 
 by (metis "1" "2" "20" "71" "8" One_nat_def Suc_ile_eq finite_nidx_bounded_nlast zero_enat_def 
     zero_order(1))
qed

lemma aschopstar_wpower_chain_b:
assumes  "(\<exists> k. 0 \<le>k \<and> k \<le> nlength \<sigma> \<and> 0< k \<and>
               (nsubn \<sigma> 0 k  \<Turnstile> f) \<and> 
               (\<exists> ls. nfinite ls \<and> (nlength ls) = n \<and> nidx ls \<and> (nnth ls 0) = 0 \<and>  
                      (enat (nlast ls)) = (nlength (ndropn k \<sigma>)) \<and> nfinite(ndropn k \<sigma>) \<and>
                  (\<forall>(i::nat).  i< (nlength ls) \<longrightarrow> 
                               ((nsubn (ndropn k \<sigma>) (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)
                    ))
         )"
shows  "(\<exists> (l:: nat nellist). 
                 (nlength l) = (Suc n) \<and> nfinite l \<and> nidx l \<and> (nnth l 0) = 0 \<and>
                  (enat (nlast l)) = (nlength \<sigma>) \<and> nfinite \<sigma> \<and>
                           (\<forall>(i::nat). i< (nlength l) \<longrightarrow> 
                               ((nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> f)
                           )
                )" 
proof -  
 obtain k where 1: "0 \<le>k \<and> k \<le> nlength \<sigma> \<and> k > 0 \<and>
               (nsubn \<sigma> 0 k  \<Turnstile> f) \<and> 
               (\<exists> ls. nfinite ls \<and> (nlength ls) = n \<and> nidx ls \<and> (nnth ls 0) = 0 \<and>  
                      (enat (nlast ls)) = (nlength (ndropn k \<sigma>)) \<and> nfinite(ndropn k \<sigma>) \<and>
                  (\<forall>(i::nat).  i< (nlength ls) \<longrightarrow> 
                               ((nsubn (ndropn k \<sigma>) (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)
                    ))" 
 using assms by auto
 have 2: "(\<exists> ls. nfinite ls \<and> (nlength ls) = n \<and> nidx ls \<and> (nnth ls 0) = 0 \<and>  
                      (enat (nlast ls)) = (nlength (ndropn k \<sigma>)) \<and> nfinite(ndropn k \<sigma>) \<and>
                  (\<forall>(i::nat).  i< (nlength ls) \<longrightarrow> 
                               ((nsubn (ndropn k \<sigma>) (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)
                    ))" 
  using 1 by auto
 obtain ls where 3: "nfinite ls \<and> (nlength ls) = n \<and> nidx ls \<and> (nnth ls 0) = 0 \<and>  
                      (enat (nlast ls)) = (nlength (ndropn k \<sigma>)) \<and> nfinite(ndropn k \<sigma>) \<and>
                  (\<forall>(i::nat).  i< (nlength ls) \<longrightarrow> 
                               ((nsubn (ndropn k \<sigma>) (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)
                    )"
  using 2 by auto
 have 4: "nidx  (nmap (\<lambda>x. x+ k) ls)  "
   using "3"
   by (simp add: nidx_expand)  
 have 41: "\<And>j. j \<le> nlength (nmap (\<lambda>x. x+ k) ls) \<longrightarrow>
                0 < (nnth  (nmap (\<lambda>x. x+ k) ls) j) " 
     by (simp add: "1")
 have 5: "nidx (NCons 0 (nmap (\<lambda>x. x+ k) ls))"

   using 3 4 nidx_expand[of ls]  nidx_expand[of "(nmap (\<lambda>x. x+ k) ls)"] 
     nidx_expand[of "(NCons 0 (nmap (\<lambda>x. x+ k) ls))"]
   by (metis (no_types, lifting) "1" Suc_ile_eq diff_zero iless_Suc_eq le_add_diff_inverse lessI 
       less_Suc_eq_0_disj nlength_NCons nlength_nmap nnth_0 nnth_Suc_NCons nnth_nmap)
 have 6: "(nlength ((NCons 0 (nmap (\<lambda>x. x+ k) ls)))) = (Suc n)"
   by (simp add: "3" eSuc_enat)
 have 7: "(enat (nlast ((NCons 0 (nmap (\<lambda>x. x+ k) ls)))))  = (nlength \<sigma>)"
    by (metis "1" "3" add.commute enat.distinct(2) enat_add_sub_same less_eqE ndropn_nlength 
        nlast_NCons nlast_nmap plus_enat_simps(1))
 have 8: "(nsubn \<sigma> 0 k  \<Turnstile> f)"
  using 1 by auto

 have 9: "(\<forall>(i::nat).  i< (nlength ls) \<longrightarrow> 
                               ((nsubn (ndropn k \<sigma>) (nnth ls i) (nnth ls (Suc i)) ) \<Turnstile> f)
                    )"
   using 3 by auto
 have 10: "(\<forall>(i::nat). i< (nlength ls) \<longrightarrow> 
                ((nsubn \<sigma> ((nnth ls (i))+k) ((nnth ls ((i)+1))+k) ) \<Turnstile> f)
                    )" 
    by (metis "3" Suc_ile_eq add.commute add.right_neutral nidx_less nsubn_ndropn plus_1_eq_Suc)


 have 11: "(\<forall> (i::nat). i < nlength ((( (nmap (\<lambda>x. x+ k) ls)))) \<longrightarrow>
              (nsubn \<sigma> (nnth ((( (nmap (\<lambda>x. x+ k) ls)))) i) (nnth ((( (nmap (\<lambda>x. x+ k) ls)))) (Suc i)) \<Turnstile> f))"
   by (metis "10" add.commute eSuc_enat ileI1 nlength_nmap nnth_nmap order_less_imp_le plus_1_eq_Suc)
 have 12: "(\<forall>i. (0<i \<and> i< 1+ (nlength (nmap (\<lambda>x. x+ k) ls))) \<longrightarrow> 
                               ((nsubn \<sigma> ((nnth (nmap (\<lambda>x. x+ k) ls) (i-1))) 
                                     ((nnth (nmap (\<lambda>x. x+ k) ls) ((i)))) ) \<Turnstile> f)
                    )" 
  using 11
  by (metis "3" Suc_diff_1 Suc_ile_eq eSuc_enat iless_Suc_eq nlength_nmap one_enat_def plus_1_eq_Suc 
      plus_enat_simps(1))
 have 13: "(\<forall> (i::nat). i < nlength ((NCons 0 ( (nmap (\<lambda>x. x+ k) ls)))) \<longrightarrow>
              (nsubn \<sigma> (nnth (((NCons 0 (nmap (\<lambda>x. x+ k) ls)))) i) 
                       (nnth (((NCons 0 (nmap (\<lambda>x. x+ k) ls)))) (Suc i)) \<Turnstile> f))" 
    using 12
    using "1" "11" "3" "6" less_Suc_eq_0_disj by auto    

 have 14: " (nnth (NCons 0 (nmap (\<lambda>x. x+ k) ls)) 0) = 0"
   by simp
 have 15: "(nnth (NCons 0 (nmap (\<lambda>x. x+ k) ls)) (1)) = k"  
  using "4" by (simp add: "3")
 have 16: "(nsubn \<sigma> (nnth (NCons 0 (nmap (\<lambda>x. x+ k) ls)) 0) k  \<Turnstile> f) " 
  by (simp add: "8")
 have 17: "nfinite (NCons 0 (nmap (\<lambda>x. x+ k) ls))"
    using "6" nlength_eq_enat_nfiniteD by blast 
 show ?thesis  
 by (metis "13" "3" "5" "6" "7"  nfinite_NCons nfinite_ndropn nfinite_nmap nnth_0)
qed 

lemma chop_wpower_eqv_sem:
  "( (\<sigma> \<Turnstile>(\<exists> n. (wpower ((f \<and> more) \<and> finite) n)))) = 
   ((\<sigma> \<Turnstile> empty) \<or>  ( (\<sigma> \<Turnstile> ((f \<and> more) \<and> finite);(\<exists> n. (wpower ((f \<and> more) \<and> finite) n)))))"
using wpowersem by fastforce

lemma aschopstar_eqv_wpower_chop_help:
  " ( \<sigma> \<Turnstile> wpower ((f \<and> more) \<and> finite) n) = 
    (\<exists> (l:: nat nellist). 
                 (nlength l) = (n) \<and> nfinite l \<and> nidx l \<and> (nnth l 0) = 0 \<and>
                  (enat (nlast l)) = (nlength \<sigma>) \<and> nfinite \<sigma> \<and>
                           (\<forall>(i::nat). i< (nlength l) \<longrightarrow> 
                               ((nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> f)
                           )
                )
   "
proof 
 (induct n arbitrary: \<sigma>)
 case 0
 then show ?case
    by (auto simp add: empty_defs nidx_expand nnth_nlast zero_enat_def)
     (metis eSuc_enat enat_0_iff(1) iless_Suc_eq leD nfinite_conv_nlength_enat nlast_NNil nlength_NNil 
      nnth_NNil zero_le)
 next
 case (Suc n)
 then show ?case 
  proof -
   have "(\<sigma> \<Turnstile> wpower ((f \<and> more) \<and> finite) (Suc n)) = 
         (\<sigma> \<Turnstile> (((f \<and> more) \<and> finite);(wpower ((f \<and> more) \<and> finite) n)))  " 
   by simp
   also have " ... =
             (\<exists> k. 0 \<le>k \<and> k \<le> nlength (\<sigma>) \<and> k > 0 \<and>
               (ntaken k (\<sigma>) \<Turnstile> f) \<and> 
               (ndropn k (\<sigma>) \<Turnstile> wpower ((f \<and> more) \<and> finite) n)
             )
           " 
  using enat_0_iff(1) le_zero_eq by (auto simp add: more_defs chop_defs finite_defs ) 
  also have "... =
           (\<exists> k. 0 \<le>k \<and> k \<le>nlength (\<sigma>) \<and> k > 0 \<and>
               (nsubn \<sigma> 0 k  \<Turnstile> f) \<and> 
               (ndropn  k (\<sigma>) \<Turnstile> wpower ((f \<and> more) \<and> finite) n)
           )" 
    by (metis One_nat_def Suc_diff_1 Suc_diff_Suc diff_add ndropn_0 ntaken_ndropn)
  also have  "... =
           (\<exists> k. 0 \<le>k \<and> k \<le> nlength (\<sigma>) \<and> k > 0 \<and>
               (nsubn \<sigma> 0 k  \<Turnstile> f) \<and> 
               (\<exists> l. nfinite l \<and> (nlength l) = n \<and> nidx l \<and> (nnth l 0) = 0 \<and>  
                      (enat (nlast l)) = (nlength (ndropn k \<sigma>)) \<and> nfinite(ndropn k \<sigma>) \<and>
                  (\<forall>(i::nat).  i< (nlength l) \<longrightarrow> 
                               ((nsubn (ndropn k \<sigma>) (nnth l i) (nnth l (Suc i)) ) \<Turnstile> f)
                    ))
             )" 
    using Suc.hyps by blast
   also have " ... =
           (\<exists> (l:: nat nellist). 
                 (nlength l) = (Suc n) \<and> nfinite l \<and> nidx l \<and> (nnth l 0) = 0 \<and>
                  (enat (nlast l)) = (nlength \<sigma>) \<and> nfinite \<sigma> \<and>
                           (\<forall>(i::nat). i< (nlength l) \<longrightarrow> 
                               ((nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> f)
                           )
                )
           " 
    using aschopstar_wpower_chain_a[of "n" "\<sigma>" "f"]
          aschopstar_wpower_chain_b [of "\<sigma>" "f" "n"] by auto
   finally show "(\<sigma> \<Turnstile> wpower ((f \<and> more) \<and> finite) (Suc n)) = 
                 (\<exists> (l:: nat nellist). 
                 (nlength l) = (Suc n) \<and> nfinite l \<and> nidx l \<and> (nnth l 0) = 0 \<and>
                  (enat (nlast l)) = (nlength \<sigma>) \<and> nfinite \<sigma> \<and>
                  
                           (\<forall>(i::nat). i< (nlength l) \<longrightarrow> 
                               ((nsubn \<sigma> (nnth l i) (nnth l (Suc i))) \<Turnstile> f)
                           )
                ) " . 
  qed
qed

lemma aschopstar_eqv_power_chop:
  "(\<sigma> \<Turnstile> aschopstar f) = (  (\<sigma> \<Turnstile> (\<exists> k. wpower ((f \<and> more) \<and> finite) k))) "
using nfinite_conv_nlength_enat by (simp add: aschopstar_d_def aschopstar_eqv_wpower_chop_help )
blast


lemma ASChopstarEqvSem:
  " (\<sigma> \<Turnstile> (aschopstar f =  (empty \<or>  ((f \<and> more) \<and> finite); (aschopstar f))) )"
proof -
 have 1: "(\<sigma> \<Turnstile> aschopstar f) = (  (\<sigma> \<Turnstile> (\<exists> k. wpower ((f \<and> more)\<and> finite) k)))"
 using aschopstar_eqv_power_chop by simp
 have 2: "(  (\<sigma> \<Turnstile> (\<exists> k. wpower ((f \<and> more)\<and> finite) k))) = 
          ( (\<sigma> \<Turnstile> empty) \<or> (\<sigma> \<Turnstile> ((f \<and> more) \<and> finite);(\<exists>n. wpower ((f \<and> more)\<and>finite) n))) " 
 using chop_wpower_eqv_sem by simp
 have 3: "(\<sigma> \<Turnstile> ((f \<and> more)\<and>finite);(\<exists>n. wpower ((f \<and> more)\<and>finite) n)) =
         ( \<exists>n. n\<le>nlength \<sigma> \<and> ((ntaken n \<sigma>) \<Turnstile> (f  \<and> more) \<and> finite) \<and> 
                         ((ndropn n \<sigma>) \<Turnstile> (\<exists>x. (wpower ((f \<and> more)\<and>finite) x) )))" 
  by (simp add: chop_defs finite_defs) blast
 have 4: "( \<exists>n. n\<le>nlength \<sigma> \<and> ((ntaken n \<sigma>) \<Turnstile> (f  \<and> more) \<and> finite) \<and> 
                          ((ndropn n \<sigma>) \<Turnstile> (\<exists>x. (wpower ((f \<and> more)\<and>finite) x) ))) =
          ( \<exists>n. n\<le>nlength \<sigma> \<and> ((ntaken n \<sigma>) \<Turnstile> (f  \<and> more)\<and>finite) \<and> ((ndropn n \<sigma>) \<Turnstile>  aschopstar f ))"
   by (simp add: aschopstar_eqv_power_chop) 
 have 5: "( \<exists>n. n\<le>nlength \<sigma> \<and> ((ntaken n \<sigma>) \<Turnstile> (f  \<and> more)\<and>finite) \<and> ((ndropn n \<sigma>) \<Turnstile>  aschopstar f )) =
          (\<sigma> \<Turnstile>  ((f \<and> more)\<and>finite); (aschopstar f) )"
    by (simp add: chop_defs finite_defs) blast
 have 6: "( (\<sigma> \<Turnstile> empty) \<or> (\<sigma> \<Turnstile> ((f \<and> more)\<and>finite);(\<exists>n. wpower ((f \<and> more)\<and>finite) n))) =
          ( \<sigma> \<Turnstile> (empty \<or>  ((f \<and> more)\<and>finite); (aschopstar f)))"
   using "3" "4" "5" by auto 
 show ?thesis using 1 2 6 by auto
qed

lemma ASChopstarEqvSChopstar:
 " \<turnstile> (aschopstar f) = (schopstar f)"
by (simp add: Valid_def schopstar_d_def aschopstar_eqv_power_chop fpowerstar_d_def fpower_d_def)


lemma len_defs :
 " (w \<Turnstile> len n) = (nlength w =n) "
proof  
   (simp add: len_d_def ) 
   show " (w \<Turnstile> (wpower skip n)) = (nlength w = n)" 
   proof (induct n arbitrary:w)
   case 0
   then show ?case by (simp add: empty_defs zero_enat_def)
   next
   case (Suc n)
   then show ?case 
   by (auto simp add: min_def len_d_def empty_defs chop_defs skip_defs finite_defs nlength_eq_enat_nfiniteD)
      (metis One_nat_def enat.distinct(2) enat_add_sub_same le_iff_add plus_1_eq_Suc plus_enat_simps(1))
   qed
qed



lemma PowerstarEqvSemhelp1:
 " \<turnstile> empty;(empty \<or> (f \<and> inf)) = (empty \<or> (f \<and> inf))"
using EmptyChopSem by blast


lemma PowerstarEqvSemhelp2:
 " \<turnstile> (f \<and> inf) = (f \<and> inf);g"
by (auto simp add: Valid_def itl_defs)
  
lemma PowerstarEqvSemhelp3:
 " \<turnstile> ((f \<and> inf);g \<or> (f \<and> finite);g) = (f ;g)"
by (auto simp add: Valid_def itl_defs)



lemma WPowerstarEqvSem:
  " (\<sigma> \<Turnstile> (wpowerstar f) =  (empty \<or>  f;(wpowerstar f) ))"
by (metis intD wpowersem wpowerstar_d_def)

lemma FPowerstarEqvSem:
  " (\<sigma> \<Turnstile> (fpowerstar f) =  (empty \<or>  (f \<and> finite);(fpowerstar f) ))"
by (metis fpowersem fpowerstar_d_def intD)



lemma PowerstarEqvSem:
  " (\<sigma> \<Turnstile> (powerstar f) =  (empty \<or>  f;(powerstar f) ))"
proof -
 have 1: "(\<sigma> \<Turnstile> (powerstar f)) = 
          ( \<sigma> \<Turnstile> (\<exists> k. fpower f k);(empty \<or> f \<and> inf)) "
   by (simp add: powerstar_d_def fpowerstar_d_def)
 have 2: "( \<sigma> \<Turnstile> (\<exists> k. fpower f k);(empty \<or> f \<and> inf)) =
          (\<sigma> \<Turnstile> (empty \<or> (f \<and> finite);(\<exists> k. (fpower f k)));(empty \<or> f \<and> inf))"
   using fpowersem by (metis inteq_reflection)
 have 3: " (\<sigma> \<Turnstile> (empty \<or> (f \<and> finite);(\<exists> k. (fpower f k)));(empty \<or> f \<and> inf)) =
           (\<sigma> \<Turnstile> empty;(empty \<or> (f \<and> inf)) \<or> 
                ((f \<and> finite);(\<exists> k. (fpower f k)));(empty \<or> f \<and> inf))"
   by (metis OrChopEqv inteq_reflection) 
 have 4: "(\<sigma> \<Turnstile> empty;(empty \<or> (f \<and> inf)) \<or> 
                ((f \<and> finite);(\<exists> k. (fpower f k)));(empty \<or> f \<and> inf)) =
          (\<sigma> \<Turnstile> (empty \<or> (f \<and> inf) \<or> ((f \<and> finite);(\<exists> k. (fpower f k)));(empty \<or> f \<and> inf))) "
   using PowerstarEqvSemhelp1 
   by (metis (mono_tags, lifting) inteq_reflection unl_lift2)
 have 5: "(\<sigma> \<Turnstile> (empty \<or> (f \<and> inf) \<or> ((f \<and> finite);(\<exists> k. (fpower f k)));(empty \<or> f \<and> inf))) =
          (\<sigma> \<Turnstile> (empty \<or> (f \<and> inf);((\<exists> k. (fpower f k));(empty \<or> f \<and> inf)) \<or>
                     ((f \<and> finite);(\<exists> k. (fpower f k)));(empty \<or> f \<and> inf))) " 
   using PowerstarEqvSemhelp2 
   by (metis (mono_tags, lifting) inteq_reflection)
 have 51: "(\<sigma> \<Turnstile> ((f \<and> finite);(\<exists> k. (fpower f k)));(empty \<or> f \<and> inf) =
                 (f \<and> finite);((\<exists> k. (fpower f k));(empty \<or> f \<and> inf)))"
   by (auto simp add: ChopAssocSemHelp1)
 have 6: "(\<sigma> \<Turnstile> (empty \<or> (f \<and> inf);((\<exists> k. (fpower f k));(empty \<or> f \<and> inf)) \<or>
                     ((f \<and> finite);(\<exists> k. (fpower f k)));(empty \<or> f \<and> inf)) ) =
          (\<sigma> \<Turnstile> (empty \<or> (f \<and> inf);((\<exists> k. (fpower f k));(empty \<or> f \<and> inf)) \<or>
                      (f \<and> finite);((\<exists> k. (fpower f k));(empty \<or> f \<and> inf))))
         "
   using "51" by auto
 have 7 : "(\<sigma> \<Turnstile> (empty \<or> (f \<and> inf);((\<exists> k. (fpower f k));(empty \<or> f \<and> inf)) \<or>
                      (f \<and> finite);((\<exists> k. (fpower f k));(empty \<or> f \<and> inf)))) =
           (\<sigma> \<Turnstile> (empty \<or> f;((\<exists> k. (fpower f k));(empty \<or> f \<and> inf))))"
   using PowerstarEqvSemhelp3 by fastforce
 have 8: "(\<sigma> \<Turnstile> (empty \<or> f;((\<exists> k. (fpower f k));(empty \<or> f \<and> inf)))) =
          (\<sigma> \<Turnstile> (empty \<or> f;(powerstar f))) "
   by (simp add:  powerstar_d_def fpowerstar_d_def)
 from 1 2 3 4 5 6 7 8 show ?thesis by fastforce
qed


lemma wpowerchopsem:
 " \<turnstile> (\<exists> k. wpower (f \<and> more) k) = 
      ( empty \<or> (f \<and> more);(\<exists> k. (wpower (f \<and> more) k))
      )"
by (simp add: wpowersem)


lemma powerchopsem:
 " \<turnstile> (\<exists> k. fpower (f \<and> more) k) = 
      ( empty \<or> ((f \<and> more) \<and> finite);(\<exists> k. (fpower (f \<and> more) k))
      )"
using fpowersem by auto


lemma ChopstarEqvSem:
  " (\<sigma> \<Turnstile> f\<^sup>\<star> =  (empty \<or>  (f \<and> more); f\<^sup>\<star>) )"
by (metis PowerstarEqvSem chopstar_d_def)

lemma SChopstarEqvSem:
  " (\<sigma> \<Turnstile> (schopstar f) =  (empty \<or>  (f \<and> more)\<frown> (schopstar f) ))"
by (metis FPowerstarEqvSem schop_d_def schopstar_d_def)
  
lemma ChopstarEqv :
    "\<turnstile>  f\<^sup>\<star> =  (empty \<or>  (f \<and> more); f\<^sup>\<star>) " 
using ChopstarEqvSem Valid_def by blast

lemma IStarIntros:
 " \<turnstile>  empty \<or> f;(istar f) \<longrightarrow> (istar f)"
unfolding Valid_def using istar_d.intros by fastforce    


lemma IStarCases:
 " (w \<Turnstile> (istar F))  \<Longrightarrow> 
 ((w \<Turnstile> empty \<or> (F;istar F))  \<Longrightarrow> P) \<Longrightarrow> P " 
  using istar_d.cases[of F w P] by auto

lemma IStarEqvIStarSem: 
 " (w \<Turnstile> (istar f)) = (w \<Turnstile> empty \<or> f;(istar f))"
using IStarCases[of f  ]  
by (metis istar_d.intros(1) istar_d.intros(2) unl_lift2)

lemma IStarEqvIStar:
 "\<turnstile> (istar f) = (empty \<or> f;(istar f)) " 
using IStarEqvIStarSem[of f] unfolding Valid_def by auto 

lemma IStarInductSem:
 assumes "(\<And>s. (s \<Turnstile> empty ) \<Longrightarrow> P s)"
       " (\<And>s. (s \<Turnstile> (F;((istar F)  \<and> P)))  \<Longrightarrow> P s) "
 shows " (w \<Turnstile> (istar F) \<longrightarrow> P) " 
 using assms istar_d.induct[of F w P] 
  chop_defs[of F "LIFT ((istar F)  \<and> P)" ]
  unfolding chop_d_def  by (metis intensional_rews(3))

lemma IStarInduct:
 assumes "\<turnstile> empty \<or> f;((istar f) \<and> g) \<longrightarrow> g"
 shows   "\<turnstile> (istar f) \<longrightarrow> g "
using assms IStarInductSem[of g] unfolding Valid_def by (metis intensional_rews(3))

  
lemma IStarWeakInductSem:
 assumes "(\<And>s. (s \<Turnstile> empty ) \<Longrightarrow> P s)"
       " (\<And>s. (s \<Turnstile> (F;P))  \<Longrightarrow> P s) "
 shows " (w \<Turnstile> (istar F) \<longrightarrow> P) " 
using assms istar_d.induct[of F w P] using chop_defs[of F P] 
chop_defs[of F "LIFT ((istar F) \<and> P)"] unfolding chop_d_def 
by (metis intensional_rews(3))

lemma IStarWeakInduct:
 assumes "\<turnstile> empty \<or> f; g \<longrightarrow> g"
 shows   "\<turnstile> (istar f) \<longrightarrow> g "
using assms IStarWeakInductSem[of g] unfolding Valid_def
 by (metis intensional_rews(3))





subsection \<open>Helper lemmas\<close>

lemma AndEmptyChopAndEmptyEqvAndEmpty:
 "\<turnstile> (f \<and> empty);(f \<and> empty) = (f \<and> empty)"
proof -
 have 1: "\<turnstile> (f \<and> empty);(f \<and> empty) \<longrightarrow> (f \<and> empty)"
   by (metis ChopAndB ChopEmpty int_eq)
 have 2: "\<turnstile> (f \<and> empty) \<longrightarrow> (f \<and> empty);(f \<and> empty)"
   by (auto simp add: Valid_def itl_defs zero_enat_def)
     (metis iless_Suc_eq less_numeral_extra(1) ntake_0 ntake_all one_eSuc zero_enat_def)
 show ?thesis
   by (simp add: "1" "2" int_iffI) 
qed


subsection \<open>Properties of Chopstar and Chopplus\<close>

lemma FPowerstardef:
 "\<turnstile> fpowerstar f = (\<exists> n. fpower f n)"
by (simp add: fpowerstar_d_def)

lemma Powerstardef:
 "\<turnstile> powerstar f = (fpowerstar f);(empty \<or> (f \<and> inf))"
by (simp add: fpowerstar_d_def powerstar_d_def)

lemma Chopstardef:
 "\<turnstile> chopstar f = powerstar (f \<and> more)"
by (simp add: chopstar_d_def)

lemma SChopstardef:
 "\<turnstile> schopstar f = fpowerstar (f \<and> more)"
by (simp add: schopstar_d_def)

lemma WPowerEqRule: 
assumes "\<turnstile> f = g "
 shows  "\<turnstile> wpower f n = wpower g n " 
using assms
by (metis int_eq int_simps(20))


lemma WPowerCommute:
 "\<turnstile> (f) ;(wpower f n) = (wpower f n);(f)" 
proof
 (induct n)
 case 0
 then show ?case 
 by (metis ChopEmpty EmptyChop inteq_reflection wpow_0)
 next
 case (Suc n)
 then show ?case 
   by (metis ChopAssoc inteq_reflection wpow_Suc)
qed


lemma FPowerCommute:
 "\<turnstile> (f \<and> finite) ;fpower f n = fpower f n;(f \<and> finite)" 
unfolding fpower_d_def  using WPowerCommute[of "LIFT (f \<and> finite)"]
by blast
 
lemma WPowerChopInductL:
 assumes "\<turnstile> g \<or> f;h \<longrightarrow> h"
 shows   "\<turnstile> (wpower f n);g \<longrightarrow> h"
using assms 
proof 
 (induct n)
 case 0
 then show ?case using EmptyChop 
 by (metis MP Prop12 int_eq int_iffD1 int_simps(33) wpow_0) 
 next
 case (Suc n)
 then show ?case 
 by (metis ChopAssoc Prop05 Prop11 RightChopImpChop lift_imp_trans wpow_Suc) 
qed

lemma FPowerChopInductL:
 assumes "\<turnstile> g \<or> f;h \<longrightarrow> h"
 shows   "\<turnstile> (fpower f n);g \<longrightarrow> h"
unfolding fpower_d_def using assms WPowerChopInductL[of g "LIFT (f \<and> finite)" h n]
using AndChopA by fastforce

lemma FPowerChopInductFiniteL:
 assumes "\<turnstile> g \<or> (f \<and> finite);h \<longrightarrow> h"
 shows   "\<turnstile> (fpower f n);g \<longrightarrow> h"
unfolding fpower_d_def using assms WPowerChopInductL[of g "LIFT (f \<and> finite)" h n]
by blast

lemma WPowerChopInductMoreL:
 assumes "\<turnstile> g \<or> (f \<and> more);h \<longrightarrow> h"
 shows   "\<turnstile> (wpower f n);g \<longrightarrow> h"
using assms
proof 
 (induct n)
 case 0
 then show ?case 
 by (metis WPowerChopInductL wpow_0)
 next
 case (Suc n)
 then show ?case 
  proof - 
  have 1: "\<turnstile> wpower f (Suc n);g  = (f;wpower f n);g " 
   by simp
 have 2: "\<turnstile> (f;wpower f n);g = f;((wpower f n);g)"
   by (meson ChopAssoc Prop11)
 have 3: "\<turnstile> f;((wpower f n);g) \<longrightarrow> f;h"
   using RightChopImpChop Suc.hyps Suc.prems by blast
 have 31: "\<turnstile> f = ((f \<and> more) \<or> (f \<and> empty))"
   unfolding empty_d_def by fastforce
 have 4: "\<turnstile> f;h = (( (f \<and> more)); h \<or> ((f \<and> empty));h)"
    using "31" OrChopEqvRule by blast
 have 5: "\<turnstile> ( (f \<and> more)); h \<longrightarrow> h" 
   by (metis Prop03 Prop10 assms inteq_reflection lift_imp_trans)
 have 6: "\<turnstile> ( (f \<and> empty)); h \<longrightarrow> h" 
   by (metis  AndChopB EmptyChop inteq_reflection )
 from "5" "6" "4" "3" "2" "1" show ?thesis
 by (metis Prop02 inteq_reflection lift_imp_trans)
 qed
qed


lemma FPowerChopInductFiniteMoreL:
 assumes "\<turnstile> g \<or> ((f \<and> finite) \<and> more);h \<longrightarrow> h"
 shows   "\<turnstile> (fpower f n);g \<longrightarrow> h"
unfolding fpower_d_def using assms
 WPowerChopInductMoreL[of g "LIFT (f \<and> finite)" h n]
by blast


lemma FPowerChopInductInfL:
 assumes "\<turnstile> g \<or> f;h \<longrightarrow> h"
 shows   "\<turnstile> ((fpower f n);(f \<and> inf));g \<longrightarrow> h"
using assms 
proof
 (induct n)
 case 0
 then show ?case 
 by (metis (no_types, lifting) AndInfChopEqvAndInf ChopAssoc FPowerChopInductFiniteL PowerstarEqvSemhelp3
     Prop03 Prop10 Prop12 inteq_reflection)
 next
 case (Suc n)
 then show ?case 
 proof -
 have "\<turnstile>(f \<and> finite);(fpower f n;((f \<and> inf);g)) = (fpower f (Suc n);(f \<and> inf));g"
  by (metis ChopAssoc fpower_d_def int_eq wpow_Suc)
 then show ?thesis
   by (metis (no_types, lifting) AndChopA ChopAndB ChopAssoc Prop03 Prop10 Suc assms int_eq lift_imp_trans) 
 qed 
qed

lemma FChopInductInfMoreL:
 assumes "\<turnstile> g \<or> f;h \<longrightarrow> h"
 shows   "\<turnstile> ((fpower f n);((f \<and> more) \<and> inf));g \<longrightarrow> h"
using FPowerChopInductInfL 
by (metis AndMoreAndInfEqvAndInf assms inteq_reflection)

lemma WPowerChopInductR:
 assumes "\<turnstile> g \<or> h;f \<longrightarrow> h"
 shows   "\<turnstile> g;(wpower f n) \<longrightarrow> h"
using assms 
proof
 (induct n)
 case 0
 then show ?case using ChopEmpty 
 by (metis MP Prop12 int_iffD2 int_simps(33) inteq_reflection wpow_0)
 next
 case (Suc n)
 then show ?case 
 by (metis AndChopB ChopAssoc Prop03 Prop10 WPowerCommute inteq_reflection lift_imp_trans wpow_Suc)
qed


lemma FPowerChopInductR:
 assumes "\<turnstile> g \<or> h;f \<longrightarrow> h"
 shows   "\<turnstile> g;(fpower f n) \<longrightarrow> h"
unfolding fpower_d_def using assms WPowerChopInductR[of g h "LIFT (f \<and> finite)" n]
using ChopAndA by fastforce

lemma FpowerChopInductInfR:
 assumes "\<turnstile> g \<or> h;f \<longrightarrow> h"
 shows   "\<turnstile> g;((fpower f n);(f \<and> inf)) \<longrightarrow> h"
using assms
by (metis ChopAndA ChopAssoc FPowerChopInductR LeftChopImpChop Prop05 int_iffD1 lift_imp_trans) 

lemma WPowerStarCommute:
 "\<turnstile> f;(\<exists> n. wpower f n) = (\<exists> n. wpower f n);f"
proof - 
 have 1: "\<turnstile> f ;(\<exists> n. wpower f n) = (\<exists> n. f ;wpower f n)  "
  by (metis ChopExist Prop11)
 have 2: "\<turnstile> (\<exists> n. f ;wpower f n) = (\<exists> n. (wpower f n);f )"
   using WPowerCommute by (metis ExEqvRule)
 have 3: "\<turnstile> (\<exists> n. (wpower f n);f) = (\<exists> n. (wpower f n));f "
  by (simp add: ExistChop)
 from 1 2 3  show ?thesis by fastforce
qed

lemma FPowerStarCommute:
 "\<turnstile> (f \<and> finite);(\<exists> n. fpower f n) = (\<exists> n. fpower f n);(f \<and> finite)"
unfolding fpower_d_def using WPowerStarCommute[of "LIFT (f \<and> finite)"]
by blast

lemma WPowerSucAndEmptyEqvAndEmpty:
 "\<turnstile> (wpower (f \<and> empty) (Suc n)) = (f \<and> empty)"
proof
 (induct n)
 case 0
 then show ?case 
 by (metis ChopEmpty wpow_0 wpow_Suc)
 next
 case (Suc n)
 then show ?case 
 by (metis AndEmptyChopAndEmptyEqvAndEmpty int_eq wpow_Suc)
qed

lemma FPowerSucAndEmptyEqvAndEmpty:
 "\<turnstile> (fpower (f \<and> empty) (Suc n)) = (f \<and> empty)"
unfolding fpower_d_def using WPowerSucAndEmptyEqvAndEmpty[of "LIFT (f \<and> finite)" n ]
WPowerEqRule[of "LIFT ((f \<and> empty) \<and> finite)" "LIFT ((f \<and> finite) \<and> empty)" "(Suc n)"]
by (meson EmptyImpFinite Prop01 Prop04 Prop05 Prop10 WPowerEqRule WPowerSucAndEmptyEqvAndEmpty)

lemma WPowerOr:
 "\<turnstile>  (wpower (f \<or> g) (Suc n)) = ( (f;wpower (f \<or> g) n) \<or> (g;wpower (f \<or> g) n))"
by (simp add: OrChopEqv)

lemma FPowerOr:
 "\<turnstile>  (fpower (f \<or> g) (Suc n)) = ( ((f \<and> finite);fpower (f \<or> g) n) \<or> 
                                 ((g \<and> finite);fpower (f \<or> g) n))"
by (simp add: FiniteOr OrChopEqvRule fpower_d_def)

lemma WPowerEmptyOrMore:
 "\<turnstile> (wpower ( (f \<and> empty) \<or> (f \<and> more)) (Suc n)) =
    ((f \<and> empty);(wpower ( (f \<and> empty) \<or> (f \<and> more)) n) \<or> 
     (f \<and> more );(wpower ( (f \<and> empty) \<or> (f \<and> more)) n))"
using WPowerOr by blast

lemma FPowerEmptyOrMore:
 "\<turnstile> (fpower ( (f \<and> empty) \<or> (f \<and> more)) (Suc n)) =
    ((f \<and> empty);(fpower ( (f \<and> empty) \<or> (f \<and> more)) n) \<or> 
     (f \<and> fmore );(fpower ( (f \<and> empty) \<or> (f \<and> more)) n))"
using FPowerOr[of "LIFT (f \<and> empty)" "LIFT (f \<and> more)" n]
by (metis (no_types, lifting) AndMoreAndFiniteEqvAndFmore EmptyImpFinite Prop01 Prop05 Prop10 int_eq)


lemma WPowerstarInductL:
 assumes "\<turnstile> g \<or> f;h \<longrightarrow> h"
 shows   "\<turnstile> (wpowerstar f);g \<longrightarrow> h"
proof -
 have 1: "\<turnstile> (wpowerstar f);g = ((\<exists> n. wpower (f) n));g"
   by (simp add: wpowerstar_d_def  LeftChopEqvChop)
 have 2: "\<turnstile> (\<exists> n. wpower (f) n);g = 
            (\<exists> n. (wpower (f) n);g)"
        by (metis ExistChop inteq_reflection)
 have 3: "\<And> n. \<turnstile> (wpower (f) n);g \<longrightarrow> h" 
   using  WPowerChopInductL[of "g" "LIFT(f)" "h"] assms by auto
 have 4: "\<turnstile> (\<exists> n. ((wpower (f) n));g) \<longrightarrow> h"
        by (metis (mono_tags, lifting) "3" Prop10 intI int_eq unl_Rex unl_lift2)
 from 1  2 4 show ?thesis 
   by (metis  inteq_reflection)
qed 

lemma FPowerstar_WPowerstar: 
 "\<turnstile> fpowerstar f = wpowerstar (f \<and> finite) " 
unfolding fpowerstar_d_def wpowerstar_d_def fpower_d_def by simp

lemma FPowerstarInductL:
 assumes "\<turnstile> g \<or> (f \<and> finite);h \<longrightarrow> h"
 shows   "\<turnstile> (fpowerstar f);g \<longrightarrow> h"
using assms WPowerstarInductL[of g "LIFT (f \<and> finite)" h]
FPowerstar_WPowerstar[of f] by (metis int_eq) 

lemma WPowerstarInductR:
 assumes "\<turnstile> g \<or> h;f \<longrightarrow> h"
 shows   "\<turnstile> g;(wpowerstar f) \<longrightarrow> h"
proof -
 have 1: "\<turnstile> g;(wpowerstar f) = g;(\<exists> n. wpower f n)"
   by (simp add:  wpowerstar_d_def)
 have 2: "\<turnstile> (g;(\<exists> n. wpower f n)) = (\<exists> n. g;(wpower f n))"
   by (metis Prop04 ChopExist int_simps(31))
 have 3: "\<And> n. \<turnstile> g;(wpower f n) \<longrightarrow> h" 
   using WPowerChopInductR assms by blast
 have 4: "\<turnstile> (\<exists> n. g;(wpower f n)) \<longrightarrow> h"
    by (metis (mono_tags, lifting) "3" Prop10 intI int_eq unl_Rex unl_lift2)
 from 1  2   4 show ?thesis by (metis  inteq_reflection)
qed 

lemma FPowerstarInductR:
 assumes "\<turnstile> g \<or> h;f \<longrightarrow> h"
 shows   "\<turnstile> g;(fpowerstar f) \<longrightarrow> h"
proof -
 have 1:"\<turnstile> g \<or> h;(f \<and> finite) \<longrightarrow> h"
   using assms using ChopAndA by fastforce
 show ?thesis 
using 1 WPowerstarInductR[of g h "LIFT (f \<and> finite)" ]
FPowerstar_WPowerstar[of f] 
by (metis inteq_reflection)
qed


lemma WPowerstarEqv :
    "\<turnstile>  (wpowerstar f) =  (empty \<or>  f; (wpowerstar f)) " 
using WPowerstarEqvSem by blast

lemma FPowerstarEqv :
    "\<turnstile>  (fpowerstar f) =  (empty \<or>  (f \<and> finite); (fpowerstar f)) " 
by (simp add: fpowersem fpowerstar_d_def)

lemma SChopstarEqv :
    "\<turnstile>  (schopstar f) =  (empty \<or>  ((f \<and> more) \<and> finite); (schopstar f)) " 
by (simp add: FPowerstarEqv schopstar_d_def)

lemma WPowerstar_more_absorb: 
 "\<turnstile> (wpowerstar (f \<and> more)) = (wpowerstar f) "
proof -
 have 1: " \<turnstile> (wpowerstar (f \<and> more)) \<longrightarrow> (wpowerstar f) "
    using WPowerstarInductL[of "LIFT empty" "LIFT f \<and> more" "LIFT (wpowerstar f)"] 
    by (metis AndChopA ChopEmpty Prop02 Prop05 WPowerChopInductL WPowerstarEqv int_iffD2 inteq_reflection wpow_0)
 have 2: "\<turnstile> empty \<longrightarrow> wpowerstar (f \<and> more)"
    using WPowerstarEqv[of "LIFT f \<and> more"] by fastforce
 have 20: "\<turnstile> (f \<and> more);wpowerstar (f \<and> more) \<longrightarrow> wpowerstar (f \<and> more)"
   by (meson Prop03 WPowerstarEqv)
 have 21: "\<turnstile> (f \<and> empty);wpowerstar (f \<and> more) \<longrightarrow> wpowerstar (f \<and> more)"
    by (metis AndChopB EmptyChop int_eq)
 have 22: "\<turnstile> ((f \<and> more) \<or> (f \<and> empty));wpowerstar (f \<and> more) =
             ((f \<and> more);wpowerstar (f \<and> more) \<or> (f \<and> empty);wpowerstar (f \<and> more))  "
      by (simp add: OrChopEqv)
 have 23: "\<turnstile> ((f \<and> more) \<or> (f \<and> empty)) = f" 
   unfolding empty_d_def by fastforce
 have 3: "\<turnstile> f;wpowerstar (f \<and> more) \<longrightarrow> wpowerstar (f \<and> more)"
    by (metis "20" "21" "22" "23" Prop02 inteq_reflection)    
 have 4: " \<turnstile> empty \<or> f;wpowerstar (f \<and> more) \<longrightarrow> wpowerstar (f \<and> more)"
   using 2 3 by fastforce
 have 5: "\<turnstile> (wpowerstar f) \<longrightarrow> (wpowerstar (f \<and> more)) "
    using 4 WPowerstarInductL[of "LIFT empty" f "LIFT (wpowerstar (f \<and> more))" ]
      by (metis ChopEmpty inteq_reflection)
 show ?thesis using 1 5 by fastforce
qed

lemma FPowerstar_more_absorb: 
 "\<turnstile> (fpowerstar (f \<and> more)) = (fpowerstar f) "
proof -
 have 1: "\<turnstile> ((f \<and> more) \<and> finite) = ((f \<and> finite) \<and> more)" 
   by fastforce
 show ?thesis 
 using FPowerstar_WPowerstar[of f]  FPowerstar_WPowerstar[of "LIFT (f \<and> more)" ] 
 WPowerstar_more_absorb[of "LIFT (f \<and> finite)"] 1
 by (metis int_eq)
qed

lemma  SChopstar_WPowerstar: 
 "\<turnstile> (schopstar f) = (wpowerstar (f \<and> finite))"
by (metis FPowerstar_WPowerstar FPowerstar_more_absorb inteq_reflection schopstar_d_def)

lemma SChopstar_and_more: 
"\<turnstile> (schopstar (f \<and> more)) = (schopstar f) " 
by (simp add: FPowerstar_more_absorb schopstar_d_def)


lemma IStarWPowerstar: 
 "\<turnstile> (istar f) = (wpowerstar f) " 
proof -
 have 1: "\<turnstile> (istar f) \<longrightarrow> (wpowerstar f) " 
   by (metis  IStarWeakInduct WPowerstarEqv int_iffD2  )
 have 2: "\<turnstile> (wpowerstar f) \<longrightarrow> (istar f) " 
   using WPowerstarInductL[of "LIFT empty" f "LIFT istar f" ] 
    by (metis ChopEmpty IStarIntros inteq_reflection)
 show ?thesis 
 by (simp add: "1" "2" Prop11)
qed



subsection \<open>Kleene Algebra\<close>

lemma WPowerstar_imp_empty:
 "\<turnstile> empty \<longrightarrow> (wpowerstar f) " 
using WPowerstarEqv[of f] by fastforce

lemma SChopstar_imp_empty:
 "\<turnstile> empty \<longrightarrow> (schopstar f) " 
using SChopstarEqv[of f] by fastforce

lemma WPowerstar_swap: 
 "\<turnstile> (wpowerstar (f \<or> g)) = (wpowerstar (g \<or>f)) " 
proof -
 have 1: "\<turnstile> (f \<or> g) = (g \<or> f) "
   by fastforce
 show ?thesis 
 by (metis "1" WPowerstarEqv inteq_reflection)
qed

lemma SChopstar_swap: 
 "\<turnstile> (schopstar (f \<or> g)) = (schopstar (g \<or> f)) " 
proof -
 have 1: "\<turnstile> (f \<or> g) = (g \<or> f) "
   by fastforce
 show ?thesis 
 by (metis "1" SChopstardef int_eq)
qed

lemma WPowerstar_1L:
 "\<turnstile> f;(wpowerstar f) \<longrightarrow> (wpowerstar f) " 
by (meson Prop03 WPowerstarEqv)

lemma SChopstar_1L: 
"\<turnstile> (f)\<frown>(schopstar f) \<longrightarrow> (schopstar f) " 
by (metis SChopstar_WPowerstar WPowerstar_1L inteq_reflection schop_d_def)

lemma SChopstarMore_1L: 
"\<turnstile> (f \<and> more)\<frown>(schopstar f) \<longrightarrow> (schopstar f) " 
by (meson AndSChopA SChopstar_1L lift_imp_trans)

lemma WPowerstar_trans_eq:
 "\<turnstile> (wpowerstar f);(wpowerstar f) = (wpowerstar f) " 
proof -
 have 1: "\<turnstile> (wpowerstar f) \<or> f;(wpowerstar f) \<longrightarrow> (wpowerstar f)" 
   by (simp add: WPowerstar_1L)
 have 2: "\<turnstile> (wpowerstar f);(wpowerstar f) \<longrightarrow> (wpowerstar f)"
    using "1" WPowerstarInductL by blast
 have 3: "\<turnstile> (wpowerstar f) \<longrightarrow> (wpowerstar f);(wpowerstar f) " 
    by (metis AndChopB EmptyChop Prop10 WPowerstar_imp_empty  inteq_reflection)
 show ?thesis 
 by (simp add: "2" "3" int_iffI)
qed

lemma SChopstar_trans_eq:
 "\<turnstile> (schopstar f);(schopstar f) = (schopstar f) " 
by (metis SChopstar_WPowerstar WPowerstar_trans_eq inteq_reflection)

lemma WPowerstar_trans:
 "\<turnstile> (wpowerstar f);(wpowerstar f) \<longrightarrow> (wpowerstar f) " 
using WPowerstar_trans_eq by fastforce

lemma SChopstar_trans:
 "\<turnstile> (schopstar f);(schopstar f) \<longrightarrow> (schopstar f) " 
using SChopstar_trans_eq by fastforce

lemma WPowerstar_induct_lvar:
 assumes "\<turnstile> f;g \<longrightarrow> g"
 shows "\<turnstile> (wpowerstar f);g \<longrightarrow> g" 
using assms
by (simp add: WPowerstarInductL)

lemma SChopstar_induct_lvar: 
 assumes "\<turnstile> (f)\<frown>g \<longrightarrow> g"
 shows   "\<turnstile> (schopstar f)\<frown>g \<longrightarrow> g" 
using assms 
by (metis AndChopA SChopstar_WPowerstar WPowerstar_induct_lvar inteq_reflection lift_imp_trans schop_d_def)

lemma SChopstarMore_induct_lvar: 
 assumes "\<turnstile> (f \<and> more)\<frown>g \<longrightarrow> g"
 shows   "\<turnstile> (schopstar f)\<frown>g \<longrightarrow> g" 
using assms 
by (metis FPowerstar_WPowerstar SChopstar_induct_lvar SChopstar_WPowerstar inteq_reflection schopstar_d_def)

lemma WPowerstar_inductL_var_equiv: 
 "(\<turnstile> (wpowerstar f);g \<longrightarrow> g) = (\<turnstile> f;g\<longrightarrow>g)"
proof -
 have 1: "(\<turnstile> f;g\<longrightarrow>g) \<Longrightarrow> (\<turnstile> (wpowerstar f);g \<longrightarrow> g) "
   by (simp add: WPowerstar_induct_lvar)
 have 2: " (\<turnstile> (wpowerstar f);g \<longrightarrow> g) \<Longrightarrow> (\<turnstile> f;g\<longrightarrow>g) "
     by (metis (no_types, lifting) AndChopB ChopAssoc EmptyChop Prop10 WPowerstar_1L 
         WPowerstar_imp_empty int_eq lift_and_com)
 show ?thesis 
 using "1" "2" by blast
qed

lemma SChopstar_inductL_var_equiv: 
 "(\<turnstile> (schopstar f)\<frown>g \<longrightarrow> g) = (\<turnstile> (f)\<frown>g\<longrightarrow>g)"
proof -
 have 1: "(\<turnstile> (f)\<frown>g\<longrightarrow>g) \<Longrightarrow> (\<turnstile> (schopstar f)\<frown>g \<longrightarrow> g) "
   by (simp add: SChopstar_induct_lvar)
 have 10: "\<turnstile> (schopstar f) \<longrightarrow> (empty \<or> (f \<and> finite) \<or> f\<frown>(schopstar f)) "
 by (metis AndSChopA FPowerstarEqv Prop05 Prop08 Prop11 schop_d_def schopstar_d_def) 
 have 101: "\<turnstile> (f \<and> finite) \<longrightarrow> (schopstar f)"
 by (metis ChopEmpty RightChopImpChop SChopstar_1L SChopstar_imp_empty int_eq lift_imp_trans schop_d_def)
 have 11: "\<turnstile> (empty \<or> (f \<and> finite) \<or> f\<frown>(schopstar f)) \<longrightarrow> (schopstar f)"
 by (meson "101" Prop02 SChopstar_1L SChopstar_imp_empty)
 have 12: "\<turnstile> (schopstar f) = (empty \<or> (f \<and> finite) \<or> f\<frown>(schopstar f)) " 
    using "10" "11" int_iffI by blast
 have 2: " (\<turnstile> (schopstar f)\<frown>g \<longrightarrow> g) \<Longrightarrow> (\<turnstile> (f)\<frown>g\<longrightarrow>g) "
    using 12
    by (metis (no_types, opaque_lifting) EmptySChop LeftSChopImpSChop SChopAssoc SChopstar_1L 
         SChopstar_imp_empty int_iffI inteq_reflection) 
 show ?thesis 
 using "1" "2" by blast
qed

lemma SChopstarMore_induct_lvar_equiv: 
 "(\<turnstile> (schopstar f)\<frown>g \<longrightarrow> g) = (\<turnstile> (f \<and> more)\<frown>g\<longrightarrow>g)"
using SChopstar_inductL_var_equiv[of "LIFT f \<and> more" g] 
SChopstar_and_more[of f] by (metis int_eq) 


lemma WPowerstar_induct_lvar_eq: 
assumes "\<turnstile> f;g = g"
 shows   "\<turnstile> (wpowerstar f);g \<longrightarrow> g"
using assms 
using WPowerstar_induct_lvar int_iffD1 by blast

lemma SChopstar_induct_lvar_eq: 
 assumes "\<turnstile> (f)\<frown>g = g"
 shows   "\<turnstile> (schopstar f)\<frown>g \<longrightarrow> g"
using assms 
using SChopstar_induct_lvar int_iffD1 by blast

lemma SChopstarMore_induct_lvar_eq: 
 assumes "\<turnstile> (f \<and> more)\<frown>g = g"
 shows   "\<turnstile> (schopstar f)\<frown>g \<longrightarrow> g"
using assms SChopstar_induct_lvar_eq[of "LIFT f \<and> more" g] SChopstar_and_more[of f] by (metis int_eq)

lemma WPowerstar_induct_lvar_eq2: 
 assumes "\<turnstile> f;g = g"
 shows   "\<turnstile> (wpowerstar f);g = g"
using assms
by (meson ChopImpChop EmptyChop Prop11 WPowerstar_imp_empty WPowerstar_induct_lvar_eq lift_imp_trans)

lemma SChopstar_induct_lvar_eq2: 
 assumes "\<turnstile> (f)\<frown>g = g"
 shows   "\<turnstile> (schopstar f)\<frown>g = g"
using assms 
by (metis AndSChopB EmptySChop Prop10 SChopstar_imp_empty SChopstar_induct_lvar int_eq int_iffD1 int_iffI)

lemma SChopstarMore_induct_lvar_eq2: 
 assumes "\<turnstile> (f \<and> more)\<frown>g = g"
 shows   "\<turnstile> (schopstar f)\<frown>g = g"
using assms SChopstar_induct_lvar_eq2[of "LIFT f \<and> more" g] SChopstar_and_more[of f] by (metis int_eq)


lemma WPowerstar_induct_lvar_empty:
 assumes "\<turnstile> empty \<or> f ; g \<longrightarrow> g" 
 shows   "\<turnstile> (wpowerstar f) \<longrightarrow> g " 
using assms 
by (metis ChopEmpty WPowerstarInductL inteq_reflection)

lemma SChopstar_induct_lvar_empty:
 assumes "\<turnstile> empty \<or> (f) \<frown> g \<longrightarrow> g" 
 shows   "\<turnstile> (schopstar f) \<longrightarrow> g " 
using assms 
by (metis FPowerstar_WPowerstar FPowerstar_more_absorb WPowerstar_induct_lvar_empty inteq_reflection 
    schop_d_def schopstar_d_def)

lemma SChopstarMore_induct_lvar_empty:
 assumes "\<turnstile> empty \<or> (f \<and> more) \<frown> g \<longrightarrow> g" 
 shows   "\<turnstile> (schopstar f) \<longrightarrow> g " 
using assms SChopstar_induct_lvar_empty[of "LIFT f \<and> more" g] SChopstar_and_more[of f] by (metis int_eq)



lemma WPowerstar_induct_lvar_star: 
 assumes "\<turnstile>  f  ; (wpowerstar g) \<longrightarrow> (wpowerstar g)"
 shows   "\<turnstile> (wpowerstar f) \<longrightarrow> (wpowerstar g) " 
using assms 
by (meson Prop02 WPowerstar_imp_empty WPowerstar_induct_lvar_empty)

lemma SChopstar_induct_lvar_star: 
 assumes "\<turnstile>  (f ) \<frown> (schopstar g) \<longrightarrow> (schopstar g)"
 shows   "\<turnstile> (schopstar f) \<longrightarrow> (schopstar g) " 
using assms 
by (meson Prop02 SChopstar_imp_empty SChopstar_induct_lvar_empty)

lemma SChopstarMore_induct_lvar_star: 
 assumes "\<turnstile>  (f \<and> more) \<frown> (schopstar g) \<longrightarrow> (schopstar g)"
 shows   "\<turnstile> (schopstar f) \<longrightarrow> (schopstar g) " 
using assms using SChopstar_induct_lvar_star[of "LIFT f \<and> more" g] SChopstar_and_more[of f] by (metis int_eq) 



lemma WPowerstar_induct_leq:
 assumes "\<turnstile> (h \<or> f;g) = g "
 shows   "\<turnstile> (wpowerstar f);h \<longrightarrow> g " 
using assms 
using WPowerstarInductL int_iffD1 by blast

lemma SChopstar_induct_leq:
 assumes "\<turnstile> (h \<or> (f)\<frown>g) = g "
 shows   "\<turnstile> (schopstar f)\<frown>h \<longrightarrow> g " 
using assms 
by (metis AndChopA SChopstar_WPowerstar WPowerstar_induct_leq inteq_reflection lift_imp_trans schop_d_def)

lemma SChopstarMore_induct_leq:
 assumes "\<turnstile> (h \<or> (f \<and> more)\<frown>g) = g "
 shows   "\<turnstile> (schopstar f)\<frown>h \<longrightarrow> g " 
using assms SChopstar_induct_leq[of h "LIFT f \<and> more"  g] SChopstar_and_more[of f] by (metis int_eq) 



lemma WPowerstar_subid: 
 assumes "\<turnstile> f \<longrightarrow> empty "
 shows   "\<turnstile> (wpowerstar f) = empty " 
using assms 
by (meson ChopEmpty Prop02 Prop11 WPowerstar_imp_empty WPowerstar_induct_lvar_empty lift_imp_trans)

lemma SChopstar_subid: 
 assumes "\<turnstile> f \<longrightarrow> empty "
 shows   "\<turnstile> (schopstar f) = empty " 
using assms 
by (metis EmptyImpFinite FPowerstar_WPowerstar FPowerstar_more_absorb Prop10 WPowerstar_subid int_eq
    lift_imp_trans schopstar_d_def)

lemma WPowerstar_subdist: 
"\<turnstile> (wpowerstar f) \<longrightarrow> (wpowerstar (f \<or> g)) " 
proof -
 have 1: "\<turnstile> f;(wpowerstar (f \<or> g)) \<longrightarrow> (f \<or> g);(wpowerstar (f \<or> g)) " 
   using OrChopEqv by fastforce
 have 2: "\<turnstile> (f \<or> g);(wpowerstar (f \<or> g)) \<longrightarrow> (wpowerstar (f \<or> g))"
   by (simp add: WPowerstar_1L)
 show ?thesis 
 using "1" "2" WPowerstar_induct_lvar_star lift_imp_trans by blast
qed

lemma SChopstar_subdist: 
"\<turnstile> (schopstar f) \<longrightarrow> (schopstar (f \<or> g)) " 
by (metis FPowerstar_WPowerstar FPowerstar_more_absorb FiniteOr WPowerstar_subdist int_eq schopstar_d_def)

lemma WPowerstar_subdist_var: 
"\<turnstile> (wpowerstar f) \<or> (wpowerstar g) \<longrightarrow> (wpowerstar (f \<or> g)) " 
by (metis Prop02 WPowerstar_subdist WPowerstar_swap inteq_reflection)

lemma SChopstar_subdist_var: 
"\<turnstile> (schopstar f) \<or> (schopstar g)\<longrightarrow> (schopstar (f \<or> g)) " 
by (metis Prop02 SChopstar_subdist SChopstar_swap inteq_reflection)

lemma WPowerstar_iso:
 assumes "\<turnstile> f \<longrightarrow> g"
 shows "\<turnstile> (wpowerstar f) \<longrightarrow> (wpowerstar g) " 
by (metis AndChopB Prop05 Prop10 WPowerstar_induct_lvar_star WPowerstarEqv assms inteq_reflection)

lemma SChopstar_iso:
 assumes "\<turnstile> f \<longrightarrow> g"
 shows "\<turnstile> (schopstar f) \<longrightarrow> (schopstar g) " 
using assms 
by (metis AndSChopB Prop10 SChopstar_1L SChopstar_induct_lvar_star inteq_reflection lift_imp_trans)


lemma WPowerstar_invol:
 "\<turnstile> (wpowerstar (wpowerstar f)) = (wpowerstar f) " 
proof -
 have 1: "\<turnstile>  (wpowerstar f);(wpowerstar f) = (wpowerstar f)"
   by (simp add: WPowerstar_trans_eq)
 have 2: "\<turnstile> (wpowerstar (wpowerstar f)) \<longrightarrow> (wpowerstar f)"
   using "1" WPowerstar_induct_lvar_star int_iffD1 by blast
 have 3: "\<turnstile> (wpowerstar (wpowerstar f));(wpowerstar (wpowerstar f)) \<longrightarrow> (wpowerstar (wpowerstar f))"
   by (simp add: WPowerstar_trans)
 have 4: "\<turnstile> f;(wpowerstar (wpowerstar f)) \<longrightarrow>  (wpowerstar (wpowerstar f))"
   using WPowerstar_1L WPowerstar_inductL_var_equiv by blast
 have 5: "\<turnstile> (wpowerstar f) \<longrightarrow>  (wpowerstar (wpowerstar f))"
   by (simp add: "4" WPowerstar_induct_lvar_star)
 show ?thesis 
 by (simp add: "2" "5" Prop11) 
qed

lemma SChopstar_invol:
 "\<turnstile> (schopstar (schopstar f)) = (schopstar f) " 
by (meson Prop11 SChopstar_1L SChopstar_inductL_var_equiv SChopstar_induct_lvar_star)

lemma WPowerstar_star2: 
 "\<turnstile> (wpowerstar (empty \<or> f)) = (wpowerstar f) " 
by (meson EmptyOrChopEqv Prop02 Prop03 Prop11 WPowerstar_imp_empty WPowerstar_induct_lvar_empty 
    WPowerstarEqv lift_imp_trans) 

lemma SChopstar_star2: 
 "\<turnstile> (schopstar (empty \<or> f)) = (schopstar f) " 
by (metis EmptyImpFinite FiniteOr Prop10 SChopstar_WPowerstar WPowerstar_star2 int_eq)


lemma Chop_WPowerstar_Closure: 
 assumes "\<turnstile> f \<longrightarrow> (wpowerstar h) "
         "\<turnstile> g \<longrightarrow> (wpowerstar h) "
 shows   "\<turnstile> f;g \<longrightarrow> (wpowerstar h) "
proof -
have 1: "\<turnstile> g \<or> (wpowerstar h);(wpowerstar h) \<longrightarrow> (wpowerstar h) " 
 by (metis Prop02 WPowerstar_1L WPowerstar_induct_lvar assms(2))
have 2: "\<turnstile> (wpowerstar h); g \<longrightarrow> (wpowerstar h) " 
  by (meson Prop02 WPowerstarInductL WPowerstar_1L assms(2))
have 3: "\<turnstile> f;g \<longrightarrow> (wpowerstar h);g "
 by (simp add: LeftChopImpChop assms(1))
show ?thesis 
using "2" "3" lift_imp_trans by blast
qed

lemma SChop_SChopstar_Closure: 
 assumes "\<turnstile> f \<longrightarrow> (schopstar h) "
         "\<turnstile> g \<longrightarrow> (schopstar h) "
 shows   "\<turnstile> f\<frown>g \<longrightarrow> (schopstar h) "
using assms
by (metis AndSChopA Prop10 SChopAndB SChopstarMore_1L SChopstarMore_induct_lvar inteq_reflection 
    lift_and_com lift_imp_trans)

lemma WPowerstar_wpowerstar_closure: 
assumes "\<turnstile> (wpowerstar f) \<longrightarrow> (wpowerstar h) "
shows "\<turnstile> (wpowerstar (wpowerstar f)) \<longrightarrow> (wpowerstar h) "
using assms
by (simp add: Chop_WPowerstar_Closure WPowerstar_induct_lvar_star)

lemma SChopstar_SChopstar_closure: 
assumes "\<turnstile> (schopstar f) \<longrightarrow> (schopstar h) "
shows "\<turnstile> (schopstar (schopstar f)) \<longrightarrow> (schopstar h) "
using assms 
by (metis SChopstar_invol inteq_reflection)


lemma WPowerstar_closed_unfold: 
assumes "\<turnstile> (wpowerstar f) = f"
shows   "\<turnstile> f = (empty \<or> f;f) " 
using assms 
by (metis WPowerstarEqv int_eq)

lemma SChopstar_closed_unfold: 
assumes "\<turnstile> (schopstar f) = f"
shows   "\<turnstile> f = (empty \<or> (f)\<frown>f) " 
using assms 
by (metis SChopstar_WPowerstar WPowerstarEqv inteq_reflection schop_d_def)

lemma SChopstarMore_closed_unfold: 
assumes "\<turnstile> (schopstar f) = f"
shows   "\<turnstile> f = (empty \<or> (f \<and> more)\<frown>f) " 
using assms 
by (metis SChopstarEqv int_eq schop_d_def)

lemma WPowerstar_ext:
 "\<turnstile> f \<longrightarrow> (wpowerstar f) " 
proof -
 have 1: "\<turnstile> f \<longrightarrow> f;(wpowerstar f) " 
   by (metis ChopEmpty RightChopImpChop WPowerstar_imp_empty int_eq)
 show ?thesis by (meson "1" WPowerstar_1L lift_imp_trans)
qed

lemma SChopstar_ext:
 "\<turnstile> f \<and> finite\<longrightarrow> (schopstar f) " 
by (metis SChopstar_WPowerstar WPowerstar_ext inteq_reflection)

lemma SChopstarMore_ext:
 "\<turnstile> f \<and> more \<and> finite\<longrightarrow> (schopstar f) " 
by (metis AndMoreAndFiniteEqvAndFmore FPowerstar_WPowerstar SChopstar_ext SChopstar_WPowerstar 
    fmore_d_def int_eq schopstar_d_def)

lemma WPowerstar_1R: 
 "\<turnstile> (wpowerstar f) ; f \<longrightarrow> (wpowerstar f) "
by (simp add: Chop_WPowerstar_Closure WPowerstar_ext)

lemma SChopstar_1R: 
 "\<turnstile> (schopstar f) \<frown> (f \<and> finite) \<longrightarrow> (schopstar f) "
by (simp add: SChop_SChopstar_Closure SChopstar_ext)

lemma SChopstarMore_1R: 
 "\<turnstile> (schopstar f) \<frown> (f \<and> fmore) \<longrightarrow> (schopstar f) "
by (simp add: SChop_SChopstar_Closure SChopstarMore_ext fmore_d_def)

lemma WPowerstar_unfoldR: 
 "\<turnstile> empty \<or> (wpowerstar f) ; f  \<longrightarrow> (wpowerstar f)"
by (meson Prop02 WPowerstar_1R WPowerstar_imp_empty)

lemma SChopstar_unfoldR: 
 "\<turnstile> empty \<or> (schopstar f) \<frown> (f \<and> finite) \<longrightarrow> (schopstar f)"
by (meson Prop02 SChopstar_1R SChopstar_imp_empty)

lemma SChopstarMore_unfoldR: 
 "\<turnstile> empty \<or> (schopstar f) \<frown> (f \<and> fmore) \<longrightarrow> (schopstar f)"
by (meson Prop02 SChopstarMore_1R SChopstar_imp_empty)


lemma WPowerstar_sim1: 
assumes "\<turnstile> f ; h \<longrightarrow> h ; g"
shows "\<turnstile> (wpowerstar f) ;h \<longrightarrow> h ;(wpowerstar g) "
proof -
 have 1: "\<turnstile> (f ;h);(wpowerstar g) \<longrightarrow> (h ; g);(wpowerstar g)"
   by (simp add: LeftChopImpChop assms)
 have 2: "\<turnstile> (h ; g);(wpowerstar g) \<longrightarrow> h ;(wpowerstar g) "
    by (metis ChopAssoc RightChopImpChop WPowerstar_1L inteq_reflection)
 have 3: "\<turnstile> (f ; h);(wpowerstar g) \<longrightarrow>  h ;(wpowerstar g) "
   using "1" "2" lift_imp_trans by blast
 have 4: "\<turnstile> (wpowerstar g) =(empty \<or> g;(wpowerstar g)) "
    by (simp add: WPowerstarEqv) 
 have 41: "\<turnstile> h;(empty \<or> g;(wpowerstar g)) = (h;empty \<or> h;(g;(wpowerstar g)))"
    by (simp add: ChopOrEqv) 
 have 42: "\<turnstile>  h;(g;(wpowerstar g)) =    (h;g);(wpowerstar g) "
    by (simp add: ChopAssoc)
 have 5: "\<turnstile> h ;(wpowerstar g) = (h  \<or> (h;g);(wpowerstar g)) "
    by (metis "4" "41" "42" ChopEmpty inteq_reflection) 
 have 6: "\<turnstile> h    \<longrightarrow>  h ;(wpowerstar g)"
   using "5" by fastforce
 have 7: "\<turnstile> h  \<or> (f ; h);(wpowerstar g) \<longrightarrow>  h ;(wpowerstar g)" 
   using "3" "6" Prop02 by blast
 show ?thesis 
    using WPowerstarInductL[of "LIFT h"  f "LIFT h ;(wpowerstar g)"]
    by (metis "3" "6" ChopAssoc Prop02 inteq_reflection)
qed

lemma SChopstar_sim1: 
assumes "\<turnstile> f \<frown> h \<longrightarrow> h \<frown> g"
shows "\<turnstile> (schopstar f) \<frown>(h \<and> finite) \<longrightarrow> h \<frown>(schopstar g) "
proof -
 have 1: "\<turnstile> (f \<frown>h)\<frown>(schopstar g) \<longrightarrow> (h \<frown> g)\<frown>(schopstar g)"
   by (simp add: LeftSChopImpSChop assms)
 have 2: "\<turnstile> (h \<frown> g)\<frown>(schopstar g) \<longrightarrow> h \<frown>(schopstar g) "
   by (metis RightSChopImpSChop SChopAssoc SChopstar_1L inteq_reflection)
 have 3: "\<turnstile> (f \<frown> h)\<frown>(schopstar g) \<longrightarrow>  h \<frown>(schopstar g) "
   using "1" "2" lift_imp_trans by blast
 have 4: "\<turnstile> (schopstar g) =(empty \<or> (g \<and> more)\<frown>(schopstar g)) "
 by (simp add: SChopstarEqv schop_d_def) 
 have 5: "\<turnstile> h \<frown>(schopstar g) = ((h \<and> finite) \<or> (h\<frown>(g \<and> more))\<frown>(schopstar g)) " 
   by (metis ChopEmpty SChopAssoc SChopOrEqv SChopstarEqv int_eq schop_d_def)
 have 6: "\<turnstile> h \<and> finite   \<longrightarrow>  h \<frown>(schopstar g)"
   using "5" by fastforce
 have 7: "\<turnstile> (h \<and> finite) \<or> (f \<frown> h)\<frown>(schopstar g) \<longrightarrow>  h \<frown>(schopstar g)" 
   using "3" "6" Prop02 by blast
 show ?thesis using 7
 by (metis "3" "6" Prop10 SChopAndB SChopAssoc SChopstar_induct_lvar inteq_reflection lift_imp_trans)
qed   

lemma SChopstarMore_sim1: 
assumes "\<turnstile> (f \<and> more) \<frown> h \<longrightarrow> h \<frown> (g \<and> more)"
shows "\<turnstile> (schopstar f) \<frown>(h \<and> finite) \<longrightarrow> h \<frown>(schopstar g) "
using assms SChopstar_sim1[of "LIFT f \<and> more" h "LIFT g \<and> more"] 
by (metis SChopstar_and_more int_eq)


lemma WPowerstar_Quasicomm_varA: 
assumes "\<turnstile> (g; f \<longrightarrow> f ; (wpowerstar (f \<or> g))) " 
shows " \<turnstile> ((wpowerstar g);f \<longrightarrow> f ; (wpowerstar (f \<or> g))) " 
proof - 
 have 0: "\<turnstile> (wpowerstar (wpowerstar (f \<or> g))) = (wpowerstar (f \<or> g)) "  
   by (meson WPowerstar_1L WPowerstar_inductL_var_equiv WPowerstar_induct_lvar_star int_iffI)
 have 2: "\<turnstile> f  ; wpowerstar wpowerstar (f \<or> g) = 
             f ; (wpowerstar (f \<or> g)) "
    by (simp add: "0" RightChopEqvChop) 
 have 4: " \<turnstile> (wpowerstar g ; f \<longrightarrow>
     f ; wpowerstar (f \<or> g)) = 
        (wpowerstar g ; f \<longrightarrow>
     f ; wpowerstar wpowerstar (f \<or> g)) " 
   using "2"  by fastforce
 show ?thesis  
   using 4 WPowerstar_sim1[of "LIFT g " "LIFT f  " "LIFT (wpowerstar (f \<or> g))" ]
    by (metis "0"  assms int_eq)
qed

lemma SChopstar_Quasicomm_varA: 
assumes "\<turnstile> ((g)\<frown> ((f) \<and> finite) \<longrightarrow> (f) \<frown> (schopstar (f \<or> g))) " 
shows " \<turnstile> ((schopstar g)\<frown>((f) \<and> finite) \<longrightarrow> (f) \<frown> (schopstar (f \<or> g))) " 
proof - 
 have 0: "\<turnstile> (schopstar (schopstar (f \<or> g))) = (schopstar (f \<or> g)) " 
   by (simp add: SChopstar_invol)
 have 2: "\<turnstile>((f) \<and> finite) \<frown> schopstar schopstar (f \<or> g) = 
             (f) \<frown> (schopstar (f \<or> g)) "
     by (metis (no_types, lifting) "0" AndSChopA Prop11 Prop12 inteq_reflection itl_def(9) lift_and_com) 
 have 3: "\<turnstile> (schopstar (g) \<frown> (((f) \<and> finite) \<and> finite)) =
            ((schopstar g) \<frown> ((f) \<and> finite)) " 
   by (metis Prop12 RightSChopImpSChop int_iffD2 int_iffI lift_and_com)
 have 4: " \<turnstile> (schopstar (g) \<frown> (((f) \<and> finite)) \<longrightarrow>
     (f) \<frown> schopstar (f \<or> g)) = 
        (schopstar (g) \<frown> (((f) \<and> finite) \<and> finite) \<longrightarrow>
     ((f) \<and> finite) \<frown> schopstar schopstar (f \<or> g)) 
             " 
   using "2" "3" by fastforce
 show ?thesis  
   using 4 SChopstar_sim1[of "LIFT g " "LIFT (f ) \<and> finite " "LIFT (schopstar (f \<or> g))" ]
    by (metis "0" "2" assms int_eq)
qed

lemma SChopstarMore_or_and: 
"\<turnstile> schopstar (f \<and> more \<or> g \<and> more) = (schopstar ((f \<or> g) \<and> more)) "
proof -
 have 1: "\<turnstile>  (f \<and> more \<or> g \<and> more) = ((f \<or> g) \<and> more)" 
  by fastforce
 show ?thesis 
 by (metis "1" SChopstardef int_eq)
qed

lemma SChopstar_QuasicommMore_varA: 
assumes "\<turnstile> ((g \<and> more)\<frown> ((f \<and> more) \<and> finite) \<longrightarrow> (f \<and> more) \<frown> (schopstar (f \<or> g))) " 
shows " \<turnstile> ((schopstar g)\<frown>((f \<and> more) \<and> finite) \<longrightarrow> (f \<and> more) \<frown> (schopstar (f \<or> g))) " 
using assms SChopstar_Quasicomm_varA[of "LIFT g \<and> more" "LIFT f \<and> more"] 
SChopstar_and_more[of f] SChopstar_and_more[of g] SChopstar_and_more[of "LIFT (f \<or> g)"]  
AndMoreAndFiniteEqvAndFmore[of f] 
 by (metis SChopstarMore_or_and inteq_reflection)
 

lemma WPowerstar_Quasicomm_varB: 
assumes "\<turnstile> ((wpowerstar g);f \<longrightarrow> f ; (wpowerstar (f \<or> g))) " 
shows " \<turnstile> (g; f \<longrightarrow> f ; (wpowerstar (f \<or> g))) "
proof -
 have 1: "\<turnstile> g; f \<longrightarrow> (wpowerstar g);f " 
   by (simp add: LeftChopImpChop WPowerstar_ext)
  show ?thesis 
 using "1" assms lift_imp_trans by blast
qed

lemma SChopstar_Quasicomm_varB: 
assumes "\<turnstile> ((schopstar g)\<frown>((f) \<and> finite) \<longrightarrow> (f) \<frown> (schopstar (f \<or> g))) " 
shows " \<turnstile> ((g)\<frown> ((f) \<and> finite) \<longrightarrow> (f) \<frown> (schopstar (f \<or> g))) "
proof -
 have 1: "\<turnstile> (g)\<frown> ((f) \<and> finite) \<longrightarrow> (schopstar g)\<frown>((f) \<and> finite) "
   by (metis LeftChopImpChop Prop12 SChopstar_ext int_iffD2 lift_and_com schop_d_def) 
 show ?thesis 
 using "1" assms lift_imp_trans by blast
qed

lemma SChopstar_QuasicommMore_varB: 
assumes "\<turnstile> ((schopstar g)\<frown>((f \<and> more) \<and> finite) \<longrightarrow> (f \<and> more) \<frown> (schopstar (f \<or> g))) " 
shows " \<turnstile> ((g \<and> more)\<frown> ((f \<and> more) \<and> finite) \<longrightarrow> (f \<and> more) \<frown> (schopstar (f \<or> g))) "
using assms SChopstar_Quasicomm_varB[of "LIFT g \<and> more" "LIFT f \<and> more"] 
SChopstar_and_more[of f] SChopstar_and_more[of g] SChopstar_and_more[of "LIFT (f \<or> g)"]  
AndMoreAndFiniteEqvAndFmore[of f] 
 by (metis SChopstarMore_or_and inteq_reflection)


lemma WPowerstar_Quasicomm_var:
 "(\<turnstile> (g; f \<longrightarrow> f ; (wpowerstar (f \<or> g))) ) =
  (\<turnstile>  ((wpowerstar g);f \<longrightarrow> f ; (wpowerstar (f \<or> g)))) "
using WPowerstar_Quasicomm_varA WPowerstar_Quasicomm_varB by blast

lemma SChopstar_Quasicomm_var:
 "(\<turnstile> ((g)\<frown> ((f) \<and> finite) \<longrightarrow> (f) \<frown> (schopstar (f \<or> g))) ) =
  (\<turnstile>  ((schopstar g)\<frown>((f) \<and> finite) \<longrightarrow> (f) \<frown> (schopstar (f \<or> g)))) "
using SChopstar_Quasicomm_varA SChopstar_Quasicomm_varB by blast

lemma SChopstar_QuasicommMore_var:
 "(\<turnstile> ((g \<and> more)\<frown> ((f \<and> more) \<and> finite) \<longrightarrow> (f \<and> more) \<frown> (schopstar (f \<or> g))) ) =
  (\<turnstile>  ((schopstar g)\<frown>((f \<and> more) \<and> finite) \<longrightarrow> (f \<and> more) \<frown> (schopstar (f \<or> g)))) "
using SChopstar_QuasicommMore_varA SChopstar_QuasicommMore_varB by blast
 

lemma WPowerstar_slide1: 
 "\<turnstile> (wpowerstar (f ; g));f \<longrightarrow> f;(wpowerstar (g;f)) "
by (simp add: ChopAssoc WPowerstar_sim1 int_iffD2)

lemma SChopstar_slide1: 
 "\<turnstile> (schopstar (f \<frown> g))\<frown>(f \<and> finite) \<longrightarrow> f\<frown>(schopstar (g\<frown>f)) " 
using SChopstar_sim1 
by (metis SChopAssoc int_iffD2)
 
lemma WPowerstar_slide1_var1: 
 "\<turnstile> (wpowerstar f);f \<longrightarrow> f;(wpowerstar f) " 
by (meson Prop04 WPowerstar_sim1 int_iffD1 int_simps(27))

lemma SChopstar_slide1_var1: 
 "\<turnstile> (schopstar f)\<frown>(f \<and> finite) \<longrightarrow> f\<frown>(schopstar f) " 
by (simp add: SChopstar_sim1)

lemma SChopstarMore_slide1_var1: 
 "\<turnstile> (schopstar f)\<frown>((f \<and> more) \<and> finite) \<longrightarrow> (f \<and> more)\<frown>(schopstar f) " 
using SChopstar_slide1_var1[of "LIFT f \<and> more" ]  SChopstar_and_more[of f]
by (metis inteq_reflection)


lemma wpowerstar_unfoldl_eq: 
 "\<turnstile> (empty \<or> f;(wpowerstar f)) = (wpowerstar f) " 
by (meson Prop04 WPowerstar_1L WPowerstar_induct_lvar_star WPowerstarEqv int_iffI)

lemma SChopstar_unfoldl_eq: 
 "\<turnstile> (empty \<or> f\<frown>(schopstar f)) = (schopstar f) " 
by (metis SChopstar_WPowerstar WPowerstarEqv inteq_reflection schop_d_def)

lemma SChopstarMore_unfoldl_eq: 
 "\<turnstile> (empty \<or> (f \<and> more)\<frown>(schopstar f)) = (schopstar f) " 
using SChopstar_unfoldl_eq[of "LIFT f \<and> more"] SChopstar_and_more[of f]
by (metis inteq_reflection)


lemma WPowerstar_rtc1_eq: 
 "\<turnstile> (empty \<or> f \<or> (wpowerstar f);(wpowerstar f)) = (wpowerstar f) " 
by (meson Prop02 Prop05 Prop11 WPowerstar_ext WPowerstar_imp_empty WPowerstar_trans_eq)

lemma SChopstar_rtc1_eq: 
 "\<turnstile> (empty \<or> (f \<and> finite) \<or> (schopstar f)\<frown>(schopstar f)) = (schopstar f) " 
by (meson EmptySChop LeftSChopImpSChop Prop02 Prop04 Prop05 Prop08 Prop11 SChop_SChopstar_Closure 
    SChopstar_ext SChopstar_imp_empty)

lemma SChopstarMore_rtc1_eq: 
 "\<turnstile> (empty \<or> ((f \<and> more) \<and> finite) \<or> (schopstar f)\<frown>(schopstar f)) = (schopstar f) " 
using SChopstar_rtc1_eq[of "LIFT f \<and> more"] SChopstar_and_more[of f]
by (metis inteq_reflection)

lemma WPowerstar_rtc1: 
"\<turnstile> (empty \<or> f  \<or> (wpowerstar f);(wpowerstar f)) \<longrightarrow> (wpowerstar f) "
by (meson WPowerstar_rtc1_eq int_iffD1)

lemma SChopstar_rtc1: 
"\<turnstile> (empty \<or> (f \<and> finite) \<or> (schopstar f)\<frown>(schopstar f)) \<longrightarrow> (schopstar f) "
by (meson SChopstar_rtc1_eq int_iffD1)

lemma SChopstarMore_rtc1: 
"\<turnstile> (empty \<or> ((f \<and> more) \<and> finite) \<or> (schopstar f)\<frown>(schopstar f)) \<longrightarrow> (schopstar f) "
using SChopstar_rtc1[of "LIFT f \<and> more"] SChopstar_and_more[of f]
by (metis inteq_reflection)


lemma WPowerstar_rtc2: 
"(\<turnstile> empty \<or> f;f \<longrightarrow> f) = (\<turnstile> f = (wpowerstar f)) " 
proof -
 have 1: "(\<turnstile> empty \<or> f;f \<longrightarrow> f) \<Longrightarrow>   (\<turnstile> f  = (wpowerstar f)) "
   using WPowerstar_induct_lvar_empty[of f "LIFT f" ] 
    by (simp add: WPowerstar_ext int_iffI)
 have 2: "(\<turnstile> f = (wpowerstar f)) \<Longrightarrow>   (\<turnstile> empty \<or> f;f \<longrightarrow> f)" 
  by (metis WPowerstar_unfoldR inteq_reflection)
 show ?thesis 
 by (metis "1" "2" inteq_reflection)
qed

lemma SChopstar_rtc2: 
"(\<turnstile> empty \<or> (f)\<frown>(f \<and> finite) \<longrightarrow> (f \<and> finite)) = (\<turnstile> (f \<and> finite) = (schopstar f)) " 
proof -
 have 1: "(\<turnstile> empty \<or> (f)\<frown>(f \<and> finite) \<longrightarrow> (f \<and> finite)) \<Longrightarrow>   (\<turnstile> (f \<and> finite) = (schopstar f)) "
   using SChopstar_induct_lvar_empty[of f "LIFT f \<and> finite" ] 
     by (simp add: Prop11 SChopstar_ext)
 have 2: "(\<turnstile> (f \<and> finite) = (schopstar f)) \<Longrightarrow>   (\<turnstile> empty \<or> (f)\<frown>(f \<and> finite) \<longrightarrow> (f \<and> finite))" 
   by (metis Prop02 SChopstar_1L SChopstar_imp_empty inteq_reflection)
 show ?thesis 
 by (metis "1" "2" inteq_reflection)
qed

lemma SChopstarMore_rtc2: 
"(\<turnstile> empty \<or> (f \<and> more)\<frown>((f \<and> more) \<and> finite) \<longrightarrow> ((f \<and> more) \<and> finite)) =
 (\<turnstile> ((f \<and> more) \<and> finite) = (schopstar f)) " 
using SChopstar_rtc2[of "LIFT f \<and> more"] SChopstar_and_more[of f]
by (metis inteq_reflection)

lemma SChopstarMore_rtc2_alt: 
"(\<turnstile> empty \<or> (f \<and> more)\<frown>(f \<and> finite) \<longrightarrow> (f \<and> finite)) = (\<turnstile> (f \<and> finite) = (schopstar f)) " 
proof -
 have 1: "(\<turnstile> empty \<or> (f \<and> more)\<frown>(f \<and> finite) \<longrightarrow> (f \<and> finite)) \<Longrightarrow>   (\<turnstile> (f \<and> finite) = (schopstar f)) "
   using SChopstarMore_induct_lvar_empty[of f "LIFT f \<and> finite" ]
   by (simp add: SChopstar_ext int_iffI)
 have 2: "(\<turnstile> (f \<and> finite) = (schopstar f)) \<Longrightarrow>   (\<turnstile> empty \<or> (f \<and> more)\<frown>(f \<and> finite) \<longrightarrow> (f \<and> finite))" 
  by (metis SChopstarMore_unfoldl_eq int_eq int_iffD1)
 show ?thesis 
 by (metis "1" "2" inteq_reflection)
qed


lemma WPowerstar_rtc3: 
"(\<turnstile> (empty \<or> f;f) = f) = (\<turnstile> f = (wpowerstar f)) " 
by (metis WPowerstar_rtc2 int_iffD1 inteq_reflection wpowerstar_unfoldl_eq)

lemma SChopstar_rtc3: 
"(\<turnstile> (empty \<or> (f)\<frown>(f \<and> finite)) = (f \<and> finite)) = (\<turnstile> (f \<and> finite) = (schopstar f)) " 
by (metis SChopstar_rtc2 SChopstar_unfoldl_eq int_iffD1 inteq_reflection)

lemma SChopstarMore_rtc3: 
"(\<turnstile> (empty \<or> (f \<and> more)\<frown>((f \<and> more) \<and> finite)) = ((f \<and> more) \<and> finite)) =
 (\<turnstile> ((f \<and> more) \<and> finite) = (schopstar f)) " 
using SChopstar_rtc3[of "LIFT f \<and> more"] SChopstar_and_more[of f]
by (metis inteq_reflection)

lemma SChopstarMore_rtc3_alt: 
"(\<turnstile> (empty \<or> (f \<and> more)\<frown>(f \<and> finite)) = (f \<and> finite)) = (\<turnstile> (f \<and> finite) = (schopstar f)) " 
by (metis SChopstarMore_induct_lvar_empty SChopstarMore_unfoldl_eq SChopstar_ext int_iffD1 int_iffI 
    inteq_reflection)


lemma WPowerstar_rtc_least:
assumes "\<turnstile> empty \<or> f  \<or> g;g \<longrightarrow> g" 
shows "\<turnstile> (wpowerstar f) \<longrightarrow> g " 
proof -
 have 1: "\<turnstile> f  \<longrightarrow> g " 
   using assms by fastforce
 have 2: "\<turnstile> g;g \<longrightarrow> g  " 
    using assms by fastforce
 have 3: "\<turnstile> f;g \<longrightarrow> g ;g "
    by (metis "1"  LeftChopImpChop)
 have 4: "\<turnstile> f ;g \<longrightarrow>  g"
   using "2" "3" lift_imp_trans by blast
 have 5: "\<turnstile> empty \<longrightarrow> g"
   using assms by fastforce
 show ?thesis 
   by (meson "4" "5" Prop02 WPowerstar_induct_lvar_empty)
qed 

lemma SChopstar_rtc_least:
assumes "\<turnstile> empty \<or> (f \<and> finite) \<or> (g)\<frown>(g) \<longrightarrow> (g)" 
shows "\<turnstile> (schopstar f) \<longrightarrow> g " 
proof -
 have 1: "\<turnstile> (f \<and> finite) \<longrightarrow> (g \<and> finite) " 
   using assms by fastforce
 have 2: "\<turnstile> (g)\<frown>(g) \<longrightarrow> g  " 
    using assms by fastforce
 have 3: "\<turnstile> (f)\<frown>(g) \<longrightarrow> (g) \<frown>(g) "
    by (metis "1"  LeftChopImpChop  schop_d_def)
 have 4: "\<turnstile> (f )\<frown>(g) \<longrightarrow>  g"
   using "2" "3" lift_imp_trans by blast
 have 5: "\<turnstile> empty \<longrightarrow> g"
   using assms by fastforce
 show ?thesis 
 by (meson "4" "5" Prop02 SChopstar_induct_lvar_empty)
qed 

lemma SChopstarMore_rtc_least:
assumes "\<turnstile> empty \<or> ((f \<and> more) \<and> finite) \<or> (g)\<frown>(g) \<longrightarrow> (g)" 
shows "\<turnstile> (schopstar f) \<longrightarrow> g " 
using assms SChopstar_rtc_least[of "LIFT f \<and> more"] SChopstar_and_more[of f]
by (metis inteq_reflection)


lemma WPowerstar_rtc_least_eq:
assumes "\<turnstile> (empty \<or> f \<or> g;g) = g" 
shows "\<turnstile> (wpowerstar f) \<longrightarrow> g " 
using assms 
using WPowerstar_rtc_least int_iffD1 by blast

lemma SChopstar_rtc_least_eq:
assumes "\<turnstile> (empty \<or> (f \<and> finite) \<or> (g)\<frown>(g)) = (g)" 
shows "\<turnstile> (schopstar f) \<longrightarrow> g " 
using SChopstar_rtc_least assms int_iffD1 by blast

lemma SChopstarMore_rtc_least_eq:
assumes "\<turnstile> (empty \<or> ((f \<and> more) \<and> finite) \<or> (g)\<frown>(g)) = (g)" 
shows "\<turnstile> (schopstar f) \<longrightarrow> g " 
using assms SChopstar_rtc_least_eq[of "LIFT f \<and> more"] SChopstar_and_more[of f]
by (metis inteq_reflection)

lemma WPowerstar_subdist_var_1: 
"\<turnstile> f  \<longrightarrow> (wpowerstar (f \<or> g) ) " 
by (meson WPowerstar_ext WPowerstar_subdist lift_imp_trans)

lemma SChopstar_subdist_var_1: 
"\<turnstile> f \<and> finite \<longrightarrow> (schopstar (f \<or> g) ) " 
by (meson SChopstar_ext SChopstar_subdist lift_imp_trans)

lemma WPowerstar_subdist_var_2: 
"\<turnstile> f;g  \<longrightarrow> (wpowerstar (f \<or> g) ) " 
by (metis Chop_WPowerstar_Closure WPowerstar_subdist_var_1 WPowerstar_swap inteq_reflection)

lemma SChopstar_subdist_var_2: 
"\<turnstile> (f)\<frown>(g \<and> finite)  \<longrightarrow> (schopstar (f \<or> g) ) " 
by (metis Chop_WPowerstar_Closure SChopstar_swap SChopstar_WPowerstar SChopstar_subdist_var_1 
    inteq_reflection schop_d_def)

lemma SChopstarMore_subdist_var_2: 
"\<turnstile> (f \<and> more)\<frown>((g \<and> more) \<and> finite)  \<longrightarrow> (schopstar (f \<or> g) ) " 
using SChopstar_subdist_var_2[of "LIFT f \<and> more" "LIFT g \<and> more"] 
SChopstar_and_more[of f] SChopstar_and_more[of g]
SChopstar_and_more[of "LIFT (f \<or> g)"]
SChopstarMore_or_and[of f g] 
by (metis  inteq_reflection)

lemma WPowerstar_subdist_var_3: 
 "\<turnstile> (wpowerstar f); (wpowerstar g) \<longrightarrow> (wpowerstar (f \<or> g) )"
by (metis Chop_WPowerstar_Closure WPowerstar_subdist WPowerstar_swap inteq_reflection)

lemma SChopstar_subdist_var_3: 
 "\<turnstile> (schopstar f)\<frown> (schopstar g) \<longrightarrow> (schopstar (f \<or> g) )"
by (metis SChop_SChopstar_Closure SChopstar_subdist SChopstar_swap inteq_reflection)

lemma WPowerstar_denest: 
 "\<turnstile> (wpowerstar (f \<or> g)) = (wpowerstar ((wpowerstar f);(wpowerstar g))) " 
proof -
 have 1: "\<turnstile> f  \<longrightarrow> (wpowerstar f) " 
   by (simp add: WPowerstar_ext)
 have 2: "\<turnstile> (wpowerstar f) \<longrightarrow> (wpowerstar f);(wpowerstar g) "
   by (metis ChopEmpty RightChopImpChop WPowerstar_imp_empty int_eq) 
 have 3: "\<turnstile> f  \<longrightarrow> (wpowerstar f);(wpowerstar g)"
  using 1 2 by fastforce
 have 4: "\<turnstile> g   \<longrightarrow>  (wpowerstar g) " 
   by (simp add: WPowerstar_ext)
 have 5: "\<turnstile> (wpowerstar g) \<longrightarrow> (wpowerstar f);(wpowerstar g) "
   by (meson EmptyChop LeftChopImpChop Prop11 WPowerstar_imp_empty lift_imp_trans) 
 have 6: "\<turnstile> g  \<longrightarrow> (wpowerstar f);(wpowerstar g)"
   using 4 5 by fastforce
 have 7: "\<turnstile> f \<or> g \<longrightarrow> (wpowerstar f);(wpowerstar g) "
   using 3 6 by fastforce
 have 9: "\<turnstile> (wpowerstar (f \<or> g)) \<longrightarrow> (wpowerstar ((wpowerstar f);(wpowerstar g))) " 
   using "7" WPowerstar_iso by blast
 have 10: "\<turnstile> (wpowerstar f);(wpowerstar g) \<longrightarrow> (wpowerstar (f \<or> g)) "
    by (simp add: WPowerstar_subdist_var_3) 
 have 11: "\<turnstile> (wpowerstar ((wpowerstar f);(wpowerstar g))) \<longrightarrow> (wpowerstar (f \<or> g)) "
    by (simp add: "10" Chop_WPowerstar_Closure WPowerstar_induct_lvar_star)
 show ?thesis  using "11" "9" int_iffI by blast
qed

lemma SChopstar_denest: 
 "\<turnstile> (schopstar (f \<or> g)) = (schopstar ((schopstar f)\<frown>(schopstar g))) " 
proof -
 have 1: "\<turnstile> (f \<and> finite)  \<longrightarrow> (schopstar f) " 
   by (simp add: SChopstar_ext)
 have 2: "\<turnstile> (schopstar f) \<longrightarrow> (schopstar f)\<frown>(schopstar g) "
   by (metis (no_types, lifting) AndEmptyChopAndEmptyEqvAndEmpty ChopImpChop EmptyImpFinite 
       FiniteAndEmptyEqvEmpty Prop02 Prop12 SChopAssoc SChopImpSChop SChopstar_1L SChopstar_imp_empty 
       SChopstar_induct_lvar_empty inteq_reflection schop_d_def) 
 have 3: "\<turnstile> (f \<and> finite)  \<longrightarrow> (schopstar f)\<frown>(schopstar g)"
  using 1 2 by fastforce
 have 4: "\<turnstile> (g \<and> finite)  \<longrightarrow>  (schopstar g) "
    by (simp add: SChopstar_ext)
 have 5: "\<turnstile> (schopstar g) \<longrightarrow> (schopstar f)\<frown>(schopstar g) " 
     by (meson EmptySChop LeftSChopImpSChop Prop11 SChopstar_imp_empty lift_imp_trans)
 have 6: "\<turnstile> (g \<and> finite)  \<longrightarrow> (schopstar f)\<frown>(schopstar g)"
   using 4 5 by fastforce
 have 7: "\<turnstile> (f \<and> finite) \<or> (g \<and> finite) \<longrightarrow> (schopstar f)\<frown>(schopstar g) "
   using 3 6 by fastforce
 have 8: "\<turnstile> ((f \<and> finite) \<or> (g \<and> finite)) = ((f \<or> g) \<and> finite)"
   by fastforce
 have 9: "\<turnstile> (schopstar (f \<or> g)) \<longrightarrow> (schopstar ((schopstar f)\<frown>(schopstar g))) " 
   by (metis (no_types, opaque_lifting) "7" "8" ChopEmpty EmptySChop FPowerstar_WPowerstar 
       FPowerstar_more_absorb SChopAssoc SChopstar_iso inteq_reflection schop_d_def schopstar_d_def)
 have 10: "\<turnstile> (schopstar f)\<frown>(schopstar g) \<longrightarrow> (schopstar (f \<or> g)) " 
   by (simp add: SChopstar_subdist_var_3)
 have 11: "\<turnstile> (schopstar ((schopstar f)\<frown>(schopstar g))) \<longrightarrow> (schopstar (f \<or> g)) "
   by (simp add: "10" SChop_SChopstar_Closure SChopstar_induct_lvar_star)
 show ?thesis  using "11" "9" int_iffI by blast
qed

lemma WPowerstar_or_var: 
"\<turnstile> (wpowerstar ((wpowerstar f) \<or> (wpowerstar g))) = (wpowerstar (f \<or> g)) " 
using WPowerstar_denest[of f g] 
      WPowerstar_denest[of "LIFT (wpowerstar f)" ]
 WPowerstar_invol[of f] WPowerstar_invol[of g] 
by (metis  int_eq)

lemma SChopstar_or_var: 
"\<turnstile> (schopstar ((schopstar f) \<or> (schopstar g))) = (schopstar (f \<or> g)) " 
by (metis (no_types, opaque_lifting) FPowerstar_WPowerstar FPowerstar_more_absorb FiniteOr 
    SChopstar_invol WPowerstar_denest inteq_reflection schopstar_d_def)

lemma WPowerstar_denest_var: 
"\<turnstile> (wpowerstar f) ; (wpowerstar (g;(wpowerstar f))) = (wpowerstar (f \<or> g)) "
proof -
 have 1: "\<turnstile> empty \<longrightarrow> (wpowerstar f);(wpowerstar (g;(wpowerstar f)))"
   by (metis AndEmptyChopAndEmptyEqvAndEmpty ChopImpChop FiniteAndEmptyEqvEmpty WPowerstar_imp_empty 
       inteq_reflection)
 have 2: "\<turnstile> (f;(wpowerstar f));(wpowerstar (g;(wpowerstar f))) \<longrightarrow>
           (wpowerstar f);(wpowerstar (g;(wpowerstar f))) "
   by (simp add: LeftChopImpChop WPowerstar_1L)
 have 3: "\<turnstile> (g;(wpowerstar f));(wpowerstar (g;(wpowerstar f))) \<longrightarrow>
            (wpowerstar f);(wpowerstar (g;(wpowerstar f))) "
    by (metis EmptyChop LeftChopImpChop WPowerstar_1L WPowerstar_imp_empty inteq_reflection lift_imp_trans) 
 have 4: "\<turnstile> ((f;(wpowerstar f));(wpowerstar (g;(wpowerstar f))) \<or>
             (g;(wpowerstar f));(wpowerstar (g;(wpowerstar f)))) =
             ((f \<or> g);(wpowerstar f));(wpowerstar (g;(wpowerstar f))) "
    by (metis OrChopEqv int_eq) 
 have 5: "\<turnstile> empty \<or> ((f \<or> g);(wpowerstar f));(wpowerstar (g;(wpowerstar f))) \<longrightarrow>
            (wpowerstar f);(wpowerstar (g;(wpowerstar f))) " 
   using 1 2 3 4 by (metis Prop02 int_eq)
 have 6: "\<turnstile> (wpowerstar (f \<or> g)) \<longrightarrow> (wpowerstar f);(wpowerstar (g;(wpowerstar f)))"
    by (metis "5" ChopAssoc WPowerstar_induct_lvar_empty int_eq)  
 have 7: "\<turnstile> (wpowerstar (g ;(wpowerstar f))) \<longrightarrow> (wpowerstar ((wpowerstar g);(wpowerstar f))) "
   by (simp add: LeftChopImpChop WPowerstar_ext WPowerstar_iso)
 have 8: "\<turnstile> (wpowerstar (g ;(wpowerstar f))) \<longrightarrow> (wpowerstar (f \<or> g)) " 
   by (metis "7" WPowerstar_denest WPowerstar_swap inteq_reflection)
 have 9: "\<turnstile> (wpowerstar f) \<longrightarrow>  (wpowerstar (f \<or> g)) "
    by (simp add: WPowerstar_subdist)
 have 10:"\<turnstile> (wpowerstar f) ; (wpowerstar (g;(wpowerstar f))) \<longrightarrow> (wpowerstar (f \<or> g)) "
    using "8" "9" Chop_WPowerstar_Closure by blast
 show ?thesis 
 by (simp add: "10" "6" int_iffI)
qed

lemma SChopstar_denest_var: 
"\<turnstile> (schopstar f) \<frown> (schopstar ((g)\<frown>(schopstar f))) = (schopstar (f \<or> g)) "
proof -
 have 1: "\<turnstile> empty \<longrightarrow> (schopstar ((g)\<frown>(schopstar f)))"
   by (simp add: SChopstar_imp_empty)
 have 11: "\<turnstile> empty \<longrightarrow> (schopstar f)\<frown>(schopstar ((g)\<frown>(schopstar f)))"
     by (metis AndEmptyChopAndEmptyEqvAndEmpty ChopImpChop EmptyImpFinite FiniteAndEmptyEqvEmpty 
         Prop12 SChopstar_imp_empty inteq_reflection itl_def(9))
 have 2: "\<turnstile> ((f)\<frown>(schopstar f))\<frown>(schopstar ((g)\<frown>(schopstar f))) \<longrightarrow>
           (schopstar f)\<frown>(schopstar ((g)\<frown>(schopstar f))) "
    using LeftSChopImpSChop SChopstar_1L by blast
 have 3: "\<turnstile> ((g)\<frown>(schopstar f))\<frown>(schopstar ((g )\<frown>(schopstar f))) \<longrightarrow>
            (schopstar f)\<frown>(schopstar ((g)\<frown>(schopstar f))) " 
    by (metis EmptySChop LeftSChopImpSChop SChopstar_1L SChopstar_imp_empty inteq_reflection lift_imp_trans)
 have 4: "\<turnstile> ((f \<or> g)\<frown>(schopstar f))\<frown>(schopstar ((g)\<frown>(schopstar f))) = 
             (((f)\<frown>(schopstar f))\<frown>(schopstar ((g)\<frown>(schopstar f))) \<or>
             ((g)\<frown>(schopstar f))\<frown>(schopstar ((g )\<frown>(schopstar f)))) 
              " 
     by (metis OrSChopEqv inteq_reflection)
 have 5: "\<turnstile> empty \<or> ((f \<or> g)\<frown>(schopstar f))\<frown>(schopstar ((g)\<frown>(schopstar f))) \<longrightarrow>
            (schopstar f)\<frown>(schopstar ((g)\<frown>(schopstar f))) " 
   using 11 2 3 4 
   by (metis Prop02 inteq_reflection)
 have 6: "\<turnstile> (schopstar (f \<or> g)) \<longrightarrow> (schopstar f)\<frown>(schopstar ((g)\<frown>(schopstar f)))"
    by (metis "5" SChopAssoc SChopstar_induct_lvar_empty inteq_reflection)
 have 7: "\<turnstile> (schopstar (g \<frown>(schopstar f))) \<longrightarrow> (schopstar ((schopstar g)\<frown>(schopstar f))) " 
    by (metis AndChopB ChopEmpty LeftChopImpChop Prop12 SChopstar_ext SChopstar_iso inteq_reflection schop_d_def)
 have 8: "\<turnstile> (schopstar (g \<frown>(schopstar f))) \<longrightarrow> (schopstar (f \<or> g)) " 
    by (metis "7" SChopstar_denest SChopstar_swap inteq_reflection)
 have 9: "\<turnstile> (schopstar f) \<longrightarrow>  (schopstar (f \<or> g)) "
    by (simp add: SChopstar_subdist)
 have 10:"\<turnstile> (schopstar f) \<frown> (schopstar ((g)\<frown>(schopstar f))) \<longrightarrow> (schopstar (f \<or> g)) "
    by (simp add: "8" "9" SChop_SChopstar_Closure)
 show ?thesis 
 by (simp add: "10" "6" int_iffI)
qed

lemma SChopstarMore_denest_var: 
"\<turnstile> (schopstar f) \<frown> (schopstar ((g \<and> more)\<frown>(schopstar f))) = (schopstar (f \<or> g)) "
using SChopstar_denest_var[of "LIFT f \<and> more" "LIFT g \<and> more"] 
SChopstar_and_more[of f] SChopstar_and_more[of g]
SChopstar_and_more[of "LIFT (f \<or> g)"]
SChopstarMore_or_and[of f g] 
by (metis inteq_reflection) 

lemma WPowerstar_denest_var_2: 
"\<turnstile> (wpowerstar f); (wpowerstar (g;(wpowerstar f))) = 
   (wpowerstar ((wpowerstar f);(wpowerstar g))) " 
by (metis WPowerstar_denest WPowerstar_denest_var int_eq)

lemma SChopstar_denest_var_2: 
"\<turnstile> (schopstar f)\<frown> (schopstar (g\<frown>(schopstar f))) = (schopstar ((schopstar f)\<frown>(schopstar g))) " 
by (metis SChopstar_denest SChopstar_denest_var inteq_reflection)

lemma SChopstarMore_denest_var_2: 
"\<turnstile> (schopstar f)\<frown> (schopstar ((g \<and> more)\<frown>(schopstar f))) = (schopstar ((schopstar f)\<frown>(schopstar g))) " 
using SChopstar_denest_var_2[of f "LIFT g \<and> more"]  SChopstar_and_more[of g]
by (metis inteq_reflection)

lemma WPowerstar_denest_var_3: 
"\<turnstile> (wpowerstar f); (wpowerstar ((wpowerstar g);(wpowerstar f))) = 
   (wpowerstar ((wpowerstar f);(wpowerstar g))) " 
by (metis WPowerstar_denest WPowerstar_denest_var WPowerstar_ext WPowerstar_induct_lvar_star 
    WPowerstar_trans int_iffI inteq_reflection)

lemma SChopstar_denest_var_3: 
"\<turnstile> (schopstar f)\<frown> (schopstar ((schopstar g)\<frown>(schopstar f))) = (schopstar ((schopstar f)\<frown>(schopstar g))) " 
by (metis SChopstar_denest_var_2 SChopstar_invol int_eq)



lemma WPowerstar_denest_var_4: 
"\<turnstile> (wpowerstar ((wpowerstar g);(wpowerstar f))) = 
   (wpowerstar ((wpowerstar f);(wpowerstar g))) "
by (metis WPowerstar_denest WPowerstar_swap inteq_reflection)

lemma SChopstar_denest_var_4: 
"\<turnstile> (schopstar ((schopstar g)\<frown>(schopstar f))) = (schopstar ((schopstar f)\<frown>(schopstar g))) "
by (metis SChopstar_denest SChopstar_swap inteq_reflection)


lemma WPowerstar_denest_var_5: 
"\<turnstile> (wpowerstar f); (wpowerstar (g;(wpowerstar f))) = 
   (wpowerstar g); (wpowerstar (f;(wpowerstar g))) " 
by (metis WPowerstar_denest_var WPowerstar_swap int_eq)

lemma SChopstar_denest_var_5: 
"\<turnstile> (schopstar f)\<frown> (schopstar (g\<frown>(schopstar f))) = (schopstar g)\<frown> (schopstar (f\<frown>(schopstar g))) " 
by (metis SChopstar_denest_var SChopstar_swap inteq_reflection) 

lemma SChopstarMore_denest_var_5: 
"\<turnstile> (schopstar f)\<frown> (schopstar ((g \<and> more)\<frown>(schopstar f))) = (schopstar g)\<frown> (schopstar (f\<frown>(schopstar g))) " 
using SChopstar_denest_var_5[of f "LIFT g \<and> more"] 
SChopstar_and_more[of g]
by (metis inteq_reflection)


lemma WPowerstar_denest_var_6: 
"\<turnstile> ((wpowerstar f);(wpowerstar g));(wpowerstar (f \<or> g)) = (wpowerstar (f \<or> g)) "
by (metis ChopAssoc WPowerstar_denest WPowerstar_denest_var_3 inteq_reflection)

lemma SChopstar_denest_var_6: 
"\<turnstile> ((schopstar f)\<frown>(schopstar g))\<frown>(schopstar (f \<or> g)) = (schopstar (f \<or> g)) "
by (metis SChopAssoc SChopstar_denest SChopstar_denest_var_3 inteq_reflection)


lemma WPowerstar_denest_var_7: 
"\<turnstile> (wpowerstar (f \<or> g)) ;  ((wpowerstar f);(wpowerstar g)) =  (wpowerstar (f \<or> g)) "
proof -
 have 1: "\<turnstile> (wpowerstar (f \<or> g)) ;  ((wpowerstar f);(wpowerstar g)) \<longrightarrow>
            (wpowerstar (f \<or> g)) ;  (wpowerstar ((wpowerstar f);(wpowerstar g))) "
     by (simp add: RightChopImpChop WPowerstar_ext)
 have 2: "\<turnstile>  (wpowerstar (f \<or> g)) ;  ((wpowerstar f);(wpowerstar g)) \<longrightarrow> (wpowerstar (f \<or> g)) "
   by (simp add: Chop_WPowerstar_Closure WPowerstar_subdist_var_3)
 have 3: "\<turnstile> empty \<longrightarrow> (wpowerstar (f \<or> g)) ;  ((wpowerstar f);(wpowerstar g)) "
   by (metis ChopEmpty ChopImpChop WPowerstar_imp_empty inteq_reflection)
 have 4: "\<turnstile> (f \<or> g) ; ((wpowerstar (f \<or> g)) ;  ((wpowerstar f);(wpowerstar g))) \<longrightarrow> 
             ((wpowerstar (f \<or> g)) ;  ((wpowerstar f);(wpowerstar g))) " 
    by (metis ChopAssoc LeftChopImpChop WPowerstar_1L inteq_reflection)
 have 5: "\<turnstile> empty \<or> (f \<or> g) ; ((wpowerstar (f \<or> g)) ;  ((wpowerstar f);(wpowerstar g))) \<longrightarrow>
            ((wpowerstar (f \<or> g)) ;  ((wpowerstar f);(wpowerstar g))) " 
   using "3" "4" Prop02 by blast
 have 6: "\<turnstile> (wpowerstar (f \<or> g)) \<longrightarrow> ((wpowerstar (f \<or> g)) ;  ((wpowerstar f);(wpowerstar g))) "
    using "5" WPowerstar_induct_lvar_empty by blast
 show ?thesis using "2" "6" int_iffI by blast
qed

lemma SChopstar_denest_var_7: 
"\<turnstile> (schopstar (f \<or> g)) \<frown>  ((schopstar f)\<frown>(schopstar g)) =  (schopstar (f \<or> g)) "
proof -
 have 1: "\<turnstile> (schopstar (f \<or> g)) \<frown>  ((schopstar f)\<frown>(schopstar g)) \<longrightarrow>
            (schopstar (f \<or> g)) \<frown>  (schopstar ((schopstar f)\<frown>(schopstar g))) "
   by (meson RightSChopImpSChop SChopstar_denest SChopstar_subdist_var_3 int_iffD1 lift_imp_trans)
 have 2: "\<turnstile>  (schopstar (f \<or> g)) \<frown>  ((schopstar f)\<frown>(schopstar g)) \<longrightarrow> (schopstar (f \<or> g)) "
   by (simp add: SChop_SChopstar_Closure SChopstar_subdist_var_3)
 have 3: "\<turnstile> empty \<longrightarrow> (schopstar (f \<or> g)) \<frown>  ((schopstar f)\<frown>(schopstar g)) "
 by (metis EmptySChop SChopImpSChop SChopstar_imp_empty inteq_reflection)
 have 4: "\<turnstile> (f \<or> g) \<frown> ((schopstar (f \<or> g)) \<frown>  ((schopstar f)\<frown>(schopstar g))) \<longrightarrow> 
             ((schopstar (f \<or> g)) \<frown>  ((schopstar f)\<frown>(schopstar g))) " 
   by (metis LeftSChopImpSChop SChopAssoc SChopstar_1L inteq_reflection)
 have 5: "\<turnstile> empty \<or> (f \<or> g) \<frown> ((schopstar (f \<or> g)) \<frown>  ((schopstar f)\<frown>(schopstar g))) \<longrightarrow>
            ((schopstar (f \<or> g)) \<frown>  ((schopstar f)\<frown>(schopstar g))) " 
   using "3" "4" Prop02 by blast
 have 6: "\<turnstile> (schopstar (f \<or> g)) \<longrightarrow> ((schopstar (f \<or> g)) \<frown>  ((schopstar f)\<frown>(schopstar g))) "
   using "5" SChopstar_induct_lvar_empty by blast
 show ?thesis using "2" "6" int_iffI by blast
qed

lemma WPowerstar_denest_var_8: 
"\<turnstile> ((wpowerstar f);(wpowerstar g));(wpowerstar ((wpowerstar f);(wpowerstar g))) =
   (wpowerstar ((wpowerstar f);(wpowerstar g))) "
by (metis ChopAssoc WPowerstar_denest_var_3 int_eq)

lemma SChopstar_denest_var_8: 
"\<turnstile> ((schopstar f)\<frown>(schopstar g))\<frown>(schopstar ((schopstar f)\<frown>(schopstar g))) =
   (schopstar ((schopstar f)\<frown>(schopstar g))) "
by (metis SChopstar_denest SChopstar_denest_var_6 inteq_reflection)


lemma WPowerstar_denest_var_9: 
"\<turnstile> (wpowerstar ((wpowerstar f);(wpowerstar g)));((wpowerstar f);(wpowerstar g)) =
   (wpowerstar ((wpowerstar f);(wpowerstar g))) "
by (metis WPowerstar_denest WPowerstar_denest_var_7 inteq_reflection)

lemma SChopstar_denest_var_9: 
"\<turnstile> (schopstar ((schopstar f)\<frown>(schopstar g)))\<frown>((schopstar f)\<frown>(schopstar g)) =
   (schopstar ((schopstar f)\<frown>(schopstar g))) "
by (metis SChopstar_denest SChopstar_denest_var_7 inteq_reflection)


lemma WPowerstar_confluence: 
"(\<turnstile> g ; (wpowerstar f) \<longrightarrow> (wpowerstar f);(wpowerstar g)) =
 (\<turnstile> (wpowerstar g) ; (wpowerstar f) \<longrightarrow> (wpowerstar f);(wpowerstar g) )  "  
proof -
 have 1: " (\<turnstile> g ; (wpowerstar f) \<longrightarrow> (wpowerstar f);(wpowerstar g)) \<Longrightarrow>
          (\<turnstile> (wpowerstar g) ; (wpowerstar f) \<longrightarrow> (wpowerstar f);(wpowerstar g) ) "
    by (metis Prop02 WPowerstar_imp_empty WPowerstar_rtc2 WPowerstar_sim1 WPowerstar_trans inteq_reflection)
 have 2: " (\<turnstile> (wpowerstar g) ; (wpowerstar f) \<longrightarrow> (wpowerstar f);(wpowerstar g) ) \<Longrightarrow>
           (\<turnstile> g ; (wpowerstar f) \<longrightarrow> (wpowerstar f);(wpowerstar g)) "
     by (metis AndChopB Prop10 WPowerstar_ext inteq_reflection lift_imp_trans)
 show ?thesis  using "1" "2" by blast
qed

lemma SChopstar_finite:
 "\<turnstile> schopstar f \<longrightarrow> finite" 
by (metis EmptyImpFinite FiniteChopEqvDiamond FiniteChopFiniteEqvFinite Prop02 SChopImpDiamond 
    SChopstar_induct_lvar_empty inteq_reflection)

lemma SChopstar_confluence: 
"(\<turnstile> g \<frown> (schopstar f) \<longrightarrow> (schopstar f)\<frown>(schopstar g)) =
 (\<turnstile> (schopstar g) \<frown> (schopstar f) \<longrightarrow> (schopstar f)\<frown>(schopstar g) )  "  
proof -
 have 1: " (\<turnstile> g \<frown> (schopstar f) \<longrightarrow> (schopstar f)\<frown>(schopstar g)) \<Longrightarrow>
          (\<turnstile> (schopstar g) \<frown> (schopstar f) \<longrightarrow> (schopstar f)\<frown>(schopstar g) ) "
    using SChopstar_sim1[of g "LIFT (schopstar f)" "LIFT (schopstar g)"] 
     by (metis Prop10 SChopstar_finite SChopstar_invol inteq_reflection)
 have 2: " (\<turnstile> (schopstar g) \<frown> (schopstar f) \<longrightarrow> (schopstar f)\<frown>(schopstar g) ) \<Longrightarrow>
           (\<turnstile> g \<frown> (schopstar f) \<longrightarrow> (schopstar f)\<frown>(schopstar g)) "
   by (metis AndChopB Prop10 SChopstar_ext SChopstar_finite int_eq lift_imp_trans schop_d_def)
 show ?thesis  using "1" "2" by blast
qed

lemma SChopstarMore_confluence: 
"(\<turnstile> (g \<and> more) \<frown> (schopstar f) \<longrightarrow> (schopstar f)\<frown>(schopstar g)) =
 (\<turnstile> (schopstar g) \<frown> (schopstar f) \<longrightarrow> (schopstar f)\<frown>(schopstar g) )  " 
using  SChopstar_confluence[of "LIFT g \<and> more" f] 
SChopstar_and_more[of g]
by (metis inteq_reflection)

lemma WPowerstar_church_rosser: 
assumes "\<turnstile> (wpowerstar g) ; (wpowerstar f) \<longrightarrow> (wpowerstar f) ; (wpowerstar g) "
shows "\<turnstile> (wpowerstar (f \<or> g)) = (wpowerstar f) ; (wpowerstar g) "
proof -
 have 1: "\<turnstile> ((wpowerstar f) ; (wpowerstar g)); ((wpowerstar f) ; (wpowerstar g)) \<longrightarrow>
            ((wpowerstar f) ; (wpowerstar f)); ((wpowerstar g) ; (wpowerstar g)) "
   by (metis (no_types, lifting) ChopAssoc ChopImpChop WPowerstar_trans WPowerstar_trans_eq assms int_eq)
 have 2: "\<turnstile> ((wpowerstar f) ; (wpowerstar g)); ((wpowerstar f) ; (wpowerstar g)) \<longrightarrow>
             ((wpowerstar f) ; (wpowerstar g)) "
   by (metis "1" WPowerstar_trans_eq inteq_reflection)
 have 3: "\<turnstile> empty \<longrightarrow> ((wpowerstar f) ; (wpowerstar g)) "
   by (metis "2" WPowerstar_denest_var_9 WPowerstar_imp_empty WPowerstar_induct_lvar int_eq lift_imp_trans)
  have 4: "\<turnstile> empty \<or> ((wpowerstar f) ; (wpowerstar g)); ((wpowerstar f) ; (wpowerstar g)) \<longrightarrow>
             ((wpowerstar f) ; (wpowerstar g)) " 
   using "2" "3" Prop02 by blast
 have 5: "\<turnstile> (wpowerstar ((wpowerstar f) ; (wpowerstar g))) \<longrightarrow> 
            ((wpowerstar f) ; (wpowerstar g)) "
    using "4" WPowerstar_induct_lvar_empty by blast
 have 6: "\<turnstile> (wpowerstar (f \<or> g)) \<longrightarrow> (wpowerstar f) ; (wpowerstar g) "
   by (metis "5" WPowerstar_denest inteq_reflection)
 have 7: "\<turnstile> (wpowerstar f) ; (wpowerstar g) \<longrightarrow> (wpowerstar (f \<or> g)) "
   by (simp add: WPowerstar_subdist_var_3)
 show ?thesis 
 by (simp add: "6" "7" int_iffI)
qed

lemma SChopstar_church_rosser: 
assumes "\<turnstile> (schopstar g) \<frown> (schopstar f) \<longrightarrow> (schopstar f) \<frown> (schopstar g) "
shows "\<turnstile> (schopstar (f \<or> g)) = (schopstar f) \<frown> (schopstar g) "
proof -
 have 0: "\<turnstile> ((schopstar f) \<frown> (schopstar g))\<frown> ((schopstar f)) \<longrightarrow>
            ((schopstar f) \<frown> (schopstar f))\<frown> ((schopstar g))" 
    by (metis RightSChopImpSChop SChopAssoc assms inteq_reflection)    
  have 01: "\<turnstile> (((schopstar f) \<frown> (schopstar g))\<frown> ((schopstar f)))\<frown> (schopstar g) \<longrightarrow>
            (((schopstar f) \<frown> (schopstar f))\<frown> ((schopstar g)))\<frown> (schopstar g) " 
       by (simp add: "0" LeftSChopImpSChop)
 have 1: "\<turnstile> ((schopstar f) \<frown> (schopstar g))\<frown> ((schopstar f) \<frown> (schopstar g)) \<longrightarrow>
            ((schopstar f) \<frown> (schopstar f))\<frown> ((schopstar g) \<frown> (schopstar g)) "
    by (metis "01" SChopAssoc inteq_reflection)
 have 2: "\<turnstile> ((schopstar f) \<frown> (schopstar g))\<frown> ((schopstar f) \<frown> (schopstar g)) \<longrightarrow>
             ((schopstar f) \<frown> (schopstar g)) "
   by (metis (no_types, lifting) "1" SChopstar_denest SChopstar_denest_var_3 int_eq int_simps(27))
 have 3: "\<turnstile> empty \<longrightarrow> ((schopstar f) \<frown> (schopstar g)) "
   by (meson EmptySChop Prop11 SChopImpSChop SChopstar_imp_empty lift_imp_trans)
 have 4: "\<turnstile> empty \<or> ((schopstar f) \<frown> (schopstar g))\<frown> ((schopstar f) \<frown> (schopstar g)) \<longrightarrow>
             ((schopstar f) \<frown> (schopstar g)) " 
   using "2" "3" Prop02 by blast
 have 5: "\<turnstile> (schopstar ((schopstar f) \<frown> (schopstar g))) \<longrightarrow> ((schopstar f) \<frown> (schopstar g)) "
   using "4" SChopstar_induct_lvar_empty by blast
 have 6: "\<turnstile> (schopstar (f \<or> g)) \<longrightarrow> (schopstar f) \<frown> (schopstar g) "
   by (metis "5" SChopstar_denest inteq_reflection)
 have 7: "\<turnstile> (schopstar f) \<frown> (schopstar g) \<longrightarrow> (schopstar (f \<or> g)) "
   by (simp add: SChopstar_subdist_var_3)
 show ?thesis 
 by (simp add: "6" "7" int_iffI)
qed

lemma WPowerstar_church_rosser_var: 
assumes "\<turnstile> g ; (wpowerstar f) \<longrightarrow> (wpowerstar f) ; (wpowerstar g) "
shows "\<turnstile> (wpowerstar (f \<or> g)) = (wpowerstar f) ; (wpowerstar g) "
using assms 
by (simp add: WPowerstar_church_rosser WPowerstar_confluence)

lemma SChopstar_church_rosser_var: 
assumes "\<turnstile> g \<frown> (schopstar f) \<longrightarrow> (schopstar f) \<frown> (schopstar g) "
shows "\<turnstile> (schopstar (f \<or> g)) = (schopstar f) \<frown> (schopstar g) "
using assms 
using SChopstar_church_rosser SChopstar_confluence by blast

lemma SChopstarMore_church_rosser_var: 
assumes "\<turnstile> (g \<and> more) \<frown> (schopstar f) \<longrightarrow> (schopstar f) \<frown> (schopstar g) "
shows "\<turnstile> (schopstar (f \<or> g)) = (schopstar f) \<frown> (schopstar g) "
using assms SChopstar_church_rosser_var[of "LIFT g \<and> more" "LIFT f \<and> more" ] 
 SChopstar_and_more[of g] SChopstar_and_more[of f]
SChopstar_and_more[of "LIFT (f \<or> g)"]   
SChopstarMore_or_and[of f g] 
by (metis inteq_reflection)

lemma WPowerstar_church_rosser_to_confluence: 
 assumes "\<turnstile> (wpowerstar (f \<or> g)) = (wpowerstar f) ; (wpowerstar g) "
 shows "\<turnstile> (wpowerstar g) ; (wpowerstar f) \<longrightarrow> (wpowerstar f) ; (wpowerstar g) "
using assms 
by (metis WPowerstar_subdist_var_3 WPowerstar_swap inteq_reflection)

lemma SChopstar_church_rosser_to_confluence: 
 assumes "\<turnstile> (schopstar (f \<or> g)) = (schopstar f) \<frown> (schopstar g) "
 shows "\<turnstile> (schopstar g) \<frown> (schopstar f) \<longrightarrow> (schopstar f) \<frown> (schopstar g) "
using assms 
by (metis SChopstar_subdist_var_3 SChopstar_swap inteq_reflection)



lemma WPowerstar_church_rosser_equiv: 
 "(\<turnstile> (wpowerstar g) ; (wpowerstar f) \<longrightarrow> (wpowerstar f) ; (wpowerstar g)) =
  (\<turnstile> (wpowerstar (f \<or> g)) = (wpowerstar f) ; (wpowerstar g)) "
using WPowerstar_church_rosser WPowerstar_church_rosser_to_confluence by blast

lemma SChopstar_church_rosser_equiv: 
 "(\<turnstile> (schopstar g) \<frown> (schopstar f) \<longrightarrow> (schopstar f) \<frown> (schopstar g)) =
  (\<turnstile> (schopstar (f \<or> g)) = (schopstar f) \<frown> (schopstar g)) "
using SChopstar_church_rosser SChopstar_church_rosser_to_confluence by blast


lemma WPowerstar_confluence_to_local_confluence: 
 assumes "\<turnstile>  (wpowerstar g) ; (wpowerstar f) \<longrightarrow> (wpowerstar f) ; (wpowerstar g)"
 shows " \<turnstile> g ; f \<longrightarrow> (wpowerstar f) ; (wpowerstar g) " 
using assms 
WPowerstar_church_rosser[of g f]
by (metis  WPowerstar_denest WPowerstar_denest_var_4 WPowerstar_subdist_var_2 inteq_reflection)

lemma SChopstar_confluence_to_local_confluence: 
 assumes "\<turnstile>  (schopstar g) \<frown> (schopstar f) \<longrightarrow> (schopstar f) \<frown> (schopstar g)"
 shows " \<turnstile> (g) \<frown> (f \<and> finite) \<longrightarrow> (schopstar f) \<frown> (schopstar g) " 
using assms 
SChopstar_church_rosser[of g f] 
by (metis SChopstar_denest SChopstar_denest_var_4 SChopstar_subdist_var_2 inteq_reflection)

lemma SChopstarMore_confluence_to_local_confluence: 
 assumes "\<turnstile>  (schopstar g) \<frown> (schopstar f) \<longrightarrow> (schopstar f) \<frown> (schopstar g)"
 shows " \<turnstile> (g \<and> more) \<frown> (f \<and> finite) \<longrightarrow> (schopstar f) \<frown> (schopstar g) " 
using assms 
SChopstar_confluence_to_local_confluence[of "LIFT g \<and> more" f]
SChopstar_and_more[of g]
by (metis inteq_reflection)

lemma WPowerstar_sup_id_star1: 
assumes "\<turnstile> empty \<longrightarrow> f" 
shows   "\<turnstile> f;(wpowerstar f) = (wpowerstar f) " 
using assms 
by (metis (no_types, lifting) AndEmptyChopAndEmptyEqvAndEmpty ChopImpChop ChopOrEqv ChopOrImp 
    FiniteAndEmptyEqvEmpty Prop02 WPowerstarEqv WPowerstar_1L WPowerstar_imp_empty int_iffI inteq_reflection)

lemma SChopstar_sup_id_star1: 
assumes "\<turnstile> empty \<longrightarrow> f" 
shows   "\<turnstile> f\<frown>(schopstar f) = (schopstar f) " 
using assms 
by (metis EmptyImpFinite Prop12 SChopstar_WPowerstar WPowerstar_sup_id_star1 inteq_reflection schop_d_def)

lemma WPowerstar_sup_id_star2: 
assumes "\<turnstile> empty \<longrightarrow> f" 
shows "\<turnstile> (wpowerstar f);f = (wpowerstar f) "
using assms
by (metis ChopEmpty ChopImpChop  WPowerstar_1R int_eq int_iffD2 int_iffI)

lemma SChopstar_sup_id_star2: 
assumes "\<turnstile> empty \<longrightarrow> f" 
shows "\<turnstile> (schopstar f)\<frown>(f \<and> finite) = (schopstar f) "
using assms
WPowerstar_sup_id_star2[of "LIFT (f \<and> finite)"]  SChopstar_WPowerstar[of f]  
by (metis EmptyImpFinite Prop10 Prop12 SChopstar_finite inteq_reflection schop_d_def)

lemma WPowerstar_unfoldr_eq:
"\<turnstile> (empty \<or> (wpowerstar f);f) = (wpowerstar f) " 
proof -
 have 1: "\<turnstile> (empty \<or> (wpowerstar f);f) \<longrightarrow> (wpowerstar f) "
  using WPowerstar_unfoldR by auto
 have 2: "\<turnstile> (empty \<or> f;(empty \<or> (wpowerstar f);f)) =
            (empty \<or> (empty \<or> f;(wpowerstar f));f) " 
   by (metis (no_types, lifting) ChopAssoc ChopEmpty ChopOrEqv EmptyOrChopEqv inteq_reflection)
 have 3: "\<turnstile> (empty \<or> (empty \<or> f;(wpowerstar f));f) =
            (empty \<or> (wpowerstar f);f) " 
    by (metis "2" inteq_reflection wpowerstar_unfoldl_eq)
 have 4: "\<turnstile> (wpowerstar f) \<longrightarrow> (empty \<or> (wpowerstar f);f) "
    by (metis "2" "3" WPowerstar_induct_lvar_empty int_eq int_iffD1)
 show ?thesis 
 using "1" "4" int_iffI by blast
qed

lemma SChopstar_unfoldr_eq:
"\<turnstile> (empty \<or> (schopstar f)\<frown>(f \<and> finite)) = (schopstar f) " 
proof -
 have 1: "\<turnstile> (empty \<or> (schopstar f)\<frown>(f \<and> finite)) \<longrightarrow> (schopstar f) "
    using SChopstar_unfoldR by auto
 have 2: "\<turnstile> (empty \<or> f\<frown>(empty \<or> (schopstar f)\<frown>(f \<and> finite))) =
            (empty \<or> (empty \<or> f\<frown>(schopstar f))\<frown>(f \<and> finite)) "
    by (metis (no_types, lifting) ChopEmpty EmptyOrSChopEqv SChopAssoc SChopOrEqv inteq_reflection itl_def(9)) 
 have 3: "\<turnstile> (empty \<or> (empty \<or> f\<frown>(schopstar f))\<frown>(f \<and> finite)) =
            (empty \<or> (schopstar f)\<frown>(f \<and> finite)) " 
   by (metis "2" SChopstar_unfoldl_eq inteq_reflection)
 have 4: "\<turnstile> (schopstar f) \<longrightarrow> (empty \<or> (schopstar f)\<frown>(f \<and> finite)) "
   by (metis "2" "3" SChopstar_induct_lvar_empty int_eq int_iffD1)
 show ?thesis 
 using "1" "4" int_iffI by blast
qed

lemma SChopstarMore_unfoldr_eq:
"\<turnstile> (empty \<or> (schopstar f)\<frown>((f \<and> more) \<and> finite)) = (schopstar f) " 
using SChopstar_unfoldr_eq[of "LIFT f \<and> more"] 
SChopstar_and_more[of f]
by (metis inteq_reflection)
 
lemma WPowerstar_star_prod_unfold: 
" \<turnstile> (empty \<or> f;((wpowerstar (g;f));g) ) = (wpowerstar (f ; g)) " 
proof -
 have 1: "\<turnstile> (wpowerstar (f ; g)) = (empty \<or> (wpowerstar (f;g));(f;g)) " 
   by (meson WPowerstar_unfoldR WPowerstar_unfoldr_eq int_iffD2 int_iffI)
 have 2: "\<turnstile> (wpowerstar (f;g));(f;g) \<longrightarrow> f;((wpowerstar (g;f));g) "
   using WPowerstar_slide1[of f g]
   by (metis AndChopB ChopAssoc Prop10 int_eq) 
 have 3: "\<turnstile> (wpowerstar (f ; g)) \<longrightarrow> (empty \<or> f;((wpowerstar (g;f));g)) " 
    using 1 2 by fastforce
 have 4: "\<turnstile> f;((wpowerstar (g;f));g) \<longrightarrow> (f;g);(wpowerstar (f ; g)) "
    using WPowerstar_slide1[of g f] 
     by (metis ChopAssoc RightChopImpChop inteq_reflection)
 have 5: "\<turnstile> (empty \<or> f;((wpowerstar (g;f));g) ) \<longrightarrow>
            empty \<or> (f;g);(wpowerstar (f ; g)) " 
  using 4 by fastforce
 have 6: "\<turnstile> (empty \<or> (f;g);(wpowerstar (f ; g))) \<longrightarrow> (wpowerstar (f ; g)) "
     by (meson WPowerstarEqv int_iffD2)
 show ?thesis 
 by (meson "3" "5" "6" int_iffI lift_imp_trans)
qed

lemma FiniteImportSChopRight: 
 "\<turnstile> (finite \<and> (f \<frown> g)) = f\<frown>(g \<and> finite) " 
by (metis ChopEmpty SChopAssoc inteq_reflection lift_and_com schop_d_def)


lemma SChopstar_star_prod_unfold: 
" \<turnstile> (empty \<or> (f)\<frown>((schopstar (g\<frown>f))\<frown>(g \<and> finite)) ) = (schopstar (f \<frown> g)) " 
proof -
 have 1: "\<turnstile> (schopstar (f \<frown> g)) = (empty \<or> (schopstar (f\<frown>g))\<frown>(f\<frown>(g \<and> finite))) " 
    by (metis FiniteImportSChopRight SChopAndCommute SChopstar_unfoldr_eq inteq_reflection)
 have 2: "\<turnstile> (schopstar (f\<frown>g))\<frown>(f\<frown>(g \<and> finite)) \<longrightarrow> (f)\<frown>((schopstar (g\<frown>f))\<frown>(g \<and> finite)) "
   using SChopstar_slide1[of f g]
   by (metis (no_types, opaque_lifting) ChopAndFiniteDist ChopEmpty LeftSChopImpSChop SChopAssoc
        inteq_reflection schop_d_def)
 have 3: "\<turnstile> (schopstar (f \<frown> g)) \<longrightarrow> (empty \<or> (f)\<frown>((schopstar (g\<frown>f))\<frown>(g \<and> finite))) " 
    using 1 2 by fastforce
 have 4: "\<turnstile> (f)\<frown>((schopstar (g\<frown>f))\<frown>(g \<and> finite)) \<longrightarrow> (f\<frown>g)\<frown>(schopstar (f \<frown> g)) "
    using SChopstar_slide1[of g f] 
    by (metis RightSChopImpSChop SChopAssoc inteq_reflection)
 have 5: "\<turnstile> (empty \<or> (f)\<frown>((schopstar (g\<frown>f))\<frown>(g \<and> finite)) ) \<longrightarrow>
            empty \<or> (f\<frown>g)\<frown>(schopstar (f \<frown> g)) " 
  using 4 by fastforce
 have 6: "\<turnstile> (empty \<or> (f\<frown>g)\<frown>(schopstar (f \<frown> g))) \<longrightarrow> (schopstar (f \<frown> g)) "
   by (meson Prop02 SChopstar_1L SChopstar_imp_empty)
 show ?thesis 
 by (meson "3" "5" "6" int_iffI lift_imp_trans)
qed

lemma WPowerstar_slide :
 "\<turnstile> (wpowerstar (f;g));f = f;(wpowerstar (g;f)) " 
proof -
 have 1: "\<turnstile> f;(wpowerstar (g;f)) = f;(empty \<or> g;((wpowerstar (f;g));f) ) "
   by (metis RightChopEqvChop WPowerstar_star_prod_unfold inteq_reflection)
 have 2: "\<turnstile> f;(empty \<or> g;((wpowerstar (f;g));f) ) = 
            (f;empty \<or> f;(g;((wpowerstar (f;g));f))) " 
   by (simp add: ChopOrEqv)
 have 3: "\<turnstile> f;empty = empty;f "
   by (metis ChopEmpty EmptyChop int_eq)
 have 4: "\<turnstile> f;(g;((wpowerstar (f;g));f)) =
            ((f;g);(wpowerstar (f;g)));f "
  by (metis ChopAssoc int_eq)
 have 5: "\<turnstile> (f;empty \<or> f;(g;((wpowerstar (f;g));f))) =
            (empty;f \<or> ((f;g);(wpowerstar (f;g)));f) " 
    using 3 4 by fastforce
 have 6: "\<turnstile> (empty;f \<or> ((f;g);(wpowerstar (f;g)));f) =
            (empty \<or> ((f;g);(wpowerstar (f;g))));f "
    by (meson OrChopEqv Prop11)
 have 7: "\<turnstile> f;(empty \<or> g;((wpowerstar (f;g));f) ) =
            (empty \<or> (f;g);(wpowerstar (f;g)));f " 
   by (metis "2" "3" "4" "6" inteq_reflection)
 have 8: "\<turnstile> (empty \<or> (f;g);(wpowerstar (f;g))) = (wpowerstar (f;g)) "
    by (simp add: wpowerstar_unfoldl_eq) 
 show ?thesis by (metis "1" "7" "8" int_eq)
qed

lemma SChopstar_slide :
 "\<turnstile> (schopstar (f\<frown>g))\<frown>(f \<and> finite) = f\<frown>(schopstar (g\<frown>f)) " 
proof -
 have 1: "\<turnstile> f\<frown>(schopstar (g\<frown>f)) = f\<frown>(empty \<or> g\<frown>((schopstar (f\<frown>g))\<frown>(f \<and> finite)) ) "
  by (metis RightSChopEqvSChop SChopstar_star_prod_unfold inteq_reflection) 
 have 2: "\<turnstile> f\<frown>(empty \<or> g\<frown>((schopstar (f\<frown>g))\<frown>(f \<and> finite)) ) = 
            (f\<frown>empty \<or> f\<frown>(g\<frown>((schopstar (f\<frown>g))\<frown>(f \<and> finite)))) " 
   by (simp add: SChopOrEqv)
 have 3: "\<turnstile> f\<frown>empty = empty\<frown>(f \<and> finite) " 
   by (metis DiamondEmptyEqvFinite EmptySChop FiniteAndEmptyEqvEmpty SChopAndCommute
        SChopAndEmptyEqvSChopAndEmpty TrueSChopEqvDiamond inteq_reflection)
 have 4: "\<turnstile> f\<frown>(g\<frown>((schopstar (f\<frown>g))\<frown>(f \<and> finite))) =
            ((f\<frown>g)\<frown>(schopstar (f\<frown>g)))\<frown>(f \<and> finite) " 
   by (metis SChopAssoc inteq_reflection)
 have 5: "\<turnstile> (f\<frown>empty \<or> f\<frown>(g\<frown>((schopstar (f\<frown>g))\<frown>(f \<and> finite)))) =
            (empty\<frown>(f \<and> finite) \<or> ((f\<frown>g)\<frown>(schopstar (f\<frown>g)))\<frown>(f \<and> finite)) " 
    using 3 4 by fastforce
 have 6: "\<turnstile> (empty\<frown>(f \<and> finite) \<or> ((f\<frown>g)\<frown>(schopstar (f\<frown>g)))\<frown>(f \<and> finite)) =
            (empty \<or> ((f\<frown>g)\<frown>(schopstar (f\<frown>g))))\<frown>(f \<and> finite) "
    by (meson OrSChopEqv Prop11) 
 have 7: "\<turnstile> f\<frown>(empty \<or> g\<frown>((schopstar (f\<frown>g))\<frown>(f \<and> finite)) ) =
            (empty \<or> (f\<frown>g)\<frown>(schopstar (f\<frown>g)))\<frown>(f \<and> finite) " 
   by (metis "2" "3" "4" "6" inteq_reflection)
 have 8: "\<turnstile> (empty \<or> (f\<frown>g)\<frown>(schopstar (f\<frown>g))) = (schopstar (f\<frown>g)) " 
     by (simp add: SChopstar_unfoldl_eq)
 show ?thesis by (metis "1" "7" "8" int_eq)
qed

lemma WPowerstar_slide_var :     
"\<turnstile> (wpowerstar f) ; f = f;(wpowerstar f) " 
by (metis EmptyOrChopEqv Prop11 WPowerstarInductR WPowerstar_slide1_var1 WPowerstar_unfoldr_eq int_eq)

lemma SChopstar_slide_var :     
"\<turnstile> (schopstar f) \<frown> (f \<and> finite) = f\<frown>(schopstar f) " 
using WPowerstar_slide_var
by (metis (no_types, lifting)  FPowerstar_WPowerstar Prop10 SChopstar_finite SChopstar_WPowerstar 
    int_eq  schop_d_def)

lemma SChopstarMore_slide_var :     
"\<turnstile> (schopstar f) \<frown> ((f \<and> more) \<and> finite) = (f \<and> more)\<frown>(schopstar f) " 
using SChopstar_slide_var[of "LIFT f \<and> more"] 
SChopstar_and_more[of f] 
by (metis  inteq_reflection)

lemma WPowerstar_or_unfold_var: 
 "\<turnstile> (empty \<or> (wpowerstar f);((wpowerstar (f \<or> g));(wpowerstar g))) = (wpowerstar (f \<or> g)) "
by (metis ChopAssoc WPowerstar_denest WPowerstar_denest_var_4 WPowerstar_slide inteq_reflection 
    wpowerstar_unfoldl_eq) 

lemma SChopstar_or_unfold_var: 
 "\<turnstile> (empty \<or> (schopstar f)\<frown>((schopstar (f \<or> g))\<frown>(schopstar g))) = (schopstar (f \<or> g)) " 
proof -
 have 1: "\<turnstile> (schopstar f)\<frown>((schopstar (f \<or> g))\<frown>(schopstar g)) =
            (schopstar f)\<frown>((schopstar (f \<or> g))\<frown>((schopstar g) \<and> finite)) "   
   by (simp add: Prop10 RightSChopEqvSChop SChopstar_finite)
 have 2: "\<turnstile> (schopstar f)\<frown>((schopstar (f \<or> g))\<frown>((schopstar g) \<and> finite)) =
            (schopstar (f \<or> g)) " 
   by (metis (no_types, lifting) SChopstar_denest_var SChopstar_denest_var_2 SChopstar_denest_var_3 
       SChopstar_slide SChopstar_swap int_eq) 
 show ?thesis 
 using "1" "2" SChopstar_imp_empty by fastforce
qed


lemma WPowerstar_or_unfold:
 "\<turnstile> ((wpowerstar f) \<or> (wpowerstar f);(g;(wpowerstar (f \<or> g)))) = (wpowerstar (f \<or> g)) "
proof -
 have 1: "\<turnstile> (wpowerstar (f \<or> g)) = (wpowerstar f);(wpowerstar (g;(wpowerstar f))) " 
    by (meson Prop11 WPowerstar_denest_var)
 have 2: "\<turnstile> (wpowerstar f);(wpowerstar (g;(wpowerstar f))) =
            (wpowerstar f);(empty \<or> (g;(wpowerstar f));(wpowerstar (g;(wpowerstar f)))) " 
   using RightChopEqvChop WPowerstarEqv by blast
 have 3: "\<turnstile> (wpowerstar f);(empty \<or> (g;(wpowerstar f));(wpowerstar (g;(wpowerstar f)))) =
            (wpowerstar f);(empty \<or> g;(wpowerstar (f \<or> g)))"
   by (metis "1" "2" ChopAssoc int_eq)
 have 4: "\<turnstile> (wpowerstar f);(empty \<or> g;(wpowerstar (f \<or> g))) =
            ((wpowerstar f);empty \<or> (wpowerstar f);(g;(wpowerstar (f \<or> g)))) " 
   using ChopOrEqv by blast
 have 5: "\<turnstile> (wpowerstar f);empty = (wpowerstar f) "
    by (simp add: ChopEmpty)
 have 6: "\<turnstile> ((wpowerstar f);empty \<or> (wpowerstar f);(g;(wpowerstar (f \<or> g)))) =
            ((wpowerstar f) \<or> (wpowerstar f);(g;(wpowerstar (f \<or> g)))) " 
    using "5" by auto
 show ?thesis 
 by (metis "1" "2" "3" "4" "6" int_eq)
qed

lemma SChopstar_or_unfold:
 "\<turnstile> ((schopstar f) \<or> (schopstar f)\<frown>(g\<frown>(schopstar (f \<or> g)))) = (schopstar (f \<or> g)) "
proof -
 have 1: "\<turnstile> (schopstar (f \<or> g)) = (schopstar f)\<frown>(schopstar (g\<frown>(schopstar f))) " 
    by (meson Prop11 SChopstar_denest_var)
 have 2: "\<turnstile> (schopstar f)\<frown>(schopstar (g\<frown>(schopstar f))) =
            (schopstar f)\<frown>(empty \<or> (g\<frown>(schopstar f))\<frown>(schopstar (g\<frown>(schopstar f)))) " 
   by (meson Prop11 RightSChopEqvSChop SChopstar_unfoldl_eq)
 have 3: "\<turnstile> (schopstar f)\<frown>(empty \<or> (g\<frown>(schopstar f))\<frown>(schopstar (g\<frown>(schopstar f)))) =
            (schopstar f)\<frown>(empty \<or> g\<frown>(schopstar (f \<or> g)))"
   by (metis "1" "2" SChopAssoc inteq_reflection)
 have 4: "\<turnstile> (schopstar f)\<frown>(empty \<or> g\<frown>(schopstar (f \<or> g))) =
            ((schopstar f)\<frown>empty \<or> (schopstar f)\<frown>(g\<frown>(schopstar (f \<or> g)))) " 
   using SChopOrEqv by blast
 have 5: "\<turnstile> (schopstar f)\<frown>empty = (schopstar f) "
   by (metis ChopEmpty Prop10 SChopstar_finite inteq_reflection schop_d_def)
 have 6: "\<turnstile> ((schopstar f)\<frown>empty \<or> (schopstar f)\<frown>(g\<frown>(schopstar (f \<or> g)))) =
            ((schopstar f) \<or> (schopstar f)\<frown>(g\<frown>(schopstar (f \<or> g)))) " 
    using "5" by auto
 show ?thesis 
 by (metis "1" "2" "3" "4" "6" int_eq)
qed

lemma SChopstarMore_or_unfold:
 "\<turnstile> ((schopstar f) \<or> (schopstar f)\<frown>((g \<and> more)\<frown>(schopstar (f \<or> g)))) = (schopstar (f \<or> g)) "
using SChopstar_or_unfold[of "LIFT f \<and> more" "LIFT g \<and> more"]
SChopstar_and_more[of f] SChopstar_and_more[of g]
SChopstar_and_more[of "LIFT (f \<or> g)"] 
SChopstarMore_or_and[of f g]
by (metis  inteq_reflection) 


lemma WPowerstar_troeger: 
 "\<turnstile> (wpowerstar (f \<or> g));h = 
    (wpowerstar f);(g;((wpowerstar (f \<or> g));h) \<or> h) " 
proof -
 have 1: "\<turnstile> (wpowerstar (f \<or> g));h =
            ((wpowerstar f) \<or> (wpowerstar f);(g;(wpowerstar (f \<or> g))));h "
   using WPowerstar_or_unfold[of f g] by (metis LeftChopEqvChop int_eq)
 have 2: "\<turnstile> ((wpowerstar f) \<or> (wpowerstar f);(g;(wpowerstar (f \<or> g))));h =
            ((wpowerstar f);h \<or> ((wpowerstar f);(g;(wpowerstar (f \<or> g))));h) " 
   by (simp add: OrChopEqv)
 have 3: "\<turnstile> ((wpowerstar f);(g;(wpowerstar (f \<or> g))));h =
            (wpowerstar f);(g;((wpowerstar (f \<or> g));h)) " 
    by (metis ChopAssoc inteq_reflection)
 have 3: "\<turnstile> ((wpowerstar f);h \<or> ((wpowerstar f);(g;(wpowerstar (f \<or> g))));h) =
            (wpowerstar f);(h \<or> g;((wpowerstar (f \<or> g));h)) "
    by (metis "3" ChopOrEqv inteq_reflection)
 have 4: "\<turnstile> (h \<or> g;((wpowerstar (f \<or> g));h)) =
            (g;((wpowerstar (f \<or> g));h)  \<or> h) "
     by fastforce
 show ?thesis 
 by (metis "1" "2" "3" "4" int_eq)
qed

lemma SChopstar_troeger: 
 "\<turnstile> (schopstar (f \<or> g))\<frown>(h) = 
    (schopstar f)\<frown>(g\<frown>((schopstar (f \<or> g))\<frown>(h)) \<or> (h)) " 
proof -
 have 1: "\<turnstile> (schopstar (f \<or> g))\<frown>(h) =
            ((schopstar f) \<or> (schopstar f)\<frown>(g\<frown>(schopstar (f \<or> g))))\<frown>(h) "
   using SChopstar_or_unfold[of f g] by (metis LeftSChopEqvSChop int_eq)
 have 2: "\<turnstile> ((schopstar f) \<or> (schopstar f)\<frown>(g\<frown>(schopstar (f \<or> g))))\<frown>(h) =
            ((schopstar f)\<frown>h \<or> ((schopstar f)\<frown>(g\<frown>(schopstar (f \<or> g))))\<frown>h) " 
   by (simp add: OrSChopEqv)
 have 3: "\<turnstile> ((schopstar f)\<frown>(g\<frown>(schopstar (f \<or> g))))\<frown>h =
            (schopstar f)\<frown>(g\<frown>((schopstar (f \<or> g))\<frown>(h))) " 
    by (metis SChopAssoc inteq_reflection)
 have 3: "\<turnstile> ((schopstar f)\<frown>h \<or> ((schopstar f)\<frown>(g\<frown>(schopstar (f \<or> g))))\<frown>h) =
            (schopstar f)\<frown>((h) \<or> g\<frown>((schopstar (f \<or> g))\<frown>(h))) "
    by (metis "3" SChopOrEqv inteq_reflection)
 have 4: "\<turnstile> ((h) \<or> g\<frown>((schopstar (f \<or> g))\<frown>(h))) =
            (g\<frown>((schopstar (f \<or> g))\<frown>(h))  \<or> (h)) "
     by fastforce
 show ?thesis 
 by (metis "1" "2" "3" "4" int_eq)
qed

lemma SChopstarMore_troeger: 
 "\<turnstile> (schopstar (f \<or> g))\<frown>(h) = 
    (schopstar f)\<frown>((g \<and> more)\<frown>((schopstar (f \<or> g))\<frown>(h)) \<or> (h)) " 
using SChopstar_troeger[of "LIFT f \<and> more" "LIFT g \<and> more" h]
SChopstar_and_more[of f] SChopstar_and_more[of g]
SChopstar_and_more[of "LIFT (f \<or> g)"] 
SChopstarMore_or_and[of f g]
by (metis  inteq_reflection)

lemma WPowerstar_square:
 "\<turnstile> (wpowerstar (f;f)) \<longrightarrow> (wpowerstar f) " 
proof -
 have 1: "\<turnstile> (f;f);(wpowerstar f) \<longrightarrow> (wpowerstar f) " 
   by (simp add: Chop_WPowerstar_Closure WPowerstar_ext) 
 show ?thesis 
 by (simp add: "1" WPowerstar_induct_lvar_star)
qed

lemma SChopstar_square:
 "\<turnstile> (schopstar (f\<frown>f)) \<longrightarrow> (schopstar f) " 
proof -
 have 1: "\<turnstile> (f\<frown>f)\<frown>(schopstar f) \<longrightarrow> (schopstar f) "  
   by (metis Prop05 RightSChopImpSChop SChopAssoc SChopstar_1L SChopstar_unfoldl_eq inteq_reflection)
 show ?thesis 
 by (simp add: "1" SChopstar_induct_lvar_star)
qed

lemma WPowerstar_meyer_1: 
"\<turnstile> (empty \<or> f);(wpowerstar (f ; f)) = (wpowerstar f) " 
proof -
 have 1: "\<turnstile> f;((empty \<or> f);(wpowerstar (f ; f))) =
            f;(empty;(wpowerstar (f ; f)) \<or> f;(wpowerstar (f ; f))) "
   by (simp add: OrChopEqv RightChopEqvChop)
 have 2: "\<turnstile> (empty;(wpowerstar (f ; f)) \<or> f;(wpowerstar (f ; f))) =
            ((wpowerstar (f ; f)) \<or> f;(wpowerstar (f ; f))) "
   by (meson EmptyOrChopEqv OrChopEqv Prop04)
 have 3: "\<turnstile> f;(empty;(wpowerstar (f ; f)) \<or> f;(wpowerstar (f ; f))) =
            f;((wpowerstar (f ; f)) \<or> f;(wpowerstar (f ; f))) " 
   using "2" RightChopEqvChop by blast
 have 4: "\<turnstile> f;((wpowerstar (f ; f)) \<or> f;(wpowerstar (f ; f))) \<longrightarrow>
            f;(wpowerstar (f ; f)) \<or> (wpowerstar (f ; f)) " 
     by (metis ChopAssoc ChopOrImp Prop05 Prop08 int_eq int_iffD2 wpowerstar_unfoldl_eq)
 have 41: "\<turnstile> f;((empty \<or> f);(wpowerstar (f ; f))) \<longrightarrow>  f;(wpowerstar (f;f) \<or> f;wpowerstar (f;f))" 
     using  EmptyOrChopEqv[of f "LIFT (wpowerstar (f ; f))"] 
       by (metis "1" Prop11 inteq_reflection)
 have 42: "\<turnstile>  f;(wpowerstar (f;f) \<or> f;wpowerstar (f;f)) = (f;wpowerstar (f;f) \<or> f;(f;wpowerstar (f;f)))" 
   using ChopOrEqv[of f "LIFT wpowerstar (f;f)" "LIFT f;wpowerstar (f;f)"] 
     by auto
 have 43: "\<turnstile> (f;wpowerstar (f;f) \<or> f;(f;wpowerstar (f;f))) =
              ( f;wpowerstar (f;f) \<or> (f;f);wpowerstar (f;f)) "
     using ChopAssoc[of f f "LIFT wpowerstar (f;f)"]   by auto
 have 44: "\<turnstile> ( f;wpowerstar (f;f) \<or> (f;f);wpowerstar (f;f)) \<longrightarrow> ((empty \<or> f);(wpowerstar (f ; f)))"
    using wpowerstar_unfoldl_eq[of "LIFT (f;f)" ]  
    using EmptyOrChopEqv by fastforce
 have 5: "\<turnstile> f;((empty \<or> f);(wpowerstar (f ; f))) \<longrightarrow> 
            ((empty \<or> f);(wpowerstar (f ; f))) "
  by (metis "1" "3" "42" "43" "44" int_eq)
 have 6: "\<turnstile> empty \<longrightarrow> ((empty \<or> f);(wpowerstar (f ; f))) "
  using EmptyOrChopEqv wpowerstar_unfoldl_eq by fastforce
 have 7: "\<turnstile> empty \<or> f;((empty \<or> f);(wpowerstar (f ; f))) \<longrightarrow>
            ((empty \<or> f);(wpowerstar (f ; f))) "
     using "5" "6" Prop02 by blast
 have 8: "\<turnstile> (wpowerstar f) \<longrightarrow> ((empty \<or> f);(wpowerstar (f ; f))) "
    using "7" WPowerstar_induct_lvar_empty by blast
 have 9: "\<turnstile> ((empty \<or> f);(wpowerstar (f ; f))) \<longrightarrow> (empty \<or> f) ;(wpowerstar f) "
    using WPowerstar_square[of f] 
    using RightChopImpChop by blast
 have 10: "\<turnstile> (empty \<or> f) ;(wpowerstar f) \<longrightarrow> (wpowerstar f) "
    by (meson Prop02 WPowerstarInductR WPowerstar_1R WPowerstar_ext WPowerstar_imp_empty)
 show ?thesis 
 by (meson "10" "8" "9" int_iffI lift_imp_trans)
qed

lemma SChopstar_meyer_1: 
"\<turnstile> (empty \<or> f)\<frown>(schopstar (f \<frown> f)) = (schopstar f) " 
proof -
 have 1: "\<turnstile> f\<frown>((empty \<or> f)\<frown>(schopstar (f \<frown> f))) =
            f\<frown>(empty\<frown>(schopstar (f \<frown> f)) \<or> f\<frown>(schopstar (f \<frown> f))) "
   by (simp add: OrSChopEqv RightSChopEqvSChop)
 have 2: "\<turnstile> (empty\<frown>(schopstar (f \<frown> f)) \<or> f\<frown>(schopstar (f \<frown> f))) =
            ((schopstar (f \<frown> f)) \<or> f\<frown>(schopstar (f \<frown> f))) "
   by (meson EmptyOrSChopEqv OrSChopEqv Prop04)
 have 3: "\<turnstile> f\<frown>(empty\<frown>(schopstar (f \<frown> f)) \<or> f\<frown>(schopstar (f \<frown> f))) =
            f\<frown>((schopstar (f \<frown> f)) \<or> f\<frown>(schopstar (f \<frown> f))) " 
   using "2" RightSChopEqvSChop by blast
 have 4: "\<turnstile> f\<frown>((schopstar (f \<frown> f)) \<or> f\<frown>(schopstar (f \<frown> f))) \<longrightarrow>
            (schopstar (f \<frown> f)) \<or> f\<frown>(schopstar (f \<frown> f))   " 
    using SChopAssoc SChopOrEqv SChopstar_unfoldl_eq by fastforce
 have 41: "\<turnstile> ((schopstar (f \<frown> f)) \<or> f\<frown>(schopstar (f \<frown> f))) =
             ((empty \<or> f)\<frown>(schopstar (f \<frown> f))) "
      by (metis "2" OrSChopEqv inteq_reflection)
 have 5: "\<turnstile> f\<frown>((empty \<or> f)\<frown>(schopstar (f \<frown> f))) \<longrightarrow> 
            ((empty \<or> f)\<frown>(schopstar (f \<frown> f))) "
     by (metis "4" "41" int_eq)
 have 6: "\<turnstile> empty \<longrightarrow> ((empty \<or> f)\<frown>(schopstar (f \<frown> f))) "
   using EmptyOrSChopEqv SChopstar_imp_empty by fastforce
 have 7: "\<turnstile> empty \<or> f\<frown>((empty \<or> f)\<frown>(schopstar (f \<frown> f))) \<longrightarrow>
            ((empty \<or> f)\<frown>(schopstar (f \<frown> f))) "
     using "5" "6" Prop02 by blast
 have 8: "\<turnstile> (schopstar f) \<longrightarrow> ((empty \<or> f)\<frown>(schopstar (f \<frown> f))) "
   using "7" SChopstar_induct_lvar_empty by blast
 have 9: "\<turnstile> ((empty \<or> f)\<frown>(schopstar (f \<frown> f))) \<longrightarrow> (empty \<or> f) \<frown>(schopstar f) "
    using SChopstar_square[of f] 
    using RightSChopImpSChop by blast
 have 10: "\<turnstile> (empty \<or> f) \<frown>(schopstar f) \<longrightarrow> (schopstar f) "
    by (metis SChopstar_1L SChopstar_star2 inteq_reflection)
 show ?thesis 
 by (meson "10" "8" "9" int_iffI lift_imp_trans)
qed
  
lemma SChopstarMore_meyer_1: 
"\<turnstile> (empty \<or> (f \<and> more))\<frown>(schopstar (f \<frown> f)) = (schopstar f) " 
proof -
 have 1: "\<turnstile> (f \<and> more)\<frown>((empty \<or> (f \<and> more))\<frown>(schopstar (f \<frown> f))) =
            (f \<and> more)\<frown>(empty\<frown>(schopstar (f \<frown> f)) \<or> (f \<and> more)\<frown>(schopstar (f \<frown> f))) "
   by (simp add: OrSChopEqv RightSChopEqvSChop)
 have 2: "\<turnstile> (empty\<frown>(schopstar (f \<frown> f)) \<or> (f \<and> more)\<frown>(schopstar (f \<frown> f))) =
            ((schopstar (f \<frown> f)) \<or> (f \<and> more)\<frown>(schopstar (f \<frown> f))) "
   by (meson EmptyOrSChopEqv OrSChopEqv Prop04)
 have 3: "\<turnstile> (f \<and> more)\<frown>(empty\<frown>(schopstar (f \<frown> f)) \<or> (f \<and> more)\<frown>(schopstar (f \<frown> f))) =
            (f \<and> more)\<frown>((schopstar (f \<frown> f)) \<or> (f \<and> more)\<frown>(schopstar (f \<frown> f))) " 
   using "2" RightSChopEqvSChop by blast
 have 30: "\<turnstile> ((f \<and> more)\<frown>(f \<and> more)) \<longrightarrow> f \<frown> f " 
    by (metis Prop11 Prop12 SChopImpSChop lift_and_com)
 have 31: "\<turnstile> ((f \<and> more)\<frown>(f \<and> more)) \<frown> (schopstar (f \<frown> f)) \<longrightarrow> (schopstar (f \<frown> f)) "
   by (metis "30" AndSChopA Prop05 Prop10 SChopstar_unfoldl_eq int_eq lift_and_com)
 have 4: "\<turnstile> (f \<and> more)\<frown>((schopstar (f \<frown> f)) \<or> (f \<and> more)\<frown>(schopstar (f \<frown> f))) \<longrightarrow>
            (f \<and> more)\<frown>(schopstar (f \<frown> f)) \<or> (schopstar (f \<frown> f)) " 
       using "31" SChopAssoc SChopOrImp by fastforce
 have 5: "\<turnstile> (f \<and> more)\<frown>((empty \<or> (f \<and> more))\<frown>(schopstar (f \<frown> f))) \<longrightarrow> 
             ((empty \<or> (f \<and> more))\<frown>(schopstar (f \<frown> f))) "
   by (metis (no_types, opaque_lifting) "4" EmptyOrSChopEqv Prop11 int_simps(33) 
    inteq_reflection lift_and_com lift_imp_trans)
 have 6: "\<turnstile> empty \<longrightarrow> ((empty \<or> (f \<and> more))\<frown>(schopstar (f \<frown> f))) "
   using EmptyOrSChopEqv SChopstar_unfoldl_eq by fastforce
 have 7: "\<turnstile> empty \<or> (f \<and> more)\<frown>((empty \<or> (f \<and> more))\<frown>(schopstar (f \<frown> f))) \<longrightarrow>
            ((empty \<or> (f \<and> more))\<frown>(schopstar (f \<frown> f))) "
     using "5" "6" Prop02 by blast
 have 8: "\<turnstile> (schopstar f) \<longrightarrow> ((empty \<or> (f \<and> more))\<frown>(schopstar (f \<frown> f))) "
    using "7" SChopstarMore_induct_lvar_empty by blast
 have 9: "\<turnstile> ((empty \<or> (f \<and> more))\<frown>(schopstar (f \<frown> f))) \<longrightarrow> (empty \<or> (f \<and> more)) \<frown>(schopstar f) "
    using SChopstar_square[of f] 
    using RightSChopImpSChop by blast
 have 10: "\<turnstile> (empty \<or> (f \<and> more)) \<frown>(schopstar f) \<longrightarrow> (schopstar f) "
   by (metis SChopstar_1L SChopstar_and_more SChopstar_star2 inteq_reflection)
 show ?thesis 
 by (meson "10" "8" "9" int_iffI lift_imp_trans)
qed

lemma WPowerstar_tc: 
assumes "\<turnstile> f;f \<longrightarrow> f "
shows "\<turnstile> (wpowerstar f);f = f " 
proof -
 have 1: "\<turnstile> f \<or> f; f \<longrightarrow> f " 
   using assms by fastforce
 have 2: "\<turnstile> (wpowerstar f);f \<longrightarrow> f" 
  by (simp add: WPowerstar_inductL_var_equiv assms)
 have 3: "\<turnstile> f \<longrightarrow> (wpowerstar f);f "
   by (metis AndChopB EmptyChop Prop10 WPowerstar_imp_empty inteq_reflection)
 show ?thesis 
 by (simp add: "2" "3" Prop11)
qed

lemma SChopstar_tc: 
assumes "\<turnstile> f\<frown>f \<longrightarrow> f "
shows "\<turnstile> (schopstar f)\<frown>(f \<and> finite) = (f \<and> finite) " 
proof -
 have 1: "\<turnstile> f \<or> f\<frown> f \<longrightarrow> f " 
   using assms by fastforce
 have 2: "\<turnstile> (schopstar f)\<frown>(f \<and> finite) \<longrightarrow> f \<and> finite" 
  by (metis Prop12 SChopAndA SChopstar_1R SChopstar_finite SChopstar_induct_lvar assms lift_imp_trans)
 have 3: "\<turnstile> f \<and> finite\<longrightarrow> (schopstar f)\<frown>(f \<and> finite) "
  by (metis EmptySChop LeftSChopImpSChop SChopstar_imp_empty inteq_reflection)
 show ?thesis 
 by (simp add: "2" "3" Prop11)
qed
 
lemma WPowerstar_tc_eq: 
assumes "\<turnstile> f;f = f "
shows "\<turnstile> (wpowerstar f);f = f " 
using assms 
by (simp add: WPowerstar_induct_lvar_eq2)

lemma SChopstar_tc_eq: 
assumes "\<turnstile> f\<frown>f = f "
shows "\<turnstile> (schopstar f)\<frown>(f \<and> finite) = (f \<and> finite) " 
using assms 
by (simp add: SChopstar_tc int_iffD1)

lemma WPowerstar_boffa_var:
 assumes "\<turnstile> f;f \<longrightarrow> f "
 shows "\<turnstile> (wpowerstar f) = (empty \<or> f) " 
proof -
 have 1: "\<turnstile> (wpowerstar f) = (empty \<or> (wpowerstar f);f) "
    by (metis WPowerstar_slide_var WPowerstarEqv int_eq) 
 show ?thesis 
 using "1" Prop06 WPowerstar_tc assms by blast
qed

lemma SChopstar_boffa_var:
 assumes "\<turnstile> f\<frown>f \<longrightarrow> f "
 shows "\<turnstile> (schopstar f) = (empty \<or> (f \<and> finite)) " 
proof -
 have 1: "\<turnstile> (schopstar f) = (empty \<or> (schopstar f)\<frown>(f \<and> finite)) " 
   by (meson SChopstar_unfoldR SChopstar_unfoldr_eq int_iffD2 int_iffI)
 show ?thesis 
 using "1" Prop06 SChopstar_tc assms by blast
qed

lemma WPowerstar_boffa:
 assumes "\<turnstile> f;f = f "
 shows "\<turnstile> (wpowerstar f) = (empty \<or> f) " 
using assms 
by (simp add: WPowerstar_boffa_var int_iffD1)

lemma SChopstar_boffa:
 assumes "\<turnstile> f\<frown>f = f "
 shows "\<turnstile> (schopstar f) = (empty \<or> (f \<and> finite)) " 
using assms 
by (simp add: SChopstar_boffa_var int_iffD1)


lemma WPowerstar_sim2: 
 assumes "\<turnstile> h ; f \<longrightarrow> g ; h"
 shows "\<turnstile> h ; (wpowerstar f) \<longrightarrow> (wpowerstar g) ; h"
proof -
 have 1: "\<turnstile> (wpowerstar g) ; (h ; f) \<longrightarrow>  (wpowerstar g) ;( g ; h ) " 
    by (simp add: RightChopImpChop assms)
 have 2: "\<turnstile> (wpowerstar g) ;( g ; h ) \<longrightarrow> (wpowerstar g) ; h " 
   by (metis ChopAssoc LeftChopImpChop WPowerstar_1L WPowerstar_slide_var inteq_reflection)
 have 3: "\<turnstile> (wpowerstar g) ; (h ; f) \<longrightarrow> (wpowerstar g) ; h "
  using "1" "2" lift_imp_trans by blast
 have 4: "\<turnstile> h \<longrightarrow> (wpowerstar g) ; h " 
   by (metis AndChopB EmptyChop Prop10 WPowerstar_imp_empty inteq_reflection)
 have 5: "\<turnstile> h \<or> (wpowerstar g) ; (h ; f) \<longrightarrow> (wpowerstar g) ; h "
    using "3" "4" Prop02 by blast
 show ?thesis
   by (metis "5" ChopAssoc WPowerstarInductR inteq_reflection)
qed

lemma SChopstarInductR:
 assumes "\<turnstile> g \<or> h \<frown> f \<longrightarrow> h "
 shows " \<turnstile> g \<frown> schopstar f \<longrightarrow> h"
proof -
 have 1: "\<turnstile>   g \<and> finite  \<longrightarrow> h" 
   using assms by fastforce
 have 2: " \<turnstile> h \<frown> f \<longrightarrow> h "
    using assms by fastforce
 have 3: "\<turnstile> g \<and> finite \<or>  h \<frown> f \<longrightarrow> h "
   using 1 2 by fastforce
 have 4: "\<turnstile>  (g \<and> finite);fpowerstar f \<longrightarrow> h" 
   by (metis "1" "2" ChopSChopdef FPowerstarInductR OrFiniteInf Prop02 Prop03 inteq_reflection)
 show ?thesis 
 using FPowerstar_more_absorb[of f] 4 
 by (metis int_eq schop_d_def schopstar_d_def)
qed

lemma SChopstar_sim2: 
 assumes "\<turnstile> h \<frown> f \<longrightarrow> g \<frown> h"
 shows "\<turnstile> h \<frown> (schopstar f) \<longrightarrow> (schopstar g) \<frown> h"
proof -
 have 1: "\<turnstile> (schopstar g) \<frown> (h \<frown> (f)) \<longrightarrow>  (schopstar g) \<frown>( g \<frown> h ) " 
    by (simp add: RightSChopImpSChop assms)
 have 2: "\<turnstile> (schopstar g) \<frown>( g \<frown> h ) \<longrightarrow> (schopstar g) \<frown> h " 
    by (metis (no_types, lifting) ChopAssoc LeftChopImpChop Prop10 SChopstar_1L SChopstar_finite 
        SChopstar_WPowerstar WPowerstar_slide_var inteq_reflection schop_d_def)
 have 3: "\<turnstile> (schopstar g) \<frown> (h \<frown> (f)) \<longrightarrow> (schopstar g) \<frown> h "
  using "1" "2" lift_imp_trans by blast
 have 4: "\<turnstile> h \<longrightarrow> (schopstar g) \<frown> h " 
   by (meson EmptySChop LeftSChopImpSChop Prop11 SChopstar_imp_empty lift_imp_trans)
 have 5: "\<turnstile> h \<or> (schopstar g) \<frown> (h \<frown> (f)) \<longrightarrow> (schopstar g) \<frown> h "
    using "3" "4" Prop02 by blast
 show ?thesis 
  by (metis "5" SChopAssoc SChopstarInductR inteq_reflection)
qed

lemma SChopImpFinite: 
 " \<turnstile> f \<longrightarrow> finite \<Longrightarrow> \<turnstile> g \<frown> f \<longrightarrow> finite " 
by (metis DiamondEmptyEqvFinite FiniteAndEmptyEqvEmpty Prop10 SChopSFinExportA 
    SFinEqvTrueSChopAndEmpty TrueSChopEqvDiamond inteq_reflection)

lemma SChopstar_sim2_finite: 
 assumes "\<turnstile> h \<frown> f \<longrightarrow> g \<frown> h"
 shows "\<turnstile> h \<frown> (schopstar f) \<longrightarrow> (schopstar g) \<frown> (h \<and> finite)"
using assms SChopstar_sim2 
by (metis FiniteImportSChopRight Prop12 SChopImpFinite SChopstar_finite inteq_reflection)

lemma WPowerstar_inductr_var:
 assumes "\<turnstile> g ; f \<longrightarrow> g "
 shows   "\<turnstile> g ; (wpowerstar f) \<longrightarrow> g "
using assms 
by (simp add: WPowerstarInductR)

lemma SChopstar_inductr_var:
 assumes "\<turnstile> g \<frown> f \<longrightarrow> g "
 shows   "\<turnstile> g \<frown> (schopstar f) \<longrightarrow> g "
using assms 
by (simp add: SChopstarInductR)

lemma SChopstarMore_inductr_var:
 assumes "\<turnstile> g \<frown> (f \<and> more) \<longrightarrow> g "
 shows   "\<turnstile> g \<frown> (schopstar f) \<longrightarrow> g "
using assms 
by (metis SChopstar_and_more SChopstar_inductr_var inteq_reflection)

lemma SChopstar_finite_absorb: 
"\<turnstile> schopstar (f \<and> finite) = schopstar f"
by (metis Prop10 SChopstar_ext SChopstar_finite SChopstar_WPowerstar int_eq lift_imp_trans) 

lemma SChopstar_inductr_finite_var:
 assumes "\<turnstile> g \<frown> (f \<and> finite) \<longrightarrow> g "
 shows   "\<turnstile> g \<frown> (schopstar f) \<longrightarrow> g "
using assms 
by (metis SChopstar_finite_absorb SChopstar_inductr_var inteq_reflection)

lemma SChopstarMore_inductr_finite_var:
 assumes "\<turnstile> g \<frown> ((f \<and> more) \<and> finite) \<longrightarrow> g "
 shows   "\<turnstile> g \<frown> (schopstar f) \<longrightarrow> g "
using assms 
by (metis SChopstar_and_more SChopstar_inductr_finite_var inteq_reflection)


lemma WPowerstar_inductr_var_equiv:
 "(\<turnstile> g ; f \<longrightarrow> g ) = (\<turnstile> g ; (wpowerstar f) \<longrightarrow> g) " 
proof -
 have 1: " (\<turnstile> g ; f \<longrightarrow> g ) \<Longrightarrow> (\<turnstile> g ; (wpowerstar f) \<longrightarrow> g) "
   by (simp add: WPowerstar_inductr_var)
 have 2: " (\<turnstile> g ; (wpowerstar f) \<longrightarrow> g) \<Longrightarrow> (\<turnstile> g ; f \<longrightarrow> g ) "
   by (metis ChopAndB Prop10 WPowerstar_ext int_eq lift_imp_trans)
 show ?thesis  using "1" "2" by blast
qed

lemma SChopstar_inductr_var_equiv:
 "(\<turnstile> g \<frown> (f \<and> finite) \<longrightarrow> g ) = (\<turnstile> g \<frown> (schopstar f) \<longrightarrow> g) " 
proof -
 have 1: " (\<turnstile> g \<frown> (f \<and> finite) \<longrightarrow> g ) \<Longrightarrow> (\<turnstile> g \<frown> (schopstar f) \<longrightarrow> g) "
    by (simp add: SChopstar_inductr_finite_var)
 have 2: " (\<turnstile> g \<frown> (schopstar f) \<longrightarrow> g) \<Longrightarrow> (\<turnstile> g \<frown> (f \<and> finite) \<longrightarrow> g ) "
   by (metis Prop10 Prop12 SChopAndB SChopstar_ext inteq_reflection)
 show ?thesis  using "1" "2" by blast
qed

lemma SChopstarMore_inductr_var_equiv:
 "(\<turnstile> g \<frown> ((f \<and> more) \<and> finite) \<longrightarrow> g ) = (\<turnstile> g \<frown> (schopstar f) \<longrightarrow> g) " 
using SChopstar_inductr_var_equiv[of g "LIFT f \<and> more"]
SChopstar_and_more[of f]
 by (metis  inteq_reflection) 

lemma WPowerstar_sim3: 
 assumes "\<turnstile> h ; f = g ; h"
 shows "\<turnstile> h ; (wpowerstar f) = (wpowerstar g) ;h"
using assms 
by (simp add: Prop11 WPowerstar_sim1 WPowerstar_sim2)

lemma SChopstar_sim3: 
 assumes "\<turnstile> h \<frown> f = g \<frown> h"
 shows "\<turnstile> h \<frown> (schopstar f) = (schopstar g) \<frown> (h \<and> finite)"
using SChopstar_sim1 SChopstar_sim2_finite 
by (metis Prop11 assms)

lemma SChopstarMore_sim3: 
 assumes "\<turnstile> h \<frown> (f \<and> more) = (g \<and> more) \<frown> h"
 shows "\<turnstile> h \<frown> (schopstar f) = (schopstar g) \<frown> (h \<and> finite)"
by (metis SChopstar_and_more SChopstar_sim3 assms inteq_reflection)


lemma WPowerstar_sim4: 
assumes "\<turnstile> f ; g \<longrightarrow> g ; f "
shows   "\<turnstile> (wpowerstar f);(wpowerstar g) \<longrightarrow> (wpowerstar g) ; (wpowerstar f) " 
using assms 
by (simp add: WPowerstar_sim1 WPowerstar_sim2)

lemma SChopstar_sim4: 
assumes "\<turnstile> f \<frown> g \<longrightarrow> g \<frown> f "
shows   "\<turnstile> (schopstar f)\<frown>(schopstar g) \<longrightarrow> (schopstar g) \<frown> (schopstar f) " 
using assms 
by (metis Prop10 SChopstar_finite SChopstar_sim1 SChopstar_sim2 inteq_reflection)

lemma WPowerstar_inductr_eq: 
assumes "\<turnstile> (h \<or> g ; f) = g "
 shows  "\<turnstile> h;(wpowerstar f) \<longrightarrow> g "
using assms
using WPowerstarInductR int_iffD1 by blast

lemma SChopstar_inductr_eq: 
assumes "\<turnstile> (h \<or> g \<frown> f) = g "
 shows  "\<turnstile> h\<frown>(schopstar f) \<longrightarrow> g "
using assms using SChopstarInductR int_iffD1 by blast

lemma SChopstarMore_inductr_eq: 
assumes "\<turnstile> (h \<or> g \<frown> (f \<and> more)) = g "
 shows  "\<turnstile> h\<frown>(schopstar f) \<longrightarrow> g "
using assms 
by (metis SChopstar_and_more SChopstar_inductr_eq inteq_reflection)


lemma WPowerstar_inductr_var_eq: 
assumes "\<turnstile> ( g ; f) = g "
 shows  "\<turnstile> g;(wpowerstar f) \<longrightarrow> g "
using assms 
using WPowerstar_inductr_var int_iffD1 by blast

lemma SChopstar_inductr_var_eq: 
assumes "\<turnstile> ( g \<frown> f) = g "
 shows  "\<turnstile> g\<frown>(schopstar f) \<longrightarrow> g "
using assms 
using SChopstar_inductr_var int_iffD1 by blast

lemma SChopstarMore_inductr_var_eq: 
assumes "\<turnstile> ( g \<frown> (f \<and> more)) = g "
 shows  "\<turnstile> g\<frown>(schopstar f) \<longrightarrow> g "
using assms 
by (simp add: SChopstarMore_inductr_var int_iffD1)

lemma WPowerstar_inductr_var_eq2: 
assumes "\<turnstile> ( g ; f) = g "
 shows  "\<turnstile> g;(wpowerstar f) = g "
using assms 
by (metis Prop11 RightChopImpChop WPowerstar_ext WPowerstar_inductr_var_eq inteq_reflection)

lemma SChopstar_inductr_var_eq2: 
assumes "\<turnstile> ( g \<frown> (f \<and> finite)) = g "
 shows  "\<turnstile> g\<frown>(schopstar f) = g "
using assms 
by (metis Prop11 RightSChopImpSChop SChopstar_ext SChopstar_inductr_finite_var inteq_reflection)

lemma SChopstarMore_inductr_var_eq2: 
assumes "\<turnstile> ( g \<frown> ((f \<and> more) \<and> finite)) = g "
 shows  "\<turnstile> g\<frown>(schopstar f) = g "
using assms 
SChopstar_inductr_var_eq2[of g "LIFT f \<and> more"] 
SChopstar_and_more[of f] 
by (metis inteq_reflection)

lemma WPowerstar_bubble_sort:
assumes "\<turnstile> g ; f \<longrightarrow> f ; g "
shows "\<turnstile> (wpowerstar (f \<or> g)) = (wpowerstar f) ; (wpowerstar g) " 
using assms
using WPowerstar_church_rosser WPowerstar_sim4 by blast

lemma SChopstar_bubble_sort:
assumes "\<turnstile> g \<frown> f \<longrightarrow> f \<frown> g "
shows "\<turnstile> (schopstar (f \<or> g)) = (schopstar f) \<frown> (schopstar g) " 
using assms by (simp add: SChopstar_church_rosser SChopstar_sim4)


lemma WPowerstar_indepence1: 
assumes "\<turnstile> f ; g = #False"
shows   "\<turnstile> (wpowerstar f);g = g "
using assms 
by (metis EmptyOrChopEqv WPowerstar_induct_lvar_eq2 WPowerstar_star2 assms int_eq int_simps(25))

lemma SChopRightFalse: 
 "\<turnstile> f\<frown>#False = #False " 
by (simp add: intI itl_defs)


lemma SChopstar_indepence1: 
assumes "\<turnstile> f \<frown> g = #False"
shows   "\<turnstile> (schopstar f)\<frown>g = g "
proof -
 have 0 : "\<turnstile> (schopstar f) = (empty \<or> ( schopstar f) \<frown> (f \<and> finite)) "
   by (meson Prop11 SChopstar_unfoldr_eq) 
  have 01: "\<turnstile> ( schopstar f) \<frown> (f \<and> finite) = ((schopstar f)\<frown>f \<and> finite) "
  by (meson FiniteImportSChopRight Prop04 lift_and_com) 
    
 have 1: "\<turnstile> (empty \<or> ( schopstar f) \<frown> (f \<and> finite))\<frown>g = (empty\<frown>g \<or> ((schopstar f) \<frown> f)\<frown>(g)) " 
   using 01  unfolding schop_d_def
   by (metis "0" EmptyImpFinite OrChopEqv Prop10 SChopstar_finite inteq_reflection schop_d_def) 
 have 11: "\<turnstile>   ((schopstar f) \<frown> f)\<frown>(g) =  ((schopstar f))\<frown>(f\<frown>g) "
   by (meson Prop11 SChopAssoc) 
 have 12: "\<turnstile>  empty\<frown>g = g "
   by (simp add: EmptySChop) 
 have 2: "\<turnstile> (schopstar f)\<frown>(f \<frown>g) = (schopstar f)\<frown>(#False) "
   by (simp add: RightSChopEqvSChop assms)
 have 3: "\<turnstile> (schopstar f)\<frown>(#False) = #False "
   by (simp add: SChopRightFalse)
 show ?thesis
 by (metis "0" "1" "11" "12" "3" assms int_simps(25) inteq_reflection)  
qed


lemma WPowerstar_indepence2: 
assumes "\<turnstile> f ; g = #False"
shows   "\<turnstile> f ; (wpowerstar g) = f "
using assms WPowerstar_inductr_var_eq2[of f g] WPowerstar_star2[of g]
by (metis ChopEmpty RightChopImpChop WPowerstar_imp_empty WPowerstar_inductr_var_equiv 
    int_eq int_iffI int_simps(11))


lemma SChopstar_indepence2: 
assumes "\<turnstile> f \<frown> g = #False"
shows   "\<turnstile> f \<frown> (schopstar g) = (f \<and> finite) "
proof -
 have 1: "\<turnstile> f \<frown> (schopstar g) = ((f \<and> finite) \<or> (f\<frown>g)\<frown>(schopstar g)) "
   by (metis ChopEmpty FPowerstarEqv FPowerstar_more_absorb SChopAssoc SChopOrEqv int_eq 
       schop_d_def schopstar_d_def)
 have 2: "\<turnstile> (f\<frown>g)\<frown>(schopstar g) = #False "
   by (metis AndInfChopEqvAndInf assms int_eq int_simps(19) schop_d_def)
 show ?thesis 
 by (metis "1" "2" int_simps(25) inteq_reflection)
qed

lemma WPowerstar_lazy_comm:
"(\<turnstile> g;f \<longrightarrow> f;(wpowerstar (f \<or> g)) \<or> g) = 
 (\<turnstile> g ;(wpowerstar f) \<longrightarrow> f ; (wpowerstar (f \<or> g)) \<or> g) "
proof 
 assume 1: "\<turnstile> g;f \<longrightarrow> f;(wpowerstar (f \<or> g)) \<or> g"
 show " \<turnstile> g ;(wpowerstar f) \<longrightarrow> f ; (wpowerstar (f \<or> g)) \<or> g"
  proof -
   have 2: "\<turnstile> (f;(wpowerstar (f \<or> g)) \<or> g);f =  
              ((f;(wpowerstar (f \<or> g)));f  \<or> g;f) "
    by (simp add: OrChopEqv)
   have 3: "\<turnstile> ((f;(wpowerstar (f \<or> g)));f  \<or> g;f) \<longrightarrow>
              ((f;(wpowerstar (f \<or> g)));f  \<or> (f;(wpowerstar (f \<or> g))) \<or> g) "
       using "1"  by fastforce
   have 4: "\<turnstile> (f;(wpowerstar (f \<or> g)));f \<longrightarrow> (f;(wpowerstar (f \<or> g))) "
      by (metis ChopAssoc RightChopImpChop WPowerstar_subdist_var_1 WPowerstar_trans_eq inteq_reflection)
   have 5: "\<turnstile> ((f;(wpowerstar (f \<or> g)));f  \<or> (f;(wpowerstar (f \<or> g))) \<or> g) \<longrightarrow>
              ((f;(wpowerstar (f \<or> g))) \<or> g) "
     using 4 by fastforce
   have 6: "\<turnstile> g \<or> ((f;(wpowerstar (f \<or> g))) \<or> g) \<longrightarrow> ((f;(wpowerstar (f \<or> g))) \<or> g)"
        by fastforce
   show ?thesis using WPowerstarInductR[of g "LIFT (f;wpowerstar (f \<or> g) \<or> g)" f]
     using "2" "3" "5" by fastforce 
 qed
next
 assume 7: "\<turnstile> g ;(wpowerstar f) \<longrightarrow> f ; (wpowerstar (f \<or> g)) \<or> g"
 show " \<turnstile> g;f \<longrightarrow> f;(wpowerstar (f \<or> g)) \<or> g"
 proof -
 have 80: "\<turnstile> f;(wpowerstar f) = (f \<or> f;(wpowerstar f))"
    by (meson ChopEmpty Prop02 Prop05 Prop11 RightChopImpChop WPowerstar_imp_empty lift_imp_trans) 
  have 8: "\<turnstile> (wpowerstar f) = (empty \<or> f \<or> f;(wpowerstar f)) "
    by (meson "80" Prop06 WPowerstarEqv)
  have 9: "\<turnstile> g ;(wpowerstar f) = (g;empty \<or> g;f \<or> g;(f;(wpowerstar f))) "
    by (metis "8" ChopOrEqv int_eq)
  show ?thesis 
   using 7 9 by fastforce
 qed
qed  

lemma SChopstar_lazy_comm:
"(\<turnstile> g\<frown>(f \<and> finite) \<longrightarrow> f\<frown>(schopstar (f \<or> g)) \<or> g) = 
 (\<turnstile> g \<frown>(schopstar f) \<longrightarrow> f \<frown> (schopstar (f \<or> g)) \<or> g) "
proof 
 assume 1: "\<turnstile> g\<frown>(f \<and> finite) \<longrightarrow> f\<frown>(schopstar (f \<or> g)) \<or> g"
 show " \<turnstile> g \<frown>(schopstar f) \<longrightarrow> f \<frown> (schopstar (f \<or> g)) \<or> g"
  proof -
   have 2: "\<turnstile> (f\<frown>(schopstar (f \<or> g)) \<or> g)\<frown>(f \<and> finite) =  
              ((f\<frown>(schopstar (f \<or> g)))\<frown>(f \<and> finite)  \<or> g\<frown>(f \<and> finite)) "
    by (simp add: OrSChopEqv)
   have 3: "\<turnstile> ((f\<frown>(schopstar (f \<or> g)))\<frown>(f \<and> finite)  \<or> g\<frown>(f\<and> finite)) \<longrightarrow>
              ((f\<frown>(schopstar (f \<or> g)))\<frown>(f \<and> finite)  \<or> (f\<frown>(schopstar (f \<or> g))) \<or> g) "
       using "1" 
       using SChopAndA by fastforce
   have 4: "\<turnstile> (f\<frown>(schopstar (f \<or> g)))\<frown>(f \<and> finite) \<longrightarrow> (f\<frown>(schopstar (f \<or> g))) "
      using  SChopstar_subdist_var_1 
      by (metis Prop11 RightSChopImpSChop SChopAssoc SChop_SChopstar_Closure SChopstardef lift_imp_trans)
   have 5: "\<turnstile> ((f\<frown>(schopstar (f \<or> g)))\<frown>(f \<and> finite)  \<or> (f\<frown>(schopstar (f \<or> g))) \<or> g) \<longrightarrow>
              ((f\<frown>(schopstar (f \<or> g))) \<or> g) "
     using 4 by fastforce
   have 6: "\<turnstile> g \<or> ((f\<frown>(schopstar (f \<or> g))) \<or> g) \<longrightarrow> ((f\<frown>(schopstar (f \<or> g))) \<or> g)"
        by fastforce
   show ?thesis
   by (metis (mono_tags, lifting) "3" "5" OrSChopEqv Prop03 SChopstar_inductr_var_equiv int_eq lift_imp_trans) 
 qed
next
 assume 7: "\<turnstile> g \<frown>(schopstar f) \<longrightarrow> f \<frown> (schopstar (f \<or> g)) \<or> g"
 show " \<turnstile> g\<frown>(f \<and> finite) \<longrightarrow> f\<frown>(schopstar (f \<or> g)) \<or> g"
 proof -
  have 8: "\<turnstile> (schopstar f) = (empty \<or> (f \<and> finite) \<or> f\<frown>(schopstar f)) "
  by (meson AndSChopA Prop02 Prop05 Prop08 SChopstarMore_unfoldl_eq SChopstar_1L SChopstar_ext 
      SChopstar_imp_empty int_iffD2 int_iffI)
  have 9: "\<turnstile> g \<frown>(schopstar f) = (g\<frown>empty \<or> g\<frown>(f \<and> finite) \<or> g\<frown>(f\<frown>(schopstar f))) "
    by (metis "8" SChopOrEqv inteq_reflection) 
  show ?thesis 
   using 7 9 by fastforce
 qed
qed  

subsection \<open>WPowerstar\<close>

lemma WPowerstar_and_inf: 
 "\<turnstile> (wpowerstar (f \<and> inf)) = (empty \<or> (f \<and> inf)) " 
by (simp add: AndInfChopEqvAndInf WPowerstar_boffa)

lemma WPowerstar_chop_and_finite_inf: 
 "\<turnstile> (wpowerstar f) = (wpowerstar (f \<and> finite));(wpowerstar (f \<and> inf)) " 
by (metis AndInfChopEqvAndInf OrFiniteInf WPowerstar_denest_var inteq_reflection)


lemma WPowerstar_SChopstar: 
 "\<turnstile> (wpowerstar f) = (schopstar f);(empty \<or> (f \<and> inf)) " 
by (metis SChopstar_WPowerstar WPowerstar_and_inf WPowerstar_chop_and_finite_inf int_eq)


lemma WPowerstar_empty: 
 "\<turnstile> (wpowerstar empty) = empty " 
by (simp add: WPowerstar_subid)

lemma  WPowerstar_more: 
 "\<turnstile> (wpowerstar more)  "
by (metis ChopImpDi DiMoreEqvMore TrueW WPowerstar_boffa_var empty_d_def int_simps(29) inteq_reflection)

lemma WPowerstar_false:
 "\<turnstile> (wpowerstar #False) = empty "
by (simp add: WPowerstar_subid)

lemma WPowerstar_true:
 "\<turnstile> (wpowerstar #True)  "
by (metis MP TrueW WPowerstar_ext)

lemma WPowerstar_inf: 
 "\<turnstile> (wpowerstar inf) = (empty \<or> inf)" 
by (simp add: InfChopInfEqvInf WPowerstar_boffa)

lemma WPowerstar_finite:  
 "\<turnstile> (wpowerstar finite) = finite " 
by (meson EmptyImpFinite FiniteChopFiniteEqvFinite Prop02 Prop11 WPowerstar_ext WPowerstar_induct_lvar_empty)

lemma WPowerstar_imp_finite:
assumes "\<turnstile> f \<longrightarrow> finite"
shows   "\<turnstile> wpowerstar f \<longrightarrow> finite " 
using assms
by (metis WPowerstar_finite WPowerstar_iso inteq_reflection)


lemma WPowerstar_and_empty: 
"\<turnstile> (wpowerstar (f \<and> empty)) = empty " 
by (metis AndEmptyChopAndEmptyEqvAndEmpty Prop11 Prop12 WPowerstar_subid inteq_reflection)


lemma WPowerstarIntro:
 assumes "\<turnstile>   f \<longrightarrow> (g \<and> more); f "
 shows   "\<turnstile> f   \<and> finite \<longrightarrow> wpowerstar g "
proof -
 have 14: "\<turnstile>  wpowerstar g = (empty \<or> (g \<and> more);(wpowerstar g)) "
    using WPowerstar_more_absorb[of g] WPowerstarEqv[of "LIFT (g \<and> more)" ]
       by (metis int_eq)
 have 15: "\<turnstile> ( \<not>(wpowerstar g) ) = (more \<and> \<not>((g \<and> more);wpowerstar g)) " 
     using 14 unfolding empty_d_def by fastforce
 have 20: "\<turnstile> f \<and> \<not>(wpowerstar g) \<longrightarrow> (g \<and> more);f \<and> \<not>((g \<and> more);wpowerstar g) " 
     using assms 15 by fastforce
 have 21: "\<turnstile> g \<and>  more  \<longrightarrow>  more  " 
   by auto
 from 20 21  show ?thesis using   ChopContraB[of f "LIFT wpowerstar g" "LIFT (g \<and> more) " ]
    by blast
qed
    
lemma WPowerstarIntroMore:
 assumes "\<turnstile>   f \<and> more\<longrightarrow> (g \<and> more); f "
 shows   "\<turnstile> f   \<and> finite \<longrightarrow> wpowerstar g "
proof -
 have 14: "\<turnstile>  wpowerstar g = (empty \<or> (g \<and> more);(wpowerstar g)) "
    using WPowerstar_more_absorb[of g] WPowerstarEqv[of "LIFT (g \<and> more)" ]
       by (metis int_eq)
 have 15: "\<turnstile> ( \<not>(wpowerstar g) ) = (more \<and> \<not>((g \<and> more);wpowerstar g)) " 
     using 14 unfolding empty_d_def by fastforce
 have 20: "\<turnstile> f \<and> \<not>(wpowerstar g) \<longrightarrow> (g \<and> more);f \<and> \<not>((g \<and> more);wpowerstar g) " 
     using assms 15 by fastforce
 have 21: "\<turnstile> g \<and>  more  \<longrightarrow>  more  " 
   by auto
 from 20 21  show ?thesis using   ChopContraB[of f "LIFT wpowerstar g" "LIFT (g \<and> more) " ]
    by blast
qed


lemma Chopstar_WPowerstar: 
 "\<turnstile> chopstar f = wpowerstar (f \<and> more) "
 by (metis FPowerstar_WPowerstar WPowerstar_and_inf WPowerstar_chop_and_finite_inf chopstar_d_def 
     inteq_reflection powerstar_d_def) 

lemma Chopstar_FPowerstar: 
 "\<turnstile> chopstar f = (fpowerstar f);(empty \<or> (f \<and> inf)) " 
by (metis Chopstar_WPowerstar FPowerstar_WPowerstar WPowerstar_and_inf WPowerstar_chop_and_finite_inf
    WPowerstar_more_absorb inteq_reflection)

lemma SChopstar_WPowerstar_more: 
 "\<turnstile> schopstar f = wpowerstar ((f \<and> more) \<and> finite) " 
by (simp add: FPowerstar_WPowerstar schopstar_d_def)


lemma SChopstar_FPowerstar: 
 "\<turnstile> schopstar f = (fpowerstar f) "
by (simp add: FPowerstar_more_absorb schopstar_d_def) 

lemma WPowerstar_skip_finite: 
 "\<turnstile> (wpowerstar skip ) = finite "
 by (metis EmptyImpFinite EmptyNextInducta Prop02 SkipChopFiniteImpFinite WPowerstar_1L 
     WPowerstar_imp_empty WPowerstar_induct_lvar_empty int_iffI next_d_def) 
  
lemma SPSEqvEmptyOrSChopSPS:
 "\<turnstile> fpowerstar f = (empty \<or> f\<frown>fpowerstar f)"
by (simp add: FPowerstarEqv schop_d_def)


lemma ChopstarEqvPowerstar: 
 "\<turnstile> f\<^sup>\<star> = powerstar f"
by (metis Chopstar_FPowerstar  powerstar_d_def)

lemma ChopplusCommute:
 "\<turnstile> f;f\<^sup>\<star> = f\<^sup>\<star>;f"
by (metis Chopstar_WPowerstar WPowerstar_more_absorb WPowerstar_slide_var int_eq)

lemma SChopplusCommute:
 "\<turnstile> f\<frown>schopstar f = schopstar f\<frown>(f \<and> finite)"
 by (meson Prop04 SChopstar_sim3 int_simps(27))




lemma CSEqvOrChopCSB:
 "\<turnstile>   f\<^sup>\<star> =  (empty  \<or>  (f\<^sup>\<star>;f)) " 
by (metis ChopplusCommute ChopstarEqvPowerstar PowerstarEqvSem intI inteq_reflection)

lemma SCSEqvOrChopSCSB:
 "\<turnstile>   schopstar f =  (empty  \<or>  (schopstar f\<frown> (f \<and> finite))) "
by (metis SChopplusCommute SChopstar_FPowerstar SPSEqvEmptyOrSChopSPS inteq_reflection)

lemma CSChopCS:
 "\<turnstile> f\<^sup>\<star> ; f\<^sup>\<star> = f\<^sup>\<star>"
by (metis Chopstar_WPowerstar WPowerstar_trans_eq inteq_reflection)

lemma SCSSChopSCS:
 "\<turnstile> schopstar f \<frown> schopstar f = schopstar f"
by (metis SChopstar_imp_empty SChopstar_invol SChopstar_sup_id_star1 inteq_reflection)


lemma CSCSEqvCS:
 "\<turnstile> (f\<^sup>\<star>)\<^sup>\<star>  = f\<^sup>\<star>"
by (metis (no_types, lifting) Chopstar_WPowerstar WPowerstar_invol WPowerstar_more_absorb int_eq)


lemma ChopPlusEqvOrChopChopPlus:
 "\<turnstile>   (f;f\<^sup>\<star>)  = (f \<or>  f; (f;f\<^sup>\<star>))  "
 by (metis CSEqvOrChopCSB ChopEmpty ChopOrEqv ChopplusCommute inteq_reflection)    

lemma SChopPlusEqvOrSChopSChopPlus:
 "\<turnstile>   (f\<frown>schopstar f)  = ((f \<and> finite) \<or>  f\<frown> (f\<frown>schopstar f))  "
 by (metis ChopEmpty SChopOrEqvRule SChopstar_WPowerstar WPowerstarEqv inteq_reflection schop_d_def)

lemma ChopPlusEqv:
 "\<turnstile> (f;f\<^sup>\<star>)  = (f \<or>  (f\<and>  more ); (f;f\<^sup>\<star>))"
using ChopplusCommute[of f] 
 by (metis ChopAssoc ChopstarEqv EmptyOrChopEqv int_eq)

lemma SChopPlusEqv:
 "\<turnstile> (f\<frown>schopstar f)  = ((f \<and> finite) \<or>  (f \<and>  more )\<frown> (f\<frown>schopstar f))" 
by (metis (no_types, opaque_lifting) ChopAssoc EmptyOrChopEqv Prop10 SChopstarEqv 
    SChopplusCommute SChopstar_finite inteq_reflection schop_d_def)


lemma CSIntro:
 assumes "\<turnstile>   f \<longrightarrow> (g \<and>  more ); f "
 shows   "\<turnstile> f  \<and> finite \<longrightarrow> g\<^sup>\<star> " 
by (metis Chopstar_WPowerstar WPowerstarIntro WPowerstar_more_absorb assms inteq_reflection)

lemma CSIntroMore:
 assumes "\<turnstile>   f \<and> more\<longrightarrow> (g \<and>  more ); f "
 shows   "\<turnstile> f  \<and> finite \<longrightarrow> g\<^sup>\<star> " 
by (metis Chopstar_WPowerstar WPowerstarIntroMore WPowerstar_more_absorb assms inteq_reflection)

lemma SCSIntro:
 assumes "\<turnstile>   f \<longrightarrow> (g \<and>  more )\<frown> f "
 shows   "\<turnstile> f  \<and> finite \<longrightarrow> schopstar g " 
proof -
 have 1: "\<turnstile>((g \<and> more) \<and> finite) = ((g \<and> finite) \<and> more)" 
    by auto
 show ?thesis  
 using assms SChopstar_WPowerstar[of g] WPowerstarIntro[of f "LIFT (g \<and> finite)"] 1
 by (metis inteq_reflection schop_d_def)
qed

lemma SCSIntroMore:
 assumes "\<turnstile>   f \<and> more \<longrightarrow> (g \<and>  more )\<frown> f "
 shows   "\<turnstile> f  \<and> finite \<longrightarrow> schopstar g " 
proof -
 have 1: "\<turnstile>((g \<and> more) \<and> finite) = ((g \<and> finite) \<and> more)" 
    by auto
 show ?thesis  
 using assms SChopstar_WPowerstar[of g] WPowerstarIntroMore[of f "LIFT (g \<and> finite)"] 1
 by (metis inteq_reflection schop_d_def)
qed

lemma CSElim:
 assumes "\<turnstile>    empty  \<longrightarrow> g " 
         "\<turnstile> (f \<and>  more ); g \<longrightarrow> g "
 shows   "\<turnstile> f\<^sup>\<star> \<longrightarrow> g "
using assms
by (metis Chopstar_WPowerstar Prop02 WPowerstar_induct_lvar_empty inteq_reflection)

lemma SCSElim:
 assumes "\<turnstile>    empty  \<longrightarrow> g " 
         "\<turnstile> (f \<and>  more )\<frown> g \<longrightarrow> g "
 shows   "\<turnstile> schopstar f \<longrightarrow> g "
using assms 
by (metis Prop02 SChopstar_WPowerstar_more WPowerstar_induct_lvar_empty inteq_reflection schop_d_def)

lemma CSElimWithoutMore: 
 assumes "\<turnstile> empty  \<longrightarrow> g "
         "\<turnstile> f; g \<longrightarrow> g "
 shows   "\<turnstile> f\<^sup>\<star>  \<longrightarrow> g "
using assms 
by (metis AndChopA CSElim Prop10 Prop12 int_eq)

lemma SCSElimWithoutMore: 
 assumes "\<turnstile>    empty  \<longrightarrow> g "
         "\<turnstile> f\<frown> g \<longrightarrow> g "
 shows   "\<turnstile> schopstar f  \<longrightarrow> g "
using assms 
by (metis Prop02 SChopstar_WPowerstar WPowerstar_induct_lvar_empty inteq_reflection schop_d_def)


lemma WPowerstarChopEqvChopOrRule: 
 assumes "\<turnstile>   f= ((wpowerstar g); h) "
 shows   "\<turnstile> f = ((g; f) \<or>  h) "
proof -
 have   1: "\<turnstile> f = ((wpowerstar g); h) " using assms by auto
 have   2: "\<turnstile> (wpowerstar g) =  (empty  \<or>  (g; (wpowerstar g))) " by (simp add: WPowerstarEqv) 
 hence  3: "\<turnstile> (wpowerstar g); h = (h \<or>  ((g; (wpowerstar g)); h)) " by (rule EmptyOrChopEqvRule)
 have   4: "\<turnstile> (g; (wpowerstar g)); h =  g; ((wpowerstar g); h)   " by (meson ChopAssoc Prop11)
 hence 41: "\<turnstile> (wpowerstar g); h = (h \<or> g; ((wpowerstar g); h))" using "3" by fastforce
 have   5: "\<turnstile> g; f = g; ((wpowerstar g); h) " using "1" by (rule RightChopEqvChop)
 hence  6: "\<turnstile> ((wpowerstar g); h) = (h \<or>  g; f)" using "41" by fastforce
 hence 61: "\<turnstile> ((wpowerstar g); h) = ((g; f) \<or> h) " by auto
 from 1 61 show ?thesis  by fastforce
qed

lemma CSChopEqvChopOrRule:
 assumes "\<turnstile>   f= (g\<^sup>\<star>; h) "
 shows   "\<turnstile> f = ((g; f) \<or>  h) "
using assms 
by (metis Chopstar_WPowerstar WPowerstarChopEqvChopOrRule WPowerstar_more_absorb int_eq)


lemma SCSChopEqvSChopOrRule:
 assumes "\<turnstile>   f= (schopstar g\<frown> h) "
 shows   "\<turnstile> f = ((g\<frown> f) \<or>  h) "
using assms 
by (metis (no_types, lifting) FPowerstar_WPowerstar FPowerstar_more_absorb Prop10 SChopstar_finite 
    WPowerstarChopEqvChopOrRule int_eq schop_d_def schopstar_d_def)

lemma WPowerstarChopIntroRule:
 assumes "\<turnstile> f \<and> \<not>  h \<longrightarrow> g; f "
         "\<turnstile> g \<longrightarrow>  more "
 shows   "\<turnstile> f \<and> finite \<longrightarrow> (wpowerstar g); h"
proof -
 have  1: "\<turnstile> f \<and> \<not>  h \<longrightarrow> g; f " 
   using assms by blast
 have  2: "\<turnstile> g \<longrightarrow>  more " 
   using assms by blast
 hence 3: "\<turnstile> g \<longrightarrow> g \<and>  more " 
   by auto
 hence 4: "\<turnstile> g; f \<longrightarrow> (g ); f "
    by auto 
 have  5: "\<turnstile> f \<longrightarrow> (g ); f \<or>  h " 
   using "1" "4" by fastforce
 have  6: "\<turnstile> wpowerstar g =  (empty  \<or>  g ; wpowerstar g) " 
   by (simp add: WPowerstarEqv)
 hence 7: "\<turnstile> (wpowerstar g); h = (h \<or>  (g; wpowerstar g); h) " 
   by (rule EmptyOrChopEqvRule)
 have  8: "\<turnstile> (g ; wpowerstar g); h = g ; (wpowerstar g; h) " 
   by (meson ChopAssoc Prop11)
 have  9: "\<turnstile> (wpowerstar g); h = (h \<or>  g ; (wpowerstar g; h)) " 
   using "7" "8" by fastforce
 have 10: "\<turnstile> f \<and> \<not> (wpowerstar g; h) \<longrightarrow> (g); f \<and> \<not> ((g); (wpowerstar g; h)) " 
   using "5" "9" by fastforce
 have 11: "\<turnstile> g \<and>  more  \<longrightarrow>  more " 
   by fastforce
 from 10 11 show ?thesis using ChopContraB using "2" by blast
qed


lemma CSChopIntroRule:
 assumes "\<turnstile> f \<and> \<not>  h \<longrightarrow> g; f "
         "\<turnstile> g \<longrightarrow>  more "
 shows   "\<turnstile> f \<and> finite \<longrightarrow> g\<^sup>\<star>; h"
using assms 
by (metis Chopstar_WPowerstar Prop10 WPowerstarChopIntroRule inteq_reflection)


lemma SCSChopIntroRule:
 assumes "\<turnstile>   f \<and> \<not>  h \<longrightarrow> g\<frown> f "
         "\<turnstile> g \<longrightarrow>  more "
 shows   "\<turnstile> f \<and> finite \<longrightarrow> schopstar g\<frown> h"
using assms SChopstar_WPowerstar[of g] WPowerstarChopIntroRule[of f h "LIFT (g \<and> finite)"]
unfolding schop_d_def 
by (metis Prop05 Prop07 Prop10 SChopstar_finite finite_d_def inteq_reflection)


lemma DiamondAndEmptyEqvAndEmpty:
 "\<turnstile> (\<diamond> f \<and> empty) = (f \<and> empty)"
 by (metis ChopAndEmptyEqvEmptyChopEmpty EmptyChop FiniteAndEmptyEqvEmpty int_eq sometimes_d_def)

lemma InitAndEmptyEqvAndEmpty:
 "\<turnstile> ((init w) \<and> empty) = (w \<and> empty) "
proof -
 have 1: "\<turnstile> ((init w) \<and> empty) = ((w \<and> empty);#True \<and> empty)" 
   by (metis init_d_def int_eq lift_and_com) 
 have 2: "\<turnstile> ((w \<and> empty);#True \<and> empty) = (w \<and> empty);(#True \<and> empty)" 
   by (meson AndChopA ChopAndA ChopAndEmptyEqvEmptyChopEmpty lift_imp_trans Prop11 Prop12)
 have 3: "\<turnstile> (w \<and> empty);(#True \<and> empty) = (w \<and> empty);empty" 
   using RightChopEqvChop by fastforce
 have 4: "\<turnstile> (w \<and> empty);empty = (w \<and> empty)" 
   using ChopEmpty by blast
 from 1 2 3 4 show ?thesis by fastforce
qed

lemma InitAndNotBoxInitImpNotEmpty:
 "\<turnstile> init w \<and> \<not>( \<box> (init w)) \<longrightarrow> \<not>  empty" 
proof -
 have 1: "\<turnstile> ((init w) \<and> empty) = (w \<and> empty) " 
   by (rule InitAndEmptyEqvAndEmpty)
 have 2: "\<turnstile> (\<not>( \<box> (init w)) \<and> empty) = (\<diamond> (\<not> (init w)) \<and> empty)" 
   by (auto simp: always_d_def)
 have 3: "\<turnstile> (\<diamond> (\<not> (init w)) \<and> empty) = (\<not> (init w) \<and> empty)" 
   by (simp add: DiamondAndEmptyEqvAndEmpty)
 have 4: "\<turnstile> (\<not> (init w)) = (init (\<not> w)) " 
   using Initprop(2) by blast
 have 5: "\<turnstile> (\<not> (init w) \<and> empty) = (\<not> w \<and> empty)" 
   using "4" InitAndEmptyEqvAndEmpty by (metis inteq_reflection)
 have 6: "\<turnstile> (\<not>( \<box> (init w)) \<and> empty) = (\<not> w \<and> empty) " 
   using "2" "3" "5" by fastforce
 have 7: "\<turnstile> \<not>(init w \<and> \<not>( \<box> (init w)) \<and> empty) " 
   using "1" "6" by fastforce  
 from 7 show ?thesis by auto
qed

lemma BoxImpTrueChopAndEmpty: 
 "\<turnstile> \<box> f \<longrightarrow> #True;(f \<and> empty) "
using BoxAndChopImport Finprop(3) by fastforce

lemma BoxInitAndMoreImpBoxInitAndMoreAndFinInit:
 "\<turnstile> \<box>( init w) \<and>  more  \<longrightarrow> (\<box> (init w) \<and>  more ) \<and>  fin ( init w) "
proof -
 have 1: "\<turnstile> fin ( init w) = #True ; (init w \<and> empty)" using FinEqvTrueChopAndEmpty by blast
 have 2: "\<turnstile> \<box>( init w) \<longrightarrow> #True;(init w \<and> empty) " by (rule BoxImpTrueChopAndEmpty)
 from 1 2 show ?thesis by fastforce
qed

lemma BoxImpTrueSChopAndEmpty: 
 "\<turnstile> \<box> f \<and> finite\<longrightarrow> #True\<frown>(f \<and> empty) "
by (metis BoxAndSChopImport DiamondEmptyEqvFinite TrueSChopEqvDiamond inteq_reflection)

lemma BoxInitAndMoreImpBoxInitAndMoreAndSFinInit:
 "\<turnstile> \<box>( init w) \<and>  more  \<and> finite \<longrightarrow> (\<box> (init w) \<and>  more ) \<and>  sfin ( init w) "
proof -
 have 1: "\<turnstile> sfin ( init w) = #True \<frown> (init w \<and> empty)" using SFinEqvTrueSChopAndEmpty by blast
 have 2: "\<turnstile> \<box>( init w) \<and> finite \<longrightarrow> #True\<frown>(init w \<and> empty) " by (rule BoxImpTrueSChopAndEmpty)
 from 1 2 show ?thesis by fastforce
qed

lemma CSImpBox:
 assumes "\<turnstile> f \<longrightarrow>  empty  \<or>  ((\<box> (init w) \<and>  more) \<and> finite ); f "
 shows   "\<turnstile> (init w \<and> f) \<and> finite \<longrightarrow> \<box> (init w) "
proof -
 have  1: "\<turnstile> f \<longrightarrow>  empty  \<or>  ((\<box>( init w) \<and>  more) \<and> finite ); f " 
   using assms by auto
 have  2: "\<turnstile> init w \<and> \<not>( \<box> (init w)) \<longrightarrow> \<not>  empty " 
   by (rule InitAndNotBoxInitImpNotEmpty)
 have  3: "\<turnstile> init w \<and> f \<and> \<not>( \<box> (init w)) \<longrightarrow> ((\<box> (init w) \<and>  more) \<and> finite ); f " 
   using "1" "2" by fastforce
 have  4: "\<turnstile> \<box> (init w) \<and>  more  \<longrightarrow> (\<box> (init w) \<and>  more ) \<and>  fin ( init w) " 
   by (rule BoxInitAndMoreImpBoxInitAndMoreAndFinInit)
 have 41: "\<turnstile> (\<box> (init w) \<and>  more) \<and> finite \<longrightarrow> 
             ((\<box> (init w) \<and>  more ) \<and> finite) \<and>  fin ( init w) "
   using "4" by auto
 hence 5: "\<turnstile> ((\<box>( init w) \<and>  more) \<and> finite ); f \<longrightarrow> 
              (((\<box> (init w) \<and>  more ) \<and> finite) \<and>  fin ( init w) ); f " 
   by (rule LeftChopImpChop)
 have  6: "\<turnstile> (((\<box> (init w) \<and>  more ) \<and> finite) \<and>  fin ( init w) ); f =
              ((\<box>( init w) \<and>  more) \<and> finite ); (init w \<and> f) " 
   using AndFinChopEqvStateAndChop by blast
 have  7: "\<turnstile> \<not>( \<box>( init w)) \<longrightarrow> (\<box> (init w)) yields (\<not>( \<box> (init w))) " 
   by (rule NotBoxStateImpBoxYieldsNotBox)
 have  8: "\<turnstile> (\<box>( init w)) yields (\<not> (\<box> (init w))) \<longrightarrow> 
             ((\<box> (init w) \<and>  more) \<and> finite ) yields (\<not>( \<box>( init w))) " 
   using AndYieldsA 
   by (metis AndMoreAndFiniteEqvAndFmore inteq_reflection)
 have  9: "\<turnstile> ((\<box>( init w) \<and>  more) \<and> finite ); (init w \<and> f) \<and> 
              ((\<box>( init w) \<and>  more) \<and> finite ) yields (\<not>( \<box> (init w)))
             \<longrightarrow>  
             ((\<box> (init w) \<and>  more) \<and> finite ); ((init w \<and> f) \<and> \<not> (\<box> (init w))) " 
   by (rule ChopAndYieldsImp)
 have 10: "\<turnstile> (init w \<and> f) \<and> \<not>( \<box> (init w)) \<longrightarrow>
             ((\<box>( init w) \<and>  more) \<and> finite ); ((init w \<and> f) \<and> \<not>( \<box> (init w))) " 
   using "3" "5" "6" "7" "8" "9" by fastforce
 have 11: "\<turnstile> ((\<box>( init w) \<and>  more) \<and> finite ); ((init w \<and> f) \<and> \<not>( \<box> (init w))) \<longrightarrow>
              more ; ((init w \<and> f) \<and> \<not>( \<box> (init w)) ) " 
   by (metis "41" LeftChopImpChop Prop12)
 have 12: "\<turnstile> (init w \<and> f) \<and> \<not>( \<box> (init w)) \<longrightarrow> 
             more ; ((init w \<and> f) \<and> \<not>( \<box> (init w)) ) " 
   using "10" "11" by fastforce
 from 12 show ?thesis using MoreChopContraFiniteB by blast
qed

lemma SCSImpBox:
 assumes "\<turnstile> f \<longrightarrow>  empty  \<or>  ((\<box> (init w) \<and>  more) )\<frown> f "
 shows   "\<turnstile> (init w \<and> f) \<and> finite \<longrightarrow> \<box> (init w) "
using assms by (simp add: CSImpBox schop_d_def)

lemma ChopstarInductR:
 assumes "\<turnstile> g \<or> h;f \<longrightarrow> h"
 shows   "\<turnstile> g;(chopstar f) \<longrightarrow> h"
by (metis Chopstar_WPowerstar WPowerstarInductR WPowerstar_more_absorb assms int_eq)


lemma BoxWPowerstarEqvBox:
 "\<turnstile>   (init w \<and> wpowerstar (\<box>( init w)) )  = \<box> (init w) "
proof -
 have 1: "\<turnstile> \<box>( init w);\<box>( init w) \<longrightarrow> \<box>( init w)"
   by (simp add: BoxStateChopBoxEqvBox int_iffD1)
 have 2: "\<turnstile> (init w \<and> empty) \<longrightarrow> \<box>( init w)"
   by (simp add: StateAndEmptyImpBoxState)
 have 3: "\<turnstile> (init w \<and> empty) \<or> \<box>( init w);\<box>( init w) \<longrightarrow> \<box>( init w) "
   using "1" "2" by fastforce
 have 4: "\<turnstile> (init w \<and> empty);wpowerstar (\<box>( init w)) \<longrightarrow> \<box> (init w) "
   using "3" WPowerstarInductR by blast
 have 5: "\<turnstile> init w \<and> wpowerstar (\<box>( init w)) \<longrightarrow> \<box> (init w) "
   using "4" StateAndEmptyChop by fastforce
 have 11: "\<turnstile> \<box> (init w) \<longrightarrow> (init w) " 
   using BoxElim by blast
 have 12: "\<turnstile> \<box>( init w) \<longrightarrow> wpowerstar (\<box> (init w)) " 
   by (simp add: WPowerstar_ext)
 have 13: "\<turnstile> \<box> (init w) \<longrightarrow> init w \<and> wpowerstar (\<box> (init w)) " 
   using "11" "12" by fastforce
 from 5 13 show ?thesis by fastforce
qed


lemma BoxCSEqvBox:
 "\<turnstile>   (init w \<and> (\<box>( init w))\<^sup>\<star>) = \<box> (init w) "
by (metis BoxWPowerstarEqvBox Chopstar_WPowerstar WPowerstar_more_absorb inteq_reflection)


lemma BoxSCSEqvBox:
 "\<turnstile>   (init w \<and> schopstar (\<box>( init w)) ) = (\<box> (init w) \<and> finite) "
proof -
 have 1: "\<turnstile> \<box>( init w)\<frown>(\<box>( init w) \<and> finite)\<longrightarrow> \<box>( init w) \<and> finite"
   by (metis BoxStateAndChopEqvChop FiniteChopFiniteEqvFinite int_iffD2 inteq_reflection schop_d_def)
 have 2: "\<turnstile> (init w \<and> empty) \<longrightarrow> \<box>( init w) \<and> finite"
   using EmptyImpFinite StateAndEmptyImpBoxState by fastforce
 have 3: "\<turnstile> (init w \<and> empty) \<or> \<box>( init w)\<frown>(\<box>( init w) \<and> finite) \<longrightarrow> \<box>( init w) \<and> finite"
   using "1" "2" by fastforce
 have 4: "\<turnstile> (init w \<and> empty)\<frown>schopstar (\<box>( init w)) \<longrightarrow> \<box> (init w) \<and> finite "
   using SChopstarInductR "3" 
   by (metis Prop12 SChopImpFinite SChopstar_finite SChopstar_finite_absorb inteq_reflection)
 have 5: "\<turnstile> init w \<and> schopstar (\<box>( init w)) \<longrightarrow> \<box> (init w) \<and> finite "
   using "4" StateAndEmptySChop by fastforce
 have 11: "\<turnstile> \<box> (init w) \<longrightarrow> (init w) " 
   using BoxElim by blast
 have 12: "\<turnstile> \<box>( init w) \<and> finite \<longrightarrow> schopstar (\<box> (init w)) " 
   by (metis SChopstar_WPowerstar WPowerstar_ext inteq_reflection)
 have 13: "\<turnstile> \<box> (init w) \<and> finite \<longrightarrow> init w \<and> schopstar (\<box> (init w)) " 
   using "11" "12" by fastforce
 from 5 13 show ?thesis by fastforce
qed

lemma WpowerstarAndMoreEqvAndMoreChop:
 " \<turnstile>  (wpowerstar f \<and>  more)  = (f \<and>  more ); wpowerstar f "
proof -
 have  1: "\<turnstile> ( empty  \<or>  (f \<and>  more ); wpowerstar f) \<and>  more \<longrightarrow> (f \<and>  more ); wpowerstar f " 
   by (auto simp: empty_d_def)
 have  2: "\<turnstile> wpowerstar f = (empty \<or> (f \<and> more);wpowerstar f)" 
   by (metis ChopstarEqv Chopstar_WPowerstar WPowerstar_more_absorb inteq_reflection)
 have  3: "\<turnstile> wpowerstar f \<and>  more  \<longrightarrow> (f \<and>  more ); wpowerstar f " 
   using "1" "2" by fastforce
 have  4: "\<turnstile> (f \<and>  more ); wpowerstar f \<longrightarrow> wpowerstar f " 
   using "2" by fastforce
 have  5: "\<turnstile> (f \<and>  more ) \<longrightarrow> more" 
   by auto
 hence 6: "\<turnstile> (f \<and>  more ); wpowerstar f \<longrightarrow>  more " 
   by (rule LeftChopImpMoreRule)
 have  7: "\<turnstile> (f \<and>  more ); wpowerstar f \<longrightarrow> wpowerstar f \<and>  more " 
   using "4" "6" by fastforce
 from 3 7 show ?thesis by fastforce
qed


lemma CSAndMoreEqvAndMoreChop:
 " \<turnstile>  (f\<^sup>\<star> \<and>  more)  = (f \<and>  more ); f\<^sup>\<star> "
by (metis Chopstar_WPowerstar WPowerstar_more_absorb WpowerstarAndMoreEqvAndMoreChop int_eq)


lemma SCSAndMoreEqvAndMoreSChop:
 " \<turnstile>  (schopstar f \<and>  more)  = (f \<and>  more )\<frown> schopstar f "
proof -
 have 1: "\<turnstile>((f \<and> more) \<and> finite) = ((f \<and> finite) \<and> more)" 
    by auto
 show ?thesis
 using SChopstar_WPowerstar[of f]  WPowerstar_more_absorb[of "LIFT (f \<and> finite)"] 
 WpowerstarAndMoreEqvAndMoreChop[of "LIFT (f \<and> finite)" ] 
 by (metis "1" int_eq schop_d_def)
qed

lemma SCSAndMoreEqvAndFMoreSChop:
 " \<turnstile>  (schopstar f \<and>  fmore)  = (f \<and>  more )\<frown> schopstar f "
by (metis AndMoreAndFiniteEqvAndFmore Prop10 SCSAndMoreEqvAndMoreSChop SChopImpFinite SChopstar_finite
    inteq_reflection)

lemma BoxStateAndWPowerstarEqvWPowerstar:
 "\<turnstile>   (\<box>( init w) \<and> wpowerstar f \<and> finite) = (init w \<and> wpowerstar (\<box>( init w) \<and> f) \<and> finite) "
proof -
 have  1: "\<turnstile> \<box> (init w) \<longrightarrow> init w " 
   using BoxElim by blast
 have  2: "\<turnstile> (wpowerstar f \<and>  more)  = (f \<and>  more ); wpowerstar f" 
    by (simp add: WpowerstarAndMoreEqvAndMoreChop)
 have  3: "\<turnstile> (\<box>( init w) \<and> ((f \<and>  more ); wpowerstar f)) =
            ((\<box> (init w) \<and> f \<and>  more ); (\<box> (init w) \<and> wpowerstar f)) " 
   by (rule BoxStateAndChopEqvChop)
 have  4: "\<turnstile> \<box> (init w) \<and> f \<and>  more  \<longrightarrow> (\<box> (init w) \<and> f) \<and>  more " 
   by auto
 hence 5: "\<turnstile> (\<box> (init w) \<and> f \<and>  more ); (\<box> (init w) \<and> wpowerstar f) \<longrightarrow>  
            ((\<box> (init w) \<and> f) \<and>  more ); (\<box> (init w) \<and> wpowerstar f) " 
   by (rule LeftChopImpChop)
 have  6: "\<turnstile> (\<box>( init w) \<and> wpowerstar f) \<and>  more  \<longrightarrow>
            ((\<box> (init w) \<and> f) \<and>  more ); (\<box> (init w) \<and> wpowerstar f) " 
   using "2" "3" "5" by fastforce
 hence 7: "\<turnstile> (\<box> (init w) \<and> wpowerstar f) \<and> finite \<longrightarrow> wpowerstar (\<box> (init w) \<and> f)" 
   using WPowerstarIntroMore[of "LIFT (\<box> (init w) \<and> wpowerstar f)"  "LIFT (\<box> (init w) \<and> f)" ]
      by blast
 have 71: "\<turnstile> init w \<and> \<box> (init w) \<and> wpowerstar f \<and> finite \<longrightarrow> init w \<and> wpowerstar (\<box> (init w) \<and> f) \<and> finite" 
   using "7" by fastforce
 have  8: "\<turnstile> \<box>( init w) \<and> wpowerstar f \<and> finite \<longrightarrow> init w \<and> wpowerstar (\<box> (init w) \<and> f) \<and> finite" 
   using "1" "71"  by fastforce
 have 11: "\<turnstile> wpowerstar (\<box> (init w) \<and> f) \<longrightarrow> wpowerstar (\<box> (init w)) " 
     by (meson Prop12 WPowerstar_iso int_iffD2 lift_and_com)
 have 12: "\<turnstile> (init w \<and> wpowerstar (\<box> (init w))) = \<box> (init w)" 
    by (simp add: BoxWPowerstarEqvBox)
 have 13: "\<turnstile> wpowerstar (\<box> (init w) \<and> f) \<longrightarrow> wpowerstar f "
    by (meson Prop12 WPowerstar_iso int_iffD2 lift_and_com) 
 have 14: "\<turnstile> init w \<and> wpowerstar (\<box> (init w) \<and> f) \<longrightarrow> init w \<and> wpowerstar (\<box> (init w)) \<and> wpowerstar f " 
   using "11" "13" by fastforce
 have 15: "\<turnstile> init w \<and> wpowerstar (\<box> (init w)) \<and> wpowerstar f \<longrightarrow> \<box> (init w) \<and> wpowerstar f " 
   using "12" by auto
 have 16: "\<turnstile> init w \<and> wpowerstar (\<box> (init w) \<and> f) \<longrightarrow> \<box> (init w) \<and> wpowerstar f " 
   using "14" "15" lift_imp_trans by blast
 from 8 16 show ?thesis by fastforce
qed

lemma BoxStateAndCSEqvCS:
 "\<turnstile>   (\<box>( init w) \<and> f\<^sup>\<star> \<and> finite) = (init w \<and> (\<box>( init w) \<and> f)\<^sup>\<star> \<and> finite) "
by (metis BoxStateAndWPowerstarEqvWPowerstar Chopstar_WPowerstar WPowerstar_more_absorb int_eq)


lemma BoxStateAndSCSEqvSCS:
 "\<turnstile>   (\<box>( init w) \<and> schopstar f) = (init w \<and> schopstar (\<box>( init w) \<and> f)) "
proof -
 have 1: "\<turnstile> (\<box> (init w) \<and> f \<and> finite) = ((\<box> (init w) \<and> f) \<and> finite)" 
    by auto
 show ?thesis 
 using BoxStateAndWPowerstarEqvWPowerstar[of w "LIFT (f \<and> finite)" ] 
 SChopstar_WPowerstar[of f] SChopstar_WPowerstar[of "LIFT (\<box>( init w) \<and> f)" ] 
 SChopstar_finite[of f] SChopstar_finite[of "LIFT (\<box>( init w) \<and> f)"]  
 by (metis "1" Prop10 inteq_reflection)
qed
 
lemma BaWPowerstarImpWPowerstar:
 "\<turnstile>    ba (f \<longrightarrow> g) \<and> finite \<longrightarrow> wpowerstar f \<longrightarrow> wpowerstar g "
proof -
 have   1: "\<turnstile> wpowerstar f =  (empty  \<or>  (f \<and>  more ); wpowerstar f) " 
   by (metis ChopstarEqv Chopstar_WPowerstar WPowerstar_more_absorb inteq_reflection)
 have   2: "\<turnstile> wpowerstar g =  (empty  \<or>  (g \<and>  more ); wpowerstar g) " 
  by (metis ChopstarEqv Chopstar_WPowerstar WPowerstar_more_absorb inteq_reflection)
 have  21: "\<turnstile> \<not>(wpowerstar g) =  (\<not>empty  \<and> \<not>( (g \<and>  more ); wpowerstar g))" 
   using "2" by fastforce
 hence 22: "\<turnstile> \<not>(wpowerstar g) =  (more  \<and> \<not>( (g \<and>  more ); wpowerstar g)) " 
    by (simp add: empty_d_def)
 have   3: "\<turnstile> wpowerstar f \<and> \<not> (wpowerstar g) \<longrightarrow>
            (empty  \<or> (f \<and>  more ); wpowerstar f) \<and> more \<and> \<not> ((g \<and>  more ); wpowerstar g) " 
   using "1" "22" by fastforce
 have  31: "\<turnstile> ((empty  \<or> (f \<and>  more ); wpowerstar f) \<and> more) = ((f \<and>  more ); wpowerstar f \<and> more)" 
   by (auto simp: empty_d_def)
 have  32: "\<turnstile> wpowerstar f \<and> \<not> (wpowerstar g) \<longrightarrow> (f \<and>  more ); wpowerstar f \<and> \<not> ((g \<and>  more ); wpowerstar g) " 
   using "3" "31" by fastforce
 have   4: "\<turnstile> (f \<longrightarrow> g) \<longrightarrow> (f \<and>  more  \<longrightarrow> g \<and>  more ) " 
   by auto
 hence  5: "\<turnstile>  ba (f \<longrightarrow> g) \<longrightarrow>  ba (f \<and>  more  \<longrightarrow> g \<and>  more ) " 
   by (rule BaImpBa)
 have   6: "\<turnstile>  ba (f \<and>  more  \<longrightarrow> g \<and>  more ) \<longrightarrow>
              (f \<and>  more ); wpowerstar f \<longrightarrow>  (g \<and>  more ); wpowerstar f " 
   by (rule BaLeftChopImpChop)
 have   7: "\<turnstile>  ba (f \<longrightarrow> g) \<and> (f \<and> more ); wpowerstar f \<longrightarrow> (g \<and>  more ); wpowerstar f " 
   using "5" "6" by fastforce
 have   8: "\<turnstile> (g \<and>  more ); wpowerstar f \<and> \<not> ((g \<and>  more ); wpowerstar g)
           \<longrightarrow>  (g \<and>  more ); (wpowerstar f \<and> \<not> (wpowerstar g)) " 
   by (rule ChopAndNotChopImp)
 have   9: "\<turnstile> (g \<and>  more ); (wpowerstar f \<and> \<not> (wpowerstar g)) \<longrightarrow>  more ; (wpowerstar f \<and> \<not> (wpowerstar g)) " 
   by (rule AndChopB)
 have  10: "\<turnstile>  ba (f \<longrightarrow> g) \<longrightarrow> more ; (wpowerstar f \<and> \<not> (wpowerstar g)) \<longrightarrow>  
              more ; ( ba (f \<longrightarrow> g) \<and> wpowerstar f \<and> \<not> (wpowerstar g)) " 
   by (rule BaChopImpChopBa)
 have  11: "\<turnstile>  ba (f \<longrightarrow> g) \<and> wpowerstar f \<and> \<not> (wpowerstar g) \<longrightarrow>  
              more ; ( ba (f \<longrightarrow> g) \<and> wpowerstar f \<and> \<not> (wpowerstar g)) " 
   using "32" "7" "8" "9" "10" by fastforce
 hence 12: "\<turnstile> finite \<longrightarrow> \<not> ( (ba (f \<longrightarrow> g)) \<and> (wpowerstar f) \<and> (\<not> (wpowerstar g))) " 
   using MoreChopLoopFiniteB by blast
 from 12 show ?thesis by (simp add: Valid_def)  
qed


lemma BaCSImpCS:
 "\<turnstile>    ba (f \<longrightarrow> g) \<and> finite \<longrightarrow> f\<^sup>\<star> \<longrightarrow> g\<^sup>\<star> "
by (metis BaWPowerstarImpWPowerstar Chopstar_WPowerstar WPowerstar_more_absorb int_eq)


lemma SDa_Da: 
"\<turnstile> sda f = da (f \<and> finite) " 
unfolding sda_d_def da_d_def schop_d_def 
by simp

lemma SBa_Ba:
 "\<turnstile> sba f = ba (finite \<longrightarrow> f) " 
proof - 
 have 1: "\<turnstile> (\<not> (finite \<longrightarrow> f)) = ( finite \<and> \<not> f ) " 
   by auto
 show ?thesis 
 unfolding sba_d_def ba_d_def sda_d_def da_d_def schop_d_def
  by (metis "1" AndChopCommute int_eq int_simps(17))
qed 


lemma SBaSCSImpSCS:
 "\<turnstile>    sba (f \<longrightarrow> g) \<longrightarrow> schopstar f \<longrightarrow> schopstar g "
proof -
 have 1: "\<turnstile> ba (finite \<longrightarrow> f \<longrightarrow> g) \<longrightarrow> ba (f \<and> finite \<longrightarrow> g \<and> finite)"
   by (simp add: BaImpBa intI) 
 have 2: "\<turnstile> schopstar f \<longrightarrow> finite" 
    by (simp add: SChopstar_finite)
 show ?thesis 
 using  BaWPowerstarImpWPowerstar[of "LIFT (f \<and> finite)" "LIFT (g \<and> finite)"]
 SChopstar_WPowerstar[of f] SChopstar_WPowerstar[of g]
 SBa_Ba[of "LIFT (f \<longrightarrow> g)"]  1 2 by fastforce
qed


lemma BaWPowerstarEqvWPowerstar:
 "\<turnstile>    ba (f = g) \<and> finite \<longrightarrow> (wpowerstar f = wpowerstar g)"
proof -
 have 0: "\<turnstile> (f = g) = ( (f \<longrightarrow> g) \<and>  (g \<longrightarrow> f))"  
   by fastforce
 have 1: "\<turnstile> ba (f = g) = (ba (f \<longrightarrow> g) \<and> ba (g \<longrightarrow> f))  " 
   by (metis "0" BaAndEqv int_eq) 
 have 2: "\<turnstile> ba (f \<longrightarrow> g) \<and> finite \<longrightarrow> (wpowerstar f \<longrightarrow> wpowerstar g)  " 
   by (rule BaWPowerstarImpWPowerstar )
 have 3: "\<turnstile> ba (g \<longrightarrow> f) \<and> finite \<longrightarrow> (wpowerstar g \<longrightarrow> wpowerstar f) " 
   by (rule BaWPowerstarImpWPowerstar ) 
 have 4: "\<turnstile> ba (f = g) \<and> finite \<longrightarrow> (wpowerstar f \<longrightarrow> wpowerstar g) \<and>  (wpowerstar g \<longrightarrow> wpowerstar f) " 
   using "1" "2" "3" by fastforce
 have 5: "\<turnstile> ((wpowerstar f \<longrightarrow> wpowerstar g) \<and> (wpowerstar g \<longrightarrow> wpowerstar f)) = (wpowerstar f = wpowerstar g) " 
   by auto
 from 4 5 show ?thesis by auto
qed

lemma BaCSEqvCS:
 "\<turnstile>    ba (f = g) \<and> finite \<longrightarrow> (f\<^sup>\<star> = g\<^sup>\<star>)" 
by (metis BaWPowerstarEqvWPowerstar Chopstar_WPowerstar WPowerstar_more_absorb int_eq)

lemma SBaSCSEqvSCS:
 "\<turnstile>    sba (f = g) \<longrightarrow> (schopstar f = schopstar g)"
proof -
 have 0: "\<turnstile> (f = g) = ( (f \<longrightarrow> g) \<and> (g \<longrightarrow> f))" 
   by fastforce
 have 1: "\<turnstile> sba (f = g) = (sba (f \<longrightarrow> g) \<and> sba (g \<longrightarrow> f))  " 
   by (metis "0" SBaAndEqv int_eq) 
 show ?thesis using  SBaSCSImpSCS[of f g] SBaSCSImpSCS[of g f]   
    1 by fastforce
qed

lemma BaAndWPowerstarImport:
 "\<turnstile>    ba  f \<and> wpowerstar g \<and> finite \<longrightarrow> wpowerstar (f \<and> g) "
proof -
 have  1: "\<turnstile> f \<longrightarrow> (g \<longrightarrow> f \<and> g) " by auto
 hence 2: "\<turnstile>  ba  f \<longrightarrow>  ba  (g \<longrightarrow> f \<and> g)" by (rule BaImpBa)
 have  3: "\<turnstile>  ba  (g \<longrightarrow> f \<and> g) \<and> finite \<longrightarrow> wpowerstar g \<longrightarrow> wpowerstar (f \<and> g) " 
   by (rule BaWPowerstarImpWPowerstar)
 from 2 3 show ?thesis by fastforce
qed

lemma BaAndCSImport:
 "\<turnstile>    ba  f \<and> g\<^sup>\<star> \<and> finite \<longrightarrow> (f \<and> g)\<^sup>\<star> "
by (metis BaAndWPowerstarImport Chopstar_WPowerstar WPowerstar_more_absorb int_eq)

lemma SBaAndSCSImport:
 "\<turnstile>    sba  f \<and> schopstar g \<longrightarrow> schopstar (f \<and> g) "
proof -
 have  1: "\<turnstile> f \<longrightarrow> (g \<longrightarrow> f \<and> g) " by auto
 hence 2: "\<turnstile>  sba  f \<longrightarrow>  sba  (g \<longrightarrow> f \<and> g)" by (rule SBaImpSBa)
 have  3: "\<turnstile>  sba  (g \<longrightarrow> f \<and> g) \<longrightarrow> schopstar g \<longrightarrow> schopstar (f \<and> g) " by (rule SBaSCSImpSCS)
 from 2 3 show ?thesis by fastforce
qed

subsection \<open>Len\<close>

lemma wpower_len: 
 "\<turnstile> wpower skip k = len k" 
by (simp add: len_d_def)

lemma wpowerstar_skip_finite:
 "\<turnstile> finite = wpowerstar skip " 
using WPowerstar_skip_finite by fastforce

lemma schopstar_skip_finite: 
 "\<turnstile> finite = schopstar skip" 
by (metis Prop10 SChopstar_WPowerstar WPowerstar_ext WPowerstar_skip_finite inteq_reflection)

lemma wpowerstar_skip_fmore: 
 "\<turnstile> fmore = skip;wpowerstar skip"
by (metis FmoreEqvSkipChopFinite inteq_reflection wpowerstar_skip_finite)

lemma schopstar_skip_fmore:
 "\<turnstile> fmore = skip\<frown>schopstar skip " 
by (metis NextSChopdef WPowerstar_skip_finite inteq_reflection next_d_def schopstar_skip_finite
     wpowerstar_skip_fmore)

lemma len_k_finite: 
 "\<turnstile> len k \<longrightarrow> finite "
proof (induct k)
case 0
then show ?case 
by (metis EmptyImpFinite len_d_def wpow_0)
next
case (Suc k)
then show ?case 
  proof -
   have 1: "\<turnstile> len (Suc k) = skip;len k" 
       by (simp add: len_d_def)
   have 2: "\<turnstile> skip \<longrightarrow> finite" 
     by (metis WPowerstar_ext WPowerstar_skip_finite inteq_reflection)
   show ?thesis 
   by (metis "1" "2" ChopImpChop FiniteChopFiniteEqvFinite Suc inteq_reflection)
  qed
qed

lemma len_k_schop: 
 "\<turnstile> len Suc k = len k \<frown> skip "
unfolding len_d_def unfolding schop_d_def 
by (metis Prop10 WPowerCommute int_eq len_k_finite wpow_Suc wpower_len)

lemma SkipChopAnd: 
 "\<turnstile> ((skip;f) \<and> (skip;g)) = skip;(f \<and> g) " 
proof -
 have 1: "\<turnstile> skip;(f \<and> g) \<longrightarrow> ((skip;f) \<and> (skip;g))" 
  by (simp add: ChopAndA ChopAndB Prop12)
 have 2: "\<turnstile> ((skip;f) \<and> (skip;g)) \<longrightarrow> skip;(f \<and> g) "
  by (metis NextAndEqvNextAndNext SkipChopEqvNext int_eq int_iffD2)
 show ?thesis 
 using "1" "2" int_iffI by blast
qed

lemma SkipSChopAnd: 
 "\<turnstile> ((skip\<frown>f) \<and> (skip\<frown>g)) = skip\<frown>(f \<and> g) " 
by (metis NextAndNextEqvNextRule NextSChopdef inteq_reflection lift_and_com)


lemma LenChopAnd: 
 "\<turnstile> (len k;f \<and> len k;g) = len k;(f \<and> g) " 
proof -
 have 2: "\<turnstile> len k;(f \<and> g) \<longrightarrow> (len k;f \<and> len k;g) " 
    by (simp add: ChopAndA ChopAndB Prop12)
 have 1: "\<turnstile>  (len k;f \<and> len k;g) \<longrightarrow> len k;(f \<and> g) "
   proof (induct k arbitrary: f g)
   case 0
   then show ?case 
   by (metis EmptyChop int_iffD2 inteq_reflection wpow_0 wpower_len)
   next
   case (Suc k)
   then show ?case 
      proof -
       have 1: "\<turnstile> len Suc k;f = len k;(skip;f) " 
         using WPowerCommute[of "LIFT skip" k] ChopAssoc[of "LIFT len k" "LIFT skip" f]
         by (metis inteq_reflection wpow_Suc wpower_len)
       have 2: "\<turnstile> len Suc k;g = len k;(skip;g)"
        using WPowerCommute[of "LIFT skip" k] ChopAssoc[of "LIFT len k" "LIFT skip" g]
         by (metis inteq_reflection wpow_Suc wpower_len)
       have 3: "\<turnstile> (len k;(skip;f) \<and> len k;(skip;g)) \<longrightarrow> len k;((skip;f) \<and> (skip;g)) " 
         by (simp add: Suc)    
       have 4: "\<turnstile> ((skip;f) \<and> (skip;g)) = skip;(f \<and> g) " 
          by (simp add: SkipChopAnd)
       have 5: "\<turnstile> len k;((skip;f) \<and> (skip;g)) = len (Suc k);(f \<and> g) " 
         using WPowerCommute[of "LIFT skip" k] ChopAssoc[of "LIFT len k" "LIFT skip" "LIFT (f \<and> g)" ]
         by (metis "4"  inteq_reflection wpow_Suc wpower_len)
       show ?thesis 
       by (metis "1" "2" "3" "5" int_eq)
     qed
   qed
  show ?thesis
  using "1" "2" int_iffI by blast
qed

lemma LenSChopAnd: 
 "\<turnstile> (len k\<frown>f \<and> len k\<frown>g) = len k\<frown>(f \<and> g) " 
proof -
 have 1: "\<turnstile> len k \<longrightarrow> finite "
  using len_k_finite by blast
 show ?thesis unfolding schop_d_def 
 by (metis "1" LenChopAnd Prop10 int_eq)
qed

lemma LenEqvLenChopLen:
 "\<turnstile> len(i+j) = len(i);len(j)"
proof 
 (induct i)
 case 0
 then show ?case 
 by (metis EmptyChop Prop11 add_cancel_left_left len_d_def wpow_0)
 next
 case (Suc i)
 then show ?case
 by (metis ChopAssoc add_Suc int_eq len_d_def wpow_Suc) 
qed 

lemma LenChopFalse: 
 "\<turnstile> len k ;#False \<longrightarrow> #False " 
by (metis AndInfEqvChopFalse ChopImpChop InfEqvNotFinite int_iffD1 int_simps(21) inteq_reflection 
    len_k_finite)

lemma LenSChopFalse: 
 "\<turnstile> len k \<frown>#False \<longrightarrow> #False " 
by (metis AndChopB InfEqvNotFinite TrueChopAndFiniteEqvAndFiniteChopFinite infinite_d_def
     int_eq int_simps(19) int_simps(22) schop_d_def)

lemma len_len_suc_not: 
 "\<turnstile> \<not>(len k \<and> len (Suc k);f)"
proof -
 have 1: "\<turnstile> len k = len k;empty " 
   by (meson ChopEmpty Prop11)
 have 2: "\<turnstile> len (Suc k);f = len k;(skip;f) " 
   using ChopAssoc[of  "LIFT len k" "LIFT skip" f] WPowerCommute[of "LIFT skip" k]
    by (metis inteq_reflection wpow_Suc wpower_len)
 have 3: "\<turnstile> (len k;empty \<and> len k;(skip;f)) = len k;(empty \<and> skip;f) " 
   by (simp add: LenChopAnd)
 have 4: "\<turnstile> (empty \<and> skip;f) \<longrightarrow> #False" 
   unfolding empty_d_def more_d_def next_d_def
   by (metis ChopAndB Prop01 int_simps(16) int_simps(25) int_simps(4) inteq_reflection) 
 have 5: "\<turnstile> (len k \<and> len (Suc k);f) \<longrightarrow> len k;#False " 
    by (metis "1" "2" "3" "4" RightChopImpChop inteq_reflection)
 have 6: "\<turnstile> len k ;#False \<longrightarrow> #False " 
   using LenChopFalse by blast
 show ?thesis 
 by (metis "5" "6" int_simps(14) inteq_reflection lift_imp_trans)
qed

lemma len_len_suc_not_schop: 
 "\<turnstile> \<not>(len k \<and> len (Suc k)\<frown>f)"
unfolding schop_d_def 
by (metis AndInfEqvChopFalse LenChopFalse Prop09 Prop10 int_eq int_simps(14) itl_def(8) len_len_suc_not)

lemma Finite_exist_len: 
 "\<turnstile> finite = (\<exists>k. len k) "
by (metis ExEqvRule WPowerstar_skip_finite int_eq wpower_len wpowerstar_d_def)

lemma LenNPlusOneB:
 "\<turnstile> len(n+1) = len(n);skip"
proof -
 have  1: "\<turnstile> len(n+1) = len(n);len(1)" by (rule LenEqvLenChopLen)
 have  2: "\<turnstile> len(1) = skip" by (simp add: ChopEmpty len_d_def)
 hence 3: "\<turnstile> len(n);len(1) = len(n);skip" using RightChopEqvChop by blast
 from 1 3 show ?thesis by fastforce
qed

lemma LenCommute:
 "\<turnstile> (skip;(len n)) = (len n);skip"
by (simp add: WPowerCommute len_d_def)

lemma NotFixedChop:
 "\<turnstile> (\<not>((g \<and> len(k));f)) = (\<not>(di(g \<and> len(k))) \<or> ((g \<and> len(k));(\<not>f)))"
by (auto simp add: itl_defs len_defs Valid_def min_def nlength_eq_enat_nfiniteD) 

subsection \<open>Properties of While\<close>


lemma SChopstar_Chopstar: 
 "\<turnstile> schopstar f = chopstar (f \<and> finite) " 
by (metis Chopstar_WPowerstar SChopstar_WPowerstar WPowerstar_more_absorb inteq_reflection)

lemma SWhile_While: 
 "\<turnstile> swhile  g  do  f  = while  g  do  (f \<and> finite) "
proof -
 have 1: "\<turnstile> ((g \<and> f) \<and> finite) = (g \<and> f \<and> finite)" 
   by auto
 have 2: "\<turnstile> chopstar ((g \<and> f) \<and> finite) = chopstar (g \<and> f \<and> finite)  "
   using 1 by (metis Chopstardef int_eq)
 show ?thesis
 unfolding swhile_d_def while_d_def 
 using SFinEQvFinAndFinite[of "LIFT \<not> g"] SChopstar_Chopstar[of "LIFT (g \<and> f)" ] 
 2 SChopstar_finite[of "LIFT (g \<and> f)"]   by fastforce
qed


lemma IfAndFiniteDist:
 "\<turnstile> (if\<^sub>i (init w) then (f;g)  else  empty \<and> finite) = 
    (if\<^sub>i (init w) then ((f \<and> finite);(g \<and> finite))  else  empty)"
proof -
 have 1: "\<turnstile> (if\<^sub>i (init w) then (f;g)  else  empty \<and> finite) =
            (( (init w \<and> (f;g)) \<or> (\<not>(init w) \<and> empty)) \<and> finite)"
   by (auto simp: ifthenelse_d_def)
 have 2: "\<turnstile> (( (init w \<and> (f;g)) \<or> (\<not>(init w) \<and> empty)) \<and> finite) =
            (( (init w \<and> (f;g) \<and> finite) \<or> (\<not>(init w) \<and> empty \<and> finite)))"
   by auto
 have 3: "\<turnstile> (init w \<and> (f;g) \<and> finite) = (init w \<and> (f \<and> finite);(g \<and> finite))"
   using ChopAndFiniteDist by fastforce
 have 4: "\<turnstile> (\<not>(init w) \<and> empty \<and> finite) = (\<not>(init w) \<and> empty)"
   using FiniteAndEmptyEqvEmpty by auto   
 have 5: "\<turnstile> (( (init w \<and> (f;g) \<and> finite) \<or> (\<not>(init w) \<and> empty \<and> finite))) =
            ((init w \<and> (f \<and> finite);(g \<and> finite)) \<or> (\<not>(init w) \<and> empty))"
   by (metis "2" "3" "4" inteq_reflection)
 have 6: "\<turnstile> ((init w \<and> (f \<and> finite);(g \<and> finite)) \<or> (\<not>(init w) \<and> empty)) =
            (if\<^sub>i (init w) then ((f \<and> finite);(g \<and> finite))  else  empty)"
   by (auto simp: ifthenelse_d_def)
 from 1 2 5 6 show ?thesis by (metis inteq_reflection)
qed

lemma WhileEqvIf:
 "\<turnstile> ((while  (init w)  do  f) \<and> finite) =  
     ( if\<^sub>i  (init w)  then  (f; ( while  (init w)  do  f))  else   empty  \<and> finite)"
proof -
 have  1: "\<turnstile>  (while  (init w)  do  f \<and> finite) = 
              ((((init w) \<and> f)\<^sup>\<star> \<and>  fin (\<not>  (init w))) \<and> finite) " 
   by (simp add: while_d_def)
 have  2: "\<turnstile> (init w \<and> f)\<^sup>\<star> =  (empty  \<or>  ((init w \<and> f); (init w \<and> f)\<^sup>\<star>)) " 
  by (metis CSEqvOrChopCSB ChopplusCommute int_eq)
 have 21: "\<turnstile> (((init w) \<and> f)\<^sup>\<star> \<and>  fin (\<not>  (init w)) \<and> finite) =
             ((empty  \<or>  ((init w \<and> f); (init w \<and> f)\<^sup>\<star>)) \<and>  fin (\<not>  (init w)) \<and> finite)  " 
   using "2" by fastforce
 have 22: "\<turnstile> ((empty  \<or>  ((init w \<and> f); (init w \<and> f)\<^sup>\<star>)) \<and>  fin (\<not>  (init w)) \<and> finite) = 
            (( empty \<and> fin (\<not>(init w)) \<and> finite) \<or> 
             ( ((init w \<and> f); (init w \<and> f)\<^sup>\<star>) \<and>  fin (\<not>  (init w)) \<and> finite))  " 
   by auto
 have 23: "\<turnstile> ((((init w) \<and> f)\<^sup>\<star> \<and>  fin (\<not>  (init w))) \<and> finite) =
             (( empty \<and> fin (\<not>(init w)) \<and> finite) \<or> 
             ( ((init w \<and> f); (init w \<and> f)\<^sup>\<star>) \<and>  fin (\<not>  (init w)) \<and> finite))"
   using "21" "22" by auto
 have  3: "\<turnstile>  (empty  \<and>  fin (\<not> ( init w))) = (\<not> ( init w) \<and>  empty) "
   by (metis FinAndEmpty Prop04 lift_and_com)
 hence 31: "\<turnstile>  (empty  \<and>  fin (\<not> ( init w)) \<and> finite) = (\<not> ( init w) \<and>  empty \<and> finite) "
   by auto
 have  32: "\<turnstile> (\<not> ( init w) \<and>  empty \<and> finite) = (\<not> ( init w) \<and>  empty)"
   using FiniteAndEmptyEqvEmpty by auto
 have  33: "\<turnstile> ( empty \<and> fin (\<not>(init w)) \<and> finite) = (\<not> ( init w) \<and>  empty)"
   using "31" "32" by fastforce
 have  34: "\<turnstile> (( empty \<and> fin (\<not>(init w)) \<and> finite) \<or> 
             ( ((init w \<and> f); (init w \<and> f)\<^sup>\<star>) \<and>  fin (\<not>  (init w)) \<and> finite)) =
             ((\<not> ( init w) \<and>  empty) \<or> 
             ( ((init w \<and> f); (init w \<and> f)\<^sup>\<star>) \<and>  fin (\<not>  (init w)) \<and> finite))"
   using "23" "33" by auto 
 have  4: "\<turnstile> (init w \<and> f); (init w \<and> f)\<^sup>\<star> = (init w \<and> (f; (init w \<and> f)\<^sup>\<star>)) " 
   by (rule StateAndChop)
 have 41: "\<turnstile> (((init w \<and> f); (init w \<and> f)\<^sup>\<star>) \<and>  fin (\<not>  (init w)) \<and> finite) =
             (init w \<and> (f; (init w \<and> f)\<^sup>\<star>) \<and>  fin (\<not>  (init w)) \<and> finite) " 
   using "4" by auto
 have 42: "\<turnstile> (init w \<and> ((f; (init w \<and> f)\<^sup>\<star>) \<and>  fin (\<not>  (init w)) \<and> finite)) = 
             (init w \<and> ((f; (init w \<and> f)\<^sup>\<star>) \<and>  fin   (init (\<not> w)) \<and> finite)) " 
   using Initprop(2) by (metis StateAndEmptyChop int_eq)
 have  5: "\<turnstile> ((f; ((init w \<and> f)\<^sup>\<star>)) \<and>  (fin  ( init (\<not> w))) \<and> finite)
             = ((f \<and> finite); ((init w \<and> f)\<^sup>\<star> \<and>  (fin  ( init (\<not> w))) \<and> finite)) " 
   using ChopAndFin by fastforce
 hence 49: "\<turnstile> (init w \<and> (f; (init w \<and> f)\<^sup>\<star>) \<and>  fin (\<not>  (init w)) \<and> finite ) =
              (init  w \<and> (f \<and> finite); ((init w \<and> f)\<^sup>\<star> \<and>  (fin  ( init (\<not> w))) \<and> finite))"
   using "42" by fastforce
 have 50: "\<turnstile>  (((init w \<and> f); (init w \<and> f)\<^sup>\<star>) \<and>  fin (\<not>  (init w)) \<and> finite) =
             (init  w \<and> (f \<and> finite); ((init w \<and> f)\<^sup>\<star> \<and>  (fin  ( init (\<not> w))) \<and> finite))"
   by (meson "41" "49" Prop04 lift_and_com)
 have 51: "\<turnstile> (init  w \<and> (f \<and> finite); ((init w \<and> f)\<^sup>\<star> \<and>  (fin  ( init (\<not> w))) \<and> finite )) = 
             (init  w \<and> (f \<and> finite); ((init w \<and> f)\<^sup>\<star> \<and>  (fin (\<not> ( init  w))) \<and> finite))  "
   by (metis (no_types) EmptyChop Initprop(2) inteq_reflection) 
 have 52: "\<turnstile> (((init w \<and> f); (init w \<and> f)\<^sup>\<star>) \<and>  fin (\<not>  (init w)) \<and> finite) =
           (init w \<and> ((f \<and> finite); ((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init  w)) \<and> finite)) ) " 
   using "50" "51" by fastforce
 have 53: "\<turnstile> ((\<not> ( init w) \<and>  empty) \<or> 
             ( ((init w \<and> f); (init w \<and> f)\<^sup>\<star>) \<and>  fin (\<not>  (init w)) \<and> finite)) =
             ((\<not> ( init w) \<and>  empty) \<or> 
             (init w \<and> ((f \<and> finite); ((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init  w)) \<and> finite)) ))"
   using "52" "34" by auto
 have  6: "\<turnstile> ((f \<and> finite); (((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w))) \<and> finite)) = 
             (f \<and> finite);  (while  (init w)  do  f \<and> finite)" 
   by (simp add: while_d_def)
 have 61: "\<turnstile> (init w \<and> ((f \<and> finite); (((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init  w))) \<and> finite))) =
            (init w \<and>  ((f \<and> finite);  (while  (init w)  do  f \<and> finite))) " 
   using "6" by auto
 have 62: "\<turnstile> ((\<not> ( init w) \<and>  empty) \<or> 
              (init w \<and> ((f \<and> finite); (((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init  w))) \<and> finite)) ))
            =((\<not> ( init w) \<and>  empty ) \<or>  
              (init w \<and> ((f \<and> finite);(while  (init w)  do  f \<and> finite)) ) ) " 
   using  "61" by fastforce
 have  7: "\<turnstile>  (while  (init w)  do  f   \<and> finite)   
           = (((\<not> ( init w) \<and>  empty ) \<or>  
              (init w \<and> (f ;  while  (init w)  do  f) \<and> finite))) " 
   proof -
    have "\<turnstile>(while (init w) do f \<and> finite) = 
        (\<not> init w \<and> empty \<or> (init w \<and> f);(init w \<and> f)\<^sup>\<star> \<and> fin (\<not> init w) \<and> finite)"
      by (metis "1" "23" "34" inteq_reflection)
    then show ?thesis
      by (metis "21" "22" "34" "52" ChopAndFiniteDist int_eq)
   qed  
 have 71: "\<turnstile> ((if\<^sub>i  (init w) then (f; ( while  (init w)  do  f))  else   empty)  \<and> finite)=
              (((\<not> ( init w) \<and>  empty ) \<or>  (init w \<and> (f;  while  (init w)  do  f)) \<and> finite )) " 
   using FiniteAndEmptyEqvEmpty by (auto simp: ifthenelse_d_def) 
 from 7 71 show ?thesis by fastforce
qed

lemma WPower_import_finite: 
 "\<turnstile> (wpower f k \<and> finite) = wpower (f \<and> finite) k " 
proof (induct k)
case 0
then show ?case 
using FiniteAndEmptyEqvEmpty by fastforce
next
case (Suc k)
then show ?case 
by (metis ChopAndFiniteDist inteq_reflection wpow_Suc)
qed


lemma WPowerstar_import_finite: 
 "\<turnstile> (wpowerstar f \<and> finite) = (wpowerstar (f \<and> finite)) " 
proof -
  have 1: "\<turnstile> (wpowerstar (f \<and> finite)) \<longrightarrow> (wpowerstar f \<and> finite) "
    by (metis OrFiniteInf Prop12 SChopstar_finite SChopstar_WPowerstar WPowerstar_subdist inteq_reflection)
  have 2: "\<turnstile> wpowerstar f \<and> finite \<longrightarrow>  (wpowerstar (f \<and> finite)) " 
   unfolding wpowerstar_d_def using WPower_import_finite[of f] by fastforce 
  show ?thesis 
  using "1" "2" int_iffI by blast
qed       

lemma Chopstar_import_finite: 
 "\<turnstile> (chopstar f \<and> finite) = (chopstar (f \<and> finite)) " 
using Chopstar_WPowerstar[of f]   Chopstar_WPowerstar[of "LIFT (f \<and> finite)" ]
  by (metis WPowerstar_import_finite WPowerstar_more_absorb int_eq)

lemma AndMoreSChopAndMoreEqvAndMoreSChop:
 "\<turnstile> ((f \<and> more)\<frown>g \<and> more) = (f \<and> more)\<frown>g"
by (meson AndSChopB MoreSChopImpMore Prop10 Prop11 lift_imp_trans)


lemma WPowerstar_chopstar: 
 "\<turnstile> (wpowerstar (f \<and> more)) = (chopstar f) "
by (meson Chopstar_WPowerstar Prop11)



lemma While_import_finite: 
 "\<turnstile> (while  g  do  f \<and> finite) = (while  g  do   (f \<and> finite)) " 
proof -
 have 1: "\<turnstile>(g \<and> f \<and> finite) = ((g \<and> f) \<and> finite) " 
    by auto
 have 2: "\<turnstile> chopstar (g \<and> f \<and> finite) = chopstar ((g \<and> f) \<and> finite)" 
   by (metis "1" Chopstardef int_eq)
 show ?thesis
 unfolding while_d_def 
 using Chopstar_import_finite[of "LIFT (g \<and> f)" ]  2 by fastforce
qed
 

lemma SWhileEqvIf:
 "\<turnstile> swhile  (init w)  do  f =  if\<^sub>i  (init w)  then  (f\<frown> ( swhile  (init w)  do  f))  else   empty "
proof -
 have 1: "\<turnstile> ((init w \<and> f) \<and> finite) = (init w \<and> f \<and> finite) " 
   by auto
 have 2: "\<turnstile> chopstar ((init w \<and> f) \<and> finite) = chopstar (init w \<and> f \<and> finite) "
    using 1 by (metis Chopstardef inteq_reflection)
 have 3: "\<turnstile> ((while  (init w)  do  f) \<and> finite) = while (init w) do (f \<and> finite) " 
   unfolding while_d_def 
   using Chopstar_import_finite[of "LIFT (init w \<and> f)" ] 2 by fastforce
 have 4: "\<turnstile>  while (init w) do (f \<and> finite) = swhile  (init w)  do  f "
    by (metis Prop04 SWhile_While lift_and_com swhile_d_def)
 have 5: "\<turnstile> ((init w \<and> (f \<and> finite);while (init w) do (f \<and> finite)   \<or> \<not> init w \<and> empty) \<and> finite) =
            ((init w \<and> (f \<and> finite);while (init w) do (f \<and> finite) \<and> finite   \<or> \<not> init w \<and> empty \<and> finite)) "
      by fastforce
 have 6: "\<turnstile> ((f \<and> finite);while (init w) do (f \<and> finite) \<and> finite) =
            (f \<and> finite);while (init w) do (f \<and> finite)" 
    by (metis "3" ChopAndFiniteDist Prop10 Prop12 int_eq int_iffD2) 
 have 7: "\<turnstile>  (\<not> init w \<and> empty \<and> finite) = (\<not> init w \<and> empty ) "
     using FiniteAndEmptyEqvEmpty by auto
 have 8: "\<turnstile> ( if\<^sub>i  (init w)  then  ((f \<and> finite); ( while  (init w)  do  (f \<and> finite)))  else   empty  \<and> finite) =
            ( if\<^sub>i  (init w)  then  (f\<frown> ( swhile  (init w)  do  f))  else   empty) " 
   unfolding ifthenelse_d_def schop_d_def
   by (metis "4" "5" "6" "7" inteq_reflection)
 have 9: "\<turnstile> ((while  (init w)  do  (f \<and> finite)) \<and> finite) =  
            ( if\<^sub>i  (init w)  then  ((f \<and> finite); ( while  (init w)  do  (f \<and> finite)))  else   empty  \<and> finite)"
   by (simp add: WhileEqvIf)
 show ?thesis 
 by (metis "3" "4" "8" "9" Prop10 Prop12 int_iffD2 inteq_reflection)
qed



lemma WhileChopEqvIf:
 "\<turnstile>   ( (while ( init w) do f) \<and> finite); g =  
       if\<^sub>i (init w) then ((f \<and> finite); ( ((while ( init w) do f) \<and> finite) ; g))  else  g "
proof -
 have  1: "\<turnstile>  (while  (init w)  do  f \<and> finite)=  
              (if\<^sub>i  (init w)  then  (f; ( while  (init w)  do  f))  else   empty  \<and> finite)" 
   by (rule WhileEqvIf)
 have 11: "\<turnstile> (if\<^sub>i  (init w)  then  (f; ( while  (init w)  do  f))  else   empty  \<and> finite) =
             (if\<^sub>i  (init w) then  ((f \<and> finite); ( while  (init w)  do  f \<and> finite))  else   empty)"
   using IfAndFiniteDist by fastforce
 have 12: "\<turnstile> (while  (init w)  do  f \<and> finite) =
             (if\<^sub>i  (init w) then  ((f \<and> finite); ( while  (init w)  do  f \<and> finite))  else   empty)"
   using "1" "11" by fastforce
 hence 2: "\<turnstile> ( (while  (init w)  do  f) \<and> finite); g = 
              (if\<^sub>i  (init w) 
               then (((f \<and> finite); ( while  (init w)  do  f \<and> finite));g)  
               else   (empty;g)) " 
   by (rule IfChopEqvRule)
 have  3: "\<turnstile>  empty ; g = g " 
   by (rule EmptyChop)
 have  4: "\<turnstile> (if\<^sub>i  (init w) 
               then (((f \<and> finite); ( while  (init w)  do  f \<and> finite));g)  
               else   (empty;g))  =
              (if\<^sub>i  (init w) 
               then (((f \<and> finite); ( while  (init w)  do  f \<and> finite));g)  
               else   g)" 
   using "3" using inteq_reflection by fastforce
 have  5: "\<turnstile> (((f \<and> finite); ( while  (init w)  do  f \<and> finite));g) =
             ((f \<and> finite);( ( while  (init w)  do  f \<and> finite);g))" 
  by (meson ChopAssoc Prop11)
 have  6: "\<turnstile> (if\<^sub>i  (init w) 
               then (((f \<and> finite); ( while  (init w)  do  f \<and> finite));g)  
               else   g) =
              (if\<^sub>i  (init w) 
               then ((f \<and> finite);( ( while  (init w)  do  f \<and> finite);g))  
               else   g)
             " 
   using "5" using inteq_reflection by fastforce
 show ?thesis
   using "2" "4" "6" by fastforce 
qed

lemma SWhileSChopEqvIf:
 "\<turnstile>   ( swhile ( init w) do f)\<frown>g =  if\<^sub>i (init w) then (f\<frown>((swhile ( init w) do f)\<frown> g))  else  g "
unfolding schop_d_def
by (metis (no_types, opaque_lifting) ChopEmpty EmptySChop SChopAssoc SWhile_While WhileChopEqvIf 
    inteq_reflection schop_d_def) 

lemma WhileChopEqvIfRule:
 assumes "\<turnstile>  f = ( while  (init w)  do  g \<and> finite); h "
 shows   "\<turnstile> f =  if\<^sub>i  (init w)  then  ((g \<and> finite); f)  else  h "
proof -
 have 1: "\<turnstile> f = ( while  (init w)  do  g \<and> finite); h " 
   using assms by auto
 have 2: "\<turnstile> ( while  (init w)  do  g \<and> finite); h =  
            if\<^sub>i  (init w)  then  ((g \<and> finite); (( while  (init w)  do  g \<and> finite); h))  else  h " 
   by (rule WhileChopEqvIf)
 have 3: "\<turnstile> ((g \<and> finite); f) = ((g \<and> finite); (( while  (init w)  do  g \<and> finite); h)) " 
   using "1" by (rule RightChopEqvChop)
 have 4: "\<turnstile> ((g \<and> finite); (( while  (init w)  do  g \<and> finite); h)) = ((g \<and> finite); f) " 
   using "3" by auto
 have 5: "\<turnstile> if\<^sub>i  (init w)  then  ((g \<and> finite); (( while  (init w)  do  g \<and> finite); h))  else  h =
           if\<^sub>i  (init w)  then  ((g \<and> finite); f)  else  h " 
   using "4" using inteq_reflection by fastforce
 from 1 2 5 show ?thesis by fastforce
qed

lemma SWhileSChopEqvIfRule:
 assumes "\<turnstile>  f = ( swhile  (init w)  do  g)\<frown> h "
 shows   "\<turnstile> f =  if\<^sub>i  (init w)  then  (g\<frown> f)  else  h "
using assms 
by (metis SWhileSChopEqvIf inteq_reflection)


lemma WhileImpFin:
 "\<turnstile>    while  (init w)  do  f \<longrightarrow>  fin (\<not> ( init w)) "
proof -
 have 1: "\<turnstile> (init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<longrightarrow>  fin (\<not> ( init w))" by auto
 from 1 show ?thesis by (simp add: while_d_def)
qed

lemma SWhileImpFin:
 "\<turnstile>    swhile  (init w)  do  f \<longrightarrow>  sfin (\<not> ( init w)) "
by (simp add: Prop01 Prop05 swhile_d_def)


lemma WhileEqvEmptyOrChopWhile:
 "\<turnstile>  (while (init w) do f \<and> finite)= 
     ((\<not> (init w) \<and> empty) \<or> (init w \<and> ((f \<and> more) \<and> finite);(while (init w) do f \<and> finite))) "
proof -
 have  1: "\<turnstile> (init w \<and> f)\<^sup>\<star>  =  (empty  \<or>  ((init w \<and> f) \<and> more ); (init w \<and> f)\<^sup>\<star>) " 
   by (rule ChopstarEqv)
 have  2: "\<turnstile> ((init w \<and> f) \<and>  more)  = (init w \<and> (f \<and> more)) " 
   by auto
 hence 3: "\<turnstile> ((init w \<and> f) \<and>  more); (init w \<and> f)\<^sup>\<star>  = (init w \<and> f \<and> more); (init w \<and> f)\<^sup>\<star> " 
   by (rule LeftChopEqvChop)
 have  4: "\<turnstile> (init w \<and> f)\<^sup>\<star>  =  (empty  \<or>  (init w \<and> f \<and>  more ); (init w \<and> f)\<^sup>\<star>) "  
   using "1" "3" by fastforce
 have 40: "\<turnstile> fin (\<not> ( init w)) = fin (\<not> w) " 
    by (metis FinEqvTrueChopAndEmpty InitAndEmptyEqvAndEmpty Initprop(2) inteq_reflection)
 have  5: "\<turnstile> ((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<and> finite)  =  
             (( empty  \<and>  fin  (\<not> (init w)) \<and> finite) \<or>  
             ((init w \<and> f \<and>  more ); (init w \<and> f)\<^sup>\<star> \<and> fin (\<not> ( init w)) \<and> finite)) " 
   using "1"  "4" by fastforce
 have  51: "\<turnstile>  ((empty  \<and>  fin  (\<not> ( init w))) \<and> finite) = ((\<not> ( init w) \<and>  empty) \<and> finite) " 
   by (metis FinAndEmpty inteq_reflection lift_and_com)
 have  52:"\<turnstile> ((\<not> ( init w) \<and>  empty) \<and> finite) = (\<not> ( init w) \<and>  empty)"
   using EmptyImpFinite by auto  
 have  6: "\<turnstile> ((empty  \<and>  fin  (\<not> ( init w))) \<and> finite) = (\<not> ( init w) \<and>  empty)"
   using "51" "52" by fastforce
 have 60: "\<turnstile> (( empty  \<and>  fin  (\<not> (init w)) \<and> finite) \<or>  
             ((init w \<and> f \<and>  more ); (init w \<and> f)\<^sup>\<star> \<and> fin (\<not> ( init w)) \<and> finite)) =
             ((\<not> ( init w) \<and>  empty) \<or>  
             ((init w \<and> f \<and>  more ); (init w \<and> f)\<^sup>\<star> \<and> fin (\<not> ( init w)) \<and> finite))" 
   using 6 by fastforce
 have 61: "\<turnstile>  ((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<and> finite)  = 
              ((\<not> ( init w) \<and>  empty) \<or>  
             ((init w \<and> f \<and>  more ); (init w \<and> f)\<^sup>\<star> \<and> fin (\<not> ( init w)) \<and> finite))"
             by (metis "5" "60" inteq_reflection)
 have  70: "\<turnstile> (init w \<and> f \<and>  more ); (init w \<and> f)\<^sup>\<star> = (init w \<and> (f \<and>  more ); (init w \<and> f)\<^sup>\<star>) " 
   by (rule StateAndChop)
 have  7: "\<turnstile> ((init w \<and> f \<and>  more ); (init w \<and> f)\<^sup>\<star> \<and> fin  ( init (\<not> w)) \<and> finite) =
             (init w \<and> (f \<and>  more ); (init w \<and> f)\<^sup>\<star> \<and> fin  ( init (\<not> w)) \<and> finite)"
   using "70" by auto 
 have  8: "\<turnstile> (( (f \<and>  more ); (init w \<and> f)\<^sup>\<star>) \<and>  fin  ( init (\<not> w)) \<and> finite)  =  
            (((f \<and>  more) \<and> finite); ((init w \<and> f)\<^sup>\<star> \<and>  fin (  init (\<not> w)) \<and> finite))  " 
   using ChopAndFin by fastforce
 have 81: "\<turnstile> fin (  init (\<not> w)) = fin (\<not> ( init w)) "
   by (meson FinEqvFin Initprop(2) Prop11) 
 have 82: "\<turnstile> ((f \<and>  more ); (init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<and> finite) =  
            (((f \<and>  more) \<and> finite); ((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<and> finite) ) " 
   using "8" "81" by (metis inteq_reflection)
 have 83: "\<turnstile> (init w \<and> (f \<and>  more ); (init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<and> finite) =
             (init w \<and> ((f \<and>  more) \<and> finite); ((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<and> finite) )"
   using "82" by fastforce
 have 84: "\<turnstile> ((\<not> ( init w) \<and>  empty) \<or>  
             (init w \<and> (f \<and>  more ); (init w \<and> f)\<^sup>\<star> \<and> fin  ( \<not>(init ( w))) \<and> finite) )  =
             ((\<not> ( init w) \<and>  empty) \<or>  
             (init w \<and> ((f \<and>  more) \<and> finite); ((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<and> finite) ) )"
      using "82" by auto
 have  9: "\<turnstile> ((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<and> finite)  =  
            ((\<not> ( init w) \<and>  empty ) \<or>  
                (init w \<and> ((f \<and>  more) \<and> finite); 
                          ((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<and> finite) ) ) "
     by (metis "61" "7" "81" "84" inteq_reflection) 
 have 10: "\<turnstile> ((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<and> finite) =
             (((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w))) \<and> finite)"
   by auto
 hence 11: "\<turnstile> ((\<not> ( init w) \<and>  empty ) \<or>  
                (init w \<and> ((f \<and>  more) \<and> finite); 
                          ((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<and> finite) ) ) =
             ((\<not> ( init w) \<and>  empty ) \<or>  
                (init w \<and> ((f \<and>  more) \<and> finite); 
                           (((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w))) \<and> finite) ) )"
  by (metis EmptyChop int_eq)               
 have 12: "\<turnstile> ((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w)) \<and> finite)  =
             ((\<not> ( init w) \<and>  empty ) \<or>  
                (init w \<and> ((f \<and>  more) \<and> finite); 
                          (((init w \<and> f)\<^sup>\<star> \<and>  fin (\<not> ( init w))) \<and> finite) ) )"
   by (metis "11" "9" inteq_reflection)
 from 12 show ?thesis 
by (metis "10" inteq_reflection while_d_def)
qed

lemma SWhileEqvEmptyOrSChopSWhile:
 "\<turnstile>  swhile (init w) do f = ((\<not> (init w) \<and> empty) \<or> (init w \<and> (f \<and> more )\<frown>swhile (init w) do f)) "
proof -
 have 1: "\<turnstile> (((f \<and> finite) \<and> more) \<and> finite) = ((f \<and> more) \<and> finite)  " 
   by auto
 show ?thesis 
 unfolding schop_d_def 
 using WhileEqvEmptyOrChopWhile[of w "LIFT (f)" ] SWhile_While[of "LIFT (init w)" f]
 by (metis While_import_finite inteq_reflection) 
qed

lemma WhileIntro:
 assumes "\<turnstile>   \<not> ( init w) \<and> f \<longrightarrow>  empty "
         "\<turnstile> init w \<and> f \<longrightarrow> ((g \<and>  more) \<and> finite ); f"
 shows   "\<turnstile> f \<and> finite \<longrightarrow>  while ( init w) do  g "
proof -
 have  1: "\<turnstile>   \<not> ( init w) \<and> f \<longrightarrow>  empty " 
   using assms by blast
 have  2: "\<turnstile> init w \<and> f \<longrightarrow> ((g \<and>  more) \<and> finite ); f " 
   using assms by blast
 have  3: "\<turnstile>  (while ( init w)  do  g \<and> finite)=  
           ((\<not>  (init w) \<and>  empty ) \<or>  
            (init w \<and> ((g \<and>  more) \<and> finite ); (while  (init w)  do  g \<and> finite))) " 
   by (rule WhileEqvEmptyOrChopWhile)
 hence 31: "\<turnstile> \<not> ( while  (init w)  do  g \<and> finite) = 
           (\<not>( (\<not>  (init w) \<and>  empty ) \<or>  
               (init w \<and> ((g \<and>  more) \<and> finite );  (while  (init w)  do  g \<and> finite))))" 
   by fastforce
 hence 32: "\<turnstile> (f \<and> \<not> ( while  (init w)  do  g \<and> finite)) = 
              (f \<and> \<not>( (\<not>(init w) \<and> empty) \<or>  
                      (init w \<and> ((g \<and>  more) \<and> finite );  (while  (init w)  do  g \<and> finite)))) "  
   by fastforce
 have  33: "\<turnstile> (f \<and> \<not>( (\<not>(init w) \<and> empty) \<or> 
                         (init w \<and> ((g \<and> more) \<and> finite);  (while  (init w)  do  g \<and> finite)))) =
              (f \<and> \<not>(\<not>(init w) \<and>  empty ) \<and> 
                     \<not>(init w \<and> ((g \<and>  more) \<and> finite );  (while  (init w)  do  g \<and> finite))) " 
   by auto
 have  34: "\<turnstile> (f \<and> \<not>(\<not>(init w) \<and> empty) \<and> 
                     \<not>((init w) \<and> (((g \<and>  more) \<and> finite );  (while  (init w)  do  g \<and> finite)))) =
               (f \<and>  ( (init w) \<or> more ) \<and> 
                       (\<not>(init w) \<or> \<not>(((g \<and> more) \<and> finite); (while (init w)  do  g \<and> finite)))) "
   by (auto simp: empty_d_def)
 have  35: "\<turnstile> (f \<and> ((init w) \<or> more) \<and> 
                   (\<not>(init w) \<or> \<not>(((g \<and> more) \<and> finite);(while (init w) do g \<and> finite)))) =
              ((f \<and> (init w) \<and> \<not> (((g \<and>  more) \<and> finite );  (while ( init w) do  g \<and> finite))) \<or>
              (f \<and> (init w) \<and> \<not>(init w)) \<or> 
              (f \<and> more \<and> \<not> (((g \<and>  more) \<and> finite );  (while ( init w) do  g \<and> finite))) \<or>
              (f \<and> more \<and> \<not>(init w))) "  
   by auto
 have  36: "\<turnstile> (f \<and> \<not> ( while  (init w)  do  g \<and> finite)) =
              ((f \<and> (init w) \<and> \<not> (((g \<and>  more) \<and> finite );  (while ( init w) do  g \<and> finite))) \<or>
              (f \<and> (init w) \<and> \<not>(init w)) \<or> 
              (f \<and> more \<and> \<not> (((g \<and>  more) \<and> finite );  (while ( init w) do  g \<and> finite))) \<or>
              (f \<and> more \<and> \<not>(init w)))"
   by (metis "32" "33" "34" "35" int_eq) 
 have  37: "\<turnstile> \<not>(f \<and> more \<and> \<not>(init w))" 
   using "1" by (auto simp: empty_d_def)
 have  38: "\<turnstile> (f \<and> more \<and> \<not> (((g \<and>  more) \<and> finite );  (while ( init w) do  g \<and> finite))) \<longrightarrow> 
             (((g \<and>  more) \<and> finite ); f \<and> 
               \<not> (((g \<and>  more) \<and> finite );  (while ( init w) do  g \<and> finite))) " 
   using "1" "2" by (auto simp: empty_d_def Valid_def)
 have  39: "\<turnstile> (f \<and> (init w) \<and> \<not> (((g \<and>  more)\<and> finite );  (while ( init w) do  g \<and> finite))) \<longrightarrow> 
             (((g \<and>  more) \<and> finite ); f  \<and> 
               \<not> (((g \<and>  more) \<and> finite );  (while ( init w) do  g \<and> finite)))" 
   using "2" by auto
 have  40: "\<turnstile> ((f \<and> (init w) \<and> \<not> (((g \<and>  more) \<and> finite );  (while ( init w) do  g \<and> finite))) \<or>
              (f \<and> (init w) \<and> \<not>(init w)) \<or> 
              (f \<and> more \<and> \<not> (((g \<and>  more) \<and> finite );  (while ( init w) do  g \<and> finite))) \<or>
              (f \<and> more \<and> \<not>(init w))) \<longrightarrow>
              ((g \<and>  more) \<and> finite ); f \<and> 
                \<not> (((g \<and>  more) \<and> finite );  (while ( init w) do  g \<and> finite)) " 
   using "39" "38" "37" "38" by fastforce
 have   4: "\<turnstile> f \<and> \<not> ( while  (init w)  do  g \<and> finite) \<longrightarrow>  
              ((g \<and>  more) \<and> finite ); f \<and> 
               \<not> (((g \<and>  more) \<and> finite );  (while ( init w) do  g \<and> finite)) "
   by (meson "36" "40" Prop11 lift_imp_trans) 
 have   50: "\<turnstile> g \<and>  more  \<longrightarrow>  more " 
   by auto
 have   5: "\<turnstile> (g \<and> more) \<and> finite \<longrightarrow> more"
   by (simp add: "50" Prop05 Prop07 finite_d_def)
 have   6: "\<turnstile> f \<and> finite \<longrightarrow>  ( while  (init w)  do  g \<and> finite)"
   using 4 5 ChopContraB by blast
 from 6 show ?thesis by (simp add: Prop12) 
qed

lemma SWhileIntro:
 assumes "\<turnstile>   \<not> ( init w) \<and> f \<longrightarrow>  empty "
         "\<turnstile> init w \<and> f \<longrightarrow> (g \<and>  more )\<frown> f"
 shows   "\<turnstile> f \<and> finite \<longrightarrow>  swhile ( init w) do  g "
proof -
 have 1: "\<turnstile> f \<and> finite \<longrightarrow>  while ( init w) do  g "
   using assms 
   using assms
   unfolding schop_d_def 
   using WhileIntro[of w f g]
   by blast
 have 2: "\<turnstile> f \<and> finite \<longrightarrow>  while ( init w) do  g  \<and> finite"
   using 1 by auto
 show ?thesis 
 using 2
 SWhile_While[of "LIFT (init w)" g] 
 While_import_finite[of "LIFT (init w)" g]
 by fastforce
qed

lemma WhileElim:
 assumes "\<turnstile>  \<not> ( init w) \<and>  empty  \<longrightarrow> g "
         "\<turnstile> init w \<and> ((f \<and>  more) \<and> finite ); g \<longrightarrow> g"
 shows   "\<turnstile>  while  (init w) do  f \<and> finite \<longrightarrow> g "
proof -
 have   1: "\<turnstile>  (while ( init w)  do  f \<and> finite)=  
             ((\<not> ( init w) \<and>  empty ) \<or>  
               (init w \<and> ((f \<and>  more) \<and> finite );  (while ( init w)  do  f \<and> finite))) " 
   by (rule WhileEqvEmptyOrChopWhile)
 hence 11: "\<turnstile> ((while ( init w)  do  f \<and> finite) \<and> \<not> g) =
             (((\<not>(init w) \<and> empty) \<or>  
               (init w \<and> ((f \<and> more) \<and> finite);(while (init w) do f \<and> finite))) \<and> \<not> g) "
    by (metis inteq_reflection lift_and_com) 
 have   2: "\<turnstile> \<not> ( init w) \<and>  empty  \<longrightarrow> g " 
   using assms by blast
 hence 21: "\<turnstile> \<not> g \<longrightarrow> \<not>(\<not> ( init w) \<and>  empty)" 
   by auto
 have  22: "\<turnstile>  ((\<not> (init w) \<and> empty) \<or> 
                 (init w \<and> ((f \<and> more) \<and> finite);(while (init w) do f \<and> finite))) \<and> \<not>  g \<longrightarrow>
              (init w \<and> ((f \<and>  more) \<and> finite );  (while ( init w)  do  f \<and> finite))  " 
   using "21" by auto
 have  23: "\<turnstile> (while ( init w)  do  f \<and> finite) \<and> \<not> g \<longrightarrow>
              (init w \<and> ((f \<and>  more) \<and> finite );  (while ( init w)  do  f \<and> finite)) \<and> \<not>  g" 
   using "11" "21" by fastforce
 have   3: "\<turnstile> (init w) \<and> (((f \<and>  more) \<and> finite ); g) \<longrightarrow> g " 
   using assms by blast
 hence 31: "\<turnstile> \<not> g \<longrightarrow> \<not>((init w) \<and> (((f \<and>  more) \<and> finite ); g))" 
   by fastforce
 have  32: "\<turnstile> (init w \<and> ((f \<and>  more) \<and> finite );  (while ( init w)  do  f \<and> finite)) \<and> \<not> g \<longrightarrow> 
             ((((f \<and>  more) \<and> finite );  (while  (init w)  do  f \<and> finite)) \<and> 
                 \<not> (((f \<and>  more) \<and> finite ); g)) \<and> \<not>g" 
   using "31" by fastforce
 have   4: "\<turnstile>  (while  (init w)  do  f \<and> finite) \<and> \<not>  g \<longrightarrow>  
              (((f \<and>  more) \<and> finite );  (while  (init w)  do  f \<and> finite)) \<and> 
                 \<not> (((f \<and>  more) \<and> finite ); g) "
   by (meson "23" "32" Prop12 lift_imp_trans) 
 have   5: "\<turnstile> (f \<and>  more) \<and> finite  \<longrightarrow>  more " 
   by auto
 from 4 5 show ?thesis using 
   ChopContraB[of "LIFT(while  (init w)  do  f \<and> finite)" "LIFT(g)" "LIFT(((f \<and>  more) \<and> finite ))"]
   by auto
qed

lemma SWhileElim:
 assumes "\<turnstile>  \<not> ( init w) \<and>  empty  \<longrightarrow> g "
         "\<turnstile> init w \<and> (f \<and>  more )\<frown> g \<longrightarrow> g"
 shows   "\<turnstile>  swhile  (init w) do  f \<longrightarrow> g "
using assms
unfolding schop_d_def 
using WhileElim[of w g f] SWhile_While[of "LIFT (init w)" f] 
 While_import_finite[of "LIFT (init w)" f]
by fastforce


lemma BaWhileImpWhile:
 "\<turnstile>    ba (f \<longrightarrow> g) \<and> finite \<longrightarrow> ( while  (init w)  do  f) \<longrightarrow> ( while  (init w)  do  g)"
proof -
 have  1: "\<turnstile> (f \<longrightarrow> g) \<longrightarrow> ((init w \<and> f)  \<longrightarrow> (init w \<and> g)) " 
   by auto
 hence 2: "\<turnstile>  ba (f \<longrightarrow> g)  \<longrightarrow>  ba ((init w \<and> f)  \<longrightarrow> (init w\<and> g)) " 
   using BaImpBa by blast
 have  3: " \<turnstile>  ba ((init w \<and> f)  \<longrightarrow> (init w\<and> g)) \<and> finite \<longrightarrow> ((init w\<and> f)\<^sup>\<star> \<longrightarrow> (init w\<and> g)\<^sup>\<star>) " 
   by (rule BaCSImpCS)
 have  4: "\<turnstile>  ba (f \<longrightarrow> g) \<and> finite \<longrightarrow> ((init w\<and> f)\<^sup>\<star>\<and> fin (\<not> ( init w)) 
             \<longrightarrow> (init w\<and> g)\<^sup>\<star>\<and> fin (\<not> (init w))) " 
   using "2" "3" by fastforce
 from 4 show ?thesis by (simp add: while_d_def)
qed

lemma SBaSWhileImpSWhile:
 "\<turnstile>    sba (f \<longrightarrow> g) \<longrightarrow> ( swhile  (init w)  do  f) \<longrightarrow> ( swhile  (init w)  do  g)"
proof -
 have  1: "\<turnstile> (f \<longrightarrow> g) \<longrightarrow> ((init w \<and> f)  \<longrightarrow> (init w \<and> g)) " 
   by auto
 hence 2: "\<turnstile>  sba (f \<longrightarrow> g) \<longrightarrow>  sba ((init w \<and> f)  \<longrightarrow> (init w\<and> g)) " 
   by (rule SBaImpSBa)
 have  3: " \<turnstile> sba ((init w \<and> f) \<longrightarrow> (init w\<and> g)) \<longrightarrow> 
               (schopstar (init w\<and> f) \<longrightarrow> schopstar (init w\<and> g)) " 
   by (rule SBaSCSImpSCS)
 have  4: "\<turnstile>  sba (f \<longrightarrow> g) \<longrightarrow> (schopstar (init w \<and> f)\<and> sfin (\<not> ( init w)) \<longrightarrow> 
          schopstar (init w\<and> g) \<and> sfin (\<not> (init w))) " 
   using "2" "3" by fastforce
 from 4 show ?thesis by (simp add: swhile_d_def)
qed

lemma WhileImpWhile:
 assumes "\<turnstile>   f\<longrightarrow> g "
 shows "\<turnstile> ( while  (init w)  do  f) \<and> finite \<longrightarrow> ( while  (init w)  do  g) "
proof -
 have  1: "\<turnstile> f \<longrightarrow> g " 
   using assms by auto
 hence 2: "\<turnstile>  ba (f \<longrightarrow> g) " 
   by (rule BaGen)
 have  3: "\<turnstile>  ba (f \<longrightarrow> g) \<and> finite \<longrightarrow> ( while  (init w)  do  f) \<longrightarrow> ( while  (init w)  do  g) " 
   by (rule BaWhileImpWhile)
 have  4: "\<turnstile> ba (f \<longrightarrow> g)  \<longrightarrow> ( while  (init w)  do  f \<and> finite) \<longrightarrow> ( while  (init w)  do  g)"
   using "3" by (auto simp: Valid_def)
 from 2 4 show ?thesis using MP by blast 
qed

lemma SWhileImpSWhile:
 assumes "\<turnstile>   f\<longrightarrow> g "
 shows "\<turnstile> ( swhile  (init w)  do  f) \<longrightarrow> ( swhile  (init w)  do  g) "
proof -
 have  1: "\<turnstile> f \<longrightarrow> g " 
   using assms by auto
 hence 2: "\<turnstile>  sba (f \<longrightarrow> g) " 
   by (rule SBaGen)
 have  3: "\<turnstile>  sba (f \<longrightarrow> g) \<longrightarrow> ( swhile  (init w)  do  f) \<longrightarrow> ( swhile  (init w)  do  g) " 
   by (rule SBaSWhileImpSWhile)
 from 2 3 show ?thesis using MP by blast
qed

lemma NotSkipEqvEmptyOrSkipChopMore: 
 "\<turnstile> (\<not> skip) = (empty \<or> skip;more)" 
using ChopEmpty WnextEqvEmptyOrNext unfolding  next_d_def wnext_d_def empty_d_def
by (metis inteq_reflection)

lemma FiniteInfSplitAnd: 
 "\<turnstile> f = ((f \<or> inf) \<and> (f \<or> finite)) " 
unfolding finite_d_def by fastforce

lemma ChopAndInf_alt_1: 
 "\<turnstile> ((f;g) \<and> inf) = (f;(g \<and> inf) \<or> (f \<and> inf);(g \<and> finite)) " 
proof -
 have 1: "\<turnstile> ((f;g) \<and> inf) = (((f \<and> finite);g \<and> inf) \<or> ((f \<and> inf);g \<and> inf)) " 
  by (metis ChopAndInf OrChopEqv OrFiniteInf inteq_reflection)
 have 2: "\<turnstile> ((f \<and> finite);g \<and> inf) = (f \<and> finite);(g \<and> inf) " 
   by (simp add: ChopAndInf)
 have 3: "\<turnstile> ((f \<and> inf);g \<and> inf) = ( (f \<and> inf);(g \<and> inf) \<or> (f \<and> inf);(g \<and> finite)) " 
   by (metis AndInfChopEqvAndInf ChopAndInf int_simps(27) inteq_reflection)
 have 4: "\<turnstile> ((f \<and> finite);(g \<and> inf) \<or> (f \<and> inf);(g \<and> inf)) = (f;(g \<and> inf)) " 
  by (metis "1" ChopAndInf inteq_reflection)
 have 5: "\<turnstile> (((f \<and> finite);g \<and> inf) \<or> ((f \<and> inf);g \<and> inf)) =
            (f;(g \<and> inf) \<or> (f \<and> inf);(g \<and> finite)) " 
  using 2 3 4 by fastforce
 show ?thesis 
 by (meson "1" "5" Prop04 Prop11)
qed

lemma ChopAndInf_alt_2: 
 "\<turnstile> ((f;g) \<and> inf) = ( (f \<and> finite);(g \<and> inf) \<or> (f \<and> inf);(g \<and> inf) \<or> ((f \<and> inf);(g \<and> finite))) " 
proof -
 have 1: "\<turnstile> (f;(g \<and> inf)) = ((f \<and> finite);(g \<and> inf) \<or> (f \<and> inf);(g \<and> inf)) " 
  using OrChopEqvRule OrFiniteInf by blast
 show ?thesis using ChopAndInf_alt_1 1 by fastforce
qed

lemma AndEmptyChopAndMore: 
 "\<turnstile> ((f \<and> empty);g \<and> more) = (f \<and> empty);(g \<and> more) " 
proof -
 have 1: "\<turnstile> (f \<and> empty);g = (init f \<and> g) " 
   by (meson InitAndEmptyEqvAndEmpty LeftChopEqvChop Prop04 StateAndEmptyChop)
 have 2: "\<turnstile> ((f \<and> empty);g \<and> more) = (init f \<and> g \<and> more) " 
  using 1 by fastforce
 have 3: "\<turnstile> (init f \<and> g \<and> more) = (f \<and> empty);(g \<and> more) " 
  by (meson InitAndEmptyEqvAndEmpty LeftChopEqvChop Prop04 StateAndEmptyChop)
 show ?thesis using 2 3 
 by (meson Prop04 Prop11)
qed


lemma OrChopOr:
 "\<turnstile> (f1 \<or> f2);(g1 \<or> g2) = ( f1;g1 \<or> f1;g2 \<or> f2;g1 \<or> f2;g2) " 
using ChopOrEqv[of "LIFT (f1 \<or> f2)" g1 g2 ] OrChopEqv[of f1 f2 g1] OrChopEqv[of f1 f2 g2]
by fastforce 

lemma ChopAndMore_alt_1: 
 "\<turnstile> (f;g \<and> more) = ( (f \<and> empty);(g \<and> more) \<or> (f \<and> more);g) " 
proof -
 have 1: "\<turnstile> (f;g) = ((f \<and> empty);g \<or> (f \<and> more);g) " 
  using EmptyOrMoreSplit OrChopEqvRule by blast
 have 2: "\<turnstile> (f;g \<and> more) = (((f \<and> empty);g \<and> more) \<or> ((f \<and> more);g \<and> more)) " 
  using 1 by fastforce
 have 3: "\<turnstile> ((f \<and> empty);g \<and> more) = (f \<and> empty);(g \<and> more) " 
  using AndEmptyChopAndMore by blast
 have 4: "\<turnstile> ((f \<and> more);g \<and> more) = ((f \<and> more);g ) " 
   by (meson AndChopB MoreChopImpMore Prop04 Prop10 int_simps(4) lift_imp_trans)
 show ?thesis 
 by (metis "2" "3" "4" int_eq)
qed

lemma ChopAndMore_alt_2: 
 "\<turnstile> (f;g \<and> more) = ( (f \<and> empty);(g \<and> more) \<or> (f \<and> more);(g \<and> empty) \<or> (f \<and> more);(g \<and> more) ) " 
using ChopAndMore_alt_1[of f g] 
by (meson ChopOrEqvRule EmptyOrMoreSplit Prop06) 


lemma WPowerstar_And_Inf_Eqv_WPowerplus_And_Inf: 
 "\<turnstile> (wpowerstar f \<and> inf) = (wpowerstar f;f \<and> inf) " 
proof -
 have 1: "\<turnstile> wpowerstar f = ( empty \<or> wpowerstar f;f) " 
   by (meson ChopEmpty Prop04 WPowerstar_unfoldr_eq)
 have 2: "\<turnstile> (( empty \<or> wpowerstar f;f) \<and> inf) = (wpowerstar f;f \<and> inf) "
   using EmptyImpFinite unfolding finite_d_def by fastforce
 show ?thesis using 1 2 
 by fastforce
qed

lemma WPowerstar_Inf_Import: 
 "\<turnstile> ((wpowerstar f) \<and> inf) = ((wpowerstar f) \<and> finite);(f \<and> inf) " 
proof -
 have 2: "\<turnstile> ((istar f) \<and> inf) = ((wpowerstar f) \<and> inf) " 
   by (metis IStarWPowerstar int_eq lift_and_com)
 have 3:"\<turnstile> (wpowerstar f) = (empty \<or> (wpowerstar f);f) " 
   by (meson WPowerstar_unfoldR WPowerstar_unfoldr_eq int_iffD2 int_iffI) 
 have 4: "\<turnstile> ((empty \<or> (wpowerstar f);f) \<and> inf) = ((wpowerstar f);f \<and> inf) " 
   using EmptyImpFinite unfolding finite_d_def by fastforce
 have 6: "\<turnstile> empty \<longrightarrow> inf \<longrightarrow> (wpowerstar f \<and> finite);(f \<and> inf)"
    using  EmptyImpFinite unfolding finite_d_def by fastforce
have 61: "\<turnstile> (f;( (wpowerstar f \<and> finite);(f \<and> inf)) \<and> inf) =
              ((f \<and> inf) \<or>  ( (f \<and> finite);( ( (wpowerstar f \<and> finite);(f \<and> inf)) \<and> inf))) " 
   by (simp add: ChopAndInfB)
 have 62: "\<turnstile> ( (f \<and> finite);( ( (wpowerstar f \<and> finite);(f \<and> inf)) \<and> inf)) =
             ( (f \<and> finite);( ( (wpowerstar f \<and> finite);(f \<and> inf)))) " 
   by (metis (no_types, lifting) ChopAndInfB Prop03 Prop10 Prop12 inteq_reflection)
 have 63: "\<turnstile> (f \<and> inf) \<and> inf \<longrightarrow> (wpowerstar f \<and> finite);(f \<and> inf) "
   by (meson ChopEqvChop Prop03 Prop10 Prop11 Prop12 WPowerstarChopEqvChopOrRule WPowerstar_import_finite lift_and_com)
 have 64: "\<turnstile> ( (f \<and> finite);( ( (wpowerstar f \<and> finite);(f \<and> inf)))) \<and> inf \<longrightarrow> (wpowerstar f \<and> finite);(f \<and> inf)"
 by (metis (no_types, lifting) ChopAndInf ChopAssoc EmptyOrChopEqv Prop03 Prop12 WPowerstarEqv WPowerstar_import_finite inteq_reflection)
 have 7: "\<turnstile> f;( (wpowerstar f \<and> finite);(f \<and> inf)) \<and> inf \<longrightarrow> (wpowerstar f \<and> finite);(f \<and> inf) "
  using "61" "63" "64" ChopAndInf by fastforce
 have 8: "\<turnstile>  (f;finite \<and> inf) = (f \<and> inf) " 
   using ChopAndInfB[of f "LIFT finite"] unfolding finite_d_def 
   by (metis AndInfEqvChopFalse ChopAndInf FiniteInfSplitAnd finite_d_def int_simps(26) inteq_reflection lift_and_com) 
 have 9: "\<turnstile> f;finite \<and> inf \<longrightarrow> (wpowerstar f \<and> finite);(f \<and> inf)" 
   by (metis "7" "8" ChopAndInf_alt_1 PowerstarEqvSemhelp2 Prop03 inteq_reflection lift_imp_trans)
 have 10: "\<turnstile> (inf \<longrightarrow> (wpowerstar f \<and> finite);(f \<and> inf)) = 
             (finite \<or> (wpowerstar f \<and> finite);(f \<and> inf)) " 
           unfolding finite_d_def by fastforce
 have 11: "\<turnstile> f;(inf \<longrightarrow> (wpowerstar f \<and> finite);(f \<and> inf)) = ((f;finite) \<or> f;((wpowerstar f \<and> finite);(f \<and> inf))) " 
    by (metis "10" ChopOrEqv inteq_reflection)   
 have 12: " \<turnstile> empty \<or> f;(inf \<longrightarrow> (wpowerstar f \<and> finite);(f \<and> inf)) \<longrightarrow> inf \<longrightarrow> (wpowerstar f \<and> finite);(f \<and> inf)"
   by (metis "11" "6" "7" "9" Prop02 Prop09 int_eq)
 have 13:"\<turnstile> ((istar f) ) \<longrightarrow> (inf \<longrightarrow> ((wpowerstar f) \<and> finite);(f \<and> inf))" 
   using IStarWeakInduct[of f "LIFT  (inf \<longrightarrow> ((wpowerstar f) \<and> finite);(f \<and> inf))"  ] 
   using "12" by blast
 have 14:"\<turnstile> ((wpowerstar f) \<and> finite);(f \<and> inf) \<longrightarrow> ((wpowerstar f);f \<and> inf)"
  by (meson AndChopA ChopAndInf int_iffD2 lift_imp_trans)
 show ?thesis
   by (metis "10" "13" "14" "2" "3" "4" InfEqvNotFinite Prop07 int_iffI inteq_reflection)
qed

lemma Prop_Var_Empty_Finite_Import: 
"\<turnstile> ( ($p \<and> empty) \<and> finite) =  ($p \<and> empty) " 
using EmptyImpFinite by auto

lemma Not_Prop_Var_And_Empty_Inf: 
"\<turnstile>  \<not>(($p \<and> empty) \<and> inf) " 
using FiniteAndEmptyEqvEmpty InfEqvNotFinite by fastforce

lemma Init_And_Finite_Absorb: 
"\<turnstile> init (f \<and> finite) = init f"
by (metis EmptySChop Initprop(1) init_d_def int_simps(17) inteq_reflection lift_and_com schop_d_def)

lemma Chop_And_More: 
 "\<turnstile> (f;g \<and> more) = ( (f \<and> more);g \<or> f;(g \<and> more)) " 
proof -
 have 1: "\<turnstile> f;g = 
   ((f \<and> empty);(g \<and> empty) \<or> (f \<and> empty);(g \<and> more)   \<or> 
    (f \<and> more);(g \<and> empty) \<or> (f \<and> more);(g \<and> more) ) "
    by (meson ChopEqvChop EmptyOrMoreSplit OrChopOr Prop04 Prop11) 
 have 2: "\<turnstile> (f;g \<and> more) = (
            ( (f \<and> empty);(g \<and> empty) \<and> more) \<or> 
            ( (f \<and> empty);(g \<and> more) \<and> more) \<or>
            ( (f \<and> more);(g \<and> empty) \<and> more) \<or>
            ( (f \<and> more);(g \<and> more) \<and> more) ) " 
   using 1 by fastforce
 have 3: "\<turnstile> ( ( (f \<and> empty);(g \<and> empty) \<and> more)) =
            ( (f;g \<and> empty) \<and> more)  "
   by (metis ChopAndEmptyEqvEmptyChopEmpty int_simps(1) inteq_reflection) 
 have 4: "\<turnstile> \<not> ( (f;g \<and> empty) \<and> more) " 
    unfolding empty_d_def by fastforce
 have 5:"\<turnstile> ( (f \<and> empty);(g \<and> more) \<and> more) = ( (f \<and> empty);(g \<and> more) ) "
  by (meson ChopAndB ChopEmpty ChopMoreImpMore Prop04 Prop10 lift_imp_trans)
 have 6: "\<turnstile> ( (f \<and> more);(g \<and> empty) \<and> more) = ( (f \<and> more);(g \<and> empty) ) "
  using AndChopB MoreChopImpMore by fastforce
 have 7: "\<turnstile> ( (f \<and> more);(g \<and> more) \<and> more) = ( (f \<and> more);(g \<and> more)) "
  by (meson ChopAndB ChopMoreImpMore Prop10 Prop11 lift_imp_trans)
 have 8: "\<turnstile> (f \<and> more);g = ( (f \<and> more);(g \<and> empty) \<or> (f \<and> more);(g \<and> more) ) "
  using ChopOrEqvRule EmptyOrMoreSplit by blast 
 have 9: "\<turnstile> f;(g \<and> more) = ( (f \<and> empty);(g \<and> more) \<or> (f \<and> more);(g \<and> more) ) "
  by (simp add: EmptyOrMoreSplit OrChopEqvRule) 
 have 10: "\<turnstile> ( (f \<and> more);g \<or> f;(g \<and> more)) =
            ( (f \<and> more);(g \<and> empty) \<or> (f \<and> more);(g \<and> more) \<or>
              (f \<and> empty);(g \<and> more) \<or> (f \<and> more);(g \<and> more) ) "
  using 8 9 by fastforce
 have 11: "\<turnstile>  ( ( (f \<and> empty);(g \<and> more) ) \<or> ( (f \<and> more);(g \<and> empty) ) \<or> ( (f \<and> more);(g \<and> more)) ) =
             ( (f \<and> more);g \<or> f;(g \<and> more)) " 
   using 10 8 9 by fastforce
 show ?thesis
 by (meson "11" ChopAndMore_alt_2 Prop11 lift_imp_trans)     
qed

end
